// Programming Catalog
//
#ifndef CATALOG_CPP
#define CATALOG_CPP
#ifndef CATALOG_H
  #include "catalog.h"
#endif

/****************************************************************************/
/*************************** Key Word Token Class ***************************/
/****************************************************************************/
KeyWordToken::KeyWordToken():
_SearchStr(NULL),
_Delims(NULL),
_WhiteSpace(NULL),
_Index(0),
_Start(0),
_Space(strcpy(new char[2], " "))
{}

/****************************************************************************/
KeyWordToken::KeyWordToken(char* Str_, short Start_, char* Delims_, char* Wspc_):
_SearchStr(Str_),
_Delims(Delims_),
_WhiteSpace(Wspc_),
_Index(Start_),
_Start(Start_),
_Space(strcpy(new char[2], " "))
{}

/****************************************************************************/
KeyWordToken::~KeyWordToken()
{
  delete[] _Space;
  _Space = NULL;
}

/****************************************************************************/
short KeyWordToken::FindToken(char* Token_)
{
  char* Delimiters_ = (_Delims == NULL) ? _Space:_Delims;

  size_t NextPos_;
  size_t Index_ = _Index;
  short CopyLen_ = FindNextWord(_SearchStr, Delimiters_, Index_, NextPos_);
  _Index = Index_;

  if (CopyLen_)
  {
    strncpy(Token_, &_SearchStr[_Index], CopyLen_);
    Token_[CopyLen_] = 0;

    if (_WhiteSpace)
    {
      RemoveChar(Token_, _WhiteSpace);
      CopyLen_ = strlen(Token_);
    }

    _Index = NextPos_;
  }

  return CopyLen_;
}

/****************************************************************************/
/****************************************************************************/
Catalog::WeightedMatch::WeightedMatch(size_t Match_, long RecNum_):
_Weight(Match_),
_RecNum(RecNum_),
_Next(NULL)
{}

/****************************************************************************/
Catalog::WeightedMatch::WeightedMatch(size_t Match_, long RecNum_, WeightedMatch* Link_):
_Weight(Match_),
_RecNum(RecNum_),
_Next(Link_)
{}

/****************************************************************************/
Catalog::WeightedMatch::~WeightedMatch()
{
  delete _Next;
  _Next = NULL;
}

/****************************************************************************/
/************************ Text Graphics Screen Setup ************************/
/****************************************************************************/
void Catalog::Initialize()
{
  GetConfiguration();

  InitDialogs();
  InitWindows();

  SetLabelWindow();
  SetRadioWindow();
}

/****************************************************************************/
/****************************************************************************/
char* Catalog::TrimAndDnCase(char* str)
{
  size_t x, l;

  for (l = strlen(str); l;)
    if (isspace(str[--l]))
      str[l] = '\0';
    else
      break;

  for (l = 0; str[l] && isspace(str[l]); l++);
  for (x = 0; (str[x] = tolower(str[l])); x++, l++);

  return str;
}

/****************************************************************************/
char* Catalog::RdStr(istream& In_, char* Dest_, int Size_, char Delimiter_)
{
  int EndOfRead_;

  In_.getline(Dest_, Size_, Delimiter_);
  if (strlen(Dest_) == size_t(Size_ - 1))
    EndOfRead_ = In_.eof();
  else
    EndOfRead_ = 1;

  if (!EndOfRead_)
    for (size_t Count_ = 0; !In_.eof() && In_.get() != Delimiter_; Count_++);

  return Dest_;
}

/****************************************************************************/
void Catalog::ReadConfigFile(ifstream& Fin_, char* PrintFile_,
			     char* PrintDev_, char* TempFile_, Boolean& ViewDeleted_)
{
  char Buffer_[80];
  char* Field_;
  char* Value_;
  int FieldNo_ = 0;
  Boolean FieldDone_[4];

  FieldDone_[0] =
  FieldDone_[1] =
  FieldDone_[2] =
  FieldDone_[3] = 0;

  if (strcmp(TrimAndDnCase(RdStr(Fin_, Buffer_, 80)), "[catalog]") == 0)
    while (!Fin_.eof() && FieldNo_ < 4)
    {
      Field_ = TrimAndDnCase(RdStr(Fin_, Buffer_, 80, '='));

      if (!strcmp(Field_, "printerfilename") && FieldDone_[0] == 0)
      {
	Value_ = TrimAndDnCase(RdStr(Fin_, Buffer_, 80));
	if (Value_ && strlen(Value_))
	  strcpy(PrintFile_, Value_);
	FieldDone_[0] = 1;
	++FieldNo_;
      }
      else if (!strcmp(Field_, "printerdevicename") && FieldDone_[1] == 0)
      {
	Value_ = TrimAndDnCase(RdStr(Fin_, Buffer_, 80));
	if (Value_ && strlen(Value_))
	  strcpy(PrintDev_, Value_);
	FieldDone_[1] = 1;
	++FieldNo_;
      }
      else if (!strcmp(Field_, "temporaryfilename") && FieldDone_[2] == 0)
      {
	Value_ = TrimAndDnCase(RdStr(Fin_, Buffer_, 80));
	if (Value_ && strlen(Value_))
	  strcpy(TempFile_, Value_);
	FieldDone_[2] = 1;
	++FieldNo_;
      }
      else if (!strcmp(Field_, "viewdeletedrecords") && FieldDone_[3] == 0)
      {
	Value_ = TrimAndDnCase(RdStr(Fin_, Buffer_, 80));
	if (Value_ && strlen(Value_))
	  ViewDeleted_ = !strcmp(Value_, "false") || !strcmp(Value_, "0") ? 0:
			 !strcmp(Value_, "true") || !strcmp(Value_, "1") ? 1:
			 0;

	FieldDone_[3] = 1;
	++FieldNo_;
      }
      else
	RdStr(Fin_, Buffer_, 80);
    }
}

/****************************************************************************/
void Catalog::GetConfiguration()
{
  ifstream cfgin_("catalog.cfg");

  strcpy(_PrinterFileName, "");
  strcpy(_PrinterDeviceName, "");
  strcpy(_TempFileName, "");
  _ViewDeleted = 0;

  // Read tweak file and parse field value into variables
  if (cfgin_.good() && !cfgin_.eof())
    ReadConfigFile(cfgin_,
		   _PrinterFileName,
		   _PrinterDeviceName,
		   _TempFileName,
		   _ViewDeleted);

  if (!strlen(_PrinterFileName))
    strcpy(_PrinterFileName, CATALOG_DEFAULT_PRINTFILE);

  if (!strlen(_PrinterDeviceName))
    strcpy(_PrinterDeviceName, CATALOG_DEFAULT_PRINTDEVICE);

  if (!strlen(_TempFileName))
    strcpy(_TempFileName, CATALOG_DEFAULT_TEMPFILE);

  _PrinterDeviceNumber = _PrinterDeviceName[3] - '1';
  _ShowErased = _ViewDeleted;
  cfgin_.close();
}

/****************************************************************************/
void Catalog::InitDialogs()
{
  CatalogScreenPromptBoxesSetup();
  CatalogScreenPushButtonsSetup();
  FileScreenSetup();
  TextLabelSetup();
  RadioBtnSetup();
}

/****************************************************************************/
void Catalog::InitWindows()
{
  _EntryWindow.MaxControl(16);
  _BtnWindow.MaxControl(12);
  _LblWindow.MaxControl(5);
  _RadWindow.MaxControl(3);
}

/****************************************************************************/
void Catalog::DeleteWindows()
{
  _EntryWindow.Delete();
  _BtnWindow.Delete();
}

/****************************************************************************/
void Catalog::ShowWindows()
{
  _EntryWindow.Show();
  _BtnWindow.Show();

  _FileNameLbl->Show();
}

/****************************************************************************/
void Catalog::RemoveWindows()
{
  _EntryWindow.Remove();
  _BtnWindow.Remove();
}

/****************************************************************************/
void Catalog::HideWindows()
{
  _EntryWindow.Hide();
  _BtnWindow.Hide();

  _FileStatusLbl->Hide();
  _FileNameLbl->Hide();
  _RecordNumLbl->Hide();
}

/****************************************************************************/
/****************************************************************************/
void Catalog::TextLabelSetup()
{
  // Title Label setup
  _TitleLbl = TextLabel::Make("Programming Catalog");
  _TitleLbl->SetXYpos(30, 1);

  // File status label setup
  _FileStatusLbl = TextLabel::Make("End Of File");
  _FileStatusLbl->SetXYpos(55, 3);

  // File saftey label setup
  _FileSafetyLbl = TextLabel::Make("Specified File Already Exists Do You Wish To Overwrite?");
  _FileSafetyLbl->SetXYpos(20, 3);

  // Record number label setup
  _RecordNumLbl = TextLabel::Make("Record #0");
  _RecordNumLbl->SetXYpos(2, 3);

  // File name label setup
  _FileNameLbl = TextLabel::Make("No File Loaded");
  _FileNameLbl->SetXYpos(2, 2);
}

/****************************************************************************/
void Catalog::CatalogScreenPushButtonsSetup()
{
  _NextBtn = PushButton::Make("&Next");
  _NextBtn->SetXYpos(3, 23);

  _PreviousBtn = PushButton::Make("P&revious");
  _PreviousBtn->SetXYpos(15, 23);

  _FirstBtn = PushButton::Make("&First");
  _FirstBtn->SetXYpos(29, 23);

  _LastBtn = PushButton::Make("Las&t");
  _LastBtn->SetXYpos(40, 23);

  _SearchBtn = PushButton::Make("&Search");
  _SearchBtn->SetXYpos(51, 23);

  _RecordNumBtn = PushButton::Make("R&ecord #");
  _RecordNumBtn->SetXYpos(63, 23);

  _DeleteBtn = PushButton::Make("&Delete");
  _DeleteBtn->SetXYpos(3, 24);

  _ModifyBtn = PushButton::Make("M&odify");
  _ModifyBtn->SetXYpos(15, 24);

  _AddBtn = PushButton::Make("&Add");
  _AddBtn->SetXYpos(29, 24);

  _PrintBtn = PushButton::Make("&Print");
  _PrintBtn->SetXYpos(40, 24);

  _LoadBtn = PushButton::Make("&Load");
  _LoadBtn->SetXYpos(51, 24);

  _SaveQuitBtn = PushButton::Make("&Quit/Backup");
  _SaveQuitBtn->SetXYpos(63, 24);
}

/****************************************************************************/
void Catalog::CatalogScreenPromptBoxesSetup()
{
  int Index_;
  PromptBox* ChainPb_;

  _NamePb = PromptBox::Make("&Name: ");
  _NamePb->SetInputLength(64);
  _NamePb->SetXYpos(2, 5);
  _NamePb->SetNoEmpty();

  _PurposePb = PromptBox::Make("&Purpose: ");
  _PurposePb->SetInputLength(64);
  _PurposePb->SetXYpos(2, 6);

  ChainPb_ = PromptBox::Make("         ");
  ChainPb_->SetInputLength(64);
  ChainPb_->SetXYpos(2, 7);
  _PurposePb->Chain(ChainPb_);

  _UsesPb = PromptBox::Make("&Uses: ");
  _UsesPb->SetInputLength(64);
  _UsesPb->SetXYpos(2, 8);

  for (Index_ = 9; Index_ < 12; Index_++)
  {
    ChainPb_ = PromptBox::Make("      ");
    ChainPb_->SetInputLength(64);
    ChainPb_->SetXpos(2);
    ChainPb_->SetYpos(Index_);
    _UsesPb->Chain(ChainPb_);
  }

  _BenefitsPb = PromptBox::Make("&Benefits: ");
  _BenefitsPb->SetInputLength(64);
  _BenefitsPb->SetXYpos(2, 12);

  for (Index_ = 13; Index_ < 16; Index_++)
  {
    ChainPb_ = PromptBox::Make("          ");
    ChainPb_->SetInputLength(64);
    ChainPb_->SetXpos(2);
    ChainPb_->SetYpos(Index_);
    _BenefitsPb->Chain(ChainPb_);
  }

  _RelatedPb = PromptBox::Make("&Related: ");
  _RelatedPb->SetInputLength(64);
  _RelatedPb->SetXYpos(2, 16);

  _LocationPb = PromptBox::Make("&Location: ");
  _LocationPb->SetInputLength(64);
  _LocationPb->SetXYpos(2, 17);

  ChainPb_ = PromptBox::Make("          ");
  ChainPb_->SetInputLength(64);
  ChainPb_->SetXYpos(2, 18);
  _LocationPb->Chain(ChainPb_);

  _SearchPb = PromptBox::Make("Search: ");
  _SearchPb->SetInputLength(66);
  _SearchPb->SetXYpos(2, 20);
  _SearchPb->Deactivate();

  _RecordNumPb = PromptBox::Make("Record Number: ");
  _RecordNumPb->SetInputLength(16);
  _RecordNumPb->SetXYpos(2, 20);
  _RecordNumPb->Deactivate();
}

/****************************************************************************/
void Catalog::FileScreenSetup()
{
  // Creating file operations buttons
  _OkBtn = PushButton::Make("&OK");
  _OkBtn->SetXYpos(5, 8);
  _CancelBtn = PushButton::Make("&Cancel");
  _CancelBtn->SetXYpos(21, 8);
  _CreateNewBtn = PushButton::Make("Create &New");
  _CreateNewBtn->SetXYpos(41, 8);

  // Creating File operations PromptBoxes
  _FilePb = PromptBox::Make("");
  _FilePb->SetXYpos(5, 6);
  _FilePb->SetInputLength(20);

  // Creating Temporary File operations PromptBoxes
  _TempFilePb = PromptBox::Make("");
  _TempFilePb->SetXYpos(5, 6);
  _TempFilePb->SetInputLength(20);
  _TempFilePb->SetInputString(_TempFileName);
  _TempFilePb->Select();
}

/****************************************************************************/
void Catalog::RadioBtnSetup()
{
  // Creating Save Quit operations buttons
  _BackupToADriveBtn = RadioButton::Make("Backup To Diskette On Drive &A: ", 1);
  _BackupToADriveBtn->SetXYpos(5, 5);
  _BackupToBDriveBtn = RadioButton::Make("Backup To Diskette On Drive &B: ", 0);
  _BackupToBDriveBtn->SetXYpos(5, 6);
  _NoBackupBtn = RadioButton::Make("&Quit Without Backing Up      : ", 0);
  _NoBackupBtn->SetXYpos(5, 7);
}

/****************************************************************************/
/****************************************************************************/
void Catalog::SetRadioWindow()
{
  _RadWindow[0] = _BackupToADriveBtn;
  _RadWindow[1] = _BackupToBDriveBtn;
  _RadWindow[2] = _NoBackupBtn;

  _RadWindow.UseColor();
}

/****************************************************************************/
void Catalog::SetLabelWindow()
{
  _LblWindow[0] = _TitleLbl;
  _LblWindow[1] = _FileStatusLbl;
  _LblWindow[2] = _RecordNumLbl;
  _LblWindow[3] = _FileNameLbl;
  _LblWindow[4] = _FileSafetyLbl;
}

/****************************************************************************/
void Catalog::SetCatalogButtonWindow()
{
  _BtnWindow[0] = _NextBtn;
  _BtnWindow[1] = _PreviousBtn;
  _BtnWindow[2] = _FirstBtn;
  _BtnWindow[3] = _LastBtn;
  _BtnWindow[4] = _SearchBtn;
  _BtnWindow[5] = _RecordNumBtn;
  _BtnWindow[6] = _DeleteBtn;
  _BtnWindow[7] = _ModifyBtn;
  _BtnWindow[8] = _AddBtn;
  _BtnWindow[9] = _PrintBtn;
  _BtnWindow[10] = _LoadBtn;
  _BtnWindow[11] = _SaveQuitBtn;

  _BtnWindow.UseColor();
  _BtnWindow.NoDelay();
}

/****************************************************************************/
void Catalog::SetCatalogPromptWindow()
{
  _EntryWindow[0] = _NamePb;

  _EntryWindow[1] = _PurposePb;
  _EntryWindow[2] = _PurposePb->NextChain();

  _EntryWindow[3] = _UsesPb;
  _EntryWindow[4] = _UsesPb->NextChain();
  _EntryWindow[5] = _UsesPb->NextChain()->NextChain();
  _EntryWindow[6] = _UsesPb->NextChain()->NextChain()->NextChain();

  _EntryWindow[7]  = _BenefitsPb;
  _EntryWindow[8]  = _BenefitsPb->NextChain();
  _EntryWindow[9]  = _BenefitsPb->NextChain()->NextChain();
  _EntryWindow[10] = _BenefitsPb->NextChain()->NextChain()->NextChain();

  _EntryWindow[11]  = _RelatedPb;

  _EntryWindow[12]  = _LocationPb;
  _EntryWindow[13]  = _LocationPb->NextChain();

  _EntryWindow[14] = _SearchPb;
  _EntryWindow[15] = _RecordNumPb;

  _EntryWindow.SetHotKeyColor(LIGHTGREEN);
  _EntryWindow.UseColor();
}

/****************************************************************************/
void Catalog::SetCatalogWindow()
{
  HideWindows();
  RemoveWindows();

  SetCatalogButtonWindow();
  SetCatalogPromptWindow();

  _RecordNumLbl->Activate();
  _FileNameLbl->Activate();
}

/****************************************************************************/
void Catalog::SetFileWindow()
{
  // Reseting dialog windows to contain no dialogs
  HideWindows();
  RemoveWindows();

  // Add dialogs to prompt window
  _BtnWindow[0] = _OkBtn;
  _BtnWindow[1] = _CancelBtn;
  _BtnWindow[2] = _CreateNewBtn;
  _EntryWindow[0] = _FilePb;

  _BtnWindow.UseColor();
  _BtnWindow.SetDelay();

  _EntryWindow.SetHotKeyColor(LIGHTGREEN);  
  _EntryWindow.UseColor();

  _RecordNumLbl->Deactivate();
  _FileNameLbl->Deactivate();
}

/****************************************************************************/
void Catalog::SetTempFileWindow()
{
  SetFileWindow();
  _EntryWindow[0] = _TempFilePb;
  _EntryWindow.UseColor();
}

/****************************************************************************/
void Catalog::ShowFileStatus()
{
  if (!_UpdateDialogs ||
      _PrintBtn->IsSelected() && _FileStatus == FILE_GOOD)
    return;

  _FileStatusLbl->Hide();

  if (_FileStatus == END_OF_FILE)
  {
    if (_RecordNum == 0 && _FileSizeInRecords == 0)
      _FileStatusLbl->SetMessage("Record File Empty");
    else
      _FileStatusLbl->SetMessage("End Of File");
  }
  else if (_FileStatus == START_OF_FILE)
    _FileStatusLbl->SetMessage("Start Of File");
  else if (_FileStatus == FILE_ERROR)
    _FileStatusLbl->SetMessage("File Access Error");

  if (_FileStatus != FILE_GOOD)
    _FileStatusLbl->Show();
}

/****************************************************************************/
void Catalog::ShowRecordNumber()
{
  if (!_UpdateDialogs)
    return;

  char Buffer_[32];

  strcpy(Buffer_, "Record #");
  LongToStr(_RecordNum, &Buffer_[strlen(Buffer_)]);

  if (_Entry._Erased)
    strcat(Buffer_, " (Erased)");

  _RecordNumLbl->Hide();
  _RecordNumLbl->SetMessage(Buffer_);
  _RecordNumLbl->Show();
}

/****************************************************************************/
void Catalog::UpdateRecordWindow()
{
  if (!_UpdateDialogs)
    return;

  ShowFileStatus();

#if CATALOG_DEBUG
  // testing
  cout <<"Entry erased: " <<_Entry._Erased <<endl;
#endif

  if (_FileStatus != FILE_GOOD || (_Entry._Erased && !_ShowErased))
    return;

#if CATALOG_DEBUG
  // testing
  cout <<"Not returning: " <<endl;
#endif

  short Index_;
  char* TempStr_;
  PromptBox* ChainPb_;

  _EntryWindow.Erase();

  _NamePb->SetInputString(_Entry._Name);
  _PurposePb->SetInputString(_Entry._Purpose);
  _PurposePb->NextChain()->SetInputString(&_Entry._Purpose[strlen(_Entry._Purpose) + 1]);

  _UsesPb->SetInputString(_Entry._Uses);
  ChainPb_ = _UsesPb;
  TempStr_ = _Entry._Uses;
  for (Index_ = 0; Index_ < 3; Index_++)
  {
    ChainPb_ = ChainPb_->NextChain();
    TempStr_ = &TempStr_[strlen(TempStr_) + 1];
    ChainPb_->SetInputString(TempStr_);
  }

  _BenefitsPb->SetInputString(_Entry._Benefits);
  ChainPb_ = _BenefitsPb;
  TempStr_ = _Entry._Benefits;
  for (Index_ = 0; Index_ < 3; Index_++)
  {
    ChainPb_ = ChainPb_->NextChain();
    TempStr_ = &TempStr_[strlen(TempStr_) + 1];
    ChainPb_->SetInputString(TempStr_);
  }

  _RelatedPb->SetInputString(_Entry._Related);
  _LocationPb->PushInputString(_Entry._Location);
  _LocationPb->NextChain()->SetInputString(&_Entry._Location[strlen(_Entry._Location) + 1]);

  ShowRecordNumber();
  _EntryWindow.Show();
}

/****************************************************************************/
void Catalog::WriteToRecordEntry()
{
  short Index_;
  char* TempStr_;
  PromptBox* ChainPb_;

  strcpy(_Entry._Name, _NamePb->Retrieve());
  strcpy(_Entry._Purpose, _PurposePb->Retrieve());
  strcpy(&_Entry._Purpose[strlen(_Entry._Purpose) + 1], _PurposePb->NextChain()->Retrieve());

  strcpy(_Entry._Uses, _UsesPb->Retrieve());
  ChainPb_ = _UsesPb;
  TempStr_ = _Entry._Uses;
  for (Index_ = 0; Index_ < 3; Index_++)
  {
    ChainPb_ = ChainPb_->NextChain();
    TempStr_ = &TempStr_[strlen(TempStr_) + 1];
    strcpy(TempStr_, ChainPb_->Retrieve());
  }

  strcpy(_Entry._Benefits, _BenefitsPb->Retrieve());
  ChainPb_ = _BenefitsPb;
  TempStr_ = _Entry._Benefits;
  for (Index_ = 0; Index_ < 3; Index_++)
  {
    ChainPb_ = ChainPb_->NextChain();
    TempStr_ = &TempStr_[strlen(TempStr_) + 1];
    strcpy(TempStr_, ChainPb_->Retrieve());
  }

  strcpy(_Entry._Related, _RelatedPb->Retrieve());
  strcpy(_Entry._Location, _LocationPb->Retrieve());
  strcpy(&_Entry._Location[strlen(_Entry._Location) + 1], _LocationPb->NextChain()->Retrieve());

  _Entry._Erased = 0;

#if CATALOG_DEBUG
  // testing.
  clrscr();
  // fwrite(&_Entry, sizeof(RecordEntry), 1, stdout);
  cout <<"WriteToRecordEntry: " <<endl
       <<_Entry._Name <<endl
       <<_Entry._Purpose <<endl
       <<_Entry._Uses <<endl
       <<_Entry._Benefits <<endl
       <<_Entry._Related <<endl
       <<_Entry._Location <<endl
       <<(int)_Entry._Erased <<endl;
  getch();
#endif
}

/****************************************************************************/
/**************************** Program Processing ****************************/
/****************************************************************************/
void Catalog::ClearEof(fstream& Fs_)
{
  if (Fs_.eof())
    Fs_.clear();
}

/****************************************************************************/
void Catalog::GetFileSize(fstream& Fs_)
{
  Fs_.seekg(0, ios::end);
  _FileSizeInRecords = Fs_.tellg() / sizeof(RecordEntry);
  Fs_.seekg(0, ios::beg);
}

/****************************************************************************/
Boolean Catalog::OpenPrintFile()
{
  _PrtFile.open(_PrinterFileName, ios::out);

  _PrtDevice.open(_PrinterDeviceName, ios::out);
  *((int*)&_PrtDevStat) = prtstat(_PrinterDeviceNumber);

  _PrtDeviceGood = _PrtDevice.good() && _PrtDevStat.selected &&
		   !_PrtDevStat.IOerror && !_PrtDevStat.paperout;

  return (_PrtFile.good() || _PrtDeviceGood);
}

/****************************************************************************/
Boolean Catalog::OpenBackupFile(char* DiskDrive_)
{
  if (!DiskDrive_)
    return 0;

  char Fname_[128];

  strcpy(Fname_, DiskDrive_);
  strcat(Fname_, _FileName);

  _BackupFile.open(Fname_, ios::binary | ios::out);
  return _BackupFile.good();
}

/****************************************************************************/
Boolean Catalog::OpenCatalogFile(char* Fname_)
{
  if (Fname_)
    _FileLoaded = GetFileHelper(_CatFile, ios::in | ios::out | ios::binary, 0, 0, 0, 0);
  else
  {
    _FilePb->SetMessage("Catalog File: ");
    _FileLoaded = GetFile(_CatFile, ios::in | ios::out | ios::binary, 0, 1);
  }

  return _FileLoaded;
}

/****************************************************************************/
Boolean Catalog::OpenTempFile()
{
  _TempFilePb->SetMessage("Save To Temporary File: ");
  return GetTempFile(_OutFile, ios::out | ios::binary);
}

/****************************************************************************/
Boolean Catalog::OpenOutputFile()
{
  _FilePb->SetMessage("Output File: ");
  return GetFile(_OutFile, ios::out | ios::binary, 1, 1);
}

/****************************************************************************/
void Catalog::ClosePrintFile()
{
  if (_PrtFile.good())
    _PrtFile.flush();
  _PrtFile.close();

  if (_PrtDeviceGood)
    _PrtDevice.flush();
  _PrtDevice.close();
}

/****************************************************************************/
void Catalog::CloseBackupFile()
{
  _BackupFile.flush();
  _BackupFile.close();
}

/****************************************************************************/
void Catalog::CloseOutputFile()
{
  _OutFile.flush();
  _OutFile.close();
}

/****************************************************************************/
void Catalog::CloseCatalogFile()
{
  _CatFile.flush();
  _CatFile.close();
  _FileLoaded = 0;
}

/****************************************************************************/
void Catalog::CloseAllFiles()
{
  CloseCatalogFile();
  CloseOutputFile();
  ClosePrintFile();
  CloseBackupFile();
}

/****************************************************************************/
Boolean Catalog::OpenFile(char* Fname_, fstream& File_, std::_Ios_Openmode BitFlags_)
{
  File_.open(Fname_, BitFlags_);
  ClearEof(File_);
  return File_.good();
}

/****************************************************************************/
Boolean Catalog::FileExists(const char* Name_)
{
   int Handle_;

   if ((Handle_ = open(Name_, O_RDONLY | O_TEXT)) >= 0)
   {
      close(Handle_);
      return 1;
   }

   return 0;
}

/****************************************************************************/
Boolean Catalog::GetFileHelper(fstream& File_, std::_Ios_Openmode BitFlags_,
                               Boolean IsTemp_, Boolean PromptUser_,
                               Boolean SafeOpen_, Boolean FileCreationOk_)
{
  if (IsTemp_)
    SetTempFileWindow();
  else
    SetFileWindow();

  if (PromptUser_)
    ShowWindows();

  Boolean FileSelected_ = 0;
  Boolean CreateNew_ = 0;
  Boolean FileOk_ = 0;

  for (;;)
  {
    if (PromptUser_)
    {
      if (FileCreationOk_)      
        _BtnWindow[2] = _CreateNewBtn;
      else
      {      
        _BtnWindow.Hide();        
        _BtnWindow[2] = NULL;
      }

      _BtnWindow.UseColor();
      _BtnWindow.Show();      
      
      do
      {
	if (!IsTemp_)
	  PromptFileBoxes();
      }
      while (!PromptFileButtons(FileSelected_, CreateNew_));

      if (FileCreationOk_)
      {
        _BtnWindow.Hide();      
        _BtnWindow[2] = NULL;
        _BtnWindow.Show();
      }
    }
    else
    {
      _BtnWindow.Hide();    
      _BtnWindow[2] = NULL;
      _BtnWindow.Show();
      FileSelected_ = 1;
    }

    if (!FileSelected_)
      break;

    #if CATALOG_DEBUG_2
      cout <<"GetFileHelper(), FileSelected_ = " <<FileSelected_ <<endl
           <<"GetFileHelper(), CreateNew_    = " <<CreateNew_ <<endl;
    #endif

    if (FileSelected_)
    {
      Boolean Overwrite_ = 0;
      Boolean Fexists_ = FileExists(_EntryWindow[0]->Retrieve());
      Boolean Safety_ = (SafeOpen_ || CreateNew_) &&
                        Fexists_ && (BitFlags_ & ios::out);
      Boolean AccessOk_ = Fexists_ || (CreateNew_ &&
                          (BitFlags_ & ios::out) && errno == ENOENT);

      #if CATALOG_DEBUG_2
        cout <<"Fexists  = " <<Fexists_ <<endl
             <<"Safety   = " <<Safety_ <<endl
             <<"AccessOk = " <<AccessOk_ <<endl;
      #endif

      if (AccessOk_)
      {
	if (Safety_)
	{
	  _FileSafetyLbl->Show();

	  for (;;)
	  {
	    _BtnWindow.Prompt();
	    if (_CancelBtn->IsSelected())
	      break;

	    if (Overwrite_ = _OkBtn->IsSelected())
	      break;
	  }

	  _FileSafetyLbl->Hide();
	}

	if (!Safety_ || Overwrite_)
	{
          _FileStatusLbl->Hide();
	  _FileStatusLbl->SetMessage("Error Opening File");

          if (Overwrite_ || CreateNew_)
          {
            #if CATALOG_DEBUG_2
              cout <<"Overwriting new file: "
                   <<_EntryWindow[0]->Retrieve() <<endl;
              cout <<"Reopening new file" <<endl;                   
            #endif
            
            RecordEntry DummyRec_;
  	    OpenFile(_EntryWindow[0]->Retrieve(), File_, ios::out | ios::binary);
            memset(&DummyRec_, 0, sizeof(RecordEntry));
            DummyRec_._Erased = 0;
            File_.write((char*)&DummyRec_, sizeof(RecordEntry));            
            File_.close();
            File_.clear();
          }

          #if CATALOG_DEBUG_2
            cout <<"Opening file: "
                 <<_EntryWindow[0]->Retrieve() <<endl;
          #endif          
                    
	  FileOk_ = OpenFile(_EntryWindow[0]->Retrieve(), File_, BitFlags_);
          File_.clear();

          #if CATALOG_DEBUG_2
            cout <<"FileOk_ = " <<FileOk_ <<endl;
          #endif          

	  if (FileOk_)
	    break;
	  else
	    _FileStatusLbl->Show();
	}
      }
      else
      {
        _FileStatusLbl->Hide();
        
        if (!Fexists_ && !CreateNew_)
	  _FileStatusLbl->SetMessage("Error: No such file");
	else if (BitFlags_ & ios::in && !(BitFlags_ & ios::out))
	  _FileStatusLbl->SetMessage("Error: No such file");
	else if (errno == EMFILE)
	  _FileStatusLbl->SetMessage("Error: Too many open files");
	else if (errno == EACCES)
	  _FileStatusLbl->SetMessage("Error: Permission denied");

	_FileStatusLbl->Show();
      } // AccessOk_
    } // FileSelected_
  } // for (;;)

  _FileStatusLbl->Hide();
  HideWindows();
  
  return FileOk_;
}

/****************************************************************************/
Boolean Catalog::PromptFileButtons(Boolean& FileSelected_, Boolean& CreateNew_)
{
  _BtnWindow.Prompt();
  CreateNew_ = _CreateNewBtn->IsSelected();  

  if (_CancelBtn->IsSelected())
  {
    FileSelected_ = 0;
    return 1;
  }

  FileSelected_ = _EntryWindow[0]->IsSelected();

  #if CATALOG_DEBUG_2
    cout <<"FileSelected = " <<FileSelected_ <<endl;
    cout <<"CreatingNew  = " <<CreateNew_ <<endl;
  #endif  
  
  if (_OkBtn->IsSelected() || CreateNew_)
    return FileSelected_;

  return 0;
}

/****************************************************************************/
/****************************************************************************/
Boolean Catalog::GetRecord(long RecRequest_)
{
  _FileStatus =
	(RecRequest_ < 0) 		    	? START_OF_FILE:
	(RecRequest_ >= _FileSizeInRecords)	? END_OF_FILE:
						  FILE_GOOD;

  if (_FileStatus != FILE_GOOD)
    return 0;

#if CATALOG_DEBUG
  // testing.
  clrscr();  
  cout <<"GetRecord Pre: " <<endl
       <<_Entry._Name <<endl
       <<_Entry._Purpose <<endl
       <<_Entry._Uses <<endl
       <<_Entry._Benefits <<endl
       <<_Entry._Related <<endl
       <<_Entry._Location <<endl
       <<(int)_Entry._Erased <<endl;  
  // fwrite(&_Entry, sizeof(RecordEntry), 1, stdout);
  getch();
#endif

  _CatFile.seekg(RecRequest_ * sizeof(RecordEntry), ios::beg);

#if CATALOG_DEBUG
  // testing
  clrscr();
  cout <<"GetRecord() File pos: " <<_CatFile.tellg() <<endl;
  getch();
#endif
  
  _CatFile.read((char*)&_Entry, sizeof(RecordEntry));

#if CATALOG_DEBUG
  // testing.
  clrscr();
  cout <<"GetRecord Post: " <<endl
       <<_Entry._Name <<endl
       <<_Entry._Purpose <<endl
       <<_Entry._Uses <<endl
       <<_Entry._Benefits <<endl
       <<_Entry._Related <<endl
       <<_Entry._Location <<endl
       <<(int)_Entry._Erased <<endl;  
  // fwrite(&_Entry, sizeof(RecordEntry), 1, stdout);
  getch();
#endif

  ClearEof(_CatFile);
  if (!_CatFile.good())
  {
    _FileStatus = FILE_ERROR;
    return 0;
  }

  _RecordNum = RecRequest_;
  return 1;
}

/****************************************************************************/
void Catalog::PutNewRecord(ofstream& Fout_)
{
#if CATALOG_DEBUG
  // testing.
  clrscr();
  cout <<"PutNewRecord Pre: " <<endl
       <<_Entry._Name <<endl
       <<_Entry._Purpose <<endl
       <<_Entry._Uses <<endl
       <<_Entry._Benefits <<endl
       <<_Entry._Related <<endl
       <<_Entry._Location <<endl
       <<(int)_Entry._Erased <<endl;  
  // fwrite(&_Entry, sizeof(RecordEntry), 1, stdout);
  getch();
#endif

  Fout_.seekp(0, ios::end);

#if CATALOG_DEBUG
  // testing
  clrscr();
  cout <<"PutNewRecord(ofstream) File pos: " <<Fout_.tellp() <<endl;
  getch();
#endif
  
  Fout_.write((char*)&_Entry, sizeof(RecordEntry));

#if CATALOG_DEBUG
  // testing.
  clrscr();
  cout <<"PutNewRecord Post: " <<endl
       <<_Entry._Name <<endl
       <<_Entry._Purpose <<endl
       <<_Entry._Uses <<endl
       <<_Entry._Benefits <<endl
       <<_Entry._Related <<endl
       <<_Entry._Location <<endl
       <<(int)_Entry._Erased <<endl;  
  // fwrite(&_Entry, sizeof(RecordEntry), 1, stdout);
  getch();
#endif  
}

/****************************************************************************/
void Catalog::PutNewRecord(fstream& File_)
{
#if CATALOG_DEBUG
  // testing.
  clrscr();
  cout <<"PutNewRecord Pre: " <<endl
       <<_Entry._Name <<endl
       <<_Entry._Purpose <<endl
       <<_Entry._Uses <<endl
       <<_Entry._Benefits <<endl
       <<_Entry._Related <<endl
       <<_Entry._Location <<endl
       <<(int)_Entry._Erased <<endl;  
  // fwrite(&_Entry, sizeof(RecordEntry), 1, stdout);
  getch();
#endif

  File_.seekp(0, ios::end);

#if CATALOG_DEBUG
  // testing
  clrscr();
  cout <<"PutNewRecord(fstream) File pos: " <<File_.tellp() <<endl;
  getch();
#endif

  File_.write((char*)&_Entry, sizeof(RecordEntry));

#if CATALOG_DEBUG
  // testing.
  clrscr();
  cout <<"PutNewRecord Post: " <<endl
       <<_Entry._Name <<endl
       <<_Entry._Purpose <<endl
       <<_Entry._Uses <<endl
       <<_Entry._Benefits <<endl
       <<_Entry._Related <<endl
       <<_Entry._Location <<endl
       <<(int)_Entry._Erased <<endl;  
  // fwrite(&_Entry, sizeof(RecordEntry), 1, stdout);
  getch();
#endif
}

/****************************************************************************/
void Catalog::PutRecord(fstream& File_)
{
  File_.seekp(_RecordNum * sizeof(RecordEntry));
  File_.write((char*)&_Entry, sizeof(RecordEntry));
  ShowRecordNumber();
}

/****************************************************************************/
void Catalog::GetNextRecord()
{
  Boolean ReadOk_;
  long OldPos_ = _RecordNum;

  for (long RecRequest_ = _RecordNum;;)
  {
    ++RecRequest_;
    ReadOk_ = GetRecord(RecRequest_);

    if (!ReadOk_ || !_Entry._Erased || _ViewDeleted)
      break;
  }

  UpdateRecordWindow();

  if (!ReadOk_)
    _RecordNum = OldPos_;
}

/****************************************************************************/
void Catalog::GetPreviousRecord()
{
  Boolean ReadOk_;
  long OldPos_ = _RecordNum;

  for (long RecRequest_ = _RecordNum;;)
  {
    --RecRequest_;
    ReadOk_ = GetRecord(RecRequest_);

    if (!ReadOk_ || !_Entry._Erased || _ViewDeleted)
      break;
  }

  UpdateRecordWindow();

  if (!ReadOk_)
    _RecordNum = OldPos_;
}

/****************************************************************************/
void Catalog::GetLastRecord()
{
  GetFileSize(_CatFile);

  _RecordNum = _FileSizeInRecords - 1;
  if (_RecordNum < 0)
    _RecordNum = 0;

#if CATALOG_DEBUG
  // testing
  clrscr();
  cout <<"Last Record number: " <<_RecordNum <<endl;
  getch();
#endif

  Boolean ReadOk_ = GetRecord(_RecordNum);

#if CATALOG_DEBUG
  // testing
  clrscr();
  cout <<"Read Ok: " <<ReadOk_ <<endl;
  cout <<"ViewDeleted: " <<_ViewDeleted <<endl;
  cout <<"Entry Erased: " <<(int)_Entry._Erased <<endl;
  getch();
#endif

  if (ReadOk_ && _Entry._Erased && !_ViewDeleted)
    GetPreviousRecord();
  else
    UpdateRecordWindow();
}

/****************************************************************************/
void Catalog::GetFirstRecord()
{
  _RecordNum = 0;
  Boolean ReadOk_ = GetRecord(_RecordNum);

  if (ReadOk_ && _Entry._Erased && !_ViewDeleted)
    GetNextRecord();
  else
    UpdateRecordWindow();
}

/****************************************************************************/
void Catalog::GetThisRecord(long RecRequest_)
{
  GetRecord(RecRequest_);
  UpdateRecordWindow();
}

/****************************************************************************/
Boolean Catalog::DeleteRecord()
{
  if (_FileSizeInRecords && !_Entry._Erased)
  {
    _Entry._Erased = 1;
    PutRecord(_CatFile);
    return 1;
  }

  return 0;
}

/****************************************************************************/
void Catalog::ModifyRecord()
{
  _EntryWindow.Prompt();
  WriteToRecordEntry();
  PutRecord(_CatFile);

  GetFileSize(_CatFile);
  ShowFileStatus();
}

/****************************************************************************/
void Catalog::AddRecord()
{
  _RecordNumLbl->Hide();

  _FileStatusLbl->Hide();
  _FileStatusLbl->SetMessage("New Record");
  _FileStatusLbl->Show();

  _EntryWindow.Erase();
  _EntryWindow.Show();
  _EntryWindow.Prompt();

  WriteToRecordEntry();
  PutNewRecord(_CatFile);

#if CATALOG_DEBUG
  // testing
  cout <<"Getting last record" <<endl;
  getch();
  clrscr();
#endif

  GetLastRecord();
}

/****************************************************************************/
void Catalog::GetUses(char* Uses_)
{
  char* TempStr_ = _Entry._Uses;

  strcpy(Uses_, TempStr_);
  for (short Index_ = 0; Index_ < 3; Index_++)
  {
    TempStr_ += (strlen(TempStr_) + 1);
    strcat(Uses_, TempStr_);
  }
}

/****************************************************************************/
short Catalog::SearchForUses(int StartIndex_)
{
  KeyWordToken SrchToken_(_SearchPb->Retrieve(), StartIndex_, ",", " ");
  KeyWordToken UsesToken_(NULL, 0, ",", " ");

  char KeyWord_[100];
  char UseWord_[100];
  char RecnumBuf_[16];
  char Uses_[USES_FIELDSIZE];

  short UsePos_;
  short Length_;
  short MatchCount_ = 0;
  Boolean Match_;

  GetFirstRecord();

  while (_FileStatus != END_OF_FILE)
  {
    GetUses(Uses_);
    UsesToken_.SetString(Uses_);

    Match_ = 0;
    while (!Match_ && UsesToken_.FindToken(UseWord_))
    {
      while (!Match_ && SrchToken_.FindToken(KeyWord_))
	for (UsePos_ = 0, Length_ = strlen(KeyWord_); strlen(&UseWord_[UsePos_]) >= Length_; UsePos_++)
	  if (Match_ = !strncmpi(KeyWord_, &UseWord_[UsePos_], Length_))
	    break;

      SrchToken_.ResetIndex();
    }

    if (Match_)
    {
      ++MatchCount_;
      _MatchingRecNum = _RecordNum;

      strcat(
	strcat(
	  strcpy(_Entry._Related, _FileName), ": Record #"), LongToStr(_RecordNum, RecnumBuf_));

      PutNewRecord(_OutFile);
    }

    GetNextRecord();
    UsesToken_.ResetIndex();
  }

  if (MatchCount_)
    _Entry._Related[0] = 0;

  return MatchCount_;
}

/****************************************************************************/
short Catalog::SearchForName(int StartIndex_)
{
  KeyWordToken SrchToken_(_SearchPb->Retrieve(), StartIndex_, ",", " ");

  char KeyWord_[100];
  char RecnumBuf_[16];
  char Name_[NAME_FIELDSIZE];

  short NamePos_;
  short Length_;
  short MatchCount_ = 0;

  Boolean Match_;
  Boolean ExactMatch_;

  while (SrchToken_.FindToken(KeyWord_))
  {
    ExactMatch_ = 0;
    GetFirstRecord();

    while (!ExactMatch_ && _FileStatus != END_OF_FILE)
    {
      strcpy(Name_, _Entry._Name);
      RemoveChar(Name_, " ");
      ExactMatch_ = !strcmpi(KeyWord_, Name_);

      if (ExactMatch_)
      {
	++MatchCount_;
	_MatchingRecNum = _RecordNum;

	strcat(
	  strcat(
	    strcpy(_Entry._Related, _FileName), ": Record #"), LongToStr(_RecordNum, RecnumBuf_));

	PutNewRecord(_OutFile);
      }
      else
	GetNextRecord();
    }

    if (!ExactMatch_)
    {
      WeightedMatch* MatchWeight_ = NULL;
      size_t Weight_ = 0;

      GetFirstRecord();

      while (_FileStatus != END_OF_FILE)
      {
	strcpy(Name_, _Entry._Name);
	RemoveChar(Name_, " ");

	size_t Short_ = 0;
	Weight_ = 0;
	Match_ = 0;
	Length_ = strlen(KeyWord_);

	size_t Index_;
	for (Index_ = Length_; Index_; --Index_)
	  if (strlen(Name_) >= Index_)
	  {
	    if (Match_ = !strncmpi(KeyWord_, Name_, Index_))
	      break;
	  }
	  else
	    ++Short_;

	if (Match_)
	{
	  Weight_ = (Length_ - Index_) * Length_ + Short_;

	  if (!Weight_)
	    break;

	  MatchWeight_ = MatchWeight_ ?
		new WeightedMatch(Weight_, _RecordNum, MatchWeight_):
		new WeightedMatch(Weight_, _RecordNum);
	}

	GetNextRecord();
      }

      WeightedMatch* Lightest_ = MatchWeight_;
      Boolean ClosestFound_ = !Weight_ && _FileStatus != END_OF_FILE;

      if (!ClosestFound_)
	for (WeightedMatch* Current_ = MatchWeight_; Current_; Current_ = Current_->_Next)
	  if (Current_->_Weight < Lightest_->_Weight)
	    Lightest_ = Current_;

      if (Lightest_ || ClosestFound_)
      {
	if (!ClosestFound_)
	  GetRecord(Lightest_->_RecNum);

	delete MatchWeight_;
	++MatchCount_;
	_MatchingRecNum = _RecordNum;

	strcat(
	  strcat(
	    strcpy(_Entry._Related, _FileName), ": Record #"), LongToStr(_RecordNum, RecnumBuf_));

	PutNewRecord(_OutFile);
      }
    }
  }

  if (MatchCount_)
    _Entry._Related[0] = 0;

  return MatchCount_;
}

/****************************************************************************/
void Catalog::SearchForRecord()
{
  _UpdateDialogs = 0;
  _SearchPb->Activate();
  _SearchPb->Prompt(1);

  short MatchCount_ = 0;
  int Extkey_;
  int Key_;

  RemovePadding(_SearchPb->Retrieve(), " ");
  Key_ = _SearchPb->LastKeyRead(&Extkey_);

  if (!_SearchPb->IsSelected() ||
      Extkey_ == 0 && Key_ == KeyboardCodes::CODETABLE(KeyboardCodes::ESC).ASCII)
  {
    _SearchPb->Deselect();
    _SearchPb->Hide();
    _SearchPb->Deactivate();
    _UpdateDialogs = 1;
    return;
  }

  Boolean TempFileUsed_;
  char Field_[16];
  long LastRecPos_ = _RecordNum;

  KeyWordToken FieldToken_(_SearchPb->Retrieve(), 0, NULL, " ");
  FieldToken_.FindToken(Field_);

  if ((TempFileUsed_ = OpenTempFile()) || OpenOutputFile())
  {
    if (!strcmpi(Field_, "Name:"))
      MatchCount_ = SearchForName(FieldToken_.GetIndex());
    else if (!strcmpi(Field_, "Uses:"))
      MatchCount_ = SearchForUses(FieldToken_.GetIndex());
    else
      MatchCount_ = SearchForUses(0);

    CloseOutputFile();

    if (MatchCount_ == 0)
    {
      _FileStatusLbl->Hide();
      _FileStatusLbl->SetMessage("No Matching Records Found");
      _FileStatusLbl->Show();
      ShowWindows();
      _BtnWindow.Prompt();

      if (TempFileUsed_)
	UnsetTempFileWindow();

      TempFileUsed_ = 0;
    }
    else if (MatchCount_ == 1)
    {
      LastRecPos_ = _MatchingRecNum;
      if (TempFileUsed_)
	UnsetTempFileWindow();
      TempFileUsed_ = 0;
    }
    else
    {
      if (!TempFileUsed_)
      {
	_FileStatusLbl->Hide();
	_FileStatusLbl->SetMessage("Records Written To File");
	_FileStatusLbl->Show();
	ShowWindows();
	_BtnWindow.Prompt();
      }
      else
      {
	_FilePb->SetInputString(_TempFilePb->Retrieve());
	_LoadBtn->Select();
      }
    }
  }

  _SearchPb->Hide();
  _SearchPb->Erase();
  _SearchPb->Deactivate();
  _UpdateDialogs = 1;

  if (!TempFileUsed_)
  {
    _EntryWindow.Erase();
    SetCatalogWindow();
    ShowWindows();
    GetThisRecord(LastRecPos_);
  }
}

/****************************************************************************/
void Catalog::SearchForRecordNumber()
{
  int Extkey_;
  int Key_;
  long RecRequest_;
  Boolean ValidRequest_;

  _RecordNumPb->Activate();
  _RecordNumPb->Prompt(1);

  RemovePadding(_RecordNumPb->Retrieve(), " ");
  Key_ = _RecordNumPb->LastKeyRead(&Extkey_);

  ValidRequest_ = _RecordNumPb->IsSelected() &&
		  !(Extkey_ == 0 && Key_ == KeyboardCodes::CODETABLE(KeyboardCodes::ESC).ASCII) &&
		  IsLong(_RecordNumPb->Retrieve());
  if (ValidRequest_)
  {
    RecRequest_ = atol(_RecordNumPb->Retrieve());
    ValidRequest_ = RecRequest_ >= 0 && RecRequest_ < _FileSizeInRecords;
  }

  _RecordNumPb->Hide();
  _RecordNumPb->Erase();
  _RecordNumPb->Deactivate();

  if (ValidRequest_)
  {
    _ShowErased = 1;
    GetThisRecord(RecRequest_);
    _ShowErased = _ViewDeleted;
  }
}

/****************************************************************************/
void Catalog::LoadFile(char* Fname_)
{
  char Buffer_[80];

  if (_FileLoaded)
    CloseCatalogFile();

  if (Fname_)
    OpenCatalogFile(Fname_);
  else
    OpenCatalogFile();

  strcpy(_FileName, _FilePb->Retrieve());

  if (_FileLoaded)
  {
    GetFileSize(_CatFile);

    strcpy(Buffer_, "Current File: ");
    strcat(Buffer_, _FilePb->Retrieve());
  }
  else
    strcpy(Buffer_, "No File Loaded");

  // Setting file name label
  _FileNameLbl->SetMessage(Buffer_);

  // Erasing previous contents of windows
  _EntryWindow.Erase();
  SetCatalogWindow();
  _EntryWindow.Erase();
  ShowWindows();
}

/****************************************************************************/
char* Catalog::PadWithSpace(int Max_, char* Rstr_)
{
  static char PadLine_[80];

  Max_ -= strlen(Rstr_);
  for (int i = 0; i < Max_; i++)
    PadLine_[i] = ' ';
  PadLine_[Max_] = 0;

  return PadLine_;
}

/****************************************************************************/
std::ostrstream* Catalog::PrintStrStream(std::ostrstream* Ostr_, ofstream& Fout_)
{
  *Ostr_ <<ends;
  char* TempLine_ = Ostr_->str();
  Fout_ <<TempLine_;
  delete TempLine_;

  delete Ostr_;
  return new std::ostrstream;
}

/****************************************************************************/
void Catalog::PrintRecords(ofstream& Fout_, char* FormFeed_)
{
  short Index_;
  short Dash_ = 0;
  char* NextLine_;
  std::ostrstream* Ostr_ = new std::ostrstream;

  GetFirstRecord();

  for (short Count_ = 0; _FileStatus != END_OF_FILE; Count_++)
  {
    for (;;)
    {
      for (Dash_ = 0; Dash_ < 80; Dash_++)
	*Ostr_ <<"-";
      *Ostr_ <<endl;

      if (FormFeed_ && Count_ && !(Count_ % 3))
      {
	*Ostr_ <<FormFeed_;
	Count_ -= 3;
      }
      else
	break;
    }

    *Ostr_ <<_FileNameLbl->GetMessage() <<endl;
    *Ostr_ <<"Record #" <<_RecordNum <<endl
	  <<endl;

    *Ostr_ <<"Name: [" <<_Entry._Name <<PadWithSpace(64, _Entry._Name) <<"]" <<endl;
    Ostr_ = PrintStrStream(Ostr_, Fout_);

    *Ostr_ <<"Purpose: [" <<_Entry._Purpose <<PadWithSpace(64, _Entry._Purpose) <<"]" <<endl;
    NextLine_ = GetNextLine(_Entry._Purpose);
    *Ostr_ <<"         [" <<NextLine_ <<PadWithSpace(64, NextLine_) <<"]" <<endl;
    Ostr_ = PrintStrStream(Ostr_, Fout_);

    *Ostr_ <<"Uses: [" <<_Entry._Uses <<PadWithSpace(64, _Entry._Uses) <<"]" <<endl;
    Ostr_ = PrintStrStream(Ostr_, Fout_);
    NextLine_ = _Entry._Uses;

    for (Index_ = 0; Index_ < 3; Index_++)
    {
      NextLine_ = GetNextLine(NextLine_);
      *Ostr_ <<"      [" <<NextLine_ <<PadWithSpace(64, NextLine_) <<"]" <<endl;
    }
    Ostr_ = PrintStrStream(Ostr_, Fout_);

    *Ostr_ <<"Benefits: [" <<_Entry._Benefits <<PadWithSpace(64, _Entry._Benefits) <<"]" <<endl;
    Ostr_ = PrintStrStream(Ostr_, Fout_);
    NextLine_ = _Entry._Benefits;

    for (Index_ = 0; Index_ < 3; Index_++)
    {
      NextLine_ = GetNextLine(NextLine_);
      *Ostr_ <<"          [" <<NextLine_ <<PadWithSpace(64, NextLine_) <<"]" <<endl;
    }
    Ostr_ = PrintStrStream(Ostr_, Fout_);

    *Ostr_ <<"Related: [" <<_Entry._Related <<PadWithSpace(64, _Entry._Related) <<"]" <<endl;
    *Ostr_ <<"Location: [" <<_Entry._Location <<PadWithSpace(64, _Entry._Location) <<"]" <<endl;
    NextLine_ = GetNextLine(_Entry._Location);
    *Ostr_ <<"          [" <<NextLine_ <<PadWithSpace(64, NextLine_) <<"]" <<endl;
    Ostr_ = PrintStrStream(Ostr_, Fout_);

    GetNextRecord();
  }

  if (Dash_)
  {
    for (Dash_ = 0; Dash_ < 80; Dash_++)
      *Ostr_ <<"-";
    *Ostr_ <<endl;
  }

  if (FormFeed_)
    *Ostr_ <<FormFeed_;

  Ostr_ = PrintStrStream(Ostr_, Fout_);
  delete Ostr_;
}

/****************************************************************************/
void Catalog::PrintFile()
{
  long LastRecPos_ = _RecordNum;
  _UpdateDialogs = 0;

  if (OpenPrintFile())
  {
    if (_PrtFile.good())
      PrintRecords(_PrtFile, NULL);
    if (_PrtDeviceGood)
      PrintRecords(_PrtDevice, "\x0B\x0C");
    ClosePrintFile();

    _FileStatusLbl->Hide();

    if (_PrtDeviceGood)
      _FileStatusLbl->SetMessage("File Records Printed");
    else if (!_PrtDevStat.selected)
      _FileStatusLbl->SetMessage("Printer Off Line");
    else if (_PrtDevStat.paperout)
      _FileStatusLbl->SetMessage("Printer Paper Out");
    else if (_PrtDevStat.IOerror)
      _FileStatusLbl->SetMessage("Printer I/O Error");
    else if (!_PrtDevStat.notbusy)
      _FileStatusLbl->SetMessage("Printer Busy");
    else if (_PrtDevStat.timeout)
      _FileStatusLbl->SetMessage("Printer Timeout");

    _FileStatusLbl->Show();
  }
  else
  {
    _FileStatusLbl->Hide();
    _FileStatusLbl->SetMessage("Error In Printing File");
    _FileStatusLbl->Show();
  }

  _UpdateDialogs = 1;
  GetThisRecord(LastRecPos_);
}

/****************************************************************************/
void Catalog::SaveRecords(ofstream& Fout_)
{
  for (long RecIndex_ = 0; RecIndex_ < _FileSizeInRecords; RecIndex_++)
    if (GetRecord(RecIndex_))
      PutNewRecord(Fout_);
}

/****************************************************************************/
void Catalog::SaveAndQuit()
{
  _BackupStatus = NO_BACKUP;
  if (!_FileLoaded)
    return;

  HideWindows();
  _FileNameLbl->Show();
  _RadWindow.Show();
  _RadWindow.Prompt();
  _RadWindow.Hide();
  _FileNameLbl->Hide();

  if (_NoBackupBtn->IsSelected())
    return;

  char* DiskDrive_ = NULL;
  if (_BackupToADriveBtn->IsSelected())
    DiskDrive_ = "A:\\";
  else if (_BackupToBDriveBtn->IsSelected())
    DiskDrive_ = "B:\\";

  if (OpenBackupFile(DiskDrive_))
  {
    _BackupStatus = FILE_GOOD;
    _UpdateDialogs = 0;

    SaveRecords(_BackupFile);
    CloseBackupFile();
  }
  else
    _BackupStatus = FILE_ERROR;
}

/****************************************************************************/
void Catalog::PromptCatalogFile()
{
  for (;;)
  {
    _BtnWindow.Prompt();

    if (_LoadBtn->IsSelected())
    {
      LoadFile();
      _LoadBtn->Deselect();

      if (_FileLoaded)
	break;
    }
    else if (_SaveQuitBtn->IsSelected())
      break;
  }
}

/****************************************************************************/
Boolean Catalog::PromptCatalogButtons()
{
  int Start_ = 0;
  PushButton* SelectedBtn_;

  if (!_FileLoaded)
  {
    PromptCatalogFile();
    if (_SaveQuitBtn->IsSelected())
    {
      SaveAndQuit();
      return 0;
    }
  }

  GetFirstRecord();

  for (;;)
  {
    _BtnWindow.Prompt(Start_);

    Start_ = 0;
    SelectedBtn_ = _BtnWindow.NextSelected(Start_);
    --Start_;

    if (_NextBtn->IsSelected())
      GetNextRecord();
    else if (_PreviousBtn->IsSelected())
      GetPreviousRecord();
    else if (_DeleteBtn->IsSelected())
    {
      if (!DeleteRecord())
	GetNextRecord();
    }
    else if (_ModifyBtn->IsSelected())
      ModifyRecord();
    else if (_AddBtn->IsSelected())
      AddRecord();
    else if (_FirstBtn->IsSelected())
      GetFirstRecord();
    else if (_LastBtn->IsSelected())
      GetLastRecord();
    else if (_SearchBtn->IsSelected())
    {
      SearchForRecord();
      if (_LoadBtn->IsSelected())
      {
	LoadFile(_TempFileName);
	_LoadBtn->Deselect();
	return 1;
      }
    }
    else if (_RecordNumBtn->IsSelected())
      SearchForRecordNumber();
    else if (_LoadBtn->IsSelected())
    {
      LoadFile();
      _LoadBtn->Deselect();
      return 1;
    }
    else if (_PrintBtn->IsSelected())
      PrintFile();
    else if (_SaveQuitBtn->IsSelected())
    {
      SaveAndQuit();
      break;
    }

    if (SelectedBtn_)
      SelectedBtn_->Deselect();
  }

  return 0;
}

/****************************************************************************/
const char* Catalog::GetPostRunMessage()
{
  return
  (
    (_BackupStatus == FILE_ERROR) 	? "File Backup Error":
    (_BackupStatus == FILE_GOOD) 	? "File Backup Successful":
					  NULL
  );
}

/****************************************************************************/
void Catalog::Run()
{
  _TitleLbl->Show();
  LoadFile();

  while (PromptCatalogButtons());
}

/****************************************************************************/
/****************************************************************************/
Catalog::~Catalog()
{
  CloseAllFiles();

  SetCatalogWindow();
  DeleteWindows();

  SetFileWindow();
  DeleteWindows();
  delete _TempFilePb;

  _LblWindow.Hide();
  _LblWindow.Delete();

  _RadWindow.Hide();
  _RadWindow.Delete();
}

/****************************************************************************/
Catalog::Catalog():
// Program flags
_FileLoaded(0),
_UpdateDialogs(1),
_ShowErased(0),
_ViewDeleted(0),

// Prompt Boxes for record fields
_NamePb(NULL),
_PurposePb(NULL),
_UsesPb(NULL),
_BenefitsPb(NULL),
_RelatedPb(NULL),
_LocationPb(NULL),
_SearchPb(NULL),

// Prompt Boxes for file operations
_FilePb(NULL),

// Push buttons uses in catalog program
_NextBtn(NULL),
_PreviousBtn(NULL),
_DeleteBtn(NULL),
_ModifyBtn(NULL),
_SearchBtn(NULL),
_SaveQuitBtn(NULL),

// Push buttons for file operations
_OkBtn(NULL),
_CancelBtn(NULL),
_CreateNewBtn(NULL),

// Program Title Label
_TitleLbl(NULL),

// File Record information
_RecordNum(0),

// Printer device number
_PrinterDeviceNumber(0)
{
  Initialize();
}

/****************************************************************************/
int main()
{
  textcolor(YELLOW);
  textbackground(BLUE);
  clrscr();

  Catalog* Catp_ = new Catalog;
  Catp_->Run();
  const char* Message_ = Catp_->GetPostRunMessage();
  delete Catp_;

  textcolor(LIGHTGRAY);
  textbackground(BLACK);
  clrscr();

  if (Message_)
    cout <<endl <<endl
	 <<Message_;

  return 0;
}

/****************************************************************************/
#endif
