// Programming Catalog
//
#ifndef CATALOG_H
#define CATALOG_H
#include <io.h>
#include <errno.h>
#include <fcntl.h>
#include <backward/strstream>

#ifndef STRUTIL_H
  #include "strutil.h"
#endif
#ifndef PRTSTAT_H
  #include "prtstat.h"
#endif
#ifndef TXTINTER_H
  #include "txtinter.h"
#endif

// Default settings
#define CATALOG_DEFAULT_PRINTFILE	"PrntFile.Prt"
#define CATALOG_DEFAULT_PRINTDEVICE	"LPT1"
#define CATALOG_DEFAULT_TEMPFILE	"TEMPF.$$$"

// debugging switch
#define CATALOG_DEBUG                   0
#define CATALOG_DEBUG_2                 0

class KeyWordToken
{
  private:
    char* _SearchStr;
    char* _Delims;
    char* _WhiteSpace;

    short _Start;
    short _Index;

    char* _Space;

  public:
    KeyWordToken();
    KeyWordToken(char* Str_, short Start_, char* Delims_, char* Wspc_);
    ~KeyWordToken();

    inline void SetStartIndex(short Index_)
	{ _Start = Index_; }
    inline void ResetIndex()
	{ _Index = _Start; }
    inline short GetIndex()
	{ return _Index; }
    inline void SetDelimiters(char* Str_)
	{ _Delims = Str_; }
    inline void SetWhiteSpace(char* Str_)
	{ _WhiteSpace = Str_; }
    inline void SetString(char* Str_)
	{ _SearchStr = Str_; }

    short FindToken(char* Token_);
};

class Catalog
{
  public:
    enum
    {
      NO_BACKUP			= 0,
      END_OF_FILE		= 1,
      START_OF_FILE		= 2,
      FILE_ERROR		= 3,
      FILE_GOOD			= 4,
      KEEP_MESSAGE		= 5,

      NAME_FIELDSIZE 		= 65,
      PURPOSE_FIELDSIZE 	= 130,
      USES_FIELDSIZE 		= 260,
      BENEFITS_FIELDSIZE	= 260,
      RELATED_FIELDSIZE 	= 65,
      LOCATION_FIELDSIZE	= 130,
      SEARCH_FIELDSIZE 		= 100,
    };

  private:
    // Print file and device name constants
    char _PrinterFileName[16];
    char _PrinterDeviceName[8];
    int _PrinterDeviceNumber;

    // Temporary file name constant
    char _TempFileName[16];

    // View deleted records flag
    Boolean _ViewDeleted;

    // Data record structure
    struct RecordEntry
    {
      char _Name[NAME_FIELDSIZE];
      char _Purpose[PURPOSE_FIELDSIZE];
      char _Uses[USES_FIELDSIZE];
      char _Benefits[BENEFITS_FIELDSIZE];
      char _Related[RELATED_FIELDSIZE];
      char _Location[LOCATION_FIELDSIZE];
      char _Erased;
    };

    struct WeightedMatch
    {
      WeightedMatch(size_t Match_, long RecNum_);
      WeightedMatch(size_t Match_, long RecNum_, WeightedMatch* Link_);

      ~WeightedMatch();

      size_t _Weight;
      long _RecNum;
      WeightedMatch* _Next;
    };

    char _FileName[128];

    // Catalog file and output file streams
    fstream _CatFile;
    fstream _OutFile;

    // Backup file stream and printer file and device streams
    ofstream _BackupFile;
    ofstream _PrtFile;
    ofstream _PrtDevice;

    // Printer device status structure
    PrStatus _PrtDevStat;

    // Program status flags and switches
    int _FileStatus;
    int _BackupStatus;
    Boolean _FileLoaded;
    Boolean _UpdateDialogs;
    Boolean _ShowErased;
    Boolean _PrtDeviceGood;

    // File position as record number
    long _RecordNum;
    long _MatchingRecNum;
    long _FileSizeInRecords;

    // Catalog record entry
    RecordEntry _Entry;

    // Prompt Boxes for record fields
    PromptBox* _NamePb;
    PromptBox* _PurposePb;
    PromptBox* _UsesPb;
    PromptBox* _BenefitsPb;
    PromptBox* _RelatedPb;
    PromptBox* _LocationPb;
    PromptBox* _SearchPb;
    PromptBox* _RecordNumPb;

    // Prompt Boxes for file operations
    PromptBox* _FilePb;
    PromptBox* _TempFilePb;

    // Prompt Window containing prompt boxes
    PromptWindow _EntryWindow;

    // Push buttons uses in catalog program
    PushButton* _NextBtn;
    PushButton* _PreviousBtn;
    PushButton* _DeleteBtn;
    PushButton* _ModifyBtn;
    PushButton* _AddBtn;
    PushButton* _SearchBtn;
    PushButton* _RecordNumBtn;
    PushButton* _SaveQuitBtn;
    PushButton* _FirstBtn;
    PushButton* _LastBtn;
    PushButton* _PrintBtn;
    PushButton* _LoadBtn;

    // Push buttons for file operations
    PushButton* _OkBtn;
    PushButton* _CancelBtn;
    PushButton* _CreateNewBtn;

    // Button Window containing the push buttons
    PushWindow _BtnWindow;

    // Radio buttons for quit and backup operations
    RadioButton* _BackupToADriveBtn;
    RadioButton* _BackupToBDriveBtn;
    RadioButton* _NoBackupBtn;

    // Radio button Window containing the radio buttons
    RadioWindow _RadWindow;

    // Program Title Label
    TextLabel* _TitleLbl;
    TextLabel* _FileStatusLbl;
    TextLabel* _RecordNumLbl;
    TextLabel* _FileNameLbl;
    TextLabel* _FileSafetyLbl;

    // Label Window containing all the text labels for the program
    LabelWindow _LblWindow;

    // Program initialization method
    void Initialize();

    // Configuration file parsing procedures
    void GetConfiguration();
    void ReadConfigFile(ifstream& Fin_, char* PrintFile_, char* PrintDev_, char* TempFile_, Boolean& ViewDeleted_);
    char* RdStr(istream& In_, char* Dest_, int Size_, char Delimiter_='\n');
    char* TrimAndDnCase(char* str);

    // Program Text Labels Setup methods
    void TextLabelSetup();
    Boolean ShouldHideLabels();

    // File Screen Setup method
    void FileScreenSetup();

    // Program Radio Button Setup methods
    void RadioBtnSetup();

    // Main Program Screen Setup method
    void CatalogScreenPromptBoxesSetup();
    void CatalogScreenPushButtonsSetup();

    // Text dialog windows methods
    void RemoveWindows();
    void InitWindows();
    void DeleteWindows();
    void ShowWindows();
    void HideWindows();

    // Text dialogs setup methods
    void InitDialogs();
    void SetCatalogButtonWindow();
    void SetCatalogPromptWindow();
    void SetCatalogWindow();
    void SetFileWindow();
    void SetTempFileWindow();
    void UnsetTempFileWindow()
	{ _EntryWindow[0] = NULL; }
    void SetLabelWindow();
    void SetRadioWindow();

    // File operations methods
    void ClearEof(fstream& Fs_);
    void GetFileSize(fstream& Fs_);
    Boolean OpenFile(char* Fname_, fstream& File_, std::_Ios_Openmode BitFlags_);
    Boolean FileExists(const char* Name_);
    Boolean GetFileHelper(fstream& File_, std::_Ios_Openmode BitFlags_,
			  Boolean IsTemp_, Boolean PromptUser_,
                          Boolean SafeOpen_, Boolean FileCreationOk_);
    inline Boolean GetTempFile(fstream& File_, std::_Ios_Openmode BitFlags_)
	{ return GetFileHelper(File_, BitFlags_, TRUE, TRUE, FALSE, FALSE); }
    inline Boolean GetFile(fstream& File_, std::_Ios_Openmode BitFlags_,
                           Boolean SafeOpen_, Boolean FileCreationOk_)
	{ return GetFileHelper(File_, BitFlags_, FALSE, TRUE,
                               SafeOpen_, FileCreationOk_); }
    inline void PromptFileBoxes()
	{ _EntryWindow.Prompt(); }
    Boolean PromptFileButtons(Boolean& FileSelected_, Boolean& CreateNew_);

    // File open/close methods
    Boolean OpenPrintFile();
    Boolean OpenBackupFile(char* DiskDrive_);
    Boolean OpenCatalogFile(char* Fname_=NULL);
    Boolean OpenOutputFile();
    Boolean OpenTempFile();
    void ClosePrintFile();
    void CloseBackupFile();
    void CloseCatalogFile();
    void CloseOutputFile();
    void CloseAllFiles();

    // Catalog dialog methods
    void ShowFileStatus();
    void ShowRecordNumber();
    void WriteToRecordEntry();
    void UpdateRecordWindow();
    void PromptCatalogFile();
    Boolean PromptCatalogButtons();

    // Binary data file record access methods
    void PutRecord(fstream& File_);
    void PutNewRecord(fstream& File_);
    void PutNewRecord(ofstream& Fout_);
    Boolean GetRecord(long RecRequest_);
    void GetThisRecord(long RecRequest_);

    // Keyword search helper methods
    void GetUses(char* Uses_);
    short SearchForUses(int StartIndex_);
    short SearchForName(int StartIndex_);
    inline Boolean IsDelimiter(char Ch_)
	{ return (isspace(Ch_) || Ch_ == ','); }

    // File Printing helper methods
    inline char* GetNextLine(char* Str_)
	{ return &Str_[strlen(Str_) + 1]; }
    std::ostrstream* PrintStrStream(std::ostrstream* Ostr_, ofstream& Fout_);
    char* PadWithSpace(int Max_, char* Rstr_);
    void PrintRecords(ofstream& Fout_, char* FormFeed_);

    // File Backup helper methods
    void SaveRecords(ofstream& Fout_);

    // Action methods for each button in catalog screen
    void GetNextRecord();
    void GetPreviousRecord();
    void GetFirstRecord();
    void GetLastRecord();
    Boolean DeleteRecord();
    void ModifyRecord();
    void AddRecord();
    void SearchForRecord();
    void SearchForRecordNumber();
    void LoadFile(char* Fname_=NULL);
    void PrintFile();
    void SaveAndQuit();

  public:
    Catalog();
    ~Catalog();

    // Main execution method
    void Run();
    const char* GetPostRunMessage();
};

#endif
