//=============================================================
//  Numerics Library
//=============================================================
//  Matrix.h
//
//  Templates for matrices
//
//=============================================================

#ifndef MATRIX_H
#define MATRIX_H

#ifndef INCL_STDDEF_H
  #include <stddef.h>
  #define INCL_STDDEF_H
#endif
#ifndef INCL_LIMITS_H
  #include <limits.h>
  #define INCL_LIMITS_H
#endif
#ifndef INCL_MATH_H
  #include <math.h>
  #define INCL_MATH_H
#endif
#ifndef INCL_STRING_H
  #include <string.h>
  #define INCL_STRING_H
#endif

#ifndef FALLIBLE_H
  #include "fallible.h"
#endif
#ifndef OBJECTIMP_H
  #include "objimp.h"
#endif
#ifndef OBJECTACCEPTOR_H
  #include "objaccpt.h"
#endif
#ifndef BUILTIN_H
  #include "builtin.h"
#endif
#ifndef FPUTIL_H
  #include "fputil.h"
#endif

//------------------------------------
// Grid: A two-dimensional array class
//------------------------------------
template <class T>
class Grid : public ObjectSkeleton
{
  ROOTOBJECTFNCS_DEFN(Object, Grid<T>)

  friend Grid<T> Apply(const Grid<T>& m, T (*func)(const T& n))
  {
    Grid<T> result(m);
    for (size_t i = 0; i < m.N; ++i)
      result.Data[i] = func(result.Data[i]);
    return result;
  }

  protected:
    T * Data;
    size_t R;
    size_t C;
    size_t N;

    size_t Index(size_t row, size_t col) const;

    void Xalloc() const
        { THROW (FallibleBase::MatrixAllocationErr()); }
    void Xincompat() const
        { THROW (FallibleBase::IncompatibleMatrixErr()); }
    void Xindex() const
        { THROW (FallibleBase::InvalidIndexErr()); }
    void Xsingular() const
        { THROW (FallibleBase::SingularMatrixErr()); }
    void Xtoobig() const
        { THROW (FallibleBase::MatrixTooBigErr()); }
    void Xzerodim() const
        { THROW (FallibleBase::ZeroDimensionErr()); }
    void Xzerodiv() const
        { THROW (FallibleBase::ZeroDivisionErr()); }

  public:
    // constructors
    Grid();
    Grid(size_t rows, size_t cols);
    Grid(const Grid<T>& m);

    // destructors
    virtual ~Grid();

    // Virtual constructor and assignment methods
    static Grid<T>* Make();
    static Grid<T>* Make(size_t rows, size_t cols);
    virtual Object* Clone(ObjectAcceptor* ObjComp_=NULL) const;

    // Static null object accessor methods
    static Grid<T>& GiveNullObject();

    // Null object accessors
    virtual const Object& NULLOBJECT() const;
    virtual Object& NULLOBJECT();

    // Class Identification methods
    static int StaticID()
	{ return TypeInfo::GRID_TYPE; }
    virtual int ClassID() const;

    // assignment operator
    Grid<T>& operator = (const Grid<T> & m);

    Grid<T>& operator = (const T * array);

    // interrogation
    size_t GetRows() const;
    size_t GetCols() const;

    bool IsVector() const;
    bool IsRowVector() const;
    bool IsColVector() const;
    bool IsSquare() const;

    // retrieve elements
    T Get(size_t row, size_t col) const;
    T & operator () (size_t row, size_t col);

    // internal function w/o exception check!
    T & Elem(size_t row, size_t col) const;

    // apply a function to each element
    void Apply(T (* func)(const T & n));

    // fill matrix with specific value
    void Fill(const T & x);

    // create a row matrix
    Grid<T> VectorRow(size_t row);

    // create a column vector
    Grid<T> VectorCol(size_t col);

    // transpose a matrix
    Grid<T> Transpose();

    // change size (destroying contents)
    void Resize(size_t rows, size_t cols);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif    
};

//=============================================================
template <class T>
inline size_t Grid<T>::Index(size_t row, size_t col) const
{
  return (row * C + col);
}

//=============================================================
template <class T>
Grid<T>::Grid()
{
  R = 1;
  C = 1;
  N = 1;

  Data = New<T>(N);

  if (Data == NULL)
    Xalloc();
}

//=============================================================
template <class T>
Grid<T>::Grid(size_t rows, size_t cols)
{
  R = rows;
  C = cols;
  N = rows * cols;

  if ((R == 0) || (C == 0))
    Xzerodim();

  if ((N < R) || (N < C))
    Xtoobig();

  Data = New<T>(N);

  if (Data == NULL)
    Xalloc();
}

//=============================================================
template <class T>
Grid<T>::Grid(const Grid<T> & m)
{
  R = m.R;
  C = m.C;
  N = m.N;

  Data = New<T>(N);

  if (Data == NULL)
    Xalloc();

  for (size_t i = 0; i < N; ++i)
    Data[i] = m.Data[i];
}

//=============================================================
// destructors
template <class T>
Grid<T>::~Grid()
{
  ::DeleteArray(Data);
}

//=============================================================
template <class T>
Grid<T>* Grid<T>::Make()
{
  return (new Grid<T>());
}

//=============================================================
template <class T>
Grid<T>* Grid<T>::Make(size_t rows, size_t cols)
{
  return (new Grid<T>(rows, cols));
}

//=============================================================
template <class T>
Object* Grid<T>::Clone(ObjectAcceptor* ObjComp_) const
{
  if (ObjComp_ == NULL)
    return (new Grid<T>(*this));

  return NULL;
}

//=============================================================
template <class T>
Grid<T>& Grid<T>::GiveNullObject()
{
  // The NULLOBJECT method must be overwritten in the derived classes to
  // actually return an instance of the respective class with the null byte
  // set to null otherwise mismatched class objects will be returned.
  static Grid<T> _NULLOBJECT;
  _NULLOBJECT.SetAsNullObject();
  return _NULLOBJECT;
}

//=============================================================
template <class T>
const Object& Grid<T>::NULLOBJECT() const
{
  // The NULLOBJECT method must be overwritten in the derived classes to
  // actually return an instance of the respective class with the null byte
  // set to null otherwise mismatched class objects will be returned.
  return Grid<T>::GiveNullObject();
}

//=============================================================
template <class T>
Object& Grid<T>::NULLOBJECT()
{
  // The NULLOBJECT method must be overwritten in the derived classes to
  // actually return an instance of the respective class with the null byte
  // set to null otherwise mismatched class objects will be returned.
  return Grid<T>::GiveNullObject();
}

//=============================================================
template <class T>
int Grid<T>::ClassID() const
{
  return TypeInfo::GRID_TYPE;
}

//=============================================================
// assignment operator
template <class T>
Grid<T>& Grid<T>::operator = (const Grid<T> & m)
{
  if (this != &m)
  {
    R = m.R;
    C = m.C;
    N = m.N;

    ::DeleteArray(Data);

    Data = New<T>(N);

    if (Data == NULL)
      Xalloc();

    for (size_t i = 0; i < N; ++i)
      Data[i] = m.Data[i];
  }

  return *this;
}

//=============================================================
template <class T>
Grid<T>& Grid<T>::operator = (const T * array)
{
  // note: must assume that array has N members
  const T * aptr = array;
  T * mptr = Data;

  for (size_t i = 0; i < N; ++i)
  {
    (*mptr) = (*aptr);
    ++aptr;
    ++mptr;
  }

  return *this;
}

//=============================================================
// interrogation
template <class T>
inline size_t Grid<T>::GetRows() const
{
  return R;
}

//=============================================================
template <class T>
inline size_t Grid<T>::GetCols() const
{
  return C;
}

//=============================================================
template <class T>
inline bool Grid<T>::IsVector() const
{
  return ((R == 1) || (C == 1));
}

//=============================================================
template <class T>
inline bool Grid<T>::IsRowVector() const
{
  return (C == 1);
}

//=============================================================
template <class T>
inline bool Grid<T>::IsColVector() const
{
  return (R == 1);
}

//=============================================================
template <class T>
inline bool Grid<T>::IsSquare() const
{
  return (R == C);
}

//=============================================================
template <class T>
T Grid<T>::Get(size_t row, size_t col) const
{
  if ((row >= R) || (col >= C))
    Xindex();

  return Data[Grid<T>::Index(row,col)];
}

//=============================================================
template <class T>
T & Grid<T>::operator () (size_t row, size_t col)
{
  if ((row >= R) || (col >= C))
    Xindex();
    
  return Data[Grid<T>::Index(row,col)];
}

//=============================================================
template <class T>
inline T & Grid<T>::Elem(size_t row, size_t col) const
{
  return Data[Grid<T>::Index(row,col)];
}

//=============================================================
// apply a function to each element
template <class T>
void Grid<T>::Apply(T (* func)(const T & n))
{
  for (size_t i = 0; i < N; ++i)
    Data[i] = func(Data[i]);
}

//=============================================================
// fill matrix with specific value
template <class T>
void Grid<T>::Fill(const T & x)
{
  for (size_t i = 0; i < N; ++i)
    Data[i] = x;
}

//=============================================================
// create a row matrix
template <class T>
Grid<T> Grid<T>::VectorRow(size_t row)
{
  if (row >= R)
    Xindex();

  Grid<T> vector(C,1);

  for (size_t i = 0; i < C; ++i)
    vector(i,0) = Data[Grid<T>::Index(row,i)];

  return vector;
}

//=============================================================
// create a column vector
template <class T>
Grid<T> Grid<T>::VectorCol(size_t col)
{
  if (col >= C)
    Xindex();

  Grid<T> vector(1,R);

  for (size_t i = 0; i < R; ++i)
    vector(0,i) = Data[Grid<T>::Index(i,col)];

  return vector;
}

//=============================================================
// transpose a matrix
template <class T>
Grid<T> Grid<T>::Transpose()
{
  Grid<T> result(C,R);

  const T * tptr = Data;
  T * rptr;

  for (size_t i = 0; i < R; ++i)
  {
    rptr = result.Data + i;

    for (size_t j = 0; j < C; ++j)
    {
      (*rptr) = (*tptr);
      ++tptr;
      rptr += R;
    }
  }

  return result;
}

//=============================================================
template <class T>
void Grid<T>::Resize(size_t rows, size_t cols)
{
  ::DeleteArray(Data);

  R = rows;
  C = cols;
  N = rows * cols;

  if ((R == 0) || (C == 0))
    Xzerodim();

  if ((N < R) || (N < C))
    Xtoobig();

  Data = New<T>(N);

  if (Data == NULL)
    Xalloc();
}

//=============================================================
#if OVERLOAD_NEW
template <class T>
void* Grid<T>::operator new (size_t Bytes_)
{
  return MemMatrix::Matrix().Allocate(Bytes_);
}

//=============================================================
template <class T>
void Grid<T>::operator delete (void* Space_)
{
  MemMatrix::Matrix().Deallocate(Space_);
}

//=============================================================
#if HAS_ARRAY_NEW
template <class T>
void* Grid<T>::operator new[] (size_t Bytes_)
{
  return MemMatrix::Matrix().Allocate(Bytes_);
}

//=============================================================
template <class T>
void Grid<T>::operator delete[] (void* Space_)
{
  MemMatrix::Matrix().Deallocate(Space_);
}
#endif
#endif

//-----------------------------------------------------
//  Matrix: A two-dimensional mathematical matrix class
//-----------------------------------------------------
template <class T>
class Matrix : public Grid<T>
{
  protected:
    // internal recursive function for determinant
    T DetRecursive();

  public:
    // constructors
    Matrix();
    Matrix(size_t rows, size_t cols, const T & init = T(0));
    Matrix(const Matrix<T>& m);
    Matrix(const Grid<T>& m);

    // Virtual constructor and assignment methods
    static Matrix<T>* Make();
    static Matrix<T>* Make(size_t rows, size_t cols, const T& init = T(0));
    virtual Object* Clone(ObjectAcceptor* ObjComp_=NULL) const;

    // Static null object accessor methods
    static Matrix<T>& GiveNullObject();

    // Null object accessors
    virtual const Object& NULLOBJECT() const;
    virtual Object& NULLOBJECT();

    // Class Identification methods
    static int StaticID()
	{ return TypeInfo::MATRIX_TYPE; }
    virtual int ClassID() const;

    // assignment operator
    Matrix<T>& operator = (const Matrix<T> & m);
    Matrix<T>& operator = (const Grid<T> & m);
    Matrix<T>& operator = (const T * array);

    // interrogation
    bool IsZero() const;
    bool IsDiagonal() const;
    bool IsIdentity() const;
    bool IsTridiagonal() const;
    bool IsUpperTriangular() const;
    bool IsLowerTriangular() const;
    bool IsPermutation() const;
    bool IsSingular() const;

    // scalar addition
    Matrix<T> operator + (const Matrix<T> & m);
    void operator += (const Matrix<T> & m);
    Matrix<T> operator + (const T & x);
    void operator += (const T & x);

    // scalar subtraction
    Matrix<T> operator - (const Matrix<T> & m);
    void operator -= (const Matrix<T> & m);
    Matrix<T> operator - (const T & x);
    void operator -= (const T & x);

    // scalar multiplication
    Matrix<T> operator * (const Matrix<T> & m);
    void operator *= (const Matrix<T> & m);
    Matrix<T> operator * (const T & x);
    void operator *= (const T & x);

    // scalar division
    Matrix<T> operator / (const Matrix<T> & m);
    void operator /= (const Matrix<T> & m);
    Matrix<T> operator / (const T & x);
    void operator /= (const T & x);

    // matrix multiplication
    Matrix<T> operator % (const Matrix<T> & m);

    // comparison operators
    bool Equals(const Matrix<T> & m);
    Matrix<bool> operator == (const Matrix<T> & m);
    Matrix<bool> operator != (const Matrix<T> & m);
    Matrix<bool> operator < (const Matrix<T> & m);
    Matrix<bool> operator <= (const Matrix<T> & m);
    Matrix<bool> operator > (const Matrix<T> & m);
    Matrix<bool> operator >= (const Matrix<T> & m);

    // negate a matrix
    Matrix<T> operator - ();

    // change size (destroying contents)
    void Resize(size_t rows, size_t cols, const T & init = T(0));

    // inner and outer products
    T InnerProduct(const Matrix<T> & m);

    // calculation euclidean norm
    double Norm();

    // calculate determinant value
    T Determinant();

    // create a minor matrix
    Matrix<T> Minor(size_t rdel, size_t cdel);

    // solve system of linear equations
    Matrix<T> LinSolve();

    // LUP decomposition
    Matrix<size_t> LUPDecompose();

    // LUP decomposition (call w/ result of LUPDecomp)
    Matrix<T> LUPSolve(const Matrix<size_t> & perm, const Matrix<T> & b);

    // LUP inversion (call w/ result of LUPDecomp)
    Matrix<T> LUPInvert(const Matrix<size_t> & perm);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif    
};

//=============================================================
// constructors
template <class T>
inline Matrix<T>::Matrix():
Grid<T>()
{
  *Grid<T>::Data = T(0);
}

//=============================================================
template <class T>
Matrix<T>::Matrix(size_t rows, size_t cols, const T & init):
Grid<T>(rows,cols)
{
  for (size_t i = 0; i < Grid<T>::N; ++i)
    Grid<T>::Data[i] = init;
}

//=============================================================
template <class T>
inline Matrix<T>::Matrix(const Matrix<T> & m):
Grid<T>(m)
{
  // place holder
}

//=============================================================
template <class T>
inline Matrix<T>::Matrix(const Grid<T> & m):
Grid<T>(m)
{
  // place holder
}

//=============================================================
template <class T>
Matrix<T>* Matrix<T>::Make()
{
  return (new Matrix<T>());
}

//=============================================================
template <class T>
Matrix<T>* Matrix<T>::Make(size_t rows, size_t cols, const T& init)
{
  return (new Matrix<T>(rows, cols, init));
}

//=============================================================
template <class T>    
Object* Matrix<T>::Clone(ObjectAcceptor* ObjComp_) const
{
  if (ObjComp_ == NULL)
    return (new Matrix<T>(*this));

  return NULL;
}

//=============================================================
template <class T>
Matrix<T>& Matrix<T>::GiveNullObject()
{
  // The NULLOBJECT method must be overwritten in the derived classes to
  // actually return an instance of the respective class with the null byte
  // set to null otherwise mismatched class objects will be returned.
  static Matrix<T> _NULLOBJECT;
  _NULLOBJECT.SetAsNullObject();
  return _NULLOBJECT;
}

//=============================================================
template <class T>
const Object& Matrix<T>::NULLOBJECT() const
{
  // The NULLOBJECT method must be overwritten in the derived classes to
  // actually return an instance of the respective class with the null byte
  // set to null otherwise mismatched class objects will be returned.
  return Matrix<T>::GiveNullObject();
}

//=============================================================
template <class T>    
Object& Matrix<T>::NULLOBJECT()
{
  // The NULLOBJECT method must be overwritten in the derived classes to
  // actually return an instance of the respective class with the null byte
  // set to null otherwise mismatched class objects will be returned.
  return Matrix<T>::GiveNullObject();
}

//=============================================================
template <class T>    
int Matrix<T>::ClassID() const
{
  return TypeInfo::MATRIX_TYPE;
}

//=============================================================
// assignment operator
template <class T>
inline Matrix<T>& Matrix<T>::operator = (const Matrix<T> & m)
{
  if (this != &m)
    Grid<T>::operator = (m);
  return *this;
}

//=============================================================
template <class T>
inline Matrix<T>& Matrix<T>::operator = (const Grid<T> & m)
{
  if (this != &m)
    Grid<T>::operator = (m);
  return *this;
}

//=============================================================
template <class T>
inline Matrix<T>& Matrix<T>::operator = (const T * array)
{
  Grid<T>::operator = (array);
  return *this;
}

//=============================================================
// interrogation
template <class T>
bool Matrix<T>::IsZero() const
{
  const T * ptr = Grid<T>::Data;
  
  for (size_t i = 0; i < Grid<T>::N; ++i)
  {
    if ((*ptr) != (T)0)
      return false;

    ++ptr;
  }

  return true;
}

//=============================================================
template <class T>
bool Matrix<T>::IsDiagonal() const
{
  if (Grid<T>::C != Grid<T>::R)
    return false;

  const T * ptr = Grid<T>::Data;

  for (size_t ir = 0; ir < Grid<T>::R; ++ir)
  {
    for (size_t ic = 0; ic < Grid<T>::C; ++ic)
    {
      if (ir == ic)
      {
        if ((*ptr) == (T)0)
          return false;
      }
      else
      {
        if ((*ptr) != (T)0)
          return false;
      }

      ++ptr;
    }
  }

  return true;
}

//=============================================================
template <class T>
bool Matrix<T>::IsIdentity() const
{
  if (Grid<T>::C != Grid<T>::R)
    return false;

  const T * ptr = Grid<T>::Data;

  for (size_t ir = 0; ir < Grid<T>::R; ++ir)
  {
    for (size_t ic = 0; ic < Grid<T>::C; ++ic)
    {
      if (ir == ic)
      {
        if ((*ptr) != (T)1)
          return false;
      }
      else
      {
        if ((*ptr) != (T)0)
          return false;
      }
      
      ++ptr;
    }
  }

  return true;
}

//=============================================================
template <class T>
bool Matrix<T>::IsTridiagonal() const
{
  if ((Grid<T>::C != Grid<T>::R) || (Grid<T>::C < 3))
    return false;

  const T * ptr = Grid<T>::Data;

  for (size_t ir = 0; ir < Grid<T>::R; ++ir)
  {
    for (size_t ic = 0; ic < Grid<T>::C; ++ic)
    {
      if (ir != ic)
      {
        if (ir > ic)
        {
          if ((ir - ic) > 1)
          {
            if ((*ptr) != (T)0)
              return false;
          }
        }
        else
        {
          if ((ic - ir) > 1)
          {
            if ((*ptr) != (T)0)
            return false;
          }
        }
      }

      ++ptr;
    }
  }

  return true;
}

//=============================================================
template <class T>
bool Matrix<T>::IsUpperTriangular() const
{
  if ((Grid<T>::C != Grid<T>::R) || (Grid<T>::C < 2))
    return false;

  size_t steps = 1;
  const T * ptr = Grid<T>::Data + Grid<T>::C;
  
  for (size_t ir = 1; ir < Grid<T>::R; ++ir)
  {
    for (size_t s = 0; s < steps; ++s)
    {
      if (ptr[s] != (T)0)
        return false;
    }
    
    ++steps;
    ptr += Grid<T>::C;
  }

  return true;
}

//=============================================================
template <class T>
bool Matrix<T>::IsLowerTriangular() const
{
  if ((Grid<T>::C != Grid<T>::R) || (Grid<T>::C < 2))
    return false;

  size_t steps = Grid<T>::C - 1;
  const T * ptr = Grid<T>::Data;

  for (size_t ir = 1; ir < Grid<T>::R; ++ir)
  {
    for (size_t s = steps; s > 0; --s)
    {
      if (ptr[s] != (T)0)
        return false;
    }
    
    --steps;
    ptr += Grid<T>::C + 1;
  }

  return true;
}

//=============================================================
template <class T>
bool Matrix<T>::IsPermutation() const
{
  if (Grid<T>::C != Grid<T>::R)
    return false;

  char * ctags = new_char_array(Grid<T>::C, NULL);

  if (ctags == NULL)
    Grid<T>::Xalloc();

  char * rtags = new_char_array(Grid<T>::R, NULL);

  if (rtags == NULL)
    Grid<T>::Xalloc();

  memset(ctags,0,Grid<T>::C);
  memset(rtags,0,Grid<T>::R);

  bool result = true;
  const T * ptr = Grid<T>::Data;

  for (size_t ri = 0; ri < Grid<T>::R; ++ri)
  {
    for (size_t ci = 0; ci < Grid<T>::C; ++ci)
    {
      if ((*ptr))
      {
        if (((*ptr) > 1) ||
            (rtags[ri] == 1) ||
            (ctags[ci] == 1))
        {
          result = false;
          goto finished;
        }

        rtags[ri] = 1;
        ctags[ci] = 1;
      }

      ++ptr;
    }
  }

  // a goto label!
  finished:

  ::DeleteArray(ctags);
  ::DeleteArray(rtags);
  return result;
}

//=============================================================
template <class T>
bool Matrix<T>::IsSingular() const
{
  if (Grid<T>::C != Grid<T>::R)
    return false;

  Matrix<T> csum(1,Grid<T>::C), rsum(Grid<T>::R,1);
  const T * ptr = Grid<T>::Data;

  for (size_t ri = 0; ri < Grid<T>::R; ++ri)
  {
    for (size_t ci = 0; ci < Grid<T>::C; ++ci)
    {
      csum(0,ci) += *ptr;
      rsum(ri,0) += *ptr;
      ++ptr;
    }
  }

  for (size_t i = 0; i < Grid<T>::R; ++i)
  {
    if ((csum(0,i) == 0) || (rsum(i,0) == 0))
      return true;
  }

  return false;
}

//=============================================================
// addition
template <class T>
Matrix<T> Matrix<T>::operator + (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  Matrix<T> result(m);

  for (size_t i = 0; i < Grid<T>::N; ++i)
    result.Data[i] += Grid<T>::Data[i];

  return result;
}

//=============================================================
template <class T>
void Matrix<T>::operator += (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  for (size_t i = 0; i < Grid<T>::N; ++i)
    Grid<T>::Data[i] += m.Data[i];
}

//=============================================================
template <class T>
Matrix<T> Matrix<T>::operator + (const T & x)
{
  Matrix<T> result(*this);

  for (size_t i = 0; i < Grid<T>::N; ++i)
    result.Data[i] += x;

  return result;
}

//=============================================================
template <class T>
void Matrix<T>::operator += (const T & x)
{
  for (size_t i = 0; i < Grid<T>::N; ++i)
    Grid<T>::Data[i] += x;
}

//=============================================================
// subtraction
template <class T>
Matrix<T> Matrix<T>::operator - (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  Matrix<T> result(*this);

  for (size_t i = 0; i < Grid<T>::N; ++i)
    result.Data[i] -= m.Data[i];

  return result;
}

//=============================================================
template <class T>
void Matrix<T>::operator -= (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  for (size_t i = 0; i < Grid<T>::N; ++i)
    Grid<T>::Data[i] -= m.Data[i];
}

//=============================================================
template <class T>
Matrix<T> Matrix<T>::operator - (const T & x)
{
  Matrix<T> result(*this);

  for (size_t i = 0; i < Grid<T>::N; ++i)
    result.Data[i] -= x;

  return result;
}

//=============================================================
template <class T>
void Matrix<T>::operator -= (const T & x)
{
  for (size_t i = 0; i < Grid<T>::N; ++i)
    Grid<T>::Data[i] -= x;
}

//=============================================================
// scalar product
template <class T>
Matrix<T> Matrix<T>::operator * (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  Matrix<T> result(m);

  for (size_t i = 0; i <Grid<T>:: N; ++i)
    result.Data[i] *= Grid<T>::Data[i];

  return result;
}

//=============================================================
template <class T>
void Matrix<T>::operator *= (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  for (size_t i = 0; i < Grid<T>::N; ++i)
    Grid<T>::Data[i] *= m.Data[i];
}

//=============================================================
template <class T>
Matrix<T> Matrix<T>::operator * (const T & x)
{
  Matrix<T> result(*this);

  for (size_t i = 0; i < Grid<T>::N; ++i)
    result.Data[i] *= x;

  return result;
}

//=============================================================
template <class T>
void Matrix<T>::operator *= (const T & x)
{
  for (size_t i = 0; i < Grid<T>::N; ++i)
    Grid<T>::Data[i] *= x;
}

//=============================================================
// division
template <class T>
Matrix<T> Matrix<T>::operator / (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  Matrix<T> result(*this);

  for (size_t i = 0; i < Grid<T>::N; ++i)
  {
    if (m.Data[i] == (T)0)
      Grid<T>::Xzerodiv();
      
    result.Data[i] /= m.Data[i];
  }

  return result;
}

//=============================================================
template <class T>
void Matrix<T>::operator /= (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  for (size_t i = 0; i < Grid<T>::N; ++i)
  {
    if (m.Data[i] == (T)0)
      Grid<T>::Xzerodiv();

    Grid<T>::Data[i] /= m.Data[i];
  }
}

//=============================================================
template <class T>
Matrix<T> Matrix<T>::operator / (const T & x)
{
  if (x == (T)0)
    Grid<T>::Xzerodiv();

  Matrix<T> result(*this);

  for (size_t i = 0; i < Grid<T>::N; ++i)
    result.Data[i] /= x;

  return result;
}

//=============================================================
template <class T>
void Matrix<T>::operator /= (const T & x)
{
  if (x == (T)0)
    Grid<T>::Xzerodiv();

  for (size_t i = 0; i < Grid<T>::N; ++i)
    Grid<T>::Data[i] /= x;
}

//=============================================================
// matrix multiplication
template <class T>
Matrix<T> Matrix<T>::operator % (const Matrix<T> & m)
{
  if (Grid<T>::C != m.R)
    Grid<T>::Xincompat();

  Matrix<T> result(Grid<T>::R,m.C);

  T * rptr = result.Data;
  const T * tptr, * mptr;

  for (size_t i = 0; i < Grid<T>::R; ++i)
  {
    for (size_t j = 0; j < m.C; ++j)
    {
      tptr =   Grid<T>::Data + (i * Grid<T>::R);
      mptr = m.Data +  j;

      for (size_t k = 0; k < Grid<T>::C; ++k)
      {
        (*rptr) += (*tptr) * (*mptr);
        ++tptr;
        mptr += m.C;
      }

      ++rptr;
    }
  }

  return result;
}

//=============================================================
// comparison operators
template <class T>
bool Matrix<T>::Equals(const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  for (size_t ri = 0; ri < Grid<T>::R; ++ri)
  {
    for (size_t ci = 0; ci < Grid<T>::C; ++ci)
    {
      if (Grid<T>::Data[Grid<T>::Index(ri,ci)] != m.Data[Grid<T>::Index(ri,ci)])
        return false;
    }
  }

  return true;
}

//=============================================================
template <class T>
Matrix<bool> Matrix<T>::operator == (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  Matrix<bool> result(Grid<T>::R,Grid<T>::C);

  for (size_t ri = 0; ri < Grid<T>::R; ++ri)
  {
    for (size_t ci = 0; ci < Grid<T>::C; ++ci)
    {
      result(ri,ci) = (Grid<T>::Data[Grid<T>::Index(ri,ci)] == m.Data[Grid<T>::Index(ri,ci)]);
    }
  }

  return result;
}

//=============================================================
template <class T>
Matrix<bool> Matrix<T>::operator != (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  Matrix<bool> result(Grid<T>::R,Grid<T>::C);

  for (size_t ri = 0; ri < Grid<T>::R; ++ri)
  {
    for (size_t ci = 0; ci < Grid<T>::C; ++ci)
    {
      result(ri,ci) = (Grid<T>::Data[Grid<T>::Index(ri,ci)] != m.Data[Grid<T>::Index(ri,ci)]);
    }
  }

  return result;
}

//=============================================================
template <class T>
Matrix<bool> Matrix<T>::operator < (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  Matrix<bool> result(Grid<T>::R,Grid<T>::C);

  for (size_t ri = 0; ri < Grid<T>::R; ++ri)
  {
    for (size_t ci = 0; ci < Grid<T>::C; ++ci)
    {
      result(ri,ci) = (Grid<T>::Data[Grid<T>::Index(ri,ci)] < m.Data[Grid<T>::Index(ri,ci)]);
    }
  }

  return result;
}

//=============================================================
template <class T>
Matrix<bool> Matrix<T>::operator <= (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  Matrix<bool> result(Grid<T>::R,Grid<T>::C);

  for (size_t ri = 0; ri < Grid<T>::R; ++ri)
  {
    for (size_t ci = 0; ci < Grid<T>::C; ++ci)
    {
      result(ri,ci) = (Grid<T>::Data[Grid<T>::Index(ri,ci)] <= m.Data[Grid<T>::Index(ri,ci)]);
    }
  }

  return result;
}

//=============================================================
template <class T>
Matrix<bool> Matrix<T>::operator > (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  Matrix<bool> result(Grid<T>::R,Grid<T>::C);

  for (size_t ri = 0; ri < Grid<T>::R; ++ri)
  {
    for (size_t ci = 0; ci < Grid<T>::C; ++ci)
    {
      result(ri,ci) = (Grid<T>::Data[Grid<T>::Index(ri,ci)] > m.Data[Grid<T>::Index(ri,ci)]);
    }
  }

  return result;
}

//=============================================================
template <class T>
Matrix<bool> Matrix<T>::operator >= (const Matrix<T> & m)
{
  if ((Grid<T>::R != m.R) || (Grid<T>::C != m.C))
    Grid<T>::Xincompat();

  Matrix<bool> result(Grid<T>::R,Grid<T>::C);

  for (size_t ri = 0; ri < Grid<T>::R; ++ri)
  {
    for (size_t ci = 0; ci < Grid<T>::C; ++ci)
    {
      result(ri,ci) = (Grid<T>::Data[Grid<T>::Index(ri,ci)] >= m.Data[Grid<T>::Index(ri,ci)]);
    }
  }

  return result;
}

//=============================================================
// negate a matrix
template <class T>
Matrix<T> Matrix<T>::operator - ()
{
  Matrix<T> result(Grid<T>::R,Grid<T>::C);

  for (size_t ri = 0; ri < Grid<T>::R; ++ri)
  {
    for (size_t ci = 0; ci < Grid<T>::C; ++ci)
      result(ri,ci) = -(Grid<T>::Get(ri,ci));
  }

  return result;
}

//=============================================================
// change size (destroying contents)
template <class T>
void Matrix<T>::Resize(size_t rows, size_t cols, const T & init)
{
  this->Grid<T>::Resize(rows,cols);

  for (size_t i = 0; i < Grid<T>::N; ++i)
    Grid<T>::Data[i] = init;
}

//=============================================================
// inner products
template <class T>
T Matrix<T>::InnerProduct(const Matrix<T> & m)
{
  T result = (T)0;

  if (((Grid<T>::R == 1) && (m.C == 1) && (Grid<T>::C == m.R)) ||
      ((Grid<T>::C == 1) && (m.R == 1) && (Grid<T>::R == m.C)))
  {
    const T * ptr1 = Grid<T>::Data;
    const T * ptr2 = m.Data;

    size_t end = (Grid<T>::C == 1 ? Grid<T>::R : Grid<T>::C);

    for (size_t n = 0; n < end; ++n)
    {
      result += (*ptr1) * (*ptr2);
      ++ptr1;
      ++ptr2;
    }
  }
  else
    Grid<T>::Xincompat();

  return result;
}

//=============================================================
// euclidean norm
template <class T>
inline double Matrix<T>::Norm()
{
  return sqrt(double(InnerProduct(Grid<T>::Transpose())));
}

//=============================================================
// calculate determinant value
template <class T>
T Matrix<T>::Determinant()
{
  if (Grid<T>::R != Grid<T>::C)
    Grid<T>::Xincompat();

  if (Grid<T>::C == 1)
    return (*Grid<T>::Data);

  if (IsSingular())
    return (T)0;

  return DetRecursive();
}

//=============================================================
template <class T>
T Matrix<T>::DetRecursive()
{
  if (Grid<T>::C == 2)
  {
    return ((*Grid<T>::Data) * (*(Grid<T>::Data + 3))) - ((*(Grid<T>::Data + 1)) * (*(Grid<T>::Data + 2)));
  }

  T result      = (T)0;
  const T * ptr = Grid<T>::Data;

  for (size_t x = 0; x < Grid<T>::C; ++x)
  {
    if (x & 1) // if on an even column
      result -= (*ptr) * (Minor(0,x)).DetRecursive();
    else
      result += (*ptr) * (Minor(0,x)).DetRecursive();

    ++ptr;
  }

  return result;
}

//=============================================================
// create a minor matrix
template <class T>
Matrix<T> Matrix<T>::Minor(size_t rdel, size_t cdel)
{
  if ((Grid<T>::R != Grid<T>::C) || (Grid<T>::R < 2))
    Grid<T>::Xincompat();

  Matrix<T> result(Grid<T>::R-1,Grid<T>::C-1);

  const T * psrc = Grid<T>::Data;
  T * pdest = result.Data;

  for (size_t rsrc = 0; rsrc < Grid<T>::R; ++rsrc)
  {
    if (rsrc != rdel)
    {
      for (size_t csrc = 0; csrc < Grid<T>::C; ++csrc)
      {
        if (csrc != cdel)
        {
          *pdest = *psrc;
          ++pdest;
        }
        
        ++psrc;
      }
    }
    else
      psrc += Grid<T>::C;
  }

  return result;
}

//=============================================================
// solve a system of linear equations via Gaussian elimination
template <class T>
Matrix<T> Matrix<T>::LinSolve()
{
  if (((Grid<T>::C - Grid<T>::R) != 1) || (IsSingular()))
    Grid<T>::Xincompat();

  size_t i, j, k, max;
  T temp;

  // forward elimination
  for (i = 0; i < Grid<T>::R; ++i)
  {
    max = i;

    for (j = i + 1; j < Grid<T>::R; ++j)
    {
      if (AbsVal(Grid<T>::Elem(j,i)) > AbsVal(Grid<T>::Elem(max,i)))
        max = j;
    }

    for (k = i; k < Grid<T>::C; ++k)
    {
      temp = Grid<T>::Elem(i,k);
      Grid<T>::Elem(i,k) = Grid<T>::Elem(max,k);
      Grid<T>::Elem(max,k) = temp;
    }

    for (j = i + 1; j < Grid<T>::R; ++j)
    {
      for (k = Grid<T>::R; k >= i; --k)
      {
        Grid<T>::Elem(j,k) -= Grid<T>::Elem(i,k) * Grid<T>::Elem(j,i) / Grid<T>::Elem(i,i);

        if (k == 0)
          break;
      }
    }
  }

  // backward substitution
  Matrix<T> X(Grid<T>::R,1); // results

  for (j = Grid<T>::R - 1; ; --j)
  {
    temp = (T)0;

    for (k = j + 1; k < Grid<T>::R; ++k)
      temp += Grid<T>::Elem(j,k) * X.Elem(k,0);

    X.Elem(j,0) = (Grid<T>::Elem(j,Grid<T>::R) - temp) / Grid<T>::Elem(j,j);

    if (j == 0)
      break;
  }

  return X;
}

//=============================================================
// LUP decomposition
template <class T>
Matrix<size_t> Matrix<T>::LUPDecompose()
{
  // make sure its square
  if ((Grid<T>::R != Grid<T>::C) || (Grid<T>::R < 2))
    Grid<T>::Xincompat();

  // LU decomposition
  size_t i, j, k, k2, t;
  T p, temp;
  Matrix<size_t> perm(Grid<T>::R,1); // permutation matrix

  // initialize permutation
  for (i = 0; i < Grid<T>::R; ++i)
    perm(i,0) = i;

  for (k = 0; k < (Grid<T>::R - 1); ++k)
  {
    p = T(0);

    for (i = k; i < Grid<T>::R; ++i)
    {
      temp = AbsVal(Grid<T>::Elem(i,k));

      if (temp > p)
      {
        p  = temp;
        k2 = i;
      }
    }

    if (p == T(0))
      Grid<T>::Xsingular();

    // exchange rows
    t = perm(k,0);
    perm(k,0)  = perm(k2,0);
    perm(k2,0) = t;

    for (i = 0; i < Grid<T>::R; ++i)
    {
      temp = Grid<T>::Elem(k,i);
      Grid<T>::Elem(k,i) = Grid<T>::Elem(k2,i);
      Grid<T>::Elem(k2,i) = temp;
    }

    for (i = k + 1; i < Grid<T>::R; ++i)
    {
      Grid<T>::Elem(i,k) /= Grid<T>::Elem(k,k);

      for (j = k + 1; j < Grid<T>::R; ++j)
        Grid<T>::Elem(i,j) -= Grid<T>::Elem(i,k) * Grid<T>::Elem(k,j);
    }
  }

  // return values
  return perm;
}

//=============================================================
// LUP decomposition (call w/ result of LUPDecomp)
template <class T>
Matrix<T> Matrix<T>::LUPSolve(const Matrix<size_t> & perm, const Matrix<T> & b)
{
  if ((Grid<T>::R != b.R) || (Grid<T>::R != perm.GetRows()))
    Grid<T>::Xincompat();

  size_t i, j, j2;
  T sum, u;
  Matrix<T> y(Grid<T>::R,1), x(Grid<T>::R,1);

  for (i = 0; i < Grid<T>::R; ++i)
  {
    sum = T(0);
    j2  = 0;

    for (j = 1; j <= i; ++j)
    {
      sum += Grid<T>::Elem(i,j2) * y.Elem(j2,0);
      ++j2;
    }

    y.Elem(i,0) = b.Elem(perm.Elem(i,0),0) - sum;
  }

  i = Grid<T>::R - 1;

  while (1)
  {
    sum = T(0);
    u   = Grid<T>::Elem(i,i);

    for (j = i + 1; j < Grid<T>::R; ++j)
      sum += Grid<T>::Elem(i,j) * x.Elem(j,0);

    x.Elem(i,0) = (y.Elem(i,0) - sum) / u;

    if (i == 0)
      break;

    --i;
  }

  return x;
}

//=============================================================
// LUP inversion (call for result of LUPDecomp)
template <class T>
Matrix<T> Matrix<T>::LUPInvert(const Matrix<size_t> & perm)
{
  size_t i, j;
  Matrix<T> p(Grid<T>::R,1);
  Matrix<T> result(Grid<T>::R,Grid<T>::R);

  for (j = 0; j < Grid<T>::R; ++j)
  {
    for (i = 0; i < Grid<T>::R; ++i)
      p.Elem(i,0) = T(0);

    p.Elem(j,0) = T(1);

    p = LUPSolve(perm,p);

    for (i = 0; i < Grid<T>::R; ++i)
      result.Elem(i,j) = p.Elem(i,0);
  }

  return result;
}

//=============================================================
#if OVERLOAD_NEW
template <class T>
void* Matrix<T>::operator new (size_t Bytes_)
{
  return MemMatrix::Matrix().Allocate(Bytes_);
}

//=============================================================
template <class T>
void Matrix<T>::operator delete (void* Space_)
{
  MemMatrix::Matrix().Deallocate(Space_);
}

//=============================================================
#if HAS_ARRAY_NEW
template <class T>
void* Matrix<T>::operator new[] (size_t Bytes_)
{
  return MemMatrix::Matrix().Allocate(Bytes_);
}

//=============================================================
template <class T>
void Matrix<T>::operator delete[] (void* Space_)
{
  MemMatrix::Matrix().Deallocate(Space_);
}
#endif
#endif

//=============================================================
#endif




