#ifndef NESTARRAY_CPP
#define NESTARRAY_CPP
#ifndef NESTARRAY_H
  #include "nestarray.h"
#endif

/****************************************************************************/
// PURPOSE:
//   Linear position to matrix coordinates conversion function using
//   calulated linear offsets vector to determine results.
//
// PRE:
//   int Pos_      : Linear position value to be converted into data matrix
//                   coordinates.
//   int* Coords_  : int vector to be filled with the data matrix coordinates.
//   int CoordLen_ : Length of the data matrix coordinates vector excluding
//                   zero index element.
//   int* OffSets_ : Progressive linear position offsets to be used in
//                   continuous divisions of the linear position value to
//                   give the equivalent data matrix coordinates.
//
//   int OffsetsLen_ : Length of the linear offsets vector.
//
// POST:
//   Returns the int vector that is filled with the data matrix coordinates
//   that is equivalent to the specified linear position value.
//
int* LinearToCoords(int Pos_, int* Coords_, int CoordsLen_,
                    int* OffSets_, int OffSetsLen_)
{
  // null pointer checks
  if (!Coords_ || !OffSets_)
    return NULL;

  int y;
  int x = 1;

  if (Pos_ / OffSets_[x])
    return NULL;

  x += 2;
  for (y = 1; y < CoordsLen_ && x < OffSetsLen_; y++)
  {
    Coords_[y] = Pos_ / OffSets_[x];
    Pos_ %= OffSets_[x];
    x += 2;
  }
  
  if (y == CoordsLen_)
  {  
    Coords_[0] = CoordsLen_;
    Coords_[CoordsLen_] = Pos_;
    return Coords_;
  }

  return NULL;
}

/****************************************************************************/
// PURPOSE:
//   Matrix coordinates to linear position conversion function using
//   calculated linear vector offsets to determine results.
//
// PRE:
//   int* Coords_    : Data matrix coordinates.
//   int CoordsLen_  : Length of the data matrix coordinates vector excluding
//                     zero index element.
//   int* Offsets_   : Progressive linear position offsets to be multiplied
//                     with the data matrix coordinates to give the
//                     equivalent linear position value.
//   int OffSetsLen_ : Length of the linear position offsets vector.
//
// POST;
//   Returns the equivalent linear position value for the given data matrix
//   coordinates. If an error occurs, returns -1.
//
int CoordsToLinear(int* Coords_, int CoordsLen_,
                   int* OffSets_, int OffSetsLen_)
{
  // null pointer checks
  if (!Coords_ || !OffSets_)
    return -1;

  int y;
  int x = 2;  
  int pos = 0;

  for (y = 1; y < CoordsLen_ && x < OffSetsLen_ &&
       InRangeX(-1, Coords_[y], OffSets_[x++]); y++)
    pos += (Coords_[y] * OffSets_[x++]);
  pos += Coords_[CoordsLen_];
  
  return ((y == CoordsLen_) ? pos:-1);
}

/****************************************************************************/
// PURPOSE:
//   Method to calculate and return an allocated int array of linear
//   position offsets given a int vector filled with the array lengths of
//   the constituent arrays within the data matrix.
//
// Offset calculation scheme example:
//
//   [asz1][cnt1]
//   [asz2][cnt2==cnt1*asz1]
//   [asz3][cnt3==cnt2*asz2]
//   [asz4][cnt4==cnt3*asz3]
//   [asz5][cnt5==cnt4*asz4]
//   [asz6][cnt6==cnt5*asz5]
//   [ 1  ][cnt7==cnt6*asz6]
//
// Created offset array in reverse order
// [asz1][cnt1][asz2][cnt2][asz3][cnt3][asz4][cnt4][asz5][cnt5][asz6][cnt6]
// [ 1  ][cnt7]
//
// Resorted to go from largest base matrix vector to smallest element vector
// [ 1  ][cnt7][asz6][cnt6][asz5][cnt5][asz4][cnt4][asz3][cnt3][asz2][cnt2]
// [asz1][cnt1]
//
// Matrix position coordinates scheme example:
//
//   PosVect[0] = (Not Used or Set With Position Vector Length)
//   PosVect[1] = Holds index position into 2nd level array
//   PosVect[2] = Holds index position into 3rd level array
//   ...
//   PosVect[Dim] = Holds index into basic array element data in 1d vector
//
// PRE:
//   int* ArraySizes_ : The array lengths of the constituent arrays within
//                      the data matrix.
//   int ArrayLen_    : The length of the int vector containing the array
//                      lengths.
//
// POST:
//   Returns an int array filled with the linear position offsets of the
//   constituent arrays within the data matrix.
//
int* CalculateOffsets(int* ArraySizes_, int ArrayLen_, int& OffSetsLen_)
{
  // null pointer checks
  if (!ArraySizes_)
    return NULL;

  int x;
  int i = 1;
  int cnt = 1;
  int* OffSets_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * ArrayLen_ * 2);

  // Working backwards from array length of deepest 1d element to top level
  // container matrix length.
  for (x = ArrayLen_ - 1; x >= 0; x--)
  {
    cnt *= ArraySizes_[x];
    OffSets_[i] = cnt;
    OffSets_[i-1] = x ? ArraySizes_[x-1]:1;
    i += 2;
  }

  OffSetsLen_ = i - 1;
  i -= 3;
  x = 0;

  // Resorting offest array to go from largest base matrix vector to
  // smallest element vector.
  while (x < i)
  {
    cnt = OffSets_[x];
    OffSets_[x++] = OffSets_[i];
    OffSets_[i++] = cnt;

    cnt = OffSets_[x];
    OffSets_[x++] = OffSets_[i];
    OffSets_[i++] = cnt;
    i -= 4;
  }

  return OffSets_;
}

/****************************************************************************/
// PURPOSE:
//   Method to initialize individual vectors withing the data matrix.
//
// PRE:
//   void* Array_  : Indexed constituent array within the data matrix.
//   int Dim_      : The dimensions of the matrix, value range = [3...8]
//   int Off_      : The offset into the data matrix, value range = [0...5]
//   int Index_    : Index accessing constituent array of the data matrix,
//                   value range = [1...7]
//   int* LenVect_ : Length vector containing array lengths of constituent
//                   vectors from the general data matrix.
//
// POST:
//   Returns the indexed array after it has been initiallzed.
//   Return NULL on null pointer arguments (error).
//
void* InitMatrix(void* Array_, int Dim_, int Off_, int Index_, int* LenVect_)
{
  // null pointer checks
  if (!Array_ || !LenVect_)
    return NULL;

  int x;
  int sz = SizeOfVector(Dim_-Off_, Index_-Off_, LenVect_);
  
  int******** Vect8_ = (Index_ == 0) ? (int********)Array_:NULL;
  int******* Vect7_ = (Index_ == 1) ? (int*******)Array_:NULL;
  int****** Vect6_ = (Index_ == 2) ? (int******)Array_:NULL;
  int***** Vect5_ = (Index_ == 3) ? (int*****)Array_:NULL;
  int**** Vect4_ = (Index_ == 4) ? (int****)Array_:NULL;
  int*** Vect3_ = (Index_ == 5) ? (int***)Array_:NULL;
  int** Vect2_ = (Index_ == 6) ? (int**)Array_:NULL;
  int* Vect1_ = (Index_ == 7) ? (int*)Array_:NULL;
  
  for (x = 1; x < sz; x++)
  {
    switch (Index_)
    {
      case 0:
        Vect8_[x] = NULL;
        break;
        
      case 1:
        Vect7_[x] = NULL;
        break;
    
      case 2:
        Vect6_[x] = NULL;
        break;
    
      case 3:
        Vect5_[x] = NULL;
        break;
        
      case 4:
        Vect4_[x] = NULL;
        break;

      case 5:
        Vect3_[x] = NULL;
        break;

      case 6:
        Vect2_[x] = NULL;
        break;

      case 7:
        Vect1_[x] = 0;
    }    
  }

  return Array_;
}

/****************************************************************************/
// PURPOSE:
//   Method to set the parent references vector from the general data matrix
//   with values from a position vector indexed with a parent index value
//
// array[1...n]
//      [i=2...n][0][0][0][0] = 3rdArr array size=x index=i=2 of
//                              2nd...nth 2ndArr element. 0 = deleted.
//
// array[1...n]
//      [i=2...n][0][0][0][(i+1)=3...x+i+1] = Deallocated array flag = 0/1.
//                                          = Shallow array copy flag = 0/2.
//
// array[1...n]
//      [i=2...n][1][0][0][(i+1)=3...x+i+1] = 3rdArr to TopPar pointer
//      reference number, referencing TopPar pointer in 1st...nth TopArr
//      element for 3rdArr arrays from 3...x.
//
// array[1...n]
//      [i=2...n][i=2][0][0][(i+1)=3...x+i+1] = 3rdArr to 2ndArr pointer
//      reference number, referencing 2ndArr pointer in 2nd...nth 2ndArr
//      element for 3rdArr arrays from 3...x.
//
// array[1...n]
//      [i=2...n][(i+1)=3...x+i+1]* = 1st starting 3rdArr value element to
//                                    ending 3rdArr value element x.
//
// PRE:
//   int* Vector_     : Parent reference vector retrieved from the data matrix
//   int Dim_         : The dimensions of the matrix, value range = [3...8]
//   int ParentIndex_ : Index into the position vector containing parent
//                      reference values, denoted by variable i = 1...Index_.
//   int Index_       : Index level into position vector, denoted by constant
//                      i == Index_, max = Index_ + 1
//   int* LenVect_    : Length vector containing array lengths of constituent
//                      vectors from the general data matrix.
//   int* PosVect_    : Position vector containing array element positions of
//                      constituent vectors from the general data matrix.
//   bool& Error_     : Error in method if set TRUE, otherwise set to FALSE.
//
int* SetParentReference(int* Vector_, int Dim_,
                        int ParentIndex_, int Index_,
                        int* LenVect_, int* PosVect_, bool& Error_)
{
  // null pointer checks
  if (!Vector_ || !LenVect_ || !PosVect_)
    return NULL;

  int pos;
  int x = Index_;        // x = max - 1

  if (InRange(x+1, PosVect_[x+1], SizeOfVector(Dim_, x, LenVect_) - 1))
    pos = PosVect_[x+1]; // = [(i+1)=3...x+i+1]
  else
    pos = 0;

  Error_ = !pos;
  
  if (pos)               // array[n]...[(i+1)=3...x+i+1] = posvect[1...i]
    Vector_[pos] = PosVect_[ParentIndex_];

  return Vector_;
}

/****************************************************************************/
// PURPOSE:
//   Method to get the parent references vector from the index into the data
//   matrix.
//
// array[1...n]
//      [i=2...n][0][0][0][0] = 3rdArr array size=x index=i=2 of
//                              2nd...nth 2ndArr element. 0 = deleted.
//
// array[1...n]
//      [i=2...n][0][0][0][(i+1)=3...x+i+1] = Deallocated array flag = 0/1.
//                                          = Shallow array copy flag = 0/2.
//
// array[1...n]
//      [i=2...n][1][0][0][(i+1)=3...x+i+1] = 3rdArr to TopPar pointer
//      reference number, referencing TopPar pointer in 1st...nth TopArr
//      element for 3rdArr arrays from 3...x.
//
// array[1...n]
//      [i=2...n][i=2][0][0][(i+1)=3...x+i+1] = 3rdArr to 2ndArr pointer
//      reference number, referencing 2ndArr pointer in 2nd...nth 2ndArr
//      element for 3rdArr arrays from 3...x.
//
// array[1...n]
//      [i=2...n][(i+1)=3...x+i+1]* = 1st starting 3rdArr value element to
//                                    ending 3rdArr value element x.
//
// PRE:
//   int* ParentRefs_ : Parent references vector of specified constituent
//                      vector within general data matrix accessed by
//                      MatrixIndex_ argument
//   int Dim_       : The dimensions of the matrix, value range = [3...8]
//   int Off_       : The offset into the data matrix, value range = [0...5]
//   int MatrixIndex_ : Index accessing constituent array of the data matrix,
//                      value range = [1...7]
//   int* LenVect_  : Length vector containing length of constituent arrays
//   int* PosVect_  : Position vector containing coordinates into matrix
//   void* Array_     : vector retrieved from the data matrix containing
//                      parent array references to be retrieved.
//   bool& Error_   : Error in array indexing (out of range)
//
// POST:
//   Returns parent references vector.
//   Parent reference vector format:
//       ParentRefs[0] == Size of array starting from index element 1
//       ParentRefs[1] == referencing TopPar pointer in 1st...nth TopArr
//       ParentRefs[2] == referencing 2ndArr pointer in 2nd...nth 2ndArr
//       ...
//       ParentRefs[3] == referencing Dim-1 Arr pointer in Dim-1...nth Array
//
int* GetParentReference(int* ParentRefs_, int Dim_, int Off_, int MatrixIndex_,
                        int* LenVect_, int* PosVect_, void* Array_, bool& Error_)
{
  // null pointer checks
  if (!ParentRefs_ || !LenVect_ || !PosVect_ || !Array_)
    return NULL;

  int pos;
  int y;
  int x = MatrixIndex_;
  int xOff_ = x - Off_;
  int xDim_ = Dim_ - Off_;
  int* Vector_;

  int******* Vect7_ = (x == 1) ? (int*******)Array_:NULL;
  int****** Vect6_ = (x == 2) ? (int******)Array_:NULL;
  int***** Vect5_ = (x == 3) ? (int*****)Array_:NULL;
  int**** Vect4_ = (x == 4) ? (int****)Array_:NULL;
  int*** Vect3_ = (x == 5) ? (int***)Array_:NULL;
  int** Vect2_ = (x == 6) ? (int**)Array_:NULL;
  int* Vect1_ = (x == 7) ? (int*)Array_:NULL;

  if (InRange(xOff_+1, PosVect_[xOff_+1], SizeOfVector(xDim_, xOff_, LenVect_) - 1))
    pos = PosVect_[xOff_+1];   // = [(i+1)=3...x+i+1]
  else
    pos = 0;

  Error_ = !pos;  
    
  if (pos)
  {    
    // 0th index of parent reference array is the size of the array
    ParentRefs_[0] = xOff_;
    
    for (y = 1; y <= xOff_; y++)
    {
      // ParentRefs[1...n] = array[1...n]...[(i+1)=3...x+i+1]
      Vector_ =
      (x == 1) ? Vect7_[y][0][0][0][0][0]:
      (x == 2) ? Vect6_[y][0][0][0][0]:
      (x == 3) ? Vect5_[y][0][0][0]:
      (x == 4) ? Vect4_[y][0][0]:
      (x == 5) ? Vect3_[y][0]:
      (x == 6) ? Vect2_[y]:
      (x == 7) ? Vect1_:NULL;

      if (Vector_)
        ParentRefs_[y] = Vector_[pos];
    }
  }

  return ParentRefs_;
}

/****************************************************************************/
// PURPOSE:
//   Method to set the reference array for each constituent array
//
// array[1...n]
//      [i=2...n][0][0][0][0] = 3rdArr array size=x index=i=2 of
//                              2nd...nth 2ndArr element. 0 = deleted.
//
// array[1...n]
//      [i=2...n][0][0][0][(i+1)=3...x+i+1] = Deallocated array flag = 0/1.
//                                          = Shallow array copy flag = 0/2.
//
// array[1...n]
//      [i=2...n][1][0][0][(i+1)=3...x+i+1] = 3rdArr to TopPar pointer
//      reference number, referencing TopPar pointer in 1st...nth TopArr
//      element for 3rdArr arrays from 3...x.
//
// array[1...n]
//      [i=2...n][i=2][0][0][(i+1)=3...x+i+1] = 3rdArr to 2ndArr pointer
//      reference number, referencing 2ndArr pointer in 2nd...nth 2ndArr
//      element for 3rdArr arrays from 3...x.
//
// array[1...n]
//      [i=2...n][(i+1)=3...x+i+1]* = 1st starting 3rdArr value element to
//                                    ending 3rdArr value element x.
//
// PRE:
//   void* Array   : Data matrix
//   int Dim_      : The dimensions of the matrix, value range = [3...8]
//   int Off_      : The offset into the data matrix, value range = [0...5]
//   int Index_    : Index accessing constituent array of the data matrix,
//                   value range = [1...6]
//   int* LenVect_ : Length vector containing length of constituent arrays
//   int* PosVect_ : Position vector containing coordinates into matrix
//   bool& Error_  : Error in array indexing (out of range)
//   bool TestMem_ : Test constituent arrays for memory allocation.
//                   If constituent array is non-null then no memory
//                   allocation takes place
//
// POST:
//   Returns initialized data matrix matrix of given dimensions of length
//
void* SetRefArrays(void* Array_, int Dim_, int Off_, int Index_,
                   int* LenVect_, int* PosVect_, bool& Error_, bool TestMem_)
{
  // null pointer checks
  if (!LenVect_ || !PosVect_ || !Array_)
    return NULL;

  int x, y;
  int sz = SizeOfVector(Dim_-Off_, Index_-Off_, LenVect_);
  int vsz = 0;

  int******* Vect7_ = (Index_ == 1) ? (int*******)Array_:NULL;
  int****** Vect6_ = (Index_ == 2) ? (int******)Array_:NULL;
  int***** Vect5_ = (Index_ == 3) ? (int*****)Array_:NULL;
  int**** Vect4_ = (Index_ == 4) ? (int****)Array_:NULL;
  int*** Vect3_ = (Index_ == 5) ? (int***)Array_:NULL;
  int** Vect2_ = (Index_ == 6) ? (int**)Array_:NULL;
  int* Vect1_ = (Index_ == 7) ? (int*)Array_:NULL;

  for (x = Off_+1; x <= Index_; x++)
  {
    // array[1]...array[n]
    switch (Index_)
    {
      case 1:
        if (Vect7_ && !Vect7_[x-Off_] || !TestMem_)
        {
          vsz = (x-Off_ >= 1) ? x-Off_:1;
          Vect6_ = Vect7_[x-Off_] =
          (int******)RawCallocateWith(MEMMATRIX, vsz * sizeof(int*****));
        }
        break;
        
      case 2:
        if (Vect6_ && !Vect6_[x-Off_] || !TestMem_)
        {
          vsz = (x-Off_ >= 1) ? x-Off_:1;
          Vect5_ = Vect6_[x-Off_] =
          (int*****)RawCallocateWith(MEMMATRIX, vsz * sizeof(int****));
        }
        break;
    
      case 3:
        if (Vect5_ && !Vect5_[x-Off_] || !TestMem_)
        {
          vsz = (x-Off_ >= 1) ? x-Off_:1;
          Vect4_ = Vect5_[x-Off_] =
          (int****)RawCallocateWith(MEMMATRIX, vsz * sizeof(int***));
        }
        break;
        
      case 4:
        if (Vect4_ && !Vect4_[x-Off_] || !TestMem_)
        {
          vsz = (x-Off_ >= 1) ? x-Off_:1;
          Vect3_ = Vect4_[x-Off_] =
          (int***)RawCallocateWith(MEMMATRIX, vsz * sizeof(int**));
        }
        break;

      case 5:
        if (Vect3_ && !Vect3_[x-Off_] || !TestMem_)
        {
          vsz = (x-Off_ >= 1) ? x-Off_:1;
          Vect2_ = Vect3_[x-Off_] =
          (int**)RawCallocateWith(MEMMATRIX, vsz * sizeof(int*));
        }
        break;

      case 6:
        vsz = (x-Off_ >= 1) ? x-Off_:1;
        if (vsz < sz)
          vsz = sz;
      
        if (Vect2_ && !Vect2_[x-Off_] || !TestMem_ ||
              (Vect2_[x-Off_] &&
               MemMatrix::Matrix().HasThis(Vect2_[x-Off_]) &&
               !MemPool::MemFit(Vect2_[x-Off_], vsz * sizeof(int))))
        {
          Vect2_[x-Off_] =
          (int*)RawReallocateWith(MEMMATRIX, Vect2_[x-Off_], vsz * sizeof(int));
          Vect1_ = Vect2_[x-Off_];
        }
        else if (Vect2_)
        {
          if (Vect2_[x-Off_])
            Vect1_ = Vect2_[x-Off_];
          else
            Vect1_ = Vect2_[x-Off_] =
            (int*)RawCallocateWith(MEMMATRIX, vsz * sizeof(int));
        }
    }
  
    for (y = Index_ + 2; y <= Dim_; y++)
    {
      // array[n]...[0]...[0]
      switch (y)
      {
        case 3:
          if (!Vect6_[0] || !TestMem_)
            Vect5_ = Vect6_[0] =
            (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****));
          break;
          
        case 4:
          if (!Vect5_[0] || !TestMem_)
            Vect4_ = Vect5_[0] =
            (int****)RawCallocateWith(MEMMATRIX, sizeof(int***));
          break;
      
        case 5:
          if (!Vect4_[0] || !TestMem_)
            Vect3_ = Vect4_[0] =
            (int***)RawCallocateWith(MEMMATRIX, sizeof(int**));
          break;

        case 6:
          if (!Vect3_[0] || !TestMem_)
            Vect2_ = Vect3_[0] =
            (int**)RawCallocateWith(MEMMATRIX, sizeof(int*));
          break;

        case 7:
          if (!Vect2_[0] || !TestMem_ ||
                (Vect2_[0] &&
                 MemMatrix::Matrix().HasThis(Vect2_[0]) &&
                 !MemPool::MemFit(Vect2_[0], sizeof(int) * sz)))
          {
            Vect2_[0] =
            (int*)RawReallocateWith(MEMMATRIX, Vect2_[0], sizeof(int) * sz);
            Vect1_ = Vect2_[0];
          }
          else if (Vect2_)
          {
            if (Vect2_[0])            
              Vect1_ = Vect2_[0];
            else
              Vect1_ = Vect2_[0] =
              (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
          }
          break;

        case 8:
          SetParentReference(Vect1_, Dim_-Off_, x-Off_, Index_-Off_, LenVect_, PosVect_, Error_);
          break;
      }
    }
  }

  return Array_;
}

/****************************************************************************/
// PURPOSE:
//   Method to remove the specified vector from the data matrix given the
//   element index into the matrix where the vector is contained.
//
// PRE:
//   void* Array_     : vector retrieved from the data matrix that is to be
//                      removed.
//   int Dim_         : The dimensions of the matrix, value range = [3...8]
//   int Off_         : The offset into the data matrix, value range = [0...5]
//   int MatrixIndex_ : Index accessing constituent array of the data matrix,
//                      value range = [1...7]
//   int* LenVect_    : Length vector containing length of constituent arrays
//
// POST:
//   The specified vector is removed from the data matrix.
//
void RemoveArray(void* Array_, int Dim_, int Off_, int MatrixIndex_, int* LenVect_)
{
  // null pointer checks
  if (Dim_ && LenVect_ != NULL && Array_ != NULL)
  {
    int y;
    int x = MatrixIndex_;
    int xOff_ = x - Off_;
    int xDim_ = Dim_ - Off_;
    int sz = SizeOfVector(xDim_, xOff_, LenVect_);
    int min;
    // [i=2...n][(i+1)=3...x+i+1]*

    int******* Vect7_ = (x == 1) ? (int*******)Array_:NULL;
    int****** Vect6_ = (x == 2) ? (int******)Array_:NULL;
    int***** Vect5_ = (x == 3) ? (int*****)Array_:NULL;
    int**** Vect4_ = (x == 4) ? (int****)Array_:NULL;
    int*** Vect3_ = (x == 5) ? (int***)Array_:NULL;
    int** Vect2_ = (x == 6) ? (int**)Array_:NULL;
    int* Vect1_ = (x == 7) ? (int*)Array_:NULL;
    
    int* Contain_ =
    (x == 1) ? Vect7_[0][0][0][0][0][0]:
    (x == 2) ? Vect6_[0][0][0][0][0]:
    (x == 3) ? Vect5_[0][0][0][0]:
    (x == 4) ? Vect4_[0][0][0]:
    (x == 5) ? Vect3_[0][0]:
    (x == 6) ? Vect2_[0]:
    (x == 7) ? Vect1_:NULL;

    // Contain_[pos] = 1;    // deleted
    // Contain_[pos] = 2;    // shared
    // SizeOfVector(Dim_, x, LenVect_)
    // CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_)
    // StatusOfVector(Dim_, x, Off_, y, Vectl_, Contain_)
    // SetStatusOfVector(Dim_, x, Off_, y, stat, Contain_)

    if (Dim_ <= 2)
    {
      if (Dim_ == 2 && Vect2_)
      {
        if (Vect2_[0])
          for (y = 1; y < sz; y++)
            if (Vect2_[y])
            {
              ::DeleteArray(Vect2_[y]);
              Vect2_[y] = NULL;
            }
      }
      else
      {
        ::DeleteArray(Vect1_);
        Vect1_ = NULL;
      }
    }
    else if (1 <= x && x < 7)
    {
      for (y = xOff_+1; y < sz; y++)
      {
        if (x == 1 && Vect7_ && Vect7_[y] &&
            StatusOfVector(Dim_, x, Off_, y, Vect1_, Contain_) == 0)
        {
          SetStatusOfVector(Dim_, x, Off_, y, 1, Contain_);
          RemoveArray(Vect7_[y], Dim_, Off_, x+1, LenVect_);
          
          if (Vect7_[0])
          {
            min = int(Contain_[0]);
            if (y < min && Vect7_[y])
            {
              ::DeleteArray(Vect7_[y]);
              Vect7_[y] = NULL;
            }
          }
        }
        else if (x == 2 && Vect6_ && Vect6_[y] &&
                 StatusOfVector(Dim_, x, Off_, y, Vect1_, Contain_) == 0)
        {
          SetStatusOfVector(Dim_, x, Off_, y, 1, Contain_);
          RemoveArray(Vect6_[y], Dim_, Off_, x+1, LenVect_);

          if (Vect6_[0])
          {
            min = int(Contain_[0]);
            if (y < min && Vect6_[y])
            {
              ::DeleteArray(Vect6_[y]);
              Vect6_[y] = NULL;
            }
          }
        }
        else if (x == 3 && Vect5_ && Vect5_[y] &&
                 StatusOfVector(Dim_, x, Off_, y, Vect1_, Contain_) == 0)
        {
          SetStatusOfVector(Dim_, x, Off_, y, 1, Contain_);        
          RemoveArray(Vect5_[y], Dim_, Off_, x+1, LenVect_);

          if (Vect5_[0])
          {
            min = int(Contain_[0]);
            if (y < min && Vect5_[y])
            {
              ::DeleteArray(Vect5_[y]);
              Vect5_[y] = NULL;
            }
          }
        }
        else if (x == 4 && Vect4_ && Vect4_[y] &&
                 StatusOfVector(Dim_, x, Off_, y, Vect1_, Contain_) == 0)
        {
          SetStatusOfVector(Dim_, x, Off_, y, 1, Contain_);
          RemoveArray(Vect4_[y], Dim_, Off_, x+1, LenVect_);

          if (Vect4_[0])
          {
            min = int(Contain_[0]);
            if (y < min && Vect4_[y])
            {
              ::DeleteArray(Vect4_[y]);
              Vect4_[y] = NULL;
            }
          }
        }
        else if (x == 5 && Vect3_ && Vect3_[y] &&
                 StatusOfVector(Dim_, x, Off_, y, Vect1_, Contain_) == 0)
        {
          SetStatusOfVector(Dim_, x, Off_, y, 1, Contain_);
          RemoveArray(Vect3_[y], Dim_, Off_, x+1, LenVect_);

          if (Vect3_[0])
          {
            min = int(Contain_[0]);
            if (y < min && Vect3_[y])
            {
              ::DeleteArray(Vect3_[y]);
              Vect3_[y] = NULL;
            }
          }
        }
        else if (x == 6 && Vect2_ && Vect2_[y] &&
                 StatusOfVector(Dim_, x, Off_, y, Vect1_, Contain_) == 0)
        {
          SetStatusOfVector(Dim_, x, Off_, y, 1, Contain_);

          if (Vect2_[0])
          {
            min = int(Contain_[0]);
            if (y < min && Vect2_[y])
            {
              ::DeleteArray(Vect2_[y]);
              Vect2_[y] = NULL;
            }
          }          
        }
      }

      if (x == 1 && Vect7_)
      {
        ::DeleteArray(Vect7_);
        Vect7_ = NULL;
      }
      else if (x == 2 && Vect6_)
      {
        ::DeleteArray(Vect6_);
        Vect6_ = NULL;
      }        
      else if (x == 3 && Vect5_)
      {
        ::DeleteArray(Vect5_);
        Vect5_ = NULL;
      }        
      else if (x == 4 && Vect4_)
      {
        ::DeleteArray(Vect4_);
        Vect4_ = NULL;
      }        
      else if (x == 5 && Vect3_)
      {
        ::DeleteArray(Vect3_);
        Vect3_ = NULL;
      }
      else if (x == 6 && Vect2_)
      {
        ::DeleteArray(Vect2_);
        Vect2_ = NULL;
      }
    }
    else if (x == 7 && Vect1_)
    {
      ::DeleteArray(Vect1_);
      Vect1_ = NULL;
    }
  }
}

/****************************************************************************/
// PURPOSE:
//   Method to set ancestor references on an array given an array filled with
//   parent references.
//
// PRE:
//   int* ParentRefs_ : The parent reference array filled with parent
//                      references of the specified vector that is passed.
//   int Dim_         : The dimensions of the matrix, value range = [3...8]
//   int Off_         : The offset into the data matrix, value range = [0...5]
//   int MatrixIndex_ : Index accessing constituent array of the data matrix,
//                      value range = [1...7]
//   int* LenVect_    : Length vector containing length of constituent arrays
//   int* PosVect_    : Position vector containing coordinates into matrix
//   void* Array_     : vector retrieved from the data matrix containing the
//                      array to be assigned parent references.
//   bool& Error_     : Error in array indexing (out of range)
//
// POST:
//   Returns the data matrix with the specified array assigned with parent
//   references.
//
void* SetReferences(int* ParentRefs_, int Dim_, int Off_, int MatrixIndex_,
                    int* LenVect_, int* PosVect_, void* Array_, bool& Error_)
{
  // null pointer checks
  if (!ParentRefs_ || !LenVect_ || !PosVect_ || !Array_)
    return NULL;

  int pos;
  int y;
  int x = MatrixIndex_;
  int xOff_ = x - Off_;
  int xDim_ = Dim_ - Off_;
  int* Vector_;

  int******* Vect7_ = (x == 1) ? (int*******)Array_:NULL;
  int****** Vect6_ = (x == 2) ? (int******)Array_:NULL;
  int***** Vect5_ = (x == 3) ? (int*****)Array_:NULL;
  int**** Vect4_ = (x == 4) ? (int****)Array_:NULL;
  int*** Vect3_ = (x == 5) ? (int***)Array_:NULL;
  int** Vect2_ = (x == 6) ? (int**)Array_:NULL;
  int* Vect1_ = (x == 7) ? (int*)Array_:NULL;

  if (InRange(xOff_+1, PosVect_[xOff_+1], SizeOfVector(xDim_, xOff_, LenVect_) - 1))
    pos = PosVect_[xOff_+1];   // = [(i+1)=3...x+i+1]
  else
    pos = 0;

  Error_ = !pos;  

  if (pos)
  {        
    for (y = 1; y <= ParentRefs_[0]; y++)
    {
      // array[1...n]...[(i+1)=3...x+i+1] = ParentRefs[1...n]
      Vector_ =
      (x == 1) ? Vect7_[y][0][0][0][0][0]:
      (x == 2) ? Vect6_[y][0][0][0][0]:
      (x == 3) ? Vect5_[y][0][0][0]:
      (x == 4) ? Vect4_[y][0][0]:
      (x == 5) ? Vect3_[y][0]:
      (x == 6) ? Vect2_[y]:
      (x == 7) ? Vect1_:NULL;

      if (Vector_)
        Vector_[pos] = ParentRefs_[y];
    }
  }

  return Array_;
}

/****************************************************************************/
// PURPOSE:
//   Method to remove the parent references from the general data matrix
//   with values from a position vector indexed with a parent index value
//
// array[1...n]
//      [i=2...n][0][0][0][0] = 3rdArr array size=x index=i=2 of
//                              2nd...nth 2ndArr element. 0 = deleted.
//
// array[1...n]
//      [i=2...n][0][0][0][(i+1)=3...x+i+1] = Deallocated array flag = 0/1.
//                                          = Shallow array copy flag = 0/2.
//
// array[1...n]
//      [i=2...n][1][0][0][(i+1)=3...x+i+1] = 3rdArr to TopPar pointer
//      reference number, referencing TopPar pointer in 1st...nth TopArr
//      element for 3rdArr arrays from 3...x.
//
// array[1...n]
//      [i=2...n][i=2][0][0][(i+1)=3...x+i+1] = 3rdArr to 2ndArr pointer
//      reference number, referencing 2ndArr pointer in 2nd...nth 2ndArr
//      element for 3rdArr arrays from 3...x.
//
// array[1...n]
//      [i=2...n][(i+1)=3...x+i+1]* = 1st starting 3rdArr value element to
//                                    ending 3rdArr value element x.
//
// PRE:
//   void* Container_ : Container array 1 level up which contains the
//                      data vector.
//   int Dim_         : The dimensions of the matrix, value range = [3...8]
//   int Off_         : The offset into the data matrix, value range = [0...5]
//   int MatrixIndex_ : Index accessing constituent array of the data matrix,
//                      value range = [1...7]
//   int* LenVect_    : Length vector containing length of constituent arrays
//   int* PosVect_    : Position vector containing coordinates into matrix
//   void* Replacement_ : Replacement vector for deleted existing vector
//                        if not set as NULL, otherwise deleted vector in
//                        data matrix is set as NULL in the array entry.
//   bool DeAlloc_      : Delete the specified array in the data vector rather
//                        than replace it or set the vector size to zero.
//   int* Shared_       : Set to the deallocation status flag of the
//                        container vector of the specified vector to be
//                        deleted. Valid values include: 0=filled,
//                        1=deallocated, 2=shared
//   int* Size_         : Set to the size in elements of the container vector.
//   int* Max_          : Set to the maximum length of the container vector.
//   bool& Error_       : Error in array indexing (out of range)
//
void* RemoveReferences(void* Container_, int Dim_, int Off_,
                       int MatrixIndex_, int* LenVect_, int* PosVect_,
                       void* Replacement_, bool DeAlloc_, int* Shared_,
                       int* Size_, int* Max_, bool& Error_)
{
  // null pointer checks
  if (!Container_ || !LenVect_ || !PosVect_)
    return NULL;

  int CurPos_;
  int PrevPos_;
  int x = MatrixIndex_;
  int lenpos = Dim_ - x;
  int xOff_ = x - Off_;
  int xDim_ = Dim_ - Off_;
  bool inc;

  if (InRange(xOff_, PosVect_[xOff_], SizeOfVector(xDim_, xOff_-1, LenVect_) - 1))
    PrevPos_ = PosVect_[xOff_];
  else
    PrevPos_ = 0;

  if (x < 7)
  {
    if (InRange(xOff_+1, PosVect_[xOff_+1], SizeOfVector(xDim_, xOff_, LenVect_) - 1))
      CurPos_ = PosVect_[xOff_+1];
    else
      CurPos_ = 0;
  }
  else
    CurPos_ = 0;

  Error_ = PrevPos_ == 0 || (x < 7 && CurPos_ == 0);
  
  if (Error_)
    return NULL;    

  int******** Vect8_ = (x==1) ? (int********)Container_:NULL;
  int******* Vect7_ = (x==2) ? (int*******)Container_:((Vect8_ && x==1) ? Vect8_[PrevPos_]:NULL);
  int****** Vect6_ = (x==3) ? (int******)Container_:((Vect7_ && x==2) ? Vect7_[PrevPos_]:NULL);
  int***** Vect5_ = (x==4) ? (int*****)Container_:((Vect6_ && x==3) ? Vect6_[PrevPos_]:NULL);
  int**** Vect4_ = (x==5) ? (int****)Container_:((Vect5_ && x==4) ? Vect5_[PrevPos_]:NULL);
  int*** Vect3_ = (x==6) ? (int***)Container_:((Vect4_ && x==5) ? Vect4_[PrevPos_]:NULL);
  int** Vect2_ = (x==7) ? (int**)Container_:((Vect3_ && x==6) ? Vect3_[PrevPos_]:NULL);
  int* Vect1_ = (x==7) ? (Vect2_ ? Vect2_[PrevPos_]:NULL):NULL;

  int pos = PrevPos_;
  int* Vector_;
  int** Contain2_;
  int* Contain_;
  
  int* ParentRefs_ =
  (x < 7) ? ((int*)RawCallocateWith(MEMMATRIX, sizeof(int) * (xDim_ + 2))):NULL;
  ParentRefs_[0] = ParentRefs_[xDim_ + 1] = 0;

  if (x < 7)
  {      
    Vector_ =
    (x == 1 && Vect7_) ? Vect7_[0][0][0][0][0][0]:
    (x == 2 && Vect6_) ? Vect6_[0][0][0][0][0]:
    (x == 3 && Vect5_) ? Vect5_[0][0][0][0]:
    (x == 4 && Vect4_) ? Vect4_[0][0][0]:
    (x == 5 && Vect3_) ? Vect3_[0][0]:
    (x == 6 && Vect2_) ? Vect2_[0]:NULL;

    Contain2_ =
    (x == 1 && Vect8_) ? Vect8_[0][0][0][0][0][0]:
    (x == 2 && Vect7_) ? Vect7_[0][0][0][0][0]:
    (x == 3 && Vect6_) ? Vect6_[0][0][0][0]:
    (x == 4 && Vect5_) ? Vect5_[0][0][0]:
    (x == 5 && Vect4_) ? Vect4_[0][0]:
    (x == 6 && Vect3_) ? Vect3_[0]:NULL;

    Contain_ = Contain2_ ? Contain2_[0]:NULL;
  }
  else
  {
    Vector_ = Vect1_;
    Contain2_ = Vect2_;
    Contain_ = Vect2_[0];
  }

  // Contain_[pos] = 1;    // deleted
  // Contain_[pos] = 2;    // shared
  // SizeOfVector(Dim_, x, LenVect_)
  // CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_)
  // StatusOfVector(Dim_, x, Off_, y, Vectl_, Contain_)
  // SetStatusOfVector(Dim_, x, Off_, y, stat, Contain_)
  
  // 0th index of parent reference array is the size of the array
  switch (x)
  {
    case 1:
      if (Vect8_[pos] || Replacement_)
      {
        if (Replacement_)
        {
          inc = !Vect7_ || !Vector_[0];
          if (inc)                            // increment size
            IncCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

          Vect8_[pos] = (int*******)Replacement_;
          if (Vect7_ && Vect7_[xOff_] && Vect8_[pos][xOff_])
            GetParentReference(ParentRefs_, Dim_, Off_, MatrixIndex_,
                               LenVect_, PosVect_, Vect7_, Error_);

          RemoveArray(Vect7_, Dim_, Off_, MatrixIndex_, LenVect_);
          Vect7_ = Vect8_[pos];
          SetStatusOfVector(Dim_, x, Off_, pos, 2, Contain_);
          Vector_ = Vect7_[0][0][0][0][0][0];

          if (inc)
            Vector_[0] = LenVect_[lenpos];    // array[n]...[0] = Length[n]

          if (Vect7_[xOff_])
            SetReferences(ParentRefs_, Dim_, Off_, MatrixIndex_,
                          LenVect_, PosVect_, Vect7_, Error_);
          else
            SetRefArrays(Vect7_, Dim_, Off_, MatrixIndex_, LenVect_, PosVect_, Error_);
        }
        else
        {
          // Setting vector size
          if (CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_) > 0)       // decrement size
            DecCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

          if (DeAlloc_)
          {
            RemoveArray(Vect7_, Dim_, Off_, MatrixIndex_, LenVect_);
            Vect7_ = Vect8_[pos] = NULL;
            SetStatusOfVector(Dim_, x, Off_, pos, 1, Contain_);
            Vector_ = NULL;
          }
          else
            Vector_[0] = 0;          // subarray size = 0, (deleted)
        }

        if (Shared_)
          *Shared_ = StatusOfVector(Dim_, x, Off_, pos, Vect1_, Contain_);
          
        if (Size_)
          *Size_ = CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

        if (Max_)
          *Max_ = Contain_[0];
      }
      break;
    
    case 2:
      if (Vect7_[pos] || Replacement_)
      {
        if (Replacement_)
        {
          inc = !Vect6_ || !Vector_[0];
          if (inc)                            // increment size
            IncCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

          Vect7_[pos] = (int******)Replacement_;
          if (Vect6_ && Vect6_[xOff_] && Vect7_[pos][xOff_])
            GetParentReference(ParentRefs_, Dim_, Off_, MatrixIndex_,
                               LenVect_, PosVect_, Vect6_, Error_);

          RemoveArray(Vect6_, Dim_, Off_, MatrixIndex_, LenVect_);
          Vect6_ = Vect7_[pos];
          SetStatusOfVector(Dim_, x, Off_, pos, 2, Contain_);
          Vector_ = Vect6_[0][0][0][0][0];

          if (inc)
            Vector_[0] = LenVect_[lenpos];    // array[n]...[0] = Length[n]

          if (Vect6_[xOff_])
            SetReferences(ParentRefs_, Dim_, Off_, MatrixIndex_,
                          LenVect_, PosVect_, Vect6_, Error_);
          else
            SetRefArrays(Vect6_, Dim_, Off_, MatrixIndex_, LenVect_, PosVect_, Error_);
        }
        else
        {
          // Setting vector size
          if (CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_) > 0)       // decrement size
            DecCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

          if (DeAlloc_)
          {
            RemoveArray(Vect6_, Dim_, Off_, MatrixIndex_, LenVect_);
            Vect6_ = Vect7_[pos] = NULL;
            SetStatusOfVector(Dim_, x, Off_, pos, 1, Contain_);
            Vector_ = NULL;
          }
          else
            Vector_[0] = 0;          // subarray size = 0, (deleted)
        }

        if (Shared_)
          *Shared_ = StatusOfVector(Dim_, x, Off_, pos, Vect1_, Contain_);

        if (Size_)
          *Size_ = CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

        if (Max_)
          *Max_ = Contain_[0];        
      }
      break;
    
    case 3:
      if (Vect6_[pos] || Replacement_)
      {
        if (Replacement_)
        {
          inc = !Vect5_ || !Vector_[0];
          if (inc)                            // increment size
            IncCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

          Vect6_[pos] = (int*****)Replacement_;
          if (Vect5_ && Vect5_[xOff_] && Vect6_[pos][xOff_])
            GetParentReference(ParentRefs_, Dim_, Off_, MatrixIndex_,
                               LenVect_, PosVect_, Vect5_, Error_);

          RemoveArray(Vect5_, Dim_, Off_, MatrixIndex_, LenVect_);
          Vect5_ = Vect6_[pos];
          SetStatusOfVector(Dim_, x, Off_, pos, 2, Contain_);          
          Vector_ = Vect5_[0][0][0][0];

          if (inc)
            Vector_[0] = LenVect_[lenpos];    // array[n]...[0] = Length[n]

          if (Vect5_[xOff_])
            SetReferences(ParentRefs_, Dim_, Off_, MatrixIndex_,
                          LenVect_, PosVect_, Vect5_, Error_);
          else
            SetRefArrays(Vect5_, Dim_, Off_, MatrixIndex_, LenVect_, PosVect_, Error_);
        }
        else
        {
          // Setting vector size
          if (CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_) > 0)       // decrement size
            DecCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

          if (DeAlloc_)
          {
            RemoveArray(Vect5_, Dim_, Off_, MatrixIndex_, LenVect_);
            Vect5_ = Vect6_[pos] = NULL;
            SetStatusOfVector(Dim_, x, Off_, pos, 1, Contain_);            
            Vector_ = NULL;
          }
          else
            Vector_[0] = 0;          // subarray size = 0, (deleted)
        }

        if (Shared_)
          *Shared_ = StatusOfVector(Dim_, x, Off_, pos, Vect1_, Contain_);

        if (Size_)
          *Size_ = CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

        if (Max_)
          *Max_ = Contain_[0];        
      }      
      break;

    case 4:
      if (Vect5_[pos] || Replacement_)
      {
        if (Replacement_)
        {
          inc = !Vect4_ || !Vector_[0];
          if (inc)                            // increment size
            IncCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

          Vect5_[pos] = (int****)Replacement_;
          if (Vect4_ && Vect4_[xOff_] && Vect5_[pos][xOff_])
            GetParentReference(ParentRefs_, Dim_, Off_, MatrixIndex_,
                               LenVect_, PosVect_, Vect4_, Error_);

          RemoveArray(Vect4_, Dim_, Off_, MatrixIndex_, LenVect_);
          Vect4_ = Vect5_[pos];
          SetStatusOfVector(Dim_, x, Off_, pos, 2, Contain_);          
          Vector_ = Vect4_[0][0][0];

          if (inc)
            Vector_[0] = LenVect_[lenpos];    // array[n]...[0] = Length[n]

          if (Vect4_[xOff_])
            SetReferences(ParentRefs_, Dim_, Off_, MatrixIndex_,
                          LenVect_, PosVect_, Vect4_, Error_);
          else
            SetRefArrays(Vect4_, Dim_, Off_, MatrixIndex_, LenVect_, PosVect_, Error_);
        }
        else
        {
          // Setting vector size
          if (CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_) > 0)       // decrement size
            DecCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

          if (DeAlloc_)
          {
            RemoveArray(Vect4_, Dim_, Off_, MatrixIndex_, LenVect_);
            Vect4_ = Vect5_[pos] = NULL;
            SetStatusOfVector(Dim_, x, Off_, pos, 1, Contain_);
            Vector_ = NULL;
          }
          else
            Vector_[0] = 0;          // subarray size = 0, (deleted)
        }

        if (Shared_)
          *Shared_ = StatusOfVector(Dim_, x, Off_, pos, Vect1_, Contain_);

        if (Size_)
          *Size_ = CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

        if (Max_)
          *Max_ = Contain_[0];
      }      
      break;
    
    case 5:
      if (Vect4_[pos] || Replacement_)
      {
        if (Replacement_)
        {
          inc = !Vect3_ || !Vector_[0];
          if (inc)                            // increment size
            IncCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

          Vect4_[pos] = (int***)Replacement_;
          if (Vect3_ && Vect3_[xOff_] && Vect4_[pos][xOff_])
            GetParentReference(ParentRefs_, Dim_, Off_, MatrixIndex_,
                               LenVect_, PosVect_, Vect3_, Error_);

          RemoveArray(Vect3_, Dim_, Off_, MatrixIndex_, LenVect_);
          Vect3_ = Vect4_[pos];
          SetStatusOfVector(Dim_, x, Off_, pos, 2, Contain_);
          Vector_ = Vect3_[0][0];

          if (inc)
            Vector_[0] = LenVect_[lenpos];    // array[n]...[0] = Length[n]

          if (Vect3_[xOff_])
            SetReferences(ParentRefs_, Dim_, Off_, MatrixIndex_,
                          LenVect_, PosVect_, Vect3_, Error_);
          else
            SetRefArrays(Vect3_, Dim_, Off_, MatrixIndex_, LenVect_, PosVect_, Error_);
        }
        else
        {
          // Setting vector size
          if (CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_) > 0)       // decrement size
            DecCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);
          
          if (DeAlloc_)
          {
            RemoveArray(Vect3_, Dim_, Off_, MatrixIndex_, LenVect_);
            Vect3_ = Vect4_[pos] = NULL;
            SetStatusOfVector(Dim_, x, Off_, pos, 1, Contain_);            
            Vector_ = NULL;
          }
          else
            Vector_[0] = 0;          // subarray size = 0, (deleted)
        }

        if (Shared_)
          *Shared_ = StatusOfVector(Dim_, x, Off_, pos, Vect1_, Contain_);

        if (Size_)
          *Size_ = CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

        if (Max_)
          *Max_ = Contain_[0];
      }      
      break;
    
    case 6:
      if (Vect3_[pos] || Replacement_)
      {
        if (Replacement_)
        {
          inc = !Vect2_ || !Vector_[0];
          if (inc)                            // increment size
            IncCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

          Vect3_[pos] = (int**)Replacement_;
          if (Vect2_ && Vect2_[xOff_] && Vect3_[pos][xOff_])
            GetParentReference(ParentRefs_, Dim_, Off_, MatrixIndex_,
                               LenVect_, PosVect_, Vect2_, Error_);

          RemoveArray(Vect2_, Dim_, Off_, MatrixIndex_, LenVect_);
          Vect2_ = Vect3_[pos];
          SetStatusOfVector(Dim_, x, Off_, pos, 2, Contain_);
          Vector_ = Vect2_[0];

          if (inc)
            Vector_[0] = LenVect_[lenpos];    // array[n]...[0] = Length[n]

          if (Vect2_[xOff_])
            SetReferences(ParentRefs_, Dim_, Off_, MatrixIndex_,
                          LenVect_, PosVect_, Vect2_, Error_);
          else
            SetRefArrays(Vect2_, Dim_, Off_, MatrixIndex_, LenVect_, PosVect_, Error_);
        }
        else
        {
          // Setting vector size
          if (CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_) > 0)       // decrement size
            DecCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);
          
          if (DeAlloc_)
          {
            RemoveArray(Vect2_, Dim_, Off_, MatrixIndex_, LenVect_);
            Vect2_ = Vect3_[pos] = NULL;
            SetStatusOfVector(Dim_, x, Off_, pos, 1, Contain_);
            Vector_ = NULL;
          }
          else
            Vector_[0] = 0;          // subarray size = 0, (deleted)
        }

        if (Shared_)
          *Shared_ = StatusOfVector(Dim_, x, Off_, pos, Vect1_, Contain_);

        if (Size_)
          *Size_ = CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);

        if (Max_)
          *Max_ = Contain_[0];
      }      
      break;
    
    case 7:
      if (Vect2_[pos] || Replacement_)
      {      
        if (Replacement_)
        {
          inc = !Vector_[0];
          if (inc)                    // increment size
            if (x == Dim_ - 1 && x == 7)
            {
              IncCountOfVector(Dim_, 0, Off_, Vect1_, Contain_, Contain2_);
            }
            else
            {
              IncCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);
            }
          
          ::DeleteArray(Vect1_);
          Vect1_ = Vect2_[pos] = (int*)Replacement_;
          if (x == Dim_ - 1 && x == 7)
          {
            SetStatusOfVector(Dim_, 0, Off_, pos, 2, Contain_);
          }
          else
          {
            SetStatusOfVector(Dim_, x, Off_, pos, 2, Contain_);
          }
          Vector_ = Vect1_;

          if (inc)
            Vector_[0] = LenVect_[lenpos];  // array[n]...[0] = Length[n]
        }
        else
        {
          // Setting vector size, decrement size
          if (x == Dim_ - 1 && x == 7)
          {
            if (CountOfVector(Dim_, 0, Off_, Vect1_, Contain_, Contain2_) > 0)
              DecCountOfVector(Dim_, 0, Off_, Vect1_, Contain_, Contain2_);
          }
          else
          {
            if (CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_) > 0)
              DecCountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);
          }

          if (DeAlloc_)
          {
            ::DeleteArray(Vect1_);
            Vect1_ = Vect2_[pos] = NULL;
            if (x == Dim_ - 1 && x == 7)
            {
              SetStatusOfVector(Dim_, 0, Off_, pos, 1, Contain_);
            }
            else
            {
              SetStatusOfVector(Dim_, x, Off_, pos, 1, Contain_);
            }
            Vector_ = NULL;
          }
          else
            Vector_[0] = 0;          // subarray size = 0, (deleted)
        }

        if (Shared_)
          if (x == Dim_ - 1 && x == 7)
          {
            *Shared_ = StatusOfVector(Dim_, 0, Off_, pos, Vect1_, Contain_);
          }
          else
          {
            *Shared_ = StatusOfVector(Dim_, x, Off_, pos, Vect1_, Contain_);
          }

        if (Size_)
          if (x == Dim_ - 1 && x == 7)
          {
            *Size_ = CountOfVector(Dim_, 0, Off_, Vect1_, Contain_, Contain2_);
          }
          else
          {
            *Size_ = CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_);
          }

        if (Max_)
          *Max_ = Contain_[0];
      }
  }

  ::DeleteArray(ParentRefs_);
  return Container_;
}

/****************************************************************************/
// PURPOSE:
//   Helper method to delete the specified element index position in the
//   data matrix and set the element index position to NULL.
//
// array[1...n]
//      [i=2...n][0][0][0][0] = 3rdArr array size=x index=i=2 of
//                              2nd...nth 2ndArr element. 0 = deleted.
//
// array[1...n]
//      [i=2...n][0][0][0][(i+1)=3...x+i+1] = Deallocated array flag = 0/1.
//                                          = Shallow array copy flag = 0/2.
//
// array[1...n]
//      [i=2...n][1][0][0][(i+1)=3...x+i+1] = 3rdArr to TopPar pointer
//      reference number, referencing TopPar pointer in 1st...nth TopArr
//      element for 3rdArr arrays from 3...x.
//
// array[1...n]
//      [i=2...n][i=2][0][0][(i+1)=3...x+i+1] = 3rdArr to 2ndArr pointer
//      reference number, referencing 2ndArr pointer in 2nd...nth 2ndArr
//      element for 3rdArr arrays from 3...x.
//
// array[1...n]
//      [i=2...n][(i+1)=3...x+i+1]* = 1st starting 3rdArr value element to
//                                    ending 3rdArr value element x.
//
// PRE:
//   void* Array   : Data matrix
//   int Dim_      : The dimensions of the matrix, value range = [2...8]
//   int Index_    : Index level into position vector, denoted by constant
//                   i == Index_, max = Index_ + 1, lowest possible value = 1
//   int* LenVect_ : Length vector containing length of constituent arrays
//   int* PosVect_ : Position vector containing coordinates into matrix
//
//   void* Replacement_ : Replacement vector for deleted existing vector
//                        if not set as NULL, otherwise deleted vector in
//                        data matrix is set as NULL in the array entry.
//                        Shared_ set to TRUE if replacement vector present,
//                        otherwise Shared_ is set to FALSE.
//   bool Replace_      : If set to true then if Replacement_ is NULL a new
//                        vector is created and then that is used as the
//                        replacement vector, otherwise if false then if
//                        Replacement_ is NULL the element position in the
//                        data matrix is deleted and set to NULL.
//   bool DeAlloc_      : Should actually perform memory deallocation in
//                        data matrix element position instead of setting
//                        vector size to zero.
//   bool& Error_       : Error in array indexing (out of range)
//   bool CheckDealloc_ : Do not perform deletion or replacement, but
//                        retrieve size and deallocation/shared information
//                        of the specified vector from the data matrix.
//   int* Shared_       : Set to the deallocation status flag of the
//                        container vector of the specified vector to be
//                        deleted. Valid values include: 0=filled,
//                        1=deallocated, 2=shared
//   int* Size_         : Set to the size in elements of the container vector.
//   int* Max_          : Set to the maximum length of the container vector.
//
// POST:
//   Returns data matrix with specified constituent vector deleted and/or
//   replaced and has containing array entry set to NULL or has containing
//   array entry shared with replacement vector. If CheckDealloc is set TRUE
//   then no deletion or replacement is actually performed, but size and
//   deallocation/shared information of the specified vector from the data
//   matrix is retrieved.
//
int DeleteVectorHelper(void* Array_, int Dim_, int Index_, int* LenVect_,
                       int* PosVect_, void* Replacement_, bool Replace_,
                       bool DeAlloc_, bool& Error_, bool CheckDealloc_,
                       int* Shared_, int* Size_, int* Max_)
{
  if (!LenVect_ || !PosVect_)
    return 0;

  int******** Vect8_;
  int******* Vect7_;
  int****** Vect6_;
  int***** Vect5_;
  int**** Vect4_;
  int*** Vect3_;
  int** Vect2_;
  int* Vect1_;
  int** Contain2_;
  int* Contain_;

  int off = 8 - Dim_; // Maximum dimensions is: 8
  int pos;  
  int inc;
  int ret;
  int sz;
  int x;  

  int xIndex_;
  int xDim_;
  int xOff_;
  
  int TempSz_;
  int TempSh_;
  int TempMx_;
  
  bool HaveSize_ = Size_ != NULL;
  bool HaveShared_ = Shared_ != NULL;
  bool HaveMax_ = Max_ != NULL;

  if (!HaveSize_)
    Size_ = &TempSz_;

  if (!HaveShared_)
    Shared_ = &TempSh_;

  if (!HaveMax_)
    Max_ = &TempMx_;

  // Contain_[pos] = 1;    // deleted
  // Contain_[pos] = 2;    // shared
  // SizeOfVector(Dim_, x, LenVect_)
  // CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_)
  // StatusOfVector(Dim_, x, Off_, y, Vect1_, Contain_)
  // SetStatusOfVector(Dim_, x, Off_, y, stat, Contain_)  

  if (2 <= Dim_ && Dim_ <= 8 &&
      0 <= Index_ && Index_ < Dim_)
  {
    if (Index_ != 0)
    {
      if (Dim_ == 2)
      {
        if (InRange(1, PosVect_[1], SizeOfVector(Dim_, 0, LenVect_) - 1))
          pos = PosVect_[1];
        else
          pos = 0;

        if (pos)
        {
          Vect2_ = (int**)Array_;

          if (CheckDealloc_ && Vect2_ && Vect2_[0])
          {
            if (Shared_)
              *Shared_ = Vect2_[0][pos];

            if (Max_)
              *Max_ = Vect2_[0][0];

            if (Size_)
              *Size_ = Vect2_[0][0];

            ret = ((Shared_ && Size_) ? (*Size_ > 0 && *Shared_ != 1):
                   Size_ ? *Size_ > 0:
                   Shared_ ? *Shared_ != 1:0);
            return (ret && Vect2_[pos]);
          }
          else if ((Vect2_ && Vect2_[pos]) || Replacement_ || Replace_)
          {
            Vect1_ = Vect2_[pos];

            if (Replacement_ || Replace_)
            {
              inc = !Vect1_ || !Vect1_[0];
              
              ::DeleteArray(Vect1_);
              if (Replace_ && !Replacement_)
              {
                sz = SizeOfVector(Dim_, 1, LenVect_);
                Vect1_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
                memset(Vect1_, 0, sizeof(int) * sz);
              }

              Vect1_ = (int*)Replacement_;

              if (Vect2_)
              {
                Vect2_[pos] = Vect1_;
                Vect2_[0][pos] = 2;    // shared
              }

              if (inc && Vect1_)
                Vect1_[0] = LenVect_[1];  // array[n]...[0] = Length[n]
            }
            else
            {
              if (DeAlloc_)
              {
                ::DeleteArray(Vect1_);
                Vect1_ = NULL;

                if (Vect2_)
                {
                  Vect2_[pos] = Vect1_;
                  Vect2_[0][pos] = 1;  // deleted
                }
              }
              else if (Vect1_)
                Vect1_[0] = 0;            // subarray size = 0, (deleted)
            }

            if (Vect2_)
            {
              if (Shared_)
                *Shared_ = Vect2_[0][pos];

              if (Max_)
                *Max_ = Vect2_[0][0];

              if (Size_)
                *Size_ = Vect2_[0][0];
            }
          }

          Error_ = false;
        }
        else
        {
          Error_ = true;
          return NULL;
        }
      }
      else
      {
        Vect8_ = (off == 0) ? (int********)Array_:NULL;
        Vect7_ = (off == 1) ? (int*******)Array_:NULL;
        Vect6_ = (off == 2) ? (int******)Array_:NULL;
        Vect5_ = (off == 3) ? (int*****)Array_:NULL;
        Vect4_ = (off == 4) ? (int****)Array_:NULL;
        Vect3_ = (off == 5) ? (int***)Array_:NULL;
        Vect2_ = (off == 6) ? (int**)Array_:NULL;

        xDim_ = Dim_;
        Dim_ += off;
        xIndex_ = Index_;
        Index_ += off;

        for (x = 1; x < xDim_; x++)
        {
          xOff_ = x + off;          

          if ((Error_ = (x <= xIndex_)) &&
              InRange(x, PosVect_[x], SizeOfVector(xDim_, x-1, LenVect_) - 1))
            pos = PosVect_[x];
          else
            pos = 0;

          if (!pos && Error_)
            return NULL;
          else
            Error_ = false;
                        
          switch (xOff_)
          {
            case 1:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (CheckDealloc_ && Vect8_ && Vect8_[0])
                  {
                    Contain2_ = Vect8_[0][0][0][0][0][0];
                    Vect1_ = Contain_ = Contain2_[0];
                    
                    if (Shared_)
                      *Shared_ = StatusOfVector(Dim_, Index_, off, pos, Vect1_, Contain_);
                      
                    if (Size_)
                      *Size_ = CountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);

                    if (Max_)
                      *Max_ = Contain_[0];

                    ret = ((Shared_ && Size_) ? (*Size_ > 0 && *Shared_ != 1):
                           Size_ ? *Size_ > 0:
                           Shared_ ? *Shared_ != 1:0);
                    return (ret && Vect8_[pos]);
                  }
                  else
                  {
                    if (Replace_ && !Replacement_)
                    {
                      sz = SizeOfVector(xDim_, xIndex_, LenVect_);
                      Vect7_ = (int*******)RawCallocateWith(MEMMATRIX, sizeof(int******) * sz);
                      Replacement_ = InitMatrix(Vect7_, Dim_, off, Index_, LenVect_);
                      Replacement_ = SetRefArrays(Vect7_, Dim_, off, Index_, LenVect_, PosVect_, Error_);
                    }
                    
                    if (Vect8_)
                      if (Vect8_[pos])
                      {
                        Vect7_ = Vect8_[pos];
                        RemoveReferences(Vect8_, Dim_, off, Index_, LenVect_,
                                         PosVect_, Replacement_, DeAlloc_,
                                         Shared_, Size_, Max_, Error_);
                      }
                      else if (Replacement_)
                      {
                        Vect8_[pos] = Vect7_ = (int*******)Replacement_;
                        Contain2_ = Vect8_[0][0][0][0][0][0];
                        Vect1_ = Contain_ = Contain2_[0];
                        IncCountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);
                      }
                  }
                }                
                else
                {
                  if (Vect8_ && Vect8_[pos])
                    Vect7_ = Vect8_[pos];
                  else
                    return 0;
                }
              }
              break;

            case 2:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (CheckDealloc_ && Vect7_ && Vect7_[0])
                  {
                    Contain2_ = Vect7_[0][0][0][0][0];
                    Vect1_ = Contain_ = Contain2_[0];
                    
                    if (Shared_)
                      *Shared_ = StatusOfVector(Dim_, Index_, off, pos, Vect1_, Contain_);

                    if (Size_)
                      *Size_ = CountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);

                    if (Max_)
                      *Max_ = Contain_[0];

                    ret = ((Shared_ && Size_) ? (*Size_ > 0 && *Shared_ != 1):
                           Size_ ? *Size_ > 0:
                           Shared_ ? *Shared_ != 1:0);
                    return (ret && Vect7_[pos]);
                  }
                  else
                  {
                    if (Replace_ && !Replacement_)
                    {
                      sz = SizeOfVector(xDim_, xIndex_, LenVect_);
                      Vect6_ = (int******)RawCallocateWith(MEMMATRIX, sizeof(int*****) * sz);
                      Replacement_ = InitMatrix(Vect6_, Dim_, off, Index_, LenVect_);
                      Replacement_ = SetRefArrays(Vect6_, Dim_, off, Index_, LenVect_, PosVect_, Error_);
                    }

                    if (Vect7_)
                      if (Vect7_[pos])
                      {
                        Vect6_ = Vect7_[pos];
                        RemoveReferences(Vect7_, Dim_, off, Index_, LenVect_,
                                         PosVect_, Replacement_, DeAlloc_,
                                         Shared_, Size_, Max_, Error_);
                      }
                      else if (Replacement_)
                      {
                        Vect7_[pos] = Vect6_ = (int******)Replacement_;
                        Contain2_ = Vect7_[0][0][0][0][0];
                        Vect1_ = Contain_ = Contain2_[0];
                        IncCountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);
                      }
                      
                  }
                }
                else
                {
                  if (Vect7_ && Vect7_[pos])
                    Vect6_ = Vect7_[pos];
                  else
                    return 0;
                }
              }
              else if (x != 1)
              {
                if (Vect7_ && Vect7_[0])
                  Vect6_ = Vect7_[0];
                else
                  return 0;
              }              
              break;
              
            case 3:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (CheckDealloc_ && Vect6_ && Vect6_[0])
                  {
                    Contain2_ = Vect6_[0][0][0][0];
                    Vect1_ = Contain_ = Contain2_[0];

                    if (Shared_)
                      *Shared_ = StatusOfVector(Dim_, Index_, off, pos, Vect1_, Contain_);
                      
                    if (Size_)
                      *Size_ = CountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);

                    if (Max_)
                      *Max_ = Contain_[0];

                    ret = ((Shared_ && Size_) ? (*Size_ > 0 && *Shared_ != 1):
                           Size_ ? *Size_ > 0:
                           Shared_ ? *Shared_ != 1:0);
                    return (ret && Vect6_[pos]);
                  }
                  else
                  {
                    if (Replace_ && !Replacement_)
                    {
                      sz = SizeOfVector(xDim_, xIndex_, LenVect_);
                      Vect5_ = (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****) * sz);
                      Replacement_ = InitMatrix(Vect5_, Dim_, off, Index_, LenVect_);
                      Replacement_ = SetRefArrays(Vect5_, Dim_, off, Index_, LenVect_, PosVect_, Error_);
                    }

                    if (Vect6_)
                      if (Vect6_[pos])
                      {
                        Vect5_ = Vect6_[pos];
                        RemoveReferences(Vect6_, Dim_, off, Index_, LenVect_,
                                         PosVect_, Replacement_, DeAlloc_,
                                         Shared_, Size_, Max_, Error_);
                      }
                      else if (Replacement_)
                      {
                        Vect6_[pos] = Vect5_ = (int*****)Replacement_;
                        Contain2_ = Vect6_[0][0][0][0];
                        Vect1_ = Contain_ = Contain2_[0];
                        IncCountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);
                      }
                  }
                }
                else
                {
                  if (Vect6_ && Vect6_[pos])
                    Vect5_ = Vect6_[pos];
                  else
                    return 0;
                }
              }
              else if (x != 1)
              {
                if (Vect6_ && Vect6_[0])
                  Vect5_ = Vect6_[0];
                else
                  return 0;
              }              
              break;
            
            case 4:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (CheckDealloc_ && Vect5_ && Vect5_[0])
                  {
                    Contain2_ = Vect5_[0][0][0];
                    Vect1_ = Contain_ = Contain2_[0];

                    if (Shared_)
                      *Shared_ = StatusOfVector(Dim_, Index_, off, pos, Vect1_, Contain_);
                      
                    if (Size_)
                      *Size_ = CountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);

                    if (Max_)
                      *Max_ = Contain_[0];

                    ret = ((Shared_ && Size_) ? (*Size_ > 0 && *Shared_ != 1):
                           Size_ ? *Size_ > 0:
                           Shared_ ? *Shared_ != 1:0);
                    return (ret && Vect5_[pos]);
                  }
                  else
                  {
                    if (Replace_ && !Replacement_)
                    {
                      sz = SizeOfVector(xDim_, xIndex_, LenVect_);
                      Vect4_ = (int****)RawCallocateWith(MEMMATRIX, sizeof(int***) * sz);
                      Replacement_ = InitMatrix(Vect4_, Dim_, off, Index_, LenVect_);
                      Replacement_ = SetRefArrays(Vect4_, Dim_, off, Index_, LenVect_, PosVect_, Error_);
                    }

                    if (Vect5_)
                      if (Vect5_[pos])
                      {
                        Vect4_ = Vect5_[pos];
                        RemoveReferences(Vect5_, Dim_, off, Index_, LenVect_,
                                         PosVect_, Replacement_, DeAlloc_,
                                         Shared_, Size_, Max_, Error_);
                      }
                      else if (Replacement_)
                      {
                        Vect5_[pos] = Vect4_ = (int****)Replacement_;
                        Contain2_ = Vect5_[0][0][0];
                        Vect1_ = Contain_ = Contain2_[0];
                        IncCountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);
                      }
                  }
                }
                else
                {
                  if (Vect5_ && Vect5_[pos])
                    Vect4_ = Vect5_[pos];
                  else
                    return 0;
                }
              }
              else if (x != 1)
              {
                if (Vect5_ && Vect5_[0])
                  Vect4_ = Vect5_[0];
                else
                  return 0;
              }
              break;
            
            case 5:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (CheckDealloc_ && Vect4_ && Vect4_[0])
                  {
                    Contain2_ = Vect4_[0][0];
                    Vect1_ = Contain_ = Contain2_[0];

                    if (Shared_)
                      *Shared_ = StatusOfVector(Dim_, Index_, off, pos, Vect1_, Contain_);
                      
                    if (Size_)
                      *Size_ = CountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);

                    if (Max_)
                      *Max_ = Contain_[0];

                    ret = ((Shared_ && Size_) ? (*Size_ > 0 && *Shared_ != 1):
                           Size_ ? *Size_ > 0:
                           Shared_ ? *Shared_ != 1:0);
                    return (ret && Vect4_[pos]);
                  }
                  else
                  {
                    if (Replace_ && !Replacement_)
                    {
                      sz = SizeOfVector(xDim_, xIndex_, LenVect_);
                      Vect3_ = (int***)RawCallocateWith(MEMMATRIX, sizeof(int**) * sz);
                      Replacement_ = InitMatrix(Vect3_, Dim_, off, Index_, LenVect_);
                      Replacement_ = SetRefArrays(Vect3_, Dim_, off, Index_, LenVect_, PosVect_, Error_);
                    }

                    if (Vect4_)
                      if (Vect4_[pos])
                      {
                        Vect3_ = Vect4_[pos];
                        RemoveReferences(Vect4_, Dim_, off, Index_, LenVect_,
                                         PosVect_, Replacement_, DeAlloc_,
                                         Shared_, Size_, Max_, Error_);
                      }
                      else if (Replacement_)
                      {
                        Vect4_[pos] = Vect3_ = (int***)Replacement_;
                        Contain2_ = Vect4_[0][0];
                        Vect1_ = Contain_ = Contain2_[0];
                        IncCountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);
                      }
                  }
                }
                else
                {
                  if (Vect4_ && Vect4_[pos])
                    Vect3_ = Vect4_[pos];
                  else
                    return 0;
                }
              }
              else if (x != 1)
              {
                if (Vect4_ && Vect4_[0])
                  Vect3_ = Vect4_[0];
                else
                  return 0;
              }
              break;

            case 6:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (CheckDealloc_ && Vect3_ && Vect3_[0])
                  {
                    Contain2_ = Vect3_[0];
                    Vect1_ = Contain_ = Contain2_[0];

                    if (Shared_)
                      *Shared_ = StatusOfVector(Dim_, Index_, off, pos, Vect1_, Contain_);
                      
                    if (Size_)
                      *Size_ = CountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);

                    if (Max_)
                      *Max_ = Contain_[0];

                    ret = ((Shared_ && Size_) ? (*Size_ > 0 && *Shared_ != 1):
                           Size_ ? *Size_ > 0:
                           Shared_ ? *Shared_ != 1:0);
                    return (ret && Vect3_[pos]);
                  }
                  else
                  {
                    if (Replace_ && !Replacement_)
                    {
                      sz = SizeOfVector(xDim_, xIndex_, LenVect_);
                      Vect2_ = (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * sz);
                      Replacement_ = InitMatrix(Vect2_, Dim_, off, Index_, LenVect_);
                      Replacement_ = SetRefArrays(Vect2_, Dim_, off, Index_, LenVect_, PosVect_, Error_);
                    }

                    if (Vect3_)
                      if (Vect3_[pos])
                      {
                        Vect2_ = Vect3_[pos];
                        RemoveReferences(Vect3_, Dim_, off, Index_, LenVect_,
                                         PosVect_, Replacement_, DeAlloc_,
                                         Shared_, Size_, Max_, Error_);
                      }
                      else if (Replacement_)
                      {
                        Vect3_[pos] = Vect2_ = (int**)Replacement_;
                        Contain2_ = Vect3_[0];
                        Vect1_ = Contain_ = Contain2_[0];
                        IncCountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);
                      }
                  }
                }
                else
                {
                  if (Vect3_ && Vect3_[pos])
                    Vect2_ = Vect3_[pos];
                  else
                    return 0;
                }
              }
              else if (x != 1)              
              {
                if (Vect3_ && Vect3_[0])
                  Vect2_ = Vect3_[0];                  
                else
                  return 0;
              }
              break;

            case 7:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (CheckDealloc_ && Vect2_ && Vect2_[0])
                  {
                    Contain2_ = Vect2_;
                    Vect1_ = Contain_ = Vect2_[0];

                    if (Index_ == Dim_ - 1 && Index_ == 7)
                    {
                      if (Shared_)
                        *Shared_ = StatusOfVector(Dim_, 0, off, pos, Vect1_, Contain_);
                    }
                    else
                    {
                      if (Shared_)
                        *Shared_ = StatusOfVector(Dim_, Index_, off, pos, Vect1_, Contain_);
                    }

                    if (Index_ == Dim_ - 1 && Index_ == 7)
                    {
                      if (Size_)
                        *Size_ = CountOfVector(Dim_, 0, off, Vect1_, Contain_, Contain2_);
                    }
                    else
                    {
                      if (Size_)
                        *Size_ = CountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);
                    }

                    if (Max_)
                      *Max_ = Contain_[0];

                    ret = ((Shared_ && Size_) ? (*Size_ > 0 && *Shared_ != 1):
                           Size_ ? *Size_ > 0:
                           Shared_ ? *Shared_ != 1:0);
                    return (ret && Vect2_[pos]);
                  }
                  else
                  {
                    if (Replace_ && !Replacement_)
                    {
                      sz = SizeOfVector(xDim_, xIndex_, LenVect_);
                      Vect1_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
                      Replacement_ = InitMatrix(Vect1_, Dim_, off, Index_, LenVect_);
                    }

                    if (Vect2_)
                      if (Vect2_[pos])
                      {
                        Vect1_ = Vect2_[pos];
                        RemoveReferences(Vect2_, Dim_, off, Index_, LenVect_,
                                         PosVect_, Replacement_, DeAlloc_,
                                         Shared_, Size_, Max_, Error_);
                      }
                      else if (Replacement_)
                      {
                        Vect2_[pos] = Vect1_ = (int*)Replacement_;
                        Contain2_ = Vect2_;
                        Vect1_ = Contain_ = Vect2_[0];

                        if (Index_ == Dim_ - 1 && Index_ == 7)
                        {
                          IncCountOfVector(Dim_, 0, off, Vect1_, Contain_, Contain2_);
                        }
                        else
                        {
                          IncCountOfVector(Dim_, Index_, off, Vect1_, Contain_, Contain2_);
                        }
                      }
                  }
                }
                else
                {
                  if (Vect2_ && Vect2_[pos])
                    Vect1_ = Vect2_[pos];
                  else
                    return 0;
                }
              }
              else if (x != 1)
              {
                if (Vect2_ && Vect2_[0])
                  Vect1_ = Vect2_[0];
                else
                  return 0;
              }
          }
        }
      }
    }
  }
  else if (Dim_ == 1)
  {
    // In calling InitArrayData function for 1D vector,
    // Index argument is assumed to hold the vector length.
    // LenVect argument is assumed to hold deallocated flag and
    // PosVect argument is assumed to hold shallowcopy flag.

    if (CheckDealloc_ && Vect1_)
    {
      if (Shared_)
      {
        *Shared_ = 0;
        
        if (LenVect_ && *LenVect_ == 1)
          *Shared_ = 1;
          
        if (PosVect_ && *PosVect_ == 1)
          *Shared_ = 2;
      }

      if (Max_ && Vect1_)
        *Max_ = Vect1_[0];

      if (Size_ && Vect1_)
        *Size_ = Vect1_[0];

      ret = ((Shared_ && Size_) ? (*Size_ > 0 && *Shared_ != 1):
             Size_ ? *Size_ > 0:
             Shared_ ? *Shared_ != 1:0);
      return (ret && Vect1_);
    }
    else
    {
      if (Replacement_ || Replace_)
      {
        ::DeleteArray(Vect1_);

        if (Replace_ && !Replacement_)
        {
          sz = Index_+1;
          Vect1_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
          memset(Vect1_, 0, sizeof(int) * sz);
        }
      
        Vect1_ = (int*)Replacement_;
        if (Vect1_)
          Vect1_[0] = Index_;
      
        if (PosVect_)
          *PosVect_ = 1;
      }
      else
      {
        if (DeAlloc_)
        {
          ::DeleteArray(Vect1_);
          Vect1_ = NULL;
        
          if (LenVect_)
            *LenVect_ = 1;
        }
        else if (Vect1_)
          Vect1_[0] = 0;
      }

      if (Shared_)
      {
        *Shared_ = 0;
        
        if (LenVect_ && *LenVect_ == 1)
          *Shared_ = 1;
          
        if (PosVect_ && *PosVect_ == 1)
          *Shared_ = 2;
      }

      if (Max_ && Vect1_)
        *Max_ = Vect1_[0];

      if (Size_ && Vect1_)
        *Size_ = Vect1_[0];

      Error_ = false;
    }
  }

  return 1;
}

/****************************************************************************/
// PURPOSE:
//   Method to retrieve the deallocated and shared status flags of the
//   specified vector contained within the data matrix and return TRUE if
//   specified vector is deleted (not set to zero size) from the matrix.
//
// PRE:
//   void* Array_  : the generalized multidimensional data matrix
//   int Dim_      : The dimensions of the matrix, value range = [1...8]
//   int Index_    : Index accessing constituent array of the data matrix,
//                   value range = [1...7], lowest possible value = 1
//   int* LenVect_ : Length vector containing length of constituent arrays
//   int* PosVect_ : Position vector containing coordinates into matrix
//   bool& Error_  : Error in array indexing (out of range)
//   int* Shared_  : Set to the deallocation status flag of the container
//                   vector of the specified vector to be deleted.
//                   Valid values include: 0=filled, 1=deallocated, 2=shared
//   int* Size_    : Set to the size in elements of the container vector.
//   int* Max_     : Set to the maximum length of the container vector.
//
int IsDeallocated(void* Array_, int Dim_, int Index_, int* LenVect_,
                  int* PosVect_, bool& Error_, int* Shared_, int* Size_,
                  int* Max_)
{
  int Shval_;  
  int Szval_;
  int Mxval_;

  if (!Shared_)
    Shared_ = &Shval_;

  if (!Size_)
    Size_ = &Szval_;

  if (!Max_)
    Max_ = &Mxval_;

  int ret =
  DeleteVectorHelper(Array_, Dim_, Index_, LenVect_, PosVect_,
                     NULL, false, false, Error_, true, Shared_, Size_, Max_);

  return (!ret);
}

/****************************************************************************/
// PURPOSE:
//   Method to delete the specified element index position in the data matrix
//   and set the element index position to NULL.
//
// PRE:
//   void* Array_  : the generalized multidimensional data matrix
//   int Dim_      : The dimensions of the matrix, value range = [1...8]
//   int Index_    : Index accessing constituent array of the data matrix,
//                   value range = [1...7], lowest possible value = 1
//   int* LenVect_ : Length vector containing length of constituent arrays
//   int* PosVect_ : Position vector containing coordinates into matrix
//   bool& Error_  : Error in array indexing (out of range)
//   int* Shared_  : Set to the deallocation status flag of the container
//                   vector of the specified vector to be deleted.
//                   Valid values include: 0=filled, 1=deallocated, 2=shared
//   int* Size_    : Set to the size in elements of the container vector.
//   int* Max_     : Set to the maximum length of the container vector.
//
int DeleteVector(void* Array_, int Dim_, int Index_, int* LenVect_,
                 int* PosVect_, bool& Error_, int* Shared_, int* Size_,
                 int* Max_)
{
  int ret =
  DeleteVectorHelper(Array_, Dim_, Index_, LenVect_, PosVect_,
                     NULL, false, true, Error_, false, Shared_, Size_, Max_);
  return ret;
}

/****************************************************************************/
// PURPOSE:
//   Method to replace a vector within the data matrix with another vector
//   If the element index within the data matrix doesn't contain a vector
//   then an option to create a new vector and replace it with the newly
//   created vector is available.
//
// PRE:
//   void* Array_  : the generalized multidimensional data matrix
//   int Dim_      : The dimensions of the matrix, value range = [1...8]
//   int Index_    : Index accessing constituent array of the data matrix,
//                   value range = [1...7], lowest possible value = 1
//   int* LenVect_ : Length vector containing length of constituent arrays
//   int* PosVect_ : Position vector containing coordinates into matrix
//
//   void* Replacement_ : The replacement vector to replace the existing
//                        vector in the data matrix with or if the element
//                        position is empty then to set as the new vector.
//   bool Replace_      : If set to true then if Replacement_ is NULL a new
//                        vector is created and then that is used as the
//                        replacement vector, otherwise if false then if
//                        Replacement_ is NULL the element position in the
//                        data matrix is deleted and set to NULL.
//   bool DeAlloc_      : Should actually perform memory deallocation in
//                        data matrix element position instead of setting
//                        vector size to zero.
//   bool& Error_  : Error in array indexing (out of range)
//   int* Shared_  : Set to the deallocation status flag of the container
//                   vector of the specified vector to be deleted.
//                   Valid values include: 0=filled, 1=deallocated, 2=shared
//   int* Size_    : Set to the size in elements of the container vector.
//   int* Max_     : Set to the maximum length of the container vector.
//
int ReplaceVector(void* Array_, int Dim_, int Index_, int* LenVect_,
                  int* PosVect_, void* Replacement_, bool Replace_,
                  bool DeAlloc_, bool& Error_, int* Shared_, int* Size_,
                  int* Max_)
{
  int ret =
  DeleteVectorHelper(Array_, Dim_, Index_, LenVect_, PosVect_,
                     Replacement_, Replace_, DeAlloc_, Error_,
                     false, Shared_, Size_, Max_);
  return ret;
}

/****************************************************************************/
// PURPOSE:
//   Method to return the real vector size within the data matrix given the
//   index into the matrix.
//
// PRE:
//   void* Array_  : the generalized multidimensional data matrix
//   int Dim_      : The dimensions of the matrix, value range = [1...8]
//   int Index_    : Index accessing constituent array of the data matrix,
//                   value range = [0...7]
//   int* LenVect_ : Length vector containing length of constituent arrays
//   int* PosVect_ : Position vector containing coordinates into matrix
//   bool& Error_  : Error in array indexing (out of range)
//
// POST:
//   returns the actual size of the indexed vector in number of elements.
//   The elements of the vector could also be pointer arrays.
//
int RealVectorCount(void* Array_, int Dim_, int Index_,
                    int* LenVect_, int* PosVect_, bool& Error_)
{
  // null pointer checks
  if (!Array_ || !LenVect_ || !PosVect_)
    return NULL;

  int off = 8 - Dim_; // Maximum dimensions is: 8
  int pos;  
  int sz;
  int x;

  int xIndex_;
  int xOff_;
  int xMax_;
  int max;

  xIndex_ = Index_;
  Index_ += off;

  int******** Vect8_ = (off == 0) ? (int********)Array_:NULL;
  int******* Vect7_ = (off == 1) ? (int*******)Array_:NULL;
  int****** Vect6_ = (off == 2) ? (int******)Array_:NULL;
  int***** Vect5_ = (off == 3) ? (int*****)Array_:NULL;
  int**** Vect4_ = (off == 4) ? (int****)Array_:NULL;
  int*** Vect3_ = (off == 5) ? (int***)Array_:NULL;
  int** Vect2_ = (off == 6) ? (int**)Array_:NULL;
  int* Vect1_ = (off == 7) ? (int*)Array_:NULL;
  int* Contain_;

  if (xIndex_ == 0)
  {
    Vect7_ = (off == 0) ? Vect8_[0]:Vect7_;
    Vect6_ = (off <= 1) ? Vect7_[0]:Vect6_;
    Vect5_ = (off <= 2) ? Vect6_[0]:Vect5_;
    Vect4_ = (off <= 3) ? Vect5_[0]:Vect4_;
    Vect3_ = (off <= 4) ? Vect4_[0]:Vect3_;
    Vect2_ = (off <= 5) ? Vect3_[0]:Vect2_;
    Vect1_ = (off <= 6) ? Vect2_[0]:Vect1_;
    sz = (Dim_ >= 3) ? Vect2_[1][0]:
         (Dim_ == 2) ? Vect2_[0][0]:
                       Vect1_[0];
      
    Error_ = false;
    return sz;
  }
  else  
  {
    max = Index_;
    xMax_ = xIndex_;
    
    for (x = 1+off; x <= max; x++)
    {
      Error_ = true;
      xOff_ = x - off;
    
      if (xOff_ <= xMax_ &&
          InRange(xOff_, PosVect_[xOff_], SizeOfVector(Dim_, xOff_-1, LenVect_) - 1))
        pos = PosVect_[xOff_];
      else
        pos = 0;

      if (!pos)
        return 0;
      else
        Error_ = false;
  
      switch (x)
      {
        case 1:
          if (x == max)
          {
            if (Vect8_[pos])
            {
              Vect7_ = Vect8_[pos];
              Contain_ = Vect7_[0][0][0][0][0][0];
              sz = Contain_[1];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect8_[pos])
              Vect7_ = Vect8_[pos];
          }
          break;
          
        case 2:
          if (x == max)
          {
            if (Vect7_[pos])
            {
              Vect6_ = Vect7_[pos];
              Contain_ = Vect6_[0][0][0][0][0];
              sz = Contain_[1];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect7_[pos])
              Vect6_ = Vect7_[pos];
          }
          break;
      
        case 3:
          if (x == max)
          {
            if (Vect6_[pos])
            {
              Vect5_ = Vect6_[pos];
              Contain_ = Vect5_[0][0][0][0];
              sz = Contain_[1];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect6_[pos])
              Vect5_ = Vect6_[pos];
          }
          break;
        
        case 4:
          if (x == max)
          {
            if (Vect5_[pos])
            {
              Vect4_ = Vect5_[pos];
              Contain_ = Vect4_[0][0][0];
              sz = Contain_[1];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect5_[pos])
              Vect4_ = Vect5_[pos];
          }
          break;

        case 5:
          if (x == max)
          {
            if (Vect4_[pos])
            {            
              Vect3_ = Vect4_[pos];
              Contain_ = Vect3_[0][0];
              sz = Contain_[1];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect4_[pos])
              Vect3_ = Vect4_[pos];
          }
          break;

        case 6:
          if (x == max)
          {
            if (Vect3_[pos])
            {
              Vect2_ = Vect3_[pos];
              Contain_ = Vect2_[0];
              sz = Contain_[1];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect3_[pos])
              Vect2_ = Vect3_[pos];
          }
          break;

        case 7:
          if (x == max)
          {
            if (Vect2_[pos])
            {
              Vect1_ = Vect2_[pos];
              sz = Vect1_[0];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect2_[pos])
            {
              Vect1_ = Vect2_[pos];
              sz = Vect1_[0];
              Error_ = false;
              return sz;
            }
          }
          break;
      }
    }
  }

  return 0;
}

/****************************************************************************/
// PURPOSE:
//   Method to return the real vector maximum length within the data matrix
//   given the index into the matrix.
//
// PRE:
//   void* Array_  : the generalized multidimensional data matrix
//   int Dim_      : The dimensions of the matrix, value range = [1...8]
//   int Index_    : Index accessing constituent array of the data matrix,
//                   value range = [0...7]
//   int* LenVect_ : Length vector containing length of constituent arrays
//   int* PosVect_ : Position vector containing coordinates into matrix
//   bool& Error_  : Error in array indexing (out of range)
//
// POST:
//   returns the actual size of the indexed vector in number of elements.
//   The elements of the vector could also be pointer arrays.
//
int RealVectorMax(void* Array_, int Dim_, int Index_,
                  int* LenVect_, int* PosVect_, bool& Error_)
{
  // null pointer checks
  if (!Array_ || !LenVect_ || !PosVect_)
    return NULL;

  int off = 8 - Dim_; // Maximum dimensions is: 8
  int pos;  
  int sz;
  int x;

  int xIndex_;
  int xOff_;
  int xMax_;
  int max;

  xIndex_ = Index_;
  Index_ += off;

  int******** Vect8_ = (off == 0) ? (int********)Array_:NULL;
  int******* Vect7_ = (off == 1) ? (int*******)Array_:NULL;
  int****** Vect6_ = (off == 2) ? (int******)Array_:NULL;
  int***** Vect5_ = (off == 3) ? (int*****)Array_:NULL;
  int**** Vect4_ = (off == 4) ? (int****)Array_:NULL;
  int*** Vect3_ = (off == 5) ? (int***)Array_:NULL;
  int** Vect2_ = (off == 6) ? (int**)Array_:NULL;
  int* Vect1_ = (off == 7) ? (int*)Array_:NULL;

  if (xIndex_ == 0)
  {
    Vect7_ = (off == 0) ? Vect8_[0]:Vect7_;
    Vect6_ = (off <= 1) ? Vect7_[0]:Vect6_;
    Vect5_ = (off <= 2) ? Vect6_[0]:Vect5_;
    Vect4_ = (off <= 3) ? Vect5_[0]:Vect4_;
    Vect3_ = (off <= 4) ? Vect4_[0]:Vect3_;
    Vect2_ = (off <= 5) ? Vect3_[0]:Vect2_;
    Vect1_ = (off <= 6) ? Vect2_[0]:Vect1_;
    
    sz = Vect1_[0];
    Error_ = false;
    return sz;
  }
  else
  {
    max = Index_;
    xMax_ = xIndex_;
  
    for (x = 1+off; x <= max; x++)
    {
      Error_ = true;
      xOff_ = x - off;
    
      if (xOff_ <= xMax_ &&
          InRange(xOff_, PosVect_[xOff_], SizeOfVector(Dim_, xOff_-1, LenVect_) - 1))
        pos = PosVect_[xOff_];
      else
        pos = 0;

      if (!pos)
        return 0;
      else
        Error_ = false;
  
      switch (x)
      {
        case 1:
          if (x == max)
          {
            if (Vect8_[pos])
            {
              Vect7_ = Vect8_[pos];
              sz = Vect7_[0][0][0][0][0][0][0];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect8_[pos])
              Vect7_ = Vect8_[pos];
          }
          break;
          
        case 2:
          if (x == max)
          {
            if (Vect7_[pos])
            {
              Vect6_ = Vect7_[pos];
              sz = Vect6_[0][0][0][0][0][0];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect7_[pos])
              Vect6_ = Vect7_[pos];
          }
          break;
      
        case 3:
          if (x == max)
          {
            if (Vect6_[pos])
            {
              Vect5_ = Vect6_[pos];
              sz = Vect5_[0][0][0][0][0];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect6_[pos])
              Vect5_ = Vect6_[pos];
          }
          break;
        
        case 4:
          if (x == max)
          {
            if (Vect5_[pos])
            {
              Vect4_ = Vect5_[pos];
              sz = Vect4_[0][0][0][0];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect5_[pos])
              Vect4_ = Vect5_[pos];
          }
          break;

        case 5:
          if (x == max)
          {
            if (Vect4_[pos])
            {
              Vect3_ = Vect4_[pos];
              sz = Vect3_[0][0][0];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect4_[pos])
              Vect3_ = Vect4_[pos];
          }
          break;

        case 6:
          if (x == max)
          {
            if (Vect3_[pos])
            {
              Vect2_ = Vect3_[pos];
              sz = Vect2_[0][0];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect3_[pos])
              Vect2_ = Vect3_[pos];
          }
          break;

        case 7:
          if (x == max)
          {
            if (Vect2_[pos])
            {
              Vect1_ = Vect2_[pos];
              sz = Vect1_[0];
              Error_ = false;
              return sz;
            }
          }
          else
          {
            if (Vect2_[pos])
            {
              Vect1_ = Vect2_[pos];
              sz = Vect1_[0];
              Error_ = false;
              return sz;
            }
          }
          break;
      }
    }
  }

  return 0;
}

/****************************************************************************/
// PURPOSE:
//   Method to retrieve the parent reference array given the index into the
//   general data matrix, valid only for matrix dim > 2.
//
// PRE:
//   void* Array_     : the generalized multidimensional data matrix
//   int Dim_         : The dimensions of the matrix, value range = [1...8]
//   int Index_       : Index accessing constituent array of the data matrix,
//                      value range = [0...7]
//   int* LenVect_    : Length vector containing length of constituent arrays
//   int* PosVect_    : Position vector containing coordinates into matrix
//   int* ParentRefs_ : Parent reference array of specified constituent vector
//   bool& Error_     : Error in array indexing (out of range)
//
// POST:
//   If Dim_ > 3 then returns the parent reference array as specified by the
//   Index into the general data matrix, otherwise returns NULL.
//
void* ParentReference(void* Array_, int Dim_, int Index_,
                      int* LenVect_, int* PosVect_, int* ParentRefs_,
                      bool& Error_)
{
  // null pointer checks
  if (!Array_ || !LenVect_ || !PosVect_ || !ParentRefs_)
    return NULL;

  if (Dim_ < 3)
    return NULL;

  int off = 8 - Dim_; // Maximum dimensions is: 8
  int pos;  
  int sz;
  int x;

  int xIndex_;
  int xOff_;
  int xDim_;
  int xMax_;
  int max;

  xIndex_ = Index_;
  Index_ += off;
  xDim_ = Dim_;
  Dim_ += off;
  
  int******** Vect8_ = (off == 0) ? (int********)Array_:NULL;;
  int******* Vect7_ = (off == 1) ? (int*******)Array_:NULL;;
  int****** Vect6_ = (off == 2) ? (int******)Array_:NULL;
  int***** Vect5_ = (off == 3) ? (int*****)Array_:NULL;
  int**** Vect4_ = (off == 4) ? (int****)Array_:NULL;
  int*** Vect3_ = (off == 5) ? (int***)Array_:NULL;
  int** Vect2_ = (off == 6) ? (int**)Array_:NULL;
  int* Vect1_ = (off == 7) ? (int*)Array_:NULL;

  if (xIndex_ != 0)
  {
    max = Index_;
    xMax_ = xIndex_;
  
    for (x = 1+off; x <= max; x++)
    {
      Error_ = true;
      xOff_ = x - off;  // real index position into matrix minus offsets
                        // x is index position with offsets included
    
      if (xOff_ <= xMax_ &&
          InRange(xOff_, PosVect_[xOff_], SizeOfVector(xDim_, xOff_-1, LenVect_) - 1))
        pos = PosVect_[xOff_];
      else
        pos = 0;

      if (!pos)
        return 0;
      else
        Error_ = false;

      switch (x)
      {
        case 1:
          if (x == max)
          {
            if (Vect8_[pos])
            {
              Vect7_ = Vect8_[pos];
              ParentRefs_ =
              GetParentReference(ParentRefs_, Dim_, off, Index_, LenVect_, PosVect_, Vect7_, Error_);
              return ParentRefs_;
            }
          }
          else
          {
            if (Vect8_[pos])
              Vect7_ = Vect8_[pos];
          }
          break;
          
        case 2:
          if (x == max)
          {
            if (Vect7_[pos])
            {
              Vect6_ = Vect7_[pos];
              ParentRefs_ =
              GetParentReference(ParentRefs_, Dim_, off, Index_, LenVect_, PosVect_, Vect6_, Error_);
              return ParentRefs_;
            }
          }
          else
          {
            if (Vect7_[pos])
              Vect6_ = Vect7_[pos];
          }
          break;
      
        case 3:
          if (x == max)
          {
            if (Vect6_[pos])
            {
              Vect5_ = Vect6_[pos];
              ParentRefs_ =
              GetParentReference(ParentRefs_, Dim_, off, Index_, LenVect_, PosVect_, Vect5_, Error_);
              return ParentRefs_;              
            }
          }
          else
          {
            if (Vect6_[pos])
              Vect5_ = Vect6_[pos];
          }
          break;
        
        case 4:
          if (x == max)
          {
            if (Vect5_[pos])
            {
              Vect4_ = Vect5_[pos];
              ParentRefs_ =
              GetParentReference(ParentRefs_, Dim_, off, Index_, LenVect_, PosVect_, Vect4_, Error_);
              return ParentRefs_;
            }
          }
          else
          {
            if (Vect5_[pos])
              Vect4_ = Vect5_[pos];
          }
          break;

        case 5:
          if (x == max)
          {
            if (Vect4_[pos])
            {
              Vect3_ = Vect4_[pos];
              ParentRefs_ =
              GetParentReference(ParentRefs_, Dim_, off, Index_, LenVect_, PosVect_, Vect3_, Error_);
              return ParentRefs_;
            }
          }
          else
          {
            if (Vect4_[pos])
              Vect3_ = Vect4_[pos];
          }
          break;

        case 6:
          if (x == max)
          {
            if (Vect3_[pos])
            {
              Vect2_ = Vect3_[pos];
              ParentRefs_ =
              GetParentReference(ParentRefs_, Dim_, off, Index_, LenVect_, PosVect_, Vect2_, Error_);
              return ParentRefs_;              
            }
          }
          else
          {
            if (Vect3_[pos])
              Vect2_ = Vect3_[pos];
          }
          break;

        case 7:
          if (x == max)
          {
            if (Vect2_[pos])
            {
              Vect1_ = Vect2_[pos];
              ParentRefs_ =
              GetParentReference(ParentRefs_, Dim_, off, Index_, LenVect_, PosVect_, Vect1_, Error_);
              return ParentRefs_;              
            }
          }
          else
          {
            if (Vect2_[pos])
            {
              Vect1_ = Vect2_[pos];
              ParentRefs_ =
              GetParentReference(ParentRefs_, Dim_, off, Index_, LenVect_, PosVect_, Vect1_, Error_);
              return ParentRefs_;
            }
          }
          break;        
      }
    }
  }

  return NULL;
}

/****************************************************************************/
// PURPOSE:
//   Method to read integer data from the general (void*) multidimensional
//   data matrix.
//
// PRE:
//   void* Array  : Data matrix to initialize
//   int Dim      : Dimensions of matrix
//   int Index    : Index position into matrix of arrays
//   int* LenVect : Length vector containing length of constituent arrays
//   int* PosVect : Position vector containing coordinates into matrix
//   bool& Error_ : Error in array indexing (out of range)
//
// POST:
//   Returns integer data in the specified positions from the general data
//   matrix.
//
int ReadFromArray(void* Array_, int Dim_, int* LenVect_,
                  int* PosVect_, bool& Error_)
{
  // null pointer checks
  if (!Array_ || !LenVect_ || !PosVect_)
    return 0;

  int******** Vect8_;
  int******* Vect7_;
  int****** Vect6_;
  int***** Vect5_;
  int**** Vect4_;
  int*** Vect3_;
  int** Vect2_;
  int* Vect1_;

  int off = 8 - Dim_; // Maximum dimensions is: 8
  int pos;  
  int x;

  int xIndex_;
  int xDim_;
  int xOff_;
  int ret;
  int Index_ = Dim_ - 1;

  if (2 <= Dim_ && Dim_ <= 8)
  {
    if (Index_ != 0)
    {
      if (Dim_ == 2)
      {
        x = 1;
        if (InRange(x, PosVect_[x], SizeOfVector(Dim_, x-1, LenVect_) - 1))
          pos = PosVect_[x];
        else
          pos = 0;

        if (pos && Array_)
        {      
          Vect2_ = (int**)Array_;
          Vect1_ = Vect2_[pos];
          ++x;

          if (InRange(x, PosVect_[x], SizeOfVector(Dim_, x-1, LenVect_) - 1))
            pos = PosVect_[x];
          else
            pos = 0;          

          // Setting vector size
          if (pos && Vect1_)
          {
            ret = Vect1_[pos];
            Error_ = false;
            return ret;
          }
          else
          {
            Error_ = true;
            return 0;
          }
        }
        else
        {
          Error_ = true;
          return 0;
        }
      }
      else
      {
        Vect8_ = (off == 0) ? (int********)Array_:NULL;
        Vect7_ = (off == 1) ? (int*******)Array_:NULL;
        Vect6_ = (off == 2) ? (int******)Array_:NULL;
        Vect5_ = (off == 3) ? (int*****)Array_:NULL;
        Vect4_ = (off == 4) ? (int****)Array_:NULL;
        Vect3_ = (off == 5) ? (int***)Array_:NULL;
        Vect2_ = (off == 6) ? (int**)Array_:NULL;      

        xDim_ = Dim_;
        Dim_ += off;
        xIndex_ = Index_;
        Index_ += off;

        for (x = 1; x < xDim_; x++)
        {
          xOff_ = x + off;

          if ((Error_ = (x <= xIndex_)) &&
              InRange(x, PosVect_[x], SizeOfVector(xDim_, x-1, LenVect_) - 1))
            pos = PosVect_[x];
          else
            pos = 0;

          if (!pos && Error_)
          {
            Error_ = true;
            return 0;
          }
          else
            Error_ = false;
                        
          switch (xOff_)
          {
            case 1:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect8_[pos])
                  {
                    Vect7_ = Vect8_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect7_[pos])
                    {
                      Error_ = false;
                      return 0;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
                else
                {
                  if (Vect8_[pos])
                    Vect7_ = Vect8_[pos];
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
              }
              else
              {
                Error_ = true;
                return 0;              
              }
              break;

            case 2:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect7_[pos])
                  {
                    Vect6_ = Vect7_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect6_[pos])
                    {
                      Error_ = false;
                      return 0;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
                else
                {
                  if (Vect7_[pos])
                    Vect6_ = Vect7_[pos];
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
              }
              else
              {
                Error_ = true;
                return 0;
              }
              break;
              
            case 3:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect6_[pos])
                  {
                    Vect5_ = Vect6_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect5_[pos])
                    {
                      Error_ = false;
                      return 0;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
                else
                {
                  if (Vect6_[pos])
                    Vect5_ = Vect6_[pos];
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
              }
              else
              {
                Error_ = true;
                return 0;
              }            
              break;
            
            case 4:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect5_[pos])
                  {
                    Vect4_ = Vect5_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect4_[pos])
                    {
                      Error_ = false;
                      return 0;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
                else
                {
                  if (Vect5_[pos])
                    Vect4_ = Vect5_[pos];
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
              }
              else
              {
                Error_ = true;
                return 0;
              }            
              break;
            
            case 5:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect4_[pos])
                  {
                    Vect3_ = Vect4_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect3_[pos])
                    {
                      Error_ = false;
                      return 0;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
                else
                {
                  if (Vect4_[pos])
                    Vect3_ = Vect4_[pos];
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
              }
              else
              {
                Error_ = true;
                return 0;
              }
              break;

            case 6:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect3_[pos])
                  {
                    Vect2_ = Vect3_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect2_[pos])
                    {
                      Error_ = false;
                      return 0;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
                else
                {
                  if (Vect3_[pos])
                    Vect2_ = Vect3_[pos];
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
              }
              else
              {
                Error_ = true;
                return 0;
              }            
              break;

            case 7:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect2_[pos])
                  {
                    Vect1_ = Vect2_[pos];

                    if (InRange(1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos)
                    {
                      ret = Vect1_[pos];
                      Error_ = false;
                      return ret;
                    }
                    else
                      Error_ = true;
                  }
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
                else
                {
                  if (Vect2_[pos])
                    Vect1_ = Vect2_[pos];
                  else
                  {
                    Error_ = true;
                    return 0;
                  }
                }
              }
              else
              {
                Error_ = true;
                return 0;
              }
          }
        }
      }
    }
  }
  else if (Dim_ == 1)
  {
    // In calling InitArrayData function for 1D vector,
    // Index argument is assumed to hold the array index position.
    if (InRange(1, PosVect_[1], SizeOfVector(xDim_, 0, LenVect_) - 1))
      pos = PosVect_[1];
    else
      pos = 0;
    
    if (pos)
    {
      ret = Vect1_[pos];
      Error_ = false;
      return ret;
    }
    else
      Error_ = true;
  }

  return 0;
}

/****************************************************************************/
// PURPOSE:
//   Method to write into the general (void*) multidimensional data matrix
//
// PRE:
//   void* Array  : Data matrix to initialize
//   int Dim      : Dimensions of matrix
//   int Index    : Index position into matrix of arrays
//   int* LenVect : Length vector containing length of constituent arrays
//   int* PosVect : Position vector containing coordinates into matrix
//   int Data_    : Integer data to assign into the data matrix.
//   bool& Error_ : Error in array indexing (out of range)
//
// POST:
//   Returns passed data matrix assigned with the specified integer data.
//
void* WriteToArray(void* Array_, int Dim_, int* LenVect_,
                   int* PosVect_, int Data_, bool& Error_)
{
  // null pointer checks
  if (!Array_ || !LenVect_ || !PosVect_)
    return NULL;

  int******** Vect8_;
  int******* Vect7_;
  int****** Vect6_;
  int***** Vect5_;
  int**** Vect4_;
  int*** Vect3_;
  int** Vect2_;
  int* Vect1_;

  int off = 8 - Dim_; // Maximum dimensions is: 8
  int pos;  
  int x;

  int xIndex_;
  int xDim_;
  int xOff_;
  int Index_ = Dim_ - 1;

  if (2 <= Dim_ && Dim_ <= 8)
  {
    if (Index_ != 0)
    {
      if (Dim_ == 2)
      {
        x = 1;
        if (InRange(x, PosVect_[x], SizeOfVector(Dim_, x-1, LenVect_) - 1))
          pos = PosVect_[x];
        else
          pos = 0;

        if (pos && Array_)
        {      
          Vect2_ = (int**)Array_;
          Vect1_ = Vect2_[pos];
          ++x;

          if (InRange(x, PosVect_[x], SizeOfVector(Dim_, x-1, LenVect_) - 1))
            pos = PosVect_[x];
          else
            pos = 0;          

          // Setting vector size
          if (pos && Vect1_)
          {
            Vect1_[pos] = Data_;
            Error_ = false;
            return Array_;
          }
          else
          {
            Error_ = true;
            return Array_;
          }
        }
        else
        {
          Error_ = true;
          return Array_;
        }
      }
      else
      {
        Vect8_ = (off == 0) ? (int********)Array_:NULL;
        Vect7_ = (off == 1) ? (int*******)Array_:NULL;
        Vect6_ = (off == 2) ? (int******)Array_:NULL;
        Vect5_ = (off == 3) ? (int*****)Array_:NULL;
        Vect4_ = (off == 4) ? (int****)Array_:NULL;
        Vect3_ = (off == 5) ? (int***)Array_:NULL;
        Vect2_ = (off == 6) ? (int**)Array_:NULL;      

        xDim_ = Dim_;
        Dim_ += off;
        xIndex_ = Index_;
        Index_ += off;

        for (x = 1; x < xDim_; x++)
        {
          xOff_ = x + off;

          if ((Error_ = (x <= xIndex_)) &&
              InRange(x, PosVect_[x], SizeOfVector(xDim_, x-1, LenVect_) - 1))
            pos = PosVect_[x];
          else
            pos = 0;

          if (!pos && Error_)
          {
            Error_ = true;
            return Array_;
          }
          else
            Error_ = false;
                        
          switch (xOff_)
          {
            case 1:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect8_[pos])
                  {
                    Vect7_ = Vect8_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect7_[pos])
                    {
                      Error_ = false;
                      return Array_;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
                else
                {
                  if (Vect8_[pos])
                    Vect7_ = Vect8_[pos];
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
              }
              else
              {
                Error_ = true;
                return Array_;              
              }
              break;

            case 2:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect7_[pos])
                  {
                    Vect6_ = Vect7_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect6_[pos])
                    {
                      Error_ = false;
                      return Array_;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
                else
                {
                  if (Vect7_[pos])
                    Vect6_ = Vect7_[pos];
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
              }
              else
              {
                Error_ = true;
                return Array_;
              }
              break;
              
            case 3:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect6_[pos])
                  {
                    Vect5_ = Vect6_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect5_[pos])
                    {
                      Error_ = false;
                      return Array_;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
                else
                {
                  if (Vect6_[pos])
                    Vect5_ = Vect6_[pos];
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
              }
              else
              {
                Error_ = true;
                return Array_;
              }            
              break;
            
            case 4:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect5_[pos])
                  {
                    Vect4_ = Vect5_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect4_[pos])
                    {
                      Error_ = false;
                      return Array_;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
                else
                {
                  if (Vect5_[pos])
                    Vect4_ = Vect5_[pos];
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
              }
              else
              {
                Error_ = true;
                return Array_;
              }            
              break;
            
            case 5:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect4_[pos])
                  {
                    Vect3_ = Vect4_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect3_[pos])
                    {
                      Error_ = false;
                      return Array_;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
                else
                {
                  if (Vect4_[pos])
                    Vect3_ = Vect4_[pos];
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
              }
              else
              {
                Error_ = true;
                return Array_;
              }
              break;

            case 6:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect3_[pos])
                  {
                    Vect2_ = Vect3_[pos];

                    if (InRange(x+1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos && Vect2_[pos])
                    {
                      Error_ = false;
                      return Array_;
                    }
                    else
                      Error_ = true;                    
                  }
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
                else
                {
                  if (Vect3_[pos])
                    Vect2_ = Vect3_[pos];
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
              }
              else
              {
                Error_ = true;
                return Array_;
              }            
              break;

            case 7:
              if (pos)
              {
                if (xOff_ == Index_)
                {
                  if (Vect2_[pos])
                  {
                    Vect1_ = Vect2_[pos];

                    if (InRange(1, PosVect_[x+1], SizeOfVector(xDim_, x, LenVect_) - 1))
                      pos = PosVect_[x+1];
                    else
                      pos = 0;

                    if (pos)
                    {
                      Vect1_[pos] = Data_;
                      Error_ = false;
                      return Array_;
                    }
                    else
                      Error_ = true;
                  }
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
                else
                {
                  if (Vect2_[pos])
                    Vect1_ = Vect2_[pos];
                  else
                  {
                    Error_ = true;
                    return Array_;
                  }
                }
              }
              else
              {
                Error_ = true;
                return Array_;
              }
          }
        }
      }
    }
  }
  else if (Dim_ == 1)
  {
    // In calling InitArrayData function for 1D vector,
    // Index argument is assumed to hold the array index position.
    if (InRange(1, PosVect_[1], SizeOfVector(xDim_, 0, LenVect_) - 1))
      pos = PosVect_[1];
    else
      pos = 0;
    
    if (pos)
    {
      Vect1_[pos] = Data_;
      Array_ = Vect1_;      
      Error_ = false;
      return Array_;
    }
    else
      Error_ = true;
  }

  return Array_;
}

/****************************************************************************/
// PURPOSE:
//   Helper function to deallocate memory in the arrays of the data matrix.
//
// PRE:
//   void* Array_  : Data matrix to deallocate
//   int Dim_      : Dimensions of matrix
//   int Cnt_      : Current dimension count of the current call to this function
//   int pos       : Index position to access an element within the data matrix
//   int* LenVect_ : Length vector containing length of constituent arrays
//
// POST:
//   Returns the deallocated data matrix
//
void* DestroyArrayDataHelper(void* Array_, int Dim_, int Cnt_, int pos, int* LenVect_)
{
  if (!Array_ || !LenVect_)
    return Array_;

  int******** Vect8_;
  int******* Vect7_;
  int****** Vect6_;
  int***** Vect5_;
  int**** Vect4_;
  int*** Vect3_;
  int** Vect2_;
  int* Vect1_;

  int off = 8 - Dim_; // Maximum dimensions is: 8
  int PosVect_[9];    // Maximum dimensions is: 8  
  int xOff_;
  int x, sz;
  int Shft_ = Dim_ - Cnt_; // Shift value down staggered data matrix
  int Index_;
  bool Error_;
  memset(PosVect_, 0, sizeof(int) * 9);

  if (2 <= Cnt_ && Cnt_ <= Dim_)
  {
    if (Cnt_ <= 2)
    {
      int max = (Dim_ == 2) ? SizeOfVector(Dim_, 0, LenVect_):
                              SizeOfVector(Dim_, Shft_, LenVect_);
                              
      if (max)
      {
        Vect2_ = (int**)Array_;

        if (Vect2_[0])
        {
          if (Cnt_ == 2)
          {
            ::DeleteArray(Vect2_[0]);
            Vect2_[0] = NULL;
          }
          
          for (x = Shft_ + pos; x < max; x++)
            if ((Shft_ <= x && x < max) && Vect2_[x])
            {
              ::DeleteArray(Vect2_[x]);
              Vect2_[x] = NULL;
            }
        }

        if (Dim_ == 2)
        {
          ::DeleteArray(Vect2_);
          Vect2_ = NULL;
          Array_ = NULL;
        }
      }
      else
        return NULL;
    }
    else
    {
      Vect8_ = (off == 0) ? (int********)Array_:NULL;
      Vect7_ = (off == 1) ? (int*******)Array_:NULL;
      Vect6_ = (off == 2) ? (int******)Array_:NULL;
      Vect5_ = (off == 3) ? (int*****)Array_:NULL;
      Vect4_ = (off == 4) ? (int****)Array_:NULL;
      Vect3_ = (off == 5) ? (int***)Array_:NULL;
      Vect2_ = (off == 6) ? (int**)Array_:NULL;
      x = Shft_;
      
      if (x < Dim_)
      {
        pos += Shft_;
        xOff_ = x + off;
        sz = SizeOfVector(Dim_, x, LenVect_);

        #if NESTARRAY_DEBUG3c
          cout <<"*** DestroyArrayDataHelper(void*, int, int, int, int*) ***" <<endl;
          cout <<"off = " <<off <<", Shft = " <<Shft_
               <<", Dim = " <<Dim_
               <<", pos = " <<pos
               <<", xOff = " <<xOff_
               <<", sz = " <<sz <<endl
               <<"Cnt = " <<Cnt_ <<endl;
        #endif

        if (x <= pos && pos < sz)
        {
          switch (xOff_)
          {
            case 0:
              if (Vect8_[pos])
              {
                #if NESTARRAY_DEBUG3c
                  cout <<", del Vect8[pos] = " <<Vect8_[pos] <<endl;
                #endif
              
                Vect7_ = Vect8_[pos];
                Index_ = Dim_ - (Cnt_-1);
                PosVect_[Index_] = Index_;
                PosVect_[Index_+1] = Index_+1;

                RemoveReferences(Vect8_, Dim_+off, off, Index_+off,
                                 LenVect_, PosVect_, NULL, true,
                                 NULL, NULL, NULL, Error_);
                
                DestroyArrayDataHelper(Vect7_, Dim_, Cnt_-1, 1, LenVect_);
                if (pos+Shft_ < sz-1)
                  DestroyArrayDataHelper(Vect8_, Dim_, Cnt_, pos+1, LenVect_);
                if (pos >= 1 - off && (pos >= 1 + Shft_ || !Shft_))
                  ::DeleteArray(Vect8_[pos]);
                Vect8_[pos] = NULL;

                if (pos == 1)
                  Vect8_[0] = NULL;
              }
              else if (0 <= Shft_ && Shft_ < Dim_-2 && pos+Shft_ < sz-1)
                DestroyArrayDataHelper(Vect8_, Dim_, Cnt_, pos+1, LenVect_);
              break;

            case 1:
              if (Vect7_[pos])
              {
                #if NESTARRAY_DEBUG3c
                  cout <<", del Vect7[pos] = " <<Vect7_[pos] <<endl;
                #endif

                Vect6_ = Vect7_[pos];
                Index_ = Dim_ - (Cnt_-1);
                PosVect_[Index_] = Index_;
                PosVect_[Index_+1] = Index_+1;

                RemoveReferences(Vect7_, Dim_+off, off, Index_+off,
                                 LenVect_, PosVect_, NULL, true,
                                 NULL, NULL, NULL, Error_);

                Vect6_ = Vect7_[pos];
                DestroyArrayDataHelper(Vect6_, Dim_, Cnt_-1, 1, LenVect_);
                if (pos+Shft_ < sz-1)
                  DestroyArrayDataHelper(Vect7_, Dim_, Cnt_, pos+1, LenVect_);
                if (pos >= 2 - off && (pos >= 1 + Shft_ || !Shft_))
                  ::DeleteArray(Vect7_[pos]);
                Vect7_[pos] = NULL;

                if (pos == 1)
                  Vect7_[0] = NULL;
              }
              else if (0 <= Shft_ && Shft_ < Dim_-2 && pos+Shft_ < sz-1)
                DestroyArrayDataHelper(Vect7_, Dim_, Cnt_, pos+1, LenVect_);
              break;

            case 2:
              if (Vect6_[pos])
              {
                #if NESTARRAY_DEBUG3c
                  cout <<", del Vect6[pos] = " <<Vect6_[pos] <<endl;
                #endif
                
                Vect5_ = Vect6_[pos];
                Index_ = Dim_ - (Cnt_-1);
                PosVect_[Index_] = Index_;
                PosVect_[Index_+1] = Index_+1;

                RemoveReferences(Vect6_, Dim_+off, off, Index_+off,
                                 LenVect_, PosVect_, NULL, true,
                                 NULL, NULL, NULL, Error_);
                
                DestroyArrayDataHelper(Vect5_, Dim_, Cnt_-1, 1, LenVect_);
                if (pos+Shft_ < sz-1)
                  DestroyArrayDataHelper(Vect6_, Dim_, Cnt_, pos+1, LenVect_);
                if (pos >= 3 - off && (pos >= 1 + Shft_ || !Shft_))
                  ::DeleteArray(Vect6_[pos]);
                Vect6_[pos] = NULL;

                if (pos == 1)
                  Vect6_[0] = NULL;
              }
              else if (0 <= Shft_ && Shft_ < Dim_-2 && pos+Shft_ < sz-1)
                DestroyArrayDataHelper(Vect6_, Dim_, Cnt_, pos+1, LenVect_);
              break;

            case 3:
              if (Vect5_[pos])
              {
                #if NESTARRAY_DEBUG3c
                  cout <<", del Vect5[pos] = " <<Vect5_[pos] <<endl;
                #endif
                
                Vect4_ = Vect5_[pos];
                Index_ = Dim_ - (Cnt_-1);
                PosVect_[Index_] = Index_;
                PosVect_[Index_+1] = Index_+1;

                RemoveReferences(Vect5_, Dim_+off, off, Index_+off,
                                 LenVect_, PosVect_, NULL, true,
                                 NULL, NULL, NULL, Error_);
                
                DestroyArrayDataHelper(Vect4_, Dim_, Cnt_-1, 1, LenVect_);
                if (pos+Shft_ < sz-1)
                  DestroyArrayDataHelper(Vect5_, Dim_, Cnt_, pos+1, LenVect_);
                if (pos >= 4 - off && (pos >= 1 + Shft_ || !Shft_))
                  ::DeleteArray(Vect5_[pos]);
                Vect5_[pos] = NULL;

                if (pos == 1)
                  Vect5_[0] = NULL;
              }
              else if (0 <= Shft_ && Shft_ < Dim_-2 && pos+Shft_ < sz-1)
                DestroyArrayDataHelper(Vect5_, Dim_, Cnt_, pos+1, LenVect_);
              break;

            case 4:
              if (Vect4_[pos])
              {
                #if NESTARRAY_DEBUG3c
                  cout <<", del Vect4[pos] = " <<Vect4_[pos] <<endl;
                #endif
                
                Vect3_ = Vect4_[pos];
                Index_ = Dim_ - (Cnt_-1);
                PosVect_[Index_] = Index_;
                PosVect_[Index_+1] = Index_+1;

                RemoveReferences(Vect4_, Dim_+off, off, Index_+off,
                                 LenVect_, PosVect_, NULL, true,
                                 NULL, NULL, NULL, Error_);
                
                DestroyArrayDataHelper(Vect3_, Dim_, Cnt_-1, 1, LenVect_);
                if (pos+Shft_ < sz-1)
                  DestroyArrayDataHelper(Vect4_, Dim_, Cnt_, pos+1, LenVect_);
                if (pos >= 5 - off && (pos >= 1 + Shft_ || !Shft_))
                  ::DeleteArray(Vect4_[pos]);
                Vect4_[pos] = NULL;

                if (pos == 1)
                  Vect4_[0] = NULL;
              }
              else if (0 <= Shft_ && Shft_ < Dim_-2 && pos+Shft_ < sz-1)
                DestroyArrayDataHelper(Vect4_, Dim_, Cnt_, pos+1, LenVect_);
              break;

            case 5:
              if (Vect3_[pos])
              {
                #if NESTARRAY_DEBUG3c
                  cout <<", del Vect3[pos] = " <<Vect3_[pos] <<endl;
                #endif
                
                Vect2_ = Vect3_[pos];
                Index_ = Dim_ - (Cnt_-1);
                PosVect_[Index_] = Index_;
                PosVect_[Index_+1] = Index_+1;

                RemoveReferences(Vect3_, Dim_+off, off, Index_+off,
                                 LenVect_, PosVect_, NULL, true,
                                 NULL, NULL, NULL, Error_);
                
                DestroyArrayDataHelper(Vect2_, Dim_, Cnt_-1, 1, LenVect_);
                if (pos+Shft_ < sz-1)
                  DestroyArrayDataHelper(Vect3_, Dim_, Cnt_, pos+1, LenVect_);
                if (pos >= 6 - off && (pos >= 1 + Shft_ || !Shft_))
                  ::DeleteArray(Vect3_[pos]);
                Vect3_[pos] = NULL;

                if (pos == 1)
                  Vect3_[0] = NULL;
              }
              else if (0 <= Shft_ && Shft_ < Dim_-2 && pos+Shft_ < sz-1)
                DestroyArrayDataHelper(Vect3_, Dim_, Cnt_, pos+1, LenVect_);
              break;
          }        
        }
      }
    }
  }

  return Array_;
}

/****************************************************************************/
// PURPOSE:
//   Main function to deallocate memory in the arrays of the data matrix.
//   This function will call the helper function:
//     DestroyArrayDataHelper(void*, int, int, int, int*);
//
// PRE:
//   void* Array_  : Data matrix to deallocate
//   int Dim_      : Dimensions of matrix
//   int* LenVect_ : Length vector containing length of constituent arrays
//
// POST:
//   Returns the deallocated data matrix
//
void* DestroyArrayData(void* Array_, int Dim_, int* LenVect_)
{
  int******** Vect8_;
  int******* Vect7_;
  int****** Vect6_;
  int***** Vect5_;
  int**** Vect4_;
  int*** Vect3_;
  int** Vect2_;
  int* Vect1_;

  int off = 8 - Dim_; // Maximum dimensions is: 8
  int max;
  int x;

  if (2 <= Dim_ && Dim_ <= 8)
  {
    if (Dim_ == 2)
    {
      max = SizeOfVector(Dim_, 0, LenVect_);

      if (max)
      {
        Vect2_ = (int**)Array_;
        
        if (Vect2_ && Vect2_[0])
        {
          ::DeleteArray(Vect2_[0]);
          Vect2_[0] = NULL;
          
          for (x = 1; x < max; x++)
            if (Vect2_[x])
            {
              ::DeleteArray(Vect2_[x]);
              Vect2_[x] = NULL;
            }
        }

        ::DeleteArray(Vect2_);
        Vect2_ = NULL;
        Array_ = NULL;
      }
      else
        return NULL;
    }
    else
    {
      Vect8_ = (off == 0) ? (int********)Array_:NULL;
      Vect7_ = (off == 1) ? (int*******)Array_:NULL;
      Vect6_ = (off == 2) ? (int******)Array_:NULL;
      Vect5_ = (off == 3) ? (int*****)Array_:NULL;
      Vect4_ = (off == 4) ? (int****)Array_:NULL;
      Vect3_ = (off == 5) ? (int***)Array_:NULL;
      Vect2_ = (off == 6) ? (int**)Array_:NULL;
    
      DestroyArrayDataHelper(Array_, Dim_, Dim_, 1, LenVect_);

      if (Vect8_)
      {
        ::DeleteArray(Vect8_);
        Vect8_ = NULL;
      }
      else if (Vect7_)
      {
        ::DeleteArray(Vect7_);
        Vect7_ = NULL;
      }        
      else if (Vect6_)
      {
        ::DeleteArray(Vect6_);
        Vect6_ = NULL;
      }
      else if (Vect5_)
      {
        ::DeleteArray(Vect5_);
        Vect5_ = NULL;
      }
      else if (Vect4_)
      {
        ::DeleteArray(Vect4_);
        Vect4_ = NULL;
      }
      else if (Vect3_)
      {
        ::DeleteArray(Vect3_);
        Vect3_ = NULL;
      }
      else if (Vect2_)
      {
        ::DeleteArray(Vect2_);
        Vect2_ = NULL;
      }
    }
  }
  else if (Dim_ == 1)
  {
    Vect1_ = (int*)Array_;
    ::DeleteArray(Vect1_);
    
    Vect1_ = NULL;
    Array_ = NULL;
  }

  return Array_;
}

/****************************************************************************/
void* CheckVector(void* Array_, int Dim_, int Index_, int Incr_, int Level_)
{
  int******** Vect8_ = (Dim_ == 8) ? (int********)Array_:NULL;
  int******* Vect7_ = (Dim_ == 7) ? (int*******)Array_:NULL;
  int****** Vect6_ = (Dim_ == 6) ? (int******)Array_:NULL;
  int***** Vect5_ = (Dim_ == 5) ? (int*****)Array_:NULL;
  int**** Vect4_ = (Dim_ == 4) ? (int****)Array_:NULL;
  int*** Vect3_ = (Dim_ == 3) ? (int***)Array_:NULL;  
  int** Vect2_ = (Dim_ == 2) ? (int**)Array_:NULL;
  int max = Index_ + 1;

  if (Dim_ == 8)
  {
    if (!Vect8_)
      Vect8_ = (int********)RawCallocateWith(MEMMATRIX, sizeof(int*******) * max);
      
    Vect8_[Index_] = (int*******)CheckVector(Vect8_[Index_], Dim_-1, Index_+Incr_, Incr_, Level_+1);
    return Vect8_;
  }
  else if (Dim_ == 7)
  {
    if (!Vect7_)
      Vect7_ = (int*******)RawCallocateWith(MEMMATRIX, sizeof(int******) * max);
      
    Vect7_[Index_] = (int******)CheckVector(Vect7_[Index_], Dim_-1, Index_+Incr_, Incr_, Level_+1);
    return Vect7_;
  }
  else if (Dim_ == 6)
  {
    if (!Vect6_)
      Vect6_ = (int******)RawCallocateWith(MEMMATRIX, sizeof(int*****) * max);
      
    Vect6_[Index_] = (int*****)CheckVector(Vect6_[Index_], Dim_-1, Index_+Incr_, Incr_, Level_+1);
    return Vect6_;
  }
  else if (Dim_ == 5)
  {
    if (!Vect5_)
      Vect5_ = (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****) * max);

    Vect5_[Index_] = (int****)CheckVector(Vect5_[Index_], Dim_-1, Index_+Incr_, Incr_, Level_+1);
    return Vect5_;
  }
  else if (Dim_ == 4)
  {
    if (!Vect4_)
      Vect4_ = (int****)RawCallocateWith(MEMMATRIX, sizeof(int***) * max);

    Vect4_[Index_] = (int***)CheckVector(Vect4_[Index_], Dim_-1, Index_+Incr_, Incr_, Level_+1);
    return Vect4_;
  }
  else if (Dim_ == 3)
  {
    if (!Vect3_)
      Vect3_ = (int***)RawCallocateWith(MEMMATRIX, sizeof(int**) * max);

    Vect3_[Index_] = (int**)CheckVector(Vect3_[Index_], Dim_-1, Index_+Incr_, Incr_, Level_+1);
    return Vect3_;
  }
  else if (Dim_ == 2)
  {
    if (!Vect2_)
      Vect2_ = (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * max);

    return Vect2_;
  }

  return Array_;
}

/****************************************************************************/
// PURPOSE:
//   Method to intialize general (void*) multidimensional data matrix
//
//   OLD VERSION: works for 6th dimensional matrix
//
// PRE:
//   void* Array  : Data matrix to initialize
//   int Dim      : Dimensions of matrix
//   int Index    : Index position into matrix of arrays
//   int* LenVect : Length vector containing length of constituent arrays
//   int* PosVect : Position vector containing coordinates into matrix
//   bool& Error_ : Error in array indexing (out of range)
//
// POST:
//   Returns initialized data matrix of given dimensions and of given lengths
//
void* InitArrayData2(void* Array_, int Dim_, int Index_,
                     int* LenVect_, int* PosVect_, bool& Error_, int Shift_)
{
  int******** Vect8_ = NULL;
  int******* Vect7_ = NULL;
  int****** Vect6_ = NULL;
  int***** Vect5_ = NULL;
  int**** Vect4_ = NULL;
  int*** Vect3_ = NULL;
  int** Vect2_ = NULL;
  int* Vect1_ = NULL;

  int******** Rpt8_ = NULL;
  int******* Rpt7_ = NULL;
  int****** Rpt6_ = NULL;
  int***** Rpt5_ = NULL;
  int**** Rpt4_ = NULL;
  int*** Rpt3_ = NULL;
  int** Rpt2_ = NULL;
  int* Rpt1_ = NULL;

  int** InitVect_ = NULL;
  int** Contain2_ = NULL;
  int* Contain_ = NULL;

  int off = 8 - Dim_; // Maximum dimensions is: 8
  int pos;
  int oldpos;
  int newpos;
  int max;
  int sz;
  int vsz;
  int oldsz;
  int newsz;
  int addsz;
  int x;
  int cnt = 0;  
  int incr;

  int mIndex_;
  int xIndex_;
  int xDim_;
  int xOff_;

  bool RangeCond_;
  bool RangeValid_;
  bool rv;
  
  // Contain_[pos] = 1;    // deleted
  // Contain_[pos] = 2;    // shared
  // SizeOfVector(Dim_, x, LenVect_)
  // CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_)
  // StatusOfVector(Dim_, x, Off_, y, Vect1_, Contain_)
  // SetStatusOfVector(Dim_, x, Off_, y, stat, Contain_)

  if (2 <= Dim_ && Dim_ <= 8 &&
      0 <= Index_ && Index_ < Dim_)
  {
    x = cnt + Shift_;
    max = vsz = (x == 0) ? LenVect_[0]+1:LenVect_[x]+x+1;
    sz = SizeOfVector(Dim_, Index_, LenVect_);
    
    if (sz < 2)
      return NULL;

    if (Index_ == 0)
    {      
      Vect8_ = (off == 0) ?
      (int********)RawCallocateWith(MEMMATRIX, sizeof(int*******) * max):
      NULL;
      
      if (Vect8_ && off == 0)
        memset(Vect8_, 0, sizeof(int*******) * max);
      
      Vect7_ = (off == 1) ?
      (int*******)RawCallocateWith(MEMMATRIX, sizeof(int******) * max):
      (off < 1) ? (Vect8_[0] = (int*******)RawCallocateWith(MEMMATRIX, sizeof(int******))):
      NULL;
      
      if (Vect7_)
        if (off == 1)
          memset(Vect7_, 0, sizeof(int******) * max);
        else
          Vect7_[0] = 0;
      
      Vect6_ = (off == 2) ?
      (int******)RawCallocateWith(MEMMATRIX, sizeof(int*****) * max):
      (off < 2) ? (Vect7_[0] = (int******)RawCallocateWith(MEMMATRIX, sizeof(int*****))):
      NULL;
      
      if (Vect6_)
        if (off == 2)
          memset(Vect6_, 0, sizeof(int*****) * max);
        else
          Vect6_[0] = 0;

      Vect5_ = (off == 3) ?
      (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****) * max):
      (off < 3) ? (Vect6_[0] = (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****))):
      NULL;
      
      if (Vect5_)
        if (off == 3)
          memset(Vect5_, 0, sizeof(int****) * max);
        else
          Vect5_[0] = 0;

      Vect4_ = (off == 4) ?
      (int****)RawCallocateWith(MEMMATRIX, sizeof(int***) * max):
      (off < 4) ? (Vect5_[0] = (int****)RawCallocateWith(MEMMATRIX, sizeof(int***))):
      NULL;
      
      if (Vect4_)
        if (off == 4)
          memset(Vect4_, 0, sizeof(int***) * max);
        else
          Vect4_[0] = 0;

      Vect3_ = (off == 5) ?
      (int***)RawCallocateWith(MEMMATRIX, sizeof(int**) * max):
      (off < 5) ? (Vect4_[0] = (int***)RawCallocateWith(MEMMATRIX, sizeof(int**))):
      NULL;
      
      if (Vect3_)
        if (off == 5)
          memset(Vect3_, 0, sizeof(int**) * max);
        else
          Vect3_[0] = 0;

      #if NESTARRAY_DEBUG3
        cout <<endl
             <<"*** vect3 initialization stage 1 ***" <<endl;
        cout <<"off = " <<off <<", vsz = " <<vsz <<endl
             <<"Vect3 = " <<Vect3_ <<endl;
      #endif

      if (off < 6)
        max = 1;

      if (Dim_ >= 2 && max > 0)
        if (max < 2)
          max = Dim_ + 1;

      cnt = 0;
      vsz = max;
      max *= sizeof(int*);
      
      Vect2_ = (off == 6) ?
      (int**)RawCallocateWith(MEMMATRIX, max):
      (off < 6) ? (Vect3_[0] = (int**)RawCallocateWith(MEMMATRIX, max)):
      NULL;
      
      if (Vect2_)
        if (off == 6)
          memset(Vect2_, 0, max);
        else
          Vect2_[0] = 0;

      #if NESTARRAY_DEBUG3
        cout <<endl
             <<"*** vect2 initialization stage 1 ***" <<endl;
        cout <<"off = " <<off <<", vsz = " <<vsz <<endl
             <<"Vect2 = " <<Vect2_ <<endl;
      #endif

      vsz = max = Dim_ + 2;
      max *= sizeof(int);

      Vect1_ = Vect2_[0] = (int*)RawCallocateWith(MEMMATRIX, max);
      memset(Vect1_, 0, max);
      Vect1_[0] = vsz;

      #if NESTARRAY_DEBUG3
        cout <<endl
             <<"*** vect1 initialization stage 1 ***" <<endl;
        cout <<"off = " <<off <<", vsz = " <<vsz <<endl
             <<"Vect1 = " <<Vect1_ <<endl;
      #endif

      // Initializing array root points
      Rpt8_ = Vect8_;
      Rpt7_ = Vect7_;
      Rpt6_ = Vect6_;
      Rpt5_ = Vect5_;
      Rpt4_ = Vect4_;
      Rpt3_ = Vect3_;
      Rpt2_ = Vect2_;
      Rpt1_ = Vect1_;

      // array[(i+1)=1...x+i+1]*  = 1st starting TopArr value element.
      if (Vect8_)
      {
        vsz = LenVect_[0]+1;
        if (vsz < Dim_ + 1)
          vsz = Dim_ + 1;
        
        Contain2_ = (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
        memset(Contain2_, 0, sizeof(int*) * vsz);
        Vect8_[0][0][0][0][0][0] = Contain2_;
        
        for (x = 1; x <= Dim_; x++)
        {
          Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
          Contain2_[x][0] = 0;
        }

        Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
        memset(Contain_, 0, sizeof(int) * vsz);
        Contain_[0] = LenVect_[0];
        Contain2_[0] = Contain_;
        Contain2_[1][0] = LenVect_[0];
                
        ++cnt;
        x = cnt + Shift_;
        Vect7_ = Vect8_[x];
      }

      if (!Vect7_)
      {
        x = cnt + Shift_;
        max = vsz = (x == 0 && off >= 1) ? LenVect_[0]+1:LenVect_[x]+x+1;
        max *= sizeof(int******);
        Vect7_ = (off == 1) ?
        (int*******)RawCallocateWith(MEMMATRIX, max):
        (off < 1) ? (Vect8_[x] = (int*******)RawCallocateWith(MEMMATRIX, max)):
        NULL;
      }

      // array[i=1...n][(i+1)=2...x+i+1]*  = 1st starting 2ndArr value element
      if (Vect7_)
      {
        x = cnt + Shift_;
        max = vsz = (x == 0 && off >= 1) ? LenVect_[0]+1:LenVect_[x]+x+1;
        max *= sizeof(int******);
            
        if (x && off < 1)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;
        
          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          Contain_[0] = Contain_[1] = LenVect_[x];
          Vect7_[0] = (int******)CheckVector(Vect7_[0], 6, 0, 0, 0);
          Vect7_[0][0][0][0][0][0] = Contain_;
        }
        else if (Rpt7_)
        {
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;
        
          Contain2_ = (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
          memset(Contain2_, 0, sizeof(int*) * vsz);
          Rpt7_[0][0][0][0][0] = Contain2_;
          
          for (x = 0; x <= Dim_; x++)
          {
            Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
            Contain2_[x][0] = 0;
          }

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          x = cnt + Shift_;
          vsz = (x == 0 && off >= 1) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;
          Contain2_[0] = Contain_;
          Contain2_[1][0] = vsz;
        }

        ++cnt;
        x = cnt + Shift_;
        Vect6_ = Vect7_[x];
      }

      if (!Vect6_)
      {
        x = cnt + Shift_;
        max = vsz = (x == 0 && off >= 2) ? LenVect_[0]+1:LenVect_[x]+x+1;
        max *= sizeof(int*****);
        Vect6_ = (off == 2) ?
        (int******)RawCallocateWith(MEMMATRIX, max):
        (off < 2) ? (Vect7_[x] = (int******)RawCallocateWith(MEMMATRIX, max)):
        NULL;
      }

      // array[1...n]
      //      [i=2...n][(i+1)=3...x+i+1]* = 1st starting 3rdArr value element
      if (Vect6_)
      {
        x = cnt + Shift_;
        max = vsz = (x == 0 && off >= 2) ? LenVect_[0]+1:LenVect_[x]+x+1;
        max *= sizeof(int*****);
        
        if (x && off < 2)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          Contain_[0] = Contain_[1] = LenVect_[x];
          Vect6_[0] = (int*****)CheckVector(Vect6_[0], 5, 0, 0, 0);
          Vect6_[0][0][0][0][0] = Contain_;
        }
        else if (Rpt6_)
        {
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain2_ = (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
          memset(Contain2_, 0, sizeof(int*) * vsz);          
          Rpt6_[0][0][0][0] = Contain2_;
          
          for (x = 0; x <= Dim_; x++)
          {
            Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
            Contain2_[x][0] = 0;
          }

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          x = cnt + Shift_;
          vsz = (x == 0 && off >= 2) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;
          Contain2_[0] = Contain_;
          Contain2_[1][0] = vsz;
        }

        ++cnt;
        x = cnt + Shift_;
        Vect5_ = Vect6_[x];
      }

      if (!Vect5_)
      {
        x = cnt + Shift_;
        max = vsz = (x == 0 && off >= 3) ? LenVect_[0]+1:LenVect_[x]+x+1;
        max *= sizeof(int****);
        Vect5_ = (off == 3) ?
        (int*****)RawCallocateWith(MEMMATRIX, max):
        (off < 3) ? (Vect6_[x] = (int*****)RawCallocateWith(MEMMATRIX, max)):
        NULL;
      }

      // array[1...n]
      //      [2...n]
      //      [i=3...n][(i+1)=4...x+i+1]* = 1st starting 4thArr value element
      if (Vect5_)
      {
        x = cnt + Shift_;
        max = vsz = (x == 0 && off >= 3) ? LenVect_[0]+1:LenVect_[x]+x+1;
        max *= sizeof(int****);
      
        if (x && off < 3)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          Contain_[0] = Contain_[1] = LenVect_[x];
          Vect5_[0] = (int****)CheckVector(Vect5_[0], 4, 0, 0, 0);
          Vect5_[0][0][0][0] = Contain_;
        }
        else if (Rpt5_)
        {
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain2_ = (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
          memset(Contain2_, 0, sizeof(int*) * vsz);
          Rpt5_[0][0][0] = Contain2_;
          
          for (x = 0; x <= Dim_; x++)
          {
            Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
            Contain2_[x][0] = 0;
          }

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          x = cnt + Shift_;
          vsz = (x == 0 && off >= 3) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;
          Contain2_[0] = Contain_;
          Contain2_[1][0] = vsz;
        }
      
        ++cnt;
        x = cnt + Shift_;
        Vect4_ = Vect5_[x];
      }

      if (!Vect4_)
      {
        x = cnt + Shift_;
        max = vsz = (x == 0 && off >= 4) ? LenVect_[0]+1:LenVect_[x]+x+1;
        max *= sizeof(int***);
        Vect4_ = (off == 4) ?
        (int****)RawCallocateWith(MEMMATRIX, max):
        (off < 4) ? (Vect5_[x] = (int****)RawCallocateWith(MEMMATRIX, max)):
        NULL;
      }

      // array[1...n]
      //      [2...n]
      //      [3...n]
      //      [i=4...n][(i+1)=5...x+i+1]* = 1st starting 5thArr value element
      if (Vect4_)
      {
        x = cnt + Shift_;
        max = vsz = (x == 0 && off >= 4) ? LenVect_[0]+1:LenVect_[x]+x+1;
        max *= sizeof(int***);
            
        if (x && off < 4)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          Contain_[0] = Contain_[1] = LenVect_[x];
          Vect4_[0] = (int***)CheckVector(Vect4_[0], 3, 0, 0, 0);
          Vect4_[0][0][0] = Contain_;
        }
        else if (Rpt4_)
        {
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain2_ = (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
          memset(Contain2_, 0, sizeof(int*) * vsz);
          Rpt4_[0][0] = Contain2_;
          
          for (x = 0; x <= Dim_; x++)
          {
            Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
            Contain2_[x][0] = 0;
          }

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          x = cnt + Shift_;
          vsz = (x == 0 && off >= 4) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;
          Contain2_[0] = Contain_;
          Contain2_[1][0] = vsz;
        }
      
        ++cnt;
        x = cnt + Shift_;
        Vect3_ = Vect4_[x];
      }

      if (!Vect3_)
      {
        x = cnt + Shift_;
        max = vsz = (x == 0 && off >= 5) ? LenVect_[0]+1:LenVect_[x]+x+1;
        max *= sizeof(int**);
        Vect3_ = (off == 5) ?
        (int***)RawCallocateWith(MEMMATRIX, max):
        (off < 5) ? (Vect4_[x] = (int***)RawCallocateWith(MEMMATRIX, max)):
        NULL;

        #if NESTARRAY_DEBUG3
          cout <<endl
               <<"*** vect3 initialization stage 2a ***" <<endl;
          cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
               <<", vsz = " <<vsz <<", LenVect[x] = " <<LenVect_[x] <<endl
               <<"Vect3 = " <<Vect3_ <<", Dim = " <<Dim_ <<endl;
        #endif
      }

      // array[1...n]
      //      [2...n]
      //      [3...n]
      //      [4...n]
      //      [i=5...n][(i+1)=6...x+i+1]* = 1st starting 6thArr value element
      if (Vect3_)
      {
        x = cnt + Shift_;
        max = vsz = (x == 0 && off >= 5) ? LenVect_[0]+1:LenVect_[x]+x+1;
        max *= sizeof(int**);

        if (x && off < 5)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          Contain_[0] = Contain_[1] = LenVect_[x];
          Vect3_[0] = (int**)CheckVector(Vect3_[0], 2, 0, 0, 0);
          Vect3_[0][0] = Contain_;

          #if NESTARRAY_DEBUG3
            cout <<endl
                 <<"*** vect3 initialization stage 2b ***" <<endl;
            cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
                 <<", vsz = " <<vsz <<", LenVect[x] = " <<LenVect_[x] <<endl
                 <<"Contain = " <<Contain_ <<", Dim = " <<Dim_ <<endl;
          #endif
        }
        else if (Rpt3_)
        {
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain2_ = (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
          memset(Contain2_, 0, sizeof(int*) * vsz);
          Rpt3_[0] = Contain2_;
          
          for (x = 0; x <= Dim_; x++)
          {
            Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
            Contain2_[x][0] = 0;
          }
          
          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          x = cnt + Shift_;
          vsz = (x == 0 && off >= 5) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;
          Contain2_[0] = Contain_;
          Contain2_[1][0] = vsz;

          #if NESTARRAY_DEBUG3
            cout <<endl
                 <<"*** vect3 initialization stage 2c ***" <<endl;
            cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
                 <<", vsz = " <<vsz <<", LenVect[0] = " <<LenVect_[0] <<endl
                 <<"Contain2 = " <<Contain2_ <<", Dim = " <<Dim_ <<endl;
          #endif
        }

        ++cnt;
        x = cnt + Shift_;
        Vect2_ = Vect3_[x];
      }

      x = cnt + Shift_;
      max = vsz = (x == 0 && off >= 6) ? LenVect_[0]+1:LenVect_[x]+x+1;

      if (Dim_ >= 2 && max > 0)
        if (max < x && x > 2)
          max = x;
        else if (max < 2)
          max = Dim_ + 2;

      max *= sizeof(int*);

      if (!Vect2_)
      {
        Vect2_ = (off == 6) ?
        (int**)RawCallocateWith(MEMMATRIX, max):
        (off < 6) ? (Vect3_[x] = (int**)RawCallocateWith(MEMMATRIX, max)):
        NULL;

        if (Vect2_ && off == 6)
          memset(Vect2_, 0, max);

        #if NESTARRAY_DEBUG3
          cout <<endl
               <<"*** vect2 initialization stage 2a ***" <<endl;
          cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
               <<", vsz = " <<vsz <<", LenVect[x] = " <<LenVect_[x] <<endl
               <<"Vect2 = " <<Vect2_ <<", Dim = " <<Dim_ <<endl;
        #endif
      }

      if (Vect2_)
      {
        x = cnt + Shift_;
        max = vsz = (x == 0 && off >= 6) ? LenVect_[0]+1:LenVect_[x]+x+1;
        max *= sizeof(int*);
        
        if (x && off < 6)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          Contain_[0] = LenVect_[x];
          Vect2_[0] = Contain_;

          #if NESTARRAY_DEBUG3
            cout <<endl
                 <<"*** vect2 initialization stage 2b ***" <<endl;
            cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
                 <<", vsz = " <<vsz <<", LenVect[x] = " <<LenVect_[x] <<endl
                 <<"Contain = " <<Contain_ <<", Dim = " <<Dim_ <<endl;
          #endif
        }
        else if (Rpt2_)
        {
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          memset(Contain_, 0, sizeof(int) * vsz);
          Rpt2_[0] = Contain_;
          
          for (x = 0; x < vsz; x++)
            Contain_[x] = 0;

          x = cnt + Shift_;
          vsz = (x == 0 && off >= 6) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;

          #if NESTARRAY_DEBUG3
            cout <<endl
                 <<"*** vect2 initialization stage 2c ***" <<endl;
            cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
                 <<", vsz = " <<vsz <<", LenVect[0] = " <<LenVect_[0] <<endl
                 <<"Contain = " <<Contain_ <<", Dim = " <<Dim_ <<endl;
          #endif
        }

        ++cnt;
        x = cnt + Shift_;
        Vect1_ = Vect2_[x];
      }

      x = cnt + Shift_;
      sz = LenVect_[x] + 1;
      max = sz;

      if (max > 0 && max < 2)
        max = Dim_ + 2;

      max *= sizeof(int);

      if (!Vect1_)
      {
        x = cnt + Shift_;
        Vect1_ = Vect2_[x] = (int*)RawCallocateWith(MEMMATRIX, max);
        memset(Vect1_, 0, max);
        Vect1_[0] = sz;

        #if NESTARRAY_DEBUG3
          cout <<endl
               <<"*** vect1 initialization stage 2a ***" <<endl;
          cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
               <<", sz = " <<sz <<", LenVect[x] = " <<LenVect_[x] <<endl
               <<"Vect1 = " <<Vect1_ <<", Dim = " <<Dim_ <<endl;
        #endif
      }

      Array_ = (off == 0) ? (void*)Vect8_:
               (off == 1) ? (void*)Vect7_:
               (off == 2) ? (void*)Vect6_:
               (off == 3) ? (void*)Vect5_:
               (off == 4) ? (void*)Vect4_:
               (off == 5) ? (void*)Vect3_:
               (off == 6) ? (void*)Vect2_:NULL;

      if (Array_)
      {
        // Setting dealloc/sharedmem flags
        InitMatrix(Array_, Dim_+off, off, Index_+off, LenVect_);
      }
    }
    else
    {
      if (Dim_ == 2)
      {
        vsz = SizeOfVector(Dim_, 0, LenVect_);
        RangeValid_ = InRange(1, PosVect_[1], vsz - 1);
      
        if (RangeValid_)
          pos = PosVect_[1];
        else
          pos = 0;

        if (pos || !RangeValid_)
        {
          sz = SizeOfVector(Dim_, 1, LenVect_);
          if (sz < 2)
            return NULL;

          Vect2_ = (int**)Array_;

          if (!Vect2_[0])
          {
            Array_ = InitArrayData2(Array_, Dim_, 0, LenVect_, PosVect_, Error_);
            Vect2_ = (int**)Array_;
          }

          if (RangeValid_ && (pos < Vect2_[0][0] && Vect2_[pos]))
            Vect1_ = Vect2_[pos];
          else
          {
            newsz = vsz;
            Contain2_ = Vect2_;
            Vect1_ = Contain2_[0];
            oldsz = CapacityOfVector(Dim_, Index_, Vect1_, Contain2_);

            if (!RangeValid_)
            {
              if (oldsz)
                Vect2_ =
                (int**)RawReallocateWith(MEMMATRIX, Vect2_, sizeof(int*) * (oldsz+vsz));
              else
                Vect2_ =
                (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);

              vsz += oldsz;
              newsz = vsz;

              Vect2_[0][0] = newsz;
              Contain2_ = Vect2_;
              Vect1_ = Contain2_[0];
            }

            vsz = sz - 1;
            if (vsz < 2)
              vsz = 2;

            // Setting vector size and max and deallocation flag
            Vect1_ = Vect2_[pos] = (int*)InitArrayData2(Vect2_[pos], 1, vsz+1, LenVect_, PosVect_, Error_, 1);
            Vect1_[0] = vsz;
            Vect2_[0][pos] = 0;
                  
            if (Vect2_[0][0] < newsz)
              Vect2_[0][0] = newsz;

            Error_ = false;
          }
        }
        else
        {
          Error_ = true;
          return NULL;
        }
      }
      else
      {
        Vect8_ = (off == 0) ? (int********)Array_:NULL;
        Vect7_ = (off == 1) ? (int*******)Array_:NULL;
        Vect6_ = (off == 2) ? (int******)Array_:NULL;
        Vect5_ = (off == 3) ? (int*****)Array_:NULL;
        Vect4_ = (off == 4) ? (int****)Array_:NULL;
        Vect3_ = (off == 5) ? (int***)Array_:NULL;
        Vect2_ = (off == 6) ? (int**)Array_:NULL;
      
        sz = SizeOfVector(Dim_, Index_, LenVect_);
        if (sz < 2)
          return NULL;

        xDim_ = Dim_;
        Dim_ += off;
        xIndex_ = Index_;
        Index_ += off;

        for (x = 1; x < xDim_; x++)
        {
          xOff_ = x + off;
          vsz = SizeOfVector(xDim_, x-1, LenVect_);
          
          if ((RangeValid_ = InRange(x, PosVect_[x], vsz - 1)))
            pos = PosVect_[x];
          else
            pos = 0;

          rv = RangeValid_;
          newpos = PosVect_[x];

          switch (xOff_)
          {
            case 1:
              if (pos || !RangeValid_)
              {
                if (!Vect8_ || !Vect8_[0])
                {
                  Array_ = InitArrayData2(Array_, Dim_, 0, LenVect_, PosVect_, Error_);
                  Vect8_ = (int********)Array_;
                }
              
                if (Vect8_ && Vect8_[0])
                  InitVect_ = Vect8_[0][0][0][0][0][0];
                
                if (RangeValid_ && (pos < InitVect_[2][0] && Vect8_[pos]))
                  Vect7_ = Vect8_[pos];
                else
                {
                  Contain2_ = Vect8_[0][0][0][0][0][0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);

                  if (!RangeValid_ || LenVect_[x-1] > oldsz)
                  {
                    if (LenVect_[x-1] > oldsz)
                    {
                      vsz = LenVect_[x-1] - oldsz;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);
                    }
                  
                    if (oldsz)
                      Vect8_ =
                      (int********)RawReallocateWith(MEMMATRIX, Vect8_, sizeof(int*******) * (oldsz+vsz));
                    else
                      Vect8_ =
                      (int********)RawCallocateWith(MEMMATRIX, sizeof(int*******) * vsz);

                    InitVect_ = Vect8_[0][0][0][0][0][0];
                    InitVect_[2][0] = oldsz+vsz;
                    Contain2_ = Vect8_[0][0][0][0][0][0];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect8_[pos])
                  {                      
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect7_ = Vect8_[pos] = (int*******)InitArrayData2(Vect8_[pos], 7, 0, LenVect_, PosVect_, Error_, xDim_-7);
                  }
                  else
                    Vect7_ = Vect8_[pos];
                  
                  InitVect_[2][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect7_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);
                }
              }
              break;

            case 2:
              if (pos || !RangeValid_)
              {
                if (!Vect7_[0])
                {
                  Array_ = InitArrayData2(Array_, Dim_, 0, LenVect_, PosVect_, Error_);
                  Vect7_ = (int*******)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 7 <= 0)
                  {
                    incr = 0;
                    if (Vect7_ && Vect7_[0])
                      InitVect_ = Vect7_[0][0][0][0][0];
                  }
                }
                else
                {
                  incr = xDim_ - 7;
                  if (!incr)
                    InitVect_ = Vect7_[0][0][0][0][0];
                }
                
                if (RangeValid_ && (pos < InitVect_[2+incr][0] && Vect7_[pos]))
                {
                  Vect6_ = Vect7_[pos];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  SetRefArrays(Vect6_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_, true);
                }
                else
                {
                  Contain2_ = Vect7_[0][0][0][0][0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);

                  if (!RangeValid_ || LenVect_[x-1] > oldsz)
                  {
                    if (LenVect_[x-1] > oldsz)
                    {
                      vsz = LenVect_[x-1] - oldsz;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);
                    }
                    
                    if (oldsz)
                      Vect7_ =
                      (int*******)RawReallocateWith(MEMMATRIX, Vect7_, sizeof(int******) * (oldsz+vsz));
                    else
                      Vect7_ =
                      (int*******)RawCallocateWith(MEMMATRIX, sizeof(int******) * vsz);

                    if (xOff_ == Index_)                    
                    {
                      if (xDim_ - 7 <= 0)
                      {
                        incr = 0;
                        if (Vect7_ && Vect7_[0])
                          InitVect_ = Vect7_[0][0][0][0][0];
                      }
                    }
                    else
                    {
                      incr = xDim_ - 7;
                      if (!incr)
                        InitVect_ = Vect7_[0][0][0][0][0];
                    }

                    InitVect_[2+incr][0] = oldsz+vsz;
                    Contain2_ = Vect7_[0][0][0][0][0];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect7_[pos])
                  {
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect6_ = Vect7_[pos] = (int******)InitArrayData2(Vect7_[pos], 6, 0, LenVect_, PosVect_, Error_, xDim_-6);
                  }
                  else
                    Vect6_ = Vect7_[pos];

                  InitVect_[2+incr][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect6_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);
                }
              }
              break;
              
            case 3:
              if (pos || !RangeValid_)
              {
                if (!Vect6_[0])
                {
                  Array_ = InitArrayData2(Array_, Dim_, 0, LenVect_, PosVect_, Error_);
                  Vect6_ = (int******)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 6 <= 0)
                  {
                    incr = 0;
                    if (Vect6_ && Vect6_[0])
                      InitVect_ = Vect6_[0][0][0][0];
                  }
                }
                else
                {
                  incr = xDim_ - 6;
                  if (!incr)
                    InitVect_ = Vect6_[0][0][0][0];
                }
                
                if (RangeValid_ && (pos < InitVect_[2+incr][0] && Vect6_[pos]))
                {
                  Vect5_ = Vect6_[pos];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  SetRefArrays(Vect5_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_, true);
                }
                else
                {
                  Contain2_ = Vect6_[0][0][0][0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);

                  if (!RangeValid_ || LenVect_[x-1] > oldsz)
                  {
                    if (LenVect_[x-1] > oldsz)
                    {
                      vsz = LenVect_[x-1] - oldsz;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);
                    }
                    
                    if (oldsz)
                      Vect6_ =
                      (int******)RawReallocateWith(MEMMATRIX, Vect6_, sizeof(int*****) * (oldsz+vsz));
                    else
                      Vect6_ =
                      (int******)RawCallocateWith(MEMMATRIX, sizeof(int*****) * vsz);

                    if (xOff_ == Index_)
                    {
                      if (xDim_ - 6 <= 0)
                      {
                        incr = 0;
                        if (Vect6_ && Vect6_[0])
                          InitVect_ = Vect6_[0][0][0][0];
                      }
                    }
                    else
                    {
                      incr = xDim_ - 6;
                      if (!incr)
                        InitVect_ = Vect6_[0][0][0][0];
                    }

                    InitVect_[2+incr][0] = oldsz+vsz;
                    Contain2_ = Vect6_[0][0][0][0];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect6_[pos])                  
                  {
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect5_ = Vect6_[pos] = (int*****)InitArrayData2(Vect6_[pos], 5, 0, LenVect_, PosVect_, Error_, xDim_-5);
                  }
                  else
                    Vect5_ = Vect6_[pos];

                  InitVect_[2+incr][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect5_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);
                }
              }
              break;
            
            case 4:
              if (pos || !RangeValid_)
              {
                if (!Vect5_[0])
                {
                  Array_ = InitArrayData2(Array_, Dim_, 0, LenVect_, PosVect_, Error_);
                  Vect5_ = (int*****)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 5 <= 0)
                  {
                    incr = 0;
                    if (Vect5_ && Vect5_[0])
                      InitVect_ = Vect5_[0][0][0];
                  }
                }
                else
                {
                  incr = xDim_ - 5;
                  if (!incr)
                    InitVect_ = Vect5_[0][0][0];
                }
                
                if (RangeValid_ && (pos < InitVect_[2+incr][0] && Vect5_[pos]))
                {
                  Vect4_ = Vect5_[pos];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  SetRefArrays(Vect4_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_, true);
                }
                else
                {
                  Contain2_ = Vect5_[0][0][0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);

                  if (!RangeValid_ || LenVect_[x-1] > oldsz)
                  {
                    if (LenVect_[x-1] > oldsz)
                    {
                      vsz = LenVect_[x-1] - oldsz;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);
                    }
                    
                    if (oldsz)
                      Vect5_ =
                      (int*****)RawReallocateWith(MEMMATRIX, Vect5_, sizeof(int****) * (oldsz+vsz));
                    else
                      Vect5_ =
                      (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****) * vsz);

                    if (xOff_ == Index_)
                    {
                      if (xDim_ - 5 <= 0)
                      {
                        incr = 0;
                        if (Vect5_ && Vect5_[0])
                          InitVect_ = Vect5_[0][0][0];
                      }
                    }
                    else
                    {
                      incr = xDim_ - 5;
                      if (!incr)
                        InitVect_ = Vect5_[0][0][0];
                    }

                    InitVect_[2+incr][0] = oldsz+vsz;
                    Contain2_ = Vect5_[0][0][0];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect5_[pos])
                  {
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect4_ = Vect5_[pos] = (int****)InitArrayData2(Vect5_[pos], 4, 0, LenVect_, PosVect_, Error_, xDim_-4);
                  }
                  else
                    Vect4_ = Vect5_[pos];

                  InitVect_[2+incr][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect4_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);
                }
              }
              break;
            
            case 5:
              if (pos || !RangeValid_)
              {
                if (!Vect4_[0])
                {
                  Array_ = InitArrayData2(Array_, Dim_, 0, LenVect_, PosVect_, Error_);
                  Vect4_ = (int****)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 4 <= 0)
                  {
                    incr = 0;
                    if (Vect4_ && Vect4_[0])
                      InitVect_ = Vect4_[0][0];
                  }
                }
                else
                {
                  incr = xDim_ - 4;
                  if (!incr)
                    InitVect_ = Vect4_[0][0];
                }
                
                if (RangeValid_ && (pos < InitVect_[2+incr][0] && Vect4_[pos]))
                {
                  Vect3_ = Vect4_[pos];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  SetRefArrays(Vect3_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_, true);
                }
                else
                {
                  Contain2_ = Vect4_[0][0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);

                  if (!RangeValid_ || LenVect_[x-1] > oldsz)
                  {
                    if (LenVect_[x-1] > oldsz)
                    {
                      vsz = LenVect_[x-1] - oldsz;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);
                    }
                    
                    if (oldsz)
                      Vect4_ =
                      (int****)RawReallocateWith(MEMMATRIX, Vect4_, sizeof(int***) * (oldsz+vsz));
                    else
                      Vect4_ =
                      (int****)RawCallocateWith(MEMMATRIX, sizeof(int***) * vsz);

                    if (xOff_ == Index_)
                    {
                      if (xDim_ - 4 <= 0)
                      {
                        incr = 0;
                        if (Vect4_ && Vect4_[0])
                          InitVect_ = Vect4_[0][0];
                      }
                    }
                    else
                    {
                      incr = xDim_ - 4;
                      if (!incr)
                        InitVect_ = Vect4_[0][0];
                    }

                    InitVect_[2+incr][0] = oldsz+vsz;
                    Contain2_ = Vect4_[0][0];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect4_[pos])
                  {
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect3_ = Vect4_[pos] = (int***)InitArrayData2(Vect4_[pos], 3, 0, LenVect_, PosVect_, Error_, xDim_-3);
                  }
                  else
                    Vect3_ = Vect4_[pos];

                  InitVect_[2+incr][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect3_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);
                }
              }
              break;

            case 6:
              if (pos || !RangeValid_)
              {
                if (!Vect3_[0])
                {
                  Array_ = InitArrayData2(Array_, Dim_, 0, LenVect_, PosVect_, Error_);
                  Vect3_ = (int***)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 3 <= 0)
                  {
                    incr = 0;
                    if (Vect3_ && Vect3_[0])
                      InitVect_ = Vect3_[0];
                  }
                }
                else
                {
                  incr = xDim_ - 3;
                  if (!incr)
                    InitVect_ = Vect3_[0];
                }

                #if NESTARRAY_DEBUG3
                  cout <<endl
                       <<"*** vect3 indexing stage 1 ***" <<endl;
                  cout <<"Dim = " <<Dim_ <<", xOff = " <<xOff_ <<", Index = " <<Index_
                       <<", incr = " <<incr <<", Vect3[pos] = " <<Vect3_[pos] <<endl
                       <<"RangeValid = " <<RangeValid_
                       <<", InitVect[2+incr][0] = " <<InitVect_[2+incr][0]
                       <<", pos = " <<pos <<endl;
                #endif

                if (RangeValid_ && (pos < InitVect_[2+incr][0] && Vect3_[pos]))
                {
                  Vect2_ = Vect3_[pos];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  SetRefArrays(Vect2_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_, true);
                }                  
                else
                {
                  Contain2_ = Vect3_[0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);

                  if (!RangeValid_ || LenVect_[x-1] > oldsz)
                  {
                    if (LenVect_[x-1] > oldsz)
                    {
                      vsz = LenVect_[x-1] - oldsz;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);
                    }
                    
                    if (oldsz)
                      Vect3_ =
                      (int***)RawReallocateWith(MEMMATRIX, Vect3_, sizeof(int**) * (oldsz+vsz));
                    else
                      Vect3_ =
                      (int***)RawCallocateWith(MEMMATRIX, sizeof(int**) * vsz);

                    if (xOff_ == Index_)
                    {
                      if (xDim_ - 3 <= 0)
                      {
                        incr = 0;
                        if (Vect3_ && Vect3_[0])
                          InitVect_ = Vect3_[0];
                      }
                    }
                    else
                    {
                      incr = xDim_ - 3;
                      if (!incr)
                        InitVect_ = Vect3_[0];
                    }

                    InitVect_[2+incr][0] = oldsz+vsz;
                    Contain2_ = Vect3_[0];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect3_[pos])
                  {
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect2_ = Vect3_[pos] = (int**)InitArrayData2(Vect3_[pos], 2, 0, LenVect_, PosVect_, Error_, xDim_-2);
                  }
                  else
                    Vect2_ = Vect3_[pos];

                  InitVect_[2+incr][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect2_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);

                  #if NESTARRAY_DEBUG3
                    cout <<endl
                         <<"*** vect3 indexing stage 2 ***" <<endl;
                    cout <<"oldsz = " <<oldsz <<", vsz = " <<vsz <<", mIndex = " <<mIndex_
                         <<", xOff = " <<xOff_ <<", incr = " <<incr
                         <<", Vect3[pos] = " <<Vect3_[pos] <<endl
                         <<"xDim_ = " <<xDim_
                         <<", InitVect[2+incr][0] = " <<InitVect_[2+incr][0]
                         <<", Vect1[0] = " <<Vect1_[0]
                         <<", Contain[0] = " <<Contain_[0] <<endl;
                  #endif
                }
              }            
              break;

            case 7:
              if (pos || !RangeValid_)
              {
                if (!Vect2_ || !Vect2_[0])
                {
                  Array_ = InitArrayData2(Array_, Dim_, 0, LenVect_, PosVect_, Error_);
                  Vect2_ = (int**)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 2 <= 0)
                  {
                    incr = 0;
                    if (Vect2_ && Vect2_[0])
                      InitVect_ = Vect2_;
                  }
                  else if (xOff_ == xIndex_ + off)
                    incr = xDim_ - 2;
                }
                else
                {
                  incr = xDim_ - 2;
                  if (!incr)
                    InitVect_ = Vect2_;
                }

                #if NESTARRAY_DEBUG3
                  cout <<endl
                       <<"*** vect2 indexing stage 1 ***" <<endl;
                  cout <<"Dim = " <<Dim_ <<", xOff = " <<xOff_ <<", Index = " <<Index_
                       <<", incr = " <<incr <<", Vect2[pos] = " <<Vect2_[pos] <<endl
                       <<"RangeValid = " <<RangeValid_
                       <<", InitVect[2+incr][0] = " <<InitVect_[2+incr][0]
                       <<", pos = " <<pos
                       <<", Vect2[0][0] = " <<Vect2_[0][0] <<endl;
                #endif

                if (RangeValid_ &&
                    ((!incr && Vect2_[0][0] > pos) || pos < InitVect_[2+incr][0]) &&
                    Vect2_[pos])
                  Vect1_ = Vect2_[pos];
                else
                {
                  Contain2_ = Vect2_;
                  Vect1_ = Contain2_[0];
                  newsz = Vect1_[0];
                  oldsz = CapacityOfVector(Dim_, Index_, Vect1_, Contain2_);

                  if (!RangeValid_ || LenVect_[x-1] > oldsz)
                  {
                    if (LenVect_[x-1] > oldsz)
                    {
                      vsz = LenVect_[x-1] - oldsz;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);
                    }
                    
                    if (oldsz)
                      Vect2_ =
                      (int**)RawReallocateWith(MEMMATRIX, Vect2_, sizeof(int*) * (oldsz+vsz));
                    else
                      Vect2_ =
                      (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);

                    vsz += oldsz;
                    newsz = vsz;

                    if (xOff_ == Index_)
                    {
                      if (xDim_ - 2 <= 0)
                      {
                        incr = 0;
                        if (Vect2_ && Vect2_[0])
                          InitVect_ = Vect2_;
                      }
                      else if (xOff_ == xIndex_ + off)
                        incr = xDim_ - 2;
                    }
                    else
                    {
                      incr = xDim_ - 2;
                      if (!incr)
                        InitVect_ = Vect2_;
                    }

                    if (incr)
                      InitVect_[2+incr][0] = vsz;
                    else
                      InitVect_[0][0] = vsz;
                      
                    Contain2_ = Vect2_;
                    Vect1_ = Contain2_[0];
                    if (!oldsz)
                      Vect1_[0] = Vect1_[1] = 0;

                    if (!RangeValid_)
                      pos = newpos;
                  }

                  if (xDim_ >= 3)
                    Vect1_[0] = newsz;
                  else
                    Vect2_[0][0] = newsz;

                  // Setting vector size and max and deallocation flag
                  if (!RangeValid_ || !Vect2_[pos])
                  {
                    if (xDim_ >= 3)
                      Vect1_[1]++;
                      
                    Vect1_ = Vect2_[pos] = (int*)InitArrayData2(Vect2_[pos], 1, vsz+1, LenVect_, PosVect_, Error_, xDim_-1);
                  }
                  else
                    Vect1_ = Vect2_[pos];

                  vsz = sz - 1;
                  if (vsz < 2)
                    vsz = 2;

                  Vect1_[0] = vsz;
                  InitVect_[0][pos] = 0;

                  #if NESTARRAY_DEBUG3
                    cout <<endl
                         <<"*** vect2 indexing stage 2 ***" <<endl;
                    cout <<"oldsz = " <<oldsz <<", vsz = " <<vsz <<", Index = " <<Index_
                         <<", xOff = " <<xOff_ <<", incr = " <<incr
                         <<", Vect2[pos] = " <<Vect2_[pos] <<endl
                         <<"xDim_ = " <<xDim_
                         <<", newsz = " <<newsz
                         <<", Vect1[0] = " <<Vect1_[0]
                         <<", Vect2[0][0] = " <<Vect2_[0][0] <<endl;
                  #endif

                  if (incr && InitVect_[2+incr][0] < newsz)
                    InitVect_[2+incr][0] = newsz;
                }
              }
          }
        }

        Array_ = (off == 0) ? (void*)Vect8_:
                 (off == 1) ? (void*)Vect7_:
                 (off == 2) ? (void*)Vect6_:
                 (off == 3) ? (void*)Vect5_:
                 (off == 4) ? (void*)Vect4_:
                 (off == 5) ? (void*)Vect3_:
                 (off == 6) ? (void*)Vect2_:NULL;
      }
    }
  }
  else if (Dim_ == 1)
  {
    // In calling InitArrayData2 function for 1D vector,
    // Index argument is assumed to hold the vector length.
    // LenVect argument is assumed to hold deallocated flag and
    // PosVect argument is assumed to hold shallowcopy flag.
    
    max = sz = Index_;
    if (max > 0 && max < 2)
      max = 2;

    ++max;
    max *= sizeof(int);
    Vect1_ = (int*)RawCallocateWith(MEMMATRIX, max);
    memset(Vect1_, 0, max);
    Vect1_[0] = sz;
    Array_ = Vect1_;

    if (!Shift_)
    {
      if (LenVect_)
        *LenVect_ = 0;

      if (PosVect_)
        *PosVect_ = 0;
    }

    Error_ = false;
  }

  return Array_;
}

/****************************************************************************/
// PURPOSE:
//   Method to intialize general (void*) multidimensional data matrix
//
// PRE:
//   void* Array  : Data matrix to initialize
//   int Dim      : Dimensions of matrix
//   int Index    : Index position into matrix of arrays
//   int* LenVect : Length vector containing length of constituent arrays
//   int* PosVect : Position vector containing coordinates into matrix
//   bool& Error_ : Error in array indexing (out of range)
//
// POST:
//   Returns initialized data matrix of given dimensions and of given lengths
//
void* InitArrayData(void* Array_, int Dim_, int Index_, int* LenVect_,
                    int* PosVect_, int ReInit_, bool& Error_, int Shift_)
{
  if (!LenVect_ || !PosVect_)
    return Array_;

  int******** Vect8_ = NULL;
  int******* Vect7_ = NULL;
  int****** Vect6_ = NULL;
  int***** Vect5_ = NULL;
  int**** Vect4_ = NULL;
  int*** Vect3_ = NULL;
  int** Vect2_ = NULL;
  int* Vect1_ = NULL;

  int******** Rpt8_ = NULL;
  int******* Rpt7_ = NULL;
  int****** Rpt6_ = NULL;
  int***** Rpt5_ = NULL;
  int**** Rpt4_ = NULL;
  int*** Rpt3_ = NULL;
  int** Rpt2_ = NULL;
  int* Rpt1_ = NULL;

  bool RootChanged7_ = false;
  bool RootChanged6_ = false;
  bool RootChanged5_ = false;
  bool RootChanged4_ = false;
  bool RootChanged3_ = false;
  bool RootChanged2_ = false;

  int** InitVect_ = NULL;
  int** Contain2_ = NULL;
  int* Contain_ = NULL;

  int off = 8 - Dim_; // Maximum dimensions is: 8
  int pos;
  int newpos;
  int oldpos;
  int max;
  int sz;
  int vsz;
  int oldsz;
  int newsz;
  int addsz;
  int x;
  int cnt = 0;  
  int incr;

  int mIndex_;
  int xIndex_;
  int xDim_;
  int xOff_;

  bool RangeValid_;
  bool RangeCond_;
  bool rv;
  
  // Contain_[pos] = 1;    // deleted
  // Contain_[pos] = 2;    // shared
  // SizeOfVector(Dim_, x, LenVect_)
  // CountOfVector(Dim_, x, Off_, Vect1_, Contain_, Contain2_)
  // StatusOfVector(Dim_, x, Off_, y, Vect1_, Contain_)
  // SetStatusOfVector(Dim_, x, Off_, y, stat, Contain_)

  if (2 <= Dim_ && Dim_ <= 8 &&
      0 <= Index_ && Index_ < Dim_)
  {    
    x = cnt + Shift_;
    max = vsz = (x == 0) ? LenVect_[0]+1:LenVect_[x]+x+1;
    sz = SizeOfVector(Dim_, Index_, LenVect_);
    
    if (sz < 2)
      return NULL;

    if (Index_ == 0)
    {    
      Vect8_ = (off == 0) ?
      (int********)RawCallocateWith(MEMMATRIX, sizeof(int*******) * max):
      NULL;
      
      if (Vect8_ && off == 0)
        memset(Vect8_, 0, sizeof(int*******) * max);

      Vect7_ = (off == 1) ?
      (int*******)RawCallocateWith(MEMMATRIX, sizeof(int******) * max):
      (off < 1) ? (Vect8_[0] = (int*******)RawCallocateWith(MEMMATRIX, sizeof(int******))):
      NULL;

      if (Vect7_)
        if (off == 1)
          memset(Vect7_, 0, sizeof(int******) * max);
        else
          Vect7_[0] = 0;
      
      Vect6_ = (off == 2) ?
      (int******)RawCallocateWith(MEMMATRIX, sizeof(int*****) * max):
      (off < 2) ? (Vect7_[0] = (int******)RawCallocateWith(MEMMATRIX, sizeof(int*****))):
      NULL;
      
      if (Vect6_)
        if (off == 2)
          memset(Vect6_, 0, sizeof(int*****) * max);
        else
          Vect6_[0] = 0;

      Vect5_ = (off == 3) ?
      (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****) * max):
      (off < 3) ? (Vect6_[0] = (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****))):
      NULL;
      
      if (Vect5_)
        if (off == 3)
          memset(Vect5_, 0, sizeof(int****) * max);
        else
          Vect5_[0] = 0;

      Vect4_ = (off == 4) ?
      (int****)RawCallocateWith(MEMMATRIX, sizeof(int***) * max):
      (off < 4) ? (Vect5_[0] = (int****)RawCallocateWith(MEMMATRIX, sizeof(int***))):
      NULL;
      
      if (Vect4_)
        if (off == 4)
          memset(Vect4_, 0, sizeof(int***) * max);
        else
          Vect4_[0] = 0;

      Vect3_ = (off == 5) ?
      (int***)RawCallocateWith(MEMMATRIX, sizeof(int**) * max):
      (off < 5) ? (Vect4_[0] = (int***)RawCallocateWith(MEMMATRIX, sizeof(int**))):
      NULL;
      
      if (Vect3_)
        if (off == 5)
          memset(Vect3_, 0, sizeof(int**) * max);
        else
          Vect3_[0] = 0;

      #if NESTARRAY_DEBUG3
        cout <<endl
             <<"*** vect3 initialization stage 1 ***" <<endl;
        cout <<"off = " <<off <<", vsz = " <<vsz <<endl
             <<"Vect3 = " <<Vect3_ <<endl;
      #endif

      if (off < 6)
        max = 1;
        
      if (Dim_ >= 2 && max > 0)
        if (max < 2)
          max = Dim_ + 1;

      cnt = 0;
      vsz = max;
      max *= sizeof(int*);
      
      Vect2_ = (off == 6) ?
      (int**)RawCallocateWith(MEMMATRIX, max):
      (off < 6) ? (Vect3_[0] = (int**)RawCallocateWith(MEMMATRIX, max)):
      NULL;
      
      if (Vect2_)
        if (off == 6)
          memset(Vect2_, 0, max);
        else
          Vect2_[0] = 0;

      #if NESTARRAY_DEBUG3
        cout <<endl
             <<"*** vect2 initialization stage 1 ***" <<endl;
        cout <<"off = " <<off <<", vsz = " <<vsz <<endl
             <<"Vect2 = " <<Vect2_ <<endl;
      #endif

      vsz = max = Dim_ + 2;
      max *= sizeof(int);

      Vect1_ = Vect2_[0] = (int*)RawCallocateWith(MEMMATRIX, max);
      memset(Vect1_, 0, max);
      Vect1_[0] = vsz;

      #if NESTARRAY_DEBUG3
        cout <<endl
             <<"*** vect1 initialization stage 1 ***" <<endl;
        cout <<"off = " <<off <<", vsz = " <<vsz <<endl
             <<"Vect1 = " <<Vect1_ <<endl;
      #endif

      // Initializing array root points
      Rpt8_ = Vect8_;
      Rpt7_ = Vect7_;
      Rpt6_ = Vect6_;
      Rpt5_ = Vect5_;
      Rpt4_ = Vect4_;
      Rpt3_ = Vect3_;
      Rpt2_ = Vect2_;
      Rpt1_ = Vect1_;

      RootChanged7_ = false;
      RootChanged6_ = false;
      RootChanged5_ = false;
      RootChanged4_ = false;
      RootChanged3_ = false;
      RootChanged2_ = false;
      
      // array[(i+1)=1...x+i+1]*  = 1st starting TopArr value element.
      if (Rpt8_)
      {      
        vsz = LenVect_[0]+1;
        if (vsz < Dim_ + 1 || vsz < 2)
          vsz = Dim_ + 1;

        Contain2_ = (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
        Rpt8_[0][0][0][0][0][0] = Contain2_;
        
        for (x = 1; x <= Dim_; x++)
        {
          Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
          Contain2_[x][0] = 0;
        }

        if (vsz < Dim_ + 2)
          vsz = Dim_ + 2;

        Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
        Contain_[0] = LenVect_[0];
        Contain2_[0] = Contain_;
        Contain2_[1][0] = LenVect_[0];
                
        ++cnt;
        x = cnt + Shift_;
        vsz = LenVect_[0]+1;
        Vect7_ = (ReInit_ & NestedVectorEnums::RESIZE) ?
                    ((x < vsz) ? Vect8_[x]:NULL):
                    Vect8_[x];

        if (x >= vsz && (ReInit_ & NestedVectorEnums::RESIZE))
        {
          Rpt8_ = Vect8_ =
          (int********)RawReallocateWith(MEMMATRIX, Vect8_, sizeof(int*******) * (x+1));
          vsz = x+1;
          LenVect_[0] = vsz-1;
          Vect7_ = Vect8_[x];
        }
      }

      if (!Vect7_)
      {
        RootChanged7_ = false;
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 1) ? LenVect_[0]+1:LenVect_[x]+x+1;
        if (vsz < Dim_ + 1 || vsz < 2)
          vsz = Dim_ + 1;

        if (vsz < x && x > 2)
          vsz = x;
        else if (vsz < 2)
          vsz = Dim_ + 2;

        max = vsz;
        max *= sizeof(int******);
        Vect7_ = (off == 1) ?
        (int*******)RawCallocateWith(MEMMATRIX, max):
        (off < 1) ? (Vect8_[x] = (int*******)RawCallocateWith(MEMMATRIX, max)):
        NULL;

        if (Vect7_ && (ReInit_ & NestedVectorEnums::REINIT))
        {
          if (!Rpt7_ && (off == 1 || (Rpt8_ && off < 1 && x == 0)))
          {
            Rpt7_ = Vect7_;
            RootChanged7_ = true;
          }

          Vect7_[0] = (int******)RawCallocateWith(MEMMATRIX, sizeof(int*****));
          Vect7_[0][0] = (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****));
          Vect7_[0][0][0] = (int****)RawCallocateWith(MEMMATRIX, sizeof(int***));
          Vect7_[0][0][0][0] = (int***)RawCallocateWith(MEMMATRIX, sizeof(int**));
        
          Vect7_[0][0][0][0][0] = (int**)RawCallocateWith(MEMMATRIX, vsz * sizeof(int*));
          Vect7_[0][0][0][0][0][0] = NULL;
        }
      }

      // array[i=1...n][(i+1)=2...x+i+1]*  = 1st starting 2ndArr value element
      // InitArrayData(Vect7_[pos], 6, 0, LenVect_, PosVect_, Error_, xDim_-6);
      if ((Vect7_ && off < 1) || ((Rpt7_ || Vect7_) && off == 1))
      {
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 1) ? LenVect_[0]+1:LenVect_[x]+x+1;
            
        if (x && off < 1)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 2)
            vsz = Dim_ + 2;
        
          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          Contain_[0] = Contain_[1] = LenVect_[x];
          Vect7_[0] = (int******)CheckVector(Vect7_[0], 6, 0, 0, 0);
          Vect7_[0][0][0][0][0][0] = Contain_;
        }
        else if (Rpt7_ || Vect7_)
        {
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;
        
          Contain2_ = RootChanged7_ ? Vect7_[0][0][0][0][0]:
                                      (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
          if (Rpt7_)
            Rpt7_[0][0][0][0][0] = Contain2_;
          else if (Vect7_)
            Vect7_[0][0][0][0][0] = Contain2_;
          
          for (x = 1; x <= Dim_; x++)
          {
            Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
            Contain2_[x][0] = 0;
          }

          if (vsz < Dim_ + 2)
            vsz = Dim_ + 2;
            
          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          x = cnt + Shift_;
          vsz = (x == 0 && off >= 1) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;
          Contain2_[0] = Contain_;
          Contain2_[1][0] = vsz;
        }

        ++cnt;
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 1) ? LenVect_[0]+1:LenVect_[x]+x+1;
        Vect6_ = (ReInit_ & NestedVectorEnums::RESIZE) ?
                    ((x < vsz) ? Vect7_[x]:NULL):
                    Vect7_[x];

        if (x >= vsz && (ReInit_ & NestedVectorEnums::RESIZE))
        {
          RootChanged7_ = Rpt7_ == Vect7_;
          Vect7_ =
          (int*******)RawReallocateWith(MEMMATRIX, Vect7_, sizeof(int******) * (x+1));
          vsz = x+1;
          LenVect_[x] = (x == 0 && off >= 1) ? (vsz-1):x;
          Vect6_ = Vect7_[x];

          if (RootChanged7_)
            Rpt7_ = Vect7_;
        }
      }

      if (!Vect6_)
      {
        RootChanged6_ = false;
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 2) ? LenVect_[0]+1:LenVect_[x]+x+1;
        if (vsz < Dim_ + 1 || vsz < 2)
          vsz = Dim_ + 1;

        if (vsz < x && x > 2)
          vsz = x;
        else if (vsz < 2)
          vsz = Dim_ + 2;

        max = vsz;
        max *= sizeof(int*****);
        Vect6_ = (off == 2) ?
        (int******)RawCallocateWith(MEMMATRIX, max):
        (off < 2) ? (Vect7_[x] = (int******)RawCallocateWith(MEMMATRIX, max)):
        NULL;
        
        if (Vect6_ && (ReInit_ & NestedVectorEnums::REINIT))
        {
          if (!Rpt6_ && (off == 2 || (Rpt7_ && off < 2 && x == 0)))
          {
            Rpt6_ = Vect6_;
            RootChanged6_ = true;
          }

          Vect6_[0] = (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****));
          Vect6_[0][0] = (int****)RawCallocateWith(MEMMATRIX, sizeof(int***));
          Vect6_[0][0][0] = (int***)RawCallocateWith(MEMMATRIX, sizeof(int**));
          Vect6_[0][0][0][0] = NULL;

          Vect6_[0][0][0][0] = (int**)RawCallocateWith(MEMMATRIX, vsz * sizeof(int*));
          Vect6_[0][0][0][0][0] = NULL;
        }
      }

      // array[1...n]
      //      [i=2...n][(i+1)=3...x+i+1]* = 1st starting 3rdArr value element
      if ((Vect6_ && off < 2) || ((Rpt6_ || Vect6_) && off == 2))
      {
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 2) ? LenVect_[0]+1:LenVect_[x]+x+1;
        
        if (x && off < 2)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 2)
            vsz = Dim_ + 2;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          Contain_[0] = Contain_[1] = LenVect_[x];
          Vect6_[0] = (int*****)CheckVector(Vect6_[0], 5, 0, 0, 0);
          Vect6_[0][0][0][0][0] = Contain_;
        }
        else if (Rpt6_ || Vect6_)
        {
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain2_ = RootChanged6_ ? Vect6_[0][0][0][0]:
                                      (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
          if (Rpt6_)
            Rpt6_[0][0][0][0] = Contain2_;
          else if (Vect6_)
            Vect6_[0][0][0][0] = Contain2_;
          
          for (x = 1; x <= Dim_; x++)
          {
            Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
            Contain2_[x][0] = 0;
          }

          if (vsz < Dim_ + 2)
            vsz = Dim_ + 2;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          x = cnt + Shift_;
          vsz = (x == 0 && off >= 2) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;
          Contain2_[0] = Contain_;
          Contain2_[1][0] = vsz;
        }

        ++cnt;
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 2) ? LenVect_[0]+1:LenVect_[x]+x+1;
        Vect5_ = (ReInit_ & NestedVectorEnums::RESIZE) ?
                    ((x < vsz) ? Vect6_[x]:NULL):
                    Vect6_[x];

        if (x >= vsz && (ReInit_ & NestedVectorEnums::RESIZE))
        {
          RootChanged6_ = Rpt6_ == Vect6_;
          Vect6_ =
          (int******)RawReallocateWith(MEMMATRIX, Vect6_, sizeof(int*****) * (x+1));
          vsz = x+1;
          LenVect_[x] = (x == 0 && off >= 2) ? (vsz-1):x;
          Vect5_ = Vect6_[x];

          if (RootChanged6_)
            Rpt6_ = Vect6_;
        }
      }

      if (!Vect5_)
      {
        RootChanged5_ = false;
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 3) ? LenVect_[0]+1:LenVect_[x]+x+1;
        if (vsz < Dim_ + 1 || vsz < 2)
          vsz = Dim_ + 1;

        if (vsz < x && x > 2)
          vsz = x;
        else if (vsz < 2)
          vsz = Dim_ + 2;

        max = vsz;
        max *= sizeof(int****);
        Vect5_ = (off == 3) ?
        (int*****)RawCallocateWith(MEMMATRIX, max):
        (off < 3) ? (Vect6_[x] = (int*****)RawCallocateWith(MEMMATRIX, max)):
        NULL;

        if (Vect5_ && (ReInit_ & NestedVectorEnums::REINIT))
        {
          if (!Rpt5_ && (off == 3 || (Rpt6_ && off < 3 && x == 0)))
          {
            Rpt5_ = Vect5_;
            RootChanged5_ = true;
          }

          Vect5_[0] = (int****)RawCallocateWith(MEMMATRIX, sizeof(int***));
          Vect5_[0][0] = (int***)RawCallocateWith(MEMMATRIX, sizeof(int**));
          Vect5_[0][0][0] = NULL;

          Vect5_[0][0][0] = (int**)RawCallocateWith(MEMMATRIX, vsz * sizeof(int*));
          Vect5_[0][0][0][0] = NULL;
        }
      }

      // array[1...n]
      //      [2...n]
      //      [i=3...n][(i+1)=4...x+i+1]* = 1st starting 4thArr value element
      if ((Vect5_ && off < 3) || ((Rpt5_ || Vect5_) && off == 3))
      {
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 3) ? LenVect_[0]+1:LenVect_[x]+x+1;
      
        if (x && off < 3)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 2)
            vsz = Dim_ + 2;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          Contain_[0] = Contain_[1] = LenVect_[x];
          Vect5_[0] = (int****)CheckVector(Vect5_[0], 4, 0, 0, 0);
          Vect5_[0][0][0][0] = Contain_;
        }
        else if (Rpt5_ || Vect5_)
        {
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain2_ = RootChanged5_ ? Vect5_[0][0][0]:
                                      (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
          if (Rpt5_)
            Rpt5_[0][0][0] = Contain2_;
          else if (Vect5_)
            Vect5_[0][0][0] = Contain2_;
          
          for (x = 1; x <= Dim_; x++)
          {
            Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
            Contain2_[x][0] = 0;
          }

          if (vsz < Dim_ + 2)
            vsz = Dim_ + 2;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          x = cnt + Shift_;
          vsz = (x == 0 && off >= 3) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;
          Contain2_[0] = Contain_;
          Contain2_[1][0] = vsz;
        }

        ++cnt;
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 3) ? LenVect_[0]+1:LenVect_[x]+x+1;
        Vect4_ = (ReInit_ & NestedVectorEnums::RESIZE) ?
                    ((x < vsz) ? Vect5_[x]:NULL):
                    Vect5_[x];

        if (x >= vsz && (ReInit_ & NestedVectorEnums::RESIZE))
        {
          RootChanged5_ = Rpt5_ == Vect5_;
          Vect5_ =
          (int*****)RawReallocateWith(MEMMATRIX, Vect5_, sizeof(int****) * (x+1));
          vsz = x+1;
          LenVect_[x] = (x == 0 && off >= 3) ? (vsz-1):x;
          Vect4_ = Vect5_[x];

          if (RootChanged5_)
            Rpt5_ = Vect5_;
        }
      }

      if (!Vect4_)
      {
        RootChanged4_ = false;
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 4) ? LenVect_[0]+1:LenVect_[x]+x+1;
        if (vsz < Dim_ + 1 || vsz < 2)
          vsz = Dim_ + 1;

        if (vsz < x && x > 2)
          vsz = x;
        else if (vsz < 2)
          vsz = Dim_ + 2;          

        max = vsz;
        max *= sizeof(int***);
        Vect4_ = (off == 4) ?
        (int****)RawCallocateWith(MEMMATRIX, max):
        (off < 4) ? (Vect5_[x] = (int****)RawCallocateWith(MEMMATRIX, max)):
        NULL;

        if (Vect4_ && (ReInit_ & NestedVectorEnums::REINIT))
        {
          if (!Rpt4_ && (off == 4 || (Rpt5_ && off < 4 && x == 0)))
          {
            Rpt4_ = Vect4_;
            RootChanged4_ = true;
          }

          Vect4_[0] = (int***)RawCallocateWith(MEMMATRIX, sizeof(int**));
          Vect4_[0][0] = NULL;

          Vect4_[0][0] = (int**)RawCallocateWith(MEMMATRIX, vsz * sizeof(int*));
          Vect4_[0][0][0] = NULL;
        }
      }

      // array[1...n]
      //      [2...n]
      //      [3...n]
      //      [i=4...n][(i+1)=5...x+i+1]* = 1st starting 5thArr value element
      if ((Vect4_ && off < 4) || ((Rpt4_ || Vect4_) && off == 4))
      {
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 4) ? LenVect_[0]+1:LenVect_[x]+x+1;
            
        if (x && off < 4)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 2)
            vsz = Dim_ + 2;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          Contain_[0] = Contain_[1] = LenVect_[x];
          Vect4_[0] = (int***)CheckVector(Vect4_[0], 3, 0, 0, 0);
          Vect4_[0][0][0] = Contain_;
        }
        else if (Rpt4_ || Vect4_)
        {
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain2_ = RootChanged4_ ? Vect4_[0][0]:
                      (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
          if (Rpt4_)
            Rpt4_[0][0] = Contain2_;
          else if (Vect4_)
            Vect4_[0][0] = Contain2_;
          
          for (x = 1; x <= Dim_; x++)
          {
            Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
            Contain2_[x][0] = 0;
          }

          if (vsz < Dim_ + 2)
            vsz = Dim_ + 2;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          x = cnt + Shift_;
          vsz = (x == 0 && off >= 4) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;
          Contain2_[0] = Contain_;
          Contain2_[1][0] = vsz;
        }

        ++cnt;
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 4) ? LenVect_[0]+1:LenVect_[x]+x+1;
        Vect3_ = (ReInit_ & NestedVectorEnums::RESIZE) ?
                    ((x < vsz) ? Vect4_[x]:NULL):
                    Vect4_[x];

        if (x >= vsz && (ReInit_ & NestedVectorEnums::RESIZE))
        {
          RootChanged4_ = Rpt4_ == Vect4_;
          Vect4_ =
          (int****)RawReallocateWith(MEMMATRIX, Vect4_, sizeof(int***) * (x+1));
          vsz = x+1;
          LenVect_[x] = (x == 0 && off >= 4) ? (vsz-1):x;
          Vect3_ = Vect4_[x];

          if (RootChanged4_)
            Rpt4_ = Vect4_;
        }
      }

      if (!Vect3_)
      {
        RootChanged3_ = false;
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 5) ? LenVect_[0]+1:LenVect_[x]+x+1;
        if (vsz < Dim_ + 1 || vsz < 2)
          vsz = Dim_ + 1;

        if (vsz < x && x > 2)
          vsz = x;
        else if (vsz < 2)
          vsz = Dim_ + 2;

        max = vsz;
        max *= sizeof(int**);
        Vect3_ = (off == 5) ?
        (int***)RawCallocateWith(MEMMATRIX, max):
        (off < 5) ? (Vect4_[x] = (int***)RawCallocateWith(MEMMATRIX, max)):
        NULL;

        if (Vect3_ && (ReInit_ & NestedVectorEnums::REINIT))
        {
          if (!Rpt3_ && (off == 5 || (Rpt4_ && off < 5 && x == 0)))
          {
            Rpt3_ = Vect3_;
            RootChanged3_ = true;
          }

          Vect3_[0] = (int**)RawCallocateWith(MEMMATRIX, vsz * sizeof(int*));
          Vect3_[0][0] = NULL;

          #if NESTARRAY_DEBUG3
            cout <<endl
                 <<"*** vect3 initialization stage 2a ***" <<endl;
            cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
                 <<", vsz = " <<vsz <<", LenVect[x] = " <<LenVect_[x] <<endl
                 <<"Vect3 = " <<Vect3_ <<", Dim = " <<Dim_ <<endl;
          #endif
        }
      }

      // array[1...n]
      //      [2...n]
      //      [3...n]
      //      [4...n]
      //      [i=5...n][(i+1)=6...x+i+1]* = 1st starting 6thArr value element
      if ((Vect3_ && off < 5) || ((Rpt3_ || Vect3_) && off == 5))
      {
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 5) ? LenVect_[0]+1:LenVect_[x]+x+1;

        if (x && off < 5)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 2)
            vsz = Dim_ + 2;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          Contain_[0] = Contain_[1] = LenVect_[x];
          Vect3_[0] = (int**)CheckVector(Vect3_[0], 2, 0, 0, 0);
          Vect3_[0][0] = Contain_;

          #if NESTARRAY_DEBUG3
            cout <<endl
                 <<"*** vect3 initialization stage 2b ***" <<endl;
            cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
                 <<", vsz = " <<vsz <<", LenVect[x] = " <<LenVect_[x] <<endl
                 <<"Contain = " <<Contain_ <<", Dim = " <<Dim_ <<endl;
          #endif
        }
        else if (Rpt3_ || Vect3_)
        {
          if (vsz < Dim_ + 1)
            vsz = Dim_ + 1;

          Contain2_ = RootChanged3_ ? Vect3_[0]:
                                      (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
          if (Rpt3_)
            Rpt3_[0] = Contain2_;
          else if (Vect3_)
            Vect3_[0] = Contain2_;
          
          for (x = 1; x <= Dim_; x++)
          {
            Contain2_[x] = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * 2);
            Contain2_[x][0] = 0;
          }

          if (vsz < Dim_ + 2)
            vsz = Dim_ + 2;
          
          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          x = cnt + Shift_;
          vsz = (x == 0 && off >= 5) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;
          Contain2_[0] = Contain_;
          Contain2_[1][0] = vsz;

          #if NESTARRAY_DEBUG3
            cout <<endl
                 <<"*** vect3 initialization stage 2c ***" <<endl;
            cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
                 <<", vsz = " <<vsz <<", LenVect[0] = " <<LenVect_[0] <<endl
                 <<"Contain2 = " <<Contain2_ <<", Dim = " <<Dim_ <<endl;
          #endif
        }

        ++cnt;
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 5) ? LenVect_[0]+1:LenVect_[x]+x+1;
        Vect2_ = (ReInit_ & NestedVectorEnums::RESIZE) ?
                    ((x < vsz) ? Vect3_[x]:NULL):
                    Vect3_[x];

        if (x >= vsz && (ReInit_ & NestedVectorEnums::RESIZE))
        {
          RootChanged3_ = Rpt3_ == Vect3_;
          Vect3_ =
          (int***)RawReallocateWith(MEMMATRIX, Vect3_, sizeof(int**) * (x+1));
          vsz = x+1;
          LenVect_[x] = (x == 0 && off >= 5) ? (vsz-1):x;
          Vect2_ = Vect3_[x];

          if (RootChanged3_)
            Rpt3_ = Vect3_;
        }
      }

      x = cnt + Shift_;
      max = vsz = (x == 0 && off >= 6) ? LenVect_[0]+1:LenVect_[x]+x+1;

      if (Dim_ >= 2 && max > 0)
        if (max < x && x > 2)
          max = x;
        else if (max < 2)
          max = Dim_ + 2;

      max *= sizeof(int*);

      if (!Vect2_)
      {
        RootChanged2_ = false;
        Vect2_ = (off == 6) ?
        (int**)RawCallocateWith(MEMMATRIX, max):
        (off < 6) ? (Vect3_[x] = (int**)RawCallocateWith(MEMMATRIX, max)):
        NULL;

        if (Vect2_)
        {
          if (!Rpt2_ && (off == 6 || (Rpt3_ && off < 6 && x == 0)))
          {
            Rpt2_ = Vect2_;
            RootChanged2_ = true;
          }

          #if NESTARRAY_DEBUG3
            cout <<endl
                 <<"*** vect2 initialization stage 2a ***" <<endl;
            cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
                 <<", vsz = " <<vsz <<", LenVect[x] = " <<LenVect_[x] <<endl
                 <<"Vect2 = " <<Vect2_ <<", Dim = " <<Dim_ <<endl;
          #endif
        }
      }

      if ((Vect2_ && off < 6) || ((Rpt2_ || Vect2_) && off == 6))
      {
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 6) ? LenVect_[0]+1:LenVect_[x]+x+1;
        
        if (x && off < 6)
        {
          vsz = LenVect_[x]+x+1;
          if (vsz < Dim_ + 2 || vsz < 2)
            vsz = Dim_ + 2;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          Contain_[0] = LenVect_[x];
          Vect2_[0] = Contain_;

          #if NESTARRAY_DEBUG3
            cout <<endl
                 <<"*** vect2 initialization stage 2b ***" <<endl;
            cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
                 <<", vsz = " <<vsz <<", LenVect[x] = " <<LenVect_[x] <<endl
                 <<"Contain = " <<Contain_ <<", Dim = " <<Dim_ <<endl;
          #endif
        }
        else if (Rpt2_ || Vect2_)
        {
          if (vsz < Dim_ + 2 || vsz < 2)
            vsz = Dim_ + 2;

          Contain_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);
          if (Rpt2_)
            Rpt2_[0] = Contain_;
          else if (Vect2_)
            Vect2_[0] = Contain_;
          
          for (x = 0; x < vsz; x++)
            Contain_[x] = 0;

          x = cnt + Shift_;
          vsz = (x == 0 && off >= 6) ? LenVect_[0]:LenVect_[x];
          Contain_[0] = vsz;

          #if NESTARRAY_DEBUG3
            cout <<endl
                 <<"*** vect2 initialization stage 2c ***" <<endl;
            cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
                 <<", vsz = " <<vsz <<", LenVect[0] = " <<LenVect_[0] <<endl
                 <<"Contain = " <<Contain_ <<", Dim = " <<Dim_ <<endl;
          #endif
        }

        ++cnt;
        x = cnt + Shift_;
        vsz = (x == 0 && off >= 6) ? LenVect_[0]+1:LenVect_[x]+x+1;
        Vect1_ = (ReInit_ & NestedVectorEnums::RESIZE) ?
                    ((x < vsz) ? Vect2_[x]:NULL):
                    Vect2_[x];

        if (x >= vsz && (ReInit_ & NestedVectorEnums::RESIZE))
        {
          RootChanged2_ = Rpt2_ == Vect2_;
          Vect2_ =
          (int**)RawReallocateWith(MEMMATRIX, Vect2_, sizeof(int*) * (x+1));
          vsz = x+1;
          LenVect_[x] = (x == 0 && off >= 6) ? (vsz-1):x;
          Vect1_ = Vect2_[x];

          if (RootChanged2_)
            Rpt2_ = Vect2_;
        }
      }

      x = cnt + Shift_;
      sz = LenVect_[x] + 1;
      max = sz;

      if (max > 0 && max < 2)
        max = Dim_ + 2;

      max *= sizeof(int);

      if (!Vect1_)
      {
        x = cnt + Shift_;
        Vect1_ = Vect2_[x] = (int*)RawCallocateWith(MEMMATRIX, max);
        Vect1_[0] = sz;

        #if NESTARRAY_DEBUG3
          cout <<endl
               <<"*** vect1 initialization stage 2a ***" <<endl;
          cout <<"cnt = " <<cnt <<", Shift = " <<Shift_ <<", x = " <<x
               <<", sz = " <<sz <<", LenVect[x] = " <<LenVect_[x] <<endl
               <<"Vect1 = " <<Vect1_ <<", Dim = " <<Dim_ <<endl;
        #endif
      }

      Array_ = (off == 0) ? (void*)Vect8_:
               (off == 1) ? (void*)Vect7_:
               (off == 2) ? (void*)Vect6_:
               (off == 3) ? (void*)Vect5_:
               (off == 4) ? (void*)Vect4_:
               (off == 5) ? (void*)Vect3_:
               (off == 6) ? (void*)Vect2_:NULL;

      if (Array_)
      {
        // Setting dealloc/sharedmem flags
        InitMatrix(Array_, Dim_+off, off, Index_+off, LenVect_);
      }
    }
    else
    {
      if (Dim_ == 2)
      {
        vsz = SizeOfVector(Dim_, 0, LenVect_);
        RangeValid_ = InRange(1, PosVect_[1], vsz - 1);
      
        if (RangeValid_)
          pos = PosVect_[1];
        else
          pos = 0;

        if (pos || !RangeValid_)
        {
          sz = SizeOfVector(Dim_, 1, LenVect_);
          if (sz < 2)
            return NULL;

          Vect2_ = (int**)Array_;

          if (!Vect2_[0])
          {
            Array_ = InitArrayData(Array_, Dim_, 0, LenVect_, PosVect_, ReInit_, Error_);
            Vect2_ = (int**)Array_;
          }

          if (Vect2_)
          {
            RangeCond_ = RangeValid_ && Vect2_[0] && Vect2_[pos] && pos < Vect2_[0][0];
            Contain2_ = Vect2_;
            Vect1_ = RangeCond_ ? Contain2_[pos]:Contain2_[0];
            newsz = vsz;
            oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);
          }
          else
            RangeCond_ = false;

          if (RangeCond_ && pos < oldsz)
            Vect1_ = Vect2_[pos];
          else
          {
            newsz = vsz;
          
            if (!RangeValid_ || (pos >= Vect2_[0][0] || pos >= oldsz))
            {
              if (oldsz)
                Vect2_ =
                (int**)RawReallocateWith(MEMMATRIX, Vect2_, sizeof(int*) * (oldsz+vsz));
              else
                Vect2_ =
                (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);

              vsz += oldsz;
              newsz = vsz;

              Vect2_[0][0] = newsz;
              Contain2_ = Vect2_;
              Vect1_ = Contain2_[0];
            }

            vsz = sz - 1;
            if (vsz < 2)
              vsz = 2;

            // Setting vector size and max and deallocation flag
            Vect1_ = Vect2_[pos] = (int*)InitArrayData(Vect2_[pos], 1, vsz+1, LenVect_, PosVect_, ReInit_, Error_, 1);
            Vect1_[0] = vsz;
            Vect2_[0][pos] = 0;
                  
            if (Vect2_[0][0] < newsz)
              Vect2_[0][0] = newsz;

            Error_ = false;
          }
        }
        else
        {
          Error_ = true;
          return NULL;
        }
      }
      else
      {
        Vect8_ = (off == 0) ? (int********)Array_:NULL;
        Vect7_ = (off == 1) ? (int*******)Array_:NULL;
        Vect6_ = (off == 2) ? (int******)Array_:NULL;
        Vect5_ = (off == 3) ? (int*****)Array_:NULL;
        Vect4_ = (off == 4) ? (int****)Array_:NULL;
        Vect3_ = (off == 5) ? (int***)Array_:NULL;
        Vect2_ = (off == 6) ? (int**)Array_:NULL;
      
        sz = SizeOfVector(Dim_, Index_, LenVect_);
        if (sz < 2)
          return NULL;

        xDim_ = Dim_;
        Dim_ += off;
        xIndex_ = Index_;
        Index_ += off;

        for (x = 1; x < xDim_; x++)
        {
          xOff_ = x + off;
          vsz = SizeOfVector(xDim_, x-1, LenVect_);
          
          if ((RangeValid_ = InRange(x, PosVect_[x], vsz - 1)))
            pos = PosVect_[x];
          else
            pos = 0;

          rv = RangeValid_;
          newpos = PosVect_[x];

          switch (xOff_)
          {
            case 1:
              if (pos || !RangeValid_)
              {
                if (!Vect8_ || !Vect8_[0])
                {
                  Array_ = InitArrayData(Array_, xDim_, 0, LenVect_, PosVect_, ReInit_, Error_);
                  Vect8_ = (int********)Array_;
                }
              
                if (Vect8_ && Vect8_[0])
                {
                  InitVect_ = Vect8_[0][0][0][0][0][0];

                  RangeCond_ = RangeValid_ &&
                               (InitVect_ && pos < InitVect_[2][0] &&
                                Vect8_ && Vect8_[pos]);
                                
                  Contain2_ = RangeCond_ ? Vect8_[pos][0][0][0][0][0]:
                                           Vect8_[0][0][0][0][0][0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);
                }
                else
                  RangeCond_ = false;

                if (RangeCond_ && mIndex_ < oldsz)
                  Vect7_ = Vect8_[pos];
                else
                {
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  RangeCond_ = RangeCond_ && mIndex_ >= oldsz;
                
                  if (!RangeValid_ || LenVect_[x-1] > oldsz || RangeCond_)
                  {
                    if (RangeCond_ && RangeValid_)
                    {
                      vsz = (mIndex_ - oldsz) + 1;
                      oldpos = pos;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);

                      if (oldsz)
                        Vect7_ = Vect8_[pos] =
                        (int*******)RawReallocateWith(MEMMATRIX, Vect8_[pos], sizeof(int******) * (oldsz+vsz));
                      else
                        Vect7_ = Vect8_[pos] =
                        (int*******)RawCallocateWith(MEMMATRIX, sizeof(int******) * vsz);
                    }
                    else
                    {
                      Contain2_ = Vect8_[0][0][0][0][0][0];
                      Vect1_ = Contain2_[0];
                      oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);

                      if (LenVect_[x-1] > oldsz)
                      {
                        vsz = LenVect_[x-1] - oldsz;
                        RangeValid_ = rv ? (pos < oldsz + x):                        
                                           (newpos < oldsz + x);
                      }
                      
                      oldpos = 0;

                      if (oldsz)
                        Vect8_ =
                        (int********)RawReallocateWith(MEMMATRIX, Vect8_, sizeof(int*******) * (oldsz+vsz));
                      else
                        Vect8_ =
                        (int********)RawCallocateWith(MEMMATRIX, sizeof(int*******) * vsz);
                    }

                    InitVect_ = Vect8_[oldpos][0][0][0][0][0];
                    InitVect_[2][0] = oldsz+vsz;
                    Contain2_ = Vect8_[oldpos][0][0][0][0][0];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect8_[pos])
                  {                      
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect7_ = Vect8_[pos] = (int*******)InitArrayData(Vect8_[pos], 7, 0, LenVect_, PosVect_, ReInit_, Error_, xDim_-7);
                  }
                  else
                    Vect7_ = Vect8_[pos];
                  
                  InitVect_[2][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect7_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);
                }
              }
              break;

            case 2:
              if (pos || !RangeValid_)
              {
                if (!Vect7_ || !Vect7_[0])
                {
                  Array_ = InitArrayData(Array_, xDim_, 0, LenVect_, PosVect_, ReInit_, Error_);
                  Vect7_ = (int*******)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 7 <= 0)
                  {
                    incr = 0;
                    if (Vect7_ && Vect7_[0])
                      InitVect_ = Vect7_[0][0][0][0][0];
                  }
                  else if (xOff_ == xIndex_ + off)
                    incr = xDim_ - 7;
                }
                else
                {
                  incr = xDim_ - 7;
                  if (!incr)
                    InitVect_ = Vect7_[0][0][0][0][0];
                }

                if (Vect7_ && Vect7_[0])
                {
                  RangeCond_ = RangeValid_ &&
                               (InitVect_ && pos < InitVect_[2+incr][0] &&
                                Vect7_ && Vect7_[pos]);
                
                  Contain2_ = RangeCond_ ? Vect7_[pos][0][0][0][0]:
                                           Vect7_[0][0][0][0][0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);
                }
                else
                  RangeCond_ = false;
                
                if (RangeCond_ && mIndex_ < oldsz)
                {
                  Vect6_ = Vect7_[pos];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  SetRefArrays(Vect6_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_, true);
                }
                else
                {
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  RangeCond_ = RangeCond_ && mIndex_ >= oldsz;
                
                  if (!RangeValid_ || LenVect_[x-1] > oldsz || RangeCond_)
                  {
                    if (RangeCond_ && RangeValid_)
                    {
                      vsz = (mIndex_ - oldsz) + 1;
                      oldpos = pos;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);

                      if (oldsz)
                        Vect6_ = Vect7_[pos] =
                        (int******)RawReallocateWith(MEMMATRIX, Vect7_[pos], sizeof(int*****) * (oldsz+vsz));
                      else
                        Vect6_ = Vect7_[pos] =
                        (int******)RawCallocateWith(MEMMATRIX, sizeof(int*****) * vsz);
                    }
                    else
                    {
                      Contain2_ = Vect7_[0][0][0][0][0];
                      Vect1_ = Contain2_[0];
                      oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);

                      if (LenVect_[x-1] > oldsz)
                      {
                        vsz = LenVect_[x-1] - oldsz;
                        RangeValid_ = rv ? (pos < oldsz + x):
                                           (newpos < oldsz + x);
                      }

                      oldpos = 0;

                      if (oldsz)
                        Vect7_ =
                        (int*******)RawReallocateWith(MEMMATRIX, Vect7_, sizeof(int******) * (oldsz+vsz));
                      else
                        Vect7_ =
                        (int*******)RawCallocateWith(MEMMATRIX, sizeof(int******) * vsz);
                    }

                    if (xOff_ == Index_)                    
                    {
                      if (xDim_ - 7 <= 0)
                      {
                        incr = 0;
                        if (Vect7_ && Vect7_[oldpos])
                          InitVect_ = Vect7_[oldpos][0][0][0][0];
                      }
                      else if (xOff_ == xIndex_ + off)
                        incr = xDim_ - 7;
                    }
                    else
                    {
                      incr = xDim_ - 7;
                      if (!incr)
                        InitVect_ = Vect7_[oldpos][0][0][0][0];
                    }

                    InitVect_[2+incr][0] = oldsz+vsz;
                    Contain2_ = Vect7_[oldpos][0][0][0][0];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect7_[pos])
                  {
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect6_ = Vect7_[pos] = (int******)InitArrayData(Vect7_[pos], 6, 0, LenVect_, PosVect_, ReInit_, Error_, xDim_-6);
                  }
                  else
                    Vect6_ = Vect7_[pos];

                  InitVect_[2+incr][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect6_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);
                }
              }
              break;
              
            case 3:
              if (pos || !RangeValid_)
              {
                if (!Vect6_ || !Vect6_[0])
                {
                  Array_ = InitArrayData(Array_, xDim_, 0, LenVect_, PosVect_, ReInit_, Error_);
                  Vect6_ = (int******)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 6 <= 0)
                  {
                    incr = 0;
                    if (Vect6_ && Vect6_[0])
                      InitVect_ = Vect6_[0][0][0][0];
                  }
                  else if (xOff_ == xIndex_ + off)
                    incr = xDim_ - 6;
                }
                else
                {
                  incr = xDim_ - 6;
                  if (!incr)
                    InitVect_ = Vect6_[0][0][0][0];
                }

                if (Vect6_ && Vect6_[0])
                {
                  RangeCond_ = RangeValid_ &&
                               (InitVect_ && pos < InitVect_[2+incr][0] &&
                                Vect6_ && Vect6_[pos]);
                
                  Contain2_ = RangeCond_ ? Vect6_[pos][0][0][0]:
                                           Vect6_[0][0][0][0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);
                }
                else
                  RangeCond_ = false;
                
                if (RangeCond_ && mIndex_ < oldsz)
                {
                  Vect5_ = Vect6_[pos];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  SetRefArrays(Vect5_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_, true);
                }
                else
                {
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  RangeCond_ = RangeCond_ && mIndex_ >= oldsz;
                
                  if (!RangeValid_ || LenVect_[x-1] > oldsz || RangeCond_)
                  {
                    if (RangeCond_ && RangeValid_)
                    {
                      vsz = (mIndex_ - oldsz) + 1;
                      oldpos = pos;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);

                      if (oldsz)
                        Vect5_ = Vect6_[pos] =
                        (int*****)RawReallocateWith(MEMMATRIX, Vect6_[pos], sizeof(int****) * (oldsz+vsz));
                      else
                        Vect5_ = Vect6_[pos] =
                        (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****) * vsz);
                    }
                    else
                    {
                      Contain2_ = Vect6_[0][0][0][0];
                      Vect1_ = Contain2_[0];
                      oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);

                      if (LenVect_[x-1] > oldsz)
                      {
                        vsz = LenVect_[x-1] - oldsz;
                        RangeValid_ = rv ? (pos < oldsz + x):
                                           (newpos < oldsz + x);
                      }

                      oldpos = 0;
                    
                      if (oldsz)
                        Vect6_ =
                        (int******)RawReallocateWith(MEMMATRIX, Vect6_, sizeof(int*****) * (oldsz+vsz));
                      else
                        Vect6_ =
                        (int******)RawCallocateWith(MEMMATRIX, sizeof(int*****) * vsz);
                    }

                    if (xOff_ == Index_)
                    {
                      if (xDim_ - 6 <= 0)
                      {
                        incr = 0;
                        if (Vect6_ && Vect6_[oldpos])
                          InitVect_ = Vect6_[oldpos][0][0][0];
                      }
                      else if (xOff_ == xIndex_ + off)
                        incr = xDim_ - 6;
                    }
                    else
                    {
                      incr = xDim_ - 6;
                      if (!incr)
                        InitVect_ = Vect6_[oldpos][0][0][0];
                    }

                    InitVect_[2+incr][0] = oldsz+vsz;
                    Contain2_ = Vect6_[oldpos][0][0][0];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect6_[pos])                  
                  {
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect5_ = Vect6_[pos] = (int*****)InitArrayData(Vect6_[pos], 5, 0, LenVect_, PosVect_, ReInit_, Error_, xDim_-5);
                  }
                  else
                    Vect5_ = Vect6_[pos];

                  InitVect_[2+incr][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect5_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);
                }
              }
              break;
            
            case 4:
              if (pos || !RangeValid_)
              {
                if (!Vect5_ || !Vect5_[0])
                {
                  Array_ = InitArrayData(Array_, xDim_, 0, LenVect_, PosVect_, ReInit_, Error_);
                  Vect5_ = (int*****)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 5 <= 0)
                  {
                    incr = 0;
                    if (Vect5_ && Vect5_[0])
                      InitVect_ = Vect5_[0][0][0];
                  }
                  else if (xOff_ == xIndex_ + off)
                    incr = xDim_ - 5;
                }
                else
                {
                  incr = xDim_ - 5;
                  if (!incr)
                    InitVect_ = Vect5_[0][0][0];
                }

                if (Vect5_ && Vect5_[0])
                {
                  RangeCond_ = RangeValid_ &&
                               (InitVect_ && pos < InitVect_[2+incr][0] &&
                                Vect5_ && Vect5_[pos]);

                  Contain2_ = RangeCond_ ? Vect5_[pos][0][0]:
                                           Vect5_[0][0][0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);
                }
                else
                  RangeCond_ = false;
                
                if (RangeCond_ && mIndex_ < oldsz)
                {
                  Vect4_ = Vect5_[pos];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  SetRefArrays(Vect4_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_, true);
                }
                else
                {
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  RangeCond_ = RangeCond_ && mIndex_ >= oldsz;

                  if (!RangeValid_ || LenVect_[x-1] > oldsz || RangeCond_)
                  {
                    if (RangeCond_ && RangeValid_)
                    {
                      vsz = (mIndex_ - oldsz) + 1;
                      oldpos = pos;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);

                      if (oldsz)
                        Vect4_ = Vect5_[pos] =
                        (int****)RawReallocateWith(MEMMATRIX, Vect5_[pos], sizeof(int***) * (oldsz+vsz));
                      else
                        Vect4_ = Vect5_[pos] =
                        (int****)RawCallocateWith(MEMMATRIX, sizeof(int***) * vsz);
                    }
                    else
                    {
                      Contain2_ = Vect5_[0][0][0];
                      Vect1_ = Contain2_[0];
                      oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);

                      if (LenVect_[x-1] > oldsz)
                      {
                        vsz = LenVect_[x-1] - oldsz;
                        RangeValid_ = rv ? (pos < oldsz + x):
                                           (newpos < oldsz + x);
                      }

                      oldpos = 0;
                    
                      if (oldsz)
                        Vect5_ =
                        (int*****)RawReallocateWith(MEMMATRIX, Vect5_, sizeof(int****) * (oldsz+vsz));
                      else
                        Vect5_ =
                        (int*****)RawCallocateWith(MEMMATRIX, sizeof(int****) * vsz);
                    }

                    if (xOff_ == Index_)
                    {
                      if (xDim_ - 5 <= 0)
                      {
                        incr = 0;
                        if (Vect5_ && Vect5_[oldpos])
                          InitVect_ = Vect5_[oldpos][0][0];
                      }
                      else if (xOff_ == xIndex_ + off)
                        incr = xDim_ - 5;
                    }
                    else
                    {
                      incr = xDim_ - 5;
                      if (!incr)
                        InitVect_ = Vect5_[oldpos][0][0];
                    }

                    InitVect_[2+incr][0] = oldsz+vsz;
                    Contain2_ = Vect5_[oldpos][0][0];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect5_[pos])
                  {
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect4_ = Vect5_[pos] = (int****)InitArrayData(Vect5_[pos], 4, 0, LenVect_, PosVect_, ReInit_, Error_, xDim_-4);
                  }
                  else
                    Vect4_ = Vect5_[pos];

                  InitVect_[2+incr][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect4_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);
                }
              }
              break;
            
            case 5:
              if (pos || !RangeValid_)
              {
                if (!Vect4_ || !Vect4_[0])
                {
                  Array_ = InitArrayData(Array_, xDim_, 0, LenVect_, PosVect_, ReInit_, Error_);
                  Vect4_ = (int****)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 4 <= 0)
                  {
                    incr = 0;
                    if (Vect4_ && Vect4_[0])
                      InitVect_ = Vect4_[0][0];
                  }
                  else if (xOff_ == xIndex_ + off)
                    incr = xDim_ - 4;
                }
                else
                {
                  incr = xDim_ - 4;
                  if (!incr)
                    InitVect_ = Vect4_[0][0];
                }

                if (Vect4_ && Vect4_[0])
                {
                  RangeCond_ = RangeValid_ &&
                               (InitVect_ && pos < InitVect_[2+incr][0] &&
                                Vect4_ && Vect4_[pos]);

                  Contain2_ = RangeCond_ ? Vect4_[pos][0]:Vect4_[0][0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);
                }
                else
                  RangeCond_ = false;
                
                if (RangeCond_ && mIndex_ < oldsz)
                {
                  Vect3_ = Vect4_[pos];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  SetRefArrays(Vect3_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_, true);
                }
                else
                {
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  RangeCond_ = RangeCond_ && mIndex_ >= oldsz;

                  if (!RangeValid_ || LenVect_[x-1] > oldsz || RangeCond_)
                  {
                    if (RangeCond_ && RangeValid_)
                    {
                      vsz = (mIndex_ - oldsz) + 1;
                      oldpos = pos;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);

                      if (oldsz)
                        Vect3_ = Vect4_[pos] =
                        (int***)RawReallocateWith(MEMMATRIX, Vect4_[pos], sizeof(int**) * (oldsz+vsz));
                      else
                        Vect3_ = Vect4_[pos] =
                        (int***)RawCallocateWith(MEMMATRIX, sizeof(int**) * vsz);
                    }
                    else
                    {
                      Contain2_ = Vect4_[0][0];
                      Vect1_ = Contain2_[0];
                      oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);
                    
                      if (LenVect_[x-1] > oldsz)
                      {
                        vsz = LenVect_[x-1] - oldsz;
                        RangeValid_ = rv ? (pos < oldsz + x):
                                           (newpos < oldsz + x);
                      }

                      oldpos = 0;
                    
                      if (oldsz)
                        Vect4_ =
                        (int****)RawReallocateWith(MEMMATRIX, Vect4_, sizeof(int***) * (oldsz+vsz));
                      else
                        Vect4_ =
                        (int****)RawCallocateWith(MEMMATRIX, sizeof(int***) * vsz);
                    }

                    if (xOff_ == Index_)
                    {
                      if (xDim_ - 4 <= 0)
                      {
                        incr = 0;
                        if (Vect4_ && Vect4_[oldpos])
                          InitVect_ = Vect4_[oldpos][0];
                      }
                      else if (xOff_ == xIndex_ + off)
                        incr = xDim_ - 4;
                    }
                    else
                    {
                      incr = xDim_ - 4;
                      if (!incr)
                        InitVect_ = Vect4_[oldpos][0];
                    }

                    InitVect_[2+incr][0] = oldsz+vsz;
                    Contain2_ = Vect4_[oldpos][0];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect4_[pos])
                  {
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect3_ = Vect4_[pos] = (int***)InitArrayData(Vect4_[pos], 3, 0, LenVect_, PosVect_, ReInit_, Error_, xDim_-3);
                  }
                  else
                    Vect3_ = Vect4_[pos];

                  InitVect_[2+incr][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect3_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);
                }
              }
              break;

            case 6:
              if (pos || !RangeValid_)
              {
                if (!Vect3_ || !Vect3_[0])
                {
                  Array_ = InitArrayData(Array_, xDim_, 0, LenVect_, PosVect_, ReInit_, Error_);
                  Vect3_ = (int***)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 3 <= 0)
                  {
                    incr = 0;
                    if (Vect3_ && Vect3_[0])
                      InitVect_ = Vect3_[0];
                  }
                  else if (xOff_ == xIndex_ + off)
                    incr = xDim_ - 3;
                }
                else
                {
                  incr = xDim_ - 3;
                  if (!incr)
                    InitVect_ = Vect3_[0];
                }

                #if NESTARRAY_DEBUG3
                  cout <<endl
                       <<"*** vect3 indexing stage 1 ***" <<endl;
                  cout <<"Dim = " <<Dim_ <<", xOff = " <<xOff_ <<", Index = " <<Index_
                       <<", incr = " <<incr <<", Vect3[pos] = " <<Vect3_[pos] <<endl
                       <<"RangeValid = " <<RangeValid_
                       <<", InitVect[2+incr][0] = " <<InitVect_[2+incr][0]
                       <<", pos = " <<pos <<endl;
                #endif

                if (Vect3_ && Vect3_[0])
                {
                  RangeCond_ = RangeValid_ &&
                               (InitVect_ && pos < InitVect_[2+incr][0] &&
                                Vect3_ && Vect3_[pos]);
                       
                  Contain2_ = RangeCond_ ? Vect3_[pos]:Vect3_[0];
                  Vect1_ = Contain2_[0];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);
                }
                else
                  RangeCond_ = false;

                if (RangeCond_ && mIndex_ < oldsz)
                {
                  Vect2_ = Vect3_[pos];
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;

                  #if NESTARRAY_DEBUG3b
                    cout <<"Dim = " <<Dim_ <<", xOff = " <<xOff_ <<", Index = " <<Index_
                         <<", incr = " <<incr <<", Vect3[pos] = " <<Vect3_[pos]
                         <<", RangeValid = " <<RangeValid_ <<endl
                         <<"InitVect[2+incr][0] = " <<InitVect_[2+incr][0]
                         <<", pos = " <<pos <<", mIndex = " <<mIndex_
                         <<", off = " <<off <<", oldsz = " <<oldsz <<endl;
                  #endif
                  
                  SetRefArrays(Vect2_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_, true);
                }
                else
                {
                  mIndex_ = (xOff_ >= Index_) ? Index_:x+off;
                  RangeCond_ = RangeCond_ && mIndex_ >= oldsz;
                
                  if (!RangeValid_ || LenVect_[x-1] > oldsz || RangeCond_)
                  {
                    if (RangeCond_ && RangeValid_)
                    {
                      vsz = (mIndex_ - oldsz) + 1;
                      oldpos = pos;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);

                      if (oldsz)
                        Vect2_ = Vect3_[pos] =
                        (int**)RawReallocateWith(MEMMATRIX, Vect3_[pos], sizeof(int*) * (oldsz+vsz));
                      else
                        Vect2_ = Vect3_[pos] =
                        (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
                    }
                    else
                    {
                      Contain2_ = Vect3_[0];
                      Vect1_ = Contain2_[0];
                      oldsz = CapacityOfVector(Dim_, mIndex_, Vect1_, Contain2_);
                    
                      if (LenVect_[x-1] > oldsz)
                      {
                        vsz = LenVect_[x-1] - oldsz;
                        RangeValid_ = rv ? (pos < oldsz + x):
                                           (newpos < oldsz + x);
                      }

                      oldpos = 0;
                    
                      if (oldsz)
                        Vect3_ =
                        (int***)RawReallocateWith(MEMMATRIX, Vect3_, sizeof(int**) * (oldsz+vsz));
                      else
                        Vect3_ =
                        (int***)RawCallocateWith(MEMMATRIX, sizeof(int**) * vsz);
                    }

                    if (xOff_ == Index_)
                    {
                      if (xDim_ - 3 <= 0)
                      {
                        incr = 0;
                        if (Vect3_ && Vect3_[oldpos])
                          InitVect_ = Vect3_[oldpos];
                      }
                      else if (xOff_ == xIndex_ + off)
                        incr = xDim_ - 3;
                    }
                    else
                    {
                      incr = xDim_ - 3;
                      if (!incr)
                        InitVect_ = Vect3_[oldpos];
                    }

                    InitVect_[2+incr][0] = oldsz+vsz;
                    Contain2_ = Vect3_[oldpos];
                    Vect1_ = Contain2_[0];
                    newsz = oldsz+vsz;

                    if (!RangeValid_)
                      pos = newpos;
                  }
                  else
                    newsz = oldsz ? 0:vsz;

                  if (newsz)
                    SetCapacityOfVector(Dim_, mIndex_, newsz, Vect1_, Contain2_);

                  if (!RangeValid_ || !Vect3_[pos])
                  {
                    Vect1_ = Contain_ = Contain2_[0];
                    IncCountOfVector(Dim_, mIndex_, off, Vect1_, Contain_, Contain2_);
                    Vect2_ = Vect3_[pos] = (int**)InitArrayData(Vect3_[pos], 2, 0, LenVect_, PosVect_, ReInit_, Error_, xDim_-2);
                  }
                  else
                    Vect2_ = Vect3_[pos];

                  InitVect_[2+incr][0] = Vect1_[0];
                  InitVect_[0][pos] = 0;
                  SetRefArrays(Vect2_, Dim_, off, mIndex_, LenVect_, PosVect_, Error_);

                  #if NESTARRAY_DEBUG3
                    cout <<endl
                         <<"*** vect3 indexing stage 2 ***" <<endl;
                    cout <<"oldsz = " <<oldsz <<", vsz = " <<vsz <<", mIndex = " <<mIndex_
                         <<", xOff = " <<xOff_ <<", incr = " <<incr
                         <<", Vect3[pos] = " <<Vect3_[pos] <<endl
                         <<"xDim_ = " <<xDim_
                         <<", InitVect[2+incr][0] = " <<InitVect_[2+incr][0]
                         <<", Vect1[0] = " <<Vect1_[0]
                         <<", Contain[0] = " <<Contain_[0] <<endl;
                  #endif
                }
              }            
              break;

            case 7:
              if (pos || !RangeValid_)
              {
                if (!Vect2_ || !Vect2_[0])
                {
                  Array_ = InitArrayData(Array_, Dim_, 0, LenVect_, PosVect_, ReInit_, Error_);
                  Vect2_ = (int**)Array_;
                }
              
                if (xOff_ == Index_)
                {
                  if (xDim_ - 2 <= 0)
                  {
                    incr = 0;
                    if (Vect2_ && Vect2_[0])
                      InitVect_ = Vect2_;
                  }
                  else if (xOff_ == xIndex_ + off)
                    incr = xDim_ - 2;
                }
                else
                {
                  incr = xDim_ - 2;
                  if (!incr)
                    InitVect_ = Vect2_;
                }

                #if NESTARRAY_DEBUG3
                  cout <<endl
                       <<"*** vect2 indexing stage 1 ***" <<endl;
                  cout <<"Dim = " <<Dim_ <<", xOff = " <<xOff_ <<", Index = " <<Index_
                       <<", incr = " <<incr <<", Vect2[pos] = " <<Vect2_[pos] <<endl
                       <<"RangeValid = " <<RangeValid_
                       <<", InitVect[2+incr][0] = " <<InitVect_[2+incr][0]
                       <<", pos = " <<pos
                       <<", Vect2[0][0] = " <<Vect2_[0][0] <<endl;
                #endif

                if (Vect2_)                
                {
                  RangeCond_ = RangeValid_ &&
                               ((!incr && Vect2_ && Vect2_[0][0] > pos) ||
                                pos < InitVect_[2+incr][0]) &&
                               Vect2_ && Vect2_[pos];

                  Contain2_ = Vect2_;
                  Vect1_ = RangeCond_ ? Contain2_[pos]:Contain2_[0];
                  newsz = Vect1_[0];
                  mIndex_ = (PosVect_[xDim_] >= 0) ? PosVect_[xDim_]:Index_;
                  oldsz = CapacityOfVector(Dim_, Index_, Vect1_, Contain2_);
                }
                else
                  RangeCond_ = false;

                if (RangeCond_ && mIndex_ < oldsz)
                  Vect1_ = Vect2_[pos];
                else
                {
                  newsz = Vect1_[0];
                  mIndex_ = (PosVect_[xDim_] >= 0) ? PosVect_[xDim_]:Index_;
                  RangeCond_ = RangeCond_ && mIndex_ >= oldsz;
                
                  if (!RangeValid_ || LenVect_[x-1] > oldsz || RangeCond_)
                  {
                    if (RangeCond_ && RangeValid_)
                    {
                      vsz = (mIndex_ - oldsz) + 1;
                      oldpos = pos;
                      RangeValid_ = rv ? (pos < oldsz + x):
                                         (newpos < oldsz + x);

                      if (oldsz)
                        Vect1_ = Vect2_[pos] =
                        (int*)RawReallocateWith(MEMMATRIX, Vect2_[pos], sizeof(int) * (oldsz+vsz));
                      else
                        Vect1_ = Vect2_[pos] =
                        (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * vsz);

                      vsz += oldsz;
                      Vect1_[0] = vsz;
                      vsz = 0;
                      
                      Vect1_ = Contain2_[0];
                      newsz = Vect1_[0];
                      oldsz = CapacityOfVector(Dim_, Index_, Vect1_, Contain2_);
                    }
                    else
                    {
                      Vect1_ = Contain2_[0];
                      newsz = Vect1_[0];                    
                      oldsz = CapacityOfVector(Dim_, Index_, Vect1_, Contain2_);
                      
                      if (LenVect_[x-1] > oldsz)
                      {
                        vsz = LenVect_[x-1] - oldsz;
                        RangeValid_ = rv ? (pos < oldsz + x):
                                           (newpos < oldsz + x);
                      }

                      oldpos = 0;
                    
                      if (oldsz)
                        Vect2_ =
                        (int**)RawReallocateWith(MEMMATRIX, Vect2_, sizeof(int*) * (oldsz+vsz));
                      else
                        Vect2_ =
                        (int**)RawCallocateWith(MEMMATRIX, sizeof(int*) * vsz);
                    }

                    vsz += oldsz;
                    newsz = vsz;

                    if (xOff_ == Index_)
                    {
                      if (xDim_ - 2 <= 0)
                      {
                        incr = 0;
                        if (Vect2_ && Vect2_[oldpos])
                          InitVect_ = Vect2_;
                      }
                      else if (xOff_ == xIndex_ + off)
                        incr = xDim_ - 2;
                    }
                    else
                    {
                      incr = xDim_ - 2;
                      if (!incr)
                        InitVect_ = Vect2_;
                    }

                    if (incr)
                      InitVect_[2+incr][0] = vsz;
                    else
                      InitVect_[0][0] = vsz;
                      
                    Contain2_ = Vect2_;
                    Vect1_ = Contain2_[0];
                    if (!oldsz)
                      Vect1_[0] = Vect1_[1] = 0;

                    if (!RangeValid_)
                      pos = newpos;
                  }

                  if (xDim_ >= 3)
                    Vect1_[0] = newsz;
                  else
                    Vect2_[0][0] = newsz;

                  // Setting vector size and max and deallocation flag
                  if (!RangeValid_ || !Vect2_[pos])
                  {
                    if (xDim_ >= 3)
                      Vect1_[1]++;
                      
                    Vect1_ = Vect2_[pos] = (int*)InitArrayData(Vect2_[pos], 1, vsz+1, LenVect_, PosVect_, ReInit_, Error_, xDim_-1);
                  }
                  else
                    Vect1_ = Vect2_[pos];

                  vsz = sz - 1;
                  if (vsz < 2)
                    vsz = 2;

                  Vect1_[0] = vsz;
                  InitVect_[0][pos] = 0;

                  #if NESTARRAY_DEBUG3
                    cout <<endl
                         <<"*** vect2 indexing stage 2 ***" <<endl;
                    cout <<"oldsz = " <<oldsz <<", vsz = " <<vsz <<", Index = " <<Index_
                         <<", xOff = " <<xOff_ <<", incr = " <<incr
                         <<", Vect2[pos] = " <<Vect2_[pos] <<endl
                         <<"xDim_ = " <<xDim_
                         <<", newsz = " <<newsz
                         <<", Vect1[0] = " <<Vect1_[0]
                         <<", Vect2[0][0] = " <<Vect2_[0][0] <<endl;
                  #endif

                  if (incr && InitVect_[2+incr][0] < newsz)
                    InitVect_[2+incr][0] = newsz;
                }
              }
          }
        }

        Array_ = (off == 0) ? (void*)Vect8_:
                 (off == 1) ? (void*)Vect7_:
                 (off == 2) ? (void*)Vect6_:
                 (off == 3) ? (void*)Vect5_:
                 (off == 4) ? (void*)Vect4_:
                 (off == 5) ? (void*)Vect3_:
                 (off == 6) ? (void*)Vect2_:NULL;
      }
    }
  }
  else if (Dim_ == 1)
  {
    // In calling InitArrayData function for 1D vector,
    // Index argument is assumed to hold the vector length.
    // LenVect argument is assumed to hold deallocated flag and
    // PosVect argument is assumed to hold shallowcopy flag.
    
    max = sz = Index_;
    if (max > 0 && max < 2)
      max = 2;

    ++max;
    max *= sizeof(int);
    Vect1_ = (int*)RawCallocateWith(MEMMATRIX, max);
    Vect1_[0] = sz;
    Array_ = Vect1_;

    if (!Shift_)
    {
      if (LenVect_)
        *LenVect_ = 0;

      if (PosVect_)
        *PosVect_ = 0;
    }

    Error_ = false;
  }

  return Array_;
}

/****************************************************************************/
void* GiveInformationArray(void* Array_, int Dim_)
{
  // null pointer checks
  if (!Array_)
    return NULL;

  int******** Vect8_;
  int******* Vect7_;
  int****** Vect6_;
  int***** Vect5_;
  int**** Vect4_;
  int*** Vect3_;
  int** Vect2_;
  int* Vect1_;

  int off = 8 - Dim_; // Maximum dimensions is: 8
  int x;

  if (1 <= Dim_ && Dim_ <= 8)
  {
    if (Dim_ <= 2)
      return NULL;
    else
    {
      Vect8_ = (off == 0) ? (int********)Array_:NULL;
      Vect7_ = (off == 1) ? (int*******)Array_:NULL;
      Vect6_ = (off == 2) ? (int******)Array_:NULL;
      Vect5_ = (off == 3) ? (int*****)Array_:NULL;
      Vect4_ = (off == 4) ? (int****)Array_:NULL;
      Vect3_ = (off == 5) ? (int***)Array_:NULL;
      Vect2_ = (off == 6) ? (int**)Array_:NULL;
    
      if (Vect8_)
        return Vect8_[0][0][0][0][0][0];
      else if (Vect7_)
        return Vect7_[0][0][0][0][0];
      else if (Vect6_)
        return Vect6_[0][0][0][0];
      else if (Vect5_)
        return Vect5_[0][0][0];
      else if (Vect4_)
        return Vect4_[0][0];
      else if (Vect3_)
        return Vect3_[0];
      else if (Vect2_)
        return Vect2_;
    }
  }

  return Array_;
}

/****************************************************************************/
void* RemoveInformationArray(void* Array_, int Dim_)
{
  if (!Array_)
    return Array_;

  int******** Vect8_;
  int******* Vect7_;
  int****** Vect6_;
  int***** Vect5_;
  int**** Vect4_;
  int*** Vect3_;
  int** Vect2_;

  int** InfoArray_ = (int**)(::GiveInformationArray(Array_, Dim_));
  int off = 8 - Dim_; // Maximum dimensions is: 8
  int x;

  for (x = 0; x <= Dim_; x++)
  {
    ::DeleteArray(InfoArray_[x]);
    InfoArray_[x] = NULL;
  }

  ::DeleteArray(InfoArray_);
  InfoArray_ = NULL;

  if (1 <= Dim_ && Dim_ <= 8)
  {
    if (Dim_ <= 2)
      return NULL;
    else
    {
      Vect8_ = (off == 0) ? (int********)Array_:NULL;
      Vect7_ = (off == 1) ? (int*******)Array_:NULL;
      Vect6_ = (off == 2) ? (int******)Array_:NULL;
      Vect5_ = (off == 3) ? (int*****)Array_:NULL;
      Vect4_ = (off == 4) ? (int****)Array_:NULL;
      Vect3_ = (off == 5) ? (int***)Array_:NULL;
      Vect2_ = (off == 6) ? (int**)Array_:NULL;
    
      if (Vect8_)
      {
        Vect8_[0][0][0][0][0][0] = NULL;
        Array_ = Vect8_;
      }
      else if (Vect7_)
      {
        Vect7_[0][0][0][0][0] = NULL;
        Array_ = Vect7_;
      }
      else if (Vect6_)
      {
        Vect6_[0][0][0][0] = NULL;
        Array_ = Vect6_;
      }        
      else if (Vect5_)
      {
        Vect5_[0][0][0] = NULL;
        Array_ = Vect5_;
      }        
      else if (Vect4_)
      {
        Vect4_[0][0] = NULL;
        Array_ = Vect4_;
      }
      else if (Vect3_)
      {
        Vect3_[0] = NULL;
        Array_ = Vect3_;
      }
      else if (Vect2_)
      {
        Vect2_ = NULL;
        Array_ = Vect2_;
      }        
    }
  }

  return Array_;
}

/****************************************************************************/
/****************************************************************************/
template <class T>
IntVector<T>::IntVector():
_Deallocated(T(0)),
_ShallowCopy(T(0)),
_Error(false),
_Shared(T(0)),
_Owner(NULL),
_Array(NULL)
{}

/****************************************************************************/
template <class T>
IntVector<T>::IntVector(T Length_):
_Deallocated(T(0)),
_ShallowCopy(T(0)),
_Error(false),
_Shared(T(0)),
_Owner(NULL),
_Array(NULL)
{
  _Array = (T*)::InitArrayData(_Array, SIZE, Length_, &_Deallocated, &_ShallowCopy, true, _Error);
}

/****************************************************************************/
template <class T>
IntVector<T>::~IntVector()
{
  if (!_Shared)
    ::DestroyArrayData(_Array, 1, NULL);
  else if (_ShallowCopy && _Owner)
  {
    _Owner->DecrShare();
    _Owner = NULL;
  }
}

/****************************************************************************/
template <class T>
IntVector<T>& IntVector<T>::Assign(const IntVector& Obj_)
{
  if (this == &Obj_)
    return *this;

  int srcsz = Obj_.VectorSize();

  if (_Array && !_Deallocated && !_Shared)
  {
    if (srcsz > _Array[0])
    {
      T* NewArray_;
      NewArray_ = (T*)::InitArrayData(NewArray_, SIZE, srcsz, &_Deallocated, &_ShallowCopy, true, _Error);
      memmove(&NewArray_[1], Obj_.AtPos(0), srcsz * sizeof(T));
      NewArray_[0] = srcsz;

      ::DestroyArrayData(_Array, 1, NULL);        
      _Array = NewArray_;
    }
    else
    {
      memmove(&NewArray_[1], Obj_.AtPos(0), srcsz * sizeof(T));
      NewArray_[0] = srcsz;
    }
  }
  else if (!_Shared)
  {
    _Array = NULL;
    _Array = (T*)::InitArrayData(_Array, SIZE, srcsz, &_Deallocated, &_ShallowCopy, true, _Error);

    memmove(&_Array[1], Obj_.AtPos(0), srcsz * sizeof(T));
    _Array[0] = srcsz;
  }

  return *this;
}

/****************************************************************************/
template <class T>
IntVector<T>& IntVector<T>::ShallowCopy(IntVector& Obj_)
{
  if (this == &Obj_)
    return *this;

  if (_Array && !_Deallocated && !_Shared)
  {
    int srcsz = Obj_.VectorSize();
    ::DestroyArrayData(_Array, 1, NULL);    
    _Array = Obj_._Array;
    _Owner = &Obj_;
    _Owner->IncrShare();
    _ShallowCopy = true;
  }

  return *this;
}

/****************************************************************************/
template <class T>
void IntVector<T>::SetVectorSize(T sz)
{
  if (_Array && !_Deallocated && !_Shared)
  {
    int oldsz = _Array[0];
    _Array[0] = sz;

    if (sz > oldsz)
    {
      T* NewArray_;
      NewArray_ = (T*)::InitArrayData(NewArray_, SIZE, sz, &_Deallocated, &_ShallowCopy, true, _Error);
      memmove(NewArray_, _Array, (oldsz+1) * sizeof(T));
      NewArray_[0] = sz;
      
      ::DestroyArrayData(_Array, 1, NULL);
      _Array = NewArray_;
    }
  }
  else if (!_Shared)
  {
    _Array = NULL;
    _Array = (T*)::InitArrayData(_Array, SIZE, sz, &_Deallocated, &_ShallowCopy, true, _Error);
    _Array[0] = sz;
  }
}

/****************************************************************************/
template <class T>
T IntVector<T>::VectorSize() const
{
  return ((_Array && !_Deallocated) ? _Array[0]:T(0));
}

/****************************************************************************/
template <class T>
T IntVector<T>::ReadFromVector(T pos) const
{
  return ((_Array && !_Deallocated && 0 <= pos && pos < _Array[0]) ? _Array[pos+1]:T(0));
}

/****************************************************************************/
template <class T>
void* IntVector<T>::WriteToVector(T pos, T Data_)
{
  if (_Array && !_Deallocated && 0 <= pos && pos < _Array[0])
    _Array[pos+1] = Data_;

  return _Array;
}

/****************************************************************************/
template <class T>
T IntVector<T>::AtPos(T pos, bool& Found_)
{
  Found_ = (_Array && !_Deallocated && 0 <= pos && pos < _Array[0]);
  return (Found_ ? &_Array[pos+1]:0);
}

/****************************************************************************/
template <class T>
const T IntVector<T>::AtPos(T pos, bool& Found_) const
{
  Found_ = (_Array && !_Deallocated && 0 <= pos && pos < _Array[0]);
  return (Found_ ? &_Array[pos+1]:0);
}

/****************************************************************************/
template <class T>
bool IntVector<T>::Error() const
{
  return _Error;
}

/****************************************************************************/
template <class T>
T IntVector<T>::Shared() const
{
  return _Shared;
}

/****************************************************************************/
template <class T>
T IntVector<T>::Size() const
{
  return SIZE;
}

/****************************************************************************/
template <class T>
T IntVector<T>::MaxLength() const
{
  return SIZE;
}

/****************************************************************************/
template <class T>
T IntVector<T>::Dim() const
{
  return SIZE;
}

/****************************************************************************/
/****************************************************************************/
template <class T>
NestedArray2<T>::NestedArray2(T* Lengths_, int MaxLength_):
MAXLENGTH(MaxLength_ ? MaxLength_:SIZE),
_Array(NULL),
_MatrixIndex(0),
_LenVect(Lengths_ ?
           (T*)memmove(RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE), Lengths_, sizeof(T) * SIZE):
           NULL),
_PosVect((T*)memset(RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1)), 0, sizeof(T) * (SIZE+1))),
_ReInitArrays(REINIT|RESIZE),
_Error(false),
_Shared(0),
_Size(0),
_OffSets(NULL),
_OffSetsLen(SIZE*2)
{
  T x, cnt;
  for (x = 1; x < SIZE; x++)
    _PosVect[x] = x;

  _PosVect[0] = SIZE;
  _PosVect[SIZE] = 1;

  if (!Lengths_)
  {
    _LenVect = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
    
    for (x = 0; x < SIZE; x++)
      _LenVect[x] = MAXLENGTH;
  }

  _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);

  InitArrayData(0);
  InitArrayData(1);
}

/****************************************************************************/
template <class T>
NestedArray2<T>::~NestedArray2()
{
  ::DestroyArrayData(_Array, 2, _LenVect);
  
  ::DeleteArray(_LenVect);
  ::DeleteArray(_PosVect);
  ::DeleteArray(_OffSets);
}

/****************************************************************************/
template <class T>
T* NestedArray2<T>::NewLengthsVector() const
{
  T* Lengths_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
  return GetLengths(Lengths_);
}

/****************************************************************************/
template <class T>
T* NestedArray2<T>::NewCoordinatesVector() const
{
  T* Coords_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1));
  return GetCoordinates(Coords_);
}

/****************************************************************************/
template <class T>
T* NestedArray2<T>::GetCoordinates(T* Coords_) const
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;    
    for (x = 1; x < SIZE; x++)
      Coords_[x] = _PosVect[x] - x;

    Coords_[0] = SIZE;
    Coords_[SIZE] = _PosVect[SIZE] - 1;    
  }

  return Coords_;  
}

/****************************************************************************/
template <class T>
T* NestedArray2<T>::GetLengths(T* Lengths_) const
{
  if (Lengths_ && Lengths_ != _LenVect)
    memmove(Lengths_, _LenVect, sizeof(T) * SIZE);
    
  return Lengths_;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray2<T>::SetReInitArrays(int Flag_)
{
  _ReInitArrays = Flag_;
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray2<T>::SetCoordinates(T* Coords_)
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;
    for (x = 1; x < SIZE; x++)
      if (x + Coords_[x] < SizeOfVector(SIZE, x-1, _LenVect))
        _PosVect[x] = x + Coords_[x];

    if (1 + Coords_[SIZE] < SizeOfVector(SIZE, SIZE-1, _LenVect))
      _PosVect[SIZE] = 1 + Coords_[SIZE];

    _PosVect[0] = SIZE;
    InitArrayData(1);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray2<T>::SetLengths(T* Lengths_)
{
  if (Lengths_ && Lengths_ != _LenVect)
  {
    memmove(_LenVect, Lengths_, sizeof(T) * SIZE);

    ::DeleteArray(_OffSets);
    _OffSets = NULL;
    _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);
  
    InitArrayData(1);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
void* NestedArray2<T>::InitArrayData(T Index_)
{
  _MatrixIndex = Index_;
  _Array = (T**)::InitArrayData(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _ReInitArrays, _Error);
  // _Array = (T**)::InitArrayData2(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _Error);
  return _Array;
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::DeleteVector(T Index_)
{
  _MatrixIndex = Index_;
  return ::DeleteVector(_Array, 2, _MatrixIndex, _LenVect, _PosVect,
                        _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::IsDeallocated(T Index_)
{
  _MatrixIndex = Index_;
  return ::IsDeallocated(_Array, 2, _MatrixIndex, _LenVect, _PosVect,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::ReplaceVector(T Index_, void* Replacement_, bool Replace_,
                                bool DeAlloc_)
{
  _MatrixIndex = Index_;
  return ::ReplaceVector(_Array, 2, _MatrixIndex, _LenVect, _PosVect,
                         Replacement_, Replace_, DeAlloc_,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::RealVectorCount(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorCount(_Array, 2, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::RealVectorSize(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorMax(_Array, 2, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T** NestedArray2<T>::GiveInfoArray()
{
  return NULL;
}

/****************************************************************************/
template <class T>
const T* NestedArray2<T>::ParentReference(T Index_)
{
  return NULL;
}

/****************************************************************************/
template <class T>
T* NestedArray2<T>::LinearToCoords(T Pos_, T* Coords_)
{
  return ::LinearToCoords(Pos_, Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::CoordsToLinear(T* Coords_)
{
  return ::CoordsToLinear(Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::ReadFromArray() const
{
  return ::ReadFromArray(_Array, 2, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
void* NestedArray2<T>::WriteToArray(T Data_)
{
  return ::WriteToArray(_Array, 2, _LenVect, _PosVect, Data_, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::ReadFromVector(T pos) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray2<T>* This_ = (NestedArray2<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 2, _LenVect, _PosVect, _Error);

  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return ret;
}

/****************************************************************************/
template <class T>
void* NestedArray2<T>::WriteToVector(T pos, T Data_)
{
  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  
  return WriteToArray(Data_);
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::AtPos(T pos, bool& Found_)
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 2, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
const T NestedArray2<T>::AtPos(T pos, bool& Found_) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray6<T>* This_ = (NestedArray6<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 2, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
int NestedArray2<T>::ReInitArrays() const
{
  return _ReInitArrays;
}

/****************************************************************************/
template <class T>
bool NestedArray2<T>::Error() const
{
  return _Error;
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::Shared() const
{
  return _Shared;
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::Size() const
{
  return SIZE;
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::MaxLength() const
{
  return MAXLENGTH;
}

/****************************************************************************/
template <class T>
T NestedArray2<T>::Dim() const
{
  return SIZE;
}

/****************************************************************************/
/****************************************************************************/
template <class T>
NestedArray3<T>::NestedArray3(T* Lengths_, int MaxLength_):
MAXLENGTH(MaxLength_ ? MaxLength_:SIZE),
_Array(NULL),
_MatrixIndex(0),
_LenVect(Lengths_ ?
           (T*)memmove(RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE), Lengths_, sizeof(T) * SIZE):
           NULL),
_PosVect((T*)memset(RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1)), 0, sizeof(T) * (SIZE+1))),
_ParentRefs((T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE)),
_ReInitArrays(REINIT|RESIZE),
_Error(false),
_Shared(0),
_Size(0),
_OffSets(NULL),
_OffSetsLen(SIZE*2)
{
  T x, cnt;
  for (x = 1; x < SIZE; x++)
    _PosVect[x] = x;

  _PosVect[0] = SIZE;
  _PosVect[SIZE] = 1;

  if (!Lengths_)
  {
    _LenVect = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
    
    for (x = 0; x < SIZE; x++)
      _LenVect[x] = MAXLENGTH;
  }

  _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);

  InitArrayData(0);
  InitArrayData(1);
  InitArrayData(2);
}

/****************************************************************************/
template <class T>
NestedArray3<T>::~NestedArray3()
{
  ::RemoveInformationArray(_Array, 3);
  ::DestroyArrayData(_Array, 3, _LenVect);
  
  ::DeleteArray(_LenVect);
  ::DeleteArray(_PosVect);
  ::DeleteArray(_ParentRefs);
  ::DeleteArray(_OffSets);
}

/****************************************************************************/
template <class T>
T* NestedArray3<T>::NewLengthsVector() const
{
  T* Lengths_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
  return GetLengths(Lengths_);
}

/****************************************************************************/
template <class T>
T* NestedArray3<T>::NewCoordinatesVector() const
{
  T* Coords_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1));
  return GetCoordinates(Coords_);
}

/****************************************************************************/
template <class T>
T* NestedArray3<T>::GetCoordinates(T* Coords_) const
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;    
    for (x = 1; x < SIZE; x++)
      Coords_[x] = _PosVect[x] - x;

    Coords_[0] = SIZE;
    Coords_[SIZE] = _PosVect[SIZE] - 1;    
  }

  return Coords_;  
}

/****************************************************************************/
template <class T>
T* NestedArray3<T>::GetLengths(T* Lengths_) const
{
  if (Lengths_ && Lengths_ != _LenVect)
    memmove(Lengths_, _LenVect, sizeof(T) * SIZE);
    
  return Lengths_;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray3<T>::SetReInitArrays(int Flag_)
{
  _ReInitArrays = Flag_;
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray3<T>::SetCoordinates(T* Coords_)
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;
    for (x = 1; x < SIZE; x++)
      if (x + Coords_[x] < SizeOfVector(SIZE, x-1, _LenVect))
        _PosVect[x] = x + Coords_[x];

    if (1 + Coords_[SIZE] < SizeOfVector(SIZE, SIZE-1, _LenVect))
      _PosVect[SIZE] = 1 + Coords_[SIZE];

    _PosVect[0] = SIZE;      
    InitArrayData(1);
    InitArrayData(2);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray3<T>::SetLengths(T* Lengths_)
{
  if (Lengths_ && Lengths_ != _LenVect)
  {
    memmove(_LenVect, Lengths_, sizeof(T) * SIZE);

    ::DeleteArray(_OffSets);
    _OffSets = NULL;
    _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);
  
    InitArrayData(1);
    InitArrayData(2);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
void* NestedArray3<T>::InitArrayData(T Index_)
{
  _MatrixIndex = Index_;
  _Array = (T***)::InitArrayData(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _ReInitArrays, _Error);
  // _Array = (T***)::InitArrayData2(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _Error);
  return _Array;
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::DeleteVector(T Index_)
{
  _MatrixIndex = Index_;
  return ::DeleteVector(_Array, 3, _MatrixIndex, _LenVect, _PosVect,
                        _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::IsDeallocated(T Index_)
{
  _MatrixIndex = Index_;
  return ::IsDeallocated(_Array, 3, _MatrixIndex, _LenVect, _PosVect,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::ReplaceVector(T Index_, void* Replacement_, bool Replace_,
                                bool DeAlloc_)
{
  _MatrixIndex = Index_;
  return ::ReplaceVector(_Array, 3, _MatrixIndex, _LenVect, _PosVect,
                         Replacement_, Replace_, DeAlloc_,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::RealVectorCount(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorCount(_Array, 3, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::RealVectorSize(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorMax(_Array, 3, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T** NestedArray3<T>::GiveInfoArray()
{
  T** retval = (T**)(::GiveInformationArray(_Array, 3));
  return retval;
}

/****************************************************************************/
template <class T>
const T* NestedArray3<T>::ParentReference(T Index_)
{
  _MatrixIndex = Index_;
  _ParentRefs = (T*)(::ParentReference(_Array, 3, _MatrixIndex, _LenVect,
                                         _PosVect, _ParentRefs, _Error));
  return _ParentRefs;
}

/****************************************************************************/
template <class T>
T* NestedArray3<T>::LinearToCoords(T Pos_, T* Coords_)
{
  return ::LinearToCoords(Pos_, Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::CoordsToLinear(T* Coords_)
{
  return ::CoordsToLinear(Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::ReadFromArray() const
{
  return ::ReadFromArray(_Array, 3, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
void* NestedArray3<T>::WriteToArray(T Data_)
{
  return ::WriteToArray(_Array, 3, _LenVect, _PosVect, Data_, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::ReadFromVector(T pos) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray3<T>* This_ = (NestedArray3<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 3, _LenVect, _PosVect, _Error);

  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return ret;
}

/****************************************************************************/
template <class T>
void* NestedArray3<T>::WriteToVector(T pos, T Data_)
{
  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  
  return WriteToArray(Data_);
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::AtPos(T pos, bool& Found_)
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 3, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
const T NestedArray3<T>::AtPos(T pos, bool& Found_) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray6<T>* This_ = (NestedArray6<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 3, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
int NestedArray3<T>::ReInitArrays() const
{
  return _ReInitArrays;
}

/****************************************************************************/
template <class T>
bool NestedArray3<T>::Error() const
{
  return _Error;
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::Shared() const
{
  return _Shared;
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::Size() const
{
  return SIZE;
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::MaxLength() const
{
  return MAXLENGTH;
}

/****************************************************************************/
template <class T>
T NestedArray3<T>::Dim() const
{
  return SIZE;
}

/****************************************************************************/
/****************************************************************************/
template <class T>
NestedArray4<T>::NestedArray4(T* Lengths_, int MaxLength_):
MAXLENGTH(MaxLength_ ? MaxLength_:SIZE),
_Array(NULL),
_MatrixIndex(0),
_LenVect(Lengths_ ?
           (T*)memmove(RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE), Lengths_, sizeof(T) * SIZE):
           NULL),
_PosVect((T*)memset(RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1)), 0, sizeof(T) * (SIZE+1))),
_ParentRefs((T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE)),
_ReInitArrays(REINIT|RESIZE),
_Error(false),
_Shared(0),
_Size(0),
_OffSets(NULL),
_OffSetsLen(SIZE*2)
{
  T x, cnt;
  for (x = 1; x < SIZE; x++)
    _PosVect[x] = x;

  _PosVect[0] = SIZE;
  _PosVect[SIZE] = 1;

  if (!Lengths_)
  {
    _LenVect = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
    
    for (x = 0; x < SIZE; x++)
      _LenVect[x] = MAXLENGTH;
  }

  _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);

  InitArrayData(0);
  InitArrayData(1);
  InitArrayData(2);
  InitArrayData(3);
}

/****************************************************************************/
template <class T>
NestedArray4<T>::~NestedArray4()
{
  ::RemoveInformationArray(_Array, 4);
  ::DestroyArrayData(_Array, 4, _LenVect);
  
  ::DeleteArray(_LenVect);
  ::DeleteArray(_PosVect);
  ::DeleteArray(_ParentRefs);
  ::DeleteArray(_OffSets);
}

/****************************************************************************/
template <class T>
T* NestedArray4<T>::NewLengthsVector() const
{
  T* Lengths_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
  return GetLengths(Lengths_);
}

/****************************************************************************/
template <class T>
T* NestedArray4<T>::NewCoordinatesVector() const
{
  T* Coords_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1));
  return GetCoordinates(Coords_);
}

/****************************************************************************/
template <class T>
T* NestedArray4<T>::GetCoordinates(T* Coords_) const
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;    
    for (x = 1; x < SIZE; x++)
      Coords_[x] = _PosVect[x] - x;

    Coords_[0] = SIZE;
    Coords_[SIZE] = _PosVect[SIZE] - 1;    
  }

  return Coords_;  
}

/****************************************************************************/
template <class T>
T* NestedArray4<T>::GetLengths(T* Lengths_) const
{
  if (Lengths_ && Lengths_ != _LenVect)
    memmove(Lengths_, _LenVect, sizeof(T) * SIZE);
    
  return Lengths_;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray4<T>::SetReInitArrays(int Flag_)
{
  _ReInitArrays = Flag_;
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray4<T>::SetCoordinates(T* Coords_)
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;
    for (x = 1; x < SIZE; x++)
      if (x + Coords_[x] < SizeOfVector(SIZE, x-1, _LenVect))
        _PosVect[x] = x + Coords_[x];

    if (1 + Coords_[SIZE] < SizeOfVector(SIZE, SIZE-1, _LenVect))
      _PosVect[SIZE] = 1 + Coords_[SIZE];

    _PosVect[0] = SIZE;      
    InitArrayData(1);
    InitArrayData(2);
    InitArrayData(3);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray4<T>::SetLengths(T* Lengths_)
{
  if (Lengths_ && Lengths_ != _LenVect)
  {
    memmove(_LenVect, Lengths_, sizeof(T) * SIZE);

    ::DeleteArray(_OffSets);
    _OffSets = NULL;
    _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);
  
    InitArrayData(1);
    InitArrayData(2);
    InitArrayData(3);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
void* NestedArray4<T>::InitArrayData(T Index_)
{
  _MatrixIndex = Index_;
  // _Array = (T****)::InitArrayData(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _ReInitArrays, _Error);
  _Array = (T****)::InitArrayData2(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _Error);
  return _Array;
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::DeleteVector(T Index_)
{
  _MatrixIndex = Index_;
  return ::DeleteVector(_Array, 4, _MatrixIndex, _LenVect, _PosVect,
                        _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::IsDeallocated(T Index_)
{
  _MatrixIndex = Index_;
  return ::IsDeallocated(_Array, 4, _MatrixIndex, _LenVect, _PosVect,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::ReplaceVector(T Index_, void* Replacement_, bool Replace_,
                                bool DeAlloc_)
{
  _MatrixIndex = Index_;
  return ::ReplaceVector(_Array, 4, _MatrixIndex, _LenVect, _PosVect,
                         Replacement_, Replace_, DeAlloc_,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::RealVectorCount(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorCount(_Array, 4, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::RealVectorSize(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorMax(_Array, 4, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T** NestedArray4<T>::GiveInfoArray()
{
  T** retval = (T**)(::GiveInformationArray(_Array, 4));
  return retval;
}

/****************************************************************************/
template <class T>
const T* NestedArray4<T>::ParentReference(T Index_)
{
  _MatrixIndex = Index_;
  _ParentRefs = (T*)(::ParentReference(_Array, 4, _MatrixIndex, _LenVect,
                                         _PosVect, _ParentRefs, _Error));
  return _ParentRefs;
}

/****************************************************************************/
template <class T>
T* NestedArray4<T>::LinearToCoords(T Pos_, T* Coords_)
{
  return ::LinearToCoords(Pos_, Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::CoordsToLinear(T* Coords_)
{
  return ::CoordsToLinear(Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::ReadFromArray() const
{
  return ::ReadFromArray(_Array, 4, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
void* NestedArray4<T>::WriteToArray(T Data_)
{
  return ::WriteToArray(_Array, 4, _LenVect, _PosVect, Data_, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::ReadFromVector(T pos) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray4<T>* This_ = (NestedArray4<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 4, _LenVect, _PosVect, _Error);

  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return ret;
}

/****************************************************************************/
template <class T>
void* NestedArray4<T>::WriteToVector(T pos, T Data_)
{
  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  
  return WriteToArray(Data_);
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::AtPos(T pos, bool& Found_)
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 4, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
const T NestedArray4<T>::AtPos(T pos, bool& Found_) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray6<T>* This_ = (NestedArray6<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 4, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
int NestedArray4<T>::ReInitArrays() const
{
  return _ReInitArrays;
}

/****************************************************************************/
template <class T>
bool NestedArray4<T>::Error() const
{
  return _Error;
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::Shared() const
{
  return _Shared;
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::Size() const
{
  return SIZE;
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::MaxLength() const
{
  return MAXLENGTH;
}

/****************************************************************************/
template <class T>
T NestedArray4<T>::Dim() const
{
  return SIZE;
}

/****************************************************************************/
/****************************************************************************/
template <class T>
NestedArray5<T>::NestedArray5(T* Lengths_, int MaxLength_):
MAXLENGTH(MaxLength_ ? MaxLength_:SIZE),
_Array(NULL),
_MatrixIndex(0),
_LenVect(Lengths_ ?
           (T*)memmove(RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE), Lengths_, sizeof(T) * SIZE):
           NULL),
_PosVect((T*)memset(RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1)), 0, sizeof(T) * (SIZE+1))),
_ParentRefs((T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE)),
_ReInitArrays(REINIT|RESIZE),
_Error(false),
_Shared(0),
_Size(0),
_OffSets(NULL),
_OffSetsLen(SIZE*2)
{
  T x, cnt;
  for (x = 1; x < SIZE; x++)
    _PosVect[x] = x;

  _PosVect[0] = SIZE;
  _PosVect[SIZE] = 1;

  if (!Lengths_)
  {
    _LenVect = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
    
    for (x = 0; x < SIZE; x++)
      _LenVect[x] = MAXLENGTH;
  }

  _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);

  InitArrayData(0);
  InitArrayData(1);
  InitArrayData(2);
  InitArrayData(3);
  InitArrayData(4);
}

/****************************************************************************/
template <class T>
NestedArray5<T>::~NestedArray5()
{
  ::RemoveInformationArray(_Array, 5);
  ::DestroyArrayData(_Array, 5, _LenVect);
  
  ::DeleteArray(_LenVect);
  ::DeleteArray(_PosVect);
  ::DeleteArray(_ParentRefs);
  ::DeleteArray(_OffSets);
}

/****************************************************************************/
template <class T>
T* NestedArray5<T>::NewLengthsVector() const
{
  T* Lengths_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
  return GetLengths(Lengths_);
}

/****************************************************************************/
template <class T>
T* NestedArray5<T>::NewCoordinatesVector() const
{
  T* Coords_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1));
  return GetCoordinates(Coords_);
}

/****************************************************************************/
template <class T>
T* NestedArray5<T>::GetCoordinates(T* Coords_) const
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;    
    for (x = 1; x < SIZE; x++)
      Coords_[x] = _PosVect[x] - x;

    Coords_[0] = SIZE;
    Coords_[SIZE] = _PosVect[SIZE] - 1;    
  }

  return Coords_;  
}

/****************************************************************************/
template <class T>
T* NestedArray5<T>::GetLengths(T* Lengths_) const
{
  if (Lengths_ && Lengths_ != _LenVect)
    memmove(Lengths_, _LenVect, sizeof(T) * SIZE);
    
  return Lengths_;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray5<T>::SetReInitArrays(int Flag_)
{
  _ReInitArrays = Flag_;
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray5<T>::SetCoordinates(T* Coords_)
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;
    for (x = 1; x < SIZE; x++)
      if (x + Coords_[x] < SizeOfVector(SIZE, x-1, _LenVect))
        _PosVect[x] = x + Coords_[x];

    if (1 + Coords_[SIZE] < SizeOfVector(SIZE, SIZE-1, _LenVect))
      _PosVect[SIZE] = 1 + Coords_[SIZE];

    _PosVect[0] = SIZE;      
    InitArrayData(1);
    InitArrayData(2);
    InitArrayData(3);
    InitArrayData(4);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray5<T>::SetLengths(T* Lengths_)
{
  if (Lengths_ && Lengths_ != _LenVect)
  {
    memmove(_LenVect, Lengths_, sizeof(T) * SIZE);

    ::DeleteArray(_OffSets);
    _OffSets = NULL;
    _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);
  
    InitArrayData(1);
    InitArrayData(2);
    InitArrayData(3);
    InitArrayData(4);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
void* NestedArray5<T>::InitArrayData(T Index_)
{
  _MatrixIndex = Index_;
  _Array = (T*****)::InitArrayData(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _ReInitArrays, _Error);
  // _Array = (T*****)::InitArrayData2(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _Error);
  return _Array;
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::DeleteVector(T Index_)
{
  _MatrixIndex = Index_;
  return ::DeleteVector(_Array, 5, _MatrixIndex, _LenVect, _PosVect,
                        _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::IsDeallocated(T Index_)
{
  _MatrixIndex = Index_;
  return ::IsDeallocated(_Array, 5, _MatrixIndex, _LenVect, _PosVect,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::ReplaceVector(T Index_, void* Replacement_, bool Replace_,
                                bool DeAlloc_)
{
  _MatrixIndex = Index_;
  return ::ReplaceVector(_Array, 5, _MatrixIndex, _LenVect, _PosVect,
                         Replacement_, Replace_, DeAlloc_,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::RealVectorCount(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorCount(_Array, 5, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::RealVectorSize(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorMax(_Array, 5, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T** NestedArray5<T>::GiveInfoArray()
{
  T** retval = (T**)(::GiveInformationArray(_Array, 5));
  return retval;
}

/****************************************************************************/
template <class T>
const T* NestedArray5<T>::ParentReference(T Index_)
{
  _MatrixIndex = Index_;
  _ParentRefs = (T*)(::ParentReference(_Array, 5, _MatrixIndex, _LenVect,
                                         _PosVect, _ParentRefs, _Error));
  return _ParentRefs;
}

/****************************************************************************/
template <class T>
T* NestedArray5<T>::LinearToCoords(T Pos_, T* Coords_)
{
  return ::LinearToCoords(Pos_, Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::CoordsToLinear(T* Coords_)
{
  return ::CoordsToLinear(Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::ReadFromArray() const
{
  return ::ReadFromArray(_Array, 5, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
void* NestedArray5<T>::WriteToArray(T Data_)
{
  return ::WriteToArray(_Array, 5, _LenVect, _PosVect, Data_, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::ReadFromVector(T pos) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray5<T>* This_ = (NestedArray5<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 5, _LenVect, _PosVect, _Error);

  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return ret;
}

/****************************************************************************/
template <class T>
void* NestedArray5<T>::WriteToVector(T pos, T Data_)
{
  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  
  return WriteToArray(Data_);
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::AtPos(T pos, bool& Found_)
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 5, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
const T NestedArray5<T>::AtPos(T pos, bool& Found_) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray6<T>* This_ = (NestedArray6<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 5, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
int NestedArray5<T>::ReInitArrays() const
{
  return _ReInitArrays;
}

/****************************************************************************/
template <class T>
bool NestedArray5<T>::Error() const
{
  return _Error;
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::Shared() const
{
  return _Shared;
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::Size() const
{
  return SIZE;
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::MaxLength() const
{
  return MAXLENGTH;
}

/****************************************************************************/
template <class T>
T NestedArray5<T>::Dim() const
{
  return SIZE;
}

/****************************************************************************/
/****************************************************************************/
template <class T>
NestedArray6<T>::NestedArray6(T* Lengths_, int MaxLength_):
MAXLENGTH(MaxLength_ ? MaxLength_:SIZE),
_Array(NULL),
_MatrixIndex(0),
_LenVect(Lengths_ ?
           (T*)memmove(RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE), Lengths_, sizeof(T) * SIZE):
           NULL),
_PosVect((T*)memset(RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1)), 0, sizeof(T) * (SIZE+1))),
_ParentRefs((T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE)),
_ReInitArrays(REINIT|RESIZE),
_Error(false),
_Shared(0),
_Size(0),
_OffSets(NULL),
_OffSetsLen(SIZE*2)
{
  T x, cnt;
  for (x = 1; x < SIZE; x++)
    _PosVect[x] = x;

  _PosVect[0] = SIZE;
  _PosVect[SIZE] = 1;

  if (!Lengths_)
  {
    _LenVect = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
    
    for (x = 0; x < SIZE; x++)
      _LenVect[x] = MAXLENGTH;
  }

  _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);

  InitArrayData(0);
  InitArrayData(1);
  InitArrayData(2);
  InitArrayData(3);
  InitArrayData(4);
  InitArrayData(5);
}

/****************************************************************************/
template <class T>
NestedArray6<T>::~NestedArray6()
{
  ::RemoveInformationArray(_Array, 6);
  ::DestroyArrayData(_Array, 6, _LenVect);
  
  ::DeleteArray(_LenVect);
  ::DeleteArray(_PosVect);
  ::DeleteArray(_ParentRefs);
  ::DeleteArray(_OffSets);
}

/****************************************************************************/
template <class T>
T* NestedArray6<T>::NewLengthsVector() const
{
  T* Lengths_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
  return GetLengths(Lengths_);
}

/****************************************************************************/
template <class T>
T* NestedArray6<T>::NewCoordinatesVector() const
{
  T* Coords_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1));
  return GetCoordinates(Coords_);
}

/****************************************************************************/
template <class T>
T* NestedArray6<T>::GetCoordinates(T* Coords_) const
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;    
    for (x = 1; x < SIZE; x++)
      Coords_[x] = _PosVect[x] - x;

    Coords_[0] = SIZE;
    Coords_[SIZE] = _PosVect[SIZE] - 1;    
  }

  return Coords_;  
}

/****************************************************************************/
template <class T>
T* NestedArray6<T>::GetLengths(T* Lengths_) const
{
  if (Lengths_ && Lengths_ != _LenVect)
    memmove(Lengths_, _LenVect, sizeof(T) * SIZE);
    
  return Lengths_;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray6<T>::SetReInitArrays(int Flag_)
{
  _ReInitArrays = Flag_;
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray6<T>::SetCoordinates(T* Coords_)
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;
    for (x = 1; x < SIZE; x++)
      if (x + Coords_[x] < SizeOfVector(SIZE, x-1, _LenVect))
        _PosVect[x] = x + Coords_[x];

    if (1 + Coords_[SIZE] < SizeOfVector(SIZE, SIZE-1, _LenVect))
      _PosVect[SIZE] = 1 + Coords_[SIZE];

    _PosVect[0] = SIZE;      
    InitArrayData(1);
    InitArrayData(2);
    InitArrayData(3);
    InitArrayData(4);
    InitArrayData(5);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray6<T>::SetLengths(T* Lengths_)
{
  if (Lengths_ && Lengths_ != _LenVect)
  {
    memmove(_LenVect, Lengths_, sizeof(T) * SIZE);

    ::DeleteArray(_OffSets);
    _OffSets = NULL;
    _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);
  
    InitArrayData(1);
    InitArrayData(2);
    InitArrayData(3);
    InitArrayData(4);
    InitArrayData(5);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
void* NestedArray6<T>::InitArrayData(T Index_)
{
  _MatrixIndex = Index_;
  // _Array = (T******)::InitArrayData(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _ReInitArrays, _Error);
  _Array = (T******)::InitArrayData2(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _Error);
  return _Array;
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::DeleteVector(T Index_)
{
  _MatrixIndex = Index_;
  return ::DeleteVector(_Array, 6, _MatrixIndex, _LenVect, _PosVect,
                        _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::IsDeallocated(T Index_)
{
  _MatrixIndex = Index_;
  return ::IsDeallocated(_Array, 6, _MatrixIndex, _LenVect, _PosVect,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::ReplaceVector(T Index_, void* Replacement_, bool Replace_,
                                bool DeAlloc_)
{
  _MatrixIndex = Index_;
  return ::ReplaceVector(_Array, 6, _MatrixIndex, _LenVect, _PosVect,
                         Replacement_, Replace_, DeAlloc_,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::RealVectorCount(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorCount(_Array, 6, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::RealVectorSize(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorMax(_Array, 6, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T** NestedArray6<T>::GiveInfoArray()
{
  T** retval = (T**)(::GiveInformationArray(_Array, 6));
  return retval;
}

/****************************************************************************/
template <class T>
const T* NestedArray6<T>::ParentReference(T Index_)
{
  _MatrixIndex = Index_;
  _ParentRefs = (T*)(::ParentReference(_Array, 6, _MatrixIndex, _LenVect,
                                         _PosVect, _ParentRefs, _Error));
  return _ParentRefs;
}

/****************************************************************************/
template <class T>
T* NestedArray6<T>::LinearToCoords(T Pos_, T* Coords_)
{
  return ::LinearToCoords(Pos_, Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::CoordsToLinear(T* Coords_)
{
  return ::CoordsToLinear(Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::ReadFromArray() const
{
  return ::ReadFromArray(_Array, 6, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
void* NestedArray6<T>::WriteToArray(T Data_)
{
  return ::WriteToArray(_Array, 6, _LenVect, _PosVect, Data_, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::ReadFromVector(T pos) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray6<T>* This_ = (NestedArray6<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 6, _LenVect, _PosVect, _Error);

  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return ret;
}

/****************************************************************************/
template <class T>
void* NestedArray6<T>::WriteToVector(T pos, T Data_)
{
  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  
  return WriteToArray(Data_);
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::AtPos(T pos, bool& Found_)
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 6, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
const T NestedArray6<T>::AtPos(T pos, bool& Found_) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray6<T>* This_ = (NestedArray6<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 6, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
int NestedArray6<T>::ReInitArrays() const
{
  return _ReInitArrays;
}

/****************************************************************************/
template <class T>
bool NestedArray6<T>::Error() const
{
  return _Error;
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::Shared() const
{
  return _Shared;
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::Size() const
{
  return SIZE;
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::MaxLength() const
{
  return MAXLENGTH;
}

/****************************************************************************/
template <class T>
T NestedArray6<T>::Dim() const
{
  return SIZE;
}

/****************************************************************************/
/****************************************************************************/
template <class T>
NestedArray7<T>::NestedArray7(T* Lengths_, int MaxLength_):
MAXLENGTH(MaxLength_ ? MaxLength_:SIZE),
_Array(NULL),
_MatrixIndex(0),
_LenVect(Lengths_ ?
           (T*)memmove(RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE), Lengths_, sizeof(T) * SIZE):
           NULL),
_PosVect((T*)memset(RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1)), 0, sizeof(T) * (SIZE+1))),
_ParentRefs((T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE)),
_ReInitArrays(REINIT|RESIZE),
_Error(false),
_Shared(0),
_Size(0),
_OffSets(NULL),
_OffSetsLen(SIZE*2)
{
  T x, cnt;
  for (x = 1; x < SIZE; x++)
    _PosVect[x] = x;

  _PosVect[0] = SIZE;
  _PosVect[SIZE] = 1;

  if (!Lengths_)
  {
    _LenVect = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
    
    for (x = 0; x < SIZE; x++)
      _LenVect[x] = MAXLENGTH;
  }

  _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);

  InitArrayData(0);
  InitArrayData(1);
  InitArrayData(2);
  InitArrayData(3);
  InitArrayData(4);
  InitArrayData(5);
  InitArrayData(6);
}

/****************************************************************************/
template <class T>
NestedArray7<T>::~NestedArray7()
{
  ::RemoveInformationArray(_Array, 7);
  ::DestroyArrayData(_Array, 7, _LenVect);
  
  ::DeleteArray(_LenVect);
  ::DeleteArray(_PosVect);
  ::DeleteArray(_ParentRefs);
  ::DeleteArray(_OffSets);
}

/****************************************************************************/
template <class T>
T* NestedArray7<T>::NewLengthsVector() const
{
  T* Lengths_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
  return GetLengths(Lengths_);
}

/****************************************************************************/
template <class T>
T* NestedArray7<T>::NewCoordinatesVector() const
{
  T* Coords_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1));
  return GetCoordinates(Coords_);
}

/****************************************************************************/
template <class T>
T* NestedArray7<T>::GetCoordinates(T* Coords_) const
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;    
    for (x = 1; x < SIZE; x++)
      Coords_[x] = _PosVect[x] - x;

    Coords_[0] = SIZE;
    Coords_[SIZE] = _PosVect[SIZE] - 1;    
  }

  return Coords_;  
}

/****************************************************************************/
template <class T>
T* NestedArray7<T>::GetLengths(T* Lengths_) const
{
  if (Lengths_ && Lengths_ != _LenVect)
    memmove(Lengths_, _LenVect, sizeof(T) * SIZE);
    
  return Lengths_;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray7<T>::SetReInitArrays(int Flag_)
{
  _ReInitArrays = Flag_;
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray7<T>::SetCoordinates(T* Coords_)
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;
    for (x = 1; x < SIZE; x++)
      if (x + Coords_[x] < SizeOfVector(SIZE, x-1, _LenVect))
        _PosVect[x] = x + Coords_[x];

    if (1 + Coords_[SIZE] < SizeOfVector(SIZE, SIZE-1, _LenVect))
      _PosVect[SIZE] = 1 + Coords_[SIZE];

    _PosVect[0] = SIZE;      
    InitArrayData(1);
    InitArrayData(2);
    InitArrayData(3);
    InitArrayData(4);
    InitArrayData(5);
    InitArrayData(6);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray7<T>::SetLengths(T* Lengths_)
{
  if (Lengths_ && Lengths_ != _LenVect)
  {
    memmove(_LenVect, Lengths_, sizeof(T) * SIZE);

    ::DeleteArray(_OffSets);
    _OffSets = NULL;
    _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);
  
    InitArrayData(1);
    InitArrayData(2);
    InitArrayData(3);
    InitArrayData(4);
    InitArrayData(5);
    InitArrayData(6);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
void* NestedArray7<T>::InitArrayData(T Index_)
{
  _MatrixIndex = Index_;
  // _Array = (T*******)::InitArrayData(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _ReInitArrays, _Error);
  _Array = (T*******)::InitArrayData2(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _Error);
  return _Array;
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::DeleteVector(T Index_)
{
  _MatrixIndex = Index_;
  return ::DeleteVector(_Array, 7, _MatrixIndex, _LenVect, _PosVect,
                        _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::IsDeallocated(T Index_)
{
  _MatrixIndex = Index_;
  return ::IsDeallocated(_Array, 7, _MatrixIndex, _LenVect, _PosVect,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::ReplaceVector(T Index_, void* Replacement_, bool Replace_,
                                bool DeAlloc_)
{
  _MatrixIndex = Index_;
  return ::ReplaceVector(_Array, 7, _MatrixIndex, _LenVect, _PosVect,
                         Replacement_, Replace_, DeAlloc_,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::RealVectorCount(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorCount(_Array, 7, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::RealVectorSize(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorMax(_Array, 7, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T** NestedArray7<T>::GiveInfoArray()
{
  T** retval = (T**)(::GiveInformationArray(_Array, 7));
  return retval;
}

/****************************************************************************/
template <class T>
const T* NestedArray7<T>::ParentReference(T Index_)
{
  _MatrixIndex = Index_;
  _ParentRefs = (T*)(::ParentReference(_Array, 7, _MatrixIndex, _LenVect,
                                         _PosVect, _ParentRefs, _Error));
  return _ParentRefs;
}

/****************************************************************************/
template <class T>
T* NestedArray7<T>::LinearToCoords(T Pos_, T* Coords_)
{
  return ::LinearToCoords(Pos_, Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::CoordsToLinear(T* Coords_)
{
  return ::CoordsToLinear(Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::ReadFromArray() const
{
  return ::ReadFromArray(_Array, 7, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
void* NestedArray7<T>::WriteToArray(T Data_)
{
  return ::WriteToArray(_Array, 7, _LenVect, _PosVect, Data_, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::ReadFromVector(T pos) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray7<T>* This_ = (NestedArray7<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 7, _LenVect, _PosVect, _Error);

  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return ret;
}

/****************************************************************************/
template <class T>
void* NestedArray7<T>::WriteToVector(T pos, T Data_)
{
  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  
  return WriteToArray(Data_);
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::AtPos(T pos, bool& Found_)
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 7, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
const T NestedArray7<T>::AtPos(T pos, bool& Found_) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray6<T>* This_ = (NestedArray6<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 7, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
int NestedArray7<T>::ReInitArrays() const
{
  return _ReInitArrays;
}

/****************************************************************************/
template <class T>
bool NestedArray7<T>::Error() const
{
  return _Error;
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::Shared() const
{
  return _Shared;
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::Size() const
{
  return SIZE;
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::MaxLength() const
{
  return MAXLENGTH;
}

/****************************************************************************/
template <class T>
T NestedArray7<T>::Dim() const
{
  return SIZE;
}

/****************************************************************************/
/****************************************************************************/
template <class T>
NestedArray8<T>::NestedArray8(T* Lengths_, int MaxLength_):
MAXLENGTH(MaxLength_ ? MaxLength_:SIZE),
_Array(NULL),
_MatrixIndex(0),
_LenVect(Lengths_ ?
           (T*)memmove(RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE), Lengths_, sizeof(T) * SIZE):
           NULL),
_PosVect((T*)memset(RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1)), 0, sizeof(T) * (SIZE+1))),
_ParentRefs((T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE)),
_ReInitArrays(REINIT|RESIZE),
_Error(false),
_Shared(0),
_Size(0),
_OffSets(NULL),
_OffSetsLen(SIZE*2)
{
  T x, cnt;
  for (x = 1; x < SIZE; x++)
    _PosVect[x] = x;

  _PosVect[0] = SIZE;
  _PosVect[SIZE] = 1;

  if (!Lengths_)
  {
    _LenVect = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
    
    for (x = 0; x < SIZE; x++)
      _LenVect[x] = MAXLENGTH;
  }

  _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);

  InitArrayData(0);
  InitArrayData(1);
  InitArrayData(2);
  InitArrayData(3);
  InitArrayData(4);
  InitArrayData(5);
  InitArrayData(6);
  InitArrayData(7);
}

/****************************************************************************/
template <class T>
NestedArray8<T>::~NestedArray8()
{
  ::RemoveInformationArray(_Array, 8);
  ::DestroyArrayData(_Array, 8, _LenVect);
  
  ::DeleteArray(_LenVect);
  ::DeleteArray(_PosVect);
  ::DeleteArray(_ParentRefs);
  ::DeleteArray(_OffSets);
}

/****************************************************************************/
template <class T>
T* NestedArray8<T>::NewLengthsVector() const
{
  T* Lengths_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * SIZE);
  return GetLengths(Lengths_);
}

/****************************************************************************/
template <class T>
T* NestedArray8<T>::NewCoordinatesVector() const
{
  T* Coords_ = (T*)RawCallocateWith(MEMMATRIX, sizeof(T) * (SIZE+1));
  return GetCoordinates(Coords_);
}

/****************************************************************************/
template <class T>
T* NestedArray8<T>::GetCoordinates(T* Coords_) const
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;    
    for (x = 1; x < SIZE; x++)
      Coords_[x] = _PosVect[x] - x;

    Coords_[0] = SIZE;
    Coords_[SIZE] = _PosVect[SIZE] - 1;    
  }

  return Coords_;  
}

/****************************************************************************/
template <class T>
T* NestedArray8<T>::GetLengths(T* Lengths_) const
{
  if (Lengths_ && Lengths_ != _LenVect)
    memmove(Lengths_, _LenVect, sizeof(T) * SIZE);
    
  return Lengths_;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray8<T>::SetReInitArrays(int Flag_)
{
  _ReInitArrays = Flag_;
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray8<T>::SetCoordinates(T* Coords_)
{
  if (Coords_ && Coords_ != _PosVect)
  {
    T x;
    for (x = 1; x < SIZE; x++)
      if (x + Coords_[x] < SizeOfVector(SIZE, x-1, _LenVect))
        _PosVect[x] = x + Coords_[x];

    if (1 + Coords_[SIZE] < SizeOfVector(SIZE, SIZE-1, _LenVect))
      _PosVect[SIZE] = 1 + Coords_[SIZE];

    _PosVect[0] = SIZE;      
    InitArrayData(1);
    InitArrayData(2);
    InitArrayData(3);
    InitArrayData(4);
    InitArrayData(5);
    InitArrayData(6);
    InitArrayData(7);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
NestedVector<T>& NestedArray8<T>::SetLengths(T* Lengths_)
{
  if (Lengths_ && Lengths_ != _LenVect)
  {
    memmove(_LenVect, Lengths_, sizeof(T) * SIZE);

    ::DeleteArray(_OffSets);
    _OffSets = NULL;
    _OffSets = ::CalculateOffsets(_LenVect, SIZE, _OffSetsLen);
  
    InitArrayData(1);
    InitArrayData(2);
    InitArrayData(3);
    InitArrayData(4);
    InitArrayData(5);
    InitArrayData(6);
    InitArrayData(7);
  }
  
  return *this;
}

/****************************************************************************/
template <class T>
void* NestedArray8<T>::InitArrayData(T Index_)
{
  _MatrixIndex = Index_;
  // _Array = (T********)::InitArrayData(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _ReInitArrays, _Error);
  _Array = (T********)::InitArrayData2(_Array, SIZE, _MatrixIndex, _LenVect, _PosVect, _Error);
  return _Array;
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::DeleteVector(T Index_)
{
  _MatrixIndex = Index_;
  return ::DeleteVector(_Array, 8, _MatrixIndex, _LenVect, _PosVect,
                        _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::IsDeallocated(T Index_)
{
  _MatrixIndex = Index_;
  return ::IsDeallocated(_Array, 8, _MatrixIndex, _LenVect, _PosVect,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::ReplaceVector(T Index_, void* Replacement_, bool Replace_,
                                bool DeAlloc_)
{
  _MatrixIndex = Index_;
  return ::ReplaceVector(_Array, 8, _MatrixIndex, _LenVect, _PosVect,
                         Replacement_, Replace_, DeAlloc_,
                         _Error, &_Shared, &_Size);
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::RealVectorCount(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorCount(_Array, 8, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::RealVectorSize(T Index_) const
{
  _MatrixIndex = Index_;
  return ::RealVectorMax(_Array, 8, _MatrixIndex, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
T** NestedArray8<T>::GiveInfoArray()
{
  T** retval = (T**)(::GiveInformationArray(_Array, 8));
  return retval;
}

/****************************************************************************/
template <class T>
const T* NestedArray8<T>::ParentReference(T Index_)
{
  _MatrixIndex = Index_;
  _ParentRefs = (T*)(::ParentReference(_Array, 8, _MatrixIndex, _LenVect,
                                         _PosVect, _ParentRefs, _Error));
  return _ParentRefs;
}

/****************************************************************************/
template <class T>
T* NestedArray8<T>::LinearToCoords(T Pos_, T* Coords_)
{
  return ::LinearToCoords(Pos_, Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::CoordsToLinear(T* Coords_)
{
  return ::CoordsToLinear(Coords_, SIZE, _OffSets, _OffSetsLen);
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::ReadFromArray() const
{
  return ::ReadFromArray(_Array, 8, _LenVect, _PosVect, _Error);
}

/****************************************************************************/
template <class T>
void* NestedArray8<T>::WriteToArray(T Data_)
{
  return ::WriteToArray(_Array, 8, _LenVect, _PosVect, Data_, _Error);
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::ReadFromVector(T pos) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray8<T>* This_ = (NestedArray8<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 8, _LenVect, _PosVect, _Error);

  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return ret;
}

/****************************************************************************/
template <class T>
void* NestedArray8<T>::WriteToVector(T pos, T Data_)
{
  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  
  return WriteToArray(Data_);
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::AtPos(T pos, bool& Found_)
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  T* Coords_ = NewCoordinatesVector();
  Coords_ = LinearToCoords(pos, Coords_);
  SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 8, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
const T NestedArray8<T>::AtPos(T pos, bool& Found_) const
{
  T* OldCoords_ = NewCoordinatesVector();
  OldCoords_ = GetCoordinates(OldCoords_);

  NestedArray6<T>* This_ = (NestedArray6<T>*)this;
  T* Coords_ = NewCoordinatesVector();
  Coords_ = This_->LinearToCoords(pos, Coords_);
  This_->SetCoordinates(Coords_);
  ::DeleteArray(Coords_);
  T ret = ::ReadFromArray(_Array, 8, _LenVect, _PosVect, _Error);

  Found_ = !_Error;
  This_->SetCoordinates(OldCoords_);
  ::DeleteArray(OldCoords_);
  
  return (Found_ ? ret:0);
}

/****************************************************************************/
template <class T>
int NestedArray8<T>::ReInitArrays() const
{
  return _ReInitArrays;
}

/****************************************************************************/
template <class T>
bool NestedArray8<T>::Error() const
{
  return _Error;
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::Shared() const
{
  return _Shared;
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::Size() const
{
  return SIZE;
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::MaxLength() const
{
  return MAXLENGTH;
}

/****************************************************************************/
template <class T>
T NestedArray8<T>::Dim() const
{
  return SIZE;
}

/****************************************************************************/
#if NESTARRAY_DEBUG
int main(int argc, char* argv[])
{
#if NESTARRAY_DEBUG1
#endif

#if NESTARRAY_DEBUG2
  int* Lengths_;
  int* Coords_;
  const int* ParentRefs_;
  int** Info_;

  int x, x1, x2, x3, x4, x5, x6, x7, x8;
  int sz, max;
  int retval;
  int data;
  int SIZE;
  char dum[5];

  int* ReplVect_;
  int* cpos;
  int lpos;

  system("cls");

  //////////////////////////////////
  // testing 3 dimensional matrix //
  //////////////////////////////////
  #if NESTARRAY_ARRLEN3_DEBUG
  NestedArray3<int> na3(NULL);
  Lengths_ = na3.NewLengthsVector();
  Coords_ = na3.NewCoordinatesVector();
  SIZE = na3.MaxLength();

  cpos = (int*)memset(RawCallocateWith(MEMMATRIX, sizeof(int) * (SIZE+1)), 0, sizeof(int) * (SIZE+1));

  cout <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x > 0)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }

  Info_ = na3.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<endl <<endl;

  for (x1 = 0; x1 < SIZE; x1++)
    for (x2 = 0; x2 < SIZE; x2++)
      for (x3 = 0; x3 < SIZE; x3++)
      {
        Coords_[1] = x1;
        Coords_[2] = x2;
        Coords_[3] = x3;
        cout <<"Coordinates: ";
  
        for (x = 1; x <= SIZE; x++)
        {
          if (x > 1)
            cout <<", ";
          cout <<"pos[" <<x <<"] = " <<Coords_[x];
        }

        cout <<" | ";
        data = (x1 * 10000) + (x2 * 100) + x3;
        na3.SetCoordinates(Coords_);
        na3.WriteToArray(data);
        retval = na3.ReadFromArray();
        cout <<"retval = " <<retval <<endl;
        lpos = na3.CoordsToLinear(Coords_);
        cpos = na3.LinearToCoords(lpos, cpos);
        cout <<"linear pos = " <<lpos <<" : coords = (";

        for (x = 1; x <= SIZE; x++)
        {
          cout <<cpos[x];
          if (x < SIZE)
            cout <<",";
        }        
        cout <<")" <<endl;
        
        for (x = 0; x < SIZE; x++)
        {
          if (x > 0)
            cout <<", ";
          cout <<"RVCount(" <<x <<") = " <<na3.RealVectorCount(x);
        }
        cout <<endl;

        for (x = 0; x < SIZE; x++)
        {
          if (x > 0)
            cout <<", ";
          cout <<"RVSize(" <<x <<") = " <<na3.RealVectorSize(x);
        }
        cout <<endl;
        
        ParentRefs_ = na3.ParentReference(1);
        max = ParentRefs_[0];
  
        for (x = 0; x <= max; x++)
        {
          if (x > 0)
            cout <<", ";
          cout <<"ParentRefs[" <<x <<"] = " <<ParentRefs_[x];
        }
        cout <<endl <<endl;
      }

  for (x = 0; x < SIZE; x++)
    Lengths_[x] = x+SIZE;

  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);

  cout <<endl;
  na3.SetLengths(Lengths_);
  
  cout <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }  

  Info_ = na3.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<endl <<endl;
  
  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);
  
  for (x1 = 0; x1 < SIZE; x1++)
    for (x2 = 0; x2 < SIZE+1; x2++)
      for (x3 = 0; x3 < SIZE+2; x3++)
      {
        Coords_[1] = x1;
        Coords_[2] = x2;
        Coords_[3] = x3;
        cout <<"Coordinates: ";
  
        for (x = 1; x <= SIZE; x++)
        {
          if (x > 1)
            cout <<", ";
          cout <<"pos[" <<x <<"] = " <<Coords_[x];
        }

        cout <<" | ";
        data = (x1 * 10000) + (x2 * 100) + x3;
        na3.SetCoordinates(Coords_);
        na3.WriteToArray(data);
        retval = na3.ReadFromArray();
        cout <<"retval = " <<retval <<endl;
        lpos = na3.CoordsToLinear(Coords_);
        cpos = na3.LinearToCoords(lpos, cpos);
        cout <<"linear pos = " <<lpos <<" : coords = (";

        for (x = 1; x <= SIZE; x++)
        {
          cout <<cpos[x];
          if (x < SIZE)
            cout <<",";
        }        
        cout <<")" <<endl;

        for (x = 0; x < SIZE; x++)
        {
          if (x > 0)
            cout <<", ";
          cout <<"RVCount(" <<x <<") = " <<na3.RealVectorCount(x);
        }
        cout <<endl;

        for (x = 0; x < SIZE; x++)
        {
          if (x > 0)
            cout <<", ";
          cout <<"RVSize(" <<x <<") = " <<na3.RealVectorSize(x);
        }
        cout <<endl;

        ParentRefs_ = na3.ParentReference(1);
        max = ParentRefs_[0];
  
        for (x = 0; x <= max; x++)
        {
          if (x > 0)
            cout <<", ";
          cout <<"ParentRefs[" <<x <<"] = " <<ParentRefs_[x];
        }
        cout <<endl <<endl;
      }

  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);

  cout <<endl <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }  

  Info_ = na3.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;  
  cout <<endl;

  // testing vector deletion
  Coords_[1] = SIZE-1;
  Coords_[2] = SIZE;
  na3.SetCoordinates(Coords_);

  na3.DeleteVector(1);
  cout <<"DeleteVector(1)" <<endl;
  cout <<"IsDeallocated(1), "
       <<"pos[1] = " <<Coords_[1] <<" : "
       <<na3.IsDeallocated(1) <<endl;

  // testing replacement with vector deallocation
  sz = SizeOfVector(SIZE, SIZE-1, Lengths_);
  ReplVect_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
  ReplVect_[0] = sz-1;
  for (x = 1; x < sz; x++)
    ReplVect_[x] = x*10+x+1;

  Coords_[1] = SIZE-2;
  Coords_[2] = SIZE;
  na3.SetCoordinates(Coords_);
  cout <<"IsDeallocated(1), "
       <<"pos[1] = " <<Coords_[1] <<" : "
       <<na3.IsDeallocated(1) <<endl;

  // replacing final nesting level vector
  na3.ReplaceVector(SIZE-1, ReplVect_, true, true);
  cout <<endl;
  cout <<"ReplaceVector(" <<(SIZE-1) <<", ReplVect, true, true)" <<endl;
  cout <<"IsDeallocated(2), "
       <<"pos[2] = " <<Coords_[2] <<" : "
       <<na3.IsDeallocated(SIZE-1) <<endl;

  for (x3 = 1; x3 < SIZE+3; x3++)
  {
    Coords_[3] = x3;
    cout <<"Coordinates: ";
  
    for (x = 1; x <= SIZE; x++)
    {
      if (x > 1)
        cout <<", ";
      cout <<"pos[" <<x <<"] = " <<Coords_[x];
    }

    cout <<" | ";
    na3.SetCoordinates(Coords_);
    retval = na3.ReadFromArray();
    cout <<"retval = " <<retval <<endl;
  }

  // testing replacement without vector deallocation
  sz = SizeOfVector(SIZE, SIZE-1, Lengths_);
  ReplVect_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
  ReplVect_[0] = sz-1;
  for (x = 1; x < sz; x++)
    ReplVect_[x] = x*100+(x+1)*10+x+2;

  Coords_[1] = 1;
  Coords_[2] = SIZE;
  na3.SetCoordinates(Coords_);

  // replacing final nesting level vector
  na3.ReplaceVector(SIZE-1, ReplVect_, true, false);
  cout <<endl;
  cout <<"ReplaceVector(" <<(SIZE-1) <<", ReplVect, true, false)" <<endl;
  cout <<"IsDeallocated(2), "
       <<"pos[2] = " <<Coords_[2] <<" : "
       <<na3.IsDeallocated(SIZE-1) <<endl;

  for (x3 = 1; x3 < SIZE+3; x3++)
  {
    Coords_[3] = x3;
    cout <<"Coordinates: ";
  
    for (x = 1; x <= SIZE; x++)
    {
      if (x > 1)
        cout <<", ";
      cout <<"pos[" <<x <<"] = " <<Coords_[x];
    }

    cout <<" | ";
    na3.SetCoordinates(Coords_);
    retval = na3.ReadFromArray();
    cout <<"retval = " <<retval <<endl;
  }

  Coords_[1] = SIZE-2;
  Coords_[2] = SIZE;
  na3.SetCoordinates(Coords_);

  na3.DeleteVector(2);
  cout <<endl <<"DeleteVector(2)" <<endl;
  cout <<"IsDeallocated(2), "
       <<"pos[2] = " <<Coords_[2] <<" : "
       <<na3.IsDeallocated(2) <<endl;
  #endif

  //////////////////////////////////
  // testing 4 dimensional matrix //
  //////////////////////////////////
  #if NESTARRAY_ARRLEN4_DEBUG
  NestedArray4<int> na4(NULL);
  Lengths_ = na4.NewLengthsVector();
  Coords_ = na4.NewCoordinatesVector();
  SIZE = na4.MaxLength();
  cpos = (int*)memset(RawCallocateWith(MEMMATRIX, sizeof(int) * (SIZE+1)), 0, sizeof(int) * (SIZE+1));

  cout <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x > 0)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }

  Info_ = na4.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<"3rd level array init/cap = " <<Info_[4][0] <<endl;
  cout <<endl <<endl;

  for (x1 = 0; x1 < SIZE; x1++)
    for (x2 = 0; x2 < SIZE; x2++)
      for (x3 = 0; x3 < SIZE; x3++)
        for (x4 = 0; x4 < SIZE; x4++)
        {
          Coords_[1] = x1;
          Coords_[2] = x2;
          Coords_[3] = x3;
          Coords_[4] = x4;
          cout <<"Coordinates: ";
  
          for (x = 1; x <= SIZE; x++)
          {
            if (x > 1)
              cout <<", ";
            cout <<"pos[" <<x <<"] = " <<Coords_[x];
          }

          cout <<" | ";
          data = (x1 * 1000000) + (x2 * 10000) + (x3 * 100) + x4;
          na4.SetCoordinates(Coords_);
          na4.WriteToArray(data);
          retval = na4.ReadFromArray();
          cout <<"retval = " <<retval <<endl;
          lpos = na4.CoordsToLinear(Coords_);
          cpos = na4.LinearToCoords(lpos, cpos);
          cout <<"linear pos = " <<lpos <<" : coords = (";

          for (x = 1; x <= SIZE; x++)
          {
            cout <<cpos[x];
            if (x < SIZE)
              cout <<",";
          }
          cout <<")" <<endl;

          for (x = 0; x < SIZE; x++)
          {
            if (x > 0)
              cout <<", ";
            cout <<"RVCount(" <<x <<") = " <<na4.RealVectorCount(x);
          }
          cout <<endl;

          for (x = 0; x < SIZE; x++)
          {
            if (x > 0)
              cout <<", ";
            cout <<"RVSize(" <<x <<") = " <<na4.RealVectorSize(x);
          }
          cout <<endl;
        
          ParentRefs_ = na4.ParentReference(1);
          max = ParentRefs_[0];
  
          for (x = 0; x <= max; x++)
          {
            if (x > 0)
              cout <<", ";
            cout <<"ParentRefs[" <<x <<"] = " <<ParentRefs_[x];
          }
          cout <<endl <<endl;
        }

  for (x = 0; x < SIZE; x++)
    Lengths_[x] = x+SIZE;

  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);

  cout <<endl;
  na4.SetLengths(Lengths_);
  
  cout <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }  

  Info_ = na4.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<"3rd level array init/cap = " <<Info_[4][0] <<endl;
  cout <<endl <<endl;
  
  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);
  
  for (x1 = 0; x1 < SIZE; x1++)
    for (x2 = 0; x2 < SIZE+1; x2++)
      for (x3 = 0; x3 < SIZE+2; x3++)
        for (x4 = 0; x4 < SIZE+3; x4++)
        {
          Coords_[1] = x1;
          Coords_[2] = x2;
          Coords_[3] = x3;
          Coords_[4] = x4;
          cout <<"Coordinates: ";
  
          for (x = 1; x <= SIZE; x++)
          {
            if (x > 1)
              cout <<", ";
            cout <<"pos[" <<x <<"] = " <<Coords_[x];
          }

          cout <<" | ";
          data = (x1 * 1000000) + (x2 * 10000) + (x3 * 100) + x4;
          na4.SetCoordinates(Coords_);
          na4.WriteToArray(data);
          retval = na4.ReadFromArray();
          cout <<"retval = " <<retval <<endl;
          lpos = na4.CoordsToLinear(Coords_);
          cpos = na4.LinearToCoords(lpos, cpos);
          cout <<"linear pos = " <<lpos <<" : coords = (";

          for (x = 1; x <= SIZE; x++)
          {
            cout <<cpos[x];
            if (x < SIZE)
              cout <<",";
          }
          cout <<")" <<endl;

          for (x = 0; x < SIZE; x++)
          {
            if (x > 0)
              cout <<", ";
            cout <<"RVCount(" <<x <<") = " <<na4.RealVectorCount(x);
          }
          cout <<endl;

          for (x = 0; x < SIZE; x++)
          {
            if (x > 0)
              cout <<", ";
            cout <<"RVSize(" <<x <<") = " <<na4.RealVectorSize(x);
          }
          cout <<endl;

          ParentRefs_ = na4.ParentReference(1);
          max = ParentRefs_[0];
  
          for (x = 0; x <= max; x++)
          {
            if (x > 0)
              cout <<", ";
            cout <<"ParentRefs[" <<x <<"] = " <<ParentRefs_[x];
          }
          cout <<endl <<endl;
        }

  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);

  cout <<endl <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }  

  Info_ = na4.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<"3rd level array init/cap = " <<Info_[4][0] <<endl;
  cout <<endl;

  // testing vector deletion
  Coords_[1] = SIZE-1;
  Coords_[2] = SIZE;
  Coords_[3] = SIZE+1;
  na4.SetCoordinates(Coords_);

  na4.DeleteVector(1);
  cout <<"DeleteVector(1)" <<endl;
  cout <<"IsDeallocated(1), "
       <<"pos[1] = " <<Coords_[1] <<" : "
       <<"pos[2] = " <<Coords_[2] <<" : "
       <<na4.IsDeallocated(1) <<endl;

  // testing replacement with vector deallocation
  sz = SizeOfVector(SIZE, SIZE-1, Lengths_);
  ReplVect_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
  ReplVect_[0] = sz-1;
  for (x = 1; x < sz; x++)
    ReplVect_[x] = x*10+x+1;

  Coords_[1] = SIZE-2;
  Coords_[2] = SIZE;
  Coords_[3] = SIZE+1;
  na4.SetCoordinates(Coords_);
  cout <<"IsDeallocated(1), "
       <<"pos[1] = " <<Coords_[1] <<" : "
       <<"pos[2] = " <<Coords_[2] <<" : "
       <<na4.IsDeallocated(1) <<endl;

  // replacing final nesting level vector
  na4.ReplaceVector(SIZE-1, ReplVect_, true, true);
  cout <<endl;
  cout <<"ReplaceVector(" <<(SIZE-1) <<", ReplVect, true, true)" <<endl;
  cout <<"IsDeallocated(3), "
       <<"pos[3] = " <<Coords_[3] <<" : "
       <<na4.IsDeallocated(SIZE-1) <<endl;

  for (x4 = 1; x4 < SIZE+3; x4++)
  {
    Coords_[4] = x4;
    cout <<"Coordinates: ";
  
    for (x = 1; x <= SIZE; x++)
    {
      if (x > 1)
        cout <<", ";
      cout <<"pos[" <<x <<"] = " <<Coords_[x];
    }

    cout <<" | ";
    na4.SetCoordinates(Coords_);
    retval = na4.ReadFromArray();
    cout <<"retval = " <<retval <<endl;
  }

  // testing replacement without vector deallocation
  sz = SizeOfVector(SIZE, SIZE-1, Lengths_);
  ReplVect_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
  ReplVect_[0] = sz-1;
  for (x = 1; x < sz; x++)
    ReplVect_[x] = x*100+(x+1)*10+x+2;

  Coords_[1] = 1;
  Coords_[2] = SIZE;
  Coords_[3] = SIZE+1;
  na4.SetCoordinates(Coords_);

  // replacing final nesting level vector
  na4.ReplaceVector(SIZE-1, ReplVect_, true, false);
  cout <<endl;
  cout <<"ReplaceVector(" <<(SIZE-1) <<", ReplVect, true, false)" <<endl;
  cout <<"IsDeallocated(3), "
       <<"pos[3] = " <<Coords_[3] <<" : "
       <<na4.IsDeallocated(SIZE-1) <<endl;

  for (x4 = 1; x4 < SIZE+3; x4++)
  {
    Coords_[4] = x4;
    cout <<"Coordinates: ";
  
    for (x = 1; x <= SIZE; x++)
    {
      if (x > 1)
        cout <<", ";
      cout <<"pos[" <<x <<"] = " <<Coords_[x];
    }

    cout <<" | ";
    na4.SetCoordinates(Coords_);
    retval = na4.ReadFromArray();
    cout <<"retval = " <<retval <<endl;
  }

  Coords_[1] = SIZE-3;
  Coords_[2] = SIZE;
  Coords_[3] = SIZE+1;
  na4.SetCoordinates(Coords_);

  na4.DeleteVector(3);
  cout <<endl <<"DeleteVector(3)" <<endl;
  cout <<"IsDeallocated(3), "
       <<"pos[3] = " <<Coords_[3] <<" : "
       <<na4.IsDeallocated(3) <<endl;
  #endif

  //////////////////////////////////
  // testing 5 dimensional matrix //
  //////////////////////////////////
  #if NESTARRAY_ARRLEN5_DEBUG
  NestedArray5<int> na5(NULL);
  Lengths_ = na5.NewLengthsVector();
  Coords_ = na5.NewCoordinatesVector();
  SIZE = na5.MaxLength();
  cpos = (int*)memset(RawCallocateWith(MEMMATRIX, sizeof(int) * (SIZE+1)), 0, sizeof(int) * (SIZE+1));

  cout <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x > 0)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }

  Info_ = na5.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<"3rd level array init/cap = " <<Info_[4][0] <<endl;
  cout <<"4th level array init/cap = " <<Info_[5][0] <<endl;
  cout <<endl <<endl;

  for (x1 = 0; x1 < SIZE; x1++)
    for (x2 = 0; x2 < SIZE; x2++)
      for (x3 = 0; x3 < SIZE; x3++)
        for (x4 = 0; x4 < SIZE; x4++)
          for (x5 = 0; x5 < SIZE; x5++)
          {
            Coords_[1] = x1;
            Coords_[2] = x2;
            Coords_[3] = x3;
            Coords_[4] = x4;
            Coords_[5] = x5;
            cout <<"Coordinates: ";
  
            for (x = 1; x <= SIZE; x++)
            {
              if (x > 1)
                cout <<", ";
              cout <<"pos[" <<x <<"] = " <<Coords_[x];
            }

            cout <<" | ";
            data = (x1 * 100000000) + (x2 * 1000000) +
                   (x3 * 10000) + (x4 * 100) + x5;
            
            na5.SetCoordinates(Coords_);
            na5.WriteToArray(data);
            retval = na5.ReadFromArray();
            cout <<"retval = " <<retval <<endl;
            lpos = na5.CoordsToLinear(Coords_);
            cpos = na5.LinearToCoords(lpos, cpos);
            cout <<"linear pos = " <<lpos <<" : coords = (";

            for (x = 1; x <= SIZE; x++)
            {
              cout <<cpos[x];
              if (x < SIZE)
                cout <<",";
            }
            cout <<")" <<endl;

            for (x = 0; x < SIZE; x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVCount(" <<x <<") = " <<na5.RealVectorCount(x);
            }
            cout <<endl;

            for (x = 0; x < SIZE; x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVSize(" <<x <<") = " <<na5.RealVectorSize(x);
            }
            cout <<endl;

            ParentRefs_ = na5.ParentReference(1);
            max = ParentRefs_[0];
  
            for (x = 0; x <= max; x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"ParentRefs[" <<x <<"] = " <<ParentRefs_[x];
            }
            cout <<endl <<endl;
          }

  Lengths_[SIZE-5] = SIZE-1;
  Lengths_[SIZE-4] = SIZE;
  Lengths_[SIZE-3] = SIZE+1;
  Lengths_[SIZE-2] = SIZE+2;
  Lengths_[SIZE-1] = SIZE+3;

  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);

  cout <<endl;
  na5.SetLengths(Lengths_);
  
  cout <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }  

  Info_ = na5.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<"3rd level array init/cap = " <<Info_[4][0] <<endl;
  cout <<"4th level array init/cap = " <<Info_[5][0] <<endl;
  cout <<endl <<endl;

  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);

  for (x1 = 0; x1 < SIZE-1; x1++)
    for (x2 = 0; x2 < SIZE; x2++)
      for (x3 = 0; x3 < SIZE+1; x3++)
        for (x4 = 0; x4 < SIZE+2; x4++)
          for (x5 = 0; x5 < SIZE+3; x5++)
          {
            Coords_[1] = x1;
            Coords_[2] = x2;
            Coords_[3] = x3;
            Coords_[4] = x4;
            Coords_[5] = x5;
            cout <<"Coordinates: ";
  
            for (x = 1; x <= SIZE; x++)
            {
              if (x > 1)
                cout <<", ";
              cout <<"pos[" <<x <<"] = " <<Coords_[x];
            }

            cout <<" | ";
            data = (x1 * 100000000) + (x2 * 1000000) +
                   (x3 * 10000) + (x4 * 100) + x5;
            
            na5.SetCoordinates(Coords_);
            na5.WriteToArray(data);
            retval = na5.ReadFromArray();
            cout <<"retval = " <<retval <<endl;
            lpos = na5.CoordsToLinear(Coords_);
            cpos = na5.LinearToCoords(lpos, cpos);
            cout <<"linear pos = " <<lpos <<" : coords = (";

            for (x = 1; x <= SIZE; x++)
            {
              cout <<cpos[x];
              if (x < SIZE)
                cout <<",";
            }
            cout <<")" <<endl;

            for (x = 0; x < SIZE; x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVCount(" <<x <<") = " <<na5.RealVectorCount(x);
            }
            cout <<endl;

            for (x = 0; x < SIZE; x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVSize(" <<x <<") = " <<na5.RealVectorSize(x);
            }
            cout <<endl;

            ParentRefs_ = na5.ParentReference(1);
            max = ParentRefs_[0];
  
            for (x = 0; x <= max; x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"ParentRefs[" <<x <<"] = " <<ParentRefs_[x];
            }
            cout <<endl <<endl;
          }

  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);

  cout <<endl <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }  

  Info_ = na5.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<"3rd level array init/cap = " <<Info_[4][0] <<endl;
  cout <<"4th level array init/cap = " <<Info_[5][0] <<endl;
  cout <<endl;

  // testing vector deletion
  Coords_[1] = SIZE-2;
  Coords_[2] = SIZE-1;
  Coords_[3] = SIZE;
  Coords_[4] = SIZE+1;
  na5.SetCoordinates(Coords_);

  na5.DeleteVector(1);
  cout <<"DeleteVector(1)" <<endl;
  cout <<"IsDeallocated(1), "
       <<"pos[1] = " <<Coords_[1] <<" : "
       <<"pos[2] = " <<Coords_[2] <<" : "
       <<"pos[3] = " <<Coords_[3] <<" : "
       <<na5.IsDeallocated(1) <<endl;

  // testing replacement with vector deallocation
  sz = SizeOfVector(SIZE, SIZE-1, Lengths_);
  ReplVect_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
  ReplVect_[0] = sz-1;
  for (x = 1; x < sz; x++)
    ReplVect_[x] = x*10+x+1;

  Coords_[1] = SIZE-3;
  Coords_[2] = SIZE-1;
  Coords_[3] = SIZE;
  Coords_[4] = SIZE+1;
  na5.SetCoordinates(Coords_);
  cout <<"IsDeallocated(1), "
       <<"pos[1] = " <<Coords_[1] <<" : "
       <<"pos[2] = " <<Coords_[2] <<" : "
       <<"pos[3] = " <<Coords_[3] <<" : "
       <<na5.IsDeallocated(1) <<endl;

  // replacing final nesting level vector
  na5.ReplaceVector(SIZE-1, ReplVect_, true, true);
  cout <<endl;
  cout <<"ReplaceVector(" <<(SIZE-1) <<", ReplVect, true, true)" <<endl;
  cout <<"IsDeallocated(4), "
       <<"pos[4] = " <<Coords_[4] <<" : "
       <<na5.IsDeallocated(SIZE-1) <<endl;

  for (x5 = 1; x5 < SIZE+3; x5++)
  {
    Coords_[5] = x5;
    cout <<"Coordinates: ";
  
    for (x = 1; x <= SIZE; x++)
    {
      if (x > 1)
        cout <<", ";
      cout <<"pos[" <<x <<"] = " <<Coords_[x];
    }

    cout <<" | ";
    na5.SetCoordinates(Coords_);
    retval = na5.ReadFromArray();
    cout <<"retval = " <<retval <<endl;
  }

  // testing replacement without vector deallocation
  sz = SizeOfVector(SIZE, SIZE-1, Lengths_);
  ReplVect_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
  ReplVect_[0] = sz-1;
  for (x = 1; x < sz; x++)
    ReplVect_[x] = x*100+(x+1)*10+x+2;

  Coords_[1] = 1;
  Coords_[2] = SIZE-1;
  Coords_[3] = SIZE;
  Coords_[4] = SIZE+1;
  na5.SetCoordinates(Coords_);

  // replacing final nesting level vector
  na5.ReplaceVector(SIZE-1, ReplVect_, true, false);
  cout <<endl;
  cout <<"ReplaceVector(" <<(SIZE-1) <<", ReplVect, true, false)" <<endl;
  cout <<"IsDeallocated(4), "
       <<"pos[4] = " <<Coords_[4] <<" : "
       <<na5.IsDeallocated(SIZE-1) <<endl;

  for (x5 = 1; x5 < SIZE+3; x5++)
  {
    Coords_[5] = x5;
    cout <<"Coordinates: ";
  
    for (x = 1; x <= SIZE; x++)
    {
      if (x > 1)
        cout <<", ";
      cout <<"pos[" <<x <<"] = " <<Coords_[x];
    }

    cout <<" | ";
    na5.SetCoordinates(Coords_);
    retval = na5.ReadFromArray();
    cout <<"retval = " <<retval <<endl;
  }

  Coords_[1] = SIZE-4;
  Coords_[2] = SIZE-1;
  Coords_[3] = SIZE;
  Coords_[4] = SIZE+1;
  na5.SetCoordinates(Coords_);

  na5.DeleteVector(4);
  cout <<endl <<"DeleteVector(4)" <<endl;
  cout <<"IsDeallocated(4), "
       <<"pos[4] = " <<Coords_[4] <<" : "
       <<na5.IsDeallocated(4) <<endl;
  #endif
       

  //////////////////////////////////
  // testing 6 dimensional matrix //
  //////////////////////////////////
  #if NESTARRAY_ARRLEN6_DEBUG
  NestedArray6<int> na6(NULL, 4);
  na6.SetReInitArrays(0);
  Lengths_ = na6.NewLengthsVector();
  Coords_ = na6.NewCoordinatesVector();
  SIZE = na6.MaxLength();  
  cpos = (int*)memset(RawCallocateWith(MEMMATRIX, sizeof(int) * (na6.Dim()+1)), 0, sizeof(int) * (na6.Dim()+1));

  cout <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x > 0)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }

  Info_ = na6.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<"3rd level array init/cap = " <<Info_[4][0] <<endl;
  cout <<"4th level array init/cap = " <<Info_[5][0] <<endl;
  cout <<"5th level array init/cap = " <<Info_[6][0] <<endl;
  cout <<endl <<endl;

  for (x1 = 0; x1 < SIZE; x1++)
    for (x2 = 0; x2 < SIZE; x2++)
      for (x3 = 0; x3 < SIZE; x3++)
        for (x4 = 0; x4 < SIZE; x4++)
          for (x5 = 0; x5 < SIZE; x5++)
          for (x6 = 0; x6 < SIZE; x6++)
          {
            Coords_[1] = x1;
            Coords_[2] = x2;
            Coords_[3] = x3;
            Coords_[4] = x4;
            Coords_[5] = x5;
            Coords_[6] = x6;
            cout <<"Coordinates: ";
  
            for (x = 1; x <= na6.Dim(); x++)
            {
              if (x > 1)
                cout <<", ";
              cout <<"pos[" <<x <<"] = " <<Coords_[x];
            }

            cout <<" | ";
            data = (x1 * 100000) + (x2 * 10000) +
                   (x3 * 1000) + (x4 * 100) + (x5 * 10) + x6;
            
            na6.SetCoordinates(Coords_);
            na6.WriteToArray(data);
            retval = na6.ReadFromArray();
            cout <<"retval = " <<retval <<endl;
            lpos = na6.CoordsToLinear(Coords_);
            cpos = na6.LinearToCoords(lpos, cpos);
            cout <<"linear pos = " <<lpos <<" : coords = (";

            for (x = 1; x <= na6.Dim(); x++)
            {
              cout <<cpos[x];
              if (x < na6.Dim())
                cout <<",";
            }
            cout <<")" <<endl;

            for (x = 0; x < na6.Dim(); x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVCount(" <<x <<") = " <<na6.RealVectorCount(x);
            }
            cout <<endl;

            for (x = 0; x < na6.Dim(); x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVSize(" <<x <<") = " <<na6.RealVectorSize(x);
            }
            cout <<endl;

            ParentRefs_ = na6.ParentReference(1);
            max = ParentRefs_[0];
  
            for (x = 0; x <= max; x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"ParentRefs[" <<x <<"] = " <<ParentRefs_[x];
            }
            cout <<endl <<endl;
          }

  Lengths_[0] = SIZE-1;
  Lengths_[1] = SIZE-1;
  Lengths_[2] = SIZE;
  Lengths_[3] = SIZE+1;
  Lengths_[4] = SIZE;
  Lengths_[5] = SIZE+1;

  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);

  cout <<endl;
  na6.SetLengths(Lengths_);
  
  cout <<"Lengths: ";
  for (x = 0; x < na6.Dim(); x++)
  {
    if (x)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }  

  Info_ = na6.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<"3rd level array init/cap = " <<Info_[4][0] <<endl;
  cout <<"4th level array init/cap = " <<Info_[5][0] <<endl;
  cout <<"5th level array init/cap = " <<Info_[6][0] <<endl;
  cout <<endl <<endl;

  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);

  for (x1 = 0; x1 < SIZE-1; x1++)
    for (x2 = 0; x2 < SIZE-1; x2++)
      for (x3 = 0; x3 < SIZE; x3++)
        for (x4 = 0; x4 < SIZE+1; x4++)
          for (x5 = 0; x5 < SIZE; x5++)
          for (x6 = 0; x6 < SIZE+1; x6++)
          {
            Coords_[1] = x1;
            Coords_[2] = x2;
            Coords_[3] = x3;
            Coords_[4] = x4;
            Coords_[5] = x5;
            Coords_[6] = x6;
            cout <<"Coordinates: ";
  
            for (x = 1; x <= na6.Dim(); x++)
            {
              if (x > 1)
                cout <<", ";
              cout <<"pos[" <<x <<"] = " <<Coords_[x];
            }

            cout <<" | ";
            data = (x1 * 100000) + (x2 * 10000) +
                   (x3 * 1000) + (x4 * 100) + (x5 * 10) + x6;
            
            na6.SetCoordinates(Coords_);
            na6.WriteToArray(data);
            retval = na6.ReadFromArray();
            cout <<"retval = " <<retval <<endl;
            lpos = na6.CoordsToLinear(Coords_);
            cpos = na6.LinearToCoords(lpos, cpos);
            cout <<"linear pos = " <<lpos <<" : coords = (";

            for (x = 1; x <= na6.Dim(); x++)
            {
              cout <<cpos[x];
              if (x < na6.Dim())
                cout <<",";
            }
            cout <<")" <<endl;

            for (x = 0; x < na6.Dim(); x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVCount(" <<x <<") = " <<na6.RealVectorCount(x);
            }
            cout <<endl;

            for (x = 0; x < na6.Dim(); x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVSize(" <<x <<") = " <<na6.RealVectorSize(x);
            }
            cout <<endl;

            ParentRefs_ = na6.ParentReference(1);
            max = ParentRefs_[0];
  
            for (x = 0; x <= max; x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"ParentRefs[" <<x <<"] = " <<ParentRefs_[x];
            }
            cout <<endl <<endl;
          }

  if (argc > 1 && strcmp(argv[1], "-i") == 0)
    cin.getline(dum, 5);

  cout <<endl <<"Lengths: ";
  for (x = 0; x < na6.Dim(); x++)
  {
    if (x)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }  

  Info_ = na6.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<"3rd level array init/cap = " <<Info_[4][0] <<endl;
  cout <<"4th level array init/cap = " <<Info_[5][0] <<endl;
  cout <<"5th level array init/cap = " <<Info_[6][0] <<endl;
  cout <<endl;

  // testing vector deletion
  Coords_[1] = SIZE-2;
  Coords_[2] = SIZE-2;
  Coords_[3] = SIZE-1;
  Coords_[4] = SIZE;
  Coords_[5] = SIZE-1;
  na6.SetCoordinates(Coords_);

  na6.DeleteVector(1);
  cout <<"DeleteVector(1)" <<endl;
  cout <<"IsDeallocated(1), "
       <<"pos[1] = " <<Coords_[1] <<" : "
       <<"pos[2] = " <<Coords_[2] <<" : "
       <<"pos[3] = " <<Coords_[3] <<" : "
       <<"pos[4] = " <<Coords_[4] <<" : "
       <<na6.IsDeallocated(1) <<endl;

  // testing replacement with vector deallocation
  sz = SizeOfVector(SIZE, SIZE-1, Lengths_);
  ReplVect_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
  ReplVect_[0] = sz-1;
  for (x = 1; x < sz; x++)
    ReplVect_[x] = x*10+x+1;

  Coords_[1] = SIZE-3;
  Coords_[2] = SIZE-2;
  Coords_[3] = SIZE-1;
  Coords_[4] = SIZE;
  Coords_[5] = SIZE-1;
  na6.SetCoordinates(Coords_);
  cout <<"IsDeallocated(1), "
       <<"pos[1] = " <<Coords_[1] <<" : "
       <<"pos[2] = " <<Coords_[2] <<" : "
       <<"pos[3] = " <<Coords_[3] <<" : "
       <<"pos[4] = " <<Coords_[4] <<" : "
       <<na6.IsDeallocated(1) <<endl;

  // replacing final nesting level vector
  na6.ReplaceVector(SIZE-1, ReplVect_, true, true);
  cout <<endl;
  cout <<"ReplaceVector(" <<(SIZE-1) <<", ReplVect, true, true)" <<endl;
  cout <<"IsDeallocated(5), "
       <<"pos[5] = " <<Coords_[5] <<" : "
       <<na6.IsDeallocated(SIZE-1) <<endl;

  for (x6 = 1; x6 < SIZE+1; x6++)
  {
    Coords_[6] = x6;
    cout <<"Coordinates: ";
  
    for (x = 1; x <= na6.Dim(); x++)
    {
      if (x > 1)
        cout <<", ";
      cout <<"pos[" <<x <<"] = " <<Coords_[x];
    }

    cout <<" | ";
    na6.SetCoordinates(Coords_);
    retval = na6.ReadFromArray();
    cout <<"retval = " <<retval <<endl;
  }

  // testing replacement without vector deallocation
  sz = SizeOfVector(SIZE, SIZE-1, Lengths_);
  ReplVect_ = (int*)RawCallocateWith(MEMMATRIX, sizeof(int) * sz);
  ReplVect_[0] = sz-1;
  for (x = 1; x < sz; x++)
    ReplVect_[x] = x*100+(x+1)*10+x+2;

  Coords_[1] = 1;
  Coords_[2] = SIZE-2;
  Coords_[3] = SIZE-1;
  Coords_[4] = SIZE;
  Coords_[5] = SIZE-1;
  na6.SetCoordinates(Coords_);

  // replacing final nesting level vector
  na6.ReplaceVector(SIZE-1, ReplVect_, true, false);
  cout <<endl;
  cout <<"ReplaceVector(" <<(SIZE-1) <<", ReplVect, true, false)" <<endl;
  cout <<"IsDeallocated(5), "
       <<"pos[5] = " <<Coords_[5] <<" : "
       <<na6.IsDeallocated(SIZE-1) <<endl;

  for (x6 = 1; x6 < SIZE+1; x6++)
  {
    Coords_[6] = x6;
    cout <<"Coordinates: ";
  
    for (x = 1; x <= na6.Dim(); x++)
    {
      if (x > 1)
        cout <<", ";
      cout <<"pos[" <<x <<"] = " <<Coords_[x];
    }

    cout <<" | ";
    na6.SetCoordinates(Coords_);
    retval = na6.ReadFromArray();
    cout <<"retval = " <<retval <<endl;
  }

  Coords_[1] = 0;
  Coords_[2] = SIZE-2;
  Coords_[3] = SIZE-1;
  Coords_[4] = SIZE;
  Coords_[5] = SIZE-1;
  na6.SetCoordinates(Coords_);

  na6.DeleteVector(5);
  cout <<endl <<"DeleteVector(5)" <<endl;
  cout <<"IsDeallocated(5), "
       <<"pos[5] = " <<Coords_[5] <<" : "
       <<na6.IsDeallocated(5) <<endl;
  #endif

  //////////////////////////////////
  // testing 7 dimensional matrix //
  //////////////////////////////////
  #if NESTARRAY_ARRLEN7_DEBUG
  NestedArray7<int> na7(NULL, 3);
  Lengths_ = na7.NewLengthsVector();
  Coords_ = na7.NewCoordinatesVector();
  SIZE = na7.MaxLength();
  cpos = (int*)memset(RawCallocateWith(MEMMATRIX, sizeof(int) * (na7.Dim()+1)), 0, sizeof(int) * (na7.Dim()+1));

  cout <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x > 0)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }

  Info_ = na7.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<"3rd level array init/cap = " <<Info_[4][0] <<endl;
  cout <<"4th level array init/cap = " <<Info_[5][0] <<endl;
  cout <<"5th level array init/cap = " <<Info_[6][0] <<endl;
  cout <<"6th level array init/cap = " <<Info_[7][0] <<endl;
  cout <<endl <<endl;

  for (x1 = 0; x1 < SIZE; x1++)
    for (x2 = 0; x2 < SIZE; x2++)
      for (x3 = 0; x3 < SIZE; x3++)
        for (x4 = 0; x4 < SIZE; x4++)
          for (x5 = 0; x5 < SIZE; x5++)
          for (x6 = 0; x6 < SIZE; x6++)
          for (x7 = 0; x7 < SIZE; x7++)
          {
            Coords_[1] = x1;
            Coords_[2] = x2;
            Coords_[3] = x3;
            Coords_[4] = x4;
            Coords_[5] = x5;
            Coords_[6] = x6;
            Coords_[7] = x7;
            cout <<"Coordinates: ";
  
            for (x = 1; x <= na7.Dim(); x++)
            {
              if (x > 1)
                cout <<", ";
              cout <<"pos[" <<x <<"] = " <<Coords_[x];
            }

            cout <<" | ";
            data = (x1 * 1000000) + (x2 * 100000) +
                   (x3 * 10000) + (x4 * 1000) +
                   (x5 * 100) + (x6 * 10) + x7;
            
            na7.SetCoordinates(Coords_);
            na7.WriteToArray(data);
            retval = na7.ReadFromArray();
            cout <<"retval = " <<retval <<endl;
            lpos = na7.CoordsToLinear(Coords_);
            cpos = na7.LinearToCoords(lpos, cpos);
            cout <<"linear pos = " <<lpos <<" : coords = (";

            for (x = 1; x <= na7.Dim(); x++)
            {
              cout <<cpos[x];
              if (x < na7.Dim())
                cout <<",";
            }
            cout <<")" <<endl;

            for (x = 0; x < na7.Dim(); x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVCount(" <<x <<") = " <<na7.RealVectorCount(x);
            }
            cout <<endl;

            for (x = 0; x < na7.Dim(); x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVSize(" <<x <<") = " <<na7.RealVectorSize(x);
            }
            cout <<endl;

            ParentRefs_ = na7.ParentReference(1);
            max = ParentRefs_[0];
  
            for (x = 0; x <= max; x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"ParentRefs[" <<x <<"] = " <<ParentRefs_[x];
            }
            cout <<endl <<endl;
          }

  #endif

  //////////////////////////////////
  // testing 8 dimensional matrix //
  //////////////////////////////////
  #if NESTARRAY_ARRLEN8_DEBUG
  NestedArray8<int> na8(NULL, 3);
  Lengths_ = na8.NewLengthsVector();
  Coords_ = na8.NewCoordinatesVector();
  SIZE = na8.MaxLength();
  cpos = (int*)memset(RawCallocateWith(MEMMATRIX, sizeof(int) * (na8.Dim()+1)), 0, sizeof(int) * (na8.Dim()+1));

  cout <<"Lengths: ";
  for (x = 0; x < SIZE; x++)
  {
    if (x > 0)
      cout <<", ";
    cout <<"len[" <<x <<"] = " <<Lengths_[x];
  }

  Info_ = na8.GiveInfoArray();
  cout <<endl <<"array info: " <<endl;
  cout <<"top array cap = " <<Info_[0][0] <<endl;
  cout <<"top running count = " <<Info_[1][0] <<endl;
  cout <<"1st level array init/cap = " <<Info_[2][0] <<endl;
  cout <<"2nd level array init/cap = " <<Info_[3][0] <<endl;
  cout <<"3rd level array init/cap = " <<Info_[4][0] <<endl;
  cout <<"4th level array init/cap = " <<Info_[5][0] <<endl;
  cout <<"5th level array init/cap = " <<Info_[6][0] <<endl;
  cout <<"6th level array init/cap = " <<Info_[7][0] <<endl;
  cout <<"7th level array init/cap = " <<Info_[8][0] <<endl;
  cout <<endl <<endl;

  for (x1 = 0; x1 < SIZE; x1++)
    for (x2 = 0; x2 < SIZE; x2++)
      for (x3 = 0; x3 < SIZE; x3++)
        for (x4 = 0; x4 < SIZE; x4++)
          for (x5 = 0; x5 < SIZE; x5++)
          for (x6 = 0; x6 < SIZE; x6++)
          for (x7 = 0; x7 < SIZE; x7++)
          for (x8 = 0; x8 < SIZE; x8++)
          {
            Coords_[1] = x1;
            Coords_[2] = x2;
            Coords_[3] = x3;
            Coords_[4] = x4;
            Coords_[5] = x5;
            Coords_[6] = x6;
            Coords_[7] = x7;
            Coords_[8] = x8;
            cout <<"Coordinates: ";
  
            for (x = 1; x <= na8.Dim(); x++)
            {
              if (x > 1)
                cout <<", ";
              cout <<"pos[" <<x <<"] = " <<Coords_[x];
            }

            cout <<" | ";
            data = (x1 * 10000000) + (x2 * 1000000) +
                   (x3 * 100000) + (x4 * 10000) +
                   (x5 * 1000) + (x6 * 100) + (x7 * 10) + x8;
            
            na8.SetCoordinates(Coords_);
            na8.WriteToArray(data);
            retval = na8.ReadFromArray();
            cout <<"retval = " <<retval <<endl;
            lpos = na8.CoordsToLinear(Coords_);
            cpos = na8.LinearToCoords(lpos, cpos);
            cout <<"linear pos = " <<lpos <<" : coords = (";

            for (x = 1; x <= na8.Dim(); x++)
            {
              cout <<cpos[x];
              if (x < na8.Dim())
                cout <<",";
            }
            cout <<")" <<endl;

            for (x = 0; x < na8.Dim(); x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVCount(" <<x <<") = " <<na8.RealVectorCount(x);
            }
            cout <<endl;

            for (x = 0; x < na8.Dim(); x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"RVSize(" <<x <<") = " <<na8.RealVectorSize(x);
            }
            cout <<endl;

            ParentRefs_ = na8.ParentReference(1);
            max = ParentRefs_[0];
  
            for (x = 0; x <= max; x++)
            {
              if (x > 0)
                cout <<", ";
              cout <<"ParentRefs[" <<x <<"] = " <<ParentRefs_[x];
            }
            cout <<endl <<endl;
          }

  #endif

  ::DeleteArray(cpos);
  ::DeleteArray(Coords_);
  ::DeleteArray(Lengths_);  
}

#endif
#endif
/****************************************************************************/
#endif  // NESTARRAY_CPP
