// Window line buffer ADT classes
// The window line buffer class define methods for maintaining a window line
// object as parts used in a window screen object in a text console. The
// window line buffer class also has methods for positioning procedures for
// setting the display position within the window screen, setting the default
// message and default input into the window line, showing, hiding, erasing
// and retrieval of the input data.
//
// The window line buffer class define methods for maintaining an array of
// window line buffers. Methods are defined to show, hide, erase and
// retrieve from the entire array of window line buffers assigned to the
// window screen object.
//
#ifndef OVWINLINE_CPP
#define OVWINLINE_CPP
#ifndef OVWINLINE_H
  #include "ovwinline.h"
#endif

/****************************************************************************/
ChainPosInfo::ChainPosInfo():
_enterb(false),
_vbuf(false),
_inwbox(false),
_chainable(false),
_error(false),
_vmax(0),
_vendpt(0),

_WrapMax(0),
_WrapIndex(0),

_reallen(0),
_copylen(0),
_index(0),
_startpt(0),
_endpt(0),
_boxpt(0),

_ChainPos(-1),
_BufInfo(NULL),
_StartInfo(NULL),
_EndInfo(NULL)
{}

/****************************************************************************/
ChainPosInfo::~ChainPosInfo()
{
  delete _StartInfo;
  delete _EndInfo;
  
  _StartInfo =
  _EndInfo = NULL;
}

/****************************************************************************/
void ChainPosInfo::RestorePrevInfo()
{
  // note: _ChainPos in reverse order from ReturnToThis method
  ChainPosInfo* ptr_ = (_ChainPos == 0) ? _EndInfo:
                       (_ChainPos == 1) ? _StartInfo:
                       (_ChainPos == 2) ? this:NULL;

  if (_ChainPos == 0 || _ChainPos == 1)
  {
    _enterb = ptr_->_enterb;
    _inwbox = ptr_->_inwbox;
    _chainable = ptr_->_chainable;
    _error = ptr_->_error;
  
    _vbuf = ptr_->_vbuf;
    _vmax = ptr_->_vmax;
    _vendpt = ptr_->_vendpt;

    _WrapMax = ptr_->_WrapMax;
    _WrapIndex = ptr_->_WrapIndex;

    _reallen = ptr_->_reallen;
    _copylen = ptr_->_copylen;
    _index = ptr_->_index;
    _startpt = ptr_->_startpt;
    _endpt = ptr_->_endpt;
    _boxpt = ptr_->_boxpt;
    _ChainPos = ptr_->_ChainPos;
  }
}

/****************************************************************************/
void ChainPosInfo::ReturnToThis()
{
  ChainPosInfo* ptr_ = (_ChainPos == 0) ? _StartInfo:
                       (_ChainPos == 1) ? _EndInfo:
                       (_ChainPos == 2) ? this:NULL;

  if (_ChainPos == 0 || _ChainPos == 1)
  {
    _enterb = ptr_->_enterb;
    _inwbox = ptr_->_inwbox;
    _chainable = ptr_->_chainable;
    _error = ptr_->_error;
  
    _vbuf = ptr_->_vbuf;
    _vmax = ptr_->_vmax;
    _vendpt = ptr_->_vendpt;

    _WrapMax = ptr_->_WrapMax;
    _WrapIndex = ptr_->_WrapIndex;

    _reallen = ptr_->_reallen;
    _copylen = ptr_->_copylen;
    _index = ptr_->_index;
    _startpt = ptr_->_startpt;
    _endpt = ptr_->_endpt;
    _boxpt = ptr_->_boxpt;
    _ChainPos = ptr_->_ChainPos;
  }
}

/****************************************************************************/
void ChainPosInfo::CopyToPosInfo(WindowLineBuffer* Ptr_)
{
  ChainPosInfo* ptr_ = (_ChainPos == 0) ? _StartInfo:
                       (_ChainPos == 1) ? _EndInfo:
                       (_ChainPos == 2) ? this:NULL;

  if (_ChainPos == 0 || _ChainPos == 1)
  {
    ptr_->_enterb = Ptr_->IsEnterable();
    ptr_->_inwbox = Ptr_->BufferChainPosition() == 0 && ptr_->_enterb &&
                    ((WindowLine*)(Ptr_->_Owner))->LineWrapAllowed();
  
    ptr_->_vbuf = Ptr_->BufferChainPosition() == 0 && ptr_->_enterb &&
                  Ptr_->VirtualBufferOffset(&ptr_->_vendpt);
    ptr_->_vmax = ptr_->_enterb ? Ptr_->VirtualBufferMax():
                                  Ptr_->_maxbuf;

    ptr_->_WrapMax = ptr_->_enterb ? ((WindowLine*)(Ptr_->_Owner))->GetMaxWrapAround():0;
    ptr_->_WrapIndex = ptr_->_enterb ? ((WindowLine*)(Ptr_->_Owner))->GetWrapAroundIndex():0;

    ptr_->_reallen =
    ptr_->_copylen = ::SafeStrLen(Ptr_->_buffer);
    ptr_->_index = Ptr_->_index;
    ptr_->_startpt = 0;
    ptr_->_endpt = Ptr_->_len;
    ptr_->_boxpt = Ptr_->_len;

    if (ptr_->_copylen > ptr_->_boxpt)
      ptr_->_copylen = ptr_->_boxpt;
  }
}

/****************************************************************************/
void ChainPosInfo::CopyInfoToThis(WindowLineBuffer* Ptr_)
{
  _enterb = Ptr_->IsEnterable();
  _inwbox = Ptr_->BufferChainPosition() == 0 && _enterb &&
            ((WindowLine*)(Ptr_->_Owner))->LineWrapAllowed();
  
  _vbuf = Ptr_->BufferChainPosition() == 0 && _enterb &&
          Ptr_->VirtualBufferOffset(&_vendpt);
  _vmax = _enterb ? Ptr_->VirtualBufferMax():
                    Ptr_->_maxbuf;

  _WrapMax = _enterb ? ((WindowLine*)(Ptr_->_Owner))->GetMaxWrapAround():0;
  _WrapIndex = _enterb ? ((WindowLine*)(Ptr_->_Owner))->GetWrapAroundIndex():0;

  _reallen =
  _copylen = ::SafeStrLen(Ptr_->_buffer);
  _index = Ptr_->_index;
  _startpt = 0;
  _endpt = Ptr_->_len;
  _boxpt = Ptr_->_len;

  if (_copylen > _boxpt)
    _copylen = _boxpt;
}

/****************************************************************************/
void ChainPosInfo::SetupInfoPtrs(WindowLineBuffer* Ptr_, int SeqPos_)
{
  _BufInfo = Ptr_;

  if (!_StartInfo)
    _StartInfo = new ChainPosInfo;

  if (!_EndInfo)
    _EndInfo = new ChainPosInfo;
  
  if (_BufInfo)
  {
    _ChainPos = (SeqPos_ == 0 && _BufInfo->VirtualBufferOffset() &&
                 _BufInfo->_bufferoffset == 0) ? 0:
                (SeqPos_ == 1 && !_BufInfo->VirtualBufferOffset() &&
                 _BufInfo->_bufferoffset != 0) ? 1:
                (SeqPos_ == 2) ? 2:-1;

    if (_ChainPos == 2)
      CopyInfoToThis(_BufInfo);
    else
      CopyToPosInfo(_BufInfo);
  }
  else
    _ChainPos = -1;
}

/****************************************************************************/
void ChainPosInfo::SetPullInfo(int start)
{
  if (!_BufInfo)
    return;

  ChainPosInfo* ptr_ = (_ChainPos == 0) ? _StartInfo:
                       (_ChainPos == 1) ? _EndInfo:
                       (_ChainPos == 2) ? this:NULL;

  if (!ptr_ || _ChainPos < 0)
    return;

  int rm_, cm_;
  int offset_ = 0;
  ptr_->_error = false;
  ptr_->_ChainPos = _ChainPos;
  ptr_->_BufInfo = _BufInfo;
  ptr_->_chainable = _BufInfo->IsChainable(false);
  ptr_->_index = start + ptr_->_startpt;

  if (_ChainPos == 0 && ptr_->_chainable && ptr_->_vbuf)
  {
    offset_ = (_BufInfo->_inboxoffset <= ptr_->_vmax) ?
                    _BufInfo->_inboxoffset:
                    ptr_->_vmax;

    ptr_->_startpt += offset_;
    ptr_->_index = start + ptr_->_startpt;
    ptr_->_endpt += offset_;
    ptr_->_boxpt = ptr_->_endpt - ptr_->_startpt;
  }
  else if (_BufInfo->_bufferoffset > 0 && !ptr_->_enterb)
  {
    ptr_->_startpt = 0;
    ptr_->_index = start;
    ptr_->_endpt = _BufInfo->_len;
    ptr_->_boxpt = _BufInfo->_len;
  }

  rm_ = cm_ = ::SafeStrLen(&_BufInfo->_buffer[ptr_->_startpt]);
  if (cm_ > ptr_->_endpt)
    cm_ = ptr_->_endpt;

  offset_ = (ptr_->_reallen - offset_) - ptr_->_copylen;
  ptr_->_copylen = (ptr_->_index < ptr_->_endpt) ?
                         ::SafeStrLen(&_BufInfo->_buffer[ptr_->_index]):0;
  ptr_->_reallen = rm_;
  ptr_->_copylen -= (offset_ > 0) ? offset_:0;
  if (ptr_->_copylen < 0)
    ptr_->_copylen = 0;
  else if (ptr_->_copylen > ptr_->_boxpt)
    ptr_->_copylen = ptr_->_boxpt;

  if (!(ptr_->_startpt <= ptr_->_index && ptr_->_index < ptr_->_endpt) ||
      !((rm_ > cm_ && ptr_->_endpt < rm_) ||
        (rm_ == cm_ && ptr_->_endpt >= rm_)) ||
      !(ptr_->_copylen <= ptr_->_reallen && cm_ <= rm_))
    ptr_->_error = true;
}

/****************************************************************************/
void ChainPosInfo::SetPushInfo(int start)
{
  if (!_BufInfo)
    return;

  ChainPosInfo* ptr_ = (_ChainPos == 0) ? _StartInfo:
                       (_ChainPos == 1) ? _EndInfo:
                       (_ChainPos == 2) ? this:NULL;

  if (!ptr_ || _ChainPos < 0)
    return;

  int rm_, cm_;
  int offset_ = 0;
  ptr_->_error = false;
  ptr_->_ChainPos = _ChainPos;
  ptr_->_BufInfo = _BufInfo;
  ptr_->_chainable = _BufInfo->IsChainable(false);
  ptr_->_index = start + ptr_->_startpt;

  if (_ChainPos == 0 && ptr_->_chainable && ptr_->_vbuf)
  {
    offset_ = (_BufInfo->_inboxoffset <= ptr_->_vmax) ?
                    _BufInfo->_inboxoffset:
                    ptr_->_vmax;

    ptr_->_startpt += offset_;
    ptr_->_index = start + ptr_->_startpt;
    ptr_->_endpt += offset_;
    ptr_->_boxpt = ptr_->_endpt - ptr_->_startpt;
  }
  else if (_BufInfo->_bufferoffset > 0 && !ptr_->_enterb)
  {
    ptr_->_startpt = 0;
    ptr_->_index = start;
    ptr_->_endpt = _BufInfo->_len;
    ptr_->_boxpt = _BufInfo->_len;
  }

  rm_ = cm_ = ::SafeStrLen(&_BufInfo->_buffer[ptr_->_startpt]);
  if (cm_ > ptr_->_endpt)
    cm_ = ptr_->_endpt;

  offset_ = (ptr_->_reallen - offset_) - ptr_->_copylen;
  ptr_->_copylen = (ptr_->_index < ptr_->_endpt) ?
                         ::SafeStrLen(&_BufInfo->_buffer[ptr_->_index]):0;
  ptr_->_reallen = rm_;
  ptr_->_copylen -= (offset_ > 0) ? offset_:0;
  if (ptr_->_copylen < 0)
    ptr_->_copylen = 0;
  else if (ptr_->_copylen > ptr_->_boxpt)
    ptr_->_copylen = ptr_->_boxpt;

  if (!(ptr_->_startpt <= ptr_->_index && ptr_->_index < ptr_->_endpt) ||
      !((rm_ > cm_ && ptr_->_endpt < rm_) ||
        (rm_ == cm_ && ptr_->_endpt >= rm_)) ||
      !(ptr_->_copylen <= ptr_->_reallen && cm_ <= rm_))
    ptr_->_error = true;
}

/****************************************************************************/
// WindowLineBuffer class definition
/****************************************************************************/
const int* InsertDetect(void* Owner_, WindowLineBuffer*, const int* Ptr_)
{
  if (Owner_)
    ((WindowLineBuffer*)Owner_)->NotifyInsertMode(Ptr_);

  return Ptr_;
}

/****************************************************************************/
// PURPOSE:
//   Prompt box default constructor
//
// PRE:
//   an uninitialize Prompt box
//
// POST:
//   a Prompt box is constructed and assigned the default values specified.
//   If the msg argument is not NULL then the Prompt message of the box
//   is assigned to the argument string.
//   The string buffer is Cleared to all 0.
//
WindowLineBuffer::WindowLineBuffer(char* msg, WindowLineBuffer* next_, int Size_, int Length_):
_Owner(NULL),
_nextline(next_),
_refline(NULL),
_ChainPosPtr(NULL),
_Enterable(TRUE),
_Selectable(FALSE),
_msg(msg ? strcpy(new char[::SafeStrLen(msg)+1], msg):NULL),
_buffer(Size_ ? (new char[Size_ + 1]):(new char[MAXBUFFER + 1])),
_retbuf(Size_ ? (new char[Size_ + 1]):(new char[MAXBUFFER + 1])),
_retbufalloc(true),
_bufferoffset(0),
_inboxoffset(0),
_inwboxcnt(0),
_MovedToEnd(false),
_EndCharInserted(0),
_EndCharReverted(false),
_maxbuf(Size_ ? Size_:MAXBUFFER),
_len((Length_ && Length_ <= Size_) ? Length_:
     Size_ ? Size_:1),   // _len should never be zero
_minlen(1),              // minimum length should be at least 1
_chainminlen(0),         // no chains at initialization so zero
_index(0),
_xpos(WHEREX()),
_ypos(WHEREY()),
_noempty(0),
_InsertMode(TRUE),
_Escaped(FALSE),
_Hidden(TRUE),
_TextWidgetActive(TRUE),
_ChainPos(0),
_InFocus(FALSE),
_SavedCursX(WHEREX()),
_SavedCursY(WHEREY()),
_BufChainNum(0),
_WinLineNum(0),
_WrapChars(false),
_RestFullBox(true),
_EndBrksActive(false),
_TrimWs(true),

// Internal data monitors
_insert_monitor(NULL),
_buffer_monitor(NULL),
_key_monitor(NULL),
_ext_monitor(NULL),
_index_monitor(NULL),
_escaped_monitor(NULL),
_hidden_monitor(NULL),
_active_monitor(NULL),
_infocus_monitor(NULL),
_inboxoffset_monitor(NULL),
_winlinenum_monitor(NULL),

// Handles to external objects, used by data monitors
_insert_handle(NULL),
_buffer_handle(NULL),
_key_handle(NULL),
_ext_handle(NULL),
_index_handle(NULL),
_escaped_handle(NULL),
_hidden_handle(NULL),
_active_handle(NULL),
_infocus_handle(NULL),
_inboxoffset_handle(NULL),
_winlinenum_handle(NULL)
{
  _ChainPosPtr = new ChainPosInfo;
  _brackets[0] = CONTMORE_LEFT;  // Continue more to the left (left triangle)
  _brackets[1] = CONTMORE_RIGHT; // Continue more to the right (right triangle)

  for (int index = 0; index <= _maxbuf; index++)
    _buffer[index] = '\0';
}

/****************************************************************************/
// PURPOSE:
//   Prompt box copy construcor
//
// PRE:
//   An uninitialized Prompt box.
//
// POST:
//   a Prompt box is constructed and assigned the corresponding values
//   in the Prompt box argument passed to the constructor.
//
WindowLineBuffer::WindowLineBuffer(WindowLineBuffer& pb, int Size_, int Length_):
_Owner(NULL),
_nextline(NULL),
_refline(NULL),
_ChainPosPtr(NULL),
_Enterable(pb._Enterable),
_Selectable(pb._Selectable),
_msg(pb._msg ? strcpy(new char[::SafeStrLen(pb._msg)+1], pb._msg):NULL),
_buffer(pb._maxbuf ? (new char[pb._maxbuf + 1]):
        Size_ ? (new char[Size_ + 1]):
                (new char[MAXBUFFER + 1])),
_retbuf(pb._maxbuf ? (new char[pb._maxbuf + 1]):
        Size_ ? (new char[Size_ + 1]):
                (new char[MAXBUFFER + 1])),
_retbufalloc(true),
_bufferoffset(0),
_inboxoffset(0),
_inwboxcnt(0),
_MovedToEnd(false),
_EndCharInserted(0),
_EndCharReverted(false),
_maxbuf(pb._maxbuf ? pb._maxbuf:
        Size_ ? Size_:MAXBUFFER),
_len(pb._len ? pb._len:
     (Length_ && Length_ <= Size_) ? Length_:
     Size_ ? Size_:1),   // _len should never be zero
_minlen(pb._minlen ? pb._minlen:1),
_chainminlen(0),
_index(pb._index),
_xpos(pb._xpos),
_ypos(pb._ypos),
_noempty(pb._noempty),
_InsertMode(TRUE),
_Escaped(FALSE),
_Hidden(TRUE),
_TextWidgetActive(TRUE),
_ChainPos(0),
_InFocus(FALSE),
_SavedCursX(WHEREX()),
_SavedCursY(WHEREY()),
_BufChainNum(pb._BufChainNum),
_WinLineNum(pb._WinLineNum),
_WrapChars(false),
_RestFullBox(pb._RestFullBox),
_EndBrksActive(pb._EndBrksActive),
_TrimWs(pb._TrimWs),

// Internal data monitors
_insert_monitor(NULL),
_buffer_monitor(NULL),
_key_monitor(NULL),
_ext_monitor(NULL),
_index_monitor(NULL),
_escaped_monitor(NULL),
_hidden_monitor(NULL),
_active_monitor(NULL),
_infocus_monitor(NULL),
_inboxoffset_monitor(NULL),
_winlinenum_monitor(NULL),

// Handles to external objects, used by data monitors
_insert_handle(NULL),
_buffer_handle(NULL),
_key_handle(NULL),
_ext_handle(NULL),
_index_handle(NULL),
_escaped_handle(NULL),
_hidden_handle(NULL),
_active_handle(NULL),
_infocus_handle(NULL),
_inboxoffset_handle(NULL),
_winlinenum_handle(NULL)
{
  _ChainPosPtr = new ChainPosInfo;
  _brackets[0] = CONTMORE_LEFT;  // Continue more to the left (left triangle)
  _brackets[1] = CONTMORE_RIGHT; // Continue more to the right (right triangle)

  strcpy(_buffer, pb._buffer);
  for (int index = ::SafeStrLen(_buffer); index <= _maxbuf; index++)
    _buffer[index] = '\0';

  if (pb._minlen)
    SetMinimumLength(pb._minlen);
  else
    SetMinimumLength(1);
}

/****************************************************************************/
// PURPOSE:
//   Prompt box copy construcor for offsetted virtual buffer
//
// PRE:
//   An uninitialized Prompt box.
//
// POST:
//   a Prompt box is constructed and assigned the corresponding values
//   in the Prompt box argument passed to the constructor.
//
WindowLineBuffer::WindowLineBuffer(WindowLineBuffer& pb, int Size_, int Length_, int Offset_):
_Owner(NULL),
_nextline(NULL),
_refline(NULL),
_ChainPosPtr(NULL),
_Enterable(FALSE),
_Selectable(FALSE),
_msg(NULL),
_buffer((pb._maxbuf > Offset_) ? (pb._buffer + Offset_):NULL),
_retbuf(pb._retbuf),
_retbufalloc(false),
_bufferoffset(Offset_),
_inboxoffset(0),
_inwboxcnt(0),
_MovedToEnd(false),
_EndCharInserted(0),
_EndCharReverted(false),
_maxbuf((pb._maxbuf > Offset_) ? (pb._maxbuf - Offset_):
        (Size_ > Offset_) ? Size_:MAXBUFFER),
_len((pb._maxbuf > Offset_) ? (pb._maxbuf - Offset_):
     (Length_ && Length_ <= Size_ &&
      Length_ > Offset_) ? Length_:
     (Size_ > Offset_) ? Size_:1),   // _len should never be zero
_minlen(pb._minlen ? pb._minlen:1),
_chainminlen(0),
_index(0),
_xpos(pb._xpos),
_ypos(pb._ypos),
_noempty(pb._noempty),
_InsertMode(TRUE),
_Escaped(FALSE),
_Hidden(TRUE),
_TextWidgetActive(TRUE),
_ChainPos(0),
_InFocus(FALSE),
_SavedCursX(WHEREX()),
_SavedCursY(WHEREY()),
_BufChainNum(pb._BufChainNum + 1),
_WinLineNum(pb._WinLineNum),
_WrapChars(false),
_RestFullBox(pb._RestFullBox),
_EndBrksActive(pb._EndBrksActive),
_TrimWs(pb._TrimWs),

// Internal data monitors
_insert_monitor(NULL),
_buffer_monitor(NULL),
_key_monitor(NULL),
_ext_monitor(NULL),
_index_monitor(NULL),
_escaped_monitor(NULL),
_hidden_monitor(NULL),
_active_monitor(NULL),
_infocus_monitor(NULL),
_inboxoffset_monitor(NULL),
_winlinenum_monitor(NULL),

// Handles to external objects, used by data monitors
_insert_handle(NULL),
_buffer_handle(NULL),
_key_handle(NULL),
_ext_handle(NULL),
_index_handle(NULL),
_escaped_handle(NULL),
_hidden_handle(NULL),
_active_handle(NULL),
_infocus_handle(NULL),
_inboxoffset_handle(NULL),
_winlinenum_handle(NULL)
{
  int index;

  _ChainPosPtr = new ChainPosInfo;
  _brackets[0] = CONTMORE_LEFT;  // Continue more to the left (left triangle)
  _brackets[1] = CONTMORE_RIGHT; // Continue more to the right (right triangle)

  if (IsEnterable())
  {
    strcpy(_buffer, pb._buffer);
    for (index = ::SafeStrLen(_buffer); index <= _maxbuf; index++)
      _buffer[index] = '\0';
  }
  else
  {
    for (index = 0; index <= _maxbuf; index++)
      _buffer[index] = '\0';
  }

  if (_buffer == pb._buffer+Offset_ &&
      _maxbuf == pb._maxbuf-Offset_ &&
      _len == pb._maxbuf-Offset_)
    _bufferoffset = Offset_;
  else
    _bufferoffset = -1;

  if (pb._minlen)
    SetMinimumLength(pb._minlen);
  else
    SetMinimumLength(1);

  if (_len <= 0)
    SetInputLength(1, false);   // _len should never be zero
}

/****************************************************************************/
const char* WindowLineBuffer::cMonitor(const char*(*monptr_)(void*, WindowLineBuffer*, const char*), void* This_, const char* ptr_)
{
  if (monptr_ != NULL)
  {
    SaveCursorPosition();
    const char* RetVal_ = (*monptr_)(This_, this, ptr_);
    RestoreCursorPosition();

    return RetVal_;
  }

  return ptr_;
}

/****************************************************************************/
const int* WindowLineBuffer::iMonitor(const int*(*monptr_)(void*, WindowLineBuffer*, const int*), void* This_, const int* ptr_)
{
  if (monptr_ != NULL)
  {
    SaveCursorPosition();
    const int* RetVal_ = (*monptr_)(This_, this, ptr_);
    RestoreCursorPosition();

    return RetVal_;
  }

  return ptr_;
}

/****************************************************************************/
void WindowLineBuffer::SaveCursorPosition()
{
  _SavedCursX = WHEREX();
  _SavedCursY = WHEREY();
}

/****************************************************************************/
void WindowLineBuffer::RestoreCursorPosition()
{
  GOTOXY(_SavedCursX, _SavedCursY);
}

/****************************************************************************/
const int* WindowLineBuffer::MonitorInsertMode(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*))
{
  _insert_handle = This_;
  _insert_monitor = FnPtr_;

  if (_nextline)
    _nextline->MonitorInsertMode(_insert_handle, _insert_monitor);

  return &_InsertMode;
}

/****************************************************************************/
const char* WindowLineBuffer::MonitorBuffer(void* This_, const char*(*FnPtr_)(void*, WindowLineBuffer*, const char*))
{
  _buffer_handle = This_;
  _buffer_monitor = FnPtr_;

  if (_nextline)
    _nextline->MonitorBuffer(_buffer_handle, _buffer_monitor);

  return _buffer;
}

/****************************************************************************/
const int* WindowLineBuffer::MonitorKey(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*))
{
  _key_handle = This_;
  _key_monitor = FnPtr_;

  if (_nextline)
    _nextline->MonitorKey(_key_handle, _key_monitor);

  return &_key;
}

/****************************************************************************/
const int* WindowLineBuffer::MonitorExtendedKey(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*))
{
  _ext_handle = This_;
  _ext_monitor = FnPtr_;

  if (_nextline)
    _nextline->MonitorExtendedKey(_ext_handle, _ext_monitor);

  return &_extended;
}

/****************************************************************************/
const int* WindowLineBuffer::MonitorIndex(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*))
{
  _index_handle = This_;
  _index_monitor = FnPtr_;

  if (_nextline)
    _nextline->MonitorIndex(_index_handle, _index_monitor);

  return &_index;
}

/****************************************************************************/
const int* WindowLineBuffer::MonitorEscapedStatus(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*))
{
  _escaped_handle = This_;
  _escaped_monitor = FnPtr_;

  if (_nextline)
    _nextline->MonitorEscapedStatus(_escaped_handle, _escaped_monitor);

  return &_Escaped;
}

/****************************************************************************/
const int* WindowLineBuffer::MonitorInFocusStatus(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*))
{
  _infocus_handle = This_;
  _infocus_monitor = FnPtr_;

  if (_nextline)
    _nextline->MonitorInFocusStatus(_infocus_handle, _infocus_monitor);

  return &_InFocus;
}

/****************************************************************************/
const int* WindowLineBuffer::MonitorHiddenStatus(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*))
{
  _hidden_handle = This_;
  _hidden_monitor = FnPtr_;
  return &_Hidden;
}

/****************************************************************************/
const int* WindowLineBuffer::MonitorActiveStatus(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*))
{
  _active_handle = This_;
  _active_monitor = FnPtr_;
  return &_TextWidgetActive;
}

/****************************************************************************/
const int* WindowLineBuffer::MonitorInboxOffset(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*))
{
  _inboxoffset_handle = This_;
  _inboxoffset_monitor = FnPtr_;

  if (_nextline)
    _nextline->MonitorInboxOffset(_inboxoffset_handle, _inboxoffset_monitor);

  return &_inboxoffset;
}

/****************************************************************************/
const int* WindowLineBuffer::MonitorWinLineNum(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*))
{
  _winlinenum_handle = This_;
  _winlinenum_monitor = FnPtr_;

  if (_nextline)
    _nextline->MonitorWinLineNum(_winlinenum_handle, _winlinenum_monitor);

  return &_WinLineNum;
}

/****************************************************************************/
// PURPOSE:
//   Prompt box data member destruction method
//
// PRE:
//   The data members of this Prompt box object with original values
//
// POST:
//   If the data members of the box object is not NULL then the data members
//   are deleted and assigned NULL
//
void WindowLineBuffer::Clear()
{
  delete[] _msg;
  _msg = NULL;

  delete _refline;
  _refline = NULL;

  if (IsEnterable() && !_bufferoffset)
  {
    delete[] _buffer;
    _buffer = NULL;
  }

  if (_retbufalloc)
  {
    delete[] _retbuf;
    _retbuf = NULL;
  }
  
  cMonitor(_buffer_monitor, _buffer_handle, _buffer);
}

/****************************************************************************/
/****************************************************************************/
// PURPOSE:
//   Prompt box assignment operator
//
// PRE:
//   The data members of this Prompt box object with original values
//
// POST:
//   The data members of this object is assigned to the corresponding
//   data members of the Prompt box argument passed to the method.
//   This object is returned.
//
WindowLineBuffer& WindowLineBuffer::operator = (WindowLineBuffer& pb)
{
  if (this != &pb)
  {
    Clear();

    _nextline = NULL;
    _Enterable = _bufferoffset ? FALSE:pb._Enterable;
    _Selectable = pb._Selectable;
    _msg = pb._msg ? strcpy(new char[::SafeStrLen(pb._msg)+1], pb._msg):NULL;
    _maxbuf = (pb._maxbuf ? pb._maxbuf:
               _len ? _len:MAXBUFFER);
    _len = (pb.InputLength() ? pb.InputLength():
           (_maxbuf <= SCREEN_STDWIDTH) ? _maxbuf:1);
    _index = pb._index;
    _xpos = pb._xpos;
    _ypos = pb._ypos;
    _noempty = pb._noempty;
    _RestFullBox = pb._RestFullBox;
    _EndBrksActive = pb._EndBrksActive;
    _brackets[0] = pb._brackets[0];
    _brackets[1] = pb._brackets[1];

    if (IsEnterable() && ::SafeStrLen(pb._buffer) > ::SafeStrLen(_buffer))
    {
      delete[] _buffer;
      _buffer = (_maxbuf ? (new char[_maxbuf + 1]):
                 _len ? (new char[_len + 1]):
                        (new char[MAXBUFFER + 1]));

      pb.SetReturnBuffer(this, pb._retbuf, true, _maxbuf ? _maxbuf:
                                                 _len ? _len:MAXBUFFER);
    }
    else if (!IsEnterable())
      pb.SetReturnBuffer(this, pb._retbuf, false);

    cMonitor(_buffer_monitor, _buffer_handle, _buffer);

    if (IsEnterable())
    {
      _inboxoffset = 0;
      _inwboxcnt = 0;
      _MovedToEnd = false;
      _EndCharInserted = 0;
      _EndCharReverted = false;
      
      strcpy(_buffer, pb._buffer);
      for (int index = ::SafeStrLen(_buffer); index <= _maxbuf; index++)
        _buffer[index] = '\0';
    }

    if (pb._minlen)
      SetMinimumLength(pb._minlen);
    else
      SetMinimumLength(1);
  }

  return *this;
}

/****************************************************************************/
WindowLineBuffer* WindowLineBuffer::Make(WindowLineBuffer& pbox, int Size_, int Length_)
{
  WindowLineBuffer* pb;
  pb = new WindowLineBuffer(pbox, Size_, Length_);
  return pb;
}

/****************************************************************************/
WindowLineBuffer* WindowLineBuffer::Make(char* msg, WindowLineBuffer* next_, int Size_, int Length_)
{
  WindowLineBuffer* pb;
  pb = new WindowLineBuffer(msg, next_, Size_, Length_);
  return pb;
}

/****************************************************************************/
WindowLineBuffer* WindowLineBuffer::MakeNextLine(WindowLineBuffer* wpb)
{
  WindowLineBuffer* nwlb;
  nwlb = wpb ? (new WindowLineBuffer(*wpb, wpb->_maxbuf, wpb->_len)):NULL;

  if (wpb && nwlb)
  {
    int msglen_ = ::SafeStrLen(wpb->_msg);
    nwlb->SetInputLength(wpb->_len, false);
    nwlb->SetXpos(wpb->_xpos + msglen_ + wpb->_len +
                  wpb->EndBrksLength(false));
    nwlb->SetYpos(wpb->_ypos);
  }

  return nwlb;
}

/****************************************************************************/
WindowLineBuffer* WindowLineBuffer::MakeVirtualLine(WindowLineBuffer* wpb)
{
  WindowLineBuffer* nwlb;
  nwlb = wpb ? (new WindowLineBuffer(*wpb, wpb->_maxbuf, wpb->_len, _len)):NULL;

  if (wpb && nwlb && nwlb->_bufferoffset == _len)
  {
    nwlb->Hide();
    nwlb->SetXpos(wpb->_xpos);
    nwlb->SetYpos(wpb->_ypos);
    Hide();
  }
  else if (nwlb->_bufferoffset <= 0 || !nwlb->_buffer)
  {
    delete nwlb;
    nwlb = NULL;
  }
  
  return nwlb;
}

/****************************************************************************/
/****************************************************************************/
void WindowLineBuffer::PrintBlanks(int Max_)
{
  for (int Index_ = 0; Index_ < Max_; Index_++)
    PUTCH(BLANK);
}

/****************************************************************************/
// PURPOSE:
//   Set Prompt message method
//
// PRE:
//   The original value of the message data member
//
// POST:
//   The message data member is assigned new memory copied from the argument
//   string.
//
char* WindowLineBuffer::SetMessage(const char* str)
{
  if (str != _msg)
  {
    delete[] _msg;
    _msg = str ? strcpy(new char[::SafeStrLen(str)+1], str):NULL;
  }

  return _msg;
}

/****************************************************************************/
void WindowLineBuffer::RemoveMessage()
{
  SetMessage(NULL);
}

/****************************************************************************/
void WindowLineBuffer::BlankMessage()
{
  int i, Len_;
  Len_ = ::SafeStrLen(_msg);

  for (i = 0; i < Len_; ++i)
    _msg[i] = ' ';
}

/****************************************************************************/
void WindowLineBuffer::PushInputString(char* str)
{
  if (!str)
    return;

  bool enterb_ = IsEnterable();
  bool vbuf_ = VirtualBufferOffset() && enterb_;
  short Index_;
  char* temps_ = strcpy(new char[::SafeStrLen(str)+1], str);

  if (vbuf_)
    _ChainPosPtr->SetupInfoPtrs(this, 0);
    
  if (IsValidString(str))
  {
    temps_ = Convert(temps_);
  
    if (vbuf_)
    {
      for (Index_ = ::SafeStrLen(temps_) - 1; Index_ >= 0; Index_--)
      {
        _ChainPosPtr->SetPushInfo(0);
        Push(0, temps_[Index_], _ChainPosPtr);
      }
    }
    else
    {
      for (Index_ = ::SafeStrLen(temps_) - 1; Index_ >= 0; Index_--)
        Push(0, temps_[Index_], NULL);
    }

    delete[] temps_;
  }
}

/****************************************************************************/
// PURPOSE:
//   Set default input method
//
// PRE:
//   The original value of the input buffer data member
//
// POST:
//   The input buffer data member is assigned new memory copied from the
//   argument string. The remaining space in the buffer is NULL filled.
//
void WindowLineBuffer::SetInputString(char* str)
{
  if (IsEnterable() && (!str || (strlen(str) <= _maxbuf && IsValidString(str))))
  {
    int index = 0;
    _inboxoffset = 0;
    _inwboxcnt = 0;
    _MovedToEnd = false;
    _EndCharInserted = 0;
    _EndCharReverted = false;

    if (str)
    {
      strcpy(_buffer, str);
      for (index = ::SafeStrLen(_buffer); index <= _maxbuf; index++)
        _buffer[index] = '\0';
      _buffer = Convert(_buffer);
    }
    else
      memset(_buffer, 0, _maxbuf);
  }
}

/****************************************************************************/
// PURPOSE:
//   Set input length method
//
// PRE:
//   The original value of the input length data member
//
// POST:
//   The input length data member is copied from the length argument.
//
void WindowLineBuffer::SetInputLength(int len, bool MakeVbuf_)
{
  if (len <= _maxbuf && len >= _minlen && len >= 1 && IsEnterable())
  {
    _len = len;

    if (MakeVbuf_ && _len < _maxbuf && _BufChainNum == 0 && !_nextline)
    {
      WindowLineBuffer* npb = MakeVirtualLine(this);
      if (npb)
        Chain(npb);
    }
  }
}

/****************************************************************************/
void WindowLineBuffer::DivBufSizeAsInputLength(int div_, bool MakeVbuf_)
{
  if (_maxbuf && IsEnterable())
  {
    int res_ = _maxbuf / div_;
    
    if (res_ >= _minlen && res_ >= 1)
    {
      _len = res_;

      if (MakeVbuf_ && _len < _maxbuf && _BufChainNum == 0 && !_nextline)
      {      
        WindowLineBuffer* npb = MakeVirtualLine(this);
        if (npb)
          Chain(npb);
      }
    }
  }
}

/****************************************************************************/
// PURPOSE:
//   Set minimum input length method
//
// PRE:
//   The original value of the minimum input length data member
//
// POST:
//   The mimimum input length data member is copied from the length argument.
//
void WindowLineBuffer::SetMinimumLength(int len)
{
  if (len <= _len)
  {
    _minlen = (len > 0) ? len:1;
    _chainminlen = 0;
  }
  else
  {
    _minlen = _len;
    _chainminlen = len - _len;

    if (_nextline && !VirtualBufferOffset())
      _nextline->SetMinimumLength(_chainminlen);
  }
}

/****************************************************************************/
int* WindowLineBuffer::WordChr(int* src_, int val_, size_t n)
{
  while (n)
    if (*src_ == val_)
      return src_;
    else
    {
      src_++;
      --n;
    }

  return NULL;
}

/****************************************************************************/
int WindowLineBuffer::EndBrksLength(bool OnGoto_) const
{
  return (_EndBrksActive ? (OnGoto_ ? 1:2):0);
}

/****************************************************************************/
void WindowLineBuffer::SetEndBrksActive(bool Flag_)
{
  _EndBrksActive = Flag_;

  if (_nextline && !VirtualBufferOffset())
    _nextline->SetEndBrksActive(Flag_);
}

/****************************************************************************/
void WindowLineBuffer::SetRestrictFullBox(bool Flag_)
{
  _RestFullBox = Flag_;

  if (_nextline && IsEnterable())
    _nextline->SetRestrictFullBox(Flag_);
}

/****************************************************************************/
void WindowLineBuffer::SetTrimWs(bool Flag_)
{
  _TrimWs = Flag_;

  if (_nextline && IsEnterable())
    _nextline->SetTrimWs(Flag_);
}

/****************************************************************************/
void WindowLineBuffer::SetNoEmpty(int flag)
{
  _noempty = flag;

  if (_nextline && IsEnterable())
    _nextline->SetNoEmpty(_noempty);
}

/****************************************************************************/
void WindowLineBuffer::NotifyInsertMode(const int* ModePtr_)
{
  if (ModePtr_)
    _InsertMode = *ModePtr_;

  iMonitor(_insert_monitor, _insert_handle, ModePtr_);
}

/****************************************************************************/
void WindowLineBuffer::SetInsertMode(int Mode_, WindowLineBuffer* Parent_)
{
  _InsertMode = Mode_;
  iMonitor(_insert_monitor, _insert_handle, &_InsertMode);

  if (Parent_)
    MonitorInsertMode(Parent_, InsertDetect);

  if (_nextline && IsEnterable())
    _nextline->SetInsertMode(_InsertMode);
}

/****************************************************************************/
void WindowLineBuffer::SetXYpos(int x, int y)
{
  SetXpos(x);
  SetYpos(y);
}

/****************************************************************************/
// PURPOSE:
//   input string validity check method
//
// PRE:
//   char* str : The original value of the input string
//
// POST:
//   Returns 1 if all the characters within the input string is defined
//   as valid, otherwise returns 0.
//
int WindowLineBuffer::IsValidString(char* str)
{
  return TRUE;
}

/****************************************************************************/
int WindowLineBuffer::IsValidLength() const
{
  return
  (
    (::SafeStrLen(_buffer) >= _minlen || !_noempty) &&
    ((_nextline && !VirtualBufferOffset()) ?
          _nextline->IsValidLength():1)
  );
}

/****************************************************************************/
#if (!(defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__)) & (defined(__linux__) | defined(__unix__)))
int WindowLineBuffer::TranslateKeys(int key)
{
  return
  (
    GetKeyMap().KeyMatch(key, INSERT)   ? KEY_IC:
    GetKeyMap().KeyMatch(key, LEFT)     ? KEY_LEFT:
    GetKeyMap().KeyMatch(key, RIGHT)    ? KEY_RIGHT:
    GetKeyMap().KeyMatch(key, HOME)     ? KEY_HOME:
    GetKeyMap().KeyMatch(key, END)      ? KEY_END:
    GetKeyMap().KeyMatch(key, DELETE)   ? KEY_DC:
    GetKeyMap().KeyMatch(key, ENTER)    ? 10:
    GetKeyMap().KeyMatch(key, BACK)     ? KEY_BACKSPACE:
    GetKeyMap().KeyMatch(key, ESCAPE)   ? 27:
    GetKeyMap().KeyMatch(key, NEXT)     ? 9:
    GetKeyMap().KeyMatch(key, PAGEUP)   ? KEY_PPAGE:
    GetKeyMap().KeyMatch(key, PAGEDOWN) ? KEY_NPAGE:
    GetKeyMap().KeyMatch(key, UP)       ? KEY_UP:
    GetKeyMap().KeyMatch(key, DOWN)     ? KEY_DOWN:
                                          key
  );
}
#endif

/****************************************************************************/
// PURPOSE:
//   Keyboard retrieval procedure
//
// PRE:
//   The original value of _key and _extended data members
//
// POST:
//   The keyboard is read for a key and is assigned to _key data member.
//   If the key is extended non-ASCII then the _extended data member is set.
//   Returns the key read from the keyboard.
//
int WindowLineBuffer::GetKey()
{
  _extkey = _key = (int)GETCH();
  iMonitor(_key_monitor, _key_handle, &_key);
  _extended = (_key == 0 || _key == 0xE0) ? 256:0;
  iMonitor(_ext_monitor, _ext_handle, &_extended);

  if (_extended)
  {
#if (!(defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__)) & (defined(__linux__) | defined(__unix__)))
    noecho();
#endif	  
    _key = (int)GETCH();
    _extkey = _key + _extended;
    iMonitor(_key_monitor, _key_handle, &_key);

#if (!(defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__)) & (defined(__linux__) | defined(__unix__)))
    _extkey = TranslateKeys(_extkey);
#endif
  }  
#if (!(defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__)) & (defined(__linux__) | defined(__unix__)))
  else
    _extkey = TranslateKeys(_extkey);
#endif

  return _key;
}

/****************************************************************************/
// PURPOSE:
//   Method for putting extended non-ASCII keyboard keys to the console
//   output.
//
// PRE:
//   int key : the key to be outputted.
//
// POST:
//   The key is outputted to the screen. Appropriate changes to the
//   internal input buffer is made.
//
void WindowLineBuffer::PutExtKey(int key)
{
  int pos;
  char nextchar_;
  int WrapIndex_ = 0;
  int WrapMax_ = 0;
  bool DoChain_ = false;
  bool GotoDone_ = false;
  int EndReached_ = 0;
  int NonWsLen_ = 0;
  bool enterb_ = false;
  bool vbuf_ = false;
  bool inwbox_ = false;
  bool NextLen_ = false;
  bool NullChar_ = false;
  bool PrevNull_ = false;
  bool CondMet_ = false;
  int endpt_ = 0;
  int index_ = 0;
  int bufoffset_ = 0;

  // Action: TOGGLE_INSERT
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  if (GetKeyMap().KeyMatch(key, INSERT))
#elif (defined(__linux__) | defined(__unix__))
  if (key == KEY_IC)
#endif
    SetInsertMode(!GetInsertMode());

  // Action: GO_LEFT (chain push if WindowLine::_wraparoundindex > 0)
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, LEFT))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_LEFT)
#endif  
  {
    if (_Owner)
    {
      enterb_ = IsEnterable();
      vbuf_ = BufferChainPosition() == 0 && enterb_ &&
              VirtualBufferOffset(&EndReached_, &NonWsLen_);
      inwbox_ = BufferChainPosition() == 0 && enterb_ &&
                 ((WindowLine*)_Owner)->LineWrapAllowed();
      WrapIndex_ = enterb_ ? ((WindowLine*)_Owner)->GetWrapAroundIndex():0;
      WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;
      DoChain_ = IsChainable(false);
      endpt_ = (!vbuf_ && !inwbox_ && _nextline) ? _len:(_len-1);
    }

    // Modified to wrap chars around first buffer in window buffer object
    if (_index > 0)
    {
      GOTOXY(wherex()-1, _ypos);
      --_index;
      iMonitor(_index_monitor, _index_handle, &_index);

      if (vbuf_)
        endpt_ += _inboxoffset;

      if (_refline)
        _refline->DecrBoxIndex();

      NextLen_ = vbuf_ ? (EndReached_ ? 0:NonWsLen_):0;
      NullChar_ = _buffer[endpt_] == 0;

      if (_index == 0 && WrapIndex_ == WrapMax_ && !NextLen_ && !NullChar_)
      {
        _MovedToEnd = false;
        _EndCharInserted = _buffer[endpt_];
        _EndCharReverted = true;
      }
    }
    else if (vbuf_)
    {
      if (_index == 0 &&
          DoChain_ && WrapIndex_ > 0 && _inboxoffset > 0)
      {
        PrevNull_ = _buffer[_inboxoffset + endpt_] == 0;
        _inboxoffset--;
        nextchar_ = _buffer[_inboxoffset];
        ((WindowLine*)_Owner)->DecrWrapAround(1);
        WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
        WrapMax_ = ((WindowLine*)_Owner)->GetMaxWrapAround();

        pos = wherex();
        ShowInput();
        GOTOXY(pos, _ypos);
        iMonitor(_inboxoffset_monitor, _inboxoffset_handle, &_inboxoffset);

        if (_inboxoffset == 0 && WrapIndex_ > 0)
        {
          ((WindowLine*)_Owner)->DecrWrapAround(WrapIndex_);
          if (_refline)
            _refline->SetBoxIndex(0);

          WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
          WrapMax_ = ((WindowLine*)_Owner)->GetMaxWrapAround();
        }
        else if (_refline)
          _refline->DecrBoxIndex();

        NextLen_ = EndReached_ ? 0:NonWsLen_;
        NullChar_ = _buffer[_inboxoffset + endpt_] == 0;

        if (WrapIndex_ == WrapMax_ && nextchar_ &&
            PrevNull_ && !NextLen_ && !NullChar_)
        {
          _MovedToEnd = false;
          _EndCharInserted = _buffer[_inboxoffset + endpt_];
          _EndCharReverted = true;
        }
      }
      else if (_index > 0)
      {
        GOTOXY(wherex()-1, _ypos);
        --_index;
        iMonitor(_index_monitor, _index_handle, &_index);
        if (_refline)
          _refline->DecrBoxIndex();
      }
    }
    else if (!vbuf_ && inwbox_)
    {
      if (_index == 0 &&
          DoChain_ && WrapIndex_ > 0 && _inwboxcnt > 0)
      {      
        _WrapChars = true;
        PrevNull_ = _buffer[endpt_] == 0;
        _inwboxcnt--;
        nextchar_ = TakeNextLastChar();
        Push(0, nextchar_, NULL);
        WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
        WrapMax_ = ((WindowLine*)_Owner)->GetMaxWrapAround();

        pos = wherex();
        ShowInput();
        GOTOXY(pos, _ypos);

        if (_inwboxcnt == 0 && WrapIndex_ > 0)
        {
          ((WindowLine*)_Owner)->DecrWrapAround(WrapIndex_);
          if (_refline)
            _refline->SetBoxIndex(0);

          WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
          WrapMax_ = ((WindowLine*)_Owner)->GetMaxWrapAround();
        }
        else if (_refline)
          _refline->DecrBoxIndex();

        NullChar_ = _buffer[endpt_] == 0;

        if (WrapIndex_ == WrapMax_ && nextchar_ &&
            PrevNull_ && !NullChar_)
        {
          _MovedToEnd = false;
          _EndCharInserted = _buffer[endpt_];
          _EndCharReverted = true;
        }
      }
      else if (_index > 0)
      {
        GOTOXY(wherex()-1, _ypos);
        --_index;
        iMonitor(_index_monitor, _index_handle, &_index);
        if (_refline)
          _refline->DecrBoxIndex();
      }
    }
  }

  // Action: GO_RIGHT (chain pull if _index >= WindowLine::_inputlen)
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, RIGHT))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_RIGHT)
#endif  
  {
    if (_Owner)
    {
      enterb_ = IsEnterable();
      vbuf_ = BufferChainPosition() == 0 && enterb_ &&
              VirtualBufferOffset(&EndReached_, &NonWsLen_);
      inwbox_ = BufferChainPosition() == 0 && enterb_ &&
                 ((WindowLine*)_Owner)->LineWrapAllowed();
      WrapIndex_ = enterb_ ? ((WindowLine*)_Owner)->GetWrapAroundIndex():0;
      WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;
      DoChain_ = IsChainable(true);
      endpt_ = (!vbuf_ && !inwbox_ && _nextline) ? _len:(_len-1);

      if (inwbox_)
        NonWsLen_ =
          (_inwboxcnt > 0 && WrapIndex_ >= WrapMax_ &&
           _inwboxcnt >= WrapMax_ && WrapIndex_ <= _inwboxcnt) ?
               HasNonNullChars(_buffer):
               HasNonWsChars(_buffer);
      else if (vbuf_)
        NonWsLen_ =
          (_inboxoffset > 0 && WrapIndex_ >= WrapMax_ &&
           _inboxoffset >= WrapMax_ && WrapIndex_ <= _inboxoffset) ?
               HasNonNullChars(_buffer + _inboxoffset):
               HasNonWsChars(_buffer + _inboxoffset);
    }

    // Modified to wrap chars around first buffer in window buffer object
    if (!vbuf_ && inwbox_)
    {
      if ((_index == ::SafeStrLen(_buffer) - 1 || _index == endpt_) &&
          WrapIndex_ <= WrapMax_ && (DoChain_ || _EndCharInserted) &&
          (!_MovedToEnd || WrapIndex_ < WrapMax_))
      {
        _WrapChars = true;
        _inwboxcnt++;
        _MovedToEnd = (WrapIndex_ >= WrapMax_);
        
        nextchar_ = _buffer[0];
        Pull(0, NULL);

        if (nextchar_)
          Join(nextchar_);

        if (WrapIndex_ == WrapMax_)
        {
          ((WindowLine*)_Owner)->IncrWrapAround(1);
          WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
          WrapMax_ = ((WindowLine*)_Owner)->GetMaxWrapAround();
          _WrapChars = false;
        }

        pos = wherex();
        ShowInput();
        GOTOXY(pos, _ypos);

        if (_EndCharInserted &&
            WrapIndex_ >= WrapMax_ && WrapIndex_ == _inwboxcnt)
        {
          _EndCharInserted = 0;
          _EndCharReverted = false;
        }

        if (_refline)
          _refline->IncrBoxIndex();
      }
      else if (_index < ::SafeStrLen(_buffer) && _index < endpt_ && (!_inwboxcnt || (_inwboxcnt > 0 &&
               (WrapIndex_ < WrapMax_ || (_inwboxcnt >= WrapMax_ && WrapIndex_ <= _inwboxcnt)))))
      {
        if (_inwboxcnt > 0)
        {
          if (WrapIndex_ < WrapMax_)
          {
            if (_inwboxcnt <= WrapMax_)
              CondMet_ = (NonWsLen_ < _len) ? (_index < NonWsLen_):
                                              (_index < endpt_);
          }
          else if (_inwboxcnt >= WrapMax_ && WrapIndex_ <= _inwboxcnt)
          {
            if (NonWsLen_ < _len)
              CondMet_ = (_inwboxcnt >= WrapMax_ && WrapIndex_ <= _inwboxcnt && _index < NonWsLen_);
            else
              CondMet_ = (_inwboxcnt == WrapMax_ && WrapIndex_ <= WrapMax_ && _index < endpt_);
          }
          else if (_inwboxcnt < WrapMax_)
            CondMet_ = _index < endpt_;
        }
        else
          CondMet_ = !_inwboxcnt && _index < ::SafeStrLen(_buffer) && _index < endpt_;

        if (CondMet_)
        {
          GOTOXY(wherex()+1, _ypos);
          ++_index;
          iMonitor(_index_monitor, _index_handle, &_index);
          if (_refline)
            _refline->IncrBoxIndex();
        }
      }
    }
    else if (vbuf_)
    {
      if (_index == endpt_ && !EndReached_ &&
          WrapIndex_ <= WrapMax_ && (DoChain_ || _EndCharInserted) &&
          (!_MovedToEnd || WrapIndex_ < WrapMax_))
      {
        _inboxoffset++;
        _MovedToEnd = (WrapIndex_ >= WrapMax_);
        ((WindowLine*)_Owner)->IncrWrapAround(1);

        if (WrapIndex_ == WrapMax_)
        {
          WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
          WrapMax_ = ((WindowLine*)_Owner)->GetMaxWrapAround();
        }

        pos = wherex();
        ShowInput();
        GOTOXY(pos, _ypos);
        iMonitor(_inboxoffset_monitor, _inboxoffset_handle, &_inboxoffset);

        if (_EndCharInserted &&
            WrapIndex_ >= WrapMax_ && WrapIndex_ == _inboxoffset)
        {
          _EndCharInserted = 0;
          _EndCharReverted = false;
        }

        if (_refline)
          _refline->IncrBoxIndex();
      }
      else if (_index < ::SafeStrLen(_buffer) && _index < endpt_ && (!_inboxoffset || (_inboxoffset > 0 &&
               (WrapIndex_ < WrapMax_ || (_inboxoffset >= WrapMax_ && WrapIndex_ <= _inboxoffset)))))
      {
        if (_inboxoffset > 0)
        {
          if (WrapIndex_ < WrapMax_)
          {
            if (_inboxoffset <= WrapMax_)
              CondMet_ = (NonWsLen_ < _len) ? (_index < NonWsLen_):
                                              (_index < endpt_);
          }
          else if (_inboxoffset >= WrapMax_ && WrapIndex_ <= _inboxoffset)
          {
            if (NonWsLen_ < _len)
              CondMet_ = (_inboxoffset >= WrapMax_ && WrapIndex_ <= _inboxoffset && _index < NonWsLen_);
            else
              CondMet_ = (_inboxoffset == WrapMax_ && WrapIndex_ <= WrapMax_ && _index < endpt_);
          }
          else if (_inboxoffset < WrapMax_)
            CondMet_ = _index < endpt_;
        }
        else
          CondMet_ = !_inboxoffset && _index < ::SafeStrLen(_buffer) && _index < endpt_;

        if (CondMet_)
        {
          GOTOXY(wherex()+1, _ypos);
          ++_index;
          iMonitor(_index_monitor, _index_handle, &_index);
          if (_refline)
            _refline->IncrBoxIndex();
        }
      }
    }
    else if (_index < ::SafeStrLen(_buffer) && _index < endpt_)
    {
      GOTOXY(wherex()+1, _ypos);
      ++_index;
      iMonitor(_index_monitor, _index_handle, &_index);
      if (_refline)
        _refline->IncrBoxIndex();
    }
  }

  // Action: GO_HOME
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, HOME))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_HOME)
#endif  
  {
    _index = 0;
    iMonitor(_index_monitor, _index_handle, &_index);
    GOTOXY(_xpos + ::SafeStrLen(_msg) + EndBrksLength(true), _ypos);
    if (_refline)
      _refline->ResetBoxIndex();

    enterb_ = IsEnterable();
    vbuf_ = BufferChainPosition() == 0 && enterb_ &&
            VirtualBufferOffset(&EndReached_, &NonWsLen_);
    inwbox_ = BufferChainPosition() == 0 && enterb_ &&
               ((WindowLine*)_Owner)->LineWrapAllowed();
    WrapIndex_ = enterb_ ? ((WindowLine*)_Owner)->GetWrapAroundIndex():0;

    if (vbuf_ && _inboxoffset > 0 && WrapIndex_ > 0)
    {
      while (_inboxoffset > 0 && WrapIndex_ > 0)
      {
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
        PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::LEFTARROW).SCAN + 256);
#elif (defined(__linux__) | defined(__unix__))
        PutExtKey(KEY_LEFT);
#endif
        WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
      }
    }
    else if (inwbox_ && _inwboxcnt > 0 && WrapIndex_ > 0)
    {
      while (_inwboxcnt > 0 && WrapIndex_ > 0)
      {
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))	      	      
        PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::LEFTARROW).SCAN + 256);
#elif (defined(__linux__) | defined(__unix__))
        PutExtKey(KEY_LEFT);
#endif
        WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
      }
    }
  }

  // Action: GO_END
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, END))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_END)
#endif 
  {
    GotoDone_ = false;
    endpt_ = _index = ::SafeStrLen(_buffer);
    iMonitor(_index_monitor, _index_handle, &_index);

    enterb_ = IsEnterable();
    vbuf_ = BufferChainPosition() == 0 && enterb_ &&
            VirtualBufferOffset(&EndReached_, &NonWsLen_);
    inwbox_ = BufferChainPosition() == 0 && enterb_ &&
               ((WindowLine*)_Owner)->LineWrapAllowed();
    bufoffset_ = vbuf_               ? _inboxoffset:
                 (!vbuf_ && inwbox_) ? _inwboxcnt:0;

    if (::SafeStrLen(_buffer) >= _len && _len > 0)
      if (endpt_ > 0)
      {
        endpt_ = _index = _len;
        GotoDone_ = true;
        GOTOXY(_xpos + ::SafeStrLen(_msg) + endpt_ + EndBrksLength(true) - 1, _ypos);
        --_index;
        iMonitor(_index_monitor, _index_handle, &_index);

        if (_refline)
          _refline->SetBoxIndex(bufoffset_ + _index);
      }

    if (!GotoDone_)
    {
      GOTOXY(_xpos + ::SafeStrLen(_msg) + endpt_ + EndBrksLength(true), _ypos);
      if (_refline)
        _refline->SetBoxIndex(bufoffset_ + _index);
    }

    WrapIndex_ = enterb_ ? ((WindowLine*)_Owner)->GetWrapAroundIndex():0;
    WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;
    DoChain_ = IsChainable(true);

    if (vbuf_ && WrapIndex_ < WrapMax_ && !EndReached_)
    {
      while (WrapIndex_ < WrapMax_ && !EndReached_)
      {
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
        PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::RIGHTARROW).SCAN + 256);
#elif (defined(__linux__) | defined(__unix__))
        PutExtKey(KEY_RIGHT);
#endif
        WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
        WrapMax_ = ((WindowLine*)_Owner)->GetMaxWrapAround();
        VirtualBufferOffset(&EndReached_, &NonWsLen_);
      }
    }
    else if (inwbox_ && WrapIndex_ < WrapMax_ && DoChain_)
    {
      while (WrapIndex_ < WrapMax_)
      {
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
        PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::RIGHTARROW).SCAN + 256);
#elif (defined(__linux__) | defined(__unix__))
        PutExtKey(KEY_RIGHT);
#endif
        WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
        WrapMax_ = ((WindowLine*)_Owner)->GetMaxWrapAround();
      }
    }
  }

  // Action: DELETE
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, DELETE))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_DC)
#endif
  {  
    if (_index < _len && IsEnterable())
    {
      enterb_ = IsEnterable();
      vbuf_ = BufferChainPosition() == 0 && enterb_ &&
              VirtualBufferOffset();

      if (vbuf_)
      {
        _ChainPosPtr->SetupInfoPtrs(this, 0);
        _ChainPosPtr->SetPullInfo(_index);
      }

      Pull(_index, vbuf_ ? _ChainPosPtr:NULL);
      pos = wherex();
      ShowInput();
      GOTOXY(pos, _ypos);      
    }
  }
}

/****************************************************************************/
// PURPOSE:
//   Method for putting standard ASCII keyboard keys to the console output.
//
// PRE:
//   int key : the key to be outputted.
//
// POST:
//   The key is outputted to the screen. Appropriate changes to the
//   internal input buffer is made.
//
void WindowLineBuffer::PutStdKey(int key)
{
  bool NextLineReady_ = FALSE;
  bool InWin_ = FALSE;
  int pos;
  int index_ = 0;
  int endpt_ = 0;
  int oldkey_ = 0;
  int EndReached_ = 0;  
  int NonWsLen_ = 0;
  int offset_ = 0;

  bool enterb_ = IsEnterable();
  bool vbuf_ = BufferChainPosition() == 0 && enterb_ &&
               VirtualBufferOffset(&EndReached_, &NonWsLen_);
  bool inwbox_ = BufferChainPosition() == 0 && enterb_ &&
                 ((WindowLine*)_Owner)->LineWrapAllowed();
  int bufoffset_ = vbuf_               ? _inboxoffset:
                   (!vbuf_ && inwbox_) ? _inwboxcnt:0;
  int WrapIndex_ = enterb_ ? ((WindowLine*)_Owner)->GetWrapAroundIndex():0;
  int WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;
  bool DoChain_ = IsChainable(true);

  // Action: NEWLINE
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  if (_InsertMode && GetKeyMap().KeyMatch(key, ENTER))
#elif (defined(__linux__) | defined(__unix__))
  if (_InsertMode && key == 10)
#endif
  {
    if (_nextline && IsEnterable())
    {
      // owner shift down to next window line instead of going to next
      // line when enter key is pressed.

      if (_Owner)
      {
        endpt_ = _len;
        offset_ = vbuf_   ? (_inboxoffset + _index):
                  inwbox_ ? (_inwboxcnt + _index):_index;
        InWin_ = enterb_ ? ((WindowLine*)_Owner)->InWindowMode(&NextLineReady_):false;

        if (InWin_ && NextLineReady_ && inwbox_)
        {
          ((WindowLine*)_Owner)->ShiftDownNext(_buffer, offset_, NULL, 0);

          if (WrapIndex_ < WrapMax_ && DoChain_)
          {
            ((WindowLine*)_Owner)->ChainFillString(_buffer, offset_, '\0', _index);

#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
            PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::HOME).SCAN + 256);
#elif (defined(__linux__) | defined(__unix__))
            PutExtKey(KEY_HOME);
#endif
            pos = wherex();
            ShowInput();
            GOTOXY(pos, _ypos);
          }
        }
        else if (enterb_)
          ((WindowLine*)_Owner)->SendUnprocessedKey(key, _extended, this);
      }
    }
  }

  // Action: IF_GO_LEFT_DELETE
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, BACK) && IsEnterable())
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_BACKSPACE && IsEnterable())
#endif
  {
    if ((_index > 0) ||
        (vbuf_ && WrapIndex_ > 0 && _inboxoffset > 0) ||
        (inwbox_ && WrapIndex_ > 0 && _inwboxcnt > 0))
    {
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
      PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::LEFTARROW).SCAN + 256);
      PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::DELETE).SCAN + 256);
#elif (defined(__linux__) | defined(__unix__))
      PutExtKey(KEY_LEFT);
      PutExtKey(KEY_DC);
#endif
      iMonitor(_index_monitor, _index_handle, &_index);
    }
  }

  // Action: PUTKEY
  else if (IsUserKey(key))
  {
    if (_index < _len && IsEnterable())
    {
      bool NullEnd_ = false;
      bool NullChar_ = false;
      bool EntryOk_ = true;
      bool EndMet_ = false;

      int endptdex_ = 0;
      int Offset_ = vbuf_ ? _inboxoffset:
                    inwbox_ ? _inwboxcnt:0;

      if (inwbox_)
        NonWsLen_ =
          (Offset_ > 0 && WrapIndex_ >= WrapMax_ &&
           Offset_ >= WrapMax_ && WrapIndex_ <= Offset_) ?
               HasNonNullChars(_buffer):
               HasNonWsChars(_buffer);
      else if (vbuf_)
        NonWsLen_ =
          (Offset_ > 0 && WrapIndex_ >= WrapMax_ &&
           Offset_ >= WrapMax_ && WrapIndex_ <= Offset_) ?
               HasNonNullChars(_buffer + Offset_):
               HasNonWsChars(_buffer + Offset_);

      index_ = vbuf_ ? (Offset_ + _index):_index;
      endpt_ = (!vbuf_ && !inwbox_ && _nextline) ? _len:(_len-1);
      endptdex_ = vbuf_ ? (Offset_ + endpt_):endpt_;
      NullEnd_ = !_buffer[endptdex_];

      if (!_buffer[index_])
      {
        NullChar_ = true;
        _buffer[index_+1] = '\0';
      }

      // replace ' ' with 0xAE, '\t' with 0xAF, '\r' with 0xB0
      // and '\n' with 0xB1: for vbuf and inwbox mode input boxes
      oldkey_ = key;
      
      if (vbuf_ || inwbox_)
        if (key == ' ')
          key = 0xAE;
        else if (key == '\t')
          key = 0xAF;
        else if (key == '\r')
          key = 0xB0;
        else if (key == '\n')
          key = 0xB1;

      if (_index == endpt_ && _index <= NonWsLen_ && NullChar_ &&
          Offset_ >= WrapMax_ && WrapIndex_ >= WrapMax_ &&
          WrapIndex_ == Offset_)
      {
        _buffer[index_] = key;
        _MovedToEnd = false;
        _EndCharInserted = key;
        _EndCharReverted = true;

        // BackFill null spaces in buffer with 0xAE instead
        while (index_ > 0 && !_buffer[index_-1])
          _buffer[--index_] = 0xAE;
      }
      else
      {
        if (_InsertMode)
        {
          if (vbuf_)
          {
            _ChainPosPtr->SetupInfoPtrs(this, 0);
            _ChainPosPtr->SetPushInfo(_index);
          }

          if (!_EndCharReverted)
            _EndCharInserted = 0;

          EntryOk_ = Push(_index, key, vbuf_ ? _ChainPosPtr:NULL);

          if (_buffer[endptdex_] && NullEnd_ && Offset_ >= WrapMax_ &&
              WrapIndex_ >= WrapMax_ && WrapIndex_ == Offset_)
          {
            _MovedToEnd = false;
            _EndCharInserted = _buffer[endptdex_];
            _EndCharReverted = true;
          }
        }
        else
        {
          _buffer[index_] = key;

          if (_index == endpt_ && _index <= NonWsLen_ &&
              Offset_ >= WrapMax_ && WrapIndex_ >= WrapMax_ &&
              WrapIndex_ == Offset_)
          {
            _MovedToEnd = false;
            _EndCharInserted = key;

            if (key && NullChar_)
              _EndCharReverted = true;
          }

          // BackFill null spaces in buffer with 0xAE instead
          while (index_ > 0 && !_buffer[index_-1])
            _buffer[--index_] = 0xAE;
        }
      }

      if (EntryOk_)
      {
        PUTCH(oldkey_);
        pos = wherex();
        ShowInput();

        if (_index + 1 <= endpt_)
        {
          GOTOXY(pos, _ypos);
          ++_index;
        }
        else
        {
          GOTOXY(pos-1, _ypos);
          EndMet_ = true;
        }
      }
      
      iMonitor(_index_monitor, _index_handle, &_index);
      if (_refline)
        _refline->SetBoxIndex(bufoffset_ + _index);

      if ((vbuf_ || inwbox_) && EndMet_)
      {
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
        PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::RIGHTARROW).SCAN + 256);
#elif (defined(__linux__) | defined(__unix__))
        PutExtKey(KEY_RIGHT);
#endif
        iMonitor(_index_monitor, _index_handle, &_index);
      }
    }
  }
}

/****************************************************************************/
int WindowLineBuffer::IsUserKey(int key)
{
  return
  (
    !_extended &&
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
    !GetKeyMap().KeyMatch(key, BACK) &&
    !GetKeyMap().KeyMatch(key, ESCAPE) &&
    !GetKeyMap().KeyMatch(key, ENTER) &&
    !GetKeyMap().KeyMatch(key, NEXT) &&
#elif (defined(__linux__) | defined(__unix__))
    key != KEY_BACKSPACE &&
    key != 27 &&
    key != 10 &&
    key != 9 &&
#endif
    isprint(key)
  );
}

/****************************************************************************/
void WindowLineBuffer::SetChainPos(int Pos_)
{
  _ChainPos = Pos_;

  if (_nextline)
    _nextline->SetChainPos(ChainPosition() + 1);
}

/****************************************************************************/
WindowLineBuffer* WindowLineBuffer::Chain(WindowLineBuffer* next_)
{
  if (_nextline)
    return _nextline->Chain(next_);
  else
  {  
    _nextline = next_;
    if (!_refline && _BufChainNum == 0)
      _refline = WindowDummyLine::Make(0, this, _len);

    if (_nextline)
    {
      SetBufferChainNum(_nextline, _BufChainNum+1);
      SetReturnBuffer(_nextline, _retbuf, false);
      
      _nextline->SetChainPos(ChainPosition() + 1);
      _nextline->SetNoEmpty(_noempty);
      _nextline->SetInsertMode(GetInsertMode(), this);

      _nextline->MonitorBuffer(_buffer_handle, _buffer_monitor);
      _nextline->MonitorKey(_key_handle, _key_monitor);
      _nextline->MonitorExtendedKey(_ext_handle, _ext_monitor);
      _nextline->MonitorIndex(_index_handle, _index_monitor);
      _nextline->MonitorEscapedStatus(_escaped_handle, _escaped_monitor);
      _nextline->MonitorInboxOffset(_inboxoffset_handle, _inboxoffset_monitor);
      _nextline->MonitorWinLineNum(_winlinenum_handle, _winlinenum_monitor);
      // _nextline->MonitorInFocusStatus(_infocus_handle, _infocus_monitor);

      if (_chainminlen)
        _nextline->SetMinimumLength(_chainminlen);

      if (_TextWidgetActive)
      {
        _nextline->Activate();

        if (_refline && _BufChainNum == 0)
          _refline->Activate();
      }
      else
        _nextline->Deactivate();
    }
  }

  return this;
}

/****************************************************************************/
/****************************************************************************/
int WindowLineBuffer::HasNonWsChars(const char* str_) const
{
  int x = 0;
  int max = str_ ? ::SafeStrLen(str_):0;

  if (max)
    for (x = max-1; x >= 0; x--)
      if (!isspace(str_[x]))
        return (x+1);

  return 0;
}

/****************************************************************************/
int WindowLineBuffer::HasNonNullChars(const char* str_) const
{
  int x = 0;
  int max = str_ ? ::SafeStrLen(str_):0;

  if (max)
    for (x = max-1; x >= 0; x--)
      if (!isspace(str_[x]) &&
          Uchar(str_[x]) != 0xAE &&
          Uchar(str_[x]) != 0xAF &&
          Uchar(str_[x]) != 0xB0 &&
          Uchar(str_[x]) != 0xB1)
        return (x+1);

  return 0;
}

/****************************************************************************/
// Has extended _nextline implemented as WindowLineBuffer with buffer offset
//
int WindowLineBuffer::VirtualBufferOffset(int* EndReached_,
                                          int* NonWsLen_, int* NonNullLen_) const
{
  int ret = 0;
  
  if (_nextline && IsEnterable())
  {  
    ret = _nextline->_bufferoffset;
    
    if (EndReached_)
    {
      *EndReached_ = (_nextline->_maxbuf > 0 &&
                      _nextline->_maxbuf - _inboxoffset) <= 0;

      if (NonWsLen_ && !(*EndReached_))
        *NonWsLen_ = HasNonWsChars(_nextline->_buffer + _inboxoffset);

      if (NonNullLen_ && !(*EndReached_))
        *NonNullLen_ = HasNonNullChars(_nextline->_buffer + _inboxoffset);
    }
  }

  return ret;
}

/****************************************************************************/
// Return the max remaining length of the extended _nextline buffer offset
//
int WindowLineBuffer::VirtualBufferMax() const
{
  int ret = 0;
  
  if (_nextline && IsEnterable())
  {
    ret = _nextline->_bufferoffset;
    if (ret)
      ret = _nextline->_maxbuf;
  }

  return ret;
}

/****************************************************************************/
int WindowLineBuffer::IsChainable(bool HasChars_) const
{
  int EndReached_ = 0;
  int NonWsLen_ = 0;
  
  bool enterb_ = IsEnterable();
  bool vbuf_ = BufferChainPosition() == 0 && enterb_ &&
               VirtualBufferOffset(&EndReached_, &NonWsLen_);

  return
  (
    vbuf_ ?
      (HasChars_ ? (_nextline && IsEnterable() && NonWsLen_ > 0):
                   (_nextline && IsEnterable())):

      (HasChars_ ? (_nextline && IsEnterable() && NextLength() > 0):
                   (_nextline && IsEnterable()))
  );
}

/****************************************************************************/
int WindowLineBuffer::NextLengthHelper() const
{
  int lastnwschar_ = HasNonWsChars(_buffer);

  if (_nextline && IsEnterable() && !lastnwschar_)
    return _nextline->NextLengthHelper();

  return lastnwschar_;
}

/****************************************************************************/
int WindowLineBuffer::NextLength() const
{
  if (!_nextline)
    return 0;

  return _nextline->NextLengthHelper();
}

/****************************************************************************/
int WindowLineBuffer::TakeNextLastCharHelper()
{
  int lastnwschar_ = 0;
  bool enterb_ = IsEnterable();
  int BoxCount_ = enterb_ ? ((WindowLine*)_Owner)->GetBoxCount():0;

  if (NextLength() > 0)
    return _nextline->TakeNextLastCharHelper();
  else
    lastnwschar_ = HasNonWsChars(_buffer);
    
  //  lastnwschar_ = (BufferChainPosition() == BoxCount_-1) ?
  //                          HasNonNullChars(_buffer):
  //                          HasNonWsChars(_buffer);

  char takech_ = lastnwschar_ ? _buffer[lastnwschar_-1]:0;

  if (takech_ && IsEnterable() && lastnwschar_)
  {
    _buffer[lastnwschar_-1] = 0;
    if (_index == lastnwschar_ && _index > 0)
      --_index;
  }

  return takech_;
}

/****************************************************************************/
int WindowLineBuffer::TakeNextLastChar()
{
  if (!_nextline)
    return 0;

  return _nextline->TakeNextLastCharHelper();
}

/****************************************************************************/
int WindowLineBuffer::IsDummyLine() const
{
  return false;
}

/****************************************************************************/
int WindowLineBuffer::IsWindowLineBuffer() const
{
  return true;
}

/****************************************************************************/
int WindowLineBuffer::ChainPosition() const
{
  return _ChainPos;
}

/****************************************************************************/
int WindowLineBuffer::BufferChainPosition() const
{
  return _BufChainNum;
}

/****************************************************************************/
int WindowLineBuffer::IsFullBox() const
{
  bool enterb_ = _Owner && IsEnterable();
  bool inwbox_ = BufferChainPosition() == 0 && enterb_ &&
                 ((WindowLine*)_Owner)->LineWrapAllowed();

  int copylen_ = _buffer ? strlen(_buffer):0;
  char lastchar_ = (copylen_ == _len) ? _buffer[_len - 1]:0;
  return ((_nextline && _nextline->_buffer) ?
                      (lastchar_ && _nextline->IsFullBox()):
                      (inwbox_ ? (lastchar_ &&
                                  !isspace(lastchar_) &&
                                  !IsEquivSpace(lastchar_)):lastchar_));
}

/****************************************************************************/
int WindowLineBuffer::RestrictFullBox() const
{
  return _RestFullBox;
}

/****************************************************************************/
bool WindowLineBuffer::EndBrksActive() const
{
  return _EndBrksActive;
}

/****************************************************************************/
bool WindowLineBuffer::ShouldTrimWs() const
{
  return _TrimWs;
}

/****************************************************************************/
int WindowLineBuffer::BufferIndex() const
{
  return _index;
}

/****************************************************************************/
int WindowLineBuffer::InBoxOffset() const
{
  return _inboxoffset;
}

/****************************************************************************/
int WindowLineBuffer::InWBoxCount() const
{
  return _inwboxcnt;
}

/****************************************************************************/
void WindowLineBuffer::ChainPush(char ch, ChainPosInfo* Info_)
{
  if (_nextline)  
  {
    int index_ = 0;
    bool enterb_ = IsEnterable();
    bool vbuf_ = VirtualBufferOffset() && enterb_;
    
    if (vbuf_)
    {
      index_ = _inboxoffset;
      _nextline->SetOwner(_Owner, _nextline);

      if (Info_)
      {
        Info_->SetupInfoPtrs(_nextline, 1);
        Info_->SetPushInfo(_inboxoffset);
      }
    }

    _nextline->Push(index_, ch, Info_);

    if (vbuf_)
    {
      _nextline->SetOwner(NULL, _nextline);

      if (Info_)
        Info_->RestorePrevInfo();
    }
  }
}

/****************************************************************************/
void WindowLineBuffer::ChainJoin(char ch)
{
  if (_nextline)
    _nextline->Join(ch);
}

/****************************************************************************/
char WindowLineBuffer::ChainPull(ChainPosInfo* Info_)
{
  if (_nextline)
  {
    int index_ = 0;
    bool enterb_ = IsEnterable();
    bool vbuf_ = VirtualBufferOffset() && enterb_;
    
    if (vbuf_)
    {
      index_ = _inboxoffset;
      _nextline->SetOwner(_Owner, _nextline);

      if (Info_)
      {
        Info_->SetupInfoPtrs(_nextline, 1);
        Info_->SetPullInfo(_inboxoffset);
      }
    }
      
    char ret_ = _nextline->Pull(index_, Info_);

    if (vbuf_)
    {
      _nextline->SetOwner(NULL, _nextline);

      if (Info_)
        Info_->RestorePrevInfo();
    }

    return ret_;
  }

  return '\0';
}

/****************************************************************************/
char WindowLineBuffer::Peek(int Index_)
{
  if (_nextline)
    return _nextline->ChrRetrieve(Index_);

  return '\0';
}

/****************************************************************************/
bool WindowLineBuffer::Push(int start, char ch, ChainPosInfo* Info_)
{
  if ((!Info_ || Info_->_ChainPos == 2) && _ChainPosPtr)
  {
    _ChainPosPtr->SetupInfoPtrs(this, 2);
    _ChainPosPtr->SetPushInfo(start);

    Info_ = _ChainPosPtr;
  }
  else if (Info_)
    Info_->ReturnToThis();

  bool enterb_ = _Owner && IsEnterable();
  int WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;
  int WrapIndex_;
  int totallen_ = ::SafeStrLen(&_buffer[Info_->_startpt]);
  int atend_ = totallen_ >= (Info_->_endpt - Info_->_startpt);
  char nextchar_ = 0;
  char lastchar_ = atend_ ? _buffer[Info_->_endpt - 1]:0;
  atend_ = lastchar_ != 0;
  bool DoChain_ = atend_ && Info_->_chainable;
  bool UserKey_ = false;
  bool ChainDone_ = false;
  bool FullBox_ = (_nextline && IsEnterable()) ? _nextline->IsFullBox():
                                                 IsFullBox();

  if (atend_)
  {
    --Info_->_copylen;
    atend_ = 1;
  }

  if (Info_->_copylen < 0)
    Info_->_copylen = 0;

  if (!Info_->_error && (!_RestFullBox || !FullBox_ || !atend_))
  {
    if (Info_->_copylen)
    {
      nextchar_ = Info_->_vbuf ? _buffer[Info_->_index + Info_->_copylen + atend_]:0;
      memmove(&_buffer[Info_->_index + 1], &_buffer[Info_->_index], Info_->_copylen);
      _buffer[(Info_->_index+1) + Info_->_copylen] = nextchar_;
    }

    if (Info_->_index < Info_->_endpt)
    {
      _buffer[Info_->_index] = ch;

      if (DoChain_ && lastchar_)
      {
        ChainPush(lastchar_, Info_);
        ChainDone_ = true;
      }
    }

    if ((ChainDone_ || (!atend_ && Info_->_chainable && _inwboxcnt > 0)) &&
        Info_->_inwbox && _Owner && IsEnterable())
      if ((UserKey_=IsUserKey(_extkey)) || _WrapChars)
      {
        WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
        DoChain_ = Info_->_chainable;

        if (_WrapChars && !UserKey_ && WrapIndex_ > 0)
        {
          ((WindowLine*)_Owner)->DecrWrapAround(1);
          --WrapIndex_;
        }
        else if (UserKey_)
        {
          WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;
          _WrapChars = false;
        }

        if (!UserKey_)
          _WrapChars = _nextline && DoChain_ && !UserKey_ &&
                       Info_->_inwbox && WrapIndex_ > 0;
      }

    return true;
  }

  return false;
}

/****************************************************************************/
void WindowLineBuffer::Join(char ch)
{
  int slen_ = ::SafeStrLen(_buffer);
  bool HasRoom_ = slen_ < _len;
  
  char added_[2];
  added_[0] = ch;
  added_[1] = 0;

  if (_nextline && (NextLength() > 0 || !HasRoom_ || _EndCharInserted))
    ChainJoin(ch);
  else if (HasRoom_)
    strcat(_buffer, added_);
}

/****************************************************************************/
char WindowLineBuffer::Pull(int start, ChainPosInfo* Info_)
{
  if ((!Info_ || Info_->_ChainPos == 2) && _ChainPosPtr)
  {
    _ChainPosPtr->SetupInfoPtrs(this, 2);
    _ChainPosPtr->SetPullInfo(start);

    Info_ = _ChainPosPtr;
  }
  else if (Info_)
    Info_->ReturnToThis();

  bool FindNws_ = false;
  bool enterb_ = _Owner && IsEnterable();
  int BoxCount_ = enterb_ ? ((WindowLine*)_Owner)->GetBoxCount():0;
  int WrapIndex_ = enterb_ ? ((WindowLine*)_Owner)->GetWrapAroundIndex():0;
  int WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;
  int NullChar_ = 0;
  bool inwbox_ = Info_->_inwbox;
  bool vbuf_ = Info_->_vbuf;
  bool DoChain_ = Info_->_chainable && !Info_->_error;
  bool ChainDone_ = false;
  bool HasEndChar_ = false;
  bool DelKey_ = false;
  char firstchar_ = _buffer[Info_->_index];
  char nextchar_ = 0;
  char lastchar_;
  int lastnwspos_;
  int blanklen_ = 0;
  int index_;

  if (Info_->_copylen < 0)
    Info_->_copylen = 0;

  if (!Info_->_copylen && !Info_->_error)
  {
    if (_nextline && IsEnterable())
    {
      int x;
      start = Info_->_index;

      for (x = start; start < Info_->_endpt; start++)
      {
        _buffer[start] = ChainPull(Info_);
        ChainDone_ = true;

        if (!_buffer[start])
          break;
      }

      if (x != start && _WrapChars && ChainDone_ &&
          _Owner && Info_->_inwbox && IsEnterable())
      {
        WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
        WrapMax_ = ((WindowLine*)_Owner)->GetMaxWrapAround();
        DoChain_ = Info_->_chainable;

        // lastnwspos_ = HasNonWsChars(_buffer);
        lastnwspos_ =
          (BufferChainPosition() < BoxCount_-1 &&
           _inwboxcnt > 0 && WrapIndex_ >= WrapMax_ &&
           _inwboxcnt >= WrapMax_ && WrapIndex_ <= _inwboxcnt) ?
               HasNonNullChars(_buffer):
               HasNonWsChars(_buffer);

        if (WrapIndex_ < WrapMax_)
        {
          ((WindowLine*)_Owner)->IncrWrapAround(start - x);
          WrapIndex_ += (start - x);
        }

        _WrapChars = (_nextline && DoChain_ && Info_->_inwbox) &&
                     (WrapIndex_ < WrapMax_ ||
                       ((lastnwspos_ < _len) ? (_inwboxcnt >= WrapMax_ && WrapIndex_ <= _inwboxcnt):
                                               (_inwboxcnt == WrapMax_ && WrapIndex_ <= WrapMax_)));
      }
    }

    return 0;
  }
  else if (_buffer[Info_->_index] && Info_->_copylen && !Info_->_error)
  {
    blanklen_ = 0;
    NullChar_ = (_buffer[(Info_->_index+1) + Info_->_copylen - 1] == 0) ? 1:0;

    if (DoChain_ || _bufferoffset > 0)
    {
      if (_bufferoffset == 0 && DoChain_)
      {
        FindNws_ = IsChainable(true);
      
        if (FindNws_)
        {
          inwbox_ = enterb_ && ((WindowLine*)_Owner)->LineWrapAllowed();
          WrapIndex_ = enterb_ ? ((WindowLine*)_Owner)->GetWrapAroundIndex():0;
          WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;
          DoChain_ = inwbox_ ?
                     (BufferChainPosition() > 0 ||
                      (BufferChainPosition() == 0 &&
                       0 <= WrapIndex_ && WrapIndex_ < WrapMax_)):true;

          if (DoChain_)
          {
            nextchar_ = ChainPull(Info_);
            ChainDone_ = true;
          }
        }
      }
      else
        FindNws_ = IsChainable(true);

      memmove(&_buffer[Info_->_index], &_buffer[Info_->_index + 1], Info_->_copylen);
      HasEndChar_ = ChainDone_ || (_MovedToEnd && _EndCharInserted);

      if (FindNws_)
      {
        lastchar_ = _buffer[Info_->_endpt];
        if (Info_->_vbuf)
          _buffer[Info_->_endpt] = 0;
        
        // lastnwspos_ = HasNonWsChars(&_buffer[Info_->_startpt]);
        if (inwbox_)
          lastnwspos_ =
            (BufferChainPosition() < BoxCount_-1 &&
             _inwboxcnt > 0 && WrapIndex_ >= WrapMax_ &&
             _inwboxcnt >= WrapMax_ && WrapIndex_ <= _inwboxcnt) ?
                 HasNonNullChars(&_buffer[Info_->_startpt]):
                 HasNonWsChars(&_buffer[Info_->_startpt]);
        else if (vbuf_)
          lastnwspos_ =
            (_bufferoffset == 0 &&
             _inboxoffset > 0 && WrapIndex_ >= WrapMax_ &&
             _inboxoffset >= WrapMax_ && WrapIndex_ <= _inboxoffset) ?
                 HasNonNullChars(&_buffer[Info_->_startpt]):
                 HasNonWsChars(&_buffer[Info_->_startpt]);

        blanklen_ = (Info_->_endpt - Info_->_startpt) - lastnwspos_ - 1;

        if (Info_->_vbuf)
          _buffer[Info_->_endpt] = lastchar_;

        if (!ChainDone_)
          nextchar_ = 0;
      }
      
      if (blanklen_ > 0 && FindNws_ && nextchar_ && HasEndChar_)
        ::memset(&_buffer[Info_->_index + lastnwspos_], 0xAE, blanklen_);

      if (!nextchar_ && (!FindNws_ || (HasEndChar_ && !Info_->_vbuf)))
      {
        index_ = Info_->_index + Info_->_copylen - NullChar_;
        _buffer[index_] = 0;
        
        while (index_ <= Info_->_endpt)
          _buffer[index_++] = 0;
      }

      if (HasEndChar_)
      {
        _buffer[Info_->_endpt - 1] = nextchar_;

        if (!Info_->_vbuf)
          _buffer[Info_->_endpt] = 0;
      }
    }
    else
    {
      memmove(&_buffer[Info_->_index], &_buffer[Info_->_index + 1], Info_->_copylen);

      if (!Info_->_vbuf)
      {
        _buffer[Info_->_index + Info_->_copylen - NullChar_] = 0;
        _buffer[Info_->_endpt] = 0;
      }
    }

    if (ChainDone_ && Info_->_inwbox && _Owner && IsEnterable())
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
      if ((DelKey_=GetKeyMap().KeyMatch(_extkey, DELETE)) || _WrapChars)
#elif (defined(__linux__) | defined(__unix__))
      if ((DelKey_=(_extkey == KEY_DC)) || _WrapChars)
#endif
      {
        WrapIndex_ = ((WindowLine*)_Owner)->GetWrapAroundIndex();
        DoChain_ = Info_->_chainable;
        
        // lastnwspos_ = HasNonWsChars(_buffer);
        lastnwspos_ =
          (BufferChainPosition() < BoxCount_-1 &&
           _inwboxcnt > 0 && WrapIndex_ >= WrapMax_ &&
           _inwboxcnt >= WrapMax_ && WrapIndex_ <= _inwboxcnt) ?
               HasNonNullChars(_buffer):
               HasNonWsChars(_buffer);

        if (_WrapChars && !DelKey_ && WrapIndex_ < WrapMax_)
        {
          ((WindowLine*)_Owner)->IncrWrapAround(1);
          ++WrapIndex_;
        }
        else if (DelKey_)
        {
          WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;
          _WrapChars = false;
        }

        if (!DelKey_)
          _WrapChars = (_nextline && DoChain_ && !DelKey_ && Info_->_inwbox) &&
                        (WrapIndex_ < WrapMax_ ||
                           ((lastnwspos_ < _len) ? (_inwboxcnt >= WrapMax_ && WrapIndex_ <= _inwboxcnt):
                                                   (_inwboxcnt == WrapMax_ && WrapIndex_ <= WrapMax_)));
      }
  }

  return firstchar_;
}

/****************************************************************************/
void WindowLineBuffer::ResetRefPos(int MoveToPos_)
{
  // new
}

/****************************************************************************/
void WindowLineBuffer::ShiftDown(char* NewStr_)
{
  if (_nextline)
    _nextline->ShiftDown(_buffer);
  SetInputString(NewStr_);
}

/****************************************************************************/
bool WindowLineBuffer::InChSet(char ch, const char* chset, size_t* Index_)
{
  size_t dummy;
  
  if (Index_)
    *Index_ = 0;
  else
    Index_ = &dummy;

  if (chset == NULL || *chset == 0)
    return FALSE;

  if (*chset == '0' && ch == 0)
    return TRUE;

  if (*chset == '-' || *chset == '0')
  {
    ++chset;
    *Index_ += 1;
  }

  for (;*chset; chset++)
  {
    if (ch == *chset)
      return TRUE;
      
    *Index_ += 1;
  }

  *Index_ = 0;
  return FALSE;
}

/****************************************************************************/
char* WindowLineBuffer::RemoveTrailSpaceChars(char* Str_, bool UnReverted_)
{
  char chset_[5];

  if (UnReverted_)
  {
    chset_[0] = 0xAE;
    chset_[1] = 0xAF;
    chset_[2] = 0xB0;
    chset_[3] = 0xB1;
    chset_[4] = 0;
  }
  else
    strcpy(chset_, " \t\r\n");
  
  for (size_t l = ::SafeStrLen(Str_); l;)
    if (InChSet(Str_[--l], chset_))
      Str_[l] = '\0';
    else
      break;

  return Str_;
}

/****************************************************************************/
bool WindowLineBuffer::IsEquivSpace(char ch) const
{
  return
  (
    Uchar(ch) == 0xAE ||
    Uchar(ch) == 0xAF ||
    Uchar(ch) == 0xB0 ||
    Uchar(ch) == 0xB1
  );
}

/****************************************************************************/
char* WindowLineBuffer::Convert(char* Str_)
{
  if (!Str_)
    return Str_;

  // replace ' ' with 0xAE, '\t' with 0xAF, '\r' with 0xB0
  // and '\n' with 0xB1: for vbuf and inwbox mode input boxes
  bool enterb_ = IsEnterable();
  bool vbuf_ = BufferChainPosition() == 0 && enterb_ &&
               VirtualBufferOffset();
  bool inwbox_ = BufferChainPosition() == 0 && enterb_ &&
                 ((WindowLine*)_Owner)->LineWrapAllowed();
  
  size_t x;
  size_t len;
  
  if (vbuf_ || inwbox_ || BufferChainPosition() > 0)
  {
    len = ::SafeStrLen(Str_);

    for (x = 0; x < len; x++)
    {
      if (Str_[x] == ' ')
        Str_[x] = 0xAE;
      else if (Str_[x] == '\t')
        Str_[x] = 0xAF;
      else if (Str_[x] == '\r')
        Str_[x] = 0xB0;
      else if (Str_[x] == '\n')
        Str_[x] = 0xB1;
    }
  }

  return Str_;
}

/****************************************************************************/
char* WindowLineBuffer::Revert(char* Str_)
{
  if (!Str_)
    return Str_;

  // replace 0xAE with ' ', 0xAF with '\t', 0xB0 with '\r'
  // and 0xB1 with '\n': for vbuf and inwbox mode input boxes
  bool enterb_ = IsEnterable();
  bool vbuf_ = BufferChainPosition() == 0 && enterb_ &&
               VirtualBufferOffset();
  bool inwbox_ = BufferChainPosition() == 0 && enterb_ &&
                 ((WindowLine*)_Owner)->LineWrapAllowed();
  
  size_t x;
  size_t len;

  if (vbuf_ || inwbox_ || BufferChainPosition() > 0)
  {
    len = ::SafeStrLen(Str_);

    for (x = 0; x < len; x++)
    {
      if (Uchar(Str_[x]) == 0xAE)
        Str_[x] = ' ';
      else if (Uchar(Str_[x]) == 0xAF)
        Str_[x] = '\t';
      else if (Uchar(Str_[x]) == 0xB0)
        Str_[x] = '\r';
      else if (Uchar(Str_[x]) == 0xB1)
        Str_[x] = '\n';
    }
  }
  
  return Str_;
}

/****************************************************************************/
void WindowLineBuffer::FillString(char* Str_, char FillCh_, int Start_, int End_)
{
  for (;Start_ < End_; Start_++)
    Str_[Start_] = FillCh_;

  Str_[End_] = '\0';
}

/****************************************************************************/
/****************************************************************************/
// PURPOSE:
//   End input test procedure
//
// PRE:
//   int key : the key to be tested as the termination key for input.
//
// POST:
//   Returns 1 if the key is defined as a termination key,
//   otherwise returns 0.
//
int WindowLineBuffer::ShouldEnd(int key, int zerotozero)
{
  int status_;
  bool enterb_ = IsEnterable();
  bool vbuf_ = VirtualBufferOffset() && enterb_;
  bool inwbox_ = _BufChainNum == 0 && enterb_ &&
                 ((WindowLine*)_Owner)->LineWrapAllowed();

  // Action: GO_DOWN, ESCAPE
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  if (GetKeyMap().KeyMatch(key, ESCAPE))	
#elif (defined(__linux__) | defined(__unix__))
  if (key == 27)
#endif
  {
    _Escaped = TRUE;
    iMonitor(_escaped_monitor, _escaped_handle, &_Escaped);
    status_ = 1;
  }

  // Action: GO_DOWN, IF_NOTCHAINED_ESCAPE
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, ENTER))	
#elif (defined(__linux__) | defined(__unix__))
  else if (key == 10)
#endif  
  {
    _Escaped = _nextline == NULL;
    iMonitor(_escaped_monitor, _escaped_handle, &_Escaped);
    status_ = 1;
  }

  // Action: GO_DOWN, GO_NEXT
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, NEXT) || GetKeyMap().KeyMatch(key, DOWN))	
#elif (defined(__linux__) | defined(__unix__))
  else if (key == 9 || key == KEY_DOWN)
#endif    
    status_ = 1;

  // Action: GO_UP
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, UP))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_UP)
#endif
    status_ = -1;

  // Action: GO_BACK, GO_LEFT (Note Plural Actions For Single Key)
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, BACK) || GetKeyMap().KeyMatch(key, LEFT))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_BACKSPACE || key == KEY_LEFT)
#endif
    status_ = (_index <= 0 && zerotozero) ? ((!vbuf_ && !inwbox_) ? -1:0):0;

  // Action: GO_RIGHT (Note Plural Actions For Single Key)
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, RIGHT) || IsUserKey(key))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_RIGHT || IsUserKey(key))
#endif
    status_ = (_index >= _len) ? ((!vbuf_ && !inwbox_) ? 1:0):0;

  // Otherwise, no action
  else
    status_ = 0;

  return status_;
}

/****************************************************************************/
// PURPOSE:
//   Get line input procedure
//
// PRE:
//   The original input buffer
//
// POST:
//   Gets input from the keyboard and place it in the input buffer until
//   a termination is press.
//
int WindowLineBuffer::GetLine(int status_)
{
  int lastindex_;
  int zerotozero_;

  int WrapIndex_ = 0;
  int WrapMax_ = 0;
  bool DoChain_ = false;
  bool enterb_ = false;
  bool vbuf_ = false;  
  bool inwbox_ = false;
  int lastnwspos_ = 0;

  ResetInputState(status_);

  cMonitor(_buffer_monitor, _buffer_handle, _buffer);
  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
  iMonitor(_active_monitor, _active_handle, &_TextWidgetActive);

  for (;;)
  {
    GetKey();
    lastindex_ = _index;

    if (_extended)
      PutExtKey(_extkey);
    else
      PutStdKey(_extkey);

    enterb_ = IsEnterable();
    vbuf_ = BufferChainPosition() == 0 && enterb_ &&
            VirtualBufferOffset();
    inwbox_ = BufferChainPosition() == 0 && enterb_ &&
               ((WindowLine*)_Owner)->LineWrapAllowed();
    
    WrapIndex_ = enterb_ ? ((WindowLine*)_Owner)->GetWrapAroundIndex():0;
    WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;
    
    // lastnwspos_ = inwbox_ ? HasNonWsChars(_buffer):0;
    lastnwspos_ = inwbox_ ?
      ((_inwboxcnt > 0 && WrapIndex_ >= WrapMax_ &&
        _inwboxcnt >= WrapMax_ && WrapIndex_ <= _inwboxcnt) ?
            HasNonNullChars(_buffer):
            HasNonWsChars(_buffer)):0;
    
    DoChain_ = IsChainable(false);
    zerotozero_ = !lastindex_ && !_index && !_WrapChars;

    if (_WrapChars && WrapIndex_ > 0 && _Owner && enterb_)
      _WrapChars = (_nextline && DoChain_ && inwbox_) &&
                   (0 < WrapIndex_ &&
                      (WrapIndex_ < WrapMax_ ||
                         ((lastnwspos_ < _len) ? (_inwboxcnt >= WrapMax_ && WrapIndex_ <= _inwboxcnt):
                                                 (_inwboxcnt == WrapMax_ && WrapIndex_ <= WrapMax_))));
    else
      _WrapChars = false;

    if (status_ = ShouldEnd(_extkey, zerotozero_))
      break;
  }

  return status_;
}

/****************************************************************************/
void WindowLineBuffer::ResetInputState(int status_)
{
  _Escaped = FALSE;
  _index = (status_ == -1) ? ::SafeStrLen(_buffer):0;

  if (status_ == -1 && ::SafeStrLen(_buffer) >= _len && _len > 0)
    _index = _len - 1;

  iMonitor(_escaped_monitor, _escaped_handle, &_Escaped);
  iMonitor(_index_monitor, _index_handle, &_index);
}

/****************************************************************************/
// PURPOSE:
//   Read input procedure
//
// POST:
//   Calls procedure to Show the contents of the input buffer as default
//   input to the Prompt box then calls GetLine to get additional input.
//   from the keyboard.
//
int WindowLineBuffer::ReadInput(int status_)
{
  ShowInput(status_);
  return GetLine(status_);
}

/****************************************************************************/
// PURPOSE:
//   Method to Erase the input
//
// POST:
//   Erases the internal buffer by NULL padding entire buffer.
//
void WindowLineBuffer::Erase()
{
  bool enterb_ = IsEnterable();
  bool LwAllowed_ = enterb_ ? ((WindowLine*)_Owner)->LineWrapAllowed():0;
  bool ShowChained_ = enterb_ ? ((WindowLine*)_Owner)->ShowChainedBoxes():0;
  
  if (!IsEnterable() || _bufferoffset ||
      (_BufChainNum && LwAllowed_ && !ShowChained_))
    return;

  if (!IsActive())
    return;

  if (_nextline)
    _nextline->Erase();

  if (::SafeStrLen(_buffer))
  {
    if (!_Hidden)
    {
      _InFocus = TRUE;
      iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

      if (UsesColor())
        SetupColors(ColorInfo::BACKGROUND_COLOR);
      else
        UseColor(FALSE);

      ResetPrompt();
      _InFocus = FALSE;
      iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
    }

    int max = VirtualBufferOffset() ? _maxbuf:_len;
    for (int index = 0; index < max; index++)
      _buffer[index] = '\0';
  }
}

/****************************************************************************/
// PURPOSE:
//   reset user Prompt method
//
// POST:
//   The user Prompt is setup at the specified position and the original
//   contents of the line is Cleared
//
void WindowLineBuffer::ResetPrompt()
{
  bool StartMet_ = false;
  bool EndMet_ = false;
  bool enterb_ = IsEnterable();
  bool LwAllowed_ = enterb_ ? ((WindowLine*)_Owner)->LineWrapAllowed():0;
  bool ShowChained_ = enterb_ ? ((WindowLine*)_Owner)->ShowChainedBoxes():0;

  bool vbuf_ = BufferChainPosition() == 0 && enterb_ &&
               VirtualBufferOffset();
  bool inwbox_ = BufferChainPosition() == 0 && enterb_ &&
                 ((WindowLine*)_Owner)->LineWrapAllowed();
  int WrapIndex_ = enterb_ ? ((WindowLine*)_Owner)->GetWrapAroundIndex():0;
  int WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;
  

  if (!IsEnterable() || _bufferoffset ||
      (_BufChainNum && LwAllowed_ && !ShowChained_))
    return;

  if (EndBrksActive())
  {
    GOTOXY(_xpos + ::SafeStrLen(_msg), _ypos);
    PrintBlanks(_len + EndBrksLength(false));
    GOTOXY(_xpos + ::SafeStrLen(_msg), _ypos);

    StartMet_ = (vbuf_ && _inboxoffset == 0) || (inwbox_ && _inwboxcnt == 0);
    EndMet_ = vbuf_ ?
                ((WrapIndex_ == WrapMax_ && WrapMax_ == 0 && _inboxoffset == 0) ||
                 (_inboxoffset > WrapMax_ && WrapIndex_ == _inboxoffset) ||
                 (_inboxoffset == WrapMax_ && WrapIndex_ == WrapMax_)):
              inwbox_ ?
                ((WrapIndex_ == WrapMax_ && WrapMax_ == 0 && _inwboxcnt == 0) ||
                 (_inwboxcnt > WrapMax_ && WrapIndex_ == _inwboxcnt) ||
                 (_inwboxcnt == WrapMax_ && WrapIndex_ == WrapMax_)):0;

    if (vbuf_)
    {
      if (EndMet_ &&
          (WrapIndex_ == WrapMax_ && WrapMax_ == 0 && _inboxoffset == 0) ||
          (!(_inboxoffset > WrapMax_ && WrapIndex_ == _inboxoffset) &&
           !(_inboxoffset == WrapMax_ && WrapIndex_ == WrapMax_)))
        EndMet_ = !IsChainable(true);
    }
    else if (inwbox_)
    {
      if (EndMet_ &&
          (WrapIndex_ == WrapMax_ && WrapMax_ == 0 && _inwboxcnt == 0) ||
          (!(_inwboxcnt > WrapMax_ && WrapIndex_ == _inwboxcnt) &&
           !(_inwboxcnt == WrapMax_ && WrapIndex_ == WrapMax_)))
        EndMet_ = !IsChainable(true);
    }

    if (StartMet_ && (vbuf_ || inwbox_))
    {
      SetupColors(ColorInfo::HIGHLIGHT_COLORS);
      PUTCH(' ');
    }    
    else
    {
      SetupColors(CONTMORE_HFG, CONTMORE_HBG);
      PUTCH(_brackets[0]);
      SetupColors(ColorInfo::HIGHLIGHT_COLORS);
    }
    
    GOTOXY(wherex() + _len, _ypos);

    if (EndMet_ && (vbuf_ || inwbox_))
    {
      SetupColors(ColorInfo::HIGHLIGHT_COLORS);
      PUTCH(' ');
    }
    else
    {
      SetupColors(CONTMORE_HFG, CONTMORE_HBG);
      PUTCH(_brackets[1]);
      SetupColors(ColorInfo::HIGHLIGHT_COLORS);
    }

    GOTOXY(_xpos + ::SafeStrLen(_msg) + EndBrksLength(true), _ypos);  
  }
  else
  {
    GOTOXY(_xpos + ::SafeStrLen(_msg), _ypos);
    PrintBlanks(_len + EndBrksLength(false));
    GOTOXY(_xpos + ::SafeStrLen(_msg), _ypos);
  }
}

/****************************************************************************/
// PURPOSE:
//   Show Prompt message method
//
// POST:
//   Goes to the specified coordinates and Shows the Prompt message
//   to the screen
//
void WindowLineBuffer::ShowMessage()
{
  bool enterb_ = IsEnterable();
  bool LwAllowed_ = enterb_ ? ((WindowLine*)_Owner)->LineWrapAllowed():0;
  bool ShowChained_ = enterb_ ? ((WindowLine*)_Owner)->ShowChainedBoxes():0;

  if (!IsEnterable() || _bufferoffset ||
      (_BufChainNum && LwAllowed_ && !ShowChained_))
    return;

  GOTOXY(_xpos, _ypos);

  if (_msg)
    CPUTS(_msg);
}

/****************************************************************************/
// PURPOSE:
//   Show input method
//
// POST:
//   Shows the Prompt box and Shows the original contents of the input
//   buffer within the box
//
void WindowLineBuffer::ShowInput(int status_)
{
  bool enterb_ = IsEnterable();
  bool LwAllowed_ = enterb_ ? ((WindowLine*)_Owner)->LineWrapAllowed():0;
  bool ShowChained_ = enterb_ ? ((WindowLine*)_Owner)->ShowChainedBoxes():0;
  int BoxCount_ = enterb_ ? ((WindowLine*)_Owner)->GetBoxCount():0;

  if (!IsEnterable() || _bufferoffset ||
      (_BufChainNum && LwAllowed_ && !ShowChained_))
    return;
    
  if (_len)
  {
    char* strptr_ = NULL;
    bool vbuf_ = enterb_ && VirtualBufferOffset();
    bool inwbox_ = BufferChainPosition() == 0 && enterb_ &&
                   ((WindowLine*)_Owner)->LineWrapAllowed();    
    int vmax_ = vbuf_ ? VirtualBufferMax():_maxbuf;
    int WrapIndex_ = enterb_ ? ((WindowLine*)_Owner)->GetWrapAroundIndex():0;
    int WrapMax_ = enterb_ ? ((WindowLine*)_Owner)->GetMaxWrapAround():0;

    char lastchar_ = 0;
    int lastnwspos_ = 0;
    int startpt_ = 0;
    int endpt_ = _len;

    ResetPrompt();
    if (ShouldTrimWs() && _buffer &&
        (vbuf_ || !inwbox_ ||
         (WrapIndex_ == 0 && BufferChainPosition() == BoxCount_-1)))
      _buffer = RemoveTrailSpaceChars(_buffer, true);

    if (vbuf_)
    {
      if (_inboxoffset <= vmax_)
        vmax_ = _inboxoffset;

      startpt_ += vmax_;
      endpt_ += vmax_;

      lastchar_ = _buffer[endpt_];
      _buffer[endpt_] = 0;

      // lastnwspos_ = HasNonWsChars(&_buffer[startpt_]);
      lastnwspos_ =
        (_inboxoffset > 0 && WrapIndex_ >= WrapMax_ &&
         _inboxoffset >= WrapMax_ && WrapIndex_ <= _inboxoffset) ?
             HasNonNullChars(&_buffer[startpt_]):
             HasNonWsChars(&_buffer[startpt_]);
      
      strptr_ = StrRetrieve();
      if (strptr_)
        CPUTS(&strptr_[startpt_]);
      _buffer[endpt_] = lastchar_;
    }
    else
    {
      strptr_ = StrRetrieve();
      if (strptr_)
        CPUTS(strptr_);
    }

    if (_nextline && !VirtualBufferOffset())
      _nextline->ShowInput();

    GOTOXY(_xpos + ::SafeStrLen(_msg) + EndBrksLength(true), _ypos);

    if (status_ == -1)
    {
      if (!vbuf_)
        lastnwspos_ = ::SafeStrLen(_buffer);
      
      GOTOXY(wherex() + lastnwspos_, _ypos);

      if (lastnwspos_ >= _len && _len > 0)
      {
        _index = lastnwspos_;
        
        while (lastnwspos_ >= _len)
        {
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
          PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::LEFTARROW).SCAN + 256);
#elif (defined(__linux__) | defined(__unix__))
          PutExtKey(KEY_LEFT);
#endif
          --lastnwspos_;
        }

        iMonitor(_index_monitor, _index_handle, &_index);
      }
    }
  }
}

/****************************************************************************/
// PURPOSE:
//   Prompt user method
//
// POST:
//   Shows the Prompt message and the Prompt box then Prompts user for
//   additional input
//
int WindowLineBuffer::Prompt(int status_)
{
  bool enterb_ = IsEnterable();
  bool LwAllowed_ = enterb_ ? ((WindowLine*)_Owner)->LineWrapAllowed():0;
  bool ShowChained_ = enterb_ ? ((WindowLine*)_Owner)->ShowChainedBoxes():0;

  if (!IsEnterable() || _bufferoffset ||
      (_BufChainNum && LwAllowed_ && !ShowChained_))
    return status_;

  if (!IsActive())
    return status_;

  if (_len)
  {
    _InFocus = TRUE;
    iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

    SetupColors(ColorInfo::HIGHLIGHT_COLORS);
    ShowMessage();
    status_ = ReadInput(status_);
    Show();

    _InFocus = FALSE;
    iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
    _Hidden = FALSE;
    iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
  }

  return status_;
}

/****************************************************************************/
// PURPOSE:
//   Shows Prompt box method
//
// POST:
//   Shows Prompt message and Prompt box with contents of input buffer
//
void WindowLineBuffer::Show()
{
  bool enterb_ = IsEnterable();
  bool LwAllowed_ = enterb_ ? ((WindowLine*)_Owner)->LineWrapAllowed():0;
  bool ShowChained_ = enterb_ ? ((WindowLine*)_Owner)->ShowChainedBoxes():0;

  if (!IsEnterable() || _bufferoffset ||
      (_BufChainNum && LwAllowed_ && !ShowChained_))
    return;

  if (!IsActive())
    return;

  if (_nextline && !VirtualBufferOffset())
    _nextline->Show();

  _InFocus = TRUE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

  SetupColors(ColorInfo::NORMAL_COLORS);
  ShowMessage();
  ShowInput(1);

  _InFocus = FALSE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
  _Hidden = FALSE;
  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
}

/****************************************************************************/
void WindowLineBuffer::ShowInput()
{
  bool enterb_ = IsEnterable();
  bool LwAllowed_ = enterb_ ? ((WindowLine*)_Owner)->LineWrapAllowed():0;
  bool ShowChained_ = enterb_ ? ((WindowLine*)_Owner)->ShowChainedBoxes():0;

  if (!IsEnterable() || _bufferoffset ||
      (_BufChainNum && LwAllowed_ && !ShowChained_))
    return;

  _InFocus = TRUE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

  ShowInput(1);

  _InFocus = FALSE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
  _Hidden = FALSE;
  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
}

/****************************************************************************/
// PURPOSE:
//   Hide Prompt box method
//
// POST:
//   goes to the specified coordinates of the Prompt box then Clears the
//   line containing the Prompt box.
//
void WindowLineBuffer::Hide()
{
  if (!_Owner)
    return;
  
  bool enterb_ = IsEnterable();  
  bool LwAllowed_ = enterb_ ? ((WindowLine*)_Owner)->LineWrapAllowed():0;
  bool ShowChained_ = enterb_ ? ((WindowLine*)_Owner)->ShowChainedBoxes():0;

  if (!IsEnterable() || _bufferoffset ||
      (_BufChainNum && LwAllowed_ && !ShowChained_))
    return;

  if (!IsActive())
    return;

  if (_nextline && !VirtualBufferOffset())
    _nextline->Hide();

  _InFocus = TRUE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

  if (UsesColor())
    SetupColors(ColorInfo::BACKGROUND_COLOR);
  else
    UseColor(FALSE);

  GOTOXY(_xpos, _ypos);
  PrintBlanks(::SafeStrLen(_msg) + _len + EndBrksLength(false));
  GOTOXY(_xpos, _ypos);

  _InFocus = FALSE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
  _Hidden = TRUE;
  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
}

/****************************************************************************/
void WindowLineBuffer::SetEnterable(Boolean Flag_)
{
  _Enterable = _bufferoffset ? FALSE:Flag_;
}

/****************************************************************************/
void WindowLineBuffer::SetSelectable(Boolean Flag_)
{
  _Selectable = Flag_;
}

/****************************************************************************/
int WindowLineBuffer::GetHotKey() const
{
  return 0;
}

/****************************************************************************/
int WindowLineBuffer::GetShortCut() const
{
  return 0;
}

/****************************************************************************/
int WindowLineBuffer::GetHotKeyType() const
{
  return 0;
}

/****************************************************************************/
void WindowLineBuffer::Select()
{
  if (!IsSelected())
    if (_maxbuf)
      *_buffer = BLANK;
}

/****************************************************************************/
void WindowLineBuffer::Deselect()
{
  if (_maxbuf)
    *_buffer = 0;
}

/****************************************************************************/
void WindowLineBuffer::Activate()
{
  _TextWidgetActive = TRUE;
  iMonitor(_active_monitor, _active_handle, &_TextWidgetActive);

  if (_nextline)
    _nextline->Activate();
}

/****************************************************************************/
void WindowLineBuffer::Deactivate()
{
  _TextWidgetActive = FALSE;
  iMonitor(_active_monitor, _active_handle, &_TextWidgetActive);

  if (_nextline)
    _nextline->Deactivate();
}

/****************************************************************************/
void WindowLineBuffer::DecrBoxIndex()
{
  bool enterb_ = IsEnterable();
  bool vbuf_ = enterb_ && VirtualBufferOffset();

  if (vbuf_)
  {
    if (_inboxoffset - 1 >= 0)
      _inboxoffset--;
  }
  else
  {
    if (_inwboxcnt - 1 >= 0)
      _inwboxcnt--;
  }
}

/****************************************************************************/
void WindowLineBuffer::IncrBoxIndex()
{
  bool enterb_ = IsEnterable();
  bool vbuf_ = enterb_ && VirtualBufferOffset();

  if (vbuf_)
  {
    if (_inboxoffset + 1 <= _maxbuf || _maxbuf == -1)
      _inboxoffset++;
  }
  else
    _inwboxcnt++;
}

/****************************************************************************/
void WindowLineBuffer::ResetBoxIndex()
{
  bool enterb_ = IsEnterable();
  bool vbuf_ = enterb_ && VirtualBufferOffset();

  if (vbuf_)
    _inboxoffset = 0;
  else
    _inwboxcnt = 0;
}

/****************************************************************************/
void WindowLineBuffer::SetBoxIndex(int Val_)
{
  bool enterb_ = IsEnterable();
  bool vbuf_ = enterb_ && VirtualBufferOffset();

  if (vbuf_)
  {
    if (0 <= Val_ && Val_ <= _maxbuf || (Val_ >= 0 && _maxbuf == -1))
      _inboxoffset = Val_;
  }
  else
  {
    if (0 <= Val_ || (Val_ >= 0 && _maxbuf == -1))
      _inwboxcnt = Val_;
  }
}

/****************************************************************************/
Boolean WindowLineBuffer::IsEmpty() const
{
  return (!_buffer || !::SafeStrLen(_buffer));
}

/****************************************************************************/
Boolean WindowLineBuffer::IsEscaped() const
{
  return _Escaped;
}

/****************************************************************************/
Boolean WindowLineBuffer::IsActive() const
{
  return _TextWidgetActive;
}

/****************************************************************************/
Boolean WindowLineBuffer::IsEnterable() const
{
  return (_Enterable && _bufferoffset <= 0);
}

/****************************************************************************/
Boolean WindowLineBuffer::IsSelectable() const
{
  return _Selectable;
}

/****************************************************************************/
Boolean WindowLineBuffer::IsSelected() const
{
  return (_buffer && ::SafeStrLen(_buffer));
}

/****************************************************************************/
Boolean WindowLineBuffer::IsValid() const
{
  return (!(IsEmpty() && _noempty) && IsValidLength());
}

/****************************************************************************/
Boolean WindowLineBuffer::IsInFocus() const
{
  return _InFocus;
}

/****************************************************************************/
char WindowLineBuffer::LastKeyRead(int* Ext_) const
{
  if (Ext_)
    *Ext_ = _extended;

  return _key;
}

/****************************************************************************/
int WindowLineBuffer::LastExtKeyRead() const
{
  return _extkey;
}

/****************************************************************************/
char* WindowLineBuffer::StrRetrieve()
{
  if (!_buffer || !_retbuf)
    return _buffer;

  strcpy(_retbuf, _buffer);
  return Revert(_retbuf);
}

/****************************************************************************/
char WindowLineBuffer::ChrRetrieve(int Index_)
{
  char retch_ = 0;
  if (_buffer && 0 <= Index_ && Index_ < ::SafeStrLen(_buffer))
    retch_ = _buffer[Index_];

  if (retch_ == 7)
    retch_ = ' ';
  else if (retch_ == 8)
    retch_ = '\t';

  return retch_;
}

/****************************************************************************/
Boolean WindowLineBuffer::IsOwner(ControlWindow* Pwin_) const
{
  return (_Owner == Pwin_);
}

/****************************************************************************/
void WindowLineBuffer::SetOwner(ControlWindow* Pwin_, TextControl* TxtCtrl_)
{
  if (_Owner && _Owner != Pwin_)
    _Owner->RemoveControl(TxtCtrl_);
  _Owner = Pwin_;
}

/****************************************************************************/
void WindowLineBuffer::SetReturnBuffer(WindowLineBufferType* Lbuf_, char* RetBufPtr_, bool Alloc_, int Size_)
{
  if (Lbuf_)
    if (this != Lbuf_)
      Lbuf_->SetReturnBuffer(Lbuf_, RetBufPtr_, Alloc_, Size_);
    else if (RetBufPtr_ && _retbuf != RetBufPtr_)
    {
      if (_retbufalloc)
        delete _retbuf;

      if (Alloc_)
      {
        _retbuf = strcpy(new char[Size_+1], RetBufPtr_);
        _retbufalloc = true;
      }
      else
      {
        _retbuf = RetBufPtr_;
        _retbufalloc = false;
      }
    }
}

/****************************************************************************/
void WindowLineBuffer::SetBufferChainNum(WindowLineBufferType* Lbuf_, int Index_)
{
  if (Lbuf_)
    if (this != Lbuf_)
      Lbuf_->SetBufferChainNum(Lbuf_, Index_);
    else
      _BufChainNum = Index_;
}

/****************************************************************************/
void WindowLineBuffer::SetWindowLineNum(int Index_)
{
  _WinLineNum = Index_;
}

/****************************************************************************/
int WindowLineBuffer::InputLength() const
{
  return _len;
}

/****************************************************************************/
void WindowLineBuffer::SetupColors(int Fg_, int Bg_)
{
  _ColorInfo.SetupColors(Fg_, Bg_);
  // GetDefaultColorInfo().SetupColors(Fg_, Bg_);
}

/****************************************************************************/
void WindowLineBuffer::SetupColors(int Mode_)
{
  if (!_ColorInfo.SetupColors(Mode_))
    GetDefaultColorInfo().SetupColors(Mode_);
}

/****************************************************************************/
void WindowLineBuffer::UseColor(Boolean Flag_)
{
  if (_ColorInfo._UseColor)
    *_ColorInfo._UseColor = Flag_;
  else
    _ColorInfo._UseColor = new Boolean(Flag_);

  GetDefaultColorInfo().UseColor(Flag_);
}

/****************************************************************************/
void WindowLineBuffer::SetNormalColors(int FgCol_, int BgCol_)
{
  if (_ColorInfo._NormalColors)
  {
    _ColorInfo._NormalColors[0] = FgCol_;
    _ColorInfo._NormalColors[1] = BgCol_;
  }
  else
  {
    _ColorInfo._NormalColors = new int[2];
    SetNormalColors(FgCol_, BgCol_);
  }
}

/****************************************************************************/
void WindowLineBuffer::SetHighLightColors(int FgCol_, int BgCol_)
{
  if (_ColorInfo._HighLightColors)
  {
    _ColorInfo._HighLightColors[0] = FgCol_;
    _ColorInfo._HighLightColors[1] = BgCol_;
  }
  else
  {
    _ColorInfo._HighLightColors = new int[2];
    SetHighLightColors(FgCol_, BgCol_);
  }
}

/****************************************************************************/
void WindowLineBuffer::SetHotKeyColor(int Col_)
{
  if (_ColorInfo._HkColor)
    *_ColorInfo._HkColor = Col_;
  else
    _ColorInfo._HkColor = new int(Col_);
}

/****************************************************************************/
void WindowLineBuffer::SetBackGroundColor(int Col_)
{
  if (_ColorInfo._BackGroundColor)
    *_ColorInfo._BackGroundColor = Col_;
  else
    _ColorInfo._BackGroundColor = new int(Col_);
}

/****************************************************************************/
void WindowLineBuffer::UseDefaultColors()
{
  _ColorInfo.ClearColors();
  UseColor();
}

/****************************************************************************/
Boolean WindowLineBuffer::UsesColor() const
{
  return
  (
    _ColorInfo._UseColor ?
      *(_ColorInfo._UseColor):
      *(GetDefaultColorInfo()._UseColor)
  );
}

/****************************************************************************/
int* WindowLineBuffer::NormalColors() const
{
  return
  (
    _ColorInfo._NormalColors ?
      _ColorInfo._NormalColors:
      GetDefaultColorInfo()._NormalColors
  );
}

/****************************************************************************/
int* WindowLineBuffer::HighLightColors() const
{
  return
  (
    _ColorInfo._HighLightColors ?
      _ColorInfo._HighLightColors:
      GetDefaultColorInfo()._HighLightColors
  );
}

/****************************************************************************/
int WindowLineBuffer::HotKeyColor() const
{
  return
  (
    _ColorInfo._HkColor ?
      *(_ColorInfo._HkColor):
      *(GetDefaultColorInfo()._HkColor)
  );
}

/****************************************************************************/
int WindowLineBuffer::BackGroundColor() const
{
  return
  (
    _ColorInfo._BackGroundColor ?
      *(_ColorInfo._BackGroundColor):
      *(GetDefaultColorInfo()._BackGroundColor)
  );
}

/****************************************************************************/
// WindowDummyLine class definition
/****************************************************************************/
const int* InsertDetect(void* Owner_, WindowDummyLine*, const int* Ptr_)
{
  if (Owner_)
    ((WindowDummyLine*)Owner_)->NotifyInsertMode(Ptr_);

  return Ptr_;
}

/****************************************************************************/
// PURPOSE:
//   Prompt box default constructor
//
// PRE:
//   an uninitialize Prompt box
//
// POST:
//   a Prompt box is constructed and assigned the default values specified.
//   If the msg argument is not NULL then the Prompt message of the box
//   is assigned to the argument string.
//   The string buffer is Cleared to all 0.
//
WindowDummyLine::WindowDummyLine(int BufChainNum_, WindowLineBuffer* Parent_, int Length_):
_Owner(NULL),
_Enterable(FALSE),
_Selectable(FALSE),

_parent((BufChainNum_ == 0) ? Parent_:NULL),
_nextline(NULL),
_nextlinealloc(false),
_bufferoffset(0),
_inboxoffset(0),
_inwboxcnt(0),
_maxbuf(-1),
_len(Length_ ? Length_:1),   // _len should never be zero
_minlen(1),                  // minimum length should be at least 1
_index(0),
_extindex(0),
_xpos(WHEREX()),
_ypos(WHEREY()),
_noempty(0),
_InsertMode(TRUE),
_Escaped(FALSE),
_Hidden(TRUE),
_TextWidgetActive(FALSE),
_InFocus(FALSE),
_SavedCursX(WHEREX()),
_SavedCursY(WHEREY()),
_BufChainNum(-99),
_WinLineNum(0),

// Internal data monitors
_insert_monitor(NULL),
_key_monitor(NULL),
_ext_monitor(NULL),
_index_monitor(NULL),
_escaped_monitor(NULL),
_hidden_monitor(NULL),
_active_monitor(NULL),
_infocus_monitor(NULL),
_inboxoffset_monitor(NULL),
_winlinenum_monitor(NULL),

// Handles to external objects, used by data monitors
_insert_handle(NULL),
_key_handle(NULL),
_ext_handle(NULL),
_index_handle(NULL),
_escaped_handle(NULL),
_hidden_handle(NULL),
_active_handle(NULL),
_infocus_handle(NULL),
_inboxoffset_handle(NULL),
_winlinenum_handle(NULL)
{
  if (_parent && BufChainNum_ == 0)
  {
    _bufferoffset = _parent->_index;
    _inboxoffset = _parent->_inboxoffset;
    _inwboxcnt = _parent->_inwboxcnt;
  }
}

/****************************************************************************/
// PURPOSE:
//   Prompt box copy construcor
//
// PRE:
//   An uninitialized Prompt box.
//
// POST:
//   a Prompt box is constructed and assigned the corresponding values
//   in the Prompt box argument passed to the constructor.
//
WindowDummyLine::WindowDummyLine(WindowDummyLine& pb, int Length_):
_Owner(NULL),
_Enterable(FALSE),
_Selectable(pb._Selectable),

_parent(pb._parent),
_nextline(NULL),
_nextlinealloc(false),
_bufferoffset(pb._bufferoffset),
_inboxoffset(pb._inboxoffset),
_inwboxcnt(pb._inwboxcnt),
_maxbuf(-1),
_len(pb._len ? pb._len:
     Length_ ? Length_:1),   // _len should never be zero

_minlen(pb._minlen ? pb._minlen:1),     
_index(pb._index),
_extindex(pb._extindex),
_xpos(pb._xpos),
_ypos(pb._ypos),
_noempty(pb._noempty),
_InsertMode(TRUE),
_Escaped(FALSE),
_Hidden(TRUE),
_TextWidgetActive(FALSE),
_InFocus(FALSE),
_SavedCursX(WHEREX()),
_SavedCursY(WHEREY()),
_BufChainNum(-99),
_WinLineNum(0),

// Internal data monitors
_insert_monitor(NULL),
_key_monitor(NULL),
_ext_monitor(NULL),
_index_monitor(NULL),
_escaped_monitor(NULL),
_hidden_monitor(NULL),
_active_monitor(NULL),
_infocus_monitor(NULL),
_inboxoffset_monitor(NULL),
_winlinenum_monitor(NULL),

// Handles to external objects, used by data monitors
_insert_handle(NULL),
_key_handle(NULL),
_ext_handle(NULL),
_index_handle(NULL),
_escaped_handle(NULL),
_hidden_handle(NULL),
_active_handle(NULL),
_infocus_handle(NULL),
_inboxoffset_handle(NULL),
_winlinenum_handle(NULL)
{
  if (pb._minlen)
    SetMinimumLength(pb._minlen);
  else
    SetMinimumLength(1);
}

/****************************************************************************/
const char* WindowDummyLine::cMonitor(const char*(*monptr_)(void*, WindowDummyLine*, const char*), void* This_, const char* ptr_)
{
  if (monptr_ != NULL)
  {
    SaveCursorPosition();
    const char* RetVal_ = (*monptr_)(This_, this, ptr_);
    RestoreCursorPosition();

    return RetVal_;
  }

  return ptr_;
}

/****************************************************************************/
const int* WindowDummyLine::iMonitor(const int*(*monptr_)(void*, WindowDummyLine*, const int*), void* This_, const int* ptr_)
{
  if (monptr_ != NULL)
  {
    SaveCursorPosition();
    const int* RetVal_ = (*monptr_)(This_, this, ptr_);
    RestoreCursorPosition();

    return RetVal_;
  }

  return ptr_;
}

/****************************************************************************/
void WindowDummyLine::SaveCursorPosition()
{
  _SavedCursX = WHEREX();
  _SavedCursY = WHEREY();
}

/****************************************************************************/
void WindowDummyLine::RestoreCursorPosition()
{
  GOTOXY(_SavedCursX, _SavedCursY);
}

/****************************************************************************/
const int* WindowDummyLine::MonitorInsertMode(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*))
{
  _insert_handle = This_;
  _insert_monitor = FnPtr_;

  return &_InsertMode;
}

/****************************************************************************/
const int* WindowDummyLine::MonitorKey(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*))
{
  _key_handle = This_;
  _key_monitor = FnPtr_;

  return &_key;
}

/****************************************************************************/
const int* WindowDummyLine::MonitorExtendedKey(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*))
{
  _ext_handle = This_;
  _ext_monitor = FnPtr_;

  return &_extended;
}

/****************************************************************************/
const int* WindowDummyLine::MonitorIndex(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*))
{
  _index_handle = This_;
  _index_monitor = FnPtr_;

  return &_index;
}

/****************************************************************************/
const int* WindowDummyLine::MonitorEscapedStatus(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*))
{
  _escaped_handle = This_;
  _escaped_monitor = FnPtr_;

  return &_Escaped;
}

/****************************************************************************/
const int* WindowDummyLine::MonitorInFocusStatus(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*))
{
  _infocus_handle = This_;
  _infocus_monitor = FnPtr_;

  return &_InFocus;
}

/****************************************************************************/
const int* WindowDummyLine::MonitorHiddenStatus(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*))
{
  _hidden_handle = This_;
  _hidden_monitor = FnPtr_;
  return &_Hidden;
}

/****************************************************************************/
const int* WindowDummyLine::MonitorActiveStatus(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*))
{
  _active_handle = This_;
  _active_monitor = FnPtr_;
  return &_TextWidgetActive;
}

/****************************************************************************/
// PURPOSE:
//   Prompt box data member destruction method
//
// PRE:
//   The data members of this Prompt box object with original values
//
// POST:
//   If the data members of the box object is not NULL then the data members
//   are deleted and assigned NULL
//
void WindowDummyLine::Clear()
{}

/****************************************************************************/
/****************************************************************************/
// PURPOSE:
//   Prompt box assignment operator
//
// PRE:
//   The data members of this Prompt box object with original values
//
// POST:
//   The data members of this object is assigned to the corresponding
//   data members of the Prompt box argument passed to the method.
//   This object is returned.
//
WindowDummyLine& WindowDummyLine::operator = (WindowDummyLine& pb)
{
  if (this != &pb)
  {
    Clear();

    _Enterable = FALSE;
    _Selectable = pb._Selectable;

    _parent = pb._parent;    
    _bufferoffset = pb._bufferoffset;
    _inboxoffset = pb._inboxoffset;
    _inwboxcnt = pb._inwboxcnt;
    _maxbuf = -1;
    _len = pb.InputLength() ? pb.InputLength():1;
    
    _minlen = pb._minlen ? pb._minlen:1;
    _index = pb._index;
    _extindex = pb._extindex;
    _xpos = pb._xpos;
    _ypos = pb._ypos;
    _noempty = pb._noempty;    

    if (pb._minlen)
      SetMinimumLength(pb._minlen);
    else
      SetMinimumLength(1);
  }

  return *this;
}

/****************************************************************************/
WindowDummyLine* WindowDummyLine::Make(int BufChainNum_, WindowLineBuffer* parent_, int Length_)
{
  WindowDummyLine* pb;
  pb = new WindowDummyLine(BufChainNum_, parent_, Length_);
  return pb;
}

/****************************************************************************/
WindowDummyLine* WindowDummyLine::Make(WindowDummyLine& pbox, int Length_)
{
  WindowDummyLine* pb;
  pb = new WindowDummyLine(pbox, Length_);
  return pb;
}

/****************************************************************************/
WindowLineBuffer* WindowDummyLine::MakeNextLine(WindowLineBuffer* wpb)
{
  WindowLineBuffer* nwlb;
  nwlb = wpb ? (new WindowLineBuffer(*wpb, wpb->_maxbuf, wpb->_len)):NULL;

  if (wpb && nwlb)
  {
    int msglen_ = ::SafeStrLen(wpb->_msg);
    nwlb->SetInputLength(wpb->_len, false);
    nwlb->SetXpos(wpb->_xpos + msglen_ + wpb->_len +
                  wpb->EndBrksLength(false));
    nwlb->SetYpos(wpb->_ypos);
  }
  
  return nwlb;
}

/****************************************************************************/
/****************************************************************************/
void WindowDummyLine::PrintBlanks(int Max_)
{
  for (int Index_ = 0; Index_ < Max_; Index_++)
    PUTCH(BLANK);
}

/****************************************************************************/
// PURPOSE:
//   Set Prompt message method
//
// PRE:
//   The original value of the message data member
//
// POST:
//   The message data member is assigned new memory copied from the argument
//   string.
//
char* WindowDummyLine::SetMessage(char* str)
{
  return NULL;
}

/****************************************************************************/
void WindowDummyLine::RemoveMessage()
{
  SetMessage(NULL);
}

/****************************************************************************/
void WindowDummyLine::BlankMessage()
{}

/****************************************************************************/
void WindowDummyLine::PushInputString(char* str)
{}

/****************************************************************************/
// PURPOSE:
//   Set default input method
//
// PRE:
//   The original value of the input buffer data member
//
// POST:
//   The input buffer data member is assigned new memory copied from the
//   argument string. The remaining space in the buffer is NULL filled.
//
void WindowDummyLine::SetInputString(char* str)
{}

/****************************************************************************/
// PURPOSE:
//   Set input length method
//
// PRE:
//   The original value of the input length data member
//
// POST:
//   The input length data member is copied from the length argument.
//
void WindowDummyLine::SetInputLength(int len, bool MakeVbuf_)
{
  _len = len;
}

/****************************************************************************/
void WindowDummyLine::DivBufSizeAsInputLength(int div_, bool MakeVbuf_)
{
  int res_ = _len / div_;
  _len = res_;
}

/****************************************************************************/
// PURPOSE:
//   Set minimum input length method
//
// PRE:
//   The original value of the minimum input length data member
//
// POST:
//   The mimimum input length data member is copied from the length argument.
//
void WindowDummyLine::SetMinimumLength(int len)
{
  if (len <= _len)
    _minlen = (len > 0) ? len:1;
  else
    _minlen = _len;
}

/****************************************************************************/
int* WindowDummyLine::WordChr(int* src_, int val_, size_t n)
{
  while (n)
    if (*src_ == val_)
      return src_;
    else
    {
      src_++;
      --n;
    }

  return NULL;
}

/****************************************************************************/
void WindowDummyLine::SetNoEmpty(int flag)
{
  _noempty = flag;
}

/****************************************************************************/
void WindowDummyLine::NotifyInsertMode(const int* ModePtr_)
{
  if (ModePtr_)
    _InsertMode = *ModePtr_;

  iMonitor(_insert_monitor, _insert_handle, ModePtr_);
}

/****************************************************************************/
void WindowDummyLine::SetInsertMode(int Mode_, WindowDummyLine* Parent_)
{
  _InsertMode = Mode_;
  iMonitor(_insert_monitor, _insert_handle, &_InsertMode);

  if (Parent_)
    MonitorInsertMode(Parent_, InsertDetect);
}

/****************************************************************************/
void WindowDummyLine::SetXYpos(int x, int y)
{
  SetXpos(x);
  SetYpos(y);
}

/****************************************************************************/
#if (!(defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__)) & (defined(__linux__) | defined(__unix__)))
int WindowDummyLine::TranslateKeys(int key)
{
  return
  (
    GetKeyMap().KeyMatch(key, INSERT)   ? KEY_IC:
    GetKeyMap().KeyMatch(key, LEFT)     ? KEY_LEFT:
    GetKeyMap().KeyMatch(key, RIGHT)    ? KEY_RIGHT:
    GetKeyMap().KeyMatch(key, HOME)     ? KEY_HOME:
    GetKeyMap().KeyMatch(key, END)      ? KEY_END:
    GetKeyMap().KeyMatch(key, DELETE)   ? KEY_DC:
    GetKeyMap().KeyMatch(key, ENTER)    ? 10:
    GetKeyMap().KeyMatch(key, BACK)     ? KEY_BACKSPACE:
    GetKeyMap().KeyMatch(key, ESCAPE)   ? 27:
    GetKeyMap().KeyMatch(key, NEXT)     ? 9:
    GetKeyMap().KeyMatch(key, PAGEUP)   ? KEY_PPAGE:
    GetKeyMap().KeyMatch(key, PAGEDOWN) ? KEY_NPAGE:
    GetKeyMap().KeyMatch(key, UP)       ? KEY_UP:
    GetKeyMap().KeyMatch(key, DOWN)     ? KEY_DOWN:
                                          key
  );
}
#endif

/****************************************************************************/
// PURPOSE:
//   Keyboard retrieval procedure
//
// PRE:
//   The original value of _key and _extended data members
//
// POST:
//   The keyboard is read for a key and is assigned to _key data member.
//   If the key is extended non-ASCII then the _extended data member is set.
//   Returns the key read from the keyboard.
//
int WindowDummyLine::GetKey()
{
  _extkey = _key = (int)GETCH();
  iMonitor(_key_monitor, _key_handle, &_key);
  _extended = (_key == 0 || _key == 0xE0) ? 256:0;
  iMonitor(_ext_monitor, _ext_handle, &_extended);

  if (_extended)
  {
#if (!(defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__)) & (defined(__linux__) | defined(__unix__)))
    noecho();
#endif	  
    _key = (int)GETCH();
    _extkey = _key + _extended;
    iMonitor(_key_monitor, _key_handle, &_key);

#if (!(defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__)) & (defined(__linux__) | defined(__unix__)))
    _extkey = TranslateKeys(_extkey);
#endif
  }
#if (!(defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__)) & (defined(__linux__) | defined(__unix__)))
  else
    _extkey = TranslateKeys(_extkey);
#endif

  return _key;
}

/****************************************************************************/
// PURPOSE:
//   Method for putting extended non-ASCII keyboard keys to the console
//   output.
//
// PRE:
//   int key : the key to be outputted.
//
// POST:
//   The key is outputted to the screen. Appropriate changes to the
//   internal input buffer is made.
//
void WindowDummyLine::PutExtKey(int key)
{
  int pos;
  char nextchar_;
  int index_ = 0;
  int endpt_ = _len-1;

  // Action: TOGGLE_INSERT
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  if (GetKeyMap().KeyMatch(key, INSERT))
#elif (defined(__linux__) | defined(__unix__))
  if (key == KEY_IC)
#endif
    SetInsertMode(!GetInsertMode());

  // Action: GO_LEFT (chain push if WindowLine::_wraparoundindex > 0)
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, LEFT))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_LEFT)
#endif  
  {
    // Modified to wrap chars around first buffer in window buffer object
    if (_extindex > 0)    
      if (_boxpos > 0)
      {
        GOTOXY(wherex()-1, _ypos);
        --_extindex;

        index_ = _index + _extindex;
        iMonitor(_index_monitor, _index_handle, &index_);
      }
      else if (_boxpos == 0)
      {
        --_extindex;
        pos = wherex();
        ShowInput();
        GOTOXY(pos, _ypos);

        index_ = _index + _extindex;
        iMonitor(_index_monitor, _index_handle, &index_);
      }
  }

  // Action: GO_RIGHT (chain pull if _extindex >= WindowLine::_inputlen)
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, RIGHT))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_RIGHT)
#endif  
  {
    if (_extindex + 1 <= _maxbuf || _maxbuf == -1)
      if (_boxpos == endpt_)
      {
        ++_extindex;
        pos = wherex();
        ShowInput();
        GOTOXY(pos, _ypos);
      
        index_ = _index + _extindex;
        iMonitor(_index_monitor, _index_handle, &index_);
      }
      else if (_boxpos < endpt_)
      {
        GOTOXY(wherex()+1, _ypos);
        ++_extindex;

        index_ = _index + _extindex;
        iMonitor(_index_monitor, _index_handle, &index_);
      }
  }

  // Action: GO_HOME
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, HOME))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_HOME)
#endif  
  {
    _extindex = 0;
    _index = 0;
    _boxpos = 0;
    if (_parent)
      _parent->ResetRefPos(0);

    iMonitor(_index_monitor, _index_handle, &_index);
  }

  // Action: GO_END
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, END))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_END)
#endif
  {
    _extindex = 0;
    _index = 0;
    _boxpos = 0;
    if (_parent)
      _parent->ResetRefPos(1);
  
    iMonitor(_index_monitor, _index_handle, &_index);
  }

  // Action: DELETE
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, DELETE))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_DC)
#endif
  {
    ; // new
  }
}

/****************************************************************************/
// PURPOSE:
//   Method for putting standard ASCII keyboard keys to the console output.
//
// PRE:
//   int key : the key to be outputted.
//
// POST:
//   The key is outputted to the screen. Appropriate changes to the
//   internal input buffer is made.
//
void WindowDummyLine::PutStdKey(int key)
{
  bool NextLineReady_ = FALSE;
  bool InWin_ = FALSE;
  int pos;
  int index_ = 0;
  int endpt_ = 0;
  int oldkey_ = 0;
  int EndReached_ = 0;  
  int NonWsLen_ = 0;

  // Action: NEWLINE
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  if (_InsertMode && GetKeyMap().KeyMatch(key, ENTER))
#elif (defined(__linux__) | defined(__unix__))
  if (_InsertMode && key == 10)
#endif
  {
      ShiftDown(NULL);

      pos = wherex();
      ShowInput();
      GOTOXY(pos, _ypos);
  }

  // Action: IF_GO_LEFT_DELETE
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, BACK) && IsEnterable())
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_BACKSPACE && IsEnterable())
#endif
  {
    if (_index > 0)
    {
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
      PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::LEFTARROW).SCAN + 256);
      PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::DELETE).SCAN + 256);
#elif (defined(__linux__) | defined(__unix__))
      PutExtKey(KEY_LEFT);
      PutExtKey(KEY_DC);
#endif
      iMonitor(_index_monitor, _index_handle, &_index);
    }
  }

  // Action: PUTKEY
  else if (IsUserKey(key))
  {
    if (_index < _len)
    {
      if (_InsertMode)
        Push(_index, key);

      PUTCH(key);
      pos = wherex();
      ShowInput();
      GOTOXY(pos, _ypos);
      ++_index;
      iMonitor(_index_monitor, _index_handle, &_index);
    }
  }
}

/****************************************************************************/
int WindowDummyLine::IsUserKey(int key)
{
  return
  (
    !_extended &&
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
    !GetKeyMap().KeyMatch(key, BACK) &&
    !GetKeyMap().KeyMatch(key, ESCAPE) &&
    !GetKeyMap().KeyMatch(key, ENTER) &&
    !GetKeyMap().KeyMatch(key, NEXT) &&
#elif (defined(__linux__) | defined(__unix__))
    key != KEY_BACKSPACE &&
    key != 27 &&
    key != 10 &&
    key != 9 &&
#endif
    isprint(key)
  );
}

/****************************************************************************/
void WindowDummyLine::ShiftDown(char* NewStr_)
{
  // if (_nextline)
  //   _nextline->ShiftDown(_buffer);
  SetInputString(NewStr_);
}

/****************************************************************************/
void WindowDummyLine::FillString(char* Str_, char FillCh_, int Start_, int End_)
{
  for (;Start_ < End_; Start_++)
    Str_[Start_] = FillCh_;

  Str_[End_] = '\0';
}

/****************************************************************************/
/****************************************************************************/
// PURPOSE:
//   End input test procedure
//
// PRE:
//   int key : the key to be tested as the termination key for input.
//
// POST:
//   Returns 1 if the key is defined as a termination key,
//   otherwise returns 0.
//
int WindowDummyLine::ShouldEnd(int key, int zerotozero)
{
  int status_;

  // Action: GO_DOWN, ESCAPE
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  if (GetKeyMap().KeyMatch(key, ESCAPE))	
#elif (defined(__linux__) | defined(__unix__))
  if (key == 27)
#endif
  {
    _Escaped = TRUE;
    iMonitor(_escaped_monitor, _escaped_handle, &_Escaped);
    status_ = 1;
  }

  // Action: GO_DOWN, IF_NOTCHAINED_ESCAPE
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, ENTER))	
#elif (defined(__linux__) | defined(__unix__))
  else if (key == 10)
#endif  
  {
    _Escaped = TRUE;
    iMonitor(_escaped_monitor, _escaped_handle, &_Escaped);
    status_ = 1;
  }

  // Action: GO_DOWN, GO_NEXT
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, NEXT) || GetKeyMap().KeyMatch(key, DOWN))	
#elif (defined(__linux__) | defined(__unix__))
  else if (key == 9 || key == KEY_DOWN)
#endif    
    status_ = 1;

  // Action: GO_UP
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, UP))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_UP)
#endif
    status_ = -1;

  // Action: GO_BACK, GO_LEFT (Note Plural Actions For Single Key)
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, BACK) || GetKeyMap().KeyMatch(key, LEFT))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_BACKSPACE || key == KEY_LEFT)
#endif
    status_ = (_index <= 0 && zerotozero) ? -1:0;

  // Action: GO_RIGHT (Note Plural Actions For Single Key)
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
  else if (GetKeyMap().KeyMatch(key, RIGHT) || IsUserKey(key))
#elif (defined(__linux__) | defined(__unix__))
  else if (key == KEY_RIGHT || IsUserKey(key))
#endif
    status_ = (_index >= _len) ? 1:0;

  // Otherwise, no action
  else
    status_ = 0;

  return status_;
}

/****************************************************************************/
// PURPOSE:
//   Get line input procedure
//
// PRE:
//   The original input buffer
//
// POST:
//   Gets input from the keyboard and place it in the input buffer until
//   a termination is press.
//
int WindowDummyLine::GetLine(int status_)
{
  int lastindex_;
  int zerotozero_;

  ResetInputState(status_);

  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
  iMonitor(_active_monitor, _active_handle, &_TextWidgetActive);

  for (;;)
  {
    GetKey();
    lastindex_ = _index;

    if (_extended)
      PutExtKey(_extkey);
    else
      PutStdKey(_extkey);

    zerotozero_ = !lastindex_ && !_index;

    if (status_ = ShouldEnd(_extkey, zerotozero_))
      break;
  }

  return status_;
}

/****************************************************************************/
void WindowDummyLine::ResetInputState(int status_)
{
  _Escaped = FALSE;
  // _index = (status_ == -1) ? 1:0;
  _index = 0;

  iMonitor(_escaped_monitor, _escaped_handle, &_Escaped);
  iMonitor(_index_monitor, _index_handle, &_index);
}

/****************************************************************************/
// PURPOSE:
//   Read input procedure
//
// POST:
//   Calls procedure to Show the contents of the input buffer as default
//   input to the Prompt box then calls GetLine to get additional input.
//   from the keyboard.
//
int WindowDummyLine::ReadInput(int status_)
{
  ShowInput(status_);
  return GetLine(status_);
}

/****************************************************************************/
// PURPOSE:
//   Method to Erase the input
//
// POST:
//   Erases the internal buffer by NULL padding entire buffer.
//
void WindowDummyLine::Erase()
{
  if (!IsActive())
    return;

  if (!_Hidden)
  {
    _InFocus = TRUE;
    iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

    if (UsesColor())
      SetupColors(ColorInfo::BACKGROUND_COLOR);
    else
      UseColor(FALSE);
    
    ResetPrompt();
    _InFocus = FALSE;
    iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
  }
}

/****************************************************************************/
// PURPOSE:
//   reset user Prompt method
//
// POST:
//   The user Prompt is setup at the specified position and the original
//   contents of the line is Cleared
//
void WindowDummyLine::ResetPrompt()
{
  if (!IsActive())
    return;

  GOTOXY(_xpos, _ypos);
  PrintBlanks(_len);
  GOTOXY(_xpos, _ypos);
}

/****************************************************************************/
// PURPOSE:
//   Show Prompt message method
//
// POST:
//   Goes to the specified coordinates and Shows the Prompt message
//   to the screen
//
void WindowDummyLine::ShowMessage()
{
  GOTOXY(_xpos, _ypos);
}

/****************************************************************************/
// PURPOSE:
//   Show input method
//
// POST:
//   Shows the Prompt box and Shows the original contents of the input
//   buffer within the box
//
void WindowDummyLine::ShowInput(int status_)
{
  if (!IsActive())
    return;

  if (_len)
  {
    ResetPrompt();

    GOTOXY(_xpos, _ypos);

    if (status_ == -1)
      GOTOXY(wherex(), _ypos);
  }
}

/****************************************************************************/
// PURPOSE:
//   Prompt user method
//
// POST:
//   Shows the Prompt message and the Prompt box then Prompts user for
//   additional input
//
int WindowDummyLine::Prompt(int status_)
{
  if (!IsActive())
    return status_;

  if (_len)
  {
    _InFocus = TRUE;
    iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

    SetupColors(ColorInfo::HIGHLIGHT_COLORS);
    ShowMessage();
    SetupColors(ColorInfo::NORMAL_COLORS);
    status_ = ReadInput(status_);
    Show();

    _InFocus = FALSE;
    iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
    _Hidden = FALSE;
    iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
  }

  return status_;
}

/****************************************************************************/
// PURPOSE:
//   Shows Prompt box method
//
// POST:
//   Shows Prompt message and Prompt box with contents of input buffer
//
void WindowDummyLine::Show()
{
  if (!IsActive())
    return;

  _InFocus = TRUE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

  SetupColors(ColorInfo::NORMAL_COLORS);
  ShowMessage();
  ShowInput(1);

  _InFocus = FALSE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
  _Hidden = FALSE;
  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
}

/****************************************************************************/
void WindowDummyLine::ShowInput()
{
  if (!IsActive())
    return;

  _InFocus = TRUE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

  ShowInput(1);

  _InFocus = FALSE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
  _Hidden = FALSE;
  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
}

/****************************************************************************/
// PURPOSE:
//   Hide Prompt box method
//
// POST:
//   goes to the specified coordinates of the Prompt box then Clears the
//   line containing the Prompt box.
//
void WindowDummyLine::Hide()
{
  if (!IsActive())
    return;

  _InFocus = TRUE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

  GOTOXY(_xpos, _ypos);


  if (UsesColor())
    SetupColors(ColorInfo::BACKGROUND_COLOR);
  else
    UseColor(FALSE);

  PrintBlanks(_len);
  GOTOXY(_xpos, _ypos);

  _InFocus = FALSE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
  _Hidden = TRUE;
  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
}

/****************************************************************************/
int WindowDummyLine::NextLengthHelper() const
{
  return _inboxoffset;
}

/****************************************************************************/
int WindowDummyLine::NextLength() const
{
  return _inboxoffset;
}

/****************************************************************************/
int WindowDummyLine::TakeNextLastCharHelper()
{
  return 0;
}

/****************************************************************************/
int WindowDummyLine::TakeNextLastChar()
{
  return 0;
}

/****************************************************************************/
int WindowDummyLine::VirtualBufferOffset(int* EndReached_) const
{
  return 0;
}

/****************************************************************************/
int WindowDummyLine::VirtualBufferMax() const
{
  return 0;
}

/****************************************************************************/
int WindowDummyLine::IsChainable(bool HasChars_) const
{
  return true;
}

/****************************************************************************/
int WindowDummyLine::IsDummyLine() const
{
  return true;
}

/****************************************************************************/
int WindowDummyLine::IsWindowLineBuffer() const
{
  return false;
}

/****************************************************************************/
int WindowDummyLine::ChainPosition() const
{
  return -99;
}

/****************************************************************************/
int WindowDummyLine::BufferChainPosition() const
{
  return -99;
}

/****************************************************************************/
int WindowDummyLine::IsFullBox() const
{
  return false;
}

/****************************************************************************/
int WindowDummyLine::RestrictFullBox() const
{
  return false;
}

/****************************************************************************/
bool WindowDummyLine::EndBrksActive() const
{
  return false;
}

/****************************************************************************/
bool WindowDummyLine::ShouldTrimWs() const
{
  return false;
}

/****************************************************************************/
int WindowDummyLine::BufferIndex() const
{
  return _index;
}

/****************************************************************************/
int WindowDummyLine::InBoxOffset() const
{
  return _inboxoffset;
}

/****************************************************************************/
int WindowDummyLine::InWBoxCount() const
{
  return _inwboxcnt;
}

/****************************************************************************/
void WindowDummyLine::SetEnterable(Boolean Flag_)
{}

/****************************************************************************/
void WindowDummyLine::SetSelectable(Boolean Flag_)
{
  _Selectable = Flag_;
}

/****************************************************************************/
int WindowDummyLine::GetHotKey() const
{
  return 0;
}

/****************************************************************************/
int WindowDummyLine::GetShortCut() const
{
  return 0;
}

/****************************************************************************/
int WindowDummyLine::GetHotKeyType() const
{
  return 0;
}

/****************************************************************************/
void WindowDummyLine::Select()
{}

/****************************************************************************/
void WindowDummyLine::Deselect()
{}

/****************************************************************************/
void WindowDummyLine::Activate()
{
  _TextWidgetActive = TRUE;
  iMonitor(_active_monitor, _active_handle, &_TextWidgetActive);
}

/****************************************************************************/
void WindowDummyLine::Deactivate()
{
  _TextWidgetActive = FALSE;
  iMonitor(_active_monitor, _active_handle, &_TextWidgetActive);
}

/****************************************************************************/
void WindowDummyLine::DecrBoxIndex()
{
  if (_inboxoffset - 1 >= 0)
    _inboxoffset--;

  if (_parent)
  {
    _bufferoffset = _parent->_index;
    _inboxoffset = _parent->_inboxoffset;
    _inwboxcnt = _parent->_inwboxcnt;  
  }
}

/****************************************************************************/
void WindowDummyLine::IncrBoxIndex()
{
  if (_inboxoffset + 1 <= _maxbuf || _maxbuf == -1)
    _inboxoffset++;

  if (_parent)
  {
    _bufferoffset = _parent->_index;
    _inboxoffset = _parent->_inboxoffset;
    _inwboxcnt = _parent->_inwboxcnt;
  }
}

/****************************************************************************/
void WindowDummyLine::ResetBoxIndex()
{
  _inboxoffset = 0;

  if (_parent)
  {
    _bufferoffset = _parent->_index;
    _inboxoffset = _parent->_inboxoffset;
    _inwboxcnt = _parent->_inwboxcnt;
  }  
}

/****************************************************************************/
void WindowDummyLine::SetBoxIndex(int Val_)
{
  if (0 <= Val_ && Val_ < _maxbuf || (Val_ >= 0 && _maxbuf == -1))
    _inboxoffset = Val_;

  if (_parent)
  {
    _bufferoffset = _parent->_index;
    _inboxoffset = _parent->_inboxoffset;
    _inwboxcnt = _parent->_inwboxcnt;
  }
}

/****************************************************************************/
Boolean WindowDummyLine::IsEmpty() const
{
  return TRUE;
}

/****************************************************************************/
Boolean WindowDummyLine::IsEscaped() const
{
  return _Escaped;
}

/****************************************************************************/
Boolean WindowDummyLine::IsActive() const
{
  return _TextWidgetActive;
}

/****************************************************************************/
Boolean WindowDummyLine::IsEnterable() const
{
  return _Enterable;
}

/****************************************************************************/
Boolean WindowDummyLine::IsSelectable() const
{
  return _Selectable;
}

/****************************************************************************/
Boolean WindowDummyLine::IsSelected() const
{
  return TRUE;
}

/****************************************************************************/
Boolean WindowDummyLine::IsValid() const
{
  return (!(IsEmpty() && _noempty) && IsValidLength());
}

/****************************************************************************/
Boolean WindowDummyLine::IsInFocus() const
{
  return _InFocus;
}

/****************************************************************************/
char WindowDummyLine::LastKeyRead(int* Ext_) const
{
  if (Ext_)
    *Ext_ = _extended;

  return _key;
}

/****************************************************************************/
int WindowDummyLine::LastExtKeyRead() const
{
  return _extkey;
}

/****************************************************************************/
char WindowDummyLine::ChrRetrieve(int Index_)
{
  return 0;
}

/****************************************************************************/
Boolean WindowDummyLine::IsOwner(ControlWindow* Pwin_) const
{
  return (_Owner == Pwin_);
}

/****************************************************************************/
void WindowDummyLine::SetOwner(ControlWindow* Pwin_, TextControl* TxtCtrl_)
{
  if (_Owner && _Owner != Pwin_)
    _Owner->RemoveControl(TxtCtrl_);
  _Owner = Pwin_;
}

/****************************************************************************/
void WindowDummyLine::SetBufferChainNum(WindowLineBufferType* Lbuf_, int Index_)
{
  if (Lbuf_)
    if (this != Lbuf_)
      Lbuf_->SetBufferChainNum(Lbuf_, Index_);
    else
      _BufChainNum = Index_;
}

/****************************************************************************/
void WindowDummyLine::SetReturnBuffer(WindowLineBufferType* Lbuf_,
                                      char* RetBufPtr_, bool Alloc_, int Size_)
{}

/****************************************************************************/
void WindowDummyLine::SetWindowLineNum(int Index_)
{
  _WinLineNum = Index_;
}

/****************************************************************************/
int WindowDummyLine::InputLength() const
{
  return _len;
}

/****************************************************************************/
void WindowDummyLine::SetupColors(int Mode_)
{
  if (!_ColorInfo.SetupColors(Mode_))
    GetDefaultColorInfo().SetupColors(Mode_);
}

/****************************************************************************/
void WindowDummyLine::UseColor(Boolean Flag_)
{
  if (_ColorInfo._UseColor)
    *_ColorInfo._UseColor = Flag_;
  else
    _ColorInfo._UseColor = new Boolean(Flag_);

  GetDefaultColorInfo().UseColor(Flag_);
}

/****************************************************************************/
void WindowDummyLine::SetNormalColors(int FgCol_, int BgCol_)
{
  if (_ColorInfo._NormalColors)
  {
    _ColorInfo._NormalColors[0] = FgCol_;
    _ColorInfo._NormalColors[1] = BgCol_;
  }
  else
  {
    _ColorInfo._NormalColors = new int[2];
    SetNormalColors(FgCol_, BgCol_);
  }
}

/****************************************************************************/
void WindowDummyLine::SetHighLightColors(int FgCol_, int BgCol_)
{
  if (_ColorInfo._HighLightColors)
  {
    _ColorInfo._HighLightColors[0] = FgCol_;
    _ColorInfo._HighLightColors[1] = BgCol_;
  }
  else
  {
    _ColorInfo._HighLightColors = new int[2];
    SetHighLightColors(FgCol_, BgCol_);
  }
}

/****************************************************************************/
void WindowDummyLine::SetHotKeyColor(int Col_)
{
  if (_ColorInfo._HkColor)
    *_ColorInfo._HkColor = Col_;
  else
    _ColorInfo._HkColor = new int(Col_);
}

/****************************************************************************/
void WindowDummyLine::SetBackGroundColor(int Col_)
{
  if (_ColorInfo._BackGroundColor)
    *_ColorInfo._BackGroundColor = Col_;
  else
    _ColorInfo._BackGroundColor = new int(Col_);
}

/****************************************************************************/
void WindowDummyLine::UseDefaultColors()
{
  _ColorInfo.ClearColors();
  UseColor();
}

/****************************************************************************/
Boolean WindowDummyLine::UsesColor() const
{
  return
  (
    _ColorInfo._UseColor ?
      *(_ColorInfo._UseColor):
      *(GetDefaultColorInfo()._UseColor)
  );
}

/****************************************************************************/
int* WindowDummyLine::NormalColors() const
{
  return
  (
    _ColorInfo._NormalColors ?
      _ColorInfo._NormalColors:
      GetDefaultColorInfo()._NormalColors
  );
}

/****************************************************************************/
int* WindowDummyLine::HighLightColors() const
{
  return
  (
    _ColorInfo._HighLightColors ?
      _ColorInfo._HighLightColors:
      GetDefaultColorInfo()._HighLightColors
  );
}

/****************************************************************************/
int WindowDummyLine::HotKeyColor() const
{
  return
  (
    _ColorInfo._HkColor ?
      *(_ColorInfo._HkColor):
      *(GetDefaultColorInfo()._HkColor)
  );
}

/****************************************************************************/
int WindowDummyLine::BackGroundColor() const
{
  return
  (
    _ColorInfo._BackGroundColor ?
      *(_ColorInfo._BackGroundColor):
      *(GetDefaultColorInfo()._BackGroundColor)
  );
}

/****************************************************************************/
// WindowLine class definition
/****************************************************************************/
// PURPOSE:
//   Prompt window default constructor
//
// PRE:
//   an uninitialize Prompt window
//
// POST:
//   a Prompt window is constructed and assigned the default values
//   specified.
//
WindowLine::WindowLine(int Max_, int WinLineNum_,
                       int WinStartx_, int WinStarty_,
                       int WinMaxx_, int WinMaxy_):
_pbox(NULL),
_Info(new ControlInfo(Max_)),
_nextline(NULL),
_RefPtr(NULL),
_DummyRef(NULL),

_InWinMode(true),
_LineTerminated(false),
_LineWrapAllowed(true),
_ReturnOnLastLine(false),
_ShowChainedBoxes(false),

_maxbox(Max_),
_boxcount(0),

_screenxpos(0),
_screenypos(0),
_screenmaxx(0),
_screenmaxy(0),

_winboxxpos(0),
_winboxypos(0),
_winboxstartx(WinStartx_),
_winboxstarty(WinStarty_),
_winboxmaxx(WinMaxx_),
_winboxmaxy(WinMaxy_),
    
_winviewxpos(0),
_winviewypos(0),
_winviewxoffset(0),
_winviewyoffset(0),

_inputlen(0),
_maxwraparound(0),
_wraparoundindex(0),
_WindowLineNum(WinLineNum_)
{
  TextGraphics::RetrieveScreenDimensions();
  _screenmaxx = TextGraphics::SCREENLEN;
  _screenmaxy = TextGraphics::SCREENHT;

  _DummyRef = new WindowLineRef(this, -1);
  MaxControl(Max_);
}

/****************************************************************************/
// PURPOSE:
//   Prompt window copy construcor
//
// PRE:
//   An uninitialized Prompt window.
//
// POST:
//   a Prompt window is constructed and assigned the corresponding values
//   in the Prompt window argument passed to the constructor.
//
WindowLine::WindowLine(WindowLine& pw):
_pbox(NULL),
_Info(new ControlInfo(pw._maxbox)),
_nextline(NULL),
_RefPtr(NULL),
_DummyRef(NULL),

_InWinMode(pw._InWinMode),
_LineTerminated(pw._LineTerminated),
_LineWrapAllowed(pw._LineWrapAllowed),
_ReturnOnLastLine(pw._ReturnOnLastLine),
_ShowChainedBoxes(pw._ShowChainedBoxes),

_maxbox(pw._maxbox),
_boxcount(pw._boxcount),

_screenxpos(0),
_screenypos(0),
_screenmaxx(0),
_screenmaxy(0),

_winboxxpos(0),
_winboxypos(0),
_winboxstartx(pw._winboxstartx),
_winboxstarty(pw._winboxstarty),
_winboxmaxx(pw._winboxmaxx),
_winboxmaxy(pw._winboxmaxy),

_winviewxpos(0),
_winviewypos(0),
_winviewxoffset(0),
_winviewyoffset(0),

_inputlen(pw._inputlen),
_maxwraparound(pw._maxwraparound),
_wraparoundindex(0),
_WindowLineNum(pw._WindowLineNum)
{
  TextGraphics::RetrieveScreenDimensions();
  _screenmaxx = TextGraphics::SCREENLEN;
  _screenmaxy = TextGraphics::SCREENHT;

  _pbox = new TextControl*[_maxbox];
  _RefPtr = new WindowLineRef*[_maxbox];
  _DummyRef = new WindowLineRef(this, -1);

  for (int index = 0; index < _maxbox; index++)
  {
    if (_RefPtr)
      _RefPtr[index] = NULL;

    if (pw[index] && _pbox)
    {
      _pbox[index] = WindowLineBuffer::Make(*pw[index]);
      SetControlData(_pbox, _Info, index);
    }
    else
      RemoveControlData(_pbox, _Info, index);
  }

  if (pw._nextline)
    _nextline = new WindowLine(*pw._nextline);
}

/****************************************************************************/
WindowLine::~WindowLine()
{
  Clear();
  delete _DummyRef;
  _DummyRef = NULL;
}

/****************************************************************************/
int WindowLine::FindMaxWrapAround()
{
  int ret = 0;
  int box_ = 0;
  int endpt_ = 0;
  int NonWsLen_ = 0;
  int BufChainPos_ = 0;

  bool DoChain_ = false;
  bool WrapCond_ = false;
  bool OverLimit_ = false;
  bool enterb_ = false;
  bool vbuf_ = false;
  bool inwbox_ = false;

  if (_pbox && _pbox[0])
  {
    BufChainPos_ = _pbox[0]->BufferChainPosition();
    
    enterb_ = _pbox[0]->IsEnterable();
    vbuf_ = BufChainPos_ == 0 && enterb_ &&
            _pbox[0]->VirtualBufferOffset();
    inwbox_ = BufChainPos_ == 0 && enterb_ &&
              LineWrapAllowed();
  }
  
  WindowLineBuffer* wlbptr_ = NULL;
  
  char* Str_ = NULL;

  if (_maxbox && vbuf_)
  {
    ret = _pbox[0]->VirtualBufferMax();

    if (ret && _pbox[0]->NextLength() > 0)
      ret = _pbox[0]->NextLength();
    else
      ret = 0;
      
    _maxwraparound = ret;
  }
  else if (_pbox && _pbox[0])
  {
    ret = FindControlCount();

    if (_boxcount > 1)
    {
      for (ret = _boxcount - 1; ret && _pbox[ret-1]->NextLength() <= 0; ret--);

      if (ret)
      {
        box_ = _pbox[ret-1]->NextLength();
        _maxwraparound = (ret-1) * _inputlen + box_;
        ret = _maxwraparound;
      }
      else
        ret = box_ = _maxwraparound = 0;

      wlbptr_ = (WindowLineBuffer*)(_pbox[0]);

      if (wlbptr_ && inwbox_)
      {
        Str_ = wlbptr_->StrRetrieve();
        endpt_ = wlbptr_->InputLength() - 1;
        DoChain_ = wlbptr_->IsChainable(true);
        
        WrapCond_ = (wlbptr_->BufferIndex() == ::SafeStrLen(Str_) - 1 ||
                     wlbptr_->BufferIndex() == endpt_) &&
                    DoChain_ && _wraparoundindex < _maxwraparound;

        OverLimit_ = wlbptr_->BufferIndex() <= ::SafeStrLen(Str_) &&
                     wlbptr_->BufferIndex() <= endpt_ &&
                     (!wlbptr_->InWBoxCount() ||
                       (wlbptr_->InWBoxCount() > 0 &&
                         (_wraparoundindex < _maxwraparound ||
                           (wlbptr_->InWBoxCount() >= _maxwraparound &&
                              _wraparoundindex <= wlbptr_->InWBoxCount()))));

        if (!WrapCond_ && OverLimit_ && box_)
        {
          NonWsLen_ = wlbptr_->HasNonNullChars(Str_);
        
          if (NonWsLen_ > 0 && NonWsLen_ < _inputlen)
          {
            _maxwraparound -= (_inputlen - NonWsLen_);
            ret = _maxwraparound;
          }
        }
      }
    }
    else
      ret = 0;
  }

  return ret;
}

/****************************************************************************/
int WindowLine::FindControlCount()
{
  int index;
  int cnt = 0;

  if (_pbox && _maxbox)
  {
    for (index = 0; index < _maxbox; index++)
      if (_pbox[index])
        ++cnt;

    _boxcount = cnt;
  }

  return cnt;
}

/****************************************************************************/
int WindowLine::GetWrapAroundIndex()
{
  return _wraparoundindex;
}

/****************************************************************************/
char* WindowLine::RetrieveInOrder()
{
  char* RetBuf_ = NULL;
  char* Str_ = NULL;
  
  int BufChainPos_ = 0;
  int BufIndex_ = 0;
  
  bool DoChain_ = false;
  bool enterb_ = false;
  bool vbuf_ = false;
  bool inwbox_ = false;

  if (_pbox && _pbox[0] &&
      !_pbox[0]->IsDummyLine() && _pbox[0]->IsWindowLineBuffer())
  {
    BufIndex_ = ((WindowLineBuffer*)_pbox[0])->BufferIndex();
    BufChainPos_ = _pbox[0]->BufferChainPosition();
    
    DoChain_ = _pbox[0]->IsChainable(false);
    enterb_ = _pbox[0]->IsEnterable();
    vbuf_ = BufChainPos_ == 0 && enterb_ &&
            _pbox[0]->VirtualBufferOffset();
    inwbox_ = BufChainPos_ == 0 && enterb_ &&
              LineWrapAllowed();

    while (BufIndex_ > 0 ||
           (GetWrapAroundIndex() > 0 && DoChain_ && (vbuf_ || inwbox_)))
    {
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
      ((WindowLineBuffer*)_pbox[0])->PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::LEFTARROW).SCAN + 256);
#elif (defined(__linux__) | defined(__unix__))
      ((WindowLineBuffer*)_pbox[0])->PutExtKey(KEY_LEFT);
#endif

      BufIndex_ = ((WindowLineBuffer*)_pbox[0])->BufferIndex();
      BufChainPos_ = _pbox[0]->BufferChainPosition();

      DoChain_ = _pbox[0]->IsChainable(false);
      enterb_ = _pbox[0]->IsEnterable();
      vbuf_ = BufChainPos_ == 0 && enterb_ &&
              _pbox[0]->VirtualBufferOffset();
      inwbox_ = BufChainPos_ == 0 && enterb_ &&
                LineWrapAllowed();
    }

    int x;
    int sz = ((WindowLineBuffer*)_pbox[0])->InputLength();
    sz += FindMaxWrapAround();

    int max = FindControlCount();
    RetBuf_ = sz ? (new char[sz + 1]):(new char[WindowLineBuffer::MAXBUFFER + 1]);
    RetBuf_[0] = 0;    

    for (x = 0; x < max; x++)
    {
      Str_ = ((WindowLineBuffer*)_pbox[x])->StrRetrieve();
      if (Str_)
        strcat(RetBuf_, Str_);
    }

    Hide();

    if (((WindowLineBuffer*)_pbox[0])->ShouldTrimWs() && RetBuf_)
      RetBuf_ = WindowLineBuffer::RemoveTrailSpaceChars(RetBuf_, false);
  }

  return RetBuf_;
}

/****************************************************************************/
WindowLine& WindowLine::SetWinBlockProperties(int InWin_,
                                              int RetOnLast_)
{
  if (InWin_ != -99)
    _InWinMode = InWin_;
  else
    _InWinMode = true;

  if (RetOnLast_ != -99)
    _ReturnOnLastLine = RetOnLast_;
  else
    _ReturnOnLastLine = !_InWinMode;

  return *this;
}

/****************************************************************************/
WindowLine& WindowLine::SetWinLineProperties(int LwAllowed_,
                                             int ShowChained_,
                                             int RestrictFull_,
                                             int TrimWs_,
                                             int EndBrksActive_)
{
  if (LwAllowed_ != -99)
    _LineWrapAllowed = LwAllowed_;
  else
    _LineWrapAllowed = true;
  
  if (ShowChained_ != -99)
    _ShowChainedBoxes = ShowChained_;
  else
    _ShowChainedBoxes = !_LineWrapAllowed;

  if (_pbox && _pbox[0] &&
      !_pbox[0]->IsDummyLine() && _pbox[0]->IsWindowLineBuffer())
  {
    if (RestrictFull_ != -99)
      ((WindowLineBuffer*)_pbox[0])->SetRestrictFullBox(RestrictFull_);
    else
      ((WindowLineBuffer*)_pbox[0])->SetRestrictFullBox(_LineWrapAllowed);

    if (TrimWs_ != -99)
      ((WindowLineBuffer*)_pbox[0])->SetTrimWs(TrimWs_);
    else
      ((WindowLineBuffer*)_pbox[0])->SetTrimWs(((WindowLineBuffer*)_pbox[0])->RestrictFullBox());

    if (EndBrksActive_ != -99)
      ((WindowLineBuffer*)_pbox[0])->SetEndBrksActive(EndBrksActive_);
    else
      ((WindowLineBuffer*)_pbox[0])->SetEndBrksActive(!_ShowChainedBoxes);
  }
    
  return *this;
}

/****************************************************************************/
void WindowLine::SendUnprocessedKey(int Key_, int Extended_,
                                    WindowLineBuffer* WinBufPtr_)
{
  if (!Extended_ && WinBufPtr_->GetKeyMap().KeyMatch(Key_, TextControl::ENTER))
  {
    if (!_InWinMode || (!_nextline && _ReturnOnLastLine))
      _LineTerminated = true;
  }
}

/****************************************************************************/
bool WindowLine::InWindowMode(bool* HaveNext_) const
{
  if (HaveNext_)
    *HaveNext_ = _nextline != NULL;

  return _InWinMode;
}

/****************************************************************************/
int WindowLine::SetInputLength(int Length_)
{
  if (0 < Length_)
    _inputlen = (Length_ > _winboxmaxx) ? _winboxmaxx:Length_;

  return _inputlen;
}

/****************************************************************************/
void WindowLine::IncrWrapAround(int Num_)
{
  if (_wraparoundindex <= _maxwraparound &&
      (0 < Num_ && Num_ <= (_maxwraparound-_wraparoundindex)+1))
  {
    _wraparoundindex += Num_;

    if (_wraparoundindex > _maxwraparound+1)
      _wraparoundindex = _maxwraparound+1;
  }

  if (_wraparoundindex < 0)
    _wraparoundindex = 0;
}

/****************************************************************************/
void WindowLine::DecrWrapAround(int Num_)
{
  if (_wraparoundindex > 0 &&
      (0 < Num_ && Num_ <= _wraparoundindex))
  {
    _wraparoundindex -= Num_;

    if (_wraparoundindex < 0)
      _wraparoundindex = 0;
  }

  if (_wraparoundindex < 0)
    _wraparoundindex = 0;
}

/****************************************************************************/
int WindowLine::WinBoxToScreenX(int xpos_)
{
  if (xpos_ < 0)
    xpos_ = _winboxxpos;

  xpos_ += _winboxstartx;

  if (xpos_ >= _winboxmaxx || xpos_ < 1)
    xpos_ = -1;

  return xpos_;
}

/****************************************************************************/
int WindowLine::WinBoxToScreenY(int ypos_)
{
  if (ypos_ < 0)
    ypos_ = _winboxypos;

  ypos_ += _winboxstarty;

  if (ypos_ >= _winboxmaxy || ypos_ < 1)
    ypos_ = -1;

  return ypos_;
}

/****************************************************************************/
int WindowLine::WinViewToWinBoxX(int xpos_)
{
  if (xpos_ < 0)
    xpos_ = _winviewxpos;

  xpos_ -= _winviewxoffset;

  if (xpos_ >= _winboxmaxx || xpos_ < 1)
    xpos_ = -1;

  return xpos_;
}

/****************************************************************************/
int WindowLine::WinViewToWinBoxY(int ypos_)
{
  if (ypos_ < 0)
    ypos_ = _winviewypos;

  ypos_ -= _winviewyoffset;

  if (ypos_ >= _winboxmaxy || ypos_ < 1)
    ypos_ = -1;

  return ypos_;
}

/****************************************************************************/
// Peform shift down of entered line to next WindowLine object
void WindowLine::ShiftDown(char* NewStr_, int Offset_, TextControl** Pbox_, int Maxbox_)
{
  // new
  if (_nextline)
  {
    _nextline->ShiftDown(NewStr_ + Offset_, 0, _pbox, _maxbox);
    
  }

  if (NewStr_)
    SetLineToString(NewStr_);
  else
  {
    _pbox = Pbox_;
    _maxbox = Maxbox_;

    FindControlCount();
  }
}

/****************************************************************************/
// Peform shift down of entered line to next WindowLine object
void WindowLine::ShiftDownNext(char* NewStr_, int Offset_, TextControl** Pbox_, int Maxbox_)
{
  // new
  if (_nextline)
    _nextline->ShiftDown(NewStr_, Offset_, _pbox, _maxbox);
}

/****************************************************************************/
void WindowLine::ShiftUp(char* NewStr_, int Offset_, TextControl** Pbox_, int Maxbox_)
{
  // new
}

/****************************************************************************/
void WindowLine::ShiftUpNext(char* NewStr_, int Offset_, TextControl** Pbox_, int Maxbox_)
{
  // new
}

/****************************************************************************/
void WindowLine::ChainFillString(char* Str_, int Offset_, char FillCh_, int Start_)
{
  // new
}

/****************************************************************************/
void WindowLine::SetLineToString(char* NewStr_)
{
  if (_pbox && _pbox[0])
  {
    _pbox[0]->Erase();
    ((WindowLineBuffer*)_pbox[0])->PushInputString(NewStr_);
  }
}

/****************************************************************************/
void WindowLine::RemoveControlData(TextControl** CtrlArray_,
                                   ControlInfo* CtrlInfo_, int index)
{
  CtrlArray_[index] = NULL;
  CtrlInfo_->SetHotKeys(index, NULL);
}

/****************************************************************************/
void WindowLine::SetControlData(TextControl** CtrlArray_,
                                ControlInfo* CtrlInfo_, int index)
{
  CtrlArray_[index]->SetOwner(this, CtrlArray_[index]);
  ((WindowLineBuffer*)CtrlArray_[index])->SetWindowLineNum(_WindowLineNum);
  int InputLen_ = ((WindowLineBuffer*)CtrlArray_[index])->InputLength();
  if (!_inputlen || (0 < InputLen_ && _inputlen < InputLen_))
    _inputlen = InputLen_;
  CtrlInfo_->SetHotKeys(index, CtrlArray_[index]);
}

/****************************************************************************/
// PURPOSE:
//   Prompt window data member destruction method
//
// PRE:
//   The data members of this Prompt window object with original values
//
// POST:
//   If the data members of the window object is not NULL then the data
//   members are deleted and assigned NULL. The array of Prompt boxes
//   are destroyed.
//
void WindowLine::Clear()
{
  if (_pbox)
  {
    Delete();

    delete[] _pbox;
    _pbox = NULL;
  }

  if (_RefPtr)
  {
    delete[] _RefPtr;
    _RefPtr = NULL;
  }

  if (_Info)
  {
    delete _Info;
    _Info = NULL;
  }
}

/****************************************************************************/
TextControl* WindowLine::Control(int Index_)
{
  return _pbox[Index_];
}

/****************************************************************************/
void WindowLine::Control(int Index_, TextControl* TxtCtrl_)
{
  _pbox[Index_] = TxtCtrl_;

  if (_pbox[Index_])
    SetControlData(_pbox, _Info, Index_); 
}

/****************************************************************************/
// PURPOSE:
//   Sets the maximum Prompt box limit on the box array
//
// PRE:
//   The previous size of the Prompt box array
//
// POST:
//   The size of the Prompt box array is set to the specified maximum
//   All contents of the array are Cleared first before resizing the array
//
void WindowLine::MaxControl(int max)
{
  if (_pbox)
  {
    Delete();

    delete[] _pbox;
    _pbox = NULL;
  }

  if (_RefPtr)
  {
    delete[] _RefPtr;
    _RefPtr = NULL;  
  }

  _maxbox = max;

  if (max)
  {
    _pbox = new TextControl*[_maxbox];
    _RefPtr = new WindowLineRef*[_maxbox];

    if (_pbox && _RefPtr)    
      for (int index = 0; index < _maxbox; index++)
      {
        _RefPtr[index] = NULL;
        RemoveControlData(_pbox, _Info, index);
      }
  }
}

/****************************************************************************/
int WindowLine::MaxControl()
{
  return _maxbox;
}

/****************************************************************************/
// PURPOSE:
//   Method to Grow the size of the array by some number passed as an
//   argument to the method. The array contents are preserved.
//
// PRE:
//   The old size of the Prompt box array and its contents
//
// POST:
//   The array size is Grown by the extra size given and the original
//   contents of the array are preserved in the new array.
//
void WindowLine::Grow(int extra)
{
  int index;
  int newlen = _maxbox + extra;

  WindowLineRef** newr = new WindowLineRef*[newlen];
  TextControl** newp = new TextControl*[newlen];
  ControlInfo* newi = new ControlInfo(newlen);

  for (index = 0; index < _maxbox; index++)
  {
    newr[index] = NULL;
    newp[index] = WindowLineBuffer::Make(*((WindowLineBuffer*)_pbox[index]));
    SetControlData(newp, newi, index);

    if (_RefPtr && _RefPtr[index])
    {
      delete _RefPtr[index];
      _RefPtr[index] = NULL;
    }
  }

  for (;index < newlen; index++)
  {
    newr[index] = NULL;
    RemoveControlData(newp, newi, index);
  }

  Clear();

  _maxbox = newlen;
  _pbox = newp;
  _Info = newi;
  _RefPtr = newr;
}

/****************************************************************************/
Boolean WindowLine::AnyActive()
{
  for (int Index_ = 0; Index_ < _maxbox; Index_++)
    if (_pbox[Index_] && _pbox[Index_]->IsActive())
      return TRUE;

  return FALSE;
}

/****************************************************************************/
Boolean WindowLine::ValidityCheck()
{
  for (int Index_ = 0; Index_ < _maxbox; Index_++)
    if (_pbox[Index_] && _pbox[Index_]->IsActive() &&
        !_pbox[Index_]->IsValid())
      return FALSE;

  return TRUE;
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and Prompt the user with
//   each one.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box element
//   within the array is used to Prompt the user.
//
int WindowLine::Prompt(int StartIndex_)
{
  int index = (StartIndex_ >= _maxbox) ? 0:
              (StartIndex_ < 0)        ? (_maxbox - 1):
               StartIndex_;
  int status_ = 1;

  if (AnyActive())
    while (!LineTerminated())
    {
      if (_pbox[index] && _pbox[index]->IsActive())
      {
        status_ = _pbox[index]->Prompt(status_);
        Boolean hkeypressed_ = _Info->HotKeyPressed();

        if (hkeypressed_)
        {
          if (status_ != 0)
            index = _Info->GetIndex();

          _Info->ResetHotKeys();
        }

        if (!hkeypressed_ || !status_)
	     {
	       if (_pbox[index]->IsEscaped())
	         if (ValidityCheck())
	           break;
            else
              status_ = 0;

	       index += status_;
        }
      }
      else
	     ++index;

      index = (index >= _maxbox) ? 0:
	      (index < 0) 	 ? (_maxbox - 1):
		   index;
    }

  return index;
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and Show to the screen
//   each one.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box element
//   within the array is Shown to the screen.
//
void WindowLine::Show()
{
  for (int index = 0; index < _maxbox; index++)
    if (_pbox[index] && _pbox[index]->IsActive())
      _pbox[index]->Show();
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and Clear each one from the
//   screen.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box element
//   within the array is Cleared from the screen.
//
void WindowLine::Hide()
{
  for (int index = 0; index < _maxbox; index++)
    if (_pbox[index] && _pbox[index]->IsActive())
      _pbox[index]->Hide();
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and Erase each Prompt box.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box element
//   within the array is Erased.
//
void WindowLine::Erase()
{
  for (int index = 0; index < _maxbox; index++)
    if (_pbox[index] && _pbox[index]->IsActive())
      _pbox[index]->Erase();
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and delete each Prompt box.
//   This method does not depend on whether the text control is in an
//   active state or not. All controls regardless of their active state will
//   be deleted.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box deleted
//
void WindowLine::Delete()
{
  for (int index = 0; index < _maxbox; index++)
  {
    if (_pbox && _pbox[index] && _pbox[index]->IsOwner(this))
    {    
      delete _pbox[index];
      RemoveControlData(_pbox, _Info, index);
    }

    if (_RefPtr && _RefPtr[index])
    {
      delete _RefPtr[index];
      _RefPtr[index] = NULL;
    }
  }
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and remove each Prompt box.
//   This method does not depend on whether the text control is in an
//   active state or not. All controls regardless of their active state will
//   be removed.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box removed,
//   but not deleted or modified
//
void WindowLine::Remove()
{
  TextControl* TxtCtrl_;

  for (int index = 0; index < _maxbox; index++)
    if (_pbox[index] && _pbox[index]->IsOwner(this))
    {
      TxtCtrl_ = _pbox[index];
      TxtCtrl_->SetOwner(NULL, TxtCtrl_);
      _Info->SetHotKeys(index, NULL);
    }
}

/****************************************************************************/
// PURPOSE:
//   Prompt box Index operator. To index individual Prompt boxes from the
//   Prompt window given the position of the Prompt box within the array.
//
// PRE:
//   int index : the index position of the Prompt box in the array
//
// POST:
//   Go into the specified index position in the Prompt box array and
//   Retrieve its element.
//
WindowLineRef& WindowLine::operator [] (int index)
{
  ASSERT((index >= 0 && index < _maxbox), USER_Message_t::ERRMSG_INDEXRANGE);

  if (!_RefPtr)
    return *_DummyRef;
  
  if (!_RefPtr[index])
    _RefPtr[index] = new WindowLineRef(this, index);
    
  return *_RefPtr[index];
}

/****************************************************************************/
WindowLineBuffer* WindowLine::NextSelected(int& Start_)
{
  Boolean Found_ = FALSE;

  for (;Start_ < _maxbox; Start_++)
    if (Found_ = _pbox[Start_] != NULL && _pbox[Start_]->IsSelected())
      break;

  if (Found_)
  {
    ++Start_;
    return (*this)[Start_ - 1];
  }

  return NULL;
}

/****************************************************************************/
void WindowLine::DeleteControl(TextControl* TxtCtrl_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index] == TxtCtrl_ && _pbox[index]->IsOwner(this))
      break;

  if (index < _maxbox)
  {
    if (_pbox)
    {
      delete _pbox[index];
      RemoveControlData(_pbox, _Info, index);
    }

    if (_RefPtr)
    {
      delete _RefPtr[index];
      _RefPtr[index] = NULL;
    }
  }
}

/****************************************************************************/
void WindowLine::RemoveControl(TextControl* TxtCtrl_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index] == TxtCtrl_ && _pbox[index]->IsOwner(this))
      break;

  if (index < _maxbox)
    RemoveControlData(_pbox, _Info, index);
}

/****************************************************************************/
ControlInfo* WindowLine::GetControlInfo()
{
  return _Info;
}

/****************************************************************************/
void WindowLine::UseColor(Boolean Flag_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index])
      _pbox[index]->UseColor(Flag_);
}

/****************************************************************************/
void WindowLine::SetNormalColors(int FgCol_, int BgCol_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index])
      _pbox[index]->SetNormalColors(FgCol_, BgCol_);
}

/****************************************************************************/
void WindowLine::SetHighLightColors(int FgCol_, int BgCol_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index])
      _pbox[index]->SetHighLightColors(FgCol_, BgCol_);
}

/****************************************************************************/
void WindowLine::SetHotKeyColor(int Col_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index])
      _pbox[index]->SetHotKeyColor(Col_);
}

/****************************************************************************/
void WindowLine::UseDefaultColors()
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index])
      _pbox[index]->UseDefaultColors();
}

/****************************************************************************/
/****************************************************************************/
WindowLineRef::WindowLineRef(WindowLine* Boss_, int Index_):
_BossPtr(Boss_),
_Index(Index_),
_Cleared(false),
_InProcess(false)
{}

/****************************************************************************/
WindowLineRef& WindowLineRef::operator = (WindowLineBuffer* Pbox_)
{
  if (_Index >= 0)
  {
    _Cleared = false;
    _InProcess = true;
    _BossPtr->Control(_Index, Pbox_);

    _InProcess = false;
    _Cleared = true;
  }
  
  return *this;
}

/****************************************************************************/
WindowLineRef::operator WindowLineBuffer* ()
{
  WindowLineBuffer* ret = NULL;

  if (_Index >= 0)
  {
    _Cleared = false;
    _InProcess = true;
    ret = WindowLineRef::CastToWindowLineBuffer(_BossPtr->Control(_Index));

    _InProcess = false;
    _Cleared = true;
  }

  ASSERT(_Index >= 0, USER_Message_t::ERRMSG_NULLPTR);
  return ret;
}

/****************************************************************************/
WindowLineBuffer* WindowLineRef::operator -> ()
{
  WindowLineBuffer* ret = NULL;

  if (_Index >= 0)
  {
    _Cleared = false;
    _InProcess = true;
    ret = WindowLineRef::CastToWindowLineBuffer(_BossPtr->Control(_Index));

    _InProcess = false;
    _Cleared = true;
  }

  ASSERT(ret != NULL, USER_Message_t::ERRMSG_NULLPTR);
  return ret;
}

/****************************************************************************/
/****************************************************************************/
#if WINLINE_DEBUG
int main()
{
  CLRSCR();

  WindowLineBuffer* pb;
  WindowLineBuffer* npb;
  WindowLine pw(1, 1, 5, 1, 75, 6);

  #if WINLINE_DEBUG1A
    pw.SetWinLineProperties(true);
    pw.SetWinBlockProperties(true, true);
  #else
    pw.SetWinLineProperties(false);
    pw.SetWinBlockProperties(false, true);
  #endif

  #if WINLINE_DEBUG1
    pb->GetKeyMap()[TextControl::UP].Add(KeyboardCodes::CODETABLE(KeyboardCodes::F1).SCAN, TRUE);
    pb->GetKeyMap()[TextControl::LEFT].Add(KeyboardCodes::CODETABLE(KeyboardCodes::F2).SCAN, TRUE);
    pb->GetKeyMap()[TextControl::RIGHT].Add(KeyboardCodes::CODETABLE(KeyboardCodes::F3).SCAN, TRUE);
    pb->GetKeyMap()[TextControl::DOWN].Add(KeyboardCodes::CODETABLE(KeyboardCodes::F4).SCAN, TRUE);

    pb = WindowLineBuffer::Make(NULL, NULL, 5);  // 20
    pb->SetInputLength(5, false);  // 20
    pb->SetXpos(2);
    pb->SetYpos(2);

    pw.SetWindowLineNum(1);
    pw.SetInputLength(5);  // 20    
    pw[0] = pb;
    pw.Grow(3);

    npb = WindowLineBuffer::MakeNextLine(pb);
    pb = npb;
    pw[1] = pb;

    npb = WindowLineBuffer::MakeNextLine(pb);
    pb = npb;
    pw[2] = pb;

    npb = WindowLineBuffer::MakeNextLine(pb);
    pb = npb;
    pw[3] = pb;

    // chaining box0 to box1 to box 2 to box 3
    ((WindowLineBuffer*)pw[0])->Chain(pw[1]);
    ((WindowLineBuffer*)pw[1])->Chain(pw[2]);
    ((WindowLineBuffer*)pw[2])->Chain(pw[3]);

    int index;
    pw.UseColor();
    pw.Show();
    pw.Prompt();
    pw.UseColor(FALSE);
    pw.Hide();

    GOTOXY(1, 5);
    for (index = 0; index < pw.MaxControl(); index++)
    {
      pb = pw[index];
      cout <<pb->StrRetrieve() <<endl;
      GETCH();
    }

    char* retstr_ = pw.RetrieveInOrder();
    GOTOXY(1, 10);
    if (retstr_)
    {
      cout <<endl <<"In Order: " <<retstr_ <<endl;
      delete retstr_;
    }    
  #endif

  #if WINLINE_DEBUG2
    #if WINLINE_DEBUG2A
      pb = WindowLineBuffer::Make(NULL, NULL, 40);  // 40
      
      if (pb)
      {
        pb->SetEndBrksActive(true);
        pb->SetInputLength(10, true);  // 40
      }
    #else
      pb = WindowLineBuffer::Make(NULL, NULL, 20);  // 20
      
      if (pb)
      {
        pb->SetEndBrksActive(true);
        pb->SetInputLength(5, true);   // 20
      }
    #endif

    pb->SetTrimWs(false);
    pb->SetXpos(2);
    pb->SetYpos(2);

    pw.SetWindowLineNum(1);
    #if WINLINE_DEBUG2A
      pw.SetInputLength(10);  // 40
    #else
      pw.SetInputLength(5);   // 20
    #endif
    pw[0] = pb;

    int index;
    pw.UseColor();
    pw.Show();
    pw.Prompt();
    pw.UseColor(FALSE);
    pw.Hide();

    GOTOXY(1, 5);
    for (index = 0; index < pw.MaxControl(); index++)
    {
      pb = pw[index];
      cout <<pb->StrRetrieve() <<endl;
      GETCH();
    }

    GETCH();
  #endif
}

#endif
/****************************************************************************/
#endif
