// Window line buffer ADT classes
// The window line buffer class define methods for maintaining a window line
// object as parts used in a window screen object in a text console. The
// window line buffer class also has methods for positioning procedures for
// setting the display position within the window screen, setting the default
// message and default input into the window line, showing, hiding, erasing
// and retrieval of the input data.
//
// The window line buffer class define methods for maintaining an array of
// window line buffers. Methods are defined to show, hide, erase and
// retrieve from the entire array of window line buffers assigned to the
// window screen object.
//
#ifndef ovWINLINE_H
#define ovWINLINE_H
/****************************************************************************/
#ifndef CTRLDEFS_H
  #include "ctrldefs.h"
#endif
#ifndef TEXTGRAPHICS_H
  #include "txtgraph.h"
#endif
#if ((defined(__linux__) | defined(__unix__)) & (!(defined(__DJGPP__))))
  #include <ncurses/form.h>
#endif

#define WINLINE_DEBUG      1
#define WINLINE_DEBUG1     0  // multi-line window buffers
#define WINLINE_DEBUG1A    0  // test line wrapping and in-window properties
#define WINLINE_DEBUG2     1  // single line window buffer (extended)
#define WINLINE_DEBUG2A    1  // single line window buffer at length 15


// WinLineBuffer operations:
//
// [winlinebuffer x] <-- [winlinebuffer] ... [winlinebuffer] <-- from start.
//
// pull characters from adjacent buffers and end buffer will pull characters
// from start buffer until max. wraparound length is reached which is:
//
//   ((total buffers) - 1) x Input Length
//
//   Input length is the same as buffer length or _maxbuf
//
// WinLineBuffer objects will be placed relative to the windows bounded box
// as specified by WindowsLine::winboxmaxx and WindowsLine::winboxmaxy
//
// Any winlinebuffer objects with size and starting positions surpassing
// the winbox dimensions will be hidden and not shown.
//
// The WinLineBuffer InFocusMonitor is disabled for this class's purpose
//
// The right arrow key would perform the chain wraparound as described above
// until the wraparound index reaches the wraparound length limit.
//
// The left arrow key would reverse the chain wraparound until the
// wraparound index reaches zero.
//
/****************************************************************************/
class ControlWindow;
class WindowDummyLine;
class WindowLineBuffer;

class ChainPosInfo
{
  protected:
    void CopyInfoToThis(WindowLineBuffer* Ptr_);
    void CopyToPosInfo(WindowLineBuffer* Ptr_);

  public:
    bool _enterb;
    bool _vbuf;
    bool _inwbox;
    bool _chainable;
    bool _error;
    int _vmax;
    int _vendpt;

    int _WrapMax;
    int _WrapIndex;

    int _reallen;
    int _copylen;
    int _index;
    int _startpt;
    int _endpt;
    int _boxpt;

    int _ChainPos;
    WindowLineBuffer* _BufInfo;
    ChainPosInfo* _StartInfo;
    ChainPosInfo* _EndInfo;

    ChainPosInfo();
    ~ChainPosInfo();

    void SetupInfoPtrs(WindowLineBuffer* Ptr_, int SeqPos_);
    void SetPushInfo(int start);
    void SetPullInfo(int start);

    void RestorePrevInfo();
    void ReturnToThis();
};

class WindowLineBufferType : public TextControl
{
  public:
    virtual void SetReturnBuffer(WindowLineBufferType* Lbuf_,
                                 char* RetBufPtr_, bool Alloc_, int Size_=0) = 0;
    virtual void SetBufferChainNum(WindowLineBufferType* Lbuf_, int Index_) = 0;
    virtual void SetWindowLineNum(int Index_) = 0;
    virtual int InputLength() const = 0;

    virtual int IsFullBox() const = 0;
    virtual int RestrictFullBox() const = 0;
    virtual bool EndBrksActive() const = 0;
    virtual bool ShouldTrimWs() const = 0;
    virtual int BufferIndex() const = 0;
    virtual int InBoxOffset() const = 0;
    virtual int InWBoxCount() const = 0;
};

class WindowLineBuffer : public WindowLineBufferType
{
  friend const int* InsertDetect(void* Owner_, WindowLineBuffer*, const int* Ptr_);
  friend class WindowDummyLine;
  friend class WindowLine;
  friend class ChainPosInfo;

  protected:
    enum { BLANK = ' ', MAXBUFFER = 128 };
#if (defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__))
    enum { CONTMORE_RIGHT = 0x10,
           CONTMORE_LEFT = 0x11 };
    enum { CONTMORE_HFG = LIGHTGREEN,
           CONTMORE_HBG = BLUE };
#elif (defined(__linux__) | defined(__unix__))
    enum { CONTMORE_RIGHT = '>',
           CONTMORE_LEFT = '<' };
    enum { CONTMORE_HFG = PINK,
           CONTMORE_HBG = GREEN };
#endif

    ColorInfo _ColorInfo;
    ControlWindow* _Owner;
    WindowLineBuffer* _nextline;
    WindowDummyLine* _refline;
    ChainPosInfo* _ChainPosPtr;

    char _brackets[2];
    Boolean _Enterable;    // Control could be entered into
    Boolean _Selectable;   // Indicate whether control can be
                           // selected like a button or checkbox
    char* _msg;				// prompt message buffer
    char* _buffer;			// input buffer
    char* _retbuf;         // return buffer
    bool _retbufalloc;     // memory allocated to return buffer
    int _bufferoffset;     // virtual buffer memory offset
    int _inboxoffset;      // sliding inputbox offset into virtual buffer
    int _inwboxcnt;        // wraparound boxex wrap count
    bool _MovedToEnd;      // cursor moved to end of text box chain
    int _EndCharInserted;  // latest end point char. inserted into text box
    bool _EndCharReverted; // end point char. reverted back into position

    int _maxbuf;			// maximum size of input buffer
					         // not including NULL terminator
    int _key;			   // input key
    int _extkey;			// extended input key
    int _extended;		// extended key indicator
    int _index;			// input buffer index
    int _len;				// maximum input length
    int _minlen;			// minimum input length
    int _chainminlen;	// remaindered min length for chained box
    int _xpos;				// horizontal coordinate position
    int _ypos;				// vertical coordinate position
    int _noempty;			// no input disallow flag
    int _InsertMode;		// insert mode
    int _Escaped;			// prompt box Escaped
    int _Hidden;			// Text dialog hidden flag
    int _TextWidgetActive;	// Active flag for this text widget
    int _ChainPos;			// The position within the chain
    int _InFocus;			   // Text widget in focus flag

    int _SavedCursX;			// Saved cursor x position
    int _SavedCursY;			// Saved cursor y position

    int _BufChainNum;      // Buffer chain number
    int _WinLineNum;       // Window line number
    bool _WrapChars;       // Wrap chars from next buffer
    
    bool _RestFullBox;     // Further input into full box is disallowed
    bool _EndBrksActive;   // Ending brackets active at left-right ends
    bool _TrimWs;          // Trim white space from input line (default: false)

    // Internal data monitoring function pointers
    const int*(*_insert_monitor)(void*, WindowLineBuffer*, const int*);    // internal use
    const char*(*_buffer_monitor)(void*, WindowLineBuffer*, const char*);
    const int*(*_key_monitor)(void*, WindowLineBuffer*, const int*);
    const int*(*_ext_monitor)(void*, WindowLineBuffer*, const int*);
    const int*(*_index_monitor)(void*, WindowLineBuffer*, const int*);
    const int*(*_escaped_monitor)(void*, WindowLineBuffer*, const int*);
    const int*(*_hidden_monitor)(void*, WindowLineBuffer*, const int*);
    const int*(*_active_monitor)(void*, WindowLineBuffer*, const int*);
    const int*(*_infocus_monitor)(void*, WindowLineBuffer*, const int*);
    const int*(*_inboxoffset_monitor)(void*, WindowLineBuffer*, const int*);
    const int*(*_winlinenum_monitor)(void*, WindowLineBuffer*, const int*);

    // Handles to external objects, used by data monitors
    void* _insert_handle;
    void* _buffer_handle;
    void* _key_handle;
    void* _ext_handle;
    void* _index_handle;
    void* _escaped_handle;
    void* _hidden_handle;
    void* _active_handle;
    void* _infocus_handle;
    void* _inboxoffset_handle;
    void* _winlinenum_handle;

    // Monitor functions execution and management methods
    const char* cMonitor(const char*(*monptr_)(void*, WindowLineBuffer*, const char*), void* This_, const char* ptr_);
    const int* iMonitor(const int*(*monptr_)(void*, WindowLineBuffer*, const int*), void* This_, const int* ptr_);
    void SaveCursorPosition();
    void RestoreCursorPosition();

    const int* MonitorInsertMode(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));

    // Promptbox chaining internal methods
    void SetChainPos(int Pos_);

    // Insert state methods
    void NotifyInsertMode(const int* ModePtr_);
    void SetInsertMode(int Mode_, WindowLineBuffer* Parent_=NULL);
    inline int GetInsertMode()
	{ return _InsertMode; }

    // show prompt box methods
    virtual void ShowMessage();
    virtual void ShowInput(int status_);
    virtual void ResetPrompt();

    // Clear prompt box method
    void PrintBlanks(int Max_);

    // input validation / testing methods
    int* WordChr(int* src_, int val_, size_t n);
    int IsValidString(char* str);
    int IsValidLength() const;
    int IsUserKey(int key);
    int ShouldEnd(int key, int zerotozero);
    int IsValidCtrlKey(int key);

    // console input retrieval methods
    int GetKey();
    inline int GetExtKey()
	{ return _extkey; }
    int GetLine(int status_);
    int ReadInput(int status_);
    void ResetInputState(int status_);
    void SetEnterable(Boolean Flag_);
    void SetSelectable(Boolean Flag_);
#if !(defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__)) & (defined(__linux__) | defined(__unix__))
    int TranslateKeys(int key);
#endif

    // keyboard character output methods
    void PutExtKey(int key);
    void PutStdKey(int key);

    // String push/pull methods
    bool IsEquivSpace(char ch) const;
    char* Convert(char* Str_);
    char* Revert(char* Str_);
    static char* RemoveTrailSpaceChars(char* Str_, bool UnReverted_);
    static bool InChSet(char ch, const char* chset, size_t* Index_=NULL);
    static void FillString(char* Str_, char FillCh_, int Start_, int End_);
    int HasNonNullChars(const char* str_) const;
    int HasNonWsChars(const char* str_) const;
    void ShiftDown(char* NewStr_);
    int NextLengthHelper() const;
    int TakeNextLastCharHelper();
    void ChainPush(char ch, ChainPosInfo* Info_);
    void ChainJoin(char ch);
    char ChainPull(ChainPosInfo* Info_);
    bool Push(int start, char ch, ChainPosInfo* Info_);
    void Join(char ch);
    char Pull(int start, ChainPosInfo* Info_);
    char Peek(int Index_=0);
    void ResetRefPos(int MoveToPos_);

    // data member destruction method
    virtual void Clear();

    // promptbox constructors    
    WindowLineBuffer(char* msg, WindowLineBuffer* next_, int Size_, int Length_=0);
    WindowLineBuffer(WindowLineBuffer& pb, int Size_, int Length_, int Offset_);
    WindowLineBuffer(WindowLineBuffer& pb, int Size_, int Length_=0);

    WindowLineBuffer* MakeVirtualLine(WindowLineBuffer* wpb);

    // Text color methods
    void SetupColors(int Mode_);
    void SetupColors(int Fg_, int Bg_);

  public:
    // WindowLineBuffer Destructor
    virtual ~WindowLineBuffer()
	{ Clear(); }

    // Promptbox chaining method
    WindowLineBuffer* Chain(WindowLineBuffer* next_);    
    inline void EraseChain()
	{ _nextline = NULL; }
    inline WindowLineBuffer* NextChain()
	{ return _nextline; }

    virtual int IsFullBox() const;
    virtual int RestrictFullBox() const;
    virtual bool EndBrksActive() const;
    virtual bool ShouldTrimWs() const;
    virtual int BufferIndex() const;
    virtual int InBoxOffset() const;
    virtual int InWBoxCount() const;

    void SetEndBrksActive(bool Flag_);
    void SetRestrictFullBox(bool Flag_);
    void SetTrimWs(bool Flag_);
    int EndBrksLength(bool OnGoto_) const;

    // virtual memory buffer methods
    virtual int VirtualBufferOffset(int* EndReached_=NULL,
                                    int* NonWsLen_=NULL,
                                    int* NonNullLen_=NULL) const;
    virtual int VirtualBufferMax() const;
    virtual int IsChainable(bool HasChars_) const;
    virtual int NextLength() const;
    virtual int TakeNextLastChar();
    virtual int IsDummyLine() const;
    virtual int IsWindowLineBuffer() const;
    virtual int ChainPosition() const;
    virtual int BufferChainPosition() const;
    
    // promptbox assignment operator
    virtual WindowLineBuffer& operator = (WindowLineBuffer& pb);
    static WindowLineBuffer* Make(char* msg=NULL, WindowLineBuffer* next_=NULL,
                                  int Size_=0, int Length_=0);
    static WindowLineBuffer* Make(WindowLineBuffer& pbox, int Size_=0, int Length_=0);
    static WindowLineBuffer* MakeNextLine(WindowLineBuffer* wpb);

    // prompt box properties mutators
    virtual char* SetMessage(const char* str);
    virtual void RemoveMessage();
    virtual void BlankMessage();
    virtual void SetInputString(char* str);
    virtual void PushInputString(char* str);
    virtual void SetInputLength(int len, bool MakeVbuf_);
    virtual int InputLength() const;
    virtual void DivBufSizeAsInputLength(int div_, bool MakeVbuf_);
    virtual void SetMinimumLength(int len);
    void SetNoEmpty(int flag=1);
    void SetXYpos(int x, int y);
    inline void SetXpos(int x)
        { Hide(); _xpos = x; }
    inline void SetYpos(int y)
        { Hide(); _ypos = y; }
        
    // prompt box manipulators
    virtual void Show();
    virtual void ShowInput();
    virtual void Hide();
    virtual int Prompt(int status_);
    virtual void Erase();

    // Accelerator key methods
    virtual int GetHotKey() const;
    virtual int GetShortCut() const;
    virtual int GetHotKeyType() const;

    // text control selection/deselection mutators
    virtual void Select();
    virtual void Deselect();

    // Empty, valid and selected Text Control test
    virtual Boolean IsEnterable() const;
    virtual Boolean IsSelectable() const;
    virtual Boolean IsSelected() const;
    virtual Boolean IsEmpty() const;
    virtual Boolean IsEscaped() const;
    virtual Boolean IsValid() const;
    virtual Boolean IsActive() const;
    virtual Boolean IsInFocus() const;

    // inbox index methods
    virtual void DecrBoxIndex();
    virtual void IncrBoxIndex();
    virtual void ResetBoxIndex();
    virtual void SetBoxIndex(int Val_);

    // Activation/Deactivation for this text widget
    virtual void Activate();
    virtual void Deactivate();

    // input data retrieval method
    char LastKeyRead(int* Ext_=NULL) const;
    int LastExtKeyRead() const;
    char ChrRetrieve(int Index_);
    inline char* GetMessage()
	{ return _msg; }
    char* StrRetrieve();

    // Text control ownership methods
    virtual void SetOwner(ControlWindow* Pwin_, TextControl* TxtCtrl_);
    virtual Boolean IsOwner(ControlWindow* Pwin_) const;
    
    // Text color methods
    virtual void UseColor(Boolean Flag_=TRUE);
    virtual void SetNormalColors(int FgCol_, int BgCol_);
    virtual void SetHighLightColors(int FgCol_, int BgCol_);
    virtual void SetHotKeyColor(int Col_);
    virtual void SetBackGroundColor(int Col_);
    virtual void UseDefaultColors();

    virtual Boolean UsesColor() const;
    virtual int* NormalColors() const;
    virtual int* HighLightColors() const;    
    virtual int HotKeyColor() const;
    virtual int BackGroundColor() const;

    // WindowLineBufferType specific methods
    virtual void SetWindowLineNum(int Index_);
    virtual void SetBufferChainNum(WindowLineBufferType* Lbuf_, int Index_);
    virtual void SetReturnBuffer(WindowLineBufferType* Lbuf_,
                                 char* RetBufPtr_, bool Alloc_, int Size_=0);

    // Internal data monitoring methods
    const char* MonitorBuffer(void* This_, const char*(*FnPtr_)(void*, WindowLineBuffer*, const char*));
    const int* MonitorKey(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));
    const int* MonitorExtendedKey(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));
    const int* MonitorIndex(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));
    const int* MonitorEscapedStatus(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));
    const int* MonitorHiddenStatus(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));
    const int* MonitorActiveStatus(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));
    const int* MonitorInFocusStatus(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));
    const int* MonitorInboxOffset(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));
    const int* MonitorWinLineNum(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));
};

/****************************************************************************/
class WindowDummyLine : public WindowLineBufferType
{
  friend const int* InsertDetect(void* Owner_, WindowDummyLine*, const int* Ptr_);
  friend class WindowLine;
  friend class WindowLineBuffer;

  protected:
    enum { BLANK = ' ', MAXBUFFER = 128 };

    ColorInfo _ColorInfo;
    ControlWindow* _Owner;

    Boolean _Enterable;       // Control could be entered into
    Boolean _Selectable;      // Indicate whether control can be
                              // selected like a button or checkbox

    WindowLineBuffer* _parent;   // same as pbox[0]
    WindowLineBuffer* _nextline; // newly created chained line buffer to
                                 // be chained to pbox[0]
    bool _nextlinealloc;         // memory allocated for next line buffer

    int _bufferoffset;    // same as _parent->_index
    int _inboxoffset;     // same as _parent->_inboxoffset
    int _inwboxcnt;       // same as _parent->_inwboxcnt

    int _maxbuf;        // maximum size of input buffer
                        // not including NULL terminator
    int _len;           // maximum input length
    int _minlen;			// minimum input length
    int _index;			// input buffer index
    int _extindex;      // extra char. lengths tacked onto end of line
    int _boxpos;        // actual cursor position relative to _len in box

    int _key;			   // input key
    int _extkey;			// extended input key
    int _extended;		// extended key indicator
    int _xpos;				// horizontal coordinate position
    int _ypos;				// vertical coordinate position
    int _noempty;			// no input disallow flag
    int _InsertMode;		// insert mode
    int _Escaped;			// prompt box Escaped
    int _Hidden;			// Text dialog hidden flag
    int _TextWidgetActive;	// Active flag for this text widget
    int _InFocus;			   // Text widget in focus flag

    int _SavedCursX;			// Saved cursor x position
    int _SavedCursY;			// Saved cursor y position

    int _BufChainNum;      // Buffer chain number
    int _WinLineNum;       // Window line number

    // Internal data monitoring function pointers
    const int*(*_insert_monitor)(void*, WindowDummyLine*, const int*);    // internal use
    const int*(*_key_monitor)(void*, WindowDummyLine*, const int*);
    const int*(*_ext_monitor)(void*, WindowDummyLine*, const int*);
    const int*(*_index_monitor)(void*, WindowDummyLine*, const int*);
    const int*(*_escaped_monitor)(void*, WindowDummyLine*, const int*);
    const int*(*_hidden_monitor)(void*, WindowDummyLine*, const int*);
    const int*(*_active_monitor)(void*, WindowDummyLine*, const int*);
    const int*(*_infocus_monitor)(void*, WindowDummyLine*, const int*);
    const int*(*_inboxoffset_monitor)(void*, WindowLineBuffer*, const int*);
    const int*(*_winlinenum_monitor)(void*, WindowLineBuffer*, const int*);    

    // Handles to external objects, used by data monitors
    void* _insert_handle;
    void* _key_handle;
    void* _ext_handle;
    void* _index_handle;
    void* _escaped_handle;
    void* _hidden_handle;
    void* _active_handle;
    void* _infocus_handle;
    void* _inboxoffset_handle;
    void* _winlinenum_handle;

    // Monitor functions execution and management methods
    const char* cMonitor(const char*(*monptr_)(void*, WindowDummyLine*, const char*), void* This_, const char* ptr_);
    const int* iMonitor(const int*(*monptr_)(void*, WindowDummyLine*, const int*), void* This_, const int* ptr_);
    void SaveCursorPosition();
    void RestoreCursorPosition();

    const int* MonitorInsertMode(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*));

    // Promptbox chaining internal methods
    void SetChainPos(int Pos_);

    // Insert state methods
    void NotifyInsertMode(const int* ModePtr_);
    void SetInsertMode(int Mode_, WindowDummyLine* Parent_=NULL);
    inline int GetInsertMode()
	{ return _InsertMode; }

    // show prompt box methods
    virtual void ShowMessage();
    virtual void ShowInput(int status_);
    virtual void ResetPrompt();

    // Clear prompt box method
    void PrintBlanks(int Max_);

    // input validation / testing methods
    int* WordChr(int* src_, int val_, size_t n);
    inline int IsValidString(char* str)
        { return 1; }
    inline int IsValidLength() const
        { return 1; }
    int IsUserKey(int key);
    int ShouldEnd(int key, int zerotozero);
    int IsValidCtrlKey(int key);

    // console input retrieval methods
    int GetKey();
    inline int GetExtKey()
        { return _extkey; }
    int GetLine(int status_);
    int ReadInput(int status_);
    void ResetInputState(int status_);
    void SetEnterable(Boolean Flag_);
    void SetSelectable(Boolean Flag_);
#if !(defined(__TURBOC__) | defined(__BORLANDC__) | defined(__DJGPP__)) & (defined(__linux__) | defined(__unix__))
    int TranslateKeys(int key);
#endif

    // keyboard character output methods
    void PutExtKey(int key);
    void PutStdKey(int key);

    // String push/pull methods
    static void FillString(char* Str_, char FillCh_, int Start_, int End_);
    int HasNonWsChars(const char* str_) const;
    void ShiftDown(char* NewStr_);
    inline void ChainPush(char ch) {}
    inline void ChainJoin(char ch) {}
    inline char ChainPull()
        { return 0; }
    inline void Push(int start, char ch=' ') {}
    inline void Join(char ch) {}
    inline char Pull(int start)
        { return 0; }
    inline char Peek(int Index_=0)
        { return 0; }

    // data member destruction method
    virtual void Clear();

    // promptbox constructors
    WindowDummyLine(int BufChainNum_, WindowLineBuffer* Parent_, int Length_=0);
    WindowDummyLine(WindowDummyLine& pb, int Length_=0);

    // Text color methods
    void SetupColors(int Mode_);

  public:
    // WindowDummyLine Destructor
    virtual ~WindowDummyLine()
	{ Clear(); }

    // Promptbox chaining method
    WindowDummyLine* Chain(WindowDummyLine* next_);
    inline void EraseChain()
	{ return; }
    inline WindowDummyLine* NextChain()
	{ return NULL; }

    virtual int IsFullBox() const;
    virtual int RestrictFullBox() const;
    virtual bool EndBrksActive() const;
    virtual bool ShouldTrimWs() const;
    virtual int BufferIndex() const;
    virtual int InBoxOffset() const;
    virtual int InWBoxCount() const;

    // promptbox assignment operator
    virtual WindowDummyLine& operator = (WindowDummyLine& pb);
    static WindowDummyLine* Make(int BufChainNum_, WindowLineBuffer* Parent_, int Length_=0);
    static WindowDummyLine* Make(WindowDummyLine& pbox, int Length_=0);
    static WindowLineBuffer* MakeNextLine(WindowLineBuffer* wpb);

    // prompt box properties mutators
    virtual char* SetMessage(char* str);
    virtual void RemoveMessage();
    virtual void BlankMessage();
    virtual void SetInputString(char* str);
    virtual void PushInputString(char* str);
    virtual void SetInputLength(int len, bool MakeVbuf_);
    virtual int InputLength() const;
    virtual void DivBufSizeAsInputLength(int div_, bool MakeVbuf_);
    virtual void SetMinimumLength(int len);
    void SetNoEmpty(int flag=1);
    void SetXYpos(int x, int y);
    int NextLengthHelper() const;
    int TakeNextLastCharHelper();
    inline void SetXpos(int x)
        { Hide(); _xpos = x; }
    inline void SetYpos(int y)
        { Hide(); _ypos = y; }

    // virtual memory buffer methods
    virtual int VirtualBufferOffset(int* EndReached_=NULL) const;    
    virtual int VirtualBufferMax() const;
    virtual int IsChainable(bool HasChars_) const;
    virtual int NextLength() const;
    virtual int TakeNextLastChar();
    virtual int IsDummyLine() const;
    virtual int IsWindowLineBuffer() const;
    virtual int ChainPosition() const;
    virtual int BufferChainPosition() const;

    // prompt box manipulators
    virtual void Show();
    virtual void ShowInput();
    virtual void Hide();
    virtual int Prompt(int status_);
    virtual void Erase();

    // Accelerator key methods
    virtual int GetHotKey() const;
    virtual int GetShortCut() const;
    virtual int GetHotKeyType() const;

    // text control selection/deselection mutators
    virtual void Select();
    virtual void Deselect();

    // Empty, valid and selected Text Control test
    virtual Boolean IsEnterable() const;
    virtual Boolean IsSelectable() const;
    virtual Boolean IsSelected() const;
    virtual Boolean IsEmpty() const;
    virtual Boolean IsEscaped() const;
    virtual Boolean IsValid() const;
    virtual Boolean IsActive() const;
    virtual Boolean IsInFocus() const;

    // Activation/Deactivation for this text widget
    virtual void Activate();
    virtual void Deactivate();

    // inbox index methods
    virtual void DecrBoxIndex();
    virtual void IncrBoxIndex();
    virtual void ResetBoxIndex();
    virtual void SetBoxIndex(int Val_);

    // input data retrieval method
    char LastKeyRead(int* Ext_=NULL) const;
    int LastExtKeyRead() const;
    char ChrRetrieve(int Index_);
    inline char* GetMessage()
	{ return NULL; }
    inline char* StrRetrieve()
	{ return NULL; }

    // Text control ownership methods
    virtual void SetOwner(ControlWindow* Pwin_, TextControl* TxtCtrl_);
    virtual Boolean IsOwner(ControlWindow* Pwin_) const;

    // Text color methods
    virtual void UseColor(Boolean Flag_=TRUE);
    virtual void SetNormalColors(int FgCol_, int BgCol_);
    virtual void SetHighLightColors(int FgCol_, int BgCol_);
    virtual void SetHotKeyColor(int Col_);
    virtual void SetBackGroundColor(int Col_);
    virtual void UseDefaultColors();

    virtual Boolean UsesColor() const;
    virtual int* NormalColors() const;
    virtual int* HighLightColors() const;    
    virtual int HotKeyColor() const;
    virtual int BackGroundColor() const;

    // WindowLineBufferType specific methods
    virtual void SetWindowLineNum(int Index_);
    virtual void SetBufferChainNum(WindowLineBufferType* Lbuf_, int Index_);
    virtual void SetReturnBuffer(WindowLineBufferType* Lbuf_,
                                 char* RetBufPtr_, bool Alloc_, int Size_=0);

    // Internal data monitoring methods
    const char* MonitorBuffer(void* This_, const char*(*FnPtr_)(void*, WindowDummyLine*, const char*));
    const int* MonitorKey(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*));
    const int* MonitorExtendedKey(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*));
    const int* MonitorIndex(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*));
    const int* MonitorEscapedStatus(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*));
    const int* MonitorHiddenStatus(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*));
    const int* MonitorActiveStatus(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*));
    const int* MonitorInFocusStatus(void* This_, const int*(*FnPtr_)(void*, WindowDummyLine*, const int*));
    const int* MonitorInboxOffset(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));
    const int* MonitorWinLineNum(void* This_, const int*(*FnPtr_)(void*, WindowLineBuffer*, const int*));
};

/****************************************************************************/
/****************************************************************************/
class WindowLineRef;

class WindowLine : public ControlWindow
{
  friend class WindowLineRef;

  protected:
    TextControl** _pbox;   // prompt box array
    ControlInfo* _Info;
    WindowLine* _nextline; // next window line in window block
    WindowLineRef** _RefPtr;
    WindowLineRef* _DummyRef;
    
    bool _InWinMode;        // in windows block mode
    bool _LineTerminated;   // Winbuffer series terminated
    bool _LineWrapAllowed;  // Line wrapping is allowed
    bool _ReturnOnLastLine; // Inactivate WindowLine upon ENTER key on last line
    bool _ShowChainedBoxes; // Show chained buffer boxes.
    
    int _maxbox;        // maximum number of prompt boxes
    int _boxcount;      // total non-null pointers to windows buffers stored

    int _screenxpos;   // horizontal coordinate position of physical text screen
    int _screenypos;   // vertical coordinate position of physical text screen
    int _screenmaxx;   // maximum horizontal extent of physical text screen
    int _screenmaxy;   // maximum vertical extent of physical text screen

    int _winboxxpos;   // horizontal coordinate position of bounded windows box
    int _winboxypos;   // vertical coordinate position of bounded windows box
    int _winboxstartx; // starting x of bounded windows box in text screen coordinates
    int _winboxstarty; // starting y of bounded windows box in text screen coordinates
    int _winboxmaxx;   // maximum horizontal extent of bounded windows box
    int _winboxmaxy;   // maximum vertical extent of bounded windows box

    int _winviewxoffset; // shifted x offset of sliding windows view
    int _winviewyoffset; // shifted y offset of sliding window view
    int _winviewxpos;    // horizontal coordinate position of sliding windows view
    int _winviewypos;    // vertical coordinate position of sliding window view

    int _inputlen;        // maximum input length (should set to same for all buffers)
    int _maxwraparound;   // maximum wrap around for window buffer contents
    int _wraparoundindex; // running wrap around index count
    int _WindowLineNum;   // Window line number for this line object

    // Finds total number of valid control objects stored within storage array
    virtual int FindControlCount();

    // Text Control accessor method
    virtual void Control(int Index_, TextControl* TxtCtrl_);
    virtual TextControl* Control(int Index_);

    // Dialog controls validity check method
    virtual Boolean AnyActive();
    virtual Boolean ValidityCheck();

    // data member destruction method
    virtual void Clear();

    // convenience methods
    void SetControlData(TextControl** CtrlArray_, ControlInfo* CtrlInfo_, int index);
    void RemoveControlData(TextControl** CtrlArray_, ControlInfo* CtrlInfo_, int index);

    // WindowLine class specific methods
    int FindMaxWrapAround();
    void SetLineToString(char* NewStr_);

  public:
    // prompt window constructors
    WindowLine(int Max_, int WinLineNum_,
               int WinStartx_, int WinStarty_,
               int WinMaxx_, int WinMaxy_);
    WindowLine(WindowLine& pw);
    virtual ~WindowLine();

    // prompt window data mutators
    virtual void MaxControl(int max);
    virtual void Grow(int extra);

    // data member accessors
    virtual int MaxControl();
    virtual void RemoveControl(TextControl* TxtCtrl_);
    virtual void DeleteControl(TextControl* TxtCtrl_);

    // Promptbox specific accessors
    WindowLineRef& operator [] (int index);
    WindowLineBuffer* NextSelected(int& Start_);

    // prompt window manipulators
    virtual void Show();
    virtual void Hide();
    virtual int Prompt(int StartIndex_=0);
    virtual void Erase();
    virtual void Delete();
    virtual void Remove();

    // Text color methods
    virtual void UseColor(Boolean Flag_=TRUE);
    virtual void SetNormalColors(int FgCol_, int BgCol_);
    virtual void SetHighLightColors(int FgCol_, int BgCol_);
    virtual void SetHotKeyColor(int Col_);
    virtual void UseDefaultColors();

    // Retrieve control information object
    virtual ControlInfo* GetControlInfo();

    // new WindowLine specific methods
    WindowLine& SetWinLineProperties(int LwAllowed_=-99,
                                     int ShowChained_=-99,
                                     int RestrictFull_=-99,
                                     int TrimWs_=-99,
                                     int EndBrksActive_=-99);
    WindowLine& SetWinBlockProperties(int InWin_=-99,
                                      int RetOnLast_=-99);
    void SendUnprocessedKey(int Key_, int Extended_,
                            WindowLineBuffer* WinBufPtr_);
    
    int SetInputLength(int Length_);
    void IncrWrapAround(int Num_);
    void DecrWrapAround(int Num_);
    bool InWindowMode(bool* HaveNext_=NULL) const;

    char* RetrieveInOrder();
    int GetWrapAroundIndex();
    inline void SetWindowLineNum(int Num_)
        { _WindowLineNum = Num_; }
    inline int GetMaxWrapAround()
        { return FindMaxWrapAround(); }
    inline bool LineWrapAllowed() const
        { return _LineWrapAllowed; }
    inline bool LineTerminated() const
        { return _LineTerminated; }
    inline bool ReturnOnLastLine() const
        { return _ReturnOnLastLine; }
    inline bool ShowChainedBoxes() const
        { return _ShowChainedBoxes; }
    inline int GetBoxCount() const
        { return _boxcount; }
    inline void ResetLine()
        { _LineTerminated = false; }

    int WinBoxToScreenX(int xpos_=-1);
    int WinBoxToScreenY(int ypos_=-1);
    int WinViewToWinBoxX(int xpos_=-1);
    int WinViewToWinBoxY(int ypos_=-1);

    // Perform shift down of entered line to next WindowLine object
    void ShiftDown(char* NewStr_, int Offset_, TextControl** Pbox_, int Maxbox_);
    void ShiftDownNext(char* NewStr_, int Offset_, TextControl** Pbox_, int Maxbox_);

    // Perform shift up of deleted line to current WindowLine object
    void ShiftUp(char* NewStr_, int Offset_, TextControl** Pbox_, int Maxbox_);
    void ShiftUpNext(char* NewStr_, int Offset_, TextControl** Pbox_, int Maxbox_);

    // Chain fill all following boxes with specified fill character
    void ChainFillString(char* Str_, int Offset_, char FillCh_, int Start_);
};

/****************************************************************************/
class WindowLineRef
{
  friend class WindowLine;
  private:
    WindowLine* _BossPtr;
    int _Index;
    bool _Cleared;
    bool _InProcess;

    WindowLineRef(WindowLine* Boss_, int Index_);

    inline bool InProcess() const
        { return _InProcess; }
    inline bool Cleared() const
        { return (_Cleared && !_InProcess); }
    static WindowLineBuffer* CastToWindowLineBuffer(TextControl* TxtCtrl_)
        { return ((WindowLineBuffer*)TxtCtrl_); }

  public:
    WindowLineRef& operator = (WindowLineBuffer* Pbox_);
    operator WindowLineBuffer* ();
    WindowLineBuffer* operator -> ();
};

/****************************************************************************/
#endif
