#ifndef POINTERSTACK_CPP
#define POINTERSTACK_CPP
#ifndef POINTERSTACK_H
  #include "pointerstack.h"
#endif

int PointerStackBase::_PrintSpec = 0;
int(*PointerStackBase::_StrOutFnc)(const char*) = NULL;
int(*PointerStackBase::_MemOutFnc)(void*) = NULL;

#if DELAYEDMEMORYDELETER_DEBUG
int ShowString(const char* str_)
{
  fprintf(stdout, "%s: deleted by DelayedMemoryDeleter\n", str_);
  return 0;
}
#endif

#if DELAYEDOBJECTDELETER_DEBUG
int ShowObject(void* memaddr_)
{
  ChrString* Record_ = (ChrString*)memaddr_;
  fprintf(stdout, "%s: deleted by DelayedObjectDeleter\n", Record_->c_str());
  return 0;
}
#endif

/*****************************************************************************/
// PointerStackBase class definition
/*****************************************************************************/
PointerStackBase::PointerStackBase():
_DataType(0)
{}

/****************************************************************************/
bool PointerStackBase::HasData(int TypeSpec_) const
{
  return (IsNonNullData() && RawData() && (!TypeSpec_ || (_DataType & TypeSpec_)));
}

/*****************************************************************************/
void PointerStackBase::SetStrOutputFnc(int(*FuncPtr_)(const char*))
{
  _StrOutFnc = FuncPtr_; 
}

/*****************************************************************************/
void PointerStackBase::SetMemOutputFnc(int(*FuncPtr_)(void*))
{
  _MemOutFnc = FuncPtr_;
}

/*****************************************************************************/
int PointerStackBase::PrintString(const char* str_)
{ 
  if (_StrOutFnc)
    return (*_StrOutFnc)(str_); 
}
            
/*****************************************************************************/
int PointerStackBase::PrintAddress(void* addr_)
{ 
  if (_MemOutFnc)
    return (*_MemOutFnc)(addr_);
}

/*****************************************************************************/
// CstringStack class definition
/*****************************************************************************/
CstringStack::CstringStack()
{
  _DataType = 0;
  _Active = NULL;
  _cNext = NULL;
  _Next = NULL;
}

/*****************************************************************************/
char* CstringStack::ActiveData(CstringStack* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_NONCONST |
                                    PointerStackBase::IS_NONVOID);

  return (ActiveStr_ ? ActiveStr_->GiveData():NULL);
}

/*****************************************************************************/
const char* CstringStack::ActiveData(const CstringStack* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_NONCONST |
                                    PointerStackBase::IS_NONVOID);

  return (ActiveStr_ ? ActiveStr_->GiveData():NULL);
}

/*****************************************************************************/
const char* CstringStack::ActiveConstData(const CstringStack* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_CONST |
                                    PointerStackBase::IS_NONVOID);

  return (ActiveStr_ ? ActiveStr_->GiveConstData():NULL);
}

/*****************************************************************************/
void* CstringStack::ActiveVoidData(CstringStack* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_NONCONST |
                                    PointerStackBase::IS_VOID);

  return (ActiveStr_ ? ActiveStr_->GiveVoidpData():NULL);
}

/*****************************************************************************/
const void* CstringStack::ActiveVoidData(const CstringStack* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_NONCONST |
                                    PointerStackBase::IS_VOID);

  return (ActiveStr_ ? ActiveStr_->GiveVoidpData():NULL);
}

/*****************************************************************************/
const void* CstringStack::ActiveConstVoidData(const CstringStack* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_CONST |
                                    PointerStackBase::IS_VOID);

  return (ActiveStr_ ? ActiveStr_->GiveConstVoidpData():NULL);
}

/*****************************************************************************/
CstringStack* CstringStack::PushActiveData(CstringStack* ActiveStr_, char* Ptr_)
{
  if (Ptr_ && ActiveStr_)
  {
    CstringStack* New_ = new CstringStack;
      
    if (New_)
    {
      New_->Push(ActiveStr_);
      New_->Set(Ptr_);
      ActiveStr_ = New_;

      return ActiveStr_;
    }
  }
  
  return NULL;
}

/*****************************************************************************/
CstringStack* CstringStack::PopActiveData(CstringStack* ActiveStr_, int PopAndDelete_)
{
  if (ActiveStr_ && MemMatrix::Matrix().HasThis(ActiveStr_))
  {
    CstringStack* Top_ = ActiveStr_;
    
    if (ActiveStr_)
      ActiveStr_ = ActiveStr_->_cNext;
    
    if (Top_)
    {
      if (PopAndDelete_)
        Top_->PopDelete(true);
      else
      {
        Top_->Set(NULL);
        Top_->Pop();
      }
    
      delete Top_;
      return ActiveStr_;
    }
  }
  else if (ActiveStr_)
  {
    ActiveStr_ = ActiveStr_->_cNext;
    return (MemMatrix::Matrix().HasThis(ActiveStr_) ? ActiveStr_:NULL);
  }
  
  return NULL;
}

/****************************************************************************/
PointerStack<char>* CstringStack::Next()
{
  return _Next;
}

/****************************************************************************/
const PointerStack<char>* CstringStack::Next() const
{
  return _Next;
}

/****************************************************************************/
CstringStack* CstringStack::NextCstr()
{
  return _cNext;
}

/****************************************************************************/
const CstringStack* CstringStack::NextCstr() const
{
  return _cNext;
}

/****************************************************************************/
CstringStack* CstringStack::CreateNewCstr(char* Ptr_)
{
  CstringStack* New_ = new CstringStack;
  New_->Set(Ptr_);
  return New_;
}

/****************************************************************************/
CstringStack* CstringStack::CreateNewVoidp(void* Ptr_)
{
  CstringStack* New_ = new CstringStack;
  New_->SetVoidp(Ptr_);
  return New_;
}

/****************************************************************************/
CstringStack* CstringStack::CreateNewConstCstr(const char* Ptr_)
{
  CstringStack* New_ = new CstringStack;
  New_->SetConst(Ptr_);
  return New_;
}

/****************************************************************************/
CstringStack* CstringStack::CreateNewConstVoidp(const void* Ptr_)
{
  CstringStack* New_ = new CstringStack;
  New_->SetConstVoidp(Ptr_);
  return New_;
}

/****************************************************************************/
CstringStack* CstringStack::PushNewCstr(CstringStack* Old_, char* Ptr_, bool SubstDel_)
{
  CstringStack* ret = NULL;
  
  if (SubstDel_)
  {    
    if (Old_)
      ret = (CstringStack*)(Old_->PushNew(Ptr_));
    else
      ret = CreateNewCstr(Ptr_);
  }
  else
  {
    MemMatrix::Matrix().Deallocate(Ptr_);
    Ptr_ = NULL;
  }
  
  return ret;
}

/****************************************************************************/
CstringStack* CstringStack::PushNewVoidp(CstringStack* Old_, void* Ptr_, bool SubstDel_)
{
  CstringStack* ret = NULL;
  
  if (SubstDel_)
  {    
    if (Old_)
      ret = (CstringStack*)(Old_->PushNewVoidp(Ptr_));
    else
      ret = CstringStack::CreateNewVoidp(Ptr_);
  }
  else
  {
    MemMatrix::Matrix().Deallocate(Ptr_);
    Ptr_ = NULL;
  }
  
  return ret;
}

/****************************************************************************/
CstringStack* CstringStack::PushNewConstCstr(CstringStack* Old_, const char* Ptr_, bool SubstDel_)
{
  CstringStack* ret = NULL;
  
  if (SubstDel_)
  {    
    if (Old_)
      ret = (CstringStack*)(Old_->PushNewConst(Ptr_));
    else
      ret = CreateNewConstCstr(Ptr_);
  }
  else
  {
    MemMatrix::Matrix().Deallocate((void*)Ptr_);
    Ptr_ = NULL;
  }
  
  return ret;
}

/****************************************************************************/
CstringStack* CstringStack::PushNewConstVoidp(CstringStack* Old_, const void* Ptr_, bool SubstDel_)
{
  CstringStack* ret = NULL;
  
  if (SubstDel_)
  {    
    if (Old_)
      ret = (CstringStack*)(Old_->PushNewConstVoidp(Ptr_));
    else
      ret = CreateNewConstVoidp(Ptr_);
  }
  else
  {
    MemMatrix::Matrix().Deallocate((void*)Ptr_);
    Ptr_ = NULL;
  }
  
  return ret;
}
    
/****************************************************************************/
void CstringStack::Set(char* Ptr_)
{
  _Active = Ptr_;
  SetVoidDataType(false);
  SetConstDataType(false);
}

/****************************************************************************/
void CstringStack::SetVoidp(void* Ptr_)
{
  _ActiveVoidPtr = Ptr_;
  SetVoidDataType(true);
  SetConstDataType(false);
}

/****************************************************************************/
void CstringStack::SetConst(const char* Ptr_)
{
  _ConstActive = Ptr_;
  SetVoidDataType(false);
  SetConstDataType(true);
}

/****************************************************************************/
void CstringStack::SetConstVoidp(const void* Ptr_)
{
  _ConstActiveVoidPtr = Ptr_;
  SetVoidDataType(true);
  SetConstDataType(true);
}

/****************************************************************************/
char* CstringStack::GiveData(bool OverrideConst_, bool OverrideTypeCheck_)
{
  return (((IsNonConstDataType() || OverrideConst_) && 
           (IsNonVoidDataType() || OverrideTypeCheck_)) ? _Active:NULL);
}

/****************************************************************************/
const char* CstringStack::GiveData(bool OverrideConst_, bool OverrideTypeCheck_) const
{
  return (((IsNonConstDataType() || OverrideConst_) && 
           (IsNonVoidDataType() || OverrideTypeCheck_)) ? _Active:NULL);
}

/****************************************************************************/
const char* CstringStack::GiveConstData(bool OverrideConst_, bool OverrideTypeCheck_) const
{
  return (((IsConstDataType() || OverrideConst_) && 
           (IsNonVoidDataType() || OverrideTypeCheck_)) ? _ConstActive:NULL);
}

/****************************************************************************/
void* CstringStack::GiveVoidpData(bool OverrideConst_, bool OverrideTypeCheck_)
{
  return (((IsNonConstDataType() || OverrideConst_) && 
           (IsVoidDataType() || OverrideTypeCheck_)) ? _ActiveVoidPtr:NULL);
}

/****************************************************************************/
const void* CstringStack::GiveVoidpData(bool OverrideConst_, bool OverrideTypeCheck_) const
{
  return (((IsNonConstDataType() || OverrideConst_) && 
           (IsVoidDataType() || OverrideTypeCheck_)) ? _ActiveVoidPtr:NULL);
}

/****************************************************************************/
const void* CstringStack::GiveConstVoidpData(bool OverrideConst_, bool OverrideTypeCheck_) const
{
  return (((IsConstDataType() || OverrideConst_) && 
           (IsVoidDataType() || OverrideTypeCheck_)) ? _ConstActiveVoidPtr:NULL);
}

/****************************************************************************/
PointerStack<char>* CstringStack::PushNew(char* Ptr_)
{
  CstringStack* New_ = new CstringStack;
  New_->Set(Ptr_);
  New_->Push(this);
  return New_;
}

/****************************************************************************/
PointerStack<char>* CstringStack::PushNewVoidp(void* Ptr_)
{
  CstringStack* New_ = new CstringStack;
  New_->SetVoidp(Ptr_);
  New_->Push(this);
  return New_;
}

/****************************************************************************/
PointerStack<char>* CstringStack::PushNewConst(const char* Ptr_)
{
  CstringStack* New_ = new CstringStack;
  New_->SetConst(Ptr_);
  New_->Push(this);
  return New_;
}

/****************************************************************************/
PointerStack<char>* CstringStack::PushNewConstVoidp(const void* Ptr_)
{
  CstringStack* New_ = new CstringStack;
  New_->SetConstVoidp(Ptr_);
  New_->Push(this);
  return New_;
}

/****************************************************************************/
void CstringStack::Push(PointerStack<char>* Ptr_)
{
  _cNext = (CstringStack*)Ptr_;
  _Next = Ptr_;
}

/****************************************************************************/
void CstringStack::Pop()
{
  if (ShowDataOnDelete())
  {
    if (IsVoidDataType())
    {
      if (ShowDataAsAddress() && GiveVoidpData(true))
        PointerStackBase::PrintAddress(GiveVoidpData(true));
    }
    else
    {
      if (ShowDataAsString() && GiveData(true))
        PointerStackBase::PrintString(GiveData(true));
      
      if (ShowDataAsAddress() && GiveVoidpData(true, true))
        PointerStackBase::PrintAddress(GiveVoidpData(true, true));
    }
  }

  _cNext = NULL;
  _Next = NULL;
  
  _Active = NULL;
}

/****************************************************************************/
bool CstringStack::PopDelete(bool SetNull_)
{
  bool Found_ = false;
  _cNext = NULL;
  _Next = NULL;
  
  if (IsNonNullData())
  {    
    if (ShowDataOnDelete())
    {
      if (IsVoidDataType())
      {
        if (ShowDataAsAddress() && GiveVoidpData(true))
          PointerStackBase::PrintAddress(GiveVoidpData(true));

        if (MemMatrix::Matrix().Deallocate(_ActiveVoidPtr))
        {
          _ActiveVoidPtr = NULL;
          Found_ = true;
        }
        else if (SetNull_)
          _Active = NULL;
      }
      else
      {
        if (ShowDataAsString() && GiveData(true))
          PointerStackBase::PrintString(GiveData(true));

        if (ShowDataAsAddress() && GiveVoidpData(true, true))
          PointerStackBase::PrintAddress(GiveVoidpData(true, true));

        if (MemMatrix::Matrix().Deallocate(_Active))
        {
          _Active = NULL;
          Found_ = true;
        }
        else if (SetNull_)
          _Active = NULL;
      }
    }
    else
    {
      if (IsVoidDataType())
      {
        if (MemMatrix::Matrix().Deallocate(_ActiveVoidPtr))
        {
          _ActiveVoidPtr = NULL;
          Found_ = true;
        }
        else if (SetNull_)
          _Active = NULL;
      }
      else
      {
        if (MemMatrix::Matrix().Deallocate(_Active))
        {
          _Active = NULL;
          Found_ = true;
        }
        else if (SetNull_)
          _Active = NULL;
      }
    }
  }
      
  SetNullData(true);
  _Active = NULL;

  return Found_;
}

/****************************************************************************/
void CstringStack::ShowStackEntry()
{
  if (IsVoidDataType())
  {
    if (ShowDataAsAddress() && GiveVoidpData(true))
      PointerStackBase::PrintAddress(GiveVoidpData(true));
  }
  else
  {
    if (ShowDataAsString() && GiveData(true))
      PointerStackBase::PrintString(GiveData(true));
    
    if (ShowDataAsAddress() && GiveVoidpData(true, true))
      PointerStackBase::PrintAddress(GiveVoidpData(true, true));
  }
}

/****************************************************************************/
// DelayedMemoryDeleter class definition
/****************************************************************************/
DelayedMemoryDeleter::DelayedMemoryDeleter():
_Destroyed(false),
_EvalRunCount(0),
_CheckInterval(DEFAULT_CHECK_INTERVAL),
_SwapInterval(DEFAULT_SWAP_INTERVAL),
_ActiveCstr(new CstringStack),
_BackupCstr(new CstringStack)
{}

/****************************************************************************/
DelayedMemoryDeleter::~DelayedMemoryDeleter()
{
  _Destroyed = true;
  _EvalRunCount = 0;
  DestroyStringStack(true);
  _ActiveCstr = _BackupCstr;
  DestroyStringStack(true);
  _ActiveCstr = _BackupCstr = NULL;
}

/****************************************************************************/
CstringStack* DelayedMemoryDeleter::CreateStringStack()
{
  if (!_ActiveCstr)
    _ActiveCstr = new CstringStack;
  
  if (!_BackupCstr)
    _BackupCstr = new CstringStack;
  
  return _ActiveCstr;
}

/****************************************************************************/
void DelayedMemoryDeleter::DestroyStringStack(int PopAndDelete_)
{
  size_t x = 0;
  size_t max = 2;
  
  bool fnd = true;
  bool reloop_ = false;
  CstringStack* InitialValue_ = _ActiveCstr;
  
  while (_ActiveCstr && x < max)
  {
    fnd = PopData(PopAndDelete_);
    reloop_ = _ActiveCstr == InitialValue_;
    
    if (!fnd || reloop_)
      x += reloop_ ? 2:1;
    else
      x = 0;
  }
}

/****************************************************************************/
void DelayedMemoryDeleter::SwapCstrStack()
{
  CstringStack* tempstk = _ActiveCstr;
  _ActiveCstr = _BackupCstr;
  _BackupCstr = tempstk;
}

/****************************************************************************/
void DelayedMemoryDeleter::ResetDefaults()
{
  _CheckInterval = DEFAULT_CHECK_INTERVAL;
  _SwapInterval = DEFAULT_SWAP_INTERVAL;
}

/****************************************************************************/
bool DelayedMemoryDeleter::SetTrashEmptyingInterval(size_t gap_)
{
  size_t chk_ = 0;
  
  if (gap_)
  {
    chk_ = _SwapInterval * gap_;
    if (chk_ == _SwapInterval || !chk_)
      ++chk_;
    
    _CheckInterval = chk_;
    if (!_CheckInterval)
      _CheckInterval++;
  }
  
  return (gap_ && chk_ && _SwapInterval);
}

/****************************************************************************/
bool DelayedMemoryDeleter::SetCheckInterval(size_t chk_)
{
  bool valid_ = chk_ && chk_ > _SwapInterval;
  if (valid_)
    _CheckInterval = chk_;
  
  return valid_;
}

/****************************************************************************/
bool DelayedMemoryDeleter::SetSwapInterval(size_t swp_)
{
  bool valid_ = swp_ && swp_ < _CheckInterval;
  if (valid_)
    _SwapInterval = swp_;
  
  return valid_;
}

/****************************************************************************/
size_t DelayedMemoryDeleter::MinIterTrashEmptied() const
{
  return (_SwapInterval ? (_CheckInterval / _SwapInterval):1);
}

/****************************************************************************/
size_t DelayedMemoryDeleter::ExtraIterTrashEmptied() const
{
  return (_SwapInterval ? (_CheckInterval % _SwapInterval):1);
}

/****************************************************************************/
bool DelayedMemoryDeleter::CheckIfEraseStoredTrash(bool SubstDel_)
{
  if (!SubstDel_)
    return false;
  
  bool Erase_ = false;
  _EvalRunCount++;
    
  if ((!_EvalRunCount ||
      (_CheckInterval && (_EvalRunCount % Ulong(_CheckInterval)) == 0)) && _BackupCstr->HasData())
  {    
    SwapCstrStack();
    Erase_ = true;
    DestroyStringStack(Erase_);
    _ActiveCstr = CreateStringStack();
    
    if (_SwapInterval && _EvalRunCount % Ulong(_SwapInterval))
      SwapCstrStack();
  }
  else if ((!_EvalRunCount ||
           (_CheckInterval && (_EvalRunCount % Ulong(_SwapInterval)) == 0)))
    SwapCstrStack();

  if (!_CheckInterval)
    _CheckInterval = DEFAULT_CHECK_INTERVAL;

  if (!_SwapInterval)
    _SwapInterval = DEFAULT_SWAP_INTERVAL;
    
  return Erase_;
}

/****************************************************************************/
char* DelayedMemoryDeleter::DelayedDeleteCstr(char* Ptr_, bool SubstDel_)
{
  if (Ptr_) _ActiveCstr = CstringStack::PushNewCstr(_ActiveCstr, Ptr_, SubstDel_);
  return (_ActiveCstr ? NULL:(SubstDel_ ? Ptr_:NULL));
}

/****************************************************************************/
const char* DelayedMemoryDeleter::DelayedDeleteConstCstr(const char* Ptr_, bool SubstDel_)
{
  if (Ptr_) _ActiveCstr = CstringStack::PushNewConstCstr(_ActiveCstr, Ptr_, SubstDel_);
  return (_ActiveCstr ? NULL:(SubstDel_ ? Ptr_:NULL));
}

/****************************************************************************/
void* DelayedMemoryDeleter::DelayedDeleteVoidp(void* Ptr_, bool SubstDel_)
{
  if (Ptr_) _ActiveCstr = CstringStack::PushNewVoidp(_ActiveCstr, Ptr_, SubstDel_);
  return (_ActiveCstr ? NULL:(SubstDel_ ? Ptr_:NULL));
}

/****************************************************************************/
const void* DelayedMemoryDeleter::DelayedDeleteConstVoidp(const void* Ptr_, bool SubstDel_)
{
  if (Ptr_) _ActiveCstr = CstringStack::PushNewConstVoidp(_ActiveCstr, Ptr_, SubstDel_);
  return (_ActiveCstr ? NULL:(SubstDel_ ? Ptr_:NULL));
}

/****************************************************************************/
bool DelayedMemoryDeleter::ClearAllBuffers(bool SubstDel_)
{
  if (!SubstDel_)
    return false;
  
  _Destroyed = true;
  _EvalRunCount = 0;
  DestroyStringStack(true);
  _ActiveCstr = _BackupCstr;
  DestroyStringStack(true);
  
  _ActiveCstr = new CstringStack;
  _BackupCstr = new CstringStack;
}

/****************************************************************************/
bool DelayedMemoryDeleter::PopData(int PopAndDelete_)
{
  bool ret = false;
  bool fnd = false;
  CstringStack* Top_ = _ActiveCstr;
  
  if (_ActiveCstr)
    _ActiveCstr = _ActiveCstr->NextCstr();
      
  if (Top_)
  {
    if (PopAndDelete_)
      Top_->PopDelete(true);
    else
    {
      Top_->Set(NULL);
      Top_->Pop();
    }
      
    if (MemMatrix::Matrix().HasThis(Top_))
    {
      fnd = true;
      delete Top_;
    }
    else
    {
      fnd = false;
      Top_ = NULL;
    }
    
    ret = fnd;
  }
  
  return ret;
}

/****************************************************************************/
MEMORYOPS_DEFN(PointerStackBase)
MEMORYOPS_DEFN(CstringStack)
MEMORYOPS_DEFN(DelayedMemoryDeleter)

/****************************************************************************/
/****************************************************************************/
#if DELAYEDDELETER_DEBUG
int main()
{
  int x = 0;
  int ones = 0;
  int tens = 0;
  int len;
  int max = 100;
  char cntstr[3];
  
  #if DELAYEDMEMORYDELETER_DEBUG
    char* Cstr_;
    DelayedMemoryDeleter* MemDel_ = new DelayedMemoryDeleter;
    MemDel_->SetShowDataOnDelete(true);
    MemDel_->SetShowDataAsString(true);
    MemDel_->SetStrOutputFnc(ShowString);
  
    for (x = 0; x < max; x++)
    {
      Cstr_ = ::NewString("iteration string: xx");
      len = strlen(Cstr_);
      ones = (x % 10) + '0';
      tens = (x / 10) + '0';
      Cstr_[len-1] = ones;
      Cstr_[len-2] = tens ? tens:' ';
    
      fprintf(stdout, "Cstr = %s\n", Cstr_);
      MemDel_->DelayedDeleteCstr(Cstr_, true);
      MemDel_->CheckIfEraseStoredTrash();
    }
  
    delete MemDel_;
  #endif
    
  #if DELAYEDOBJECTDELETER_DEBUG
    ChrString* Record_;
    ChrString From("xx");
    ChrString To;
    
    DelayedObjectDeleter<ChrString>* ObjDel_ = new DelayedObjectDeleter<ChrString>;
    ObjDel_->SetShowDataOnDelete(true);
    ObjDel_->SetShowDataAsAddress(true);
    ObjDel_->SetMemOutputFnc(ShowObject);
  
    for (x = 0; x < max; x++)
    {
      Record_ = new ChrString("iteration ChrString Object: xx");
      len = Record_->strlen();      
      ones = (x % 10) + '0';
      tens = (x / 10) + '0';
      cntstr[2] = 0;
      cntstr[1] = ones;
      cntstr[0] = tens ? tens:' ';
      To = cntstr;
      Record_->Replace(len-3, From, To);
    
      fprintf(stdout, "Record = %s\n", Record_->c_str());
      ObjDel_->DelayedDeleteObject(Record_, true);
      ObjDel_->CheckIfEraseStoredTrash();
    }
  
    delete ObjDel_;    
  #endif
}
#endif


/****************************************************************************/
#endif
