#ifndef POINTERSTACK_H
#define POINTERSTACK_H

/* standard libraries */
#ifndef INCL_STDIO_H
  #include <stdio.h>
  #define INCL_STDIO_H
#endif
#ifndef INCL_STRING_H
  #include <string.h>
  #define INCL_STRING_H
#endif
#ifndef INCL_STDLIB_H
  #include <stdlib.h>
  #define INCL_STDLIB_H
#endif

/* Custom libraries */
// ../cppds-4.2/source/
#ifndef TYPEDEFS_H
  #include "typedefs.h"
#endif
#ifndef CHRSTRING_H
  #include "chrstring.h"
#endif

#define DELAYEDDELETER_DEBUG                0
#define DELAYEDMEMORYDELETER_DEBUG          0
#define DELAYEDOBJECTDELETER_DEBUG          0

/****************************************************************************/
class PointerStackBase
{
  protected:
    static int _PrintSpec;
    Uchar _DataType;

    static int(*_StrOutFnc)(const char*);     // pointer to string output function
    static int(*_MemOutFnc)(void*);     // pointer to memory output function
    
  public:    
    enum
    {
      IS_NONCONST    = 0x01, // bit pattern: 0000 0001
      IS_CONST       = 0x02, // bit pattern: 0000 0010
      IS_NONVOID     = 0x04, // bit pattern: 0000 0100
      IS_VOID        = 0x08, // bit pattern: 0000 1000
            
      CONST_MASK     = 0x03, // bit pattern: 0000 0011
      VOID_MASK      = 0x0C, // bit pattern: 0000 1100      
      DATA_MASK      = 0x0F, // bit pattern: 0000 1111
      
      AS_STRING      = 0x10, // bit pattern: 0001 0000
      AS_ADDRESS     = 0x20, // bit pattern: 0010 0000
      SHOW_ON_DELETE = 0x40, // bit pattern: 0100 0000
      
      PRINT_MASK     = 0x30, // bit pattern: 0011 0000
      DELNSTR_MASK   = 0x50, // bit pattern: 0101 0000
      DELNADDR_MASK  = 0x60  // bit pattern: 0110 0000
    };  
    
    PointerStackBase();
    
    inline void SetVoidDataType(bool flag_)
        { _DataType = flag_ ? ((_DataType & CONST_MASK) | IS_VOID):
                              ((_DataType & CONST_MASK) | IS_NONVOID); }
    inline bool IsVoidDataType() const
        { return (_DataType & IS_VOID); }
    inline bool IsNonVoidDataType() const
        { return (_DataType & IS_NONVOID); }
    
    inline void SetConstDataType(bool flag_)
        { _DataType = flag_ ? ((_DataType & VOID_MASK) | IS_CONST):
                              ((_DataType & VOID_MASK) | IS_NONCONST); }
    inline bool IsConstDataType() const
        { return (_DataType & IS_CONST); }
    inline bool IsNonConstDataType() const
        { return (_DataType & IS_NONCONST); }
    
    inline void SetNullData(bool flag_)
        { if (flag_) _DataType = 0; }
    inline bool IsNullData() const
        { return ((_DataType & DATA_MASK) == 0); }
    inline bool IsNonNullData() const
        { return ((_DataType & DATA_MASK) != 0); }

    inline void SetShowNoData()
        { _PrintSpec = _PrintSpec & SHOW_ON_DELETE; }
    inline void SetShowDataAsString(bool flag_)
        { _PrintSpec = flag_ ? ((_PrintSpec & DELNADDR_MASK) | AS_STRING):(_PrintSpec & ~AS_STRING); }
    inline void SetShowDataAsAddress(bool flag_)
        { _PrintSpec = flag_ ? ((_PrintSpec & DELNSTR_MASK) | AS_ADDRESS):(_PrintSpec & ~AS_ADDRESS); }
    inline void SetShowDataOnDelete(bool flag_)
        { _PrintSpec = flag_ ? ((_PrintSpec & PRINT_MASK) | SHOW_ON_DELETE):(_PrintSpec & ~SHOW_ON_DELETE); }

    inline bool ShowNoData() const
        { return ((_PrintSpec & PRINT_MASK) == 0); }
    inline bool ShowDataAsString() const
        { return  (_PrintSpec & AS_STRING); }
    inline bool ShowDataAsAddress() const
        { return (_PrintSpec & AS_ADDRESS); }
    inline bool ShowDataOnDelete() const
        { return (_PrintSpec & SHOW_ON_DELETE); }
        
    static void SetStrOutputFnc(int(*FuncPtr_)(const char*));
    static void SetMemOutputFnc(int(*FuncPtr_)(void*));

    static int PrintString(const char* str_);
    static int PrintAddress(void* addr_);
            
    virtual void Pop() = 0;
    virtual bool PopDelete(bool SetNull_=false) = 0;

    virtual ptrdiff_t RawData() const = 0;
    virtual bool HasData(int TypeSpec_=0) const;
    
#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

template <class T>
class DelayedObjectDeleter;

template <class T>
class PointerStack : public PointerStackBase
{
  friend class DelayedObjectDeleter<T>;
  
  protected:
    PointerStack<T>* _Next;
    
    union
    {  
      T* _Active;
      const T* _ConstActive;            
      void* _ActiveVoidPtr;
      const void* _ConstActiveVoidPtr;  
    };
    
    static PointerStack<T>* CreateNewType(T* Ptr_);
    static PointerStack<T>* CreateNewVoidp(void* Ptr_);
    static PointerStack<T>* CreateNewConstType(const T* Ptr_);
    static PointerStack<T>* CreateNewConstVoidp(const void* Ptr_);
    
  public:
    PointerStack();
    
    // PointerStack: -- PointerStack<T> methods
    static T* ActiveData(PointerStack<T>* ActiveStr_, bool* HasData_=NULL);    
    static const T* ActiveData(const PointerStack<T>* ActiveStr_, bool* HasData_=NULL);    
    static const T* ActiveConstData(const PointerStack<T>* ActiveStr_, bool* HasData_=NULL);
    
    static void* ActiveVoidData(PointerStack<T>* ActiveStr_, bool* HasData_=NULL);
    static const void* ActiveVoidData(const PointerStack<T>* ActiveStr_, bool* HasData_=NULL);
    static const void* ActiveConstVoidData(const PointerStack<T>* ActiveStr_, bool* HasData_=NULL);
    
    static PointerStack<T>* PushActiveData(PointerStack<T>* ActiveStr_, T* Ptr_);
    static PointerStack<T>* PopActiveData(PointerStack<T>* ActiveStr_, int PopAndDelete_);
    
    static PointerStack<T>* PushNewType(PointerStack<T>* Old_, T* Ptr_, bool SubstDel_);
    static PointerStack<T>* PushNewVoidp(PointerStack<T>* Old_, void* Ptr_, bool SubstDel_);
    static PointerStack<T>* PushNewConstType(PointerStack<T>* Old_, const T* Ptr_, bool SubstDel_);
    static PointerStack<T>* PushNewConstVoidp(PointerStack<T>* Old_, const void* Ptr_, bool SubstDel_);

    virtual ptrdiff_t RawData() const;

    virtual PointerStack<T>* Next();
    virtual const PointerStack<T>* Next() const;
    
    virtual void Set(T* Ptr_);
    virtual void SetVoidp(void* Ptr_);
    virtual void SetConst(const T* Ptr_);
    virtual void SetConstVoidp(const void* Ptr_);

    virtual T* GiveData(bool OverrideConst_=false, bool OverrideTypeCheck_=false);
    virtual const T* GiveData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const;
    virtual const T* GiveConstData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const;
    virtual void* GiveVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false);
    virtual const void* GiveVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const;
    virtual const void* GiveConstVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const;
    
    virtual PointerStack<T>* PushNew(T* Ptr_);
    virtual PointerStack<T>* PushNewVoidp(void* Ptr_);
    virtual PointerStack<T>* PushNewConst(const T* Ptr_);
    virtual PointerStack<T>* PushNewConstVoidp(const void* Ptr_);
    
    virtual void Push(PointerStack<T>* Ptr_);    
    virtual void Pop();
    virtual bool PopDelete(bool SetNull_=false);
    virtual void ShowStackEntry();

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

class CstringStack : public PointerStack<char>
{
  friend class DelayedMemoryDeleter;
  
  protected:
    CstringStack* _cNext;
    
    static CstringStack* CreateNewCstr(char* Ptr_);        
    static CstringStack* CreateNewVoidp(void* Ptr_);
    static CstringStack* CreateNewConstCstr(const char* Ptr_);
    static CstringStack* CreateNewConstVoidp(const void* Ptr_);
  
  public:    
    CstringStack();
    
    // CstringStack: -- PointerStack<char> methods
    static char* ActiveData(CstringStack* ActiveStr_, bool* HasData_=NULL);
    static const char* ActiveData(const CstringStack* ActiveStr_, bool* HasData_=NULL);
    static const char* ActiveConstData(const CstringStack* ActiveStr_, bool* HasData_=NULL);
    
    static void* ActiveVoidData(CstringStack* ActiveStr_, bool* HasData_=NULL);
    static const void* ActiveVoidData(const CstringStack* ActiveStr_, bool* HasData_=NULL);
    static const void* ActiveConstVoidData(const CstringStack* ActiveStr_, bool* HasData_=NULL);
    
    static CstringStack* PushActiveData(CstringStack* ActiveStr_, char* Ptr_);
    static CstringStack* PopActiveData(CstringStack* ActiveStr_, int PopAndDelete_);
    
    static CstringStack* PushNewCstr(CstringStack* Old_, char* Ptr_, bool SubstDel_);
    static CstringStack* PushNewVoidp(CstringStack* Old_, void* Ptr_, bool SubstDel_);
    static CstringStack* PushNewConstCstr(CstringStack* Old_, const char* Ptr_, bool SubstDel_);
    static CstringStack* PushNewConstVoidp(CstringStack* Old_, const void* Ptr_, bool SubstDel_);
    
    virtual PointerStack<char>* Next();
    virtual const PointerStack<char>* Next() const;
    
    virtual CstringStack* NextCstr();
    virtual const CstringStack* NextCstr() const;
    
    virtual void Set(char* Ptr_);
    virtual void SetVoidp(void* Ptr_);
    virtual void SetConst(const char* Ptr_);
    virtual void SetConstVoidp(const void* Ptr_);

    virtual char* GiveData(bool OverrideConst_=false, bool OverrideTypeCheck_=false);
    virtual const char* GiveData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const;
    virtual const char* GiveConstData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const;
    virtual void* GiveVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false);
    virtual const void* GiveVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const;
    virtual const void* GiveConstVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const;

    virtual PointerStack<char>* PushNew(char* Ptr_);
    virtual PointerStack<char>* PushNewVoidp(void* Ptr_);
    virtual PointerStack<char>* PushNewConst(const char* Ptr_);
    virtual PointerStack<char>* PushNewConstVoidp(const void* Ptr_);
    
    virtual void Push(PointerStack<char>* Ptr_);
    virtual void Pop();
    virtual bool PopDelete(bool SetNull_=false);
    virtual void ShowStackEntry();
    
#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
class DelayedMemoryDeleter
{
  public:
    enum
    {
      DEFAULT_CHECK_INTERVAL = 3,
      DEFAULT_SWAP_INTERVAL  = 2,
      
      AS_STRING  = 16,
      AS_ADDRESS = 32
    };
  
  protected:
    mutable bool _Destroyed;
    Ulong _EvalRunCount;
    size_t _CheckInterval;
    size_t _SwapInterval;
    
    mutable CstringStack* _ActiveCstr;
    mutable CstringStack* _BackupCstr;
    
    int _ShowDeletedMemory;

    inline CstringStack* CreateNewCstr(char* Ptr_)
        { return _ActiveCstr->CreateNewCstr(Ptr_); }
    inline CstringStack* CreateNewConstCstr(const char* Ptr_)
        { return _ActiveCstr->CreateNewConstCstr(Ptr_); }
    inline CstringStack* CreateNewVoidp(void* Ptr_)
        { return _ActiveCstr->CreateNewVoidp(Ptr_); }
    inline CstringStack* CreateNewConstVoidp(const void* Ptr_)
        { return _ActiveCstr->CreateNewConstVoidp(Ptr_); }
    
    CstringStack* CreateStringStack();
    void DestroyStringStack(int PopAndDelete_);
    void SwapCstrStack();

  public:
    DelayedMemoryDeleter();
    ~DelayedMemoryDeleter();    
    
    bool CheckIfEraseStoredTrash(bool SubstDel_=true);

    inline ptrdiff_t RawData() const
        { return _ActiveCstr->RawData(); }
    inline bool HasData(int TypeSpec_=0) const
        { return _ActiveCstr->HasData(TypeSpec_); }

    char* DelayedDeleteCstr(char* Ptr_, bool SubstDel_);
    const char* DelayedDeleteConstCstr(const char* Ptr_, bool SubstDel_);
    void* DelayedDeleteVoidp(void* Ptr_, bool SubstDel_);
    const void* DelayedDeleteConstVoidp(const void* Ptr_, bool SubstDel_);
    bool ClearAllBuffers(bool SubsDel_);
    bool PopData(int PopAndDelete_);
    
    void ResetDefaults();
    bool SetTrashEmptyingInterval(size_t gap_);
    bool SetCheckInterval(size_t chk_);
    bool SetSwapInterval(size_t swp_);        
    
    size_t MinIterTrashEmptied() const;
    size_t ExtraIterTrashEmptied() const;

    inline void ShowStackTop()
        { _ActiveCstr->ShowStackEntry(); }
    inline void SetShowNoData()
        { _ActiveCstr->SetShowNoData(); }
    inline void SetShowDataAsString(bool flag_)
        { _ActiveCstr->SetShowDataAsString(flag_); }
    inline void SetShowDataAsAddress(bool flag_)
        { _ActiveCstr->SetShowDataAsAddress(flag_); }
    inline void SetShowDataOnDelete(bool flag_)
        { _ActiveCstr->SetShowDataOnDelete(flag_); }
        
    inline bool ShowNoData() const
        { return _ActiveCstr->ShowNoData(); }
    inline bool ShowDataAsString() const
        { return _ActiveCstr->ShowDataAsString(); }
    inline bool ShowDataAsAddress() const
        { return _ActiveCstr->ShowDataAsAddress(); }
    inline bool ShowDataOnDelete() const
        { return _ActiveCstr->ShowDataOnDelete(); }
    
    inline void SetStrOutputFnc(int(*FuncPtr_)(const char*))
        { PointerStackBase::SetStrOutputFnc(FuncPtr_); }
    inline void SetMemOutputFnc(int(*FuncPtr_)(void*))
        { PointerStackBase::SetMemOutputFnc(FuncPtr_); }
    
    inline char* ActiveData(bool* HasData_=NULL)
        { return CstringStack::ActiveData(_ActiveCstr, HasData_); }
    inline const char* ActiveData(bool* HasData_=NULL) const
        { return CstringStack::ActiveData(_ActiveCstr, HasData_); }
    inline const char* ActiveConstData(bool* HasData_=NULL)
        { return CstringStack::ActiveConstData(_ActiveCstr, HasData_); }
    
    inline void* ActiveVoidData(bool* HasData_=NULL)
        { return CstringStack::ActiveVoidData(_ActiveCstr, HasData_); }
    inline const void* ActiveVoidData(bool* HasData_=NULL) const
        { return CstringStack::ActiveVoidData(_ActiveCstr, HasData_); }
    inline const void* ActiveConstVoidData(bool* HasData_=NULL)
        { return CstringStack::ActiveConstVoidData(_ActiveCstr, HasData_); }

    inline CstringStack* GiveBackupCstr()
        { return _BackupCstr; }
    inline const CstringStack* GiveBackupCstr() const
        { return _BackupCstr; }
    
    inline CstringStack* GiveActiveCstr()
        { return _ActiveCstr; }
    inline const CstringStack* GiveActiveCstr() const
        { return _ActiveCstr; }

    inline char* GiveData(bool OverrideConst_=false, bool OverrideTypeCheck_=false)
        { return _ActiveCstr->GiveData(OverrideConst_, OverrideTypeCheck_); }
    inline const char* GiveData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const
        { return _ActiveCstr->GiveData(OverrideConst_, OverrideTypeCheck_); }
    inline const char* GiveConstData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const
        { return _ActiveCstr->GiveConstData(OverrideConst_, OverrideTypeCheck_); }

    inline void* GiveVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false)
        { return _ActiveCstr->GiveVoidpData(OverrideConst_, OverrideTypeCheck_); }
    inline const void* GiveVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const
        { return _ActiveCstr->GiveVoidpData(OverrideConst_, OverrideTypeCheck_); }
    inline const void* GiveConstVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const
        { return _ActiveCstr->GiveConstVoidpData(OverrideConst_, OverrideTypeCheck_); }
    
    inline void SetDestroyed(bool flag_)
        { _Destroyed = true; }
    inline bool Destroyed() const
        { return _Destroyed; }
    inline Ulong RunCount() const
        { return _EvalRunCount; }
        
#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif    
};

/****************************************************************************/
template <class T>
class DelayedObjectDeleter
{
  public:
    enum
    {
      DEFAULT_CHECK_INTERVAL = 3,
      DEFAULT_SWAP_INTERVAL  = 2,
    };

  protected:
    mutable bool _Destroyed;
    Ulong _EvalRunCount;
    size_t _CheckInterval;
    size_t _SwapInterval;
    
    mutable PointerStack<T>* _ActiveObject;
    mutable PointerStack<T>* _BackupObject;
       
    inline PointerStack<T>* CreateNewObject(T* Ptr_)
        { return _ActiveObject->CreateNewType(Ptr_); }
    inline PointerStack<T>* CreateNewConstObject(const T* Ptr_)
        { return _ActiveObject->CreateNewConstType(Ptr_); }
    inline PointerStack<T>* CreateNewVoidp(void* Ptr_)
        { return _ActiveObject->CreateNewVoidp(Ptr_); }
    inline PointerStack<T>* CreateNewConstVoidp(const void* Ptr_)
        { return _ActiveObject->CreateNewConstVoidp(Ptr_); }
    
    PointerStack<T>* CreateObjectStack();
    void DestroyObjectStack(int PopAndDelete_);
    void SwapObjectStack();

  public:
    DelayedObjectDeleter();
    ~DelayedObjectDeleter();
    
    bool CheckIfEraseStoredTrash(bool SubstDel_=true);

    inline ptrdiff_t RawData() const
        { return _ActiveObject->RawData(); }
    inline bool HasData(int TypeSpec_=0) const
        { return _ActiveObject->HasData(TypeSpec_); }
    
    T* DelayedDeleteObject(T* Ptr_, bool SubstDel_);
    const T* DelayedDeleteConstObject(const T* Ptr_, bool SubstDel_);
    void* DelayedDeleteVoidp(void* Ptr_, bool SubstDel_);
    const void* DelayedDeleteConstVoidp(const void* Ptr_, bool SubstDel_);
    bool ClearAllBuffers(bool SubsDel_);
    bool PopData(int PopAndDelete_);

    void ResetDefaults();
    bool SetTrashEmptyingInterval(size_t gap_);
    bool SetCheckInterval(size_t chk_);
    bool SetSwapInterval(size_t swp_);
    
    size_t MinIterTrashEmptied() const;
    size_t ExtraIterTrashEmptied() const;

    inline void ShowStackTop()
        { _ActiveObject->ShowStackEntry(); }
    inline void SetShowData()
        { _ActiveObject->SetShowNoData(); }
    inline void SetShowDataAsString(bool flag_)
        { _ActiveObject->SetShowDataAsString(flag_); }
    inline void SetShowDataAsAddress(bool flag_)
        { _ActiveObject->SetShowDataAsAddress(flag_); }
    inline void SetShowDataOnDelete(bool flag_)
        { _ActiveObject->SetShowDataOnDelete(flag_); }
        
    inline bool ShowNoData() const
        { return _ActiveObject->ShowNoData(); }
    inline bool ShowDataAsString() const
        { return _ActiveObject->ShowDataAsString(); }
    inline bool ShowDataAsAddress() const
        { return _ActiveObject->ShowDataAsAddress(); }
    inline bool ShowDataOnDelete() const
        { return _ActiveObject->ShowDataOnDelete(); }

    inline void SetStrOutputFnc(int(*FuncPtr_)(const char*))
        { PointerStackBase::SetStrOutputFnc(FuncPtr_); }
    inline void SetMemOutputFnc(int(*FuncPtr_)(void*))
        { PointerStackBase::SetMemOutputFnc(FuncPtr_); }
        
    inline T* ActiveData(bool* HasData_=NULL)
        { return PointerStack<T>::ActiveData(_ActiveObject, HasData_); }
    inline const T* ActiveData(bool* HasData_=NULL) const
        { return PointerStack<T>::ActiveData(_ActiveObject, HasData_); }        
    inline const T* ActiveConstData(bool* HasData_=NULL)
        { return PointerStack<T>::ActiveConstData(_ActiveObject, HasData_); }
    
    inline void* ActiveVoidData(bool* HasData_=NULL)
        { return PointerStack<T>::ActiveVoidData(_ActiveObject, HasData_); }
    inline const void* ActiveVoidData(bool* HasData_=NULL) const
        { return PointerStack<T>::ActiveVoidData(_ActiveObject, HasData_); }
    inline const void* ActiveConstVoidData(bool* HasData_=NULL)
        { return PointerStack<T>::ActiveConstVoidData(_ActiveObject, HasData_); }

    inline PointerStack<T>* GiveBackupObject()
        { return _BackupObject; }
    inline const PointerStack<T>* GiveBackupObject() const
        { return _BackupObject; }
    
    inline PointerStack<T>* GiveActiveObject()
        { return _ActiveObject; }
    inline const PointerStack<T>* GiveActiveObject() const
        { return _ActiveObject; }

    inline T* GiveData(bool OverrideConst_=false, bool OverrideTypeCheck_=false)
        { return _ActiveObject->GiveData(OverrideConst_, OverrideTypeCheck_); }
    inline const T* GiveData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const
        { return _ActiveObject->GiveData(OverrideConst_, OverrideTypeCheck_); }
    inline const T* GiveConstData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const
        { return _ActiveObject->GiveConstData(OverrideConst_, OverrideTypeCheck_); }

    inline void* GiveVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false)
        { return _ActiveObject->GiveVoidpData(OverrideConst_, OverrideTypeCheck_); }
    inline const void* GiveVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const
        { return _ActiveObject->GiveVoidpData(OverrideConst_, OverrideTypeCheck_); }
    inline const void* GiveConstVoidpData(bool OverrideConst_=false, bool OverrideTypeCheck_=false) const
        { return _ActiveObject->GiveConstVoidpData(OverrideConst_, OverrideTypeCheck_); }
    
    inline void SetDestroyed(bool flag_)
        { _Destroyed = true; }
    inline bool Destroyed() const
        { return _Destroyed; }
    inline Ulong RunCount() const
        { return _EvalRunCount; }
        
#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif    
};

/*****************************************************************************/
/*****************************************************************************/
template <class T>
PointerStack<T>::PointerStack()
{
  _DataType = 0;
  _Active = NULL;
}

/****************************************************************************/
template <class T>
ptrdiff_t PointerStack<T>::RawData() const
{
  ptrdiff_t AddrVal_ = 0;
  if (IsNonNullData())
    AddrVal_ = IsNonVoidDataType() ?
                  (IsNonConstDataType() ? ptrdiff_t(GiveData()):ptrdiff_t(GiveConstData())):
               IsVoidDataType() ?
                  (IsNonConstDataType() ? ptrdiff_t(GiveVoidpData()):ptrdiff_t(GiveConstVoidpData())):0;

  return AddrVal_;
}

/*****************************************************************************/
template <class T>
T* PointerStack<T>::ActiveData(PointerStack<T>* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_NONCONST |
                                    PointerStackBase::IS_NONVOID);
  
  return (ActiveStr_ ? ActiveStr_->GiveData():NULL);
}

/*****************************************************************************/
template <class T>
const T* PointerStack<T>::ActiveData(const PointerStack<T>* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_NONCONST |
                                    PointerStackBase::IS_NONVOID);

  return (ActiveStr_ ? ActiveStr_->GiveData():NULL);
}

/*****************************************************************************/
template <class T>
const T* PointerStack<T>::ActiveConstData(const PointerStack<T>* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_CONST |
                                    PointerStackBase::IS_NONVOID);

  return (ActiveStr_ ? ActiveStr_->GiveConstData():NULL);
}

/*****************************************************************************/
template <class T>
void* PointerStack<T>::ActiveVoidData(PointerStack<T>* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_NONCONST |
                                    PointerStackBase::IS_VOID);

  return (ActiveStr_ ? ActiveStr_->GiveVoidpData():NULL);
}

/*****************************************************************************/
template <class T>
const void* PointerStack<T>::ActiveVoidData(const PointerStack<T>* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_NONCONST |
                                    PointerStackBase::IS_VOID);

  return (ActiveStr_ ? ActiveStr_->GiveVoidpData():NULL);
}

/*****************************************************************************/
template <class T>
const void* PointerStack<T>::ActiveConstVoidData(const PointerStack<T>* ActiveStr_, bool* HasData_)
{
  if (HasData_ && ActiveStr_)
    *HasData_ = ActiveStr_->HasData(PointerStackBase::IS_CONST |
                                    PointerStackBase::IS_VOID);

  return (ActiveStr_ ? ActiveStr_->GiveConstVoidpData():NULL);
}

/*****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::PushActiveData(PointerStack<T>* ActiveStr_, T* Ptr_)
{
  if (Ptr_ && ActiveStr_)
  {
    PointerStack<T>* New_ = new PointerStack<T>;
      
    if (New_)
    {
      New_->Push(ActiveStr_);
      New_->Set(Ptr_);
      ActiveStr_ = New_;

      return ActiveStr_;
    }
  }
  
  return NULL;
}

/*****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::PopActiveData(PointerStack<T>* ActiveStr_, int PopAndDelete_)
{
  if (ActiveStr_ && MemMatrix::Matrix().HasThis(ActiveStr_))
  {
    PointerStack<T>* Top_ = ActiveStr_;
    
    if (ActiveStr_)
      ActiveStr_ = ActiveStr_->_Next;
    
    if (Top_)
    {
      if (PopAndDelete_)
        Top_->PopDelete(true);
      else
      {
        Top_->Set(NULL);
        Top_->Pop();
      }
    
      delete Top_;
      return ActiveStr_;
    }
  }
  else if (ActiveStr_)
  {
    ActiveStr_ = ActiveStr_->_Next;
    return (MemMatrix::Matrix().HasThis(ActiveStr_) ? ActiveStr_:NULL);
  }
  
  return NULL;
}

/****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::Next()
{
  return _Next;
}

/****************************************************************************/
template <class T>
const PointerStack<T>* PointerStack<T>::Next() const
{
  return _Next;
}

/****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::CreateNewType(T* Ptr_)
{
  PointerStack<T>* New_ = new PointerStack<T>;
  New_->Set(Ptr_);
  return New_;
}

/****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::CreateNewVoidp(void* Ptr_)
{
  PointerStack<T>* New_ = new PointerStack<T>;
  New_->SetVoidp(Ptr_);
  return New_;
}

/****************************************************************************/
template <class T>
PointerStack<T>* CreateNewConstType(const T* Ptr_)
{
  PointerStack<T>* New_ = new PointerStack<T>;
  New_->SetConst(Ptr_);
  return New_;
}

/****************************************************************************/
template <class T>
PointerStack<T>* CreateNewConstVoidp(const void* Ptr_)
{
  PointerStack<T>* New_ = new PointerStack<T>;
  New_->SetConstVoidp(Ptr_);
  return New_;
}

/****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::PushNewType(PointerStack<T>* Old_, T* Ptr_, bool SubstDel_)
{
  PointerStack<T>* ret = NULL;
  
  if (SubstDel_)
  {
    if (Old_)
      ret = Old_->PushNew(Ptr_);
    else
      ret = CreateNewType(Ptr_);
  }
  else
  {
    MemMatrix::Matrix().Deallocate(Ptr_);
    return NULL;
  }  
  
  return ret;
}

/****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::PushNewVoidp(PointerStack<T>* Old_, void* Ptr_, bool SubstDel_)
{
  PointerStack<T>* ret = NULL;
  
  if (SubstDel_)
  {
    if (Old_)
      ret = Old_->PushNewVoidp(Ptr_);
    else
      ret = CreateNewVoidp(Ptr_);
  }
  else
  {
    MemMatrix::Matrix().Deallocate(Ptr_);
    return NULL;
  }  
  
  return ret;
}

/****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::PushNewConstType(PointerStack<T>* Old_, const T* Ptr_, bool SubstDel_)
{
  PointerStack<T>* ret = NULL;
  
  if (SubstDel_)
  {
    if (Old_)
      ret = Old_->PushNewConst(Ptr_);
    else
      ret = CreateNewConstType(Ptr_);
  }
  else
  {
    MemMatrix::Matrix().Deallocate((void*)Ptr_);
    return NULL;
  }  
  
  return ret;
}

/****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::PushNewConstVoidp(PointerStack<T>* Old_, const void* Ptr_, bool SubstDel_)
{
  PointerStack<T>* ret = NULL;
  
  if (SubstDel_)
  {
    if (Old_)
      ret = Old_->PushNewConstVoidp(Ptr_);
    else
      ret = CreateNewConstVoidp(Ptr_);
  }
  else
  {
    MemMatrix::Matrix().Deallocate((void*)Ptr_);
    return NULL;
  }  
  
  return ret;
}

/****************************************************************************/
template <class T>
void PointerStack<T>::Set(T* Ptr_)
{
  _Active = Ptr_;
  SetVoidDataType(false);
  SetConstDataType(false);
}

/****************************************************************************/
template <class T>
void PointerStack<T>::SetVoidp(void* Ptr_)
{
  _ActiveVoidPtr = Ptr_;
  SetVoidDataType(true);
  SetConstDataType(false);
}

/****************************************************************************/
template <class T>
void PointerStack<T>::SetConst(const T* Ptr_)
{
  _ConstActive = Ptr_;
  SetVoidDataType(false);
  SetConstDataType(true);
}

/****************************************************************************/
template <class T>
void PointerStack<T>::SetConstVoidp(const void* Ptr_)
{
  _ConstActiveVoidPtr = Ptr_;
  SetVoidDataType(true);
  SetConstDataType(true);
}

/****************************************************************************/
template <class T>
T* PointerStack<T>::GiveData(bool OverrideConst_, bool OverrideTypeCheck_)
{
  return (((IsNonConstDataType() || OverrideConst_) &&
           (IsNonVoidDataType() || OverrideTypeCheck_)) ? _Active:NULL);
}

/****************************************************************************/
template <class T>
const T* PointerStack<T>::GiveData(bool OverrideConst_, bool OverrideTypeCheck_) const
{
  return (((IsNonConstDataType() || OverrideConst_) && 
           (IsNonVoidDataType() || OverrideTypeCheck_)) ? _Active:NULL);
}

/****************************************************************************/
template <class T>
const T* PointerStack<T>::GiveConstData(bool OverrideConst_, bool OverrideTypeCheck_) const
{
  return (((IsConstDataType() || OverrideConst_) &&
           (IsNonVoidDataType() || OverrideTypeCheck_)) ? _ConstActive:NULL);
}

/****************************************************************************/
template <class T>
void* PointerStack<T>::GiveVoidpData(bool OverrideConst_, bool OverrideTypeCheck_)
{
  return (((IsNonConstDataType() || OverrideConst_) && 
           (IsVoidDataType() || OverrideTypeCheck_)) ? _ActiveVoidPtr:NULL);
}

/****************************************************************************/
template <class T>
const void* PointerStack<T>::GiveVoidpData(bool OverrideConst_, bool OverrideTypeCheck_) const
{
  return (((IsNonConstDataType() || OverrideConst_) && 
           (IsVoidDataType() || OverrideTypeCheck_)) ? _ActiveVoidPtr:NULL);
}

/****************************************************************************/
template <class T>
const void* PointerStack<T>::GiveConstVoidpData(bool OverrideConst_, bool OverrideTypeCheck_) const
{
  return (((IsConstDataType() || OverrideConst_) && 
           (IsVoidDataType() || OverrideTypeCheck_)) ? _ConstActiveVoidPtr:NULL);
}

/****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::PushNew(T* Ptr_)
{
  PointerStack<T>* New_ = new PointerStack<T>;
  New_->Set(Ptr_);
  New_->Push(this);
  return New_;
}

/****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::PushNewVoidp(void* Ptr_)
{
  PointerStack<T>* New_ = new PointerStack<T>;
  New_->SetVoidp(Ptr_);
  New_->Push(this);
  return New_;
}

/****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::PushNewConst(const T* Ptr_)
{
  PointerStack<T>* New_ = new PointerStack<T>;
  New_->SetConst(Ptr_);
  New_->Push(this);
  return New_;
}

/****************************************************************************/
template <class T>
PointerStack<T>* PointerStack<T>::PushNewConstVoidp(const void* Ptr_)
{
  PointerStack<T>* New_ = new PointerStack<T>;
  New_->SetConstVoidp(Ptr_);
  New_->Push(this);
  return New_;
}

/****************************************************************************/
template <class T>
void PointerStack<T>::Push(PointerStack<T>* Ptr_)
{
  _Next = Ptr_;
}

/****************************************************************************/
template <class T>
void PointerStack<T>::Pop()
{
  if (ShowDataOnDelete())
  {
    if (IsVoidDataType())
    {
      if (ShowDataAsAddress() && GiveVoidpData(true))
        PointerStackBase::PrintAddress(GiveVoidpData(true));
    }
    else
    {
      if (ShowDataAsString() && GiveData(true, true))
        PointerStackBase::PrintString((const char*)GiveData(true, true));
      
      if (ShowDataAsAddress() && GiveVoidpData(true, true))
        PointerStackBase::PrintAddress(GiveVoidpData(true, true));
    }
  }

  _Next = NULL;
  _Active = NULL;
}

/****************************************************************************/
template <class T>
bool PointerStack<T>::PopDelete(bool SetNull_)
{
  bool Found_ = false;
  _Next = NULL;
    
  if (IsNonNullData())
  {
    if (ShowDataOnDelete())
    {
      if (IsVoidDataType())
      {
        if (ShowDataAsAddress() && GiveVoidpData(true))
          PointerStackBase::PrintAddress(GiveVoidpData(true));
      
        if (MemMatrix::Matrix().Deallocate(_ActiveVoidPtr))
        {
          _ActiveVoidPtr = NULL;
          Found_ = true;
        }
        else if (SetNull_)
          _ActiveVoidPtr = NULL;
      }
      else
      {
        if (ShowDataAsString() && GiveData(true, true))
          PointerStackBase::PrintString((const char*)GiveData(true, true));

        if (ShowDataAsAddress() && GiveVoidpData(true, true))
          PointerStackBase::PrintAddress(GiveVoidpData(true, true));

        if (MemMatrix::Matrix().HasThis(_Active))
        {
          delete _Active;
          _Active = NULL;

          Found_ = true;
        }
        else if (SetNull_)
          _Active = NULL;
      }
    }
    else
    {
      if (IsVoidDataType())
      {      
        if (MemMatrix::Matrix().Deallocate(_ActiveVoidPtr))
        {
          _ActiveVoidPtr = NULL;
          Found_ = true;
        }
        else if (SetNull_)
          _Active = NULL;
      }
      else
      {      
        if (MemMatrix::Matrix().HasThis(_Active))
        {
          delete _Active;
          _Active = NULL;

          Found_ = true;
        }
        else if (SetNull_)
          _Active = NULL;
      }
    }
  }
    
  SetNullData(true);
  _Active = NULL;

  return Found_;
}

/****************************************************************************/
template <class T>
void PointerStack<T>::ShowStackEntry()
{
  if (IsVoidDataType())
  {
    if (ShowDataAsAddress() && GiveVoidpData(true))
      PointerStackBase::PrintAddress(GiveVoidpData(true));
  }
  else
  {
    if (ShowDataAsString() && GiveData(true, true))
      PointerStackBase::PrintString((const char*)GiveData(true, true));
    
    if (ShowDataAsAddress() && GiveVoidpData(true, true))
      PointerStackBase::PrintAddress(GiveVoidpData(true, true));
  }
}

/****************************************************************************/
/****************************************************************************/
template <class T>
DelayedObjectDeleter<T>::DelayedObjectDeleter():
_Destroyed(false),
_EvalRunCount(0),
_CheckInterval(DEFAULT_CHECK_INTERVAL),
_SwapInterval(DEFAULT_SWAP_INTERVAL),
_ActiveObject(new PointerStack<T>),
_BackupObject(new PointerStack<T>)
{}

/****************************************************************************/
template <class T>
DelayedObjectDeleter<T>::~DelayedObjectDeleter()
{
  _Destroyed = true;
  _EvalRunCount = 0;
  DestroyObjectStack(true);
  _ActiveObject = _BackupObject;
  DestroyObjectStack(true);
  _ActiveObject = _BackupObject = NULL;
}

/****************************************************************************/
template <class T>
PointerStack<T>* DelayedObjectDeleter<T>::CreateObjectStack()
{
  if (!_ActiveObject)
    _ActiveObject = new PointerStack<T>;
  
  if (!_BackupObject)
    _BackupObject = new PointerStack<T>;
  
  return _ActiveObject;
}

/****************************************************************************/
template <class T>
void DelayedObjectDeleter<T>::DestroyObjectStack(int PopAndDelete_)
{
  size_t x = 0;
  size_t max = 2;
  
  bool fnd = true;
  bool reloop_ = false;
  PointerStack<T>* InitialValue_ = _ActiveObject;
  
  while (_ActiveObject && x < max)
  {
    fnd = PopData(PopAndDelete_);
    reloop_ = _ActiveObject == InitialValue_;
    
    if (!fnd || reloop_)
      x += reloop_ ? 2:1;
    else
      x = 0;
  }
}

/****************************************************************************/
template <class T>
void DelayedObjectDeleter<T>::SwapObjectStack()
{
  PointerStack<T>* tempstk = _ActiveObject;
  _ActiveObject = _BackupObject;
  _BackupObject = tempstk;
}

/****************************************************************************/
template <class T>
void DelayedObjectDeleter<T>::ResetDefaults()
{
  _CheckInterval = DEFAULT_CHECK_INTERVAL;
  _SwapInterval = DEFAULT_SWAP_INTERVAL;
}

/****************************************************************************/
template <class T>
bool DelayedObjectDeleter<T>::SetTrashEmptyingInterval(size_t gap_)
{
  size_t chk_ = 0;
  
  if (gap_)
  {
    chk_ = _SwapInterval * gap_;
    if (chk_ == _SwapInterval || !chk_)
      ++chk_;
    
    _CheckInterval = chk_;
    if (!_CheckInterval)
      _CheckInterval++;
  }
  
  return (gap_ && chk_ && _SwapInterval);
}

/****************************************************************************/
template <class T>
bool DelayedObjectDeleter<T>::SetCheckInterval(size_t chk_)
{
  bool valid_ = chk_ && chk_ > _SwapInterval;
  if (valid_)
    _CheckInterval = chk_;
  
  return valid_;
}

/****************************************************************************/
template <class T>
bool DelayedObjectDeleter<T>::SetSwapInterval(size_t swp_)
{
  bool valid_ = swp_ && swp_ < _CheckInterval;
  if (valid_)
    _SwapInterval = swp_;
  
  return valid_;
}

/****************************************************************************/
template <class T>
size_t DelayedObjectDeleter<T>::MinIterTrashEmptied() const
{
  return (_SwapInterval ? (_CheckInterval / _SwapInterval):1);
}

/****************************************************************************/
template <class T>
size_t DelayedObjectDeleter<T>::ExtraIterTrashEmptied() const
{
  return (_SwapInterval ? (_CheckInterval % _SwapInterval):1);
}

/****************************************************************************/
template <class T>
bool DelayedObjectDeleter<T>::CheckIfEraseStoredTrash(bool SubstDel_)
{
  if (!SubstDel_)
    return false;
  
  bool Erase_ = false;
  _EvalRunCount++;
  
  if ((!_EvalRunCount ||
      (_CheckInterval && (_EvalRunCount % Ulong(_CheckInterval)) == 0)) && _BackupObject->HasData())
  {    
    SwapObjectStack();
    Erase_ = true;
    DestroyObjectStack(Erase_);
    _ActiveObject = CreateObjectStack();
    
    if (_SwapInterval && _EvalRunCount % Ulong(_SwapInterval))
      SwapObjectStack();
  }
  else if ((!_EvalRunCount ||
           (_CheckInterval && (_EvalRunCount % Ulong(_SwapInterval)) == 0)))
    SwapObjectStack();

  if (!_CheckInterval)
    _CheckInterval = DEFAULT_CHECK_INTERVAL;

  if (!_SwapInterval)
    _SwapInterval = DEFAULT_SWAP_INTERVAL;
  
  return Erase_;
}

/****************************************************************************/
template <class T>
T* DelayedObjectDeleter<T>::DelayedDeleteObject(T* Ptr_, bool SubstDel_)
{
  if (Ptr_) _ActiveObject = PointerStack<T>::PushNewType(_ActiveObject, Ptr_, SubstDel_);
  return (_ActiveObject ? NULL:(SubstDel_ ? Ptr_:NULL));
}

/****************************************************************************/
template <class T>
const T* DelayedObjectDeleter<T>::DelayedDeleteConstObject(const T* Ptr_, bool SubstDel_)
{
  if (Ptr_) _ActiveObject = PointerStack<T>::PushNewConstType(_ActiveObject, Ptr_, SubstDel_);
  return (_ActiveObject ? NULL:(SubstDel_ ? Ptr_:NULL));
}

/****************************************************************************/
template <class T>
void* DelayedObjectDeleter<T>::DelayedDeleteVoidp(void* Ptr_, bool SubstDel_)
{
  if (Ptr_) _ActiveObject = PointerStack<T>::PushNewVoidp(_ActiveObject, Ptr_, SubstDel_);
  return (_ActiveObject ? NULL:(SubstDel_ ? Ptr_:NULL));
}

/****************************************************************************/
template <class T>
const void* DelayedObjectDeleter<T>::DelayedDeleteConstVoidp(const void* Ptr_, bool SubstDel_)
{
  if (Ptr_) _ActiveObject = PointerStack<T>::PushNewConstVoidp(_ActiveObject, Ptr_, SubstDel_);
  return (_ActiveObject ? NULL:(SubstDel_ ? Ptr_:NULL));
}

/****************************************************************************/
template <class T>
bool DelayedObjectDeleter<T>::ClearAllBuffers(bool SubstDel_)
{
  if (!SubstDel_)
    return false;
  
  _Destroyed = true;
  _EvalRunCount = 0;
  DestroyObjectStack(true);
  _ActiveObject = _BackupObject;
  DestroyObjectStack(true);
  
  _ActiveObject = new PointerStack<T>;
  _BackupObject = new PointerStack<T>;
}

/****************************************************************************/
template <class T>
bool DelayedObjectDeleter<T>::PopData(int PopAndDelete_)
{
  bool fnd = false;
  bool ret = false;
  PointerStack<T>* Top_ = _ActiveObject;
  
  if (_ActiveObject)
    _ActiveObject = _ActiveObject->Next();
      
  if (Top_)
  {
    if (PopAndDelete_)
      Top_->PopDelete(true);
    else
    {
      Top_->Set(NULL);
      Top_->Pop();
    }
      
    if (MemMatrix::Matrix().HasThis(Top_))
    {
      fnd = true;
      delete Top_;
    }
    else
    {
      fnd = false;
      Top_ = NULL;
    }
    
    ret = fnd;
  }
  
  return ret;
}

/****************************************************************************/
MEMORYOPS_TEMPLATE_DEFN(DelayedObjectDeleter)
MEMORYOPS_TEMPLATE_DEFN(PointerStack)

/****************************************************************************/
#endif
