// Prompt box and Prompt window ADT classes
// The Prompt box class define methods for maintaining a Prompt box object
// used for Prompting a user for input from the console. The Prompt box class
// also has methods for filtering the input to define and disallow invalid
// input, positioning procedures for setting the display position of the
// Prompt box, setting the default Prompt message and default input into
// the Prompt box, Showing, hiding, Prompting, erasing and retrieval of
// the input data.
//
// The Prompt window class define methods for maintaining an array of Prompt
// boxes. Methods are defined to Show, Hide, Prompt, Erase and Retrieve from
// the entire array of Prompt boxes assigned to the Prompt window object.
//
#ifndef PROMPTBOX_CPP
#define PROMPTBOX_CPP
#ifndef PROMPTBOX_H
  #include "promptbox.h"
#endif

/****************************************************************************/
/****************************************************************************/
const int* InsertDetect(void* Owner_, PromptBox*, const int* Ptr_)
{
  if (Owner_)
    ((PromptBox*)Owner_)->NotifyInsertMode(Ptr_);

  return Ptr_;
}

/****************************************************************************/
// PURPOSE:
//   Prompt box default constructor
//
// PRE:
//   an uninitialize Prompt box
//
// POST:
//   a Prompt box is constructed and assigned the default values specified.
//   If the msg argument is not NULL then the Prompt message of the box
//   is assigned to the argument string.
//   The string buffer is Cleared to all 0.
//
PromptBox::PromptBox(char* msg, PromptBox* next_, int Size_, int Length_):
_Owner(NULL),
_chainbox(next_),
_HotKey(0),
_ShortCut(0),
_HotIndex(0),
_HotKeyStatus(NONE),
_Enterable(TRUE),
_Selectable(FALSE),
_msg(TextControl::SetMessage_Method(msg, _ShortCut, _HotKey, _HotIndex)),
_filter(NULL),
_negfilter(NULL),
_CtrlFilter(NULL),
_CtrlFilterLen(0),
_buffer(Size_ ? (new char[Size_ + 1]):(new char[MAXBUFFER + 1])),
_maxbuf(Size_ ? Size_:MAXBUFFER),
_len(Length_ ? Length_:
     Size_ ? Size_:1),   // _len should never be zero
_minlen(1),              // minimum length should be at least 1
_chainminlen(0),         // no chains at initialization so zero
_index(0),
_xpos(wherex()),
_ypos(wherey()),
_testf(NULL),
_noempty(0),
_InsertMode(FALSE),
_Escaped(FALSE),
_Hidden(TRUE),
_TextWidgetActive(TRUE),
_ChainPos(0),
_InFocus(FALSE),
_SavedCursX(WHEREX()),
_SavedCursY(WHEREY()),
_RestFullBox(true),
_EndBrksActive(true),
_TrimWs(false),

// Internal data monitors
_insert_monitor(NULL),
_buffer_monitor(NULL),
_key_monitor(NULL),
_ext_monitor(NULL),
_index_monitor(NULL),
_escaped_monitor(NULL),
_hidden_monitor(NULL),
_active_monitor(NULL),
_infocus_monitor(NULL),

// Handles to external objects, used by data monitors
_insert_handle(NULL),
_buffer_handle(NULL),
_key_handle(NULL),
_ext_handle(NULL),
_index_handle(NULL),
_escaped_handle(NULL),
_hidden_handle(NULL),
_active_handle(NULL),
_infocus_handle(NULL)
{
  _brackets[0] = '[';
  _brackets[1] = ']';

  for (int index = 0; index <= _maxbuf; index++)
    _buffer[index] = '\0';
}

/****************************************************************************/
// PURPOSE:
//   Prompt box copy construcor
//
// PRE:
//   An uninitialized Prompt box.
//
// POST:
//   a Prompt box is constructed and assigned the corresponding values
//   in the Prompt box argument passed to the constructor.
//
PromptBox::PromptBox(PromptBox& pb, int Size_, int Length_):
_Owner(NULL),
_chainbox(NULL),
_HotKey(pb._HotKey),
_ShortCut(pb._ShortCut),
_HotIndex(pb._HotIndex),
_HotKeyStatus(NONE),
_Enterable(pb._Enterable),
_Selectable(pb._Selectable),
_msg(pb._msg ? strcpy(new char[::SafeStrLen(pb._msg)+1], pb._msg):NULL),
_filter(pb._filter?strcpy(new char[::SafeStrLen(pb._filter)+1], pb._filter):NULL),
_negfilter(pb._negfilter?strcpy(new char[::SafeStrLen(pb._negfilter)+1], pb._negfilter):NULL),
_CtrlFilter(pb._CtrlFilterLen?(int*)memmove(new int[pb._CtrlFilterLen], pb._CtrlFilter, pb._CtrlFilterLen * sizeof(int)):NULL),
_CtrlFilterLen(pb._CtrlFilterLen),
_buffer(pb._maxbuf ? (new char[pb._maxbuf + 1]):
        Size_ ? (new char[Size_ + 1]):
                (new char[MAXBUFFER + 1])),
_maxbuf(pb._maxbuf ? pb._maxbuf:
        Size_ ? Size_:MAXBUFFER),
_len(pb._len ? pb._len:
     Length_ ? Length_:
     Size_ ? Size_:1),   // _len should never be zero
_minlen(pb._minlen ? pb._minlen:1),
_chainminlen(0),
_index(pb._index),
_xpos(pb._xpos),
_ypos(pb._ypos),
_testf(pb._testf),
_noempty(pb._noempty),
_InsertMode(FALSE),
_Escaped(FALSE),
_Hidden(TRUE),
_TextWidgetActive(TRUE),
_ChainPos(0),
_InFocus(FALSE),
_SavedCursX(WHEREX()),
_SavedCursY(WHEREY()),
_RestFullBox(true),
_EndBrksActive(true),
_TrimWs(false),

// Internal data monitors
_insert_monitor(NULL),
_buffer_monitor(NULL),
_key_monitor(NULL),
_ext_monitor(NULL),
_index_monitor(NULL),
_escaped_monitor(NULL),
_hidden_monitor(NULL),
_active_monitor(NULL),
_infocus_monitor(NULL),

// Handles to external objects, used by data monitors
_insert_handle(NULL),
_buffer_handle(NULL),
_key_handle(NULL),
_ext_handle(NULL),
_index_handle(NULL),
_escaped_handle(NULL),
_hidden_handle(NULL),
_active_handle(NULL),
_infocus_handle(NULL)
{
  _brackets[0] = '[';
  _brackets[1] = ']';

  strcpy(_buffer, pb._buffer);
  for (int index = ::SafeStrLen(_buffer); index <= _maxbuf; index++)
    _buffer[index] = '\0';

  if (pb._minlen)
    SetMinimumLength(pb._minlen);
  else
    SetMinimumLength(1);
}

/****************************************************************************/
const char* PromptBox::cMonitor(const char*(*monptr_)(void*, PromptBox*, const char*), void* This_, const char* ptr_)
{
  if (monptr_ != NULL)
  {
    SaveCursorPosition();
    const char* RetVal_ = (*monptr_)(This_, this, ptr_);
    RestoreCursorPosition();

    return RetVal_;
  }

  return ptr_;
}

/****************************************************************************/
const int* PromptBox::iMonitor(const int*(*monptr_)(void*, PromptBox*, const int*), void* This_, const int* ptr_)
{
  if (monptr_ != NULL)
  {
    SaveCursorPosition();
    const int* RetVal_ = (*monptr_)(This_, this, ptr_);
    RestoreCursorPosition();

    return RetVal_;
  }

  return ptr_;
}

/****************************************************************************/
void PromptBox::SaveCursorPosition()
{
  _SavedCursX = WHEREX();
  _SavedCursY = WHEREY();
}

/****************************************************************************/
void PromptBox::RestoreCursorPosition()
{
  GOTOXY(_SavedCursX, _SavedCursY);
}

/****************************************************************************/
const int* PromptBox::MonitorInsertMode(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*))
{
  _insert_handle = This_;
  _insert_monitor = FnPtr_;

  if (_chainbox)
    _chainbox->MonitorInsertMode(_insert_handle, _insert_monitor);

  return &_InsertMode;
}

/****************************************************************************/
const char* PromptBox::MonitorBuffer(void* This_, const char*(*FnPtr_)(void*, PromptBox*, const char*))
{
  _buffer_handle = This_;
  _buffer_monitor = FnPtr_;

  if (_chainbox)
    _chainbox->MonitorBuffer(_buffer_handle, _buffer_monitor);

  return _buffer;
}

/****************************************************************************/
const int* PromptBox::MonitorKey(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*))
{
  _key_handle = This_;
  _key_monitor = FnPtr_;

  if (_chainbox)
    _chainbox->MonitorKey(_key_handle, _key_monitor);

  return &_key;
}

/****************************************************************************/
const int* PromptBox::MonitorExtendedKey(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*))
{
  _ext_handle = This_;
  _ext_monitor = FnPtr_;

  if (_chainbox)
    _chainbox->MonitorExtendedKey(_ext_handle, _ext_monitor);

  return &_extended;
}

/****************************************************************************/
const int* PromptBox::MonitorIndex(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*))
{
  _index_handle = This_;
  _index_monitor = FnPtr_;

  if (_chainbox)
    _chainbox->MonitorIndex(_index_handle, _index_monitor);

  return &_index;
}

/****************************************************************************/
const int* PromptBox::MonitorEscapedStatus(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*))
{
  _escaped_handle = This_;
  _escaped_monitor = FnPtr_;

  if (_chainbox)
    _chainbox->MonitorEscapedStatus(_escaped_handle, _escaped_monitor);

  return &_Escaped;
}

/****************************************************************************/
const int* PromptBox::MonitorInFocusStatus(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*))
{
  _infocus_handle = This_;
  _infocus_monitor = FnPtr_;

  if (_chainbox)
    _chainbox->MonitorInFocusStatus(_infocus_handle, _infocus_monitor);

  return &_InFocus;
}

/****************************************************************************/
const int* PromptBox::MonitorHiddenStatus(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*))
{
  _hidden_handle = This_;
  _hidden_monitor = FnPtr_;
  return &_Hidden;
}

/****************************************************************************/
const int* PromptBox::MonitorActiveStatus(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*))
{
  _active_handle = This_;
  _active_monitor = FnPtr_;
  return &_TextWidgetActive;
}

/****************************************************************************/
// PURPOSE:
//   Prompt box data member destruction method
//
// PRE:
//   The data members of this Prompt box object with original values
//
// POST:
//   If the data members of the box object is not NULL then the data members
//   are deleted and assigned NULL
//
void PromptBox::Clear()
{
  delete[] _msg;
  _msg = NULL;

  delete[] _buffer;
  _buffer = NULL;
  cMonitor(_buffer_monitor, _buffer_handle, _buffer);

  delete[] _filter;
  _filter = NULL;

  delete[] _CtrlFilter;
  _CtrlFilter = NULL;

  delete[] _negfilter;
  _negfilter = NULL;
}

/****************************************************************************/
/****************************************************************************/
// PURPOSE:
//   Prompt box assignment operator
//
// PRE:
//   The data members of this Prompt box object with original values
//
// POST:
//   The data members of this object is assigned to the corresponding
//   data members of the Prompt box argument passed to the method.
//   This object is returned.
//
PromptBox& PromptBox::operator = (PromptBox& pb)
{
  if (this != &pb)
  {
    Clear();

    _chainbox = NULL;
    _HotKey = pb._HotKey;
    _ShortCut = pb._ShortCut;
    _HotIndex = pb._HotIndex;
    _HotKeyStatus = NONE;
    _Enterable = pb._Enterable;
    _Selectable = pb._Selectable;
    _msg = pb._msg ? strcpy(new char[::SafeStrLen(pb._msg)+1], pb._msg):NULL;
    _filter = pb._filter ? strcpy(new char[::SafeStrLen(pb._filter)+1], pb._filter):NULL;
    _negfilter = pb._negfilter ? strcpy(new char[::SafeStrLen(pb._negfilter)+1], pb._negfilter):NULL;
    _CtrlFilter = pb._CtrlFilterLen ? (int*)memmove(new int[pb._CtrlFilterLen], pb._CtrlFilter, pb._CtrlFilterLen * sizeof(int)):NULL;
    _CtrlFilterLen = pb._CtrlFilterLen;
    _maxbuf = (pb._maxbuf ? pb._maxbuf:
               _len ? _len:MAXBUFFER);
    _len = (pb.InputLength() ? pb.InputLength():
           (_maxbuf <= SCREEN_STDWIDTH) ? _maxbuf:1);           
    _index = pb._index;
    _xpos = pb._xpos;
    _ypos = pb._ypos;
    _testf = pb._testf;
    _noempty = pb._noempty;
    _RestFullBox = pb._RestFullBox;
    _brackets[0] = pb._brackets[0];
    _brackets[1] = pb._brackets[1];

    if (::SafeStrLen(pb._buffer) > ::SafeStrLen(_buffer))
    {    
      delete[] _buffer;
      _buffer= (_maxbuf ? (new char[_maxbuf + 1]):
                _len ? (new char[_len + 1]):
                       (new char[MAXBUFFER + 1]));
      cMonitor(_buffer_monitor, _buffer_handle, _buffer);
    }

    strcpy(_buffer, pb._buffer);
    for (int index = ::SafeStrLen(_buffer); index <= _maxbuf; index++)
      _buffer[index] = '\0';

    if (pb._minlen)
      SetMinimumLength(pb._minlen);
    else
      SetMinimumLength(1);
  }

  return *this;
}

/****************************************************************************/
PromptBox* PromptBox::Make(PromptBox& pbox, int Size_, int Length_)
{
  PromptBox* pb;
  pb = new PromptBox(pbox, Size_, Length_);
  return pb;
}

/****************************************************************************/
PromptBox* PromptBox::Make(char* msg, PromptBox* next_, int Size_, int Length_)
{
  PromptBox* pb;
  pb = new PromptBox(msg, next_, Size_, Length_);
  return pb;
}

/****************************************************************************/
PromptBox* PromptBox::MakeNextChain(PromptBox* pb)
{
  PromptBox* nwpb;
  nwpb = pb ? (new PromptBox(*pb, pb->_maxbuf, pb->_len)):NULL;

  if (pb && nwpb)
  {
    int msglen_ = pb->_msg ? ::SafeStrLen(pb->_msg):0;
    nwpb->SetInputLength(pb->_len);
    nwpb->SetXpos(pb->_xpos + msglen_ + pb->_len + 2);
    nwpb->SetYpos(pb->_ypos);
  }

  return nwpb;
}

/****************************************************************************/
/****************************************************************************/
void PromptBox::SetRoundBrackets()
{
  _brackets[0] = '(';
  _brackets[1] = ')';
}

/****************************************************************************/
void PromptBox::SetSquareBrackets()
{
  _brackets[0] = '[';
  _brackets[1] = ']';
}

/****************************************************************************/
void PromptBox::PrintBlanks(int Max_)
{
  for (int Index_ = 0; Index_ < Max_; Index_++)
    PUTCH(BLANK);
}

/****************************************************************************/
int PromptBox::GetHotKeyType() const
{
  return _HotKeyStatus;
}

/****************************************************************************/
int PromptBox::GetHotKey() const
{
  return _HotKey;
}

/****************************************************************************/
int PromptBox::GetShortCut() const
{
  return _ShortCut;
}

/****************************************************************************/
// PURPOSE:
//   Set Prompt message method
//
// PRE:
//   The original value of the message data member
//
// POST:
//   The message data member is assigned new memory copied from the argument
//   string.
//
char* PromptBox::SetMessage(const char* str)
{
  if (str != _msg)
  {
    delete[] _msg;
    _msg = TextControl::SetMessage_Method(str, _ShortCut, _HotKey, _HotIndex);
  }

  return _msg;
}

/****************************************************************************/
void PromptBox::RemoveMessage()
{
  SetMessage(NULL);
}

/****************************************************************************/
char* PromptBox::SetHotKeys(char* str)
{
  char* newstr_ = TextControl::SetMessage_Method(str, _ShortCut, _HotKey, _HotIndex);
  return newstr_;
}

/****************************************************************************/
void PromptBox::RemoveHotKeys()
{
  _ShortCut = _HotKey = _HotIndex = 0;
}

/****************************************************************************/
void PromptBox::BlankMessage()
{
  int i, Len_;
  Len_ = ::SafeStrLen(_msg);

  for (i = 0; i < Len_; ++i)
    _msg[i] = ' ';
}

/****************************************************************************/
void PromptBox::PushInputString(char* str)
{
  if (IsValidString(str))
    for (short Index_ = ::SafeStrLen(str) - 1; Index_ >= 0; Index_--)
      Push(0, str[Index_]);
}

/****************************************************************************/
// PURPOSE:
//   Set default input method
//
// PRE:
//   The original value of the input buffer data member
//
// POST:
//   The input buffer data member is assigned new memory copied from the
//   argument string. The remaining space in the buffer is NULL filled.
//
void PromptBox::SetInputString(char* str)
{
  if (str && ::SafeStrLen(str) <= _maxbuf && IsValidString(str))
  {
    strcpy(_buffer, str);
    for (int index = ::SafeStrLen(_buffer); index <= _maxbuf; index++)
      _buffer[index] = '\0';
  }
}

/****************************************************************************/
// PURPOSE:
//   Set input length method
//
// PRE:
//   The original value of the input length data member
//
// POST:
//   The input length data member is copied from the length argument.
//
void PromptBox::SetInputLength(int len)
{
  if (len <= _maxbuf && len >= _minlen && len >= 1)
    _len = len;
}

/****************************************************************************/
int PromptBox::InputLength() const
{
  return _len;
}

/****************************************************************************/
void PromptBox::DivBufSizeAsInputLength(int div_)
{
  if (_maxbuf)
  {
    int res_ = _maxbuf / div_;
    if (res_ >= _minlen && res_ >= 1)
      _len = res_;
  }
}

/****************************************************************************/
// PURPOSE:
//   Set minimum input length method
//
// PRE:
//   The original value of the minimum input length data member
//
// POST:
//   The mimimum input length data member is copied from the length argument.
//
void PromptBox::SetMinimumLength(int len)
{
  if (len <= _len)
  {
    _minlen = (len > 0) ? len:1;
    _chainminlen = 0;
  }
  else
  {
    _minlen = _len;
    _chainminlen = len - _len;

    if (_chainbox)
      _chainbox->SetMinimumLength(_chainminlen);
  }
}

/****************************************************************************/
// PURPOSE:
//   Set input filter method
//
// PRE:
//   The original value of the filter
//
// POST:
//   The input filter is set to argument string passed.
//
void PromptBox::SetFilter(char* str)
{
  if (str != _filter)
  {
    delete[] _filter;
    _filter = str ?
    strcpy(new char[::SafeStrLen(str)+1], str):NULL;

    if (_chainbox)
      _chainbox->SetFilter(_filter);
  }
}

/****************************************************************************/
void PromptBox::AddFilter(char* str)
{
  int i, Len_, FLen_, OldLen_;

  if (!str)
    return;

  Len_ = ::SafeStrLen(str);
  OldLen_ = FLen_ = ::SafeStrLen(_filter);

  char* news_ = strcpy(new char[Len_ + FLen_ + 1], _filter);

  for (i = 0; i < Len_; ++i)
    if (!memchr(news_, str[i], FLen_))
      news_[FLen_++] = str[i];

  if (FLen_ != OldLen_)
  {
    news_[FLen_] = 0;
    delete[] _filter;
    _filter = news_;
  }
  else
    delete[] news_;
}

/****************************************************************************/
void PromptBox::RemoveFilter(char* str)
{
  char *cptr_, *dptr_, *news_, *loc_;
  int i, Cnt_, Len_, FLen_, OldLen_;

  if (!str)
    return;

  Len_ = ::SafeStrLen(str);
  OldLen_ = FLen_ = ::SafeStrLen(_filter);
  dptr_ = cptr_ = news_ = _filter;

  for (Cnt_ = i = 0; i < Len_; ++i)
  {
    while (loc_ = (char*)memchr(cptr_, str[i], FLen_))
    {
      ++Cnt_;
      *loc_ = 0;
      dptr_ = cptr_;
      cptr_ = loc_ + 1;
      FLen_ -= (cptr_ - dptr_);

      if (FLen_ <= 0)
	break;
    }

    cptr_ = news_;
    FLen_ = OldLen_;
  }

  if (Cnt_)
  {
    _filter = new char[OldLen_ - Cnt_ + 1];

    for (FLen_ = i = 0; i < OldLen_; ++i)
      if (news_[i])
	_filter[FLen_++] = news_[i];

    _filter[FLen_] = 0;
    delete[] news_;
  }
}

/****************************************************************************/
int* PromptBox::WordChr(int* src_, int val_, size_t n)
{
  while (n)
    if (*src_ == val_)
      return src_;
    else
    {
      src_++;
      --n;
    }

  return NULL;
}

/****************************************************************************/
void PromptBox::SetCtrlFilter(const KeyNode& Obj_)
{
  SetCtrlFilter(Obj_.GetCodes(), Obj_.Size());
}

/****************************************************************************/
void PromptBox::SetCtrlFilter(const int* str, int Len_)
{
  if (str != _CtrlFilter)
  {
    delete[] _CtrlFilter;
    _CtrlFilter = str ? (int*)memmove(new int[Len_], str, sizeof(int) * Len_):NULL;
    _CtrlFilterLen = str ? Len_:0;

    if (_chainbox)
      _chainbox->SetCtrlFilter(_CtrlFilter, Len_);
  }
}

/****************************************************************************/
void PromptBox::AddCtrlFilter(const int* str, int Len_)
{
  int* news_;
  int i, FLen_;

  if (!str || !Len_)
    return;

  FLen_ = _CtrlFilterLen;
  news_ = (int*)memmove(new int[Len_ + FLen_ + 1], _CtrlFilter, sizeof(int) * FLen_);

  for (i = 0; i < Len_; ++i)
    if (!WordChr(news_, str[i], FLen_))
      news_[FLen_++] = str[i];

  if (FLen_ != _CtrlFilterLen)
  {
    delete[] _CtrlFilter;
    _CtrlFilter = news_;
    _CtrlFilterLen = FLen_;
  }
  else
    delete[] news_;
}

/****************************************************************************/
void PromptBox::AddCtrlFilter(const KeyNode& Obj_)
{
  AddCtrlFilter(Obj_.GetCodes(), Obj_.Size());
}

/****************************************************************************/
void PromptBox::RemoveCtrlFilter(const int* str, int Len_)
{
  int *cptr_, *dptr_, *news_, *loc_;
  int i, Cnt_, FLen_;

  if (!str || !Len_)
    return;

  FLen_ = _CtrlFilterLen;
  dptr_ = cptr_ = news_ = _CtrlFilter;

  for (Cnt_ = i = 0; i < Len_; ++i)
  {
    while (loc_ = WordChr(cptr_, str[i], FLen_))
    {
      ++Cnt_;
      *loc_ = 0;
      dptr_ = cptr_;
      cptr_ = loc_ + 1;
      FLen_ -= (cptr_ - dptr_);

      if (FLen_ <= 0)
	break;
    }

    cptr_ = news_;
    FLen_ = _CtrlFilterLen;
  }

  if (Cnt_)
  {
    _CtrlFilter = new int[_CtrlFilterLen - Cnt_ + 1];

    for (FLen_ = i = 0; i < _CtrlFilterLen; ++i)
      if (news_[i])
	_CtrlFilter[FLen_++] = news_[i];

    _CtrlFilterLen = FLen_;
    delete[] news_;
  }
}

/****************************************************************************/
void PromptBox::RemoveCtrlFilter(const KeyNode& Obj_)
{
  RemoveCtrlFilter(Obj_.GetCodes(), Obj_.Size());
}

/****************************************************************************/
int* PromptBox::GetCtrlFilter(int& Len_)
{
  Len_ = _CtrlFilterLen;
  return _CtrlFilter;
}

/****************************************************************************/
// PURPOSE:
//   Set negative input filter method
//
// PRE:
//   The original value of the negative filter
//
// POST:
//   The negative input filter is set to argument string passed.
//
void PromptBox::SetNegFilter(char* str)
{
  if (str != _negfilter)
  {
    delete[] _negfilter;
    _negfilter = str ?
    strcpy(new char[::SafeStrLen(str)+1], str):NULL;

    if (_chainbox)
      _chainbox->SetNegFilter(_negfilter);
  }
}

/****************************************************************************/
void PromptBox::AddNegFilter(char* str)
{
  int i, Len_, FLen_, OldLen_;

  if (!str)
    return;

  Len_ = ::SafeStrLen(str);
  OldLen_ = FLen_ = ::SafeStrLen(_negfilter);

  char* news_ = strcpy(new char[Len_ + FLen_ + 1], _negfilter);

  for (i = 0; i < Len_; ++i)
    if (!memchr(news_, str[i], FLen_))
      news_[FLen_++] = str[i];

  if (FLen_ != OldLen_)
  {
    news_[FLen_] = 0;
    delete[] _negfilter;
    _negfilter = news_;
  }
  else
    delete[] news_;
}

/****************************************************************************/
void PromptBox::RemoveNegFilter(char* str)
{
  char *cptr_, *dptr_, *news_, *loc_;
  int i, Cnt_, Len_, FLen_, OldLen_;

  if (!str)
    return;

  Len_ = ::SafeStrLen(str);
  OldLen_ = FLen_ = ::SafeStrLen(_negfilter);
  dptr_ = cptr_ = news_ = _negfilter;

  for (Cnt_ = i = 0; i < Len_; ++i)
  {
    while (loc_ = (char*)memchr(cptr_, str[i], FLen_))
    {
      ++Cnt_;
      *loc_ = 0;
      dptr_ = cptr_;
      cptr_ = loc_ + 1;
      FLen_ -= (cptr_ - dptr_);

      if (FLen_ <= 0)
	break;
    }

    cptr_ = news_;
    FLen_ = OldLen_;
  }

  if (Cnt_)
  {
    _negfilter = new char[OldLen_ - Cnt_ + 1];

    for (FLen_ = i = 0; i < OldLen_; ++i)
      if (news_[i])
	_negfilter[FLen_++] = news_[i];

    _negfilter[FLen_] = 0;
    delete[] news_;
  }
}

/****************************************************************************/
void PromptBox::SetRestrictFullBox(int Flag_)
{
  _RestFullBox = Flag_;

  if (_chainbox)
    _chainbox->SetRestrictFullBox(Flag_);
}

/****************************************************************************/
void PromptBox::SetEndBrksActive(bool Flag_)
{
  _EndBrksActive = Flag_;

  if (_chainbox)
    _chainbox->SetEndBrksActive(Flag_);
}

/****************************************************************************/
void PromptBox::SetTrimWs(bool Flag_)
{
  _TrimWs = Flag_;

  if (_chainbox)
    _chainbox->SetTrimWs(Flag_);
}

/****************************************************************************/
void PromptBox::SetNoEmpty(int flag)
{
  _noempty = flag;

  if (_chainbox)
    _chainbox->SetNoEmpty(_noempty);
}

/****************************************************************************/
void PromptBox::NotifyInsertMode(const int* ModePtr_)
{
  if (ModePtr_)
    _InsertMode = *ModePtr_;

  iMonitor(_insert_monitor, _insert_handle, ModePtr_);
}

/****************************************************************************/
void PromptBox::SetInsertMode(int Mode_, PromptBox* Parent_)
{
  _InsertMode = Mode_;
  iMonitor(_insert_monitor, _insert_handle, &_InsertMode);

  if (Parent_)
    MonitorInsertMode(Parent_, InsertDetect);

  if (_chainbox)
    _chainbox->SetInsertMode(_InsertMode);
}

/****************************************************************************/
// PURPOSE:
//   Search filter method. Searches the filter string for the given
//   keyboard character
//
// PRE:
//   int key : the keyboard character to search for
//   char* fil : the filter string used in the search
//
// POST:
//   Returns 1 if the key character is found within the filter otherwise,
//   returns 0.
//
int PromptBox::SearchFilter(int key, char* fil)
{
  for (int index = 0; index < ::SafeStrLen(fil); index++)
    if (key == fil[index])
      return 1;

  return 0;
}

/****************************************************************************/
// PURPOSE:
//   Search filter method. Searches the filter string for the given
//   keyboard character
//
// PRE:
//   int key : the keyboard character to search for
//   char* fil : the filter string used in the search
//
// POST:
//   Returns 1 if the key character is found within the filter otherwise,
//   returns 0.
//
int PromptBox::SearchIntFilter(int key, int* fil, int Len_)
{
  for (int index = 0; index < Len_; index++)
    if (key == fil[index])
      return 1;

  return 0;
}

/****************************************************************************/
// PURPOSE:
//   Search positive filter method. Searches the filter string for the given
//   keyboard character. If SrchCtrl_ flag is specified also searches for the
//   key within the control key filter.
//
// PRE:
//   int key : the keyboard character to search for
//   Boolean SrchCtrl_ : Specifies whether or not to search for control keys
//			 in the control key filter.
//
// POST:
//   Returns 1 if the key character is found within the filter(s) otherwise,
//   returns 0.
//
int PromptBox::SearchPosFilter(int key, Boolean SrchCtrl_, Boolean DefaultVal_)
{
  if (SrchCtrl_)
    return (SearchIntFilter(key, _CtrlFilter, _CtrlFilterLen) ||
	    _Owner->GetControlInfo()->TestHotKey(key) ||
	    SearchPosFilter(key, FALSE, FALSE));

  return ((_filter && ::SafeStrLen(_filter)) ? SearchFilter(key, _filter):DefaultVal_);
}

/****************************************************************************/
void PromptBox::SetXYpos(int x, int y)
{
  SetXpos(x);
  SetYpos(y);
}

/****************************************************************************/
// PURPOSE:
//   key input validity check method
//
// PRE:
//   int key : The key input from the keyboard
//
// POST:
//   Returns 1 if the key is defined as valid, otherwise returns 0.
//
int PromptBox::TestKey(int key, Boolean SrchCtrl_)
{
  return
  (
    SearchPosFilter(key, SrchCtrl_, TRUE) &&
    ((_negfilter && ::SafeStrLen(_negfilter)) ? !SearchFilter(key, _negfilter):TRUE) &&
    (_testf ? _testf(key):TRUE)
  );
}

/****************************************************************************/
int PromptBox::IsValidCtrlKey(int key)
{
  return ((_CtrlFilter && _CtrlFilterLen) ? TestKey(key, TRUE):TRUE);
}

/****************************************************************************/
// PURPOSE:
//   input string validity check method
//
// PRE:
//   char* str : The original value of the input string
//
// POST:
//   Returns 1 if all the characters within the input string is defined
//   as valid, otherwise returns 0.
//
int PromptBox::IsValidString(char* str)
{
  for (int index = 0; index < ::SafeStrLen(str); index++)
    if (!IsValidKey(str[index]))
      return FALSE;

  return TRUE;
}

/****************************************************************************/
int PromptBox::IsValidLength() const
{
  return
  (
    (::SafeStrLen(_buffer) >= _minlen || !_noempty) &&
    (_chainbox ? _chainbox->IsValidLength():1)
  );
}

/****************************************************************************/
// PURPOSE:
//   Keyboard retrieval procedure
//
// PRE:
//   The original value of _key and _extended data members
//
// POST:
//   The keyboard is read for a key and is assigned to _key data member.
//   If the key is extended non-ASCII then the _extended data member is set.
//   Returns the key read from the keyboard.
//
int PromptBox::GetKey()
{
  _extkey = _key = (int)GETCH();
  iMonitor(_key_monitor, _key_handle, &_key);
  _extended = (_key == 0 || _key == 0xE0) ? 256:0;
  iMonitor(_ext_monitor, _ext_handle, &_extended);

  if (_extended)
  {
    _key = (int)GETCH();
    _extkey = _key + _extended;
    iMonitor(_key_monitor, _key_handle, &_key);
  }

  return _key;
}

/****************************************************************************/
// PURPOSE:
//   Method for putting extended non-ASCII keyboard keys to the console
//   output.
//
// PRE:
//   int key : the key to be outputted.
//
// POST:
//   The key is outputted to the screen. Appropriate changes to the
//   internal input buffer is made.
//
void PromptBox::PutExtKey(int key)
{
  int pos;

  if (!IsValidCtrlKey(key))
    return;

  // Action: TOGGLE_INSERT
  if (GetKeyMap().KeyMatch(key, INSERT))
    SetInsertMode(!GetInsertMode());

  // Action: GO_LEFT
  else if (GetKeyMap().KeyMatch(key, LEFT))
  {
    if (_index > 0)
    {
      GOTOXY(wherex()-1, _ypos);
      --_index;
      iMonitor(_index_monitor, _index_handle, &_index);
    }
  }

  // Action: GO_RIGHT
  else if (GetKeyMap().KeyMatch(key, RIGHT))
  {
    if (_index < ::SafeStrLen(_buffer) && _index < _len)
    {
      GOTOXY(wherex()+1, _ypos);
      ++_index;
      iMonitor(_index_monitor, _index_handle, &_index);
    }
  }

  // Action: GO_HOME
  else if (GetKeyMap().KeyMatch(key, HOME))
  {
    _index = 0;
    iMonitor(_index_monitor, _index_handle, &_index);
    GOTOXY(_xpos + ::SafeStrLen(_msg) + 1, _ypos);
  }

  // Action: GO_END
  else if (GetKeyMap().KeyMatch(key, END))
  {
    _index = ::SafeStrLen(_buffer);
    iMonitor(_index_monitor, _index_handle, &_index);
    GOTOXY(_xpos + ::SafeStrLen(_msg) + ::SafeStrLen(_buffer) + 1, _ypos);

    if (::SafeStrLen(_buffer) >= _len && _len > 0)
      if (_index > 0)
      {
        _index = _len;
        GOTOXY(wherex()-1, _ypos);
        --_index;
        iMonitor(_index_monitor, _index_handle, &_index);
      }
  }

  // Action: DELETE
  else if (GetKeyMap().KeyMatch(key, DELETE))
  {
    if (_index < _len)
    {
      Pull(_index);
      pos = wherex();
      ShowInput();
      GOTOXY(pos, _ypos);
    }
  }

  // Action: Go to item with matching hotkey
  else
  {
    if (key == _HotKey)
      _HotKeyStatus = HOTKEY;
    else
      _HotKeyStatus = NONE;
  }
}

/****************************************************************************/
// PURPOSE:
//   Method for putting standard ASCII keyboard keys to the console output.
//
// PRE:
//   int key : the key to be outputted.
//
// POST:
//   The key is outputted to the screen. Appropriate changes to the
//   internal input buffer is made.
//
void PromptBox::PutStdKey(int key)
{
  int pos;

  if (!IsValidCtrlKey(key))
    return;

  // Action: NEWLINE
  if (_InsertMode && _chainbox &&
      GetKeyMap().KeyMatch(key, ENTER))
  {
    if (_chainbox)
    {
      _chainbox->ShiftDown(&_buffer[_index]);
      if (_index < ::SafeStrLen(_buffer))
      {
	FillString(_buffer, '\0', _index, _len);

	pos = wherex();
	ShowInput();
	GOTOXY(pos, _ypos);
      }
    }
  }

  // Action: IF_GO_LEFT_DELETE
  else if (GetKeyMap().KeyMatch(key, BACK))
  {
    if (_index > 0)
    {
      PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::LEFTARROW).SCAN + 256);
      PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::DELETE).SCAN + 256);
      iMonitor(_index_monitor, _index_handle, &_index);
    }
  }

  // Action: PUTKEY
  else if (IsUserKey(key))
  {
    if (_index < _len)
    {
      int EntryOk_ = 1;
      if (!_buffer[_index])
	     _buffer[_index+1] = '\0';

      if (_InsertMode)
	     EntryOk_ = Push(_index, key);
      else
	     _buffer[_index] = key;

      if (EntryOk_)
      {
        PUTCH(key);
        pos = wherex();
        ShowInput();
        GOTOXY(pos, _ypos);
        ++_index;
      }
      
      iMonitor(_index_monitor, _index_handle, &_index);
    }
  }

  // Action: Go to item with matching shortcut and select it
  else
  {
    if (key == _ShortCut)
      _HotKeyStatus = SHORTCUT;
    else
      _HotKeyStatus = NONE;
  }
}

/****************************************************************************/
int PromptBox::IsUserKey(int key)
{
  return
  (
    !_extended &&
    !GetKeyMap().KeyMatch(key, BACK) &&
    !GetKeyMap().KeyMatch(key, ESCAPE) &&
    !GetKeyMap().KeyMatch(key, ENTER) &&
    !GetKeyMap().KeyMatch(key, NEXT) &&
    isprint(key) && IsValidKey(key)
  );
}

/****************************************************************************/
void PromptBox::SetChainPos(int Pos_)
{
  _ChainPos = Pos_;

  if (_chainbox)
    _chainbox->SetChainPos(ChainPosition() + 1);
}

/****************************************************************************/
PromptBox* PromptBox::Chain(PromptBox* next_)
{
  if (_chainbox)
    return _chainbox->Chain(next_);
  else
  {
    _chainbox = next_;

    if (_chainbox)
    {
      _chainbox->SetChainPos(ChainPosition() + 1);
      _chainbox->SetFilter(_filter);
      _chainbox->SetNegFilter(_negfilter);
      _chainbox->SetCtrlFilter(_CtrlFilter, _CtrlFilterLen);
      _chainbox->SetNoEmpty(_noempty);
      _chainbox->SetInsertMode(GetInsertMode(), this);

      _chainbox->MonitorBuffer(_buffer_handle, _buffer_monitor);
      _chainbox->MonitorKey(_key_handle, _key_monitor);
      _chainbox->MonitorExtendedKey(_ext_handle, _ext_monitor);
      _chainbox->MonitorIndex(_index_handle, _index_monitor);
      _chainbox->MonitorEscapedStatus(_escaped_handle, _escaped_monitor);
      _chainbox->MonitorInFocusStatus(_infocus_handle, _infocus_monitor);

      if (_chainminlen)
	_chainbox->SetMinimumLength(_chainminlen);

      if (_TextWidgetActive)
	_chainbox->Activate();
      else
	_chainbox->Deactivate();
    }
  }

  return this;
}

/****************************************************************************/
/****************************************************************************/
int PromptBox::IsFullBox() const
{
  int copylen_ = _buffer ? ::SafeStrLen(_buffer):0;
  char lastchar_ = (copylen_ == _len) ? _buffer[_len - 1]:0;
  return ((_chainbox && _chainbox->_buffer) ?
                      (lastchar_ && _chainbox->IsFullBox()):
                      (lastchar_ && !isspace(lastchar_)));
}

/****************************************************************************/
void PromptBox::ChainPush(char ch)
{
  if (_chainbox)
    _chainbox->Push(0, ch);
}

/****************************************************************************/
int PromptBox::Push(int start, char ch)
{
  if (!IsValidKey(ch))
    return 0;

  int copylen_ = ::SafeStrLen(_buffer);
  int atend_ = copylen_ == _len;
  char lastchar_ = atend_ ? _buffer[_len - 1]:0;
  atend_ = lastchar_ != 0;
  Boolean DoChain_ = atend_ &&
                     _chainbox && _chainbox->IsValidKey(lastchar_);
  Boolean FullBox_ = _chainbox ? _chainbox->IsFullBox():IsFullBox();

  if (atend_)
    --copylen_;

  copylen_ -= start;
  if (copylen_ < 0)
    copylen_ = 0;

  if (!_RestFullBox || !FullBox_ || !atend_)
  {
    if (copylen_)
    {
      memmove(&_buffer[start + 1], &_buffer[start], copylen_);
      _buffer[start + 1 + copylen_] = '\0';
    }

    _buffer[start] = ch;
    if (DoChain_)
      ChainPush(lastchar_);

    return 1;
  }

  return 0;
}

/****************************************************************************/
char PromptBox::ChainPull()
{
  if (_chainbox)
    return _chainbox->Pull(0);

  return '\0';
}

/****************************************************************************/
char PromptBox::Peek(int Index_)
{
  if (_chainbox)
    return _chainbox->Retrieve(Index_);

  return '\0';
}

/****************************************************************************/
char PromptBox::Pull(int start)
{
  int lastpos_;
  int copylen_ = ::SafeStrLen(_buffer);
  char firstchar_ = _buffer[start];
  char nextchar_;
  Boolean DoChain_;
  Boolean Pad_;

  copylen_ -= start;
  if (copylen_ < 0)
    copylen_ = 0;

  if (!copylen_)
  {
    if (_chainbox)
      for (;start < _len; start++)
      {
        _buffer[start] = ChainPull();
        if (!_buffer[start])
          break;
      }

    return 0;
  }
  else if (_buffer[start] && copylen_)
  {
    memmove(&_buffer[start], &_buffer[start + 1], copylen_);

    nextchar_ = Peek();
    DoChain_ = _chainbox && IsValidKey(nextchar_);
    Pad_ = (_buffer ? ::SafeStrLen(_buffer):0) < (_len - 1);

    if (Pad_ && _buffer)
    {
      lastpos_ = ::SafeStrLen(_buffer);
      copylen_ = (_len - 1) - lastpos_;
      ::memset(&_buffer[lastpos_], ' ', copylen_);
      Pad_ = ::SafeStrLen(_buffer) < _len - 1;
    }

    if (DoChain_ && !Pad_)
    {
      nextchar_ = ChainPull();

      _buffer[_len - 1] = nextchar_;
      _buffer[_len] = '\0';
    }
  }

  return firstchar_;
}

/****************************************************************************/
void PromptBox::ShiftDown(char* NewStr_)
{
  if (_chainbox)
    _chainbox->ShiftDown(_buffer);
  SetInputString(NewStr_);
}

/****************************************************************************/
void PromptBox::FillString(char* Str_, char FillCh_, int Start_, int End_)
{
  for (;Start_ < End_; Start_++)
    Str_[Start_] = FillCh_;

  Str_[End_] = '\0';
}

/****************************************************************************/
/****************************************************************************/
// PURPOSE:
//   End input test procedure
//
// PRE:
//   int key : the key to be tested as the termination key for input.
//
// POST:
//   Returns 1 if the key is defined as a termination key,
//   otherwise returns 0.
//
int PromptBox::ShouldEnd(int key, int zerotozero)
{
  int status_;

  if (!IsValidCtrlKey(key))
    return 0;

  // Action: GO_DOWN, ESCAPE
  if (GetKeyMap().KeyMatch(key, ESCAPE))
  {
    _Escaped = TRUE;
    iMonitor(_escaped_monitor, _escaped_handle, &_Escaped);
    status_ = 1;
  }

  // Action: GO_DOWN, IF_NOTCHAINED_ESCAPE
  else if (GetKeyMap().KeyMatch(key, ENTER))
  {
    _Escaped = _chainbox == NULL;
    iMonitor(_escaped_monitor, _escaped_handle, &_Escaped);
    status_ = 1;
  }

  // Action: GO_DOWN, GO_NEXT
  else if (GetKeyMap().KeyMatch(key, NEXT) || GetKeyMap().KeyMatch(key, DOWN))
    status_ = 1;

  // Action: GO_UP
  else if (GetKeyMap().KeyMatch(key, UP))
    status_ = -1;

  // Action: GO_BACK, GO_LEFT (Note Plural Actions For Single Key)
  else if (GetKeyMap().KeyMatch(key, BACK) || GetKeyMap().KeyMatch(key, LEFT))
    status_ = (_index <= 0 && zerotozero) ? -1:0;

  // Action: GO_RIGHT (Note Plural Actions For Single Key)
  else if (GetKeyMap().KeyMatch(key, RIGHT) || IsUserKey(key))
    status_ = (_index >= _len) ? 1:0;

  else if (_Owner->GetControlInfo()->HotKeyPressed() ||
	   _Owner->GetControlInfo()->TestHotKey(key))
  {
    // Action: Go to item with matching hotkey
    if (_HotKeyStatus == HOTKEY)
    {
      status_ = 0;
      _Owner->GetControlInfo()->ResetHotKeys();
    }
    else if (_HotKeyStatus == SHORTCUT)
    {
      status_ = IsSelectable();
      if (!status_)
	_Owner->GetControlInfo()->ResetHotKeys();
    }
    else
      status_ = 1;
  }

  // Otherwise, no action
  else
    status_ = 0;

  return status_;
}

/****************************************************************************/
// PURPOSE:
//   Get line input procedure
//
// PRE:
//   The original input buffer
//
// POST:
//   Gets input from the keyboard and place it in the input buffer until
//   a termination is press.
//
int PromptBox::GetLine(int status_)
{
  int lastindex_;
  int zerotozero_;

  ResetInputState(status_);

  cMonitor(_buffer_monitor, _buffer_handle, _buffer);
  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
  iMonitor(_active_monitor, _active_handle, &_TextWidgetActive);

  for (;;)
  {
    GetKey();
    lastindex_ = _index;

    if (_extended)
      PutExtKey(_extkey);
    else
      PutStdKey(_extkey);

    zerotozero_ = !lastindex_ && !_index;

    if (status_ = ShouldEnd(_extkey, zerotozero_))
      break;
  }

  return status_;
}

/****************************************************************************/
void PromptBox::ResetInputState(int status_)
{
  _Escaped = FALSE;
  _HotKeyStatus = NONE;
  _index = (status_ == -1) ? ::SafeStrLen(_buffer):0;

  if (status_ == -1 && ::SafeStrLen(_buffer) >= _len && _len > 0)
    _index = _len - 1;

  iMonitor(_escaped_monitor, _escaped_handle, &_Escaped);
  iMonitor(_index_monitor, _index_handle, &_index);
}

/****************************************************************************/
// PURPOSE:
//   Read input procedure
//
// POST:
//   Calls procedure to Show the contents of the input buffer as default
//   input to the Prompt box then calls GetLine to get additional input.
//   from the keyboard.
//
int PromptBox::ReadInput(int status_)
{
  ShowInput(status_);
  return GetLine(status_);
}

/****************************************************************************/
// PURPOSE:
//   Method to Erase the input
//
// POST:
//   Erases the internal buffer by NULL padding entire buffer.
//
void PromptBox::Erase()
{
  if (!IsActive())
    return;

  if (_chainbox)
    _chainbox->Erase();

  if (::SafeStrLen(_buffer))
  {
    if (!_Hidden)
    {
      _InFocus = TRUE;
      iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

      if (UsesColor())
        SetupColors(ColorInfo::BACKGROUND_COLOR);
      else
        UseColor(FALSE);

      ResetPrompt();
      _InFocus = FALSE;
      iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
    }

    for (int index = 0; index < _len; index++)
      _buffer[index] = '\0';
  }
}

/****************************************************************************/
// PURPOSE:
//   reset user Prompt method
//
// POST:
//   The user Prompt is setup at the specified position and the original
//   contents of the line is Cleared
//
void PromptBox::ResetPrompt()
{
  GOTOXY(_xpos + ::SafeStrLen(_msg), _ypos);
  PrintBlanks(_len + 2);
  GOTOXY(_xpos + ::SafeStrLen(_msg), _ypos);

  PUTCH(EndBrksActive() ? _brackets[0]:' ');
  GOTOXY(wherex() + _len, _ypos);
  PUTCH(EndBrksActive() ? _brackets[1]:' ');

  GOTOXY(_xpos + ::SafeStrLen(_msg) + 1, _ypos);
}

/****************************************************************************/
// PURPOSE:
//   Show Prompt message method
//
// POST:
//   Goes to the specified coordinates and Shows the Prompt message
//   to the screen
//
void PromptBox::ShowMessage()
{
  GOTOXY(_xpos, _ypos);

  if (_msg)
    if (_HotKey && UsesColor())
    {
      text_info txtinfo_;
      gettextinfo(&txtinfo_);
      int OldFg_ = txtinfo_.attribute & 0x0F;
      char* Temps_ = strncpy(new char[_HotIndex + 1], _msg, _HotIndex);
      Temps_[_HotIndex] = 0;

      CPUTS(Temps_);
      textcolor(HotKeyColor());
      PUTCH(_msg[_HotIndex]);
      textcolor(OldFg_);
      CPUTS(&_msg[_HotIndex + 1]);

      delete[] Temps_;
    }
    else
      CPUTS(_msg);
}

/****************************************************************************/
// PURPOSE:
//   Show input method
//
// POST:
//   Shows the Prompt box and Shows the original contents of the input
//   buffer within the box
//
void PromptBox::ShowInput(int status_)
{
  if (_len)
  {
    ResetPrompt();
    if (ShouldTrimWs() && _buffer)
      for (int x = ::SafeStrLen(_buffer) - 1; isspace(_buffer[x]) && x >= 0; x--)
        _buffer[x] = 0;
    
    CPUTS(_buffer);

    if (_chainbox)
      _chainbox->ShowInput();

    GOTOXY(_xpos + ::SafeStrLen(_msg) + 1, _ypos);

    if (status_ == -1)
    {
      GOTOXY(wherex() + ::SafeStrLen(_buffer), _ypos);

      if (::SafeStrLen(_buffer) >= _len && _len > 0)
      {
        _index = _len;
        PutExtKey((int)KeyboardCodes::CODETABLE(KeyboardCodes::LEFTARROW).SCAN + 256);
        iMonitor(_index_monitor, _index_handle, &_index);
      }
    }
  }
}

/****************************************************************************/
// PURPOSE:
//   Prompt user method
//
// POST:
//   Shows the Prompt message and the Prompt box then Prompts user for
//   additional input
//
int PromptBox::Prompt(int status_)
{
  if (!IsActive())
    return status_;

  if (_len)
  {
    _InFocus = TRUE;
    iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

    SetupColors(ColorInfo::HIGHLIGHT_COLORS);
    ShowMessage();
    status_ = ReadInput(status_);
    Show();

    _InFocus = FALSE;
    iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
    _Hidden = FALSE;
    iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
  }

  return status_;
}

/****************************************************************************/
// PURPOSE:
//   Shows Prompt box method
//
// POST:
//   Shows Prompt message and Prompt box with contents of input buffer
//
void PromptBox::Show()
{
  if (!IsActive())
    return;

  if (_chainbox)
    _chainbox->Show();

  _InFocus = TRUE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

  SetupColors(ColorInfo::NORMAL_COLORS);
  ShowMessage();
  ShowInput(1);

  _InFocus = FALSE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
  _Hidden = FALSE;
  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
}

/****************************************************************************/
void PromptBox::ShowInput()
{
  _InFocus = TRUE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

  ShowInput(1);

  _InFocus = FALSE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
  _Hidden = FALSE;
  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
}

/****************************************************************************/
// PURPOSE:
//   Hide Prompt box method
//
// POST:
//   goes to the specified coordinates of the Prompt box then Clears the
//   line containing the Prompt box.
//
void PromptBox::Hide()
{
  if (!IsActive())
    return;

  if (_chainbox)
    _chainbox->Hide();

  _InFocus = TRUE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);

  if (UsesColor())
    SetupColors(ColorInfo::BACKGROUND_COLOR);
  else
    UseColor(FALSE);

  GOTOXY(_xpos, _ypos);
  PrintBlanks(::SafeStrLen(_msg) + _len + 2);
  GOTOXY(_xpos, _ypos);

  _InFocus = FALSE;
  iMonitor(_infocus_monitor, _infocus_handle, &_InFocus);
  _Hidden = TRUE;
  iMonitor(_hidden_monitor, _hidden_handle, &_Hidden);
}

/****************************************************************************/
char PromptBox::AnyValidChar()
{
  // Try the blank character first
  if (IsValidKey(BLANK))
    return BLANK;

  // We assume the range of char to be 0 - 255
  for (Byte_t Index_ = 0; Index_ < 255; Index_++)
    if (IsValidKey(Index_))
      return Index_;

  return 0;
}

/****************************************************************************/
void PromptBox::SetEnterable(Boolean Flag_)
{
  _Enterable = Flag_;
}

/****************************************************************************/
void PromptBox::SetSelectable(Boolean Flag_)
{
  _Selectable = Flag_;
}

/****************************************************************************/
void PromptBox::Select()
{
  if (!IsSelected())
    if (_maxbuf)
      *_buffer = AnyValidChar();
}

/****************************************************************************/
void PromptBox::Deselect()
{
  if (_maxbuf)
    *_buffer = 0;
}

/****************************************************************************/
void PromptBox::Activate()
{
  _TextWidgetActive = TRUE;
  iMonitor(_active_monitor, _active_handle, &_TextWidgetActive);

  if (_chainbox)
    _chainbox->Activate();
}

/****************************************************************************/
void PromptBox::Deactivate()
{
  _TextWidgetActive = FALSE;
  iMonitor(_active_monitor, _active_handle, &_TextWidgetActive);

  if (_chainbox)
    _chainbox->Deactivate();
}

/****************************************************************************/
Boolean PromptBox::IsEmpty() const
{
  return (!_buffer || !::SafeStrLen(_buffer));
}

/****************************************************************************/
Boolean PromptBox::IsEscaped() const
{
  return _Escaped;
}

/****************************************************************************/
Boolean PromptBox::IsActive() const
{
  return _TextWidgetActive;
}

/****************************************************************************/
Boolean PromptBox::IsEnterable() const
{
  return _Enterable;
}

/****************************************************************************/
Boolean PromptBox::IsSelectable() const
{
  return _Selectable;
}

/****************************************************************************/
Boolean PromptBox::IsSelected() const
{
  return (_buffer && ::SafeStrLen(_buffer));
}

/****************************************************************************/
Boolean PromptBox::IsValid() const
{
  return (!(IsEmpty() && _noempty) && IsValidLength());
}

/****************************************************************************/
Boolean PromptBox::IsInFocus() const
{
  return _InFocus;
}

/****************************************************************************/
char PromptBox::LastKeyRead(int* Ext_) const
{
  if (Ext_)
    *Ext_ = _extended;

  return _key;
}

/****************************************************************************/
int PromptBox::LastExtKeyRead() const
{
  return _extkey;
}

/****************************************************************************/
char PromptBox::Retrieve(int Index_)
{
  if (_buffer && 0 <= Index_ && Index_ < ::SafeStrLen(_buffer))
    return _buffer[Index_];

  return 0;
}

/****************************************************************************/
Boolean PromptBox::IsOwner(ControlWindow* Pwin_) const
{
  return (_Owner == Pwin_);
}

/****************************************************************************/
void PromptBox::SetOwner(ControlWindow* Pwin_, TextControl* TxtCtrl_)
{
  if (_Owner && _Owner != Pwin_)
    _Owner->RemoveControl(TxtCtrl_);
  _Owner = Pwin_;
}

/****************************************************************************/
void PromptBox::SetupColors(int Mode_)
{
  if (!_ColorInfo.SetupColors(Mode_))
    GetDefaultColorInfo().SetupColors(Mode_);
}

/****************************************************************************/
void PromptBox::UseColor(Boolean Flag_)
{
  if (_ColorInfo._UseColor)
    *_ColorInfo._UseColor = Flag_;
  else
    _ColorInfo._UseColor = new Boolean(Flag_);

  GetDefaultColorInfo().UseColor(Flag_);
}

/****************************************************************************/
void PromptBox::SetNormalColors(int FgCol_, int BgCol_)
{
  if (_ColorInfo._NormalColors)
  {
    _ColorInfo._NormalColors[0] = FgCol_;
    _ColorInfo._NormalColors[1] = BgCol_;
  }
  else
  {
    _ColorInfo._NormalColors = new int[2];
    SetNormalColors(FgCol_, BgCol_);
  }
}

/****************************************************************************/
void PromptBox::SetHighLightColors(int FgCol_, int BgCol_)
{
  if (_ColorInfo._HighLightColors)
  {
    _ColorInfo._HighLightColors[0] = FgCol_;
    _ColorInfo._HighLightColors[1] = BgCol_;
  }
  else
  {
    _ColorInfo._HighLightColors = new int[2];
    SetHighLightColors(FgCol_, BgCol_);
  }
}

/****************************************************************************/
void PromptBox::SetHotKeyColor(int Col_)
{
  if (_ColorInfo._HkColor)
    *_ColorInfo._HkColor = Col_;
  else
    _ColorInfo._HkColor = new int(Col_);
}

/****************************************************************************/
void PromptBox::SetBackGroundColor(int Col_)
{
  if (_ColorInfo._BackGroundColor)
    *_ColorInfo._BackGroundColor = Col_;
  else
    _ColorInfo._BackGroundColor = new int(Col_);
}

/****************************************************************************/
void PromptBox::UseDefaultColors()
{
  _ColorInfo.ClearColors();
  UseColor();
}

/****************************************************************************/
Boolean PromptBox::UsesColor() const
{
  return
  (
    _ColorInfo._UseColor ?
      *(_ColorInfo._UseColor):
      *(GetDefaultColorInfo()._UseColor)
  );
}

/****************************************************************************/
int* PromptBox::NormalColors() const
{
  return
  (
    _ColorInfo._NormalColors ?
      _ColorInfo._NormalColors:
      GetDefaultColorInfo()._NormalColors
  );
}

/****************************************************************************/
int* PromptBox::HighLightColors() const
{
  return
  (
    _ColorInfo._HighLightColors ?
      _ColorInfo._HighLightColors:
      GetDefaultColorInfo()._HighLightColors
  );
}

/****************************************************************************/
int PromptBox::HotKeyColor() const
{
  return
  (
    _ColorInfo._HkColor ?
      *(_ColorInfo._HkColor):
      *(GetDefaultColorInfo()._HkColor)
  );
}

/****************************************************************************/
int PromptBox::BackGroundColor() const
{
  return
  (
    _ColorInfo._BackGroundColor ?
      *(_ColorInfo._BackGroundColor):
      *(GetDefaultColorInfo()._BackGroundColor)
  );
}

/****************************************************************************/
/****************************************************************************/
// PURPOSE:
//   Prompt window default constructor
//
// PRE:
//   an uninitialize Prompt window
//
// POST:
//   a Prompt window is constructed and assigned the default values
//   specified.
//
PromptWindow::PromptWindow(int Max_):
_pbox(NULL),
_Info(new ControlInfo(Max_)),
_maxbox(Max_),
_boxcount(0)
{
  MaxControl(Max_);
}

/****************************************************************************/
// PURPOSE:
//   Prompt window copy construcor
//
// PRE:
//   An uninitialized Prompt window.
//
// POST:
//   a Prompt window is constructed and assigned the corresponding values
//   in the Prompt window argument passed to the constructor.
//
PromptWindow::PromptWindow(PromptWindow& pw):
_Info(new ControlInfo(pw._maxbox)),
_maxbox(pw._maxbox),
_boxcount(pw._boxcount)
{
  _pbox = new TextControl*[_maxbox];
  for (int index = 0; index < _maxbox; index++)
    if (pw[index])
    {
      _pbox[index] = PromptBox::Make(*pw[index]);
      SetControlData(_pbox, _Info, index);
    }
    else
      RemoveControlData(_pbox, _Info, index);
}

/****************************************************************************/
int PromptWindow::FindControlCount()
{
  int index;
  int cnt = 0;

  if (_pbox && _maxbox)
  {
    for (index = 0; index < _maxbox; index++)
      if (_pbox[index])
        ++cnt;

    _boxcount = cnt;
  }

  return cnt;
}

/****************************************************************************/
void PromptWindow::RemoveControlData(TextControl** CtrlArray_,
				     ControlInfo* CtrlInfo_, int index)
{
  CtrlArray_[index] = NULL;
  CtrlInfo_->SetHotKeys(index, NULL);
}

/****************************************************************************/
void PromptWindow::SetControlData(TextControl** CtrlArray_,
				  ControlInfo* CtrlInfo_, int index)
{
  CtrlArray_[index]->SetOwner(this, CtrlArray_[index]);
  CtrlInfo_->SetHotKeys(index, CtrlArray_[index]);
}

/****************************************************************************/
// PURPOSE:
//   Prompt window data member destruction method
//
// PRE:
//   The data members of this Prompt window object with original values
//
// POST:
//   If the data members of the window object is not NULL then the data
//   members are deleted and assigned NULL. The array of Prompt boxes
//   are destroyed.
//
void PromptWindow::Clear()
{
  if (_pbox)
  {
    Delete();

    delete[] _pbox;
    _pbox = NULL;
  }

  if (_Info)
  {
    delete _Info;
    _Info = NULL;
  }
}

/****************************************************************************/
TextControl* PromptWindow::Control(int Index_)
{
  return _pbox[Index_];
}

/****************************************************************************/
void PromptWindow::Control(int Index_, TextControl* TxtCtrl_)
{
  _pbox[Index_] = TxtCtrl_;

  if (_pbox[Index_])
    SetControlData(_pbox, _Info, Index_);
}

/****************************************************************************/
// PURPOSE:
//   Sets the maximum Prompt box limit on the box array
//
// PRE:
//   The previous size of the Prompt box array
//
// POST:
//   The size of the Prompt box array is set to the specified maximum
//   All contents of the array are Cleared first before resizing the array
//
void PromptWindow::MaxControl(int max)
{
  if (_pbox)
  {
    Delete();
    delete[] _pbox;
    _pbox = NULL;
  }

  _maxbox = max;

  if (max)
  {
    _pbox = new TextControl*[_maxbox];
    for (int index = 0; index < _maxbox; index++)
      RemoveControlData(_pbox, _Info, index);
  }
}

/****************************************************************************/
int PromptWindow::MaxControl()
{
  return _maxbox;
}

/****************************************************************************/
// PURPOSE:
//   Method to Grow the size of the array by some number passed as an
//   argument to the method. The array contents are preserved.
//
// PRE:
//   The old size of the Prompt box array and its contents
//
// POST:
//   The array size is Grown by the extra size given and the original
//   contents of the array are preserved in the new array.
//
void PromptWindow::Grow(int extra)
{
  int index;
  int newlen = _maxbox + extra;

  TextControl** newp = new TextControl*[newlen];
  ControlInfo* newi = new ControlInfo(newlen);

  for (index = 0; index < _maxbox; index++)
  {
    newp[index] = PromptBox::Make(*((PromptBox*)_pbox[index]));
    SetControlData(newp, newi, index);
  }

  for (;index < newlen; index++)
    RemoveControlData(newp, newi, index);

  Clear();

  _maxbox = newlen;
  _pbox = newp;
  _Info = newi;
}

/****************************************************************************/
Boolean PromptWindow::AnyActive()
{
  for (int Index_ = 0; Index_ < _maxbox; Index_++)
    if (_pbox[Index_] && _pbox[Index_]->IsActive())
      return TRUE;

  return FALSE;
}

/****************************************************************************/
Boolean PromptWindow::ValidityCheck()
{
  for (int Index_ = 0; Index_ < _maxbox; Index_++)
    if (_pbox[Index_] && _pbox[Index_]->IsActive() &&
	!_pbox[Index_]->IsValid())
      return FALSE;

  return TRUE;
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and Prompt the user with
//   each one.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box element
//   within the array is used to Prompt the user.
//
int PromptWindow::Prompt(int StartIndex_)
{
  int index = (StartIndex_ >= _maxbox) ? 0:
	      (StartIndex_ < 0)        ? (_maxbox - 1):
					 StartIndex_;
  int status_ = 1;

  if (AnyActive())
    for (;;)
    {
      if (_pbox[index] && _pbox[index]->IsActive())
      {
	status_ = _pbox[index]->Prompt(status_);
	Boolean hkeypressed_ = _Info->HotKeyPressed();

	if (hkeypressed_)
	{
	  if (status_ != 0)
	    index = _Info->GetIndex();

	  _Info->ResetHotKeys();
	}

	if (!hkeypressed_ || !status_)
	{
	  if (_pbox[index]->IsEscaped())
	    if (ValidityCheck())
	      break;
	    else
	      status_ = 0;

	  index += status_;
	}
      }
      else
	++index;

      index = (index >= _maxbox) ? 0:
	      (index < 0) 	 ? (_maxbox - 1):
				   index;
    }

  return index;
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and Show to the screen
//   each one.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box element
//   within the array is Shown to the screen.
//
void PromptWindow::Show()
{
  for (int index = 0; index < _maxbox; index++)
    if (_pbox[index] && _pbox[index]->IsActive())
      _pbox[index]->Show();
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and Clear each one from the
//   screen.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box element
//   within the array is Cleared from the screen.
//
void PromptWindow::Hide()
{
  for (int index = 0; index < _maxbox; index++)
    if (_pbox[index] && _pbox[index]->IsActive())
      _pbox[index]->Hide();
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and Erase each Prompt box.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box element
//   within the array is Erased.
//
void PromptWindow::Erase()
{
  for (int index = 0; index < _maxbox; index++)
    if (_pbox[index] && _pbox[index]->IsActive())
      _pbox[index]->Erase();
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and delete each Prompt box.
//   This method does not depend on whether the text control is in an
//   active state or not. All controls regardless of their active state will
//   be deleted.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box deleted
//
void PromptWindow::Delete()
{
  for (int index = 0; index < _maxbox; index++)
    if (_pbox[index] && _pbox[index]->IsOwner(this))
    {
      delete _pbox[index];
      RemoveControlData(_pbox, _Info, index);
    }
}

/****************************************************************************/
// PURPOSE:
//   Cycle through the entire Prompt box array and remove each Prompt box.
//   This method does not depend on whether the text control is in an
//   active state or not. All controls regardless of their active state will
//   be removed.
//
// POST:
//   The Prompt box array is cycled through and each Prompt box removed,
//   but not deleted or modified
//
void PromptWindow::Remove()
{
  TextControl* TxtCtrl_;

  for (int index = 0; index < _maxbox; index++)
    if (_pbox[index] && _pbox[index]->IsOwner(this))
    {
      TxtCtrl_ = _pbox[index];
      TxtCtrl_->SetOwner(NULL, TxtCtrl_);
      _Info->SetHotKeys(index, NULL);
    }
}

/****************************************************************************/
// PURPOSE:
//   Prompt box Index operator. To index individual Prompt boxes from the
//   Prompt window given the position of the Prompt box within the array.
//
// PRE:
//   int index : the index position of the Prompt box in the array
//
// POST:
//   Go into the specified index position in the Prompt box array and
//   Retrieve its element.
//
PromptWindowRef PromptWindow::operator [] (int index)
{
  ASSERT((index >= 0 && index < _maxbox), USER_Message_t::ERRMSG_INDEXRANGE);

  PromptWindowRef PboxRef(this, index);
  return PboxRef;
}

/****************************************************************************/
PromptBox* PromptWindow::NextSelected(int& Start_)
{
  Boolean Found_ = FALSE;

  for (;Start_ < _maxbox; Start_++)
    if (Found_ = _pbox[Start_] != NULL && _pbox[Start_]->IsSelected())
      break;

  if (Found_)
  {
    ++Start_;
    return (*this)[Start_ - 1];
  }

  return NULL;
}

/****************************************************************************/
void PromptWindow::DeleteControl(TextControl* TxtCtrl_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index] == TxtCtrl_ && _pbox[index]->IsOwner(this))
      break;

  if (index < _maxbox)
  {
    delete _pbox[index];
    RemoveControlData(_pbox, _Info, index);
  }
}

/****************************************************************************/
void PromptWindow::RemoveControl(TextControl* TxtCtrl_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index] == TxtCtrl_ && _pbox[index]->IsOwner(this))
      break;

  if (index < _maxbox)
    RemoveControlData(_pbox, _Info, index);
}

/****************************************************************************/
ControlInfo* PromptWindow::GetControlInfo()
{
  return _Info;
}

/****************************************************************************/
void PromptWindow::UseColor(Boolean Flag_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index])
      _pbox[index]->UseColor(Flag_);
}

/****************************************************************************/
void PromptWindow::SetNormalColors(int FgCol_, int BgCol_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index])
      _pbox[index]->SetNormalColors(FgCol_, BgCol_);
}

/****************************************************************************/
void PromptWindow::SetHighLightColors(int FgCol_, int BgCol_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index])
      _pbox[index]->SetHighLightColors(FgCol_, BgCol_);
}

/****************************************************************************/
void PromptWindow::SetHotKeyColor(int Col_)
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index])
      _pbox[index]->SetHotKeyColor(Col_);
}

/****************************************************************************/
void PromptWindow::UseDefaultColors()
{
  int index;
  for (index = 0; index < _maxbox; index++)
    if (_pbox[index])
      _pbox[index]->UseDefaultColors();
}

/****************************************************************************/
/****************************************************************************/
PromptWindowRef::PromptWindowRef(PromptWindow* Boss_, int Index_):
_BossPtr(Boss_),
_Index(Index_)
{}

/****************************************************************************/
PromptWindowRef& PromptWindowRef::operator = (PromptBox* Pbox_)
{
  _BossPtr->Control(_Index, Pbox_);
  return *this;
}

/****************************************************************************/
PromptWindowRef::operator PromptBox* ()
{
  return CastToPromptBox(_BossPtr->Control(_Index));
}

/****************************************************************************/
PromptBox* PromptWindowRef::operator -> ()
{
  return CastToPromptBox(_BossPtr->Control(_Index));
}

/****************************************************************************/
/****************************************************************************/
#if PROMPTBOX_DEBUG
int main()
{
  CLRSCR();

  PromptBox* pb;
  PromptWindow pw;

  pw.MaxControl(1);

  pb->GetKeyMap()[TextControl::UP].Add(KeyboardCodes::CODETABLE(KeyboardCodes::F1).SCAN, TRUE);
  pb->GetKeyMap()[TextControl::LEFT].Add(KeyboardCodes::CODETABLE(KeyboardCodes::F2).SCAN, TRUE);
  pb->GetKeyMap()[TextControl::RIGHT].Add(KeyboardCodes::CODETABLE(KeyboardCodes::F3).SCAN, TRUE);
  pb->GetKeyMap()[TextControl::DOWN].Add(KeyboardCodes::CODETABLE(KeyboardCodes::F4).SCAN, TRUE);

  pb = PromptBox::Make("Enter an &integer : ");
  pb->SetFilter("+-0123456789");
  pb->SetInputLength(15);
  pb->SetInputString("10");
  pb->SetXpos(3);
  pb->SetYpos(2);
  pw[0] = pb;

  pw.Grow(3);

  pb = PromptBox::Make("Enter a &word : \n");
  pb->SetFilter(isgraph);
  pb->SetInputLength(20);
  pb->SetInputString("ACE");
  pb->SetXpos(3);
  pb->SetYpos(4);
  pw[1] = pb;

  pb = PromptBox::Make("Enter 2nd wo&rd : ");
  pb->SetFilter(isgraph);
  pb->SetRoundBrackets();
  pb->SetInputLength(20);
  pb->SetInputString("char");
  pb->SetXpos(3);
  pb->SetYpos(6);
  pw[2] = pb;

  // chaining box 2 to box 3
  ((PromptBox*)pw[1])->Chain(pw[2]);
  ((PromptBox*)pw[1])->PushInputString("HelloWorld");

  pb = PromptBox::Make("Enter a &floating point : ");
  pb->SetFilter("+-0123456789.");
  pb->SetNoEmpty();
  pb->SetInputLength(20);
  pb->SetInputString("BEE");
  pb->SetXpos(3);
  pb->SetYpos(8);
  pw[3] = pb;

  int index;

  pw.UseColor();
  pw.Show();
  pw.Prompt();
  pw.UseColor(FALSE);
  pw.Hide();

  GOTOXY(1, 10);
  for (index = 0; index < pw.MaxControl(); index++)
  {
    pb = pw[index];
    cout <<pb->Retrieve() <<endl;
    GETCH();
  }

// Setting All promptbox message to NULL
  pw[0]->SetMessage(NULL);
  pw[1]->SetMessage(NULL);
  pw[2]->SetMessage(NULL);
  pw[3]->SetMessage(NULL);
  
  pw.UseColor();
  pw.Show();
  pw.Prompt();
  pw.UseColor(FALSE);
  pw.Hide();

  GOTOXY(1, 10);
  for (index = 0; index < pw.MaxControl(); index++)
  {
    pb = pw[index];
    cout <<pb->Retrieve() <<endl;
    GETCH();
  }

  return 0;
}

#endif
/****************************************************************************/
#endif

