// Prompt box and prompt window ADT classes
// The prompt box class define methods for maintaining a prompt box object
// used for prompting a user for input from the console. The prompt box class
// also has methods for filtering the input to define and disallow invalid
// input, positioning procedures for setting the display position of the
// prompt box, setting the default prompt message and default input into
// the prompt box, showing, hiding, prompting, erasing and retrieval of
// the input data.
//
// The prompt window class define methods for maintaining an array of prompt
// boxes. Methods are defined to show, hide, prompt, erase and retrieve from
// the entire array of prompt boxes assigned to the prompt window object.
//
#ifndef PROMPTBOX_H
#define PROMPTBOX_H
/****************************************************************************/
#ifndef CTRLDEFS_H
  #include "ctrldefs.h"
#endif

#define PROMPTBOX_DEBUG		0

/****************************************************************************/
class PromptWindow;

class PromptBox : public TextControl
{
  friend const int* InsertDetect(void* Owner_, PromptBox*, const int* Ptr_);
  friend class PromptWindow;

  protected:
    enum { BLANK = ' ', MAXBUFFER = 128 };

    ColorInfo _ColorInfo;
    ControlWindow* _Owner;
    PromptBox* _chainbox;

    char _brackets[2];
    int _HotKey;			// Hot Letter for Alt-key shifting
    int _ShortCut;			// Shortcut Letter for Ctrl-key shifting
    int _HotIndex;			// Index of Hot Letter in label
    int _HotKeyStatus;			// Indicate which hot key was pressed
    Boolean _Enterable;       // Control could be entered into
    Boolean _Selectable;		// Indicate whether control can be
					// selected like a button or checkbox
    char* _msg;				// prompt message buffer
    char* _filter;			// input filter string
    char* _negfilter;			// negative input filter string
    int* _CtrlFilter;			// Control keys filter
    int _CtrlFilterLen;			// Control keys filter length
    char* _buffer;			// input buffer

    int _maxbuf;			// maximum size of input buffer
					// not including NULL terminator
    int _key;			       	// input key
    int _extkey;			// extended input key
    int _extended;			// extended key indicator
    int _index;				// input buffer index
    int _len;				// maximum input length
    int _minlen;			// minimum input length
    int _chainminlen;			// remaindered min length for chained box
    int _xpos;				// horizontal coordinate position
    int _ypos;				// vertical coordinate position
    int _noempty;			// no input disallow flag
    int _InsertMode;			// insert mode
    int _Escaped;			// prompt box Escaped
    int _Hidden;			// Text dialog hidden flag
    int _TextWidgetActive;		// Active flag for this text widget
    int _ChainPos;			// The position within the chain
    int _InFocus;			// Text widget in focus flag
    
    int _RestFullBox;   // Further input into full box is disallowed
    int _EndBrksActive; // Ending brackets active at left-right ends
    int _TrimWs;        // Trim white space from input line (default: false)

    int _SavedCursX;			// Saved cursor x position
    int _SavedCursY;			// Saved cursor y position

    int(*_testf)(int);			// character set function filter

    // Internal data monitoring function pointers
    const int*(*_insert_monitor)(void*, PromptBox*, const int*);    // internal use
    const char*(*_buffer_monitor)(void*, PromptBox*, const char*);
    const int*(*_key_monitor)(void*, PromptBox*, const int*);
    const int*(*_ext_monitor)(void*, PromptBox*, const int*);
    const int*(*_index_monitor)(void*, PromptBox*, const int*);
    const int*(*_escaped_monitor)(void*, PromptBox*, const int*);
    const int*(*_hidden_monitor)(void*, PromptBox*, const int*);
    const int*(*_active_monitor)(void*, PromptBox*, const int*);
    const int*(*_infocus_monitor)(void*, PromptBox*, const int*);

    // Handles to external objects, used by data monitors
    void* _insert_handle;
    void* _buffer_handle;
    void* _key_handle;
    void* _ext_handle;
    void* _index_handle;
    void* _escaped_handle;
    void* _hidden_handle;
    void* _active_handle;
    void* _infocus_handle;

    // Monitor functions execution and management methods
    const char* cMonitor(const char*(*monptr_)(void*, PromptBox*, const char*), void* This_, const char* ptr_);
    const int* iMonitor(const int*(*monptr_)(void*, PromptBox*, const int*), void* This_, const int* ptr_);
    void SaveCursorPosition();
    void RestoreCursorPosition();

    const int* MonitorInsertMode(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*));

    // Promptbox chaining internal methods
    void SetChainPos(int Pos_);

    // Insert state methods
    void NotifyInsertMode(const int* ModePtr_);
    void SetInsertMode(int Mode_, PromptBox* Parent_=NULL);
    inline int GetInsertMode()
	{ return _InsertMode; }

    // any valid char retrieval method
    char AnyValidChar();

    // show prompt box methods
    virtual void ShowMessage();
    virtual void ShowInput(int status_);
    virtual void ResetPrompt();

    // Clear prompt box method
    void PrintBlanks(int Max_);

    // input validation / testing methods
    int* WordChr(int* src_, int val_, size_t n);
    int SearchFilter(int key, char* fil);
    int SearchIntFilter(int key, int* fil, int Len_);
    int SearchPosFilter(int key, Boolean SrchCtrl_, Boolean DefaultVal_);
    int TestKey(int key, Boolean SrchCtr_);
    int IsValidString(char* str);
    int IsValidLength() const;
    int IsUserKey(int key);
    int ShouldEnd(int key, int zerotozero);
    inline int IsValidKey(int key)
	{ return TestKey(key, FALSE); }
    int IsValidCtrlKey(int key);

    // console input retrieval methods
    int GetKey();
    inline int GetExtKey()
	{ return _extkey; }
    int GetLine(int status_);
    int ReadInput(int status_);
    void ResetInputState(int status_);
    void SetEnterable(Boolean Flag_);
    void SetSelectable(Boolean Flag_);

    // keyboard character output methods
    void PutExtKey(int key);
    void PutStdKey(int key);

    // String push/pull methods
    void FillString(char* Str_, char FillCh_, int Start_, int End_);
    void ShiftDown(char* NewStr_);
    void ChainPush(char ch);
    char ChainPull();
    int Push(int start, char ch=' ');
    char Pull(int start);
    char Peek(int Index_=0);

    // data member destruction method
    virtual void Clear();

    // promptbox constructors
    PromptBox(char* msg, PromptBox* next_, int Size_, int Length_=0);
    PromptBox(PromptBox& pb, int Size_, int Length_=0);

    // Text color methods
    void SetupColors(int Mode_);

  public:
    // PromptBox Destructor
    virtual ~PromptBox()
	{ Clear(); }

    // Promptbox chaining method
    PromptBox* Chain(PromptBox* next_);
    inline int ChainPosition()
	{ return _ChainPos; }
    inline void EraseChain()
	{ _chainbox = NULL; }
    inline PromptBox* NextChain()
	{ return _chainbox; }

    // promptbox assignment operator
    virtual PromptBox& operator = (PromptBox& pb);
    static PromptBox* Make(char* msg=NULL, PromptBox* next_=NULL, int Size_=0, int Length_=0);
    static PromptBox* Make(PromptBox& pbox, int Size_=0, int Length_=0);
    static PromptBox* MakeNextChain(PromptBox* pb);

    // prompt box properties mutators
    void SetRoundBrackets();
    void SetSquareBrackets();
    virtual char* SetMessage(const char* str);
    virtual void RemoveMessage();
    virtual void BlankMessage();
    virtual void SetInputString(char* str);
    virtual void PushInputString(char* str);
    virtual void SetInputLength(int len);
    virtual int InputLength() const;
    virtual void DivBufSizeAsInputLength(int div_);
    virtual void SetMinimumLength(int len);
    virtual char* SetHotKeys(char* str);
    virtual void RemoveHotKeys();
    void SetNegFilter(char* str);
    void AddNegFilter(char* str);
    void RemoveNegFilter(char* str);
    void SetFilter(char* str);
    void AddFilter(char* str);
    void RemoveFilter(char* str);
    void SetCtrlFilter(const int* str, int Len_);
    void SetCtrlFilter(const KeyNode& Obj_);
    void AddCtrlFilter(const int* str, int Len_);
    void AddCtrlFilter(const KeyNode& Obj_);
    void RemoveCtrlFilter(const int* str, int Len_);
    void RemoveCtrlFilter(const KeyNode& Obj_);
    void SetNoEmpty(int flag=1);
    void SetXYpos(int x, int y);
    inline void SetFilter(int(*fncp)(int))
	{ _testf = fncp; }
    inline void SetXpos(int x)
	{ _xpos = x; }
    inline void SetYpos(int y)
	{ _ypos = y; }
    void SetRestrictFullBox(int Flag_);

    void SetEndBrksActive(bool Flag_);
    void SetRestrictFullBox(bool Flag_);
    void SetTrimWs(bool Flag_);

    // Prompt box properties accessors
    inline int RestrictFullBox() const
   { return _RestFullBox; }
    inline bool EndBrksActive() const
   { return _EndBrksActive; }
    inline bool ShouldTrimWs() const
   { return _TrimWs; }
    inline char* GetFilter()
	{ return _filter; }
    inline char* GetNegFilter()
	{ return _negfilter; }
    int* GetCtrlFilter(int& Len_);
    int IsFullBox() const;

    // prompt box manipulators
    virtual void Show();
    virtual void ShowInput();
    virtual void Hide();
    virtual int Prompt(int status_);
    virtual void Erase();

    // text control selection/deselection mutators
    virtual void Select();
    virtual void Deselect();

    // Empty, valid and selected Text Control test
    virtual Boolean IsEnterable() const;
    virtual Boolean IsSelectable() const;
    virtual Boolean IsSelected() const;
    virtual Boolean IsEmpty() const;
    virtual Boolean IsEscaped() const;
    virtual Boolean IsValid() const;
    virtual Boolean IsActive() const;
    virtual Boolean IsInFocus() const;

    // Activation/Deactivation for this text widget
    virtual void Activate();
    virtual void Deactivate();

    // Accelerator key methods
    virtual int GetHotKey() const;
    virtual int GetShortCut() const;
    virtual int GetHotKeyType() const;

    // input data retrieval method
    char LastKeyRead(int* Ext_=NULL) const;
    int LastExtKeyRead() const;
    char Retrieve(int Index_);
    inline char* GetMessage()
	{ return _msg; }
    inline char* Retrieve()
	{ return _buffer; }
    inline int HotKeyStatus()
	{ return _HotKeyStatus; }

    // Text control ownership methods
    virtual void SetOwner(ControlWindow* Pwin_, TextControl* TxtCtrl_);
    virtual Boolean IsOwner(ControlWindow* Pwin_) const;

    // Text color methods
    virtual void UseColor(Boolean Flag_=TRUE);
    virtual void SetNormalColors(int FgCol_, int BgCol_);
    virtual void SetHighLightColors(int FgCol_, int BgCol_);
    virtual void SetHotKeyColor(int Col_);
    virtual void SetBackGroundColor(int Col_);
    virtual void UseDefaultColors();

    virtual Boolean UsesColor() const;
    virtual int* NormalColors() const;
    virtual int* HighLightColors() const;    
    virtual int HotKeyColor() const;
    virtual int BackGroundColor() const;

    // Internal data monitoring methods
    const char* MonitorBuffer(void* This_, const char*(*FnPtr_)(void*, PromptBox*, const char*));
    const int* MonitorKey(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*));
    const int* MonitorExtendedKey(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*));
    const int* MonitorIndex(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*));
    const int* MonitorEscapedStatus(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*));
    const int* MonitorHiddenStatus(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*));
    const int* MonitorActiveStatus(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*));
    const int* MonitorInFocusStatus(void* This_, const int*(*FnPtr_)(void*, PromptBox*, const int*));
};

/****************************************************************************/
/****************************************************************************/
class PromptWindowRef;

class PromptWindow : public ControlWindow
{
  friend class PromptWindowRef;

  protected:
    TextControl** _pbox;       	// prompt box array
    ControlInfo* _Info;
    int _maxbox;		// maximum number of prompt boxes
    int _boxcount;   // total non-null pointers to prompt boxes stored

    // Total number of valid control objects stored within storage array
    virtual int FindControlCount();

    // Text Control accessor method
    virtual void Control(int Index_, TextControl* TxtCtrl_);
    virtual TextControl* Control(int Index_);

    // Dialog controls validity check method
    virtual Boolean AnyActive();
    virtual Boolean ValidityCheck();

    // data member destruction method
    virtual void Clear();

    // convenience methods
    void SetControlData(TextControl** CtrlArray_, ControlInfo* CtrlInfo_, int index);
    void RemoveControlData(TextControl** CtrlArray_, ControlInfo* CtrlInfo_, int index);

  public:
    // prompt window constructors
    PromptWindow(int Max_=0);
    PromptWindow(PromptWindow& pw);
    virtual ~PromptWindow()
	{ Clear(); }

    // prompt window data mutators
    virtual void MaxControl(int max);
    virtual void Grow(int extra);

    // data member accessors
    virtual int MaxControl();
    virtual void RemoveControl(TextControl* TxtCtrl_);
    virtual void DeleteControl(TextControl* TxtCtrl_);

    // Promptbox specific accessors
    PromptWindowRef operator [] (int index);
    PromptBox* NextSelected(int& Start_);

    // prompt window manipulators
    virtual void Show();
    virtual void Hide();
    virtual int Prompt(int StartIndex_=0);
    virtual void Erase();
    virtual void Delete();
    virtual void Remove();

    // Text color methods
    virtual void UseColor(Boolean Flag_=TRUE);
    virtual void SetNormalColors(int FgCol_, int BgCol_);
    virtual void SetHighLightColors(int FgCol_, int BgCol_);
    virtual void SetHotKeyColor(int Col_);
    virtual void UseDefaultColors();

    // Retrieve control information object
    virtual ControlInfo* GetControlInfo();
};

/****************************************************************************/
class PromptWindowRef
{
  friend class PromptWindow;
  private:
    PromptWindow* _BossPtr;
    int _Index;

    PromptWindowRef(PromptWindow* Boss_, int Index_);
    static PromptBox* CastToPromptBox(TextControl* TxtCtrl_)
	{ return ((PromptBox*)TxtCtrl_); }

  public:
    PromptWindowRef& operator = (PromptBox* Pbox_);
    operator PromptBox* ();
    PromptBox* operator -> ();
};

/****************************************************************************/
#endif

