#ifndef SIMPLEARRAY_H
#define SIMPLEARRAY_H
#ifndef BUILTIN_H
  #include "builtin.h"
#endif
#ifndef INDEX_H
  #include "index.h"
#endif

#define SIMPLEARRAY_DEBUG       0
#define SIMPLEARRAY_DEBUG1      0
#define SIMPLEARRAY_DEBUG1a     0

#define SIMPARR_ARRAYTESTS      0
#define SIMPARR_ARRITERTESTS    0
#define SIMPARR_VECTORTESTS     0
#define SIMPARR_VECTITERTESTS   0
#define SIMPARR_MATRIXTESTS     0
#define SIMPARR_3DMATRIXTESTS   0

#if SIMPLEARRAY_DEBUG
  #ifndef SORTS_CPP
    #include "sorts.cpp"
  #endif
#endif

/****************************************************************************/
// SimpleVector template class forward declarations
template <class T>
class SimpleVector;

/****************************************************************************/
// SimpleMatrix template class forward declarations
template <class T>
class SimpleMatrix;

/****************************************************************************/
// Simple3DMatrix template class forward declarations
template <class T>
class Simple3DMatrix;

/****************************************************************************/
class VectorInfo
{
  public:
    enum
    {    
      NONE     = 0,
      ARRAY    = 1,
      VECTOR   = 2,
      MATRIX   = 3,
      MATRIX3D = 4,

      MTOSTRANS = 100,
      STOMTRANS = 110,
      STARTSC   = 120,
      STARTMO   = 140,
      MAXTRANS  = 4,

      DEFAULT_SIZE = 10
    };

    // class identification
    virtual bool IsClassType(int Type_) const = 0;
    virtual bool IsMultiDimType() const = 0;
};

/****************************************************************************/
template <class P, class B>
class SimpleBaseArray : public VectorInfo
{
  friend class SimpleVector<B>;

  public:
    typedef int(*CompFuncType)(const B&, const B&);

  protected:
    Boolean _Error;
    mutable size_t _Size;
    P* _Buffer;
    bool _UseMemMgr;

    static CompFuncType _CompFunc;

    // constructors
    SimpleBaseArray();
    SimpleBaseArray(size_t n, bool UseMemMgr_=true);
    SimpleBaseArray(const P* a, size_t n, bool UseMemMgr_=true);
    SimpleBaseArray(const SimpleBaseArray<P,B>& Arr_,
                    size_t start_, size_t end_, bool UseMemMgr_=true);
    SimpleBaseArray(const SimpleBaseArray<P,B>& Arr_, bool UseMemMgr_=true);
    
    // concatenate constructor
    SimpleBaseArray(const SimpleBaseArray<P,B> & a1,
                    const SimpleBaseArray<P,B> & a2,
                    bool UseMemMgr_=true);

  public:
    virtual ~SimpleBaseArray();

    // item comparison function pointer setting method
    static void SetComparisonFunction(int(*CompFunc_)(const B&, const B&));

    // static distance method
    static size_t FindDist(size_t start_, size_t end_,
                           const SimpleBaseArray<P,B>& Arr_);

    // error set method
    void SetError(Boolean Flag_=TRUE)
        { _Error = Flag_; }
    
    // assignment operator
    SimpleBaseArray<P,B>& operator = (const SimpleBaseArray<P,B>& a);

    // conversion operator
    operator const P* () const;

    // conversion operator
    operator P* ();

    // raw C array retrieval methods
    P* Base()
        { return _Buffer; }
    const P* Base() const
        { return _Buffer; }
    
    // get number of elements
    size_t GetCount() const;

    // index operators
    P& operator [] (const Index & i);
    P& operator [] (int i);
    const P operator [] (const Index & i) const;
    const P operator [] (int i) const;

    const P Read(const Index & i) const;
    const P Read(int i) const;

    // fill functions
    void Fill(P i);
    void FillFrom(int i, P v);
    void FillFrom(const Index& i, P v);

    // array growing operation
    void GrowArray(size_t Incr_=10);

    // memory manager methods
    inline void UseMemMgr(bool f)
        { _UseMemMgr = f; }
    inline bool UseMemMgr() const
        { return _UseMemMgr; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif    
};

/****************************************************************************/
template <class T>
class SimpleArray : private SimpleBaseArray<T,T>
{
  friend SimpleArray<Boolean> SeriesArrayCompare(const SimpleArray<T> &ia1, const SimpleArray<T> &ia2);
  friend SimpleArray<T> Apply(const SimpleArray<T> & ia, T (* func)(T));

  friend Boolean operator == (const SimpleArray<T>& Obj1_, const SimpleArray<T>& Obj2_);
  friend Boolean operator != (const SimpleArray<T>& Obj1_, const SimpleArray<T>& Obj2_);
  friend Boolean operator > (const SimpleArray<T>& Obj1_, const SimpleArray<T>& Obj2_);
  friend Boolean operator < (const SimpleArray<T>& Obj1_, const SimpleArray<T>& Obj2_);
  friend Boolean operator >= (const SimpleArray<T>& Obj1_, const SimpleArray<T>& Obj2_);
  friend Boolean operator <= (const SimpleArray<T>& Obj1_, const SimpleArray<T>& Obj2_);

  EXTERNALCLASSFNCS_TEMPLATE_DECL(SimpleArray, SimpleArray)

  private:
    // static members for single instance object creation
    static SimpleArray<T>* _StaticClassPtr;
    static bool _StaticIsBuiltIn;
    static size_t m_ObjectCnt;
    static bool m_AutoRelease;
    static bool m_StaticCreate;
    static bool m_StaticDestroy;    
    static void Init();
    size_t _ObjectID;

  public:
    // static members for single instance object creation
    static SimpleArray<T>* Instance();
    static void Release();

  public:
    // type definitions
    typedef int(*CompFuncType)(const T&, const T&);
  
    // constructors
    SimpleArray();
    SimpleArray(size_t n);
    SimpleArray(const T* a, size_t n);
    SimpleArray(const SimpleArray<T>& Arr_, size_t start_, size_t end_);
    SimpleArray(const SimpleArray<T>& Arr_);
    
    // concatenate constructor
    SimpleArray(const SimpleArray<T> & a1,
                const SimpleArray<T> & a2);

    virtual ~SimpleArray();

    // Class Identification methods
    static int StaticID()
	{ return TypeInfo::SIMPLEARRAY_TYPE; }
    Boolean IsCommonClass(int IDvalue_) const;
    virtual int ClassID() const;
    const char* ClassName() const;

    // class identification
    virtual bool IsClassType(int Type_) const;
    virtual bool IsMultiDimType() const;    

    // Object Identification method
    virtual size_t ObjectID() const;

    // data type identification
    static bool IsBuiltInTypeData();

    // item comparison function pointer setting method
    static void SetComparisonFunction(int(*CompFunc_)(const T&, const T&));

    // error set method
    void SetError(Boolean Flag_=TRUE)
        { _Error = Flag_; }
    
    // assignment operator
    SimpleArray<T>& operator = (const SimpleArray<T>& a);

    // conversion operator
    operator const T* () const;

    // raw C array retrieval methods
    T* Base()
        { return _Buffer; }
    const T* Base() const
        { return _Buffer; }
    
    // get number of elements
    size_t GetCount() const
        { return _Size; }

    // index operators
    virtual T& operator [] (const Index & i);
    virtual T& operator [] (int i);
    virtual const T& operator [] (const Index & i) const;
    virtual const T& operator [] (int i) const;

    virtual const T Read(const Index & i) const;
    virtual const T Read(int i) const;

    // comparison operators
    virtual SimpleArray<Boolean> SeriesArrayCompare(const SimpleArray<T> &ia);
    virtual Boolean IsEqual(const SimpleArray<T> & ia) const;
    virtual Boolean IsLesser(const SimpleArray<T> & ia) const;
    virtual Boolean IsGreater(const SimpleArray<T> & ia) const;

    // apply function to elements
    virtual SimpleArray<T>& Apply(T(*func)(T));

    // fill functions
    void Fill(T i);
    void FillFrom(int i, T v);
    void FillFrom(const Index& i, T v);

    // array growing operation
    virtual void GrowArray(size_t Incr_=10);
    virtual size_t RunLength();

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

    CLONEFNCS_TEMPLATE_DECL(SimpleArray)
};

/****************************************************************************/
template <class T>
class SimpleArrayIter
{
  protected:
    SimpleArray<T>& _Target; // reference to target array
    T* _ElemFirst;     // address of first element
    T* _ElemLast;      // address of last  element
    T* _ElemPtr;       // pointer to element
    
  public:
    // constructors
    SimpleArrayIter(SimpleArray<T>& a);
    SimpleArrayIter(const SimpleArrayIter<T>& aptr);
    // ~SimpleArrayIter();

    // assignment operators
    SimpleArrayIter<T>& operator = (const SimpleArrayIter<T> & aptr);
    SimpleArrayIter<T>& operator = (const Index& i);
    SimpleArrayIter<T>& operator = (size_t i);

    // null pointer method
    bool IsNull() const;

    // conversion operator
    T& operator * ();

    // increment and decrement
    SimpleArrayIter<T>& operator ++ ();          // prefix
    SimpleArrayIter<T>  operator ++ (int dummy); // postfix
    SimpleArrayIter<T>& operator -- ();          // prefix
    SimpleArrayIter<T>  operator -- (int dummy); // postfix

    // set to first or last element
    void SetFirst();
    void SetLast();

    // value check functions
    int IsFirst();
    int IsLast();

    // comparison operators
    int operator == (const SimpleArrayIter<T> & aptr);
    int operator != (const SimpleArrayIter<T> & aptr);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif    
};

/****************************************************************************/
template <class T>
class SimpleVectorIter
{
  protected:
    SimpleVector<T>& _Target; // reference to target array
    T** _ElemFirst;     // address of first element
    T** _ElemLast;      // address of last  element
    T** _ElemPtr;       // pointer to element
    
  public:
    // constructors
    SimpleVectorIter(SimpleVector<T>& a);
    SimpleVectorIter(const SimpleVectorIter<T>& aptr);
    // ~SimpleVectorIter();

    // assignment operators
    SimpleVectorIter<T>& operator = (const SimpleVectorIter<T> & aptr);
    SimpleVectorIter<T>& operator = (const Index& i);
    SimpleVectorIter<T>& operator = (size_t i);

    // null pointer method
    bool IsNull() const;

    // conversion operator
    T*& operator * ();

    // increment and decrement
    SimpleVectorIter<T>& operator ++ ();          // prefix
    SimpleVectorIter<T>  operator ++ (int dummy); // postfix
    SimpleVectorIter<T>& operator -- ();          // prefix
    SimpleVectorIter<T>  operator -- (int dummy); // postfix

    // set to first or last element
    void SetFirst();
    void SetLast();

    // value check functions
    int IsFirst();
    int IsLast();

    // comparison operators
    int operator == (const SimpleVectorIter<T> & aptr);
    int operator != (const SimpleVectorIter<T> & aptr);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif    
};

/****************************************************************************/
#if SIMPLEARRAY_DEBUG
#ifndef STRUTIL_H
  #include "strutil.h"
#endif

class SimpleStr
{
  DEFAULT_ROOTOBJECT_DEFN(SimpleStr)

  friend ostream & operator << (ostream & buffer, SimpleStr& s)
      { buffer <<s.Str(); return buffer; }

  private:
    char* _str;

  public:
    SimpleStr():
      _str(NULL) {}
    SimpleStr(const char* str_):
      _str(new_char_string(str_)) {}
    ~SimpleStr()
      { DeleteArray(_str); }

    SimpleStr& operator = (const char* str_)
      { DeleteArray(_str);
        _str = new_char_string(str_);
        return *this; }

    char* Str()
      { return _str; }
};
#endif

/****************************************************************************/
template <class T, class C>
class PointerReturner
{
  private:
    T** _Data;
    T** _Base;
    C* _Container;
    size_t _Index;
    size_t _Size;
    int _TransCode;

  public:
    PointerReturner();
    PointerReturner(C* Cont_, size_t Index_, size_t Size_,
                    int Code_, T** Ptr_=NULL);
    PointerReturner(const PointerReturner<T,C>& Obj_);

    PointerReturner<T,C>& operator = (PointerReturner<T,C>& Obj_);
    PointerReturner<T,C>& operator = (T* Val_);

    operator T* ();
};

/****************************************************************************/
template <class T>
class PointerCopyNode
{
  public:
    typedef PointerCopyNode<T>* NPTYPE;

  private:
    T* _Data;
    PointerCopyNode<T>* _Next;

    void SetNext(PointerCopyNode<T>* Ptr_);

  public:
    PointerCopyNode();
    PointerCopyNode(T* Ptr_);
    ~PointerCopyNode();

    bool HasThis(T* Val_) const;
    bool HasThis(T* Val_, PointerCopyNode<T>::NPTYPE* Node_);
    PointerCopyNode<T>* Push(T* Val_);
    PointerCopyNode<T>* Pop(PointerCopyNode<T>*& NewHead_);
    PointerCopyNode<T>* Swap(PointerCopyNode<T>* Head_);

    inline T* Data()
        { return _Data; }
    inline PointerCopyNode<T>* Next()
        { return _Next; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

template <class T, class D>
class PointerCopyTracker
{
  friend class SimpleVector<D>;
  friend class PointerReturner<D,PointerCopyTracker>;

  private:
    // Containing object can either have a linked list node of vector
    // objects or have a pointer that references the master copy node
    //
    PointerCopyNode<T>* _Head; // Head of linked list of vector objects
    T* _Parent;                // Parent of PointerCopyTracker object
    T* _Master;  // Pointer to master list node of vector objects
    D** _Data;   // Shallow copy of parent array or newly assigned data
    
    size_t _Size;      // Size of data vector
    bool _SelfErase;   // Self erase object upon last node removal
    bool _EraseData;   // Flag to indicate data deletion allowed
    bool _OwnsData;    // The pointer tracker object owns the data it stores
    bool _ShallowCopy; // Shallow copy of master data
    bool _DataCreated; // Data array created from this object
    bool _Unlinked;    // Master object unlinked and destroyed

    bool ValidAttribTrans(bool Od_, bool Se_, bool Sc_,
                          bool NewOd_, bool NewSe_, bool NewSc_,
                          bool TransOd_, bool TransSe_, bool TransSc_) const;

    void EraseElement(D** Data_, size_t Index_, size_t sz);
    void DoErase(MemAllocMethod Method_, D** Data_, size_t sz);
    void EraseData();
    bool PtrUnlinked(T* Ptr_);
    T* MasterPtr();
    
    D** Data()
        { return _Data; }
    D** Buffer()
        { return (_Parent ? _Parent->Base():NULL); }

  public:
    PointerCopyTracker(T* Parent_);
    PointerCopyTracker(const PointerCopyTracker<T,D>& Obj_);
    ~PointerCopyTracker();

    void SetDataRefs(D** Element_, size_t Index_);
    void RemoveCopyRefs(T* Ptr_);
    void AppendCopyRefs(T* Ptr_, bool ChainAppend_);
    void RetCallToShallow(bool Unlinked_=true, bool InitCall_=false);
    void SetMaster(T* Ptr_, bool OwnsData_=false);    
    void SetData(D** Data_, size_t sz, bool AutoErase_=false);
    void SetOwnershipAttrib(bool OwnsData_, bool SelfErase_, bool Shallow_);
    int IsTransitionAttrib(int TransCode_) const;

    bool HasCopyRefs() const;
    bool HasThis(T* ToFind_) const;
    bool HasThis(T* ToFind_, PointerCopyNode<T>::NPTYPE* Node_);
    T* Push(T* Val_);
    T* Swap(T* Head_, PointerCopyNode<T>* Node_=NULL);
    T* Pop();

    void CheckUnlinked();
    void UntrackCopyRefs();
    D** InitDataArray(D** Data_, size_t sz);

    inline void SetSelfErase(bool f)
        { _SelfErase = f; }
    inline void SetEraseData(bool f)
        { _EraseData = f; }
        
    inline bool SelfErase() const
        { return _SelfErase; }
    inline bool EraseData() const
        { return _EraseData; }
    inline bool OwnsData() const
        { return _OwnsData; }
    inline bool ShallowCopy() const
        { return _ShallowCopy; }
    inline bool IsUnlinked() const
        { return _Unlinked; }
    inline bool IsParentUnlinked() const
        { return (_Parent ? _Parent->IsUnlinked():false); }
    inline bool IsMasterUnlinked() const
        { return (_Master ? _Master->IsUnlinked():false); }
        
    inline PointerCopyNode<T>* Head()
        { return _Head; }
    inline bool IsOwnershipAttrib(bool OwnsData_, bool SelfErase_, bool Shallow_) const
        { return (_OwnsData == OwnsData_ &&
                  _SelfErase == SelfErase_ &&
                  _ShallowCopy == Shallow_); }

    int IsOwnershipAttribExt(bool OwnsData_, bool SelfErase_, bool Shallow_,
                             int TransCode_) const;
    bool IsValidTrans(int TransCode_, bool TransCond_) const;
        

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
template <class T>
class SimpleVector : private SimpleBaseArray<T*,T>
{
  public:
    typedef SimpleVector<T> TTYPE;
    typedef T DTYPE;
    typedef PointerCopyTracker<TTYPE,DTYPE> CTYPE;
    typedef PointerCopyNode<TTYPE> NTYPE;

  friend class SimpleMatrix<T>;
  friend class Simple3DMatrix<T>;
  
  friend class PointerCopyTracker<TTYPE,DTYPE>;

  friend SimpleVector<Boolean> SeriesArrayCompare(const SimpleVector<T> &ia1, const SimpleVector<T> &ia2);
  friend SimpleVector<T> Apply(const SimpleVector<T>& ia, T(*func)(T));

  friend Boolean operator == (const SimpleVector<T>& Obj1_, const SimpleVector<T>& Obj2_);
  friend Boolean operator != (const SimpleVector<T>& Obj1_, const SimpleVector<T>& Obj2_);
  friend Boolean operator > (const SimpleVector<T>& Obj1_, const SimpleVector<T>& Obj2_);
  friend Boolean operator < (const SimpleVector<T>& Obj1_, const SimpleVector<T>& Obj2_);
  friend Boolean operator >= (const SimpleVector<T>& Obj1_, const SimpleVector<T>& Obj2_);
  friend Boolean operator <= (const SimpleVector<T>& Obj1_, const SimpleVector<T>& Obj2_);

  EXTERNALCLASSFNCS_TEMPLATE_DECL(SimpleVector, SimpleVector)

  private:
    // static members for single instance object creation
    static SimpleVector<T>* _StaticClassPtr;
    static int* _StaticTrIndex;
    static bool _StaticIsBuiltIn;
    static size_t m_ObjectCnt;
    static bool m_AutoRelease;
    static bool m_StaticCreate;
    static bool m_StaticDestroy;    
    static void Init();
    size_t _ObjectID;

  public:
    // static members for single instance object creation
    static int* Instance();
    static void Release();

  protected:
    static bool _AutoErase;    
    static MemAllocMethod _AllocMethod;
    
    mutable size_t _Length;
    mutable size_t _Index;

    PointerCopyTracker<TTYPE,DTYPE>* _CopyTracker;
    bool _Unlinked;
    bool _CpTrkInit;

    // data references for matrix objects
    static int TrIndex(int i);
    PointerCopyTracker<SimpleVector<T>::TTYPE,SimpleVector<T>::DTYPE>* CopyTrackPtr();
    const PointerCopyTracker<SimpleVector<T>::TTYPE,SimpleVector<T>::DTYPE>* CopyTrackPtr() const;
    void SetDataRefs(T** Data_, size_t Index_);
    void SetDataReferences(SimpleMatrix<T>* Ptr_, size_t i);
    SimpleVector<T>::DTYPE** AssignDataArray(SimpleVector<T>::DTYPE** Vect_, size_t sz);
    void ReplaceCopyTracker(PointerCopyTracker<SimpleVector<T>::TTYPE,SimpleVector<T>::DTYPE>* Ptr_,
                            SimpleVector<T>::TTYPE* Master_);

    T* AllocData(const T* ptr);
    bool HasThis(SimpleVector<T>::TTYPE* Val_) const;
    bool HasThis(SimpleVector<T>::TTYPE* Val_, SimpleVector<T>::NTYPE** Node_);
    SimpleVector<T>::TTYPE* Push(SimpleVector<T>::TTYPE* Val_);
    SimpleVector<T>::TTYPE* Swap(SimpleVector<T>::TTYPE* Head_, SimpleVector<T>::NTYPE* Node_=NULL);
    SimpleVector<T>::TTYPE* Pop();

  public:
    // type definitions
    typedef int(*CompFuncType)(const T&, const T&);

    // constructors
    SimpleVector();
    SimpleVector(size_t n);
    SimpleVector(const T** a, size_t n);
    SimpleVector(const SimpleVector<T>& Arr_, size_t start_, size_t end_);
    SimpleVector(const SimpleVector<T>& Arr_);

    // concatenate constructor
    SimpleVector(const SimpleVector<T> & a1,
                 const SimpleVector<T> & a2);

    virtual ~SimpleVector();

    // Class Identification methods
    static int StaticID()
	{ return TypeInfo::SIMPLEVECTOR_TYPE; }
    Boolean IsCommonClass(int IDvalue_) const;
    virtual int ClassID() const;
    const char* ClassName() const;

    // class identification
    virtual bool IsClassType(int Type_) const;
    virtual bool IsMultiDimType() const;

    // Object Identification method
    virtual size_t ObjectID() const;

    // set master object for shallow copied pointers
    void RetCallToShallow(bool Unlinked_=true, bool InitCall_=false);
    void SetMaster(SimpleVector<T>::TTYPE* Ptr_, bool OwnsData_=false);
    void SetConstMaster(const SimpleVector<T>::TTYPE* Ptr_, bool OwnsData_=false);
    void AppendCopyRefs(SimpleVector<T>::TTYPE* Ptr_, bool ChainAppend_);
    void AppendConstCopyRefs(const SimpleVector<T>::TTYPE* Ptr_, bool ChainAppend_);
    bool IsOwnershipAttrib(bool OwnsData_, bool SelfErase_, bool Shallow_) const;

    // data type identification
    static bool IsBuiltInTypeData();

    // memory allocation method for auto-erase
    static void SetAllocationMethod(MemAllocMethod Method_);

    // item comparison function pointer setting method
    static void SetComparisonFunction(int(*CompFunc_)(const T&, const T&));

    // auto erase vector content upon object destruction
    static void SetAutoErase(bool f, SimpleVector<T>* This_);
    void SetAutoErase(bool f);
    void SetDataErase(bool f);

    // object instance creation methods
    static SimpleVector<T>* Make();
    static SimpleVector<T>* Make(size_t n);
    static SimpleVector<T>* Make(const T** a, size_t n);
    static SimpleVector<T>* Make(const SimpleVector<T>& Arr_, size_t start_, size_t end_);
    static SimpleVector<T>* Make(const SimpleVector<T>& Arr_);
    static SimpleVector<T>* Make(const SimpleVector<T> & a1, const SimpleVector<T> & a2);

    // error set method
    inline void SetError(Boolean Flag_=TRUE)
        { _Error = Flag_; }
    
    // raw C array retrieval methods
    inline T** Base()
        { return _Buffer; }
    inline T** const Base() const
        { return _Buffer; }
    
    // get number of elements
    inline size_t GetCount() const
        { return _Size; }
    inline bool IsUnlinked() const
        { return _Unlinked; }

    // assignment operators
    virtual SimpleVector<T>& operator = (const SimpleVector<T>& Vect_);
    virtual SimpleVector<T>& DeepCopy(const SimpleVector<T>& Vect_);
    virtual SimpleVector<T>& ShallowCopy(const SimpleVector<T>& Vect_);

    // index operators
    virtual PointerReturner<SimpleVector<T>::DTYPE,SimpleVector<T>::CTYPE>
            operator [] (const Index& i);
    virtual PointerReturner<SimpleVector<T>::DTYPE,SimpleVector<T>::CTYPE>
            operator [] (int i);

    virtual const T* Read(const Index & i) const;
    virtual const T* Read(int i) const;
    const T* IndexedElement() const;
    T* IndexedElement();

    virtual void GrowVector(size_t Incr_=10);
    virtual void SetIndex(size_t Index_);
    virtual void EraseVector(MemAllocMethod Method_=MEMMATRIX);
    virtual void FindVectorRunLength();
    virtual size_t RunLength();

    // comparison operators
    virtual SimpleArray<Boolean> SeriesArrayCompare(const SimpleVector<T> &ia);
    virtual Boolean IsEqual(const SimpleVector<T> & ia) const;
    virtual Boolean IsLesser(const SimpleVector<T> & ia) const;
    virtual Boolean IsGreater(const SimpleVector<T> & ia) const;

    // apply function to elements
    virtual SimpleVector<T>& Apply(T(*func)(T));

    // fill functions
    virtual void Fill(T* v);
    virtual void FillFrom(int i, T* v);
    virtual void FillFrom(const Index& i, T* v);

    inline MemAllocMethod AllocMethod() const
        { return _AllocMethod; }
    inline bool AutoErase() const
        { return _AutoErase; }
    inline size_t Length() const
        { return _Length; }
    inline size_t Capacity() const
        { return GetCount(); }
    inline size_t CurrentIndex() const
        { return _Index; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

    CLONEFNCS_TEMPLATE_DECL(SimpleVector)
};

/****************************************************************************/
template <class T>
class SimpleMatrix
{
  public:
    typedef SimpleMatrix<T> TTYPE;
    typedef SimpleVector<T> DTYPE;
    typedef PointerCopyTracker<TTYPE,DTYPE> CTYPE;
    typedef PointerCopyNode<TTYPE> NTYPE;

  friend class Simple3DMatrix<T>;
  friend class PointerCopyTracker<TTYPE,DTYPE>;
  
  friend class SimpleVector<T>;
  friend class SimpleBaseArray<T,T>;

  EXTERNALCLASSFNCS_TEMPLATE_DECL(SimpleMatrix, SimpleMatrix)

  private:
    // static members for single instance object creation
    static SimpleMatrix<T>* _StaticClassPtr;
    static int* _StaticTrIndex;
    static bool _StaticIsBuiltIn;
    static size_t m_ObjectCnt;
    static bool m_AutoRelease;
    static bool m_StaticCreate;
    static bool m_StaticDestroy;
    static void Init();
    size_t _ObjectID;

  public:
    // static members for single instance object creation
    static int* Instance();
    static void Release();

  protected:
    static bool _AutoErase;
    static MemAllocMethod _AllocMethod;

    Boolean _Error;
    SimpleVector<T>** _Matrix;
  
    mutable size_t** _VectorSizes;
    mutable size_t** _VectorLengths;
    mutable size_t** _VectorIndexes;

    mutable size_t _MatrixSize;
    mutable size_t _MatrixLength;
    mutable size_t _MatrixIndex;

    PointerCopyTracker<TTYPE,DTYPE>* _CopyTracker;
    bool _Unlinked;
    bool _CpTrkInit;

    static int TrIndex(int i);
    PointerCopyTracker<SimpleMatrix<T>::TTYPE,SimpleMatrix<T>::DTYPE>* CopyTrackPtr();
    const PointerCopyTracker<SimpleMatrix<T>::TTYPE,SimpleMatrix<T>::DTYPE>* CopyTrackPtr() const;
    void SetDataRefs(SimpleVector<T>** Data_, size_t Index_);
    void SetDataReferences(Simple3DMatrix<T>* Ptr_, size_t i);
    SimpleMatrix<T>::DTYPE** AssignDataArray(SimpleMatrix<T>::DTYPE** Vect_, size_t sz);
    void ReplaceCopyTracker(PointerCopyTracker<SimpleMatrix<T>::TTYPE,SimpleMatrix<T>::DTYPE>* Ptr_,
                            SimpleMatrix<T>::TTYPE* Master_);

    bool HasThis(SimpleMatrix<T>::TTYPE* Val_) const;
    bool HasThis(SimpleMatrix<T>::TTYPE* Val_, SimpleMatrix<T>::NTYPE** Node_);
    SimpleMatrix<T>::TTYPE* Push(SimpleMatrix<T>::TTYPE* Val_);
    SimpleMatrix<T>::TTYPE* Swap(SimpleMatrix<T>::TTYPE* Head_, SimpleMatrix<T>::NTYPE* Node_=NULL);
    SimpleMatrix<T>::TTYPE* Pop();

  public:
    SimpleMatrix(size_t msz=VectorInfo::DEFAULT_SIZE, size_t vsz=0);
    SimpleMatrix(size_t msz, const T** a, size_t n);
    SimpleMatrix(const SimpleMatrix<T>& Arr_, size_t start_, size_t end_);
    SimpleMatrix(const SimpleMatrix<T>& Arr_);

    // concatenate constructor
    SimpleMatrix(const SimpleMatrix<T> & a1,
                 const SimpleMatrix<T> & a2);

    virtual ~SimpleMatrix();

    // Class Identification methods
    static int StaticID()
	{ return TypeInfo::SIMPLEMATRIX_TYPE; }
    Boolean IsCommonClass(int IDvalue_) const;
    virtual int ClassID() const;
    const char* ClassName() const;

    // Object Identification method
    virtual size_t ObjectID() const;

    // set master object for shallow copied pointers
    void RetCallToShallow(bool Unlinked_=true, bool InitCall_=false);
    void SetMaster(SimpleMatrix<T>* Ptr_, bool OwnsData_=false);
    void SetConstMaster(const SimpleMatrix<T>* Ptr_, bool OwnsData_=false);
    void AppendCopyRefs(SimpleMatrix<T>::TTYPE* Ptr_, bool ChainAppend_);
    void AppendConstCopyRefs(const SimpleMatrix<T>::TTYPE* Ptr_, bool ChainAppend_);
    bool IsOwnershipAttrib(bool OwnsData_, bool SelfErase_, bool Shallow_) const;

    // data type identification
    static bool IsBuiltInTypeData();

    // memory allocation method for auto-erase
    static void SetAllocationMethod(MemAllocMethod Method_);

    // auto erase matrix content upon object destruction
    static void SetAutoErase(bool f, SimpleMatrix<T>* This_);
    void SetAutoErase(bool f);
    void SetDataErase(bool f);

    // object instance creation methods
    static SimpleMatrix<T>* Make();
    static SimpleMatrix<T>* Make(size_t n);
    static SimpleMatrix<T>* Make(const T** a, size_t n);
    static SimpleMatrix<T>* Make(const SimpleMatrix<T>& Arr_, size_t start_, size_t end_);
    static SimpleMatrix<T>* Make(const SimpleMatrix<T>& Arr_);
    static SimpleMatrix<T>* Make(const SimpleMatrix<T> & a1, const SimpleMatrix<T> & a2);

    // class identification
    virtual bool IsClassType(int Type_) const;
    virtual bool IsMultiDimType() const;

    // error set method
    inline void SetError(Boolean Flag_=TRUE)
        { _Error = Flag_; }
    
    // conversion operator
    const SimpleVector<T>* IndexedElement() const;
    SimpleVector<T>* IndexedElement();
    
    // raw C array retrieval methods
    inline SimpleVector<T>** Base()
        { return _Matrix; }
    inline const SimpleVector<T>** Base() const
        { return _Matrix; }
    
    // get number of elements
    inline size_t GetCount() const
        { return _MatrixSize; }
    inline bool IsUnlinked() const
        { return _Unlinked; }

    // assignment operator
    virtual SimpleMatrix<T>& operator = (const SimpleMatrix<T>& Matr_);
    virtual SimpleMatrix<T>& DeepCopy(const SimpleMatrix<T>& a);
    virtual SimpleMatrix<T>& ShallowCopy(const SimpleMatrix<T>& a);

    // index operators
    virtual PointerReturner<SimpleMatrix<T>::DTYPE,SimpleMatrix<T>::CTYPE>
            operator [] (const Index& i);
    virtual PointerReturner<SimpleMatrix<T>::DTYPE,SimpleMatrix<T>::CTYPE>
            operator [] (int i);

    virtual const SimpleVector<T>* Read(const Index & i) const;
    virtual const SimpleVector<T>* Read(int i) const;

    virtual void GrowMatrix(size_t Incr_=10);
    virtual void SetMatrixIndex(size_t Index_);
    virtual void EraseVector(MemAllocMethod Method_=MEMMATRIX);
    virtual void FindVectorRunLength();
    virtual size_t VectorRunLength();

    virtual void GrowVector(size_t Incr_=10);
    virtual void SetVectorIndex(size_t Index_);
    virtual void EraseMatrix(MemAllocMethod Method_=MEMMATRIX);
    virtual void FindMatrixRunLength();
    virtual size_t RunLength();
    
    size_t VectorLength(size_t Index_) const;
    size_t VectorSize(size_t Index_) const;
    size_t VectorIndex(size_t Index_) const;

    // fill functions
    virtual void Fill(T* v);
    virtual void FillFrom(int ix, int iy, T* v);
    virtual void FillFrom(const Index& ix, const Index& iy, T* v);

    inline MemAllocMethod AllocMethod() const
        { return _AllocMethod; }
    inline bool AutoErase() const
        { return _AutoErase; }
    inline size_t Length() const
        { return _MatrixLength; }
    inline size_t Capacity() const
        { return _MatrixSize; }
    inline size_t CurrentIndex() const
        { return _MatrixIndex; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

    CLONEFNCS_TEMPLATE_DECL(SimpleMatrix)
};

/****************************************************************************/
template <class T>
class Simple3DMatrix
{
  public:
    typedef Simple3DMatrix<T> TTYPE;
    typedef SimpleMatrix<T> DTYPE;
    typedef PointerCopyTracker<TTYPE,DTYPE> CTYPE;
    typedef PointerCopyNode<TTYPE> NTYPE;

  friend class PointerCopyTracker<TTYPE,DTYPE>;

  friend class SimpleMatrix<T>;
  friend class SimpleVector<T>;
  friend class SimpleBaseArray<T,T>;

  EXTERNALCLASSFNCS_TEMPLATE_DECL(Simple3DMatrix, Simple3DMatrix)

  private:
    // static members for single instance object creation
    static Simple3DMatrix<T>* _StaticClassPtr;
    static int* _StaticTrIndex;
    static bool _StaticIsBuiltIn;
    static size_t m_ObjectCnt;
    static bool m_AutoRelease;
    static bool m_StaticCreate;
    static bool m_StaticDestroy;
    static void Init();
    size_t _ObjectID;

  public:
    // static members for single instance object creation
    static int* Instance();
    static void Release();

  protected:
    static bool _AutoErase;
    static MemAllocMethod _AllocMethod;

    Boolean _Error;
    SimpleMatrix<T>** _Matrix3D;
  
    mutable size_t** _MatrixSizes;
    mutable size_t** _MatrixLengths;
    mutable size_t** _MatrixIndexes;

    mutable size_t _Matrix3DSize;
    mutable size_t _Matrix3DLength;
    mutable size_t _Matrix3DIndex;

    PointerCopyTracker<TTYPE,DTYPE>* _CopyTracker;
    bool _Unlinked;
    bool _CpTrkInit;

    static int TrIndex(int i);
    PointerCopyTracker<Simple3DMatrix<T>::TTYPE,Simple3DMatrix<T>::DTYPE>* CopyTrackPtr();
    const PointerCopyTracker<Simple3DMatrix<T>::TTYPE,Simple3DMatrix<T>::DTYPE>* CopyTrackPtr() const;
    void SetDataRefs(SimpleMatrix<T>** Data_, size_t Index_);
    Simple3DMatrix<T>::DTYPE** AssignDataArray(Simple3DMatrix<T>::DTYPE** Vect_, size_t sz);
    void ReplaceCopyTracker(PointerCopyTracker<Simple3DMatrix<T>::TTYPE,Simple3DMatrix<T>::DTYPE>* Ptr_,
                            Simple3DMatrix<T>::TTYPE* Master_);

    bool HasThis(Simple3DMatrix<T>::TTYPE* Val_) const;
    bool HasThis(Simple3DMatrix<T>::TTYPE* Val_, Simple3DMatrix<T>::NTYPE** Node_);
    Simple3DMatrix<T>::TTYPE* Push(Simple3DMatrix<T>::TTYPE* Val_);
    Simple3DMatrix<T>::TTYPE* Swap(Simple3DMatrix<T>::TTYPE* Head_, Simple3DMatrix<T>::NTYPE* Node_=NULL);
    Simple3DMatrix<T>::TTYPE* Pop();

  public:
    Simple3DMatrix(size_t msz=VectorInfo::DEFAULT_SIZE, size_t vsz=0);
    Simple3DMatrix(size_t msz, const T** a, size_t n);
    Simple3DMatrix(const Simple3DMatrix<T>& Arr_, size_t start_, size_t end_);
    Simple3DMatrix(const Simple3DMatrix<T>& Arr_);

    // concatenate constructor
    Simple3DMatrix(const Simple3DMatrix<T> & a1,
                   const Simple3DMatrix<T> & a2);

    virtual ~Simple3DMatrix();

    // Class Identification methods
    static int StaticID()
	{ return TypeInfo::SIMPLE3DMATRIX_TYPE; }
    Boolean IsCommonClass(int IDvalue_) const;
    virtual int ClassID() const;
    const char* ClassName() const;

    // Object Identification method
    virtual size_t ObjectID() const;

    // set master object for shallow copied pointers
    void RetCallToShallow(bool Unlinked_=true, bool InitCall_=false);
    void SetMaster(Simple3DMatrix<T>* Ptr_, bool OwnsData_=false);
    void SetConstMaster(const Simple3DMatrix<T>* Ptr_, bool OwnsData_=false);
    void AppendCopyRefs(Simple3DMatrix<T>::TTYPE* Ptr_, bool ChainAppend_);
    void AppendConstCopyRefs(const Simple3DMatrix<T>::TTYPE* Ptr_, bool ChainAppend_);
    bool IsOwnershipAttrib(bool OwnsData_, bool SelfErase_, bool Shallow_) const;

    // data type identification
    static bool IsBuiltInTypeData();

    // memory allocation method for auto-erase
    static void SetAllocationMethod(MemAllocMethod Method_);

    // auto erase matrix content upon object destruction
    static void SetAutoErase(bool f, Simple3DMatrix<T>* This_);
    void SetAutoErase(bool f);
    void SetDataErase(bool f);

    // object instance creation methods
    static Simple3DMatrix<T>* Make();
    static Simple3DMatrix<T>* Make(size_t n);
    static Simple3DMatrix<T>* Make(const T** a, size_t n);
    static Simple3DMatrix<T>* Make(const Simple3DMatrix<T>& Arr_, size_t start_, size_t end_);
    static Simple3DMatrix<T>* Make(const Simple3DMatrix<T>& Arr_);
    static Simple3DMatrix<T>* Make(const Simple3DMatrix<T> & a1, const Simple3DMatrix<T> & a2);

    // class identification
    virtual bool IsClassType(int Type_) const;
    virtual bool IsMultiDimType() const;

    // error set method
    inline void SetError(Boolean Flag_=TRUE)
        { _Error = Flag_; }
    
    // conversion operator
    const SimpleMatrix<T>* IndexedElement() const;
    SimpleMatrix<T>* IndexedElement();

    // raw C array retrieval methods
    inline SimpleMatrix<T>** Base()
        { return _Matrix3D; }
    inline const SimpleMatrix<T>** Base() const
        { return _Matrix3D; }
    
    // get number of elements
    inline size_t GetCount() const
        { return _Matrix3DSize; }
    inline bool IsUnlinked() const
        { return _Unlinked; }

    // assignment operator
    virtual Simple3DMatrix<T>& operator = (const Simple3DMatrix<T>& Matr_);
    virtual Simple3DMatrix<T>& DeepCopy(const Simple3DMatrix<T>& a);
    virtual Simple3DMatrix<T>& ShallowCopy(const Simple3DMatrix<T>& a);

    // index operators
    virtual PointerReturner<Simple3DMatrix<T>::DTYPE,Simple3DMatrix<T>::CTYPE>
            operator [] (const Index& i);
    virtual PointerReturner<Simple3DMatrix<T>::DTYPE,Simple3DMatrix<T>::CTYPE>
            operator [] (int i);

    virtual const SimpleMatrix<T>* Read(const Index & i) const;
    virtual const SimpleMatrix<T>* Read(int i) const;

    virtual void GrowMatrix(size_t Incr_=10);
    virtual void SetMatrixIndex(size_t Index_);
    virtual void EraseMatrix(MemAllocMethod Method_=MEMMATRIX);
    virtual void FindMatrixRunLength();
    virtual size_t MatrixRunLength();

    virtual void GrowMatrix3D(size_t Incr_=10);
    virtual void SetMatrix3DIndex(size_t Index_);
    virtual void EraseMatrix3D(MemAllocMethod Method_=MEMMATRIX);
    virtual void FindMatrix3DRunLength();
    virtual size_t RunLength();
    
    size_t MatrixLength(size_t Index_) const;
    size_t MatrixSize(size_t Index_) const;
    size_t MatrixIndex(size_t Index_) const;

    // fill functions
    virtual void Fill(T* v);
    virtual void FillFrom(int ix, int iy, int iz, T* v);
    virtual void FillFrom(const Index& ix, const Index& iy, const Index& iz, T* v);

    inline MemAllocMethod AllocMethod() const
        { return _AllocMethod; }
    inline bool AutoErase() const
        { return _AutoErase; }
    inline size_t Length() const
        { return _Matrix3DLength; }
    inline size_t Capacity() const
        { return _Matrix3DSize; }
    inline size_t CurrentIndex() const
        { return _Matrix3DIndex; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

    CLONEFNCS_TEMPLATE_DECL(Simple3DMatrix)
};

/****************************************************************************/
#endif




