#ifndef SIMPLELIST_H
#define SIMPLELIST_H
#ifndef LISTDEFN_H
  #include "listdefn.h"
#endif

#define SIMPLELIST_DEBUG	0
#define SIMPLELIST_INTERDEBUG	0
#define SIMPLELIST_LISTDUMP	0

/****************************************************************************/
template <class T>
class SimpleList;

template <class T>
class SearchableList;

template <class T>
struct SimpleNode;

template <class T>
class ConstListIterator
{
  protected:
    const SimpleList<T>* _ConstList;
    const SimpleNode<T>* _ConstHere;
    size_t _Index;

  public:
    ConstListIterator(const SimpleList<T>* List_);

    Boolean More() const;
    Boolean Less() const;
    
    const SimpleNode<T>* Next();
    const SimpleNode<T>* Prev();

    const SimpleNode<T>* operator ++ (int);
    const SimpleNode<T>* operator ++ ()
	{ return Next(); }

    const SimpleNode<T>* operator -- (int);
    const SimpleNode<T>* operator -- ()
	{ return Prev(); }

    const SimpleList<T>* List() const
	{ return _ConstList; }
    SimpleList<T>* List()
	{ return _ConstList; }
    const SimpleNode<T>* Here() const
	{ return _ConstHere; }
    SimpleNode<T>* Here()
	{ return _ConstHere; }

    ConstListIterator<T>& Head();
    ConstListIterator<T>& Tail();

    const SimpleNode<T>* IndexNode(size_t Index_);

    operator const SimpleNode<T>* () const
	{ return Here(); }
    operator SimpleNode<T>* ()
	{ return Here(); }

    size_t Position() const;
    size_t Count() const;                

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

template <class T>
class SimpleListIterator
{
  protected:
    SimpleList<T>* _List;
    SimpleNode<T>* _Here;
    size_t _Index;

  public:
    SimpleListIterator(SimpleList<T>* List_);

    Boolean More() const;
    Boolean Less() const;
    
    SimpleNode<T>* Next();
    SimpleNode<T>* Prev();

    SimpleNode<T>* operator ++ (int);
    SimpleNode<T>* operator ++ ()
	{ return Next(); }

    SimpleNode<T>* operator -- (int);
    SimpleNode<T>* operator -- ()
	{ return Prev(); }

    const SimpleList<T>* List() const
	{ return _List; }
    SimpleList<T>* List()
	{ return _List; }        
    const SimpleNode<T>* Here() const
	{ return _Here; }
    SimpleNode<T>* Here()
	{ return _Here; }

    SimpleListIterator<T>& Head();
    SimpleListIterator<T>& Tail();

    SimpleNode<T>* IndexNode(size_t Index_);
    SimpleNode<T>* Update(size_t Index_, T* Data_);

    void Remove();
    void InsertAfter(T* NewPtr_)
	{ _List->InsertAfter(_Here, NewPtr_); }
    void InsertBefore(T* NewPtr_)
	{ _List->InsertBefore(_Here, NewPtr_); }

    operator const SimpleNode<T>* () const
	{ return Here(); }
    operator SimpleNode<T>* ()
	{ return Here(); }

    size_t Position() const;
    size_t Count() const;        

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif        
};

/****************************************************************************/
// Numbered list node
template <class T>
class NumberedNode
{
  public:
    NumberedNode();
    NumberedNode(T* Obj_, size_t Num_);
    NumberedNode(const NumberedNode& Obj_);
    ~NumberedNode();
  
    size_t _Number;
    T* _Object;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
#endif    
};

// Singly linked list node
template <class T>
class SimpleNode
{
  public:
    typedef NumberedNode<T> WRAP;

    union
    {
      WRAP* _Wrapper;
      T* _Object;
    };

    SimpleNode<T>* _Next;

  protected:
    SimpleNode(T* Obj_, SimpleNode<T>* Next_, size_t Num_);
    
  public:
    SimpleNode(T* Obj_, SimpleNode<T>* Next_ = NULL);
    virtual ~SimpleNode();

    virtual bool IsValueDestroyed() const;
    virtual bool IsSinglyLinked() const;
    virtual void Insert(SimpleNode<T>* Node_);

    virtual T* Value();
    virtual const T* Value() const;
    virtual SimpleNode<T>* SetValue(T* Ptr_);

    virtual size_t Number() const;
    virtual SimpleNode<T>* SetNumber(size_t Num_);

    virtual SimpleNode<T>* SetNext(SimpleNode<T>* Ptr_);
    virtual SimpleNode<T>* Next();
    virtual const SimpleNode<T>* Next() const;

    virtual SimpleNode<T>* SetPrev(SimpleNode<T>* Ptr_);
    virtual SimpleNode<T>* Prev();
    virtual const SimpleNode<T>* Prev() const;

    static SimpleNode<T>* FindHead(SimpleNode<T>* Ptr_);
    static SimpleNode<T>* FindTail(SimpleNode<T>* Ptr_);
    static const SimpleNode<T>* FindConstHead(const SimpleNode<T>* Ptr_);
    static const SimpleNode<T>* FindConstTail(const SimpleNode<T>* Ptr_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
#endif
};

// Non-destructive linked list node
template <class T>
class SimpleKeepNode : public SimpleNode<T>
{
  protected:
    SimpleKeepNode(T* Obj_, SimpleNode<T>* Next_, size_t Num_);

  public:
    SimpleKeepNode(T* Obj_, SimpleNode<T>* Next_ = NULL);
    virtual ~SimpleKeepNode();

    virtual bool IsValueDestroyed() const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
#endif    
};

// Doubly linked list node
template <class T>
class SimpleDblNode : public SimpleNode<T>
{
  public:
    SimpleNode<T>* _Prev;

  protected:
    SimpleDblNode(T* Obj_, SimpleNode<T>* Next_,
                  SimpleNode<T>* Prev_, size_t Num_);

  public:
    SimpleDblNode(T* Obj_,
                  SimpleNode<T>* Next_=NULL,
                  SimpleNode<T>* Prev_=NULL);
    virtual ~SimpleDblNode();

    virtual bool IsValueDestroyed() const;
    virtual bool IsSinglyLinked() const;
    virtual void Insert(SimpleNode<T>* Node_);

    virtual SimpleNode<T>* SetNext(SimpleNode<T>* Ptr_);
    virtual SimpleNode<T>* Next();
    virtual const SimpleNode<T>* Next() const;

    virtual SimpleNode<T>* SetPrev(SimpleNode<T>* Ptr_);
    virtual SimpleNode<T>* Prev();
    virtual const SimpleNode<T>* Prev() const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
#endif  
};

// Non-destructive doubly linked list node
template <class T>
class SimpleDblKeepNode : public SimpleDblNode<T>
{
  protected:
    SimpleDblKeepNode(T* Obj_, SimpleNode<T>* Next_,
                      SimpleNode<T>* Prev_, size_t Num_);

  public:
    SimpleDblKeepNode(T* Obj_,
                  SimpleNode<T>* Next_=NULL,
                  SimpleNode<T>* Prev_=NULL);
    virtual ~SimpleDblKeepNode();

    virtual bool IsValueDestroyed() const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
#endif
};

// Numbered singly linked list node
template <class T>
class SimpleNumberedNode : public SimpleNode<T>
{
  public:
    SimpleNumberedNode(T* Obj_, size_t Num_, SimpleNode<T>* Next_=NULL);
    virtual ~SimpleNumberedNode();

    virtual T* Value();
    virtual const T* Value() const;
    virtual SimpleNode<T>* SetValue(T* Ptr_);

    virtual size_t Number() const;
    virtual SimpleNode<T>* SetNumber(size_t Num_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
#endif    
};

// Numbered Non-destructive linked list node
template <class T>
class SimpleNumberedKeepNode : public SimpleKeepNode<T>
{
  public:
    SimpleNumberedKeepNode(T* Obj_, size_t Num_, SimpleNode<T>* Next_=NULL);
    virtual ~SimpleNumberedKeepNode();

    virtual T* Value();
    virtual const T* Value() const;
    virtual SimpleNode<T>* SetValue(T* Ptr_);

    virtual size_t Number() const;
    virtual SimpleNode<T>* SetNumber(size_t Num_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
#endif
};

// Numbered Doubly linked list node
template <class T>
class SimpleNumberedDblNode : public SimpleDblNode<T>
{
  public:
    SimpleNumberedDblNode(T* Obj_, size_t Num_,
                       SimpleNode<T>* Next_=NULL,
                       SimpleNode<T>* Prev_=NULL);
    virtual ~SimpleNumberedDblNode();

    virtual T* Value();
    virtual const T* Value() const;
    virtual SimpleNode<T>* SetValue(T* Ptr_);

    virtual size_t Number() const;
    virtual SimpleNode<T>* SetNumber(size_t Num_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
#endif
};

// Numbered Non-destructive Doubly linked list node
template <class T>
class SimpleNumberedDblKeepNode : public SimpleDblKeepNode<T>
{
  public:
    SimpleNumberedDblKeepNode(T* Obj_, size_t Num_,
                              SimpleNode<T>* Next_=NULL,
                              SimpleNode<T>* Prev_=NULL);
    virtual ~SimpleNumberedDblKeepNode();

    virtual T* Value();
    virtual const T* Value() const;
    virtual SimpleNode<T>* SetValue(T* Ptr_);

    virtual size_t Number() const;
    virtual SimpleNode<T>* SetNumber(size_t Num_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
#endif
};

/****************************************************************************/
template <class T>
class SimpleList : public ListOperation
{
  protected:
    ObjectCloner<T>* _Cloner;
    size_t _NodeLinks;
    size_t _Count;
    bool _KeepValue;
    bool _SaveOps;
    bool _Numbered;
    int _CopyValue;

    union
    {
      mutable ListOperation* _ListOps;
      mutable const ListOperation* _ConstListOps;
    };

    union
    {
      mutable SimpleNode<T>* _Tail;
      mutable const SimpleNode<T>* _ConstTail;
    };
    
    union
    {
      mutable SimpleNode<T>* _List;
      mutable const SimpleNode<T>* _ConstList;
    };

    SimpleList(const SimpleList<T>& List_, bool SaveOps_, bool Numbered_);
    SimpleList(const SimpleNode<T>* NodePtr_, ObjectCloner<T>* ClonerPtr_=NULL,
               size_t NodeLinks_=2, bool KeepValue_=false,
               int CopyValue_=-1, bool SaveOps_=false,
               bool Numbered_=false);

    SimpleNode<T>* FindPrevious(SimpleNode<T>* Ptr_);
    
    T* PopTailNode();
    T* PopNode();

  public:
    SimpleList(ObjectCloner<T>* ClonerPtr_=NULL,
               size_t NodeLinks_=2, bool KeepValue_=false,
               int CopyValue_=-1, bool SaveOps_=false,
               bool Numbered_=false);
    SimpleList(const SimpleList<T>& List_);
    virtual ~SimpleList();

    static T* Deref(SimpleNode<T>* Ptr_)
	{ return (Ptr_ ? Ptr_->_Object:((T*)NULL)); }    

    SimpleList<T>& operator = (const SimpleList<T>& SimpList_);
    SimpleList<T>& operator += (const SimpleList<T>& SimpList_);

    SimpleNode<T>* Tail();
    const SimpleNode<T>* Tail() const;
    SimpleNode<T>* Head()
	{ return _List; }
    const SimpleNode<T>* Head() const
	{ return _List; }
    size_t NodeLinks() const
        { return _NodeLinks; }
    bool KeepValue() const
        { return _KeepValue; }
    bool CopyValue() const
        { return _CopyValue; }
    bool SaveOps() const
        { return _SaveOps; }

    // Record list operations history
    virtual void DumpVectorData(ostream* osp_=NULL);
    virtual void DumpListData(int(*dispfn)(const void*, int), ostream* osp_=NULL);
    virtual void ReNumberList();
    virtual bool IsNumbered() const;
    virtual void SetSaveOps(bool SaveOps_);
    virtual void SetNumberedList(bool Flag_);
    virtual void StoreOpToList(int OpType_, size_t NodeNum_=0);
    virtual int LastOp() const;
    virtual int FirstOp() const;
    virtual int OpAt(size_t Index_) const;
    virtual int LastNodeNum() const;
    virtual int FirstNodeNum() const;
    virtual int NodeNumAt(size_t Index_) const;    
    virtual size_t TotalOps() const;
    virtual size_t OpsListLimit() const;
    virtual size_t OpsListIndex() const;
    virtual bool IsLinkedList() const;

    // List item duplication behaviour
    SimpleList<T>& SetCloner(ObjectCloner<T>* Cloner_);
    SimpleList<T>& SetCopyValue(int Flag_);

    // List item destruction behaviour
    SimpleList<T>& SetKeepValue(bool Flag_);

    // Item/Node Removal and Insertion methods
    T* Remove(SimpleNode<T>* Ptr_, int DeleteValue_=-1);
    void InsertAfter(SimpleNode<T>* Ptr_, T* NewPtr_);
    void InsertBefore(SimpleNode<T>* Ptr_, T* NewPtr_);

    // Calls the Push method to add the node at the head of the list.
    SimpleList<T>& AppendHead(T* Obj_);

    // Calls the AddNode method to add the node at the tail of the list.
    SimpleList<T>& AppendTail(T* Obj_);

    // Delete all nodes
    virtual void DeleteAll();
    virtual void DeleteAllAndData(int DeleteValue_=1);

    // Returns TRUE if list is empty
    Boolean Empty(void) const;

    // Return size of list in number of elements
    virtual size_t Size() const;

    bool HasThis(const T* Obj_, PtrCompare<T>* Compare_=NULL) const;
    bool HasThis(const T* Obj_, PtrCompare<T>* Compare_=NULL,
                 SimpleNode<T>** LocPtr_=NULL);

    virtual bool IsInserted() const;
    virtual bool IsDetached() const;
        
    virtual size_t TimesInserted() const;
    virtual size_t TimesDetached() const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
template <class T>
class SimpleStack : public SimpleList<T>
{
  protected:  
    SimpleStack(const SimpleNode<T>* NodePtr_, ObjectCloner<T>* ClonerPtr_=NULL,
                size_t NodeLinks_=1, bool KeepValue_=true,
                int CopyValue_=-1, bool SaveOps_=false);

  public:
    SimpleStack(ObjectCloner<T>* ClonerPtr_=NULL,
                size_t NodeLinks_=1, bool KeepValue_=true,
                int CopyValue_=-1, bool SaveOps_=false);
    SimpleStack(const SimpleStack<T>& List_);
    virtual ~SimpleStack();

    SimpleStack<T>& operator = (const SimpleList<T>& SimpList_);
    SimpleStack<T>& operator += (const SimpleList<T>& SimpList_);

    // Item/Node Removal and Insertion methods
    T* Pop();
    SimpleStack<T>& Push(T* NewPtr_);

    // Record stack operations history
    virtual void SetSaveOps(bool SaveOps_);
    virtual void StoreOpToList(int OpType_, size_t NodeNum_=0);
    virtual int LastOp() const;
    virtual int FirstOp() const;
    virtual int OpAt(size_t Index_) const;
    virtual int LastNodeNum() const;
    virtual int FirstNodeNum() const;
    virtual int NodeNumAt(size_t Index_) const;
    virtual size_t TotalOps() const;
    virtual size_t OpsListLimit() const;
    virtual size_t OpsListIndex() const;
    virtual bool IsStack() const;

    // Delete all nodes
    virtual void DeleteAll();
    virtual void DeleteAllAndData(int DeleteValue_=1);

    bool IsPopped() const;
    bool IsPushed() const;

    size_t TimesPopped() const;
    size_t TimesPushed() const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
template <class T>
class SimpleQueue : public SimpleList<T>
{
  protected:
    SimpleQueue(const SimpleNode<T>* NodePtr_, ObjectCloner<T>* ClonerPtr_=NULL,
                size_t NodeLinks_=2, bool KeepValue_=true,
                int CopyValue_=-1, bool SaveOps_=false);

  public:
    SimpleQueue(ObjectCloner<T>* ClonerPtr_=NULL,
                size_t NodeLinks_=2, bool KeepValue_=true,
                int CopyValue_=-1, bool SaveOps_=false);
    SimpleQueue(const SimpleQueue<T>& List_);
    virtual ~SimpleQueue();

    SimpleQueue<T>& operator = (const SimpleList<T>& SimpList_);
    SimpleQueue<T>& operator += (const SimpleList<T>& SimpList_);

    // Item/Node Removal and Insertion methods
    T* Dequeue();
    SimpleQueue<T>& Enqueue(T* NewPtr_);

    // Record queue operations history
    virtual void SetSaveOps(bool SaveOps_);
    virtual void StoreOpToList(int OpType_, size_t NodeNum_=0);
    virtual int LastOp() const;
    virtual int FirstOp() const;
    virtual int OpAt(size_t Index_) const;
    virtual int LastNodeNum() const;
    virtual int FirstNodeNum() const;
    virtual int NodeNumAt(size_t Index_) const;
    virtual size_t TotalOps() const;
    virtual size_t OpsListLimit() const;
    virtual size_t OpsListIndex() const;
    virtual bool IsQueue() const;

    // Delete all nodes
    virtual void DeleteAll();
    virtual void DeleteAllAndData(int DeleteValue_=1);

    bool IsEnqueued() const;
    bool IsDequeued() const;

    size_t TimesEnqueued() const;
    size_t TimesDequeued() const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
template <class T>
class SimpleDeque : public SimpleList<T>
{
  protected:
    SimpleDeque(const SimpleNode<T>* NodePtr_, ObjectCloner<T>* ClonerPtr_=NULL,
                size_t NodeLinks_=2, bool KeepValue_=true,
                int CopyValue_=-1, bool SaveOps_=false);

  public:
    SimpleDeque(ObjectCloner<T>* ClonerPtr_=NULL,
                size_t NodeLinks_=2, bool KeepValue_=true,
                int CopyValue_=-1, bool SaveOps_=false);
    SimpleDeque(const SimpleDeque<T>& List_);
    virtual ~SimpleDeque();

    SimpleDeque<T>& operator = (const SimpleList<T>& SimpList_);
    SimpleDeque<T>& operator += (const SimpleList<T>& SimpList_);

    // Item/Node Removal and Insertion methods
    T* Dequeue();
    SimpleDeque<T>& Enqueue(T* NewPtr_);

    bool IsEnqueued() const;
    bool IsDequeued() const;

    size_t TimesEnqueued() const;
    size_t TimesDequeued() const;

    // Item/Node Removal and Insertion methods
    T* Pop();
    SimpleDeque<T>& Push(T* NewPtr_);

    bool IsPopped() const;
    bool IsPushed() const;
    
    size_t TimesPopped() const;
    size_t TimesPushed() const;

    // Record deque operations history
    virtual void SetSaveOps(bool SaveOps_);
    virtual void StoreOpToList(int OpType_, size_t NodeNum_=0);
    virtual int LastOp() const;
    virtual int FirstOp() const;
    virtual int OpAt(size_t Index_) const;
    virtual int LastNodeNum() const;
    virtual int FirstNodeNum() const;
    virtual int NodeNumAt(size_t Index_) const;
    virtual size_t TotalOps() const;
    virtual size_t OpsListLimit() const;
    virtual size_t OpsListIndex() const;
    virtual bool IsDeque() const;

    // Delete all nodes
    virtual void DeleteAll();
    virtual void DeleteAllAndData(int DeleteValue_=1);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
template <class T>
class SimpleListBrowser;

struct SearchEnums
{
  public:
    enum
    {
      FORWARD = 1,
      REVERSE = -1
    };
};

template <class T>
class SearchableList : public SimpleList<T>, public SearchEnums
{
  friend class SimpleListBrowser<T>;

  protected:
    mutable int _SearchDir;
    SearchableList(const SimpleNode<T>* NodePtr_, ObjectCloner<T>* ClonerPtr_=NULL,
                   size_t NodeLinks_=2, bool KeepValue_=false,
                   int CopyValue_=-1, bool SaveOps_=false);

  public:
    SearchableList(ObjectCloner<T>* ClonerPtr_=NULL,
                   size_t NodeLinks_=2, bool KeepValue_=false,
                   int CopyValue_=-1, bool SaveOps_=false);
    SearchableList(SimpleList<T>* ListPtr_);
    SearchableList(const SearchableList<T>& List_);

    SearchableList<T>& operator = (SimpleList<T>* ListPtr_);
    SearchableList<T>& operator += (SimpleList<T>* ListPtr_);
    
    SearchableList<T>& operator = (const SimpleList<T>& SimpList_);
    SearchableList<T>& operator += (const SimpleList<T>& SimpList_);
    
    #if !defined(__TURBOC__) & !defined(__BORLANDC__)
      SearchableList<T>& operator = (const SearchableList<T>& SimpList_);
      SearchableList<T>& operator += (const SearchableList<T>& SimpList_);
    #endif

    // Find next adjacent node with the same matching object
    SimpleNode<T>* NextMatch(SimpleNode<T>* Ptr_, PtrCompare<T>* Compare_);
    SimpleNode<T>* NextMatch(SimpleNode<T>* Ptr_);

    // Find next adjacent node with the same matching object
    const SimpleNode<T>* NextMatch(const SimpleNode<T>* Ptr_, PtrCompare<T>* Compare_) const;
    const SimpleNode<T>* NextMatch(const SimpleNode<T>* Ptr_) const;

    // Find next preceding node with the same matching object
    SimpleNode<T>* PrevMatch(SimpleNode<T>* Ptr_, PtrCompare<T>* Compare_);
    SimpleNode<T>* PrevMatch(SimpleNode<T>* Ptr_);

    // Find next preceding node with the same matching object
    const SimpleNode<T>* PrevMatch(const SimpleNode<T>* Ptr_, PtrCompare<T>* Compare_) const;
    const SimpleNode<T>* PrevMatch(const SimpleNode<T>* Ptr_) const;

    // Find nearest node with matching object
    SimpleNode<T>* ForwardFind(T* Obj_, PtrCompare<T>* Compare_);
    SimpleNode<T>* ForwardFind(T* Obj_);

    // Find nearest node with matching object
    const SimpleNode<T>* ForwardFind(const T* Obj_, PtrCompare<T>* Compare_) const;
    const SimpleNode<T>* ForwardFind(const T* Obj_) const;

    // Find nearest node with matching object
    SimpleNode<T>* ReverseFind(T* Obj_, PtrCompare<T>* Compare_);
    SimpleNode<T>* ReverseFind(T* Obj_);

    // Find nearest node with matching object
    const SimpleNode<T>* ReverseFind(const T* Obj_, PtrCompare<T>* Compare_) const;
    const SimpleNode<T>* ReverseFind(const T* Obj_) const;

    // Find nearest node with matching object, search direction adjustable
    SimpleNode<T>* Find(T* Obj_, PtrCompare<T>* Compare_);
    SimpleNode<T>* Find(T* Obj_);

    // Find nearest node with matching object, search direction adjustable
    const SimpleNode<T>* Find(const T* Obj_, PtrCompare<T>* Compare_) const;
    const SimpleNode<T>* Find(const T* Obj_) const;

    SearchableList<T>& SetSearchDirection(int Dir_)
        { _SearchDir = Dir_; }
    const SearchableList<T>& SetSearchDirection(int Dir_) const
        { _SearchDir = Dir_; }
    int SearchDirection() const
        { return _SearchDir; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif    
};

template <class T>
class SimpleListBrowser : private SearchableList<T>
{
  public:
    SimpleListBrowser(const SimpleNode<T>* Ptr_);
    SimpleListBrowser(const SearchableList<T>& Alist_);
    SimpleListBrowser(SimpleList<T>* ListPtr_);
    virtual ~SimpleListBrowser();

    SimpleListBrowser<T>& operator = (const SimpleNode<T>* Ptr_);
    SimpleListBrowser<T>& operator = (const SearchableList<T>& Alist_);
    SimpleListBrowser<T>& operator = (SimpleList<T>* ListPtr_);

    // Find next adjacent node with the same matching object
    const SimpleNode<T>* NextMatch(const T* Obj_, PtrCompare<T>* Compare_) const;
    const SimpleNode<T>* NextMatch(const T* Obj_) const;

    // Find next preceding node with the same matching object
    const SimpleNode<T>* PrevMatch(const T* Obj_, PtrCompare<T>* Compare_) const;
    const SimpleNode<T>* PrevMatch(const T* Obj_) const;

    const SimpleNode<T>* IndexNode(size_t Index_) const;

    Boolean More() const;
    const SimpleNode<T>* operator ++ () const;
    const SimpleNode<T>* operator ++ (int) const;

    Boolean Less() const;
    const SimpleNode<T>* operator -- () const;
    const SimpleNode<T>* operator -- (int) const;

    const SimpleNode<T>* Here() const
	{ return SearchableList<T>::_List; }

    operator const SimpleNode<T>* () const
	{ return Here(); }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif        
};

template <class T>
class SimpleListModifier : private SearchableList<T>
{
  public:
    SimpleListModifier(SimpleNode<T>* Ptr_);
    SimpleListModifier(SearchableList<T>& Alist_);
    SimpleListModifier(SimpleList<T>* ListPtr_);
    virtual ~SimpleListModifier();

    SimpleListModifier<T>& operator = (SimpleNode<T>* Ptr_);
    SimpleListModifier<T>& operator = (SearchableList<T>& Alist_);
    SimpleListModifier<T>& operator = (SimpleList<T>* ListPtr_);

    // Find next adjacent node with the same matching object
    SimpleNode<T>* NextMatch(const T* Obj_, PtrCompare<T>* Compare_);
    SimpleNode<T>* NextMatch(const T* Obj_);

    // Find next preceding node with the same matching object
    SimpleNode<T>* PrevMatch(const T* Obj_, PtrCompare<T>* Compare_);
    SimpleNode<T>* PrevMatch(const T* Obj_);

    SimpleNode<T>* IndexNode(size_t Index_);

    Boolean More() const;
    SimpleNode<T>* operator ++ ();
    SimpleNode<T>* operator ++ (int);

    Boolean Less() const;
    SimpleNode<T>* operator -- ();
    SimpleNode<T>* operator -- (int);

    SimpleNode<T>* Here()
	{ return SearchableList<T>::_List; }

    operator SimpleNode<T>* ()
	{ return Here(); }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif        
};

/****************************************************************************/
#endif


