#ifndef STRFILE_CPP
#define STRFILE_CPP
#ifndef STRFILE_H
  #include "strfile.h"
#endif

/****************************************************************************/
StrFile::StrFile(const char* Fname_,
                 Boolean AllocBuf_, Boolean AllocArray_):
_Fin(NULL),
_Fout(NULL),
_FileMode(0),
_Fname(Fname_ ? NewString(Fname_):NULL),

_Success(0),
_SpecTokenFound(0),
_QuoteFound(0),
_IgnoreEos(1),
_Eos(0),
_NoSep(0),

_StripToken(0),
_SkipEmptyFields(0),

_DataSpec(0),
_DataFound(0),
_TypeSpec(0),
_TypeFound(0),

_LineNo(0),
_TokenNo(0),
_BufLen(0),
_Sval(0),

_Tokenated(0),
_ReturnStr(0),

_Line(0),
_Delims(0),
_StrQuote(0),
_DelimsEsc(0),
_StrQuoteEsc(0),

_Assignment(0),
_LeftBracket(0),
_RightBracket(0),
_WhiteSpace(0),

_Section(0),
_Key(0),
_Value(0),

_Index(0),
_NextPos(0),
_TokenEnd(0),
_Token(0),

_TokenIndex(0),
_TokenLimit(0),
_NumTokens(0),
_Tokens(0)
{
  _ReturnStr = (char*)RawAllocateWith(MEMMATRIX, sizeof(char) * RETURNBUFFER_MAX);

  if (AllocBuf_)
    InitLine(StrFile::BUFLEN);

  if (AllocArray_)
    InitTokens(NUMTOKENS_START);

  if (_Fname)
    OpenFile();
}

/****************************************************************************/
StrFile::StrFile(const StrFile& Obj_):
_Fin(NULL),
_Fout(NULL),
_FileMode(0),
_Fname(Obj_._Fname ? NewString(Obj_._Fname):NULL),

_Success(0),
_SpecTokenFound(0),
_QuoteFound(0),
_IgnoreEos(1),
_Eos(0),
_NoSep(0),

_StripToken(Obj_._StripToken),
_SkipEmptyFields(Obj_._SkipEmptyFields),

_DataSpec(Obj_._DataSpec),
_DataFound(0),
_TypeSpec(Obj_._TypeSpec),
_TypeFound(0),

_LineNo(0),
_TokenNo(0),
_BufLen(0),
_Sval(0),

_Tokenated(0),
_ReturnStr(0),

_Line(0),
_Delims(Obj_._Delims ? NewString(Obj_._Delims):NULL),
_StrQuote(Obj_._StrQuote ? NewString(Obj_._StrQuote):NULL),
_DelimsEsc(Obj_._DelimsEsc),
_StrQuoteEsc(Obj_._StrQuoteEsc),

_Assignment(Obj_._Assignment ? NewString(Obj_._Assignment):NULL),
_LeftBracket(Obj_._LeftBracket ? NewString(Obj_._LeftBracket):NULL),
_RightBracket(Obj_._RightBracket ? NewString(Obj_._RightBracket):NULL),
_WhiteSpace(Obj_._WhiteSpace ? NewString(Obj_._WhiteSpace):NULL),

_Section(0),
_Key(0),
_Value(0),

_Index(0),
_NextPos(0),
_TokenEnd(0),
_Token(0),

_TokenIndex(0),
_TokenLimit(0),
_NumTokens(0),
_Tokens(0)
{
  _ReturnStr = (char*)RawAllocateWith(MEMMATRIX, sizeof(char) * RETURNBUFFER_MAX);
  InitLine(StrFile::BUFLEN);
  InitTokens(NUMTOKENS_START);

  if (_Fname)
    OpenFile();
}

/****************************************************************************/
StrFile::~StrFile()
{
  int x;
  for (x = 0; x < _NumTokens; x++)
    RawDeleteArray(_Tokens[x]);
  
  RawDeleteArray(_Tokens);
  RawDeleteArray(_Fname);

  RawDeleteArray(_Tokenated);
  RawDeleteArray(_ReturnStr);
  
  RawDeleteArray(_Line);  
  RawDeleteArray(_Delims);
  RawDeleteArray(_StrQuote);

  RawDeleteArray(_Assignment);
  RawDeleteArray(_LeftBracket);
  RawDeleteArray(_RightBracket);
  RawDeleteArray(_WhiteSpace);

  RawDeleteArray(_Section);
  RawDeleteArray(_Key);
  RawDeleteArray(_Value);  

  CloseFile();
  
  if (_Fin)
    delete _Fin;

  if (_Fout)
    delete _Fout;
}

/****************************************************************************/
StrFile& StrFile::WriteFile()
{
  if (_Fout)
    WriteFile(*_Fout);

  return *this;
}

/****************************************************************************/
StrFile& StrFile::WriteFile(ofstream& Out_)
{
  char* Buffer_ = NULL;
  size_t slen, llen, rlen;

  if (!_Line)
  {
    InitLine(StrFile::BUFLEN);
    _Success = 0;
    return *this;
  }

  slen = _Line ? strlen(_Line):0;
  llen = _LeftBracket ? strlen(_LeftBracket):0;
  rlen = _RightBracket ? strlen(_RightBracket):0;

  _DataFound = 0;
  _TypeFound = 0;

  if (!_DataSpec)
  {
    _Success = slen;
    
    if (_Success)
      Out_ <<_Line;
    else
      return *this;
  }
  else if ((_DataSpec & StrFile::SECTION) &&
           llen && strncmp(_Line, _LeftBracket, llen) == 0 &&
           rlen && strncmp(_Line+slen-rlen, _RightBracket, rlen) == 0)
    _Success = WriteSection();
  else if (_DataSpec & StrFile::KVPAIR)
  {    
    if (_Assignment && strstr(_Line, _Assignment))
      _Success = WritePair();
    else
      _Success = WriteValue();
  }
  else if (_DataSpec & StrFile::TOKEN)
  {
    if (_Delims && strpbrk(_Line, _Delims))
      _Success = WriteToken();
    else
      _Success = WriteSepItem();
  }

  return *this;
}

/****************************************************************************/
StrFile& StrFile::ReadFile()
{
  if (_Fin)
    ReadFile(*_Fin);

  return *this;
}

/****************************************************************************/
StrFile& StrFile::ReadFile(ifstream& In_)
{
  char* Buffer_ = NULL;
  size_t slen, llen, rlen;

  if (!_Line)
    InitLine(StrFile::BUFLEN);

  if (!_NextPos)
  {
    In_.getline(_Line, _BufLen);  
    RemovePadding(_Line, " \n\t");
    
    slen = _Line ? strlen(_Line):0;
    llen = _LeftBracket ? strlen(_LeftBracket):0;
    rlen = _RightBracket ? strlen(_RightBracket):0;

    if (!slen)
    {
      _Success = 0;
      return *this;
    }    
  }

  _DataFound = 0;
  _TypeFound = 0;

  if (!_DataSpec)
  {
    _Success = _Line ? strlen(_Line):0;
    
    if (_Success)
    {
      _DataFound |= StrFile::LINE;
      FindTokenType(_Line);
    }
  }  
  else if ((_DataSpec & StrFile::SECTION) &&
           llen && strncmp(_Line, _LeftBracket, llen) == 0 &&
           rlen && strncmp(_Line+slen-rlen, _RightBracket, rlen) == 0)
    _Success = ReadSection();
  else if ((_DataSpec & StrFile::KVPAIR) &&
           (!_NextPos || _Eos))
  {    
    if (_Assignment && strstr(_Line, _Assignment))
      _Success = ReadPair();
    else
      _Success = ReadValue();
  }
  else if ((_DataSpec & StrFile::TOKEN) &&
           (_NextPos && !_Eos))
  {
    if (_Delims && strpbrk(_Line, _Delims))
      _Success = ReadToken();
    else
      _Success = ReadSepItem();

    if (_Eos && _IgnoreEos)
    {
      ResetEos();
      ReadFile(In_);
    }
  }

  return *this;
}

/****************************************************************************/
char* StrFile::InitLine(size_t Size_)
{
  if (Size_)
  {
    _Line = (char*)RawAllocateWith(MEMMATRIX, sizeof(char) * (Size_+1));
    _Line = (char*)memset(_Line, 0, sizeof(char) * (Size_+1));
    _BufLen = Size_;    
    _Line[0] = _Line[Size_] = 0;
  }
  
  return _Line;
}

/****************************************************************************/
char** StrFile::InitTokens(int Size_)
{
  if (Size_)
  {
    _Tokens = (char**)RawAllocateWith(MEMMATRIX, sizeof(char*) * (Size_+1));
    _Tokens = (char**)memset(_Tokens, 0, sizeof(char*) * (Size_+1));
    _TokenLimit = Size_;    
    _Tokens[0] = _Tokens[Size_] = 0;    
    _NumTokens = 0;
  }
  
  return _Tokens;
}

/****************************************************************************/
char** StrFile::IncreaseTokens(int Incr_)
{
  if (Incr_)
  {
    char** Old_ = _Tokens;
    int OldLen_ = _TokenLimit;

    _TokenLimit += Incr_;
    _Tokens = (char**)RawAllocateWith(MEMMATRIX, sizeof(char*) * (_TokenLimit+1));
    
    memmove(_Tokens, Old_, sizeof(char*) * OldLen_);
    memset(Old_, 0, sizeof(char*) * OldLen_);
    memset(_Tokens+OldLen_, 0, sizeof(char*) * Incr_);
    RawDeleteArray(Old_);
  }

  return _Tokens;
}

/****************************************************************************/
int StrFile::FindTokenType(char* Token_)
{
  if (_TypeSpec == INT)
  {
    _SpecTokenFound = IsInt(Token_, 10);
    
    if (_SpecTokenFound)
    {
      _Ival = atoi(Token_);
      _TypeFound = INT;
    }
    else
    {
      _Sval = Token_;
      _TypeFound = STRING;    
    }
  }
  else if (_TypeSpec == FLOAT)
  {
    _SpecTokenFound = IsFloat(Token_, 10);
    
    if (_SpecTokenFound)
    {
      _Fval = atof(Token_);
      _TypeFound = FLOAT;
    }
    else
    {
      _Sval = Token_;
      _TypeFound = STRING;
    }
  }
  else if (_TypeSpec == STRING)
  {
    _SpecTokenFound = TRUE;  
    _Sval = Token_;
    _TypeFound = STRING;
  }
  else
  {
    _SpecTokenFound = FALSE;

    if (IsInt(Token_, 10))
    {
      _Ival = atoi(Token_);
      _TypeFound = INT;
    }
    else if (IsFloat(Token_, 10))
    {
      _Fval = atof(Token_);
      _TypeFound = FLOAT;
    }
    else
    {
      _Sval = Token_;
      _TypeFound = STRING;
    }
  }

  return _TypeFound;
}

/****************************************************************************/
char* StrFile::SetSection(const char* Line_)
{
  int slen, rlen, llen;
  slen = Line_ ? strlen(Line_):0;

  if (slen)
    if (_FileMode)
    {
      llen = _LeftBracket ? strlen(_LeftBracket):0;
      rlen = _RightBracket ? strlen(_RightBracket):0;

      if (llen && strncmp(Line_, _LeftBracket, llen) == 0 &&
          rlen && strncmp(Line_+slen-rlen, _RightBracket, rlen) == 0)
      {
        _DataSpec = StrFile::SECTION;
        AssignTrgToSrcStr(_Section, Line_);
      }
      else if (llen && rlen)
      {
        _DataSpec = StrFile::SECTION;
        AssignTrgToSrcStr(_Section, _LeftBracket);
        AppendTrgToSrcStr(_Section, Line_);
        AppendTrgToSrcStr(_Section, _RightBracket);
      }
    }
    else
      AssignTrgToSrcStr(_Section, Line_);

  return _Section;
}

/****************************************************************************/
char* StrFile::SetKey(const char* Line_)
{
  char* ret = (Line_ ? AssignTrgToSrcStr(_Key, Line_):_Key);

  if (_FileMode && Line_)
    _DataSpec = StrFile::KVPAIR;

  return ret;  
}

/****************************************************************************/
char* StrFile::SetValue(const char* Line_)
{
  char* ret = (Line_ ? AssignTrgToSrcStr(_Value, Line_):_Value);

  if (_FileMode && Line_)
  {
    if (!_Key || _DataSpec != StrFile::KVPAIR)
      _DataSpec = StrFile::VALUE;
    else
      _DataSpec = StrFile::KVPAIR;
      
    _TypeSpec = STRING;
    _Sval = ret;
  }

  return ret;
}

/****************************************************************************/
int StrFile::SetIntValue(int Val_)
{
  if (_FileMode)
  {
    IntToStr(Val_, _Line);
    AssignTrgToSrcStr(_Value, _Line);

    if (!_Key || _DataSpec != StrFile::KVPAIR)
      _DataSpec = StrFile::VALUE;
    else
      _DataSpec = StrFile::KVPAIR;

    _TypeSpec = INT;
    _Ival = Val_;    
  }

  return Val_;
}

/****************************************************************************/
double StrFile::SetFloatValue(double Val_, int prec)
{
  if (_FileMode)
  {
    FloatToStr(Val_, _Line, 0, prec);
    AssignTrgToSrcStr(_Value, _Line);

    if (!_Key || _DataSpec != StrFile::KVPAIR)
      _DataSpec = StrFile::VALUE;
    else
      _DataSpec = StrFile::KVPAIR;

    _TypeSpec = FLOAT;
    _Fval = Val_;
  }

  return Val_;
}

/****************************************************************************/
char* StrFile::SetToken(const char* Line_)
{
  if (_FileMode && Line_)
  {
    strcpy(_Line, Line_);
    _Token = AddToken(_Line, strlen(_Line));
    
    if (_Token)
    {
      if (!_Key || _DataSpec != StrFile::KVPAIR)
        _DataSpec = StrFile::TOKEN;
      else
        _DataSpec = StrFile::KVPAIR;

      _TypeSpec = STRING;
      _Sval = _Token;
      return _Token;
    }
  }
  
  return NULL;
}

/****************************************************************************/
int StrFile::SetIntToken(int Val_)
{
  if (_FileMode)
  {
    IntToStr(Val_, _Line);
    _Token = AddToken(_Line, strlen(_Line));

    if (_Token)
    {
      if (!_Key || _DataSpec != StrFile::KVPAIR)
        _DataSpec = StrFile::TOKEN;
      else
        _DataSpec = StrFile::KVPAIR;
    
      _TypeSpec = INT;
      _Ival = Val_;
      return Val_;
    }
  }

  return 0;
}

/****************************************************************************/
double StrFile::SetFloatToken(double Val_, int prec)
{
  if (_FileMode)  
  {
    FloatToStr(Val_, _Line, 0, prec);
    _Token = AddToken(_Line, strlen(_Line));

    if (_Token)
    {
      if (!_Key || _DataSpec != StrFile::KVPAIR)
        _DataSpec = StrFile::TOKEN;
      else
        _DataSpec = StrFile::KVPAIR;
    
      _TypeSpec = FLOAT;
      _Fval = Val_;
      return Val_;
    }
  }

  return 0;
}

/****************************************************************************/
char* StrFile::SetToken(size_t Start_, size_t End_)
{
  if (!_FileMode && _Value)
  {
    size_t len = strlen(_Value);
    size_t nlen = End_ - Start_;

    // start position should be less than string length and new string
    // length should be greater or at least equal to the old string length
    if (len > Start_ && len - Start_ >= nlen)
    {
      // add padded quote characters to new string length
      if (_StrQuote && _QuoteFound)
        nlen += 2 * strlen(_StrQuote);
    
      _Token = _Value + Start_;
      return AddToken(_Token, nlen);
    }
  }

  return NULL;
}

/****************************************************************************/
char* StrFile::AddToken(const char* Line_, size_t nlen)
{
  if (Line_ && nlen)
  {
    bool noqt = false;
    size_t slen = ::SafeStrLen(_StrQuote);
  
    if (!_Tokens)
      InitTokens(NUMTOKENS_START);

    if (_FileMode)
    {
      noqt = _StrQuote && strncmp(Line_, _StrQuote, slen) != 0;
      if (noqt)
        nlen += 2 * slen;
    }
  
    char* temp = (char*)RawAllocateWith(MEMMATRIX, sizeof(char) * (nlen+1));
    _Tokens[_NumTokens++] = strncpy(temp, Line_, nlen);    
    temp[nlen] = 0;

    if (_FileMode)
    {    
      if (noqt)
        AddPadding(temp, _StrQuote);
    }
    else
    {
      if (_StrQuote && _TokenEnd && _QuoteFound &&
          (_DataFound & StrFile::TOKEN))
      {
        RemovePadding(temp, _StrQuote);
        AddPadding(temp, _StrQuote);
      }
    }

    if (_WhiteSpace)
      RemovePadding(temp, _WhiteSpace);

    if (_NumTokens >= _TokenLimit)
      IncreaseTokens(TOKENS_INCREMENT);

    return temp;
  }

  return NULL;
}

/****************************************************************************/
size_t StrFile::WriteSection()
{
  size_t slen, rlen, llen;
  
  slen = _Line ? strlen(_Line):0;
  llen = _LeftBracket ? strlen(_LeftBracket):0;
  rlen = _RightBracket ? strlen(_RightBracket):0;

  if ((_DataSpec & StrFile::SECTION) && _Fout &&
      llen && strncmp(_Line, _LeftBracket, llen) == 0 &&
      rlen && strncmp(_Line+slen-rlen, _RightBracket, rlen) == 0)
  {
    *_Fout <<_Line <<endl;
    return (slen+1);
  }

  return 0;
}

/****************************************************************************/
size_t StrFile::WriteKey()
{
  size_t len = _Key ? strlen(_Key):0;
  size_t alen = _Assignment ? strlen(_Assignment):0;

  if (len && alen && _Fout)
    *_Fout <<_Key <<_Assignment;

  return (len + alen);
}

/****************************************************************************/
size_t StrFile::WriteValue()
{
  size_t len = _Value ? strlen(_Value):0;

  if (len && _Fout)
    if (_DataSpec & StrFile::TOKEN)
    {
      if (_NumTokens && _Tokens[_NumTokens-1])
        if (_Delims)
          return WriteToken();
        else
          return WriteSepItem();
    }
    else
    {
      *_Fout <<_Value <<endl;
      ++len;
    }

  return len;
}

/****************************************************************************/
size_t StrFile::WritePair()
{
  size_t len = 0;
  len += WriteKey();
  len += WriteValue();

  if ((_DataSpec & StrFile::TOKEN) && _Fout &&
      _NumTokens && _Tokens[_NumTokens-1])
  {
    if (_Delims)
    {
      while (_TokenIndex < _NumTokens)
        len += WriteToken();
    }
    else
    {
      while (_TokenIndex < _NumTokens)
        len += WriteSepItem();
    }
  }

  return len;
}

/****************************************************************************/
size_t StrFile::WriteToken()
{
  size_t len = 0;
  size_t tlen = 0;

  if (_NumTokens && _Tokens[_NumTokens-1] &&
      _TokenIndex < _NumTokens &&
      (_DataSpec & StrFile::TOKEN) && _Fout)
  {
    if (_Delims && _TokenIndex > 0)
    {
      *_Fout <<_Delims;
      len += strlen(_Delims);
    }

    tlen = ::SafeStrLen(_Tokens[_TokenIndex]);
    
    if (tlen)
    {
      *_Fout <<_Tokens[_TokenIndex++];
      len += tlen;

      if (_TokenIndex == _NumTokens)
      {
        *_Fout <<endl;
        ++len;
      }
    }
  }

  return len;
}

/****************************************************************************/
size_t StrFile::WriteSepItem()
{
  size_t len = 0;
  size_t tlen = 0;

  if (_NumTokens && _Tokens[_NumTokens-1] &&
      _TokenIndex < _NumTokens &&
      (_DataSpec & StrFile::TOKEN) && _Fout)
  {
    if (_WhiteSpace && _TokenIndex > 0)
    {
      *_Fout <<_WhiteSpace;
      len += strlen(_WhiteSpace);
    }
    else
    {
      *_Fout <<" ";
      ++len;
    }

    tlen = ::SafeStrLen(_Tokens[_TokenIndex]);

    if (tlen)
    {
      *_Fout <<_Tokens[_TokenIndex++];
      len += tlen;

      if (_TokenIndex == _NumTokens)
      {
        *_Fout <<endl;
        ++len;
      }
    }
  }

  return len;
}

/****************************************************************************/
size_t StrFile::ReadSection()
{
  size_t len = _Line ? strlen(_Line):0;

  if (len)
  {
    _DataFound |= StrFile::SECTION;
    SetSection(_Line);

    memmove(_Line, _Line+1, len);
    _Line[len-2] = 0;

    if (_DataSpec & StrFile::KVPAIR)
    {
      if (_Assignment && strstr(_Line, _Assignment))
        ReadPair();
      else
        ReadValue();
    }
    else if (_DataSpec & StrFile::VALUE)
      ReadValue();
  }

  return len;
}

/****************************************************************************/
size_t StrFile::ReadValue()
{
  if (_Key)
    _Key[0] = 0;

  size_t len = _Line ? strlen(_Line):0;

  if (len)
  {
    _DataFound |= StrFile::VALUE;  
    SetValue(_Line);

    if (_DataSpec & StrFile::TOKEN)
    {
      ResetEos();
    
      if (_Delims && strpbrk(_Line, _Delims))
        ReadToken();
      else
        ReadSepItem();
    }    
  }

  return len;
}

/****************************************************************************/
size_t StrFile::ReadPair()
{
  int len = _Line ? strlen(_Line):0;
  int alen = _Assignment ? strlen(_Assignment):0;

  if (len && alen)
  {
    char* pos = strstr(_Line, _Assignment);
    char c = *pos;
    *pos = 0;

    _DataFound |= StrFile::KVPAIR;
    SetKey(_Line);
    
    *pos = c;
    pos += alen;

    SetValue(pos);

    if (_DataSpec & StrFile::TOKEN)
    {
      ResetEos();
    
      if (_Delims && strpbrk(_Line, _Delims))
        ReadToken();
      else
        ReadSepItem();
    }
  }

  return (alen ? len:0);
}

/****************************************************************************/
size_t StrFile::ReadToken()
{
  if (_Eos)
    return 0;

  size_t CopyLen_;

  if (_NextPos > _Index)
    _Index = _NextPos;
  
  if (::SafeStrLen(_Delims) && _Value)
    CopyLen_ = FindNextWord(_Value, _Delims, _Index, _NextPos, _StrQuote,
                            _SkipEmptyFields, _DelimsEsc, _StrQuoteEsc);
  else
    CopyLen_ = ::SafeStrLen(_Value);

  if (CopyLen_)
  {
    _TokenEnd = CopyLen_;
    _QuoteFound = FALSE;
    int qlen = _StrQuote ? strlen(_StrQuote):0;
    
    if (_StrQuote && qlen &&
          (_QuoteFound = strncmp(_Value+_Index+_TokenEnd-qlen, _StrQuote, qlen) == 0))
      _TokenEnd -= qlen;

    _DataFound |= StrFile::TOKEN;
    SetToken(_Index, _NextPos);
    FindTokenType(_Tokens[_NumTokens-1]);
  }
  else
  {
    _Eos = _Value[_Index] == 0 || _Index >= strlen(_Value);    
    if (!_Eos)
      _Token = _Value + strlen(_Value);
  }

  return (_Token ? CopyLen_:0);
}

/****************************************************************************/
size_t StrFile::ReadSepItem()
{
  if (_Eos)
    return 0;

  if (_NextPos > _Index)
    _Index = _NextPos;

  size_t len = 0;

  if (_Value && strlen(_Value))
    if (!_NextPos)
    {
      AssignTrgToSrcStr(_Tokenated, _Value);
      const char* SepStr_ = _WhiteSpace ? _WhiteSpace:" \t\n";

      if (strpbrk(_Token, SepStr_))
      {
        _Token = strtok(_Tokenated, SepStr_);
        _NoSep = 0;
      }
      else
      {
        _Token = _Tokenated;
        _NoSep = 1;
      }
      
      len = _Token ? strlen(_Token):0;
      _Index = _NextPos;
      _NextPos += len;
    }
    else if (_Index == _NextPos && !_NoSep)
    {
      _Token = strtok(0, (_WhiteSpace ? _WhiteSpace:" \t\n"));
      len = _Token ? strlen(_Token):0;
      _Index = _NextPos;
      _NextPos += len;
    }
    else if (_NoSep)
    {
      len = 0;
      _Token = NULL;
      _NoSep = 0;
    }
  
  if (_Token && len)
  {
    _DataFound |= StrFile::WSSITEM;  
    AddToken(_Token, len);
    FindTokenType(_Tokens[_NumTokens-1]);
  }
  else
    _Eos = 1;

  return (_Token ? len:0);
}

/****************************************************************************/
const char* StrFile::DataSpecStr() const
{
  _ReturnStr[0] = 0;
  
  if (_DataSpec & StrFile::SECTION)
    strcpy(_ReturnStr, "Section");

  if (_DataSpec & StrFile::KVPAIR)
  {
    if (_ReturnStr[0])
      strcat(_ReturnStr, "|KVPair");
    else
      strcpy(_ReturnStr, "KVPair");
  }

  if (_DataSpec & StrFile::VALUE)
  {
    if (_ReturnStr[0])
      strcat(_ReturnStr, "|Value");
    else
      strcpy(_ReturnStr, "Value");
  }

  if (_DataSpec & StrFile::TOKEN)
  {
    if (_ReturnStr[0])
      strcat(_ReturnStr, "|Token");
    else
      strcpy(_ReturnStr, "Token");
  }

  if (_DataSpec & StrFile::WSSITEM)
  {
    if (_ReturnStr[0])
      strcat(_ReturnStr, "|WSSItem");
    else
      strcpy(_ReturnStr, "WSSItem");
  }

  return _ReturnStr;
}

/****************************************************************************/
const char* StrFile::TypeSpecStr() const
{
  _ReturnStr[0] = 0;
  
  if (_TypeSpec == StrFile::INT)
    strcpy(_ReturnStr, "Int");

  if (_TypeSpec == StrFile::FLOAT)
  {
    if (_ReturnStr[0])
      strcat(_ReturnStr, "|Float");
    else
      strcpy(_ReturnStr, "Float");
  }

  if (_TypeSpec == StrFile::STRING)
  {
    if (_ReturnStr[0])
      strcat(_ReturnStr, "|String");
    else
      strcpy(_ReturnStr, "String");
  }

  return _ReturnStr;
}

/****************************************************************************/
const char* StrFile::DataFoundStr() const
{
  _ReturnStr[0] = 0;
  
  if (_DataFound & StrFile::SECTION)
    strcpy(_ReturnStr, "Section");

  if (_DataFound & StrFile::KVPAIR)
  {
    if (_ReturnStr[0])
      strcat(_ReturnStr, "|KVPair");
    else
      strcpy(_ReturnStr, "KVPair");
  }

  if (_DataFound & StrFile::VALUE)
  {
    if (_ReturnStr[0])
      strcat(_ReturnStr, "|Value");
    else
      strcpy(_ReturnStr, "Value");
  }

  if (_DataFound & StrFile::TOKEN)
  {
    if (_ReturnStr[0])
      strcat(_ReturnStr, "|Token");
    else
      strcpy(_ReturnStr, "Token");
  }

  if (_DataFound & StrFile::WSSITEM)
  {
    if (_ReturnStr[0])
      strcat(_ReturnStr, "|WSSItem");
    else
      strcpy(_ReturnStr, "WSSItem");
  }

  return _ReturnStr;
}

/****************************************************************************/
const char* StrFile::TypeFoundStr() const
{
  if (_TypeFound == StrFile::INT)
    strcpy(_ReturnStr, "Int");

  if (_TypeFound == StrFile::FLOAT)
    strcpy(_ReturnStr, "Float");

  if (_TypeFound == StrFile::STRING)
    strcpy(_ReturnStr, "String");

  return _ReturnStr;
}

/****************************************************************************/
StrFile& StrFile::StripToken(Boolean Flag_)
{
  _StripToken = Flag_;
  return *this;
}

/****************************************************************************/
StrFile& StrFile::SkipEmptyFields(Boolean Flag_)
{
  _SkipEmptyFields = Flag_;
  return *this;
}

/****************************************************************************/
StrFile& StrFile::SetDataType(int Type_)
{
  _DataSpec = Type_;
  return *this;
}

/****************************************************************************/
StrFile& StrFile::SetTokenType(int Type_)
{
  _TypeSpec = Type_;
  return *this;
}

/****************************************************************************/
StrFile& StrFile::IgnoreEos(Boolean Flag_)
{
  _IgnoreEos = Flag_;
  return *this;
}

/****************************************************************************/
StrFile& StrFile::ResetEos()
{
  _Index = _NextPos = _TokenEnd = _Eos = 0;
  return *this;
}

/****************************************************************************/
StrFile& StrFile::operator = (const StrFile& Obj_)
{
  if (this != &Obj_)
  {
    int x;
    for (x = 0; x < _NumTokens; x++)
      RawDeleteArray(_Tokens[x]);
  
    RawDeleteArray(_Tokens);
    RawDeleteArray(_Fname);

    RawDeleteArray(_Tokenated);
    RawDeleteArray(_ReturnStr);
  
    RawDeleteArray(_Line);
    RawDeleteArray(_Delims);
    RawDeleteArray(_StrQuote);

    RawDeleteArray(_Assignment);
    RawDeleteArray(_LeftBracket);
    RawDeleteArray(_RightBracket);
    RawDeleteArray(_WhiteSpace);

    RawDeleteArray(_Section);
    RawDeleteArray(_Key);
    RawDeleteArray(_Value);
    CloseFile();
  
    _Fname = Obj_._Fname ? NewString(Obj_._Fname):NULL;
    _FileMode = Obj_._FileMode;

    _Success = 0;
    _SpecTokenFound = 0;
    _QuoteFound = 0;
    _IgnoreEos = 1;
    _Eos = 0;
    _NoSep = 0;
    
    _StripToken = Obj_._StripToken;
    _SkipEmptyFields = Obj_._SkipEmptyFields;

    _DataSpec = Obj_._DataSpec;
    _DataFound = 0;
    _TypeSpec = Obj_._TypeSpec;
    _TypeFound = 0;

    _LineNo = 0;
    _TokenNo = 0;
    _BufLen = 0;
    _Sval = 0;

    _Tokenated = 0;
    _ReturnStr = 0;

    _Line = 0;
    _Delims = Obj_._Delims ? NewString(Obj_._Delims):NULL;
    _StrQuote = Obj_._StrQuote ? NewString(Obj_._StrQuote):NULL;
    _DelimsEsc = Obj_._DelimsEsc;
    _StrQuoteEsc = Obj_._StrQuoteEsc;

    _Assignment = Obj_._Assignment ? NewString(Obj_._Assignment):NULL;
    _LeftBracket = Obj_._LeftBracket ? NewString(Obj_._LeftBracket):NULL;
    _RightBracket = Obj_._RightBracket ? NewString(Obj_._RightBracket):NULL;
    _WhiteSpace = Obj_._WhiteSpace ? NewString(Obj_._WhiteSpace):NULL;

    _Section = 0;
    _Key = 0;
    _Value = 0;

    _Index = 0;
    _NextPos = 0;
    _TokenEnd = 0;
    _Token = 0;

    _TokenIndex = 0;
    _TokenLimit = 0;
    _NumTokens = 0;
    _Tokens = 0;

    _ReturnStr = (char*)RawAllocateWith(MEMMATRIX, sizeof(char) * RETURNBUFFER_MAX);
    InitLine(StrFile::BUFLEN);
    InitTokens(NUMTOKENS_START);

    if (_Fname)
      OpenFile();
  }

  return *this;
}

/****************************************************************************/
StrFile& StrFile::SetOutputFile(const char* Fname_, Boolean Append_)
{
  if (Fname_)
  {
    AssignTrgToSrcStr(_Fname, Fname_);
    OpenFile(1, Append_);

    if (!_Line)
      InitLine(StrFile::BUFLEN);    
  }

  return *this;
}

/****************************************************************************/
StrFile& StrFile::SetInputFile(const char* Fname_, Boolean OpenFile_)
{
  if (Fname_)
  {
    AssignTrgToSrcStr(_Fname, Fname_);

    if (OpenFile_)
      OpenFile();
  }

  return *this;
}

/****************************************************************************/
int StrFile::OpenFile(int Mode_, Boolean Append_)
{
  if (Mode_)
  {  
    if (_Fname && strlen(_Fname))
    {
      if (!_FileMode)
        CloseFile();
    
      if (!_Fout)
        _Fout = new ofstream;
      else
      {
        _Fout->close();
        _NoSep = 0;
      }

      _Fout->clear(ios::goodbit);
      _Fout->open(_Fname, (Append_ ? ios::app:ios::out));
      _FileMode = Mode_;

      return _Fout->good();
    }
  }
  else
  {
    if (_Fname && strlen(_Fname))
    {
      if (_FileMode)
        CloseFile();
    
      if (!_Fin)
        _Fin = new ifstream;
      else
      {
        _Fin->close();
        ResetEos();
      }

      _Fin->clear(ios::goodbit);
      _Fin->open(_Fname, ios::in);
      _FileMode = Mode_;

      return _Fin->good();
    }
  }

  return 0;
}

/****************************************************************************/
int StrFile::CloseFile()
{
  if (_FileMode)
  {
    if (_Fout)
    {
      _Fout->close();
      _NoSep = 0;
      
      return _Fout->good();
    }
  }
  else
  {
    if (_Fin)
    {
      _Fin->close();
      ResetEos();

      return _Fin->good();
    }
  }

  return 0;
}

/****************************************************************************/
const char* StrFile::Token(size_t Index_) const
{
  if (0 <= Index_ && Index_ < _NumTokens)
    return _Tokens[Index_];

  return NULL;
}

/****************************************************************************/
// PURPOSE:
//   Returns the first piece of data available
//
// RETURNS:
//   Returns the Line read if only a line with no distinct properties is read
//   Returns the Section read if a section name is found
//   Returns the key in a kv-pair read if a key is found
//   Returns the value in a kv-pair read if a value string is found
//   Returns the last token found if a token is read
//   Returns the last whitespace found if a whitespace is read
//
const char* StrFile::GiveDataRetrieved1() const
{
  return
  (
    (_DataFound & StrFile::LINE)    ? _Line:
    (_DataFound & StrFile::SECTION) ? _Section:
    (_DataFound & StrFile::KVPAIR)  ? _Key:
    (_DataFound & StrFile::VALUE)   ? _Value:
    (_DataFound & StrFile::TOKEN)   ? _Tokens[_NumTokens-1]:
    (_DataFound & StrFile::WSSITEM) ? _Tokens[_NumTokens-1]:NULL
  );
}

/****************************************************************************/
// PURPOSE:
//   Returns the second piece of data available
//
// RETURNS:
//   Returns the value string read in a kv-pair entry
//
const char* StrFile::GiveDataRetrieved2() const
{
  return ((_DataFound & StrFile::KVPAIR) ||
             ((_DataFound & StrFile::SECTION) &&
              (_DataFound & StrFile::VALUE)) ? _Value:NULL);
}

/****************************************************************************/
// PURPOSE:
//   Returns the third piece of data available
//
// RETURNS:
//   Returns the last token found if a token is read
//
const char* StrFile::GiveDataRetrieved3() const
{
  return (((_DataFound & StrFile::TOKEN) ||
           (_DataFound & StrFile::WSSITEM)) ? _Tokens[_NumTokens-1]:NULL);
}

/****************************************************************************/
// PURPOSE:
//   Returns the data specified given the data index type passed as an
//   argument to the function.
//
// RETURNS:
//   The first, second or third piece of data corresponding to argument
//   value 1, 2 or 3 as the argument value passed to the function.
//
const char* StrFile::GiveDataRetrieved(int Index_) const
{
  return ((Index_ == 1) ? GiveDataRetrieved1():
          (Index_ == 2) ? GiveDataRetrieved2():
          (Index_ == 3) ? GiveDataRetrieved3():NULL);
}

/****************************************************************************/
// PURPOSE:
//   Returns the data available after reading an entry in the file
//
// RETURNS:
//   Returns 1 if a section name is found or a value string is found
//   Returns 2 if a Key-Value pair is found
//   REturns 3 if more than one token is found within the value string
//
int StrFile::DataAvailable() const
{
  int Avail_ = 0;
  
  if (_DataFound & StrFile::SECTION)
    Avail_ = 1;

  if (_DataFound & StrFile::VALUE)
    Avail_ = 1;

  if (_DataFound & StrFile::KVPAIR)
  {
    Avail_ = 2;

    if ((_DataFound & StrFile::TOKEN) ||
        (_DataFound & StrFile::WSSITEM))
      ++Avail_;
  }

  return Avail_;
}

/****************************************************************************/
const char* StrFile::DataReadStr(int Num_, Boolean CheckTok_) const
{
  return ((_DataFound & StrFile::SECTION) ?
              ((Num_ == 1) ? "Section":
               (Num_ == 2 && (_DataFound & StrFile::VALUE)) ? "Value":""):
          (_DataFound & StrFile::VALUE)   ?
              ((Num_ == 1) ? "Value":
               (Num_ == 3 && (_DataFound & StrFile::TOKEN))   ? "Token":
               (Num_ == 3 && (_DataFound & StrFile::WSSITEM)) ? "WssItem":""):
          (((_DataFound & StrFile::KVPAIR) && (!CheckTok_ || Num_ == 2)) ?
              ((Num_ == 1) ? "Key":
               (Num_ == 2) ? "Value":" "):
              (((_DataFound & StrFile::TOKEN) && CheckTok_)   ? "Token":
               ((_DataFound & StrFile::WSSITEM) && CheckTok_) ? "WssItem":" ")));
}

/****************************************************************************/
int StrFile::TokenAvailable() const
{
  return ((_DataFound & StrFile::TOKEN) ||
          (_DataFound & StrFile::WSSITEM));
}

/****************************************************************************/
MEMORYOPS_DEFN(StrFile)

/****************************************************************************/
#if STRFILE_DEBUG
int main(int argc, char* argv[])
{
  StrFile* sfile = new StrFile;
  sfile->SetInputFile("autorun.inf", TRUE);
  sfile->SetDataType(StrFile::SECTION | StrFile::KVPAIR | StrFile::TOKEN);
  
  sfile->SetLeftBracket("[");
  sfile->SetRightBracket("]");
  sfile->SetAssignment("=");
  sfile->SetWhiteSpace(" \t\n");
  sfile->SetDelims(", ");
  sfile->SetDelimEsc('\\');
  sfile->SetStrQuote("\"");
  sfile->SetStrQuoteEsc('\\');

  cout <<"Data Specified: " <<sfile->DataSpecStr() <<endl;

  while (!sfile->FilePtr()->eof())
  {
    sfile->ReadFile();

    if (sfile->Success())
    {
      cout <<"Data Found == " <<sfile->DataFoundStr() <<endl;
      cout <<"Type Found == " <<sfile->TypeFoundStr() <<":" <<endl;
           
      if (sfile->DataAvailable() > 1)
      {
        cout <<"\t" <<sfile->DataReadStr(1, FALSE)
             <<" == " <<sfile->GiveDataRetrieved(1);
      
        cout <<" : " <<sfile->DataReadStr(2, FALSE)
             <<" == " <<sfile->GiveDataRetrieved(2);
             
        if (sfile->TokenAvailable())
          cout <<endl <<"\t\t"
               <<sfile->DataReadStr(1, TRUE) <<" == "
               <<sfile->GiveDataRetrieved(3);
      }
      else
      {
        if ((sfile->DataFound() & StrFile::SECTION) &&
            (sfile->DataFound() & StrFile::VALUE))
        {
          cout <<"\t" <<sfile->DataReadStr(1, FALSE)
               <<" == " <<sfile->GiveDataRetrieved(1) <<endl;
               
          cout <<"\t" <<sfile->DataReadStr(2, FALSE)
               <<" == " <<sfile->GiveDataRetrieved(2);
        }
        else
          cout <<"\t" <<sfile->DataReadStr(1, TRUE)
               <<" == " <<sfile->GiveDataRetrieved(1);
      }

      cout <<endl;
    }
  }

  // game options
  int x;
  
  sfile->SetInputFile("gameopt.bj", TRUE);
  sfile->SetDataType(StrFile::KVPAIR | StrFile::TOKEN);
  sfile->SetAssignment("=");
  cout <<endl <<"parsing file: gameopt.bj" <<endl;

  sfile->SetTokenType(StrFile::INT);

  for (x = 0; x < 12; x++)
  {
    sfile->ReadFile();
    if (sfile->Success() &&
         (sfile->DataFound() & StrFile::KVPAIR) &&
         (sfile->TypeSpec() & StrFile::INT))
      cout <<"token = " <<sfile->GiveDataRetrieved(1) <<" : "
           <<"ival = " <<sfile->Ival() <<" : "
           <<"type found = " <<sfile->TypeFoundStr() <<endl;           
    else
      cout <<"type found = " <<sfile->TypeFoundStr() <<endl;
  }

  sfile->SetTokenType(StrFile::STRING);
  sfile->ReadFile();
  
  if (sfile->Success() &&
       (sfile->DataFound() & StrFile::KVPAIR) &&
       (sfile->TypeSpec() & StrFile::STRING))
    cout <<"token = " <<sfile->GiveDataRetrieved(1) <<" : "
         <<"sval = " <<sfile->Sval() <<" : "
         <<"type found = " <<sfile->TypeFoundStr() <<endl;
  else
    cout <<"type found = " <<sfile->TypeFoundStr() <<endl;

  sfile->SetTokenType(StrFile::INT);
  
  for (x = 0; x < 3; x++)
  {
    sfile->ReadFile();
    if (sfile->Success() &&
         (sfile->DataFound() & StrFile::KVPAIR) &&
         (sfile->TypeSpec() & StrFile::INT))
      cout <<"token = " <<sfile->GiveDataRetrieved(1) <<" : "
           <<"ival = " <<sfile->Ival() <<" : "
           <<"type found = " <<sfile->TypeFoundStr() <<endl;           
    else
      cout <<"type found = " <<sfile->TypeFoundStr() <<endl;
  }

  // player options
  sfile->SetInputFile("playopt.bj", TRUE);
  sfile->SetDataType(StrFile::KVPAIR | StrFile::TOKEN);  
  sfile->SetAssignment("=");
  cout <<endl <<"parsing file: playopt.bj" <<endl;
  
  sfile->SetTokenType(StrFile::INT);

  for (x = 0; x < 11; x++)
  {
    sfile->ReadFile();
    if (sfile->Success() &&
         (sfile->DataFound() & StrFile::KVPAIR) &&
         (sfile->TypeSpec() & StrFile::INT))
      cout <<"token = " <<sfile->GiveDataRetrieved(1) <<" : "
           <<"ival = " <<sfile->Ival() <<" : "
           <<"type found = " <<sfile->TypeFoundStr() <<endl;           
    else
      cout <<"type found = " <<sfile->TypeFoundStr() <<endl;
  }

  sfile->SetTokenType(StrFile::STRING);

  for (x = 0; x < 3; x++)
  {
    sfile->ReadFile();
    if (sfile->Success() &&
         (sfile->DataFound() & StrFile::KVPAIR) &&
         (sfile->TypeSpec() & StrFile::STRING))
      cout <<"token = " <<sfile->GiveDataRetrieved(1) <<" : "
           <<"sval = " <<sfile->Sval() <<" : "
           <<"type found = " <<sfile->TypeFoundStr() <<endl;
    else
      cout <<"type found = " <<sfile->TypeFoundStr() <<endl;
  }

  // game options
  sfile->SetOutputFile("strfile.out", FALSE);
  sfile->SetDataType(StrFile::KVPAIR);
  sfile->SetTokenType(StrFile::INT);
  sfile->SetAssignment("=");
  int Success_ = sfile->FilePtr()->good();
  x = 100;

  if (Success_)
  {
    sfile->SetKey("lower limit");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("table bet limit");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("numplayers");
    sfile->SetIntValue(x++);
    sfile->WritePair();
    
    sfile->SetKey("surrender allowed");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("insurance allowed");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("append to results file");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("deal 2nd card");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("hit on split aces");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("maxrounds");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("manual play");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("primary player");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("decks used");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("results file");
    sfile->SetValue("results-file");
    sfile->WritePair();

    sfile->SetKey("initial pot");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("jackpot");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("jackpot bet");
    sfile->SetIntValue(x++);
    sfile->WritePair();
    
    sfile->CloseFile();
  }

  // player options
  sfile->SetOutputFile("strfile.out", TRUE);
  sfile->SetDataType(StrFile::KVPAIR);
  sfile->SetTokenType(StrFile::INT);
  sfile->SetAssignment("=");
  Success_ = sfile->FilePtr()->good();
  x = 200;

  if (Success_)
  {
    sfile->SetKey("personal bet limit");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("preferred bet amount");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("double next bet on loss");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("break at limit");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("manual play");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("autobetting");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("quit on consec. loss");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("quit on net session loss");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("quit on net total loss");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("quit on net session win");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("quit on net total win");
    sfile->SetIntValue(x++);
    sfile->WritePair();

    sfile->SetKey("blackjack strategy file");
    sfile->SetValue("results-file");
    sfile->WritePair();

    sfile->SetKey("betting strategy file");
    sfile->SetValue("betting-strategy");
    sfile->WritePair();

    sfile->SetKey("poker strategy file");
    sfile->SetValue("poker-strategy");
    sfile->WritePair();

    sfile->CloseFile();
  }
  
  return 0;
}

#endif
#endif
