#ifndef STRFILE_H
#define STRFILE_H
#ifndef INCL_FSTREAM_H
  #define INCL_FSTREAM_H
  #include <fstream>
#endif
#ifndef STRUTIL_H
  #include "strutil.h"
#endif

#define STRFILE_DEBUG   0

// structure for storing Data Settings:
// StrFile file format:
//
// [section name]
// key=token,token,token...
// ...
//
// if StrFile file has sections then
//   section name is enclosed within "[" and "]"
//   stripping section name will put contents into
//     _Section and _Value
//     _Value will be further stripped into tokens and stored in
//     _Token with position of the token found saved starting in
//     _Index and ending in _NextPos
//
// if Line Data has assignments then
//   format is: key=value if HasKVPair is true,
//     otherwise format is: value and _Key is assigned NULL
//
//   stripping line data will put contents into
//     _Key and _Value if assignment found,
//     otherwise _Value is assigned the line data and
//     _Key assigned set NULL
//
// if Value Data is separated into tokens then
//   format is: token,token,token
//   with "," or some other character being the delimiter.
//
//   _Value data is stripped into tokens with results placed into
//   _Token and _Index and _NextPos set with starting and ending
//   position of the token found within the _Value char string.
//

class StrFile
{
  public:
    enum
    {
      NONE    = 0,
      LINE    = 1,
      SECTION = 2,
      KVPAIR  = 4,
      VALUE   = 8,
      TOKEN   = 16,
      WSSITEM = 32,
    
      STRING  = 101,
      INT     = 102,
      FLOAT   = 103,

      BUFLEN           = 512,
      RETURNBUFFER_MAX = 80,
      TOKENS_INCREMENT = 10,
      NUMTOKENS_START  = 11
    };

  private:
    ifstream* _Fin;
    ofstream* _Fout;
    char* _Fname;
    int _FileMode;
  
    // Returned
    Boolean _Success;
    Boolean _SpecTokenFound;
    Boolean _QuoteFound;
    Boolean _IgnoreEos;
    Boolean _Eos;
    Boolean _NoSep;

    // Preset
    Boolean _StripToken;
    Boolean _SkipEmptyFields;

    int _DataSpec;        // Preset
    int _DataFound;       // Returned
    int _TypeSpec;        // Preset
    int _TypeFound;       // Returned

    // Returned
    int _LineNo;
    int _TokenNo;
    size_t _BufLen;

    // Returned
    union
    {
      char* _Sval;
      double _Fval;
      int _Ival;
    };

    char* _Tokenated;     // Internal use
    char* _ReturnStr;     // Internal use
    
    char* _Line;          // Returned
    char* _Delims;        // Preset
    char* _StrQuote;      // Preset
    char _DelimsEsc;      // Preset
    char _StrQuoteEsc;    // Preset

    // Preset
    char* _Assignment;
    char* _LeftBracket;
    char* _RightBracket;
    char* _WhiteSpace;

    // Returned by function
    char* _Section;
    char* _Key;
    char* _Value;

    size_t _Index;        // Returned
    size_t _NextPos;      // Returned
    size_t _TokenEnd;     // internal use
    char* _Token;         // Returned

    size_t _TokenIndex;   // Internal use
    size_t _TokenLimit;   // Internal use
    size_t _NumTokens;    // Returned
    char** _Tokens;       // Returned

    int FindTokenType(char* Token_);

    const char* GiveDataRetrieved1() const;
    const char* GiveDataRetrieved2() const;
    const char* GiveDataRetrieved3() const;    

  public:
    StrFile(const char* Fname_ = NULL,
            Boolean AllocBuf_=TRUE, Boolean AllocArray_=TRUE);
    StrFile(const StrFile& Obj_);
    ~StrFile();

    StrFile& WriteFile();
    StrFile& WriteFile(ofstream& Out_);

    StrFile& ReadFile();
    StrFile& ReadFile(ifstream& In_);

    // Initialized
    char* InitLine(size_t Size_);
    char* ReadLine(const char* Line_)
        { return (strlen(Line_) < _BufLen ?
                    strcpy(_Line, Line_):NULL); }

    char** InitTokens(int Size_);
    char** IncreaseTokens(int Incr_);

    // Preset
    char* SetDelims(const char* Line_)
        { if (_DelimsEsc && InCharSet(_DelimsEsc, _Delims))
            return 0;
          return AssignTrgToSrcStr(_Delims, Line_); }
    char* SetStrQuote(const char* Line_)
        { if (_StrQuoteEsc && InCharSet(_StrQuoteEsc, _StrQuote))
            return 0;
          return AssignTrgToSrcStr(_StrQuote, Line_); }
    char* SetAssignment(const char* Line_)
        { return AssignTrgToSrcStr(_Assignment, Line_); }
    char* SetLeftBracket(const char* Line_)
        { return AssignTrgToSrcStr(_LeftBracket, Line_); }
    char* SetRightBracket(const char* Line_)
        { return AssignTrgToSrcStr(_RightBracket, Line_); }
    char* SetWhiteSpace(const char* Line_)
        { return AssignTrgToSrcStr(_WhiteSpace, Line_); }
    char SetDelimEsc(char Esc_)
        { return ((_DelimsEsc = Esc_)); }
    char SetStrQuoteEsc(char Esc_)
        { return ((_StrQuoteEsc = Esc_)); }

    // Return by processing function
    char* SetSection(const char* Line_);
    char* SetKey(const char* Line_);
    char* SetValue(const char* Line_);
    int SetIntValue(int Val_);
    double SetFloatValue(double Val_, int prec=6);

    char* SetToken(const char* Line_);
    int SetIntToken(int Val_);
    double SetFloatToken(double Val_, int prec=6);
    
    char* SetToken(size_t Start_, size_t End_);
    char* AddToken(const char* Line_, size_t nlen);
    void ResetTokenIndex()
        { _TokenIndex = 0; }

    size_t ReadSection();
    size_t ReadPair();
    size_t ReadValue();
    size_t ReadToken();
    size_t ReadSepItem();

    size_t WriteSection();
    size_t WritePair();
    size_t WriteKey();
    size_t WriteValue();
    size_t WriteToken();
    size_t WriteSepItem();

    Boolean Success() const
        { return _Success; }
    Boolean SpecTokenFound() const
        { return _SpecTokenFound; }
    Boolean Eos() const
        { return _Eos; }

    int FileMode() const
        { return _FileMode; }
    int DataSpec() const
        { return _DataSpec; }
    int TypeSpec() const
        { return _TypeSpec; }
    int DataFound() const
        { return _DataFound; }
    int TypeFound() const
        { return _TypeFound; }
        
    const char* DataSpecStr() const;
    const char* TypeSpecStr() const;
    const char* DataFoundStr() const;
    const char* TypeFoundStr() const;
    const char* DataReadStr(int Num_, Boolean CheckTok_) const;
        
    int LineNo() const
        { return _LineNo; }
    int TokenNo() const
        { return _TokenNo; }
    size_t BufferLength() const
        { return _BufLen; }

    const char* Sval() const
        { return _Sval; }
    int Ival() const
        { return _Ival; }
    double Fval() const
        { return _Fval; }

    int DataAvailable() const;
    int TokenAvailable() const;    
    const char* GiveDataRetrieved(int Index_) const;
    
    const char* LineRead() const
        { return _Line; }
    const char* Section() const
        { return _Section; }
    const char* Key() const
        { return _Key; }
    const char* Value() const
        { return _Value; }
    size_t TokenStart() const
        { return _Index; }
    size_t TokenEnd() const
        { return _NextPos; }
    const char* CurrentToken() const
        { return _Token; }
    size_t NumTokens() const
        { return _NumTokens; }
    ifstream* InFilePtr()
        { return _Fin; }
    ofstream* OutFilePtr()
        { return _Fout; }

    StrFile& operator = (const StrFile& Obj_);
    StrFile& SetOutputFile(const char* Fname_, Boolean Append_);
    StrFile& SetInputFile(const char* Fname_, Boolean OpenFile_);
    int OpenFile(int Mode_=0, Boolean Append_=FALSE);
    int CloseFile();
    
    StrFile& IgnoreEos(Boolean Flag_);
    StrFile& ResetEos();
    StrFile& StripToken(Boolean Flag_);
    StrFile& SkipEmptyFields(Boolean Flag_);
    StrFile& SetDataType(int Type_);
    StrFile& SetTokenType(int Type_);
    const char* Token(size_t Index_) const;

  #if OVERLOAD_NEW
      void* operator new (size_t Bytes_);
      void operator delete (void* Space_);

  #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
  #endif
  #endif
};

#endif
