#ifndef SYMBOLS_H
#define SYMBOLS_H

#ifndef INCL_STDIO_H
  #include <stdio.h>
  #define INCL_STDIO_H
#endif
#ifndef INCL_FSTREAM_H
  // #include <fstream.h>
  #include <fstream>
  #define INCL_FSTREAM_H
#endif
#ifndef INCL_IOMANIP_H
  // #include <iomanip.h>
  #include <iomanip>
  #define INCL_IOMANIP_H
#endif
#ifndef INCL_STDLIB_H
  #include <stdlib.h>
  #define INCL_STDLIB_H
#endif
#ifndef INCL_STRING_H
  #include <string.h>
  #define INCL_STRING_H
#endif
#ifndef INCL_CTYPE_H
  #include <ctype.h>
  #define INCL_CTYPE_H
#endif

#ifndef TYPEDEFS_H
  #include "typedefs.h"
#endif
#ifndef SAFESTR_H
  #include "safestr.h"
#endif
#ifndef BITVECTOR_H
  #include "bitvect.h"
#endif
#ifndef STRUTIL_H
  #include "strutil.h"
#endif
#ifndef LONGNUM_H
  #include "longnum.h"
#endif
#ifndef STRFILTER_H
  #include "strfilter.h"
#endif
#ifndef REGEXPR_H
  #include "exprclas.h"
#endif

#define SYMBOLSET_DEBUG         0
#define SYMBOLSET_DEBUG2        0
#define SYMBOLSET_DEBUG3        0

//
// (internal data)
//   (program specific data)
//     programindex 0:
//     currentscope 0:
//     scope level 1-0-1-3: /* pgrm-fnc-blk-seq */
//     brackbal 0:
//     delimcnt 0-0:        /* blk-seq */
//     argumentcnt 0-0:     /* blk-seq */
//     CdefExists 0:
//     CdefLimit 0:
//     branchblocked 0:
//     blkstklevel 0:
//     fncstklevel 0:
//     OrBranchOut 0:
//     BranchFound 0:
//     orbranchnum 0:
//
//   (symbol specific data)
//     name ccc:
//     symtype enum:
//     symattr enum:
//     datatype enum:
//     seqtype enum:
//     structtype enum:
//     decltype enum:
//     defntype enum:
//     CdefExists 0:
//     CdefLimit 0:
//     CdeChkBlock:
//     optype enum:
//     opstate 0:
//     sestate 0:
//     brksymtype 0:
//     brksymstate 0:
//
// -<symbolname:Symbol_Type:symattr:datatype:
//    [:Recur=n|Recur|End][:LeftSpace|RightSpace];>
//
// -<symbolname:OperatorSet_Type:OperatorSymbol:datatype:
//   opType[:Separator=c {if opType==TrinaryOp}]:
//   opPos[:ResultPreOp|ResultPostOp]
//   [:Resultant:defined_datatype=c[:AssignStr=c]]
//   [:Operand1:operand_datatype[=c {if opdt==defined | opdt==icasedefined}]]
//   [:Operand2:operand_datatype[=c {if opdt==defined | opdt==icasedefined}]]
//   [:Operand3:operand_datatype[=c {if opdt==defined | opdt==icasedefined}]];>
//
// -<symbolname:Sequence_Type:seqtype:
//    [decltype: {if seqtype==declaration} |
//     defntype: {if seqtype==definition}]
//   structtype:symattr:datatype:Continue=n|Recur=n|Recur|End
//    [:LeftSpace|RightSpace];>
//
// -<symbolname:BranchOut_Type:
//   structtype:symattr:datatype:Continue=n|End[:Separator=c];>
//
// -<symbolname:BracketedSymbol_Type:LeftBracket:datatype:
//   Delim=c|NoDelim:pushblk:pushfnc>
// (<BracketedSymbol_Type:Data:datatype:
//   Delim=c|NoDelim:Continue=n|End> |
//  <BracketedSymbol_Type:LeftBracket:datatype:
//   Delim=c|NoDelim:pushblk:pushfnc>)
// <BracketedSymbol_Type:RightBracket:datatype:
//  Delim=c|NoDelim:pushblk:pushfnc;>
//
// -<symbolname:Statement_Type:LeftBracket:datatype:structtype:
//   Delim=c|SeqTerm=c|AssignStr=c|NoDelim:pushblk:pushfnc>
// (<Statement_Type:(Data:datatype)|(Statement:symbolname):
//   Delim=c|SeqTerm=c|AssignStr=c|NoDelim:Continue=n|End> |
//  <Statement_Type:LeftBracket:datatype:structtype:
//   Delim=c|SeqTerm=c|AssignStr=c|NoDelim:pushblk:pushfnc>)
// (<Statement_Type:RightBracket:datatype[:Terminator]:
//   structtype:Delim=c|SeqTerm=c|AssignStr=c|NoDelim:pushblk:pushfnc;> |
//  <Statement_Type:Terminator:datatype:structtype:
//   Delim=c|SeqTerm=c|AssignStr=c|NoDelim:pushblk:pushfnc;>)
//
/****************************************************************************/
// SymbolSet class forward declaration
class SymbolSet;

struct LevelData
{
  enum
  {
    NONE                = 0,
    CLASS_LEVEL         = 400,
    DATA_MEMBERS_LEVEL  = 401,
    FUNCTION_LEVEL      = 300,
    BLOCK_LEVEL         = 200,
    DATA_VARS_LEVEL     = 201,
    STATEMENT_LEVEL     = 100,
    SEQUENCE_LEVEL      = 101,
    SUB_EXPRESSION      = 102,
    SUB_IDENTIFIER      = 103,
    SUB_VARIABLE        = 104,
    SUB_CONSTANT        = 105,
    DELIMITER           = 106,    

    CLASS_SPECS_SUBLEVEL     = 128,
    FUNCTION_PARAMS_SUBLEVEL = 64,
    BLOCK_COND_SUBLEVEL      = 32,
    FUNCTION_ARGS_SUBLEVEL   = 16,
    ENUMERATED_LIST          = 8,
    ARRAY_DIMENSION          = 4,
    STATEMENT_CONTINUATION   = 2,

    REFERRED_LINK_ACTIVE     = 128,
    FROM_LINK_ACTIVE         = 64,
    NEXT_LINK_ACTIVE         = 32,
    BACK_LINK_ACTIVE         = 16
  };

  int _Level;    // Class          = 400,
                 // Data members   = 401,
                 // Function       = 300,
                 // Block          = 200,
                 // Data Vars      = 201,
                 // Statement      = 100,
                 // Sequence       = 101,
                 // sub expression = 102,
                 // sub identifier = 103,
                 // sub variable   = 104,
                 // sub constant   = 105,
                 // delimiter      = 106,
                 // None           = 0
  int _SubLevel; // Class specs    = 128,
                 // Fnc Params     = 64,
                 // Block cond     = 32,
                 // Fnc Args       = 16,
                 // Enum List      = 8,
                 // Array Dim      = 4,
                 // Statement cont = 2,
                 
  int _LinkActive; // Referred Link = 128
                   // From Link     = 64,
                   // Next Link     = 32,
                   // Back Link     = 16,
                   // None          = 0

  int _Count;   // sequence count value of statement element
  int _Size;    // size of element array
  int _Max;     // maximum capacity of element array

  int _StackIndex;      // Referred scope stack element index
  int _ScopeLevel;      // Referred scope level
  
  ScopeInfo* _Refer;    // Referenced recursive scope stack element
  ScopeInfo* _From;     // Parent scope info

  LevelData* _BackPtr;     // Referring Back Level pointer of a SubLevel
  LevelData* _SubLevelPtr; // Pointer to level data if sublevel != None

  union
  {
    LevelData** _Elements;  // array of sequential elements
    SymbolSet* _Statement;  // pointer to statement object
  };
  
  bool _Copied;

  LevelData();
  LevelData(const LevelData& Obj_);
  ~LevelData();

  LevelData* Clone() const;
  LevelData& operator = (const LevelData& Obj_);  

  void Clear();  
  int GetLevelEnum() const;
  int GiveSizeForIndex(int Index_) const;
  int GiveLevelForIndex(int Index_) const;
  int GiveIndexForLevel(int Level_) const;
  
  LevelData& SetLevelEnum(int LvlEnum_);
  LevelData& SetLevel(int Lvl_);
  
  LevelData& SetCount(int Cnt_);
  LevelData& SetSize(int Sz_);
  LevelData& SetMax(int Max_);

  LevelData& SetRefer(LevelData* Refer_);
  LevelData& SetFrom(LevelData* From_);
  
  LevelData& SetSizeArray(int Sz_, int* Ptr_=NULL, bool* ArrAct_=NULL);
  LevelData& SetSizeIndex(int Index_, int Sz_);
  LevelData& InitSizeArray(int Sz_);
  void ShowData(ostream& os) const;
  void ShowSizeArray(ostream& os) const;

  LevelData** GrowNextArray(bool MakeNode_);
  LevelData** CreateNextArray(int Sz_, int Lv_, bool MakeNode_);
  LevelData** AppendNextArray(int Sz_, int Lv_, bool MakeNode_);
  LevelData* CreateNext(int& Index_, bool& Done_);
  
  LevelData* GetCoord(int* Coord_, int sz_, bool& Done_);  
  LevelData* GetNext(int Index_, bool& Done_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
// SymbolParserData forward declaration
class SymbolParserData;

struct ScopeInfo
{
  enum
  {
    // Scope levels
    TopLEVEL       = 4,
    ClassLEVEL     = 3,
    FunctionLEVEL  = 2,
    BlockLEVEL     = 1,
    StatementLEVEL = 0,
    
    ClassINDEX     = 0,
    FunctionINDEX  = 1,
    BlockINDEX     = 2,
    StatementINDEX = 3,
      
    StatementSCOPE = 32,
    BlockSCOPE     = 64,
    FunctionSCOPE  = 128,
    ClassSCOPE     = 256
  };

  bool _TopLevel;           // is top level or not (true/false)
  LevelData* _ReferredFrom; // Referring LevelData structure pointer
  int*** _SizeArray;        // Size of each structure pointer stack array
        
  LevelData** _ClassStack;
  LevelData** _FunctionStack;
  LevelData** _BlockStack;
  LevelData** _StatementStack;
  
  SymbolParserData* _Parserdata;

  ScopeInfo(int Levels_, int LeafSz_, int Classes_,
            SymbolParserData* Pdata_);
  ScopeInfo(const ScopeInfo& Obj_);
  ~ScopeInfo();

  ScopeInfo& operator = (const ScopeInfo& Obj_);
  ScopeInfo& InitSizeArray();

  int GiveSizeForIndex(int Index_, int FncLevel_, int BlkLevel_) const;
  int GiveLevelForIndex(int Index_) const;
  int GiveIndexForLevel(int Level_) const;
  int FindCurrentIndex(int Level_) const;
};

/****************************************************************************/
class SymbolParserData
{
  protected:
    int _PgrmIndex;
    int _Scope;
    LevelData* _ScopeLevel;
    int _BrackBal;
    
    LevelData* _DelimCnt;
    LevelData* _ArgCnt;
    bool _CdefExists;
    int _CdefLimit;
    int _BlkStkLevel;
    int _FncStkLevel;
    bool _OrBranchOut;
    bool _BranchFound;
    int _OrBranchNum;    
    bool _BranchBlocked;
    int _LeafSize;
    int _RefCount;

    LevelData* CreateLevelTree(LevelData* Ldp_, int Deepest_, int LeafSz_);
    LevelData* CreateLevelTreeRecFnc(LevelData* Ldp_, int LvIndex_, int SzIndex_,
                                     int Deepest_, int LeafSz_);

    LevelData* AppendLevelTree(LevelData* Ldp_, LevelData* Prv_,
                               int Deepest_, int LeafSz_, int Index_);
    LevelData* AppendLevelTreeRecFnc(LevelData* Ldp_, int LvIndex_, int SzIndex_,
                                     int LeafSz_, int Remain_, int Index_, int LvDecr_);

  public:
    SymbolParserData();
    SymbolParserData(const SymbolParserData& Obj_);
    ~SymbolParserData();

    SymbolParserData& operator = (const SymbolParserData& Obj_);

    SymbolParserData& SetScopeLevel(int Pgrm_, int Fnc_,
                                    int Blk_, int Seq_, int Sz_, int Cnt_);
    SymbolParserData& SetDelimiterCnt(int Blk_, int Seq_, int Sz_, int Cnt_);
    SymbolParserData& SetArgumentCnt(int Blk_, int Seq_, int Sz_, int Cnt_);
    SymbolParserData& ClearDelimiterCnt(int Blk_, int Seq_);
    SymbolParserData& ClearArgumentCnt(int Blk_, int Seq_);

    SymbolParserData& SetPgrmIndex(int Index_);
    SymbolParserData& IncPgrmIndex();
    SymbolParserData& SetScope(int Scope_);
    SymbolParserData& SetBrackBal(int Bal_);
    SymbolParserData& IncBrackBal();
    SymbolParserData& DecBrackBal();
    SymbolParserData& SetCdefExists(bool Val_);
    SymbolParserData& SetCdefLimit(int Val_);
    SymbolParserData& SetBlkStkLevel(int Lvl_);
    SymbolParserData& IncBlkStkLevel();
    SymbolParserData& DecBlkStkLevel();    
    SymbolParserData& SetFncStkLevel(int Lvl_);
    SymbolParserData& IncFncStkLevel();
    SymbolParserData& DecFncStkLevel();
    SymbolParserData& SetOrBranchOut(bool Val_);
    SymbolParserData& SetBranchFound(bool Val_);
    SymbolParserData& SetOrBranchNum(int Num_);
    SymbolParserData& SetBranchBlocked(bool Blk_);
    void IncRefObjects()
        { ++_RefCount; }
    void DecRefObjects()
        { --_RefCount; }
    
    void ShowData(ostream& os) const;

    LevelData* ScopeLevel(int Pgrm_, int Fnc_,
                          int Blk_, int Seq_, int Sz_);
    LevelData* DelimiterCnt(int Blk_, int Seq_, int Sz_);
    LevelData* ArgumentCnt(int Blk_, int Seq_, int Sz_);
    int RefCount() const
        { return _RefCount; }

    int GetPgrmIndex() const
        { return _PgrmIndex; }
    int GetScope() const
        { return _Scope; }
    int GetBrackBal() const
        { return _BrackBal; }
    bool GetCdefExists() const
        { return _CdefExists; }
    int GetCdefLimit() const
        { return _CdefLimit; }
    int GetBlkStkLevel() const
        { return _BlkStkLevel; }
    int GetFncStkLevel() const
        { return _FncStkLevel; }        
    bool GetOrBranchOut() const
        { return _OrBranchOut; }
    bool GetBranchFound() const
        { return _BranchFound; }
    int GetOrBranchNum() const
        { return _OrBranchNum; }
    bool GetBranchBlocked() const
        { return _BranchBlocked; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

// specific data members used according to the
// datatype assigned.
union SymbolBaseData
{
  char CharValue;
  Uchar UcharValue;
  long IntValue;
  Ulong UintValue;
  float FloatValue;
  double DoubleValue;
  Ldouble LdoubleValue;
  char* StrValue;
};

/****************************************************************************/
class SymbolTagArray
{
  protected:
    enum
    {
      TAGARRAYSIZE_INCR = 10,
      TOKENARRAY_LIMIT = 128,
      TOKENSTRING_LIMIT = 256
    };
  
    char* _FileName;
    StringFilter* _Filter;    
    
    char** _TagArray;
    int _TagArrayLimit;
    int _TagArraySize;
    char** _TokenArray;
    int _TokenArraySize;
  
    // Methods for creating token array
    char** ReadSymbolFileHelper(int& sz);
    char** InitTagArray();
    char** GrowTagArray(int Incr_);
    char** RemoveEndBrackets();

    SymbolSet* MakeSymbolSetType(char** Args_, int len);
    SymbolSet* MakeOperatorSetType(char** Args_, int len);
    SymbolSet* MakeSequenceType(char** Args_, int len);
    SymbolSet* MakeBranchOutType(char** Args_, int len);
    SymbolSet* MakeBracketedSymbolType(char** Args_, int len, int& BrkLvl_,
                                              int& CallIter_, SymbolSet* RetSym_,
                                              bool& Done_);

  public:
    SymbolTagArray(const char* Fname_);
    ~SymbolTagArray();

    const char* FileName() const
        { return _FileName; }
    int TagArrayLimit() const
        { return _TagArrayLimit; }
    int TagArraySize() const
        { return _TagArraySize; }
    int TokenArraySize() const
        { return _TokenArraySize; }
  
    // Symbol object instance generator methods
    char** ClearTagArray();
    void ShowTagArray(int& st);
    char** ReadSymbolFile();
    SymbolSet* MakeFromTagText(int& Index_);
};

/****************************************************************************/
#define SymState_Check(Sym, State)                              \
   (Sym && !(_AggsState & State))

#define SymDataState_Check(Sym, Dtype, State)                   \
   ((Sym || _DataType != Dtype) && !(_AggsState & State))

#define SymRequiredState_Check(Sym, Req, State)                 \
   (Sym &&  (_AggsRequired & Req) &&                            \
           !(_AggsState & State))

#define SeqRequiredState_Check(Sym, Req, State)                 \
   (Sym &&  (_SeqRequired & Req) &&                             \
           !(_SeqState & State))

class SymbolSet
{
  public:
    // binary types
    enum
    {
      // Found/NotFound states
      Found = 1,
      NotFound = 0
    };

    // Match leading or trailing whitespaces
    // in string defined symbols
    enum    
    {
      NoPaddingSpaces = 0,
      MatchLeadingSpaces = 1,
      MatchTrailingSpaces = 2      
    };
  
    // general symbol enums
    enum
    {
      // Symbol attributes
      NONE = 0,
      Starter = 1,
      Ender = 2,
      WhiteSpace = 4,
      LeftBracket = 8,
      RightBracket = 16,
      DelimiterSymbol = 32,
      AssignmentSymbol = 64,
      OperatorSymbol = 128,
      SymbolData = 256
    };

    enum
    {
      // algorithm structures
      NotStruct = 0,
      ReservedStruct = 1,
      CtrlStruct = 2,
      BranchStruct = 4,
      ListStruct = 8,
      AltElseDef = 16,
      SwitchStruct = 32,
      CondTestStruct = 64,
      LoopStruct = 128,
      LabelStruct = 256,
      IterStruct = 1024,
      FunctionStruct = 2048
    };

    enum
    {
      // Symbol types
      Symbol_Type = 1,
      Sequence_Type = 2,
      BranchOut_Type = 4,
      OperatorSet_Type = 8,
      BracketedSymbol_Type = 16
    };

    // operator enums
    enum
    {
      // Operator attributes
      PreFixed = 1,
      PostFixed = 2,
      InFixed = 4,
      UnaryOp = 8,
      BinaryOp = 16,
      TrinaryOp = 32,
      FreeStanding = 64,
      ResultPreOp = 128,
      ResultPostOp = 256
    };

    // sequence enums
    enum
    {
      // sequence attributes      
      Reserved = 1,
      Constant = 2,
      Statement = 4,
      Expression = 8,
      Label = 16,
      LabelJump = 32,
      Declaration = 64,
      Definition = 128,
      FncCall = 256,
      FncName = 512,
      Directive = 1024,
      ParserDirective = 2048
    };

    // Declaration types
    enum
    {
      ConstDecl = 1,
      VarDecl = 2,
      ListDecl = 4,
      FncDecl = 8,
      StructDecl = 16,
      ClassDecl = 32,
      MemberDecl = 64,
      MethodDecl = 128
    };

    // Definition types
    enum
    {
      FncDefn = 1,
      ClassDefn = 2,
      MethodDefn = 4
    };

    // I/O and Data retrieval types
    enum
    {
      NO_DATA = 0,
      INPUT = 1,
      OUTPUT = 2,

      UNSIGNED = 4,
      SIGNED = 8,
      
      CHAR = 16,
      INT = 32,
      FLOAT = 64,
      DOUBLE = 128,
      LONGDOUBLE = 256,
      STRING = 512,
      DEFINED = 1024,
      ICASEDEFINED = 2048,
      REGEXPR = 4096
    };

    // operator symbol state enums
    enum
    {
      NO_RESULTANT_REQ  = 0,
      RESULTANT_REQ     = 1,
      NO_ASSIGNMENT_REQ = 2,
      ASSIGNMENT_REQ    = 4,      
      NO_OPERAND1_REQ   = 8,
      OPERAND1_REQ      = 16,
      NO_OPERAND2_REQ   = 32,
      OPERAND2_REQ      = 64,
      NO_OPERANDSEP_REQ = 128,
      OPERANDSEP_REQ    = 256,
      NO_OPERAND3_REQ   = 512,
      OPERAND3_REQ      = 1024,

      NO_OPERATOR_DONE  = 0,
      RESULTANT_DONE    = 2048,
      ASSIGNMENT_DONE   = 4096,
      OPERAND1_DONE     = 8192,
      OPERATOR_DONE     = 16384,
      OPERAND2_DONE     = 32768,
      OPERANDSEP_DONE   = 65536,
      OPERAND3_DONE     = 131072
    };

    // starter, branch separator, ender state enums
    enum
    {
      NO_STARTER_REQ   = 0,
      STARTER_REQ      = 1,
      NO_ENDER_REQ     = 2,
      ENDER_REQ        = 4,
      NO_BRANCHSEP_REQ = 8,
      BRANCHSEP_REQ    = 16,

      NO_SEQUENCE_DONE = 0,
      STARTER_DONE     = 32,
      SEQUENCE_DONE    = 64,
      BRANCHSEP_DONE   = 128,
      ENDER_DONE       = 256
    };

    // bracketed symbol state enums
    enum
    {
      NO_LEFTBRK_REQ  = 0,
      LEFTBRK_REQ     = 1,
      NO_RIGHTBRK_REQ = 2,
      RIGHTBRK_REQ    = 4,
      NO_SYMDELIM_REQ = 8,
      SYMDELIM_REQ    = 16,
      NO_NEXTSYM_REQ  = 32,
      NEXTSYM_REQ     = 64,

      NO_BRKSYM_DONE  = 0,
      LEFTBRK_DONE    = 128,
      SYMBOLS_DONE    = 256,
      SYMDELIM_DONE   = 512,
      NEXTSYM_DONE    = 1024,
      RIGHTBRK_DONE   = 2048
    };

    enum
    {
      NOT_OPTIONAL = 0,
      OPTIONAL     = 1,
      FRESH        = 2,
      STARTED      = 4,
      PARSE_DONE   = 8,
      PARSE_FAIL   = 16,
      OPTION_VALID = 32,

      TRIGGER_OBJECT_SET       = 64,
      TRIGGER_FUNCTION_SET     = 128,
      TRIGGER_DONE             = 256,
      TRIGGER_FAIL             = 512,
      STATECHANGE_OBJECT_SET   = 1024,
      STATECHANGE_FUNCTION_SET = 2048,
      STATECHANGE_DONE         = 4096,
      STATECHANGE_FAIL         = 8192
    };

    struct TriggerFncData
    {    
      int(*_TriggerFnc)(int, char**, void*);
      int _FncArgc;
      char** _FncArgv;
      bool _FncArgr;
      bool _FncArgx;
      bool _DesArgs;
      void* _FncArgi;

#if OVERLOAD_NEW
      void* operator new (size_t Bytes_);
      void operator delete (void* Space_);
#if HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
#endif
#endif
    };

    struct StateChangeFncData
    {
      int(*_StateChangeFnc)(int, char**, void*);
      int _FncArgc;
      char** _FncArgv;
      bool _FncArgr;
      bool _FncArgx;
      bool _DesArgs;
      void* _FncArgi;

#if OVERLOAD_NEW
      void* operator new (size_t Bytes_);
      void operator delete (void* Space_);
#if HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
#endif
#endif
    };    

  protected:  
    int _IoType;
    int _DataType;

    int _Slen;
    int _Sterm;
    
    int _SymAttrib;
    int _SymType;
    int _RetCode;
    int _Wspace;

    char* _Name;
    char* _WspaceChar;
    char* _RegExpr;

    union
    {
      char* _Char;
      SymbolSet* _SymData;
    };

    mutable bool _Transferred;
    bool _SkipSpaces;
    bool _Required;
    bool _CaseSensitive;
    bool _Ordered;
    bool _ParseValid;
    bool _ParseDone;
    bool _SoloSymbol;
    bool _HasSymData;
    bool _IsRegExpr;
    int _ParseState;
    int _TriggerState;
    int _StateChangeState;
    int _RecurTimes;

    SymbolSet* _Enclosing;
    static SymbolParserData* _ParserData;

    // Trigger functions and functors
    union
    {
      ObjectFunctor* _TriggerObj;
      TriggerFncData* _TriggerData;
    };

    union
    {
      ObjectFunctor* _StateChangeObj;
      StateChangeFncData* _StateChangeData;
    };

    static int DecPoints(double Val_, int Pts_);
    
    virtual int ExecTriggerFunction();
    virtual int ExecStateChangeFunction(SymbolSet* Operator_, int OperandNum_, int State_);
    virtual int ExecStateChangeObject(SymbolSet* Operator_, int OperandNum_, int State_);

    // Methods for construction/destruction of Parser Data structure
    static void CreateParserData();
    static void DestroyParserData();    

    // Methods for creating token array
    static char** ReadSymbolFileHelper(StringFilter* sfilter_, int& sz, char** TokArray_);
    static char** InitTokenArray(char** TokArray_);
    static char** GrowTokenArray(char** TokArray_, int Incr_);

    static SymbolSet* MakeSymbolSetType(char** Args_, int len);
    static SymbolSet* MakeOperatorSetType(char** Args_, int len);
    static SymbolSet* MakeSequenceType(char** Args_, int len);
    static SymbolSet* MakeBranchOutType(char** Args_, int len);
    static SymbolSet* MakeBracketedSymbolType(char** Args_, int len, int& BrkLvl_,
                                              int& CallIter_, SymbolSet* RetSym_,
                                              bool& Done_);

  public:
    SymbolSet();
    SymbolSet(const char* Str_, const char* WsChar_,
              int IoType_, int DataType_, int Attrib_,
              int Type_, int Ws_, bool SkipSpaces_,
              bool Required_, bool CaseSensitive_, bool Ordered_);
    SymbolSet(SymbolSet* SymData_, const char* WsChar_,
              int IoType_, int DataType_, int Attrib_,
              int Type_, int Ws_, bool SkipSpaces_,
              bool Required_, bool CaseSensitive_, bool Ordered_);
    SymbolSet(const SymbolSet& Obj_);
    virtual ~SymbolSet();

    virtual SymbolSet* Clone() const;
    virtual SymbolSet& operator = (const SymbolSet& Obj_);
    
    int GetIoType() const
        { return _IoType; }
    int GetDataType() const
        { return _DataType; }
    int GetStringLength() const
        { return _Slen; }
    int GetStringTerm() const
        { return _Sterm; }
    int GetRecurTimes() const
        { return _RecurTimes; }
    bool IsRecurring() const
        { return (_RecurTimes != 0); }
    bool RecurForever() const
        { return (_RecurTimes == -1); }

    SymbolSet* GetEnclosing()
        { return _Enclosing; }
    const char* GetName() const
        { return _Name; }
    int GetWspace() const
        { return _Wspace; }        
    const char* GetWspaceChar() const
        { return _WspaceChar; }
    int GetSymbolType() const
        { return _SymType; }
    int GetSymbolAttrib() const
        { return _SymAttrib; }
    int GetRetCode() const
        { return _RetCode; }
    bool IsSoloSymbol() const
        { return _SoloSymbol; }
    SymbolBaseData* ConvertToBaseData(SymbolBaseData* SymData_);

    virtual void Reset();
    virtual bool IsSequence(bool FollowEnclosing_=false) const;
    virtual bool IsBranchOut(bool FollowEnclosing_=false) const;
    virtual bool IsOperatorSet(bool FollowEnclosing_=false) const;
    virtual bool IsBracketedSymbol(bool FollowEnclosing_=false) const;
    virtual bool IsAggregateSymbol(bool FollowEnclosing_=false) const;
    
    bool GetSkipSpaces() const
        { return _SkipSpaces; }
    bool IsRequired() const
        { return _Required; }
    bool IsCaseSensitive() const
        { return _CaseSensitive; }
    bool IsOrdered() const
        { return _Ordered; }
    bool IsSymbolType(int Type_)
        { return (_SymType & Type_); }

    virtual bool ChangeState(SymbolSet* Operator_, int OperandNum_, int State_, char* RetStr_);
    virtual SymbolSet& ClearTriggerData();
    virtual SymbolSet& ClearStateChangeData();
    SymbolSet& SetSoloSymbol(bool Val_);
    SymbolSet& SetIoType(int ioType_);
    SymbolSet& SetDataType(int dataType_);
    SymbolSet& SetStringLength(int Length_);
    SymbolSet& SetStringTerminator(int Term_);
    SymbolSet& SetRecurTimes(int Recur_);

    virtual SymbolSet& SetSymbol(const char* NewSymbol_);
    virtual SymbolSet& SetRegExpr(const char* NewSymbol_);
    virtual SymbolSet& SetSymbolSet(SymbolSet* NewSymbol_, bool Overwrite_=true);
    virtual char* GetSymbol(bool Transfer_=false) const;
    virtual SymbolSet* GetSymbolSet(bool Transfer_=false) const;
    virtual bool SymTransferred() const;
    
    SymbolSet& SetName(const char* Name_);    
    SymbolSet& SetWspace(int Wspace_);
    SymbolSet& AppendWhiteSpace(const char* Wspace_);
    SymbolSet& RemoveWhiteSpace(const char* Wspace_);
    SymbolSet& SetWspaceChar(const char* Wspace_);
    SymbolSet& SetRetCode(int RetCode_);
    SymbolSet& SetSymbolType(int Type_);
    virtual SymbolSet& SetSymbolAttrib(int Attr_);
    
    SymbolSet& SetRequired(bool Req_);
    SymbolSet& SetCaseSensitive(bool Sensitive_);
    SymbolSet& SetSkipSpaces(bool Skip_);
    SymbolSet& SetOrdered(bool Ordered_);

    virtual SymbolSet& SetStateChangeObject(ObjectFunctor* Obj_);
    virtual SymbolSet& SetStateChangeFunction(int(*Fnc_)(int, char**, void*),
                                      int FncArgc_, char** FncArgv_,
                                      bool ResultArg_, bool BranchArg_,
                                      void* FncArgi_);

    virtual SymbolSet& SetTriggerObject(ObjectFunctor* Obj_);
    virtual SymbolSet& SetTriggerFunction(int(*Fnc_)(int, char**, void*),
                                  int FncArgc_, char** FncArgv_,
                                  bool ResultArg_, bool BranchArg_,
                                  void* FncArgi_);

    virtual SymbolSet::TriggerFncData* GiveTriggerFncData();
    virtual SymbolSet::StateChangeFncData* GiveStateChangeFncData();
    virtual ObjectFunctor* GiveTriggerObject();
    virtual ObjectFunctor* GiveStateChangeObject();

    // convenience methods from porting old FieldInfo and TokenInfo class
    SymbolSet& SetToken(const char* Token_)
        { return SetSymbol(Token_); }
    SymbolSet& SetField(const char* Field_)
        { return SetSymbol(Field_); }
    SymbolSet& SetCode(int Code_)
        { return SetRetCode(Code_); }
    SymbolSet& SetSkipWhiteSpace(bool Skip_)
        { return SetSkipSpaces(Skip_); }

    const char* GetRegExpr() const
        { return _RegExpr; }
    char* GetToken() const
        { return GetSymbol(false); }
    char* GetField(bool Transfer_) const
        { return GetSymbol(Transfer_); }
    int GetCode() const
        { return _RetCode; }
    bool SkipWhiteSpace() const
        { return _SkipSpaces; }
    bool IsRegExpr() const
        { return _IsRegExpr; }

    // virtual methods in derived classes
    virtual int GetStructType() const;
    virtual SymbolSet& SetStructType(int StructType_);    
    virtual const char* GetSeparatorString() const;    
    virtual SymbolSet& SetSeparatorString(const char* SepChar_);
    virtual SymbolSet& ClearSeparatorString();    
    virtual SymbolSet& SetEnclosing(SymbolSet* Enclosing_);    
    virtual bool HasSymbol(const SymbolSet* SymPtr_, bool NoChain_=false);
    virtual SymbolSet& SetNext(SymbolSet* Set_, bool SeqTerm_=false);
    virtual SymbolSet* NextSymbol(int Index_=0, bool RawSym_=false);
    virtual SymbolSet& SetAssignment(SymbolSet* Assign_, bool Overwrite_=true);
    virtual SymbolSet& SetSeparator(SymbolSet* Delim_, bool Overwrite_=true);
    virtual SymbolSet* Assignment(bool Transfer_=false);
    virtual SymbolSet* Separator(bool Transfer_=false);    
    virtual int Length() const;

    // AggregateSymbol class derived methods
    virtual int AggregateType() const;
    virtual SymbolSet& SetNestingSymbol(int Index_, SymbolSet* SymPtr_=NULL);

    virtual SymbolSet& SetAggsState(int EnumIndex_);
    virtual SymbolSet& SetAggsRequired();
    virtual SymbolSet* SetAggs(int EnumIndex_, int NestLevel_,
                               SymbolSet* Sym_, bool Overwrite_=true);
    
    virtual int GetAggsState() const;    
    virtual int GetAggsRequired() const;    
    virtual SymbolSet* GetAggs(int EnumIndex_, int NestLevel_,
                               bool Transfer_=false);

    virtual bool AggsTransferred(int Index_) const;
    virtual bool HasAggs(int EnumIndex_);

    // Sequence class derived methods
    virtual bool IsSingleChain(bool& EndSeq_) const;
    virtual bool GetOrBranchOut() const;    
    virtual bool GetCdeChkBlock() const;
    virtual int GetSequenceType() const;
    virtual int GetDeclType() const;    
    virtual int GetDefnType() const;
    virtual bool CheckForCde();
    virtual bool CdefExists() const;
    virtual int CdefLimit() const;
    virtual bool IsErrorSet() const;
    virtual int GetIndex() const;
    virtual SymbolSet& ConvertToCircSeq();    
    virtual SymbolSet& HaltOnCircSeqEnd();
    virtual SymbolSet& SetCircDefLimitValue(int Limit_=0);
    virtual SymbolSet& ResetCdefLimit();    
    virtual SymbolSet& CheckForCdeBlock();
    virtual SymbolSet& SetOrBranchOut(bool Branch_);
    virtual SymbolSet& SetCdeChkBlock(bool Block_);
    virtual SymbolSet& SetSequenceType(int Type_);
    virtual SymbolSet& SetDeclType(int Type_);
    virtual SymbolSet& SetDefnType(int Type_);
    virtual SymbolSet& SetCircSeqTerm(SymbolSet* Set_);
    virtual SymbolSet& UnlinkSeqTerm(SymbolSet* Set_);
    virtual SymbolSet& SetCircSeqWrap(SymbolSet* Set_);
    virtual SymbolSet& ResetErrorFlag();

    // BranchOut class derived methods
    virtual int BranchNum() const;
    virtual bool BranchFound() const;
    virtual bool SepTransferred() const;

    // Operator class derived methods
    virtual int GetResultOrder() const;
    virtual int GetOperatorState() const;
    virtual int GetOperatorRequired() const;    
    virtual int GetOperatorType() const;
    virtual int* GetResultantState();    
    virtual SymbolSet& SetResultOrder(int Order_);
    virtual SymbolSet& SetOperatorType(int Type_);
    virtual SymbolSet& SetOperand1(SymbolSet* Operand_, bool Overwrite_=true);
    virtual SymbolSet& SetOperand2(SymbolSet* Operand_, bool Overwrite_=true);
    virtual SymbolSet& SetOperand3(SymbolSet* Operand_, bool Overwrite_=true);
    virtual SymbolSet& SetResultant(SymbolSet* Result_, bool Overwrite_=true);    
    virtual SymbolSet* Operand1(bool Transfer_=false);
    virtual SymbolSet* Operand2(bool Transfer_=false);
    virtual SymbolSet* Operand3(bool Transfer_=false);
    virtual SymbolSet* Resultant(bool Transfer_=false);
    virtual bool OpTransferred(int Index_) const;

    // BracketedSymbol class derived methods
    virtual SymbolSet& SetStartDelim(const char* Start_, bool Cs_=true, bool Ord_=true);
    virtual SymbolSet& SetEndDelim(const char* End_, bool Cs_=true, bool Ord_=true);
    virtual char* GetStartDelim() const;
    virtual char* GetEndDelim() const;
    virtual SymbolSet& SetLeftBracket(SymbolSet* Left_, bool Overwrite_=true);
    virtual SymbolSet& SetDelimiter(SymbolSet* Delim_, bool Overwrite_=true);
    virtual SymbolSet& SetRightBracket(SymbolSet* Right_, bool Overwrite_=true);
    virtual SymbolSet* NextDelimiter(int Index_=0, bool Transfer_=false);
    virtual SymbolSet* NextLeftBracket(int Index_=0, bool Transfer_=false);
    virtual SymbolSet* NextRightBracket(int Index_=0, bool Transfer_=false);
    virtual bool BrkSymTransferred(int Index_) const;
    virtual int GetBrkSymRequired() const;
    virtual int GetBrkSymState() const;
    virtual SymbolSet& IncBlockCount(int Val_);
    virtual SymbolSet& IncFunctionCount(int Val_);
    virtual bool HasLeftBracket() const;
    virtual bool HasDelimiter() const;
    virtual bool HasNext() const;
    virtual bool HasRightBracket() const;

    // symbol data parsing method    
    SymbolSet& SetParseValid(bool Valid_);    
    SymbolSet& SetParseDone(bool Done_);
    bool IsParseValid() const;
    bool IsParseDone() const;

    virtual SymbolSet& SetGroupValid(bool Valid_);
    virtual SymbolSet& SetGroupDone(bool Done_);    
    virtual bool IsGroupValid() const;
    virtual bool IsGroupDone() const;
    virtual char* ParseSymbolStep(char* Text_, bool& Done_);    
    virtual char* ParseSymbol(char* Text_);

    // Symbol object instance generator methods
    static char** ClearTokenArray(char** TokArray_);
    static void ShowTokenArray(char** TokArray_, int& st, int sz);
    static char** ReadSymbolFile(const char* FileName_, int& sz);        
    static SymbolSet* MakeFromTagText(char** TagArr_, int sz, int& Index_);

    // return parser data structure
    static SymbolParserData* ParserData();

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif        
};

/****************************************************************************/
class AggregateSymbol : public SymbolSet
{
  protected:
    int _MaxAggs;
    SymbolSet** _AggsArray;
    bool* _AggsTransferred;
    int* _StatesArray;
    char* _SepChar;

    int _NestingSymIndex;
    int _AggType;
    int _AggsRequired;
    int _AggsState;
    bool _GroupValid;
    bool _GroupDone;

    virtual void InitAggSymbolStates() = 0;
    void SetAggsStates(int Index_, int NotReqEnum_,
                       int ReqEnum_, int DoneEnum_);

  public:
    AggregateSymbol(int MaxAggs_, int AggType_);
    AggregateSymbol(int MaxAggs_, int AggType_, const char* Str_, const char* WsChar_,
              int IoType_, int DataType_,
              int Attrib_, int Type_, int Ws_,
              bool SkipSpaces_, bool Required_,
              bool CaseSensitive_, bool Ordered_);
    AggregateSymbol(int MaxAggs_, int AggType_, SymbolSet* SymData_, const char* WsChar_,
              int IoType_, int DataType_,
              int Attrib_, int Type_, int Ws_,
              bool SkipSpaces_, bool Required_,
              bool CaseSensitive_, bool Ordered_);              
    AggregateSymbol(int MaxAggs_, int AggType_, const AggregateSymbol& Obj_);
    virtual ~AggregateSymbol();

    virtual SymbolSet& operator = (const SymbolSet& Obj_);

    //**** BEGIN: AggregateSymbol class specific methods ****//
    virtual bool IsAggregateSymbol(bool FollowEnclosing_=false) const;
    virtual const char* GetSeparatorString() const;    
    virtual SymbolSet& SetSeparatorString(const char* SepChar_);
    virtual SymbolSet& ClearSeparatorString();
    
    virtual int AggregateType() const;
    virtual SymbolSet& SetNestingSymbol(int Index_, SymbolSet* SymPtr_=NULL);

    virtual SymbolSet& SetAggsState(int EnumIndex_);
    virtual SymbolSet& SetAggsRequired();
    virtual SymbolSet* SetAggs(int EnumIndex_, int NestLevel_,
                               SymbolSet* Sym_, bool Overwrite_=true);

    virtual int GetAggsState() const;
    virtual int GetAggsRequired() const;
    virtual SymbolSet* GetAggs(int EnumIndex_, int NestLevel_,
                               bool Transfer_=false);

    virtual bool AggsTransferred(int Index_) const;
    virtual bool HasAggs(int EnumIndex_);
    //**** END: AggregateSymbol class specific methods ****//
    
    virtual void Reset();
    virtual bool HasSymbol(const SymbolSet* SymPtr_, bool NoChain_=false);

    virtual SymbolSet& SetGroupValid(bool Valid_);
    virtual SymbolSet& SetGroupDone(bool Done_);    
    virtual bool IsGroupValid() const;
    virtual bool IsGroupDone() const;

    virtual SymbolSet::TriggerFncData* GiveTriggerFncData();
    virtual SymbolSet::StateChangeFncData* GiveStateChangeFncData();
    virtual ObjectFunctor* GiveTriggerObject();
    virtual ObjectFunctor* GiveStateChangeObject();    

    virtual SymbolSet& SetStateChangeObject(ObjectFunctor* Obj_);
    virtual SymbolSet& SetStateChangeFunction(int(*Fnc_)(int, char**, void*),
                                      int FncArgc_, char** FncArgv_,
                                      bool ResultArg_, bool BranchArg_,
                                      void* FncArgi_);

    virtual SymbolSet& SetTriggerObject(ObjectFunctor* Obj_);
    virtual SymbolSet& SetTriggerFunction(int(*Fnc_)(int, char**, void*),
                                  int FncArgc_, char** FncArgv_,
                                  bool ResultArg_, bool BranchArg_,
                                  void* FncArgi_);    
};

/****************************************************************************/
class Sequence : public SymbolSet
{
  protected:
    int _SetLen;
    int _Limit;
    int _Index;
    int _StructType;
    int _SeqType;
    int _SeqRequired;
    int _SeqState;
    int _DeclType;
    int _DefnType;

    int _CdefLimit;
    bool _CdefExists;
    bool _CdeChkBlock;
    bool _OrBranchOut;

    int _CdefLimitValue;
    bool _CdefLimitExplicit;
    bool _ErrorSet;
    char* _SepChar;
    
    SymbolSet** _Set;
    SymbolSet* _NullSet;
    SymbolSet* _Intersect;
    mutable SymbolSet* _CircSeqTerm;

    SymbolSet* _Separator;
    bool _SepTransferred;

  public:
    enum { SETLENINCR = 32, CIRCDEFLIMIT = 64 };
  
    Sequence();
    Sequence(const char* Str_, const char* WsChar_,
             int IoType_, int DataType_, int Struct_,
             int Attrib_, int Type_, int Ws_,
             int SeqType_, int TypeExt_, bool SkipSpaces_,
             bool Required_, bool CaseSensitive_, bool Ordered_);
    Sequence(const Sequence& Obj_);
    virtual ~Sequence();

    virtual SymbolSet* Clone() const;
    virtual SymbolSet& operator = (const SymbolSet& Obj_);

    virtual SymbolSet& SetSymbolAttrib(int Attr_);
    virtual int GetStructType() const;
    virtual SymbolSet& SetStructType(int StructType_);
    
    virtual const char* GetSeparatorString() const;    
    virtual SymbolSet& SetSeparatorString(const char* SepChar_);
    virtual SymbolSet& ClearSeparatorString();    
    
    virtual SymbolSet& SetSeparator(SymbolSet* Delim_, bool Overwrite_=true);
    virtual SymbolSet* Separator(bool Transfer_=false);
    virtual bool SepTransferred() const;    

    virtual void Reset();
    virtual bool CheckForCde();
    virtual bool HasSymbol(const SymbolSet* SymPtr_, bool NoChain_=false);
    virtual SymbolSet& ConvertToCircSeq();
    virtual SymbolSet& HaltOnCircSeqEnd();
    virtual SymbolSet& SetCircDefLimitValue(int Limit_=CIRCDEFLIMIT);
    virtual SymbolSet& ResetCdefLimit();
    virtual SymbolSet& CheckForCdeBlock();
    virtual SymbolSet& SetOrBranchOut(bool Branch_);
    virtual SymbolSet& SetCdeChkBlock(bool Block_);
    virtual SymbolSet& SetSequenceType(int Type_);    
    virtual SymbolSet& SetDeclType(int Type_);
    virtual SymbolSet& SetDefnType(int Type_);    
    virtual SymbolSet& SetNext(SymbolSet* Set_, bool SeqTerm_=false);
    virtual SymbolSet* NextSymbol(int Index_=0, bool RawSym_=false);
    virtual SymbolSet& UnlinkSeqTerm(SymbolSet* Set_);
    virtual SymbolSet& SetCircSeqTerm(SymbolSet* Set_);
    virtual SymbolSet& SetCircSeqWrap(SymbolSet* Set_);
    virtual SymbolSet& ResetErrorFlag();
    virtual int Length() const;
    virtual int GetIndex() const;

    virtual bool IsErrorSet() const;
    virtual bool CdefExists() const;
    virtual int CdefLimit() const;
    virtual bool IsSingleChain(bool& EndSeq_) const;
    virtual bool GetOrBranchOut() const;
    virtual bool GetCdeChkBlock() const;    
    virtual int GetSequenceType() const;
    virtual int GetDeclType() const;    
    virtual int GetDefnType() const;
    virtual bool IsSequence(bool FollowEnclosing_=false) const;

    virtual char* ParseSymbolStep(char* Text_, bool& Done_);
    virtual char* ParseSymbol(char* Text_);

    virtual SymbolSet& SetStateChangeObject(ObjectFunctor* Obj_);
    virtual SymbolSet& SetStateChangeFunction(int(*Fnc_)(int, char**, void*),
                                      int FncArgc_, char** FncArgv_,
                                      bool ResultArg_, bool BranchArg_,
                                      void* FncArgi_);

    virtual SymbolSet& SetTriggerObject(ObjectFunctor* Obj_);
    virtual SymbolSet& SetTriggerFunction(int(*Fnc_)(int, char**, void*),
                                  int FncArgc_, char** FncArgv_,
                                  bool ResultArg_, bool BranchArg_,
                                  void* FncArgi_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif        
};

/****************************************************************************/
class BranchOut : public Sequence
{
  protected:
    int _BranchNum;
    bool _BranchFound;
  
  public:
    BranchOut();
    BranchOut(const char* Str_, const char* WsChar_,
             int IoType_, int DataType_, int Struct_,
             int Attrib_, int Type_, int Ws_,
             int SeqType_, int TypeExt_, bool SkipSpaces_,
             bool Required_, bool CaseSensitive_, bool Ordered_);
    BranchOut(const BranchOut& Obj_);

    virtual SymbolSet* Clone() const;
    virtual SymbolSet& operator = (const SymbolSet& Obj_);

    virtual void Reset();
    virtual int BranchNum() const;
    virtual bool BranchFound() const;

    virtual bool IsBranchOut(bool FollowEnclosing_=false) const;
    virtual char* ParseSymbolStep(char* Text_, bool& Done_);    
    virtual char* ParseSymbol(char* Text_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
class OperatorSet : public AggregateSymbol
{
  protected:
    SymbolSet* _Operand1;
    SymbolSet* _Operand2;
    SymbolSet* _Operand3;
    SymbolSet* _Resultant;
    SymbolSet* _Separator;
    SymbolSet* _Assignment;
    bool* _OpTransferred;

    int _OpType;
    int _OpPos;
    int _ResultOrder;
    int _ResultantState;    

    virtual void InitAggSymbolStates();
    
    char* ParseOperandSymbol(int OpNumber_, bool OperatorDone_, char* Text_);
    char* ParseOperatorSymbol(char* Text_, bool& OperatorDone_);

  public:
    enum
    {
      RESULTANT_INDEX = 0,
      OPERAND1_INDEX = 1,
      OPERAND2_INDEX = 2,
      OPERAND3_INDEX = 3,
      OPERANDSEP_INDEX = 4,
      ASSIGNMENT_INDEX = 5
    };
  
    OperatorSet();
    OperatorSet(const char* Str_, const char* WsChar_,
                int IoType_, int DataType_,
                int SymAttrib_, int SymType_, int Ws_,
                int OpType_, int OpPos_, int ResultOrder_,
                bool SkipSpaces_, bool Required_,
                bool CaseSensitive_, bool Ordered_);
    OperatorSet(const OperatorSet& Obj_);

    virtual SymbolSet* Clone() const;
    virtual SymbolSet& operator = (const SymbolSet& Obj_);

    virtual int GetOperatorState() const;
    virtual int GetOperatorRequired() const;
    virtual int GetOperatorType() const;
    virtual int GetResultOrder() const;    
    virtual int* GetResultantState();
    
    virtual SymbolSet& SetOperatorType(int Type_);
    virtual SymbolSet& SetResultOrder(int Order_);
    virtual SymbolSet& SetSeparator(SymbolSet* Delim_, bool Overwrite_=true);
    virtual SymbolSet& SetAssignment(SymbolSet* Assign_, bool Overwrite_=true);
    virtual SymbolSet& SetOperand1(SymbolSet* Operand_, bool Overwrite_=true);
    virtual SymbolSet& SetOperand2(SymbolSet* Operand_, bool Overwrite_=true);
    virtual SymbolSet& SetOperand3(SymbolSet* Operand_, bool Overwrite_=true);
    virtual SymbolSet& SetResultant(SymbolSet* Result_, bool Overwrite_=true);

    virtual SymbolSet* Separator(bool Transfer_=false);
    virtual SymbolSet* Assignment(bool Transfer_=false);
    virtual SymbolSet* Operand1(bool Transfer_=false);
    virtual SymbolSet* Operand2(bool Transfer_=false);
    virtual SymbolSet* Operand3(bool Transfer_=false);
    virtual SymbolSet* Resultant(bool Transfer_=false);
    virtual bool OpTransferred(int Index_) const;    

    virtual bool HasOperand1() const;
    virtual bool HasOperand2() const;
    virtual bool HasOperand3() const;
    virtual bool HasResultant() const;
    virtual bool HasSeparator() const;
    virtual bool HasAssignment() const;
    
    virtual bool IsOperatorSet(bool FollowEnclosing_=false) const;
    virtual int AggregateType() const;

    virtual char* ParseSymbolStep(char* Text_, bool& Done_);
    virtual char* ParseSymbol(char* Text_);

    virtual SymbolSet::TriggerFncData* GiveTriggerFncData();
    virtual SymbolSet::StateChangeFncData* GiveStateChangeFncData();
    virtual ObjectFunctor* GiveTriggerObject();
    virtual ObjectFunctor* GiveStateChangeObject();    

    virtual SymbolSet& SetStateChangeObject(ObjectFunctor* Obj_);
    virtual SymbolSet& SetStateChangeFunction(int(*Fnc_)(int, char**, void*),
                                      int FncArgc_, char** FncArgv_,
                                      bool ResultArg_, bool BranchArg_,
                                      void* FncArgi_);

    virtual SymbolSet& SetTriggerObject(ObjectFunctor* Obj_);
    virtual SymbolSet& SetTriggerFunction(int(*Fnc_)(int, char**, void*),
                                  int FncArgc_, char** FncArgv_,
                                  bool ResultArg_, bool BranchArg_,
                                  void* FncArgi_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif        
};

/****************************************************************************/
// For handling symbols of the type: (s , ...)
//
class BracketedSymbol : public AggregateSymbol
{
  protected:
    SymbolSet* _Left;
    SymbolSet* _Delimiter;
    SymbolSet* _Next;
    SymbolSet* _Right;
    bool* _BrkSymTransferred;

    int _BrkSymRequired;
    int _BrkSymState;
    bool _GroupValid;
    bool _GroupDone;
    
    int _BlockInc;
    int _FunctionInc;
    int _BlockDec;
    int _FunctionDec;
    
    int* _ArgCounter;
    int* _DelimCounter;
    bool _SharedCnt;

    virtual void InitAggSymbolStates();

  public:
    enum
    {
      LEFTBRK_INDEX = 0,
      SYMDELIM_INDEX = 1,
      NEXTSYM_INDEX = 2,
      RIGHTBRK_INDEX = 3,
    };
  
    BracketedSymbol();
    BracketedSymbol(const char* Str_, const char* WsChar_,
                    int IoType_, int DataType_,
                    int Attrib_, int Type_, int Ws_,
                    bool SkipSpaces_, bool Required_,
                    bool CaseSensitive_, bool Ordered_);
    BracketedSymbol(const BracketedSymbol& Obj_);
    virtual ~BracketedSymbol();

    virtual SymbolSet* Clone() const;
    virtual SymbolSet& operator = (const SymbolSet& Obj_);

    // convenience methods from porting old FieldInfo and TokenInfo class
    virtual SymbolSet& SetStartDelim(const char* Start_, bool Cs_=true, bool Ord_=true);
    virtual SymbolSet& SetEndDelim(const char* End_, bool Cs_=true, bool Ord_=true);
    virtual char* GetStartDelim() const;
    virtual char* GetEndDelim() const;

    virtual SymbolSet& SetEnclosing(SymbolSet* Enclosing_);    
    virtual SymbolSet& SetLeftBracket(SymbolSet* Left_, bool Overwrite_=true);
    virtual SymbolSet& SetDelimiter(SymbolSet* Delim_, bool Overwrite_=true);
    virtual SymbolSet& SetNext(SymbolSet* Next_, bool Overwrite_=true);
    virtual SymbolSet& SetRightBracket(SymbolSet* Right_, bool Overwrite_=true);
    
    virtual SymbolSet* NextLeftBracket(int Index_=0, bool Transfer_=false);
    virtual SymbolSet* NextSymbol(int Index_=0, bool Transfer_=false);
    virtual SymbolSet* NextDelimiter(int Index_=0, bool Transfer_=false);
    virtual SymbolSet* NextRightBracket(int Index_=0, bool Transfer_=false);
    virtual bool BrkSymTransferred(int Index_) const;
    
    virtual bool HasSymbol(const SymbolSet* SymPtr_, bool NoChain_=false);
    virtual bool HasLeftBracket() const;
    virtual bool HasDelimiter() const;
    virtual bool HasNext() const;
    virtual bool HasRightBracket() const;

    virtual bool IsBracketedSymbol(bool FollowEnclosing_=false) const;
    virtual int AggregateType() const;

    virtual char* ParseSymbolStep(char* Text_, bool& Done_);
    virtual char* ParseSymbol(char* Text_);
    
    virtual int GetBrkSymRequired() const;
    virtual int GetBrkSymState() const;

    virtual SymbolSet& IncBlockCount(int Val_);
    virtual SymbolSet& IncFunctionCount(int Val_);

    virtual SymbolSet::TriggerFncData* GiveTriggerFncData();
    virtual SymbolSet::StateChangeFncData* GiveStateChangeFncData();
    virtual ObjectFunctor* GiveTriggerObject();
    virtual ObjectFunctor* GiveStateChangeObject();    

    virtual SymbolSet& SetStateChangeObject(ObjectFunctor* Obj_);
    virtual SymbolSet& SetStateChangeFunction(int(*Fnc_)(int, char**, void*),
                                      int FncArgc_, char** FncArgv_,
                                      bool ResultArg_, bool BranchArg_,
                                      void* FncArgi_);

    virtual SymbolSet& SetTriggerObject(ObjectFunctor* Obj_);
    virtual SymbolSet& SetTriggerFunction(int(*Fnc_)(int, char**, void*),
                                  int FncArgc_, char** FncArgv_,
                                  bool ResultArg_, bool BranchArg_,
                                  void* FncArgi_);    

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

typedef BracketedSymbol FieldInfo;

/****************************************************************************/
#endif

