#ifndef TAGELEMENT_H
#define TAGELEMENT_H
#ifndef TAGINFO_H
  #include "taginfo.h"
#endif
#ifndef CHRSTRING_H
  #include "chrstring.h"
#endif
#ifndef SIMPLELIST_CPP
  #include "simplelist.cpp"
#endif

#define TAGELEMENT_DEBUG        0  // main() testing
#define TAGELEMENT_DEBUG1       0  // TagElement list node deletion debug

/****************************************************************************/
// TagCountData class forward declaration
//
class TagCountData;

// 0th element: valid tag = { 0 = ignored, 1 = active, 2 = inactive }
// 1st element: nesting level = unsigned integer
// 2nd element: tag element counter = unsigned integer
// 3rd element: tag attrib bits =
//                { 0   = none,
//                  1   = singleton,
//                  2   = empty,
//                  4   = pair,               // matches with pair ender
//                  8   = optional pair,
//                  16  =   allow singleton,  // only set if optional pair
//                  32  =   allow empty,      // only set if optional pair
//                  64  = pair ender,         // single tag to match tag pair
//                  128 = tag container,      // specifies container type tag
//                  256 = tag specific }      // specific to tag element
//
// 4th element: index into individual tag element list -- TagCnt, TagBrk
// 5th element: index of matching tag in pair sequence if tag attrib is pair
// 6th element: Tag position: opening, forbidden, closing
// 7th element: Tag list node sequence number where this element is stored
// 8th element: Required tag order for start+end optional tags =
//                  { 0 = NO_REQTAG,          // no required tag
//                    1 = BEFORE_REQTAG,      // before required tag
//                    2 = AFTER_REQTAG,       // after required tag
//                    4 = SURROUND_REQTAG, }  // both before and after required tags
//
// 9th element: Implied end of tag at specific condition met =
//                  { 0 = NO_IMPLIEDEND,      // no implied end
//                    1 = AT_EOF,             // at end of file
//                    2 = AT_BEFORE_REQTAG,   // at required BEFORE tag
//                    4 = AT_NEXTTAG }        // at next encountered tag
//
// 10th element: Required tag for s+e optional tag found and condition met:
//                  { 0  = TAG_NOT_FOUND,       // tags (cur+req) not found
//                    1  = REQTAG_FOUND,        // required tag found
//                    2  = CURTAG_FOUND,        // current tag found
//                    4  = REQTAG_COND_NOT_MET, // required tag condition not met
//                    8  = REQTAG_COND_MET,     // required tag condition met
//                   16  = AFTREQTAG_FOUND,     // required AFTER tag found
//                   32  = BEFREQTAG_FOUND,     // required BEFORE tag found
//                   64  = TAG_COND_MET_NOREQ } // tag condition met no required tag
//
// 11th element: Implied end state for tags with specific implied end conditions
//                  { 0 = TNF_IMPLIEDEND_NOT_MET,      // searching for req. tags, implied end cond NOT met
//                    1 = NOREQTAG_IMPLIEDEND_NOT_MET, // req. tags cond NOT met, implied end cond NOT met
//                    2 = REQTAG_IMPLIEDEND_NOT_MET,   // req. tags cond met, implied end cond NOT met
//                    4 = NOREQTAG_IMPLIEDEND_MET,     // req. tags cond NOT met, implied end cond met
//                    8 = REQTAG_IMPLIEDEND_MET }      // req. tags cond met, implied end cond met
//
// 12th element: Tag validity state correlating with 0th element valid tag
//                  { 0 = IGNORED,
//                    1 = ACTIVE,
//                    2 = INACTIVE }
//
// 13th element: Content Tag Type
//                  { 0  = NONE,
//                    1  = HTMLCONTENT_SPECIAL,
//                    2  = HTMLCONTENT_HTML,
//                    4  = HTMLCONTENT_HEAD,
//                    8  = HTMLCONTENT_BODY,
//                    16 = HTMLCONTENT_STDTAGPAIR }
//
// Required Tag enumerated logic and sequence map:
// REQTAG_COND_MET = AFTER_REQTAG && (TAG_NOT_FOUND-->REQTAG_FOUND &&
//                                    REQTAG_FOUND-->CURTAG_FOUND)
//
//                 = BEFORE_REQTAG && (TAG_NOT_FOUND-->CURTAG_FOUND &&
//                                     CURTAG_FOUND-->REQTAG_FOUND)
//
//                 = SURROUND_REQTAG && (TAG_NOT_FOUND-->AFTREQTAG_FOUND &&
//                                       AFTREQTAG_FOUND-->CURTAG_FOUND &&
//                                       CURTAG_FOUND-->BEFREQTAG_FOUND)
//
// TAG_COND_MET_NOREQ = NO_REQTAG && TAG_NOT_FOUND-->CURTAG_FOUND
//
// Required Tag sequence check:
// if (_TagAttrib[OPTREQTAGORDER] == AFTER_REQTAG)
//   if (*this == Obj_)
//     1.0 CURTAG_FOUND-->REQTAG_COND_NOT_MET : (No further assignments)
//   else
//   {
//     if (AftTagNotFound_ && *_AftRequiredTag == Obj_) {
//       2.1 TAG_NOT_FOUND-->REQTAG_FOUND : from IsRequiredTag(&Obj_)
//       if (AftReqTagFound_)
//         2.2 REQTAG_FOUND-->CURTAG_FOUND : from IsEqual(Obj_)
//         2.3 REQTAG_COND_MET
//   }}
//
// if (_TagAttrib[OPTREQTAGORDER] == BEFORE_REQTAG)
//   if (BefTagNotFound_ && *_BefRequiredTag == Obj_)
//     1.0 REQTAG_FOUND-->REQTAG_COND_NOT_MET : (No further assignments)
//   else
//   {
//     if (BefTagNotFound_ && *this == Obj_) {
//       2.1 TAG_NOT_FOUND-->CURTAG_FOUND : from IsEqual(Obj_)
//       if (CurTagFound_)
//         2.2 CURTAG_FOUND-->REQTAG_FOUND : from IsRequiredTag(&Obj_)
//         2.3 REQTAG_COND_MET
//   }}
//
// if (_TagAttrib[OPTREQTAGORDER] == SURROUND_REQTAG)
//   if (*this == Obj_)
//     1.0 CURTAG_FOUND-->REQTAG_COND_NOT_MET : (No further assignments)
//   else
//   {
//     if (AftTagNotFound_ && *_AftRequiredTag == Obj_) {
//       2.1 TAG_NOT_FOUND-->AFTREQTAG_FOUND : from IsRequiredTag(&Obj_)
//       if (AftReqTagFound_)
//         2.1.1 AFTREQTAG_FOUND-->CURTAG_FOUND : from IsEqual(Obj_)
//         2.1.2 CurTagFound_ = true
//         if (CurTagFound_ &&
//             BefTagNotFound_ && *_BefRequiredTag == Obj_) {
//           3.1 CURTAG_FOUND-->BEFREQTAG_FOUND
//           3.2 REQTAG_COND_MET
//
//     if (CurTagNotFound_ &&
//         BefTagNotFound_ && *_BefRequiredTag == Obj_) {
//       2.2 REQTAG_FOUND-->REQTAG_COND_NOT_MET : (No further assignments)
//   }}
//
// if (_TagAttrib[OPTREQTAGORDER] == NO_REQTAG)
//   if (*this == Obj_)
//     1.1 TAG_NOT_FOUND-->CURTAG_FOUND : from IsEqual(Obj_)
//     1.2 REQTAG_COND_MET
//
// For int TagAttrib[] :
//   VALIDITY        = 0,  // Inherits parent's tag validity
//   NESTING         = 1,  // Tag element specific nesting, overwrites parent
//   TAGCOUNT        = 2,  // Tag element specific tag count. overwrites parent
//   TAGATTR         = 3,  // Tag element's attribute is set upon object creation
//   TAGINDEX        = 4,  // Inherits parent's tag index
//   MATCHINDEX      = 5,  // Inherits parent's matching tag index
//   TAGPOS          = 6,  // Tag position: opening, forbidden, closing
//   LISTSEQ         = 7,  // List seq. position is set upon object creation
//   OPTREQTAGORDER  = 8,  // Required tag order for start+end optional tags
//   IMPLIEDEND      = 9,  // Implied end of tag at specific condition met
//   REQTAGCOND      = 10, // Required tag for s+e optional tag found
//   IMPLIEDENDCOND  = 11, // Implied end state for tags with specific implied end conditions
//   VALIDITYSTATE   = 12, // Tag validity state correlating with 0th element valid tag
//   HTMLCONTTAGTYPE = 13, // Html content tag type for html tags
//
// Format of parent object's cumulative element sequence list:
//   [AbsSize.] [listsize.] [listseq][attrib][tagcount][nesting]...
//
// AbsSize: absolute size of list in number of ints allocated
// listsize: the size of the int list in (listseq+attrib+tagcount+nesting) groups
// listseq: list sequence of referred TagElement node
// attrib: attributes of the tag in the TagElement node
// tagcount: running tag count with specific value for each encountered tag
// nesting: running nesting with specific value for each encountered tag
//
// The cumulative element sequence list is specific to the parent object
// which holds only one single instance.
//

// class TagElement forward declaration
class TagElement;
class StackNodeInfo;
class TagTypeInfo;

// new : TagElementStates structure
// created by TagLinksSequencer ptr in
// _GenSearchTag ptr -- _Sequencer StackInfo stack _HeadInfo
// and then passed on to each TagElement object that is stored in each
// StackNodeInfo structure that is created and stored in _HeadInfo
//
struct TagElementStates
{
  StackNodeInfo* _Parent;  // StackNodeInfo parent object which created/owns this object
  size_t _StackNodeID;
  bool _Copied;

  // tag counts and constants
  int _Validity;           // 0th element: valid tag = { 0 = ignored, 1 = active, 2 = inactive }
  int _NestingLevel;       // 1st element: nesting level = unsigned integer
  int _TagCount;           // 2nd element: tag element counter = unsigned integer
  int _HtmlContTagType;    // 13th element: Html content tag type for html tags

  // tag conditions
  int _ReqTagCond;         // 10th element: Required tag for s+e optional tag found and condition met:
  int _ImpliedEndCond;     // 11th element: Implied end state for tags with specific implied end conditions
  int _ValidityState;      // 12th element: Tag validity state correlating with 0th element valid tag

  long _ReqTagFoundIndex;          // Index value into required tag array of found tag
  bool _ImpliedTagFound;           // Implied ending tag of tag pair found
  int _ReqTagDirFound;             // Tag Dir of found required tag = { BEFORE_REQTAG, AFTER_REQTAG }
  const TagElementStates* _EndTag; // Terminating end tag of this current tag element
  int _EndTagCond;                 // End tag termination condition corresponding to tag pair

  int _TagCountsSet           : 1; // Nesting level and tag cound values set?
  int _ReqTagCondChanged      : 1; // Required tag condition changed : 10th element
  int _ImpliedEndCondChanged  : 1; // Implied end condition changed : 11th element
  int _ValidityStateChanged   : 1; // Validity state changed : 12th element:

  TagElementStates(StackNodeInfo* Parent_, size_t NodeID_);

  inline bool IsCopied() const
      { return _Copied; }
  inline bool TagCountsSet() const
      { return (_TagCountsSet && _NestingLevel); }
  inline bool HasThisParent(const StackNodeInfo* Parent_) const
      { return (_Parent == Parent_); }

  inline void SetParent(StackNodeInfo* Parent_)
      { _Parent = Parent_; }
  inline void SetCopied(bool Copied_)
      { _Copied = Copied_; }

  bool SetTagValidityState(int Val_);

  void ResetState(bool States_, bool Conds_,
                  bool Counts_, bool TagConsts_,
                  bool ReqTagSrch_);

  void CopyFromTagStates(const TagElement* TagPtr_,
                         bool States_, bool Conds_);
  void CopyToTagStates(const TagElement* TagPtr_,
                       bool States_, bool Conds_,
                       bool Counts_, bool TagConsts_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

  EXTERNALCLASSFNCS_STUB(TagElementStates, TagElementStates)
  CLONEFNCS_INLINE_DEFN(TagElementStates)
};

class TagElement : public TagEnums
{
  friend class TagCountData;
  friend class TagElementStates;

  private:
    enum
    {
      REQTAGFOUNDINDEX_OFFSET = 100000,
      IMPLIEDTAGFOUND_OFFSET  = 10000,
      REQTAGDIRFOUND_OFFSET   = 1000,
      ENDTAGCOND_OFFSET       = 10
    };

  protected:
    const TagCountData* _Parent;
    ChrString _TagName;
    mutable int _TagAttrib[MAX_FIELDS];

    TagElement* _MatchTag;  // Matching end tag of tag pair

    TagElement** _BefRequiredTags;
    bool* _MatchingEndTags;
    bool* _EnclosingPairTags;
    long _MaxBefReqTags;
    long _NumBefReqTags;

    TagElement** _AftRequiredTags;
    long _MaxAftReqTags;
    long _NumAftReqTags;

    mutable TagElementStates* _TagStates;
    mutable TagElementStates* _SrcTagStates;

    mutable int _TagStatesChanged       : 1;
    mutable int _ReqTagCondChanged      : 1; // Required tag condition changed : 10th element
    mutable int _ImpliedEndCondChanged  : 1; // Implied end condition changed : 11th element
    mutable int _ValidityStateChanged   : 1; // Validity state changed : 12th element:

    void SetTagListSeq(int ListSeq_);
    void DestroyMatchingTag();
    void CopyVectors(const TagElement& Obj_);

    TagElement** EraseVector(int ReqTagOrder_);
    TagElement** GrowVector(int ReqTagOrder_, long Incr_=10);

    bool NonContainerOptTagMatch(const TagElement* Ptr_) const;
    bool IsMatchingTag(const TagElement* Ptr_) const;
    bool IsRequiredTag(const TagElement* Ptr_, int TestingCond_) const;
    bool TestReqTagCond(const TagElement& Obj_, int TestingCond_) const;

    bool HasRequiredTagAddr(const TagElement* Ptr_) const;
    bool MatchesRequiredTags(const TagElement* Ptr_,
                             bool* BefTagFndPtr_=NULL,
                             bool* AftTagFndPtr_=NULL) const;

    void CopyFromTagStates(bool States_, bool Conds_) const;
    void CopyToTagStates(bool States_, bool Conds_,
                         bool Counts_, bool TagConsts_) const;

    inline bool CompareMatchingTags(const TagElement& Obj_) const
        { return IsMatchingTag(Obj_._MatchTag); }

    inline void SetReqTagFoundIndex(long Index_) const
        { if (_TagStates) _TagStates->_ReqTagFoundIndex = Index_; }
    inline void SetImpliedTagFound(bool Flag_) const
        { if (_TagStates) _TagStates->_ImpliedTagFound = Flag_; }
    inline void SetReqTagDirFound(int Dir_) const
        { if (_TagStates) _TagStates->_ReqTagDirFound = Dir_; }
    inline void SetEndTag(const TagElementStates* Ptr_) const
        { if (_TagStates) _TagStates->_EndTag = Ptr_; }
    inline void SetEndTagCond(int Cond_) const
        { if (_TagStates) _TagStates->_EndTagCond = Cond_; }

  public:
    TagElement(int ListSeq_, int TagAttr_,
               const TagCountData* Parent_, int* ParentAttr_,
               bool DefnOnly_=true, TagElementStates* TagStates_=NULL);
    TagElement(const TagElement& Obj_);
    ~TagElement();

    static TagElement** InitTagElementArray(TagElement** ArrPtr_, size_t max);
    static bool IsPairTagAttr(int TagAttr_,
                              bool* Optional_=NULL, bool* Single_=NULL,
                              bool* Empty_=NULL, bool* Ender_=NULL,
                              bool* Container_=NULL, bool* TagSpecific_=NULL);

    TagElement& SetTagAttrib(int Attr_, int Value_);
    TagElement& IncrTagAttrib(int Attr_, int Incr_=1);
    TagElement& DecrTagAttrib(int Attr_, int Decr_=1);
    TagElement& SetSingletonAllowed(bool Set_);
    TagElement& SetEmptyAllowed(bool Set_);
    TagElement& SetPairEnderTag(bool PairEnder_);
    TagElement& SetSingletonTag(bool Single_);
    TagElement& SetEmptyTag(bool Empty_);
    TagElement& SetContainerTag(bool Contain_);

    TagElement& SetTagPos(int TagPos_);
    TagElement& SetTagName(const ChrString& Tag_);
    TagElement& SetMatchingTag(TagElement* MatchTag_, bool Terminate_=false);

    TagElement& ResetRequiredTags();
    TagElement& SetHtmlContentTagType(int ContentType_);
    TagElement& SetRequiredCond(int ReqTagOrder_, int ImpliedEnd_=-1,
                                int TagCond_=0, int ImpliedEndCond_=0);
    TagElement& AddRequiredTag(TagElement* ReqTag_,
                               int ReqTagOrder_, bool MatchingEnd_=false);
    TagElement& SetRequiredTags(TagElement** Vect_,
                                int ReqTagOrder_, bool MatchingEnd_=false);

    TagElement& operator = (const TagElement& Obj_);
    bool UpdateTagElement(const TagElement& Obj_);
    bool SetTagValidityState(int Val_) const;
    void ResetState(bool States_, bool Conds_,
                    bool Counts_, bool TagConsts_,
                    bool ReqTagSrch_);

    void RenewTagElementState();
    void SetTagElementState(TagElementStates* Ptr_);
    inline void SetSourceTagElementState(TagElementStates* Ptr_)
        { _SrcTagStates = Ptr_; }

    inline const TagElementStates* TagElementState() const
        { return _TagStates; }
    inline TagElementStates* GiveTagElementState()
        { return _TagStates; }

    inline const TagElementStates* SourceTagElementState() const
        { return _SrcTagStates; }
    inline TagElementStates* GiveSourceTagElementState()
        { return _SrcTagStates; }

    inline bool HasThisParent(const TagCountData* Parent_) const
        { return (_Parent == Parent_); }
    inline int ValidityChanged() const
        { return _ValidityStateChanged; }
    inline int ReqTagCondChanged() const
        { return _ReqTagCondChanged; }
    inline int ImpliedEndCondChanged() const
        { return _ImpliedEndCondChanged; }

    inline long ReqTagFoundIndex() const
        { return (_TagStates ? _TagStates->_ReqTagFoundIndex:0); }
    inline bool ImpliedTagFound() const
        { return (_TagStates ? _TagStates->_ImpliedTagFound:0); }
    inline int ReqTagDirFound() const
        { return (_TagStates ? _TagStates->_ReqTagDirFound:0); }
    inline const TagElementStates* EndTag() const
        { return (_TagStates ? _TagStates->_EndTag:0); }
    inline int EndTagCond() const
        { return (_TagStates ? _TagStates->_EndTagCond:0); }

    inline int TagAttrib(int Attr_) const
        { return _TagAttrib[Attr_]; }
    inline const int* TagAttribPtr(int Base_=0) const
        { return ((0 <= Base_ && Base_ < MAX_FIELDS) ?
                      &_TagAttrib[Base_]:
                      _TagAttrib); }
    inline const ChrString& TagName() const
        { return _TagName; }
    inline TagElement* GiveMatchingTag()
        { return _MatchTag; }
    inline TagElement** GiveRequiredTags(int Type_)
        { return ((Type_ == BEFORE_REQTAG) ? _BefRequiredTags:
                  (Type_ == AFTER_REQTAG) ? _AftRequiredTags:NULL); }

    int TagNesting(bool PairOnly_=true) const;
    inline int TagCount() const
        { return _TagAttrib[TagEnums::TAGCOUNT]; }
    inline int TagAttrib() const
        { return _TagAttrib[TagEnums::TAGATTR]; }

    inline int TagBrkIndex() const
        { return _TagAttrib[TagEnums::TAGINDEX]; }
    inline int TagMatchIndex() const
        { return _TagAttrib[TagEnums::MATCHINDEX]; }
    inline int TagPosition() const
        { return _TagAttrib[TagEnums::TAGPOS]; }
    inline int TagListSeq() const
        { return _TagAttrib[TagEnums::LISTSEQ]; }

    inline int TagValidity() const
        { return _TagAttrib[TagEnums::VALIDITY]; }
    inline int TagOptReqTagOrder() const
        { return _TagAttrib[TagEnums::OPTREQTAGORDER]; }
    inline int TagImpliedEnd() const
        { return _TagAttrib[TagEnums::IMPLIEDEND]; }
    inline int TagReqTagCond() const
        { return _TagAttrib[TagEnums::REQTAGCOND]; }
    inline int TagImpliedEndCond() const
        { return _TagAttrib[TagEnums::IMPLIEDENDCOND]; }
    inline int TagValidityState() const
        { return _TagAttrib[TagEnums::VALIDITYSTATE]; }
    inline int HtmlContentTagType() const
        { return _TagAttrib[TagEnums::HTMLCONTTAGTYPE]; }

    inline bool AtReqTagOrder(int ReqOrder_) const
        { return (_TagAttrib[TagEnums::OPTREQTAGORDER] == ReqOrder_); }
    inline bool AtReqTagCond(int ReqCond_) const
        { return (_TagAttrib[TagEnums::REQTAGCOND] & ReqCond_); }

    inline bool RequiredTagCondNotMet() const
        { return ((_TagAttrib[TagEnums::REQTAGCOND] & TagEnums::REQTAG_COND_NOT_MET) &&
                  !(_TagAttrib[TagEnums::REQTAGCOND] & TagEnums::REQTAG_COND_MET)); }
    inline bool RequiredTagCondMet() const
        { return ((_TagAttrib[TagEnums::REQTAGCOND] & TagEnums::REQTAG_COND_MET) &&
                  !(_TagAttrib[TagEnums::REQTAGCOND] & TagEnums::REQTAG_COND_NOT_MET)); }
    inline bool NoRequiredTag() const
        { return (_TagAttrib[TagEnums::OPTREQTAGORDER] == TagEnums::NO_REQTAG); }
    inline bool CurrentTagCondMet() const
        { return (_TagAttrib[TagEnums::REQTAGCOND] & TagEnums::TAG_COND_MET_NOREQ); }

    inline void ResetTagFoundCond()
        { _TagAttrib[TagEnums::REQTAGCOND] = TagEnums::TAG_NOT_FOUND; }
    inline void ResetImpliedEndCond()
        { _TagAttrib[TagEnums::IMPLIEDENDCOND] = TagEnums::TNF_IMPLIEDEND_NOT_MET; }
    inline void ResetTagValidityState()
        { _TagAttrib[TagEnums::VALIDITYSTATE] = _TagAttrib[TagEnums::VALIDITY]; }

    inline bool IsEqualToMatchTag(const TagElement& Obj_) const
        { return IsMatchingTag(&Obj_); }
    inline bool TestAftReqCond(const TagElement& Obj_) const
        { return TestReqTagCond(Obj_, TagEnums::AFTER_REQTAG); }
    inline bool TestBefReqCond(const TagElement& Obj_) const
        { return TestReqTagCond(Obj_, TagEnums::BEFORE_REQTAG); }
    inline bool TestCurTagCond(const TagElement& Obj_) const
        { return TestReqTagCond(Obj_, TagEnums::CURRENT_REQTAG); }

    // retrieve data from TagCountData class
    int GiveBrkNesting(const char* TagBrkStr_) const;
    int GiveTagCount(const char* TagBrkStr_) const;
    const char* GiveTagBrk(int Ender_=0) const;
    const char* GiveTagMatchBrk(int Ender_=0) const;

    bool IsEqual(const TagElement& Obj_) const;
    bool TagLookupEqual(const TagElement& Obj_) const;
    bool IsPairTag(bool* Optional_=NULL, bool* Single_=NULL,
                   bool* Empty_=NULL, bool* Ender_=NULL,
                   bool* Container_=NULL, bool* TagSpecific_=NULL) const;

    bool IsContainerTag(bool Optional_, bool Both_=false) const;
    bool IsOptionalTag() const;
    bool IsSingleTag(bool ExcludeEmpty_=false) const;
    bool IsPairStartTag() const;
    bool IsPairEndTag() const;

    bool NestedTagAllowed() const;
    bool DuplicateTagAllowed() const;

    long NumEnclosingPairs() const;
    TagElement* EnclosingPair(long Index_);

    TagElement* MoreRequiredBeforeTags(int& Index_, bool& Reset_);
    TagElement* MoreRequiredAfterTags(int& Index_, bool& Reset_);

    bool AtReqTagCondStopState(bool PartialMet_, bool FailOnly_=false) const;
    inline bool TagAtReqTagCondNIE() const
        { return (_TagAttrib[TagEnums::IMPLIEDENDCOND] == TagEnums::REQTAG_NO_IMPLIEDEND); }
    inline bool TagAtImpliedEnd() const
        { return (_TagAttrib[TagEnums::IMPLIEDENDCOND] == TagEnums::REQTAG_IMPLIEDEND_MET); }
    inline bool TagAtMatchingEnd() const
        { return (_MatchingEndTags ?
                     (_BefRequiredTags[ReqTagFoundIndex()] &&
                      _MatchingEndTags[ReqTagFoundIndex()]):false); }
    inline bool TagAtBeforeReqTag() const
        { return (_BefRequiredTags ?
                     (_BefRequiredTags[ReqTagFoundIndex()] &&
                      !_MatchingEndTags[ReqTagFoundIndex()]):false); }
    inline bool TagIsValid() const
        { return (_TagAttrib[TagEnums::VALIDITYSTATE] == TagEnums::ACTIVE ||
                  _TagAttrib[TagEnums::VALIDITYSTATE] == TagEnums::IGNORED); }

    void DumpTagAttrib(ostream& os_, size_t fw_) const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

    EXTERNALCLASSFNCS_STUB(TagElement, TagElement)
    CLONEFNCS_INLINE_DEFN(TagElement)
};

// TagElementComparer class
class TagElementComparer : public MatchableTraits<TagElement>
{
  STATICCLASSINSTANCE_VARS_DECL(TagElementComparer)

  public:  
    TagElementComparer();
    ~TagElementComparer();

    virtual MatchableTraits<TagElement>* Clone() const;
    virtual Boolean Equal(const TagElement& x, const TagElement& y) const;
    virtual Boolean NotEqual(const TagElement& x, const TagElement& y) const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

    EXTERNALCLASSFNCS_STUB(TagElementComparer, TagElementComparer)
    CLONEFNCS_INLINE_DEFN(TagElementComparer)
};

// TagElementMatchTagComparer class
class TagElementMatchTagComparer : public MatchableTraits<TagElement>
{
  STATICCLASSINSTANCE_VARS_DECL(TagElementMatchTagComparer)

  public:
    TagElementMatchTagComparer();
    ~TagElementMatchTagComparer();

    virtual MatchableTraits<TagElement>* Clone() const;
    virtual Boolean Equal(const TagElement& x, const TagElement& y) const;
    virtual Boolean NotEqual(const TagElement& x, const TagElement& y) const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

    EXTERNALCLASSFNCS_STUB(TagElementMatchTagComparer, TagElementMatchTagComparer)
    CLONEFNCS_INLINE_DEFN(TagElementMatchTagComparer)
};

/****************************************************************************/
struct HtmlSpecialCaseData
{
  TagTypeInfo* _Parent;

  bool _HtmlTagFound;
  bool _HeadTagFound;
  bool _HeadEndTagFound;
  bool _BodyTagFound;

  bool _HtmlTagOmitted;
  bool _HeadTagOmitted;
  bool _HeadEndTagOmitted;
  bool _BodyTagOmitted;

  bool _HtmlTagError;
  bool _HeadTagError;
  bool _HeadEndTagError;
  bool _BodyTagError;

  HtmlSpecialCaseData();

  void Reset();
  void CheckForSpecialHtmlTags(const ChrString& TagStr_);
  int GiveTagErrors() const;

  inline void SetParent(TagTypeInfo* Parent_)
      { _Parent = Parent_; }
  inline bool HasThisParent(const TagTypeInfo* Parent_) const
      { return (_Parent == Parent_); }

  inline bool HasTagErrors() const
      { return (_HtmlTagError || _HeadTagError ||
                _HeadEndTagError || _BodyTagError); }

  inline bool HasTagOmitted() const
      { return (_HtmlTagOmitted || _HeadTagOmitted ||
                _HeadEndTagOmitted || _BodyTagOmitted); }

  inline bool HasTagFound() const
      { return (_HtmlTagFound || _HeadTagFound ||
                _HeadEndTagFound || _BodyTagFound); }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

  EXTERNALCLASSFNCS_STUB(HtmlSpecialCaseData, HtmlSpecialCaseData)
  CLONEFNCS_INLINE_DEFN(HtmlSpecialCaseData)
};

/****************************************************************************/
class TagBracketList : public TagEnums
{
  private:
    // static members for single instance object creation
    static TagBracketList* m_This;
    static size_t m_ObjectCnt;
    static bool m_AutoRelease;
    static bool m_StaticCreate;
    static bool m_StaticDestroy;
    static void Init();

  public:
    // static members for single instance object creation
    static TagBracketList* Instance();
    static void Release();
    
  protected:  
    TagBracketList();

    virtual void AssignGlobalTagBrkList(char** TagBrkList_);
    virtual void AddTagBrk(const char* Str_);
    virtual void SetupTagData(bool DoAlloc_);
    virtual void DestroyTagList();
    virtual void InitTagList();
  
    // globally defined tag brackets for various tag types
    static const char* _DocTypeOpen_Str;
    static const char* _DocTypeClose_Str;
    static const char* _CommentOpen_Str;
    static const char* _CommentClose_Str;
    static const char* _NestedComment1Open_Str;
    static const char* _NestedComment1Close_Str;
    static const char* _NestedComment2Open_Str;
    static const char* _NestedComment2Close_Str;
    static const char* _StdTagOpen_Str;
    static const char* _StdTagClose_Str;
    static const char* _EndTagOpen_Str;
    static const char* _EndTagClose_Str;
    static const char* _EmptyOpen_Str;
    static const char* _EmptyClose_Str;
    static const char* _ScriptOpen_Str;
    static const char* _ScriptClose_Str;

    static const char* _CCommentBlkOpen_Str;
    static const char* _CCommentBlkClose_Str;
    static const char* _COpenCommentBlkOpen_Str;
    static const char* _COpenCommentBlkClose_Str;
    static const char* _CCloseCommentBlkOpen_Str;
    static const char* _CCloseCommentBlkClose_Str;
    static const char* _CppCommentLineOpen_Str;
    static const char* _CppCommentLineClose_Str;
    static char** _GlobalTagBrkList;

    // Tag bracket array, user specified
    size_t _ObjectID;
    bool _ManualCreate;

    char** _TagBrkMatrix;
    int _TagBrkIndex;   // index to current highest entry in _TagBrkList
    long _TagBrkMax;    // maximum allocated capacity of _TagBrkList

  public:
    TagBracketList(int ManualCreate_);
    ~TagBracketList();
  
    // static members for single instance object creation  
    static void InitStrings();
    static const char* GiveNestedCommentStr(int Index_);
    static const char* GiveNestedCommentChar(int Index_, char* chp);

    const char* GiveTagBrk(int Index_) const;
    int GiveTagBrkIndexForStr(const char* Str_) const;
    inline int GiveTagBrkIndex() const
        { return _TagBrkIndex; }
    inline char** GiveTagBrkMatrix()
        { return _TagBrkMatrix; }

    static bool IsScriptTag(const char* start_, const char* end_=NULL);
    static bool IsEmptyTag(const char* start_, const char* end_);
    static bool IsNestedCommentTag(const char* start_, const char* end_, int Type_);
    static bool IsNestedCommentOpenTag(const char* start_, const char* end_);
    static bool IsNestedCommentCloseTag(const char* start_, const char* end_);
    static bool IsCommentTag(const char* start_, const char* end_=NULL);
    static bool IsDocTypeTag(const char* start_, const char* end_);
    static bool IsStdTag(const char* start_, const char* end_=NULL);
    static bool IsEndTag(const char* start_, const char* end_=NULL);
    static bool IsCCommentBlkTag(const char* start_, const char* end_);
    static bool IsNestedCBlkTag(const char* start_, const char* end_, int Type_);
    static bool IsCOpenTag(const char* start_, const char* end_);
    static bool IsCCloseTag(const char* start_, const char* end_);
    static bool IsCppTag(const char* start_, const char* end_=NULL);
    static bool IsLangTag(const char* start_, const char* end_);
    static int GiveTagType(const ChrString& tagstr_);
    static ChrString StripTagBrackets(const ChrString& tagstr_, int tagtype_,
                                      ChrString* StartBrk_=NULL, ChrString* EndBrk_=NULL);
    static ChrString FilterTagElementStr(const ChrString& tagstr_);

    static const char* TagArrayFieldsToStr(int Field_);
    static const char* TagPositionToStr(int Pos_);
    static const char* TagErrorsToStr(int Error_);
    static const char* TagTypeToStr(int Type_);
    static const char* TagArrayIndexToStr(int Index_);
    static const char* TagAllowFlagValuesToStr(int Flag_, char* Buf_);
    static const char* TagRetrContFlagValuesToStr(int Flag_, char* Buf_);
    static const char* TagValidityToStr(int Validity_);
    static const char* TagAttributesToStr(int Attrib_, char* Buf_);
    static const char* TagOptReqTagOrderToStr(int ReqOrder_);
    static const char* TagImpliedEndToStr(int Implied_);
    static const char* TagReqTagCondToStr(int ReqCond_, char* Buf_);
    static const char* TagImpliedEndCondToStr(int ImpliedCond_, char* Buf_);
    static const char* TagDocStyleTypeToStr(int DocStyle_);
    static const char* TagDocFileTypeToStr(int DocFile_);
    static const char* TagScanModeToStr(int Mode_);
    static const char* TagSearchParameterToStr(int Param_);
    static const char* TagSearchTypeToStr(int SearchType_);
    static const char* TagHtmlContentTagTypeToStr(int HtmlContTagType_);
    static int TagAttribToTagPos(int TagAttrib_);
    static int TagPosToTagAttrib(int TagPos_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif    
};

/****************************************************************************/
class TagCountData : public TagBracketList
{
  typedef SearchableList<TagElement>* TAGLISTPTR;
  typedef SimpleListIterator<TagElement>* TAGLISTITER;

  public:
    enum
    {
      TAGLISTSEQ_OFFSET = 0,
      TAGATTRIB_OFFSET  = 1,
      TAGCOUNT_OFFSET   = 2,
      TAGNESTING_OFFSET = 3,

      TOTALSIZE_OFFSET  = 0,
      DATACOUNT_OFFSET  = 1,

      ELEMENTLIST_BASE_OFFSET = 2,
      ELEMENTLIST_OFFSET_SIZE = 4
    };

  private:
    // static members for single instance object creation
    static TagCountData* m_This;
    static size_t m_ObjectCnt;
    static bool m_AutoRelease;
    static bool m_StaticCreate;
    static bool m_StaticDestroy;
    static void Init();

  public:
    // static members for single instance object creation  
    static TagCountData* Instance();
    static void Release();
    static int** InitTagCntMatrix(int** ArrPtr_, size_t max);
    static char** InitTagBrkArray(char** ArrPtr_, size_t max);

  // Tag list indexes
  protected:
    TagCountData();

    virtual void SetupTagData(bool DoAlloc_);
    virtual void DestroyObjectData();
    virtual void InitTagList();

    // Tag brackets list counters
    int _DocTypeOpen_Cnt[MAX_FIELDS];           // Singleton tag
    int _DocTypeClose_Cnt[MAX_FIELDS];          // Singleton tag
    int _CommentOpen_Cnt[MAX_FIELDS];           // Singleton tag
    int _CommentClose_Cnt[MAX_FIELDS];          // Singleton tag
    int _NestedComment1Open_Cnt[MAX_FIELDS];    // pair, matches with NestedComment2
    int _NestedComment1Close_Cnt[MAX_FIELDS];   // pair, matches with NestedComment2
    int _NestedComment2Open_Cnt[MAX_FIELDS];    // pair ender
    int _NestedComment2Close_Cnt[MAX_FIELDS];   // pair ender
    int _StdTagOpen_Cnt[MAX_FIELDS];            // tag specific, matches with EndTag
    int _StdTagClose_Cnt[MAX_FIELDS];           // tag specific, matches with EndTag
    int _EndTagOpen_Cnt[MAX_FIELDS];            // pair ender
    int _EndTagClose_Cnt[MAX_FIELDS];           // pair ender
    int _EmptyOpen_Cnt[MAX_FIELDS];             // Singleton tag, empty tag
    int _EmptyClose_Cnt[MAX_FIELDS];            // Singleton tag, empty tag
    int _ScriptOpen_Cnt[MAX_FIELDS];            // Singleton tag
    int _ScriptClose_Cnt[MAX_FIELDS];           // Singleton tag

    int _CCommentBlkOpen_Cnt[MAX_FIELDS];       // Singleton tag
    int _CCommentBlkClose_Cnt[MAX_FIELDS];      // Singleton tag
    int _COpenCommentBlkOpen_Cnt[MAX_FIELDS];   // pair, matches with CCloseComment
    int _COpenCommentBlkClose_Cnt[MAX_FIELDS];  // pair, matches with CCloseComment
    int _CCloseCommentBlkOpen_Cnt[MAX_FIELDS];  // pair ender
    int _CCloseCommentBlkClose_Cnt[MAX_FIELDS]; // pair ender
    int _CppCommentLineOpen_Cnt[MAX_FIELDS];    // Singleton tag
    int _CppCommentLineClose_Cnt[MAX_FIELDS];   // Singleton tag

    size_t _ObjectID;

    static bool _DestroyLists;
    TAGLISTPTR _StdTagList;
    TAGLISTPTR _EndTagList;
    int* _ElementSeqList;    // Cumulative element sequence list
    int _SeqListIndex;       // Index for element sequence list
    int _SeqIndexIncr;       // Increment for sequence list index update

    HtmlSpecialCaseData* _HtmlSpecialCase;
    int** _TagCntMatrix;
    int _TagCntIndex;
    long _TagCntMax;

    TAGLISTITER _ListIter;
    int _TagIterIndex;       // tag list iterator index for _StdTagList
    int _DocType;            // document type used for selecting reset method
    bool _EofFound;          // End of file marker found
    bool _TagDataUpdated;    // Tag element and tag sequence list updated
    int _SeqListUpdateIndex; // Sequence list index used for updates to list

    int* SetupElementSeqList();
    int* GrowElementSeqList(int Incr_=GROWINCR);
    void UpdateSeqList(TagElement* Element_, int PairEnder_,
                       bool FindAndUpdate_);
    
    TagCountData& InitTagCounts();
    TagElement* MakeMatchTag(TagElement* TagPtr_, bool& CreatedTag_) const;
    TagElement* SearchForTag(int TagPos_, bool Empty_, bool Container_, const char* StartTagBrk_,
                             const ChrString& TagElement_, const char* EndTagBrk_,
                             int& PairEnder_, bool Update_);

  public:  
    TagCountData(int ManualCreate_);
    ~TagCountData();

    static void SetDestroyLists(bool dls_);
    TagCountData& SetHtmlSpecialCase(HtmlSpecialCaseData* Ptr_);
    TagCountData& CopyTagCount(const TagCountData* TagCnt_);
    TagCountData& Reset(bool ResetCounts_);
    TagCountData& DeleteTagLists();

    const int* GiveTagCountArray(int Index_) const;
    int GiveTagCount(const char* TxtFndBrk_) const;
    int GiveBrkNesting(const char* TxtFndBrk_) const;
    int PushBrkNesting(const char* UpdateBrk_, const char* TxtFndBrk_, int Nest_);
    int PopBrkNesting(const char* UpdateBrk_, const char* TxtFndBrk_, int Nest_);
    int BrkStringToIndex(const char* Brk_, int& BrkEnder_, bool Substr_=false) const;
    int BrkStringToIndex(const char* StartBrk_, const char* EndBrk_,
                         int& PairEnder_, bool Substr_=false) const;
    bool IsParentOf(const TagElement* Child_) const;
    bool IsElementTagStr(const char* SrcStartTagBrk_,
                         const ChrString& SrcTagElement_, const char* SrcEndTagBrk_,
                         const char* TrgStartTagBrk_,
                         const ChrString& TrgTagElement_, const char* TrgEndTagBrk_);

    TagElement* UpdateTag(int TagPos_, bool Empty_, bool Container_,
                          const char* StartTagBrk_, const ChrString& TagElement_,
                          const char* EndTagBrk_);
    TagElement* SearchForTag(int TagPos_, bool Empty_, bool Container_,
                             const char* StartTagBrk_, const ChrString& TagElement_,
                             const char* EndTagBrk_);
    TagElement* AddToTagList(int TagPos_, bool Empty_, bool Container_, int AppAttr_,
                             const char* StartTagBrk_, const ChrString& TagElement_,
                             const char* EndTagBrk_, TagElement* MatchingTag_);
    TagElement* MakeMatchTag(TagElement* TagPtr_,
                             bool SetMatchTag_, bool AddToList_);

    TagElement* ResetRequiredTags(TagElement* TagPtr_);
    TagElement* SetHtmlContentTagType(TagElement* TagPtr_, int ContentType_);
    TagElement* SetRequiredCond(TagElement* TagPtr_,
                                int ReqTagOrder_, int ImpliedEnd_=-1,
                                int TagCond_=0, int ImpliedEndCond_=0);
    TagElement* AddRequiredTag(TagElement* TagPtr_, TagElement* ReqTag_,
                               int ReqTagOrder_, bool MatchingEnd_=false);
    TagElement* SetRequiredTags(TagElement* TagPtr_, TagElement** Vect_,
                                int ReqTagOrder_, bool MatchingEnd_=false);

    TagElement* SetPairEnderTag(TagElement* TagPtr_, bool PairEnder_);
    TagElement* SetSingletonAllowed(TagElement* TagPtr_, bool Set_);
    TagElement* SetEmptyAllowed(TagElement* TagPtr_, bool Set_);
    TagElement* SetSingletonTag(TagElement* TagPtr_, bool Single_);
    TagElement* SetEmptyTag(TagElement* TagPtr_, bool Empty_);
    TagElement* SetContainerTag(TagElement* TagPtr_, bool Contain_);

    TagCountData& ResetTagDataUpdate();
    TagCountData& ResetTagIterIndex();
    bool SetTagIterIndex(int TagIndex_);
    bool AdvanceTagIter();
    TagElement* GetCurrentTagElement();
    TagCountData& SetDocType(int DocType_);
    TagCountData& SetEofFound(bool Eof_);

    // Resets validity property in tag count array, 0th element
    void ResetHtmlTags();
    void ResetXmlTags();
    void ResetCppTags();
    void ResetCTags();

    inline const HtmlSpecialCaseData* GiveHtmlSpecialCase() const
        { return _HtmlSpecialCase; }
    inline HtmlSpecialCaseData* GiveHtmlSpecialCase()
        { return _HtmlSpecialCase; }
    inline size_t TagListSize(int TagPos_) const
        { return ((TagPos_ == TagEnums::OPENING) ?
                       _StdTagList->Size():
                       _EndTagList->Size()); }
    inline bool AtFirstTag() const
        { return (_TagIterIndex == 0); }
    inline bool AtLastTag() const
        { return ((_TagIterIndex+1) >= _StdTagList->Size()); }

    inline int GiveTagIterIndex()
        { return _TagIterIndex; }
    inline int GiveTagCntIndex() const
        { return _TagCntIndex; }
    inline int** GiveTagCntMatrix()
        { return _TagCntMatrix; }

    inline int DocTypeSpecified() const
        { return _DocType; }
    inline bool EofFound() const
        { return _EofFound; }
    inline bool TagDataUpdated() const
        { return (_TagDataUpdated && _SeqListUpdateIndex > 0); }

    const int* GiveCurrentSeqListNode(int* IndexPtr_=NULL) const;
    inline const int* GiveElementSeqList() const
        { return _ElementSeqList; }
    inline const int GiveSeqIndexIncr() const
        { return _SeqIndexIncr; }
    inline const int GiveSeqListIndex() const
        { return _SeqListIndex; }
    inline const int GiveSeqListSize() const
        { return (_ElementSeqList ? _ElementSeqList[0]:0); }
    inline const int GiveSeqListDataCnt() const
        { return (_ElementSeqList ? _ElementSeqList[1]:0); }

    void DumpElementSeqListNode(ostream& os_, size_t fw_=20) const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

    EXTERNALCLASSFNCS_STUB(TagCountData, TagCountData)
    CLONEFNCS_INLINE_DEFN(TagCountData)
};

/****************************************************************************/
#endif

