#ifndef TAGREADER_H
#define TAGREADER_H
#ifndef TAGSTKNODE_H
  #include "tagstknode.h"
#endif

#define TAGREADER_TAGINFOLISTWALKDUMP   0  // walk stack info list from tail to head and dump each node:
                                           //   tagdump3xd.txt, tagdump4xd.txt

#define TAGREADER_TAGSTACK_DEBUG        1  // tagdump7.txt, TAGPARSEINFO_DEBUG == 0
#define TAGREADER_PARSESTR_DEBUG        0  // tagdump7a.txt, test TagReader::ParseTextStr() method
#define TAGREADER_PARSEDOC_DEBUG        1  // tagdump7b.txt, test TagReader::ParseDocument() method
#define TAGREADER_PARSEDOC2_DEBUG       0  // tagdump7c.txt, test TagReader::ParseDocumentByParts() method

#define TAGREADER_CREATEFROMSNI_DEBUG   0  // print debug info for method:
                                           //   TagReader::CreateFromStackNodeInfo, file: tagdump7cfsni.txt
#define TAGREADER_CREATEFROMSNI2_DEBUG  1  // print debug info for WalkTagStack method and submethods
                                           //   file: tagdump7trtsw.txt
#define TAGREADER_CREATEFROMSNI3_DEBUG  1  // print debug info for TagLineInfo::CreateFromStackNodeInfo
                                           // method calls and debug info from containing methods
                                           //   file: tagdump7trtsd.txt

#define TAGREADER_CREATEFROM_DEBUG \
        ((TAGREADER_CREATEFROMSNI_DEBUG)   | \
         (TAGREADER_CREATEFROMSNI_DEBUG2)  | \
         (TAGREADER_CREATEFROMSNI_DEBUG3))

/****************************************************************************/
// TagReader forward declaration
//
class TagReader;
class TagStack;

// class to contains bare tag data which includes:
//   TagParseInfo:
//     string: start bracket, _TagBrkMatrix[_TagBrkIndex]
//     string: end bracket,  _TagBrkMatrix[_TagBrkIndex+1]
//     enum: found tag type
//     enum: tag attr
//
//   TagCommentData:
//     string: intra comment end bracket
//     string: intra comment start bracket
//     bool: intra comment tag
//     enum: intra comment start/end
//     enum: intra comment active/inactive/ignored
//     enum: comment type cstyle/cppstyle/xmlstyle
//
//   KeyValuePairData:
//     string: Key/Value pair delim
//     string: White space
//     string: Tag terminator
//     KeyValuePairNode**: Key/Value pair array
//
//   bool: ending tag
//   bool: in comment
//
class TagStringData
{
  protected:
    TagParseInfo* _TagParse;          // created for each TagStringData object
    TagCommentData* _TagCommentData;  // created comment data object
                                      // only created if _InComment == true
    KeyValuePairData* _KeyValueData;  // created key value pair data object
                                      // only created if _HasKvPairs == true
    TagDictionary* _Dictionary;       // passed pointer to tag dictionary

    const char* _StartTagBrkStr;      // starting bracket string of tag
    const char* _EndTagBrkStr;        // ending bracket string of tag
    TagElement* _ElementPtr;          // Tag element ptr of matching created TagElement object
    TagStack* _RefTagStk;             // reference TagStack ptr
    StackNodeInfo* _RefStkNode;       // reference StackNode ptr
    ChrString _TagString;             // Char string showing tag found

    mutable Bit _HasTagContents:1;      // true if tag contains text content
    mutable Bit _HasIntraTagContents:1; // true if tag contains internal tag content
    mutable Bit _HasKvPairs:1;          // true if tag has internal key-value pairs
    mutable Bit _HasCommentData:1;      // true if tag has comment tag data
    mutable Bit _InComment:1;           // true if in html comment
    mutable Bit _InScript:1;            // true if within script tags
    mutable Bit _SingletonTag:1;        // true if tag is an singleton tag
    mutable Bit _DocTypeTag:1;          // true if tag is an doctype tag
    mutable Bit _EndingTag:1;           // true if tag is an ending tag

  public:
    TagStringData(TagParseInfo* Ptr_, TagDictionary* Ptr_,
                  TagStack* TstkPtr_, StackNodeInfo* StknPtr_);
    TagStringData(const TagStringData& Obj_);
    ~TagStringData();

    TagStringData& SetRefTagStk(TagStack* Ptr_);
    TagStringData& SetRefStkNode(StackNodeInfo* Ptr_);
    TagStringData& SetTagDictionary(TagDictionary* Ptr_);
    TagStringData& operator = (const TagStringData& Obj_);

    bool HasTagContents() const;
    bool HasIntraTagContents() const;
    bool HasKvPairs() const;
    bool HasCommentData() const;
    bool InComment() const;
    bool InScript() const;
    bool SingletonTag() const;
    bool DocTypeTag() const;
    bool EndingTag() const;

    bool IsParentOf(const TagParseInfo* Child_) const;
    bool IsParentOf(const TagCommentData* Child_) const;
    bool IsParentOf(const KeyValuePairData* Child_) const;

    inline const char* GetStartTagBrkStr()
        { return _StartTagBrkStr; }
    inline const char* GetEndTagBrkStr()
        { return _EndTagBrkStr; }

    inline TagElement* GiveTagElementPtr()
        { return _ElementPtr; }
    inline TagStack* GiveRefTagStk()
        { return _RefTagStk; }
    inline StackNodeInfo* GiveRefStkNode()
        { return _RefStkNode; }
    inline const ChrString& GiveTagString()
        { return _TagString; }
        
    inline TagCommentData* GiveTagCommentData()
        { return _TagCommentData; }
    inline KeyValuePairData* GiveKeyValuePairData()
        { return _KeyValueData; }
    inline TagParseInfo* GiveTagParseInfo()
        { return _TagParse; }
    inline TagDictionary* GiveTagDictionary()
        { return _Dictionary; }
    
    TagStringData& SetTextBufferPtr(const ChrString* Ptr_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
// class to contain tag string data AND file position data
class TagLineInfo;

class TagString : public TagStringData
{
  protected:
    TagLineInfo* _Parent;          // Parent TagLineInfo object
    FilePosition* _FilePosRec;     // created file position object

    mutable long _NestLevel;       // Same as _Parent->NestLevel();
    mutable long _SequenceNum;     // Same as _Parent->SequenceNum();

  public:
    TagString(TagLineInfo* Parent_,
              TagStack* TstkPtr_, StackNodeInfo* StknPtr_);
    TagString(const TagString& Obj_);
    ~TagString();

    static TagString* Make(TagLineInfo* Parent_,
                           TagStack* TstkPtr_, StackNodeInfo* StknPtr_);
    static TagString* Make(const TagString& Obj_);

    TagString& operator = (const TagString& Obj_);

    TagLineInfo* GiveParent();
    const TagLineInfo* GiveParent() const;
    FilePosition* GiveFilePosition();
    long NestLevel() const;
    long SequenceNum() const;

    void ShowTagString(const ChrString& Header_, bool AppTab_=false);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif    
};

/****************************************************************************/
class TagStack;
class TagVector;

class TagCoordData
{
  // _ValueSet[CHILDSTK]        // child stack nesting level set
  // _PrevValues[CHILDSTK]      // previous child stack nesting level
  // _CurrValues[CHILDSTK]      // child stack nesting level
  //
  // _ValueSet[CHILDVECT]       // child vector index position set
  // _PrevValues[CHILDVECT]     // previous child vector index position
  // _CurrValues[CHILDVECT]     // child vector index position
  //
  // _ValueSet[TAGSTK]          // stack nesting level set
  // _PrevValues[TAGSTK]        // previous stack nesting level
  // _CurrValues[TAGSTK]        // stack nesting level
  //
  // _ValueSet[TAGVECT]         // vector index position set
  // _PrevValues[TAGVECT]       // previous vector index position
  // _CurrValues[TAGVECT]       // vector index position

  private:
    enum
    {
      TAGSTK    = 0,
      TAGVECT   = 1,
      CHILDSTK  = 2,
      CHILDVECT = 3,
      MAXVALUES = 4
    };

    bool _ReadOnly;
    long _StackOffset;          // Child stack nesting level offset

    long _StackNestLevel;
    long _VectorIndexPos;

    TagCoordData* _PrevData;    // Previous coordinate data structure
    TagCoordData* _NextData;    // Next coordinate data structure

    TagVector* _Parent;         // Parent owning tag vector of this object
    TagStack* _Next;            // Next level of tag stack

  public:
    TagCoordData(TagStack* StkPtr_,
                 TagVector* VectPtr_, long Offset_=0);
    TagCoordData(const TagCoordData& Obj_);
    virtual ~TagCoordData();

    static TagCoordData* Make(const TagCoordData& Obj_);
    static TagCoordData* Make(TagStack* StkPtr_,
                              TagVector* VectPtr_, long Offset_=0);

    TagCoordData& operator = (const TagCoordData& Obj_);

    void SetStackNestLevel(long lvl, bool Spec_);
    void SetVectorIndexPos(long pos, bool Spec_);

    TagCoordData* SetNext(TagStack* Next_);
    TagCoordData* SetParent(TagVector* Next_, TagCoordData* Child_);

    inline bool StackNestLevelSpec() const
        { return (_StackNestLevel >= 0); }
    inline bool VectorIndexPosSpec() const
        { return (_VectorIndexPos >= 0); }

    long StackNestLevel(bool Offset_=true) const;
    long VectorIndexPos() const;

    inline void SetPrevData(TagCoordData* Ptr_)
        { _PrevData = Ptr_; }
    inline void SetNextData(TagCoordData* Ptr_)
        { _NextData = Ptr_; }

    inline long StackOffset() const
        { return _StackOffset; }
    inline bool ReadOnly() const
        { return _ReadOnly; }
    inline void SetReadOnly(bool rd)
        { _ReadOnly = rd; }

    inline TagCoordData* PrevCoord()
        { return _PrevData; }
    inline const TagCoordData* PrevCoord() const
        { return _PrevData; }

    inline TagCoordData* NextCoord()
        { return _NextData; }
    inline const TagCoordData* NextCoord() const
        { return _NextData; }

    inline TagStack* Next()
        { return _Next; }
    inline const TagStack* Next() const
        { return _Next; }

    inline TagVector* Parent()
        { return _Parent; }
    inline const TagVector* Parent() const
        { return _Parent; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
class TagCoordSpecifier
{
  private:
    enum
    {
      TAGSTK    = 0,
      TAGVECT   = 1,
      CHILDSTK  = 2,
      CHILDVECT = 3,
      MAXVALUES = 4
    };

    bool _ValueSet[MAXVALUES];
    bool _ValueSpec[MAXVALUES];
    long _PrevValues[MAXVALUES];
    long _CurrValues[MAXVALUES];

    long _StackOffset;          // Child stack nesting level offset
    bool _ReadOnly;

    TagCoordData* _CoordData;

  public:
    TagCoordSpecifier(long Offset_=0);
    TagCoordSpecifier(TagCoordData* Ptr_);
    TagCoordSpecifier(TagVector* Ptr_);
    TagCoordSpecifier(const TagCoordSpecifier& Obj_);

    virtual ~TagCoordSpecifier();

    static TagCoordSpecifier* Make(long Offset_=0);
    static TagCoordSpecifier* Make(TagCoordData* Ptr_);
    static TagCoordSpecifier* Make(TagVector* Ptr_);
    static TagCoordSpecifier* Make(const TagCoordSpecifier& Obj_);

    TagCoordSpecifier& operator = (TagCoordData* Ptr_);
    TagCoordSpecifier& operator = (TagVector* Ptr_);
    TagCoordSpecifier& operator = (const TagCoordSpecifier& Obj_);

    long StackNestLevel(bool Offset_=true) const;
    long ChildStackLevel(bool Offset_=true) const;
    long VectorIndexPos() const;
    long ChildVectorIndex() const;

    long NewStackNestLevel(bool Offset_=true) const;
    long NewChildStackLevel(bool Offset_=true) const;
    long NewVectorIndexPos() const;
    long NewChildVectorIndex() const;

    void SetStackNestLevel(long lvl,
                           bool Spec_, bool AllowUpdate_=true);
    void SetChildStackLevel(long lvl,
                            bool Spec_, bool AllowUpdate_=true);
    void SetVectorIndexPos(long pos,
                           bool Spec_, bool AllowUpdate_=true);
    void SetChildVectorIndex(long pos,
                             bool Spec_, bool AllowUpdate_=true);
    void SetMaxChildren(long max,
                        bool Spec_, bool AllowUpdate_=true);

    void SetReadOnly(bool rd);
    inline bool ReadOnly() const
        { return _ReadOnly; }

    inline bool StackNestLevelSpec() const
        { return _ValueSpec[TAGSTK]; }
    inline bool ChildStackLevelSpec() const
        { return _ValueSpec[CHILDSTK]; }
    inline bool VectorIndexPosSpec() const
        { return _ValueSpec[TAGVECT]; }
    inline bool ChildVectorIndexSpec() const
        { return _ValueSpec[CHILDVECT]; }

    inline long StackOffset() const
        { return _StackOffset; }
    inline TagCoordData* GiveCoordData()
        { return _CoordData; }

    void CommitNewPos();
    void RollbackPos();
    void RestoreFrom(TagCoordSpecifier* Ptr_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
class TagCoordWrapper : public TagCoordSpecifier
{
  private:
    bool _NewPos;
    TagReader* _TagReader;

  public:
    TagCoordWrapper(const TagCoordSpecifier& Obj_,
                    TagReader* Reader_, bool NewPos_);

    TagCoordWrapper& operator = (const TagCoordWrapper& Obj_);

    long GiveCoordVectLen() const;
    long GiveTagStkLevel(long VectLen_, bool Create_) const;
    long GiveNextTagStkLevel(bool Create_) const;
    long GiveNextTagVectIndex(bool Create_) const;

    bool IsTagStkLevelSpec(long VectLen_, long StkLvl_) const;
    bool IsNextTagStkLevelSpec(long VectLen_,
                               bool Create_, long NextStkLvl_) const;
    bool IsNextTagVectIndexSpec(bool Create_,
                                long NextVectIndex_, long VectorLen_) const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
//   tag stack
//   includes:
//     scan mode: { tagread, tagcheck }
//     nesting number
//     tag vector length
//     tag vector maximum capacity
//     tag string vector array
//
//   tag string vector
//   includes:
//     pointer to parent tag stack
//     pointer to next level tag stack
//     pointer to tagstring
//
//       tag sequence number
//       matchingnum: matching sequence number of opening tag for end tags
//
//       tagattr: { Singleton / Pair / OptionalPair / Empty }
//       tag error: { pending / valid / error / tagerr_type... }
//       anchor type: { opening / closing / forbidden }
//

class TagVector
{
  friend class TagStack;
  friend class TagCoordData;

  protected:
    bool _ReadOnly;                // Read only copy of previously constructed tag object
    TagStack* _Parent;             // parent stack object which holds this tag vector
    TagStack* _NextLevel;          // created vertical stack level links to tags (nesting)
    TagCoordData* _ParentStack;    // pointer to parent TagCoordData object
    TagCoordData* _ChildStack;     // pointer to child TagCoordData object

    TagDictionary* _Dictionary; // passed pointer to tag dictionary

    TagLineInfo* _TagLine;    // created tag line info object,
                              //   passed to _TagLine:
                              //     this->_NextLevel == _TagLine->_TagStack
    TagParseInfo* _TagParse;  // same as: _TagLine->_StartTag->_TagParse
                              //      or  _TagLine->_EndTag->_TagParse

    long _SequenceNum;  // sequence number within horizontal sequence level
                        // _TagParse->StackInfoPtr()->SequenceNumber()
    long _MatchingNum;  // matching horizontal seq. number of tag pair ender
                        // if (_TagParse->StackInfoPtr()->ForwardLink())
                        //   _TagParse->StackInfoPtr()->ForwardLink()->SequenceNumber()

    void ShowVectorEntry(long slvl, long vlvl, long nlvl);
    bool IsRootCoordData(TagCoordData* Ptr_);

    TagCoordData* PushStack(TagStack* NewStk_);
    TagCoordData* PushStack(long NestLevel_);

    TagCoordData* SetParentStack(TagCoordData* Ptr_);
    TagCoordData* SetChildStack(TagCoordData* Ptr_, TagCoordData* Coord_);

  public:
    TagVector(TagStack* Parent_, TagParseInfo* TagParse_);
    TagVector(const TagVector& Obj_);
    ~TagVector();

    static TagVector* Make(TagStack* Parent_, TagParseInfo* TagParse_);
    static TagVector* Make(const TagVector& Obj_);

    TagVector& operator = (const TagVector& Obj_);

    TagVector& SetTagParseInfo(TagParseInfo* TagParse_);
    TagVector& SetTagDictionary(TagDictionary* Ptr_);
    TagVector& SetSequenceNum(long SeqNum_);
    TagVector& SetMatchingNum(long MatchNum_);

    TagCoordData* SetNextLevel(TagStack* NextLevel_);
    TagCoordData* CreateNextLevel(long NestLevel_);
    TagCoordData* EraseNextLevel(TagCoordData* ParStk_, bool RdOnly_=false);
    TagCoordData* PopStack();

    const TagStack* Parent() const;
    const TagStack* NextLevel() const;
    const TagLineInfo* GiveTagLineInfo() const;
    const TagParseInfo* GiveTagParseInfo() const;
    const TagDictionary* GiveTagDictionary() const;
    bool IsParentOf(const TagLineInfo* Child_) const;
    void GiveCoordinate(long& NestLevel_, long& SeqNum_) const;

    TagStack* Parent();
    TagStack* NextLevel();
    TagLineInfo* GiveTagLineInfo();
    TagParseInfo* GiveTagParseInfo();
    TagDictionary* GiveTagDictionary();

    inline bool HasThisParent(const TagStack* Parent_) const
        { return (_Parent == Parent_); }
    long NestLevel() const;
    inline long SequenceNum() const
        { return _SequenceNum; }
    inline long MatchingNum() const
        { return _MatchingNum; }
    inline void SetReadOnly(bool rd)
        { _ReadOnly = rd; }

    inline TagCoordData* CurrCoordData()
        { return _ParentStack; }
    inline TagCoordData* NextCoordData()
        { return _ChildStack; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
class TagStack
{
  protected:
    int _ScanMode;    // scan mode can be either TAGREAD or TAGCHECK
    bool _ReadOnly;   // Read only copy of previously constructed tag object
    long _NestLevel;  // Nesting level of current horizontal tag sequence
                      // Default initial nesting level is 0.
                      // _TagReader->GiveTagDictionary()->StackInfoPtr()->NestLevel()

    long _VectorLen;  // horizontal sequence tag vector running tag count
    long _VectorMax;  // horizontal sequence tag vector upper limit

    // horizontal sequence level links to tags
    // _TagVector[0] ... _TagVector[n] in sequence of tags
    // of length n the same nesting level _NestLevel
    TagReader* _TagReader;         // pointer to tag reader object
    TagVector** _TagVector;        // created horizontal sequence tag vector
    TagCoordData* _FromCoord;      // enclosing TagCoordData structure

    TagVector** GrowVector(long Incr_=10);
    bool HasVector(TagVector* Ptr_);

    void ShowCoordVect(long* CoordVect_, long VectLen_) const;

    TagVector* ResetTagVectorChild(TagVector* TVect_);
    TagVector* SetTagVectorChild(TagVector* TVect_,
                                 long StkLvl_=-1, long Index_=-1);

    TagVector* GiveTFCoordHelper(bool Create_, long* CoordVect_,
                                 long Index_, TagCoordWrapper* Tcw,
                                 TagVector* FoundVectPtr_, bool* Found_);
    const TagVector* GiveTFCoordHelper(long* CoordVect_, long Index_,
                                       TagCoordWrapper* Tcw,
                                       const TagVector* FoundVectPtr_,
                                       bool* Found_) const;

    void ShowStackData();
    void ShowVectorData(long slvl, long x=-1);

  public:
    TagStack(long NestLevel_,
             TagReader* TrdPtr_, TagCoordData* CoordPtr_=NULL);
    TagStack(const TagStack& Obj_);
    ~TagStack();

    static TagStack* Make(TagCoordSpecifier* TCspec_, long NestLevel_,
                          TagReader* TrdPtr_, TagCoordData* CoordPtr_=NULL);
    static TagStack* Make(const TagStack& Obj_);

    TagStack& operator = (const TagStack& Obj_);

    TagStack& SetScanMode(int Mode_);
    TagStack& SetNestLevel(long Level_);
    TagStack& SetTagReader(TagReader* Ptr_);

    TagVector* AddTagVector(TagVector* Ptr_, bool KeepPtr_=false);
    TagVector* CreateTagVector(TagParseInfo* Ptr_, long StkLvl_, long Index_);
    TagVector* CreateTagVector(TagDictionary* Ptr_, long StkLvl_, long Index_);
    TagStack& EraseTagVectors();

    TagVector* GiveTagVector(long Index_, bool* Found_=NULL);
    const TagVector* GiveTagVector(long Index_, bool* Found_=NULL) const;

    TagVector* GiveTagFromCoordinate(bool Create_, long* CoordVect_, bool New_,
                                     TagCoordSpecifier* TCoord_, bool* Found_=NULL);
    const TagVector* GiveTagFromCoordinate(long* CoordVect_, bool New_,
                                           TagCoordSpecifier* TCoord_, bool* Found_=NULL) const;

    void WalkTagStack(long slvl);
    void SetFromCoordData(TagCoordData* From_, long StkLvl_=-1);

    TagDictionary* GiveTagDictionary();
    TagParseInfo* GiveTagParseInfo();

    bool IsRootCoordData(TagCoordData* Ptr_) const;
    bool IsParentOf(const TagVector* Child_) const;

    inline TagReader* GiveTagReader()
        { return _TagReader; }
    inline const TagReader* GiveTagReader() const
        { return _TagReader; }
    inline int ScanMode() const
        { return _ScanMode; }
    inline long NestLevel() const
        { return _NestLevel; }
    inline long VectorLen() const
        { return _VectorLen; }
    inline long VectorMax() const
        { return _VectorMax; }
    inline void SetReadOnly(bool rd)
        { _ReadOnly = rd; }

    inline TagVector* FromVect()
        { return (_FromCoord ? _FromCoord->Parent():NULL); }
    inline TagCoordData* FromCoordData()
        { return _FromCoord; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
class TagLineInfo
{
  friend class TagVector;

  protected:
    // _StartTag and _EndTag are mutually exclusive with either
    // having NULL or a pointer to a TagString object, but not both
    TagString* _StartTag;  // Passed TagString object to store data for start tag
                           // assigned to NULL if _EndTag allocated.
    TagString* _EndTag;    // Passed TagString object to store data for end tag
                           // assigned to NULL if _StartTag allocated.

    bool _UnorderedTag;         // An unordered tag (non-content type tag)
                                // found before the starting html tag <html>
    bool _ImpliedEnd;           // for tags with optional end
                                // _StartTag->GiveTagParseInfo()->StackInfoPtr()->EndImplied()
    bool _Singleton;            // Singleton tag or empty tag
                                // _StartTag->GiveTagParseInfo()->StackInfoPtr()->IsSingleton()

    TagLineInfo* _MatchingTag;  // Matching start TagLineInfo for end tags
                                // assigned to NULL if _StartTag allocated.
    TagLineInfo* _TerminalTag;  // Terminating end TagLineInfo for start tags
                                // assigned to NULL if _EndTag allocated.
    
    // Tag Content buffer
    ChrString* _TagContent;  // String containing tag contents
                             // if (_StartTag->GiveTagParseInfo()->StackInfoPtr()->HasTagContents())
                             //   _StartTag->GiveTagParseInfo()->StackInfoPtr()->GiveTagContent()

    TagVector* _Parent;      // parent TagVector object
    TagStack* _TagStack;     // same as _Parent->_NextLevel

    Bit _Matched:1;          // TagLineInfo object matches search parameters
    Bit _Shown:1;            // Contents of TagLineInfo object is shown
    Bit _Modified:1;         // Contents of TagLineInfo object is modified
    Bit _Saved:1;            // Contents of TagLineInfo object is saved

    inline void SetTagStack(TagStack* StkPtr_)
        { _TagStack = StkPtr_; }

    void ShowTagLineInfo(bool Terminate_=false);

  public:
    TagLineInfo(TagVector* Parent_, TagLineInfo* MatchTag_=NULL);
    TagLineInfo(const TagLineInfo& Obj_);
    ~TagLineInfo();

    static TagLineInfo* Make(TagVector* Parent_, TagLineInfo* MatchTag_=NULL);
    static TagLineInfo* Make(const TagLineInfo& Obj_);

    TagLineInfo& operator = (const TagLineInfo& Obj_);
    TagLineInfo* CreateFromStackNodeInfo(TagReader* Trd_,
                                         StackNodeInfo* Ptr_, bool ChildCreated_);

    TagString* SetStartTag(TagString* StartTag_, bool EndTagImplied_,
                           bool SingletonTag_, bool UnorderedTag_, bool KeepPtr_=false);
    TagString* SetEndTag(TagString* EndTag_, bool KeepPtr_=false);
    TagLineInfo* SetMatchingTag(TagLineInfo* MatchTag_);
    TagLineInfo* SetTerminalTag(TagLineInfo* TermTag_);
    TagLineInfo& SetTagContent(ChrString* Ptr_, bool KeepPtr_=false);
    TagDictionary* GiveTagDictionary();
    TagParseInfo* GiveTagParseInfo();

    inline bool UnorderedTag() const
        { return _UnorderedTag; }
    inline bool ImpliedEnd() const
        { return _ImpliedEnd; }
    inline bool Singleton() const
        { return _Singleton; }
    inline TagLineInfo* MatchingTag()
        { return _MatchingTag; }
    inline TagLineInfo* TerminalTag()
        { return _TerminalTag; }
    inline TagString* StartTag()
        { return _StartTag; }
    inline TagString* EndTag()
        { return _EndTag; }
    inline long NestLevel() const
        { return (_Parent ? _Parent->NestLevel():0); }
    inline long SequenceNum() const
        { return (_Parent ? _Parent->SequenceNum():0); }
        
    inline TagVector* Parent()
        { return _Parent; }
    inline ChrString* GiveTagContent()
        { return _TagContent; }        
    inline TagStack* GiveTagStack()
        { return _TagStack; }
    inline bool HasThisParent(const TagVector* Parent_) const
        { return (_Parent == Parent_); }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
class TagErrorTracker
{
  protected:
    TagReader* _TagReader;        // pointer to TagReader object
    FilePosition* _FilePosRec;    // pointer to file position object

    TagDictionary* _Dictionary;      // pointer to tag dictionary object
    TagCommentData* _TagCommentData; // pointer to comment data object
    KeyValuePairData* _KeyValueData; // pointer to key value pair data object

  public:
    TagErrorTracker(TagReader* Trd_);
    ~TagErrorTracker();

    static TagErrorTracker* Make(TagReader* Trd_);

    bool HasError() const;
    int TagParseError() const;
    FilePosition* GiveFilePosition();

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
// DocType : C_FILE                = 1,
//           CPP_FILE              = 2,
//           XML_FILE              = 4,
//           HTML_FILE             = 8,
//           XHTML_FILE            = 16
//
class TagParseData
{
  private:
    bool _TagLinkedFwd;           // parsed tag has forward link to matching end tag
    bool _ChildrenCreated;        // Children vector indexes created
    bool _ChildVisited;           // First child node of child vector indexes visted
    bool _PairEndingTag;          // Tag pair ending tag found
    bool _EndingTagPopped;        // Tag pair ending tag popped from stack

    long _MaxTags;                // Maximum number of valid created tag entries
    long _CreatedTags[10];        // Created tags
      
    NodeNestingData* _TagFwdLink;   // forward link of tag
    StackNodeInfo* _StkNodeFwdLink; // forward link of StackNodeInfo

    TagVector* _StartTagVector;
    TagVector* _EndTagVector;

    TagLineInfo* _CreatedStartLine;
    TagLineInfo* _CreatedEndLine;

  public:
    TagParseData();

    bool TagCreated(long snl, long sn);
    void CreateTag(long snl, long sn);

    void ClearTags();
    void Reset();

    inline void SetTagFwdLink(NodeNestingData* lnk)
        { _TagFwdLink = lnk; }
    inline void SetStkNodeFwdLink(StackNodeInfo* lnk)
        { _StkNodeFwdLink = lnk; }
    inline void SetStartTagVector(TagVector* lnk)
        { _StartTagVector = lnk; }
    inline void SetEndTagVector(TagVector* lnk)
        { _EndTagVector = lnk; }

    inline void SetChildrenCreated(bool v)
        { _ChildrenCreated = v; }
    inline void SetChildVisited(bool v)
        { _ChildVisited = v; }
    inline void SetTagLinkedFwd(bool v)
        { _TagLinkedFwd = v; }
    inline void SetPairEndingTag(bool v)
        { _PairEndingTag = v; }
    inline void SetEndingTagPopped(bool v)
        { _EndingTagPopped = v; }

    inline bool ChildrenCreated() const
        { return _ChildrenCreated; }
    inline bool ChildVisited() const
        { return _ChildVisited; }
    inline bool TagLinkedFwd() const
        { return _TagLinkedFwd; }
    inline bool HasTagFwdLink() const
        { return (_TagLinkedFwd &&
                  (_StkNodeFwdLink || _TagFwdLink)); }
    inline bool IsPairEndingTag() const
        { return _PairEndingTag; }
    inline bool EndingTagPopped() const
        { return _EndingTagPopped; }

    inline StackNodeInfo* GiveStkNodeFwdLink()
        { return _StkNodeFwdLink; }
    inline StackNodeInfo* GiveStkNodeBackLink()
        { return (_StkNodeFwdLink ?
                      _StkNodeFwdLink->BackLink():
                      NULL); }

    inline NodeNestingData* GiveTagFwdLink()
        { return _TagFwdLink; }
    inline TagVector* GiveStartTagVector()
        { return _StartTagVector; }
    inline TagVector* GiveEndTagVector()
        { return _EndTagVector; }

    inline TagLineInfo* GiveCreatedStartLine()
        { return _CreatedStartLine; }
    inline void SetCreatedStartLine(TagLineInfo* ptr)
        { _CreatedStartLine = ptr; }
    inline TagLineInfo* GiveCreatedEndLine()
        { return _CreatedEndLine; }
    inline void SetCreatedEndLine(TagLineInfo* ptr)
        { _CreatedEndLine = ptr; }
};

/****************************************************************************/
class TagReader : public TagParseInfo
{
  friend class TagStack;
  friend class TagLineInfo;
  friend class TagErrorTracker;

  enum { MAXTAG_MARGIN = 5 };      // max tag entries at root tag vector

  private:
    TextFileLineBuffer* _LineBuf;  // created text file line buffer
    FileReader* _File;             // created file reader

    bool _ParseByParts;         // parse html/xml document by parts
    bool _OrderedTagsFound;     // standard html content type tags found

    int _TagBalance;            // tag balance value
    int _TagSetsFound;          // sets of tag found counter
    int _TagPosAdvCnt;          // tag position advanced counter
    int _NumUnorderedTags;      // number of unordered tags

    long* _Coords;              // stack/vector coordinate vector
    long _CoordsSize;           // size of coordinate vector
    long _MaxNestLevel;         // maximum nesting level of TagStack object
    long _StackOffset;          // Child stack nesting level offset

    bool _AtNewPos;
    bool _AtCurrentPos;

    TagParseData* _TPData;          // Tag parsing data
    TagCoordSpecifier* _TCoord;     // Tag coordinate data of current TagStack object
    TagCoordSpecifier* _OldTCoord;  // Copy olf tag coordinate data of current TagStack object

    ChrString _Buffer;          // tag string text buffer as ChrString object
    ChrString _MainFileName;    // main document file name

    // Pointer to created tag related objects
    TagStack* _RootStack;             // root tag stack object
    TagStack* _TagStack;              // created tag stack object
    TagVector* _TagVect;              // returned from _TagStack object
    TagErrorTracker* _TagError;       // created tag error object
    NodeNestingData* _Nodep;          // created node nesting data object

    FilePosition* _FilePosRec;        // passed from _LineBuf

    int CountUnorderedTags();
    void SetTextBuffer();
    void InitLineBuffer();

    bool PosAdvCntAtMax() const;
    TagTypeInfo* GiveReadTagEntry(bool& NoContent_);
    TagLineInfo* CreateFromStackNodeInfo(StackNodeInfo* Ptr_);

    void WalkTagStack();
    void CreateCoords(int sz=0);
    void GrowCoords(int sz=0);

    TagCoordSpecifier* MakeTagCoordCopy(TagCoordSpecifier* TCoord_, bool Swap_);
    TagCoordSpecifier* RestoreTagCoordCopy(TagCoordSpecifier* TCoord_);

    TagCoordSpecifier* PopChildStack();
    TagVector* SeekPos(bool Create_, bool NewPos_,
                       TagCoordSpecifier* NewCopy_);
    inline TagVector* SeekNewPos(bool Create_, TagCoordSpecifier* NewCopy_)
        { return SeekPos(Create_, true, NewCopy_); }
    inline TagVector* SeekCurrentPos(bool Create_, TagCoordSpecifier* NewCopy_)
        { return SeekPos(Create_, false, NewCopy_); }

    bool HasChildTags(StackNodeInfo* Ptr_);
    bool IsParentTag(StackNodeInfo* Ptr_, NodeNestingData* NestNode_);
    bool IsEnclosedTag(StackNodeInfo* Ptr_,
                       NodeNestingData* NestNode_, int pnc);

    inline bool ChildVectorCreated()
        { return (_TPData &&
                  _TPData->ChildrenCreated() && _TPData->ChildVisited()); }
    inline TagStack* GiveNextTagStack()
        { return ((_TCoord && _TCoord->GiveCoordData()) ?
                      _TCoord->GiveCoordData()->Next():
                      NULL); }
    inline TagStack* GiveRootTagStack()
        { return _RootStack; }

    bool SetNestedTags(StackNodeInfo* Ptr_, int psn,
                       int pnc, int nc, int nlvl,
                       int& snl, int& chdlvl,
                       bool& SnlSet_, bool& ChdlvlSet_);

    bool HasTagLinkForward(StackNodeInfo* Ptr_, NodeNestingData* NestNode_);
    void SetTagLinkForward(StackNodeInfo* Ptr_, NodeNestingData* NestNode_,
                           bool ChildCreated_, int snl, int chdlvl,
                           bool SnlSet_, bool ChdlvlSet_);

    NodeNestingData* FindParentSequenceNumber(StackNodeInfo* Ptr_, int& psn);
    int FindNextSequenceNumber(StackNodeInfo* Ptr_);
    void FindNumChildren(StackNodeInfo* Ptr_, NodeNestingData* NestNode_,
                         int& pnc, int& nc);
    int FindCurrentLevel(StackNodeInfo* Ptr_, int& clvl, int& chdlvl);
    void FindChildLevel(StackNodeInfo* Ptr_, int& chdlvl);
    int FindSequenceNumber(StackNodeInfo* Ptr_);
    int FindEncounteredLevel(StackNodeInfo* Ptr_);

  public:
    TagReader(int DocType_=0);
    ~TagReader();

    virtual TagParseInfo& AdvanceReadPos();
    virtual TagParseInfo& ResetReadPos();

    TagReader& SetInputFile(const ChrString& Fname_);
    TagReader& ReadNextTag();
    TagReader& ReadAllTags();

    FilePosition* GiveFilePosition();
    NodeNestingData* GiveNodeNestingData();

    TagReader& ParseTextStr();
    TagReader& ParseDocument();
    TagReader& ParseDocumentByParts();

    TagVector* SeekRootPos(bool Create_);

    inline void CommitNewPos()
        { _TCoord->CommitNewPos(); }
    inline bool AtNewPos() const
        { return _AtNewPos; }
    inline bool AtCurrentPos() const
        { return _AtCurrentPos; }
    inline bool AtRootPos() const
        { return (_RootStack == _TagStack); }
    inline long CoordsSize() const
        { return _CoordsSize; }
    inline long MaxNestLevel() const
        { return _MaxNestLevel; }

    inline void SetTagCreated(long snl, long sn)
        { _TPData->CreateTag(snl, sn); }

    // new
    inline TagParseData* GiveTagParseData()
        { return _TPData; }
    inline TagCoordSpecifier* GiveTagCoordSpecifier()
        { return _TCoord; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
#endif

