#ifndef TAGSTKNODE_CPP
#define TAGSTKNODE_CPP
#ifndef TAGSTKNODE_H
  #include "tagstknode.h"
#endif

/****************************************************************************/
// TagTypeInfo class static data members
/****************************************************************************/
size_t TagTypeInfo::m_ObjectCnt = 0;
bool TagTypeInfo::_ShowDebugDump = false;
bool TagTypeInfo::_ShowErasedNodes = false;
bool TagTypeInfo::_ShowNewLines = false;

/****************************************************************************/
// TagCommentData class static members
/****************************************************************************/
bool TagCommentData::_ShowCommentDataDump = false;

/****************************************************************************/
// KeyValuePairData class static members
/****************************************************************************/
bool KeyValuePairData::_ShowKeyValuePairDump = false;

/****************************************************************************/
// QListNode class static data members
/****************************************************************************/
size_t QListNode::_StackSize = 0;
size_t QListNode::_NumErased = 0;

/****************************************************************************/
// TagSearchResult class definitions
/****************************************************************************/
TagSearchResult::TagSearchResult(TagTypeInfo* Parent_):
_Parent(Parent_),
_Xpos(0),
_Ypos(0),

_pTextBuffer(NULL),
_pTagTerminator(NULL),
_pKVDelim(NULL),
_pWspace(NULL),
_TagTermIndex(0),

_ReadPos(0),
_PrevReadPos(0),
_SearchDone(false),
_EofFound(false),

_TagBracketIndex(0),
_Index(0),
_Length(0),
_PrevReturnValue(0),
_ReturnValue(0),

_BrkIndex(0),
_BrkLength(0),
_MatchIndex(0),
_MatchLength(0),
_AtStartBrk(true),

_ElementIndex(0),
_ElementLength(0),

_DelimsArray(NULL),
_NumDelims(0),
_MaxDelims(0),

_AtKeyStart(false),
_ValidDelimSet(false),

_StartBrkFound(false),
_EndBrkFound(false),
_TagElementFound(false),
_ContentType(0)
{
  InitContentData();
  GrowVector();
}

/****************************************************************************/
TagSearchResult::TagSearchResult(const TagSearchResult& Obj_):
_Parent(Obj_._Parent),
_Xpos(Obj_._Xpos),
_Ypos(Obj_._Ypos),

_pTextBuffer(Obj_._pTextBuffer),
_pTagTerminator(Obj_._pTagTerminator),
_pKVDelim(Obj_._pKVDelim),
_pWspace(Obj_._pWspace),
_TagTermIndex(Obj_._TagTermIndex),

_ReadPos(Obj_._ReadPos),
_PrevReadPos(Obj_._PrevReadPos),
_SearchDone(Obj_._SearchDone),
_EofFound(Obj_._EofFound),

_TagBracketIndex(Obj_._TagBracketIndex),
_Index(Obj_._Index),
_Length(Obj_._Length),
_PrevReturnValue(Obj_._PrevReturnValue),
_ReturnValue(Obj_._ReturnValue),

_BrkIndex(Obj_._BrkIndex),
_BrkLength(Obj_._BrkLength),
_MatchIndex(Obj_._MatchIndex),
_MatchLength(Obj_._MatchLength),
_AtStartBrk(Obj_._AtStartBrk),

_ElementIndex(Obj_._ElementIndex),
_ElementLength(Obj_._ElementLength),

_DelimsArray(NULL),
_NumDelims(Obj_._NumDelims),
_MaxDelims(Obj_._MaxDelims),

_AtKeyStart(Obj_._AtKeyStart),
_ValidDelimSet(Obj_._ValidDelimSet),

_StartBrkFound(Obj_._StartBrkFound),
_EndBrkFound(Obj_._EndBrkFound),
_TagElementFound(Obj_._TagElementFound)
{
  Subscript max = Obj_._MaxDelims;
  InitContentData(&Obj_);

  if (max)
  {
    max *= sizeof(TagDelimiterSet);
    TagDelimiterSet* DelimSet_ = (TagDelimiterSet*)RawAllocateWith(MEMMATRIX, max);
    _DelimsArray = (TagDelimiterSet*)::memmove(DelimSet_, Obj_._DelimsArray, max);
  }
}

/****************************************************************************/
TagSearchResult::~TagSearchResult()
{
  _pTextBuffer = NULL;
  _pTagTerminator = NULL;
  _pKVDelim = NULL;
  _pWspace = NULL;
  
  RawDeleteArray(_DelimsArray);
  _DelimsArray = NULL;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::operator = (const TagSearchResult& Obj_)
{
  if (this != &Obj_)
  {
    _Xpos = Obj_._Xpos;
    _Ypos = Obj_._Ypos;

    _pTextBuffer = Obj_._pTextBuffer;
    _pTagTerminator = Obj_._pTagTerminator;
    _pKVDelim = Obj_._pKVDelim;
    _pWspace = Obj_._pWspace;
    _TagTermIndex = Obj_._TagTermIndex;

    _ReadPos = Obj_._ReadPos;
    _PrevReadPos = Obj_._PrevReadPos;
    _SearchDone = Obj_._SearchDone;
    _EofFound = Obj_._EofFound;

    _TagBracketIndex = Obj_._TagBracketIndex;
    _Index = Obj_._Index;
    _Length = Obj_._Length;
    _PrevReturnValue = Obj_._PrevReturnValue;
    _ReturnValue = Obj_._ReturnValue;

    _BrkIndex = Obj_._BrkIndex;
    _BrkLength = Obj_._BrkLength;
    _MatchIndex = Obj_._MatchIndex;
    _MatchLength = Obj_._MatchLength;
    _AtStartBrk = Obj_._AtStartBrk;
    _ElementIndex = Obj_._ElementIndex;
    _ElementLength = Obj_._ElementLength;

    _AtKeyStart = Obj_._AtKeyStart;
    _ValidDelimSet = Obj_._ValidDelimSet;

    _StartBrkFound = Obj_._StartBrkFound;
    _EndBrkFound = Obj_._EndBrkFound;
    _TagElementFound = Obj_._TagElementFound;

    InitContentData(&Obj_);

    Subscript max = Obj_._MaxDelims;
    Subscript cmax = _MaxDelims * sizeof(TagDelimiterSet);
    TagDelimiterSet* DelimSet_ = _DelimsArray;

    if (max)
    {
      max *= sizeof(TagDelimiterSet);
      _NumDelims = 0;

      if (_DelimsArray)
      {
        if (_MaxDelims < Obj_._MaxDelims)
          DelimSet_ = GrowVector(Obj_._MaxDelims - _MaxDelims);
        else if (cmax)
          _DelimsArray = (TagDelimiterSet*)::memset(DelimSet_, 0, cmax);

        if (Obj_._DelimsArray)
        {
          _DelimsArray = (TagDelimiterSet*)::memmove(DelimSet_, Obj_._DelimsArray, max);
          _NumDelims = Obj_._NumDelims;
        }
      }
      else
      {
        DelimSet_ = (TagDelimiterSet*)RawAllocateWith(MEMMATRIX, max);
        _MaxDelims = Obj_._MaxDelims;

        if (Obj_._DelimsArray)
        {
          _DelimsArray = (TagDelimiterSet*)::memmove(DelimSet_, Obj_._DelimsArray, max);
          _NumDelims = Obj_._NumDelims;
        }
        else
        {
          cmax = _MaxDelims;
          cmax *= sizeof(TagDelimiterSet);
          
          _DelimsArray = (TagDelimiterSet*)::memset(DelimSet_, 0, cmax);
          _NumDelims = 0;
        }
      }
    }
    else if (cmax)
    {
      _NumDelims = 0;

      if (!_DelimsArray)
        DelimSet_ = (TagDelimiterSet*)RawAllocateWith(MEMMATRIX, cmax);

      _DelimsArray = (TagDelimiterSet*)::memset(DelimSet_, 0, cmax);
    }
  }

  return *this;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::SetParent(TagTypeInfo* Ptr_)
{
  _Parent = Ptr_;
  return *this;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::SetFileXYpos(long Xpos_, long Ypos_)
{
  _Xpos = Xpos_;
  _Ypos = Ypos_;
  return *this;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::ResetFileXYpos()
{
  _Xpos = 0;
  _Ypos = 0;
  return *this;
}

/****************************************************************************/
void TagSearchResult::InitContentData(const TagSearchResult* Ptr_)
{
  if (Ptr_ && Ptr_ != this)
  {
    _ContentType = Ptr_->_ContentType;

    _Content._StartPos = Ptr_->_Content._StartPos;
    _Content._EndPos = Ptr_->_Content._EndPos;
    _Content._Str = Ptr_->_Content._Str;

    _IntContent._StartPos = Ptr_->_IntContent._StartPos;
    _IntContent._EndPos = Ptr_->_IntContent._EndPos;
    _IntContent._Str = Ptr_->_IntContent._Str;
  }
  else
  {
    _ContentType = NO_CONTENT;

    _Content._StartPos = 0;
    _Content._EndPos = 0;
    _Content._Str.SetEmpty();

    _IntContent._StartPos = 0;
    _IntContent._EndPos = 0;
    _IntContent._Str.SetEmpty();
  }
}

/****************************************************************************/
TagSearchResult::TagDelimiterSet* TagSearchResult::GrowVector(long Incr_)
{
  Subscript max = 0;
  TagDelimiterSet* DelimSet_ = NULL;

  if (!_DelimsArray || !_MaxDelims)
  {
    max = Incr_;
    max *= sizeof(TagDelimiterSet);
    DelimSet_ = (TagDelimiterSet*)RawAllocateWith(MEMMATRIX, max);
    _DelimsArray = DelimSet_;
    _MaxDelims = Incr_;
  }
  else
  {
    max = _MaxDelims + Incr_;
    max *= sizeof(TagDelimiterSet);
    DelimSet_ = _DelimsArray;
    _DelimsArray = (TagDelimiterSet*)RawAllocateWith(MEMMATRIX, max);
    max = _MaxDelims * sizeof(TagDelimiterSet);
    ::memmove(_DelimsArray, DelimSet_, max);
    _MaxDelims += Incr_;
    RawDeleteArray(DelimSet_);
  }

  return _DelimsArray;
}

/****************************************************************************/
const char* TagSearchResult::TagContentTypeToStr(int Type_)
{
  return
  (
    (Type_ == NO_CONTENT)              ? "NO_CONTENT":
    (Type_ == TAG_ENCL_CONTENT)        ? "TAG_ENCL_CONTENT":
    (Type_ == INTRA_TAG_CONTENT)       ? "INTRA_TAG_CONTENT":
    (Type_ == INTRA_NESTEDTAG_CONTENT) ? "INTRA_NESTEDTAG_CONTENT":""
  );
}

/****************************************************************************/
Fallible<Subscript> TagSearchResult::find(const ChrString& s)
{
  _SearchDone = false;

  if (_pTextBuffer)
  {
    // Initial find if successful will be for opening tag bracket
    // and assigned a ReturnValue of 1. Subsequent searches if successful
    // will be indicated by an assignment of the enumerated types:
    //   MATCH_TAGBRACKET or MATCH_TAGENDBRACKET or MATCH_TAGTYPE
    //   into the ReturnValue.
    ChrString SearchStr_ = _pTextBuffer->subString(_ReadPos, _pTextBuffer->strlen() - _ReadPos);
    SearchStr_.SetCaseSensitive(_pTextBuffer->CaseSensitive());
    _SrchFall = SearchStr_.find(s);
    _SearchDone = true;

    if (_ReturnValue < 0)
    {
      if (-_ReturnValue == MATCH_TAGBRACKET)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGBRACKET:0;
      else if (-_ReturnValue == MATCH_TAGENDBRACKET)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGENDBRACKET:0;
      else if (-_ReturnValue == MATCH_TAGTYPE)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGTYPE:0;

      if (_ReturnValue == MATCH_TAGBRACKET)
        _StartBrkFound = true;
      else if (_ReturnValue == MATCH_TAGENDBRACKET)
        _EndBrkFound = true;
      else if (_ReturnValue == MATCH_TAGTYPE)
        _TagElementFound = true;
    }
    else
      _ReturnValue = _SrchFall.valid() ? 1:0;

    _Index = ((_ReturnValue > 0) ? _SrchFall:
              (_ReturnValue < 0) ? _Index:0);

    if (_ReturnValue > 0)
      _Length = s.strlen();
  }

  return _SrchFall;
}

/****************************************************************************/
Fallible<Subscript> TagSearchResult::findlast(const ChrString& s)
{
  _SearchDone = false;

  if (_pTextBuffer)
  {
    ChrString SearchStr_ = _pTextBuffer->subString(_ReadPos, _pTextBuffer->strlen() - _ReadPos);
    SearchStr_.SetCaseSensitive(_pTextBuffer->CaseSensitive());
    _SrchFall = SearchStr_.findlast(s);
    _SearchDone = true;

    if (_ReturnValue < 0)
    {
      if (-_ReturnValue == MATCH_TAGBRACKET)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGBRACKET:0;
      else if (-_ReturnValue == MATCH_TAGENDBRACKET)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGENDBRACKET:0;
      else if (-_ReturnValue == MATCH_TAGTYPE)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGTYPE:0;

      if (_ReturnValue == MATCH_TAGBRACKET)
        _StartBrkFound = true;
      else if (_ReturnValue == MATCH_TAGENDBRACKET)
        _EndBrkFound = true;
      else if (_ReturnValue == MATCH_TAGTYPE)
        _TagElementFound = true;
    }
    else
      _ReturnValue = _SrchFall.valid() ? 1:0;

    _Index = ((_ReturnValue > 0) ? _SrchFall:
              (_ReturnValue < 0) ? _Index:0);

    if (_ReturnValue > 0)
      _Length = s.strlen();
  }

  return _SrchFall;
}

/****************************************************************************/
Fallible<Subscript> TagSearchResult::findnext(Subscript pos, const ChrString& s)
{
  _SearchDone = false;

  if (_pTextBuffer)
  {
    ChrString SearchStr_ = _pTextBuffer->subString(_ReadPos, _pTextBuffer->strlen() - _ReadPos);
    SearchStr_.SetCaseSensitive(_pTextBuffer->CaseSensitive());
    _SrchFall = SearchStr_.findnext(pos, s);
    _SearchDone = true;

    if (_ReturnValue < 0)
    {
      if (-_ReturnValue == MATCH_TAGBRACKET)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGBRACKET:0;
      else if (-_ReturnValue == MATCH_TAGENDBRACKET)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGENDBRACKET:0;
      else if (-_ReturnValue == MATCH_TAGTYPE)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGTYPE:0;

      if (_ReturnValue == MATCH_TAGBRACKET)
        _StartBrkFound = true;
      else if (_ReturnValue == MATCH_TAGENDBRACKET)
        _EndBrkFound = true;
      else if (_ReturnValue == MATCH_TAGTYPE)
        _TagElementFound = true;
    }
    else
      _ReturnValue = _SrchFall.valid() ? 1:0;

    _Index = ((_ReturnValue > 0) ? _SrchFall:
              (_ReturnValue < 0) ? _Index:0);

    if (_ReturnValue > 0)
      _Length = s.strlen();
  }

  return _SrchFall;
}

/****************************************************************************/
Fallible<Subscript> TagSearchResult::findprevious(Subscript pos, const ChrString& s)
{
  _SearchDone = false;

  if (_pTextBuffer)
  {
    ChrString SearchStr_ = _pTextBuffer->subString(_ReadPos, _pTextBuffer->strlen() - _ReadPos);
    SearchStr_.SetCaseSensitive(_pTextBuffer->CaseSensitive());
    _SrchFall = SearchStr_.findprevious(pos, s);
    _SearchDone = true;

    if (_ReturnValue < 0)
    {
      if (-_ReturnValue == MATCH_TAGBRACKET)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGBRACKET:0;
      else if (-_ReturnValue == MATCH_TAGENDBRACKET)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGENDBRACKET:0;
      else if (-_ReturnValue == MATCH_TAGTYPE)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGTYPE:0;

      if (_ReturnValue == MATCH_TAGBRACKET)
        _StartBrkFound = true;
      else if (_ReturnValue == MATCH_TAGENDBRACKET)
        _EndBrkFound = true;
      else if (_ReturnValue == MATCH_TAGTYPE)
        _TagElementFound = true;
    }
    else
      _ReturnValue = _SrchFall.valid() ? 1:0;

    _Index = ((_ReturnValue > 0) ? _SrchFall:
              (_ReturnValue < 0) ? _Index:0);

    if (_ReturnValue > 0)
      _Length = s.strlen();
  }

  return _SrchFall;
}

/****************************************************************************/
Subscript TagSearchResult::FindDelims(Subscript StartPos_, Subscript EndPos_,
                                      long BrkIndex_)
{
  // Sample html code segment:
  //
  // <meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">
  // <meta name="author" content="Joseph Wong">
  // <link rel="Prev"
  //       title="Cybernadian Computer Services" type="text/html"
  //       href="http://www.cybernadian.net/index.html">
  //

  // Initialize delimiter found data members
  _NumDelims = 0;
  _ValidDelimSet = false;
  _TagBracketIndex = BrkIndex_;

  // Skip nontagcontent type tags
  if (BrkIndex_ == TagEnums::COMMENT_DEX ||
      BrkIndex_ == TagEnums::SCRIPT_DEX ||
      BrkIndex_ == TagEnums::CCOMMENTBLK_DEX ||
      BrkIndex_ == TagEnums::COPENCOMMENTBLK_DEX ||
      BrkIndex_ == TagEnums::CCLOSECOMMENTBLK_DEX ||
      BrkIndex_ == TagEnums::CPPCOMMENTLINEOPEN_DEX)
  {
    SetElementData(0, 0, false);
    return 0;
  }

  // Find single white space separated contents instead of KV pairs
  // Applies to doctype tags and content within nested comment tags
  if (BrkIndex_ == TagEnums::DOCTYPE_DEX ||
      BrkIndex_ == TagEnums::END_DEX ||
      BrkIndex_ == TagEnums::NESTEDCOMMENT1_DEX ||
      BrkIndex_ == TagEnums::NESTEDCOMMENT2_DEX)
  {
    // No delimiters for end tag
    if (BrkIndex_ == TagEnums::END_DEX)
      return 0;
    else if (BrkIndex_ != TagEnums::DOCTYPE_DEX)
      SetElementData(0, 0, false);

    return FindSingleDelims(StartPos_, EndPos_);
  }

  // Initialize local variables
  size_t x, max;
  size_t quotcnt = 0;
  char c = 0;
  char NextDelim_ = _pKVDelim ? (*_pKVDelim)[0]:'=';
  char NextWs_ = _pWspace ? (*_pWspace)[0]:' ';
  bool EndCond_ = false;
  bool TabTried_ = false;
  Subscript v;
  TagDelimiterSet* dptr = NULL;

  // save previous tag search results
  Fallible<Subscript> SrchSaved_ = _SrchFall;
  int SavedRet_ = _ReturnValue;
  Subscript SavedIndex_ = _Index;
  Subscript SavedLength_ = _Length;

  // Get search substring extracted from current read position in text buffer
  ChrString SearchStr_ = _pTextBuffer->subString(_ReadPos, _pTextBuffer->strlen() - _ReadPos);
  SearchStr_.SetCaseSensitive(_pTextBuffer->CaseSensitive());
  --StartPos_;

  for (x = quotcnt = 0; _pTextBuffer &&
       !EndConditionReached(SearchStr_, StartPos_, EndPos_); x++)
  {
    if (!TabTried_)
    {
      c = _AtKeyStart ? (((IsKVdelim(c) && quotcnt == 0) ||
                          (c == '\"' && quotcnt == 1)) ? '\"':NextWs_):NextDelim_;

      if (c == '\"')
        ++quotcnt;
      else if (_AtKeyStart && IsWspace(c))
        quotcnt = 0;
    }

    if (x == 0 && StartPos_ < 0)
    {
      v = (SearchStr_[0] == c) ? 0:99;
      _SrchFall = 0;
      
      if (v == 99)
        _SrchFall.invalidate();
    }
    else
    {
      if (!IsKVdelim(c) && *SearchStr_(StartPos_) == c)
        _SrchFall = Fallible<Subscript>(StartPos_, NULL);
      else
        _SrchFall = SearchStr_.findnext(StartPos_, c);
    }

    if (_ReturnValue < 0)
    {
      if (-_ReturnValue == MATCH_TAGDELIMS)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGDELIMS:0;
    }

    _Index = ((_ReturnValue > 0) ? _SrchFall:
              (_ReturnValue < 0) ? _Index:0);

    if (_ReturnValue > 0)
    {
      _Length = 1;
      StartPos_ = _Index;

      if (!EndConditionReached(SearchStr_, StartPos_, EndPos_))
      {
        if (quotcnt > 0)
        {
          ++StartPos_;
          _ReturnValue = -_ReturnValue;
        }
        else
        {
          dptr = &_DelimsArray[_NumDelims];
          dptr->_DelimChar = c;
          dptr->_Index = StartPos_;
          _NumDelims++;

          NextDelim_ = _pKVDelim ? (*_pKVDelim)[0]:'=';
          NextWs_ = _pWspace ? (*_pWspace)[0]:' ';

          if (_NumDelims >= _MaxDelims)
            GrowVector();

          if (_AtKeyStart && TabTried_)
            TabTried_ = false;

          // skip whitespace
          max = SearchStr_.strlen();
          for (++StartPos_;
               !(EndCond_=EndConditionReached(SearchStr_, StartPos_, EndPos_)) &&
               StartPos_ < max && isspace(*SearchStr_(StartPos_));
               ++StartPos_);

          if (IsWspace(c) && !EndCond_ &&
              StartPos_ < max && !isspace(*SearchStr_(StartPos_)))
          {
            dptr->_DelimChar = c;
            dptr->_Index = StartPos_-1;
          }
             
          _AtKeyStart = !_AtKeyStart;
          _ReturnValue = -_ReturnValue;
        }
      }
      else
        _ReturnValue = 0;
    }
    else if (!_ReturnValue)
    {
      if (_pKVDelim && IsKVdelim(c))
      {
        NextDelim_ = NextKVdelim(c);
        --x;
      }
      else if (_AtKeyStart && quotcnt == 0 &&
                  ((!TabTried_ && c == ' ') || (_pWspace && IsWspace(c))))
      {
        c = NextWspace(c);
        TabTried_ = true;

        if (!c)
          break;
        else
          _ReturnValue = SavedRet_;
      }
      else
        break;
    }
  }

  if (_pTextBuffer)
  {
    if (_NumDelims)
    {
      // make sure last delimiter found (should be '=') is nearer than
      // the ending tag brk in the searched string
      dptr = &_DelimsArray[_NumDelims-1];
      _ValidDelimSet = _NumDelims >= 2 &&
                           (dptr->_Index < EndPos_ ||
                           (dptr->_Index < StartPos_ && !EndPos_));
    }
    else
      _ValidDelimSet = false;

    // No odd number of: ' ', '=' delimiter chars allowed
    if (_ValidDelimSet && _NumDelims % 2)
    {
      --_NumDelims;

      dptr = &_DelimsArray[_NumDelims];
      dptr->_DelimChar = 0;
      dptr->_Index = 0;
    }

    // restore tag search results data
    _SrchFall = SrchSaved_;
    _ReturnValue = SavedRet_;
    _Index = SavedIndex_;
    _Length = SavedLength_;
  }
  else
    _ValidDelimSet = false;

  // returns the number of delimiters found in the searched tag string
  Subscript SrchRes_ = _ValidDelimSet ? _NumDelims:0;
  return SrchRes_;
}

/****************************************************************************/
Subscript TagSearchResult::FindSingleDelims(Subscript StartPos_, Subscript EndPos_)
{
  // Sample html code segment:
  //
  // <meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">
  // <meta name="author" content="Joseph Wong">
  // <link rel="Prev"
  //       title="Cybernadian Computer Services" type="text/html"
  //       href="http://www.cybernadian.net/index.html">
  //

  // Initialize delimiter found data members
  _NumDelims = 0;
  _ValidDelimSet = false;

  // Initialize local variables
  size_t x, max, len;
  char c = ' ';
  bool EndCond_ = false;
  bool Eos_ = false;
  TagDelimiterSet* dptr = NULL;

  // save previous tag search results
  Fallible<Subscript> SrchSaved_ = _SrchFall;
  int SavedRet_ = _ReturnValue;
  Subscript SavedIndex_ = _Index;
  Subscript SavedLength_ = _Length;

  size_t Next_ = size_t(StartPos_);
  size_t Start_ = size_t(StartPos_);

  // Get search substring extracted from current read position in text buffer
  ChrString SearchStr_ = _pTextBuffer->subString(_ReadPos, _pTextBuffer->strlen() - _ReadPos);
  SearchStr_.SetCaseSensitive(_pTextBuffer->CaseSensitive());
  --StartPos_;

  for (x = 0; _pTextBuffer &&
       !EndConditionReached(SearchStr_, StartPos_, EndPos_); x++)
  {
    len = FindToken(SearchStr_, Start_, Next_, Eos_);
    StartPos_ = Subscript(Start_);

    if (len)
      _SrchFall = StartPos_;
    else
      _SrchFall = Fallible<Subscript>(StartPos_, NULL);

    if (_ReturnValue < 0)
    {
      if (-_ReturnValue == MATCH_TAGDELIMS)
        _ReturnValue = _SrchFall.valid() ? MATCH_TAGDELIMS:0;
    }

    _Index = ((_ReturnValue > 0) ? _SrchFall:
              (_ReturnValue < 0) ? _Index:0);

    if (_ReturnValue > 0)
    {
      _Length = 1;

      if (!EndConditionReached(SearchStr_, StartPos_, EndPos_))
      {
        dptr = &_DelimsArray[_NumDelims];

        if (_NumDelims > 0)
        {
          dptr->_DelimChar = c;
          for (;!isspace(*SearchStr_(StartPos_)); StartPos_--);
          dptr->_Index = StartPos_;
        }

        _NumDelims++;
        if (_NumDelims >= _MaxDelims)
          GrowVector();

        // skip whitespace
        max = SearchStr_.strlen();
        for (++StartPos_;
             !(EndCond_=EndConditionReached(SearchStr_, StartPos_, EndPos_)) &&
             StartPos_ < max && isspace(*SearchStr_(StartPos_));
             ++StartPos_);

        if (!EndCond_ && StartPos_ < max &&
            !isspace(*SearchStr_(StartPos_)))
        {
          dptr->_DelimChar = c;
          dptr->_Index = StartPos_-1;
        }

        Start_ = Next_;
        _ReturnValue = -_ReturnValue;
      }
      else
        _ReturnValue = 0;
    }
    else if (!_ReturnValue)
      break;
  }

  if (_pTextBuffer)
  {
    if (_NumDelims)
      _ValidDelimSet = _NumDelims >= 1 &&
                           (dptr->_Index < EndPos_ ||
                           (dptr->_Index < StartPos_ && !EndPos_));
    else
      _ValidDelimSet = false;

    // restore tag search results data
    _SrchFall = SrchSaved_;
    _ReturnValue = SavedRet_;
    _Index = SavedIndex_;
    _Length = SavedLength_;
  }
  else
    _ValidDelimSet = false;

  // returns the number of delimiters found in the searched tag string
  Subscript SrchRes_ = _ValidDelimSet ? _NumDelims:0;
  return SrchRes_;
}

/****************************************************************************/
Subscript TagSearchResult::FindIntraTagBrks(int TagIndex_, bool& Found_,
                                       long* StartPosx_, long* EndPosx_,
                                       ChrString* StartBrkStr_, ChrString* EndBrkStr_)
{
  if (!_StartBrkFound || !_EndBrkFound)
    return false;

  Subscript pos;
  Subscript BrkLen_ = BracketLength();
  Subscript EndLen_ = MatchingLength();

  Subscript StartPos_;
  Subscript EndPos_;

  ChrString StartStr_;
  ChrString EndStr_;
  ChrString IntStartStr_;
  ChrString IntEndStr_;

  bool StartFnd_ = false;
  bool EndFnd_ = false;
  bool RunSearch_ = false;
  bool IntHtmlCmnt_ = false;

  if (TagIndex_ == TagEnums::COMMENT_DEX ||
      TagIndex_ == TagEnums::NESTEDCOMMENT1_DEX ||
      TagIndex_ == TagEnums::NESTEDCOMMENT2_DEX)
  {
    StartStr_ = NESTEDCOMMENT1CLOSE_STR;
    EndStr_ = NESTEDCOMMENT2OPEN_STR;
    IntStartStr_ = INTNESTEDCOMMENTEND_STR;
    IntEndStr_ = INTNESTEDCOMMENTSTART_STR;

    RunSearch_ = true;
    IntHtmlCmnt_ = true;

    if (StartBrkStr_)
      *StartBrkStr_ = StartStr_;

    if (EndBrkStr_)
      *EndBrkStr_ = EndStr_;
  }
  else if (TagIndex_ == TagEnums::CCOMMENTBLK_DEX ||
           TagIndex_ == TagEnums::COPENCOMMENTBLK_DEX ||
           TagIndex_ == TagEnums::CCLOSECOMMENTBLK_DEX)
  {
    StartStr_ = COPENCOMMENTBLKCLOSE_STR;
    EndStr_ = CCLOSECOMMENTBLKOPEN_STR;
    RunSearch_ = true;

    if (StartBrkStr_)
      *StartBrkStr_ = StartStr_;

    if (EndBrkStr_)
      *EndBrkStr_ = EndStr_;
  }

  if (RunSearch_)
  {
    Fallible<Subscript> SrchRes_;
    ChrString TagStr_ = GiveTagString();
    ChrString SearchStr_ = TagStr_.subString(BrkLen_,
                                             TagStr_.strlen() - BrkLen_ - EndLen_);
    SearchStr_.SetCaseSensitive(_pTextBuffer->CaseSensitive());

    if (IntHtmlCmnt_)
      SrchRes_ = SearchStr_.find(IntStartStr_);
    else
      SrchRes_ = SearchStr_.find(StartStr_);

    StartPos_ = SrchRes_.valid() ? SrchRes_:0;

    if (SrchRes_.valid())
      StartFnd_ = true;

    if (IntHtmlCmnt_)
      ++StartPos_;

    pos = StartPos_ + StartStr_.strlen();

    if (IntHtmlCmnt_)
      SrchRes_ = SearchStr_.findnext(pos, IntEndStr_);
    else
      SrchRes_ = SearchStr_.findnext(pos, EndStr_);

    EndPos_ = SrchRes_.valid() ? SrchRes_:0;

    if (SrchRes_.valid())
      EndFnd_ = true;

    if (StartFnd_ && EndFnd_)
    {
      if (StartPosx_)
        *StartPosx_ = StartPos_;

      if (EndPosx_)
        *EndPosx_ = EndPos_;

      Found_ = true;
      return pos;
    }
  }

  Found_ = false;
  return 0;
}

/****************************************************************************/
size_t TagSearchResult::FindToken(ChrString& SearchStr_, size_t& Index_, size_t& NextPos_, bool& Eos_)
{
  size_t CopyLen_ = FindNextWord(SearchStr_.c_str(), " \t\n", Index_, NextPos_, "\"", true);

  if (CopyLen_)
  {
    if (Index_ && isspace(SearchStr_[Index_-1]))
      --Index_;
  }
  else
    Eos_ = SearchStr_[Index_] == 0 || Index_ >= SearchStr_.strlen();

  return CopyLen_;
}

/****************************************************************************/
bool TagSearchResult::IsWspace(char c)
{
  return ((_pWspace && c) ? InCharSet(c, _pWspace->c_str()):(c == ' '));
}

/****************************************************************************/
bool TagSearchResult::IsKVdelim(char c)
{
  return ((_pKVDelim && c) ? InCharSet(c, _pKVDelim->c_str()):(c == '='));
}

/****************************************************************************/
char TagSearchResult::NextWspace(char c)
{
  Subscript x = 0;
  Subscript len = _pWspace ? _pWspace->strlen():0;

  if (!len)
    return ((c == ' ') ? '\t':' ');

  for (x = 0; x < len && (*_pWspace)[x] != c; x++);
  return ((x < len) ? (*_pWspace)[x+1]:0);
}

/****************************************************************************/
char TagSearchResult::NextKVdelim(char c)
{
  Subscript x = 0;
  Subscript len = _pKVDelim ? _pKVDelim->strlen():0;

  if (!len)
    return '=';

  for (x = 0; x < len && (*_pKVDelim)[x] != c; x++);
  return ((x < len) ? (*_pKVDelim)[x+1]:0);
}

/****************************************************************************/
bool TagSearchResult::EndConditionReached(const ChrString& SearchStr_, Subscript StartPos_, Subscript EndPos_) const
{
  int ret = 3;

  if (!EndPos_ && _pTagTerminator && _TagTermIndex)
  {
    int cnt = 0;
    const ChrString* TermSelected_ = NULL;

    for (cnt = 0; cnt < _TagTermIndex && _pTagTerminator[cnt]; cnt++)
    {
      TermSelected_ = _pTagTerminator[cnt];
      ChrString CurrTxtStr_(SearchStr_(StartPos_), TermSelected_->strlen());
      CurrTxtStr_.SetCaseSensitive(TermSelected_->CaseSensitive());

      if (CurrTxtStr_ == *TermSelected_)
        return true;
      else
        ret = 0;
    }
  }

  return ((ret > 0) ? (StartPos_ >= EndPos_):false);
}

/****************************************************************************/
bool TagSearchResult::RetrieveIntraContents(bool PreserveNL_,
                                            const ChrString* StartStr_,
                                            const ChrString* EndStr_)
{
  if (!_StartBrkFound || !_EndBrkFound)
    return false;

  Subscript pos;
  Subscript RemChr_ = 0;
  Subscript BrkIndex_ = BracketIndex();
  Subscript EndIndex_ = MatchingIndex();
  Subscript BrkLen_ = BracketLength();
  Subscript EndLen_ = MatchingLength();
  Subscript StartPos_ = 0;
  Subscript EndPos_ = 0;
  ChrString TagStr_;

  bool ret = false;
  bool StartFnd_ = false;
  bool EndFnd_ = false;

  if (StartStr_ && EndStr_)
  {
    Fallible<Subscript> SrchRes_;
    TagStr_ = GiveTagString();
    ChrString SearchStr_ = TagStr_.subString(BrkLen_,
                                             TagStr_.strlen() - BrkLen_ - EndLen_);
    SearchStr_.SetCaseSensitive(_pTextBuffer->CaseSensitive());

    SrchRes_ = SearchStr_.find(*StartStr_);
    _IntContent._StartPos = SrchRes_.valid() ? SrchRes_:0;
    if (SrchRes_.valid())
      StartFnd_ = true;

    pos = _IntContent._StartPos + StartStr_->strlen();
    SrchRes_ = SearchStr_.findnext(pos, *EndStr_);
    _IntContent._EndPos = SrchRes_.valid() ? SrchRes_:0;
    if (SrchRes_.valid())
      EndFnd_ = true;

    if (StartFnd_ && EndFnd_ &&
        pos > 0 && _IntContent._EndPos > 0 &&
        _IntContent._EndPos > pos)
    {
      TagStr_ = _IntContent._Str = SearchStr_.subString(pos, _IntContent._EndPos - pos);
      StartPos_ = pos;
      EndPos_ = _IntContent._EndPos;

      RemChr_ = _IntContent._Str.RemoveTrailing(" \t\n\r\f\v");
      _IntContent._EndPos -= RemChr_;
      RemChr_ = _IntContent._Str.RemoveLeading(" \t\n\r\f\v");
      pos += RemChr_;
      ret = true;

      if (!_IntContent._Str.strlen() ||
          _IntContent._EndPos <= pos)
      {
        _IntContent._EndPos =
        _IntContent._StartPos = pos;
        _IntContent._Str.SetEmpty();
        return false;
      }
      else
      {
        if (PreserveNL_)
        {
          RemChr_ = TagStr_.RemoveTrailing(" \t");
          EndPos_ -= RemChr_;
          RemChr_ = TagStr_.RemoveLeading(" \t");
          StartPos_ += RemChr_;

          _IntContent._EndPos = EndPos_;
          pos = StartPos_;
          _IntContent._Str = TagStr_;
        }

        _IntContent._StartPos = BrkIndex_ + BrkLen_ + pos;
        _IntContent._EndPos += BrkIndex_ + BrkLen_;
      }
    }
    else
    {
      _IntContent._Str.SetEmpty();
      _IntContent._StartPos =
      _IntContent._EndPos = 0;
      return false;
    }

    if (ret)
      _ContentType |= INTRA_NESTEDTAG_CONTENT;
  }
  else
  {
    TagStr_ = c_str();
    pos = BrkIndex_ + BrkLen_;
    _IntContent._EndPos = EndIndex_;

    _IntContent._Str = TagStr_.subString(pos, _IntContent._EndPos - pos);
    TagStr_ = _IntContent._Str;
    StartPos_ = pos;
    EndPos_ = _IntContent._EndPos;

    RemChr_ = _IntContent._Str.RemoveTrailing(" \t\n\r\f\v");
    _IntContent._EndPos -= RemChr_;
    RemChr_ = _IntContent._Str.RemoveLeading(" \t\n\r\f\v");
    pos += RemChr_;
    ret = true;

    if (!_IntContent._Str.strlen() ||
        _IntContent._EndPos <= pos)
    {
      _IntContent._EndPos =
      _IntContent._StartPos = pos;
      return false;
    }
    else
    {
      if (PreserveNL_)
      {
        RemChr_ = TagStr_.RemoveTrailing(" \t");
        EndPos_ -= RemChr_;
        RemChr_ = TagStr_.RemoveLeading(" \t");
        StartPos_ += RemChr_;

        _IntContent._Str = TagStr_;
        _IntContent._EndPos = EndPos_;
        _IntContent._StartPos = StartPos_;
      }
      else
        _IntContent._StartPos = pos;
    }

    if (ret)
      _ContentType |= INTRA_TAG_CONTENT;
  }

  return ret;
}

/****************************************************************************/
bool TagSearchResult::RetrieveContents(bool PreserveNL_)
{
  Subscript pos = ReadPos();

  if (pos <= 0)
    return false;

  bool ret = false;
  Subscript RemChr_ = 0;
  Subscript SavedPos_ = pos;
  Subscript BrkIndex_ = BracketIndex();
  Subscript StartPos_;
  Subscript EndPos_;
  ChrString TagStr_;

  _Content._EndPos = _Content._StartPos = pos;
  _Content._EndPos += (BrkIndex_ > 0) ? BrkIndex_:0;

  if (_Content._EndPos == SavedPos_ && !_StartBrkFound)
    _Content._EndPos = _pTextBuffer->strlen();

  if (_Content._StartPos > 0 && _Content._EndPos > 0 &&
      _Content._EndPos > _Content._StartPos)
  {
    _Content._Str = _pTextBuffer->subString(pos, _Content._EndPos - pos);
    TagStr_ = _Content._Str;
    StartPos_ = _Content._StartPos;
    EndPos_ = _Content._EndPos;

    RemChr_ = _Content._Str.RemoveTrailing(" \t\n\r\f\v");
    _Content._EndPos -= RemChr_;
    RemChr_ = _Content._Str.RemoveLeading(" \t\n\r\f\v");
    _Content._StartPos += RemChr_;
    ret = true;

    if (!_Content._Str.strlen() ||
        _Content._EndPos <= _Content._StartPos)
    {
      _Content._EndPos = _Content._StartPos;
      _Content._Str.SetEmpty();
      return false;
    }
    else if (PreserveNL_)
    {
      RemChr_ = TagStr_.RemoveTrailing(" \t");
      EndPos_ -= RemChr_;
      RemChr_ = TagStr_.RemoveLeading(" \t");
      StartPos_ += RemChr_;

      _Content._Str = TagStr_;
      _Content._EndPos = EndPos_;
      _Content._StartPos = StartPos_;
    }

    if (ret)
      _ContentType |= TAG_ENCL_CONTENT;
  }

  return ret;
}

/****************************************************************************/
Subscript TagSearchResult::strlen() const
{
  return
    ((_pTextBuffer && _ReadPos < _pTextBuffer->strlen()) ?
       ((_ReadPos > 0) ? ::strlen((*_pTextBuffer)(_ReadPos)):
        ((_ReadPos == 0) ? _pTextBuffer->strlen():0)):0);
}

/****************************************************************************/
const char* TagSearchResult::c_str() const
{
  return
    ((_pTextBuffer && _ReadPos <= _pTextBuffer->strlen()) ?
       ((0 < _ReadPos && _ReadPos <  _pTextBuffer->strlen()) ?
            (*_pTextBuffer)(_ReadPos):
        ((_ReadPos == 0) ? _pTextBuffer->c_str():
         (_ReadPos == _pTextBuffer->strlen()) ?
            (_pTextBuffer->c_str() + _pTextBuffer->strlen()):NULL)):NULL);
}

/****************************************************************************/
const ChrString* TagSearchResult::txtbuf() const
{
  return _pTextBuffer;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::SwapStartEndBrks()
{
  bool TempFound_ = _StartBrkFound;
  Subscript TempIndex_ = _BrkIndex;
  Subscript TempLength_ = _BrkLength;

  _BrkIndex = _MatchIndex;
  _BrkLength = _MatchLength;

  _MatchIndex = TempIndex_;
  _MatchLength = TempLength_;
  _StartBrkFound = _EndBrkFound;
  _EndBrkFound = TempFound_;
  _AtStartBrk = !_AtStartBrk;

  return *this;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::SetSearchType(int Type_)
{
  // Setting return value to indicate a search for matching brackets or
  // matching tag types by assigning negative value of enumerated type
  //
  // Type_ == 0 : Search for starting tag
  // Type_ == 1 : Search for ending tag
  //
  if (Type_ > 0 &&
      (Type_ == MATCH_TAGBRACKET ||
       Type_ == MATCH_TAGENDBRACKET ||
       Type_ == MATCH_TAGTYPE ||
       Type_ == MATCH_TAGDELIMS))
  {
    // Initial setting of kvpair delimiters search
    if (Type_ == MATCH_TAGDELIMS && _ReturnValue != ::abs(Type_))
      _AtKeyStart = true;

    if (_ReturnValue >= 0)
      _PrevReturnValue = _ReturnValue;

    _ReturnValue = -Type_;
  }
  else if (Type_ == 0)
  {
    if (_ReturnValue >= 0)
      _PrevReturnValue = _ReturnValue;

    _ReturnValue = 0;
  }

  return *this;
}

/****************************************************************************/
const TagSearchResult& TagSearchResult::SetTextBufferPtr(const ChrString* Ptr_) const
{
  _pTextBuffer = Ptr_;
  if (_pTextBuffer)
    _pTextBuffer->SetCaseSensitive(FALSE);

  return *this;
}

/****************************************************************************/
const TagSearchResult& TagSearchResult::AddTagTerminatorPtr(const ChrString* Ptr_) const
{
  if (Ptr_)
  {
    _pTagTerminator[_TagTermIndex] = Ptr_;
    _pTagTerminator[_TagTermIndex]->SetCaseSensitive(FALSE);
    _TagTermIndex++;
  }

  return *this;
}

/****************************************************************************/
const TagSearchResult& TagSearchResult::SetTagTerminatorPtr(TAGTERMLISTTYPE Ptr_, int TagTermIndex_) const
{
  _pTagTerminator = Ptr_;
  _TagTermIndex = Ptr_ ? TagTermIndex_:0;

  if (_pTagTerminator)
  {
    int x;
    const ChrString* TermSelected_ = NULL;

    for (x = 0; x < _TagTermIndex && _pTagTerminator[x]; x++)
    {
      TermSelected_= _pTagTerminator[x];
      TermSelected_->SetCaseSensitive(FALSE);
    }
  }

  return *this;
}

/****************************************************************************/
const TagSearchResult& TagSearchResult::SetKVDelimPtr(const ChrString* Ptr_) const
{
  _pKVDelim = Ptr_;
  if (_pKVDelim)
    _pKVDelim->SetCaseSensitive(FALSE);

  return *this;
}

/****************************************************************************/
const TagSearchResult& TagSearchResult::SetWspacePtr(const ChrString* Ptr_) const
{
  _pWspace = Ptr_;
  if (_pWspace)
    _pWspace->SetCaseSensitive(FALSE);

  return *this;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::SetElementData(Subscript Index_, Subscript Length_, bool NoBlanks_)
{
  _ElementIndex = Index_;
  _ElementLength = Length_;
  Subscript sIndex_ = Index_ ? Index_:
                      _AtStartBrk ? (_BrkIndex + _BrkLength):
                                    (_MatchIndex + _MatchLength);

  const char* Ptr_ = c_str();
  _TagElementFound =
     Ptr_ && Length_ > 0 &&
     ((NoBlanks_ &&
        (isalpha(Ptr_[sIndex_]) ||
         Ptr_[sIndex_] == TagEnums::NESTEDCOMMENT_OPEN_CH ||
         Ptr_[sIndex_] == TagEnums::NESTEDCOMMENT_CLOSE_CH)) || !NoBlanks_);

  return *this;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::SetMatchingBrkData(Subscript Index_, Subscript Length_)
{
  _MatchIndex = Index_;
  _MatchLength = Length_;

  const char* Ptr_ = c_str();
  _EndBrkFound = Ptr_ && !isspace(Ptr_[_MatchIndex]) && Index_ + Length_ > 0;
  return *this;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::SetStartingBrkData(Subscript Index_, Subscript Length_)
{
  _BrkIndex = Index_;
  _BrkLength = Length_;
  _AtStartBrk = true;

  const char* Ptr_ = c_str();
  _StartBrkFound = Ptr_ && !isspace(Ptr_[_BrkIndex]) && Index_ + Length_ > 0;
  return *this;
}

/****************************************************************************/
int TagSearchResult::SearchReturn() const
{
  return ((_ReturnValue < 0) ? 0:_ReturnValue);
}

/****************************************************************************/
bool TagSearchResult::AtStartBrk() const
{
  return _AtStartBrk;
}

/****************************************************************************/
// At End Of Line function
//
bool TagSearchResult::AtEol() const
{
  if (_pTextBuffer && (0 <= _ReadPos && _ReadPos <= _pTextBuffer->strlen()))
  {
    if (_ReadPos == _pTextBuffer->strlen())
      return true;

    const char* EndPtr_ = (*_pTextBuffer)(_ReadPos);
    const char* StartPtr_ = (*_pTextBuffer)(_PrevReadPos);
    bool Eol_;

    for (Eol_ = false; !Eol_ && StartPtr_ < EndPtr_; StartPtr_++)
      Eol_ = (*StartPtr_ == '\n');

    return Eol_;
  }

  return false;
}

/****************************************************************************/
// At End Of Text Buffer function
//
bool TagSearchResult::AtEotb() const
{
  return (!_pTextBuffer || _pTextBuffer->strlen() - _ReadPos <= 0);
}

/****************************************************************************/
bool TagSearchResult::TargetFound() const
{
  return _SrchFall.valid();
}

/****************************************************************************/
Subscript TagSearchResult::SearchResultIndex() const
{
  return ((_ReturnValue <= 0) ?
              ((_PrevReturnValue > 0) ? _Index:0):
              _Index);
}

/****************************************************************************/
Subscript TagSearchResult::SearchResultLength() const
{
  return ((_ReturnValue <= 0) ?
              ((_PrevReturnValue > 0) ? _Length:0):
              _Length);
}

/****************************************************************************/
Subscript TagSearchResult::Xstart() const
{
  return (_StartBrkFound ? (_Xpos + _BrkIndex):0);
}

/****************************************************************************/
Subscript TagSearchResult::Xend() const
{
  return (_EndBrkFound ? (_Xpos + _MatchIndex + _MatchLength - 1):0);
}

/****************************************************************************/
Subscript TagSearchResult::BracketIndex() const
{
  return (_StartBrkFound ? _BrkIndex:0);
}

/****************************************************************************/
Subscript TagSearchResult::BracketLength() const
{
  return (_StartBrkFound ? _BrkLength:0);
}

/****************************************************************************/
Subscript TagSearchResult::MatchingIndex() const
{
  return (_EndBrkFound ? _MatchIndex:0);
}

/****************************************************************************/
Subscript TagSearchResult::MatchingLength() const
{
  return (_EndBrkFound ? _MatchLength:0);
}

/****************************************************************************/
Subscript TagSearchResult::ReadPos() const
{
  return _ReadPos;
}

/****************************************************************************/
Subscript TagSearchResult::ElementIndex() const
{
  return
  (
    _TagElementFound ?
        (_ElementIndex ? _ElementIndex:
         _AtStartBrk ? (_BrkIndex + _BrkLength):
                       (_MatchIndex + _MatchLength)):0
  );
}

/****************************************************************************/
Subscript TagSearchResult::ElementLength() const
{
  return (_TagElementFound ? _ElementLength:0);
}

/****************************************************************************/
Subscript TagSearchResult::TagContentStartPos() const
{
  return ((_ContentType & TAG_ENCL_CONTENT) ? _Content._StartPos:0);
}

/****************************************************************************/
Subscript TagSearchResult::TagContentEndPos() const
{
  return ((_ContentType & TAG_ENCL_CONTENT) ? _Content._EndPos:0);
}

/****************************************************************************/
Subscript TagSearchResult::TagContentLength() const
{
  Subscript sIndex_ = TagContentStartPos();
  sIndex_ = (TagContentEndPos() - sIndex_);
  return ((sIndex_ < 0) ? 0:sIndex_);
}

/****************************************************************************/
Subscript TagSearchResult::IntraTagContentStartPos() const
{
  return
  (
    ((_ContentType & INTRA_TAG_CONTENT) ||
     (_ContentType & INTRA_NESTEDTAG_CONTENT)) ? _IntContent._StartPos:0
  );
}

/****************************************************************************/
Subscript TagSearchResult::IntraTagContentEndPos() const
{
  return
  (
    ((_ContentType & INTRA_TAG_CONTENT) ||
     (_ContentType & INTRA_NESTEDTAG_CONTENT)) ? _IntContent._EndPos:0
  );
}

/****************************************************************************/
Subscript TagSearchResult::IntraTagContentLength() const
{
  Subscript sIndex_ = IntraTagContentStartPos();
  sIndex_ = (IntraTagContentEndPos() - sIndex_);
  return ((sIndex_ < 0) ? 0:sIndex_);
}

/****************************************************************************/
Subscript TagSearchResult::DelimPosAt(Subscript Index_, bool* Found_) const
{
  TagDelimiterSet* dptr =
  ((_ValidDelimSet && _NumDelims &&
    0 <= Index_ && Index_ < _NumDelims) ? &_DelimsArray[Index_]:NULL);

  if (Found_)
    *Found_ = dptr != NULL;
    
  return (dptr ? dptr->_Index:0);
}

/****************************************************************************/
char TagSearchResult::DelimCharAt(Subscript Index_, bool* Found_) const
{
  TagDelimiterSet* dptr =
  ((_ValidDelimSet && _NumDelims &&
    0 <= Index_ && Index_ < _NumDelims) ? &_DelimsArray[Index_]:NULL);

  if (Found_)
    *Found_ = dptr != NULL;

  return (dptr ? dptr->_DelimChar:0);
}

/****************************************************************************/
ChrString TagSearchResult::TagKeyAt(Subscript Index_, bool* Found_) const
{
  // Get search substring extracted from current read position in text buffer
  ChrString SearchStr_;
  SearchStr_.NullTerminate();
  
  bool Valid_ =
        !AtEotb() && _ValidDelimSet &&
        _NumDelims && 0 <= Index_ && (Index_ * 2) < _NumDelims;
  
  if (Valid_)
  {
    Subscript startx = Index_ * 2;
    Subscript endx = startx + 1;
    Subscript kpos = DelimPosAt(startx) + 1;
    Subscript vpos = (endx < _NumDelims) ? DelimPosAt(endx):_MatchIndex;

    if (kpos < vpos)
    {
      SearchStr_ = _pTextBuffer->subString(_ReadPos, _pTextBuffer->strlen() - _ReadPos);
      SearchStr_ = SearchStr_.subString(kpos, vpos-kpos);
      SearchStr_.SetCaseSensitive(_pTextBuffer->CaseSensitive());
    }
    else
      Valid_ = false;
  }

  if (Found_)
    *Found_ = Valid_;

  ChrString SrchRes_(SearchStr_);
  return SrchRes_;
}

/****************************************************************************/
ChrString TagSearchResult::TagValueAt(Subscript Index_, bool* Found_) const
{
  if (_ValidDelimSet &&
       (_TagBracketIndex == TagEnums::DOCTYPE_DEX ||
        _TagBracketIndex == TagEnums::NESTEDCOMMENT1_DEX ||
        _TagBracketIndex == TagEnums::NESTEDCOMMENT2_DEX))
  {
    // Get search substring extracted from current read position in text buffer
    ChrString SearchStr_;
    SearchStr_.NullTerminate();

    bool Valid_ =
          !AtEotb() && _ValidDelimSet &&
          _NumDelims && 0 <= Index_ && Index_ < _NumDelims;
  
    if (Valid_)
    {
      Subscript startx = Index_;
      Subscript endx = startx + 1;
      Subscript vpos = DelimPosAt(startx) + 1;
      Subscript kpos = (endx < _NumDelims) ? DelimPosAt(endx):_MatchIndex;

      if (vpos < kpos)
      {
        SearchStr_ = _pTextBuffer->subString(_ReadPos, _pTextBuffer->strlen() - _ReadPos);
        SearchStr_ = SearchStr_.subString(vpos, kpos-vpos);
        SearchStr_.SetCaseSensitive(_pTextBuffer->CaseSensitive());
      }
      else
        Valid_ = false;
    }

    if (Found_)
      *Found_ = Valid_;

    ChrString SrchRes_(SearchStr_);
    return SrchRes_;
  }

  // Get search substring extracted from current read position in text buffer
  ChrString SearchStr_;
  SearchStr_.NullTerminate();
  
  bool Valid_ =
        !AtEotb() && _ValidDelimSet &&
        _NumDelims && 0 <= Index_ && (Index_ * 2) < _NumDelims;
  
  if (Valid_)
  {
    Subscript startx = Index_ * 2 + 1;
    Subscript endx = startx + 1;
    Subscript vpos = DelimPosAt(startx) + 1;
    Subscript kpos = (endx < _NumDelims) ? DelimPosAt(endx):_MatchIndex;

    if (vpos < kpos)
    {
      SearchStr_ = _pTextBuffer->subString(_ReadPos, _pTextBuffer->strlen() - _ReadPos);
      SearchStr_ = SearchStr_.subString(vpos, kpos-vpos);
      SearchStr_.SetCaseSensitive(_pTextBuffer->CaseSensitive());
    }
    else
      Valid_ = false;
  }

  if (Found_)
    *Found_ = Valid_;

  ChrString SrchRes_(SearchStr_);
  return SrchRes_;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::AssignSearchResult(Subscript Index_, Subscript Length_,
                                                     Subscript MIndex_, Subscript MLength_,
                                                     Subscript EIndex_, Subscript ELength_,
                                                     bool NoBlanks_)
{
  if (_pTextBuffer)
  {
    _BrkIndex = Index_;
    _BrkLength = Length_;
    _MatchIndex = MIndex_;
    _MatchLength = MLength_;
    _ElementIndex = EIndex_;
    _ElementLength = ELength_;
    _ReturnValue = 1;

    _AtStartBrk = true;
    _NumDelims = 0;
    _ValidDelimSet = false;

    const char* Ptr_ = c_str();
    Subscript sIndex_ = Index_ + Length_;

    _StartBrkFound = Ptr_ && !isspace(Ptr_[_BrkIndex]) && Index_ + Length_ > 0;
    _EndBrkFound = Ptr_ && !isspace(Ptr_[_MatchIndex]) && MIndex_ + MLength_ > 0;
    _TagElementFound =
       Ptr_ && ELength_ > 0 &&
       ((NoBlanks_ &&
          (isalpha(Ptr_[sIndex_]) ||
           Ptr_[sIndex_] == TagEnums::NESTEDCOMMENT_OPEN_CH ||
           Ptr_[sIndex_] == TagEnums::NESTEDCOMMENT_CLOSE_CH)) || !NoBlanks_);
  }

  return *this;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::AdvanceReadPos()
{
  if (!_AtStartBrk)
    SwapStartEndBrks();

  if (_pTextBuffer && (_EndBrkFound || _StartBrkFound))
  {
    _PrevReadPos = _ReadPos;
    _ReadPos += (_MatchIndex > 0 && _EndBrkFound) ? (_MatchIndex + _MatchLength):
                (_BrkIndex > 0 && _StartBrkFound) ? (_BrkIndex + _BrkLength):0;

    if (_ReadPos == _PrevReadPos && (AtEotb() || _SearchDone))
      _ReadPos = _pTextBuffer->strlen();
    else if (0 <= _ReadPos && _ReadPos <= _pTextBuffer->strlen())
    {
      const char* EndPtr_ = (*_pTextBuffer)(_ReadPos);
      const char* StartPtr_ = (*_pTextBuffer)(_PrevReadPos);

      size_t Eol_;
      size_t ChrCnt_;

      for (Eol_ = ChrCnt_ = 0; StartPtr_ < EndPtr_; StartPtr_++)
      {
        ++ChrCnt_;
        if (*StartPtr_ == '\n')
          ++Eol_;
      }

      _Xpos += ChrCnt_;
      _Ypos += Eol_;
    }

    _BrkIndex = _BrkLength = 0;
    _MatchIndex = _MatchLength = 0;
    _ElementIndex = 0;
    _ElementLength = 0;
    _ReturnValue = 0;
    _ContentType = NO_CONTENT;

    _AtStartBrk = true;
    _NumDelims = 0;
    _ValidDelimSet = false;

    _SearchDone =
    _StartBrkFound =
    _EndBrkFound =
    _TagElementFound = false;
  }

  return *this;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::ResetReadPos()
{
  if (!_AtStartBrk)
    SwapStartEndBrks();

  if (_pTextBuffer)
  {
    _PrevReadPos =
    _ReadPos = 0;

    _Xpos = 0;
    _Ypos = 0;

    _BrkIndex = _BrkLength = 0;
    _MatchIndex = _MatchLength = 0;
    _ElementIndex = 0;
    _ElementLength = 0;
    _ReturnValue = 0;
    _ContentType = NO_CONTENT;

    _AtStartBrk = true;
    _NumDelims = 0;
    _ValidDelimSet = false;

    _SearchDone =
    _StartBrkFound =
    _EndBrkFound =
    _TagElementFound = false;
  }

  return *this;
}

/****************************************************************************/
// At End Of File function
//
bool TagSearchResult::AtEof() const
{
  return _EofFound;
}

/****************************************************************************/
TagSearchResult& TagSearchResult::SetEofFound(bool Flag_)
{
  _EofFound = Flag_;
  return *this;
}

/****************************************************************************/
void TagSearchResult::DumpXYCoordData(ostream& os_, size_t fw_) const
{
  char Buffer_[128];
  Subscript Xstart_;
  Subscript Xend_;

  UlongToStr(_Xpos, Buffer_);
  strcat(Buffer_, ",");
  UlongToStr(_Xpos, Buffer_ + ::strlen(Buffer_));
  os_.width(fw_); os_ <<std::left <<"Xpos,Ypos: " <<Buffer_ <<endl;

  Xstart_ = Xstart();
  UlongToStr(Xstart_, Buffer_);
  strcat(Buffer_, ",");
  Xend_ = Xend();
  UlongToStr(Xend_, Buffer_ + ::strlen(Buffer_));
  os_.width(fw_); os_ <<std::left <<"Xstart,Xend: " <<Buffer_ <<endl;
}

/****************************************************************************/
void TagSearchResult::DumpTagStringData(ostream& os_,
                                        size_t fw_, bool ShowNl_) const
{
  char Buffer_[256];
  ChrString TagStr_ = c_str();
  Subscript sIndex_ = BracketIndex();
  Subscript tLen_ = sIndex_ + BracketLength();
  Subscript mLen_ = MatchingIndex() + MatchingLength();
  Subscript eLen_ = ElementIndex() + ElementLength();
  Subscript rLen_ = eLen_ ? (mLen_ - eLen_):(mLen_ - tLen_);
  Subscript sLen_ = BracketLength() + ElementLength() + rLen_;

  ChrString ResultStr_ = ((sLen_ > 0) ? TagStr_.subString(sIndex_, sLen_):ChrString());
  if (sLen_ < 0)
    sLen_ = 0;

  if (!ShowNl_)
  {
    SubstChar(ResultStr_.c_str(), "\n\r", ' ');
    ResultStr_.SimplifySpaces(false);
  }
  ResultStr_.RemovePadding(" \t\n\r\f\v");

  os_.width(fw_); os_ <<std::left <<"Tag String: ";
  os_.width(fw_); os_ <<std::left <<ResultStr_;
  os_.width(fw_); os_ <<std::left <<"Tag Length: " <<sLen_ <<endl;
}

/****************************************************************************/
ChrString TagSearchResult::GiveTagString() const
{
  ChrString TagStr_ = c_str();
  Subscript sIndex_ = BracketIndex();
  Subscript tLen_ = sIndex_ + BracketLength();
  Subscript mLen_ = MatchingIndex() + MatchingLength();
  Subscript eLen_ = ElementIndex() + ElementLength();
  Subscript rLen_ = eLen_ ? (mLen_ - eLen_):(mLen_ - tLen_);
  Subscript sLen_ = BracketLength() + ElementLength() + rLen_;
  ChrString ResultStr_ = ((sLen_ > 0) ? TagStr_.subString(sIndex_, sLen_):ChrString());
  ResultStr_.RemovePadding(" \t\n\r\f\v");
  return ResultStr_;
}

/****************************************************************************/
void TagSearchResult::DumpTargetBrkData(ostream& os_, size_t fw_) const
{
  char Buffer_[256];
  ChrString TagStr_ = c_str();
  Subscript sIndex_ = BracketIndex();
  Subscript sLen_ = BracketLength();
  Subscript subLen_ = 0;
  Subscript StartPt_ = 0;
  ChrString TempStr_ = ((sLen_ > 0) ? TagStr_.subString(sIndex_, sLen_):ChrString());

  ChrString ResultStr_ = LongToStr(_ReadPos + sIndex_, Buffer_);
  ResultStr_ += ChrString(',');

  if (sLen_ > 0 &&
      (_ContentType & INTRA_NESTEDTAG_CONTENT))
  {
    // StartStr_ = NESTEDCOMMENT1CLOSE_STR;
    // EndStr_ = NESTEDCOMMENT2OPEN_STR;

    if (_TagBracketIndex == TagEnums::COMMENT_DEX ||
        _TagBracketIndex == TagEnums::NESTEDCOMMENT1_DEX ||
        _TagBracketIndex == TagEnums::NESTEDCOMMENT2_DEX)
    {
      StartPt_ = (sIndex_ <= _IntContent._StartPos) ?
                     _IntContent._StartPos:
                     _IntContent._EndPos;
      subLen_ = StartPt_;
      subLen_ -= (sIndex_ + sLen_);

      if (subLen_ > 0)
      {
        TempStr_ += TagStr_.subString(sIndex_ + sLen_, subLen_);
        sLen_ += subLen_;
      }

      ResultStr_ += TempStr_;
    }

    // StartStr_ = COPENCOMMENTBLKCLOSE_STR;
    // EndStr_ = CCLOSECOMMENTBLKOPEN_STR;

    else if (_TagBracketIndex == TagEnums::CCOMMENTBLK_DEX ||
             _TagBracketIndex == TagEnums::COPENCOMMENTBLK_DEX ||
             _TagBracketIndex == TagEnums::CCLOSECOMMENTBLK_DEX)
    {
      StartPt_ = (sIndex_ <= _IntContent._StartPos) ?
                     _IntContent._StartPos:
                     _IntContent._EndPos;
      subLen_ = StartPt_;
      subLen_ -= (sIndex_ + sLen_);

      if (subLen_ > 0)
      {
        TempStr_ += TagStr_.subString(sIndex_ + sLen_, subLen_);
        sLen_ += subLen_;
      }

      ResultStr_ += TempStr_;
    }

    // _TagBracketIndex = DOCTYPE_DEX
    // _TagBracketIndex = SCRIPT_DEX
    // _TagBracketIndex = CPPCOMMENTLINEOPEN_DEX

    else
      ResultStr_ += TempStr_;
  }
  else if (sLen_ > 0)
    ResultStr_ += TempStr_;

  if (sLen_ < 0)
    sLen_ = 0;

  ResultStr_.RemovePadding(" \t\n\r\f\v");
  os_.width(fw_); os_ <<std::left <<"Target Index+Brk: ";
  os_.width(fw_); os_ <<std::left <<ResultStr_;
  os_.width(fw_); os_ <<std::left <<"Target Length: " <<sLen_ <<endl;
}

/****************************************************************************/
ChrString TagSearchResult::GiveTargetBrkString(Subscript* Indexp_, Subscript* Lenp_, bool FullNestedTag_) const
{
  ChrString TagStr_ = c_str();
  Subscript sIndex_ = BracketIndex();
  Subscript sLen_ = BracketLength();
  Subscript subLen_ = 0;
  Subscript StartPt_ = 0;
  ChrString TempStr_ = ((sLen_ > 0) ? TagStr_.subString(sIndex_, sLen_):ChrString());
  ChrString ResultStr_;

  if (FullNestedTag_ && sLen_ > 0 &&
      (_ContentType & INTRA_NESTEDTAG_CONTENT))
  {
    // StartStr_ = NESTEDCOMMENT1CLOSE_STR;
    // EndStr_ = NESTEDCOMMENT2OPEN_STR;

    if (_TagBracketIndex == TagEnums::COMMENT_DEX ||
        _TagBracketIndex == TagEnums::NESTEDCOMMENT1_DEX ||
        _TagBracketIndex == TagEnums::NESTEDCOMMENT2_DEX)
    {
      StartPt_ = (sIndex_ <= _IntContent._StartPos) ?
                     _IntContent._StartPos:
                     _IntContent._EndPos;
      subLen_ = StartPt_;
      subLen_ -= (sIndex_ + sLen_);

      if (subLen_ > 0)
      {
        TempStr_ += TagStr_.subString(sIndex_ + sLen_, subLen_);
        sLen_ += subLen_;
      }

      ResultStr_ = TempStr_;
    }

    // StartStr_ = COPENCOMMENTBLKCLOSE_STR;
    // EndStr_ = CCLOSECOMMENTBLKOPEN_STR;

    else if (_TagBracketIndex == TagEnums::CCOMMENTBLK_DEX ||
             _TagBracketIndex == TagEnums::COPENCOMMENTBLK_DEX ||
             _TagBracketIndex == TagEnums::CCLOSECOMMENTBLK_DEX)
    {
      StartPt_ = (sIndex_ <= _IntContent._StartPos) ?
                     _IntContent._StartPos:
                     _IntContent._EndPos;
      subLen_ = StartPt_;
      subLen_ -= (sIndex_ + sLen_);

      if (subLen_ > 0)
      {
        TempStr_ += TagStr_.subString(sIndex_ + sLen_, subLen_);
        sLen_ += subLen_;
      }

      ResultStr_ = TempStr_;
    }
  }
  else if (sLen_ > 0)
    ResultStr_ = TempStr_;

  ResultStr_.RemovePadding(" \t\n\r\f\v");

  if (Indexp_)
    *Indexp_ = sIndex_;

  if (Lenp_)
    *Lenp_ = sLen_;

  return ResultStr_;
}

/****************************************************************************/
void TagSearchResult::DumpTagElementData(ostream& os_,
                                         size_t fw_, bool ShowNl_) const
{
  char Buffer_[256];
  ChrString TagStr_ = c_str();
  Subscript sIndex_ = ElementIndex();
  Subscript sLen_ = ElementLength();

  ChrString ResultStr_ = LongToStr(_ReadPos + sIndex_, Buffer_);
  ResultStr_ += ChrString(',');
  if (sLen_ > 0)
    ResultStr_ += TagStr_.subString(sIndex_, sLen_);

  if (sLen_ < 0)
    sLen_ = 0;

  if (!ShowNl_)
  {
    SubstChar(ResultStr_.c_str(), "\n\r", ' ');
    ResultStr_.SimplifySpaces(false);
  }
  ResultStr_.RemovePadding(" \t\n\r\f\v");
  os_.width(fw_); os_ <<std::left <<"Element Index+Tag: ";
  os_.width(fw_); os_ <<std::left <<ResultStr_;
  os_.width(fw_); os_ <<std::left <<"Element Length: " <<sLen_ <<endl;
}

/****************************************************************************/
ChrString TagSearchResult::GiveTagElementString() const
{
  ChrString TagStr_ = c_str();
  Subscript sIndex_ = ElementIndex();
  Subscript sLen_ = ElementLength();
  ChrString ResultStr_ = ((sLen_ > 0) ? TagStr_.subString(sIndex_, sLen_):ChrString());
  ResultStr_.RemovePadding(" \t\n\r\f\v");
  return ResultStr_;
}

/****************************************************************************/
void TagSearchResult::DumpTagAttributeData(ostream& os_,
                                           size_t fw_, bool ShowNl_) const
{
  char Buffer_[256];
  Subscript x, y, max;
  Subscript sIndex_;
  ChrString ResultStr_;
  bool Found_;
  bool SingleDelim_ = (_ValidDelimSet &&
                         (_TagBracketIndex == TagEnums::DOCTYPE_DEX ||
                          _TagBracketIndex == TagEnums::NESTEDCOMMENT1_DEX ||
                          _TagBracketIndex == TagEnums::NESTEDCOMMENT2_DEX));
  
  max = DelimsFound();

  for (x = 0; x < max; x++)
  {
    sIndex_ = DelimPosAt(x);
    ResultStr_ = LongToStr(_ReadPos + sIndex_, Buffer_);
    ResultStr_ += ChrString(',') + ChrString(DelimCharAt(x));
    
    os_.width(20); os_ <<std::left <<"Delim Index+Char: ";
    os_.width(20); os_ <<std::left <<ResultStr_ <<endl;

    if (x % 2 || SingleDelim_)
    {
      y = SingleDelim_ ? x:(x/2);

      if (SingleDelim_)
        ResultStr_ = TagValueAt(y, &Found_);
      else
      {
        ResultStr_ = TagKeyAt(y, &Found_);
        ResultStr_ += ChrString('=') + TagValueAt(y, &Found_);
      }

      if (Found_)
      {
        if (!ShowNl_)
        {
          SubstChar(ResultStr_.c_str(), "\n\r", ' ');
          ResultStr_.SimplifySpaces(false);
        }

        ResultStr_.RemovePadding(" \t\n\r\f\v");
        os_.width(20); os_ <<std::left <<(SingleDelim_ ? "IntraTagValue:":
                                                         "Key=Value: ");
        os_.width(20); os_ <<std::left <<ResultStr_ <<endl;
      }
    }
  }
}

/****************************************************************************/
ChrString** TagSearchResult::GiveTagIndexCharString(size_t& ArrayMax_) const
{
  char Buffer_[256];
  Subscript x, y, max;
  Subscript sIndex_;
  ChrString ResultStr_;
  ChrString** RetStrArray_ = NULL;
  bool Found_;
  bool SingleDelim_ = (_ValidDelimSet &&
                         (_TagBracketIndex == TagEnums::DOCTYPE_DEX ||
                          _TagBracketIndex == TagEnums::NESTEDCOMMENT1_DEX ||
                          _TagBracketIndex == TagEnums::NESTEDCOMMENT2_DEX));

  ArrayMax_ = 0;
  x = max = DelimsFound();

  if (max)
  {
    ArrayMax_ = size_t(max);
    max *= sizeof(ChrString*);
    RetStrArray_ = (ChrString**)RawAllocateWith(MEMMATRIX, max);
    max = x;

    for (x = 0; x < max; x++)
    {
      sIndex_ = DelimPosAt(x);
      ResultStr_ = LongToStr(_ReadPos + sIndex_, Buffer_);
      ResultStr_ += ChrString(',') + ChrString(DelimCharAt(x));
      ResultStr_.RemovePadding(" \t\n\r\f\v");
      RetStrArray_[x] = new ChrString(ResultStr_);
    }
  }

  return RetStrArray_;
}

/****************************************************************************/
ChrString** TagSearchResult::GiveTagKeyValueString(size_t& ArrayMax_) const
{
  char Buffer_[256];
  Subscript x, y, max;
  Subscript sIndex_;
  ChrString ResultStr_;
  ChrString** RetStrArray_ = NULL;
  bool Found_;
  bool SingleDelim_ = (_ValidDelimSet &&
                         (_TagBracketIndex == TagEnums::DOCTYPE_DEX ||
                          _TagBracketIndex == TagEnums::NESTEDCOMMENT1_DEX ||
                          _TagBracketIndex == TagEnums::NESTEDCOMMENT2_DEX));

  ArrayMax_ = 0;
  x = max = DelimsFound();
  if (!SingleDelim_)
    max /= 2;

  if (max)
  {
    ArrayMax_ = size_t(max);
    max *= sizeof(ChrString*);
    RetStrArray_ = (ChrString**)RawAllocateWith(MEMMATRIX, max);
    max = x;

    for (x = 0; x < max; x++)
    {
      if (x % 2 || SingleDelim_)
      {
        y = SingleDelim_ ? x:(x/2);

        if (SingleDelim_)
          ResultStr_ = TagValueAt(y, &Found_);
        else
        {
          ResultStr_ = TagKeyAt(y, &Found_);
          ResultStr_ += ChrString('=') + TagValueAt(y, &Found_);
        }

        if (Found_)
        {
          ResultStr_.RemovePadding(" \t\n\r\f\v");
          RetStrArray_[y] = new ChrString(ResultStr_);
        }
      }
    }
  }

  return RetStrArray_;
}

/****************************************************************************/
void TagSearchResult::DumpMatchingBrkData(ostream& os_, size_t fw_) const
{
  char Buffer_[256];
  ChrString TagStr_ = c_str();
  Subscript sIndex_ = MatchingIndex();
  Subscript sLen_ = MatchingLength();
  Subscript subLen_ = 0;
  ChrString TempStr_ = ((sLen_ > 0) ? TagStr_.subString(sIndex_, sLen_):ChrString());
  ChrString ResultStr_;

  if (sLen_ > 0 &&
      (_ContentType & INTRA_NESTEDTAG_CONTENT))
  {
    // StartStr_ = NESTEDCOMMENT1CLOSE_STR;
    // EndStr_ = NESTEDCOMMENT2OPEN_STR;

    if (_TagBracketIndex == TagEnums::COMMENT_DEX ||
        _TagBracketIndex == TagEnums::NESTEDCOMMENT1_DEX ||
        _TagBracketIndex == TagEnums::NESTEDCOMMENT2_DEX)
    {
      subLen_ = sIndex_;
      subLen_ -= _IntContent._EndPos;

      if (subLen_ > 0)
      {
        TempStr_ = (TagStr_.subString(_IntContent._EndPos, subLen_) + TempStr_);
        sIndex_ -= subLen_;
        sLen_ += subLen_;
      }

      ResultStr_ = LongToStr(_ReadPos + sIndex_, Buffer_);
      ResultStr_ += ChrString(',');
      ResultStr_ += TempStr_;
    }

    // StartStr_ = COPENCOMMENTBLKCLOSE_STR;
    // EndStr_ = CCLOSECOMMENTBLKOPEN_STR;

    else if (_TagBracketIndex == TagEnums::CCOMMENTBLK_DEX ||
             _TagBracketIndex == TagEnums::COPENCOMMENTBLK_DEX ||
             _TagBracketIndex == TagEnums::CCLOSECOMMENTBLK_DEX)
    {
      subLen_ = sIndex_;
      subLen_ -= _IntContent._EndPos;

      if (subLen_ > 0)
      {
        TempStr_ = (TagStr_.subString(_IntContent._EndPos, subLen_) + TempStr_);
        sIndex_ -= subLen_;
        sLen_ += subLen_;
      }

      ResultStr_ = LongToStr(_ReadPos + sIndex_, Buffer_);
      ResultStr_ += ChrString(',');
      ResultStr_ += TempStr_;
    }

    // _TagBracketIndex = DOCTYPE_DEX
    // _TagBracketIndex = SCRIPT_DEX
    // _TagBracketIndex = CPPCOMMENTLINEOPEN_DEX

    else
    {
      ResultStr_ = LongToStr(_ReadPos + sIndex_, Buffer_);
      ResultStr_ += ChrString(',');
      ResultStr_ += TempStr_;
    }
  }
  else if (sLen_ > 0)
  {
    ResultStr_ = LongToStr(_ReadPos + sIndex_, Buffer_);
    ResultStr_ += ChrString(',');
    ResultStr_ += TempStr_;
  }

  if (sLen_ < 0)
    sLen_ = 0;

  ResultStr_.RemovePadding(" \t\n\r\f\v");
  os_.width(20); os_ <<std::left <<"Matching Index+Brk: ";
  os_.width(20); os_ <<std::left <<ResultStr_;
  os_.width(20); os_ <<std::left <<"Matching Length: " <<sLen_ <<endl;
}

/****************************************************************************/
ChrString TagSearchResult::GiveMatchingBrkString(Subscript* Indexp_, Subscript* Lenp_, bool FullNestedTag_) const
{
  ChrString TagStr_ = c_str();
  Subscript sIndex_ = MatchingIndex();
  Subscript sLen_ = MatchingLength();
  Subscript subLen_ = 0;
  ChrString TempStr_ = ((sLen_ > 0) ? TagStr_.subString(sIndex_, sLen_):ChrString());
  ChrString ResultStr_;

  if (FullNestedTag_ && sLen_ > 0 &&
      (_ContentType & INTRA_NESTEDTAG_CONTENT))
  {
    // StartStr_ = NESTEDCOMMENT1CLOSE_STR;
    // EndStr_ = NESTEDCOMMENT2OPEN_STR;

    if (_TagBracketIndex == TagEnums::COMMENT_DEX ||
        _TagBracketIndex == TagEnums::NESTEDCOMMENT1_DEX ||
        _TagBracketIndex == TagEnums::NESTEDCOMMENT2_DEX)
    {
      subLen_ = sIndex_;
      subLen_ -= _IntContent._EndPos;

      if (subLen_ > 0)
      {
        TempStr_ = (TagStr_.subString(_IntContent._EndPos, subLen_) + TempStr_);
        sIndex_ -= subLen_;
        sLen_ += subLen_;
      }

      ResultStr_ = TempStr_;
    }

    // StartStr_ = COPENCOMMENTBLKCLOSE_STR;
    // EndStr_ = CCLOSECOMMENTBLKOPEN_STR;

    else if (_TagBracketIndex == TagEnums::CCOMMENTBLK_DEX ||
             _TagBracketIndex == TagEnums::COPENCOMMENTBLK_DEX ||
             _TagBracketIndex == TagEnums::CCLOSECOMMENTBLK_DEX)
    {
      subLen_ = sIndex_;
      subLen_ -= _IntContent._EndPos;

      if (subLen_ > 0)
      {
        TempStr_ = (TagStr_.subString(_IntContent._EndPos, subLen_) + TempStr_);
        sIndex_ -= subLen_;
        sLen_ += subLen_;
      }

      ResultStr_ = TempStr_;
    }
    else
      ResultStr_ = TempStr_;
  }
  else if (sLen_ > 0)
    ResultStr_ = TempStr_;

  ResultStr_.RemovePadding(" \t\n\r\f\v");

  if (Indexp_)
    *Indexp_ = sIndex_;

  if (Lenp_)
    *Lenp_ = sLen_;

  return ResultStr_;
}

/****************************************************************************/
bool TagSearchResult::HasTagContentString() const
{
  if ((_ContentType & TAG_ENCL_CONTENT) &&
      _Content._StartPos > 0 && _Content._EndPos > 0 &&
      _Content._EndPos > _Content._StartPos)
    return (!_Content._Str.IsEmpty());

  return false;
}

/****************************************************************************/
ChrString TagSearchResult::GiveTagContentString() const
{
  if ((_ContentType & TAG_ENCL_CONTENT) &&
      _Content._StartPos > 0 && _Content._EndPos > 0 &&
      _Content._EndPos > _Content._StartPos)
    return _Content._Str;

  return ChrString();
}

/****************************************************************************/
void TagSearchResult::DumpTagContentData(ostream& os_,
                                         size_t fw_, bool ShowNl_) const
{
  char Buffer_[256];
  Subscript sIndex_ = TagContentStartPos();
  Subscript sLen_ = TagContentEndPos() - sIndex_;

  ChrString ResultStr_ = LongToStr(sIndex_, Buffer_);
  ResultStr_ += ChrString(',');
  if (sLen_ > 0 && (_ContentType & TAG_ENCL_CONTENT))
    ResultStr_ += _Content._Str;

  if (sLen_ < 0)
    sLen_ = 0;

  if (!ShowNl_)
  {
    ResultStr_.RemovePadding(" \t\n\r\f\v");
    SubstChar(ResultStr_.c_str(), "\n\r", ' ');
    ResultStr_.SimplifySpaces(false);
  }
  os_.width(20); os_ <<std::left <<"Content Index+Str: ";
  os_.width(20); os_ <<std::left <<ResultStr_;
  os_.width(20); os_ <<std::left <<"Content Length: " <<sLen_ <<endl;
}

/****************************************************************************/
bool TagSearchResult::HasIntraTagContentString() const
{
  if (((_ContentType & INTRA_TAG_CONTENT) ||
       (_ContentType & INTRA_NESTEDTAG_CONTENT)) &&
      _IntContent._StartPos > 0 && _IntContent._EndPos > 0 &&
      _IntContent._EndPos > _IntContent._StartPos)
    return (!_IntContent._Str.IsEmpty());

  return false;
}

/****************************************************************************/
ChrString TagSearchResult::GiveIntraTagContentString() const
{
  if (((_ContentType & INTRA_TAG_CONTENT) ||
       (_ContentType & INTRA_NESTEDTAG_CONTENT)) &&
      _IntContent._StartPos > 0 && _IntContent._EndPos > 0 &&
      _IntContent._EndPos > _IntContent._StartPos)
    return _IntContent._Str;

  return ChrString();
}

/****************************************************************************/
void TagSearchResult::DumpIntraTagContentData(ostream& os_,
                                              size_t fw_, bool ShowNl_) const
{
  char Buffer_[256];
  Subscript sIndex_ = IntraTagContentStartPos();
  Subscript sLen_ = IntraTagContentEndPos() - sIndex_;

  ChrString ResultStr_ = LongToStr(sIndex_, Buffer_);
  ResultStr_ += ChrString(',');
  if (sLen_ > 0 && ((_ContentType & INTRA_TAG_CONTENT) ||
                    (_ContentType & INTRA_NESTEDTAG_CONTENT)))
    ResultStr_ += _IntContent._Str;

  if (sLen_ < 0)
    sLen_ = 0;

  if (!ShowNl_)
  {
    ResultStr_.RemovePadding(" \t\n\r\f\v");
    SubstChar(ResultStr_.c_str(), "\n\r", ' ');
    ResultStr_.SimplifySpaces(false);
  }
  os_.width(20); os_ <<std::left <<"Intra Content Index+Str: ";
  os_.width(20); os_ <<std::left <<ResultStr_;
  os_.width(20); os_ <<std::left <<"Intra Content Length: " <<sLen_ <<endl;
}

/****************************************************************************/
Subscript TagSearchResult::NumKeyValuePairs() const
{
  Subscript max;
  bool SingleDelim_ = (_ValidDelimSet &&
                         (_TagBracketIndex == TagEnums::DOCTYPE_DEX ||
                          _TagBracketIndex == TagEnums::NESTEDCOMMENT1_DEX ||
                          _TagBracketIndex == TagEnums::NESTEDCOMMENT2_DEX));

  max = DelimsFound();
  if (!SingleDelim_)
    max /= 2;

  return max;
}

/****************************************************************************/
void TagSearchResult::DumpSearchResults(ostream& os_, size_t fw_, bool ShowNl_) const
{
  if (_Parent)
  {
    _Parent->DumpElementSeqListNode(os_, fw_);
    _Parent->DumpStackNodeInfo(os_, fw_);
  }

  #if TAGREADER_ABBREV_DUMP
    os_ <<"*> Start : Search Results Dump <*" <<endl;
    DumpTagAttributeData(os_, fw_, ShowNl_);
    DumpTagStringData(os_, fw_, ShowNl_);
    os_ <<"*> End : Search Results Dump <*" <<endl;
  #else
    os_ <<"*> Start : Search Results Dump <*" <<endl;
    DumpXYCoordData(os_, fw_);
    DumpTargetBrkData(os_, fw_);
    DumpTagElementData(os_, fw_, ShowNl_);
    DumpTagAttributeData(os_, fw_, ShowNl_);
    DumpMatchingBrkData(os_, fw_);
    DumpTagStringData(os_, fw_, ShowNl_);
    os_ <<"*> End : Search Results Dump <*" <<endl;
  #endif
}

/****************************************************************************/
MEMORYOPS_DEFN(TagSearchResult)

/****************************************************************************/
// TagLinksSequencer class definition
/****************************************************************************/
TagLinksSequencer::TagLinksSequencer():
_StdTagCount(0),
_NestedCmntTagCount(0),
_CopenCmntTagCount(0),
_NestingError(0),
_TagNestCond(0),
_NCTagNestCond(0),
_ExecTagActionFails(0),
_SearchRunNumber(0),

_HeadInfo(NULL),
_TailInfo(NULL),
_InfoPtr(NULL),
_InfoPtrSet(false),

_HeadCursorTagStk(NULL),
_TailCursorTagStk(NULL),
_CursorTagStkPtr(NULL),
_CursorTagStkPtrSet(false),

_HeadTagStack(NULL),
_TailTagStack(NULL),
_TagStackPtr(NULL),
_TagStackPtrSet(false),

_HeadAppText(NULL),
_TailAppText(NULL),
_AppTextPtr(NULL),
_AppTextPtrSet(false),

_CursorStkTag(NULL),
_FoundTag(NULL),
_FoundCursorTag(false),
_StkTagFound(false),
_PrevTagInfo(NULL)
{}

/****************************************************************************/
TagLinksSequencer::TagLinksSequencer(const TagLinksSequencer& Obj_):
_StdTagCount(Obj_._StdTagCount),
_NestedCmntTagCount(Obj_._NestedCmntTagCount),
_CopenCmntTagCount(Obj_._CopenCmntTagCount),
_NestingError(Obj_._NestingError),
_TagNestCond(Obj_._TagNestCond),
_NCTagNestCond(Obj_._NCTagNestCond),
_ExecTagActionFails(Obj_._ExecTagActionFails),
_SearchRunNumber(Obj_._SearchRunNumber),

_HeadInfo(NULL),
_TailInfo(NULL),
_InfoPtr(NULL),
_InfoPtrSet(false),

_HeadCursorTagStk(NULL),
_TailCursorTagStk(NULL),
_CursorTagStkPtr(NULL),
_CursorTagStkPtrSet(false),

_HeadTagStack(NULL),
_TailTagStack(NULL),
_TagStackPtr(NULL),
_TagStackPtrSet(false),

_HeadAppText(NULL),
_TailAppText(NULL),
_AppTextPtr(NULL),
_AppTextPtrSet(false),

_CursorStkTag(NULL),
_FoundTag(NULL),
_FoundCursorTag(false),
_StkTagFound(false),
_PrevTagInfo(NULL)
{
  ResetPrevTagInfo();
}

/****************************************************************************/
TagLinksSequencer::~TagLinksSequencer()
{
  DestroyAllStacks();
}

/****************************************************************************/
TagLinksSequencer& TagLinksSequencer::operator = (const TagLinksSequencer& Obj_)
{
  if (this != &Obj_)
  {
    _StdTagCount = Obj_._StdTagCount;
    _NestedCmntTagCount = Obj_._NestedCmntTagCount;
    _CopenCmntTagCount = Obj_._CopenCmntTagCount;
    _NestingError = Obj_._NestingError;
    _TagNestCond = Obj_._TagNestCond;
    _NCTagNestCond = Obj_._NCTagNestCond;
    _ExecTagActionFails = Obj_._ExecTagActionFails;
    _SearchRunNumber = Obj_._SearchRunNumber;

    ResetPrevTagInfo();
  }

  return *this;
}

/****************************************************************************/
// Tag stack list related methods
/****************************************************************************/
void TagLinksSequencer::ConfirmTagStackErased()
{
  StackNodeInfo* More_ = NULL;
  NextTagStack(true, false);
  SetTagStackPtr(true);
  StackNodeInfo* NextPtr_ = _HeadTagStack;
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;

  #if TAGREADER_INTERNALSTACK_DEBUG
    if (TagTypeInfo::ShowDebugDump())
    {
      *os_ <<"------------------------------- debug start -------------------------------" <<endl
           <<"Pre ConfirmTagStackErased" <<endl;
      StackNodeInfo::ShowStackAddr(_HeadTagStack, StackNodeInfo::PREVNEXT, "TagStack");
    }
  #endif

  if (NextPtr_)
  {
    _TagStackPtr = NextPtr_;

    do
    {
      if (More_)
      {
        More_->Unlink();
        More_ = NULL;
      }

      More_ = PopTagStack(false, true);
    }
    while (More_ && _TagStackPtr);
  }

  #if TAGREADER_INTERNALSTACK_DEBUG
    if (TagTypeInfo::ShowDebugDump())
    {
      *os_ <<endl <<"Post ConfirmTagStackErased" <<endl;
      StackNodeInfo::ShowStackAddr(_HeadTagStack, StackNodeInfo::PREVNEXT, "TagStack");
      *os_ <<"-------------------------------- debug end --------------------------------"
           <<endl <<endl;
    }
  #endif

  NextTagStack(true, false);
  SetTagStackPtr(false);
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::PopTagStack(bool Erase_, bool ConfirmErased_)
{
  StackNodeInfo* RetPtr_ = NULL;
  StackNodeInfo* EndPt_ = NULL;
  StackNodeInfo* SavedPtr_ = NULL;
  StackNodeInfo* SavedTagPtr_ = _HeadTagStack;
  StackNodeInfo* RelPtr_ = (_TagStackPtr && _TagStackPtrSet) ?
                               _TagStackPtr:
                               (_TagStackPtr=HeadTagStack(true, Erase_ && !ConfirmErased_));

  if (SavedTagPtr_ != _HeadTagStack)
    SavedTagPtr_ = _HeadTagStack;

  bool CurNode_ = _CursorTagStkPtr == _HeadCursorTagStk;
  bool HeadCurNode_ = _HeadCursorTagStk == _HeadTagStack;
  bool HeadEnd_ = _TagStackPtr == _HeadTagStack;
  bool LastNode_ = false;
  bool HeadMoved_ = false;
  int MoveFlag_ = 0;

  if (!RelPtr_)
    return NULL;

  if (!_TagStackPtrSet && _TagStackPtr && Erase_ && !ConfirmErased_)
    SetTagStackPtr(true);

  if (HeadEnd_)
  {
    SavedPtr_ = _TagStackPtr;
    _TagStackPtr = _TagStackPtr->_Next;
    LastNode_ = _TagStackPtr == NULL;

    if (!LastNode_)
      _HeadTagStack = _TagStackPtr;
  }
  else
  {
    SavedPtr_ = RelPtr_;
    _HeadTagStack = RelPtr_->_Next;
    LastNode_ = _HeadTagStack == NULL;

    if (LastNode_)
      _TagStackPtr = _HeadTagStack = RelPtr_;
  }

  HeadMoved_ = SavedTagPtr_ != _HeadTagStack;

  if (HeadMoved_ && ConfirmErased_)
  {
    if (HeadCurNode_)
    {
      _HeadCursorTagStk = _HeadCursorTagStk->_NextCursor;
      MoveFlag_ = 1;
    }

    if (HeadCurNode_ && CurNode_)
    {
      _CursorTagStkPtr = _HeadCursorTagStk;
      MoveFlag_ = 2;
    }
  }

  if (ConfirmErased_)
  {
    SavedTagPtr_ = _HeadTagStack;
    CurNode_ = _CursorTagStkPtr == _HeadCursorTagStk;
    HeadCurNode_ = _HeadCursorTagStk == _HeadTagStack;

    if (RelPtr_->Erased())
    {
      if (RelPtr_->_Next)
      {
        RelPtr_->_Next->SetPrev(NULL);
        if (RelPtr_->_Next->_PrevCursor == RelPtr_)
          RelPtr_->_Next->_PrevCursor = NULL;

        EndPt_ = (RelPtr_ != RelPtr_->_Next) ? RelPtr_->_Next:NULL;
        RelPtr_->SetNext(NULL);
        RelPtr_->_NextCursor = NULL;
      }

      if (MoveFlag_ && RelPtr_ &&
          _HeadCursorTagStk->_PrevCursor &&
          _HeadCursorTagStk->_PrevCursor == RelPtr_)
        if (_HeadCursorTagStk->_PrevCursor->_PrevCursor)
        {
          _HeadCursorTagStk->_PrevCursor->_PrevCursor->_NextCursor = _HeadCursorTagStk;
          _HeadCursorTagStk->_PrevCursor = _HeadCursorTagStk->_PrevCursor->_PrevCursor;
        }
        else
          _HeadCursorTagStk->_PrevCursor = NULL;

      RetPtr_ = RelPtr_;
      RelPtr_ = EndPt_;

      if (LastNode_)
        _TagStackPtr = _HeadTagStack = NULL;
      else if (!HeadEnd_)
        _TagStackPtr = EndPt_;
    }
    else
    {
      if (HeadEnd_)
      {
        if (!LastNode_)
        {
          _HeadTagStack = _TagStackPtr = SavedPtr_;
          MoveFlag_ += 10;
        }
        else
          _TagStackPtr = _HeadTagStack;
      }
      else if (!LastNode_)
      {
        _TagStackPtr = _HeadTagStack = RelPtr_;
        MoveFlag_ += 10;
      }

      RelPtr_ = NULL;
    }

    HeadMoved_ = SavedTagPtr_ != _HeadTagStack;

    if (HeadMoved_)
    {
      if (MoveFlag_ < 10 && _HeadTagStack)
      {
        if (HeadCurNode_)
          _HeadCursorTagStk = _HeadCursorTagStk->_NextCursor;

        if (HeadCurNode_ && CurNode_)
          _CursorTagStkPtr = _HeadCursorTagStk;
      }
      else
      {
        if (MoveFlag_ == 12 || !_HeadTagStack)
          _CursorTagStkPtr = _HeadCursorTagStk = _HeadTagStack;
        else if (MoveFlag_ == 11)
          _HeadCursorTagStk = _HeadTagStack;
      }
    }
  }
  else if (Erase_)
  {
    if (!LastNode_)
      EndPt_ = _HeadTagStack;

    if (!ConfirmErased_)
    {
      while (RelPtr_ && RelPtr_ != EndPt_)
      {
        RelPtr_->SetErased(true);
        RelPtr_ = RelPtr_->_Next;

        if (RelPtr_)
          RelPtr_->ResumeTextAppend(RelPtr_);
      }

      if (LastNode_)
        RelPtr_ = _TailTagStack;
    }

    _HeadTagStack = SavedPtr_;
  }

  return (RetPtr_ ? RetPtr_:RelPtr_);
}

/****************************************************************************/
// Only initiate TagStack popping if:
//   EquivTag(CurTagStkPtr_, _HeadTagStack) returns true AND
//   CurTagStkPtr_ == _HeadCurTagStk AND _TagAppendDone == true
//
// Node popping will also begin for all earlier nodes in TagStack
// where _TagAppendDone == true AND _FoundTag < CurTagStkPtr_:
//   _FoundCurTag == true OR CurTagStkPtr_ == NULL
//
//   by working backwards from _HeadCurTagStk and calling
//   _HeadTagStack->TagIsEqual(CurTagStkPtr_->_Prev, this, true)
//   iteratively until no previous CurTagStk node is found while
//   also popping nodes from _HeadCurTagStk as they are found to
//   match the current pointer from _CurTagStkPtr
//
// Otherwise if CurTagStkPtr_ != _HeadTagStack OR
// EquivTag(CurTagStkPtr_, _HeadTagStack) returns false then:
// indicate a pop operation by setting the bool _Erased flag in CurTagStkPtr_
// to true, but do not actually pop the stack node. TagLinksSequencer::_CurTagStkPtr
// will also be moved to the previous node toward the head of the stack by
// assigning it to the _Prev node. Additionally,
// if EquivTag(FoundTagPtr_, CurTagStkPtr_) method returns true then:
// set the _CurStkNodeFound flag of the CurTagStkPtr_ stack node to true
//
StackNodeInfo* TagLinksSequencer::PopTagStack(StackNodeInfo* FoundTagPtr_, StackNodeInfo* CurTagStkPtr_,
                                        StackNodeInfo* EncNode_, TagTypeInfo* TagTypeInfoPtr_,
                                        bool Erase_, bool ConfirmErased_)
{
  StackNodeInfo* RetPtr_ = NULL;
  StackNodeInfo* EndPt_ = NULL;
  StackNodeInfo* SavedPtr_ = NULL;
  StackNodeInfo* SavedTagPtr_ = _HeadTagStack;
  StackNodeInfo* CurStkNode_ = NULL;
  StackNodeInfo* RelPtr_ = (_TagStackPtr && _TagStackPtrSet) ?
                               _TagStackPtr:
                               (_TagStackPtr=HeadTagStack(true, Erase_ && !ConfirmErased_));

  if (SavedTagPtr_ != _HeadTagStack)
    SavedTagPtr_ = _HeadTagStack;

  bool CurNode_ = _CursorTagStkPtr == _HeadCursorTagStk;
  bool HeadCurNode_ = _HeadCursorTagStk == _HeadTagStack;
  bool HeadEnd_ = _TagStackPtr == _HeadTagStack;
  bool LastNode_ = false;
  bool HeadMoved_ = false;
  int MoveFlag_ = 0;

  if (!_HeadTagStack || !_HeadCursorTagStk || !TagTypeInfoPtr_ || !RelPtr_)
    return NULL;

  if (!_TagStackPtrSet && _TagStackPtr && Erase_ && !ConfirmErased_)
    SetTagStackPtr(true);

  bool SearchDone_ = false;
  bool Incremented_ = false;
  bool CurTagHead_ = CurTagStkPtr_ == _HeadCursorTagStk;
  bool NodeMatch_ = CurTagStkPtr_ && _HeadTagStack &&
                    StackNodeInfo::EquivTag(CurTagStkPtr_, _HeadTagStack);

  if (NodeMatch_ &&
        (CurTagHead_ || TagTypeInfoPtr_->Sequencer()->FoundCursorTag()))
  {
    // has forw/back links of cursor stack node been processed yet?
    Incremented_ = CurTagStkPtr_->_TagAppendDone;

    // link forward if cursor stack node is of singleton or empty type
    if (!Incremented_ && CurTagStkPtr_ &&
          (CurTagStkPtr_->SingletonTagFound() ||
           CurTagStkPtr_->EmptyTagFound()))
    {
      CurTagStkPtr_->LinkForward(EncNode_);
      Incremented_ = CurTagStkPtr_->_TagAppendDone;
    }

    if (Incremented_)
    {
      if (HeadEnd_)
      {
        SavedPtr_ = _TagStackPtr;
        _TagStackPtr = _TagStackPtr->_Next;
        LastNode_ = _TagStackPtr == NULL;

        if (!LastNode_)
          _HeadTagStack = _TagStackPtr;
      }
      else
      {
        SavedPtr_ = RelPtr_;
        _HeadTagStack = RelPtr_->_Next;
        LastNode_ = _HeadTagStack == NULL;

        if (LastNode_)
          _TagStackPtr = _HeadTagStack = RelPtr_;
      }

      HeadMoved_ = SavedTagPtr_ != _HeadTagStack;

      if (HeadMoved_ && ConfirmErased_)
      {
        if (HeadCurNode_)
        {
          _HeadCursorTagStk = _HeadCursorTagStk->_NextCursor;
          MoveFlag_ = 1;
        }

        if (HeadCurNode_ && CurNode_)
        {
          _CursorTagStkPtr = _HeadCursorTagStk;
          MoveFlag_ = 2;
        }
      }
    }
  }

  if (CurTagStkPtr_ &&
        (!NodeMatch_ ||
         !(CurTagHead_ || TagTypeInfoPtr_->Sequencer()->FoundCursorTag())))
  {
    // unlink tag nodes that has already been processed
    if (!CurTagStkPtr_->Unlinked() && CurTagStkPtr_->_TagAppendDone)
      CurTagStkPtr_->SetUnlinked(true);

    // set cursor stack node found, if found tag node is equal to it
    if (!NodeMatch_ && StackNodeInfo::EquivTag(FoundTagPtr_, CurTagStkPtr_))
      CurTagStkPtr_->SetCursorStkNodeFound(true);
  }

  if (NodeMatch_ && Incremented_ &&
        (CurTagHead_ || TagTypeInfoPtr_->Sequencer()->FoundCursorTag()))
  {
    // if cursor stack tag matches with head of tag stack or is the
    // found tag result of search then unlink it
    if (CurTagStkPtr_)
      CurTagStkPtr_->SetUnlinked(true);

    if (ConfirmErased_)
    {
      SavedTagPtr_ = _HeadTagStack;
      CurNode_ = _CursorTagStkPtr == _HeadCursorTagStk;
      HeadCurNode_ = _HeadCursorTagStk == _HeadTagStack;

      if (RelPtr_->Erased())
      {
        if (RelPtr_->_Next)
        {
          RelPtr_->_Next->SetPrev(NULL);
          if (RelPtr_->_Next->_PrevCursor == RelPtr_)
            RelPtr_->_Next->_PrevCursor = NULL;

          EndPt_ = (RelPtr_ != RelPtr_->_Next) ? RelPtr_->_Next:NULL;
          RelPtr_->SetNext(NULL);
          RelPtr_->_NextCursor = NULL;
        }

        if (MoveFlag_ && RelPtr_ &&
            _HeadCursorTagStk->_PrevCursor &&
            _HeadCursorTagStk->_PrevCursor == RelPtr_)
          if (_HeadCursorTagStk->_PrevCursor->_PrevCursor)
          {
            _HeadCursorTagStk->_PrevCursor->_PrevCursor->_NextCursor = _HeadCursorTagStk;
            _HeadCursorTagStk->_PrevCursor = _HeadCursorTagStk->_PrevCursor->_PrevCursor;
          }
          else
            _HeadCursorTagStk->_PrevCursor = NULL;

        RetPtr_ = RelPtr_;
        RelPtr_ = EndPt_;

        if (LastNode_)
          _TagStackPtr = _HeadTagStack = NULL;
        else if (!HeadEnd_)
          _TagStackPtr = EndPt_;
      }
      else
      {
        if (HeadEnd_)
        {
          if (!LastNode_)
          {
            _HeadTagStack = _TagStackPtr = SavedPtr_;
            MoveFlag_ += 10;
          }
          else
            _TagStackPtr = _HeadTagStack;
        }
        else if (!LastNode_)
        {
          _TagStackPtr = _HeadTagStack = RelPtr_;
          MoveFlag_ += 10;
        }

        RelPtr_ = NULL;
      }

      HeadMoved_ = SavedTagPtr_ != _HeadTagStack;

      if (HeadMoved_)
      {
        if (MoveFlag_ < 10)
        {
          if (HeadCurNode_)
            _HeadCursorTagStk = _HeadCursorTagStk->_NextCursor;

          if (HeadCurNode_ && CurNode_)
            _CursorTagStkPtr = _HeadCursorTagStk;
        }
        else
        {
          if (MoveFlag_ == 12 || !_HeadTagStack)
            _CursorTagStkPtr = _HeadCursorTagStk = _HeadTagStack;
          else if (MoveFlag_ == 11)
            _HeadCursorTagStk = _HeadTagStack;
        }
      }
    }
    else if (Erase_)
    {
      if (!LastNode_)
        EndPt_ = _HeadTagStack;

      if (!ConfirmErased_)
      {
        while (RelPtr_ && RelPtr_ != EndPt_)
        {
          RelPtr_->SetErased(true);
          RelPtr_ = RelPtr_->_Next;

          if (RelPtr_)
            RelPtr_->ResumeTextAppend(RelPtr_);
        }

        if (LastNode_)
          RelPtr_ = _TailTagStack;
      }

      _HeadTagStack = SavedPtr_;
    }
  }

  SavedTagPtr_ = CurTagStkPtr_;

  for (SavedPtr_ = CurStkNode_ = NULL;
       (!SavedPtr_ || SavedPtr_ != CurStkNode_) && (CurStkNode_ = PrevCursorTag());
       SavedPtr_ = CurStkNode_)
  {
    TagTypeInfoPtr_->Sequencer()->SetToGlobalStackTag();
    CurTagStkPtr_ = TagTypeInfoPtr_->Sequencer()->CursorStackTag();

    if (_HeadTagStack->TagIsEqual(CurTagStkPtr_, TagTypeInfoPtr_, true))
    {
      SearchDone_ = true;
      break;
    }
    else
    {
      // link forward within cursor stack all unprocessed
      // singleton or empty tag types
      if (CurTagStkPtr_ &&
            (CurTagStkPtr_->SingletonTagFound() ||
             CurTagStkPtr_->EmptyTagFound()))
        CurTagStkPtr_->LinkForward(EncNode_);
    }
  }

  if (SearchDone_ && TagTypeInfoPtr_->Sequencer()->FoundCursorTag() &&
      (SavedTagPtr_ != TagTypeInfoPtr_->Sequencer()->CursorStackTag() ||
       FoundTagPtr_ != TagTypeInfoPtr_->Sequencer()->FoundTag()))
    return PopTagStack(TagTypeInfoPtr_->Sequencer()->FoundTag(),
                       TagTypeInfoPtr_->Sequencer()->CursorStackTag(),
                       EncNode_, TagTypeInfoPtr_, true);

  return (RetPtr_ ? RetPtr_:RelPtr_);
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::HeadTagStack(bool Reset_, bool NotErased_)
{
  StackNodeInfo* PrevPtr_ = _HeadTagStack;
  StackNodeInfo* NextPtr_ = _HeadTagStack;

  if (_HeadTagStack && NotErased_)
  {
    bool TailEnd_ = PrevPtr_ == _TailTagStack;

    if (Reset_)
    {
      for (PrevPtr_ = _HeadTagStack->_Prev;
           PrevPtr_ && (_HeadTagStack = PrevPtr_);
           PrevPtr_ = _HeadTagStack->_Prev);
      NextPtr_ = _HeadTagStack;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_Next;
      TailEnd_ = PrevPtr_ == _TailTagStack;
    }
    while (PrevPtr_ && PrevPtr_->Erased() && !TailEnd_);

    _HeadTagStack = PrevPtr_;

    if (_HeadTagStack->Erased() && TailEnd_)
    {
      for (PrevPtr_ = _HeadTagStack->_Prev;
           PrevPtr_ && (_HeadTagStack = PrevPtr_);
           PrevPtr_ = _HeadTagStack->_Prev);

      return (_HeadTagStack->Erased() ? NULL:_HeadTagStack);
    }
  }
  else if (_HeadTagStack && Reset_)
  {
    for (PrevPtr_ = _HeadTagStack->_Prev;
         PrevPtr_ && (_HeadTagStack = PrevPtr_);
         PrevPtr_ = _HeadTagStack->_Prev);
  }

  return _HeadTagStack;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::TailTagStack(bool Reset_, bool NotErased_)
{
  StackNodeInfo* PrevPtr_ = _TailTagStack;
  StackNodeInfo* NextPtr_ = _TailTagStack;

  if (_TailTagStack && NotErased_)
  {
    bool HeadEnd_ = PrevPtr_ == _HeadTagStack;

    if (Reset_)
    {
      for (NextPtr_ = _TailTagStack->_Next;
           NextPtr_ && (_TailTagStack = NextPtr_);
           NextPtr_ = _TailTagStack->_Next);
      NextPtr_ = _TailTagStack;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_Prev;
      HeadEnd_ = PrevPtr_ == _HeadTagStack;
    }
    while (PrevPtr_ && PrevPtr_->Erased() && !HeadEnd_);

    _TailTagStack = PrevPtr_;

    if (_TailTagStack->Erased() && HeadEnd_)
    {
      for (NextPtr_ = _TailTagStack->_Next;
           NextPtr_ && (_TailTagStack = NextPtr_);
           NextPtr_ = _TailTagStack->_Next);

      return (_TailTagStack->Erased() ? NULL:_TailTagStack);
    }
  }
  else if (_TailTagStack && Reset_)
  {
    for (NextPtr_ = _TailTagStack->_Next;
         NextPtr_ && (_TailTagStack = NextPtr_);
         NextPtr_ = _TailTagStack->_Next);
  }

  return _TailTagStack;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::NextTagStack(bool Reset_, bool Peek_, bool NotErased_)
{
  if (_TagStackPtr && !Reset_)
  {
    bool TailEnd_ = _TagStackPtr == _TailTagStack;
    StackNodeInfo* PrevPtr_ = _TagStackPtr;
    StackNodeInfo* NextPtr_ = _TagStackPtr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_Next;
    }
    while (NotErased_ && NextPtr_ && NextPtr_->Erased());

    if (Peek_)
      return NextPtr_;
    else
      _TagStackPtr = NextPtr_;

    if (TailEnd_)
    {
      _TailTagStack = _TagStackPtr ? _TagStackPtr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _TagStackPtr = HeadTagStack(true, NotErased_);

  return _TagStackPtr;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::PrevTagStack(bool Reset_, bool Peek_, bool NotErased_)
{
  if (_TagStackPtr && !Reset_)
  {
    bool HeadEnd_ = _TagStackPtr == _HeadTagStack;
    StackNodeInfo* PrevPtr_ = _TagStackPtr;
    StackNodeInfo* NextPtr_ = _TagStackPtr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_Prev;
    }
    while (NotErased_ && NextPtr_ && NextPtr_->Erased());

    if (Peek_)
      return NextPtr_;
    else
      _TagStackPtr = NextPtr_;

    if (HeadEnd_)
    {
      _HeadTagStack = _TagStackPtr ? _TagStackPtr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _TagStackPtr = TailTagStack(true, NotErased_);

  return _TagStackPtr;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::SetTagStackPtr(bool Set_)
{
  _TagStackPtrSet = Set_;
  return _TagStackPtr;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::PushTagStack(StackNodeInfo* Node_, bool Create_)
{
  StackNodeInfo* NodePtr_ = (Create_ && Node_) ? new StackNodeInfo(*Node_):Node_;
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;

  if (NodePtr_)
  {
    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump())
      {
        *os_ <<"------------------------------- debug start -------------------------------" <<endl
             <<"Pre PushTagStack" <<endl;
        StackNodeInfo::ShowStackAddr(_HeadTagStack, StackNodeInfo::PREVNEXT, "TagStack");
      }
    #endif

    if (Create_ && Node_ && NodePtr_)
    {
      NodePtr_->SetMatchInfoLink(Node_);
      NodePtr_->SetNodeID(Node_->NodeID());
    }

    if (_HeadTagStack)
    {
      HeadTagStack(true, false);
      _TagStackPtr =
      _HeadTagStack = (StackNodeInfo*)(_HeadTagStack->AppendHead(NodePtr_, true));
      SetTagStackPtr(false);
    }
    else
      _TagStackPtr =
      _HeadTagStack =
      _TailTagStack = NodePtr_;

    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump())
      {
        *os_ <<endl <<"Post PushTagStack" <<endl;
        StackNodeInfo::ShowStackAddr(_HeadTagStack, StackNodeInfo::PREVNEXT, "TagStack");
        *os_ <<"-------------------------------- debug end --------------------------------"
             <<endl <<endl;
      }
    #endif
  }

  return NodePtr_;
}

/****************************************************************************/
// Cursor Tag list related methods
/****************************************************************************/
void TagLinksSequencer::ConfirmCursorTagErased()
{
  StackNodeInfo* More_ = NULL;
  NextCursorTag(true, false);
  SetCursorTagStkPtr(true);
  StackNodeInfo* NextPtr_ = _HeadCursorTagStk;
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;

  #if TAGREADER_INTERNALSTACK_DEBUG
    if (TagTypeInfo::ShowDebugDump())
    {
      *os_ <<"------------------------------- debug start -------------------------------" <<endl
           <<"Pre ConfirmCursorTagErased" <<endl;
      StackNodeInfo::ShowStackAddr(_HeadCursorTagStk, StackNodeInfo::NEXTCUR, "CursorTagStk");
    }
  #endif

  if (NextPtr_)
  {
    _CursorTagStkPtr = NextPtr_;
    do (More_ = PopCursorTag(false, true));
    while (More_ && _CursorTagStkPtr);
  }

  #if TAGREADER_INTERNALSTACK_DEBUG
    if (TagTypeInfo::ShowDebugDump())
    {
      *os_ <<endl <<"Post ConfirmCursorTagErased" <<endl;
      StackNodeInfo::ShowStackAddr(_HeadCursorTagStk, StackNodeInfo::NEXTCUR, "CursorTagStk");
      *os_ <<"-------------------------------- debug end --------------------------------"
           <<endl <<endl;
    }
  #endif

  NextCursorTag(true, false);
  SetCursorTagStkPtr(false);
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::PopCursorTag(bool Erase_, bool ConfirmErased_)
{
  StackNodeInfo* RetPtr_ = NULL;
  StackNodeInfo* EndPt_ = NULL;
  StackNodeInfo* SavedPtr_ = NULL;
  StackNodeInfo* SavedTagPtr_ = _HeadCursorTagStk;
  StackNodeInfo* RelPtr_ = (_CursorTagStkPtr && _CursorTagStkPtrSet) ?
                               _CursorTagStkPtr:
                               (_CursorTagStkPtr=HeadCursorTagStk(true, Erase_ && !ConfirmErased_));

  if (SavedTagPtr_ != _HeadCursorTagStk)
    SavedTagPtr_ = _HeadCursorTagStk;

  bool TagNode_ = _TagStackPtr == _HeadTagStack;
  bool HeadTagNode_ = _HeadTagStack == _HeadCursorTagStk;
  bool HeadEnd_ = _CursorTagStkPtr == _HeadCursorTagStk;
  bool LastNode_ = false;
  bool HeadMoved_ = false;
  int MoveFlag_ = 0;

  if (!RelPtr_)
    return NULL;

  if (!_CursorTagStkPtrSet && _CursorTagStkPtr && Erase_ && !ConfirmErased_)
    SetCursorTagStkPtr(true);

  if (HeadEnd_)
  {
    SavedPtr_ = _CursorTagStkPtr;
    _CursorTagStkPtr = _CursorTagStkPtr->_NextCursor;
    LastNode_ = _CursorTagStkPtr == NULL;

    if (!LastNode_)
      _HeadCursorTagStk = _CursorTagStkPtr;
  }
  else
  {
    SavedPtr_ = RelPtr_;
    _HeadCursorTagStk = RelPtr_->_NextCursor;
    LastNode_ = _HeadCursorTagStk == NULL;

    if (LastNode_)
      _CursorTagStkPtr = _HeadCursorTagStk = RelPtr_;
  }

  HeadMoved_ = SavedTagPtr_ != _HeadCursorTagStk;

  if (HeadMoved_ && ConfirmErased_)
  {
    if (HeadTagNode_)
    {
      _HeadTagStack = _HeadTagStack->_Next;
      MoveFlag_ = 1;
    }

    if (HeadTagNode_ && TagNode_)
    {
      _TagStackPtr = _HeadTagStack;
      MoveFlag_ = 2;
    }
  }

  if (ConfirmErased_)
  {
    SavedTagPtr_ = _HeadCursorTagStk;
    TagNode_ = _TagStackPtr == _HeadTagStack;
    HeadTagNode_ = _HeadTagStack == _HeadCursorTagStk;

    // Unlinked flag is checked instead of the erase flag
    if (RelPtr_->_Unlinked)
    {
      if (RelPtr_->_NextCursor)
      {
        RelPtr_->_NextCursor->_PrevCursor = NULL;
        EndPt_ = (RelPtr_ != RelPtr_->_NextCursor) ? RelPtr_->_NextCursor:NULL;
        RelPtr_->_NextCursor = NULL;
      }

      // deletion of node is omitted and left as job for the tagstack
      // delete RelPtr_;
      RetPtr_ = RelPtr_;
      RelPtr_ = EndPt_;

      if (LastNode_)
        _CursorTagStkPtr = _HeadCursorTagStk = NULL;
      else if (!HeadEnd_)
        _CursorTagStkPtr = EndPt_;
    }
    else
    {
      if (HeadEnd_)
      {
        if (!LastNode_)
        {
          _HeadCursorTagStk = _CursorTagStkPtr = SavedPtr_;
          MoveFlag_ += 10;
        }
        else
          _CursorTagStkPtr = _HeadCursorTagStk;
      }
      else if (!LastNode_)
      {
        _CursorTagStkPtr = _HeadCursorTagStk = RelPtr_;
        MoveFlag_ += 10;
      }

      RelPtr_ = NULL;
    }

    HeadMoved_ = SavedTagPtr_ != _HeadCursorTagStk;

    if (HeadMoved_)
    {
      if (MoveFlag_ < 10)
      {
        if (HeadTagNode_)
          _HeadTagStack = _HeadTagStack->_Next;

        if (HeadTagNode_ && TagNode_)
          _TagStackPtr = _HeadTagStack;
      }
      else
      {
        if (MoveFlag_ == 12 || !_HeadCursorTagStk)
          _TagStackPtr = _HeadTagStack = _HeadCursorTagStk;
        else if (MoveFlag_ == 11);
          _HeadTagStack = _HeadCursorTagStk;
      }
    }
  }
  else if (Erase_)
  {
    if (!LastNode_)
      EndPt_ = _HeadCursorTagStk;

    if (!ConfirmErased_)
    {
      while (RelPtr_ && RelPtr_ != EndPt_)
      {
        RelPtr_->SetUnlinked(true);
        RelPtr_ = RelPtr_->_NextCursor;
      }

      if (LastNode_)
        RelPtr_ = _TailCursorTagStk;
    }

    _HeadCursorTagStk = SavedPtr_;
  }

  return (RetPtr_ ? RetPtr_:RelPtr_);
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::HeadCursorTagStk(bool Reset_, bool NotErased_)
{
  StackNodeInfo* PrevPtr_ = _HeadCursorTagStk;
  StackNodeInfo* NextPtr_ = _HeadCursorTagStk;

  if (_HeadCursorTagStk && NotErased_)
  {
    bool TailEnd_ = PrevPtr_ == _TailCursorTagStk;

    if (Reset_)
    {
      for (PrevPtr_ = _HeadCursorTagStk->_PrevCursor;
           PrevPtr_ && (_HeadCursorTagStk = PrevPtr_);
           PrevPtr_ = _HeadCursorTagStk->_PrevCursor);
      NextPtr_ = _HeadCursorTagStk;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_NextCursor;
      TailEnd_ = PrevPtr_ == _TailCursorTagStk;
    }
    while (PrevPtr_ &&
           (PrevPtr_->Erased() || PrevPtr_->Unlinked()) && !TailEnd_);

    _HeadCursorTagStk = PrevPtr_;

    if ((_HeadCursorTagStk->Erased() || _HeadCursorTagStk->Unlinked()) && TailEnd_)
    {
      for (PrevPtr_ = _HeadCursorTagStk->_PrevCursor;
           PrevPtr_ && (_HeadCursorTagStk = PrevPtr_);
           PrevPtr_ = _HeadCursorTagStk->_PrevCursor);

      return ((_HeadCursorTagStk->Erased() || _HeadCursorTagStk->Unlinked()) ?
                  NULL:_HeadCursorTagStk);
    }
  }
  else if (_HeadCursorTagStk && Reset_)
  {
    for (PrevPtr_ = _HeadCursorTagStk->_PrevCursor;
         PrevPtr_ && (_HeadCursorTagStk = PrevPtr_);
         PrevPtr_ = _HeadCursorTagStk->_PrevCursor);
  }

  return _HeadCursorTagStk;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::TailCursorTagStk(bool Reset_, bool NotErased_)
{
  StackNodeInfo* PrevPtr_ = _TailCursorTagStk;
  StackNodeInfo* NextPtr_ = _TailCursorTagStk;

  if (_TailCursorTagStk && NotErased_)
  {
    bool HeadEnd_ = PrevPtr_ == _HeadCursorTagStk;

    if (Reset_)
    {
      for (NextPtr_ = _TailCursorTagStk->_NextCursor;
           NextPtr_ && (_TailCursorTagStk = NextPtr_);
           NextPtr_ = _TailCursorTagStk->_NextCursor);
      NextPtr_ = _TailCursorTagStk;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_PrevCursor;
      HeadEnd_ = PrevPtr_ == _HeadCursorTagStk;
    }
    while (PrevPtr_ &&
           (PrevPtr_->Erased() || PrevPtr_->Unlinked()) && !HeadEnd_);

    _TailCursorTagStk = PrevPtr_;

    if ((_TailCursorTagStk->Erased() || _TailCursorTagStk->Unlinked()) && HeadEnd_)
    {
      for (NextPtr_ = _TailCursorTagStk->_NextCursor;
           NextPtr_ && (_TailCursorTagStk = NextPtr_);
           NextPtr_ = _TailCursorTagStk->_NextCursor);

      return ((_TailCursorTagStk->Erased() || _TailCursorTagStk->Unlinked()) ?
                  NULL:_TailCursorTagStk);
    }
  }
  else if (_TailCursorTagStk && Reset_)
  {
    for (NextPtr_ = _TailCursorTagStk->_NextCursor;
         NextPtr_ && (_TailCursorTagStk = NextPtr_);
         NextPtr_ = _TailCursorTagStk->_NextCursor);
  }

  return _TailCursorTagStk;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::NextCursorTag(bool Reset_, bool Peek_, bool NotErased_)
{
  if (_CursorTagStkPtr && !Reset_)
  {
    bool TailEnd_ = _CursorTagStkPtr == _TailCursorTagStk;
    StackNodeInfo* PrevPtr_ = _CursorTagStkPtr;
    StackNodeInfo* NextPtr_ = _CursorTagStkPtr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_NextCursor;
    }
    while (NotErased_ && NextPtr_ &&
           (NextPtr_->Erased() || NextPtr_->Unlinked()));

    if (Peek_)
      return NextPtr_;
    else
      _CursorTagStkPtr = NextPtr_;

    if (TailEnd_)
    {
      _TailCursorTagStk = _CursorTagStkPtr ? _CursorTagStkPtr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _CursorTagStkPtr = HeadCursorTagStk(true, NotErased_);

  return _CursorTagStkPtr;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::PrevCursorTag(bool Reset_, bool Peek_, bool NotErased_)
{
  if (_CursorTagStkPtr && !Reset_)
  {
    bool HeadEnd_ = _CursorTagStkPtr == _HeadCursorTagStk;
    StackNodeInfo* PrevPtr_ = _CursorTagStkPtr;
    StackNodeInfo* NextPtr_ = _CursorTagStkPtr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_PrevCursor;
    }
    while (NotErased_ && NextPtr_ &&
           (NextPtr_->Erased() || NextPtr_->Unlinked()));

    if (Peek_)
      return NextPtr_;
    else
      _CursorTagStkPtr = NextPtr_;

    if (HeadEnd_)
    {
      _HeadCursorTagStk = _CursorTagStkPtr ? _CursorTagStkPtr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _CursorTagStkPtr = TailCursorTagStk(true, NotErased_);

  return _CursorTagStkPtr;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::SetCursorTagStkPtr(bool Set_)
{
  _CursorTagStkPtrSet = Set_;
  return _CursorTagStkPtr;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::PushCursorTag(StackNodeInfo* Node_, bool Create_)
{
  StackNodeInfo* NodePtr_ = (Create_ && Node_) ? new StackNodeInfo(*Node_):Node_;
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;

  if (NodePtr_)
  {
    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump())
      {
        *os_ <<"------------------------------- debug start -------------------------------" <<endl
             <<"Pre PushCursorTag" <<endl;
        StackNodeInfo::ShowStackAddr(_HeadCursorTagStk, StackNodeInfo::NEXTCUR, "CursorTagStk");
      }
    #endif

    if (Create_ && Node_ && NodePtr_)
    {
      NodePtr_->SetMatchInfoLink(Node_);
      NodePtr_->SetNodeID(Node_->NodeID());
    }

    if (_HeadCursorTagStk)
    {
      HeadCursorTagStk(true, false);
      _CursorTagStkPtr =
      _HeadCursorTagStk = _HeadCursorTagStk->AppendHeadCursor(NodePtr_);
      SetCursorTagStkPtr(false);
    }
    else
      _CursorTagStkPtr =
      _HeadCursorTagStk =
      _TailCursorTagStk = NodePtr_;

    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump())
      {
        *os_ <<endl <<"Post PushCursorTag" <<endl;
        StackNodeInfo::ShowStackAddr(_HeadCursorTagStk, StackNodeInfo::NEXTCUR, "CursorTagStk");
        *os_ <<"-------------------------------- debug end --------------------------------"
             <<endl <<endl;
      }
    #endif
  }

  return NodePtr_;
}

/****************************************************************************/
// Stack info list related methods
/****************************************************************************/
void TagLinksSequencer::ConfirmStackInfoErased()
{
  StackNodeInfo* More_ = NULL;
  NextStackInfo(true, false);
  SetStackInfoPtr(true);
  StackNodeInfo* NextPtr_ = _HeadInfo;
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;

  #if TAGREADER_INTERNALSTACK_DEBUG
    if (TagTypeInfo::ShowDebugDump())
    {
      *os_ <<"------------------------------- debug start -------------------------------" <<endl
           <<"Pre ConfirmStackInfoErased" <<endl;
      StackNodeInfo::ShowStackAddr(_HeadInfo, StackNodeInfo::PREVNEXT, "Info");
    }
  #endif

  if (NextPtr_)
  {
    _InfoPtr = NextPtr_;

    do
    {
      if (More_)
      {
        More_->Unlink();
        More_ = NULL;
      }

      More_ = PopStackInfo(false, true);
    }
    while (More_ && _InfoPtr);
  }

  #if TAGREADER_INTERNALSTACK_DEBUG
    if (TagTypeInfo::ShowDebugDump())
    {
      *os_ <<endl <<"Post ConfirmStackInfoErased" <<endl;
      StackNodeInfo::ShowStackAddr(_HeadInfo, StackNodeInfo::PREVNEXT, "Info");
      *os_ <<"-------------------------------- debug end --------------------------------"
           <<endl <<endl;
    }
  #endif

  NextStackInfo(true, false);
  SetStackInfoPtr(false);
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::PopStackInfo(bool Erase_, bool ConfirmErased_)
{
  StackNodeInfo* RetPtr_ = NULL;
  StackNodeInfo* EndPt_ = NULL;
  StackNodeInfo* SavedPtr_ = NULL;
  StackNodeInfo* SavedTagPtr_ = _HeadInfo;
  StackNodeInfo* RelPtr_ = (_InfoPtr && _InfoPtrSet) ?
                               _InfoPtr:
                               (_InfoPtr=HeadInfo(true, Erase_ && !ConfirmErased_));

  if (SavedTagPtr_ != _HeadInfo)
    SavedTagPtr_ = _HeadInfo;

  bool TagNode_ = _TagStackPtr == _HeadTagStack;
  bool HeadTagNode_ = _HeadTagStack == _HeadInfo;
  bool HeadEnd_ = _InfoPtr == _HeadInfo;
  bool LastNode_ = false;
  bool HeadMoved_ = false;
  int MoveFlag_ = 0;

  if (!RelPtr_)
    return NULL;

  if (!_InfoPtrSet && _InfoPtr && Erase_ && !ConfirmErased_)
    SetStackInfoPtr(true);

  if (HeadEnd_)
  {
    SavedPtr_ = _InfoPtr;
    _InfoPtr = _InfoPtr->_Next;
    LastNode_ = _InfoPtr == NULL;

    if (!LastNode_)
      _HeadInfo = _InfoPtr;
  }
  else
  {
    SavedPtr_ = RelPtr_;
    _HeadInfo = RelPtr_->_Next;
    LastNode_ = _HeadInfo == NULL;

    if (LastNode_)
      _InfoPtr = _HeadInfo = RelPtr_;
  }

  HeadMoved_ = SavedTagPtr_ != _HeadInfo;

  if (HeadMoved_ && ConfirmErased_)
  {
    if (HeadTagNode_)
    {
      _HeadTagStack = _HeadTagStack->_Next;
      MoveFlag_ = 1;
    }

    if (HeadTagNode_ && TagNode_)
    {
      _TagStackPtr = _HeadTagStack;
      MoveFlag_ = 2;
    }
  }

  if (ConfirmErased_)
  {
    SavedTagPtr_ = _HeadInfo;
    TagNode_ = _TagStackPtr == _HeadTagStack;
    HeadTagNode_ = _HeadTagStack == _HeadInfo;

    if (RelPtr_->Erased())
    {
      if (RelPtr_->_Next)
      {
        RelPtr_->_Next->SetPrev(NULL);
        EndPt_ = (RelPtr_ != RelPtr_->_Next) ? RelPtr_->_Next:NULL;
        RelPtr_->SetNext(NULL);
      }

      if (MoveFlag_ && RelPtr_ &&
          _HeadTagStack->_Prev &&
          _HeadTagStack->_Prev == RelPtr_)
        if (_HeadTagStack->_Prev->_Prev)
        {
          _HeadTagStack->_Prev->_Prev->SetNext(_HeadTagStack);
          _HeadTagStack->SetPrev(_HeadTagStack->_Prev->_Prev);
        }
        else
          _HeadTagStack->SetPrev(NULL);

      RetPtr_ = RelPtr_;
      RelPtr_ = EndPt_;

      if (LastNode_)
        _InfoPtr = _HeadInfo = NULL;
      else if (!HeadEnd_)
        _InfoPtr = EndPt_;
    }
    else
    {
      if (HeadEnd_)
      {
        if (!LastNode_)
        {
          _HeadInfo = _InfoPtr = SavedPtr_;
          MoveFlag_ += 10;
        }
        else
          _InfoPtr = _HeadInfo;
      }
      else if (!LastNode_)
      {
        _InfoPtr = _HeadInfo = RelPtr_;
        MoveFlag_ += 10;
      }

      RelPtr_ = NULL;
    }

    HeadMoved_ = SavedTagPtr_ != _HeadInfo;

    if (HeadMoved_)
    {
      if (MoveFlag_ < 10)
      {
        if (HeadTagNode_)
          _HeadTagStack = _HeadTagStack->_Next;

        if (HeadTagNode_ && TagNode_)
          _TagStackPtr = _HeadTagStack;
      }
      else
      {
        if (MoveFlag_ == 12 || !_HeadInfo)
          _TagStackPtr = _HeadTagStack = _HeadInfo;
        else if (MoveFlag_ == 11)
          _HeadTagStack = _HeadInfo;
      }
    }
  }
  else if (Erase_)
  {
    if (!LastNode_)
      EndPt_ = _HeadInfo;

    if (!ConfirmErased_)
    {
      while (RelPtr_ && RelPtr_ != EndPt_)
      {
        RelPtr_->SetErased(true);
        RelPtr_ = RelPtr_->_Next;
      }

      if (LastNode_)
        RelPtr_ = _TailInfo;
    }

    _HeadInfo = SavedPtr_;
  }

  return (RetPtr_ ? RetPtr_:RelPtr_);
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::HeadInfo(bool Reset_, bool NotErased_)
{
  StackNodeInfo* PrevPtr_ = _HeadInfo;
  StackNodeInfo* NextPtr_ = _HeadInfo;

  if (_HeadInfo && NotErased_)
  {
    bool TailEnd_ = PrevPtr_ == _TailInfo;

    if (Reset_)
    {
      for (PrevPtr_ = _HeadInfo->_Prev;
           PrevPtr_ && (_HeadInfo = PrevPtr_);
           PrevPtr_ = _HeadInfo->_Prev);
      NextPtr_ = _HeadInfo;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_Next;
      TailEnd_ = PrevPtr_ == _TailInfo;
    }
    while (PrevPtr_ && PrevPtr_->Erased() && !TailEnd_);

    _HeadInfo = PrevPtr_;

    if (_HeadInfo->Erased() && TailEnd_)
    {
      for (PrevPtr_ = _HeadInfo->_Prev;
           PrevPtr_ && (_HeadInfo = PrevPtr_);
           PrevPtr_ = _HeadInfo->_Prev);

      return (_HeadInfo->Erased() ? NULL:_HeadInfo);
    }
  }
  else if (_HeadInfo && Reset_)
  {
    for (PrevPtr_ = _HeadInfo->_Prev;
         PrevPtr_ && (_HeadInfo = PrevPtr_);
         PrevPtr_ = _HeadInfo->_Prev);
  }

  return _HeadInfo;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::TailInfo(bool Reset_, bool NotErased_)
{
  StackNodeInfo* PrevPtr_ = _TailInfo;
  StackNodeInfo* NextPtr_ = _TailInfo;

  if (_TailInfo && NotErased_)
  {
    bool HeadEnd_ = PrevPtr_ == _HeadInfo;

    if (Reset_)
    {
      for (NextPtr_ = _TailInfo->_Next;
           NextPtr_ && (_TailInfo = NextPtr_);
           NextPtr_ = _TailInfo->_Next);
      NextPtr_ = _TailInfo;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_Prev;
      HeadEnd_ = PrevPtr_ == _HeadInfo;
    }
    while (PrevPtr_ && PrevPtr_->Erased() && !HeadEnd_);

    _TailInfo = PrevPtr_;

    if (_TailInfo->Erased() && HeadEnd_)
    {
      for (NextPtr_ = _TailInfo->_Next;
           NextPtr_ && (_TailInfo = NextPtr_);
           NextPtr_ = _TailInfo->_Next);

      return (_TailInfo->Erased() ? NULL:_TailInfo);
    }
  }
  else if (_TailInfo && Reset_)
  {
    for (NextPtr_ = _TailInfo->_Next;
         NextPtr_ && (_TailInfo = NextPtr_);
         NextPtr_ = _TailInfo->_Next);
  }

  return _TailInfo;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::NextStackInfo(bool Reset_, bool Peek_, bool NotErased_)
{
  if (_InfoPtr && !Reset_)
  {
    bool TailEnd_ = _InfoPtr == _TailInfo;
    StackNodeInfo* PrevPtr_ = _InfoPtr;
    StackNodeInfo* NextPtr_ = _InfoPtr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_Next;
    }
    while (NotErased_ && NextPtr_ && NextPtr_->Erased());

    if (Peek_)
      return NextPtr_;
    else
      _InfoPtr = NextPtr_;

    if (TailEnd_)
    {
      _TailInfo = _InfoPtr ? _InfoPtr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _InfoPtr = HeadInfo(true, NotErased_);

  return _InfoPtr;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::PrevStackInfo(bool Reset_, bool Peek_, bool NotErased_)
{
  if (_InfoPtr && !Reset_)
  {
    bool HeadEnd_ = _InfoPtr == _HeadInfo;
    StackNodeInfo* PrevPtr_ = _InfoPtr;
    StackNodeInfo* NextPtr_ = _InfoPtr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = NextPtr_->_Prev;
    }
    while (NotErased_ && NextPtr_ && NextPtr_->Erased());

    if (Peek_)
      return NextPtr_;
    else
      _InfoPtr = NextPtr_;

    if (HeadEnd_)
    {
      _HeadInfo = _InfoPtr ? _InfoPtr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _InfoPtr = TailInfo(true, NotErased_);

  return _InfoPtr;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::SetStackInfoPtr(bool Set_)
{
  _InfoPtrSet = Set_;
  return _InfoPtr;
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::PushStackInfo(StackNodeInfo* Node_, bool Create_, bool Show_)
{
  StackNodeInfo* NodePtr_ = (Create_ && Node_) ? new StackNodeInfo(*Node_):Node_;
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;

  if (NodePtr_)
  {
    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump() && Show_)
      {
        *os_ <<"------------------------------- debug start -------------------------------" <<endl
             <<"Pre PushStackInfo" <<endl;
        StackNodeInfo::ShowStackAddr(_HeadInfo, StackNodeInfo::PREVNEXT, "Info");
      }
    #endif

    NodePtr_->MakeTagElementStates();
    if (Create_ && Node_ && NodePtr_)
      NodePtr_->SetNodeID(Node_->NodeID());

    if (_HeadInfo)
    {
      HeadInfo(true, false);
      _InfoPtr =
      _HeadInfo = (StackNodeInfo*)(_HeadInfo->AppendHead(NodePtr_, false));
      SetStackInfoPtr(false);
    }
    else
      _InfoPtr =
      _HeadInfo =
      _TailInfo = NodePtr_;

    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump() && Show_)
      {
        *os_ <<endl <<"Post PushStackInfo" <<endl;
        StackNodeInfo::ShowStackAddr(_HeadInfo, StackNodeInfo::PREVNEXT, "Info");
        *os_ <<"-------------------------------- debug end --------------------------------"
             <<endl <<endl;
      }
    #endif
  }

  return NodePtr_;
}

/****************************************************************************/
// Append tag content text list related methods
/****************************************************************************/
void TagLinksSequencer::ConfirmAppContStrErased()
{
  AppTextNode* More_ = NULL;
  NextAppText(true, false);
  SetAppTextPtr(true);
  AppTextNode* NextPtr_ = _HeadAppText;

  if (NextPtr_)
  {
    _AppTextPtr = NextPtr_;

    do
    {
      if (More_)
      {
        More_->Unlink();
        More_ = NULL;
      }

      More_ = PopAppContStr(false, true,
                            NULL, NULL, NULL, NULL);
    }
    while (More_ && _AppTextPtr);
  }

  NextAppText(true, false);
  SetAppTextPtr(false);
}

/****************************************************************************/
AppTextNode* TagLinksSequencer::PopAppContStr(bool Erase_, bool ConfirmErased_,
                                              StackNodeInfo* EncNode_,
                                              StackNodeInfo* ChkNode_,
                                              bool* NodeErased_,
                                              bool* WasErased_)
{
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;
  AppTextNode* ChkPtr_ = NULL;
  AppTextNode* RetPtr_ = NULL;
  AppTextNode* EndPt_ = NULL;
  AppTextNode* SavedPtr_ = NULL;
  AppTextNode* RelPtr_ = (_AppTextPtr && _AppTextPtrSet) ?
                               _AppTextPtr:
                               (_AppTextPtr=HeadAppText(true, Erase_ && !ConfirmErased_));

  bool HeadEnd_ = _AppTextPtr == _HeadAppText;
  bool LastNode_ = false;
  bool TagError_ = false;

  if (!RelPtr_)
    return NULL;

  if (!_AppTextPtrSet && _AppTextPtr && Erase_ && !ConfirmErased_)
    SetAppTextPtr(true);

  if (HeadEnd_)
  {
    SavedPtr_ = _AppTextPtr;
    _AppTextPtr = (AppTextNode*)(_AppTextPtr->GiveNext());
    LastNode_ = _AppTextPtr == NULL;

    if (!LastNode_)
      _HeadAppText = _AppTextPtr;
  }
  else
  {
    SavedPtr_ = RelPtr_;
    _HeadAppText = (AppTextNode*)(RelPtr_->GiveNext());
    LastNode_ = _HeadAppText == NULL;

    if (LastNode_)
      _AppTextPtr = _HeadAppText = RelPtr_;
  }

  if (ConfirmErased_)
  {
    if ((RelPtr_->Erased() || RelPtr_->DoErase()) && !RelPtr_->IsCopied())
    {
      if (RelPtr_->Next())
      {
        RelPtr_->GiveNext()->SetPrev(NULL);
        EndPt_ = (RelPtr_ != RelPtr_->Next()) ? (AppTextNode*)(RelPtr_->GiveNext()):NULL;
        RelPtr_->SetNext(NULL);
      }

      RetPtr_ = RelPtr_;
      RelPtr_ = EndPt_;

      if (LastNode_)
        _AppTextPtr = _HeadAppText = NULL;
      else if (!HeadEnd_)
        _AppTextPtr = EndPt_;
    }
    else
    {
      if (HeadEnd_)
      {
        if (!LastNode_)
          _HeadAppText = _AppTextPtr = SavedPtr_;
        else
          _AppTextPtr = _HeadAppText;
      }
      else if (!LastNode_)
        _AppTextPtr = _HeadAppText = RelPtr_;

      RelPtr_ = NULL;
    }
  }
  else if (Erase_)
  {
    if (NodeErased_)
      *NodeErased_ = false;

    if (!LastNode_)
      EndPt_ = _HeadAppText;

    if (EncNode_)
    {
      if (!ConfirmErased_)
      {
        while (RelPtr_ && EncNode_ != RelPtr_->RefNode())
          RelPtr_ = (AppTextNode*)(RelPtr_->GiveNext());

        if ((!RelPtr_ && ChkNode_ && EncNode_) ||
              (ChkNode_ && EncNode_ && RelPtr_->RefNode() &&
               RelPtr_->RefNode() != EncNode_))
        {
          *os_ <<endl <<"*** TagLinksSequencer::PopAppContStr() ChkNode Failed" <<endl;
          StackNodeInfo::ShowStackAddr(EncNode_, StackNodeInfo::APPTEXT, "AppText");
          if (RelPtr_)
            StackNodeInfo::ShowStackAddr(RelPtr_, StackNodeInfo::APPTEXT, "AppText");
          *os_ <<"****************************************" <<endl <<endl;

          TagError_ = true;
          if (_Parent)
            _Parent->SetTagErrors(POPAPPCONTSTR_CHKNODE, 0);
        }
        else
          ChkNode_ = NULL;

        if (!ChkNode_)
        {
          ChkPtr_ = (RelPtr_ && EncNode_ == RelPtr_->RefNode()) ? RelPtr_:NULL;

          if (ChkPtr_)
          {
            if (!RelPtr_->Erased() && !RelPtr_->DoErase())
              RelPtr_->SetDoErase(true);
            else if (WasErased_)
              *WasErased_ = true;
          }

          TagError_ = !ChkPtr_ || !(RelPtr_->Erased() || RelPtr_->DoErase());

          if (NodeErased_)
            *NodeErased_ = !TagError_;

          if (!ChkPtr_ && WasErased_)
            *WasErased_ = false;

          if (TagError_ && _Parent)
            _Parent->SetTagErrors(POPAPPCONTSTR_CHKNODE, 0);
        }
        else
        {
          if (NodeErased_)
            *NodeErased_ = false;

          if (WasErased_)
            *WasErased_ = false;
        }
      }
    }
    else
    {
      if (!ConfirmErased_)
      {
        // Remove all preceeding NonContentTypeTag Nodes
        //
        while (RelPtr_ && RelPtr_->RefNode() &&
               RelPtr_->RefNode()->IsNonContentTypeTag())
        {
          RelPtr_->SetErased(true);
          RelPtr_ = (AppTextNode*)(RelPtr_->GiveNext());

          if (RelPtr_ == EndPt_ && !LastNode_)
          {
            _HeadAppText = (AppTextNode*)(RelPtr_->GiveNext());
            LastNode_ = _HeadAppText == NULL;
            EndPt_ = LastNode_ ? NULL:_HeadAppText;

            if (LastNode_)
              _AppTextPtr = _HeadAppText = RelPtr_;
          }
        }

        // Report error if passed ChkNode does not equal head of
        // AppTextNode stack
        //
        if ((!RelPtr_ && ChkNode_) ||
              (ChkNode_ && RelPtr_->RefNode() &&
               RelPtr_->RefNode() != ChkNode_))
        {
          *os_ <<endl <<"*** TagLinksSequencer::PopAppContStr() ChkNode Failed" <<endl;
          StackNodeInfo::ShowStackAddr(ChkNode_, StackNodeInfo::APPTEXT, "AppText");
          if (RelPtr_)
            StackNodeInfo::ShowStackAddr(RelPtr_, StackNodeInfo::APPTEXT, "AppText");
          *os_ <<"****************************************" <<endl <<endl;

          TagError_ = true;
          if (_Parent)
            _Parent->SetTagErrors(POPAPPCONTSTR_CHKNODE, 0);
        }
        else
          ChkNode_ = NULL;

        if (!ChkNode_)
        {
          ChkPtr_ = (RelPtr_ && RelPtr_ != EndPt_) ? RelPtr_:NULL;

          if (ChkPtr_)
          {
            if (WasErased_)
              *WasErased_ = false;

            while (RelPtr_ && RelPtr_ != EndPt_)
            {
              if (!RelPtr_->Erased())
                RelPtr_->SetErased(true);
              else if (WasErased_)
                *WasErased_ = true;

              RelPtr_ = (AppTextNode*)(RelPtr_->GiveNext());
            }
          }

          TagError_ = (ChkPtr_ && RelPtr_ && (RelPtr_ == ChkPtr_) && !RelPtr_->Erased()) ||
                      (!ChkPtr_ && (!RelPtr_ || !RelPtr_->Erased()));

          if (NodeErased_)
            *NodeErased_ = (ChkPtr_ && RelPtr_) ? ((RelPtr_ != ChkPtr_) || RelPtr_->Erased()):
                           RelPtr_ ? RelPtr_->Erased():false;

          if (WasErased_)
          {
            if (ChkPtr_ && RelPtr_ && RelPtr_ == ChkPtr_)
              *WasErased_ = RelPtr_->Erased();
            else if (!ChkPtr_)
            {
              if (RelPtr_)
                *WasErased_ = RelPtr_->Erased();
              else
                *WasErased_ = false;
            }
          }

          if (TagError_ && _Parent)
            _Parent->SetTagErrors(POPAPPCONTSTR_CHKNODE, 0);
        }
        else
        {
          if (NodeErased_)
            *NodeErased_ = false;

          if (WasErased_)
            *WasErased_ = false;
        }

        if (LastNode_)
          RelPtr_ = _TailAppText;
      }
    }

    _HeadAppText = SavedPtr_;
  }

  return (RetPtr_ ? RetPtr_:RelPtr_);
}

/****************************************************************************/
AppTextNode* TagLinksSequencer::HeadAppText(bool Reset_, bool NotErased_)
{
  AppTextNode* PrevPtr_ = _HeadAppText;
  AppTextNode* NextPtr_ = _HeadAppText;

  if (_HeadAppText && NotErased_)
  {
    bool TailEnd_ = PrevPtr_ == _TailAppText;

    if (Reset_)
    {
      for (PrevPtr_ = (AppTextNode*)(_HeadAppText->GivePrev());
           PrevPtr_ && (_HeadAppText = PrevPtr_);
           PrevPtr_ = (AppTextNode*)(_HeadAppText->GivePrev()));
      NextPtr_ = _HeadAppText;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (AppTextNode*)(NextPtr_->GiveNext());
      TailEnd_ = PrevPtr_ == _TailAppText;
    }
    while (PrevPtr_ && PrevPtr_->Erased() && !TailEnd_);

    _HeadAppText = PrevPtr_;

    if (_HeadAppText->Erased() && TailEnd_)
    {
      for (PrevPtr_ = (AppTextNode*)(_HeadAppText->GivePrev());
           PrevPtr_ && (_HeadAppText = PrevPtr_);
           PrevPtr_ = (AppTextNode*)(_HeadAppText->GivePrev()));

      return (_HeadAppText->Erased() ? NULL:_HeadAppText);
    }
  }
  else if (_HeadAppText && Reset_)
  {
    for (PrevPtr_ = (AppTextNode*)(_HeadAppText->GivePrev());
         PrevPtr_ && (_HeadAppText = PrevPtr_);
         PrevPtr_ = (AppTextNode*)(_HeadAppText->GivePrev()));
  }

  return _HeadAppText;
}

/****************************************************************************/
AppTextNode* TagLinksSequencer::TailAppText(bool Reset_, bool NotErased_)
{
  AppTextNode* PrevPtr_ = _TailAppText;
  AppTextNode* NextPtr_ = _TailAppText;

  if (_TailAppText && NotErased_)
  {
    bool HeadEnd_ = PrevPtr_ == _HeadAppText;

    if (Reset_)
    {
      for (NextPtr_ = (AppTextNode*)(_TailAppText->GiveNext());
           NextPtr_ && (_TailAppText = NextPtr_);
           NextPtr_ = (AppTextNode*)(_TailAppText->GiveNext()));
      NextPtr_ = _TailAppText;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (AppTextNode*)(NextPtr_->GivePrev());
      HeadEnd_ = PrevPtr_ == _HeadAppText;
    }
    while (PrevPtr_ && PrevPtr_->Erased() && !HeadEnd_);

    _TailAppText = PrevPtr_;

    if (_TailAppText->Erased() && HeadEnd_)
    {
      for (NextPtr_ = (AppTextNode*)(_TailAppText->GiveNext());
           NextPtr_ && (_TailAppText = NextPtr_);
           NextPtr_ = (AppTextNode*)(_TailAppText->GiveNext()));

      return (_TailAppText->Erased() ? NULL:_TailAppText);
    }
  }
  else if (_TailAppText && Reset_)
  {
    for (NextPtr_ = (AppTextNode*)(_TailAppText->GiveNext());
         NextPtr_ && (_TailAppText = NextPtr_);
         NextPtr_ = (AppTextNode*)(_TailAppText->GiveNext()));
  }

  return _TailAppText;
}

/****************************************************************************/
AppTextNode* TagLinksSequencer::NextAppText(bool Reset_, bool Peek_, bool NotErased_)
{
  if (_AppTextPtr && !Reset_)
  {
    bool TailEnd_ = _AppTextPtr == _TailAppText;
    AppTextNode* PrevPtr_ = _AppTextPtr;
    AppTextNode* NextPtr_ = _AppTextPtr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (AppTextNode*)(NextPtr_->GiveNext());
    }
    while (NotErased_ && NextPtr_ && NextPtr_->Erased());

    if (Peek_)
      return NextPtr_;
    else
      _AppTextPtr = NextPtr_;

    if (TailEnd_)
    {
      _TailAppText = _AppTextPtr ? _AppTextPtr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _AppTextPtr = HeadAppText(true, NotErased_);

  return _AppTextPtr;
}

/****************************************************************************/
AppTextNode* TagLinksSequencer::PrevAppText(bool Reset_, bool Peek_, bool NotErased_)
{
  if (_AppTextPtr && !Reset_)
  {
    bool HeadEnd_ = _AppTextPtr == _HeadAppText;
    AppTextNode* PrevPtr_ = _AppTextPtr;
    AppTextNode* NextPtr_ = _AppTextPtr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (AppTextNode*)(NextPtr_->GivePrev());
    }
    while (NotErased_ && NextPtr_ && NextPtr_->Erased());

    if (Peek_)
      return NextPtr_;
    else
      _AppTextPtr = NextPtr_;

    if (HeadEnd_)
    {
      _HeadAppText = _AppTextPtr ? _AppTextPtr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _AppTextPtr = TailAppText(true, NotErased_);

  return _AppTextPtr;
}

/****************************************************************************/
AppTextNode* TagLinksSequencer::SetAppTextPtr(bool Set_)
{
  _AppTextPtrSet = Set_;
  return _AppTextPtr;
}

/****************************************************************************/
AppTextNode* TagLinksSequencer::PushAppContStr(bool HasTagStk_, StackNodeInfo* Node_)
{
  AppTextNode* NodePtr_ = Node_ ? Node_->MakeAppTextNode(HasTagStk_):NULL;
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;

  if (NodePtr_)
  {
    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump())
      {
        *os_ <<"------------------------------- debug start -------------------------------" <<endl
             <<"Pre PushAppContStr" <<endl;
        StackNodeInfo::ShowStackAddr(_HeadAppText, StackNodeInfo::APPTEXT, "AppText");
      }
    #endif

    if (_HeadAppText)
    {
      HeadAppText(true, false);
      _AppTextPtr =
      _HeadAppText = (AppTextNode*)(_HeadAppText->AppendHead(NodePtr_, false));
      SetAppTextPtr(false);
    }
    else
      _AppTextPtr =
      _HeadAppText =
      _TailAppText = NodePtr_;

    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump())
      {
        *os_ <<endl <<"Post PushAppContStr" <<endl;
        StackNodeInfo::ShowStackAddr(_HeadAppText, StackNodeInfo::APPTEXT, "AppText");
        *os_ <<"-------------------------------- debug end --------------------------------"
             <<endl <<endl;
      }
    #endif
  }

  return NodePtr_;
}

/****************************************************************************/
AppTextNode* TagLinksSequencer::PushAppTentStr(bool HasTentStk_, StackNodeInfo* Node_)
{
  AppTextNode* NodePtr_ = Node_ ? Node_->MakeAppTentTextNode(HasTentStk_):NULL;
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;

  if (NodePtr_)
  {
    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump())
      {
        *os_ <<"------------------------------- debug start -------------------------------" <<endl
             <<"Pre PushAppTentStr" <<endl;
        StackNodeInfo::ShowStackAddr(_HeadAppText, StackNodeInfo::APPTEXT, "AppText");
      }
    #endif

    if (_HeadAppText)
    {
      HeadAppText(true, false);
      _AppTextPtr =
      _HeadAppText = (AppTextNode*)(_HeadAppText->AppendHead(NodePtr_, false));
      SetAppTextPtr(false);
    }
    else
      _AppTextPtr =
      _HeadAppText =
      _TailAppText = NodePtr_;

    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump())
      {
        *os_ <<endl <<"Post PushAppTentStr" <<endl;
        StackNodeInfo::ShowStackAddr(_HeadAppText, StackNodeInfo::APPTEXT, "AppText");
        *os_ <<"-------------------------------- debug end --------------------------------"
             <<endl <<endl;
      }
    #endif
  }

  return NodePtr_;
}

/****************************************************************************/
void TagLinksSequencer::SetOpStTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                                     StackNodeInfo* EncNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     SeqNum==Ume.BackLink->SeqNum+1;
  //     CurLevel==Ume.CurLevel;
  //     EncLevel==0;
  //   else
  //     SeqNum++;
  //     CurLevel==St.CurLevel;
  //     EncLevel==0;
  //   endif
  //
  //   Tag=HeadAppText(true,true);
  //   OpSt.PopAppContStr(1);
  //   OpSt.AppContStr=OpSt.HeadAppContStr();
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Tag->_TransferNode && Ume==Tag then
  //     Ume.AppendTentToContStr(1);
  //   endif
  //   if (OpSt.AtReqTagCondStopState() &&
  //       OpSt.TagAtImpliedEnd()) then
  //     OpSt.PushAppTentStr(true,OpSt);
  //   endif
  // endif

  AppTextNode* NodePtr_ = NULL;
  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);
  bool NodePopped_ = false;

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetOpStTagToUnmatchedEndData(CurNode_);
    NodePtr_ = HeadAppText(true, true);

    if ((CurNode_->AtReqTagCondStopState(false) &&
         CurNode_->RequiredTagCondMet()) ||
        (CurNode_->NoRequiredTag() &&
         CurNode_->CurrentTagCondMet()))
      NodePopped_ = CurNode_->PopAppContStr(true);

    CurNode_->AppText()->SetAppendText();
    EncNode_->AppText()->SetAppendText();

    if (EncNode_->BackLink() &&
        EncNode_->BackLink()->AppText())
      EncNode_->BackLink()->AppText()->SetAppendText();

    if (NodePtr_ && NodePtr_->HasTentativeStr())
    {
      NodePtr_ = FindAppTextNode(NodePtr_->GiveRefNode());

      if (NodePtr_ && NodePtr_->RefNode() == EncNode_)
        NodePtr_->TransferText();
    }

    if (NodePopped_ &&
          ((CurNode_->AtReqTagCondStopState(false) &&
            CurNode_->RequiredTagCondMet()) ||
           (CurNode_->NoRequiredTag() &&
            CurNode_->CurrentTagCondMet())))
      PushAppTentStr(true, CurNode_);
  }
}

/****************************************************************************/
void TagLinksSequencer::SetStdTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                                    StackNodeInfo* EncNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     SeqNum==Ume.BackLink->SeqNum+1;
  //     CurLevel==Ume.CurLevel;
  //     EncLevel==0;
  //   else
  //     SeqNum++;
  //     CurLevel==St.CurLevel;
  //     EncLevel==0;
  //   endif
  //
  //   Tag=HeadAppText(true,true);
  //   St.PopAppContStr(0);
  //   St.AppContStr=St.HeadAppContStr();
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Tag->_TransferNode && Ume==Tag then
  //     Ume.AppendTentToContStr(1);
  //   endif
  // endif

  AppTextNode* NodePtr_ = NULL;
  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetStdTagToUnmatchedEndData(CurNode_);
    NodePtr_ = HeadAppText(true, true);
    CurNode_->PopAppContStr(false, EncNode_);

    CurNode_->AppText()->SetAppendText();
    EncNode_->AppText()->SetAppendText();

    if (EncNode_->BackLink() &&
        EncNode_->BackLink()->AppText())
      EncNode_->BackLink()->AppText()->SetAppendText();

    if (NodePtr_ && NodePtr_->HasTentativeStr())
    {
      NodePtr_ = FindAppTextNode(NodePtr_->GiveRefNode());

      if (NodePtr_ && NodePtr_->RefNode() == EncNode_)
        NodePtr_->TransferText();
    }
  }
}

/****************************************************************************/
void TagLinksSequencer::SetSglTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                                    StackNodeInfo* EncNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     SeqNum==Ume.BackLink->SeqNum+1;
  //     CurLevel==Ume.CurLevel;
  //     EncLevel==0;
  //   else
  //     SeqNum++;
  //     CurLevel==St.CurLevel;
  //     EncLevel==0;
  //   endif
  //
  //   Tag=HeadAppText(true,true);
  //   Sgl.PopAppContStr(1);
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Tag->_TransferNode && Ume==Tag then
  //     Ume.AppendTentToContStr(1);
  //   endif
  // endif

  AppTextNode* NodePtr_ = NULL;
  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetSglTagToUnmatchedEndData(CurNode_);
    NodePtr_ = HeadAppText(true, true);

    if (CurNode_->SingletonTagFound())
      CurNode_->PopAppContStr(true);

    EncNode_->AppText()->SetAppendText();

    if (EncNode_->BackLink() &&
        EncNode_->BackLink()->AppText())
      EncNode_->BackLink()->AppText()->SetAppendText();

    if (NodePtr_ && NodePtr_->HasTentativeStr())
    {
      NodePtr_ = FindAppTextNode(NodePtr_->GiveRefNode());

      if (NodePtr_ && NodePtr_->RefNode() == EncNode_)
        NodePtr_->TransferText();
    }
  }
}

/****************************************************************************/
void TagLinksSequencer::SetMatchedEndTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                                           StackNodeInfo* EncNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     SeqNum==Ume.BackLink->SeqNum+1;
  //     CurLevel==Ume.CurLevel;
  //     EncLevel==0;
  //   else
  //     SeqNum++;
  //     CurLevel==St.CurLevel;
  //     EncLevel==0;
  //   endif
  //
  //   Tag=HeadAppText(true,true);
  //   Pe.PopAppContStr(0);
  //   Pe.AppContStr=Pe.HeadAppContStr();
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Tag->_TransferNode && Ume==Tag then
  //     Ume.AppendTentToContStr(1);
  //   endif
  // endif

  AppTextNode* NodePtr_ = NULL;
  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetMatchedEndTagToUnmatchedEndData(CurNode_);
    NodePtr_ = HeadAppText(true, true);
    CurNode_->PopAppContStr(false, EncNode_);

    CurNode_->AppText()->SetAppendText();
    EncNode_->AppText()->SetAppendText();

    if (EncNode_->BackLink() &&
        EncNode_->BackLink()->AppText())
      EncNode_->BackLink()->AppText()->SetAppendText();

    if (NodePtr_ && NodePtr_->HasTentativeStr())
    {
      NodePtr_ = FindAppTextNode(NodePtr_->GiveRefNode());

      if (NodePtr_ && NodePtr_->RefNode() == EncNode_)
        NodePtr_->TransferText();
    }
  }
}

/****************************************************************************/
void TagLinksSequencer::SetOpStTagToOpStTagData(StackNodeInfo* CurNode_,
                                                StackNodeInfo* EncNode_)
{
  // SeqNum==0;
  // CurLevel==St->EncLevel;
  // IncNestLevel(); EncLevel==CurLevel++;
  //
  // OpSt.PopAppContStr(1);   // 1st param indicates string termination
  // OpSt_.PushAppContStr(HeadAppText,0);
  // OpSt_.AppContStr=OpSt_.HeadAppContStr();
  //   if (OpSt.AtReqTagCondStopState() &&
  //       OpSt.TagAtImpliedEnd()) then
  //     OpSt.PushAppTentStr(true,OpSt);
  //   endif

  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);
  bool NodePopped_ = false;

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetOpStTagToOpStTagData(CurNode_, NodePopped_);
    PushAppContStr(_HeadAppText != NULL, EncNode_);
    EncNode_->AppText()->SetAppendText();

    if (NodePopped_ &&
          ((CurNode_->AtReqTagCondStopState(false) &&
            CurNode_->RequiredTagCondMet()) ||
           (CurNode_->NoRequiredTag() &&
            CurNode_->CurrentTagCondMet())))
      PushAppTentStr(true, CurNode_);
  }
}

/****************************************************************************/
void TagLinksSequencer::SetSglTagToSglTagData(StackNodeInfo* CurNode_,
                                              StackNodeInfo* EncNode_)
{
  // SeqNum++;
  // CurLevel==Sgl->CurLevel;
  // EncLevel==0;
  // Sgl.PopAppContStr(1);
  // Sgl_.PushAppContStr(HeadAppText,1);
  // Sgl_.AppContStr=Sgl_.HeadAppContStr();

  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetSglTagToSglTagData(CurNode_);
    PushAppContStr(_HeadAppText != NULL, EncNode_);
    EncNode_->AppText()->SetAppendText();
  }
}

/****************************************************************************/
void TagLinksSequencer::SetSglTagToMatchedEndData(StackNodeInfo* CurNode_,
                                                  StackNodeInfo* EncNode_)
{
  // SeqNum++;
  // CurLevel==Sgl->CurLevel;
  // EncLevel==0;
  // Sgl.PopAppContStr(1);
  // Me.AppContStr=Me.HeadAppContStr();
  // Me->BackLink.AppContStr=Me->BackLink.HeadAppContStr();

  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetSglTagToMatchedEndData(CurNode_);
    EncNode_->AppText()->SetAppendText();

    if (EncNode_->BackLink() &&
        EncNode_->BackLink()->AppText())
      EncNode_->BackLink()->AppText()->SetAppendText();
    else if (CurNode_ && CurNode_->AppText())
      CurNode_->AppText()->SetAppendText();
  }
}

/****************************************************************************/
void TagLinksSequencer::SetStdTagToMatchedEndData(StackNodeInfo* CurNode_,
                                                  StackNodeInfo* EncNode_)
{
  // if (Me->BackLink)
  //   SeqNum==Me->BackLink->SeqNum+1;
  //   CurLevel==Me->CurLevel;
  //   EncLevel==0;
  // else
  //   SeqNum++;
  //   CurLevel==Pe.CurLevel;
  //   EncLevel==0;
  // endif
  //
  // St.PopAppContStr(1);
  // Me.AppContStr=Me.HeadAppContStr();
  // Me->BackLink.AppContStr=Me->BackLink.HeadAppContStr();

  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetStdTagToMatchedEndData(CurNode_);
    EncNode_->AppText()->SetAppendText();

    if (EncNode_->BackLink() &&
        EncNode_->BackLink()->AppText())
      EncNode_->BackLink()->AppText()->SetAppendText();
    else if (CurNode_ && CurNode_->AppText())
      CurNode_->AppText()->SetAppendText();
  }
}

/****************************************************************************/
void TagLinksSequencer::SetXToSglTagData(StackNodeInfo* CurNode_,
                                         StackNodeInfo* EncNode_)
{
  // SeqNum++;
  // CurLevel==X->CurLevel;
  // EncLevel==0;
  // Sgl.PushAppContStr(HeadAppText,1);
  // Sgl.AppContStr=Sgl.HeadAppContStr();

  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetXToSglTagData(CurNode_);
    PushAppContStr(true, EncNode_);
    EncNode_->AppText()->SetAppendText();
  }
}

/****************************************************************************/
void TagLinksSequencer::SetXToStdTagData(StackNodeInfo* CurNode_,
                                         StackNodeInfo* EncNode_)
{
  // SeqNum++;
  // CurLevel==X->CurLevel;
  // IncNestLevel(); EncLevel==CurLevel++;
  //   if (X == SglTag)
  //     SglTag.PopAppContStr(1);
  // St.PushAppContStr(HeadAppText,0);
  // St.AppContStr=St.HeadAppContStr();

  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetXToStdTagData(CurNode_);
    PushAppContStr(true, EncNode_);
    EncNode_->AppText()->SetAppendText();
  }
}

/****************************************************************************/
void TagLinksSequencer::SetStdTagToStdTagData(StackNodeInfo* CurNode_,
                                              StackNodeInfo* EncNode_)
{
  // SeqNum==0;
  // CurLevel==St->EncLevel;
  // IncNestLevel(); EncLevel==CurLevel++;
  // St_.PushAppContStr(HeadAppText,0);
  // St_.AppContStr=St_.HeadAppContStr();

  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetStdTagToStdTagData(CurNode_);
    PushAppContStr(_HeadAppText != NULL, EncNode_);
    EncNode_->AppText()->SetAppendText();
  }
}

/****************************************************************************/
void TagLinksSequencer::SetStdTagToSglTagData(StackNodeInfo* CurNode_,
                                              StackNodeInfo* EncNode_)
{
  // SeqNum==0;
  // CurLevel==St->EncLevel;
  // EncLevel==0;
  // Sgl.PushAppContStr(HeadAppText,1);
  // Sgl.AppContStr=Sgl.HeadAppContStr();

  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetStdTagToSglTagData(CurNode_);
    PushAppContStr(_HeadAppText != NULL, EncNode_);
    EncNode_->AppText()->SetAppendText();
  }
}

/****************************************************************************/
void TagLinksSequencer::SetStdTagData(StackNodeInfo* CurNode_)
{
  // SeqNum==0;
  // CurLevel==0;
  // IncNestLevel(); EncLevel==CurLevel++;
  // St.PushAppContStr(HeadAppText,0);
  // St.AppContStr=St.HeadAppContStr();

  StackNodeInfo* sCurNode_ = CurNode_;
  CurNode_ = FindInfoNode(CurNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (CurNode_)
  {
    CurNode_->SetStdTagData();
    PushAppContStr(true, CurNode_);
    CurNode_->AppText()->SetAppendText();
  }
}

/****************************************************************************/
void TagLinksSequencer::SetSingleTagData(StackNodeInfo* CurNode_)
{
  // SeqNum==0;
  // CurLevel==0;
  // EncLevel==0;
  // Sgl.PushAppContStr(HeadAppText,0);
  // Sgl.AppContStr=Sgl.HeadAppContStr();

  StackNodeInfo* sCurNode_ = CurNode_;
  CurNode_ = FindInfoNode(CurNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (CurNode_)
  {
    CurNode_->SetSingleTagData();
    PushAppContStr(true, CurNode_);
    CurNode_->AppText()->SetAppendText();
  }
}

/****************************************************************************/
void TagLinksSequencer::SetNonContentTagData(StackNodeInfo* CurNode_)
{
  // ContentStr+="<!--[if lt IE 7]> // NC Tag
  //                  <style>div.page img.alpha{display:none}</style>
  //              <![endif]-->"
  //
  // SeqNum==0;
  // CurLevel==0;
  // EncLevel==0;

  StackNodeInfo* sCurNode_ = CurNode_;
  CurNode_ = FindInfoNode(CurNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (CurNode_)
  {
    CurNode_->SetNonContentTagData();
    PushAppContStr(true, CurNode_);
    CurNode_->AppText()->SetAppendText(false);
  }
}

/****************************************************************************/
void TagLinksSequencer::SetNonContentTagData(StackNodeInfo* CurNode_,
                                             StackNodeInfo* EncNode_)
{
  // ContentStr+="<!--[if lt IE 7]> // NC Tag
  //                  <style>div.page img.alpha{display:none}</style>
  //              <![endif]-->"
  //
  // SeqNum++;
  // CurLevel==CurNode->CurLevel;
  // EncLevel==0;

  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && EncNode_)
  {
    EncNode_->SetNonContentTagData(CurNode_);
    PushAppContStr(true, EncNode_);
    EncNode_->AppText()->SetAppendText(false);
  }
}

/****************************************************************************/
AppTextNode* TagLinksSequencer::FindAppTextNode(StackNodeInfo* Node_)
{
  return (_HeadAppText ? _HeadAppText->FindTagNode(Node_):NULL);
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::FindInfoNode(StackNodeInfo* Node_)
{
  return (_HeadInfo ? _HeadInfo->FindTagNode(Node_):NULL);
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::FindCursorTagNode(StackNodeInfo* Node_)
{
  return (_HeadCursorTagStk ? _HeadCursorTagStk->FindTagNode(Node_):NULL);
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::FindTagStackNode(StackNodeInfo* Node_)
{
  return (_HeadTagStack ? _HeadTagStack->FindTagNode(Node_):NULL);
}

/****************************************************************************/
/****************************************************************************/
StackNodeInfo* TagLinksSequencer::RecurseAppend(StackNodeInfo* CurNode_,
                                                StackNodeInfo* EncNode_, bool Shifted_)
{
  StackNodeInfo* sCurNode_ = CurNode_;
  StackNodeInfo* sEncNode_ = EncNode_;
  CurNode_ = FindInfoNode(CurNode_);
  EncNode_ = FindInfoNode(EncNode_);

  if (!CurNode_)
    CurNode_ = sCurNode_;

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (CurNode_ && CurNode_->_Parent &&
      CurNode_->Next() &&
      CurNode_->IsContentAppended() &&
      !CurNode_->IsNonContentTypeTag() &&
      !CurNode_->IsContentTransferred() &&
        (CurNode_->SingletonTagFound() ||
         CurNode_->EmptyTagFound()))
    RecurseAppend((StackNodeInfo*)(CurNode_->GiveNext()), CurNode_, true);

  if (CurNode_ && CurNode_->_Parent &&
      EncNode_ && CurNode_ != EncNode_ &&
      !CurNode_->PairEnderFound() &&
      !EncNode_->PairEnderFound() &&
      !CurNode_->IsNonContentTypeTag() &&
      CurNode_->AppText()->AppendText() &&
      CurNode_->AppText()->TextAppendNotPaused())
    CurNode_->AppText()->AppendNestedNode(EncNode_, Shifted_);

  if (CurNode_ && EncNode_ && CurNode_ != EncNode_)
    return RecurseAppend((StackNodeInfo*)(CurNode_->GiveNext()), EncNode_, Shifted_);

  return EncNode_;
}

/****************************************************************************/
void TagLinksSequencer::ResetLinks()
{
  HeadInfo(true, false);
  StackNodeInfo* Ptr_ = NULL;

  for (Ptr_ = _HeadInfo; Ptr_; Ptr_ = Ptr_->_Next)
  {
    Ptr_->SetErased(false);
    Ptr_->SetUnlinked(false);
  }
}

/****************************************************************************/
void TagLinksSequencer::AppendNestingError(int Value_)
{
  _NestingError |= Value_;
}

/****************************************************************************/
void TagLinksSequencer::SetNestingError(int Value_)
{
  _NestingError = Value_;
}

/****************************************************************************/
void TagLinksSequencer::ResetNestingError()
{
  _NestingError = 0;
}

/****************************************************************************/
void TagLinksSequencer::AppendTagNestCond(int Value_)
{
  _TagNestCond |= Value_;
  _NCTagNestCond = 0;
}

/****************************************************************************/
void TagLinksSequencer::SetTagNestCond(int Value_)
{
  if (Value_ == NCTAG_NESTCOND)
    _NCTagNestCond = Value_;
  else
  {
    _TagNestCond = Value_;
    _NCTagNestCond = 0;
  }
}

/****************************************************************************/
void TagLinksSequencer::ResetTagNestCond()
{
  _TagNestCond = 0;
  _NCTagNestCond = 0;
}

/****************************************************************************/
Ulong TagLinksSequencer::SearchRunNumber() const
{
  return ((_HeadInfo && _SearchRunNumber) ? _SearchRunNumber:0);
}

/****************************************************************************/
void TagLinksSequencer::ResetSearchRunNumber()
{
  _SearchRunNumber = 0;
  _ExecTagActionFails = 0;
  DestroyAllStacks();
}

/****************************************************************************/
void TagLinksSequencer::ResetExecTagActionFails()
{
  _ExecTagActionFails = 0;
}

/****************************************************************************/
long TagLinksSequencer::ExecTagActionFails() const
{
  return ((_HeadInfo && _ExecTagActionFails) ? _ExecTagActionFails:0);
}

/****************************************************************************/
void TagLinksSequencer::SetStkNodeSearchData(StackNodeInfo* CurNode_)
{
  if (CurNode_)
  {
    _CursorStkTag = CurNode_->_CursorStkTag;
    _FoundTag = CurNode_->_FoundTag;
    _FoundCursorTag = CurNode_->_FoundCursorTag;
    _StkTagFound = CurNode_->_StkTagFound;
  }
}

/****************************************************************************/
void TagLinksSequencer::DestroyAllStacks()
{
  HeadTagStack(true, false);
  delete _HeadTagStack;
  _TagStackPtr = _TailTagStack = _HeadTagStack = NULL;

  HeadInfo(true, false);
  delete _HeadInfo;
  _InfoPtr = _TailInfo = _HeadInfo = NULL;

  HeadAppText(true, false);
  delete _HeadAppText;
  _AppTextPtr = _TailAppText = _HeadAppText = NULL;

  _CursorTagStkPtr = _TailCursorTagStk = _HeadCursorTagStk = NULL;
  ErasePrevTagInfo();
}

/****************************************************************************/
const StackNodeInfo* TagLinksSequencer::PrevTagInfoNode() const
{
  return (_PrevTagInfo ? _PrevTagInfo->_InfoNode:NULL);
}

/****************************************************************************/
StackNodeInfo* TagLinksSequencer::PrevTagInfoNode()
{
  return (_PrevTagInfo ? _PrevTagInfo->_InfoNode:NULL);
}

/****************************************************************************/
bool TagLinksSequencer::PrevTagInfoSet() const
{
  return (_PrevTagInfo ? _PrevTagInfo->_TagInfoSet:false);
}

/****************************************************************************/
void TagLinksSequencer::ErasePrevTagInfo()
{
  delete _PrevTagInfo;
  _PrevTagInfo = NULL;
}

/****************************************************************************/
void TagLinksSequencer::ResetPrevTagInfo()
{
  if (!_PrevTagInfo)
    _PrevTagInfo = new TagInfoRecord;

  _PrevTagInfo->Reset();
}

/****************************************************************************/
void TagLinksSequencer::CompareWithPrevTagInfo(StackNodeInfo* EncNode_, bool PairEnderFound_)
{
  if (EncNode_ && EncNode_->_TagFound && EncNode_->_ContentTypeTag &&
      _PrevTagInfo && _PrevTagInfo->_TagInfoSet && _PrevTagInfo->_ContentTypeTag)
  {
    _PrevTagInfo->_MatchingTags =
        _PrevTagInfo->_TagName.strlen() > 0 &&
        !_PrevTagInfo->_PairEnderFound && PairEnderFound_ &&
        _PrevTagInfo->_TagName.strlen() == EncNode_->_TagElementStr.strlen() &&
        _PrevTagInfo->_TagName.StriComp(EncNode_->_TagElementStr) == 0;
  }
}

/****************************************************************************/
MEMORYOPS_DEFN(TagLinksSequencer)

/****************************************************************************/
// TagInfoRecord class definition
/****************************************************************************/
TagInfoRecord::TagInfoRecord()
{
  Reset();
}

/****************************************************************************/
void TagInfoRecord::Reset()
{
  _TagAttr             =
  _TagTypeFound        = 0;

  _NestingError        =
  _TagNestCond         = 0L;
  _NCTagNestCond       = 0L;

  _TagFound            =
  _ContentTypeTag      =
  _NonContentTypeTag   =
  _HasPairContent      =
  _EndBrkFound         =
  _TagSpcFound         =
  _SingletonTagFound   =
  _EmptyTagFound       =
  _ContainerTagFound   =
  _StdTagFound         =
  _PairEnderFound      =
  _SingletonTagAllowed =
  _EmptyTagAllowed     =
  _SearchDone          = false;

  _TagInfoSet = false;
  _MatchingTags = false;
  _InfoNode = NULL;

  _TagName.SetCaseSensitive(false);
  _TagName.SetEmpty();
}

/****************************************************************************/
MEMORYOPS_DEFN(TagInfoRecord)

/****************************************************************************/
// TagDumpStreamData class definition
/****************************************************************************/
TagDumpStreamData::TagDumpStreamData():
_Output(NULL),
_Input(NULL),
_FieldWidth(NULL)
{
  _Output = &std::cout;
  _Input = &std::cin;
  _FieldWidth = 20;
}

/****************************************************************************/
TagDumpStreamData::TagDumpStreamData(const TagDumpStreamData* Ptr_):
_Output(Ptr_ ? Ptr_->_Output:NULL),
_Input(Ptr_ ? Ptr_->_Input:NULL),
_FieldWidth(Ptr_ ? Ptr_->_FieldWidth:0)
{
  if (!Ptr_)
  {
    _Output = &std::cout;
    _Input = &std::cin;
    _FieldWidth = 20;
  }
}

/****************************************************************************/
TagDumpStreamData& TagDumpStreamData::operator = (const TagDumpStreamData* Ptr_)
{
  if (Ptr_)
  {
    _Output = Ptr_->_Output;
    _Input = Ptr_->_Input;
    _FieldWidth = Ptr_->_FieldWidth;
  }

  return *this;
}

/****************************************************************************/
void TagDumpStreamData::SetOutput(ostream* os_)
{
  _Output = os_;
}

/****************************************************************************/
void TagDumpStreamData::SetInput(istream* is_)
{
  _Input = is_;
}

/****************************************************************************/
void TagDumpStreamData::SetFieldWidth(size_t fw_)
{
  _FieldWidth = fw_;
}

/****************************************************************************/
MEMORYOPS_DEFN(TagDumpStreamData)

/****************************************************************************/
// TagTypeInfo class definition
/****************************************************************************/
//
// bool _TagSpecified;     // specified html tag and bracket to find or none
// ChrString* _ElementStr; // element str confirmed or found if not specified
//
// _ElementStr needs to be specified by SetElementStr(...)
//     if _TagSpecified is true
// _SingletonTagAllowed defaults to true
//     if not specified by SetSingletonAllowed(...)
// _EmptyTagAllowed defaults to true
//     if not specified by SetEmptyAllowed(...)
// _DocType needs to be specified by SetDocType(...)
//     in all cases
// _TagAttr needs to be specified by SetTagAttr(...)
//     if _TagSpecified is true
// _TagBrkIndex specified by SetTagBrk(...)
//     if _TagSpecified is true
//
// User set: Tag attributes:
//   bool _SingletonTagAllowed;
//   bool _EmptyTagAllowed;
//
//   int _DocType;  // Document type: tag type allowed; html, xml, cpp...
//   int _TagAttr;  // User set: Singleton / Pair / OptionalPair / Empty
//
// Tag bracket array, user specified:
//   char** _TagBrkMatrix;
//   long _TagBrkIndex;  // in +2 increments for each tag, +1 for ending tag
//
TagTypeInfo::TagTypeInfo():
_ObjectID(0),
_TagSpecified(false),
_ElementStr(NULL),
_ElementPtr(NULL),
_TagCntData(NULL),
_LookupRefTag(false),
_DictionaryTag(false),
_SingletonTagAllowed(true),
_EmptyTagAllowed(true),
_DocType(0),
_EofFound(false),
_TagAttr(0),
_TagTypeFound(0),

_TagBrkMatrix(NULL),
_TagBrkIndex(0),
_TagBrkMax(TagEnums::TAGLIST_MAX_ENTRIES),
_AltIndex(0),
_TagBrkExtend1(0),
_TagBrkExtend2(0),
_SearchResult(new TagSearchResult),

_Sequencer(NULL),
_SequencerSet(false),
_SequencerAlloc(false),
_HtmlSpecialCase(NULL),
_HtmlSpecialCaseSet(false),
_HtmlSpecialCaseAlloc(false),

_DataSpecified(0),
_TagFound(false),
_ContentTypeTag(false),
_NonContentTypeTag(false),
_HasPairContent(false),
_EndBrkFound(false),
_TagSpcFound(false),
_SingletonTagFound(false),
_EmptyTagFound(false),
_ContainerTagFound(false),
_StdTagFound(false),
_PairEnderFound(false),
_SearchDone(false),

_TagInitSelected(0),
_TagInitConfirmed(false),
_ForbiddenUnspec(false),
_DumpTagContents(false),

_IO(new TagDumpStreamData),

// Additional StackNodeInfo related data members
_TagContValid(0)
{
  _SearchResult->SetParent(this);
  _TagCntData = TagCountData::Instance();
  _TagBrkMatrix = _TagCntData->GiveTagBrkMatrix();
  _ObjectID = ++m_ObjectCnt;
}

/****************************************************************************/
TagTypeInfo::TagTypeInfo(const TagTypeInfo& Obj_):
_ObjectID(Obj_._ObjectID),
_TagSpecified(Obj_._TagSpecified),
_ElementStr((Obj_._ElementStr && Obj_._TagSpecified) ?
                (new ChrString(*Obj_._ElementStr)):NULL),

_ElementPtr(Obj_._ElementPtr),
_TagCntData(NULL),
_LookupRefTag(false),
_DictionaryTag(Obj_._DictionaryTag),
_SingletonTagAllowed(Obj_._SingletonTagAllowed),
_EmptyTagAllowed(Obj_._EmptyTagAllowed),
_DocType(Obj_._DocType),
_EofFound(Obj_._EofFound),
_TagAttr(Obj_._TagAttr),
_TagTypeFound(Obj_._TagTypeFound),

_TagBrkMatrix(NULL),
_TagBrkIndex(Obj_._TagBrkIndex),
_TagBrkMax(TagEnums::TAGLIST_MAX_ENTRIES),
_AltIndex(Obj_._AltIndex),
_TagBrkExtend1(Obj_._TagBrkExtend1),
_TagBrkExtend2(Obj_._TagBrkExtend2),
_SearchResult(Obj_._SearchResult ?
                 (new TagSearchResult(*Obj_._SearchResult)):NULL),

_Sequencer(Obj_._SequencerSet ? Obj_._Sequencer:
           (Obj_._SequencerAlloc && Obj_._Sequencer) ?
               (new TagLinksSequencer(*Obj_._Sequencer)):NULL),
_SequencerSet(Obj_._SequencerSet),
_SequencerAlloc(Obj_._SequencerAlloc),
_HtmlSpecialCase(Obj_._HtmlSpecialCaseSet ? Obj_._HtmlSpecialCase:
           (Obj_._HtmlSpecialCaseAlloc && Obj_._HtmlSpecialCase) ?
               (new HtmlSpecialCaseData(*Obj_._HtmlSpecialCase)):NULL),
_HtmlSpecialCaseSet(Obj_._HtmlSpecialCaseSet),
_HtmlSpecialCaseAlloc(Obj_._HtmlSpecialCaseAlloc),

_DataSpecified(Obj_._DataSpecified),
_TagFound(Obj_._TagFound),
_ContentTypeTag(Obj_._ContentTypeTag),
_NonContentTypeTag(Obj_._NonContentTypeTag),
_HasPairContent(Obj_._HasPairContent),
_EndBrkFound(Obj_._EndBrkFound),
_TagSpcFound(Obj_._TagSpcFound),
_SingletonTagFound(Obj_._SingletonTagFound),
_EmptyTagFound(Obj_._EmptyTagFound),
_ContainerTagFound(Obj_._ContainerTagFound),
_StdTagFound(Obj_._StdTagFound),
_PairEnderFound(Obj_._PairEnderFound),
_SearchDone(Obj_._SearchDone),

_TagInitSelected(0),
_TagInitConfirmed(false),
_ForbiddenUnspec(Obj_._ForbiddenUnspec),
_DumpTagContents(Obj_._DumpTagContents),

_IO(new TagDumpStreamData(Obj_._IO)),

// Additional StackNodeInfo related data members
_TagContValid(0)
{
  // _TagCntData not copied since specific to object type
  _SearchResult->SetParent(this);
  _TagCntData = TagCountData::Instance();
  _TagBrkMatrix = _TagCntData->GiveTagBrkMatrix();
  _ObjectID = ++m_ObjectCnt;

  if (_Sequencer && _SequencerAlloc)
    _Sequencer->SetParent(this);

  if (_HtmlSpecialCase && _HtmlSpecialCaseAlloc)
    _HtmlSpecialCase->SetParent(this);
}

/****************************************************************************/
TagTypeInfo::~TagTypeInfo()
{
  // Tag Brk matrix not destroyed within this class
  _TagCntData = NULL;
  _TagBrkMatrix = NULL;
  _ElementPtr = NULL;

  delete _ElementStr;
  _ElementStr = NULL;
  
  delete _SearchResult;
  _SearchResult = NULL;

  delete _IO;
  _IO = NULL;

  if (m_ObjectCnt > 0)
  {
    --m_ObjectCnt;

    if (m_ObjectCnt == 0)
    {
      if (_SequencerAlloc)
      {
        delete _Sequencer;
        _Sequencer = NULL;

        _SequencerAlloc = false;
        _SequencerSet = false;
      }

      if (_HtmlSpecialCaseAlloc)
      {
        delete _HtmlSpecialCase;
        _HtmlSpecialCase = NULL;

        _HtmlSpecialCaseAlloc = false;
        _HtmlSpecialCaseSet = false;
      }

      TagCountData::SetDestroyLists(false);
      TagCountData::Release();
    }
  }
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::Make()
{
  return (new TagTypeInfo());
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::Make(const TagTypeInfo& Obj_)
{
  return (new TagTypeInfo(Obj_));
}

/****************************************************************************/
void TagTypeInfo::Exec_PairEndTagUnmatchedEnd(long NestCond_,
                                              StackNodeInfo* CurNode_, StackNodeInfo* EncNode_)
{
  bool Pushed_ = false;
  StackNodeInfo* MatchTag_;
  StackNodeInfo* FoundTagPtr_;
  StackNodeInfo* CurTagStkPtr_;

  if (CurNode_ && CurNode_->PairEnderFound())
  {
    if (EncNode_ && EncNode_->PairEnderFound())
    {
      Sequencer()->_CursorStkTag = Sequencer()->HeadTagStack(true, true);

      if (Sequencer()->_HeadTagStack &&
          !Sequencer()->_HeadTagStack->IsEmpty() &&
          EncNode_ && EncNode_->PairEnderFound() &&
          Sequencer()->_HeadTagStack->TagIsEqual(EncNode_, this, true))
      {
        Pushed_ = true;
        MatchTag_ = Sequencer()->_HeadTagStack->GiveNodeFound();
        Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(MatchTag_, false);
      }
      else
      {
        Pushed_ = true;
        Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);
      }
    }
    else
    {
      Sequencer()->_CursorStkTag = Sequencer()->HeadTagStack(true, true);

      if (Sequencer()->_HeadTagStack &&
          !Sequencer()->_HeadTagStack->IsEmpty() &&
          CurNode_ && CurNode_->PairEnderFound() &&
          Sequencer()->_HeadTagStack->TagIsEqual(CurNode_, this, true))
      {
        Pushed_ = true;
        MatchTag_ = Sequencer()->_HeadTagStack->GiveNodeFound();
        Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(MatchTag_, false);
      }
      else
      {
        Pushed_ = true;
        Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(CurNode_, false);
      }
    }
  }
  else if (EncNode_ && EncNode_->PairEnderFound())
  {
    Pushed_ = true;
    Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);
  }

  Sequencer()->AssignGlobalStackTag();

  if (!Sequencer()->_HeadTagStack || Sequencer()->_HeadTagStack->IsEmpty())
  {
    Sequencer()->AppendNestingError(STDTAGENDPAIR_MISMATCH);
    _TagContValid |= TAGCONTENT_NESTINGERROR;

    // _MismatchedEndPair = EncounteredTag
    if (Sequencer()->_CursorStkTag && EncNode_)
      Sequencer()->_CursorStkTag->SetMismatchedEndPair(EncNode_);

    if (Pushed_)
      Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
    return;
  }
  else if (CurNode_)
  {
    if (!EncNode_ || !EncNode_->PairEnderFound())
    {
      if (Pushed_)
        Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
      return;
    }

    // nested tags of same type encountered
    //
    // find latest preceeding tag which is either a pair or
    // optional pair type tag that matches encountered closing tag
    // third argument to TagIsEqual skips optional tags that are unmatched
    //
    // ::TagIsEqual(StackNodeInfo*, StackNodeInfo*, bool)
    //   _StkTagFound = _CursorStkTag != NULL && _FoundTag != NULL;
    //   _FoundCursorTag = _StkTagFound && EquivTag(_CursorStkTag, _FoundTag);
    //
    // EquivTag method matches same tag type as well as matching pair
    // tag types as being equivalent
    //
    if (Sequencer()->_HeadTagStack->TagIsEqual(EncNode_, this, true) &&
        !Sequencer()->FoundCursorTag() &&
        Sequencer()->StackTagFound() &&
        Sequencer()->_FoundTag->Preceeds(Sequencer()->_CursorStkTag))
    {
      MatchTag_ = Sequencer()->_HeadTagStack->GiveNodeFound();

      if (MatchTag_)
      {
        long SavedNestCond_ = NestCond_;
        FoundTagPtr_ = Sequencer()->_FoundTag;
        CurTagStkPtr_ = MatchTag_->CursorTagStkPtr();
        MatchTag_->RemovePrevTentStr();
        MatchTag_->SetTagNestCond(STMEND_NESTCOND, EncNode_);

        // Only initiate TagStack popping if
        // EquivTag(CurTagStkPtr_, _HeadTagStack) returns true and
        // also matches _HeadCurTagStk
        //
        // Node popping will also begin for all earlier nodes in TagStack
        // where _TagAppendDone == true and
        // _FoundTag does not preceed CurTagStkPtr_ :
        //
        //   _FoundCurTag or CurTagStkPtr_ == NULL
        //   by working backwards from _HeadCurTagStk and calling
        //   _HeadTagStack->TagIsEqual(CurTagStkPtr_->_Prev, this, true)
        //   iteratively until no previous CurTagStk node is found while
        //   also popping nodes from _HeadCurTagStk as they are found to
        //   match the current pointer from _CurTagStkPtr
        //
        // Otherwise if CurTagStkPtr_ does not match _HeadTagStack or
        // _HeadCurTagStk then indicate a pop operation by setting the
        // bool _Erased flag in CurTagStkPtr_ to true, but do not actually
        // pop the stack node. Additionally,
        // if EquivTag(FoundTagPtr_, CurTagStkPtr_) method returns true then
        // Set the _CurStkNodeFound flag of the CurTagStkPtr_ stack node to true
        //
        Sequencer()->PopTagStack(FoundTagPtr_, CurTagStkPtr_, EncNode_, this, true);
        Sequencer()->ConfirmTagStackErased();
        // SetTagCondData(SavedNestCond_, CurNode_, EncNode_);

        if ((CurTagStkPtr_->Unlinked() || CurTagStkPtr_->Erased()) &&
            (Sequencer()->_HeadCursorTagStk->Unlinked() ||
             Sequencer()->_HeadCursorTagStk->Erased()))
        {
          Pushed_ = false;
          Sequencer()->_CursorStkTag = Sequencer()->HeadCursorTagStk(false, true);
        }
      }
    }
    else if (!Sequencer()->FoundCursorTag() ||
             (Sequencer()->StackTagFound() &&
              Sequencer()->_FoundTag->Follows(Sequencer()->_CursorStkTag)))
    {
      Sequencer()->AppendNestingError(TAGPAIR_OVERLAP);
      _TagContValid |= TAGCONTENT_NESTINGERROR;

      // _OverlappedPrevPair = EncounteredTag;
      // _OverlappedNextPair = _StkTagFound ? _FoundTag:NULL;
      if (EncNode_)
        if (Sequencer()->_CursorStkTag && Sequencer()->_StkTagFound &&
            EncNode_ != Sequencer()->_FoundTag)
          Sequencer()->_CursorStkTag->SetOverlappedPairs(EncNode_, Sequencer()->_FoundTag);
        else if (CurNode_ && CurNode_ != EncNode_ &&
                 EncNode_->Follows(CurNode_))
          CurNode_->SetOverlappedPairs(CurNode_, EncNode_);
    }
    else if (Sequencer()->StackTagFound() && Sequencer()->FoundCursorTag())
    {
      bool Pair_ = false;
      CurTagStkPtr_ = Sequencer()->_CursorStkTag;
      FoundTagPtr_ = Sequencer()->_FoundTag;
      TagTypeInfo::IsOptionalTag(FoundTagPtr_->_TagAttr, &Pair_);

      if (FoundTagPtr_ && FoundTagPtr_->_StdTagFound &&
          EncNode_->PairEnderFound() && Pair_)
      {
        // _CurStkTag is the pointer to the Current Stack Tag
        // 2nd argument to LinkForward indicates automatic LinkBackward
        // action after checking for correct stack node values
        //   StackNodeInfo::LinkBackward(StackNodeInfo* EncTag_, StackNodeInfo*);
        //
        // If current stack tag _CurStkTag is not of singleton tag type
        // then LinkForward method will also set the closing tag for all
        // intervening nodes between the linked tags to the ending tag node
        // if the intervening node tags is of the optional or singleton
        // tag type with the method below:
        //   SetTagEncountered(TagEnums::CLOSING, false);
        //

        // EncounteredTag->LinkBackward(_HeadTagStack);
        // _TagAppendDone = true;
        if (FoundTagPtr_ &&
            StackNodeInfo::EquivTag(FoundTagPtr_, EncNode_, NULL, StackNodeInfo::FORWARD_ORDER) &&
            FoundTagPtr_->Preceeds(EncNode_))
          FoundTagPtr_->LinkForward(EncNode_, true);
        else if (CurTagStkPtr_ &&
                 StackNodeInfo::EquivTag(CurTagStkPtr_, EncNode_, NULL, StackNodeInfo::FORWARD_ORDER) &&
                 CurTagStkPtr_->Preceeds(EncNode_))
          CurTagStkPtr_->LinkForward(EncNode_, true);

        MatchTag_ = Sequencer()->_HeadTagStack->GiveNodeFound();
        FoundTagPtr_ = Sequencer()->_FoundTag;
        CurTagStkPtr_ = MatchTag_->CursorTagStkPtr();

        // Only pop head of stack if CurTagStkPtr_ is head of stack and
        // all earlier nodes in TagStack where _TagAppendDone == true and
        // _FoundTag does not preceed CurTagStkPtr_ :
        //   _FoundCurTag or CurTagStkPtr_ == NULL
        //
        Sequencer()->PopTagStack(FoundTagPtr_, CurTagStkPtr_, EncNode_, this, true);
        Sequencer()->ConfirmTagStackErased();

        if ((CurTagStkPtr_->Unlinked() || CurTagStkPtr_->Erased()) &&
            (Sequencer()->_HeadCursorTagStk->Unlinked() ||
             Sequencer()->_HeadCursorTagStk->Erased()))
        {
          Pushed_ = false;
          Sequencer()->_CursorStkTag = Sequencer()->HeadCursorTagStk(false, true);
        }
      }
    }
    else if (Sequencer()->StackTagFound() && Sequencer()->_CursorStkTag->Follows(CurNode_) &&
             !StackNodeInfo::EquivTag(CurNode_, Sequencer()->_CursorStkTag, NULL, StackNodeInfo::FORWARD_ORDER))
    {
      Sequencer()->AppendNestingError(TAGPAIR_OVERLAP);
      _TagContValid |= TAGCONTENT_NESTINGERROR;

      // _OverlappedPrevPair = EncounteredTag;
      // _OverlappedNextPair = _StkTagFound ? _FoundTag:NULL;
      if (Sequencer()->_CursorStkTag && Sequencer()->_StkTagFound)
        if (CurNode_ && CurNode_ != Sequencer()->_CursorStkTag &&
            Sequencer()->_CursorStkTag->Follows(CurNode_))
          CurNode_->SetOverlappedPairs(CurNode_, Sequencer()->_CursorStkTag);
    }

    if (Pushed_)
      Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
  }
}

/****************************************************************************/
void TagTypeInfo::Exec_StartTagUnmatchedEnd(long NestCond_,
                                            StackNodeInfo* CurNode_, StackNodeInfo* EncNode_)
{
  bool Pushed_ = false;
  StackNodeInfo* MatchTag_;
  StackNodeInfo* FoundTagPtr_;
  StackNodeInfo* CurTagStkPtr_;

  if (CurNode_ && CurNode_->StdTagFound() && !CurNode_->PairEnderFound() &&
      !CurNode_->SingletonTagFound() && !CurNode_->EmptyTagFound())
  {
    if (EncNode_ && EncNode_->PairEnderFound() &&
        !StackNodeInfo::EquivTag(EncNode_, CurNode_))
    {
      Pushed_ = true;
      Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);
    }
    else
    {
      Pushed_ = true;
      Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(CurNode_, false);
    }
  }
  else if (Sequencer()->_HeadTagStack &&
           !Sequencer()->_HeadTagStack->IsEmpty() &&
           CurNode_ && CurNode_->PairEnderFound() &&
           StackNodeInfo::EquivTag(CurNode_, Sequencer()->_HeadTagStack))
  {
    Pushed_ = true;
    Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(Sequencer()->_HeadTagStack, false);
  }
  else if (Sequencer()->_HeadTagStack &&
           !Sequencer()->_HeadTagStack->IsEmpty() &&
           EncNode_ && EncNode_->PairEnderFound() &&
           StackNodeInfo::EquivTag(EncNode_, Sequencer()->_HeadTagStack))
  {
    Pushed_ = true;
    Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(Sequencer()->_HeadTagStack, false);
  }
  else
  {
    Pushed_ = true;
    Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);
  }

  Sequencer()->AssignGlobalStackTag();

  if (!Sequencer()->_HeadTagStack || Sequencer()->_HeadTagStack->IsEmpty())
  {
    Sequencer()->AppendNestingError(STDTAGENDPAIR_MISMATCH);
    _TagContValid |= TAGCONTENT_NESTINGERROR;

    // _MismatchedEndPair = EncounteredTag
    if (Sequencer()->_CursorStkTag)
      Sequencer()->_CursorStkTag->SetMismatchedEndPair(EncNode_);

    if (Pushed_)
      Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
    return;
  }
  else if (CurNode_)
  {
    if (!EncNode_ || !EncNode_->PairEnderFound())
    {
      if (Pushed_)
        Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
      return;
    }

    // nested tags of same type encountered
    //
    // find latest preceeding tag which is either a pair or
    // optional pair type tag that matches encountered closing tag
    // third argument to TagIsEqual skips optional tags that are unmatched
    //
    // ::TagIsEqual(StackNodeInfo*, StackNodeInfo*, bool)
    //   _StkTagFound = _CursorStkTag != NULL && _FoundTag != NULL;
    //   _FoundCursorTag = _StkTagFound && EquivTag(_CursorStkTag, _FoundTag);
    //
    // EquivTag method matches same tag type as well as matching pair
    // tag types as being equivalent
    //
    if (Sequencer()->_HeadTagStack->TagIsEqual(EncNode_, this, true) &&
        !Sequencer()->FoundCursorTag() &&
        Sequencer()->StackTagFound() &&
        Sequencer()->_FoundTag->Preceeds(Sequencer()->_CursorStkTag))
    {
      MatchTag_ = Sequencer()->_HeadTagStack->GiveNodeFound();

      if (MatchTag_)
      {
        long SavedNestCond_ = NestCond_;
        FoundTagPtr_ = Sequencer()->_FoundTag;
        CurTagStkPtr_ = MatchTag_->CursorTagStkPtr();
        MatchTag_->RemovePrevTentStr();
        MatchTag_->SetTagNestCond(STMEND_NESTCOND, EncNode_);

        // Only initiate TagStack popping if
        // EquivTag(CurTagStkPtr_, _HeadTagStack) returns true and
        // also matches _HeadCurTagStk
        //
        // Node popping will also begin for all earlier nodes in TagStack
        // where _TagAppendDone == true and
        // _FoundTag does not preceed CurTagStkPtr_ :
        //
        //   _FoundCursorTag or CurTagStkPtr_ == NULL
        //   by working backwards from _HeadCursorTagStk and calling
        //   _HeadTagStack->TagIsEqual(CurTagStkPtr_->_Prev, this, true)
        //   iteratively until no previous CurTagStk node is found while
        //   also popping nodes from _HeadCursorTagStk as they are found to
        //   match the current pointer from _CursorTagStkPtr
        //
        // Otherwise if CurTagStkPtr_ does not match _HeadTagStack or
        // _HeadCursorTagStk then indicate a pop operation by setting the
        // bool _Erased flag in CurTagStkPtr_ to true, but do not actually
        // pop the stack node. Additionally,
        // if EquivTag(FoundTagPtr_, CurTagStkPtr_) method returns true then
        // Set the _CursorStkNodeFound flag of the CurTagStkPtr_ stack node to true
        //
        Sequencer()->PopTagStack(FoundTagPtr_, CurTagStkPtr_, EncNode_, this, true);
        Sequencer()->ConfirmTagStackErased();
        // SetTagCondData(SavedNestCond_, CurNode_, EncNode_);

        if ((CurTagStkPtr_->Unlinked() || CurTagStkPtr_->Erased()) &&
            (Sequencer()->_HeadCursorTagStk->Unlinked() ||
             Sequencer()->_HeadCursorTagStk->Erased()))
        {
          Pushed_ = false;
          Sequencer()->_CursorStkTag = Sequencer()->HeadCursorTagStk(false, true);
        }
      }
    }
    else if (!Sequencer()->FoundCursorTag() ||
             (Sequencer()->StackTagFound() &&
              Sequencer()->_FoundTag->Follows(Sequencer()->_CursorStkTag)))
    {
      Sequencer()->AppendNestingError(TAGPAIR_OVERLAP);
      _TagContValid |= TAGCONTENT_NESTINGERROR;

      // _OverlappedPrevPair = EncounteredTag;
      // _OverlappedNextPair = _StkTagFound ? _FoundTag:NULL;
      if (EncNode_)
        if (Sequencer()->_CursorStkTag && Sequencer()->_StkTagFound &&
            EncNode_ != Sequencer()->_FoundTag)
          Sequencer()->_CursorStkTag->SetOverlappedPairs(EncNode_, Sequencer()->_FoundTag);
        else if (CurNode_ && CurNode_ != EncNode_ &&
                 EncNode_->Follows(CurNode_))
          CurNode_->SetOverlappedPairs(CurNode_, EncNode_);
    }
    else if (Sequencer()->StackTagFound() && Sequencer()->_CursorStkTag->Follows(CurNode_) &&
             !StackNodeInfo::EquivTag(CurNode_, Sequencer()->_CursorStkTag, NULL, StackNodeInfo::FORWARD_ORDER))
    {
      Sequencer()->AppendNestingError(TAGPAIR_OVERLAP);
      _TagContValid |= TAGCONTENT_NESTINGERROR;

      // _OverlappedPrevPair = EncounteredTag;
      // _OverlappedNextPair = _StkTagFound ? _FoundTag:NULL;
      if (Sequencer()->_CursorStkTag && Sequencer()->_StkTagFound)
        if (CurNode_ && CurNode_ != Sequencer()->_CursorStkTag &&
            Sequencer()->_CursorStkTag->Follows(CurNode_))
          CurNode_->SetOverlappedPairs(CurNode_, Sequencer()->_CursorStkTag);
    }

    if (Pushed_)
      Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
  }
}

/****************************************************************************/
void TagTypeInfo::Exec_StartTagMatchedEnd(long NestCond_,
                                          StackNodeInfo* CurNode_, StackNodeInfo* EncNode_)
{
  bool Pushed_ = false;
  StackNodeInfo* MatchTag_;
  StackNodeInfo* FoundTagPtr_;
  StackNodeInfo* CurTagStkPtr_;

  if (CurNode_ && CurNode_->StdTagFound() && !CurNode_->PairEnderFound() &&
      !CurNode_->SingletonTagFound() && !CurNode_->EmptyTagFound())
  {
    if (EncNode_ && EncNode_->PairEnderFound() &&
        !StackNodeInfo::EquivTag(EncNode_, CurNode_))
    {
      Pushed_ = true;
      Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);
    }
    else
    {
      Pushed_ = true;
      Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(CurNode_, false);
    }
  }
  else if (Sequencer()->_HeadTagStack &&
           !Sequencer()->_HeadTagStack->IsEmpty() &&
           CurNode_ && CurNode_->PairEnderFound() &&
           StackNodeInfo::EquivTag(CurNode_, Sequencer()->_HeadTagStack))
  {
    Pushed_ = true;
    Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(Sequencer()->_HeadTagStack, false);
  }
  else if (Sequencer()->_HeadTagStack &&
           !Sequencer()->_HeadTagStack->IsEmpty() &&
           EncNode_ && EncNode_->PairEnderFound() &&
           StackNodeInfo::EquivTag(EncNode_, Sequencer()->_HeadTagStack))
  {
    Pushed_ = true;
    Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(Sequencer()->_HeadTagStack, false);
  }
  else
  {
    Pushed_ = true;
    Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);
  }

  Sequencer()->AssignGlobalStackTag();

  if (!Sequencer()->_HeadTagStack || Sequencer()->_HeadTagStack->IsEmpty())
  {
    // _CursorStkTag is the pointer to the Current Stack Tag
    // 2nd argument to LinkForward indicates automatic LinkBackward
    // action after checking for correct stack node values
    //   StackNodeInfo::LinkBackward(StackNodeInfo* EncTag_, StackNodeInfo*);
    //
    // If current stack tag _CursorStkTag is not of singleton tag type
    // then LinkForward method will also set the closing tag for all
    // intervening nodes between the linked tags to the ending tag node
    // if the intervening node tags is of the optional or singleton
    // tag type with the method below:
    //   SetTagEncountered(TagEnums::CLOSING, false);
    //

    // EncounteredTag->LinkBackward(CurNode_);
    // _TagAppendDone = true;
    if (Sequencer()->_CursorStkTag)
      Sequencer()->_CursorStkTag->LinkForward(EncNode_, true);

    if (Pushed_)
      Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
    return;
  }
  else if (CurNode_)
  {
    if (!EncNode_ || !EncNode_->PairEnderFound())
    {
      if (Pushed_)
        Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
      return;
    }

    // nested tags of same type encountered
    //
    // find latest preceeding tag which is either a pair or
    // optional pair type tag that matches encountered closing tag
    // third argument to TagIsEqual skips optional tags that are unmatched
    //
    // ::TagIsEqual(StackNodeInfo*, StackNodeInfo*, bool)
    //   _StkTagFound = _CursorStkTag != NULL && _FoundTag != NULL;
    //   _FoundCursorTag = _StkTagFound && EquivTag(_CursorStkTag, _FoundTag);
    //
    // EquivTag method matches same tag type as well as matching pair
    // tag types as being equivalent
    //
    if (Sequencer()->_HeadTagStack->TagIsEqual(EncNode_, this, true) &&
        !Sequencer()->FoundCursorTag() &&
        Sequencer()->StackTagFound() &&
        Sequencer()->_FoundTag->Preceeds(Sequencer()->_CursorStkTag))
    {
      MatchTag_ = Sequencer()->_HeadTagStack->GiveNodeFound();

      if (MatchTag_)
      {
        long SavedNestCond_ = NestCond_;
        FoundTagPtr_ = Sequencer()->_FoundTag;
        CurTagStkPtr_ = MatchTag_->CursorTagStkPtr();
        MatchTag_->RemovePrevTentStr();
        MatchTag_->SetTagNestCond(STMEND_NESTCOND, EncNode_);

        // Only initiate TagStack popping if
        // EquivTag(CurTagStkPtr_, _HeadTagStack) returns true and
        // also matches _HeadCursorTagStk
        //
        // Node popping will also begin for all earlier nodes in TagStack
        // where _TagAppendDone == true and
        // _FoundTag does not preceed CurTagStkPtr_ :
        //
        //   _FoundCursorTag or CurTagStkPtr_ == NULL
        //   by working backwards from _HeadCursorTagStk and calling
        //   _HeadTagStack->TagIsEqual(CurTagStkPtr_->_Prev, this, true)
        //   iteratively until no previous CurTagStk node is found while
        //   also popping nodes from _HeadCursorTagStk as they are found to
        //   match the current pointer from _CursorTagStkPtr
        //
        // Otherwise if CurTagStkPtr_ does not match _HeadTagStack or
        // _HeadCursorTagStk then indicate a pop operation by setting the
        // bool _Erased flag in CurTagStkPtr_ to true, but do not actually
        // pop the stack node. Additionally,
        // if EquivTag(FoundTagPtr_, CurTagStkPtr_) method returns true then
        // Set the _CursorStkNodeFound flag of the CurTagStkPtr_ stack node to true
        //
        Sequencer()->PopTagStack(FoundTagPtr_, CurTagStkPtr_, EncNode_, this, true);
        Sequencer()->ConfirmTagStackErased();
        // SetTagCondData(SavedNestCond_, CurNode_, EncNode_);

        if ((CurTagStkPtr_->Unlinked() || CurTagStkPtr_->Erased()) &&
            (Sequencer()->_HeadCursorTagStk->Unlinked() ||
             Sequencer()->_HeadCursorTagStk->Erased()))
        {
          Pushed_ = false;
          Sequencer()->_CursorStkTag = Sequencer()->HeadCursorTagStk(false, true);
        }
      }
    }
    else if (!Sequencer()->FoundCursorTag() ||
             (Sequencer()->StackTagFound() &&
              Sequencer()->_FoundTag->Follows(Sequencer()->_CursorStkTag)))
    {
      Sequencer()->AppendNestingError(TAGPAIR_OVERLAP);
      _TagContValid |= TAGCONTENT_NESTINGERROR;

      // _OverlappedPrevPair = this;
      // _OverlappedNextPair = _StkTagFound ? _FoundTag:NULL;
      if (EncNode_)
        if (Sequencer()->_CursorStkTag && Sequencer()->_StkTagFound &&
            EncNode_ != Sequencer()->_FoundTag)
          Sequencer()->_CursorStkTag->SetOverlappedPairs(EncNode_, Sequencer()->_FoundTag);
        else if (CurNode_ && CurNode_ != EncNode_ &&
                 EncNode_->Follows(CurNode_))
          CurNode_->SetOverlappedPairs(CurNode_, EncNode_);
    }
    else if (Sequencer()->StackTagFound() && Sequencer()->FoundCursorTag())
    {
      bool Pair_ = false;
      CurTagStkPtr_ = Sequencer()->_CursorStkTag;
      FoundTagPtr_ = Sequencer()->_FoundTag;
      TagTypeInfo::IsOptionalTag(FoundTagPtr_->_TagAttr, &Pair_);

      if (FoundTagPtr_ && FoundTagPtr_->_StdTagFound &&
          EncNode_->PairEnderFound() && Pair_)
      {
        // _CursorStkTag is the pointer to the Current Stack Tag
        // 2nd argument to LinkForward indicates automatic LinkBackward
        // action after checking for correct stack node values
        //   StackNodeInfo::LinkBackward(StackNodeInfo* EncTag_, StackNodeInfo*);
        //
        // If current stack tag _CursorStkTag is not of singleton tag type
        // then LinkForward method will also set the closing tag for all
        // intervening nodes between the linked tags to the ending tag node
        // if the intervening node tags is of the optional or singleton
        // tag type with the method below:
        //   SetTagEncountered(TagEnums::CLOSING, false);
        //

        // EncounteredTag->LinkBackward(_HeadTagStack);
        // _TagAppendDone = true;
        if (FoundTagPtr_ && StackNodeInfo::EquivTag(FoundTagPtr_, EncNode_, NULL, StackNodeInfo::FORWARD_ORDER) &&
            FoundTagPtr_->Preceeds(EncNode_))
          FoundTagPtr_->LinkForward(EncNode_, true);
        else if (CurTagStkPtr_ && StackNodeInfo::EquivTag(CurTagStkPtr_, EncNode_, NULL, StackNodeInfo::FORWARD_ORDER) &&
                 CurTagStkPtr_->Preceeds(EncNode_))
          CurTagStkPtr_->LinkForward(EncNode_, true);

        MatchTag_ = Sequencer()->_HeadTagStack->GiveNodeFound();
        FoundTagPtr_ = Sequencer()->_FoundTag;
        CurTagStkPtr_ = MatchTag_->CursorTagStkPtr();

        // Only pop head of stack if CurTagStkPtr_ is head of stack and
        // all earlier nodes in TagStack where _TagAppendDone == true and
        // _FoundTag does not preceed CurTagStkPtr_ :
        //   _FoundCursorTag or CurTagStkPtr_ == NULL
        //
        Sequencer()->PopTagStack(FoundTagPtr_, CurTagStkPtr_, EncNode_, this, true);
        Sequencer()->ConfirmTagStackErased();

        if ((CurTagStkPtr_->Unlinked() || CurTagStkPtr_->Erased()) &&
            (Sequencer()->_HeadCursorTagStk->Unlinked() ||
             Sequencer()->_HeadCursorTagStk->Erased()))
        {
          Pushed_ = false;
          Sequencer()->_CursorStkTag = Sequencer()->HeadCursorTagStk(false, true);
        }
      }
    }
    else if (Sequencer()->StackTagFound() && Sequencer()->_CursorStkTag->Follows(CurNode_) &&
             !StackNodeInfo::EquivTag(CurNode_, Sequencer()->_CursorStkTag, NULL, StackNodeInfo::FORWARD_ORDER))
    {
      Sequencer()->AppendNestingError(TAGPAIR_OVERLAP);
      _TagContValid |= TAGCONTENT_NESTINGERROR;

      // _OverlappedPrevPair = EncounteredTag;
      // _OverlappedNextPair = _StkTagFound ? _FoundTag:NULL;
      if (Sequencer()->_CursorStkTag && Sequencer()->_StkTagFound)
        if (CurNode_ && CurNode_ != Sequencer()->_CursorStkTag &&
            Sequencer()->_CursorStkTag->Follows(CurNode_))
          CurNode_->SetOverlappedPairs(CurNode_, Sequencer()->_CursorStkTag);
    }

    if (Pushed_)
      Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
  }
}

/****************************************************************************/
void TagTypeInfo::Exec_SingletonUnmatchedEnd(long NestCond_,
                                             StackNodeInfo* CurNode_, StackNodeInfo* EncNode_)
{
  bool Pushed_ = false;
  StackNodeInfo* MatchTag_;
  StackNodeInfo* FoundTagPtr_;
  StackNodeInfo* CurTagStkPtr_;

  if (CurNode_ && CurNode_->StdTagFound() && !CurNode_->PairEnderFound() &&
      !CurNode_->SingletonTagFound() && !CurNode_->EmptyTagFound())
  {
    if (EncNode_ && EncNode_->PairEnderFound() &&
        !StackNodeInfo::EquivTag(EncNode_, CurNode_))
    {
      Pushed_ = true;
      Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);
    }
    else
    {
      Pushed_ = true;
      Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(CurNode_, false);
    }
  }
  else if (Sequencer()->_HeadTagStack &&
           !Sequencer()->_HeadTagStack->IsEmpty() &&
           CurNode_ && CurNode_->PairEnderFound() &&
           StackNodeInfo::EquivTag(CurNode_, Sequencer()->_HeadTagStack))
  {
    Pushed_ = true;
    Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(Sequencer()->_HeadTagStack, false);
  }
  else if (Sequencer()->_HeadTagStack &&
           !Sequencer()->_HeadTagStack->IsEmpty() &&
           EncNode_ && EncNode_->PairEnderFound() &&
           StackNodeInfo::EquivTag(EncNode_, Sequencer()->_HeadTagStack))
  {
    Pushed_ = true;
    Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(Sequencer()->_HeadTagStack, false);
  }
  else
  {
    Pushed_ = true;
    Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);
  }

  Sequencer()->AssignGlobalStackTag();

  if (!Sequencer()->_HeadTagStack || Sequencer()->_HeadTagStack->IsEmpty())
  {
    Sequencer()->AppendNestingError(STDTAGENDPAIR_MISMATCH | SINGLETON_UNMATCHED);
    _TagContValid |= TAGCONTENT_NESTINGERROR;

    // _MismatchedEndPair = EncounteredTag
    if (CurNode_)
      CurNode_->SetMismatchedEndPair(EncNode_);

    if (Pushed_)
      Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
    return;
  }
  else if (CurNode_)
  {
    if (!EncNode_ || !EncNode_->PairEnderFound())
    {
      if (Pushed_)
        Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
      return;
    }

    // closing off singleton tag of current node
    CurNode_->LinkForward(EncNode_);

    // find latest preceeding tag which is either a pair or
    // optional pair type tag that matches encountered closing tag
    // 2nd argument to TagIsEqual skips optional tags that are unmatched
    //
    // ::TagIsEqual(StackNodeInfo*, StackNodeInfo*, bool)
    //   _StkTagFound = _CursorStkTag != NULL && _FoundTag != NULL;
    //   _FoundCursorTag = _StkTagFound && EquivTag(_CursorStkTag, _FoundTag);
    //
    // EquivTag method matches same tag type as well as matching pair
    // tag types as being equivalent
    //
    if (Sequencer()->_HeadTagStack->TagIsEqual(EncNode_, this, true) &&
        !Sequencer()->FoundCursorTag() &&
        Sequencer()->StackTagFound() &&
        Sequencer()->_FoundTag->Preceeds(Sequencer()->_CursorStkTag))
    {
      MatchTag_ = Sequencer()->_HeadTagStack->GiveNodeFound();

      if (MatchTag_)
      {
        long SavedNestCond_ = NestCond_;
        FoundTagPtr_ = Sequencer()->_FoundTag;
        CurTagStkPtr_ = MatchTag_->CursorTagStkPtr();
        MatchTag_->RemovePrevTentStr();
        MatchTag_->SetTagNestCond(STMEND_NESTCOND, EncNode_);

        // Only initiate TagStack popping if
        // EquivTag(CurTagStkPtr_, _HeadTagStack) returns true and
        // also matches _HeadCurTagStk
        //
        // Node popping will also begin for all earlier nodes in TagStack
        // where _TagAppendDone == true and
        // _FoundTag does not preceed CurTagStkPtr_ :
        //
        //   _FoundCurTag or CurTagStkPtr_ == NULL
        //   by working backwards from _HeadCurTagStk and calling
        //   _HeadTagStack->TagIsEqual(CurTagStkPtr_->_Prev, this, true)
        //   iteratively until no previous CurTagStk node is found while
        //   also popping nodes from _HeadCurTagStk as they are found to
        //   match the current pointer from _CurTagStkPtr
        //
        // Otherwise if CurTagStkPtr_ does not match _HeadTagStack or
        // _HeadCurTagStk then indicate a pop operation by setting the
        // bool _Erased flag in CurTagStkPtr_ to true, but do not actually
        // pop the stack node. Additionally,
        // if EquivTag(FoundTagPtr_, CurTagStkPtr_) method returns true then
        // Set the _CurStkNodeFound flag of the CurTagStkPtr_ stack node to true
        //
        Sequencer()->PopTagStack(FoundTagPtr_, CurTagStkPtr_, EncNode_, this, true);
        Sequencer()->ConfirmTagStackErased();
        // SetTagCondData(SavedNestCond_, CurNode_, EncNode_);

        if ((CurTagStkPtr_->Unlinked() || CurTagStkPtr_->Erased()) &&
            (Sequencer()->_HeadCursorTagStk->Unlinked() ||
             Sequencer()->_HeadCursorTagStk->Erased()))
        {
          Pushed_ = false;
          Sequencer()->_CursorStkTag = Sequencer()->HeadCursorTagStk(false, true);
        }
      }
    }
    else if (!Sequencer()->FoundCursorTag() ||
             (Sequencer()->StackTagFound() &&
              Sequencer()->_FoundTag->Follows(Sequencer()->_CursorStkTag)))
    {
      Sequencer()->AppendNestingError(TAGPAIR_OVERLAP);
      _TagContValid |= TAGCONTENT_NESTINGERROR;

      // _OverlappedPrevPair = this;
      // _OverlappedNextPair = _StkTagFound ? _FoundTag:NULL;
      if (EncNode_)
        if (Sequencer()->_CursorStkTag && Sequencer()->_StkTagFound &&
            EncNode_ != Sequencer()->_FoundTag)
          Sequencer()->_CursorStkTag->SetOverlappedPairs(EncNode_, Sequencer()->_FoundTag);
        else if (CurNode_ && CurNode_ != EncNode_ &&
                 EncNode_->Follows(CurNode_))
          CurNode_->SetOverlappedPairs(CurNode_, EncNode_);
    }
    else if (Sequencer()->StackTagFound() && Sequencer()->FoundCursorTag())
    {
      bool Pair_ = false;
      CurTagStkPtr_ = Sequencer()->_CursorStkTag;
      FoundTagPtr_ = Sequencer()->_FoundTag;
      TagTypeInfo::IsOptionalTag(FoundTagPtr_->_TagAttr, &Pair_);

      if (FoundTagPtr_ && FoundTagPtr_->_StdTagFound &&
          EncNode_->PairEnderFound() && Pair_)
      {
        // _CurStkTag is the pointer to the Current Stack Tag
        // 2nd argument to LinkForward indicates automatic LinkBackward
        // action after checking for correct stack node values
        //   StackNodeInfo::LinkBackward(StackNodeInfo* EncTag_, StackNodeInfo*);
        //
        // If current stack tag _CurStkTag is not of singleton tag type
        // then LinkForward method will also set the closing tag for all
        // intervening nodes between the linked tags to the ending tag node
        // if the intervening node tags is of the optional or singleton
        // tag type with the method below:
        //   SetTagEncountered(TagEnums::CLOSING, false);
        //

        // EncounteredTag->LinkBackward(_HeadTagStack);
        // _TagAppendDone = true;
        if (FoundTagPtr_ && StackNodeInfo::EquivTag(FoundTagPtr_, EncNode_, NULL, StackNodeInfo::FORWARD_ORDER) &&
            FoundTagPtr_->Preceeds(EncNode_))
          FoundTagPtr_->LinkForward(EncNode_, true);
        else if (CurTagStkPtr_ && StackNodeInfo::EquivTag(CurTagStkPtr_, EncNode_, NULL, StackNodeInfo::FORWARD_ORDER) &&
                 CurTagStkPtr_->Preceeds(EncNode_))
          CurTagStkPtr_->LinkForward(EncNode_, true);

        MatchTag_ = Sequencer()->_HeadTagStack->GiveNodeFound();
        FoundTagPtr_ = Sequencer()->_FoundTag;
        CurTagStkPtr_ = MatchTag_->CursorTagStkPtr();

        // Only pop head of stack if CurTagStkPtr_ is head of stack and
        // all earlier nodes in TagStack where _TagAppendDone == true and
        // _FoundTag does not preceed CurTagStkPtr_ :
        //   _FoundCurTag or CurTagStkPtr_ == NULL
        //
        Sequencer()->PopTagStack(FoundTagPtr_, CurTagStkPtr_, EncNode_, this, true);
        Sequencer()->ConfirmTagStackErased();

        if ((CurTagStkPtr_->Unlinked() || CurTagStkPtr_->Erased()) &&
            (Sequencer()->_HeadCursorTagStk->Unlinked() ||
             Sequencer()->_HeadCursorTagStk->Erased()))
        {
          Pushed_ = false;
          Sequencer()->_CursorStkTag = Sequencer()->HeadCursorTagStk(false, true);
        }
      }
      else
      {
        Sequencer()->AppendNestingError(STDTAGENDPAIR_MISMATCH | SINGLETON_MATCHED);
        _TagContValid |= TAGCONTENT_NESTINGERROR;

        // _MismatchedEndPair = EncounteredTag
        if (Sequencer()->_CursorStkTag &&
            StackNodeInfo::EquivTag(Sequencer()->_CursorStkTag, EncNode_))
          Sequencer()->_CursorStkTag->SetMismatchedEndPair(EncNode_);
      }
    }
    else if (Sequencer()->StackTagFound() && Sequencer()->_CursorStkTag->Follows(CurNode_) &&
             !StackNodeInfo::EquivTag(CurNode_, Sequencer()->_CursorStkTag, NULL, StackNodeInfo::FORWARD_ORDER))
    {
      Sequencer()->AppendNestingError(TAGPAIR_OVERLAP);
      _TagContValid |= TAGCONTENT_NESTINGERROR;

      // _OverlappedPrevPair = EncounteredTag;
      // _OverlappedNextPair = _StkTagFound ? _FoundTag:NULL;
      if (Sequencer()->_CursorStkTag && Sequencer()->_StkTagFound)
        if (CurNode_ && CurNode_ != Sequencer()->_CursorStkTag &&
            Sequencer()->_CursorStkTag->Follows(CurNode_))
          CurNode_->SetOverlappedPairs(CurNode_, Sequencer()->_CursorStkTag);
    }

    if (Pushed_)
      Sequencer()->_CursorStkTag = Sequencer()->PopCursorTag(true);
  }
}

/****************************************************************************/
void TagTypeInfo::ExecTagCondAction(long NestCond_,
                                    StackNodeInfo* CurNode_, StackNodeInfo* EncNode_)
{
  if (Sequencer() && !NestCond_ && CurNode_ && EncNode_)
  {
    if (!IsContentTypeTag())
    {
      if (IsNonContentTypeTag())
      {
        Sequencer()->SetTagNestCond(NCTAG_NESTCOND);
        Sequencer()->ResetNestingError();
        _TagContValid = TAGCONTENT_VALID;

        // Recursively append text to previous tags contained in the stack node
        // list if AppendText is true and TextAppend has not been paused
        Sequencer()->RecurseAppend(CurNode_, EncNode_);
      }
      else
        Sequencer()->IncExecTagActionFails();
    }
    else
    {
      int TagNest_ = 0;
      bool Optional_ =
        IsOptionalTag(_ElementPtr ? _ElementPtr->TagAttrib():_TagAttr);

      TagNest_ =
      (_TagTypeFound == EMPTY_TAG ||
        (_TagTypeFound == STD_TAG && _SingletonTagFound)) ? SGL_NESTCOND:
      (_TagTypeFound == STD_TAG && _StdTagFound && !Optional_) ? ST_NESTCOND:
      (_TagTypeFound == STD_TAG && _StdTagFound && Optional_) ? OPTST_NESTCOND:
      NO_NESTCOND;

      // singleton: <single>
      if (TagNest_ == SGL_NESTCOND)
      {
        if (EncNode_ &&
            (EncNode_->SingletonTagFound() || EncNode_->EmptyTagFound()))
        {
          Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);
          Sequencer()->SetTagNestCond(SGL_NESTCOND);
        }
      }

      // start: <pair>
      else if (TagNest_ == ST_NESTCOND)
      {
        if (EncNode_ && EncNode_->StdTagFound())
        {
          Sequencer()->PushTagStack(EncNode_, true);
          Sequencer()->SetTagNestCond(ST_NESTCOND);
        }
      }

      // optional start: <opt-start>
      else if (TagNest_ == OPTST_NESTCOND)
      {
        if (EncNode_ && EncNode_->StdTagFound())
        {
          Sequencer()->PushTagStack(EncNode_, true);
          Sequencer()->SetTagNestCond(OPTST_NESTCOND);
        }
      }

      if (TagNest_ != NO_NESTCOND)
      {
        Sequencer()->ResetNestingError();
        _TagContValid = TAGCONTENT_VALID;
      }
      else
        Sequencer()->IncExecTagActionFails();
    }

    return;
  }

  bool Failed_ = false;
  bool CurPair_ = false;
  bool EncPair_ = false;
  bool CurOpt_ = false;
  bool EncOpt_ = false;

  switch (NestCond_)
  {
    // singleton: <single>
    case SGL_NESTCOND:
      if (EncNode_ &&
          (EncNode_->SingletonTagFound() || EncNode_->EmptyTagFound()))
      {
        Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);
        Sequencer()->SetTagNestCond(SGL_NESTCOND);
        Sequencer()->ResetNestingError();
        _TagContValid = TAGCONTENT_VALID;
      }
      break;

    // start: <pair>
    case ST_NESTCOND:
      if (EncNode_ && EncNode_->StdTagFound())
      {
        Sequencer()->PushTagStack(EncNode_, true);
        Sequencer()->SetTagNestCond(ST_NESTCOND);
        Sequencer()->ResetNestingError();
        _TagContValid = TAGCONTENT_VALID;
      }
      break;

    // optional start: <opt-pair>
    case OPTST_NESTCOND:
      if (EncNode_ && EncNode_->StdTagFound())
      {
        Sequencer()->PushTagStack(EncNode_, true);
        Sequencer()->SetTagNestCond(OPTST_NESTCOND);
        Sequencer()->ResetNestingError();
        _TagContValid = TAGCONTENT_VALID;
      }
      break;

    // singleton-singleton tags: <single>...<single>
    case SGLSGL_NESTCOND:
      if (CurNode_ &&
          (CurNode_->SingletonTagFound() || CurNode_->EmptyTagFound()) &&
          EncNode_ &&
          (EncNode_->SingletonTagFound() || EncNode_->EmptyTagFound()))
      {
        Sequencer()->SetTagNestCond(SGLSGL_NESTCOND);
        CurNode_->LinkForward(EncNode_);
      }
      break;

    // singleton-start tags: <single>...<pair>
    case SGLST_NESTCOND:

    // singleton-optstart tags: <single>...<opt-pair>
    case SGLOPTST_NESTCOND:
      if (CurNode_ &&
          (CurNode_->SingletonTagFound() || CurNode_->EmptyTagFound()) &&
          EncNode_ && EncNode_->StdTagFound())
      {
        Sequencer()->SetTagNestCond(NestCond_);
        CurNode_->LinkForward(EncNode_);
        Sequencer()->PushTagStack(EncNode_, true);
      }
      break;

    // singleton-matching-end tags: <single>...</single>
    case SGLMEND_NESTCOND:
      Sequencer()->SetTagNestCond(SGLMEND_NESTCOND);
      _TagContValid = TAGCONTENT_NOTINCLUDED;
      Sequencer()->AppendNestingError(ENDTAG_FORBIDDEN | SINGLETON_MATCHED);
      break;

    // singleton-unmatched-end tags: <single>...</pair>
    case SGLUMEND_NESTCOND:

    // singleton-end-unmatched-end tags: </single>...</newpair> OR
    //                                   </single>...</single>
    case SGLENDUMEND_NESTCOND:
      if (CurNode_ &&
          (CurNode_->SingletonTagFound() || CurNode_->EmptyTagFound()) &&
          EncNode_ && EncNode_->PairEnderFound())
      {
        Sequencer()->SetTagNestCond(NestCond_);
        Exec_SingletonUnmatchedEnd(NestCond_, CurNode_, EncNode_);
        Sequencer()->ConfirmCursorTagErased();
      }
      break;

    // start-singleton tags: <pair>...<single>
    case STSGL_NESTCOND:

    // optional start-singleton tags: <opt-pair>...<single>
    case OPTSTSGL_NESTCOND:
      if (CurNode_ && CurNode_->StdTagFound() &&
          EncNode_ &&
          (EncNode_->SingletonTagFound() || EncNode_->EmptyTagFound()))
      {
        Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);

        Sequencer()->SetTagNestCond(NestCond_);
        Sequencer()->ResetNestingError();
        _TagContValid = TAGCONTENT_VALID;
      }
      break;

    // singleton-end-singleton tags: </single>...<single>
    case SGLENDSGL_NESTCOND:
      if (CurNode_ &&
          (CurNode_->SingletonTagFound() || CurNode_->EmptyTagFound()) &&
          EncNode_ &&
          (EncNode_->SingletonTagFound() || EncNode_->EmptyTagFound()))
      {
        Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);

        Sequencer()->SetTagNestCond(SGLENDSGL_NESTCOND);
        Sequencer()->ResetNestingError();
        _TagContValid = TAGCONTENT_VALID;
      }
      break;

    // pair-end-singleton tags: </pair>...<single>
    case PAIRENDSGL_NESTCOND:

    // optpair-end-singleton tags: </optpair>...<single>
    case OPTENDSGL_NESTCOND:
      if (CurNode_ && CurNode_->PairEnderFound() &&
          EncNode_ &&
          (EncNode_->SingletonTagFound() || EncNode_->EmptyTagFound()))
      {
        Sequencer()->_CursorStkTag = Sequencer()->PushCursorTag(EncNode_, false);

        Sequencer()->SetTagNestCond(NestCond_);
        Sequencer()->ResetNestingError();
        _TagContValid = TAGCONTENT_VALID;
      }
      break;

    // singleton-end-start tags: </single>...<pair>
    case SGLENDST_NESTCOND:

    // singleton-end-optpair tags: </single>...<opt-pair>
    case SGLENDOPTST_NESTCOND:
      if (CurNode_ &&
          (CurNode_->SingletonTagFound() || CurNode_->EmptyTagFound()) &&
          EncNode_ && EncNode_->StdTagFound())
      {
        Sequencer()->SetTagNestCond(NestCond_);
        Sequencer()->PushTagStack(EncNode_, true);
      }
      break;

    // pair-end-start tags: </pair>...<pair>
    case PAIRENDST_NESTCOND:

    // pair-end-optpair tags: </pair>...<opt-pair>
    case PAIRENDOPTST_NESTCOND:

    // optpair-end-start tags: </opt-pair>...<pair>
    case OPTENDST_NESTCOND:

    // optpair-end-optpair tags: </opt-pair>...<opt-pair>
    case OPTENDOPTST_NESTCOND:
      if (CurNode_ && CurNode_->PairEnderFound() &&
          EncNode_ && EncNode_->StdTagFound())
      {
        Sequencer()->SetTagNestCond(NestCond_);
        Sequencer()->PushTagStack(EncNode_, true);
      }
      break;

    // start-start tags: <pair>...<new-pair>
    case STST_NESTCOND:

    // start-optstart tags: <pair>...<new-optpair>
    case STOPTST_NESTCOND:

    // optional start-start tags: <opt-pair>...<pair>
    case OPTSTST_NESTCOND:

    // optional start-optstart tags: <opt-pair>...<opt-pair>
    case OPTSTOPTST_NESTCOND:
      if (CurNode_ && CurNode_->StdTagFound() &&
          EncNode_ && EncNode_->StdTagFound())
      {
        CurOpt_ = TagTypeInfo::IsOptionalTag(CurNode_->_TagAttr, &CurPair_);
        EncOpt_ = TagTypeInfo::IsOptionalTag(EncNode_->_TagAttr, &EncPair_);
        Sequencer()->SetTagNestCond(NestCond_);

        if (CurOpt_ && CurPair_ && EncOpt_ && EncPair_ &&
            CurNode_->TagStringEqual(EncNode_))
          CurNode_->LinkForward(EncNode_);

        Sequencer()->PushTagStack(EncNode_, true);
      }
      break;

    // start-matching-end tags: <pair>...</pair>
    case STMEND_NESTCOND:

    // optional start-matching-end tags: <opt-pair>...</opt-pair>
    case OPTSTMEND_NESTCOND:
      if (CurNode_ && CurNode_->StdTagFound() &&
          EncNode_ && EncNode_->PairEnderFound())
      {
        Sequencer()->SetTagNestCond(NestCond_);
        Exec_StartTagMatchedEnd(NestCond_, CurNode_, EncNode_);
        Sequencer()->ConfirmCursorTagErased();
      }
      break;

    // start-unmatched-end tags: <pair>...</new-pair>
    case STUMEND_NESTCOND:

    // optional start-unmatched-end tags: <opt-pair>...</pair>
    case OPTSTUMEND_NESTCOND:
      if (CurNode_ && CurNode_->StdTagFound() &&
          EncNode_ && EncNode_->PairEnderFound())
      {
        Sequencer()->SetTagNestCond(NestCond_);
        Exec_StartTagUnmatchedEnd(NestCond_, CurNode_, EncNode_);
        Sequencer()->ConfirmCursorTagErased();
      }
      break;

    // pair-end-unmatched-end tags: </pair>...</newpair> OR
    //                              </pair>...</pair>
    case PAIRENDUMEND_NESTCOND:

    // optpair-end-unmatched-end tags: </opt-pair>...</newpair> OR
    //                                 </opt-pair>...</opt-pair>
    case OPTENDUMEND_NESTCOND:
      if (CurNode_ && CurNode_->PairEnderFound() &&
          EncNode_ && EncNode_->PairEnderFound())
      {
        Sequencer()->SetTagNestCond(NestCond_);
        Exec_PairEndTagUnmatchedEnd(NestCond_, CurNode_, EncNode_);
        Sequencer()->ConfirmCursorTagErased();
      }
      break;

    default:
      Failed_ = true;
      break;
  }

  if (Failed_)
    Sequencer()->IncExecTagActionFails();
  else
    Sequencer()->ResetExecTagActionFails();

  if (CurNode_ && EncNode_)
  {
    // Recursively append text to previous tags contained in the stack node
    // list if AppendText is true and TextAppend has not been paused
    Sequencer()->RecurseAppend(CurNode_, EncNode_);
  }
}

/****************************************************************************/
void TagTypeInfo::SetOpStTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                               StackNodeInfo* EncNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     SeqNum==Ume.BackLink->SeqNum+1;
  //     CurLevel==Ume.CurLevel;
  //     EncLevel==0;
  //   else
  //     SeqNum++;
  //     CurLevel==St.CurLevel;
  //     EncLevel==0;
  //   endif
  //
  //   Tag=HeadAppText(true,true);
  //   OpSt.PopAppContStr(1);
  //   OpSt.AppContStr=OpSt.HeadAppContStr();
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Tag->_TransferNode && Ume==Tag then
  //     Ume.AppendTentToContStr(1);
  //   endif
  //   if (OpSt.AtReqTagCondStopState() &&
  //       OpSt.TagAtImpliedEnd()) then
  //     OpSt.PushAppTentStr(true,OpSt);
  //   endif
  // endif

  if (CurNode_ && EncNode_)
    Sequencer()->SetOpStTagToUnmatchedEndData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetStdTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                              StackNodeInfo* EncNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     SeqNum==Ume.BackLink->SeqNum+1;
  //     CurLevel==Ume.CurLevel;
  //     EncLevel==0;
  //   else
  //     SeqNum++;
  //     CurLevel==St.CurLevel;
  //     EncLevel==0;
  //   endif
  //
  //   Tag=HeadAppText(true,true);
  //   St.PopAppContStr(0);
  //   St.AppContStr=St.HeadAppContStr();
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Tag->_TransferNode && Ume==Tag then
  //     Ume.AppendTentToContStr(1);
  //   endif
  // endif

  if (CurNode_ && EncNode_)
    Sequencer()->SetStdTagToUnmatchedEndData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetSglTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                              StackNodeInfo* EncNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     SeqNum==Ume.BackLink->SeqNum+1;
  //     CurLevel==Ume.CurLevel;
  //     EncLevel==0;
  //   else
  //     SeqNum++;
  //     CurLevel==St.CurLevel;
  //     EncLevel==0;
  //   endif
  //
  //   Tag=HeadAppText(true,true);
  //   Sgl.PopAppContStr(1);
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Tag->_TransferNode && Ume==Tag then
  //     Ume.AppendTentToContStr(1);
  //   endif
  // endif

  if (CurNode_ && EncNode_)
    Sequencer()->SetSglTagToUnmatchedEndData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetMatchedEndTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                                     StackNodeInfo* EncNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     SeqNum==Ume.BackLink->SeqNum+1;
  //     CurLevel==Ume.CurLevel;
  //     EncLevel==0;
  //   else
  //     SeqNum++;
  //     CurLevel==St.CurLevel;
  //     EncLevel==0;
  //   endif
  //
  //   Tag=Pe.PopAppContStr(0);
  //     Pe.AppContStr=Pe.HeadAppContStr();
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Ume.SrchTentAppStk(Tag)==true then
  //     Ume.DetTentAppStk(Tag);
  //     Ume.AppendTentToContStr(1);
  //   endif
  // endif

  if (CurNode_ && EncNode_)
    Sequencer()->SetMatchedEndTagToUnmatchedEndData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetOpStTagToOpStTagData(StackNodeInfo* CurNode_,
                                          StackNodeInfo* EncNode_)
{
  // SeqNum==0;
  // CurLevel==St->EncLevel;
  // IncNestLevel(); EncLevel==CurLevel++;
  //
  // OpSt.PopAppContStr(1);   // 1st param indicates string termination
  // OpSt_.PushAppContStr(HeadAppText,0);
  // OpSt_.AppContStr=OpSt_.HeadAppContStr();
  //   if (OpSt.AtReqTagCondStopState() &&
  //       OpSt.TagAtImpliedEnd()) then
  //     OpSt.PushAppTentStr(true,OpSt);
  //   endif

  if (CurNode_ && EncNode_)
    Sequencer()->SetOpStTagToOpStTagData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetSglTagToSglTagData(StackNodeInfo* CurNode_,
                                        StackNodeInfo* EncNode_)
{
  // SeqNum++;
  // CurLevel==Sgl->CurLevel;
  // EncLevel==0;
  // Sgl.PopAppContStr(1);
  // Sgl_.PushAppContStr(HeadAppText,1);
  // Sgl_.AppContStr=Sgl_.HeadAppContStr();

  if (CurNode_ && EncNode_)
    Sequencer()->SetSglTagToSglTagData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetSglTagToMatchedEndData(StackNodeInfo* CurNode_,
                                            StackNodeInfo* EncNode_)
{
  // SeqNum++;
  // CurLevel==Sgl->CurLevel;
  // EncLevel==0;
  // Sgl.PopAppContStr(1);
  // Me.AppContStr=Me.HeadAppContStr();
  // Me->BackLink.AppContStr=Me->BackLink.HeadAppContStr();

  if (CurNode_ && EncNode_)
    Sequencer()->SetSglTagToMatchedEndData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetStdTagToMatchedEndData(StackNodeInfo* CurNode_,
                                            StackNodeInfo* EncNode_)
{
  // if (Me->BackLink)
  //   SeqNum==Me->BackLink->SeqNum+1;
  //   CurLevel==Me->CurLevel;
  //   EncLevel==0;
  // else
  //   SeqNum++;
  //   CurLevel==Pe.CurLevel;
  //   EncLevel==0;
  // endif
  //
  // St.PopAppContStr(1);
  // Me.AppContStr=Me.HeadAppContStr();
  // Me->BackLink.AppContStr=Me->BackLink.HeadAppContStr();

  if (CurNode_ && EncNode_)
    Sequencer()->SetStdTagToMatchedEndData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetXToSglTagData(StackNodeInfo* CurNode_,
                                   StackNodeInfo* EncNode_)
{
  // SeqNum++;
  // CurLevel==X->CurLevel;
  // EncLevel==0;
  // Sgl.PushAppContStr(HeadAppText,1);
  // Sgl.AppContStr=Sgl.HeadAppContStr();

  if (CurNode_ && EncNode_)
    Sequencer()->SetXToSglTagData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetXToStdTagData(StackNodeInfo* CurNode_,
                                   StackNodeInfo* EncNode_)
{
  // SeqNum++;
  // CurLevel==X->CurLevel;
  // IncNestLevel(); EncLevel==CurLevel++;
  //   if (X == SglTag)
  //     SglTag.PopAppContStr(1);
  // St.PushAppContStr(HeadAppText,0);
  // St.AppContStr=St.HeadAppContStr();

  if (CurNode_ && EncNode_)
    Sequencer()->SetXToStdTagData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetStdTagToStdTagData(StackNodeInfo* CurNode_,
                                        StackNodeInfo* EncNode_)
{
  // SeqNum==0;
  // CurLevel==St->EncLevel;
  // IncNestLevel(); EncLevel==CurLevel++;
  // St_.PushAppContStr(HeadAppText,0);
  // St_.AppContStr=St_.HeadAppContStr();

  if (CurNode_ && EncNode_)
    Sequencer()->SetStdTagToStdTagData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetStdTagToSglTagData(StackNodeInfo* CurNode_,
                                        StackNodeInfo* EncNode_)
{
  // SeqNum==0;
  // CurLevel==St->EncLevel;
  // EncLevel==0;
  // Sgl.PushAppContStr(HeadAppText,1);
  // Sgl.AppContStr=Sgl.HeadAppContStr();

  if (CurNode_ && EncNode_)
    Sequencer()->SetStdTagToSglTagData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetStdTagData(StackNodeInfo* CurNode_)
{
  // SeqNum==0;
  // CurLevel==0;
  // IncNestLevel(); EncLevel==CurLevel++;
  // St.PushAppContStr(HeadAppText,0);
  // St.AppContStr=St.HeadAppContStr();

  if (CurNode_)
    Sequencer()->SetStdTagData(CurNode_);
}

/****************************************************************************/
void TagTypeInfo::SetSingleTagData(StackNodeInfo* CurNode_)
{
  // SeqNum==0;
  // CurLevel==0;
  // EncLevel==0;
  // St.PushAppContStr(HeadAppText,0);
  // St.AppContStr=St.HeadAppContStr();

  if (CurNode_)
    Sequencer()->SetSingleTagData(CurNode_);
}

/****************************************************************************/
void TagTypeInfo::SetNonContentTagData(StackNodeInfo* CurNode_)
{
  // ContentStr+="<!--[if lt IE 7]> // NC Tag
  //                  <style>div.page img.alpha{display:none}</style>
  //              <![endif]-->"
  //
  // SeqNum==0;
  // CurLevel==0;
  // EncLevel==0;

  if (CurNode_)
    Sequencer()->SetNonContentTagData(CurNode_);
}

/****************************************************************************/
void TagTypeInfo::SetNonContentTagData(StackNodeInfo* CurNode_,
                                       StackNodeInfo* EncNode_)
{
  // ContentStr+="<!--[if lt IE 7]> // NC Tag
  //                  <style>div.page img.alpha{display:none}</style>
  //              <![endif]-->"
  //
  // SeqNum++;
  // CurLevel==CurNode->CurLevel;
  // EncLevel==0;

  if (CurNode_ && EncNode_)
    Sequencer()->SetNonContentTagData(CurNode_, EncNode_);
}

/****************************************************************************/
void TagTypeInfo::SetTagCondData(long NestCond_,
                                 StackNodeInfo* CurNode_, StackNodeInfo* EncNode_)
{
  if (!NestCond_)
  {
    if (!IsContentTypeTag() && CurNode_)
    {
      if (IsNonContentTypeTag())
      {
        if (EncNode_ && CurNode_ != EncNode_)
          SetNonContentTagData(CurNode_, EncNode_);
        else
          SetNonContentTagData(CurNode_);
      }
    }
    else if (IsContentTypeTag() && CurNode_ && Sequencer())
    {
      int TagNest_ = 0;
      bool Optional_ =
        IsOptionalTag(_ElementPtr ? _ElementPtr->TagAttrib():_TagAttr);

      TagNest_ =
      (_TagTypeFound == EMPTY_TAG ||
        (_TagTypeFound == STD_TAG && _SingletonTagFound)) ? SGL_NESTCOND:
      (_TagTypeFound == STD_TAG && _StdTagFound && !Optional_) ? ST_NESTCOND:
      (_TagTypeFound == STD_TAG && _StdTagFound && Optional_) ? OPTST_NESTCOND:
      NO_NESTCOND;

      // singleton: <single>
      if (TagNest_ == SGL_NESTCOND)
      {
        SetSingleTagData(CurNode_);
        Sequencer()->SetTagNestCond(SGL_NESTCOND);
      }

      // start: <pair>
      else if (TagNest_ == ST_NESTCOND)
      {
        SetStdTagData(CurNode_);
        Sequencer()->SetTagNestCond(ST_NESTCOND);
      }

      // optional start: <opt-start>
      else if (TagNest_ == OPTST_NESTCOND)
      {
        SetOpStTagData(CurNode_);
        Sequencer()->SetTagNestCond(OPTST_NESTCOND);
      }

      if (TagNest_ != NO_NESTCOND)
      {
        _TagContValid = TAGCONTENT_VALID;
        Sequencer()->ResetNestingError();
        CurNode_->CopyTagTypeStackNodeData(this);
      }
    }

    return;
  }

  switch (NestCond_)
  {
    // singleton-singleton tags: <single>...<single>
    case SGLSGL_NESTCOND:
      SetSglTagToSglTagData(CurNode_, EncNode_);
      Sequencer()->SetTagNestCond(SGLSGL_NESTCOND);
      _TagContValid = TAGCONTENT_NOTINCLUDED;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // singleton-start tags: <single>...<pair>
    case SGLST_NESTCOND:

    // singleton-optstart tags: <single>...<optpair>
    case SGLOPTST_NESTCOND:
      if (NestCond_ == SGLST_NESTCOND)
      {
        SetXToStdTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(SGLST_NESTCOND);
      }
      else
      {
        SetXToOpStTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(SGLOPTST_NESTCOND);
      }

      _TagContValid = TAGCONTENT_NOTINCLUDED;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // singleton-matching-end tags: <single>...</single>
    case SGLMEND_NESTCOND:
      SetSglTagToMatchedEndData(CurNode_, EncNode_);
      Sequencer()->SetTagNestCond(SGLMEND_NESTCOND);
      _TagContValid = TAGCONTENT_NOTINCLUDED;
      Sequencer()->AppendNestingError(ENDTAG_FORBIDDEN | SINGLETON_MATCHED);
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // singleton-unmatched-end tags: <single>...</pair>
    case SGLUMEND_NESTCOND:
      SetSglTagToUnmatchedEndData(CurNode_, EncNode_);
      Sequencer()->SetTagNestCond(SGLUMEND_NESTCOND);
      _TagContValid = TAGCONTENT_NOTINCLUDED;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // singleton-end-unmatched-end tags: </single>...</newpair> OR
    //                                   </single>...</single>
    case SGLENDUMEND_NESTCOND:
      SetMatchedEndTagToUnmatchedEndData(CurNode_, EncNode_);
      Sequencer()->SetTagNestCond(SGLENDUMEND_NESTCOND);
      _TagContValid = TAGCONTENT_NOTINCLUDED;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // start-singleton tags: <pair>...<single>
    case STSGL_NESTCOND:

    // optional start-singleton tags: <opt-pair>...<single>
    case OPTSTSGL_NESTCOND:

    // singleton-end-singleton tags: </single>...<single>
    case SGLENDSGL_NESTCOND:

    // pair-end-singleton tags: </pair>...<single>
    case PAIRENDSGL_NESTCOND:

    // optpair-end-singleton tags: </optpair>...<single>
    case OPTENDSGL_NESTCOND:
      if (NestCond_ == STSGL_NESTCOND)
      {
        SetStdTagToSglTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(STSGL_NESTCOND);
      }
      else if (NestCond_ == OPTSTSGL_NESTCOND)
      {
        SetOpStTagToSglTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(OPTSTSGL_NESTCOND);
      }
      else if (NestCond_ == SGLENDSGL_NESTCOND)
      {
        SetXToSglTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(SGLENDSGL_NESTCOND);
      }
      else if (NestCond_ == PAIRENDSGL_NESTCOND)
      {
        SetXToSglTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(PAIRENDSGL_NESTCOND);
      }
      else
      {
        SetXToSglTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(OPTENDSGL_NESTCOND);
      }

      _TagContValid = TAGCONTENT_VALID;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // singleton-end-start tags: </single>...<pair>
    case SGLENDST_NESTCOND:

    // singleton-end-optpair tags: </single>...<optpair>
    case SGLENDOPTST_NESTCOND:
      if (NestCond_ == SGLENDST_NESTCOND)
      {
        SetPairEndToStdTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(SGLENDST_NESTCOND);
      }
      else if (NestCond_ == SGLENDOPTST_NESTCOND)
      {
        SetPairEndToOpStTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(SGLENDOPTST_NESTCOND);
      }

      _TagContValid = TAGCONTENT_VALID;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // start-start tags: <pair>...<new-pair>
    case STST_NESTCOND:

    // start-optstart tags: <pair>...<new-optpair>
    case STOPTST_NESTCOND:

    // pair-end-start tags: </pair>...<pair>
    case PAIRENDST_NESTCOND:

    // pair-end-optpair tags: </pair>...<optpair>
    case PAIRENDOPTST_NESTCOND:

    // optpair-end-start tags: </optpair>...<pair>
    case OPTENDST_NESTCOND:

    // optpair-end-optpair tags: </optpair>...<optpair>
    case OPTENDOPTST_NESTCOND:

    // singleton: <single>
    case SGL_NESTCOND:

    // start: <pair>
    case ST_NESTCOND:

    // optional start: <opt-start>
    case OPTST_NESTCOND:
      if (NestCond_ == SGL_NESTCOND)
      {
        SetSingleTagData(CurNode_);
        Sequencer()->SetTagNestCond(SGL_NESTCOND);
      }
      else if (NestCond_ == ST_NESTCOND)
      {
        SetStdTagData(CurNode_);
        Sequencer()->SetTagNestCond(ST_NESTCOND);
      }
      else if (NestCond_ == OPTST_NESTCOND)
      {
        SetOpStTagData(CurNode_);
        Sequencer()->SetTagNestCond(OPTST_NESTCOND);
      }
      else if (NestCond_ == STST_NESTCOND)
      {
        SetStdTagToStdTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(STST_NESTCOND);
      }
      else if (NestCond_ == STOPTST_NESTCOND)
      {
        SetStdTagToOpStTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(STOPTST_NESTCOND);
      }
      else if (NestCond_ == PAIRENDST_NESTCOND)
      {
        SetPairEndToStdTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(PAIRENDST_NESTCOND);
      }
      else if (NestCond_ == PAIRENDOPTST_NESTCOND)
      {
        SetPairEndToOpStTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(PAIRENDOPTST_NESTCOND);
      }
      else if (NestCond_ == OPTENDST_NESTCOND)
      {
        SetPairEndToStdTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(OPTENDST_NESTCOND);
      }
      else
      {
        SetPairEndToOpStTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(OPTENDOPTST_NESTCOND);
      }

      _TagContValid = TAGCONTENT_VALID;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // start-matching-end tags: <pair>...</pair>
    case STMEND_NESTCOND:
      SetStdTagToMatchedEndData(CurNode_, EncNode_);
      Sequencer()->SetTagNestCond(STMEND_NESTCOND);
      _TagContValid = TAGCONTENT_VALID;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // start-unmatched-end tags: <pair>...</new-pair>
    case STUMEND_NESTCOND:
      SetStdTagToUnmatchedEndData(CurNode_, EncNode_);
      Sequencer()->SetTagNestCond(STUMEND_NESTCOND);
      _TagContValid = TAGCONTENT_VALID;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // pair-end-unmatched-end tags: </pair>...</newpair> OR
    //                              </pair>...</pair>
    case PAIRENDUMEND_NESTCOND:
      SetMatchedEndTagToUnmatchedEndData(CurNode_, EncNode_);
      Sequencer()->SetTagNestCond(PAIRENDUMEND_NESTCOND);
      _TagContValid = TAGCONTENT_VALID;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // optional start-start tags: <opt-pair>...<pair>
    case OPTSTST_NESTCOND:

    // optional start-optstart tags: <opt-pair>...<optpair>
    case OPTSTOPTST_NESTCOND:
      if (NestCond_ == OPTSTST_NESTCOND)
      {
        SetOpStTagToStdTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(OPTSTST_NESTCOND);
      }
      else
      {
        SetOpStTagToOpStTagData(CurNode_, EncNode_);
        Sequencer()->SetTagNestCond(OPTSTOPTST_NESTCOND);
      }

      _TagContValid = TAGCONTENT_VALID;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // optional start-matching-end tags: <opt-pair>...</opt-pair>
    case OPTSTMEND_NESTCOND:
      SetOpStTagToMatchedEndData(CurNode_, EncNode_);
      Sequencer()->SetTagNestCond(OPTSTMEND_NESTCOND);
      _TagContValid = TAGCONTENT_VALID;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // optional start-unmatched-end tags: <opt-pair>...</pair>
    case OPTSTUMEND_NESTCOND:
      SetOpStTagToUnmatchedEndData(CurNode_, EncNode_);
      Sequencer()->SetTagNestCond(OPTSTUMEND_NESTCOND);
      _TagContValid = TAGCONTENT_VALID;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;

    // optpair-end-unmatched-end tags: </optpair>...</newpair> OR
    //                                 </optpair>...</optpair>
    case OPTENDUMEND_NESTCOND:
      SetMatchedEndTagToUnmatchedEndData(CurNode_, EncNode_);
      Sequencer()->SetTagNestCond(OPTENDUMEND_NESTCOND);
      _TagContValid = TAGCONTENT_VALID;
      Sequencer()->ResetNestingError();
      CurNode_->CopyTagTypeStackNodeData(this);
      break;
  }
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::operator = (const TagTypeInfo& Obj_)
{
  if (this != &Obj_)
  {  
    delete _ElementStr;
    
    if (Obj_._ElementStr && Obj_._TagSpecified)
      _ElementStr = new ChrString(*Obj_._ElementStr);
    else
      _ElementStr = NULL;

    _ElementPtr = Obj_._ElementPtr;

    // _TagCntData not copied since specific to object type
    if (!_TagCntData)
      _TagCntData = TagCountData::Instance();
    _TagBrkMatrix = _TagCntData->GiveTagBrkMatrix();

    _TagSpecified = Obj_._TagSpecified;
    _LookupRefTag = false;
    _DictionaryTag = Obj_._DictionaryTag;
    _SingletonTagAllowed = Obj_._SingletonTagAllowed;
    _EmptyTagAllowed = Obj_._EmptyTagAllowed;
    _DocType = Obj_._DocType;
    _EofFound = Obj_._EofFound;
    _TagAttr = Obj_._TagAttr;

    _TagBrkIndex = Obj_._TagBrkIndex;
    _TagBrkMax = TagEnums::TAGLIST_MAX_ENTRIES;
    _AltIndex = Obj_._AltIndex;
    _TagBrkExtend1 = Obj_._TagBrkExtend1;
    _TagBrkExtend2 = Obj_._TagBrkExtend2;

    _DataSpecified = Obj_._DataSpecified;

    if (Obj_._SearchResult)
    {
      if (!_SearchResult)
        _SearchResult = new TagSearchResult(*Obj_._SearchResult);
      else
        *_SearchResult = *Obj_._SearchResult;

      _SearchResult->SetParent(this);
    }

    _Sequencer = Obj_._SequencerSet ? Obj_._Sequencer:
                 (Obj_._SequencerAlloc && Obj_._Sequencer) ?
                     (new TagLinksSequencer(*Obj_._Sequencer)):NULL;
    _SequencerSet = Obj_._SequencerSet;
    _SequencerAlloc = Obj_._SequencerAlloc;
    _HtmlSpecialCase = Obj_._HtmlSpecialCaseSet ? Obj_._HtmlSpecialCase:
                 (Obj_._HtmlSpecialCaseAlloc && Obj_._HtmlSpecialCase) ?
                     (new HtmlSpecialCaseData(*Obj_._HtmlSpecialCase)):NULL;
    _HtmlSpecialCaseSet = Obj_._HtmlSpecialCaseSet;
    _HtmlSpecialCaseAlloc = Obj_._HtmlSpecialCaseAlloc;

    _TagFound = Obj_._TagFound;
    _ContentTypeTag = Obj_._ContentTypeTag;
    _NonContentTypeTag = Obj_._NonContentTypeTag;
    _HasPairContent = Obj_._HasPairContent;
    _TagTypeFound = Obj_._TagTypeFound;

    _EndBrkFound = Obj_._EndBrkFound;
    _TagSpcFound = Obj_._TagSpcFound;
    _SingletonTagFound = Obj_._SingletonTagFound;
    _EmptyTagFound = Obj_._EmptyTagFound;
    _ContainerTagFound = Obj_._ContainerTagFound;
    _StdTagFound = Obj_._StdTagFound;
    _PairEnderFound = Obj_._PairEnderFound;
    _ForbiddenUnspec = Obj_._ForbiddenUnspec;
    _DumpTagContents = Obj_._DumpTagContents;

    if (_IO)
      *_IO = Obj_._IO;
    else
      _IO = new TagDumpStreamData(Obj_._IO);

    if (_Sequencer && _SequencerAlloc)
      _Sequencer->SetParent(this);

    if (_HtmlSpecialCase && _HtmlSpecialCaseAlloc)
      _HtmlSpecialCase->SetParent(this);
  }

  return *this;
}

/****************************************************************************/
void TagTypeInfo::CopyFromReferenceObject(const TagTypeInfo& Obj_)
{
  if (this != &Obj_)
  {  
    // _TagCntData not copied since specific to object type
    if (!_TagCntData)
      _TagCntData = TagCountData::Instance();
    _TagBrkMatrix = _TagCntData->GiveTagBrkMatrix();

    _ElementPtr = Obj_._ElementPtr;
    _TagAttr = Obj_._TagAttr;
    _ContainerTagFound = (_TagAttr & TagEnums::TAG_CONTAINER);
  }
}

/****************************************************************************/
bool TagTypeInfo::operator == (const TagTypeInfo& Obj_) const
{
  if (this == &Obj_)
    return true;

  return IsEqual(Obj_);
}

/****************************************************************************/
// Raw tag lookup equality method
//
bool TagTypeInfo::TagLookupEqual(const TagTypeInfo& Obj_) const
{
  if (this != &Obj_)
  {
    _AltIndex = NO_ALTINDEX;
    bool TagEqual_ =
      (_ElementStr && Obj_._ElementStr &&
       _ElementStr->StriComp(*Obj_._ElementStr) == 0);

    bool IndexEqual_ =
      (_TagBrkIndex == Obj_._TagBrkIndex ||
       (_TagBrkIndex == TagEnums::STD_DEX &&
        Obj_._TagBrkIndex == TagEnums::EMPTY_DEX) ||
       (_TagBrkIndex == TagEnums::EMPTY_DEX &&
        Obj_._TagBrkIndex == TagEnums::STD_DEX) ||

       (_TagBrkIndex == TagEnums::COMMENT_DEX &&
        Obj_._TagBrkIndex == TagEnums::NESTEDCOMMENT1_DEX) ||
       (_TagBrkIndex == TagEnums::COMMENT_DEX &&
        Obj_._TagBrkIndex == TagEnums::NESTEDCOMMENT2_DEX) ||

       (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX &&
        Obj_._TagBrkIndex == TagEnums::COPENCOMMENTBLK_DEX) ||
       (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX &&
        Obj_._TagBrkIndex == TagEnums::CCLOSECOMMENTBLK_DEX));

    bool ElmEqual_ =
      Obj_._LookupRefTag ? true:
      (_ElementPtr && Obj_._ElementPtr) ?
           _ElementPtr->TagLookupEqual(*Obj_._ElementPtr):
           (!_ElementPtr && !Obj_._ElementPtr);

    bool Equal_ = (TagEqual_ && IndexEqual_ && ElmEqual_);

    if (Equal_)
    {
      if (_TagBrkIndex == TagEnums::STD_DEX &&
          Obj_._TagBrkIndex == TagEnums::EMPTY_DEX)
        _AltIndex = SINGLE_EMPTY;
      else if (_TagBrkIndex == TagEnums::EMPTY_DEX &&
               Obj_._TagBrkIndex == TagEnums::STD_DEX)
        _AltIndex = EMPTY_SINGLE;
      else if (_TagBrkIndex == TagEnums::COMMENT_DEX &&
               Obj_._TagBrkIndex == TagEnums::NESTEDCOMMENT1_DEX)
      {
        _AltIndex = COMMENT_NESTEDCMNT1;
        _TagBrkExtend1 = TagEnums::NESTEDCOMMENT1_DEX;
        _TagBrkExtend2 = TagEnums::NESTEDCOMMENT2_DEX;
      }
      else if (_TagBrkIndex == TagEnums::COMMENT_DEX &&
               Obj_._TagBrkIndex == TagEnums::NESTEDCOMMENT2_DEX)
      {
        _AltIndex = COMMENT_NESTEDCMNT2;
        _TagBrkExtend1 = TagEnums::NESTEDCOMMENT1_DEX;
        _TagBrkExtend2 = TagEnums::NESTEDCOMMENT2_DEX;
      }
      else if (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX &&
               Obj_._TagBrkIndex == TagEnums::COPENCOMMENTBLK_DEX)
      {
        _AltIndex = CCMNT_COPENCMNTBLK;
        _TagBrkExtend1 = TagEnums::COPENCOMMENTBLK_DEX;
        _TagBrkExtend2 = TagEnums::CCLOSECOMMENTBLK_DEX;
      }
      else if (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX &&
               Obj_._TagBrkIndex == TagEnums::CCLOSECOMMENTBLK_DEX)
      {
        _AltIndex = CCMNT_CCLOSECMNTBLK;
        _TagBrkExtend1 = TagEnums::COPENCOMMENTBLK_DEX;
        _TagBrkExtend2 = TagEnums::CCLOSECOMMENTBLK_DEX;
      }
    }

    return Equal_;
  }

  return true;
}

/****************************************************************************/
bool TagTypeInfo::IsEqual(const TagTypeInfo& Obj_) const
{
  if (this != &Obj_)
  {
    int TagCntMask_ = ~TagEnums::TAG_CONTAINER;
    _AltIndex = NO_ALTINDEX;

    bool Equal_ =
      _TagSpecified == Obj_._TagSpecified &&
      (_ElementStr && Obj_._ElementStr &&
       _ElementStr->StriComp(*Obj_._ElementStr) == 0) &&

      (((_TagAttr & TagEnums::TAG_SPECIFIC) ||
        ((_TagAttr & TagEnums::SINGLETON_TAGATTR) &&
         (Obj_._TagAttr & TagEnums::ALLOW_SINGLETON_TAG)) ||
        ((_TagAttr & TagEnums::EMPTY_TAGATTR) &&
         (Obj_._TagAttr & TagEnums::ALLOW_EMPTY_TAG)) ||
        (_ForbiddenUnspec &&
         (_TagAttr & TagEnums::SINGLETON_TAGATTR) &&
         ((Obj_._TagAttr & TagEnums::PAIR_TAGATTR) ||
          (Obj_._TagAttr & TagEnums::OPTIONALPAIR_TAGATTR)))) ||
       (((_TagAttr & TagEnums::SINGLETON_TAGATTR) ||
         (_TagAttr & TagEnums::EMPTY_TAGATTR)) &&
        ((Obj_._TagAttr & TagEnums::SINGLETON_TAGATTR) ||
         (Obj_._TagAttr & TagEnums::EMPTY_TAGATTR))) ||
       ((_TagAttr & TagEnums::PAIR_ENDER) &&
        (Obj_._TagAttr & TagEnums::PAIR_ENDER)) ||
       (_TagAttr & TagCntMask_) == (Obj_._TagAttr & TagCntMask_)) &&

      (_TagBrkIndex == Obj_._TagBrkIndex ||
       (_TagBrkIndex == TagEnums::STD_DEX &&
        Obj_._TagBrkIndex == TagEnums::EMPTY_DEX) ||
       (_TagBrkIndex == TagEnums::EMPTY_DEX &&
        Obj_._TagBrkIndex == TagEnums::STD_DEX) ||

       (_TagBrkIndex == TagEnums::COMMENT_DEX &&
        Obj_._TagBrkIndex == TagEnums::NESTEDCOMMENT1_DEX) ||
       (_TagBrkIndex == TagEnums::COMMENT_DEX &&
        Obj_._TagBrkIndex == TagEnums::NESTEDCOMMENT2_DEX) ||

       (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX &&
        Obj_._TagBrkIndex == TagEnums::COPENCOMMENTBLK_DEX) ||
       (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX &&
        Obj_._TagBrkIndex == TagEnums::CCLOSECOMMENTBLK_DEX));

    if (Equal_)
    {
      if (_TagBrkIndex == TagEnums::STD_DEX &&
          Obj_._TagBrkIndex == TagEnums::EMPTY_DEX)
        _AltIndex = SINGLE_EMPTY;
      else if (_TagBrkIndex == TagEnums::EMPTY_DEX &&
               Obj_._TagBrkIndex == TagEnums::STD_DEX)
        _AltIndex = EMPTY_SINGLE;
      else if (_TagBrkIndex == TagEnums::COMMENT_DEX &&
               Obj_._TagBrkIndex == TagEnums::NESTEDCOMMENT1_DEX)
      {
        _AltIndex = COMMENT_NESTEDCMNT1;
        _TagBrkExtend1 = TagEnums::NESTEDCOMMENT1_DEX;
        _TagBrkExtend2 = TagEnums::NESTEDCOMMENT2_DEX;
      }
      else if (_TagBrkIndex == TagEnums::COMMENT_DEX &&
               Obj_._TagBrkIndex == TagEnums::NESTEDCOMMENT2_DEX)
      {
        _AltIndex = COMMENT_NESTEDCMNT2;
        _TagBrkExtend1 = TagEnums::NESTEDCOMMENT1_DEX;
        _TagBrkExtend2 = TagEnums::NESTEDCOMMENT2_DEX;
      }
      else if (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX &&
               Obj_._TagBrkIndex == TagEnums::COPENCOMMENTBLK_DEX)
      {
        _AltIndex = CCMNT_COPENCMNTBLK;
        _TagBrkExtend1 = TagEnums::COPENCOMMENTBLK_DEX;
        _TagBrkExtend2 = TagEnums::CCLOSECOMMENTBLK_DEX;
      }
      else if (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX &&
               Obj_._TagBrkIndex == TagEnums::CCLOSECOMMENTBLK_DEX)
      {
        _AltIndex = CCMNT_CCLOSECMNTBLK;
        _TagBrkExtend1 = TagEnums::COPENCOMMENTBLK_DEX;
        _TagBrkExtend2 = TagEnums::CCLOSECOMMENTBLK_DEX;
      }
    }

    return Equal_;
  }

  return true;
}

/****************************************************************************/
bool TagTypeInfo::IsParentOf(const TagSearchResult* Child_) const
{
  return (Child_ ? Child_->HasThisParent(this):false);
}

/****************************************************************************/
bool TagTypeInfo::IsParentOf(const AppTextNode* Child_) const
{
  return (Child_ ? Child_->HasThisParent(this):false);
}

/****************************************************************************/
bool TagTypeInfo::IsParentOf(const StackNodeInfo* Child_) const
{
  return (Child_ ? Child_->HasThisParent(this):false);
}

/****************************************************************************/
bool TagTypeInfo::IsParentOf(const TagLinksSequencer* Child_) const
{
  return (Child_ ? Child_->HasThisParent(this):false);
}

/****************************************************************************/
bool TagTypeInfo::IsParentOf(const HtmlSpecialCaseData* Child_) const
{
  return (Child_ ? Child_->HasThisParent(this):false);
}

/****************************************************************************/
ChrString* TagTypeInfo::GiveEndTagTermStr(int x)
{
  return
  (
    (x == TagEnums::COMMENT_DEX)        ? (new ChrString(COMMENTCLOSE_STR)):
    (x == TagEnums::NESTEDCOMMENT2_DEX) ? (new ChrString(NESTEDCOMMENT2CLOSE_STR)):
    (x == TagEnums::EMPTY_DEX)          ? (new ChrString(EMPTYCLOSE_STR)):
    (x == TagEnums::SCRIPT_DEX)         ? (new ChrString(SCRIPTCLOSE_STR)):
                                          NULL
  );
}

/****************************************************************************/
bool TagTypeInfo::IsEndTagTermType(int x)
{
  return
  (
    x == TagEnums::COMMENT_DEX ||
    x == TagEnums::NESTEDCOMMENT2_DEX ||
    x == TagEnums::EMPTY_DEX ||
    x == TagEnums::SCRIPT_DEX
  );
}

/****************************************************************************/
bool TagTypeInfo::IsNestedTagType(int x, bool NestedCmnt_)
{
  return
  (
   (NestedCmnt_ &&
    x == TagEnums::COMMENT_DEX) ||
    x == TagEnums::NESTEDCOMMENT1_DEX ||
    x == TagEnums::NESTEDCOMMENT2_DEX ||
    x == TagEnums::CCOMMENTBLK_DEX ||
    x == TagEnums::COPENCOMMENTBLK_DEX ||
    x == TagEnums::CCLOSECOMMENTBLK_DEX
  );
}

/****************************************************************************/
bool TagTypeInfo::IsNoElementNameTagType(int x)
{
  return
  (
    x == TagEnums::COMMENT_DEX ||
    x == TagEnums::SCRIPT_DEX ||
    x == TagEnums::CCOMMENTBLK_DEX ||
    x == TagEnums::COPENCOMMENTBLK_DEX ||
    x == TagEnums::CCLOSECOMMENTBLK_DEX ||
    x == TagEnums::CPPCOMMENTLINEOPEN_DEX
  );
}

/****************************************************************************/
bool TagTypeInfo::IsIntraContentTagType(int x)
{
  return
  (
    x == TagEnums::COMMENT_DEX ||
    x == TagEnums::NESTEDCOMMENT1_DEX ||
    x == TagEnums::NESTEDCOMMENT2_DEX ||
    x == TagEnums::SCRIPT_DEX ||
    x == TagEnums::CCOMMENTBLK_DEX ||
    x == TagEnums::COPENCOMMENTBLK_DEX ||
    x == TagEnums::CCLOSECOMMENTBLK_DEX ||
    x == TagEnums::CPPCOMMENTLINEOPEN_DEX
  );
}

/****************************************************************************/
bool TagTypeInfo::IsSingleDelimTagType(int x)
{
  return
  (
    x == TagEnums::DOCTYPE_DEX ||
    x == TagEnums::NESTEDCOMMENT1_DEX ||
    x == TagEnums::NESTEDCOMMENT2_DEX
  );
}

/****************************************************************************/
bool TagTypeInfo::SetContentTypeTag(const char* start_, const char* end_)
{
  _HasPairContent = _ContentTypeTag = IsContentTypeTag(start_, end_);
  _NonContentTypeTag = IsNonContentTypeTag(start_, end_);
  return _ContentTypeTag;
}

/****************************************************************************/
void TagTypeInfo::SetShowDebugDump(bool Flag_, bool ShowErased_)
{
  _ShowDebugDump = Flag_;
  _ShowErasedNodes = ShowErased_;
}

/****************************************************************************/
void TagTypeInfo::SetShowNewLines(bool Flag_)
{
  _ShowNewLines = Flag_;
}

/****************************************************************************/
bool TagTypeInfo::ShowDebugDump()
{
  return _ShowDebugDump;
}

/****************************************************************************/
bool TagTypeInfo::ShowErasedNodes()
{
  return _ShowErasedNodes;
}

/****************************************************************************/
bool TagTypeInfo::ShowNewLines()
{
  return _ShowNewLines;
}

/****************************************************************************/
int TagTypeInfo::GiveTagPosForTagType(int tagtype_)
{
  return
  (
    (tagtype_ == TagEnums::END_TAG) ? TagEnums::CLOSING:
    ((tagtype_ == TagEnums::STD_TAG) ||
     (tagtype_ == TagEnums::EMPTY_TAG)) ? TagEnums::OPENING:
                                          TagEnums::NO_TAGPOS
  );
}

/****************************************************************************/
bool TagTypeInfo::IsContentTypeTag(const char* start_, const char* end_)
{
  bool ret = TagBracketList::IsEmptyTag(start_, end_) ||
             TagBracketList::IsStdTag(start_, end_) ||
             TagBracketList::IsEndTag(start_, end_);
  return ret;
}

/****************************************************************************/
bool TagTypeInfo::IsNonContentTypeTag(const char* start_, const char* end_)
{
  bool ret = TagBracketList::IsScriptTag(start_, end_) ||
             TagBracketList::IsNestedCommentTag(start_, end_, 3) ||
             TagBracketList::IsCommentTag(start_, end_) ||
             TagBracketList::IsDocTypeTag(start_, end_) ||
             TagBracketList::IsCCommentBlkTag(start_, end_) ||
             TagBracketList::IsNestedCBlkTag(start_, end_, 3) ||
             TagBracketList::IsCppTag(start_, end_);
  return ret;
}

/****************************************************************************/
Ulong TagTypeInfo::SearchRunNumber() const
{
  return Sequencer()->SearchRunNumber();
}

/****************************************************************************/
void TagTypeInfo::ResetSearchRunNumber()
{
  Sequencer()->ResetSearchRunNumber();
}

/****************************************************************************/
long TagTypeInfo::ExecTagActionFails() const
{
  return Sequencer()->ExecTagActionFails();
}

/****************************************************************************/
long TagTypeInfo::StdTagCount() const
{
  return Sequencer()->_StdTagCount;
}

/****************************************************************************/
long TagTypeInfo::NestedCmntTagCount() const
{
  return Sequencer()->_NestedCmntTagCount;
}

/****************************************************************************/
long TagTypeInfo::CopenCmntTagCount() const
{
  return Sequencer()->_CopenCmntTagCount;
}

/****************************************************************************/
long TagTypeInfo::NestingError() const
{
  return Sequencer()->_NestingError;
}

/****************************************************************************/
long TagTypeInfo::TagNestCondition() const
{
  return Sequencer()->_TagNestCond;
}

/****************************************************************************/
void TagTypeInfo::ResetTagInfo()
{
  // _TagCntData not copied since specific to object type
  if (!_TagCntData)
    _TagCntData = TagCountData::Instance();
  _TagBrkMatrix = _TagCntData->GiveTagBrkMatrix();

  if (!_DictionaryTag)
  {
    _ElementPtr = NULL;
    _TagAttr = 0;
  }
}

/****************************************************************************/
bool TagTypeInfo::SetTagInfoFound(TagTypeInfo* TagInfo_, bool Found_)
{
  if (TagInfo_ && Found_ && !_DictionaryTag)
  {
    CopyFromReferenceObject(*TagInfo_);  // Assigning specified tag object if found

    if (_ElementPtr)
      _TagAttr = _ElementPtr->TagAttrib();

    _TagSpecified = false;  // Searching for tag in text so not specified
    Found_ = true;
  }
  else
    Found_ = false;

  return Found_;
}

/****************************************************************************/
bool TagTypeInfo::TagSequenceCheck(const char* ptr_, long* IndexPos_,
                                   long cur_, long slen_, long endpt_)
{
  long y;
  long SkipEndDex_ = TagEnums::CCLOSECOMMENTBLK_DEX;
  long CommentDex_ = TagEnums::COMMENT_DEX;
  long CCmntBlkDex_ = TagEnums::CCOMMENTBLK_DEX;
  long CppCmntLineDex_ = TagEnums::CPPCOMMENTLINEOPEN_DEX;
  long SkipStartDex1_ = TagEnums::COPENCOMMENTBLK_DEX;
  long SkipStartDex2_ = TagEnums::CPPCOMMENTLINEOPEN_DEX;

  // end brk sequential error detection omitted for defined tags
  // with no '>' standard end brk
  if (cur_ != SkipEndDex_)
  {
    // find first end brk following detected start brk
    const char* pos = (cur_ == CommentDex_) ?
                          (const char*)::strstr(ptr_, COMMENTCLOSE_STR):
                      (cur_ == CCmntBlkDex_) ?
                          (const char*)::strstr(ptr_, CCOMMENTBLKCLOSE_STR):
                      (cur_ == CppCmntLineDex_) ?
                          (const char*)::strstr(ptr_, CPPCOMMENTLINECLOSE_STR):
                          (const char*)::memchr(ptr_, STDTAG_CLOSE_CH, strlen(ptr_));

    if (pos)
    {
      y = slen_ + (pos - ptr_);

      // end brk cannot lie within start and end pts of detected brks
      if (slen_ <= y && y < endpt_)
      {
        IndexPos_[cur_] = 0;
        IndexPos_[cur_+1] = 0;
        return false;
      }
    }
  }

  // start brk sequential error detection omitted for defined tags
  // with no '<' standard start brk
  if (cur_ != SkipStartDex1_ && cur_ != SkipStartDex2_)
  {
    // find first start brk following detected start brk
    const char* pos = (cur_ == CommentDex_) ?
                          (const char*)::strstr(ptr_, COMMENTOPEN_STR):
                      (cur_ == CCmntBlkDex_) ?
                          (const char*)::strstr(ptr_, CCOMMENTBLKOPEN_STR):
                      (cur_ == CppCmntLineDex_) ?
                          (const char*)::strstr(ptr_, CPPCOMMENTLINEOPEN_STR):
                          (const char*)::memchr(ptr_, STDTAG_OPEN_CH, strlen(ptr_));

    if (pos)
    {
      y = slen_ + (pos - ptr_);

      // start brk cannot lie within start and end pts of detected brks
      if (slen_ <= y && y < endpt_)
      {
        IndexPos_[cur_] = 0;
        IndexPos_[cur_+1] = 0;
        return false;
      }
    }
  }

  return true;
}

/****************************************************************************/
void TagTypeInfo::InitCppLangTagSwitches(TagSwitch& Switch_)
{
  _TagInitSelected = CPPCOMMENTLINE_TAG;
  _TagInitConfirmed = false;

  Switch_._IsHtml = false;
  Switch_._IsEnd = false;
  Switch_._IsScript = false;
  Switch_._IsDoc = false;
  Switch_._IsComment = false;
  Switch_._IsNestedComment = false;
  Switch_._IsSingle = false;
  Switch_._IsCLang = true;
  Switch_._IsCppLang = true;

  Switch_._AllowEnd = false;
  Switch_._AllowNestedComment = false;
  Switch_._AllowCLang = true;

  Switch_._HasEndError = false;
  Switch_._HasNestedCommentError = false;
  Switch_._HasCLangError = false;

  Switch_._EndErrorCode = 0;
  Switch_._NestedCommentErrorCode = 0;
  Switch_._CLangErrorCode = 0;

  Switch_._RetrieveEnd = false;
  Switch_._RetrieveComment = false;
  Switch_._RetrieveNestedComment = false;
  Switch_._RetrieveScript = false;
  Switch_._RetrieveCLang = true;
  Switch_._RetrieveCppLang = true;
}

/****************************************************************************/
void TagTypeInfo::InitCLangTagSwitches(TagSwitch& Switch_)
{
  _TagInitSelected = CCOMMENTBLK_TAG;
  _TagInitConfirmed = false;

  Switch_._IsHtml = false;
  Switch_._IsEnd = false;
  Switch_._IsScript = false;
  Switch_._IsDoc = false;
  Switch_._IsComment = false;
  Switch_._IsNestedComment = false;
  Switch_._IsSingle = false;
  Switch_._IsCLang = true;
  Switch_._IsCppLang = false;

  Switch_._AllowEnd = false;
  Switch_._AllowNestedComment = false;
  Switch_._AllowCLang = true;

  Switch_._HasEndError = false;
  Switch_._HasNestedCommentError = false;
  Switch_._HasCLangError = false;

  Switch_._EndErrorCode = 0;
  Switch_._NestedCommentErrorCode = 0;
  Switch_._CLangErrorCode = 0;

  Switch_._RetrieveEnd = false;
  Switch_._RetrieveComment = false;
  Switch_._RetrieveNestedComment = false;
  Switch_._RetrieveScript = false;
  Switch_._RetrieveCLang = true;
  Switch_._RetrieveCppLang = false;
}

/****************************************************************************/
void TagTypeInfo::InitCommentTagSwitches(TagSwitch& Switch_)
{
  _TagInitSelected = COMMENT_TAG;
  _TagInitConfirmed = false;

  Switch_._IsHtml = true;
  Switch_._IsEnd = false;
  Switch_._IsScript = false;
  Switch_._IsDoc = false;
  Switch_._IsComment = true;
  Switch_._IsNestedComment = true;
  Switch_._IsSingle = false;
  Switch_._IsCLang = false;
  Switch_._IsCppLang = false;

  Switch_._AllowEnd = false;
  Switch_._AllowNestedComment = true;
  Switch_._AllowCLang = false;

  Switch_._HasEndError = false;
  Switch_._HasNestedCommentError = false;
  Switch_._HasCLangError = false;

  Switch_._EndErrorCode = 0;
  Switch_._NestedCommentErrorCode = 0;
  Switch_._CLangErrorCode = 0;

  Switch_._RetrieveEnd = false;
  Switch_._RetrieveComment = true;
  Switch_._RetrieveNestedComment = true;
  Switch_._RetrieveScript = false;
  Switch_._RetrieveCLang = false;
  Switch_._RetrieveCppLang = false;
}

/****************************************************************************/
void TagTypeInfo::InitNestedCommentTagSwitches(TagSwitch& Switch_)
{
  _TagInitSelected = NESTEDCOMMENT1_TAG;
  _TagInitConfirmed = false;

  Switch_._IsHtml = true;
  Switch_._IsEnd = false;
  Switch_._IsScript = false;
  Switch_._IsDoc = false;
  Switch_._IsComment = false;
  Switch_._IsNestedComment = true;
  Switch_._IsSingle = false;
  Switch_._IsCLang = false;
  Switch_._IsCppLang = false;

  Switch_._AllowEnd = false;
  Switch_._AllowNestedComment = true;
  Switch_._AllowCLang = false;

  Switch_._HasEndError = false;
  Switch_._HasNestedCommentError = false;
  Switch_._HasCLangError = false;

  Switch_._EndErrorCode = 0;
  Switch_._NestedCommentErrorCode = 0;
  Switch_._CLangErrorCode = 0;

  Switch_._RetrieveEnd = false;
  Switch_._RetrieveComment = false;
  Switch_._RetrieveNestedComment = true;
  Switch_._RetrieveScript = false;
  Switch_._RetrieveCLang = false;
  Switch_._RetrieveCppLang = false;
}

/****************************************************************************/
void TagTypeInfo::InitSingleTagSwitches(TagSwitch& Switch_)
{
  bool HtmlStyle_ = !_DocType ||
                    _DocType == TagEnums::HTML_FILE ||
                    _DocType == TagEnums::XHTML_FILE;

  _TagInitSelected = EMPTY_TAG;
  _TagInitConfirmed = false;

  Switch_._IsHtml = true;
  Switch_._IsEnd = false;
  Switch_._IsScript = false;
  Switch_._IsDoc = false;
  Switch_._IsComment = false;
  Switch_._IsNestedComment = false;
  Switch_._IsSingle = true;
  Switch_._IsCLang = false;
  Switch_._IsCppLang = false;

  Switch_._AllowEnd = false;
  Switch_._AllowNestedComment = false;
  Switch_._AllowCLang = false;

  Switch_._HasEndError = false;
  Switch_._HasNestedCommentError = false;
  Switch_._HasCLangError = false;

  Switch_._EndErrorCode = 0;
  Switch_._NestedCommentErrorCode = 0;
  Switch_._CLangErrorCode = 0;

  Switch_._RetrieveEnd = false;
  Switch_._RetrieveComment = false;
  Switch_._RetrieveNestedComment = false;
  Switch_._RetrieveScript = false;
  Switch_._RetrieveCLang = false;
  Switch_._RetrieveCppLang = false;
}

/****************************************************************************/
void TagTypeInfo::InitAllHtmlTagSwitches(TagSwitch& Switch_)
{
  _TagInitSelected = (COMMENT_TAG |
                      STD_TAG | END_TAG | EMPTY_TAG | SCRIPT_TAG);
  _TagInitConfirmed = false;

  Switch_._IsHtml = true;
  Switch_._IsEnd = false;
  Switch_._IsScript = true;
  Switch_._IsDoc = true;
  Switch_._IsComment = true;
  Switch_._IsNestedComment = true;
  Switch_._IsSingle = true;
  Switch_._IsCLang = false;
  Switch_._IsCppLang = false;

  Switch_._AllowEnd = false;
  Switch_._AllowNestedComment = true;
  Switch_._AllowCLang = false;

  Switch_._HasEndError = false;
  Switch_._HasNestedCommentError = false;
  Switch_._HasCLangError = false;

  Switch_._EndErrorCode = 0;
  Switch_._NestedCommentErrorCode = 0;
  Switch_._CLangErrorCode = 0;

  Switch_._RetrieveEnd = true;
  Switch_._RetrieveComment = true;
  Switch_._RetrieveNestedComment = true;
  Switch_._RetrieveScript = true;
  Switch_._RetrieveCLang = false;
  Switch_._RetrieveCppLang = false;
}

/****************************************************************************/
void TagTypeInfo::InitAllXmlTagSwitches(TagSwitch& Switch_)
{
  _TagInitSelected = (STD_TAG | END_TAG | EMPTY_TAG | SCRIPT_TAG);
  _TagInitConfirmed = false;

  Switch_._IsHtml = true;
  Switch_._IsEnd = false;
  Switch_._IsScript = true;
  Switch_._IsDoc = true;
  Switch_._IsComment = false;
  Switch_._IsNestedComment = false;
  Switch_._IsSingle = true;
  Switch_._IsCLang = false;
  Switch_._IsCppLang = false;

  Switch_._AllowEnd = false;
  Switch_._AllowNestedComment = false;
  Switch_._AllowCLang = false;

  Switch_._HasEndError = false;
  Switch_._HasNestedCommentError = false;
  Switch_._HasCLangError = false;

  Switch_._EndErrorCode = 0;
  Switch_._NestedCommentErrorCode = 0;
  Switch_._CLangErrorCode = 0;

  Switch_._RetrieveEnd = true;
  Switch_._RetrieveComment = false;
  Switch_._RetrieveNestedComment = false;
  Switch_._RetrieveScript = true;
  Switch_._RetrieveCLang = false;
  Switch_._RetrieveCppLang = false;
}

/****************************************************************************/
void TagTypeInfo::InitStdTagSwitches(TagSwitch& Switch_)
{
  bool HtmlStyle_ = !_DocType ||
                    _DocType == TagEnums::HTML_FILE ||
                    _DocType == TagEnums::XHTML_FILE;

  _TagInitSelected = STD_TAG;
  _TagInitConfirmed = false;

  Switch_._IsHtml = true;
  Switch_._IsEnd = false;
  Switch_._IsScript = false;
  Switch_._IsDoc = false;
  Switch_._IsComment = false;
  Switch_._IsNestedComment = false;
  Switch_._IsSingle = false;
  Switch_._IsCLang = false;
  Switch_._IsCppLang = false;

  Switch_._AllowEnd = false;
  Switch_._AllowNestedComment = false;
  Switch_._AllowCLang = false;

  Switch_._HasEndError = false;
  Switch_._HasNestedCommentError = false;
  Switch_._HasCLangError = false;

  Switch_._EndErrorCode = 0;
  Switch_._NestedCommentErrorCode = 0;
  Switch_._CLangErrorCode = 0;

  Switch_._RetrieveEnd = true;
  Switch_._RetrieveComment = false;
  Switch_._RetrieveNestedComment = false;
  Switch_._RetrieveScript = false;
  Switch_._RetrieveCLang = false;
  Switch_._RetrieveCppLang = false;
}

/****************************************************************************/
void TagTypeInfo::InitEndTagSwitches(TagSwitch& Switch_)
{
  bool HtmlStyle_ = !_DocType ||
                    _DocType == TagEnums::HTML_FILE ||
                    _DocType == TagEnums::XHTML_FILE;

  _TagInitSelected = END_TAG;
  _TagInitConfirmed = false;

  Switch_._IsHtml = true;
  Switch_._IsEnd = true;
  Switch_._IsScript = false;
  Switch_._IsDoc = false;
  Switch_._IsComment = false;
  Switch_._IsNestedComment = false;
  Switch_._IsSingle = false;
  Switch_._IsCLang = false;
  Switch_._IsCppLang = false;

  Switch_._AllowEnd = true;
  Switch_._AllowNestedComment = false;
  Switch_._AllowCLang = false;

  Switch_._HasEndError = false;
  Switch_._HasNestedCommentError = false;
  Switch_._HasCLangError = false;

  Switch_._EndErrorCode = 0;
  Switch_._NestedCommentErrorCode = 0;
  Switch_._CLangErrorCode = 0;

  Switch_._RetrieveEnd = false;
  Switch_._RetrieveComment = false;
  Switch_._RetrieveNestedComment = false;
  Switch_._RetrieveScript = false;
  Switch_._RetrieveCLang = false;
  Switch_._RetrieveCppLang = false;
}

/****************************************************************************/
void TagTypeInfo::InitScriptTagSwitches(TagSwitch& Switch_)
{
  bool HtmlStyle_ = !_DocType ||
                    _DocType == TagEnums::HTML_FILE ||
                    _DocType == TagEnums::XHTML_FILE;

  _TagInitSelected = SCRIPT_TAG;
  _TagInitConfirmed = false;

  Switch_._IsHtml = true;
  Switch_._IsEnd = false;
  Switch_._IsScript = true;
  Switch_._IsDoc = false;
  Switch_._IsComment = false;
  Switch_._IsNestedComment = false;
  Switch_._IsSingle = false;
  Switch_._IsCLang = false;
  Switch_._IsCppLang = false;

  Switch_._AllowEnd = false;
  Switch_._AllowNestedComment = false;
  Switch_._AllowCLang = false;

  Switch_._HasEndError = false;
  Switch_._HasNestedCommentError = false;
  Switch_._HasCLangError = false;

  Switch_._EndErrorCode = 0;
  Switch_._NestedCommentErrorCode = 0;
  Switch_._CLangErrorCode = 0;

  Switch_._RetrieveEnd = false;
  Switch_._RetrieveComment = false;
  Switch_._RetrieveNestedComment = false;
  Switch_._RetrieveScript = true;
  Switch_._RetrieveCLang = false;
  Switch_._RetrieveCppLang = false;
}

/****************************************************************************/
void TagTypeInfo::InitDocTagSwitches(TagSwitch& Switch_)
{
  _TagInitSelected = DOCTYPE_TAG;
  _TagInitConfirmed = false;

  Switch_._IsHtml = true;
  Switch_._IsEnd = false;
  Switch_._IsScript = false;
  Switch_._IsDoc = true;
  Switch_._IsComment = false;
  Switch_._IsNestedComment = false;
  Switch_._IsSingle = false;
  Switch_._IsCLang = false;
  Switch_._IsCppLang = false;

  Switch_._AllowEnd = false;
  Switch_._AllowNestedComment = false;
  Switch_._AllowCLang = false;

  Switch_._HasEndError = false;
  Switch_._HasNestedCommentError = false;
  Switch_._HasCLangError = false;

  Switch_._EndErrorCode = 0;
  Switch_._NestedCommentErrorCode = 0;
  Switch_._CLangErrorCode = 0;

  Switch_._RetrieveEnd = false;
  Switch_._RetrieveComment = false;
  Switch_._RetrieveNestedComment = false;
  Switch_._RetrieveScript = false;
  Switch_._RetrieveCLang = false;
  Switch_._RetrieveCppLang = false;
}

/****************************************************************************/
void TagTypeInfo::InitTagSwitches(TagSwitch& Switch_, int TagType_, int PairEnder_)
{
  if (TagType_ == TagEnums::DOCTYPE_TAG)
    InitDocTagSwitches(Switch_);
  else if (TagType_ == TagEnums::COMMENT_TAG)
    InitCommentTagSwitches(Switch_);
  else if (TagType_ == TagEnums::NESTEDCOMMENT1_TAG ||
           TagType_ == TagEnums::NESTEDCOMMENT2_TAG)
    InitNestedCommentTagSwitches(Switch_);
  else if (TagType_ == TagEnums::STD_TAG)
  {
    if (PairEnder_)
      InitEndTagSwitches(Switch_);
    else
    {
      if ((_TagSpecified && (_DataSpecified & TAGBRKINDEX_SPECIFIED)) &&
          (_TagSpecified && (_DataSpecified & TAGATTR_SPECIFIED)))
      {
        if ((IsSingletonAllowed() && IsSingletonTag()) ||
            (IsEmptyAllowed() && IsEmptyTag()))
          InitSingleTagSwitches(Switch_);
        else
          InitStdTagSwitches(Switch_);
      }
      else
        InitStdTagSwitches(Switch_);
    }
  }
  else if (TagType_ == TagEnums::END_TAG)
    InitEndTagSwitches(Switch_);
  else if (TagType_ == TagEnums::EMPTY_TAG)
    InitSingleTagSwitches(Switch_);
  else if (TagType_ == TagEnums::SCRIPT_TAG)
    InitScriptTagSwitches(Switch_);
  else if (TagType_ == TagEnums::CPPCOMMENTLINE_TAG)
    InitCppLangTagSwitches(Switch_);
  else if (TagType_ == TagEnums::CCOMMENTBLK_TAG ||
           TagType_ == TagEnums::COPENCOMMENTBLK_TAG ||
           TagType_ == TagEnums::CCLOSECOMMENTBLK_TAG)
    InitCLangTagSwitches(Switch_);
}

/****************************************************************************/
void TagTypeInfo::ConfirmTagInit()
{
  bool HtmlStyle_ = !_DocType ||
                    _DocType == TagEnums::HTML_FILE ||
                    _DocType == TagEnums::XHTML_FILE;

  if (_TagInitConfirmed && _TagCntData)
  {
    if (_TagInitSelected == CPPCOMMENTLINE_TAG)
      _TagCntData->ResetCppTags();
    else if (_TagInitSelected == CCOMMENTBLK_TAG)
      _TagCntData->ResetCTags();
    else if (_TagInitSelected == COMMENT_TAG)
      _TagCntData->ResetHtmlTags();
    else if (_TagInitSelected == NESTEDCOMMENT1_TAG)
      _TagCntData->ResetHtmlTags();
    else if (_TagInitSelected == EMPTY_TAG)
    {
      if (HtmlStyle_ || _DocType == 0)
        _TagCntData->ResetHtmlTags();
      else
        _TagCntData->ResetXmlTags();
    }
    else if (_TagInitSelected == (COMMENT_TAG |
                                  STD_TAG | END_TAG | EMPTY_TAG | SCRIPT_TAG))
      _TagCntData->ResetHtmlTags();
    else if (_TagInitSelected == (STD_TAG | END_TAG | EMPTY_TAG | SCRIPT_TAG))
    {
      if (HtmlStyle_ || _DocType == 0)
        _TagCntData->ResetHtmlTags();
      else
        _TagCntData->ResetXmlTags();
    }
    else if (_TagInitSelected == STD_TAG)
    {
      if (HtmlStyle_ || _DocType == 0)
        _TagCntData->ResetHtmlTags();
      else
      {
        if ((_TagSpecified && (_DataSpecified & TAGBRKINDEX_SPECIFIED)) &&
            (_TagSpecified && (_DataSpecified & TAGATTR_SPECIFIED)))
        {
          if ((IsSingletonAllowed() && IsSingletonTag()) ||
              (IsEmptyAllowed() && IsEmptyTag()))
            _TagCntData->ResetHtmlTags();
          else
            _TagCntData->ResetXmlTags();
        }
        else
          _TagCntData->ResetXmlTags();
      }
    }
    else if (_TagInitSelected == END_TAG)
    {
      if (HtmlStyle_ || _DocType == 0)
        _TagCntData->ResetHtmlTags();
      else
        _TagCntData->ResetXmlTags();
    }
    else if (_TagInitSelected == SCRIPT_TAG)
    {
      if (HtmlStyle_ || _DocType == 0)
        _TagCntData->ResetHtmlTags();
      else
        _TagCntData->ResetXmlTags();
    }
    else if (_TagInitSelected == DOCTYPE_TAG)
    {
      if (HtmlStyle_ || _DocType == 0)
        _TagCntData->ResetHtmlTags();
      else
        _TagCntData->ResetXmlTags();
    }

    _TagInitConfirmed = true;
  }
}

/****************************************************************************/
void TagTypeInfo::RestoreTagSwitches()
{
  if (_TagInitConfirmed)
  {
    if (_TagInitSelected == CPPCOMMENTLINE_TAG)
      InitCppLangTagSwitches(_TagSwitch);
    else if (_TagInitSelected == CCOMMENTBLK_TAG)
      InitCLangTagSwitches(_TagSwitch);
    else if (_TagInitSelected == COMMENT_TAG)
      InitCommentTagSwitches(_TagSwitch);
    else if (_TagInitSelected == NESTEDCOMMENT1_TAG)
      InitNestedCommentTagSwitches(_TagSwitch);
    else if (_TagInitSelected == EMPTY_TAG)
      InitSingleTagSwitches(_TagSwitch);
    else if (_TagInitSelected == (COMMENT_TAG |
                                  STD_TAG | END_TAG | EMPTY_TAG | SCRIPT_TAG))
      InitAllHtmlTagSwitches(_TagSwitch);
    else if (_TagInitSelected == (STD_TAG | END_TAG | EMPTY_TAG | SCRIPT_TAG))
      InitAllXmlTagSwitches(_TagSwitch);
    else if (_TagInitSelected == STD_TAG)
    {
      if ((_TagSpecified && (_DataSpecified & TAGBRKINDEX_SPECIFIED)) &&
          (_TagSpecified && (_DataSpecified & TAGATTR_SPECIFIED)))
      {
        if ((IsSingletonAllowed() && IsSingletonTag()) ||
            (IsEmptyAllowed() && IsEmptyTag()))
          InitSingleTagSwitches(_TagSwitch);
        else
          InitStdTagSwitches(_TagSwitch);
      }
      else
        InitStdTagSwitches(_TagSwitch);
    }
    else if (_TagInitSelected == END_TAG)
      InitEndTagSwitches(_TagSwitch);
    else if (_TagInitSelected == SCRIPT_TAG)
      InitScriptTagSwitches(_TagSwitch);
    else if (_TagInitSelected == DOCTYPE_TAG)
      InitDocTagSwitches(_TagSwitch);

    _TagInitConfirmed = true;
  }
}

/****************************************************************************/
void TagTypeInfo::RetrieveIntraTagContent(bool Flag_)
{
  _TagSwitch._RetrieveComment       = _TagSwitch._IsComment && Flag_;
  _TagSwitch._RetrieveNestedComment = _TagSwitch._IsNestedComment && Flag_;
  _TagSwitch._RetrieveScript        = _TagSwitch._IsScript && Flag_;
  _TagSwitch._RetrieveCLang         = _TagSwitch._IsCLang && Flag_;
  _TagSwitch._RetrieveCppLang       = _TagSwitch._IsCppLang && Flag_;
}

/****************************************************************************/
bool TagTypeInfo::GiveNextSwitches(TagSwitch& Switch_) const
{
  // returns if "Allow" transition was made from
  // true to false or false to true
  bool ret = false;

  if (Switch_._IsHtml)
  {
    if (Switch_._IsNestedComment)
    {
      if ((!_TagSpecified && _TagFound && _SearchDone) &&
          (_TagTypeFound & TagEnums::NESTEDCOMMENT1_TAG))
      {
        ret = !Switch_._AllowNestedComment;
        Switch_._IsNestedComment = true;

        if (!Switch_._HasNestedCommentError)
        {
          Switch_._AllowNestedComment = true;
          Switch_._NestedCommentErrorCode = 0;
        }

        return ret;
      }
      else if ((!_TagSpecified && _TagFound && _SearchDone) &&
               (_TagTypeFound & TagEnums::NESTEDCOMMENT2_TAG))
      {
        // returns if binary value of tag switch have been toggled
        ret = (Sequencer()->_NestedCmntTagCount > 0) ? !Switch_._AllowNestedComment:
                                                       Switch_._AllowNestedComment;
        Switch_._IsNestedComment = true;

        if (!Switch_._HasNestedCommentError)
        {
          Switch_._AllowNestedComment = Sequencer()->_NestedCmntTagCount > 0;
          Switch_._NestedCommentErrorCode = 0;
        }

        // Disallow end nested comment tag if starting nested comment tag
        // count is greater than 0.
        // Omit this for now and allow end tag anyway : == true
        // (_NestedCmntTagCount > 0);
        return ret;
      }
    }

    if (_StdTagFound && !_SingletonTagFound)
    {
      ret = !Switch_._IsEnd;
      Switch_._IsEnd = true;

      if (!Switch_._HasEndError)
      {
        Switch_._AllowEnd = true;
        Switch_._EndErrorCode = 0;
      }

      return ret;
    }
    else if ((!_StdTagFound && _SingletonTagFound) || _PairEnderFound)
    {
      // returns if binary value of tag switch have been toggled
      ret = (Sequencer()->_StdTagCount > 0) ? !Switch_._AllowEnd:
                                              Switch_._AllowEnd;
      Switch_._IsEnd = true;

      if (!Switch_._HasEndError)
      {
        Switch_._AllowEnd = Sequencer()->_StdTagCount > 0;
        Switch_._EndErrorCode = 0;
      }

      // Disallow html end tag if html starting tag count is greater than 0.
      // Omit this for now and allow end tag anyway : == true
      // (_StdTagCount > 0);
      return ret;
    }
  }
  else
  {
    if (Switch_._IsCLang)
    {
      if ((!_TagSpecified && _TagFound && _SearchDone) &&
          (_TagTypeFound & TagEnums::COPENCOMMENTBLK_TAG))
      {
        ret = !Switch_._IsCLang;
        Switch_._IsCLang = true;

        if (!Switch_._HasCLangError)
        {
          Switch_._AllowCLang = true;
          Switch_._CLangErrorCode = 0;
        }

        return ret;
      }
      else if ((!_TagSpecified && _TagFound && _SearchDone) &&
               (_TagTypeFound & TagEnums::CCLOSECOMMENTBLK_TAG))
      {
        // returns if binary value of tag switch have been toggled
        ret = (Sequencer()->_CopenCmntTagCount > 0) ? !Switch_._AllowCLang:
                                                      Switch_._AllowCLang;
        Switch_._IsCLang = true;

        if (!Switch_._HasCLangError)
        {
          Switch_._AllowCLang = Sequencer()->_CopenCmntTagCount > 0;
          Switch_._CLangErrorCode = 0;
        }

        // Disallow end comment tag if starting comment tag count is
        // greater than 0.
        // Omit this for now and allow end tag anyway : == true
        // (_CopenCmntTagCount > 0);
        return ret;
      }
    }
  }

  return ret;
}

/****************************************************************************/
int TagTypeInfo::GiveExcludedTags(bool IsHtml_, bool IsEnd_, bool IsScript_,
                                  bool IsDoc_, bool IsComment_, bool IsNestedComment_,
                                  bool IsSingle_, bool IsCLang_, bool IsCppLang_) const
{
  int Excluded_ = 0;
  int AllTags_ = (TagEnums::STD_TAG |
                  TagEnums::END_TAG |
                  TagEnums::SCRIPT_TAG |
                  TagEnums::DOCTYPE_TAG |
                  TagEnums::COMMENT_TAG |
                  TagEnums::NESTEDCOMMENT1_TAG |
                  TagEnums::NESTEDCOMMENT2_TAG |
                  TagEnums::EMPTY_TAG);

  if (IsHtml_)
  {
    Excluded_ |= (TagEnums::CCOMMENTBLK_TAG |
                  TagEnums::COPENCOMMENTBLK_TAG |
                  TagEnums::CCLOSECOMMENTBLK_TAG |
                  TagEnums::CPPCOMMENTLINE_TAG);
  }
  else
  {
    Excluded_ |= AllTags_;
  
    if (IsCLang_ && !IsCppLang_)
      Excluded_ |= TagEnums::CPPCOMMENTLINE_TAG;

    if (IsCppLang_ && !IsCLang_)
      Excluded_ |= (TagEnums::CCOMMENTBLK_TAG |
                    TagEnums::COPENCOMMENTBLK_TAG |
                    TagEnums::CCLOSECOMMENTBLK_TAG);
  }

  if (!IsEnd_)
    Excluded_ |= TagEnums::END_TAG;

  if (!IsScript_)
    Excluded_ |= TagEnums::SCRIPT_TAG;

  if (!IsDoc_)
    Excluded_ |= TagEnums::DOCTYPE_TAG;

  if (!IsComment_)
    Excluded_ |= (TagEnums::COMMENT_TAG |
                  TagEnums::NESTEDCOMMENT1_TAG |
                  TagEnums::NESTEDCOMMENT2_TAG);

  if (!IsNestedComment_)
    Excluded_ |= (TagEnums::NESTEDCOMMENT1_TAG |
                  TagEnums::NESTEDCOMMENT2_TAG);

  if (!IsSingle_)
    Excluded_ |= TagEnums::EMPTY_TAG;

  return Excluded_;
}

/****************************************************************************/
int* TagTypeInfo::SetTagSwitches(TagSwitch& Switch_, int* TagBitsArr_)
{
  SetExcludedTags(Switch_, TagBitsArr_+TAGEXCL_INDEX);
  SetAllowedTags(Switch_, TagBitsArr_+TAGALLOW_INDEX);
  SetRetrievedTags(Switch_, TagBitsArr_+TAGRETR_INDEX);

  _TagInitConfirmed = true;
  ConfirmTagInit();

  return TagBitsArr_;
}

/****************************************************************************/
int TagTypeInfo::SetAllowedTags(TagSwitch& Switch_, int* TagErr_)
{
  if (&Switch_ != &_TagSwitch)
  {
    _TagSwitch._AllowNestedComment = Switch_._AllowNestedComment;
    _TagSwitch._AllowEnd = Switch_._AllowEnd;
    _TagSwitch._AllowCLang = Switch_._AllowCLang;
  }

  int ret = 0;
  ret |= _TagSwitch._AllowNestedComment ? ALLOW_NESTEDCOMMENT_TAG:0;
  ret |= _TagSwitch._AllowEnd ? ALLOW_END_TAG:0;
  ret |= _TagSwitch._AllowCLang ? ALLOW_CCOMMENTBLK_TAG:0;

  if (TagErr_)
    *TagErr_ = ret;

  return ret;
}

/****************************************************************************/
int TagTypeInfo::SetRetrievedTags(TagSwitch& Switch_, int* TagRetr_)
{
  if (&Switch_ != &_TagSwitch)
  {
    _TagSwitch._RetrieveEnd = Switch_._RetrieveEnd;
    _TagSwitch._RetrieveComment = Switch_._RetrieveComment;
    _TagSwitch._RetrieveNestedComment = Switch_._RetrieveNestedComment;
    _TagSwitch._RetrieveScript = Switch_._RetrieveScript;
    _TagSwitch._RetrieveCLang = Switch_._RetrieveCLang;
    _TagSwitch._RetrieveCppLang = Switch_._RetrieveCppLang;
  }

  int ret = 0;
  ret |= _TagSwitch._RetrieveEnd ? RETRIEVE_END_TAG:0;
  ret |= _TagSwitch._RetrieveComment ? RETRIEVE_COMMENT_TAG:0;
  ret |= _TagSwitch._RetrieveNestedComment ? RETRIEVE_NESTEDCOMMENT_TAG:0;
  ret |= _TagSwitch._RetrieveScript ? RETRIEVE_SCRIPT_TAG:0;
  ret |= _TagSwitch._RetrieveCLang ? RETRIEVE_CCOMMENTBLK_TAG:0;
  ret |= _TagSwitch._RetrieveCppLang ? RETRIEVE_CPPCOMMENTLINE_TAG:0;

  if (TagRetr_)
    *TagRetr_ = ret;

  return ret;
}

/****************************************************************************/
int TagTypeInfo::SetExcludedTags(TagSwitch& Switch_, int* TagExcl_)
{
  if (&Switch_ != &_TagSwitch)
  {
    _TagSwitch._IsHtml = Switch_._IsHtml;
    _TagSwitch._IsEnd = Switch_._IsEnd;
    _TagSwitch._IsScript = Switch_._IsScript;
    _TagSwitch._IsDoc = Switch_._IsDoc;
    _TagSwitch._IsComment = Switch_._IsComment;
    _TagSwitch._IsNestedComment = Switch_._IsNestedComment;
    _TagSwitch._IsSingle = Switch_._IsSingle;
    _TagSwitch._IsCLang = Switch_._IsCLang;
    _TagSwitch._IsCppLang = Switch_._IsCppLang;
  }

  int ret = GiveExcludedTags
  (
    Switch_._IsHtml,
    Switch_._IsEnd,
    Switch_._IsScript,
    Switch_._IsDoc,
    Switch_._IsComment,
    Switch_._IsNestedComment,
    Switch_._IsSingle,
    Switch_._IsCLang,
    Switch_._IsCppLang
  );

  if (TagExcl_)
    *TagExcl_ = ret;

  return ret;
}

/****************************************************************************/
int TagTypeInfo::GiveAllowedTags(int AllowedTags_) const
{
  int ret = AllowedTags_;
  bool AllowEnd_           = ret ? (ret & ALLOW_END_TAG):true;
  bool AllowNestedComment_ = ret ? (ret & ALLOW_NESTEDCOMMENT_TAG):true;
  bool AllowCLang_         = ret ? (ret & ALLOW_CCOMMENTBLK_TAG):true;

  ret = 0;
  ret |= (_TagSwitch._AllowNestedComment && AllowNestedComment_) ?
             ALLOW_NESTEDCOMMENT_TAG:0;
  ret |= (_TagSwitch._AllowEnd && AllowEnd_) ?
             ALLOW_END_TAG:0;
  ret |= (_TagSwitch._AllowCLang && AllowCLang_) ?
             ALLOW_CCOMMENTBLK_TAG:0;

  return ret;
}

/****************************************************************************/
int TagTypeInfo::GiveRetrievedTags(int RetrTags_) const
{
  int ret = RetrTags_;
  bool RetrEnd_           = ret ? (ret & RETRIEVE_END_TAG):true;
  bool RetrComment_       = ret ? (ret & RETRIEVE_COMMENT_TAG):true;
  bool RetrNestedComment_ = ret ? (ret & RETRIEVE_NESTEDCOMMENT_TAG):true;
  bool RetrScript_        = ret ? (ret & RETRIEVE_SCRIPT_TAG):true;
  bool RetrCLang_         = ret ? (ret & RETRIEVE_CCOMMENTBLK_TAG):true;
  bool RetrCppLang_       = ret ? (ret & RETRIEVE_CPPCOMMENTLINE_TAG):true;

  ret = 0;
  ret |= (_TagSwitch._RetrieveEnd && RetrEnd_) ?
             RETRIEVE_END_TAG:0;
  ret |= (_TagSwitch._RetrieveComment && RetrComment_) ?
             RETRIEVE_COMMENT_TAG:0;
  ret |= (_TagSwitch._RetrieveNestedComment && RetrNestedComment_) ?
             RETRIEVE_NESTEDCOMMENT_TAG:0;
  ret |= (_TagSwitch._RetrieveScript && RetrScript_) ?
             RETRIEVE_SCRIPT_TAG:0;
  ret |= (_TagSwitch._RetrieveCLang && RetrCLang_) ?
             RETRIEVE_CCOMMENTBLK_TAG:0;
  ret |= (_TagSwitch._RetrieveCppLang && RetrCppLang_) ?
             RETRIEVE_CPPCOMMENTLINE_TAG:0;

  return ret;
}

/****************************************************************************/
int TagTypeInfo::GiveExcludedTagsForDocType(int ExcludedTags_) const
{
  int ret = 0;
  bool CStyle_ = _DocType == TagEnums::C_FILE;
  bool CppStyle_ = _DocType == TagEnums::CPP_FILE;
  bool XmlStyle_ = _DocType == TagEnums::XML_FILE;
  bool HtmlStyle_ = !_DocType ||
                    _DocType == TagEnums::HTML_FILE ||
                    _DocType == TagEnums::XHTML_FILE;

  if (CppStyle_ || CStyle_)
  {
    ret =
    GiveExcludedTags
    (
        false,
        false,
        false,
        false,
        false,
        false,
        false,
        (CStyle_ || CppStyle_),
        CppStyle_
    );
  }
  else if (XmlStyle_)
  {
    ret =
    GiveExcludedTags
    (
        true,
        ((!(_SingletonTagAllowed &&
           (_TagAttr & TagEnums::SINGLETON_TAGATTR)) ||
          (_TagAttr & TagEnums::PAIR_ENDER) ||
          !_TagSpecified) &&
         _TagSwitch._IsEnd),
        _TagSwitch._IsScript,
        _TagSwitch._IsDoc,
        false,
        false,
        (((_EmptyTagAllowed &&
           ((_TagAttr & TagEnums::EMPTY_TAGATTR) || !_TagSpecified)) ||
          (_SingletonTagAllowed &&
           ((_TagAttr & TagEnums::SINGLETON_TAGATTR) || !_TagSpecified))) &&
         _TagSwitch._IsSingle),
        false,
        false
    );
  }
  else if (HtmlStyle_)
  {
    ret =
    GiveExcludedTags
    (
        true,
        ((!(_SingletonTagAllowed &&
           (_TagAttr & TagEnums::SINGLETON_TAGATTR)) ||
          (_TagAttr & TagEnums::PAIR_ENDER) ||
          !_TagSpecified) &&
         _TagSwitch._IsEnd),
        _TagSwitch._IsScript,
        _TagSwitch._IsDoc,
        _TagSwitch._IsComment,
        _TagSwitch._IsNestedComment,
        (((_EmptyTagAllowed &&
           ((_TagAttr & TagEnums::EMPTY_TAGATTR) || !_TagSpecified)) ||
          (_SingletonTagAllowed &&
           ((_TagAttr & TagEnums::SINGLETON_TAGATTR) || !_TagSpecified))) &&
         _TagSwitch._IsSingle),
        false,
        false
    );
  }
  else
  {
    ret = GiveExcludedTags
    (
      _TagSwitch._IsHtml,
      _TagSwitch._IsEnd,
      _TagSwitch._IsScript,
      _TagSwitch._IsDoc,
      _TagSwitch._IsComment,
      _TagSwitch._IsNestedComment,
      _TagSwitch._IsSingle,
      _TagSwitch._IsCLang,
      _TagSwitch._IsCppLang
    );
  }

  return (ExcludedTags_ ? (ret & ExcludedTags_):ret);
}

/****************************************************************************/
bool TagTypeInfo::IsAllowedTag(int AllowedTags_,
                               const char* start_, const char* end_, int* TagErr_)
{
  int ret = 0;
  bool IsAllowedTag_ = false;
  bool res = false;

  IsAllowedTag_ |= (AllowedTags_ & TagEnums::ALLOW_CCOMMENTBLK_TAG) &&
                   (res=(TagBracketList::IsCCommentBlkTag(start_, end_) ||
                         TagBracketList::IsNestedCBlkTag(start_, end_, 3)));

  if (!res)
    IsAllowedTag_ |= (AllowedTags_ & TagEnums::ALLOW_END_TAG) &&
                     (res=TagBracketList::IsEndTag(start_, end_));
  else if (!ret)
    ret = TagEnums::ALLOW_CCOMMENTBLK_TAG;

  if (!res)
    IsAllowedTag_ |= (AllowedTags_ & TagEnums::ALLOW_NESTEDCOMMENT_TAG) &&
                     (res=(TagBracketList::IsCommentTag(start_, end_) ||
                           TagBracketList::IsNestedCommentTag(start_, end_, 3)));
  else if (!ret)
    ret = TagEnums::ALLOW_END_TAG;

  if (res && !ret)
    ret = TagEnums::ALLOW_NESTEDCOMMENT_TAG;

  if (TagErr_)
    *TagErr_ = ret;

  return IsAllowedTag_;
}

/****************************************************************************/
bool TagTypeInfo::IsRetrievedTag(int RetrievedTags_,
                                 const char* start_, const char* end_, int* TagRetr_)
{
  int ret = 0;
  bool IsRetrTag_ = false;
  bool res = false;

  IsRetrTag_ |= (RetrievedTags_ & TagEnums::RETRIEVE_END_TAG) &&
                (res=TagBracketList::IsEndTag(start_, end_));

  if (!res)
    IsRetrTag_ |= (RetrievedTags_ & TagEnums::RETRIEVE_CCOMMENTBLK_TAG) &&
                  (res=(TagBracketList::IsCCommentBlkTag(start_, end_) ||
                        TagBracketList::IsNestedCBlkTag(start_, end_, 3)));
  else if (!ret)
    ret = TagEnums::RETRIEVE_END_TAG;

  if (!res)
    IsRetrTag_ |= (RetrievedTags_ & TagEnums::RETRIEVE_CPPCOMMENTLINE_TAG) &&
                  (res=TagBracketList::IsCppTag(start_, end_));
  else if (!ret)
    ret = TagEnums::RETRIEVE_CCOMMENTBLK_TAG;

  if (!res)
    IsRetrTag_ |= (RetrievedTags_ & TagEnums::RETRIEVE_SCRIPT_TAG) &&
                  (res=TagBracketList::IsScriptTag(start_, end_));
  else if (!ret)
    ret = TagEnums::RETRIEVE_CPPCOMMENTLINE_TAG;

  if (!res)
    IsRetrTag_ |= (RetrievedTags_ & TagEnums::RETRIEVE_COMMENT_TAG) &&
                  (res=TagBracketList::IsCommentTag(start_, end_));
  else if (!ret)
    ret = TagEnums::RETRIEVE_SCRIPT_TAG;

  if (!res)
    IsRetrTag_ |= (RetrievedTags_ & TagEnums::RETRIEVE_NESTEDCOMMENT_TAG) &&
                  (res=(TagBracketList::IsCommentTag(start_, end_) ||
                        TagBracketList::IsNestedCommentTag(start_, end_, 3)));
  else if (!ret)
    ret = TagEnums::RETRIEVE_COMMENT_TAG;

  if (res && !ret)
    ret = TagEnums::RETRIEVE_NESTEDCOMMENT_TAG;

  if (TagRetr_)
    *TagRetr_ = ret;

  return IsRetrTag_;
}

/****************************************************************************/
bool TagTypeInfo::IsExcludedTag(int ExcludedTags_, const char* start_, const char* end_,
                                int* TagExcl_)
{
  int ret = 0;
  bool IsExclTag_ = false;
  bool res = false;

  IsExclTag_ |= (ExcludedTags_ & TagEnums::CCOMMENTBLK_TAG) &&
                (res=TagBracketList::IsCCommentBlkTag(start_, end_));

  if (!res)
    IsExclTag_ |= (ExcludedTags_ & TagEnums::COPENCOMMENTBLK_TAG) &&
                  (res=TagBracketList::IsNestedCBlkTag(start_, end_, 1));
  else if (!ret)
    ret = TagEnums::CCOMMENTBLK_TAG;

  if (!res)
    IsExclTag_ |= (ExcludedTags_ & TagEnums::CCLOSECOMMENTBLK_TAG) &&
                  (res=TagBracketList::IsNestedCBlkTag(start_, end_, 2));
  else if (!ret)
    ret = TagEnums::COPENCOMMENTBLK_TAG;

  if (!res)
    IsExclTag_ |= (ExcludedTags_ & TagEnums::CPPCOMMENTLINE_TAG) &&
                  (res=TagBracketList::IsCppTag(start_, end_));
  else if (!ret)
    ret = TagEnums::CCLOSECOMMENTBLK_TAG;

  if (!res)
    IsExclTag_ |= (ExcludedTags_ & TagEnums::END_TAG) &&
                  (res=TagBracketList::IsEndTag(start_, end_));
  else if (!ret)
    ret = TagEnums::CPPCOMMENTLINE_TAG;

  if (!res)
    IsExclTag_ |= (ExcludedTags_ & TagEnums::SCRIPT_TAG) &&
                  (res=TagBracketList::IsScriptTag(start_, end_));
  else if (!ret)
    ret = TagEnums::END_TAG;

  if (!res)
    IsExclTag_ |= (ExcludedTags_ & TagEnums::DOCTYPE_TAG) &&
                  (res=TagBracketList::IsDocTypeTag(start_, end_));
  else if (!ret)
    ret = TagEnums::SCRIPT_TAG;

  if (!res)
    IsExclTag_ |= (ExcludedTags_ & TagEnums::COMMENT_TAG) &&
                  (res=TagBracketList::IsCommentTag(start_, end_));
  else if (!ret)
    ret = TagEnums::DOCTYPE_TAG;

  if (!res)
    IsExclTag_ |= (ExcludedTags_ & TagEnums::NESTEDCOMMENT1_TAG) &&
                  (res=TagBracketList::IsNestedCommentTag(start_, end_, 1));
  else if (!ret)
    ret = TagEnums::COMMENT_TAG;

  if (!res)
    IsExclTag_ |= (ExcludedTags_ & TagEnums::NESTEDCOMMENT2_TAG) &&
                  (res=TagBracketList::IsNestedCommentTag(start_, end_, 2));
  else if (!ret)
    ret = TagEnums::NESTEDCOMMENT1_TAG;

  if (!res)
    IsExclTag_ |= (ExcludedTags_ & TagEnums::EMPTY_TAG) &&
                  TagBracketList::IsEmptyTag(start_, end_);
  else if (!ret)
    ret = TagEnums::NESTEDCOMMENT2_TAG;

  if (res && !ret)
    ret = TagEnums::EMPTY_TAG;

  if (TagExcl_)
    *TagExcl_ = ret;

  return IsExclTag_;
}

/****************************************************************************/
int TagTypeInfo::GiveAltIndex(bool Real_) const
{
  return
  (
    (Real_ && _AltIndex) ?
        ((_AltIndex == SINGLE_EMPTY)        ? EMPTY_DEX:
         (_AltIndex == EMPTY_SINGLE)        ? STD_DEX:
         (_AltIndex == COMMENT_NESTEDCMNT1) ? COMMENT_DEX:
         (_AltIndex == COMMENT_NESTEDCMNT2) ? COMMENT_DEX:
         (_AltIndex == CCMNT_COPENCMNTBLK)  ? CCOMMENTBLK_DEX:
         (_AltIndex == CCMNT_CCLOSECMNTBLK) ? CCOMMENTBLK_DEX:0):
        _AltIndex
  );
}

/****************************************************************************/
int TagTypeInfo::GiveTagPosition() const
{
  int Opening_ = 0;
  int TagPos_ = _ElementPtr ? _ElementPtr->TagPosition():-1;

  if (TagPos_ >= 0)
    return TagPos_;

  if (_TagFound && _TagTypeFound && !_TagAttr)
  {
    Opening_ = (_TagTypeFound == TagEnums::STD_TAG &&
                   (_SearchDone ||
                    !(IsSingletonTag() || IsEmptyTag()))) ||
               (_TagTypeFound == TagEnums::NESTEDCOMMENT1_TAG) ||
               (_TagTypeFound == TagEnums::COPENCOMMENTBLK_TAG);
    TagPos_ = (!_SearchDone &&
               (IsSingletonTag() || IsEmptyTag())) ? TagEnums::FORBIDDEN:
              Opening_ ? TagEnums::OPENING:
                         TagEnums::CLOSING;

    return TagPos_;
  }

  return
  (
    ((_TagAttr & TagEnums::EMPTY_TAGATTR) ||
     (_TagAttr & TagEnums::SINGLETON_TAGATTR))    ? TagEnums::FORBIDDEN:
    (_TagAttr & TagEnums::PAIR_ENDER)             ? TagEnums::CLOSING:
    ((_TagAttr & TagEnums::PAIR_TAGATTR) ||
     (_TagAttr & TagEnums::OPTIONALPAIR_TAGATTR)) ? TagEnums::OPENING:
    (_TagAttr & TagEnums::TAG_SPECIFIC)           ? TagEnums::INDETERMINATE:0
  );
}

/****************************************************************************/
int TagTypeInfo::GiveMatchingTagAttrib(int TagType_, int TagPos_) const
{
  return
  (
    (TagType_ == TagEnums::DOCTYPE_TAG)            ? TagEnums::SINGLETON_TAGATTR:
    (TagType_ == TagEnums::COMMENT_TAG)            ? TagEnums::SINGLETON_TAGATTR:
    (TagType_ == TagEnums::NESTEDCOMMENT1_TAG)     ? TagEnums::PAIR_ENDER:
    (TagType_ == TagEnums::NESTEDCOMMENT2_TAG)     ? TagEnums::PAIR_TAGATTR:
    (TagType_ == TagEnums::STD_TAG)                ? TagEnums::PAIR_ENDER:
    (TagType_ == TagEnums::END_TAG)                ? ((TagPos_ >= 0) ?
                                                          ((TagPos_ == TagEnums::OPENING)   ? TagEnums::PAIR_TAGATTR:
                                                           (TagPos_ == TagEnums::FORBIDDEN) ? TagEnums::SINGLETON_TAGATTR:
                                                           (TagPos_ == TagEnums::CLOSING)   ? TagEnums::PAIR_ENDER:
                                                                                              TagEnums::TAG_SPECIFIC):
                                                          TagEnums::TAG_SPECIFIC):
    (TagType_ == TagEnums::EMPTY_TAG)              ? (TagEnums::SINGLETON_TAGATTR |
                                                      TagEnums::EMPTY_TAGATTR):
    (TagType_ == TagEnums::SCRIPT_TAG)             ? TagEnums::SINGLETON_TAGATTR:
    (TagType_ == TagEnums::CCOMMENTBLK_TAG)        ? TagEnums::SINGLETON_TAGATTR:
    (TagType_ == TagEnums::COPENCOMMENTBLK_TAG)    ? TagEnums::PAIR_ENDER:
    (TagType_ == TagEnums::CCLOSECOMMENTBLK_TAG)   ? TagEnums::PAIR_TAGATTR:
    (TagType_ == TagEnums::CPPCOMMENTLINE_TAG)     ? TagEnums::SINGLETON_TAGATTR:0
  );
}

/****************************************************************************/
int TagTypeInfo::GiveDefaultTagAttrib(int TagType_, int TagPos_) const
{
  return
  (
    (TagType_ == TagEnums::DOCTYPE_TAG)            ? TagEnums::SINGLETON_TAGATTR:
    (TagType_ == TagEnums::COMMENT_TAG)            ? TagEnums::SINGLETON_TAGATTR:
    (TagType_ == TagEnums::NESTEDCOMMENT1_TAG)     ? TagEnums::PAIR_TAGATTR:
    (TagType_ == TagEnums::NESTEDCOMMENT2_TAG)     ? TagEnums::PAIR_ENDER:
    (TagType_ == TagEnums::STD_TAG)                ? ((TagPos_ >= 0) ?
                                                          ((TagPos_ == TagEnums::OPENING)   ? TagEnums::PAIR_TAGATTR:
                                                           (TagPos_ == TagEnums::FORBIDDEN) ? TagEnums::SINGLETON_TAGATTR:
                                                           (TagPos_ == TagEnums::CLOSING)   ? TagEnums::PAIR_ENDER:
                                                                                              TagEnums::TAG_SPECIFIC):
                                                          TagEnums::TAG_SPECIFIC):
    (TagType_ == TagEnums::END_TAG)                ? TagEnums::PAIR_ENDER:
    (TagType_ == TagEnums::EMPTY_TAG)              ? (TagEnums::SINGLETON_TAGATTR |
                                                      TagEnums::EMPTY_TAGATTR):
    (TagType_ == TagEnums::SCRIPT_TAG)             ? TagEnums::SINGLETON_TAGATTR:
    (TagType_ == TagEnums::CCOMMENTBLK_TAG)        ? TagEnums::SINGLETON_TAGATTR:
    (TagType_ == TagEnums::COPENCOMMENTBLK_TAG)    ? TagEnums::PAIR_TAGATTR:
    (TagType_ == TagEnums::CCLOSECOMMENTBLK_TAG)   ? TagEnums::PAIR_ENDER:
    (TagType_ == TagEnums::CPPCOMMENTLINE_TAG)     ? TagEnums::SINGLETON_TAGATTR:0
  );
}

/****************************************************************************/
int TagTypeInfo::TagIndexToTagType() const
{
  bool TagSet_ = _TagBrkIndex >= 0;

  if (TagSet_)
  {
    int TagType_ = _AltIndex ?
                       (long)::pow(2, GiveAltIndex(true)/2):
                       (long)::pow(2, _TagBrkIndex/2);
    return TagType_;
  }

  return 0;
}

/****************************************************************************/
TagElement* TagTypeInfo::GiveTagElementPtr()
{
  return _ElementPtr;
}

/****************************************************************************/
TagElement* TagTypeInfo::GiveMatchingElementPtr(int TagPos_)
{
  if (_ElementPtr)
  {
    if (_ElementPtr->TagPosition() == TagEnums::FORBIDDEN ||
        TagPos_ == TagEnums::FORBIDDEN)
      return _ElementPtr;
    else if (_ElementPtr->TagPosition() == TagEnums::INDETERMINATE ||
             TagPos_ == TagEnums::INDETERMINATE ||
             TagPos_ == TagEnums::NO_TAGPOS)
      return _ElementPtr->GiveMatchingTag();
    else if (_ElementPtr->TagPosition() == TagEnums::CLOSING &&
             _ElementPtr->GiveMatchingTag())
      return (TagPos_ == TagEnums::OPENING) ?
                 _ElementPtr->GiveMatchingTag():
                 _ElementPtr;
    else if (_ElementPtr->TagPosition() == TagEnums::OPENING &&
             _ElementPtr->GiveMatchingTag())
      return (TagPos_ == TagEnums::CLOSING) ?
                 _ElementPtr->GiveMatchingTag():
                 _ElementPtr;

    return NULL;
  }

  return NULL;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::CreateStackInfoNode(int DocType_, TagTypeInfo* TagPtr_, bool UnlinkTemp_,
                                              bool GenSearchDone_, bool SpecSearchDone_)
{
  // TO DO: Before coding for tag stack logic,
  //        should build linked list to be filled with relevant
  //        tag stack helper information.
  //

  if (!_TagFound)
    return *this;

  if (!TagPtr_)
    TagPtr_ = this;

  if (!_Sequencer)
    CreateSequencer();

  if (!_HtmlSpecialCase)
    CreateHtmlSpecialCase();

  TagElement* TagElmPtr_ = TagPtr_->GiveTagElementPtr();

  Sequencer()->IncSearchRunNumber();
  StackNodeInfo* StkPtr_ = new StackNodeInfo;

  if (StkPtr_)
  {
    StkPtr_->SetFromParameters(DocType_, GenSearchDone_, SpecSearchDone_);
    StkPtr_->CopyFromTagTypeInfo(TagPtr_);

    if (TagElmPtr_)
      StkPtr_->CopyFromTagElement(TagPtr_->GiveMatchingElementPtr(TagEnums::OPENING));

    if (_SearchResult)
      StkPtr_->CopyFromTagSearchResult(_SearchResult);

    Sequencer()->PushStackInfo(StkPtr_, false, TagPtr_ == this);

    if (TagElmPtr_)
      StkPtr_->SetTagElementState();

    if (GenSearchDone_)
    {
      Sequencer()->_InfoPtr = Sequencer()->_HeadInfo;
      ShiftTagContents();
    }

    if (TagPtr_ != this && UnlinkTemp_)
    {
      SetStackInfoPtr(false);
      Sequencer()->PopStackInfo(true);

      NextStackInfo(true, false);
      SetStackInfoPtr(true);

      StkPtr_ = Sequencer()->PopStackInfo(false, true);

      NextStackInfo(true, false);
      SetStackInfoPtr(false);

      if (StkPtr_)
      {
        StkPtr_->Unlink();
        Sequencer()->_InfoPtr = Sequencer()->_HeadInfo;
      }
    }
  }

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::AdjustTagCount(int DictTagPos_)
{
  if (_TagFound && _TagTypeFound)
  {
    TagElement* TagElmPtr_ = GiveTagElementPtr();
    int Opening_ = (_TagTypeFound == TagEnums::STD_TAG &&
                       (_SearchDone ||
                        !(IsSingletonTag() || IsEmptyTag()))) ||
                   (_TagTypeFound == TagEnums::NESTEDCOMMENT1_TAG) ||
                   (_TagTypeFound == TagEnums::COPENCOMMENTBLK_TAG);
    int TagPos_ = TagElmPtr_ ? TagElmPtr_->TagPosition():
                  (!_SearchDone &&
                   (IsSingletonTag() || IsEmptyTag())) ? TagEnums::FORBIDDEN:
                  Opening_ ? TagEnums::OPENING:
                             TagEnums::CLOSING;

    bool StdTag_ = (_TagTypeFound == TagEnums::STD_TAG &&
                    TagPos_ != TagEnums::FORBIDDEN &&
                       (_SearchDone ||
                        !(IsSingletonTag() || IsEmptyTag())));

    if (StdTag_ && DictTagPos_ == TagEnums::FORBIDDEN)
    {
      Sequencer()->DecStdTagCount();

      _SingletonTagFound = true;
      _HasPairContent = false;
      _StdTagFound = false;

      _EmptyTagFound = _TagTypeFound == TagEnums::EMPTY_TAG;
      _ContainerTagFound = false;
      _PairEnderFound = false;

      PopBrkNesting(TagPos_, 1);

      if (Sequencer()->_StdTagCount < 0)
      {
        Sequencer()->AppendNestingError(TagEnums::ENDTAG_FORBIDDEN);

        if (Sequencer()->_NestingError & TagEnums::STDTAGENDPAIR_MISMATCH)
          Sequencer()->_NestingError &= ~TagEnums::STDTAGENDPAIR_MISMATCH;
      }

      if (!_TagSwitch._AllowEnd)
      {
        _TagSwitch._HasEndError = true;
        _TagSwitch._EndErrorCode |= ENDTAG_FORBIDDEN;

        if (_TagSwitch._EndErrorCode & TagEnums::STDTAGENDPAIR_MISMATCH)
          _TagSwitch._EndErrorCode &= ~TagEnums::STDTAGENDPAIR_MISMATCH;
      }
    }
  }
}

/****************************************************************************/
void TagTypeInfo::TransferTagFoundDataTo(TagTypeInfo* TagPtr_)
{
  if (TagPtr_)
  {
    TagPtr_->_TagAttr = _TagAttr;
    TagPtr_->_StdTagFound = _StdTagFound;

    if (TagPtr_->_StdTagFound)
    {
      TagPtr_->_EmptyTagFound = false;
      TagPtr_->_PairEnderFound = false;
    }
    else
    {
      TagPtr_->_EmptyTagFound = TagPtr_->_TagTypeFound == TagEnums::EMPTY_TAG;
      TagPtr_->_PairEnderFound = TagPtr_->_TagTypeFound == TagEnums::END_TAG;
    }

    if (TagPtr_->_StdTagFound)
      TagPtr_->_SingletonTagFound = false;
    else if (TagPtr_->_EmptyTagFound || TagPtr_->_PairEnderFound)
    {
      TagPtr_->_HasPairContent = false;
      TagPtr_->_ContainerTagFound = false;

      if (TagPtr_->_PairEnderFound)
        TagPtr_->_SingletonTagFound = false;
      else if (TagPtr_->_EmptyTagFound)
        TagPtr_->_SingletonTagFound = true;
    }
  }
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetTagTypeFound(long Index_, bool SetTagCnt_)
{
  bool Err_ = false;
  _TagFound = Index_ >= 0;

  if (_TagFound)
  {
    _TagTypeFound = (long)::pow(2, Index_/2);

    if (!_TagSpecified && _SearchDone)
      _TagBrkIndex = Index_;

    bool StdTag_ = false;
    bool NoContentTag_ = _TagTypeFound == TagEnums::CPPCOMMENTLINE_TAG ||
                         _TagTypeFound == TagEnums::DOCTYPE_TAG ||
                         _TagTypeFound == TagEnums::COMMENT_TAG ||
                         _TagTypeFound == TagEnums::SCRIPT_TAG ||
                         _TagTypeFound == TagEnums::EMPTY_TAG;

    TagElement* TagElmPtr_ = GiveTagElementPtr();
    int Opening_ = (_TagTypeFound == TagEnums::STD_TAG &&
                       (_SearchDone ||
                        !(IsSingletonTag() || IsEmptyTag()))) ||
                   (_TagTypeFound == TagEnums::NESTEDCOMMENT1_TAG) ||
                   (_TagTypeFound == TagEnums::COPENCOMMENTBLK_TAG);
    int TagPos_ = TagElmPtr_ ? TagElmPtr_->TagPosition():
                  ((!_SearchDone && (IsSingletonTag() || IsEmptyTag())) ||
                   NoContentTag_ || IsSingletonTag()) ? TagEnums::FORBIDDEN:
                  Opening_ ? TagEnums::OPENING:
                             TagEnums::CLOSING;

    if (!_TagSpecified && _SearchDone)
    {
      _TagAttr = GiveDefaultTagAttrib(_TagTypeFound, TagPos_);
      _StdTagFound = _TagTypeFound == TagEnums::STD_TAG &&
                     TagPos_ != TagEnums::FORBIDDEN &&
                         (_SearchDone ||
                          !(IsSingletonTag() || IsEmptyTag()));

      if (_StdTagFound)
      {
        _EmptyTagFound = false;
        _PairEnderFound = false;
      }
      else
      {
        _EmptyTagFound = _TagTypeFound == TagEnums::EMPTY_TAG;
        _PairEnderFound = _TagTypeFound == TagEnums::END_TAG;
      }

      if (_StdTagFound)
      {
        _HasPairContent = _SearchResult->HasTagContentString();
        _SingletonTagFound = false;
      }
      else if (_EmptyTagFound || _PairEnderFound)
      {
        _HasPairContent = false;
        _ContainerTagFound = false;

        if (_PairEnderFound)
          _SingletonTagFound = false;
        else if (_EmptyTagFound)
          _SingletonTagFound = true;
      }
    }

    if (!SetTagCnt_)
      return *this;

    // STDTAGOPEN_STR                  "<"    // if (prev_tagattr == optionalpair)
    //                                        //   PopBrkNesting(StdTagOpen, StdTagOpen, 1);
    //                                        // PushBrkNesting(StdTagOpen, StdTagOpen, 1);
    // STDTAGCLOSE_STR                 ">"    // if (tagattr == singleton)
    //                                        //   PopBrkNesting(StdTagOpen, StdTagClose, 1);
    //                                        // else
    //                                        //   PopBrkNesting(StdTagOpen, StdTagClose, 0);
    // ENDTAGOPEN_STR                  "</"   // if (tagattr == pair || tagattr == optionalpair)
    //                                        //   PopBrkNesting(StdTagOpen, EndTagOpen, 1);
    //                                        // else
    //                                        //   PushBrkNesting(EndTagOpen, EndTagOpen, 1);
    // ENDTAGCLOSE_STR                 ">"    // no action
    //
    // EMPTYOPEN_STR                   "<"    // if (prev_tagattr == optionalpair)
    //                                        //   PopBrkNesting(StdTagOpen, EmptyTagOpen, 1);
    //                                        // PushBrkNesting(EmptyTagOpen, EmptyTagOpen, 1);
    // EMPTYCLOSE_STR                  "/>"   // if (tagattr == empty)
    //                                        //   PopBrkNesting(EmptyTagOpen, EmptyTagClose, 1);
    //                                        // else
    //                                        //   PopBrkNesting(EmptyTagOpen, EmptyTagClose, 0);

    if (_StdTagFound)
    {
      Sequencer()->IncStdTagCount();
      PushBrkNesting(TagPos_, 1);
    }
    else if (_EmptyTagFound)
    {
      if (TagPos_ != TagEnums::CLOSING)
        PushBrkNesting(TagPos_, 1);
      else
        PopBrkNesting(TagPos_, 1);
    }
    else if (_PairEnderFound)
    {
      Sequencer()->DecStdTagCount();
      
      if (Sequencer()->_StdTagCount < 0)
      {
        Sequencer()->AppendNestingError(_SingletonTagFound ?
                                            TagEnums::ENDTAG_FORBIDDEN:
                                            TagEnums::STDTAGENDPAIR_MISMATCH);
        Err_ = true;
      }

      if (!_TagSwitch._AllowEnd)
      {
        _TagSwitch._HasEndError = true;
        _TagSwitch._EndErrorCode |= _SingletonTagFound ?
                                       TagEnums::ENDTAG_FORBIDDEN:
                                       TagEnums::STDTAGENDPAIR_MISMATCH;
        Err_ = true;
      }

      if (!Err_)
        PopBrkNesting(TagPos_, 1);
    }

    // NESTEDCOMMENT1OPEN_STR          "<!--" // PushBrkNesting(CommentOpen, NestedComment1Open, 1);
    // NESTEDCOMMENT1CLOSE_STR         ">"    // PushBrkNesting(NestedComment1Open, NestedComment1Close, 1);
    // NESTEDCOMMENT2OPEN_STR          "<!"   // if (nesting(CommentOpen) && nesting(Nested1CommentOpen))
    //                                        //   PopBrkNesting(CommentOpen, NestedComment2Open, 1);
    // NESTEDCOMMENT2CLOSE_STR         "-->"  // if (nesting(NestedComment1Open))
    //                                        //   PopBrkNesting(NestedComment1Open, NestedComment2Close, 1);
    //                                        // else
    //                                             PopBrkNesting(CommentOpen, CommentClose, 1);

    if ((!_TagSpecified && _SearchDone) &&
        (_TagTypeFound == TagEnums::NESTEDCOMMENT1_TAG))
    {
      Sequencer()->IncNestedCmntTagCount();
      PushBrkNesting(TagPos_, 1);
    }
    else if ((!_TagSpecified && _SearchDone) &&
             (_TagTypeFound == TagEnums::NESTEDCOMMENT2_TAG))
    {
      Sequencer()->DecNestedCmntTagCount();

      if (Sequencer()->_NestedCmntTagCount < 0)
      {
        Sequencer()->AppendNestingError(TagEnums::NESTEDCMNTTAG_MISMATCH);
        Err_ = true;
      }

      if (!_TagSwitch._AllowNestedComment)
      {
        _TagSwitch._HasNestedCommentError = true;
        _TagSwitch._NestedCommentErrorCode |= TagEnums::NESTEDCMNTTAG_MISMATCH;
        Err_ = true;
      }

      if (!Err_)
        PopBrkNesting(TagPos_, 1);
    }

    // COPENCOMMENTBLKOPEN_STR         "/*"   // PushBrkNesting(COpenBlkOpen, COpenBlkOpen, 1);
    // COPENCOMMENTBLKCLOSE_STR        "->"   // PushBrkNesting(COpenBlkClose, COpenBlkClose, 1);
    // CCLOSECOMMENTBLKOPEN_STR        "<-"   // if (nesting(COpenBlkOpen) && nesting(COpenBlkClose))
    //                                        //   PopBrkNesting(COpenBlkOpen, CCloseBlkOpen, 1);
    // CCLOSECOMMENTBLKCLOSE_STR       "*/"   // if (nesting(COpenBlkClose))
    //                                        //   PopBrkNesting(COpenBlkClose, CCloseBlkClose, 1);

    if ((!_TagSpecified && _SearchDone) &&
        (_TagTypeFound == TagEnums::COPENCOMMENTBLK_TAG))
    {
      Sequencer()->IncCopenCmntTagCount();
      PushBrkNesting(TagPos_, 1);
    }
    else if ((!_TagSpecified && _SearchDone) &&
             (_TagTypeFound == TagEnums::CCLOSECOMMENTBLK_TAG))
    {
      Sequencer()->DecCopenCmntTagCount();

      if (Sequencer()->_CopenCmntTagCount < 0)
      {
        Sequencer()->AppendNestingError(TagEnums::CCLOSECMNTTAG_MISMATCH);
        Err_ = true;
      }

      if (!_TagSwitch._AllowCLang)
      {
        _TagSwitch._HasCLangError = true;
        _TagSwitch._CLangErrorCode |= TagEnums::CCLOSECMNTTAG_MISMATCH;
        Err_ = true;
      }

      if (!Err_)
        PopBrkNesting(TagPos_, 1);
    }

    // DOCTYPE_TAG
    // DOCTYPEOPEN_STR                 "<!"   // PushBrkNesting(DocTypeOpen, DocTypeOpen, 1);
    // DOCTYPECLOSE_STR                ">"    // PopBrkNesting(DocTypeOpen, DocTypeClose, 1);

    if ((!_TagSpecified && _SearchDone) &&
        (_TagTypeFound == TagEnums::DOCTYPE_TAG))
    {
      if (TagPos_ != TagEnums::CLOSING)
        PushBrkNesting(TagPos_, 1);
      else
        PopBrkNesting(TagPos_, 1);
    }

    // COMMENT_TAG
    // COMMENTOPEN_STR                 "<!--" // PushBrkNesting(CommentOpen, CommentOpen, 1);
    // COMMENTCLOSE_STR                "-->"  // PopBrkNesting(CommentOpen, CommentClose, 1);

    if ((!_TagSpecified && _SearchDone) &&
        (_TagTypeFound == TagEnums::COMMENT_TAG))
    {
      if (TagPos_ != TagEnums::CLOSING)
        PushBrkNesting(TagPos_, 1);
      else
        PopBrkNesting(TagPos_, 1);
    }

    // SCRIPT_TAG
    // SCRIPTOPEN_STR                  "<?"   // PushBrkNesting(ScriptTagOpen, ScriptTagOpen, 1);
    // SCRIPTCLOSE_STR                 "?>"   // PopBrkNesting(ScriptTagOpen, ScriptTagClose, 1);

    if ((!_TagSpecified && _SearchDone) &&
        (_TagTypeFound == TagEnums::SCRIPT_TAG))
    {
      if (TagPos_ != TagEnums::CLOSING)
        PushBrkNesting(TagPos_, 1);
      else
        PopBrkNesting(TagPos_, 1);
    }

    // CPPCOMMENTLINE_TAG
    // CPPCOMMENTLINEOPEN_STR          "//"   // PushBrkNesting(CppCommentOpen, CppCommentOpen, 1);
    // CPPCOMMENTLINECLOSE_STR         "=>"   // PopBrkNesting(CppCommentOpen, CppCommentClose, 1);

    if ((!_TagSpecified && _SearchDone) &&
        (_TagTypeFound == TagEnums::CPPCOMMENTLINE_TAG))
    {
      if (TagPos_ != TagEnums::CLOSING)
        PushBrkNesting(TagPos_, 1);
      else
        PopBrkNesting(TagPos_, 1);
    }
  }
  else
  {
    _TagTypeFound = 0;
    _SingletonTagFound = false;
    _EmptyTagFound = false;
    _ContainerTagFound = false;
    _StdTagFound = false;
    _PairEnderFound = false;
  }

  return *this;
}

/****************************************************************************/
// Method for setting tag errors found in the Tag Switches object _TagSwitch
//
// Relevant tag switches within the tag switch object are:
//
//      bool _AllowEnd;
//      bool _AllowNestedComment;
//      bool _AllowCLang;
//
//      bool _HasEndError;
//      bool _HasNestedCommentError;
//      bool _HasCLangError;
//
//      int _EndErrorCode;
//      int _NestedCommentErrorCode;
//      int _CLangErrorCode;
//
void TagTypeInfo::SetTagErrors(int ErrorTag_, long Index_)
{
  if (ErrorTag_ == ALLOW_NESTEDCOMMENT_TAG &&
        (Index_ == NESTEDCOMMENT1_DEX ||
         Index_ == NESTEDCOMMENT2_DEX ||
         Index_ == COMMENT_DEX))
  {
    Sequencer()->AppendNestingError(ENDTAG_FORBIDDEN);

    if (!_TagSwitch._AllowEnd)
    {
      _TagSwitch._HasEndError = true;
      _TagSwitch._EndErrorCode |= TagEnums::ENDTAG_FORBIDDEN;
    }
  }
  else if (ErrorTag_ == ALLOW_END_TAG && Index_ == END_DEX)
  {
    Sequencer()->AppendNestingError(NESTEDCMNTTAG_FORBIDDEN);

    if (!_TagSwitch._AllowNestedComment)
    {
      _TagSwitch._HasNestedCommentError = true;
      _TagSwitch._NestedCommentErrorCode |= TagEnums::NESTEDCMNTTAG_MISMATCH;
    }
  }
  else if (ErrorTag_ == ALLOW_CCOMMENTBLK_TAG && Index_ == COMMENT_DEX)
  {
    Sequencer()->AppendNestingError(CCLOSECMNTTAG_FORBIDDEN);

    if (!_TagSwitch._AllowCLang)
    {
      _TagSwitch._HasCLangError = true;
      _TagSwitch._CLangErrorCode |= TagEnums::CCLOSECMNTTAG_MISMATCH;
    }
  }
  else if (!Index_)
  {
    if (ErrorTag_ & TagEnums::HTML_TAG_MISPLACED)
      Sequencer()->AppendNestingError(TagEnums::HTML_TAG_MISPLACED);

    if (ErrorTag_ & TagEnums::HEAD_TAG_MISPLACED)
      Sequencer()->AppendNestingError(TagEnums::HEAD_TAG_MISPLACED);

    if (ErrorTag_ & TagEnums::BODY_TAG_MISPLACED)
      Sequencer()->AppendNestingError(TagEnums::BODY_TAG_MISPLACED);

    if (ErrorTag_ & TagEnums::REQUIRED_TAG_NOT_FOUND)
      Sequencer()->AppendNestingError(TagEnums::REQUIRED_TAG_NOT_FOUND);

    if (ErrorTag_ & TagEnums::POPAPPCONTSTR_CHKNODE)
      Sequencer()->AppendNestingError(TagEnums::POPAPPCONTSTR_CHKNODE);
  }
}

/****************************************************************************/
bool TagTypeInfo::FindIndexPositions(long* IndexPos_, long* TailIndexPos_, long x, long& FirstIndex_,
                                     long& ClosestIndex_, long& ClosestLen_, long& ClosestTailIndex_, long& ClosestTailLen_,
                                     long& MaxIndex_, long& MaxLen_, long& MaxTailIndex_, long& MaxTailLen_)
{
  // Starting brks found, so save index position and brk length
  // 1st position record string index, 2nd position record brk length
  bool TailFound_ = false;
  bool CCloseTag_ = TagBracketList::IsCCommentBlkTag(_TagBrkMatrix[x], _TagBrkMatrix[x+1]);

  IndexPos_[x] = _SearchResult->SearchResultIndex();
  IndexPos_[x+1] = _SearchResult->SearchResultLength();

  _SearchResult->SetSearchType(0);
  
  if (CCloseTag_)
    _SearchResult->find(ChrString(CCLOSECOMMENTBLKCLOSE_STR));
  else
    _SearchResult->find(ChrString(STDTAG_CLOSE_CH));
          
  if (_SearchResult->TargetFound())
  {
    TailIndexPos_[x] = _SearchResult->SearchResultIndex();
    TailIndexPos_[x+1] = 1;

    if (!CCloseTag_)
    {
      _SearchResult->SetSearchType(0);
      _SearchResult->find(ChrString(_TagBrkMatrix[x+1]));

      if (_SearchResult->TargetFound() &&
          IndexPos_[x] <  _SearchResult->SearchResultIndex() &&
              (_SearchResult->SearchResultIndex() <= TailIndexPos_[x] ||
               x == TagEnums::COMMENT_DEX))
      {
        TailIndexPos_[x] = _SearchResult->SearchResultIndex();
        TailIndexPos_[x+1] = _SearchResult->SearchResultLength();
        TailFound_ = true;
      }
      else
      {
        IndexPos_[x] = 0;
        IndexPos_[x+1] = 0;
        TailIndexPos_[x] = 0;
        TailIndexPos_[x+1] = 0;
        TailFound_ = false;

        return false;
      }
    }
    else
    {
      TailIndexPos_[x+1] = 2;
      TailFound_ = true;
    }
  }

  if (TailFound_)
  {
    if (FirstIndex_ == -1)
      FirstIndex_ = x;

    if (ClosestIndex_ == -1 ||
         (IndexPos_[x] <= ClosestIndex_ &&
             (TailIndexPos_[x] <= ClosestTailIndex_ ||
              x == TagEnums::COMMENT_DEX)))
    {
      ClosestLen_ = IndexPos_[x+1];
      ClosestIndex_ = IndexPos_[x];
      ClosestTailLen_ = TailIndexPos_[x+1];
      ClosestTailIndex_ = TailIndexPos_[x];
    }
        
    if (MaxLen_ == -1 ||
         (IndexPos_[x+1] >= MaxLen_ && TailIndexPos_[x+1] >= MaxTailLen_))
    {
      // save maximum brk length found and index position
      MaxLen_ = IndexPos_[x+1];
      MaxIndex_ = IndexPos_[x];
      MaxTailLen_ = TailIndexPos_[x+1];
      MaxTailIndex_ = TailIndexPos_[x];
    }
  }

  return true;
}

/****************************************************************************/
bool TagTypeInfo::IsUnmatchedBrkPair(long ClosestIndex_, long ClosestLen_,
                                     long ClosestTailIndex_, long ClosestTailLen_,
                                     long MaxIndex_, long MaxLen_, long MaxTailLen_,
                                     long* IndexPos_, long* TailIndexPos_,
                                     long x, bool CppStyle_)
{
  // selected brk entries should match either closest found brk pair or
  // longest found tag pair, otherwise return to top of loop

  bool UnMatchedClosestBrk_ =
    (ClosestIndex_ < MaxIndex_ &&
       ((IndexPos_[x+1] < ClosestLen_ || TailIndexPos_[x+1] < ClosestTailLen_) ||
        (IndexPos_[x] > ClosestIndex_ || TailIndexPos_[x] > ClosestTailIndex_)));

  bool UnMatchedLongestBrk_ =
    (ClosestIndex_ == MaxIndex_ &&
       ((IndexPos_[x+1] < MaxLen_ || TailIndexPos_[x+1] < MaxTailLen_) ||
        (CppStyle_ &&
        (IndexPos_[x] > ClosestIndex_ || TailIndexPos_[x] > ClosestTailIndex_))));

  bool ZeroLengthBrk_ = !IndexPos_[x+1];

  return (UnMatchedClosestBrk_ || UnMatchedLongestBrk_ || ZeroLengthBrk_);
}

/****************************************************************************/
long TagTypeInfo::IndexFromBrk(long* sindex_, long* eindex_, bool NestedTag_) const
{
  StartBrk(NestedTag_, sindex_);
  EndBrk(NestedTag_, eindex_);
  return (sindex_ ? *sindex_:
          eindex_ ? *eindex_:0);
}

/****************************************************************************/
const char* TagTypeInfo::StartBrk(bool NestedTag_, long* index_) const
{
  long tbrindex_ = 0;
  TagBrkExtend2(false, &tbrindex_);

  if (index_)
    ((_TagSpecified && (_DataSpecified & TAGBRKINDEX_SPECIFIED)) ?
     ((NestedTag_ && 0 < _TagBrkExtend2 && _TagBrkExtend2 < CPPCOMMENTLINEOPEN_DEX) ?
          tbrindex_:(_AltIndex ? (*index_ = GiveAltIndex(true)):
                                 (*index_ = _TagBrkIndex))):0L);

  return ((_TagSpecified && (_DataSpecified & TAGBRKINDEX_SPECIFIED)) ?
          ((NestedTag_ && 0 < _TagBrkExtend2 && _TagBrkExtend2 < CPPCOMMENTLINEOPEN_DEX) ?
               TagBrkExtend2(false):
               (_AltIndex ? _TagBrkMatrix[GiveAltIndex(true)]:
                            _TagBrkMatrix[_TagBrkIndex])):NULL);
}

/****************************************************************************/
const char* TagTypeInfo::EndBrk(bool NestedTag_, long* index_) const
{
  long tbrindex_ = 0;
  TagBrkExtend1(false, &tbrindex_);

  if (index_)
    ((_TagSpecified && (_DataSpecified & TAGBRKINDEX_SPECIFIED)) ?
     ((NestedTag_ && 0 < _TagBrkExtend1 && _TagBrkExtend1 < CPPCOMMENTLINEOPEN_DEX) ?
          tbrindex_:(_AltIndex ? (*index_ = GiveAltIndex(true)+1):
                                 (*index_ = _TagBrkIndex+1))):0L);

  return ((_TagSpecified && (_DataSpecified & TAGBRKINDEX_SPECIFIED)) ?
          ((NestedTag_ && 0 < _TagBrkExtend1 && _TagBrkExtend1 < CPPCOMMENTLINEOPEN_DEX) ?
               TagBrkExtend1(false):
               (_AltIndex ? _TagBrkMatrix[GiveAltIndex(true)+1]:
                            _TagBrkMatrix[_TagBrkIndex+1])):NULL);
}

/****************************************************************************/
long TagTypeInfo::IndexFromBrkFound(long* sindex_, long* eindex_, bool NestedTag_) const
{
  StartBrkFound(NestedTag_, sindex_);
  EndBrkFound(NestedTag_, eindex_);
  return (sindex_ ? *sindex_:
          eindex_ ? *eindex_:0);
}

/****************************************************************************/
const char* TagTypeInfo::StartBrkFound(bool NestedTag_, long* index_) const
{
  long tbrindex_ = 0;
  TagBrkExtend2(false, &tbrindex_);

  if (index_)
    (_TagFound ?
      ((NestedTag_ && 0 < _TagBrkExtend2 && _TagBrkExtend2 < CPPCOMMENTLINEOPEN_DEX) ?
         tbrindex_:(_AltIndex ? (*index_ = GiveAltIndex(true)):
                                (*index_ = _TagBrkIndex))):IndexFromBrk(index_, NULL, false));

  return (_TagFound ?
           ((NestedTag_ && 0 < _TagBrkExtend2 && _TagBrkExtend2 < CPPCOMMENTLINEOPEN_DEX) ?
              TagBrkExtend2(false):
              (_AltIndex ? _TagBrkMatrix[GiveAltIndex(true)]:
                           _TagBrkMatrix[_TagBrkIndex])):StartBrk());
}

/****************************************************************************/
const char* TagTypeInfo::EndBrkFound(bool NestedTag_, long* index_) const
{
  long tbrindex_ = 0;
  TagBrkExtend1(false, &tbrindex_);

  if (index_)
    (_EndBrkFound ?
      ((NestedTag_ && 0 < _TagBrkExtend1 && _TagBrkExtend1 < CPPCOMMENTLINEOPEN_DEX) ?
         tbrindex_:(_AltIndex ? (*index_ = GiveAltIndex(true)+1):
                                (*index_ = _TagBrkIndex+1))):IndexFromBrk(NULL, index_, false));

  return (_EndBrkFound ?
           ((NestedTag_ && 0 < _TagBrkExtend1 && _TagBrkExtend1 < CPPCOMMENTLINEOPEN_DEX) ?
              TagBrkExtend1(false):
              (_AltIndex ? _TagBrkMatrix[GiveAltIndex(true)+1]:
                           _TagBrkMatrix[_TagBrkIndex+1])):EndBrk());
}

/****************************************************************************/
const char* TagTypeInfo::TagBrkExtend1(bool StartingBrk_, long* index_) const
{
  if (index_)
    ((0 < _TagBrkExtend1 && _TagBrkExtend1 < CPPCOMMENTLINEOPEN_DEX &&
        (_TagFound ||
         (_TagSpecified && (_DataSpecified & TAGBRKINDEX_SPECIFIED)))) ?
             (StartingBrk_ ? (*index_ = _TagBrkExtend1):
                             (*index_ = _TagBrkExtend1+1)):0);

  return ((0 < _TagBrkExtend1 && _TagBrkExtend1 < CPPCOMMENTLINEOPEN_DEX &&
             (_TagFound ||
              (_TagSpecified && (_DataSpecified & TAGBRKINDEX_SPECIFIED)))) ?
                  (StartingBrk_ ? _TagBrkMatrix[_TagBrkExtend1]:_TagBrkMatrix[_TagBrkExtend1+1]):NULL);
}

/****************************************************************************/
const char* TagTypeInfo::TagBrkExtend2(bool EndingBrk_, long* index_) const
{
  if (index_)
    ((0 < _TagBrkExtend2 && _TagBrkExtend2 < CPPCOMMENTLINEOPEN_DEX &&
        (_EndBrkFound ||
         (_TagSpecified && (_DataSpecified & TAGBRKINDEX_SPECIFIED)))) ?
             (EndingBrk_ ? (*index_ = _TagBrkExtend2+1):
                           (*index_ = _TagBrkExtend2)):0);

  return ((0 < _TagBrkExtend2 && _TagBrkExtend2 < CPPCOMMENTLINEOPEN_DEX &&
             (_EndBrkFound ||
              (_TagSpecified && (_DataSpecified & TAGBRKINDEX_SPECIFIED)))) ?
                  (EndingBrk_ ? _TagBrkMatrix[_TagBrkExtend2+1]:_TagBrkMatrix[_TagBrkExtend2]):NULL);
}

/****************************************************************************/
bool TagTypeInfo::SearchForTag(int* TagSwitchBits_, long* Index_)
{
  long x, y;
  long slen;
  long max;
  
  bool endfnd = false;
  bool spcfnd = false;
  bool ContRetr_ = false;
  bool TagFound_ = false;
  bool EndTermSet_ = false;
  bool IsExclTag_ = false;
  bool IsAllowTag_ = false;
  bool IsRetrTag_ = false;
  bool TargetFound_ = false;
  bool NestedCmnt_ = false;
  bool NoElmNameType_ = false;
  bool CppStyle_ = _TagSwitch._IsCLang ||
                   _TagSwitch._IsCppLang;
  
  long startpt;
  long startlen;
  long endpt;
  long endlen;
  long spcpt;
  long leadspc = 0;
  long elmlen = -1;
  long elmstartpt = -1;
  long elmendpt;
  
  long MaxLen_ = -1;
  long MaxIndex_ = -1;
  long MaxTailLen_ = -1;
  long MaxTailIndex_ = -1;
  
  long ClosestLen_ = -1;
  long ClosestIndex_ = -1;
  long ClosestTailLen_ = -1;
  long ClosestTailIndex_ = -1;
  long FirstIndex_ = -1;

  char* spcset = MakeCharSet(isspace);
  const char* rdptr = _SearchResult->c_str();
  ChrString* EndTermStr_ = NULL;
  ChrString StartBrkStr_;
  ChrString EndBrkStr_;

  int ExclBits_ = 0;
  int AllowedBits_ = 0;
  int RetrBits_ = 0;
  int DisallowTag_ = 0;
  int UnRetrTag_ = 0;

  Reset(SEARCH_TARGETS);
  _SearchDone = false;

  if (Index_)
    *Index_ = -1;

  if ((_TagSpecified && _DataSpecified != ALL_SPECIFIED) ||
      (!_TagSpecified && !(_DataSpecified & DOCTYPE_SPECIFIED)))
    return false;

  if (!_Sequencer)
    CreateSequencer();

  if (!_HtmlSpecialCase)
    CreateHtmlSpecialCase();

  ExclBits_ = GiveExcludedTagsForDocType(0);
  AllowedBits_ = GiveAllowedTags(0);
  RetrBits_ = GiveRetrievedTags(0);

  TagSwitchBits_[TAGEXCL_INDEX] = ExclBits_;
  TagSwitchBits_[TAGALLOW_INDEX] = AllowedBits_;
  TagSwitchBits_[TAGRETR_INDEX] = RetrBits_;

  if (_TagSpecified)
  {
    _SearchResult->SetSearchType(MATCH_TAGTYPE);
  
    if (SearchForElementStr())
    {
      elmstartpt = _SearchResult->SearchResultIndex();
      elmlen = _SearchResult->SearchResultLength();
    }

    if (_SearchResult->TargetFound() && elmstartpt > 0)
    {
      _SearchResult->SetSearchType(MATCH_TAGBRACKET);
      x = _TagBrkIndex;
      _SearchResult->findprevious(elmstartpt, ChrString(_TagBrkMatrix[x]));
    }

    if (_SearchResult->TargetFound())
    {
      startpt = _SearchResult->SearchResultIndex();
      startlen = _SearchResult->SearchResultLength();
      
      _SearchResult->SetSearchType(MATCH_TAGTYPE);
      
      if (SearchForElementStr())
      {
        if (_SearchResult->SearchResultLength() == elmlen &&
            _SearchResult->SearchResultIndex() == elmstartpt)
        {
          // Does found tag have retrievable content?
          IsRetrTag_ = IsRetrievedTag(TagSwitchBits_[TAGRETR_INDEX],
                                      _TagBrkMatrix[x], _TagBrkMatrix[x+1], &UnRetrTag_);

          elmendpt = elmstartpt + elmlen;
          _SearchResult->SetSearchType(MATCH_TAGENDBRACKET);
          x = _TagBrkIndex;
          NoElmNameType_ = IsNoElementNameTagType(x);

          _SearchResult->findnext(_SearchResult->SearchResultIndex(), ChrString(_TagBrkMatrix[x+1]));

          if (_SearchResult->TargetFound())
          {
            // reaching this point means a fully formed tag has been successfully
            // extracted from the read text buffer: <, > and tag element string
            // replace search results with valid start tag's index and length
            endpt = _SearchResult->SearchResultIndex();
            endlen = _SearchResult->SearchResultLength();
            _EndBrkFound = endfnd = true;

            _SearchResult->SetStartingBrkData(startpt, startlen);
            _SearchResult->SetMatchingBrkData(endpt, endlen);

            if (NoElmNameType_)
              _SearchResult->SetElementData(0, 0, false);
            else
              _SearchResult->SetElementData(elmstartpt, elmlen, true);

            EndTermSet_ = IsEndTagTermType(_TagBrkIndex);

            if (EndTermSet_)
              EndTermStr_ = GiveEndTagTermStr(_TagBrkIndex);

            // Search for tag delimiters of key-value pairs list
            // if present within tag
            _SearchResult->SetTagTerminatorPtr(NULL, 0);

            if (EndTermStr_)
              _SearchResult->AddTagTerminatorPtr(EndTermStr_);

            _SearchResult->SetSearchType(MATCH_TAGDELIMS);
            _SearchResult->FindDelims(elmendpt, endpt, _TagBrkIndex);

            if (Index_)
              *Index_ = _TagBrkIndex;

            TagFound_ = _TagFound = _TagBrkIndex >= 0;
            _TagBrkExtend1 = _TagBrkExtend2 = 0;

            delete EndTermStr_;
            _SearchResult->SetTagTerminatorPtr(NULL, 0);

            if (IsRetrTag_)
            {
              ContRetr_ = _SearchResult->RetrieveContents(false);

              if (IsIntraContentTagType(_TagBrkIndex))
              {
                _SearchResult->FindIntraTagBrks(_TagBrkIndex, NestedCmnt_,
                                                NULL, NULL, &StartBrkStr_, &EndBrkStr_);

                if (IsNestedTagType(_TagBrkIndex, NestedCmnt_))
                {
                  _SearchResult->RetrieveIntraContents(true, &StartBrkStr_, &EndBrkStr_);

                  if (NestedCmnt_)
                  {
                    if (_TagBrkIndex == TagEnums::COMMENT_DEX)
                    {
                      _TagBrkExtend1 = TagEnums::NESTEDCOMMENT1_DEX;
                      _TagBrkExtend2 = TagEnums::NESTEDCOMMENT2_DEX;
                    }

                    if (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX)
                    {
                      _TagBrkExtend1 = TagEnums::COPENCOMMENTBLK_DEX;
                      _TagBrkExtend2 = TagEnums::CCLOSECOMMENTBLK_DEX;
                    }
                  }
                }
                else
                  _SearchResult->RetrieveIntraContents(true);
              }
            }
            else
              ContRetr_ = _SearchResult->RetrieveContents(false);

            _TagCntData->ResetTagDataUpdate();
            SetTagTypeFound(_TagBrkIndex, false);
            return true;
          }
        }
      }
    }
  }
  else if (rdptr && ::strlen(rdptr))
  {
    // format of IndexPos array is: IndexPos[0,2,4... even n] for pos index,
    //                              IndexPos[1,3,5... odd n] for str length
    long IndexPos_[TagEnums::TAGLIST_MAX_ENTRIES];
    long TailIndexPos_[TagEnums::TAGLIST_MAX_ENTRIES];
    
    ::memset(IndexPos_, 0, sizeof(long) * TagEnums::TAGLIST_MAX_ENTRIES);
    ::memset(TailIndexPos_, 0, sizeof(long) * TagEnums::TAGLIST_MAX_ENTRIES);
    Reset(TAG_CONTENT_TYPE);
    
    for (x = 0; x < TagEnums::TAGLIST_MAX_ENTRIES; x+=2)
    {
      IsExclTag_ = IsExcludedTag(TagSwitchBits_[TAGEXCL_INDEX],
                                 _TagBrkMatrix[x], _TagBrkMatrix[x+1]);

      // Skip specified excluded tags and nested comments which are not parsed
      if (IsExclTag_ ||
          TagBracketList::IsNestedCommentTag(_TagBrkMatrix[x], _TagBrkMatrix[x+1], 3) ||
          TagBracketList::IsNestedCBlkTag(_TagBrkMatrix[x], _TagBrkMatrix[x+1], 3))
        continue;
      else
      {
        _SearchDone = true;
        _SearchResult->SetSearchType(MATCH_TAGBRACKET);
        _SearchResult->find(ChrString(_TagBrkMatrix[x]));
      }

      if (_SearchResult->TargetFound())
      {
        TargetFound_ = true;
        FindIndexPositions(IndexPos_, TailIndexPos_, x, FirstIndex_,
                           ClosestIndex_, ClosestLen_, ClosestTailIndex_, ClosestTailLen_,
                           MaxIndex_, MaxLen_, MaxTailIndex_, MaxTailLen_);
      }
    }

    if (_SearchDone && TargetFound_)
      x = (FirstIndex_ >= 0) ? FirstIndex_:0;
    else
      x = TagEnums::TAGLIST_MAX_ENTRIES;

    for (;x < TagEnums::TAGLIST_MAX_ENTRIES; x+=2)
    {
      // selected brk entries should match either closest found brk pair or
      // longest found tag pair, otherwise return to top of loop
      if (IsUnmatchedBrkPair(ClosestIndex_, ClosestLen_,
                             ClosestTailIndex_, ClosestTailLen_,
                             MaxIndex_, MaxLen_, MaxTailLen_,
                             IndexPos_, TailIndexPos_, x, CppStyle_))
        continue;

      // Is found tag allowed at given point in tag sequence?
      IsAllowTag_ = IsAllowedTag(TagSwitchBits_[TAGALLOW_INDEX],
                                 _TagBrkMatrix[x], _TagBrkMatrix[x+1], &DisallowTag_);

      // Does found tag have retrievable content?
      IsRetrTag_ = IsRetrievedTag(TagSwitchBits_[TAGRETR_INDEX],
                                  _TagBrkMatrix[x], _TagBrkMatrix[x+1], &UnRetrTag_);

      if (!IsAllowTag_)
        SetTagErrors(DisallowTag_, x);

      // get brk starting point and string length to end of start brk
      // assign string pointer to end of start brk
      startpt = IndexPos_[x];
      startlen = IndexPos_[x+1];
      slen = startpt + startlen;
      const char* ptr = _SearchResult->c_str() + slen;
      NoElmNameType_ = IsNoElementNameTagType(x);

      if (ptr)
      {
        // skip leading spaces
        for (leadspc = 0; isspace(ptr[leadspc]); leadspc++);

        // find space char position after tag element name
        spcfnd = endfnd = false;
        spcpt = StrHasChar(ptr+leadspc, spcset);
          
        if (spcpt >= 0)
        {
          spcpt += slen + leadspc;
          _TagSpcFound = spcfnd = true;
        }
      
        _SearchResult->SetSearchType(MATCH_TAGENDBRACKET);
        _SearchResult->findnext(slen, ChrString(_TagBrkMatrix[x+1]));

        if (_SearchResult->TargetFound())
        {
          endpt = _SearchResult->SearchResultIndex();
          endlen = _SearchResult->SearchResultLength();
          _EndBrkFound = endfnd = true;

          // detected spcpt must lie within tag start and end brk to be valid
          if (spcfnd && spcpt > endpt)
          {
            spcpt = -1;
            spcfnd = false;
          }
        }
        else
        {
          // end brk is critical so if not found then spcpt is irrelevant
          spcpt = -1;
          spcfnd = false;
        }

        if (endfnd)
        {
          if (!TagSequenceCheck(ptr+leadspc, IndexPos_, x, slen+leadspc, endpt))
            continue;

          // tag element string is characters extracted from end of
          // start brk to detected spc pt or detected end brk if no spaces
          // within tag. Tag not explicitly specified so assumed extracted
          // from read text buffer. Element length must be greater than 0
          // and start before space delimiter of tag is encountered.
          elmstartpt = slen + leadspc;
          
          if (spcfnd)
            elmlen = spcpt - elmstartpt;
          else
            elmlen = endpt - elmstartpt;

          if (elmlen > 0 || NoElmNameType_)
          {
            if (!NoElmNameType_)
            {
              if (!_ElementStr)
                _ElementStr = new ChrString(ptr, elmlen);
              else
                *_ElementStr = ChrString(ptr, elmlen);
            }

            // reaching this point means a fully formed tag has been successfully
            // extracted from the read text buffer: <, > and tag element string
            // replace search results with valid start tag's index and length
            _TagBrkIndex = x;
            _ElementStr->SetCaseSensitive(FALSE);

            _SearchResult->SetStartingBrkData(startpt, startlen);
            _SearchResult->SetMatchingBrkData(endpt, endlen);

            if (NoElmNameType_)
              _SearchResult->SetElementData(0, 0, false);
            else
              _SearchResult->SetElementData(elmstartpt, elmlen, true);

            EndTermSet_ = IsEndTagTermType(_TagBrkIndex);

            if (EndTermSet_)
              EndTermStr_ = GiveEndTagTermStr(_TagBrkIndex);

            // Search for tag delimiters of key-value pairs list
            // if present within tag
            elmendpt = elmstartpt + elmlen;
            _SearchResult->SetTagTerminatorPtr(NULL, 0);

            if (EndTermStr_)
              _SearchResult->AddTagTerminatorPtr(EndTermStr_);

            _SearchResult->SetSearchType(MATCH_TAGDELIMS);
            _SearchResult->FindDelims(elmendpt, endpt, _TagBrkIndex);

            if (Index_)
              *Index_ = _TagBrkIndex;

            TagFound_ = _TagFound = _TagBrkIndex >= 0;
            _TagBrkExtend1 = _TagBrkExtend2 = 0;

            SetContentTypeTag(StartBrkFound(), EndBrkFound());
            delete EndTermStr_;
            _SearchResult->SetTagTerminatorPtr(NULL, 0);

            if (IsRetrTag_)
            {
              ContRetr_ = _SearchResult->RetrieveContents(false);

              if (IsIntraContentTagType(_TagBrkIndex))
              {
                _SearchResult->FindIntraTagBrks(_TagBrkIndex, NestedCmnt_,
                                                NULL, NULL, &StartBrkStr_, &EndBrkStr_);

                if (IsNestedTagType(_TagBrkIndex, NestedCmnt_))
                {
                  _SearchResult->RetrieveIntraContents(true, &StartBrkStr_, &EndBrkStr_);

                  if (NestedCmnt_)
                  {
                    if (_TagBrkIndex == TagEnums::COMMENT_DEX)
                    {
                      _TagBrkExtend1 = TagEnums::NESTEDCOMMENT1_DEX;
                      _TagBrkExtend2 = TagEnums::NESTEDCOMMENT2_DEX;
                    }

                    if (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX)
                    {
                      _TagBrkExtend1 = TagEnums::COPENCOMMENTBLK_DEX;
                      _TagBrkExtend2 = TagEnums::CCLOSECOMMENTBLK_DEX;
                    }
                  }
                }
                else
                  _SearchResult->RetrieveIntraContents(true);
              }
            }
            else
              ContRetr_ = _SearchResult->RetrieveContents(false);

            _TagCntData->ResetTagDataUpdate();
            SetTagTypeFound(_TagBrkIndex, true);
            break;
          }
          else
          {
            // absent elmend tags are invalid so zero out index entries
            // and restart search for matching start and end brks
            IndexPos_[x] = 0;
            IndexPos_[x+1] = 0;
          }
        }
        else
        {
          // matching end brk not found so zero out index entries and
          // restart search for matching start and end brks
          IndexPos_[x] = 0;
          IndexPos_[x+1] = 0;
        }        
      }
    }
  }

  if (!TagFound_)
  {
    SetTagTypeFound(-1, false);

    if (_SearchResult && !_TagSpecified)
      ContRetr_ = _SearchResult->RetrieveContents(false);
  }

  if ((_DumpTagContents && GiveOutput() && (TagFound_ || _SearchDone)) ||
      (!_TagSpecified && (!rdptr || ::strlen(rdptr) == 0)))
  {
    if (!TagFound_)
      ContRetr_ = _SearchResult->RetrieveContents(false);

    if (ContRetr_)
      DumpTagContentData(*GiveOutput(), GiveFieldWidth());
    else if (ShowDebugDump())
      *GiveOutput() <<endl;
  }

  return TagFound_;
}

/****************************************************************************/
bool TagTypeInfo::SearchForElementStr(const ChrString* Str_)
{
  if (Str_)
  {
    _SearchResult->findnext(_SearchResult->SearchResultIndex(), *Str_);
    return _SearchResult->TargetFound();
  }
  else if (_TagSpecified && _ElementStr)
  {
    _SearchResult->findnext(_SearchResult->SearchResultIndex(), *_ElementStr);
    return _SearchResult->TargetFound();
  }

  return false;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetTagBrk(const char* Start_, const char* End_, long* Index_)
{
  if (!Start_ || !End_)
    return *this;

  long x;
  bool NestedCmnt_ = false;

  size_t slen_s, tlen_s;
  size_t slen_e, tlen_e;
  
  for (x = 0; x < _TagBrkMax; x+=2)
    if (_TagBrkMatrix[x] && _TagBrkMatrix[x+1])
    {
      slen_s = strlen(_TagBrkMatrix[x]);
      tlen_s = strlen(Start_);

      slen_e = strlen(_TagBrkMatrix[x+1]);
      tlen_e = strlen(End_);

      if (slen_s == tlen_s && slen_e == tlen_e)
        if (strcmp(Start_, _TagBrkMatrix[x]) == 0 &&
            strcmp(End_, _TagBrkMatrix[x+1]) == 0)
        {
          _TagBrkIndex = x;
          _DataSpecified |= TAGBRKINDEX_SPECIFIED;
          
          if (Index_)
            *Index_ = x;

          if (_TagSpecified)
          {
            NestedCmnt_ = IsNestedTagType(_TagBrkIndex, false);

            if (NestedCmnt_)
            {
              if (_TagBrkIndex == TagEnums::COMMENT_DEX)
              {
                _TagBrkExtend1 = TagEnums::NESTEDCOMMENT1_DEX;
                _TagBrkExtend2 = TagEnums::NESTEDCOMMENT2_DEX;
              }

              if (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX)
              {
                _TagBrkExtend1 = TagEnums::COPENCOMMENTBLK_DEX;
                _TagBrkExtend2 = TagEnums::CCLOSECOMMENTBLK_DEX;
              }
            }
          }

          break;
        }
    }

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetElementStr(bool Specified_, ChrString* Ptr_, bool KeepPtr_)
{
  if (Specified_ && Ptr_)
  {
    bool NestedCmnt_ = false;

    _TagSpecified = true;
    _DataSpecified |= ELEMENT_SPECIFIED;

    if (KeepPtr_)
    {
      if (_ElementStr)
        delete _ElementStr;
        
      _ElementStr = Ptr_;
      _ElementStr->SetCaseSensitive(FALSE);
    }
    else
    {
      if (_ElementStr)
        *_ElementStr = *Ptr_;
      else
        _ElementStr = new ChrString(*Ptr_);

      _ElementStr->SetCaseSensitive(FALSE);
    }

    if (_DataSpecified & TAGBRKINDEX_SPECIFIED)
    {
      NestedCmnt_ = IsNestedTagType(_TagBrkIndex, false);

      if (NestedCmnt_)
      {
        if (_TagBrkIndex == TagEnums::COMMENT_DEX)
        {
          _TagBrkExtend1 = TagEnums::NESTEDCOMMENT1_DEX;
          _TagBrkExtend2 = TagEnums::NESTEDCOMMENT2_DEX;
        }

        if (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX)
        {
          _TagBrkExtend1 = TagEnums::COPENCOMMENTBLK_DEX;
          _TagBrkExtend2 = TagEnums::CCLOSECOMMENTBLK_DEX;
        }
      }
    }
  }
  else if (!Specified_ && !Ptr_)
    ResetTagList();

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetSingletonAllowed(bool Set_)
{
  _SingletonTagAllowed = Set_;
  _TagCntData->SetSingletonAllowed(_ElementPtr, Set_);

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetEmptyAllowed(bool Set_)
{
  _EmptyTagAllowed = Set_;
  _TagCntData->SetEmptyAllowed(_ElementPtr, Set_);

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetDocType(int DocType_)
{
  if (DocType_ > 0)
  {
    _DocType = DocType_;
    _DataSpecified |= DOCTYPE_SPECIFIED;
    _TagCntData->SetDocType(DocType_);
  }

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetEofFound(bool Eof_)
{
  _EofFound = Eof_;
  _TagCntData->SetEofFound(Eof_);
  _SearchResult->SetEofFound(Eof_);

  return *this;
}

/****************************************************************************/
bool TagTypeInfo::IsOptionalOrNoContentTag(int TagAttr_, int TagTypeFound_,
                                           bool HasPairContent_, bool* PairPtr_)
{
  bool Optional_;
  bool Single_;
  bool Empty_;
  bool Container_;
  bool Ender_;
  bool TagSpec_;
  bool Pair_;

  Pair_ = TagElement::IsPairTagAttr(TagAttr_, &Optional_, &Single_,
                                    &Empty_, &Container_, &Ender_, &TagSpec_);

  bool Sgl_ = !HasPairContent_ && (!Pair_ && !Container_ && Single_);
  bool Emp_ = !HasPairContent_ && (!Pair_ && !Container_ && Empty_);

  if (PairPtr_)
    *PairPtr_ = Pair_;

  return
  (
    Optional_ ||
    (TagTypeFound_ == TagEnums::STD_TAG && Sgl_)  ||
    (TagTypeFound_ == TagEnums::EMPTY_TAG && Emp_)
  );
}

/****************************************************************************/
bool TagTypeInfo::IsOptionalTag(int TagAttr_, bool* PairPtr_)
{
  bool Optional_;
  bool Single_;
  bool Empty_;
  bool Container_;
  bool Ender_;
  bool TagSpec_;
  bool Pair_;

  Pair_ = TagElement::IsPairTagAttr(TagAttr_, &Optional_, &Single_,
                                    &Empty_, &Container_, &Ender_, &TagSpec_);

  if (PairPtr_)
    *PairPtr_ = Pair_;

  return Optional_;
}

/****************************************************************************/
bool TagTypeInfo::IsOptionalPairTag() const
{
  bool Pair_ = false;
  bool Opt_ = IsOptionalTag(_TagAttr, &Pair_);
  return (Pair_ && Opt_);
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetTagAttr(int TagAttr_)
{
  int Mask_ = ~TagEnums::EMPTY_TAGATTR & ~SINGLETON_TAGATTR;

  if (_EmptyTagAllowed)
    Mask_ |= TagEnums::EMPTY_TAGATTR;

  if (_SingletonTagAllowed)
    Mask_ |= TagEnums::SINGLETON_TAGATTR;

  _TagAttr = TagAttr_ & Mask_;
  _DataSpecified |= TAGATTR_SPECIFIED;

  return *this;
}

/****************************************************************************/
int TagTypeInfo::GiveTagNestingCondition(bool Initial_) const
{
  int TagNest_ = 0;
  bool PrevSearchDone_ = SearchRunNumber() && _ElementPtr &&
                         Sequencer()->_PrevTagInfo &&
                         Sequencer()->PrevTagInfoSet() &&
                         _SearchDone && _ContentTypeTag;

  if (Initial_ && SearchRunNumber() &&
      Sequencer()->_PrevTagInfo && _SearchDone && _ContentTypeTag)
  {
    bool Optional_ =
      IsOptionalTag(_ElementPtr ? _ElementPtr->TagAttrib():_TagAttr);

    TagNest_ =
    (_TagTypeFound == EMPTY_TAG ||
      (_TagTypeFound == STD_TAG && _SingletonTagFound)) ? SGL_NESTCOND:
    (_TagTypeFound == STD_TAG && _StdTagFound && !Optional_) ? ST_NESTCOND:
    (_TagTypeFound == STD_TAG && _StdTagFound && Optional_) ? OPTST_NESTCOND:
    NO_NESTCOND;

    return TagNest_;
  }
  else if (PrevSearchDone_ &&
           Sequencer()->_PrevTagInfo->_TagFound && _TagFound)
  {
    bool CurOptional_ =
      IsOptionalTag(_ElementPtr ? _ElementPtr->TagAttrib():_TagAttr);
    bool PrevOptional_ =
      IsOptionalTag(Sequencer()->_PrevTagInfo->_TagAttr);
    bool PrevSingleEnd_ = Sequencer()->_PrevTagInfo->_TagNestCond == SGLMEND_NESTCOND;
    bool PrevPairEnd_ = Sequencer()->_PrevTagInfo->_TagNestCond == SGLUMEND_NESTCOND ||
                        Sequencer()->_PrevTagInfo->_TagNestCond == STMEND_NESTCOND ||
                        Sequencer()->_PrevTagInfo->_TagNestCond == STUMEND_NESTCOND ||
                        Sequencer()->_PrevTagInfo->_TagNestCond == OPTSTMEND_NESTCOND ||
                        Sequencer()->_PrevTagInfo->_TagNestCond == OPTSTUMEND_NESTCOND ||
                        Sequencer()->_PrevTagInfo->_TagNestCond == SGLENDUMEND_NESTCOND ||
                        Sequencer()->_PrevTagInfo->_TagNestCond == PAIRENDUMEND_NESTCOND ||
                        Sequencer()->_PrevTagInfo->_TagNestCond == OPTENDUMEND_NESTCOND;

    TagNest_ =
    ((Sequencer()->_PrevTagInfo->_TagTypeFound == EMPTY_TAG ||
       (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && Sequencer()->_PrevTagInfo->_SingletonTagFound)) &&
     (_TagTypeFound == EMPTY_TAG ||
       (_TagTypeFound == STD_TAG && _SingletonTagFound))) ? SGLSGL_NESTCOND:
    ((Sequencer()->_PrevTagInfo->_TagTypeFound == EMPTY_TAG ||
       (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && Sequencer()->_PrevTagInfo->_SingletonTagFound)) &&
     _TagTypeFound == STD_TAG && _StdTagFound && !CurOptional_) ? SGLST_NESTCOND:
    ((Sequencer()->_PrevTagInfo->_TagTypeFound == EMPTY_TAG ||
       (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && Sequencer()->_PrevTagInfo->_SingletonTagFound)) &&
     _TagTypeFound == STD_TAG && _StdTagFound && CurOptional_) ? SGLOPTST_NESTCOND:
    ((Sequencer()->_PrevTagInfo->_TagTypeFound == EMPTY_TAG ||
       (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && Sequencer()->_PrevTagInfo->_SingletonTagFound)) &&
     _TagTypeFound == END_TAG && _PairEnderFound &&
     Sequencer()->_PrevTagInfo->_MatchingTags) ? SGLMEND_NESTCOND:
    ((Sequencer()->_PrevTagInfo->_TagTypeFound == EMPTY_TAG ||
       (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && Sequencer()->_PrevTagInfo->_SingletonTagFound)) &&
     _TagTypeFound == END_TAG && _PairEnderFound &&
     !Sequencer()->_PrevTagInfo->_MatchingTags) ? SGLUMEND_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG && _TagTypeFound == END_TAG &&
     Sequencer()->_PrevTagInfo->_PairEnderFound && _PairEnderFound &&
     PrevSingleEnd_) ? SGLENDUMEND_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG && Sequencer()->_PrevTagInfo->_PairEnderFound && 
      (_TagTypeFound == EMPTY_TAG || (_TagTypeFound == STD_TAG && _SingletonTagFound)) &&
     PrevSingleEnd_) ? SGLENDSGL_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG && _TagTypeFound == STD_TAG &&
     Sequencer()->_PrevTagInfo->_PairEnderFound && _StdTagFound && !CurOptional_ &&
     PrevSingleEnd_) ? SGLENDST_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG && _TagTypeFound == STD_TAG &&
     Sequencer()->_PrevTagInfo->_PairEnderFound && _StdTagFound && CurOptional_ &&
     PrevSingleEnd_) ? SGLENDOPTST_NESTCOND:

    (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG &&
     !PrevOptional_ && Sequencer()->_PrevTagInfo->_StdTagFound &&
      (_TagTypeFound == EMPTY_TAG ||
        (_TagTypeFound == STD_TAG && _SingletonTagFound))) ? STSGL_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && _TagTypeFound == STD_TAG &&
     !PrevOptional_ && Sequencer()->_PrevTagInfo->_StdTagFound && _StdTagFound && !CurOptional_) ? STST_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && _TagTypeFound == STD_TAG &&
     !PrevOptional_ && Sequencer()->_PrevTagInfo->_StdTagFound && _StdTagFound && CurOptional_) ? STOPTST_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && _TagTypeFound == END_TAG &&
     Sequencer()->_PrevTagInfo->_StdTagFound && _PairEnderFound &&
     !PrevOptional_ && Sequencer()->_PrevTagInfo->_MatchingTags) ? STMEND_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && _TagTypeFound == END_TAG &&
     Sequencer()->_PrevTagInfo->_StdTagFound && _PairEnderFound &&
     !PrevOptional_ && !Sequencer()->_PrevTagInfo->_MatchingTags) ? STUMEND_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG && _TagTypeFound == END_TAG &&
     Sequencer()->_PrevTagInfo->_PairEnderFound && _PairEnderFound &&
     !PrevOptional_ && PrevPairEnd_) ? PAIRENDUMEND_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG &&
     !PrevOptional_ && Sequencer()->_PrevTagInfo->_PairEnderFound && PrevPairEnd_ &&
      (_TagTypeFound == EMPTY_TAG ||
        (_TagTypeFound == STD_TAG && _SingletonTagFound))) ? PAIRENDSGL_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG && _TagTypeFound == STD_TAG &&
     Sequencer()->_PrevTagInfo->_PairEnderFound && _StdTagFound && !CurOptional_ &&
     !PrevOptional_ && PrevPairEnd_) ? PAIRENDST_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG && _TagTypeFound == STD_TAG &&
     Sequencer()->_PrevTagInfo->_PairEnderFound && _StdTagFound && CurOptional_ &&
     !PrevOptional_ && PrevPairEnd_) ? PAIRENDOPTST_NESTCOND:

    (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG &&
     PrevOptional_ && Sequencer()->_PrevTagInfo->_StdTagFound &&
      (_TagTypeFound == EMPTY_TAG ||
        (_TagTypeFound == STD_TAG && _SingletonTagFound))) ? OPTSTSGL_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && _TagTypeFound == STD_TAG &&
     PrevOptional_ && Sequencer()->_PrevTagInfo->_StdTagFound && _StdTagFound && !CurOptional_) ? OPTSTST_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && _TagTypeFound == STD_TAG &&
     PrevOptional_ && Sequencer()->_PrevTagInfo->_StdTagFound && _StdTagFound && CurOptional_) ? OPTSTOPTST_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && _TagTypeFound == END_TAG &&
     Sequencer()->_PrevTagInfo->_StdTagFound && _PairEnderFound &&
     PrevOptional_ && Sequencer()->_PrevTagInfo->_MatchingTags) ? OPTSTMEND_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == STD_TAG && _TagTypeFound == END_TAG &&
     Sequencer()->_PrevTagInfo->_StdTagFound && _PairEnderFound &&
     PrevOptional_ && !Sequencer()->_PrevTagInfo->_MatchingTags) ? OPTSTUMEND_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG && _TagTypeFound == END_TAG &&
     Sequencer()->_PrevTagInfo->_PairEnderFound && _PairEnderFound &&
     PrevOptional_ && PrevPairEnd_) ? OPTENDUMEND_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG &&
     PrevOptional_ && Sequencer()->_PrevTagInfo->_PairEnderFound && PrevPairEnd_ &&
      (_TagTypeFound == EMPTY_TAG ||
        (_TagTypeFound == STD_TAG && _SingletonTagFound))) ? OPTENDSGL_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG && _TagTypeFound == STD_TAG &&
     Sequencer()->_PrevTagInfo->_PairEnderFound && _StdTagFound && !CurOptional_ &&
     PrevOptional_ && PrevPairEnd_) ? OPTENDST_NESTCOND:
    (Sequencer()->_PrevTagInfo->_TagTypeFound == END_TAG && _TagTypeFound == STD_TAG &&
     Sequencer()->_PrevTagInfo->_PairEnderFound && _StdTagFound && CurOptional_ &&
     PrevOptional_ && PrevPairEnd_) ? OPTENDOPTST_NESTCOND:

    NO_NESTCOND;

    return TagNest_;
  }

  return NO_NESTCOND;
}

/****************************************************************************/
void TagTypeInfo::CompareWithPrevTagInfo(StackNodeInfo* EncNode_)
{
  Sequencer()->CompareWithPrevTagInfo(EncNode_, _PairEnderFound);
}

/****************************************************************************/
void TagTypeInfo::SetPrevTagInfo(StackNodeInfo* InfoPtr_)
{
  if (!_ElementStr || _ElementStr->strlen() == 0)
    return;

  if (!Sequencer()->_PrevTagInfo)
    Sequencer()->ResetPrevTagInfo();

  Sequencer()->_PrevTagInfo->_TagAttr             = _ElementPtr ? _ElementPtr->TagAttrib():_TagAttr;
  Sequencer()->_PrevTagInfo->_TagTypeFound        = _TagTypeFound;

  Sequencer()->_PrevTagInfo->_NestingError        = Sequencer()->_NestingError;
  Sequencer()->_PrevTagInfo->_TagNestCond         = Sequencer()->_TagNestCond;
  Sequencer()->_PrevTagInfo->_NCTagNestCond       = Sequencer()->_NCTagNestCond;

  Sequencer()->_PrevTagInfo->_TagFound            = _TagFound;
  Sequencer()->_PrevTagInfo->_ContentTypeTag      = _ContentTypeTag;
  Sequencer()->_PrevTagInfo->_NonContentTypeTag   = _NonContentTypeTag;
  Sequencer()->_PrevTagInfo->_HasPairContent      = _HasPairContent;
  Sequencer()->_PrevTagInfo->_EndBrkFound         = _EndBrkFound;
  Sequencer()->_PrevTagInfo->_TagSpcFound         = _TagSpcFound;
  Sequencer()->_PrevTagInfo->_SingletonTagFound   = _SingletonTagFound;
  Sequencer()->_PrevTagInfo->_EmptyTagFound       = _EmptyTagFound;
  Sequencer()->_PrevTagInfo->_ContainerTagFound   = _ContainerTagFound;
  Sequencer()->_PrevTagInfo->_StdTagFound         = _StdTagFound;
  Sequencer()->_PrevTagInfo->_PairEnderFound      = _PairEnderFound;
  Sequencer()->_PrevTagInfo->_SingletonTagAllowed = _SingletonTagAllowed;
  Sequencer()->_PrevTagInfo->_EmptyTagAllowed     = _EmptyTagAllowed;
  Sequencer()->_PrevTagInfo->_SearchDone          = _SearchDone;
  Sequencer()->_PrevTagInfo->_TagName             = *_ElementStr;
  Sequencer()->_PrevTagInfo->_TagInfoSet          = true;
  Sequencer()->_PrevTagInfo->_InfoNode            = InfoPtr_;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetHasPairContent(bool Flag_)
{
  _HasPairContent = Flag_;
  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetPairEnderTag(bool PairEnder_)
{
  if (PairEnder_)
    _TagAttr |= TagEnums::PAIR_ENDER;
  else if (!PairEnder_)
    _TagAttr &= ~TagEnums::PAIR_ENDER;

  _TagCntData->SetPairEnderTag(_ElementPtr, PairEnder_);

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetContainerTag(bool Contain_)
{
  bool Opt_ = (_TagAttr & TagEnums::OPTIONALPAIR_TAGATTR);
  bool Pair_ = (_TagAttr & TagEnums::PAIR_TAGATTR);
  bool ContainerTagAllowed_ = (Opt_ || Pair_);

  if (Contain_ && ContainerTagAllowed_)
    _TagAttr |= TagEnums::TAG_CONTAINER;
  else if (!Contain_)
    _TagAttr &= ~TagEnums::TAG_CONTAINER;

  _TagCntData->SetContainerTag(_ElementPtr, Contain_);
  _TagCntData->SetSingletonTag(_ElementPtr, false);

  if (_TagAttr & SINGLETON_TAGATTR)
    SetSingletonTag(false);

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetEmptyTag(bool Empty_)
{
  if (Empty_ && _EmptyTagAllowed)
  {
    _TagAttr |= TagEnums::EMPTY_TAGATTR;
    _HasPairContent = false;
  }
  else if (!Empty_)
    _TagAttr &= ~TagEnums::EMPTY_TAGATTR;

  _TagCntData->SetEmptyTag(_ElementPtr, Empty_);
  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetSingletonTag(bool Single_)
{
  if (Single_ && _SingletonTagAllowed)
  {
    _TagAttr |= TagEnums::SINGLETON_TAGATTR;
    _HasPairContent = false;
  }
  else if (!Single_)
    _TagAttr &= ~TagEnums::SINGLETON_TAGATTR;

  _TagCntData->SetSingletonTag(_ElementPtr, Single_);

  if (_TagAttr & TAG_CONTAINER)
    SetContainerTag(false);

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetTextBufferPtr(const ChrString* Ptr_)
{
  if (_SearchResult)
    _SearchResult->SetTextBufferPtr(Ptr_);
    
  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::AssignSearchResult(Subscript Index_, Subscript Length_,
                                             Subscript MIndex_, Subscript MLength_,
                                             Subscript EIndex_, Subscript ELength_)
{
  if (_SearchResult)
    _SearchResult->AssignSearchResult(Index_, Length_,
                                      MIndex_, MLength_,
                                      EIndex_, ELength_);

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::AdvanceReadPos()
{
  if (_SearchResult)
    _SearchResult->AdvanceReadPos();

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::ResetReadPos()
{
  // TagTypeInfo methods called by TagDictionary::CreateTag method
  // and the corresponding indicators set by each:
  //
  //   TagInfop_->SetDocType(_DocType);
  //     _DocType = DocType_;
  //     _DataSpecified |= DOCTYPE_SPECIFIED;
  //
  //   TagInfop_->SetTagBrk(StartBrk_, EndBrk_, NULL);
  //     _TagBrkIndex = x;
  //     _DataSpecified |= TAGBRKINDEX_SPECIFIED;
  //
  //   TagInfop_->SetElementStr(true, new ChrString(Element_), true);
  //     _TagSpecified = true;
  //     _DataSpecified |= ELEMENT_SPECIFIED;
  //     _ElementStr = new ChrString(*Ptr_);
  //
  //   TagInfop_->SetEmptyTag(Empty_);
  //   TagInfop_->SetSingletonTag(Singleton_);
  //     _TagAttr |= Flag;
  //     _HasPairContent = false;
  //
  //   TagInfop_->SetTagAttr(attr);
  //     _TagAttr = TagAttr_ & Mask_;
  //     _DataSpecified |= TAGATTR_SPECIFIED;
  //
  //   TagInfop_->SetContentTypeTag(StartBrk_, EndBrk_);
  //     _HasPairContent = _ContentTypeTag = IsContentTypeTag(start_, end_);
  //     _NonContentTypeTag = IsNonContentTypeTag(start_, end_);
  //
  //   TagInfop_->SetHasPairContent(TagPos_ == TagEnums::OPENING);
  //     _HasPairContent = Flag_;
  //

  if (_SearchResult)
  {
    _SearchResult->ResetReadPos();
    _TagFound = false;

    if (!(_DictionaryTag && _TagSpecified &&
         (_DataSpecified & TAGBRKINDEX_SPECIFIED)))
    {
      _HasPairContent =
      _ContentTypeTag = false;
      _NonContentTypeTag = false;
    }

    _TagTypeFound = 0;
    _EndBrkFound = false;
    _TagSpcFound = false;
    _SingletonTagFound = false;
    _EmptyTagFound = false;
    _ContainerTagFound = false;
    _StdTagFound = false;
    _PairEnderFound = false;

    ResetTagCount();
    Sequencer()->ResetSearchRunNumber();
    Sequencer()->ErasePrevTagInfo();
  }

  return *this;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::Reset(int Spec_)
{
  // TagTypeInfo methods called by TagDictionary::CreateTag method
  // and the corresponding indicators set by each:
  //
  //   TagInfop_->SetDocType(_DocType);
  //     _DocType = DocType_;
  //     _DataSpecified |= DOCTYPE_SPECIFIED;
  //
  //   TagInfop_->SetTagBrk(StartBrk_, EndBrk_, NULL);
  //     _TagBrkIndex = x;
  //     _DataSpecified |= TAGBRKINDEX_SPECIFIED;
  //
  //   TagInfop_->SetElementStr(true, new ChrString(Element_), true);
  //     _TagSpecified = true;
  //     _DataSpecified |= ELEMENT_SPECIFIED;
  //     _ElementStr = new ChrString(*Ptr_);
  //
  //   TagInfop_->SetEmptyTag(Empty_);
  //   TagInfop_->SetSingletonTag(Singleton_);
  //     _TagAttr |= Flag;
  //     _HasPairContent = false;
  //
  //   TagInfop_->SetTagAttr(attr);
  //     _TagAttr = TagAttr_ & Mask_;
  //     _DataSpecified |= TAGATTR_SPECIFIED;
  //
  //   TagInfop_->SetContentTypeTag(StartBrk_, EndBrk_);
  //     _HasPairContent = _ContentTypeTag = IsContentTypeTag(start_, end_);
  //     _NonContentTypeTag = IsNonContentTypeTag(start_, end_);
  //
  //   TagInfop_->SetHasPairContent(TagPos_ == TagEnums::OPENING);
  //     _HasPairContent = Flag_;
  //

  if (_SearchResult)
  {
    if (Spec_ == ALL)
    {
      _SearchResult->ResetReadPos();
      _TagFound = false;

      if (!_DictionaryTag)
        _DataSpecified = 0;

      if (!(_DictionaryTag && _TagSpecified &&
           (_DataSpecified & TAGBRKINDEX_SPECIFIED)))
      {
        _HasPairContent =
        _ContentTypeTag = false;
        _NonContentTypeTag = false;
      }

      _TagTypeFound = 0;
      _EndBrkFound = false;
      _TagSpcFound = false;
      _SingletonTagFound = false;
      _EmptyTagFound = false;
      _ContainerTagFound = false;
      _StdTagFound = false;
      _PairEnderFound = false;

      if (!(_DictionaryTag && _TagSpecified &&
           (_DataSpecified & TAGATTR_SPECIFIED)))
        _TagAttr = 0;

      if (!(_DictionaryTag && _TagSpecified &&
           (_DataSpecified & DOCTYPE_SPECIFIED)))
        _DocType = 0;

      ResetTagList();
      ResetTagCount();
      ResetSearchRunNumber();
      Sequencer()->ErasePrevTagInfo();
    }
    else
    {
      if (Spec_ & READPOS)
      {
        _SearchResult->ResetReadPos();
        _TagFound = false;

        if (!(_DictionaryTag && _TagSpecified &&
             (_DataSpecified & TAGBRKINDEX_SPECIFIED)))
        {
          _HasPairContent =
          _ContentTypeTag = false;
          _NonContentTypeTag = false;
        }

        _TagTypeFound = 0;
        _EndBrkFound = false;
        _TagSpcFound = false;
        _SingletonTagFound = false;
        _EmptyTagFound = false;
        _ContainerTagFound = false;
        _StdTagFound = false;
        _PairEnderFound = false;

        ResetTagCount();
        ResetSearchRunNumber();
        Sequencer()->ErasePrevTagInfo();
      }

      if (Spec_ & SEARCH_PARAMS)
      {
        if (!_DictionaryTag)
          _DataSpecified = 0;

        if (!(_DictionaryTag && _TagSpecified &&
             (_DataSpecified & TAGATTR_SPECIFIED)))
          _TagAttr = 0;

        if (!(_DictionaryTag && _TagSpecified &&
             (_DataSpecified & DOCTYPE_SPECIFIED)))
          _DocType = 0;
      }

      if (Spec_ & SEARCH_TARGETS)
      {
        _TagFound = false;
        _EndBrkFound = false;
        _TagSpcFound = false;
        _SingletonTagFound = false;
        _EmptyTagFound = false;
        _ContainerTagFound = false;
        _StdTagFound = false;
        _PairEnderFound = false;
      }

      if (Spec_ & TAG_CONTENT_TYPE)
        if (!(_DictionaryTag && _TagSpecified &&
             (_DataSpecified & TAGBRKINDEX_SPECIFIED)))
        {
          _ContentTypeTag = false;
          _NonContentTypeTag = false;
          _HasPairContent = false;
        }

      if (Spec_ & TAGLIST)
        ResetTagList();
    }
  }

  return *this;
}

/****************************************************************************/
void TagTypeInfo::ResetTagCount()
{
  Sequencer()->ResetNestingError();
  Sequencer()->ResetStdTagCount();
  Sequencer()->ResetNestedCmntTagCount();
  Sequencer()->ResetCopenCmntTagCount();

  _TagSwitch._HasEndError =
  _TagSwitch._HasNestedCommentError =
  _TagSwitch._HasCLangError = false;

  _TagSwitch._EndErrorCode =
  _TagSwitch._NestedCommentErrorCode =
  _TagSwitch._CLangErrorCode = 0;

  if (_TagInitConfirmed)
    RestoreTagSwitches();
}

/****************************************************************************/
void TagTypeInfo::ResetTagList()
{
  if (!(_DictionaryTag && _TagSpecified &&
       (_DataSpecified & TAGBRKINDEX_SPECIFIED)))
  {
    _TagBrkIndex = 0;
    _AltIndex = 0;
  }

  if (!(_DictionaryTag && _TagSpecified &&
       (_DataSpecified & ELEMENT_SPECIFIED)))
  {
    delete _ElementStr;
    _ElementStr = NULL;
  }

  if (!_DictionaryTag)
    _TagSpecified = false;
}

/****************************************************************************/
void TagTypeInfo::ResetSearchParams()
{
  if (_SearchResult && !_DictionaryTag)
    _DataSpecified = 0;
}

/****************************************************************************/
void TagTypeInfo::ResetSearchTargets()
{
  _TagFound = false;
  _EndBrkFound = false;
  _TagSpcFound = false;
  _SingletonTagFound = false;
  _EmptyTagFound = false;
  _ContainerTagFound = false;
  _StdTagFound = false;
  _PairEnderFound = false;
}

/****************************************************************************/
void TagTypeInfo::ResetTagContentType()
{
  if (!(_DictionaryTag && _TagSpecified &&
       (_DataSpecified & TAGBRKINDEX_SPECIFIED)))
  {
    _ContentTypeTag = false;
    _NonContentTypeTag = false;
    _HasPairContent = false;
  }
}

/****************************************************************************/
Subscript TagTypeInfo::ReadPos() const
{
  return (_SearchResult ? _SearchResult->ReadPos():0);
}

/****************************************************************************/
const ChrString& TagTypeInfo::ElementStr() const
{
  ChrString Dummy_;
  return (_ElementStr ? *_ElementStr:Dummy_);
}

/****************************************************************************/
void TagTypeInfo::DumpElementSeqListNode(ostream& os_, size_t fw_) const
{
  if (_TagCntData && _TagCntData->TagDataUpdated() &&
      TagTypeInfo::ShowDebugDump())
    _TagCntData->DumpElementSeqListNode(os_, fw_);
}

/****************************************************************************/
void TagTypeInfo::DumpStackNodeInfo(ostream& os_, size_t fw_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  StackNodeInfo* Prev_ = NULL;
  StackNodeInfo* Tail_ = Sequencer()->_TailInfo;

  while (Tail_ && Prev_ != Tail_ && Sequencer()->SearchRunNumber() > 0)
  {
    if (Tail_->HasThisParentAndSearchNum(this))
    {
      Tail_->DumpStackNodeInfo(os_, fw_);
      break;
    }

    if (Sequencer()->SearchRunNumber() > 0 && Prev_ != Tail_)
    {
      Prev_ = Tail_;
      Tail_ = (StackNodeInfo*)(Tail_->Prev());
    }
  }
}

/****************************************************************************/
void TagTypeInfo::DumpSearchResults(ostream& os_, size_t fw_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  const TagSearchResult* Result_ = SearchResult();
  ChrString RemStr_ = Result_->c_str();
  Subscript rPos_ = ReadPos();

  if (ElementStr_Specified())
  {
    os_.width(fw_);
    os_ <<std::left <<"Element String Specified: true" <<endl;
    os_.width(fw_);
    os_ <<std::left <<"Element String: " <<ElementStr() <<endl;
  }

  if (!TagTypeInfo::ShowNewLines())
  {
    RemStr_.RemovePadding(" \t\n\r\f\v");
    SubstChar(RemStr_.c_str(), "\n\r", ' ');
    RemStr_.SimplifySpaces(false);
  }

  #if TAGREADER_ABBREV_DUMP
    os_.width(fw_); os_ <<std::left <<"Read Position: " <<rPos_ <<endl;
  #else
    os_.width(fw_); os_ <<std::left <<"Read Position: " <<rPos_ <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Index: " <<_TagBrkIndex <<endl;
    os_.width(fw_); os_ <<std::left <<"Start Brk Found: " <<StartBrkFound() <<endl;
    os_.width(fw_); os_ <<std::left <<"End Brk Found: " <<EndBrkFound() <<endl;
    os_.width(fw_); os_ <<std::left <<"String Remaining: " <<RemStr_ <<endl;
  #endif

  Result_->DumpSearchResults(os_, fw_, _ShowNewLines);

  if (_DumpTagContents)
  {
    if (_IO)
    {
      _IO->SetOutput(&os_);
      _IO->SetFieldWidth(fw_);
    }

    DumpIntraTagContentData(os_, fw_);
  }
  else
    os_ <<endl;
}

/****************************************************************************/
void TagTypeInfo::DumpTagContentData(ostream& os_, size_t fw_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  const TagSearchResult* Result_ = SearchResult();
  Result_->DumpTagContentData(os_, fw_);
  os_ <<endl;
}

/****************************************************************************/
void TagTypeInfo::DumpIntraTagContentData(ostream& os_, size_t fw_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  const TagSearchResult* Result_ = SearchResult();
  int ContentType_ = Result_->ContentType();
  bool HasIntTagCont_ = (ContentType_ & TagSearchResult::INTRA_TAG_CONTENT) ||
                        (ContentType_ & TagSearchResult::INTRA_NESTEDTAG_CONTENT);

  if (HasIntTagCont_)
    Result_->DumpIntraTagContentData(os_, fw_);
}

/****************************************************************************/
void TagTypeInfo::ShouldDumpContents(bool Flag_, ostream& os_, size_t fw_) const
{
  _DumpTagContents = Flag_;

  if (Flag_ && _IO)
  {
    _IO->SetOutput(&os_);
    _IO->SetFieldWidth(fw_);
  }
}

/****************************************************************************/
StackNodeInfo* TagTypeInfo::HeadInfo(bool Reset_, bool NotErased_)
{
  Sequencer();
  return (_Sequencer ? _Sequencer->HeadInfo(Reset_, NotErased_):NULL);
}

/****************************************************************************/
StackNodeInfo* TagTypeInfo::TailInfo(bool Reset_, bool NotErased_)
{
  Sequencer();
  return (_Sequencer ? _Sequencer->TailInfo(Reset_, NotErased_):NULL);
}

/****************************************************************************/
StackNodeInfo* TagTypeInfo::NextStackInfo(bool Reset_, bool Peek_, bool NotErased_)
{
  Sequencer();
  return (_Sequencer ? _Sequencer->NextStackInfo(Reset_, Peek_, NotErased_):NULL);
}

/****************************************************************************/
StackNodeInfo* TagTypeInfo::PrevStackInfo(bool Reset_, bool Peek_, bool NotErased_)
{
  Sequencer();
  return (_Sequencer ? _Sequencer->PrevStackInfo(Reset_, Peek_, NotErased_):NULL);
}

/****************************************************************************/
StackNodeInfo* TagTypeInfo::SetStackInfoPtr(bool Set_)
{
  Sequencer();
  return (_Sequencer ? _Sequencer->SetStackInfoPtr(Set_):NULL);
}

/****************************************************************************/
StackNodeInfo* TagTypeInfo::CursorTagStkPtr()
{
  Sequencer();
  return (_Sequencer ? _Sequencer->CursorTagStkPtr():NULL);
}

/****************************************************************************/
StackNodeInfo* TagTypeInfo::TagStackPtr()
{
  Sequencer();
  return (_Sequencer ? _Sequencer->TagStackPtr():NULL);
}

/****************************************************************************/
StackNodeInfo* TagTypeInfo::StackInfoPtr()
{
  Sequencer();
  return (_Sequencer ? _Sequencer->StackInfoPtr():NULL);
}

/****************************************************************************/
// example 1:
// ----------
// a:<StdTagType1> d:-1,0,+3;b:+1,+2,+1
//      content for 1a   // tag content
//   b:<StdTagType2> d:-1,-1,+2;c:-1,0,+1;b:+1,+1,0
//      content for 2    // tag content
//   c:</StdTagType2> d:-1,-2,+1;c:-1,-1,0
//      content for 1b   // tag content
// d:</StdTagType1> d:-1,-1,0
//
// <StdTagType2>._Content._Str == "content for 1a"   // belongs to <StdTagType1>
//
// </StdTagType2>._Content._Str == "content for 2"   // belongs to <StdTagType2>
//
// </StdTagType1>._Content._Str == "content for 1b"  // belongs to <StdTagType1>
//
// example 2:
// ----------
// a:<StdTagType1> f:-1,0,+5;b:+1,+2,+1
//      content for 1a       // tag content
//   b:<StdTagType2> f:-1,-1,+4;e:-1,0,+3;c:+1,+2,+1;b:+1,+1,0
//        content for 2a     // tag content
//     c:<StdTagType3> f:-1,-2,+3;e:-1,-1,+2;d:-1,0,+1;c:+1,+1,0
//          content for 3    // tag content
//     d:</StdTagType3> f:-1,-3,+2;e:-1,-2,+1;d:-1,-1,0
//        content for 2b     // tag content
//   e:</StdTagType2> f:-1,-2,+1;e:-1,-1,0
//      content for 1b       // tag content
// f:</StdTagType1> f:-1,-1,0
//
// <StdTagType2>._Content._Str == "content for 1a"   // belongs to <StdTagType1>
//
// <StdTagType3>._Content._Str == "content for 2a"   // belongs to <StdTagType2>
//
// </StdTagType3>._Content._Str == "content for 3"   // belongs to <StdTagType3>
//
// </StdTagType2>._Content._Str == "content for 2b"  // belongs to <StdTagType2>
//
// </StdTagType1>._Content._Str == "content for 1b"  // belongs to <StdTagType1>
//
// example 3:
// ----------
// a:<StdTagType1> f:-1,0,+5;d:+1,+2;b:+1,+2,+1
//      content for 1a   // tag content
//   b:<StdTagType2> f:-1,-1,+4;d:+1,+1,+2;c:-1,0,+1;b:+1,+1,0
//      content for 2    // tag content
//   c:</StdTagType2> f:-1,-2,+3;d:+1,0,+1;c:-1,-1,0
//      content for 1b   // tag content
//   d:<StdTagType3> f:-1,-1,+2;e:-1,0,+1;d:+1,+1,0
//      content for 3    // tag content
//   e:</StdTagType3> f:-1,-2,+1;e:-1,-1,0
//      content for 1c   // tag content
// f:</StdTagType1> f:-1,-1,0;
//
// <StdTagType2>._Content._Str == "content for 1a"   // belongs to <StdTagType1>
//
// </StdTagType2>._Content._Str == "content for 2"   // belongs to <StdTagType2>
//
// <StdTagType3>._Content._Str == "content for 1b"   // belongs to <StdTagType1>
//
// </StdTagType3>._Content._Str == "content for 3"   // belongs to <StdTagType3>
//
// </StdTagType1>._Content._Str == "content for 1c"  // belongs to <StdTagType1>
//
// example 4:
// ----------
// a:<StdTagType> b:0,+1,+1
//      content for 1a   // tag content
//   b:<EmptyTagType/> c:0,+1,+1;b:0,0,0
//      content for 1b   // tag content
//   c:<SingleTagType> d:-1,0,+1;c:0,0,0
//      content for 1c   // tag content
// d:</StdTagType> d:-1,-1,0
//
// <EmptyTagType/>._Content._Str == "content for 1a"  // belongs to <StdTagType>
//
// <SingleTagType>._Content._Str == "content for 1b"  // belongs to <EmptyTagType>
//
// </StdTagType>._Content._Str == "content for 1c"    // belongs to <SingleTagType>
//
// Code:
// -----
// Starting conditions for ShiftTagContents:
// If EndTag then
//     (TagEndDir_ == -1 && TagEndVal_ == -1 && Level_ == 0)
// If StartTag then
//     (TagEndDir_ == +1 && TagEndVal_ == +1 && Level_ == 0)
// If SingletonTag or EmptyTag then
//     (TagEndDir_ == 0 && TagEndVal_ == 0 && Level_ == 0)
//
// Stopping conditions for ShiftTagContents:
// If EndTag then
//     (TagEndDir_ == -1 && TagEndVal_ == 0 && Level_ > 0)
// If StartTag then
//     (TagEndDir_ == +1 && TagEndVal_ == +2 && Level_ > 0)
// If SingletonTag or EmptyTag then
//     (TagEndDir_ == 0 && TagEndVal_ == +1 && Level_ > 0)
//
// if (HeadNode_)
//   HeadNode_->ShiftTagContents(HeadNode_->GiveNext(), TagEndDir_, TagEndVal_, 0);
//
void TagTypeInfo::ShiftTagContents()
{
  StackNodeInfo* HeadNode_ = HeadInfo(true, true);
  int TagEndDir_ = 0;

  if (HeadNode_ && HeadNode_->_TagFound)
  {
    TagEndDir_ = HeadNode_->StdTagFound() ? 1:
                 HeadNode_->PairEnderFound() ? -1:
                   (HeadNode_->EmptyTagFound() ||
                    HeadNode_->SingletonTagFound()) ? 0:2;

    if (TagEndDir_ != 2 && HeadNode_->GiveNext())
      HeadNode_->ShiftTagContents((StackNodeInfo*)(HeadNode_->GiveNext()),
                                  TagEndDir_, TagEndDir_, 0);
  }
}

/****************************************************************************/
TagLinksSequencer* TagTypeInfo::Sequencer()
{
  if (!_Sequencer)
    CreateSequencer();
  else
    _Sequencer->SetParent(this);

  return _Sequencer;
}

/****************************************************************************/
const TagLinksSequencer* TagTypeInfo::Sequencer() const
{
  if (!_Sequencer)
  {
    _Sequencer = new TagLinksSequencer;
    _SequencerAlloc = true;
    _SequencerSet = false;
  }

  return _Sequencer;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetSequencer(TagLinksSequencer* Ptr_)
{
  if (_SequencerAlloc && _Sequencer)
    delete _Sequencer;

  _Sequencer = Ptr_;
  _Sequencer->SetParent(this);
  _SequencerAlloc = false;
  _SequencerSet = true;

  return *this;
}

/****************************************************************************/
TagLinksSequencer* TagTypeInfo::CreateSequencer()
{
  if (_SequencerAlloc && _Sequencer)
    delete _Sequencer;

  _Sequencer = new TagLinksSequencer;
  _Sequencer->SetParent(this);
  _SequencerAlloc = true;
  _SequencerSet = false;
}

/****************************************************************************/
HtmlSpecialCaseData* TagTypeInfo::HtmlSpecialCase()
{
  if (!_HtmlSpecialCase)
    CreateHtmlSpecialCase();
  else
    _HtmlSpecialCase->SetParent(this);

  return _HtmlSpecialCase;
}

/****************************************************************************/
const HtmlSpecialCaseData* TagTypeInfo::HtmlSpecialCase() const
{
  if (!_HtmlSpecialCase)
  {
    _HtmlSpecialCase = new HtmlSpecialCaseData;
    _HtmlSpecialCaseAlloc = true;
    _HtmlSpecialCaseSet = false;
  }

  return _HtmlSpecialCase;
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::SetHtmlSpecialCase(HtmlSpecialCaseData* Ptr_)
{
  if (_HtmlSpecialCaseAlloc && _HtmlSpecialCase)
    delete _HtmlSpecialCase;

  _HtmlSpecialCase = Ptr_;
  _HtmlSpecialCase->SetParent(this);
  _HtmlSpecialCaseAlloc = false;
  _HtmlSpecialCaseSet = true;

  if (_TagCntData)
    _TagCntData->SetHtmlSpecialCase(_HtmlSpecialCase);

  return *this;
}

/****************************************************************************/
HtmlSpecialCaseData* TagTypeInfo::CreateHtmlSpecialCase()
{
  if (_HtmlSpecialCaseAlloc && _HtmlSpecialCase)
    delete _HtmlSpecialCase;

  _HtmlSpecialCase = new HtmlSpecialCaseData;
  _HtmlSpecialCase->SetParent(this);
  _HtmlSpecialCaseAlloc = true;
  _HtmlSpecialCaseSet = false;
}

/****************************************************************************/
// Wrapper methods for TagCountData class methods : start
/****************************************************************************/
const int* TagTypeInfo::GiveTagCountArray(int Index_) const
{
  return (_TagCntData ? _TagCntData->GiveTagCountArray(Index_):NULL);
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::AddToTagList(int TagPos_, TagTypeInfo* MatchingTag_, bool& Updated_)
{
  // long x = _TagBrkIndex;
  TagElement* vptr = NULL;  
  TagElement* MatchingElement_ = MatchingTag_ ?
                                     MatchingTag_->GiveTagElementPtr():
                                     NULL;

  if ((_DataSpecified == ALL_SPECIFIED) ||
      (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED) &&
       _TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED) &&
       _ElementStr && (_DataSpecified & ELEMENT_SPECIFIED)))
    if (!_TagCntData || !IsContentTypeTag(StartBrk(), EndBrk()))
      vptr = NULL;
    else
      vptr = _TagCntData->AddToTagList(TagPos_, IsEmptyTag(), IsContainerTag(),
                                       _TagAttr, StartBrk(), *_ElementStr,
                                       EndBrk(), MatchingElement_);

  if (vptr)
  {
    _ElementPtr = vptr;
    _TagAttr = vptr->TagAttrib();
    return this;
  }

  return NULL;
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::MakeMatchTag(TagTypeInfo* StartTag_,
                                       bool SetMatchTag_, bool AddToList_)
{
  long x = 0;
  TagTypeInfo* tiptr = NULL;
  TagElement* vptr = NULL;
  TagElement* StartElement_ = StartTag_ ?
                                  StartTag_->GiveTagElementPtr():
                                  NULL;

  if (StartElement_)
  {
    x = StartElement_->TagBrkIndex();

    if (!_TagCntData || !IsContentTypeTag(_TagBrkMatrix[x], _TagBrkMatrix[x+1]))
      vptr = NULL;
    else
      vptr = _TagCntData->MakeMatchTag(StartElement_, SetMatchTag_, AddToList_);

    if (vptr)
    {
      tiptr = new TagTypeInfo;
      tiptr->SetEmptyTag(false);

      if (tiptr)
      {
        tiptr->SetTagElementPtr(vptr);

        if (_TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED))
        {
          x = vptr->TagBrkIndex();
          tiptr->SetTagBrk(_TagBrkMatrix[x], _TagBrkMatrix[x+1]);
        }
        else
          tiptr->SetTagBrk(vptr->GiveTagBrk(), vptr->GiveTagBrk(1), NULL);

        if (_ElementStr && (_DataSpecified & ELEMENT_SPECIFIED))
        {
          ChrString TagName_ = vptr->TagName();
          tiptr->SetElementStr(_TagSpecified, &TagName_);
        }
        else
          tiptr->SetElementStr(true, new ChrString(vptr->TagName()), true);

        if (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED))
        {
          int attr = vptr->TagAttrib();
          tiptr->SetTagAttr(attr);
        }
        else
          tiptr->SetTagAttr(vptr->TagAttrib());
      }
    }
  }

  return tiptr;
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::ResetRequiredTags(TagTypeInfo* TagPtr_)
{
  long x = 0;
  TagTypeInfo* tiptr = NULL;
  TagElement* vptr = NULL;
  TagElement* StartElement_ = TagPtr_ ?
                                  TagPtr_->GiveTagElementPtr():
                                  NULL;

  if (StartElement_)
  {
    x = StartElement_->TagBrkIndex();

    if (!_TagCntData || !IsContentTypeTag(_TagBrkMatrix[x], _TagBrkMatrix[x+1]))
      vptr = NULL;
    else
      vptr = _TagCntData->ResetRequiredTags(StartElement_);

    if (vptr)
    {
      tiptr = TagPtr_;

      if (tiptr)
      {
        if (_TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED))
        {
          x = vptr->TagBrkIndex();
          tiptr->SetTagBrk(_TagBrkMatrix[x], _TagBrkMatrix[x+1]);
        }
        else
          tiptr->SetTagBrk(vptr->GiveTagBrk(), vptr->GiveTagBrk(1), NULL);

        if (_ElementStr && (_DataSpecified & ELEMENT_SPECIFIED))
        {
          ChrString TagName_ = vptr->TagName();
          tiptr->SetElementStr(_TagSpecified, &TagName_);
        }
        else
          tiptr->SetElementStr(true, new ChrString(vptr->TagName()), true);

        if (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED))
        {
          int attr = vptr->TagAttrib();
          tiptr->SetTagAttr(attr);
        }
        else
          tiptr->SetTagAttr(vptr->TagAttrib());
      }
    }
  }

  return tiptr;
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::SetHtmlContentTagType(TagTypeInfo* TagPtr_, int ContentType_)
{
  long x = 0;
  TagTypeInfo* tiptr = NULL;
  TagElement* vptr = NULL;
  TagElement* StartElement_ = TagPtr_ ?
                                  TagPtr_->GiveTagElementPtr():
                                  NULL;

  if (StartElement_)
  {
    x = StartElement_->TagBrkIndex();

    if (!_TagCntData || !IsContentTypeTag(_TagBrkMatrix[x], _TagBrkMatrix[x+1]))
      vptr = NULL;
    else
      vptr = _TagCntData->SetHtmlContentTagType(StartElement_, ContentType_);

    if (vptr)
    {
      tiptr = TagPtr_;

      if (tiptr)
      {
        if (_TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED))
        {
          x = vptr->TagBrkIndex();
          tiptr->SetTagBrk(_TagBrkMatrix[x], _TagBrkMatrix[x+1]);
        }
        else
          tiptr->SetTagBrk(vptr->GiveTagBrk(), vptr->GiveTagBrk(1), NULL);

        if (_ElementStr && (_DataSpecified & ELEMENT_SPECIFIED))
        {
          ChrString TagName_ = vptr->TagName();
          tiptr->SetElementStr(_TagSpecified, &TagName_);
        }
        else
          tiptr->SetElementStr(true, new ChrString(vptr->TagName()), true);

        if (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED))
        {
          int attr = vptr->TagAttrib();
          tiptr->SetTagAttr(attr);
        }
        else
          tiptr->SetTagAttr(vptr->TagAttrib());
      }
    }
  }

  return tiptr;
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::SetRequiredCond(TagTypeInfo* TagPtr_,
                                          int ReqTagOrder_, int ImpliedEnd_,
                                          int TagCond_, int ImpliedEndCond_)
{
  long x = 0;
  TagTypeInfo* tiptr = NULL;
  TagElement* vptr = NULL;
  TagElement* StartElement_ = TagPtr_ ?
                                  TagPtr_->GiveTagElementPtr():
                                  NULL;

  if (StartElement_)
  {
    x = StartElement_->TagBrkIndex();

    if (!_TagCntData || !IsContentTypeTag(_TagBrkMatrix[x], _TagBrkMatrix[x+1]))
      vptr = NULL;
    else
      vptr = _TagCntData->SetRequiredCond(StartElement_,
                                          ReqTagOrder_, ImpliedEnd_,
                                          TagCond_, ImpliedEndCond_);

    if (vptr)
    {
      tiptr = TagPtr_;

      if (tiptr)
      {
        if (_TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED))
        {
          x = vptr->TagBrkIndex();
          tiptr->SetTagBrk(_TagBrkMatrix[x], _TagBrkMatrix[x+1]);
        }
        else
          tiptr->SetTagBrk(vptr->GiveTagBrk(), vptr->GiveTagBrk(1), NULL);

        if (_ElementStr && (_DataSpecified & ELEMENT_SPECIFIED))
        {
          ChrString TagName_ = vptr->TagName();
          tiptr->SetElementStr(_TagSpecified, &TagName_);
        }
        else
          tiptr->SetElementStr(true, new ChrString(vptr->TagName()), true);

        if (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED))
        {
          int attr = vptr->TagAttrib();
          tiptr->SetTagAttr(attr);
        }
        else
          tiptr->SetTagAttr(vptr->TagAttrib());
      }
    }
  }

  return tiptr;
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::AddRequiredTag(TagTypeInfo* TagPtr_, TagTypeInfo* ReqTag_,
                                         int ReqTagOrder_, bool MatchingEnd_)
{
  long x = 0;
  TagTypeInfo* tiptr = NULL;
  TagElement* vptr = NULL;
  TagElement* ReqElement_ = ReqTag_ ?
                                ReqTag_->GiveTagElementPtr():
                                NULL;
  TagElement* StartElement_ = TagPtr_ ?
                                  TagPtr_->GiveTagElementPtr():
                                  NULL;

  if (StartElement_)
  {
    x = StartElement_->TagBrkIndex();

    if (!_TagCntData || !IsContentTypeTag(_TagBrkMatrix[x], _TagBrkMatrix[x+1]))
      vptr = NULL;
    else
      vptr = _TagCntData->AddRequiredTag(StartElement_, ReqElement_,
                                         ReqTagOrder_, MatchingEnd_);

    if (StartElement_)
    {
      tiptr = TagPtr_;
      vptr = StartElement_;

      if (tiptr)
      {
        if (_TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED))
        {
          x = vptr->TagBrkIndex();
          tiptr->SetTagBrk(_TagBrkMatrix[x], _TagBrkMatrix[x+1]);
        }
        else
          tiptr->SetTagBrk(vptr->GiveTagBrk(), vptr->GiveTagBrk(1), NULL);

        if (_ElementStr && (_DataSpecified & ELEMENT_SPECIFIED))
        {
          ChrString TagName_ = vptr->TagName();
          tiptr->SetElementStr(_TagSpecified, &TagName_);
        }
        else
          tiptr->SetElementStr(true, new ChrString(vptr->TagName()), true);

        if (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED))
        {
          int attr = vptr->TagAttrib();
          tiptr->SetTagAttr(attr);
        }
        else
          tiptr->SetTagAttr(vptr->TagAttrib());
      }
    }
  }

  return tiptr;
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::AddRequiredTagElement(TagTypeInfo* TagPtr_, TagElement* ReqTag_,
                                                int ReqTagOrder_, bool MatchingEnd_)
{
  long x = 0;
  TagTypeInfo* tiptr = NULL;
  TagElement* vptr = NULL;
  TagElement* ReqElement_ = ReqTag_;
  TagElement* StartElement_ = TagPtr_ ?
                                  TagPtr_->GiveTagElementPtr():
                                  NULL;

  if (StartElement_)
  {
    x = StartElement_->TagBrkIndex();

    if (!_TagCntData || !IsContentTypeTag(_TagBrkMatrix[x], _TagBrkMatrix[x+1]))
      vptr = NULL;
    else
      vptr = _TagCntData->AddRequiredTag(StartElement_, ReqElement_,
                                         ReqTagOrder_, MatchingEnd_);

    if (StartElement_)
    {
      tiptr = TagPtr_;
      vptr = StartElement_;

      if (tiptr)
      {
        if (_TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED))
        {
          x = vptr->TagBrkIndex();
          tiptr->SetTagBrk(_TagBrkMatrix[x], _TagBrkMatrix[x+1]);
        }
        else
          tiptr->SetTagBrk(vptr->GiveTagBrk(), vptr->GiveTagBrk(1), NULL);

        if (_ElementStr && (_DataSpecified & ELEMENT_SPECIFIED))
        {
          ChrString TagName_ = vptr->TagName();
          tiptr->SetElementStr(_TagSpecified, &TagName_);
        }
        else
          tiptr->SetElementStr(true, new ChrString(vptr->TagName()), true);

        if (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED))
        {
          int attr = vptr->TagAttrib();
          tiptr->SetTagAttr(attr);
        }
        else
          tiptr->SetTagAttr(vptr->TagAttrib());
      }
    }
  }

  return tiptr;
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::SetRequiredTags(TagTypeInfo* TagPtr_, TagElement** Vect_,
                                          int ReqTagOrder_, bool MatchingEnd_)
{
  long x = 0;
  TagTypeInfo* tiptr = NULL;
  TagElement* vptr = NULL;
  TagElement* StartElement_ = TagPtr_ ?
                                  TagPtr_->GiveTagElementPtr():
                                  NULL;

  if (StartElement_)
  {
    x = StartElement_->TagBrkIndex();

    if (!_TagCntData || !IsContentTypeTag(_TagBrkMatrix[x], _TagBrkMatrix[x+1]))
      vptr = NULL;
    else
      _TagCntData->SetRequiredTags(StartElement_, Vect_,
                                   ReqTagOrder_, MatchingEnd_);

    if (vptr)
    {
      tiptr = TagPtr_;
      vptr = StartElement_;

      if (tiptr)
      {
        if (_TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED))
        {
          x = vptr->TagBrkIndex();
          tiptr->SetTagBrk(_TagBrkMatrix[x], _TagBrkMatrix[x+1]);
        }
        else
          tiptr->SetTagBrk(vptr->GiveTagBrk(), vptr->GiveTagBrk(1), NULL);

        if (_ElementStr && (_DataSpecified & ELEMENT_SPECIFIED))
        {
          ChrString TagName_ = vptr->TagName();
          tiptr->SetElementStr(_TagSpecified, &TagName_);
        }
        else
          tiptr->SetElementStr(true, new ChrString(vptr->TagName()), true);

        if (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED))
        {
          int attr = vptr->TagAttrib();
          tiptr->SetTagAttr(attr);
        }
        else
          tiptr->SetTagAttr(vptr->TagAttrib());
      }
    }
  }

  return tiptr;
}

/****************************************************************************/
int TagTypeInfo::PushBrkNesting(int TagPos_, int Nest_)
{
  bool PrevSearchDone_ = SearchRunNumber() && Sequencer()->_PrevTagInfo && _SearchDone;
  bool CStyle_ = _DocType == TagEnums::C_FILE;
  bool CppStyle_ = _DocType == TagEnums::CPP_FILE;
  bool XmlStyle_ = _DocType == TagEnums::XML_FILE;
  bool HtmlStyle_ = !_DocType ||
                    _DocType == TagEnums::HTML_FILE ||
                    _DocType == TagEnums::XHTML_FILE;

  int ret = 0;
  int x = _AltIndex ? GiveAltIndex(true):_TagBrkIndex;
  const char* OpenBrk_ = (x >= 0) ? _TagBrkMatrix[x]:NULL;
  const char* CloseBrk_ = (x >= 0) ? _TagBrkMatrix[x+1]:NULL;

  if (!OpenBrk_ || !_TagCntData || !PrevSearchDone_)
    return ret;

  if (TagPos_ == TagEnums::OPENING)
  {
    if (HtmlStyle_ || XmlStyle_)
    {
      // NESTEDCOMMENT1OPEN_STR  "<!--"
      //
      if (HtmlStyle_ && x == TagEnums::NESTEDCOMMENT1_DEX)
        ret = _TagCntData ?
                  _TagCntData->PushBrkNesting(COMMENTOPEN_STR, OpenBrk_, Nest_):
                  NULL;

      // STDTAGOPEN_STR  "<"
      //
      else if (x == TagEnums::STD_DEX)
      {
        if ((_TagCntData && Sequencer()->_PrevTagInfo->_TagAttr & TagEnums::OPTIONALPAIR_TAGATTR))
          _TagCntData->PopBrkNesting(OpenBrk_, OpenBrk_, 1);

        ret = _TagCntData ?
                  _TagCntData->PushBrkNesting(OpenBrk_, OpenBrk_, Nest_):
                  NULL;
      }

      // ENDTAGOPEN_STR  "</"
      //
      else if ((x == TagEnums::END_DEX) &&
               (!(_TagAttr & TagEnums::PAIR_TAGATTR) &&
                !(_TagAttr & TagEnums::OPTIONALPAIR_TAGATTR)))
        ret = _TagCntData ?
                  _TagCntData->PushBrkNesting(OpenBrk_, OpenBrk_, Nest_):
                  NULL;
    }
    else if (CStyle_ || CppStyle_)
    {
      // COPENCOMMENTBLKOPEN_STR  "/*"
      //
      if (x == TagEnums::COPENCOMMENTBLK_DEX)
        ret = _TagCntData ?
                  _TagCntData->PushBrkNesting(OpenBrk_, OpenBrk_, Nest_):
                  NULL;
    }
  }
  else if (TagPos_ == TagEnums::FORBIDDEN)
  {
    if (CppStyle_)
    {
      // valid only for Cpp tags
      // CPPCOMMENTLINEOPEN_STR   "//"
      // CPPCOMMENTLINECLOSE_STR  "=>"
      //    
      if (x == TagEnums::CPPCOMMENTLINEOPEN_DEX)
        ret = _TagCntData ?
                  _TagCntData->PushBrkNesting(OpenBrk_, OpenBrk_, Nest_):
                  NULL;
    }
    else if (HtmlStyle_ || XmlStyle_)
    {
      // DOCTYPEOPEN_STR   "<!"
      // DOCTYPECLOSE_STR  ">"
      // COMMENTOPEN_STR   "<!--"
      // COMMENTCLOSE_STR  "-->"
      // SCRIPTOPEN_STR    "<?"
      // SCRIPTCLOSE_STR   "?>"
      //
      if (x == TagEnums::DOCTYPE_DEX ||
          x == TagEnums::COMMENT_DEX ||
          x == TagEnums::SCRIPT_DEX)
        ret = _TagCntData ?
                  _TagCntData->PushBrkNesting(OpenBrk_, OpenBrk_, Nest_):
                  NULL;

      // STDTAGOPEN_STR  "<"
      //
      else if (x == TagEnums::STD_DEX &&
               (_TagAttr & TagEnums::SINGLETON_TAGATTR))
      {
        if (_TagCntData && (Sequencer()->_PrevTagInfo->_TagAttr & TagEnums::OPTIONALPAIR_TAGATTR))
          _TagCntData->PopBrkNesting(OpenBrk_, OpenBrk_, 1);

        ret = _TagCntData ?
                  _TagCntData->PushBrkNesting(OpenBrk_, OpenBrk_, Nest_):
                  NULL;
      }

      // EMPTYOPEN_STR  "<"
      //
      else if (x == TagEnums::EMPTY_DEX &&
               (_TagAttr & TagEnums::EMPTY_TAGATTR))
      {
        if (_TagCntData && (Sequencer()->_PrevTagInfo->_TagAttr & TagEnums::OPTIONALPAIR_TAGATTR))
          _TagCntData->PopBrkNesting(STDTAGOPEN_STR, OpenBrk_, 1);

        ret = _TagCntData ?
                  _TagCntData->PushBrkNesting(OpenBrk_, OpenBrk_, Nest_):
                  NULL;
      }
    }
  }
  else if (TagPos_ == TagEnums::CLOSING)
  {
    ++x;
    
    if (HtmlStyle_)
    {
      // NESTEDCOMMENT1CLOSE_STR  ">"
      //
      if (x == TagEnums::NESTEDCOMMENT1_DEX + 1)
        ret = _TagCntData ?
                  _TagCntData->PushBrkNesting(OpenBrk_, CloseBrk_, Nest_):
                  NULL;
    }
    else if (CStyle_ || CppStyle_)
    {
      // COPENCOMMENTBLKCLOSE_STR  "->"
      //
      if (x == TagEnums::COPENCOMMENTBLK_DEX + 1)
        ret = _TagCntData ?
                  _TagCntData->PushBrkNesting(CloseBrk_, CloseBrk_, Nest_):
                  NULL;
    }
  }

  return ret;
}

/****************************************************************************/
int TagTypeInfo::PopBrkNesting(int TagPos_, int Nest_)
{
  bool PrevSearchDone_ = SearchRunNumber() && Sequencer()->_PrevTagInfo && _SearchDone;
  bool CStyle_ = _DocType == TagEnums::C_FILE;
  bool CppStyle_ = _DocType == TagEnums::CPP_FILE;
  bool XmlStyle_ = _DocType == TagEnums::XML_FILE;
  bool HtmlStyle_ = !_DocType ||
                    _DocType == TagEnums::HTML_FILE ||
                    _DocType == TagEnums::XHTML_FILE;

  int ret = 0;
  int x = _AltIndex ? GiveAltIndex(true):_TagBrkIndex;
  const char* OpenBrk_ = (x >= 0) ? _TagBrkMatrix[x]:NULL;
  const char* CloseBrk_ = (x >= 0) ? _TagBrkMatrix[x+1]:NULL;

  if (!OpenBrk_ || !_TagCntData || !PrevSearchDone_)
    return ret;

  if (TagPos_ == TagEnums::OPENING)
  {
    if (HtmlStyle_ || XmlStyle_)
    {
      // NESTEDCOMMENT2OPEN_STR  "<!"
      //
      if (HtmlStyle_ && x == TagEnums::NESTEDCOMMENT2_DEX &&
          _TagCntData->GiveBrkNesting(COMMENTOPEN_STR) &&
          _TagCntData->GiveBrkNesting(NESTEDCOMMENT1OPEN_STR))
        ret = _TagCntData->PopBrkNesting(COMMENTOPEN_STR, OpenBrk_, Nest_);

      // ENDTAGOPEN_STR  "</"
      //
      else
      {
        if ((x == TagEnums::END_DEX) &&
               ((_TagAttr & TagEnums::PAIR_TAGATTR) ||
                (_TagAttr & TagEnums::OPTIONALPAIR_TAGATTR)))
          ret = _TagCntData->PopBrkNesting(STDTAGOPEN_STR, OpenBrk_, Nest_);
        else
          ret = _TagCntData->PopBrkNesting(OpenBrk_, OpenBrk_, Nest_);
      }
    }
    else if (CStyle_ || CppStyle_)
    {
      // CCLOSECOMMENTBLKOPEN_STR  "<-"
      //
      if (x == TagEnums::CCLOSECOMMENTBLK_DEX &&
          _TagCntData->GiveBrkNesting(COPENCOMMENTBLKOPEN_STR) &&
          _TagCntData->GiveBrkNesting(COPENCOMMENTBLKCLOSE_STR))
        ret = _TagCntData->PopBrkNesting(COPENCOMMENTBLKOPEN_STR, OpenBrk_, Nest_);
    }
  }
  else if (TagPos_ == TagEnums::FORBIDDEN)
  {
    if (CppStyle_)
    {
      // valid only for Cpp tags
      // CPPCOMMENTLINEOPEN_STR   "//"
      // CPPCOMMENTLINECLOSE_STR  "=>"
      //    
      if (x == TagEnums::CPPCOMMENTLINEOPEN_DEX)
        ret = _TagCntData->PopBrkNesting(OpenBrk_, CloseBrk_, Nest_);
    }
    else if (HtmlStyle_ || XmlStyle_)
    {
      // STDTAGOPEN_STR  "<"
      //
      if (x == TagEnums::STD_DEX &&
          (_TagAttr & TagEnums::SINGLETON_TAGATTR))
      {
        if ((Sequencer()->_PrevTagInfo->_TagAttr & TagEnums::OPTIONALPAIR_TAGATTR))
          _TagCntData->PopBrkNesting(OpenBrk_, OpenBrk_, 1);

        ret = _TagCntData->PushBrkNesting(OpenBrk_, OpenBrk_, Nest_);
      }

      // EMPTYOPEN_STR  "<"
      //
      else if (x == TagEnums::EMPTY_DEX &&
               (_TagAttr & TagEnums::EMPTY_TAGATTR))
      {
        if ((Sequencer()->_PrevTagInfo->_TagAttr & TagEnums::OPTIONALPAIR_TAGATTR))
          _TagCntData->PopBrkNesting(STDTAGOPEN_STR, OpenBrk_, 1);

        ret = _TagCntData->PushBrkNesting(OpenBrk_, OpenBrk_, Nest_);
      }
    }
  }
  else if (TagPos_ == TagEnums::CLOSING)
  {
    ++x;
    
    if (HtmlStyle_)
    {
      // NESTEDCOMMENT2CLOSE_STR  "-->"
      //
      if (x == TagEnums::NESTEDCOMMENT2_DEX + 1)
      {
        if (_TagCntData->GiveBrkNesting(NESTEDCOMMENT1OPEN_STR))
          ret = _TagCntData->PopBrkNesting(NESTEDCOMMENT1OPEN_STR, CloseBrk_, Nest_);
        else
          ret = _TagCntData->PopBrkNesting(COMMENTOPEN_STR, COMMENTCLOSE_STR, Nest_);
      }

      // STDTAGCLOSE_STR  ">"
      //
      else if (x == TagEnums::STD_DEX + 1)
      {
        if ((_TagAttr & TagEnums::SINGLETON_TAGATTR))
          ret = _TagCntData->PopBrkNesting(OpenBrk_, CloseBrk_, 1);
        else
          ret = _TagCntData->PopBrkNesting(OpenBrk_, CloseBrk_, 0);
      }

      // EMPTYCLOSE_STR  "/>"
      //
      else if (x == TagEnums::EMPTY_DEX + 1)
      {
        if ((_TagAttr & TagEnums::EMPTY_TAGATTR))
          ret = _TagCntData->PopBrkNesting(OpenBrk_, CloseBrk_, 1);
        else
          ret = _TagCntData->PopBrkNesting(OpenBrk_, CloseBrk_, 0);
      }
    }
    else if (CStyle_ || CppStyle_)
    {
      // CCLOSECOMMENTBLKCLOSE_STR  "*/"
      //
      if (x == TagEnums::CCLOSECOMMENTBLK_DEX + 1 &&
          _TagCntData->GiveBrkNesting(COPENCOMMENTBLKCLOSE_STR))
        ret = _TagCntData->PopBrkNesting(COPENCOMMENTBLKCLOSE_STR, CloseBrk_, Nest_);
    }    
  }

  return ret;
}

/****************************************************************************/
int TagTypeInfo::BrkIndexToTagType(int BrkIndex_) const
{
  return ((long)::pow(2, BrkIndex_/2));
}

/****************************************************************************/
int TagTypeInfo::BrkStringToIndex(const char* Brk_, int& BrkEnder_) const
{
  return (_TagCntData ?
          _TagCntData->BrkStringToIndex(Brk_, BrkEnder_):0);
}

/****************************************************************************/
int TagTypeInfo::BrkStringToIndex(const char* StartBrk_, const char* EndBrk_,
                                  int& PairEnder_) const
{
  return (_TagCntData ?
          _TagCntData->BrkStringToIndex(StartBrk_, EndBrk_, PairEnder_):0);
}

/****************************************************************************/
int TagTypeInfo::GiveTagCount(const char* TxtFndBrk_) const
{
  return (_TagCntData ? _TagCntData->GiveTagCount(TxtFndBrk_):NULL);
}

/****************************************************************************/
int TagTypeInfo::GiveBrkNesting(const char* TxtFndBrk_) const
{
  return (_TagCntData ? _TagCntData->GiveBrkNesting(TxtFndBrk_):NULL);
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::UpdateTag(int TagPos_)
{
  // long x = _TagBrkIndex;
  TagElement* vptr = NULL;
  TagElement* savedptr = _ElementPtr;
  bool GenSearchDone_ = !_TagSpecified && _SearchDone &&
                        _TagFound && _ElementStr;

  if (GenSearchDone_ ||
      (_DataSpecified == ALL_SPECIFIED) ||
      (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED) &&
       _TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED) &&
       _ElementStr && (_DataSpecified & ELEMENT_SPECIFIED)))
  {
    const char* StartBrk_ = GenSearchDone_ ? StartBrkFound():StartBrk();
    const char* EndBrk_ = GenSearchDone_ ? EndBrkFound():EndBrk();

    if (!_TagCntData || !IsContentTypeTag(StartBrk_, EndBrk_))
      vptr = NULL;
    else
      vptr = _TagCntData->UpdateTag(TagPos_, IsEmptyTag(), IsContainerTag(),
                                    StartBrk_, *_ElementStr, EndBrk_);
  }

  if (vptr)
  {
    _ElementPtr = vptr;

    if (vptr && vptr->IsPairStartTag() &&
        savedptr && savedptr->IsPairEndTag())
      _ElementPtr = savedptr;

    _TagAttr = _ElementPtr->TagAttrib();
    return this;
  }

  return NULL;
}

/****************************************************************************/
TagTypeInfo* TagTypeInfo::SetThisTag(int TagPos_)
{
  // long x = _TagBrkIndex;
  TagElement* vptr = NULL;

  if ((_DataSpecified == ALL_SPECIFIED) ||
      (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED) &&
       _TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED) &&
       _ElementStr && (_DataSpecified & ELEMENT_SPECIFIED)))
    if (!IsContentTypeTag(StartBrk(), EndBrk()))
      vptr = NULL;
    else
      vptr = _TagCntData ?
                 _TagCntData->SearchForTag(TagPos_, IsEmptyTag(), IsContainerTag(),
                                           StartBrk(), *_ElementStr, EndBrk()):
                 NULL;

  if (vptr)
  {
    _ElementPtr = vptr;
    _TagAttr = vptr->TagAttrib();
    return this;
  }

  return NULL;
}

/****************************************************************************/
bool TagTypeInfo::HasThisTag(int TagPos_) const
{
  // long x = _TagBrkIndex;
  TagElement* vptr = NULL;

  if ((_DataSpecified == ALL_SPECIFIED) ||
      (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED) &&
       _TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED) &&
       _ElementStr && (_DataSpecified & ELEMENT_SPECIFIED)))
  {
    if (!IsContentTypeTag(StartBrk(), EndBrk()))
      return true;
    else
      vptr = _TagCntData ?
                 _TagCntData->SearchForTag(TagPos_, IsEmptyTag(), IsContainerTag(),
                                           StartBrk(), *_ElementStr, EndBrk()):
                 NULL;
  }

  return (vptr != NULL);
}

/****************************************************************************/
TagTypeInfo& TagTypeInfo::ResetTagIterIndex()
{
  if (_TagCntData)
    _TagCntData->ResetTagIterIndex();

  return *this;
}

/****************************************************************************/
bool TagTypeInfo::SetTagIterIndex(int TagIndex_)
{
  return (_TagCntData ? _TagCntData->SetTagIterIndex(TagIndex_):NULL);
}

/****************************************************************************/
bool TagTypeInfo::AdvanceTagIter()
{
  return (_TagCntData ? _TagCntData->AdvanceTagIter():NULL);
}

/****************************************************************************/
TagElement* TagTypeInfo::GetCurrentTagElement()
{
  return (_TagCntData ? _TagCntData->GetCurrentTagElement():NULL);
}

/****************************************************************************/
size_t TagTypeInfo::TagListSize(int TagPos_) const
{
  return (_TagCntData ? _TagCntData->TagListSize(TagPos_):NULL);
}

/****************************************************************************/
bool TagTypeInfo::AtFirstTag() const
{
  return (_TagCntData ? _TagCntData->AtFirstTag():NULL);
}

/****************************************************************************/
bool TagTypeInfo::AtLastTag() const
{
  return (_TagCntData ? _TagCntData->AtLastTag():NULL);
}

/****************************************************************************/
// Wrapper methods for TagCountData class methods : end
/****************************************************************************/
bool TagTypeInfo::InDocType(int DocType_) const
{
  int ExcludedTags_ = 0;
  int TagType_ = TagIndexToTagType();
  
  bool CStyle_ = DocType_ == TagEnums::C_FILE;
  bool CppStyle_ = DocType_ == TagEnums::CPP_FILE;
  bool XmlStyle_ = DocType_ == TagEnums::XML_FILE;
  bool HtmlStyle_ = DocType_ == TagEnums::HTML_FILE ||
                    DocType_ == TagEnums::XHTML_FILE;

  if (CppStyle_ || CStyle_)
  {
    ExcludedTags_ =
    GiveExcludedTags
    (
        false,
        false,
        false,
        false,
        false,
        false,
        false,
        (CStyle_ || CppStyle_),
        CppStyle_
    );
  }
  else if (XmlStyle_)
  {
    ExcludedTags_ =
    GiveExcludedTags
    (
        true,
        ((!(_SingletonTagAllowed &&
           (_TagAttr & TagEnums::SINGLETON_TAGATTR)) ||
          (_TagAttr & TagEnums::PAIR_ENDER) ||
          !_TagSpecified) &&
         _TagSwitch._IsEnd),
        _TagSwitch._IsScript,
        _TagSwitch._IsDoc,
        false,
        false,
        (((_EmptyTagAllowed &&
           ((_TagAttr & TagEnums::EMPTY_TAGATTR) || !_TagSpecified)) ||
          (_SingletonTagAllowed &&
           ((_TagAttr & TagEnums::SINGLETON_TAGATTR) || !_TagSpecified))) &&
         _TagSwitch._IsSingle),
        false,
        false
    );
  }
  else if (HtmlStyle_)
  {
    ExcludedTags_ =
    GiveExcludedTags
    (
        true,        
        ((!(_SingletonTagAllowed &&
           (_TagAttr & TagEnums::SINGLETON_TAGATTR)) ||
          (_TagAttr & TagEnums::PAIR_ENDER) ||
          !_TagSpecified) &&
         _TagSwitch._IsEnd),
        _TagSwitch._IsScript,
        _TagSwitch._IsDoc,
        _TagSwitch._IsComment,
        _TagSwitch._IsNestedComment,
        (((_EmptyTagAllowed &&
           ((_TagAttr & TagEnums::EMPTY_TAGATTR) || !_TagSpecified)) ||
          (_SingletonTagAllowed &&
           ((_TagAttr & TagEnums::SINGLETON_TAGATTR) || !_TagSpecified))) &&
         _TagSwitch._IsSingle),
        false,
        false
    );
  }

  bool ret = !(ExcludedTags_ & TagType_);
  return ret;
}

/****************************************************************************/
TagSearchResult::TAGTERMLISTTYPE
TagTypeInfo::InitChrStringPtrs(TagSearchResult::TAGTERMLISTTYPE arrptr_, size_t max)
{
  size_t x;
  for (x = 0; x < max; x++)
    arrptr_[x] = NULL;
  return arrptr_;
}

/****************************************************************************/
TagTypeInfo** TagTypeInfo::InitTagTypeArray(TagTypeInfo** arrptr_, size_t max)
{
  size_t x;
  for (x = 0; x < max; x++)
    arrptr_[x] = NULL;
  return arrptr_;
}

/****************************************************************************/
MEMORYOPS_DEFN(TagTypeInfo)

/****************************************************************************/
// QListNode class definition
/****************************************************************************/
QListNode::QListNode():
_NextBase(NULL),
_PrevBase(NULL),
_Copied(false),
_Erased(false),
_NodeID(0)
{
  // increment StackSize and assign to the Node ID of this object
  _NodeID = ++_StackSize;
}

/****************************************************************************/
QListNode::QListNode(const QListNode& Obj_):
_NextBase(NULL),
_PrevBase(NULL),
_Copied(true),
_Erased(Obj_._Erased),
_NodeID(0)
{
  // increment StackSize and assign to the Node ID of this object
  if (!_Erased)
    _NodeID = ++_StackSize;
}

/****************************************************************************/
QListNode::~QListNode()
{
  if (!_Copied)
  {
    --_StackSize;
    _Copied = true;

    if (_Erased)
    {
      --_NumErased;
      _Erased = false;
    }
  }

  SetPrev(NULL);
  delete _NextBase;
  SetNext(NULL);
}

/****************************************************************************/
int QListNode::ClassType() const
{
  return QListNode::QLISTNODE;
}

/****************************************************************************/
QListNode& QListNode::operator = (const QListNode& List_)
{
  if (this != &List_)
  {
    // remove entire stack
    SetPrev(NULL);
    delete _NextBase;
    SetNext(NULL);

    // QListNode class data members
    SetPrev(NULL);
    SetNext(NULL);
    _Copied = true;
  }

  return *this;
}

/****************************************************************************/
bool QListNode::IsEmpty()
{
  return ((_StackSize - _NumErased) <= 0);
}

/****************************************************************************/
size_t QListNode::Size()
{
  return _StackSize;
}

/****************************************************************************/
size_t QListNode::NumErased()
{
  return _NumErased;
}

/****************************************************************************/
QListNode* QListNode::Tail(int Type_)
{
  QListNode* Ptr_ = this;

  while (Ptr_ && Ptr_->Next())
    Ptr_ = Ptr_->GiveNext();

  return Ptr_;
}

/****************************************************************************/
const QListNode* QListNode::Tail(int Type_) const
{
  const QListNode* Ptr_ = this;

  while (Ptr_ && Ptr_->Next())
    Ptr_ = Ptr_->Next();

  return Ptr_;
}

/****************************************************************************/
QListNode* QListNode::Head(int Type_)
{
  QListNode* Ptr_ = this;

  while (Ptr_ && Ptr_->Prev())
    Ptr_ = Ptr_->GivePrev();

  return Ptr_;
}

/****************************************************************************/
const QListNode* QListNode::Head(int Type_) const
{
  const QListNode* Ptr_ = this;

  while (Ptr_ && Ptr_->Prev())
    Ptr_ = Ptr_->Prev();

  return Ptr_;
}

/****************************************************************************/
bool QListNode::Preceeds(QListNode* EncTag_, bool OrSameAs_)
{
  return (EncTag_ ? (OrSameAs_ ? (_NodeID <= EncTag_->NodeID()):
                                 (_NodeID < EncTag_->NodeID())):false);
}

/****************************************************************************/
bool QListNode::Follows(QListNode* EncTag_, bool OrSameAs_)
{
  return (EncTag_ ? (OrSameAs_ ? (_NodeID >= EncTag_->NodeID()):
                                 (_NodeID > EncTag_->NodeID())):false);
}

/****************************************************************************/
bool QListNode::SameAs(QListNode* EncTag_)
{
  return (EncTag_ ? (_NodeID == EncTag_->NodeID()):false);
}

/****************************************************************************/
QListNode* QListNode::GiveNext()
{
  return _NextBase;
}

/****************************************************************************/
const QListNode* QListNode::Next() const
{
  return _NextBase;
}

/****************************************************************************/
QListNode* QListNode::SetNext(QListNode* Ptr_)
{
  _NextBase = Ptr_;
  return _NextBase;
}

/****************************************************************************/
QListNode* QListNode::GivePrev()
{
  return _PrevBase;
}

/****************************************************************************/
const QListNode* QListNode::Prev() const
{
  return _PrevBase;
}

/****************************************************************************/
QListNode* QListNode::SetPrev(QListNode* Ptr_)
{
  _PrevBase = Ptr_;
  return _PrevBase;
}

/****************************************************************************/
QListNode* QListNode::RemoveHead(QListNode*& Head_)
{
  if (Head_)
  {
    QListNode* Ptr_ = Head_;
    Head_ = Head_->GiveNext();
    Ptr_->SetNext(NULL);
    Head_->SetPrev(NULL);
    Ptr_->SetPrev(NULL);
    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
QListNode* QListNode::AppendHead(QListNode* Ptr_, bool NestTag_)
{
  if (Ptr_ && Ptr_ != this)
  {
    Ptr_->SetNext(this);
    if (Next() == Ptr_)
      SetNext(NULL);

    SetPrev(Ptr_);
    Ptr_->SetPrev(NULL);
    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
void QListNode::SetNodeID(size_t ID_)
{
  _NodeID = ID_;
}

/****************************************************************************/
void QListNode::SetErased(bool Flag_)
{
  _Erased = Flag_;

  if (_Erased)
    _NumErased++;
}

/****************************************************************************/
void QListNode::SetCopied(bool Flag_)
{
  _Copied = Flag_;
}

/****************************************************************************/
bool QListNode::Unlink()
{
  if (_Erased && !_Copied)
  {
    SetPrev(NULL);
    SetNext(NULL);

    delete this;
    return true;
  }

  return false;
}

/****************************************************************************/
void QListNode::DeleteAll()
{
  QListNode* Ptr_ = Head();
  delete Ptr_;
}

/****************************************************************************/
MEMORYOPS_DEFN(QListNode)

/****************************************************************************/
// TextNodeData class definition
/****************************************************************************/
TextNodeData::TextNodeData():
QListNode(),

_TextStr(NULL),
_DataType(0),
_DataSet(false)
{}

/****************************************************************************/
TextNodeData::TextNodeData(const TextNodeData& Obj_):
QListNode(Obj_),

_TextStr(NULL),
_DataType(Obj_._DataType),
_DataSet(Obj_._DataSet)
{
  if (Obj_._DataSet)
  {
    if (Obj_._DataType == QListNode::STRING_DATA && Obj_._TextStr)
      _TextStr = new ChrString(*Obj_._TextStr);
    else if (_DataType == QListNode::INFONODE_DATA)
      _NestedNode = Obj_._NestedNode;
  }
}

/****************************************************************************/
TextNodeData::~TextNodeData()
{
  if (_DataSet)
  {
    if (_DataType == QListNode::STRING_DATA)
    {
      delete _TextStr;
      _TextStr = NULL;
    }
    else if (_DataType == QListNode::INFONODE_DATA)
      _NestedNode = NULL;
  }
}

/****************************************************************************/
TextNodeData* TextNodeData::MakeNode()
{
  return (new TextNodeData);
}

/****************************************************************************/
TextNodeData* TextNodeData::MakeNode(const TextNodeData& Obj_)
{
  return (new TextNodeData(Obj_));
}

/****************************************************************************/
TextNodeData* TextNodeData::CopyNode(TextNodeData* TrgData_, TextNodeData* SrcData_)
{
  if (SrcData_)
  {
    if (TrgData_ &&
        TrgData_->_DataSet &&
        TrgData_->_TextStr &&
        TrgData_->_DataType == QListNode::STRING_DATA)
    {
      delete TrgData_->_TextStr;
      TrgData_->_TextStr = NULL;
    }

    TextNodeData* Node_ = TrgData_ ? TrgData_:MakeNode();
    Node_->_DataType = SrcData_->_DataType;
    Node_->_DataSet = SrcData_->_DataSet;
    Node_->_Erased = SrcData_->_Erased;

    if (Node_->_DataSet)
    {
      if (Node_->_DataType == QListNode::STRING_DATA && SrcData_->_TextStr)
        Node_->_TextStr = new ChrString(*SrcData_->_TextStr);
      else if (Node_->_DataType == QListNode::INFONODE_DATA)
        Node_->_NestedNode = SrcData_->_NestedNode;
    }

    return Node_;
  }

  return NULL;
}

/****************************************************************************/
TextNodeData* TextNodeData::CopyTextNodeData(TextNodeData*& TrgData_,
                                             TextNodeData*& TrgDataHead_,
                                             size_t& TrgNumData_,
                                             TextNodeData* SrcData_,
                                             TextNodeData* SrcDataHead_,
                                             size_t SrcNumData_)
{
  TextNodeData* tempPtr_;
  bool NodeCreated_ = false;
  size_t x = TrgNumData_;
  size_t y = SrcNumData_;
  size_t z = 0;

  if (!TrgData_ || !SrcData_ ||
      !TrgDataHead_ || !SrcDataHead_ ||
      !TrgNumData_ || !SrcNumData_)
  {
    if (!SrcData_ || !SrcDataHead_ || !SrcNumData_)
    {
      if (TrgData_ && TrgDataHead_ && 
          TrgData_->ClassType() == QListNode::TEXTNODEDATA &&
          TrgDataHead_->ClassType() == QListNode::TEXTNODEDATA)
      {
        TrgData_->DeleteAll();
        TrgDataHead_ =
        TrgData_ = NULL;
        TrgNumData_ = 0;
      }

      return NULL;
    }
    else if (!TrgData_ || !TrgDataHead_ || !TrgNumData_)
    {
      if (SrcData_ && SrcDataHead_ &&
          SrcData_->ClassType() == QListNode::TEXTNODEDATA &&
          SrcDataHead_->ClassType() == QListNode::TEXTNODEDATA)
      {
        TrgDataHead_ =
        TrgData_ = CopyNode(NULL, SrcData_);
        x = TrgNumData_ = 1;
        SrcData_ = (TextNodeData*)(SrcData_->Prev());
      }

      if (!SrcData_)
        return TrgDataHead_;
      else
        NodeCreated_ = true;
    }
  }

  if (!(SrcData_ && SrcDataHead_ &&
        SrcData_->ClassType() == QListNode::TEXTNODEDATA &&
        SrcDataHead_->ClassType() == QListNode::TEXTNODEDATA) ||
      !(TrgData_ && TrgDataHead_ &&
        TrgData_->ClassType() == QListNode::TEXTNODEDATA &&
        TrgDataHead_->ClassType() == QListNode::TEXTNODEDATA))
    return NULL;

  if (x < y)
  {
    if (!NodeCreated_)
    {
      z = x;
      x = 0;

      for (tempPtr_ = TrgData_; x < z && SrcData_; x++)
      {
        tempPtr_ = CopyNode(tempPtr_, SrcData_);
        tempPtr_ = (TextNodeData*)(tempPtr_->Prev());
        SrcData_ = (TextNodeData*)(SrcData_->Prev());
      }
    }

    while (x < y && SrcData_)
    {
      TrgDataHead_ = (TextNodeData*)(TrgDataHead_->AppendHead(CopyNode(NULL, SrcData_), false));
      SrcData_ = (TextNodeData*)(SrcData_->Prev());
      x++;
    }

    TrgNumData_ = x;
  }
  else if (x >= y)
  {
    while (x > y && TrgDataHead_)
    {
      tempPtr_ = (TextNodeData*)TrgDataHead_;
      TrgDataHead_ = (TextNodeData*)(TrgDataHead_->Next());
      tempPtr_->Unlink();
      x--;
    }

    z = x;
    x = 0;

    for (tempPtr_ = TrgData_; x < z && SrcData_; x++)
    {
      tempPtr_ = CopyNode(tempPtr_, SrcData_);
      tempPtr_ = (TextNodeData*)(tempPtr_->Prev());
      SrcData_ = (TextNodeData*)(SrcData_->Prev());
    }

    TrgNumData_ = x;
  }

  return TrgDataHead_;
}

/****************************************************************************/
int TextNodeData::ClassType() const
{
  return QListNode::TEXTNODEDATA;
}

/****************************************************************************/
QListNode& TextNodeData::operator = (const QListNode& List_)
{
  if (this != &List_)
  {
    QListNode::operator = (List_);
    TextNodeData* ListPtr_ = (List_.ClassType() == TEXTNODEDATA) ?
                                 ((TextNodeData*)(&List_)):NULL;

    if (!ListPtr_)
      return *this;

    if (_TextStr && _DataSet &&
        _DataType == QListNode::STRING_DATA)
    {
      delete _TextStr;
      _TextStr = NULL;
    }

    _Erased = ListPtr_->_Erased;
    _DataSet = ListPtr_->_DataSet;
    _DataType = ListPtr_->_DataType;

    if (_DataSet)
    {
      if (ListPtr_->_DataType == QListNode::STRING_DATA && ListPtr_->_TextStr)
        _TextStr = new ChrString(*ListPtr_->_TextStr);
      else if (_DataType == QListNode::INFONODE_DATA)
        _NestedNode = ListPtr_->_NestedNode;
    }
  }

  return *this;
}

/****************************************************************************/
void TextNodeData::SetDataType(int Type_)
{
  _DataType = Type_;
}

/****************************************************************************/
void TextNodeData::SetCTextStr(const char* Str_)
{
  if (Str_)
  {
    if (_DataType == QListNode::STRING_DATA)
      delete _TextStr;
    else
      _DataType = QListNode::STRING_DATA;

    _TextStr = new ChrString(Str_);
    _DataSet = true;

    if (_TextStr && !TagTypeInfo::ShowNewLines())
    {
      SubstChar(_TextStr->c_str(), "\n\r", ' ');
      _TextStr->SimplifySpaces(false);
    }
  }
}

/****************************************************************************/
void TextNodeData::SetTextStr(const ChrString& Str_)
{
  if (_DataType == QListNode::STRING_DATA)
    delete _TextStr;
  else
    _DataType = QListNode::STRING_DATA;

  _TextStr = new ChrString(Str_);
  _DataSet = true;

  if (_TextStr && !TagTypeInfo::ShowNewLines())
  {
    SubstChar(_TextStr->c_str(), "\n\r", ' ');
    _TextStr->SimplifySpaces(false);
  }
}

/****************************************************************************/
void TextNodeData::AppendCTextStr(const char* Str_)
{
  if (Str_)
  {
    if (_DataType == QListNode::STRING_DATA)
    {
      if (!_TextStr)
        _TextStr = new ChrString(Str_);
      else
        *_TextStr += ChrString(Str_);
    }
    else
    {
      _DataType = QListNode::STRING_DATA;
      _TextStr = new ChrString(Str_);
    }

    _DataSet = true;

    if (_TextStr && !TagTypeInfo::ShowNewLines())
    {
      SubstChar(_TextStr->c_str(), "\n\r", ' ');
      _TextStr->SimplifySpaces(false);
    }
  }
}

/****************************************************************************/
void TextNodeData::AppendTextStr(const ChrString& Str_)
{
  if (_DataType == QListNode::STRING_DATA)
  {
    if (!_TextStr)
      _TextStr = new ChrString(Str_);
    else
      *_TextStr += Str_;
  }
  else
  {
    _DataType = QListNode::STRING_DATA;
    _TextStr = new ChrString(Str_);
  }

  _DataSet = true;

  if (_TextStr && !TagTypeInfo::ShowNewLines())
  {
    SubstChar(_TextStr->c_str(), "\n\r", ' ');
    _TextStr->SimplifySpaces(false);
  }
}

/****************************************************************************/
void TextNodeData::SetNestedNode(StackNodeInfo* NodePtr_)
{
  _NestedNode = NodePtr_;
  _DataSet = true;
}

/****************************************************************************/
void TextNodeData::ShowContents(const StackNodeInfo* ptr_, ostream& os_,
                                size_t fw_, size_t* Index_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  StackNodeInfo* Ptr_ = NULL;
  bool Valid_ = false;
  char Buffer_[32];

  if (Index_)
  {
    ChrString NodeStr_ = ChrString("NodeID[");
    NodeStr_ += ChrString(::UintToStr(*Index_, Buffer_));
    NodeStr_ += ChrString("]: ");
    os_.width(fw_); os_ <<std::left <<NodeStr_ <<_NodeID <<endl;
  }
  else
  {
    os_.width(fw_); os_ <<std::left <<"NodeID: " <<_NodeID <<endl;
  }

  if (_DataSet)
  {
    if (_DataType == QListNode::STRING_DATA)
    {
      os_.width(fw_); os_ <<std::left <<"- TextStr: " <<(_TextStr ? *_TextStr:ChrString("NULL")) <<endl;
    }
    else if (_DataType == QListNode::INFONODE_DATA && _NestedNode)
    {
      os_.width(fw_); os_ <<std::left <<"- NestedNode: ";
      Ptr_ = _NestedNode;

      Valid_ = (Ptr_ && !Ptr_->Erased()) &&
               (Ptr_->_StartBrkFound && Ptr_->_StartBrkStr.strlen());

      if (Valid_)
        StackNodeInfo::ShowTagStr(os_, 0, _NestedNode, StackNodeInfo::PREVNEXT);
      else
        os_ <<"NULL" <<endl;
    }
  }
  else
  {
    os_.width(fw_); os_ <<std::left <<"- DataSet: false" <<endl;
  }
}

/****************************************************************************/
MEMORYOPS_DEFN(TextNodeData)

/****************************************************************************/
// AppTextNode class definition
/****************************************************************************/
AppTextNode::AppTextNode():
QListNode(),

_Parent(NULL),
_HasTagStk(false),
_RefNode(NULL),
_Prev(NULL),
_Next(NULL),

_TextQ(new TextNodeData),
_TextQhead(NULL),
_TextQptr(NULL),
_TextQptrSet(false),
_NumTextData(1),

_SglNode(false),
_StdNode(false),
_AppendText(false),
_DoErase(false),
_PauseAppend(TXTAPP_NOT_PAUSED),
_Transferred(false),
_TransferNode(NULL)
{
  _TextQptr = _TextQhead = _TextQ;
}

/****************************************************************************/
AppTextNode::AppTextNode(bool HasTagStk_, StackNodeInfo* RefNode_):
QListNode(),

_Parent(NULL),
_HasTagStk(HasTagStk_),
_RefNode(RefNode_),
_Prev(NULL),
_Next(NULL),

_TextQ(new TextNodeData),
_TextQhead(NULL),
_TextQptr(NULL),
_TextQptrSet(false),
_NumTextData(1),

_SglNode(RefNode_ ? (RefNode_->SingletonTagFound() ||
                     RefNode_->EmptyTagFound()):false),
_StdNode(RefNode_ ? RefNode_->StdTagFound():false),
_AppendText(false),
_DoErase(false),
_PauseAppend(TXTAPP_NOT_PAUSED),
_Transferred(false),
_TransferNode(NULL)
{
  SetParent(RefNode_ ? RefNode_->_Parent:NULL);
  _TextQptr = _TextQhead = _TextQ;
}

/****************************************************************************/
AppTextNode::AppTextNode(const AppTextNode& Obj_):
QListNode(Obj_),

_Parent(Obj_._Parent),
_HasTagStk(Obj_._HasTagStk),
_RefNode(Obj_._RefNode),
_Prev(NULL),
_Next(NULL),

_TextQ(new TextNodeData),
_TextQhead(NULL),
_TextQptr(NULL),
_TextQptrSet(false),
_NumTextData(1),

_SglNode(Obj_._SglNode),
_StdNode(Obj_._StdNode),
_AppendText(Obj_._AppendText),
_DoErase(Obj_._DoErase),
_PauseAppend(Obj_._PauseAppend),
_Transferred(false),
_TransferNode(NULL)
{
  _TextQhead = _TextQ;
  TextNodeData::CopyTextNodeData(_TextQ, _TextQhead, _NumTextData,
                                 Obj_._TextQ, Obj_._TextQhead, Obj_._NumTextData);
  _TextQptr = _TextQhead;
}

/****************************************************************************/
AppTextNode::~AppTextNode()
{
  if (_RefNode)
    _RefNode->SetAppTextNode(NULL);

  SetPrev(NULL);
  _RefNode = NULL;
  _TransferNode = NULL;

  if (_TextQhead);
    _TextQhead->DeleteAll();

  _TextQptr =
  _TextQ =
  _TextQhead = NULL;

  // delete all: taken care of by QListNode virtual destructor
  // delete _Next;
  // SetNext(NULL);
}

/****************************************************************************/
AppTextNode* AppTextNode::Make()
{
  return (new AppTextNode);
}

/****************************************************************************/
AppTextNode* AppTextNode::Make(const AppTextNode& Obj_)
{
  return (new AppTextNode(Obj_));
}

/****************************************************************************/
const char* AppTextNode::PauseAppendTypeToStr(int Type_)
{
  return
  (
    (Type_ == TXTAPP_NOT_PAUSED) ? "TXTAPP_NOT_PAUSED":
    (Type_ == NEED_STDTAG)       ? "NEED_STDTAG":
    (Type_ == STDTAG_APPENDED)   ? "STDTAG_APPENDED":
    (Type_ == NEED_ENDTAG)       ? "NEED_ENDTAG":
    (Type_ == ENDTAG_APPENDED)   ? "ENDTAG_APPENDED":
    (Type_ == TAG_COMPLETED)     ? "TAG_COMPLETED":""
  );
}

/****************************************************************************/
// TextNodeData traversing methods
/****************************************************************************/
TextNodeData* AppTextNode::PopTextNode(bool Erase_, bool ConfirmErased_)
{
  TextNodeData* RetPtr_ = NULL;
  TextNodeData* EndPt_ = NULL;
  TextNodeData* SavedPtr_ = NULL;
  TextNodeData* RelPtr_ = (_TextQptr && _TextQptrSet) ?
                               _TextQptr:
                               (_TextQptr=HeadTextNode(true, Erase_ && !ConfirmErased_));

  bool HeadEnd_ = _TextQptr == _TextQhead;
  bool LastNode_ = false;

  if (!RelPtr_)
    return NULL;

  if (!_TextQptrSet && _TextQptr && Erase_ && !ConfirmErased_)
    SetTextNodePtr(true);

  if (HeadEnd_)
  {
    SavedPtr_ = _TextQptr;
    _TextQptr = (TextNodeData*)(_TextQptr->Next());
    LastNode_ = _TextQptr == NULL;

    if (!LastNode_)
      _TextQhead = _TextQptr;
  }
  else
  {
    SavedPtr_ = RelPtr_;
    _TextQhead = (TextNodeData*)(RelPtr_->Next());
    LastNode_ = _TextQhead == NULL;

    if (LastNode_)
      _TextQptr = _TextQhead = RelPtr_;
  }

  if (ConfirmErased_)
  {
    if (RelPtr_->Erased())
    {
      if (RelPtr_->Next())
      {
        RelPtr_->GiveNext()->SetPrev(NULL);
        EndPt_ = (RelPtr_ != RelPtr_->Next()) ?
                     (TextNodeData*)(RelPtr_->Next()):NULL;
        RelPtr_->SetNext(NULL);
      }

      RetPtr_ = RelPtr_;
      RelPtr_ = EndPt_;

      if (LastNode_)
        _TextQptr = _TextQhead = NULL;
      else if (!HeadEnd_)
        _TextQptr = EndPt_;
    }
    else
    {
      if (HeadEnd_)
      {
        if (!LastNode_)
          _TextQhead = _TextQptr = SavedPtr_;
        else
          _TextQptr = _TextQhead;
      }
      else if (!LastNode_)
        _TextQptr = _TextQhead = RelPtr_;

      RelPtr_ = NULL;
    }
  }
  else if (Erase_)
  {
    if (!LastNode_)
      EndPt_ = _TextQhead;

    if (!ConfirmErased_)
    {
      while (RelPtr_ != EndPt_)
      {

        RelPtr_->SetErased(true);
        RelPtr_ = (TextNodeData*)(RelPtr_->Next());
      }

      if (LastNode_)
        RelPtr_ = _TextQ;
    }

    _TextQhead = SavedPtr_;
  }

  _NumTextData = _TextQhead->Size();
  return (RetPtr_ ? RetPtr_:RelPtr_);
}

/****************************************************************************/
TextNodeData* AppTextNode::NextTextNode(bool Reset_, bool Peek_, bool NotErased_)
{
  if (_TextQptr && !Reset_)
  {
    bool TailEnd_ = _TextQptr == _TextQ;
    TextNodeData* PrevPtr_ = _TextQptr;
    TextNodeData* NextPtr_ = _TextQptr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (TextNodeData*)(NextPtr_->Next());
    }
    while (NotErased_ && NextPtr_ && NextPtr_->Erased());

    if (Peek_)
      return NextPtr_;
    else
      _TextQptr = NextPtr_;

    if (TailEnd_)
    {
      _TextQ = _TextQptr ? _TextQptr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _TextQptr = HeadTextNode(true, NotErased_);

  return _TextQptr;
}

/****************************************************************************/
TextNodeData* AppTextNode::PrevTextNode(bool Reset_, bool Peek_, bool NotErased_)
{
  if (_TextQptr && !Reset_)
  {
    bool HeadEnd_ = _TextQptr == _TextQhead;
    TextNodeData* PrevPtr_ = _TextQptr;
    TextNodeData* NextPtr_ = _TextQptr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (TextNodeData*)(NextPtr_->Prev());
    }
    while (NotErased_ && NextPtr_ && NextPtr_->Erased());

    if (Peek_)
      return NextPtr_;
    else
      _TextQptr = NextPtr_;

    if (HeadEnd_)
    {
      _TextQhead = _TextQptr ? _TextQptr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _TextQptr = TailTextNode(true, NotErased_);

  return _TextQptr;
}

/****************************************************************************/
TextNodeData* AppTextNode::PushTextNode(TextNodeData* Node_, bool Create_)
{
  TextNodeData* NodePtr_ = (Create_ && Node_) ? Node_->MakeNode(*Node_):Node_;

  if (NodePtr_)
  {
    if (_TextQhead)
    {
      HeadTextNode(true, false);
      _TextQptr =
      _TextQhead = (TextNodeData*)(_TextQhead->AppendHead(NodePtr_, false));
      SetTextNodePtr(false);
    }
    else
      _TextQptr =
      _TextQhead =
      _TextQ = NodePtr_;

    _NumTextData = _TextQhead->Size();
  }

  return NodePtr_;
}

/****************************************************************************/
TextNodeData* AppTextNode::HeadTextNode(bool Reset_, bool NotErased_) const
{
  TextNodeData* PrevPtr_ = _TextQhead;
  TextNodeData* NextPtr_ = _TextQhead;

  if (_TextQhead && NotErased_)
  {
    bool TailEnd_ = PrevPtr_ == _TextQ;

    if (Reset_)
    {
      for (PrevPtr_ = (TextNodeData*)(_TextQhead->Prev());
           PrevPtr_ && (_TextQhead = PrevPtr_);
           PrevPtr_ = (TextNodeData*)(_TextQhead->Prev()));
      NextPtr_ = _TextQhead;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (TextNodeData*)(NextPtr_->Next());
      TailEnd_ = PrevPtr_ == _TextQ;
    }
    while (PrevPtr_ && PrevPtr_->Erased() && !TailEnd_);

    _TextQhead = PrevPtr_;

    if (_TextQhead->Erased() && TailEnd_)
    {
      for (PrevPtr_ = (TextNodeData*)(_TextQhead->Prev());
           PrevPtr_ && (_TextQhead = PrevPtr_);
           PrevPtr_ = (TextNodeData*)(_TextQhead->Prev()));

      return (_TextQhead->Erased() ? NULL:_TextQhead);
    }
  }
  else if (_TextQhead && Reset_)
  {
    for (PrevPtr_ = (TextNodeData*)(_TextQhead->Prev());
         PrevPtr_ && (_TextQhead = PrevPtr_);
         PrevPtr_ = (TextNodeData*)(_TextQhead->Prev()));
  }

  return _TextQhead;
}

/****************************************************************************/
TextNodeData* AppTextNode::TailTextNode(bool Reset_, bool NotErased_) const
{
  TextNodeData* PrevPtr_ = _TextQ;
  TextNodeData* NextPtr_ = _TextQ;

  if (_TextQ && NotErased_)
  {
    bool HeadEnd_ = PrevPtr_ == _TextQhead;

    if (Reset_)
    {
      for (NextPtr_ = (TextNodeData*)(_TextQ->Next());
           NextPtr_ && (_TextQ = NextPtr_);
           NextPtr_ = (TextNodeData*)(_TextQ->Next()));
      NextPtr_ = _TextQ;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (TextNodeData*)(NextPtr_->Prev());
      HeadEnd_ = PrevPtr_ == _TextQhead;
    }
    while (PrevPtr_ && PrevPtr_->Erased() && !HeadEnd_);

    _TextQ = PrevPtr_;

    if (_TextQ->Erased() && HeadEnd_)
    {
      for (NextPtr_ = (TextNodeData*)(_TextQ->Next());
           NextPtr_ && (_TextQ = NextPtr_);
           NextPtr_ = (TextNodeData*)(_TextQ->Next()));

      return (_TextQ->Erased() ? NULL:_TextQ);
    }
  }
  else if (_TextQ && Reset_)
  {
    for (NextPtr_ = (TextNodeData*)(_TextQ->Next());
         NextPtr_ && (_TextQ = NextPtr_);
         NextPtr_ = (TextNodeData*)(_TextQ->Next()));
  }

  return _TextQ;
}

/****************************************************************************/
TextNodeData* AppTextNode::SetTextNodePtr(bool Set_) const
{
  _TextQptrSet = Set_;
  return _TextQptr;
}

/****************************************************************************/
void AppTextNode::ConfirmTextNodeErased()
{
  TextNodeData* More_ = NULL;
  NextTextNode(true, false);
  SetTextNodePtr(true);
  TextNodeData* NextPtr_ = _TextQhead;

  if (NextPtr_)
  {
    _TextQptr = NextPtr_;

    do
    {
      if (More_)
      {
        More_->Unlink();
        More_ = NULL;
      }

      More_ = PopTextNode(false, true);
    }
    while (More_ && _TextQptr);
  }

  NextTextNode(true, false);
  SetTextNodePtr(false);
}

/****************************************************************************/
/****************************************************************************/
void AppTextNode::UnsetData(bool Head_, size_t Index_)
{
  if (_TextQ && Index_ < _NumTextData)
  {
    size_t x;
    size_t max = Index_;
    TextNodeData* Ptr_ = Head_ ? _TextQhead:_TextQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (TextNodeData*)(Ptr_->Prev());

    if (Ptr_)
      Ptr_->UnsetData();
  }
}

/****************************************************************************/
bool AppTextNode::DataSet(bool Head_, size_t Index_) const
{
  if (_TextQ && Index_ < _NumTextData)
  {
    size_t x;
    size_t max = Index_;
    TextNodeData* Ptr_ = Head_ ? _TextQhead:_TextQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (TextNodeData*)(Ptr_->Prev());

    return (Ptr_ ? Ptr_->DataSet():false);
  }

  return false;
}

/****************************************************************************/
ChrString* AppTextNode::NodeTextStr(bool Head_, size_t Index_)
{
  if (_TextQ && Index_ < _NumTextData)
  {
    size_t x;
    size_t max = Index_;
    TextNodeData* Ptr_ = Head_ ? _TextQhead:_TextQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (TextNodeData*)(Ptr_->Prev());

    bool HaveData_ = Ptr_ && Ptr_->DataSet() &&
                     Ptr_->DataType() == QListNode::STRING_DATA;
    return (HaveData_ ? Ptr_->GiveTextStr():NULL);
  }

  return NULL;
}

/****************************************************************************/
StackNodeInfo* AppTextNode::NodeNestedNode(bool Head_, size_t Index_)
{
  if (_TextQ && Index_ < _NumTextData)
  {
    size_t x;
    size_t max = Index_;
    TextNodeData* Ptr_ = Head_ ? _TextQhead:_TextQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (TextNodeData*)(Ptr_->Prev());

    bool HaveData_ = Ptr_ && Ptr_->DataSet() &&
                     Ptr_->DataType() == QListNode::INFONODE_DATA;
    return (HaveData_ ? Ptr_->GiveNestedNode():NULL);
  }

  return NULL;
}

/****************************************************************************/
int AppTextNode::NodeDataType(bool Head_, size_t Index_)
{
  if (_TextQ && Index_ < _NumTextData)
  {
    size_t x;
    size_t max = Index_;
    TextNodeData* Ptr_ = Head_ ? _TextQhead:_TextQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (TextNodeData*)(Ptr_->Prev());

    return (Ptr_ ? Ptr_->DataType():0);
  }

  return 0;
}

/****************************************************************************/
TextNodeData* AppTextNode::GiveTextNode(bool Head_, size_t Index_)
{
  if (_TextQ && Index_ < _NumTextData)
  {
    size_t x;
    size_t max = Index_;
    TextNodeData* Ptr_ = Head_ ? _TextQhead:_TextQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (TextNodeData*)(Ptr_->Prev());

    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
const TextNodeData* AppTextNode::TextNode(bool Head_, size_t Index_) const
{
  if (_TextQ && Index_ < _NumTextData)
  {
    size_t x;
    size_t max = Index_;
    TextNodeData* Ptr_ = Head_ ? _TextQhead:_TextQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (TextNodeData*)(Ptr_->Prev());

    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
void AppTextNode::ShowTextNodeContentsChkPtr(const StackNodeInfo* ptr_, ostream& os_,
                                             size_t fw_, bool ShowErased_) const
{
  if (_TextQ && TagTypeInfo::ShowDebugDump() && ((!Erased() && !DoErase()) ||
      ShowErased_ || TagTypeInfo::ShowErasedNodes()))
  {
    size_t x;
    size_t max = _NumTextData;
    TextNodeData* Prev_ = NULL;
    TextNodeData* Ptr_ = _TextQ;

    if (Erased() || DoErase())
      os_ <<"==> *** Erased ***" <<endl;

    for (x = 0; x < max && Ptr_ && Ptr_ != Prev_; x++)
    {
      Prev_ = Ptr_;
      Ptr_->ShowContents(ptr_, os_, fw_, &x);
      Ptr_ = (TextNodeData*)(Ptr_->Prev());
    }
  }
}

/****************************************************************************/
void AppTextNode::ShowTextNodeContents(ostream& os_, size_t fw_, bool ShowErased_) const
{
  if (_TextQ && TagTypeInfo::ShowDebugDump() && ((!Erased() && !DoErase()) ||
      ShowErased_ || TagTypeInfo::ShowErasedNodes()))
  {
    size_t x;
    size_t max = _NumTextData;
    TextNodeData* Prev_ = NULL;
    TextNodeData* Ptr_ = _TextQ;

    if (Erased() || DoErase())
      os_ <<"==> *** Erased ***" <<endl;

    for (x = 0; x < max && Ptr_ && Ptr_ != Prev_; x++)
    {
      Prev_ = Ptr_;
      Ptr_->ShowContents(_RefNode, os_, fw_, &x);
      Ptr_ = (TextNodeData*)(Ptr_->Prev());
    }
  }
}

/****************************************************************************/
TextNodeData* AppTextNode::CreateTextNodeData(ChrString* TextStr_,
                                              StackNodeInfo* NestedNode_,
                                              int DataType_)
{
  TextNodeData* TextNodePtr_ = new TextNodeData;

  if (TextStr_ && DataType_ == QListNode::STRING_DATA)
  {
    TextNodePtr_->SetDataType(DataType_);
    TextNodePtr_->SetTextStr(*TextStr_);
  }

  if (NestedNode_ && DataType_ == QListNode::INFONODE_DATA)
  {
    TextNodePtr_->SetDataType(DataType_);
    TextNodePtr_->SetNestedNode(NestedNode_);
  }

  PushTextNode(TextNodePtr_, false);
  _TextQptr = _TextQhead;

  return _TextQhead;
}

/****************************************************************************/
/****************************************************************************/
void AppTextNode::SetCTextStr(const char* Str_)
{
  if (Str_ && _TextQhead)
  {
    if (!_RefNode && _Transferred && _TransferNode)
      _TransferNode->SetCTextStr(Str_);
    else
    {
      if (_TextQhead->DataType() == QListNode::INFONODE_DATA ||
          _TextQhead->DataSet())
        CreateTextNodeData(NULL, NULL, QListNode::STRING_DATA);
      else
        _TextQhead->SetDataType(QListNode::STRING_DATA);

      _TextQhead->SetCTextStr(Str_);
    }
  }
}

/****************************************************************************/
void AppTextNode::SetTextStr(const ChrString* StrPtr_)
{
  if (StrPtr_ && _TextQhead)
  {
    if (!_RefNode && _Transferred && _TransferNode)
      _TransferNode->SetTextStr(StrPtr_);
    else
    {
      if (_TextQhead->DataType() == QListNode::INFONODE_DATA ||
          _TextQhead->DataSet())
        CreateTextNodeData(NULL, NULL, QListNode::STRING_DATA);
      else
        _TextQhead->SetDataType(QListNode::STRING_DATA);

      _TextQhead->SetTextStr(*StrPtr_);
    }
  }
}

/****************************************************************************/
void AppTextNode::AppendCTextStr(const char* Str_)
{
  if (Str_ && _TextQhead)
  {
    if (!_RefNode && _Transferred && _TransferNode)
      _TransferNode->AppendCTextStr(Str_);
    else
    {
      if (_TextQhead->DataType() == QListNode::INFONODE_DATA)
        CreateTextNodeData(NULL, NULL, QListNode::STRING_DATA);
      else
        _TextQhead->SetDataType(QListNode::STRING_DATA);

      _TextQhead->AppendCTextStr(Str_);
    }
  }
}

/****************************************************************************/
void AppTextNode::AppendTextStr(const ChrString* StrPtr_)
{
  if (StrPtr_ && _TextQhead)
  {
    if (!_RefNode && _Transferred && _TransferNode)
      _TransferNode->AppendTextStr(StrPtr_);
    else
    {
      if (_TextQhead->DataType() == QListNode::INFONODE_DATA)
        CreateTextNodeData(NULL, NULL, QListNode::STRING_DATA);
      else
        _TextQhead->SetDataType(QListNode::STRING_DATA);

      _TextQhead->AppendTextStr(*StrPtr_);
    }
  }
}

/****************************************************************************/
void AppTextNode::SetNestedNode(StackNodeInfo* NodePtr_, bool Transfer_)
{
  if (NodePtr_ && _TextQhead)
  {
    if (!_RefNode && _Transferred && _TransferNode)
      _TransferNode->SetNestedNode(NodePtr_);
    else
    {
      if (_TextQhead->DataType() == QListNode::STRING_DATA ||
          _TextQhead->DataSet())
        CreateTextNodeData(NULL, NULL, QListNode::INFONODE_DATA);
      else
        _TextQhead->SetDataType(QListNode::INFONODE_DATA);

      _TextQhead->SetNestedNode(NodePtr_);

      if (!Transfer_)
      {
        if (_PauseAppend == NEED_STDTAG)
          _PauseAppend = STDTAG_APPENDED;
        else if (_PauseAppend == NEED_ENDTAG)
          _PauseAppend = ENDTAG_APPENDED;
      }
    }
  }
}

/****************************************************************************/
void AppTextNode::AppendNestedNode(StackNodeInfo* NodePtr_, bool Shifted_)
{
  if (NodePtr_ && _TextQhead)
  {
    if (!_RefNode && _Transferred && _TransferNode)
      _TransferNode->AppendNestedNode(NodePtr_);
    else
    {
      ChrString TempStr_;
      bool DataSet_ = false;
      bool Appended_ = false;
      bool OmitTag_ = Shifted_ &&
                      (NodePtr_->IsContentShifted() ||
                       NodePtr_->IsContentAppended());

      if (OmitTag_ && NodePtr_->IsContentTransferred())
        return;

      if (!OmitTag_)
      {
        TempStr_ = NodePtr_->GiveTagStr();
        Appended_ = !TempStr_.IsEmpty();

        if (Appended_)
        {
          if (!DataSet_)
          {
            if (_TextQhead->DataType() == QListNode::INFONODE_DATA)
              CreateTextNodeData(NULL, NULL, QListNode::STRING_DATA);
            else
              _TextQhead->SetDataType(QListNode::STRING_DATA);

            DataSet_ = true;
          }

          _TextQhead->AppendTextStr(TempStr_);
        }
      }

      TempStr_ = NodePtr_->GiveTagContent();
      Appended_ = Appended_ || !TempStr_.IsEmpty();

      if (Appended_)
      {
        if (!DataSet_)
        {
          if (_TextQhead->DataType() == QListNode::INFONODE_DATA)
            CreateTextNodeData(NULL, NULL, QListNode::STRING_DATA);
          else
            _TextQhead->SetDataType(QListNode::STRING_DATA);

          DataSet_ = true;
        }

        _TextQhead->AppendTextStr(TempStr_);
        NodePtr_->SetContentTransferred();

        if (_RefNode)
          _RefNode->SetContentReceived();
      }

      if (_RefNode && _RefNode->_Parent)
        DumpThisTextNode(*_RefNode->GiveOutput(),
                         _RefNode->GiveFieldWidth(),
                         "AppTextNode::AppendNestedNode");
    }
  }
}

/****************************************************************************/
StackNodeInfo* AppTextNode::SetRefNode(StackNodeInfo* RefNode_)
{
  _RefNode = RefNode_;
  _SglNode = RefNode_ ? (RefNode_->SingletonTagFound() ||
                         RefNode_->EmptyTagFound()):false;
  _StdNode = RefNode_ ? RefNode_->StdTagFound():false;

  return _RefNode;
}

/****************************************************************************/
StackNodeInfo* AppTextNode::GiveRefNode()
{
  return _RefNode;
}

/****************************************************************************/
const StackNodeInfo* AppTextNode::RefNode() const
{
  return _RefNode;
}

/****************************************************************************/
const StackNodeInfo* AppTextNode::RefNodeForwardLink() const
{
  return (_RefNode ? _RefNode->_ForwardLink:NULL);
}

/****************************************************************************/
const StackNodeInfo* AppTextNode::RefNodeBackLink() const
{
  return (_RefNode ? _RefNode->_BackLink:NULL);
}

/****************************************************************************/
int AppTextNode::LinkType() const
{
  return (_RefNode ? _RefNode->_LinkType:0);
}

/****************************************************************************/
void AppTextNode::SetErased(bool Flag_)
{
  if (_RefNode)
  {
    _Erased = Flag_;

    if (_Erased)
      _NumErased++;
  }
}

/****************************************************************************/
void AppTextNode::SetCopied(bool Flag_)
{
  if (_RefNode)
    _Copied = Flag_;
}

/****************************************************************************/
int AppTextNode::ClassType() const
{
  return QListNode::APPTEXTNODE;
}

/****************************************************************************/
QListNode& AppTextNode::operator = (const QListNode& List_)
{
  if (this != &List_)
  {
    QListNode::operator = (List_);
    AppTextNode* ListPtr_ = (List_.ClassType() == APPTEXTNODE) ?
                                 ((AppTextNode*)(&List_)):NULL;

    if (!ListPtr_)
      return *this;

    if (_TextQhead)
    {
      TextNodeData::CopyTextNodeData(_TextQ, _TextQhead, _NumTextData,
                                     ListPtr_->_TextQ, ListPtr_->_TextQhead, ListPtr_->_NumTextData);
      _TextQptr = _TextQhead;
    }

    _Parent = ListPtr_->_Parent;
    _HasTagStk = ListPtr_->_HasTagStk;
    _RefNode = ListPtr_->_RefNode;
    SetPrev(NULL);
    SetNext(NULL);
    _SglNode = _RefNode ? (_RefNode->SingletonTagFound() ||
                           _RefNode->EmptyTagFound()):false;
    _StdNode = _RefNode ? _RefNode->StdTagFound():false;
    _AppendText = ListPtr_->_AppendText;
    _DoErase = ListPtr_->_DoErase;
    _Copied = true;
    _Erased = ListPtr_->_Erased;
    _TransferNode = NULL;
  }

  return *this;
}

/****************************************************************************/
void AppTextNode::SetParent(TagTypeInfo* Ptr_)
{
  if (Ptr_)
    _Parent = Ptr_;
}

/****************************************************************************/
void AppTextNode::SetDoErase(bool Flag_)
{
  if (!_Erased)
    _DoErase = Flag_;
}

/****************************************************************************/
void AppTextNode::ExecDoErase()
{
  if (!_Erased && _DoErase)
  {
    SetErased(true);
    _DoErase = false;
  }
}

/****************************************************************************/
AppTextNode* AppTextNode::AppTextPtr()
{
  return (_Parent ? _Parent->Sequencer()->_AppTextPtr:NULL);
}

/****************************************************************************/
void AppTextNode::SetAppendText()
{
  _AppendText = _HasTagStk &&
                _Parent->Sequencer()->_HeadAppText != NULL;
}

/****************************************************************************/
void AppTextNode::SetAppendText(bool Flag_)
{
  _AppendText = Flag_;
}

/****************************************************************************/
bool AppTextNode::Unlink()
{
  ExecDoErase();

  if (_Erased && !_Copied)
  {
    SetPrev(SetNext(NULL));
    _Parent = NULL;

    if (_TransferNode && _RefNode)
    {
      if (HasTentativeStr())
      {
        _Transferred = false;
        _RefNode->_TextNode = _TransferNode;
        _TransferNode->_RefNode = _RefNode;
        _RefNode = NULL;
        _AppendText = false;
        _TransferNode = NULL;
        SetNext(NULL);
      }
    }
    else
    {
      _RefNode->_TextNode = NULL;
      _RefNode = NULL;
      _TransferNode = NULL;
    }

    delete this;
    return true;
  }

  return false;
}

/****************************************************************************/
// 1st param indicates string termination : AppContStr=0
//
bool AppTextNode::PopAppContStr(bool Flag_,
                                StackNodeInfo* EncNode_, bool PostLink_)
{
  AppTextNode* NodePtr_ = NULL;
  bool Equal_;
  bool Equiv_;
  bool StopAppend_ = false;
  bool PrevAppend_ = _AppendText;

  bool NodeErased_ = false;
  bool WasErased_ = false;

  StackNodeInfo* sEncNode_ = EncNode_;
  EncNode_ = _Parent->Sequencer()->FindInfoNode(EncNode_);

  if (!EncNode_)
    EncNode_ = sEncNode_;

  if (_Parent->Sequencer()->_HeadAppText)
  {
    if (Flag_)
      _AppendText = false;
    else
    {
      _AppendText = _HasTagStk &&
                    _Parent->Sequencer()->_HeadAppText != NULL &&
                    _Parent->Sequencer()->_HeadTagStack != NULL;

      if (_AppendText && EncNode_ && _RefNode)
      {
        Equiv_ = _RefNode->EquivTag(_RefNode, EncNode_, &Equal_, StackNodeInfo::FORWARD_ORDER);
        StopAppend_ = _RefNode->IsLinked(EncNode_, true) ||
                      ((Equiv_ && !Equal_) && _RefNode->Preceeds(EncNode_, false));

        if (StopAppend_)
          _AppendText = false;
      }
    }

    if (!_AppendText && _Parent->Sequencer()->_HeadAppText != NULL)
    {
      if (PostLink_ && StopAppend_)
        _Parent->Sequencer()->PopAppContStr(true, false, _RefNode, NULL,
                                            &NodeErased_, &WasErased_);
      else
      {
        sEncNode_ = Flag_ ? _RefNode:NULL;
        _Parent->Sequencer()->PopAppContStr(true, false, NULL, sEncNode_,
                                            &NodeErased_, &WasErased_);
      }
    }

    if (PrevAppend_ != _AppendText && !_AppendText &&
        (!WasErased_ && _PauseAppend != TAG_COMPLETED))
    {
      _PauseAppend = TAG_COMPLETED;

      if (_RefNode && _RefNode->_Parent &&
          TagTypeInfo::ShowDebugDump())
        DumpThisTextNode(*_RefNode->GiveOutput(),
                         _RefNode->GiveFieldWidth(),
                         "AppTextNode::PopAppContStr");
    }
  }

  return NodeErased_;
}

/****************************************************************************/
void AppTextNode::SetTransferNode(AppTextNode* Node_)
{
  if (Node_)
  {
    // copied from transfer node
    _RefNode = Node_->_RefNode;
    _SglNode = _RefNode ? (_RefNode->SingletonTagFound() ||
                           _RefNode->EmptyTagFound()):false;
    _StdNode = _RefNode ? _RefNode->StdTagFound():false;
    _PauseAppend = Node_->_PauseAppend;

    // setup date members of linking node to transfer node
    _Transferred = false;
    _TransferNode = Node_;
    _RefNode = Node_->_RefNode;
    _RefNode->_TextNode = this;
    _TransferNode->_RefNode = NULL;
    SetAppendText();
  }
}

/****************************************************************************/
void AppTextNode::RemoveTentativeNode()
{
  // remove tentative node for previous optional tags where unmatched
  // endtag has been matched and processed
  //

  if (HasTentativeStr())
  {
    _Transferred = false;
    _RefNode->_TextNode = _TransferNode;
    _TransferNode->_RefNode = _RefNode;
    _RefNode = NULL;
    _AppendText = false;
    _TransferNode = NULL;
    SetNext(NULL);

    delete this;
  }
}

/****************************************************************************/
void AppTextNode::ResumeTextAppend(StackNodeInfo* Ptr_)
{
  // resume text appending for the current tag after new end tag
  // is encountered

  if (_AppendText && _PauseAppend == STDTAG_APPENDED && Ptr_)
  {
    if (_TextQhead->DataType() == QListNode::INFONODE_DATA ||
        _TextQhead->DataSet())
      CreateTextNodeData(NULL, NULL, QListNode::INFONODE_DATA);

    _PauseAppend = NEED_ENDTAG;
    SetNestedNode(Ptr_);

    if (_PauseAppend == ENDTAG_APPENDED)
    {
      UnpauseTextAppend();

      if (_RefNode && _RefNode->_Parent &&
          TagTypeInfo::ShowDebugDump())
        DumpThisTextNode(*_RefNode->GiveOutput(),
                         _RefNode->GiveFieldWidth(),
                         "AppTextNode::ResumeTextAppend");
    }
  }
}

/****************************************************************************/
void AppTextNode::PauseTextAppend(StackNodeInfo* Ptr_)
{
  // pause text appending for the current tag after new standard or
  // optional start tag is encountered

  if (_AppendText && _PauseAppend == TXTAPP_NOT_PAUSED && Ptr_)
  {
    if (_TextQhead->DataType() == QListNode::STRING_DATA ||
        _TextQhead->DataSet())
      CreateTextNodeData(NULL, NULL, QListNode::INFONODE_DATA);

    _PauseAppend = NEED_STDTAG;
    SetNestedNode(Ptr_);

    if (_RefNode && _RefNode->_Parent &&
        TagTypeInfo::ShowDebugDump())
      DumpThisTextNode(*_RefNode->GiveOutput(),
                       _RefNode->GiveFieldWidth(),
                       "AppTextNode::PauseTextAppend");
  }
}

/****************************************************************************/
void AppTextNode::TransferText()
{
  if (HasTentativeStr() && _TextQ)
  {
    size_t x;
    size_t max = _NumTextData;

    TailTextNode(true, true);
    _TextQptr = _TextQ;

    for (x = 0; x < max && _TextQptr; x++)
    {
      if (_TextQptr->DataType() == QListNode::STRING_DATA)
        _TransferNode->AppendTextStr(_TextQptr->GiveTextStr());
      else if (_TextQptr->DataType() == QListNode::INFONODE_DATA)
        _TransferNode->SetNestedNode(_TextQptr->GiveNestedNode(), true);

      _TextQptr = (TextNodeData*)(_TextQptr->Prev());
    }

    _TransferNode->_PauseAppend = _PauseAppend;

    if (_PauseAppend == ENDTAG_APPENDED)
      _TransferNode->UnpauseTextAppend();

    _Transferred = true;
    _RefNode->_TextNode = _TransferNode;
    _TransferNode->_RefNode = _RefNode;
    _RefNode = NULL;
    _AppendText = false;
  }
}

/****************************************************************************/
AppTextNode* AppTextNode::FindTagNode(StackNodeInfo* Node_)
{
  AppTextNode* AppTextPtr_ = NULL;
  AppTextNode* HomeNode_ = this;

  if (Node_ && Node_->AppText())
  {
    for (AppTextPtr_ = Node_->AppText();
         HomeNode_ &&
         (HomeNode_ != AppTextPtr_ ||
          HomeNode_->_RefNode != AppTextPtr_->_RefNode);
         HomeNode_ = HomeNode_->_Next);

    return HomeNode_;
  }

  return AppTextPtr_;
}

/****************************************************************************/
QListNode* AppTextNode::GiveNext()
{
  return _Next;
}

/****************************************************************************/
const QListNode* AppTextNode::Next() const
{
  return _Next;
}

/****************************************************************************/
QListNode* AppTextNode::SetNext(QListNode* Ptr_)
{
  if (!Ptr_ || Ptr_->ClassType() == QListNode::APPTEXTNODE)
  {
    _Next = (AppTextNode*)Ptr_;
    _NextBase = _Next;
    return _Next;
  }

  _NextBase = Ptr_;
  return _NextBase;
}

/****************************************************************************/
QListNode* AppTextNode::GivePrev()
{
  return _Prev;
}

/****************************************************************************/
const QListNode* AppTextNode::Prev() const
{
  return _Prev;
}

/****************************************************************************/
QListNode* AppTextNode::SetPrev(QListNode* Ptr_)
{
  if (!Ptr_ || Ptr_->ClassType() == QListNode::APPTEXTNODE)
  {
    _Prev = (AppTextNode*)Ptr_;
    _PrevBase = _Prev;
    return _Prev;
  }

  _PrevBase = Ptr_;
  return _PrevBase;
}

/****************************************************************************/
void AppTextNode::DumpThisTextNode(ostream& os_, size_t fw_, const char* Str_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  ChrString RefTag_ = _RefNode->GiveTagStr();
  ChrString TitleStr_("*** ");
  TitleStr_ += ChrString(Str_);
  TitleStr_ += ChrString(": Start ***");

  os_ <<TitleStr_ <<endl;

  if (!Erased() && !DoErase())
    fprintf(stdout, "this=0x%p ref=0x%p reftag=%s prev=0x%p next=0x%p\n",
            this, _RefNode, RefTag_.c_str(), Prev(), Next());
  else
  {
    fprintf(stdout, "this=0x%p ref=0x%p reftag=%s prev=0x%p next=0x%p : ",
            this, _RefNode, RefTag_.c_str(), Prev(), Next());
    fprintf(stdout, "*** Erased ***\n");
  }

  os_.width(fw_); os_ <<std::left <<"*NumTextData: " <<_NumTextData <<endl;
  os_.width(fw_); os_ <<std::left <<"Append Text : " <<_AppendText <<endl;
  os_.width(fw_); os_ <<std::left <<"Pause Append: " <<PauseAppendTypeToStr(_PauseAppend)
                                                     <<endl;

  ShowTextNodeContents(os_, fw_, true);

  TitleStr_ = ChrString("*** ");
  TitleStr_ += ChrString(Str_);
  TitleStr_ += ChrString(": End ***");

  os_ <<TitleStr_ <<endl <<endl;
}

/****************************************************************************/
void AppTextNode::DumpAppTextNode(const StackNodeInfo* ptr_, ostream& os_, size_t fw_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  os_ <<"*** Start : StackNodeInfo->AppTextNode Dump ***" <<endl;

  os_.width(fw_); os_ <<std::left <<"Has Tag Stk : " <<_HasTagStk <<endl;

  if (_RefNode)
  {
    os_.width(fw_); os_ <<std::left <<"Ref Node  :" <<endl
                                    <<"\tString: "  <<_RefNode->GiveTagStr()
                                    <<"\tAddr: " <<_RefNode <<endl;
  }
  else
  {
    os_.width(fw_); os_ <<std::left <<"Ref Node  : NULL" <<endl;
  }

  os_.width(fw_); os_ <<std::left <<"SglNode     : " <<(_SglNode ? "true":"false") <<endl;
  os_.width(fw_); os_ <<std::left <<"StdNode     : " <<(_StdNode ? "true":"false") <<endl;
  os_.width(fw_); os_ <<std::left <<"Append Text : " <<_AppendText <<endl;
  os_.width(fw_); os_ <<std::left <<"DoErase     : " <<_DoErase <<endl;
  os_.width(fw_); os_ <<std::left <<"Pause Append: " <<PauseAppendTypeToStr(_PauseAppend)
                                                     <<endl;
  os_.width(fw_); os_ <<std::left <<"Transferred : " <<_Transferred <<endl;
  os_.width(fw_); os_ <<std::left <<"Copied      : " <<_Copied <<endl;
  os_.width(fw_); os_ <<std::left <<"Erased      : " <<_Erased <<endl;

  if (_TextQ)
  {
    os_.width(fw_); os_ <<std::left <<"*NumTextData : " <<_NumTextData <<endl;
    ShowTextNodeContentsChkPtr(ptr_, os_, fw_);
  }
  else
  {
    os_.width(fw_); os_ <<std::left <<"(No TextNode Data)" <<endl;
  }

  // AppTextNode Object data
  os_.width(fw_); os_ <<std::left <<"AppTextNode ObjectID: " <<_NodeID <<endl;
  os_ <<"*** End : StackNodeInfo->AppTextNode Dump ***" <<endl <<endl;
}

/****************************************************************************/
MEMORYOPS_DEFN(AppTextNode)

/****************************************************************************/
// NodeNestingData class definition
/****************************************************************************/
NodeNestingData::NodeNestingData():
QListNode(),

_ThisTag(NULL),
_StartTag(NULL),
_EndTag(NULL),
_TagParent(NULL)
{}

/****************************************************************************/
NodeNestingData::NodeNestingData(const NodeNestingData& Obj_):
QListNode(Obj_),

_ThisTag(Obj_._ThisTag),
_StartTag(Obj_._StartTag),
_EndTag(Obj_._EndTag),
_TagParent(Obj_._TagParent)
{}

/****************************************************************************/
NodeNestingData::~NodeNestingData()
{
  _ThisTag = NULL;
  _StartTag = NULL;
  _EndTag = NULL;
  _TagParent = NULL;
}

/****************************************************************************/
NodeNestingData* NodeNestingData::MakeNode()
{
  return (new NodeNestingData);
}

/****************************************************************************/
NodeNestingData* NodeNestingData::MakeNode(const NodeNestingData& Obj_)
{
  return (new NodeNestingData(Obj_));
}

/****************************************************************************/
NodeNestingData* NodeNestingData::CopyNode(NodeNestingData* TrgData_, NodeNestingData* SrcData_)
{
  if (SrcData_)
  {
    NodeNestingData* Node_ = TrgData_ ? TrgData_:MakeNode();

    Node_->_Erased = SrcData_->_Erased;
    Node_->_ThisTag = SrcData_->_ThisTag;
    Node_->_StartTag = SrcData_->_StartTag;
    Node_->_EndTag = SrcData_->_EndTag;
    Node_->_TagParent = SrcData_->_TagParent;

    return Node_;
  }

  return NULL;
}

/****************************************************************************/
NodeNestingData* NodeNestingData::CopyNodeNestingData(NodeNestingData*& TrgData_,
                                                      NodeNestingData*& TrgDataHead_,
                                                      size_t& TrgNumData_,
                                                      NodeNestingData* SrcData_,
                                                      NodeNestingData* SrcDataHead_,
                                                      size_t SrcNumData_)
{
  NodeNestingData* tempPtr_;
  bool NodeCreated_ = false;
  size_t x = TrgNumData_;
  size_t y = SrcNumData_;
  size_t z = 0;

  if (!TrgData_ || !SrcData_ ||
      !TrgDataHead_ || !SrcDataHead_ ||
      !TrgNumData_ || !SrcNumData_)
  {
    if (!SrcData_ || !SrcDataHead_ || !SrcNumData_)
    {
      if (TrgData_ && TrgDataHead_ && 
          TrgData_->ClassType() == QListNode::NODENESTINGDATA &&
          TrgDataHead_->ClassType() == QListNode::NODENESTINGDATA)
      {
        TrgData_->DeleteAll();
        TrgDataHead_ =
        TrgData_ = NULL;
        TrgNumData_ = 0;
      }

      return NULL;
    }
    else if (!TrgData_ || !TrgDataHead_ || !TrgNumData_)
    {
      if (SrcData_ && SrcDataHead_ &&
          SrcData_->ClassType() == QListNode::NODENESTINGDATA &&
          SrcDataHead_->ClassType() == QListNode::NODENESTINGDATA)
      {
        TrgDataHead_ =
        TrgData_ = CopyNode(NULL, SrcData_);
        x = TrgNumData_ = 1;
        SrcData_ = (NodeNestingData*)(SrcData_->Prev());
      }

      if (!SrcData_)
        return TrgDataHead_;
      else
        NodeCreated_ = true;
    }
  }

  if (!(SrcData_ && SrcDataHead_ &&
        SrcData_->ClassType() == QListNode::NODENESTINGDATA &&
        SrcDataHead_->ClassType() == QListNode::NODENESTINGDATA) ||
      !(TrgData_ && TrgDataHead_ &&
        TrgData_->ClassType() == QListNode::NODENESTINGDATA &&
        TrgDataHead_->ClassType() == QListNode::NODENESTINGDATA))
    return NULL;

  if (x < y)
  {
    if (!NodeCreated_)
    {
      z = x;
      x = 0;

      for (tempPtr_ = TrgData_; x < z && SrcData_; x++)
      {
        tempPtr_ = CopyNode(tempPtr_, SrcData_);
        tempPtr_ = (NodeNestingData*)(tempPtr_->Prev());
        SrcData_ = (NodeNestingData*)(SrcData_->Prev());
      }
    }

    while (x < y && SrcData_)
    {
      TrgDataHead_ = (NodeNestingData*)(TrgDataHead_->AppendHead(CopyNode(NULL, SrcData_), false));
      SrcData_ = (NodeNestingData*)(SrcData_->Prev());
      x++;
    }

    TrgNumData_ = x;
  }
  else if (x >= y)
  {
    while (x > y && TrgDataHead_)
    {
      tempPtr_ = (NodeNestingData*)TrgDataHead_;
      TrgDataHead_ = (NodeNestingData*)(TrgDataHead_->Next());
      tempPtr_->Unlink();
      x--;
    }

    z = x;
    x = 0;

    for (tempPtr_ = TrgData_; x < z && SrcData_; x++)
    {
      tempPtr_ = CopyNode(tempPtr_, SrcData_);
      tempPtr_ = (NodeNestingData*)(tempPtr_->Prev());
      SrcData_ = (NodeNestingData*)(SrcData_->Prev());
    }

    TrgNumData_ = x;
  }

  return TrgDataHead_;
}

/****************************************************************************/
int NodeNestingData::ClassType() const
{
  return QListNode::NODENESTINGDATA;
}

/****************************************************************************/
NodeNestingData& NodeNestingData::operator = (const QListNode& List_)
{
  if (this != &List_)
  {
    QListNode::operator = (List_);
    NodeNestingData* ListPtr_ = (List_.ClassType() == NODENESTINGDATA) ?
                                    ((NodeNestingData*)(&List_)):NULL;

    if (!ListPtr_)
      return *this;

    _Erased = ListPtr_->_Erased;
    _ThisTag = ListPtr_->_ThisTag;
    _StartTag = ListPtr_->_StartTag;
    _EndTag = ListPtr_->_EndTag;
    _TagParent = ListPtr_->_TagParent;
  }

  return *this;
}

/****************************************************************************/
void NodeNestingData::SetThisTag(StackNodeInfo* Ptr_)
{
  _ThisTag = Ptr_;
}

/****************************************************************************/
void NodeNestingData::SetStartTag(StackNodeInfo* Ptr_)
{
  _StartTag = Ptr_;
}

/****************************************************************************/
void NodeNestingData::SetEndTag(StackNodeInfo* Ptr_)
{
  _EndTag = Ptr_;
}

/****************************************************************************/
void NodeNestingData::SetTagParent(StackNodeInfo* Ptr_)
{
  _TagParent = Ptr_;
}

/****************************************************************************/
void NodeNestingData::ShowContents(const StackNodeInfo* ptr_, ostream& os_, size_t fw_) const
{
  StackNodeInfo* Ptr_ = NULL;
  bool Valid_ = false;

  if (!TagTypeInfo::ShowDebugDump())
    return;

  os_.width(fw_); os_ <<std::left <<"NodeID: " <<_NodeID <<endl;

  if (_ThisTag)
  {
    os_.width(fw_); os_ <<std::left <<"- ThisTag: ";
    Ptr_ = _ThisTag;

    Valid_ = (Ptr_ && !Ptr_->Erased()) &&
             (Ptr_->_StartBrkFound && Ptr_->_StartBrkStr.strlen());

    if (Valid_)
      StackNodeInfo::ShowTagStr(os_, 0, _ThisTag, StackNodeInfo::PREVNEXT, NULL, true);
    else
      os_ <<endl;
  }
  else
  {
    os_.width(fw_); os_ <<std::left <<"- ThisTag: NULL" <<endl;
  }

  if (_StartTag)
  {
    os_.width(fw_); os_ <<std::left <<"- StartTag: ";
    Ptr_ = _StartTag;

    Valid_ = (Ptr_ && !Ptr_->Erased()) &&
             (Ptr_->_StartBrkFound && Ptr_->_StartBrkStr.strlen());

    if (Valid_)
      StackNodeInfo::ShowTagStr(os_, 0, _StartTag, StackNodeInfo::PREVNEXT, NULL, true);
    else
      os_ <<endl;
  }
  else
  {
    os_.width(fw_); os_ <<std::left <<"- StartTag: NULL" <<endl;
  }

  if (_EndTag)
  {
    os_.width(fw_); os_ <<std::left <<"- EndTag: ";
    Ptr_ = _EndTag;

    Valid_ = (Ptr_ && !Ptr_->Erased()) &&
             (Ptr_->_StartBrkFound && Ptr_->_StartBrkStr.strlen());

    if (Valid_)
      StackNodeInfo::ShowTagStr(os_, 0, _EndTag, StackNodeInfo::PREVNEXT, NULL, true);
    else
      os_ <<endl;
  }
  else
  {
    os_.width(fw_); os_ <<std::left <<"- EndTag: NULL" <<endl;
  }

  if (_TagParent)
  {
    os_.width(fw_); os_ <<std::left <<"- TagParent: ";
    Ptr_ = _TagParent;

    Valid_ = (Ptr_ && !Ptr_->Erased()) &&
             (Ptr_->_StartBrkFound && Ptr_->_StartBrkStr.strlen());

    if (Valid_)
      StackNodeInfo::ShowTagStr(os_, 0, _TagParent, StackNodeInfo::PREVNEXT, NULL, true);
    else
      os_ <<endl;
  }
  else
  {
    os_.width(fw_); os_ <<std::left <<"- TagParent: NULL" <<endl;
  }

  if (_TagParent)
  {
    os_.width(fw_); os_ <<endl <<std::left <<"** TagParent: Start Dump **" <<endl;
    _TagParent->ShowNodeNestingContents(ptr_, os_, fw_);

    if (_TagParent->_TextNode)
    {
      os_.width(fw_); os_ <<endl <<std::left <<"** AppTextNode Data:" <<endl;
      _TagParent->DumpAppTextNodeChkPtr(ptr_, os_, fw_);
    }

    os_.width(fw_); os_ <<std::left <<"** TagParent: End Dump **" <<endl;
  }
}

/****************************************************************************/
MEMORYOPS_DEFN(NodeNestingData)

/****************************************************************************/
// StackNodeFinder class definition
/****************************************************************************/
StackNodeFinder::StackNodeFinder(StackNodeInfo* Ptr_):
_StkNodePtr(Ptr_),
_NestedQ(Ptr_ ? Ptr_->NestedQtail():NULL),
_NumNestedData(Ptr_ ? Ptr_->NestedQsize():0)
{}

/****************************************************************************/
StackNodeFinder::StackNodeFinder(const StackNodeInfo* cPtr_):
_ConstStkNodePtr(cPtr_),
_ConstNestedQ(cPtr_ ? cPtr_->NestedQtail():NULL),
_NumNestedData(cPtr_ ? cPtr_->NestedQsize():0)
{}

/****************************************************************************/
NodeNestingData* StackNodeFinder::FindNestingNode_MatchingPtr(int TagType_, StackNodeInfo* ptr,
                                                            StackNodeInfo* tpptr, bool* Found_)
{
  if (_NestedQ && _NumNestedData)
  {
    size_t x;
    size_t max = _NumNestedData;
    bool fnd = false;

    NodeNestingData* Ptr_ = _NestedQ;
    StackNodeInfo* TagAddr[4];
    int ti = THIS_TAG-1;
    int i = (THIS_TAG <= TagType_ && TagType_ <= PARENT_TAG) ?
                (TagType_-1):-1;

    if (i >= 0)
      for (x = 0; x < max && Ptr_ && !fnd; x++)
      {
        TagAddr[0] = Ptr_->GiveThisTag();
        TagAddr[1] = Ptr_->GiveStartTag();
        TagAddr[2] = Ptr_->GiveEndTag();
        TagAddr[3] = Ptr_->GiveTagParent();

        fnd = TagAddr[i] == ptr && (!tpptr || TagAddr[ti] == tpptr);
        if (!fnd)
          Ptr_ = (NodeNestingData*)(Ptr_->Prev());
      }

    if (Found_)
      *Found_ = fnd;

    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
const NodeNestingData* StackNodeFinder::FindNestingNode_MatchingPtr(int TagType_, const StackNodeInfo* ptr,
                                                                  const StackNodeInfo* tpptr, bool* Found_) const
{
  if (_ConstNestedQ && _NumNestedData)
  {
    size_t x;
    size_t max = _NumNestedData;
    bool fnd = false;

    NodeNestingData* Ptr_ = _NestedQ;
    StackNodeInfo* TagAddr[4];
    int ti = THIS_TAG-1;
    int i = (THIS_TAG <= TagType_ && TagType_ <= PARENT_TAG) ?
                (TagType_-1):-1;

    if (i >= 0)
      for (x = 0; x < max && Ptr_ && !fnd; x++)
      {
        TagAddr[0] = Ptr_->GiveThisTag();
        TagAddr[1] = Ptr_->GiveStartTag();
        TagAddr[2] = Ptr_->GiveEndTag();
        TagAddr[3] = Ptr_->GiveTagParent();

        fnd = TagAddr[i] == ptr && (!tpptr || TagAddr[ti] == tpptr);
        if (!fnd)
          Ptr_ = (NodeNestingData*)(Ptr_->Prev());
      }

    if (Found_)
      *Found_ = fnd;

    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
NodeNestingData* StackNodeFinder::FindNestingNode_TagType(int TagType_, bool* Found_)
{
  return FindNestingNode_MatchingPtr(TagType_, _StkNodePtr, NULL, Found_);
}

/****************************************************************************/
const NodeNestingData* StackNodeFinder::FindNestingNode_TagType(int TagType_, bool* Found_) const
{
  return FindNestingNode_MatchingPtr(TagType_, _ConstStkNodePtr, NULL, Found_);
}

/****************************************************************************/
NodeNestingData* StackNodeFinder::FindNextNode_MatchingPtr(NodeNestingData* ptr, int TagType_, StackNodeInfo* snptr,
                                                         StackNodeInfo* tpptr, bool* Found_)
{
  if (ptr == NULL)
    return FindNestingNode_TagType(TagType_, Found_);

  if (_NestedQ && _NumNestedData)
  {
    size_t x;
    size_t max = _NumNestedData;
    bool fnd = false;

    NodeNestingData* Ptr_ = _NestedQ;
    StackNodeInfo* TagAddr[4];
    int ti = THIS_TAG-1;
    int i = (THIS_TAG <= TagType_ && TagType_ <= PARENT_TAG) ?
                (TagType_-1):-1;

    if (i >= 0)
    {
      for (x = 0; x < max && Ptr_ && !fnd; x++)
      {
        Ptr_ = (NodeNestingData*)(Ptr_->Prev());
        fnd = Ptr_ == ptr;
      }

      if (fnd)
      {
        for (fnd = false; x < max && Ptr_ && !fnd; x++)
        {
          Ptr_ = (NodeNestingData*)(Ptr_->Prev());
          TagAddr[0] = Ptr_->GiveThisTag();
          TagAddr[1] = Ptr_->GiveStartTag();
          TagAddr[2] = Ptr_->GiveEndTag();
          TagAddr[3] = Ptr_->GiveTagParent();

          fnd = TagAddr[i] == snptr && (!tpptr || TagAddr[ti] == tpptr);
        }
      }
    }

    if (Found_)
      *Found_ = fnd;

    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
const NodeNestingData* StackNodeFinder::FindNextNode_MatchingPtr(NodeNestingData* ptr, int TagType_, const StackNodeInfo* snptr,
                                                               const StackNodeInfo* tpptr, bool* Found_) const
{
  if (ptr == NULL)
    return FindNestingNode_TagType(TagType_, Found_);

  if (_ConstNestedQ && _NumNestedData)
  {
    size_t x;
    size_t max = _NumNestedData;
    bool fnd = false;

    NodeNestingData* Ptr_ = _NestedQ;
    StackNodeInfo* TagAddr[4];
    int ti = THIS_TAG-1;
    int i = (THIS_TAG <= TagType_ && TagType_ <= PARENT_TAG) ?
                (TagType_-1):-1;

    if (i >= 0)
    {
      for (x = 0; x < max && Ptr_ && !fnd; x++)
      {
        Ptr_ = (NodeNestingData*)(Ptr_->Prev());
        fnd = Ptr_ == ptr;
      }

      if (fnd)
      {
        for (fnd = false; x < max && Ptr_ && !fnd; x++)
        {
          Ptr_ = (NodeNestingData*)(Ptr_->Prev());
          TagAddr[0] = Ptr_->GiveThisTag();
          TagAddr[1] = Ptr_->GiveStartTag();
          TagAddr[2] = Ptr_->GiveEndTag();
          TagAddr[3] = Ptr_->GiveTagParent();

          fnd = TagAddr[i] == snptr && (!tpptr || TagAddr[ti] == tpptr);
        }
      }
    }

    if (Found_)
      *Found_ = fnd;

    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
NodeNestingData* StackNodeFinder::FindNextNode_TagType(NodeNestingData* ptr, int TagType_, bool* Found_)
{
  return FindNextNode_MatchingPtr(ptr, TagType_, _StkNodePtr, NULL, Found_);
}

/****************************************************************************/
const NodeNestingData* StackNodeFinder::FindNextNode_TagType(NodeNestingData* ptr, int TagType_, bool* Found_) const
{
  return FindNextNode_MatchingPtr(ptr, TagType_, _ConstStkNodePtr, NULL, Found_);
}

/****************************************************************************/
MEMORYOPS_DEFN(StackNodeFinder)

/****************************************************************************/
// StackNodeInfo class definition
/****************************************************************************/
StackNodeInfo::StackNodeInfo():
QListNode(),

// Set From Node Creation Parameters
_DocType(0),
_SearchType(0),

// from TagSearchResult Class
_NumDelims(0),
_StartBrkFound(false),
_EndBrkFound(false),
_TagElementFound(false),
_ContentType(0),
_HasTagContents(false),
_HasIntraTagContents(false),
_NumKeyValuePairs(0),
_TagKeyValuePairs(NULL),
_TagKVArrayCopied(false),
_ContentShifted(0),

// from TagTypeInfo Class
_Parent(NULL),
_SearchRun(0),
_ElementPtr(NULL),
_TagAttr(0),
_TagBrkIndex(0),
_EndBrkIndex(0),
_AltIndex(0),

_StdTagCount(0),
_NestedCmntTagCount(0),
_CopenCmntTagCount(0),
_NestingError(0),
_TagNestCond(0),

_TagFound(false),
_ContentTypeTag(false),
_NonContentTypeTag(false),
_HasPairContent(false),
_TagTypeFound(0),
_LinkType(0),

_SingletonTagFound(false),
_EmptyTagFound(false),
_ContainerTagFound(false),
_StdTagFound(false),
_PairEnderFound(false),
_PrevUnorderedTagFound(false),
_UnorderedTag(false),

// From TagTypeInfo Class: Additional StackNodeInfo related data members
_CursorStkTag(NULL),
_FoundTag(NULL),
_FoundCursorTag(false),
_StkTagFound(false),
_TagContValid(0),

_NCTagAddr(NULL),
_NextSequenceNum(0),
_ParentSequenceNum(0),
_SequenceNum(0),
_CurLevel(0),
_EncLevel(0),
_ChdLevel(0),

_PallocChildren(NULL),
_CrefChildren(NULL),

_TextNode(NULL),
_TagStates(NULL),

// StackNodeInfo class data members
_Prev(NULL),
_Next(NULL),
_BackLink(NULL),
_ForwardLink(NULL),
_PrevCursor(NULL),
_NextCursor(NULL),

_NestedQ(new NodeNestingData),
_NestedQhead(NULL),
_NestedQptr(NULL),
_NestedQptrSet(false),
_NumNestedData(1),

_OverlappedPrevPair(NULL),
_OverlappedNextPair(NULL),
_MismatchedEndPair(NULL),
_MatchInfoLink(NULL),

_EndImplied(false),
_TagAppendDone(false),
_Unlinked(false),
_CursorStkNodeFound(false),
_NodeErased(false),

_HasCommentData(false),
_HasKVPairData(false)
{
  _CommentDataPtr = NULL;
  _NestedQptr = _NestedQhead = _NestedQ;

  // from TagElement Class
  ::memset(_TagAttrib, 0, sizeof(int) * TagEnums::MAX_FIELDS);

  if (_Parent)
    _Parent->Sequencer();
}

/****************************************************************************/
StackNodeInfo::StackNodeInfo(const StackNodeInfo& Obj_):
QListNode(Obj_),

// Set From Node Creation Parameters
_DocType(Obj_._DocType),
_SearchType(Obj_._SearchType),

// from TagSearchResult Class
_NumDelims(Obj_._NumDelims),
_StartBrkFound(Obj_._StartBrkFound),
_EndBrkFound(Obj_._EndBrkFound),
_TagElementFound(Obj_._TagElementFound),
_StartBrkStr(Obj_._StartBrkStr),
_TagElementStr(Obj_._TagElementStr),
_EndBrkStr(Obj_._EndBrkStr),
_ContentType(Obj_._ContentType),
_HasTagContents(Obj_._HasTagContents),
_HasIntraTagContents(Obj_._HasIntraTagContents),
_NumKeyValuePairs(Obj_._NumKeyValuePairs),
_TagKeyValuePairs(Obj_._TagKeyValuePairs),
_TagKVArrayCopied(true),
_ContentShifted(0),

// from TagTypeInfo Class
_Parent(Obj_._Parent),
_SearchRun(Obj_._SearchRun),
_ElementPtr(Obj_._ElementPtr),
_TagAttr(Obj_._TagAttr),
_TagBrkIndex(Obj_._TagBrkIndex),
_EndBrkIndex(Obj_._EndBrkIndex),
_AltIndex(Obj_._AltIndex),

_StdTagCount(Obj_._StdTagCount),
_NestedCmntTagCount(Obj_._NestedCmntTagCount),
_CopenCmntTagCount(Obj_._CopenCmntTagCount),
_NestingError(Obj_._NestingError),
_TagNestCond(Obj_._TagNestCond),

_TagFound(Obj_._TagFound),
_ContentTypeTag(Obj_._ContentTypeTag),
_NonContentTypeTag(Obj_._NonContentTypeTag),
_HasPairContent(Obj_._HasPairContent),
_TagTypeFound(Obj_._TagTypeFound),
_LinkType(Obj_._LinkType),

_SingletonTagFound(Obj_._SingletonTagFound),
_EmptyTagFound(Obj_._EmptyTagFound),
_ContainerTagFound(Obj_._ContainerTagFound),
_StdTagFound(Obj_._StdTagFound),
_PairEnderFound(Obj_._PairEnderFound),

// From TagTypeInfo Class: Additional StackNodeInfo related data members
_CursorStkTag(Obj_._CursorStkTag),
_FoundTag(Obj_._FoundTag),
_FoundCursorTag(Obj_._FoundCursorTag),
_StkTagFound(Obj_._StkTagFound),
_TagContValid(Obj_._TagContValid),

_NCTagAddr(Obj_._NCTagAddr),
_NextSequenceNum(Obj_._NextSequenceNum),
_ParentSequenceNum(Obj_._ParentSequenceNum),
_SequenceNum(Obj_._SequenceNum),
_CurLevel(Obj_._CurLevel),
_EncLevel(Obj_._EncLevel),
_ChdLevel(Obj_._ChdLevel),

_PallocChildren(NULL),
_CrefChildren(Obj_._CrefChildren),

_TextNode(Obj_._TextNode),
_TagStates(Obj_._TagStates),

// StackNodeInfo class data members
_Prev(NULL),
_Next(NULL),
_BackLink(Obj_._BackLink),
_ForwardLink(Obj_._ForwardLink),
_PrevCursor(NULL),
_NextCursor(NULL),

_NestedQ(new NodeNestingData),
_NestedQhead(NULL),
_NestedQptr(NULL),
_NestedQptrSet(false),
_NumNestedData(1),

_OverlappedPrevPair(Obj_._OverlappedPrevPair),
_OverlappedNextPair(Obj_._OverlappedNextPair),
_MismatchedEndPair(Obj_._MismatchedEndPair),
_MatchInfoLink(NULL),

_EndImplied(Obj_._EndImplied),
_TagAppendDone(Obj_._TagAppendDone),
_Unlinked(Obj_._Unlinked),
_CursorStkNodeFound(Obj_._CursorStkNodeFound),
_NodeErased(Obj_._NodeErased),

_HasCommentData(Obj_._HasCommentData),
_HasKVPairData(Obj_._HasKVPairData)
{
  if (_HasCommentData)
    _CommentDataPtr = Obj_._CommentDataPtr;

  if (_HasKVPairData)
    _KVPairDataPtr = Obj_._KVPairDataPtr;

  _NestedQhead = _NestedQ;
  NodeNestingData::CopyNodeNestingData(_NestedQ, _NestedQhead, _NumNestedData,
                                       Obj_._NestedQ, Obj_._NestedQhead, Obj_._NumNestedData);
  _NestedQptr = _NestedQhead;

  // from TagElement Class
  ::memmove(_TagAttrib, Obj_._TagAttrib, sizeof(int) * TagEnums::MAX_FIELDS);

  if (_Parent)
    _Parent->Sequencer();

  if (_TextNode)
    _TextNode->SetCopied(true);

  if (_TagStates)
    _TagStates->SetCopied(true);
}

/****************************************************************************/
StackNodeInfo::~StackNodeInfo()
{
  TagElement* TagElmPtr_ = _Parent ? _Parent->GiveTagElementPtr():NULL;

  SetPrev(NULL);
  _BackLink = NULL;
  _ForwardLink = NULL;
  _PrevCursor = NULL;
  _NextCursor = NULL;

  _OverlappedPrevPair = NULL;
  _OverlappedNextPair = NULL;
  _MismatchedEndPair = NULL;
  _MatchInfoLink = NULL;

  if (_TextNode)
  {
    _TextNode->SetRefNode(NULL);
    _TextNode = NULL;
  }

  if (_TagStates && !_TagStates->IsCopied())
  {
    if (TagElmPtr_)
      TagElmPtr_->SetTagElementState(NULL);

    _TagStates->SetParent(NULL);

    delete _TagStates;
    _TagStates = NULL;
  }

  if (_NestedQhead)
    _NestedQhead->DeleteAll();

  _NestedQptr =
  _NestedQ =
  _NestedQhead = NULL;

  if (!_TagKVArrayCopied && _NumKeyValuePairs)
  {
    size_t x;
    size_t max = _NumKeyValuePairs;

    for (x = 0; x < max; x++)
    {
      delete _TagKeyValuePairs[x];
      _TagKeyValuePairs[x] = NULL;
    }

    ::RawDeleteArray(_TagKeyValuePairs);
    _NumKeyValuePairs = 0;
  }

  if (_PallocChildren)
  {
    ::RawDelete(_PallocChildren);
    _PallocChildren = NULL;
  }

  if (_HasKVPairData)
  {
    delete _KVPairDataPtr;
    _KVPairDataPtr = NULL;
    _HasKVPairData = false;
  }

  if (_HasCommentData)
  {
    delete _CommentDataPtr;
    _CommentDataPtr = NULL;
    _HasCommentData = false;
  }

  // delete all: taken care of by QListNode virtual destructor
  // delete _Next;
  // SetNext(NULL);
}

/****************************************************************************/
// NodeNestingData traversing methods
/****************************************************************************/
NodeNestingData* StackNodeInfo::PopNestingNode(bool Erase_, bool ConfirmErased_)
{
  NodeNestingData* RetPtr_ = NULL;
  NodeNestingData* EndPt_ = NULL;
  NodeNestingData* SavedPtr_ = NULL;
  NodeNestingData* RelPtr_ = (_NestedQptr && _NestedQptrSet) ?
                                _NestedQptr:
                                (_NestedQptr=HeadNestingNode(true, Erase_ && !ConfirmErased_));

  bool HeadEnd_ = _NestedQptr == _NestedQhead;
  bool LastNode_ = false;

  if (!RelPtr_)
    return NULL;

  if (!_NestedQptrSet && _NestedQptr && Erase_ && !ConfirmErased_)
    SetNestingNodePtr(true);

  if (HeadEnd_)
  {
    SavedPtr_ = _NestedQptr;
    _NestedQptr = (NodeNestingData*)(_NestedQptr->Next());
    LastNode_ = _NestedQptr == NULL;

    if (!LastNode_)
      _NestedQhead = _NestedQptr;
  }
  else
  {
    SavedPtr_ = RelPtr_;
    _NestedQhead = (NodeNestingData*)(RelPtr_->Next());
    LastNode_ = _NestedQhead == NULL;

    if (LastNode_)
      _NestedQptr = _NestedQhead = RelPtr_;
  }

  if (ConfirmErased_)
  {
    if (RelPtr_->Erased())
    {
      if (RelPtr_->Next())
      {
        RelPtr_->GiveNext()->SetPrev(NULL);
        EndPt_ = (RelPtr_ != RelPtr_->Next()) ?
                     (NodeNestingData*)(RelPtr_->Next()):NULL;
        RelPtr_->SetNext(NULL);
      }

      RetPtr_ = RelPtr_;
      RelPtr_ = EndPt_;

      if (LastNode_)
        _NestedQptr = _NestedQhead = NULL;
      else if (!HeadEnd_)
        _NestedQptr = EndPt_;
    }
    else
    {
      if (HeadEnd_)
      {
        if (!LastNode_)
          _NestedQhead = _NestedQptr = SavedPtr_;
        else
          _NestedQptr = _NestedQhead;
      }
      else if (!LastNode_)
        _NestedQptr = _NestedQhead = RelPtr_;

      RelPtr_ = NULL;
    }
  }
  else if (Erase_)
  {
    if (!LastNode_)
      EndPt_ = _NestedQhead;

    if (!ConfirmErased_)
    {
      while (RelPtr_ != EndPt_)
      {

        RelPtr_->SetErased(true);
        RelPtr_ = (NodeNestingData*)(RelPtr_->Next());
      }

      if (LastNode_)
        RelPtr_ = _NestedQ;
    }

    _NestedQhead = SavedPtr_;
  }

  _NumNestedData = _NestedQhead->Size();
  return (RetPtr_ ? RetPtr_:RelPtr_);
}

/****************************************************************************/
NodeNestingData* StackNodeInfo::NextNestingNode(bool Reset_, bool Peek_, bool NotErased_)
{
  // mark
  if (_NestedQptr && !Reset_)
  {
    bool TailEnd_ = _NestedQptr == _NestedQ;
    NodeNestingData* PrevPtr_ = _NestedQptr;
    NodeNestingData* NextPtr_ = _NestedQptr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (NodeNestingData*)(NextPtr_->Next());
    }
    while (NotErased_ && NextPtr_ && NextPtr_->Erased());

    if (Peek_)
      return NextPtr_;
    else
      _NestedQptr = NextPtr_;

    if (TailEnd_)
    {
      _NestedQ = _NestedQptr ? _NestedQptr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _NestedQptr = HeadNestingNode(true, NotErased_);

  return _NestedQptr;
}

/****************************************************************************/
NodeNestingData* StackNodeInfo::PrevNestingNode(bool Reset_, bool Peek_, bool NotErased_)
{
  if (_NestedQptr && !Reset_)
  {
    bool HeadEnd_ = _NestedQptr == _NestedQhead;
    NodeNestingData* PrevPtr_ = _NestedQptr;
    NodeNestingData* NextPtr_ = _NestedQptr;

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (NodeNestingData*)(NextPtr_->Prev());
    }
    while (NotErased_ && NextPtr_ && NextPtr_->Erased());

    if (Peek_)
      return NextPtr_;
    else
      _NestedQptr = NextPtr_;

    if (HeadEnd_)
    {
      _NestedQhead = _NestedQptr ? _NestedQptr:PrevPtr_;
      return NextPtr_;
    }
  }
  else if (Reset_)
    _NestedQptr = TailNestingNode(true, NotErased_);

  return _NestedQptr;
}

/****************************************************************************/
NodeNestingData* StackNodeInfo::PushNestingNode(NodeNestingData* Node_, bool Create_)
{
  NodeNestingData* NodePtr_ = (Create_ && Node_) ? Node_->MakeNode(*Node_):Node_;

  if (NodePtr_)
  {
    if (_NestedQhead)
    {
      HeadNestingNode(true, false);
      _NestedQptr =
      _NestedQhead = (NodeNestingData*)(_NestedQhead->AppendHead(NodePtr_, false));
      SetNestingNodePtr(false);
    }
    else
      _NestedQptr =
      _NestedQhead =
      _NestedQ = NodePtr_;

    _NumNestedData = _NestedQhead->Size();
  }

  return NodePtr_;
}

/****************************************************************************/
NodeNestingData* StackNodeInfo::HeadNestingNode(bool Reset_, bool NotErased_) const
{
  NodeNestingData* PrevPtr_ = _NestedQhead;
  NodeNestingData* NextPtr_ = _NestedQhead;

  if (_NestedQhead && NotErased_)
  {
    bool TailEnd_ = PrevPtr_ == _NestedQ;

    if (Reset_)
    {
      for (PrevPtr_ = (NodeNestingData*)(_NestedQhead->Prev());
           PrevPtr_ && (_NestedQhead = PrevPtr_);
           PrevPtr_ = (NodeNestingData*)(_NestedQhead->Prev()));
      NextPtr_ = _NestedQhead;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (NodeNestingData*)(NextPtr_->Next());
      TailEnd_ = PrevPtr_ == _NestedQ;
    }
    while (PrevPtr_ && PrevPtr_->Erased() && !TailEnd_);

    _NestedQhead = PrevPtr_;

    if (_NestedQhead->Erased() && TailEnd_)
    {
      for (PrevPtr_ = (NodeNestingData*)(_NestedQhead->Prev());
           PrevPtr_ && (_NestedQhead = PrevPtr_);
           PrevPtr_ = (NodeNestingData*)(_NestedQhead->Prev()));

      return (_NestedQhead->Erased() ? NULL:_NestedQhead);
    }
  }
  else if (_NestedQhead && Reset_)
  {
    for (PrevPtr_ = (NodeNestingData*)(_NestedQhead->Prev());
         PrevPtr_ && (_NestedQhead = PrevPtr_);
         PrevPtr_ = (NodeNestingData*)(_NestedQhead->Prev()));
  }

  return _NestedQhead;
}

/****************************************************************************/
NodeNestingData* StackNodeInfo::TailNestingNode(bool Reset_, bool NotErased_) const
{
  NodeNestingData* PrevPtr_ = _NestedQ;
  NodeNestingData* NextPtr_ = _NestedQ;

  if (_NestedQ && NotErased_)
  {
    bool HeadEnd_ = PrevPtr_ == _NestedQhead;

    if (Reset_)
    {
      for (NextPtr_ = (NodeNestingData*)(_NestedQ->Next());
           NextPtr_ && (_NestedQ = NextPtr_);
           NextPtr_ = (NodeNestingData*)(_NestedQ->Next()));
      NextPtr_ = _NestedQ;
    }

    do
    {
      PrevPtr_ = NextPtr_;
      NextPtr_ = (NodeNestingData*)(NextPtr_->Prev());
      HeadEnd_ = PrevPtr_ == _NestedQhead;
    }
    while (PrevPtr_ && PrevPtr_->Erased() && !HeadEnd_);

    _NestedQ = PrevPtr_;

    if (_NestedQ->Erased() && HeadEnd_)
    {
      for (NextPtr_ = (NodeNestingData*)(_NestedQ->Next());
           NextPtr_ && (_NestedQ = NextPtr_);
           NextPtr_ = (NodeNestingData*)(_NestedQ->Next()));

      return (_NestedQ->Erased() ? NULL:_NestedQ);
    }
  }
  else if (_NestedQ && Reset_)
  {
    for (NextPtr_ = (NodeNestingData*)(_NestedQ->Next());
         NextPtr_ && (_NestedQ = NextPtr_);
         NextPtr_ = (NodeNestingData*)(_NestedQ->Next()));
  }

  return _NestedQ;
}

/****************************************************************************/
NodeNestingData* StackNodeInfo::SetNestingNodePtr(bool Set_) const
{
  _NestedQptrSet = Set_;
  return _NestedQptr;
}

/****************************************************************************/
void StackNodeInfo::ConfirmNestingNodeErased()
{
  NodeNestingData* More_ = NULL;
  NextNestingNode(true, false);
  SetNestingNodePtr(true);
  NodeNestingData* NextPtr_ = _NestedQhead;

  if (NextPtr_)
  {
    _NestedQptr = NextPtr_;

    do
    {
      if (More_)
      {
        More_->Unlink();
        More_ = NULL;
      }

      More_ = PopNestingNode(false, true);
    }
    while (More_ && _NestedQptr);
  }

  NextNestingNode(true, false);
  SetNestingNodePtr(false);
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::NestedTag(bool Head_, size_t Index_)
{
  if (_NestedQ && Index_ < _NumNestedData)
  {
    size_t x;
    size_t max = Index_;
    NodeNestingData* Ptr_ = Head_ ? _NestedQhead:_NestedQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (NodeNestingData*)(Ptr_->Prev());

    return (Ptr_ ? Ptr_->GiveStartTag():NULL);
  }

  return NULL;
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::NestedEndTag(bool Head_, size_t Index_)
{
  if (_NestedQ && Index_ < _NumNestedData)
  {
    size_t x;
    size_t max = Index_;
    NodeNestingData* Ptr_ = Head_ ? _NestedQhead:_NestedQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (NodeNestingData*)(Ptr_->Prev());

    return (Ptr_ ? Ptr_->GiveEndTag():NULL);
  }

  return NULL;
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::NestedTagParent(bool Head_, size_t Index_)
{
  if (_NestedQ && Index_ < _NumNestedData)
  {
    size_t x;
    size_t max = Index_;
    NodeNestingData* Ptr_ = Head_ ? _NestedQhead:_NestedQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (NodeNestingData*)(Ptr_->Prev());

    return (Ptr_ ? Ptr_->GiveTagParent():NULL);
  }

  return NULL;
}

/****************************************************************************/
NodeNestingData* StackNodeInfo::GiveNestingNode(bool Head_, size_t Index_)
{
  if (_NestedQ && Index_ < _NumNestedData)
  {
    size_t x;
    size_t max = Index_;
    NodeNestingData* Ptr_ = Head_ ? _NestedQhead:_NestedQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (NodeNestingData*)(Ptr_->Prev());

    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
const NodeNestingData* StackNodeInfo::NestingNode(bool Head_, size_t Index_) const
{
  if (_NestedQ && Index_ < _NumNestedData)
  {
    size_t x;
    size_t max = Index_;
    NodeNestingData* Ptr_ = Head_ ? _NestedQhead:_NestedQ;

    if (!Head_)
      for (x = 0; x < max && Ptr_; x++)
        Ptr_ = (NodeNestingData*)(Ptr_->Prev());

    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
void StackNodeInfo::ShowNodeNestingContents(const StackNodeInfo* ptr_, ostream& os_, size_t fw_) const
{
  if (_NestedQ && TagTypeInfo::ShowDebugDump() && ptr_ != this &&
      (!Erased() || TagTypeInfo::ShowErasedNodes()))
  {
    size_t x;
    size_t max = _NumNestedData;

    TailNestingNode(true, true);
    _NestedQptr = _NestedQ;

    if (Erased())
      os_ <<"--> *** Erased ***" <<endl;

    for (x = 0; x < max && _NestedQptr; x++)
    {
      _NestedQptr->ShowContents(this, os_, fw_);
      _NestedQptr = (NodeNestingData*)(_NestedQptr->Prev());
    }
  }
}

/****************************************************************************/
NodeNestingData* StackNodeInfo::CreateNestingNodeData(StackNodeInfo* ThisTag_,
                                                      StackNodeInfo* Tag_,
                                                      StackNodeInfo* EndTag_,
                                                      StackNodeInfo* TagParent_)
{
  NodeNestingData* NodeNestPtr_ = new NodeNestingData;

  if (ThisTag_)
    NodeNestPtr_->SetThisTag(ThisTag_);

  if (Tag_)
    NodeNestPtr_->SetStartTag(Tag_);

  if (EndTag_)
    NodeNestPtr_->SetEndTag(EndTag_);

  if (TagParent_)
    NodeNestPtr_->SetTagParent(TagParent_);

  PushNestingNode(NodeNestPtr_, false);
  _NestedQptr = _NestedQhead;

  return _NestedQhead;
}

/****************************************************************************/
/****************************************************************************/
StackNodeInfo* StackNodeInfo::Make()
{
  return (new StackNodeInfo);
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::Make(const StackNodeInfo& Obj_)
{
  return (new StackNodeInfo(Obj_));
}

/****************************************************************************/
void StackNodeInfo::SetSourceTagElementState(TagElementStates* Ptr_) const
{
  if (_ElementPtr)
    _ElementPtr->SetSourceTagElementState(Ptr_);
}

/****************************************************************************/
void StackNodeInfo::SetSourceTagElementState() const
{
  if (_ElementPtr && _TagStates)
    _ElementPtr->SetSourceTagElementState(_TagStates);
}

/****************************************************************************/
void StackNodeInfo::SetTagElementState() const
{
  if (_ElementPtr && _TagStates &&
      _TagStates != _ElementPtr->TagElementState())
    _ElementPtr->SetTagElementState(_TagStates);
  else if (_ElementPtr && _TagStates)
    _ElementPtr->RenewTagElementState();
}

/****************************************************************************/
void StackNodeInfo::SetPrevUnorderedTag(bool PrevSearchDone_, bool GenSrchWasDone_,
                                        int GenSearchFailed_, Ulong PrevSearchRun_,
                                        Ulong CurrSearchRun_, bool NonContentTypeTagFound_,
                                        bool ContentTypeTagFound_)
{
  if (!PrevSearchDone_)
  {
    _PrevUnorderedTagFound = GenSrchWasDone_ &&
                             GenSearchFailed_ == TagDictionary::NO_FAIL &&
                             PrevSearchRun_ < CurrSearchRun_ &&
                             (NonContentTypeTagFound_ || ContentTypeTagFound_);

    if (_PrevUnorderedTagFound)
    {
      _NextSequenceNum = CurrSearchRun_ - PrevSearchRun_;

      if (_Next)
        _Next->_UnorderedTag = true;
    }
  }
  else if (PrevSearchRun_ < CurrSearchRun_)
    _PrevUnorderedTagFound = false;
}

/****************************************************************************/
const char* StackNodeInfo::ContentShiftedTypeToStr(int Type_)
{
  return
  (
    (Type_ & CONTENT_RESET)       ? "CONTENT_RESET":
    (Type_ & CONTENT_SHIFTED)     ? "CONTENT_SHIFTED":
    (Type_ & CONTENT_APPENDED)    ? "CONTENT_APPENDED":
    (Type_ & CONTENT_RECEIVED)    ? "CONTENT_RECEIVED":
    (Type_ & CONTENT_TRANSFERRED) ? "CONTENT_TRANSFERRED":
    (Type_ & CONTENT_SHOWN)       ? "CONTENT_SHOWN":"No"
  );
}

/****************************************************************************/
int StackNodeInfo::TagNesting(bool PairOnly_) const
{
  bool EndTag_ = PairEnderFound() && _StartBrkFound && _EndBrkFound &&
                 TagBracketList::IsEndTag(_StartBrkStr.c_str(), _EndBrkStr.c_str());

  return (PairOnly_ ? (EndTag_ ? 0:_TagAttrib[TagEnums::NESTING]):
                      _TagAttrib[TagEnums::NESTING]);
}

/****************************************************************************/
void StackNodeInfo::ShowAppTextStr(ostream& os_, size_t fw_,
                                   AppTextNode* Ptr_, int Type_)
{
  if (Type_ > 0 && Ptr_ && !Ptr_->Erased() &&
      Ptr_->TextQhead() && TagTypeInfo::ShowDebugDump())
    Ptr_->ShowTextNodeContents(os_, fw_);
}

/****************************************************************************/
ChrString StackNodeInfo::GiveIntraTagContentFromSearch(StackNodeInfo* Ptr_)
{
  ChrString TagContent_;

  if (Ptr_ && !Ptr_->Erased())
    if (Ptr_->_HasIntraTagContents &&
        (Ptr_->_StartBrkFound &&
         Ptr_->_StartBrkStr.strlen()) &&
        (Ptr_->_EndBrkFound &&
         Ptr_->_EndBrkStr.strlen()))
      TagContent_ = Ptr_->_Parent->SearchResult()->GiveIntraTagContentString();

  return TagContent_;
}

/****************************************************************************/
ChrString StackNodeInfo::GiveTagContentFromSearch(StackNodeInfo* Ptr_)
{
  ChrString TagContent_;

  if (Ptr_ && !Ptr_->Erased())
    if (Ptr_->_HasTagContents &&
        Ptr_->_ContentTypeTag &&
        (Ptr_->_StartBrkFound &&
         Ptr_->_StartBrkStr.strlen()) &&
        (Ptr_->_EndBrkFound &&
         Ptr_->_EndBrkStr.strlen()))
      TagContent_ = Ptr_->_Parent->SearchResult()->GiveTagContentString();

  return TagContent_;
}

/****************************************************************************/
ChrString StackNodeInfo::GiveTagStrFromSearch(StackNodeInfo* Ptr_)
{
  ChrString TagStr_;

  if (Ptr_ && !Ptr_->Erased())
    if ((Ptr_->_StartBrkFound &&
         Ptr_->_StartBrkStr.strlen()) &&
        (Ptr_->_ContentTypeTag ||
         Ptr_->_NonContentTypeTag) &&
        (Ptr_->_EndBrkFound &&
         Ptr_->_EndBrkStr.strlen()))
    {
      TagStr_ = Ptr_->_Parent->SearchResult()->GiveTagString();

      if (!TagTypeInfo::ShowNewLines())
      {
        SubstChar(TagStr_.c_str(), "\n\r", ' ');
        TagStr_.SimplifySpaces(false);
      }
    }

  return TagStr_;
}

/****************************************************************************/
ChrString StackNodeInfo::GiveIntraTagContent() const
{
  if (_HasIntraTagContents &&
      _IntraContentStr.strlen())
    return _IntraContentStr;
}

/****************************************************************************/
ChrString StackNodeInfo::GiveTagStr() const
{
  size_t x;
  size_t max = _NumKeyValuePairs;

  ChrString TagStr_;
  ChrString kvStr_;
  const KeyValuePairNode* kvPtr_ = NULL;

  if (!Erased())
  {
    if (_StartBrkFound &&
        _StartBrkStr.strlen())
    {
      TagStr_ = _StartBrkStr;

      if (_ContentTypeTag ||
          _NonContentTypeTag)
      {
        if (_TagElementFound &&
            _TagElementStr.strlen())
          TagStr_ += _TagElementStr;

        if (_HasIntraTagContents &&
            _IntraContentStr.strlen())
          TagStr_ += (ChrString(" ") + _IntraContentStr + ChrString(" "));

        for (x = 0; x < max; x++)
          if (_HasKVPairData)
          {
            kvPtr_ = _KVPairDataPtr->GiveKvPair(x);
            kvStr_ = kvPtr_ ? kvPtr_->GiveKVPairString():"NULL";
            TagStr_ += (ChrString(" ") + kvStr_);
          }
          else if (_TagKeyValuePairs[x])
            TagStr_ += (ChrString(" ") + *_TagKeyValuePairs[x]);

        if (_EndBrkFound &&
            _EndBrkStr.strlen())
          TagStr_ += (_EndBrkStr + ChrString("\n"));
        else
          TagStr_ += ChrString("\n");
      }
      else
        TagStr_ += ChrString("\n");
    }

    if (!TagTypeInfo::ShowNewLines())
    {
      SubstChar(TagStr_.c_str(), "\n\r", ' ');
      TagStr_.SimplifySpaces(false);
    }
  }

  return TagStr_;
}

/****************************************************************************/
void StackNodeInfo::ShowTagStr(ostream& os_, size_t fw_, StackNodeInfo* Ptr_,
                               int Type_, const char* LinkStr_, bool ShowObjectID_)
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  size_t x;
  size_t max = Ptr_->_NumKeyValuePairs;

  if (Type_ > 0 && Ptr_ && !Ptr_->Erased() &&
      TagTypeInfo::ShowDebugDump())
    if (Ptr_->_StartBrkFound &&
        Ptr_->_StartBrkStr.strlen())
    {
      if (fw_)
        os_.width(fw_);
      else
        os_ <<"\t";

      if (LinkStr_)
        os_ <<LinkStr_ <<": ";
      os_ <<Ptr_->_StartBrkStr;

      if (Ptr_->_ContentTypeTag ||
          Ptr_->_NonContentTypeTag)
      {
        if (Ptr_->_TagElementFound &&
            Ptr_->_TagElementStr.strlen())
          os_ <<Ptr_->_TagElementStr;

        if (Ptr_->_HasIntraTagContents &&
            Ptr_->_IntraContentStr.strlen())
        {
          if (!TagTypeInfo::ShowNewLines())
          {
            SubstChar(Ptr_->_IntraContentStr.c_str(), "\n\r", ' ');
            Ptr_->_IntraContentStr.SimplifySpaces(false);
          }

          os_ <<" " <<Ptr_->_IntraContentStr <<" ";
        }

        for (x = 0; x < max; x++)
          if (Ptr_->_TagKeyValuePairs[x])
          {
            if (!TagTypeInfo::ShowNewLines())
            {
              SubstChar(Ptr_->_TagKeyValuePairs[x]->c_str(), "\n\r", ' ');
              Ptr_->_TagKeyValuePairs[x]->SimplifySpaces(false);
            }

            os_ <<" " <<(*Ptr_->_TagKeyValuePairs[x]);
          }

        if (Ptr_->_EndBrkFound &&
            Ptr_->_EndBrkStr.strlen())
        {
          os_ <<Ptr_->_EndBrkStr;

          if (ShowObjectID_)
            os_ <<" | " <<"NodeID: " <<Ptr_->NodeID() <<" | Addr: " <<Ptr_ <<endl;
          else
            os_ <<endl;
        }
        else
        {
          if (ShowObjectID_)
            os_ <<" | " <<"NodeID: " <<Ptr_->NodeID() <<" | Addr: " <<Ptr_ <<endl;
          else
            os_ <<endl;
        }
      }
      else
        os_ <<endl;
    }
}

/****************************************************************************/
void StackNodeInfo::ShowStackAddr(QListNode* StkPtr_, int Type_, const char* StkStr_,
                                  int Dir_, bool DumpInfo_)
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  int x;
  const char* Istr_ = StkStr_ ? StkStr_:"";
  StackNodeInfo* Ptr_ = (Type_ != APPTEXT) ? ((StackNodeInfo*)StkPtr_):NULL;
  AppTextNode* AppPtr_ = (Type_ == APPTEXT) ? ((AppTextNode*)StkPtr_):NULL;
  StackNodeInfo* AppTxtStkNode_ = AppPtr_ ? AppPtr_->GiveRefNode():NULL;

  ostream* os_ = (Type_ == APPTEXT) ?
                      (AppTxtStkNode_ ? AppTxtStkNode_->GiveOutput():NULL):
                      (Ptr_ ? Ptr_->GiveOutput():NULL);

  if (!os_)
    os_ = &std::cout;

  if (Type_ == StackNodeInfo::APPTEXT && AppPtr_ && AppPtr_->RefNode())
  {
    ChrString RefTag_;

    if (Dir_ == 1)
    {
      fprintf(stdout, "stack: %s\n", Istr_);
      for (x = 0; AppPtr_; AppPtr_ = (AppTextNode*)(AppPtr_->GiveNext()), x++)
      {
        if (AppPtr_)
        {
          RefTag_ = AppPtr_->RefNode()->GiveTagStr();

          if (!AppPtr_->Erased() && !AppPtr_->DoErase())
            fprintf(stdout, "node=%u this=0x%p ref=0x%p reftag=%s prev=0x%p next=0x%p\n",
                    x, AppPtr_, AppPtr_->RefNode(), RefTag_.c_str(), AppPtr_->Prev(), AppPtr_->Next());
          else
          {
            if (TagTypeInfo::ShowErasedNodes())
              fprintf(stdout, "node=%u this=0x%p ref=0x%p reftag=%s prev=0x%p next=0x%p : ",
                      x, AppPtr_, AppPtr_->RefNode(), RefTag_.c_str(), AppPtr_->Prev(), AppPtr_->Next());

            fprintf(stdout, "*** Erased ***\n");
          }
        }

        ShowAppTextStr(*os_, 0, AppPtr_, Type_);
      }
    }
    else if (Dir_ == -1)
    {
      fprintf(stdout, "stack: %s\n", Istr_);
      for (x = 0; AppPtr_; AppPtr_ = (AppTextNode*)(AppPtr_->GivePrev()), x++)
      {
        if (AppPtr_)
        {
          RefTag_ = AppPtr_->RefNode()->GiveTagStr();

          if (!AppPtr_->Erased() && !AppPtr_->DoErase())
            fprintf(stdout, "node=%u this=0x%p ref=0x%p reftag=%s prev=0x%p next=0x%p\n",
                    x, AppPtr_, AppPtr_->RefNode(), RefTag_.c_str(), AppPtr_->Prev(), AppPtr_->Next());
          else
          {
            if (TagTypeInfo::ShowErasedNodes())
              fprintf(stdout, "node=%u this=0x%p ref=0x%p reftag=%s prev=0x%p next=0x%p : ",
                      x, AppPtr_, AppPtr_->RefNode(), RefTag_.c_str(), AppPtr_->Prev(), AppPtr_->Next());

            fprintf(stdout, "*** Erased ***\n");
          }
        }

        ShowAppTextStr(*os_, 0, AppPtr_, Type_);
      }
    }
  }
  else if (Type_ == StackNodeInfo::PREVNEXT)
  {
    if (Dir_ == 1)
    {
      fprintf(stdout, "stack: %s\n", Istr_);
      for (x = 0; Ptr_; Ptr_ = Ptr_->_Next, x++)
      {
        if (Ptr_ && !Ptr_->Erased())
          fprintf(stdout, "node=%u this=0x%p prev=0x%p next=0x%p\n",
                  x, Ptr_, Ptr_->_Prev, Ptr_->_Next);
        else
        {
          if (TagTypeInfo::ShowErasedNodes())
            fprintf(stdout, "node=%u this=0x%p prev=0x%p next=0x%p : ",
                    x, Ptr_, Ptr_->_Prev, Ptr_->_Next);

          fprintf(stdout, "*** Erased ***\n");
        }

        ShowTagStr(*os_, 0, Ptr_, Type_);
        if (DumpInfo_)
          Ptr_->TagInfoListWalkDump(*os_, 20);
      }
    }
    else if (Dir_ == -1)
    {
      fprintf(stdout, "stack: %s\n", Istr_);
      for (x = 0; Ptr_; Ptr_ = Ptr_->_Prev, x++)
      {
        if (Ptr_ && !Ptr_->Erased())
          fprintf(stdout, "node=%u this=0x%p prev=0x%p next=0x%p\n",
                  x, Ptr_, Ptr_->_Prev, Ptr_->_Next);
        else
        {
          if (TagTypeInfo::ShowErasedNodes())
            fprintf(stdout, "node=%u this=0x%p prev=0x%p next=0x%p : ",
                    x, Ptr_, Ptr_->_Prev, Ptr_->_Next);

          fprintf(stdout, "*** Erased ***\n");
        }

        ShowTagStr(*os_, 0, Ptr_, Type_);
        if (DumpInfo_)
          Ptr_->TagInfoListWalkDump(*os_, 20);
      }
    }
  }
  else if (Type_ == StackNodeInfo::NEXTCUR)
  {
    if (Dir_ == 1)
    {
      fprintf(stdout, "stack: %s\n", Istr_);
      for (x = 0; Ptr_; Ptr_ = Ptr_->_NextCursor, x++)
      {
        if (Ptr_ && !Ptr_->Erased())
          fprintf(stdout, "node=%u this=0x%p prevcur=0x%p nextcur=0x%p\n",
                  x, Ptr_, Ptr_->_PrevCursor, Ptr_->_NextCursor);
        else
        {
          if (TagTypeInfo::ShowErasedNodes())
            fprintf(stdout, "node=%u this=0x%p prevcur=0x%p nextcur=0x%p : ",
                    x, Ptr_, Ptr_->_PrevCursor, Ptr_->_NextCursor);

          fprintf(stdout, "*** Erased ***\n");
        }

        ShowTagStr(*os_, 0, Ptr_, Type_);
      }
    }
    else if (Dir_ == -1)
    {
      fprintf(stdout, "stack: %s\n", Istr_);
      for (x = 0; Ptr_; Ptr_ = Ptr_->_PrevCursor, x++)
      {
        if (Ptr_ && !Ptr_->Erased())
          fprintf(stdout, "node=%u this=0x%p prevcur=0x%p nextcur=0x%p\n",
                  x, Ptr_, Ptr_->_PrevCursor, Ptr_->_NextCursor);
        else
        {
          if (TagTypeInfo::ShowErasedNodes())
            fprintf(stdout, "node=%u this=0x%p prevcur=0x%p nextcur=0x%p : ",
                    x, Ptr_, Ptr_->_PrevCursor, Ptr_->_NextCursor);

          fprintf(stdout, "*** Erased ***\n");
        }

        ShowTagStr(*os_, 0, Ptr_, Type_);
      }
    }
  }
  else if (Type_ == StackNodeInfo::BACKFORW)
  {
    if (Ptr_ && !Ptr_->Erased())
      fprintf(stdout, "this=0x%p back=0x%p forw=0x%p\n",
              Ptr_, Ptr_->_BackLink, Ptr_->_ForwardLink);
    else
    {
      if (TagTypeInfo::ShowErasedNodes())
        fprintf(stdout, "this=0x%p back=0x%p forw=0x%p : ",
                Ptr_, Ptr_->_BackLink, Ptr_->_ForwardLink);

      fprintf(stdout, "*** Erased ***\n");
    }

    ShowTagStr(*os_, 0, Ptr_, Type_, Istr_);

    if (Ptr_->_ForwardLink && !Ptr_->Erased() &&
        Ptr_->_ForwardLink != Ptr_ && Ptr_->_ForwardLink != Ptr_->_BackLink)
      StackNodeInfo::ShowStackAddr(Ptr_->_ForwardLink, StackNodeInfo::BACKFORW, "ForwLink");
  }
  else if (Type_ == StackNodeInfo::PAIRLINK)
  {
    if (Ptr_ && !Ptr_->Erased())
      fprintf(stdout, "this=0x%p prevPair=0x%p nextPair=0x%p\n",
              Ptr_, Ptr_->_OverlappedPrevPair, Ptr_->_OverlappedNextPair);
    else
    {
      if (TagTypeInfo::ShowErasedNodes())
        fprintf(stdout, "this=0x%p prevPair=0x%p nextPair=0x%p : ",
                Ptr_, Ptr_->_OverlappedPrevPair, Ptr_->_OverlappedNextPair);

      fprintf(stdout, "*** Erased ***\n");
    }

    if (!Ptr_->Erased())
    {
      ShowTagStr(*os_, 0, Ptr_, Type_, "this");
      ShowTagStr(*os_, 0, Ptr_->_OverlappedPrevPair, Type_, "prevPair");
      ShowTagStr(*os_, 0, Ptr_->_OverlappedNextPair, Type_, "nextPair");
    }
  }
  else if (Type_ == StackNodeInfo::MISMLINK)
  {
    if (Ptr_ && !Ptr_->Erased())
      fprintf(stdout, "this=0x%p mismatchedEnd=0x%p\n",
              Ptr_, Ptr_->_MismatchedEndPair);
    else
    {
      if (TagTypeInfo::ShowErasedNodes())
        fprintf(stdout, "this=0x%p mismatchedEnd=0x%p : ",
                Ptr_, Ptr_->_MismatchedEndPair);

      fprintf(stdout, "*** Erased ***\n");
    }

    if (!Ptr_->Erased())
    {
      ShowTagStr(*os_, 0, Ptr_, Type_, "this");
      ShowTagStr(*os_, 0, Ptr_->_MismatchedEndPair, Type_, "mismatchedEnd");
    }
  }
}

/****************************************************************************/
const TagElementStates* StackNodeInfo::TagElementState(bool FromElmPtr_) const
{
  return ((FromElmPtr_ && _ElementPtr) ? _ElementPtr->TagElementState():_TagStates);
}

/****************************************************************************/
TagElementStates* StackNodeInfo::GiveTagElementState(bool FromElmPtr_)
{
  return ((FromElmPtr_ && _ElementPtr) ? _ElementPtr->GiveTagElementState():_TagStates);
}

/****************************************************************************/
const TagElementStates* StackNodeInfo::SourceTagElementState(bool FromElmPtr_) const
{
  return ((FromElmPtr_ && _ElementPtr) ? _ElementPtr->SourceTagElementState():_TagStates);
}

/****************************************************************************/
TagElementStates* StackNodeInfo::GiveSourceTagElementState(bool FromElmPtr_)
{
  return ((FromElmPtr_ && _ElementPtr) ? _ElementPtr->GiveSourceTagElementState():_TagStates);
}

/****************************************************************************/
void StackNodeInfo::CopyFromTagElement(TagElement* Ptr_)
{
  if (Ptr_)
    ::memmove(_TagAttrib, Ptr_->TagAttribPtr(), sizeof(int) * TagEnums::MAX_FIELDS);
}

/****************************************************************************/
void StackNodeInfo::CopyFromTagSearchResult(TagSearchResult* Ptr_)
{
  if (Ptr_)
  {
    size_t x;
    size_t max;

    _NumDelims = Ptr_->DelimsFound();
    _StartBrkFound = Ptr_->StartBrkFound();
    _EndBrkFound = Ptr_->EndBrkFound();
    _TagElementFound = Ptr_->TagElementFound();
    _StartBrkStr = Ptr_->GiveTargetBrkString(NULL, NULL, true);
    _TagElementStr = Ptr_->GiveTagElementString();
    _EndBrkStr = Ptr_->GiveMatchingBrkString(NULL, NULL, true);
    _ContentType = Ptr_->ContentType();

    if (!_TagKVArrayCopied && _NumKeyValuePairs)
    {
      max = _NumKeyValuePairs;
      for (x = 0; x < max; x++)
      {
        delete _TagKeyValuePairs[x];
        _TagKeyValuePairs[x] = NULL;
      }

      ::RawDeleteArray(_TagKeyValuePairs);
      _NumKeyValuePairs = 0;
    }

    _NumKeyValuePairs = Ptr_->NumKeyValuePairs();
    _HasTagContents = Ptr_->HasTagContentString();
    _HasIntraTagContents = Ptr_->HasIntraTagContentString();

    if (_HasTagContents && Ptr_ == _Parent->SearchResult())
      _ContentStr = GiveTagContentFromSearch(this);
    else
      _ContentStr.SetEmpty();

    if (_HasIntraTagContents && Ptr_ == _Parent->SearchResult())
      _IntraContentStr = GiveIntraTagContentFromSearch(this);
    else
      _IntraContentStr.SetEmpty();

    Ptr_->Parent()->IndexFromBrkFound(&_TagBrkIndex, NULL);
    Ptr_->Parent()->IndexFromBrkFound(NULL, &_EndBrkIndex);

    _TagKeyValuePairs = Ptr_->GiveTagKeyValueString(_NumKeyValuePairs);
    _TagKVArrayCopied = false;
    ResetContentShifted();

    if (!TagTypeInfo::ShowNewLines())
    {
      max = _NumKeyValuePairs;
      for (x = 0; x < max; x++)
      {
        _TagKeyValuePairs[x]->RemovePadding(" \t\n\r\f\v");
        SubstChar(_TagKeyValuePairs[x]->c_str(), "\n\r", ' ');
        _TagKeyValuePairs[x]->SimplifySpaces(false);
      }

      _IntraContentStr.RemovePadding(" \t\n\r\f\v");
      SubstChar(_IntraContentStr.c_str(), "\n\r", ' ');
      _IntraContentStr.SimplifySpaces(false);

      _ContentStr.RemovePadding(" \t\n\r\f\v");
      SubstChar(_ContentStr.c_str(), "\n\r", ' ');
      _ContentStr.SimplifySpaces(false);
    }

    ParseKVPairCommentData(Ptr_);
  }
}

/****************************************************************************/
void StackNodeInfo::ParseKVPairCommentData(TagSearchResult* Ptr_)
{
  bool CmmtCond_ = InComment() || InNestedComment() || InScript();
  bool KVPCond_ = _NumKeyValuePairs && _TagKeyValuePairs;

  if (!CmmtCond_ && _HasCommentData)
  {
    _HasCommentData = false;

    delete _CommentDataPtr;
    _CommentDataPtr = NULL;
  }

  if (!KVPCond_ && _HasKVPairData)
  {
    _HasKVPairData = false;

    delete _KVPairDataPtr;
    _KVPairDataPtr = NULL;
  }

  if (CmmtCond_)
  {
    _HasCommentData = true;
    _CommentDataPtr = new TagCommentData(NULL);

    _CommentDataPtr->SetSearchResult(Ptr_);
    _CommentDataPtr->SetShowCommentDataDump(false);
    _CommentDataPtr->ParseComment();

    if (!_CommentDataPtr->CommentFound())
    {
      _HasCommentData = false;

      delete _CommentDataPtr;
      _CommentDataPtr = NULL;
    }
  }
  else if (KVPCond_)
  {
    _HasKVPairData = true;
    _KVPairDataPtr = new KeyValuePairData(NULL, new ChrString("="),
                                                new ChrString(" \t"),
                                                new ChrString(">"), true);

    _KVPairDataPtr->SetSearchResult(Ptr_);
    _KVPairDataPtr->SetShowKeyValuePairDump(false);
    _KVPairDataPtr->ParseKVpair();

    if (_KVPairDataPtr->NumPairs() == 0)
    {
      _HasKVPairData = false;

      delete _KVPairDataPtr;
      _KVPairDataPtr = NULL;
    }
  }
}

/****************************************************************************/
void StackNodeInfo::SetFromParameters(int DocType_,
                                      bool GenSearchDone_, bool SpecSearchDone_)
{
  _DocType = DocType_;
  _SearchType = 0;

  if (GenSearchDone_)
    _SearchType = TagEnums::GENERAL_SEARCH;

  if (SpecSearchDone_)
    _SearchType = TagEnums::SPECIFIC_SEARCH;
}

/****************************************************************************/
void StackNodeInfo::CopyFromTagTypeInfo(TagTypeInfo* Ptr_)
{
  if (Ptr_)
  {
    _Parent = Ptr_;
    _SearchRun = Ptr_->SearchRunNumber();
    _ElementPtr = Ptr_->GiveTagElementPtr();
    _TagAttr = Ptr_->TagAttribute();
    _TagBrkIndex = Ptr_->GiveTagBrkIndex();
    _EndBrkIndex = _TagBrkIndex + 1;
    _AltIndex = Ptr_->GiveAltIndex(true);

    _StdTagCount = Ptr_->StdTagCount();
    _NestedCmntTagCount = Ptr_->NestedCmntTagCount();
    _CopenCmntTagCount = Ptr_->CopenCmntTagCount();
    _NestingError = Ptr_->NestingError();
    _TagNestCond = Ptr_->TagNestCondition();

    _TagFound = Ptr_->TagFound();
    _ContentTypeTag = Ptr_->IsContentTypeTag();
    _NonContentTypeTag = Ptr_->IsNonContentTypeTag();
    _HasPairContent = Ptr_->HasPairContent();
    _TagTypeFound = Ptr_->TagTypeFound();

    _SingletonTagFound = Ptr_->_SingletonTagFound;
    _EmptyTagFound = Ptr_->_EmptyTagFound;
    _ContainerTagFound = Ptr_->_ContainerTagFound;
    _StdTagFound = Ptr_->_StdTagFound;
    _PairEnderFound = Ptr_->_PairEnderFound;

    bool ContentTag_ = _StdTagFound && !_SingletonTagFound && !_EmptyTagFound;
    _EndImplied = !ContentTag_ && _TagTypeFound != TagEnums::END_TAG;
    _TagAppendDone = false;
  }
}

/****************************************************************************/
void StackNodeInfo::CopyStackNodeDataFromPtr(StackNodeInfo* Ptr_)
{
  if (Ptr_)
  {
    _TagNestCond = Ptr_->_TagNestCond;
    _TagContValid = Ptr_->_TagContValid;
    _NestingError = Ptr_->_NestingError;
  }
}

/****************************************************************************/
void StackNodeInfo::SetStkNodeSearchDataFromPtr(StackNodeInfo* Ptr_)
{
  if (Ptr_)
  {
    _CursorStkTag = Ptr_->_CursorStkTag;
    _FoundTag = Ptr_->_FoundTag;
    _FoundCursorTag = Ptr_->_FoundCursorTag;
    _StkTagFound = Ptr_->_StkTagFound;
  }
}

/****************************************************************************/
void StackNodeInfo::CopyTagTypeStackNodeData(TagTypeInfo* Ptr_)
{
  if (Ptr_)
  {
    _TagNestCond = Ptr_->Sequencer()->_TagNestCond;
    _TagContValid = Ptr_->_TagContValid;
    _NestingError = Ptr_->Sequencer()->_NestingError;
  }
}

/****************************************************************************/
void StackNodeInfo::SetStkNodeSearchData(TagTypeInfo* Ptr_)
{
  if (Ptr_)
  {
    _CursorStkTag = Ptr_->Sequencer()->_CursorStkTag;
    _FoundTag = Ptr_->Sequencer()->_FoundTag;
    _FoundCursorTag = Ptr_->Sequencer()->_FoundCursorTag;
    _StkTagFound = Ptr_->Sequencer()->_StkTagFound;
  }
}

/****************************************************************************/
void StackNodeInfo::AssignGlobalStackTag()
{
  if (_Parent)
  {
    _Parent->Sequencer()->_CursorStkTag = _CursorStkTag;
    _Parent->Sequencer()->AssignGlobalStackTag();
  }
}

/****************************************************************************/
int StackNodeInfo::ClassType() const
{
  return QListNode::STACKNODEINFO;
}

/****************************************************************************/
QListNode& StackNodeInfo::operator = (const QListNode& List_)
{
  if (this != &List_)
  {
    size_t x;
    size_t max;

    QListNode::operator = (List_);
    StackNodeInfo* ListPtr_ = (List_.ClassType() == QListNode::STACKNODEINFO) ?
                                 ((StackNodeInfo*)(&List_)):NULL;

    if (!ListPtr_)
      return *this;

    if (_NestedQhead)
    {
      NodeNestingData::CopyNodeNestingData(_NestedQ, _NestedQhead, _NumNestedData,
                                           ListPtr_->_NestedQ, ListPtr_->_NestedQhead, ListPtr_->_NumNestedData);
      _NestedQptr = _NestedQhead;
    }

    _HasCommentData = ListPtr_->_HasCommentData;
    _HasKVPairData = ListPtr_->_HasKVPairData;

    if (_HasCommentData)
      _CommentDataPtr = ListPtr_->_CommentDataPtr;

    if (_HasKVPairData)
      _KVPairDataPtr = ListPtr_->_KVPairDataPtr;

    // Set From Node Creation Parameters
    _DocType = ListPtr_->_DocType;
    _SearchType = ListPtr_->_SearchType;

    // from TagSearchResult Class
    _NumDelims = ListPtr_->_NumDelims;
    _StartBrkFound = ListPtr_->_StartBrkFound;
    _EndBrkFound = ListPtr_->_EndBrkFound;
    _TagElementFound = ListPtr_->_TagElementFound;
    _StartBrkStr = ListPtr_->_StartBrkStr;
    _TagElementStr = ListPtr_->_TagElementStr;
    _EndBrkStr = ListPtr_->_EndBrkStr;
    _ContentType = ListPtr_->_ContentType;

    if (!_TagKVArrayCopied && _NumKeyValuePairs)
    {
      max = _NumKeyValuePairs;
      for (x = 0; x < max; x++)
      {
        delete _TagKeyValuePairs[x];
        _TagKeyValuePairs[x] = NULL;
      }

      ::RawDeleteArray(_TagKeyValuePairs);
      _NumKeyValuePairs = 0;
    }

    _HasTagContents = ListPtr_->_HasTagContents;
    _HasIntraTagContents = ListPtr_->_HasIntraTagContents;

    if (_HasTagContents)
      _ContentStr = ListPtr_->_ContentStr;
    else
      _ContentStr.SetEmpty();

    if (_HasIntraTagContents)
      _IntraContentStr = ListPtr_->_IntraContentStr;
    else
      _IntraContentStr.SetEmpty();

    _NumKeyValuePairs = ListPtr_->_NumKeyValuePairs;
    _TagKeyValuePairs = ListPtr_->_TagKeyValuePairs;
    _TagKVArrayCopied = true;
    _ContentShifted = 0;

    // from TagTypeInfo Class
    _Parent = ListPtr_->_Parent;
    _SearchRun = ListPtr_->_SearchRun;
    _ElementPtr = ListPtr_->_ElementPtr;
    _TagAttr = ListPtr_->_TagAttr;
    _TagBrkIndex = ListPtr_->_TagBrkIndex;
    _EndBrkIndex = ListPtr_->_EndBrkIndex;
    _AltIndex = ListPtr_->_AltIndex;

    _StdTagCount = ListPtr_->_StdTagCount;
    _NestedCmntTagCount = ListPtr_->_NestedCmntTagCount;
    _CopenCmntTagCount = ListPtr_->_CopenCmntTagCount;
    _NestingError = ListPtr_->_NestingError;
    _TagNestCond = ListPtr_->_TagNestCond;

    _TagFound = ListPtr_->_TagFound;
    _ContentTypeTag = ListPtr_->_ContentTypeTag;
    _NonContentTypeTag = ListPtr_->_NonContentTypeTag;
    _HasPairContent = ListPtr_->_HasPairContent;
    _TagTypeFound = ListPtr_->_TagTypeFound;
    _LinkType = ListPtr_->_LinkType;

    _SingletonTagFound = ListPtr_->_SingletonTagFound;
    _EmptyTagFound = ListPtr_->_EmptyTagFound;
    _ContainerTagFound = ListPtr_->_ContainerTagFound;
    _StdTagFound = ListPtr_->_StdTagFound;
    _PairEnderFound = ListPtr_->_PairEnderFound;

    // StackNodeInfo class data members
    SetPrev(NULL);
    SetNext(NULL);
    _BackLink = ListPtr_->_BackLink;
    _ForwardLink = ListPtr_->_ForwardLink;
    _PrevCursor = NULL;
    _NextCursor = NULL;

    _OverlappedPrevPair = ListPtr_->_OverlappedPrevPair;
    _OverlappedNextPair = ListPtr_->_OverlappedNextPair;
    _MismatchedEndPair = ListPtr_->_MismatchedEndPair;

    _EndImplied = ListPtr_->_EndImplied;
    _TagAppendDone = ListPtr_->_TagAppendDone;
    _Copied = true;
    _Erased = ListPtr_->_Erased;
    _Unlinked = ListPtr_->_Unlinked;
    _CursorStkNodeFound = ListPtr_->_CursorStkNodeFound;
    _NodeErased = ListPtr_->_NodeErased;

    // from TagElement Class
    ::memmove(_TagAttrib, ListPtr_->_TagAttrib, sizeof(int) * TagEnums::MAX_FIELDS);

    if (!TagTypeInfo::ShowNewLines())
    {
      max = _NumKeyValuePairs;
      for (x = 0; x < max; x++)
      {
        _TagKeyValuePairs[x]->RemovePadding(" \t\n\r\f\v");
        SubstChar(_TagKeyValuePairs[x]->c_str(), "\n\r", ' ');
        _TagKeyValuePairs[x]->SimplifySpaces(false);
      }

      _IntraContentStr.RemovePadding(" \t\n\r\f\v");
      SubstChar(_IntraContentStr.c_str(), "\n\r", ' ');
      _IntraContentStr.SimplifySpaces(false);

      _ContentStr.RemovePadding(" \t\n\r\f\v");
      SubstChar(_ContentStr.c_str(), "\n\r", ' ');
      _ContentStr.SimplifySpaces(false);
    }
  }

  return *this;
}

/****************************************************************************/
QListNode* StackNodeInfo::Head(int Type_)
{
  StackNodeInfo* Ptr_ = this;
  if (!Type_ || Type_ == PREVNEXT)
  {
    while (Ptr_ && Ptr_->_Prev)
      Ptr_ = Ptr_->_Prev;
  }
  else if (Type_ == NEXTCUR)
  {
    while (Ptr_ && Ptr_->_PrevCursor)
      Ptr_ = Ptr_->_PrevCursor;
  }

  return Ptr_;
}

/****************************************************************************/
const QListNode* StackNodeInfo::Head(int Type_) const
{
  const StackNodeInfo* Ptr_ = this;
  if (!Type_ || Type_ == PREVNEXT)
  {
    while (Ptr_ && Ptr_->_Prev)
      Ptr_ = Ptr_->_Prev;
  }
  else if (Type_ == NEXTCUR)
  {
    while (Ptr_ && Ptr_->_PrevCursor)
      Ptr_ = Ptr_->_PrevCursor;
  }

  return Ptr_;
}

/****************************************************************************/
QListNode* StackNodeInfo::Tail(int Type_)
{
  StackNodeInfo* Ptr_ = this;
  if (!Type_ || Type_ == PREVNEXT)
  {
    while (Ptr_ && Ptr_->_Next)
      Ptr_ = Ptr_->_Next;
  }
  else if (Type_ == NEXTCUR)
  {
    while (Ptr_ && Ptr_->_NextCursor)
      Ptr_ = Ptr_->_NextCursor;
  }

  return Ptr_;
}

/****************************************************************************/
const QListNode* StackNodeInfo::Tail(int Type_) const
{
  const StackNodeInfo* Ptr_ = this;
  if (!Type_ || Type_ == PREVNEXT)
  {
    while (Ptr_ && Ptr_->_Next)
      Ptr_ = Ptr_->_Next;
  }
  else if (Type_ == NEXTCUR)
  {
    while (Ptr_ && Ptr_->_NextCursor)
      Ptr_ = Ptr_->_NextCursor;
  }

  return Ptr_;
}

/****************************************************************************/
// Algorithm:
// <OpSt1> ... (-Stop Append) <St2> ... <End2> (+Append)
//     St-->Me
//     Continue appending tentative string of tag <OpSt1> after <End2>
//     <St2> preceeds <OpSt1>, <OpSt1> is optional tag
//     Link Structure:
//       <OpSt1:Prev> --> <St2:Prev> --> <End2> ...
//       ... <OpSt1> <-- <Next:St2> <-- <Next:End2>
//
// <St1> ... <OpSt2> (+Append) ... <End1> (-Stop Append)
//     OpSt-->Ume
//     Tentative string of <OpSt2> and preceeding tags until <End1> removed
//     <St1> follows <OpSt2>, <OpSt2> is optional tag
//     Link Structure:
//       <St1:Prev> --> <OpSt2:Prev> --> <End1> ...
//       ... <St1> <-- <Next:OpSt2> <-- <Next:End1>
//
void StackNodeInfo::RemovePrevTentStr()
{
  // remove previous tentative strings for optional tag stack nodes
  // that follows the NodeID of the current stack node

  StackNodeInfo* Node_;
  StackNodeInfo* EndPt_;
  StackNodeInfo* InfoLink_ = this;
  StackNodeInfo* MatchInfo_ = NULL;

  if (_PairEnderFound && _BackLink)
  {
    MatchInfo_ = InfoLink_->GiveIfExistMatchingInfo();
    InfoLink_ = InfoLink_->GiveIfExistMatchingInfo();
    EndPt_ = MatchInfo_;

    if (InfoLink_->_BackLink)
      for (Node_ = InfoLink_->_BackLink->_Prev; Node_ && Node_ != EndPt_;
           Node_ = Node_->_Prev)
        if (Node_->_TextNode && Node_->_TextNode->HasTentativeStr())
          Node_->_TextNode->RemoveTentativeNode();
  }
  else if (_StdTagFound && _ForwardLink)
  {
    MatchInfo_ = _ForwardLink->GiveIfExistMatchingInfo();
    InfoLink_ = InfoLink_->GiveIfExistMatchingInfo();
    EndPt_ = MatchInfo_;

    if (InfoLink_->_Prev)
      for (Node_ = InfoLink_->_Prev; Node_ && Node_ != EndPt_;
           Node_ = Node_->_Prev)
        if (Node_->_TextNode && Node_->_TextNode->HasTentativeStr())
          Node_->_TextNode->RemoveTentativeNode();
  }
}

/****************************************************************************/
QListNode* StackNodeInfo::RemoveHead(QListNode*& Head_)
{
  if (Head_ && Head_->ClassType() == QListNode::STACKNODEINFO)
  {
    StackNodeInfo* Ptr_ = (StackNodeInfo*)Head_;
    Head_ = Head_->GiveNext();
    Ptr_->SetNext(NULL);
    Head_->SetPrev(NULL);
    Ptr_->SetPrev(NULL);
    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::RemoveHeadCursor(StackNodeInfo*& Head_)
{
  if (Head_)
  {
    StackNodeInfo* Ptr_ = Head_;
    Head_ = Head_->_NextCursor;
    Ptr_->_NextCursor = NULL;
    Head_->_PrevCursor = NULL;
    Ptr_->_PrevCursor = NULL;
    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
// Algorithm:
// <OpSt1> ... (-Stop Append) <St2> ... <End2> (+Append)
//     St-->Me
//     Continue appending tentative string of tag <OpSt1> after <End2>
//     <St2> preceeds <OpSt1>, <OpSt1> is optional tag
//     Link Structure:
//       <OpSt1:Prev> --> <St2:Prev> --> <End2> ...
//       ... <OpSt1> <-- <Next:St2> <-- <Next:End2>
//
// <St1> ... <OpSt2> (+Append) ... <End1> (-Stop Append)
//     OpSt-->Ume
//     Tentative string of <OpSt2> and preceeding tags until <End1> removed
//     <St1> follows <OpSt2>, <OpSt2> is optional tag
//     Link Structure:
//       <St1:Prev> --> <OpSt2:Prev> --> <End1> ...
//       ... <St1> <-- <Next:OpSt2> <-- <Next:End1>
//
void StackNodeInfo::ResumeTextAppend(StackNodeInfo* Ptr_)
{
  // resume text appending for the current tag after new end tag
  // is encountered

  StackNodeInfo* PrevNode_ = Ptr_ ? ((Ptr_ == this) ? Ptr_->_Prev:Ptr_):NULL;
  StackNodeInfo* InfoLink_ = PrevNode_ ? PrevNode_:NULL;

  StackNodeInfo* TagStart_ = PrevNode_ ? PrevNode_->_BackLink:NULL;
  StackNodeInfo* TagEnd_ = PrevNode_ ? PrevNode_->_ForwardLink:NULL;
  StackNodeInfo* MatchInfo_ = NULL;
  StackNodeInfo* SavedPtr_ = NULL;
  bool Matched_ = false;

  if (Ptr_ && Ptr_ != this)
    Ptr_ = this;

  MatchInfo_ = Ptr_->GiveIfExistMatchingInfo();

  if (PrevNode_ && Ptr_ != PrevNode_ && MatchInfo_->_NestedQhead)
  {
    if (InfoLink_)
    {
      if (InfoLink_->_PairEnderFound && TagStart_)
      {
        SavedPtr_ = Ptr_ = MatchInfo_;
        MatchInfo_ = TagStart_ ? TagStart_->GiveIfExistMatchingInfo():NULL;
        InfoLink_ = InfoLink_->GiveIfExistMatchingInfo();

        if (!Matched_)
        while (Ptr_ && Ptr_->Follows(MatchInfo_, true))
        {
          Matched_ = Ptr_->_NestedQhead->StartTag() == MatchInfo_;

          if (Matched_)
          {
            Ptr_->_NestedQhead->SetThisTag(Ptr_);
            Ptr_->_NestedQhead->SetEndTag(InfoLink_);

            if (Ptr_->_TextNode && !Ptr_->_TextNode->TextAppendResumed())
              Ptr_->_TextNode->ResumeTextAppend(InfoLink_);

            break;
          }
          else
            Ptr_ = Ptr_->_Next;
        }

        Ptr_ = SavedPtr_;

        if (!Matched_)
        while (Ptr_ && Ptr_->Preceeds(MatchInfo_, true))
        {
          Matched_ = Ptr_->_NestedQhead->StartTag() == MatchInfo_;

          if (Matched_)
          {
            Ptr_->_NestedQhead->SetThisTag(Ptr_);
            Ptr_->_NestedQhead->SetEndTag(InfoLink_);

            if (Ptr_->_TextNode && !Ptr_->_TextNode->TextAppendResumed())
              Ptr_->_TextNode->ResumeTextAppend(InfoLink_);

            break;
          }
          else
            Ptr_ = Ptr_->_Prev;
        }
      }
      else if (InfoLink_->_StdTagFound && TagEnd_)
      {
        SavedPtr_ = Ptr_ = MatchInfo_;
        MatchInfo_ = InfoLink_ ? InfoLink_->GiveIfExistMatchingInfo():NULL;
        TagEnd_ = TagEnd_->GiveIfExistMatchingInfo();

        if (!Matched_)
        while (Ptr_ && Ptr_->Follows(MatchInfo_, true))
        {
          Matched_ = Ptr_->_NestedQhead->StartTag() == MatchInfo_;

          if (Matched_)
          {
            Ptr_->_NestedQhead->SetThisTag(Ptr_);
            Ptr_->_NestedQhead->SetEndTag(TagEnd_);

            if (Ptr_->_TextNode && !Ptr_->_TextNode->TextAppendResumed())
              Ptr_->_TextNode->ResumeTextAppend(TagEnd_);

            break;
          }
          else
            Ptr_ = Ptr_->_Next;
        }

        Ptr_ = SavedPtr_;

        if (!Matched_)
        while (Ptr_ && Ptr_->Preceeds(MatchInfo_, true))
        {
          Matched_ = Ptr_->_NestedQhead->StartTag() == MatchInfo_;

          if (Matched_)
          {
            Ptr_->_NestedQhead->SetThisTag(Ptr_);
            Ptr_->_NestedQhead->SetEndTag(TagEnd_);

            if (Ptr_->_TextNode && !Ptr_->_TextNode->TextAppendResumed())
              Ptr_->_TextNode->ResumeTextAppend(TagEnd_);

            break;
          }
          else
            Ptr_ = Ptr_->_Prev;
        }
      }
    }
    else
    {
      if (PrevNode_->_PairEnderFound && TagStart_)
      {
        SavedPtr_ = Ptr_ = MatchInfo_;
        MatchInfo_ = TagStart_ ? TagStart_->GiveIfExistMatchingInfo():NULL;
        PrevNode_ = PrevNode_->GiveIfExistMatchingInfo();

        if (!Matched_)
        while (Ptr_ && Ptr_->Follows(MatchInfo_, true))
        {
          Matched_ = Ptr_->_NestedQhead->StartTag() == MatchInfo_;

          if (Matched_)
          {
            Ptr_->_NestedQhead->SetThisTag(Ptr_);
            Ptr_->_NestedQhead->SetEndTag(PrevNode_);

            if (Ptr_->_TextNode && !Ptr_->_TextNode->TextAppendResumed())
              Ptr_->_TextNode->ResumeTextAppend(PrevNode_);

            break;
          }
          else
            Ptr_ = Ptr_->_Next;
        }

        Ptr_ = SavedPtr_;

        if (!Matched_)
        while (Ptr_ && Ptr_->Preceeds(MatchInfo_, true))
        {
          Matched_ = Ptr_->_NestedQhead->StartTag() == MatchInfo_;

          if (Matched_)
          {
            Ptr_->_NestedQhead->SetThisTag(Ptr_);
            Ptr_->_NestedQhead->SetEndTag(PrevNode_);

            if (Ptr_->_TextNode && !Ptr_->_TextNode->TextAppendResumed())
              Ptr_->_TextNode->ResumeTextAppend(PrevNode_);

            break;
          }
          else
            Ptr_ = Ptr_->_Prev;
        }
      }
      else if (PrevNode_->_StdTagFound && TagEnd_)
      {
        SavedPtr_ = Ptr_ = MatchInfo_;
        MatchInfo_ = PrevNode_ ? PrevNode_->GiveIfExistMatchingInfo():NULL;
        TagEnd_ = TagEnd_->GiveIfExistMatchingInfo();

        if (!Matched_)
        while (Ptr_ && Ptr_->Follows(MatchInfo_, true))
        {
          Matched_ = Ptr_->_NestedQhead->StartTag() == MatchInfo_;

          if (Matched_)
          {
            Ptr_->_NestedQhead->SetThisTag(Ptr_);
            Ptr_->_NestedQhead->SetEndTag(TagEnd_);

            if (Ptr_->_TextNode && !Ptr_->_TextNode->TextAppendResumed())
              Ptr_->_TextNode->ResumeTextAppend(TagEnd_);

            break;
          }
          else
            Ptr_ = Ptr_->_Next;
        }

        Ptr_ = SavedPtr_;

        if (!Matched_)
        while (Ptr_ && Ptr_->Preceeds(MatchInfo_, true))
        {
          Matched_ = Ptr_->_NestedQhead->StartTag() == MatchInfo_;

          if (Matched_)
          {
            Ptr_->_NestedQhead->SetThisTag(Ptr_);
            Ptr_->_NestedQhead->SetEndTag(TagEnd_);

            if (Ptr_->_TextNode && !Ptr_->_TextNode->TextAppendResumed())
              Ptr_->_TextNode->ResumeTextAppend(TagEnd_);

            break;
          }
          else
            Ptr_ = Ptr_->_Prev;
        }
      }
    }
  }
}

/****************************************************************************/
// Algorithm:
// <OpSt1> ... (-Stop Append) <St2> ... <End2> (+Append)
//     St-->Me
//     Continue appending tentative string of tag <OpSt1> after <End2>
//     <St2> preceeds <OpSt1>, <OpSt1> is optional tag
//     Link Structure:
//       <OpSt1:Prev> --> <St2:Prev> --> <End2> ...
//       ... <OpSt1> <-- <Next:St2> <-- <Next:End2>
//
// <St1> ... <OpSt2> (+Append) ... <End1> (-Stop Append)
//     OpSt-->Ume
//     Tentative string of <OpSt2> and preceeding tags until <End1> removed
//     <St1> follows <OpSt2>, <OpSt2> is optional tag
//     Link Structure:
//       <St1:Prev> --> <OpSt2:Prev> --> <End1> ...
//       ... <St1> <-- <Next:OpSt2> <-- <Next:End1>
//
void StackNodeInfo::PauseTextAppend(StackNodeInfo* Ptr_)
{
  // pause text appending for the current tag after new standard or
  // optional start tag is encountered

  StackNodeInfo* PrevNode_ = Ptr_ ? ((Ptr_ == this) ? Ptr_->_Prev:Ptr_):NULL;
  StackNodeInfo* InfoLink_ = PrevNode_ ? PrevNode_:NULL;
  StackNodeInfo* MatchInfo_ = NULL;
  bool TagCreated_ = false;

  if (Ptr_ && Ptr_ != this)
    Ptr_ = this;

  MatchInfo_ = Ptr_->GiveIfExistMatchingInfo();

  if (PrevNode_ && Ptr_ != PrevNode_ && MatchInfo_->_NestedQhead)
  {
    Ptr_ = MatchInfo_;

    if (InfoLink_)
    {
      if (Ptr_->_NestedQhead->GiveStartTag())
      {
        Ptr_->CreateNestingNodeData();
        TagCreated_ = _NestedQhead != NULL;
      }
      else if (Ptr_->_NestedQhead && _NumNestedData)
        TagCreated_ = true;

      MatchInfo_ = InfoLink_ ? InfoLink_->GiveIfExistMatchingInfo():NULL;

      if (TagCreated_ && MatchInfo_)
      {
        MatchInfo_->PushNestingNode(NodeNestingData::CopyNode(NULL, _NestedQhead), false);
        Ptr_->_NestedQhead->SetThisTag(Ptr_);
        Ptr_->_NestedQhead->SetStartTag(MatchInfo_);
        MatchInfo_->_NestedQhead->SetThisTag(MatchInfo_);
        MatchInfo_->_NestedQhead->SetTagParent(Ptr_);

        if (Ptr_->_TextNode && !Ptr_->_TextNode->TextAppendPaused())
          Ptr_->_TextNode->PauseTextAppend(MatchInfo_);
      }
    }
    else
    {
      if (Ptr_->_NestedQhead->GiveStartTag())
      {
        Ptr_->CreateNestingNodeData();
        TagCreated_ = _NestedQhead != NULL;
      }
      else if (Ptr_->_NestedQhead && _NumNestedData)
        TagCreated_ = true;

      MatchInfo_ = PrevNode_ ? PrevNode_->GiveIfExistMatchingInfo():NULL;

      if (TagCreated_ && MatchInfo_)
      {
        MatchInfo_->PushNestingNode(NodeNestingData::CopyNode(NULL, _NestedQhead), false);
        Ptr_->_NestedQhead->SetThisTag(Ptr_);
        Ptr_->_NestedQhead->SetStartTag(MatchInfo_);
        MatchInfo_->_NestedQhead->SetThisTag(MatchInfo_);
        MatchInfo_->_NestedQhead->SetTagParent(Ptr_);

        if (Ptr_->_TextNode && !Ptr_->_TextNode->TextAppendPaused())
          Ptr_->_TextNode->PauseTextAppend(MatchInfo_);
      }
    }
  }
}

/****************************************************************************/
QListNode* StackNodeInfo::AppendHead(QListNode* Ptr_, bool NestTag_)
{
  if (Ptr_ && Ptr_ != this && Ptr_->ClassType() == QListNode::STACKNODEINFO)
  {
    // Link Structure:
    //   <Ptr_:Prev> --> <this:Prev> --> <SomeTag> ...
    //   ... <Ptr_> <-- <Next:this> <-- <Next:SomeTag>
    //

    Ptr_->SetNext(this);
    if (Next() == (StackNodeInfo*)Ptr_)
      SetNext(NULL);

    SetPrev(Ptr_);
    Ptr_->SetPrev(NULL);

    if (NestTag_)
      PauseTextAppend((StackNodeInfo*)Ptr_);

    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::AppendHeadCursor(StackNodeInfo* Ptr_)
{
  if (Ptr_ && Ptr_ != this)
  {
    Ptr_->_NextCursor = this;
    if (_NextCursor == Ptr_)
      _NextCursor = NULL;

    _PrevCursor = Ptr_;
    Ptr_->_PrevCursor = NULL;
    return Ptr_;
  }

  return NULL;
}

/****************************************************************************/
// example 1:
// ----------
// a:<StdTagType1> d:-1,0,+3;b:+1,+2,+1
//      content for 1a   // tag content
//   b:<StdTagType2> d:-1,-1,+2;c:-1,0,+1;b:+1,+1,0
//      content for 2    // tag content
//   c:</StdTagType2> d:-1,-2,+1;c:-1,-1,0
//      content for 1b   // tag content
// d:</StdTagType1> d:-1,-1,0
//
// <StdTagType2>._Content._Str == "content for 1a"   // belongs to <StdTagType1>
//
// </StdTagType2>._Content._Str == "content for 2"   // belongs to <StdTagType2>
//
// </StdTagType1>._Content._Str == "content for 1b"  // belongs to <StdTagType1>
//
// example 2:
// ----------
// a:<StdTagType1> f:-1,0,+5;b:+1,+2,+1
//      content for 1a       // tag content
//   b:<StdTagType2> f:-1,-1,+4;e:-1,0,+3;c:+1,+2,+1;b:+1,+1,0
//        content for 2a     // tag content
//     c:<StdTagType3> f:-1,-2,+3;e:-1,-1,+2;d:-1,0,+1;c:+1,+1,0
//          content for 3    // tag content
//     d:</StdTagType3> f:-1,-3,+2;e:-1,-2,+1;d:-1,-1,0
//        content for 2b     // tag content
//   e:</StdTagType2> f:-1,-2,+1;e:-1,-1,0
//      content for 1b       // tag content
// f:</StdTagType1> f:-1,-1,0
//
// <StdTagType2>._Content._Str == "content for 1a"   // belongs to <StdTagType1>
//
// <StdTagType3>._Content._Str == "content for 2a"   // belongs to <StdTagType2>
//
// </StdTagType3>._Content._Str == "content for 3"   // belongs to <StdTagType3>
//
// </StdTagType2>._Content._Str == "content for 2b"  // belongs to <StdTagType2>
//
// </StdTagType1>._Content._Str == "content for 1b"  // belongs to <StdTagType1>
//
// example 3:
// ----------
// a:<StdTagType1> f:-1,0,+5;d:+1,+2;b:+1,+2,+1
//      content for 1a   // tag content
//   b:<StdTagType2> f:-1,-1,+4;d:+1,+1,+2;c:-1,0,+1;b:+1,+1,0
//      content for 2    // tag content
//   c:</StdTagType2> f:-1,-2,+3;d:+1,0,+1;c:-1,-1,0
//      content for 1b   // tag content
//   d:<StdTagType3> f:-1,-1,+2;e:-1,0,+1;d:+1,+1,0
//      content for 3    // tag content
//   e:</StdTagType3> f:-1,-2,+1;e:-1,-1,0
//      content for 1c   // tag content
// f:</StdTagType1> f:-1,-1,0;
//
// <StdTagType2>._Content._Str == "content for 1a"   // belongs to <StdTagType1>
//
// </StdTagType2>._Content._Str == "content for 2"   // belongs to <StdTagType2>
//
// <StdTagType3>._Content._Str == "content for 1b"   // belongs to <StdTagType1>
//
// </StdTagType3>._Content._Str == "content for 3"   // belongs to <StdTagType3>
//
// </StdTagType1>._Content._Str == "content for 1c"  // belongs to <StdTagType1>
//
// example 4:
// ----------
// a:<StdTagType> b:0,+1,+1
//      content for 1a   // tag content
//   b:<EmptyTagType/> c:0,+1,+1;b:0,0,0
//      content for 1b   // tag content
//   c:<SingleTagType> d:-1,0,+1;c:0,0,0
//      content for 1c   // tag content
// d:</StdTagType> d:-1,-1,0
//
// <EmptyTagType/>._Content._Str == "content for 1a"  // belongs to <StdTagType>
//
// <SingleTagType>._Content._Str == "content for 1b"  // belongs to <EmptyTagType>
//
// </StdTagType>._Content._Str == "content for 1c"    // belongs to <SingleTagType>
//
// Code:
// -----
// Starting conditions for ShiftTagContents:
// If EndTag then
//     (TagEndDir_ == -1 && TagEndVal_ == -1 && Level_ == 0)
// If StartTag then
//     (TagEndDir_ == +1 && TagEndVal_ == +1 && Level_ == 0)
// If SingletonTag or EmptyTag then
//     (TagEndDir_ == 0 && TagEndVal_ == 0 && Level_ == 0)
//
// Stopping conditions for ShiftTagContents:
// If EndTag then
//     (TagEndDir_ == -1 && TagEndVal_ == 0 && Level_ > 0)
// If StartTag then
//     (TagEndDir_ == +1 && TagEndVal_ == +2 && Level_ > 0)
// If SingletonTag or EmptyTag then
//     (TagEndDir_ == 0 && TagEndVal_ == +1 && Level_ > 0)
//
// if (HeadNode_)
//   HeadNode_->ShiftTagContents(HeadNode_->GiveNext(), TagEndDir_, TagEndVal_, 0);
//
StackNodeInfo* StackNodeInfo::ShiftTagContents(StackNodeInfo* EncNode_,
                                               int TagEndDir_, int TagEndVal_, int Level_)
{
  if (EncNode_ && EncNode_->_TagFound)
  {
    int Incr_ = EncNode_->StdTagFound() ? 1:
                EncNode_->PairEnderFound() ? -1:
                  (EncNode_->EmptyTagFound() ||
                   EncNode_->SingletonTagFound()) ? 1:0;
    TagEndVal_ += Incr_;
    ++Level_;

    bool Stop_ = (TagEndDir_ == 0 && TagEndVal_ == 1 && Level_ > 0) ||
                 (TagEndDir_ == 1 && TagEndVal_ == 2 && Level_ > 0) ||
                 (TagEndDir_ == -1 && TagEndVal_ == 0 && Level_ > 0);

    if (Stop_)
      return ShiftBackContents(EncNode_);
    else
      return EncNode_->ShiftTagContents((StackNodeInfo*)(EncNode_->GiveNext()),
                                         TagEndDir_, TagEndDir_, Level_);
  }

  return NULL;
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::ShiftBackContents(StackNodeInfo* EncNode_)
{
  if (EncNode_ && EncNode_ != this)
  {
    EncNode_->_ContentType |= _ContentType;
    EncNode_->_HasTagContents = EncNode_->_HasTagContents || _HasTagContents;
    EncNode_->_HasPairContent = EncNode_->_HasPairContent || _HasPairContent;
    int PrevStat_ = EncNode_->_ContentShifted;

    if (EncNode_->_HasTagContents)
    {
      if ((PrevStat_ & CONTENT_SHIFTED) ||
          (PrevStat_ & CONTENT_APPENDED))
      {
        EncNode_->_ContentStr += _ContentStr;

        if (!_ContentStr.IsEmpty())
        {
          EncNode_->ResetContentShifted();
          EncNode_->SetContentShifted();
          EncNode_->SetContentAppended();
        }
      }
      else
      {
        EncNode_->_ContentStr = _ContentStr;

        if (!_ContentStr.IsEmpty())
        {
          EncNode_->ResetContentShifted();
          EncNode_->SetContentShifted();
        }
      }
    }

    _ContentType = 0;
    _HasTagContents = false;
    _HasPairContent = false;
    _ContentStr.SetEmpty();
    ResetContentShifted();

    if (EncNode_->_Parent &&
        EncNode_->_ContentTypeTag && EncNode_->_HasTagContents)
    {
      ChrString RetStr_ = EncNode_->GiveTagContent();

      if (!RetStr_.IsEmpty())
      {
        if (EncNode_->_TextNode)
        {
          if ((PrevStat_ & CONTENT_SHIFTED) ||
              (PrevStat_ & CONTENT_APPENDED))
            EncNode_->_TextNode->AppendTextStr(&RetStr_);
          else
            EncNode_->_TextNode->SetTextStr(&RetStr_);
        }

        EncNode_->SetContentAppended();

        if (EncNode_->_TextNode && EncNode_->_Parent)
          DumpShiftBack(*EncNode_->GiveOutput(),
                        EncNode_->GiveFieldWidth(), EncNode_);
      }
    }
  }

  return EncNode_;
}

/****************************************************************************/
bool StackNodeInfo::InComment() const
{
  return
  (
    _TagTypeFound == TagEnums::COMMENT_TAG          ||
    _TagTypeFound == TagEnums::NESTEDCOMMENT1_TAG   ||
    _TagTypeFound == TagEnums::NESTEDCOMMENT2_TAG   ||
    _TagTypeFound == TagEnums::CCOMMENTBLK_TAG      ||
    _TagTypeFound == TagEnums::COPENCOMMENTBLK_TAG  ||
    _TagTypeFound == TagEnums::CCLOSECOMMENTBLK_TAG ||
    _TagTypeFound == TagEnums::CPPCOMMENTLINE_TAG
  );
}

/****************************************************************************/
bool StackNodeInfo::InNestedComment() const
{
  return
  (
    _TagTypeFound == TagEnums::NESTEDCOMMENT1_TAG   ||
    _TagTypeFound == TagEnums::NESTEDCOMMENT2_TAG
  );
}

/****************************************************************************/
bool StackNodeInfo::InScript() const
{
  return (_TagTypeFound == TagEnums::SCRIPT_TAG);
}

/****************************************************************************/
bool StackNodeInfo::IsDocTypeTag() const
{
  return (_TagTypeFound == TagEnums::DOCTYPE_TAG);
}

/****************************************************************************/
bool StackNodeInfo::IsEndTag() const
{
  return
  (
    PairEnderFound() ||
    _TagTypeFound == TagEnums::END_TAG
  );
}

/****************************************************************************/
bool StackNodeInfo::IsSingleton() const
{
  return
  (
    SingletonTagFound() ||
    EmptyTagFound() ||
    _TagTypeFound == TagEnums::EMPTY_TAG
  );
}

/****************************************************************************/
bool StackNodeInfo::IsStdTag() const
{
  return
  (
    !InComment() &&
    !InScript() &&
    !IsDocTypeTag() &&

    (_TagTypeFound == TagEnums::STD_TAG  ||
     _TagTypeFound == TagEnums::EMPTY_TAG ||
     _TagTypeFound == TagEnums::END_TAG)
  );
}

/****************************************************************************/
bool StackNodeInfo::IsCTag() const
{
  return
  (
    _TagTypeFound == TagEnums::CCOMMENTBLK_TAG      ||
    _TagTypeFound == TagEnums::COPENCOMMENTBLK_TAG  ||
    _TagTypeFound == TagEnums::CCLOSECOMMENTBLK_TAG
  );
}

/****************************************************************************/
bool StackNodeInfo::IsCppTag() const
{
  return (_TagTypeFound == TagEnums::CPPCOMMENTLINE_TAG);
}

/****************************************************************************/
void StackNodeInfo::SetContentShown() const
{
  _ContentShifted |= CONTENT_SHOWN;
}

/****************************************************************************/
void StackNodeInfo::SetContentReceived() const
{
  _ContentShifted |= CONTENT_RECEIVED;
}

/****************************************************************************/
void StackNodeInfo::SetContentTransferred() const
{
  _ContentShifted |= CONTENT_TRANSFERRED;
}

/****************************************************************************/
void StackNodeInfo::SetContentAppended() const
{
  _ContentShifted |= CONTENT_APPENDED;
}

/****************************************************************************/
void StackNodeInfo::SetContentShifted() const
{
  _ContentShifted |= CONTENT_SHIFTED;
}

/****************************************************************************/
void StackNodeInfo::ResetContentShifted() const
{
  _ContentShifted = CONTENT_RESET;
}

/****************************************************************************/
bool StackNodeInfo::IsOptionalOrNoContentTag() const
{
  return
  TagTypeInfo::IsOptionalOrNoContentTag
  (
    _TagAttr,
    _TagTypeFound,
    _HasPairContent
  );
}

/****************************************************************************/
// _CurStkTag is the pointer to the Current Stack Tag
// 2nd argument to LinkForward indicates automatic LinkBackward
// action after checking for correct stack node values
//   StackNodeInfo::LinkBackward(StackNodeInfo* EncTag_, StackNodeInfo*);
//
// If current stack tag _CurStkTag is not of singleton tag type
// then LinkForward method will also set the closing tag for all
// intervening nodes between the linked tags to the ending tag node
// if the intervening node tags is of the optional or singleton or empty
// tag type with the method below:
//   SetTagEncountered(TagEnums::CLOSING, false);
//
// if (AutoLinkBack_)
//   EncounteredTag->LinkBackward(_CurStkTag);
//   _TagAppendDone = true;
//
void StackNodeInfo::SetTagEncountered(StackNodeInfo* EndTag_, int TagPos_, bool AutoLinkBack_)
{
  if (!_TagAppendDone && EndTag_ && TagPos_ == TagEnums::END_TAG)
  {
    bool Equal_ = false;
    bool Equiv_ = EquivTag(this, EndTag_, &Equal_, StackNodeInfo::FORWARD_ORDER);
    bool ContentTag_ = _StdTagFound && !_SingletonTagFound && !_EmptyTagFound;

    _EndImplied = !ContentTag_ && _TagTypeFound != TagEnums::END_TAG;
    AutoLinkBack_ = ((Equiv_ && !Equal_) && Preceeds(EndTag_, false)) ||
                    (AutoLinkBack_ && ContentTag_ &&
                     EndTag_->_TagTypeFound == TagEnums::END_TAG);

    _ForwardLink = EndTag_;

    if (AutoLinkBack_ && !_EndImplied)
      EndTag_->LinkBackward(this);

    _TagAppendDone = _EndImplied || AutoLinkBack_;
  }
}

/****************************************************************************/
int StackNodeInfo::FindLinkType(StackNodeInfo* StartTag_, StackNodeInfo* EndTag_)
{
  if (StartTag_ && StartTag_->TagCompleted() &&
      EndTag_ && EndTag_->TagCompleted())
  {
    bool Equal_;
    bool StOp_ = TagTypeInfo::IsOptionalTag(StartTag_->_TagAttr);
    bool EnOp_ = TagTypeInfo::IsOptionalTag(EndTag_->_TagAttr);
    bool Equiv_ = EquivTag(StartTag_, EndTag_, &Equal_, StackNodeInfo::FORWARD_ORDER);

    return
    (
      (StartTag_->SingletonTagFound() &&
       EndTag_->SingletonTagFound())         ? TagEnums::SGLSGL_NESTCOND:
      (StartTag_->SingletonTagFound() &&
       EndTag_->StdTagFound() && !EnOp_)     ? TagEnums::SGLST_NESTCOND:
      (StartTag_->SingletonTagFound() &&
       EndTag_->StdTagFound() && EnOp_)      ? TagEnums::SGLOPTST_NESTCOND:
      (StartTag_->SingletonTagFound() &&
       EndTag_->PairEnderFound() && Equiv_)  ? TagEnums::SGLMEND_NESTCOND:
      (StartTag_->SingletonTagFound() &&
       EndTag_->PairEnderFound() && !Equiv_) ? TagEnums::SGLUMEND_NESTCOND:

      (StartTag_->StdTagFound() && !StOp_ &&
       EndTag_->SingletonTagFound())         ? TagEnums::STSGL_NESTCOND:
      (StartTag_->StdTagFound() && !StOp_ &&
       EndTag_->StdTagFound() && !EnOp_)     ? TagEnums::STST_NESTCOND:
      (StartTag_->StdTagFound() && !StOp_ &&
       EndTag_->StdTagFound() && EnOp_)      ? TagEnums::STOPTST_NESTCOND:
      (StartTag_->StdTagFound() && !StOp_ &&
       EndTag_->PairEnderFound() && Equiv_)  ? TagEnums::STMEND_NESTCOND:
      (StartTag_->StdTagFound() && !StOp_ &&
       EndTag_->PairEnderFound() && !Equiv_) ? TagEnums::STUMEND_NESTCOND:

      (StartTag_->StdTagFound() && StOp_ &&
       EndTag_->SingletonTagFound())         ? TagEnums::OPTSTSGL_NESTCOND:
      (StartTag_->StdTagFound() && StOp_ &&
       EndTag_->StdTagFound() && !EnOp_)     ? TagEnums::OPTSTST_NESTCOND:
      (StartTag_->StdTagFound() && StOp_ &&
       EndTag_->StdTagFound() && EnOp_)      ? TagEnums::OPTSTOPTST_NESTCOND:
      (StartTag_->StdTagFound() && StOp_ &&
       EndTag_->PairEnderFound() && Equiv_)  ? TagEnums::OPTSTMEND_NESTCOND:
      (StartTag_->StdTagFound() && StOp_ &&
       EndTag_->PairEnderFound() && !Equiv_) ? TagEnums::OPTSTUMEND_NESTCOND:

      TagEnums::NO_NESTCOND
    );
  }

  return 0;
}

/****************************************************************************/
bool StackNodeInfo::HasLinks(bool BiDir_, bool StartLink_)
{
  StackNodeInfo* FwdTag_ = _ForwardLink;
  StackNodeInfo* BckTag_ = _BackLink;

  bool Equal_ = false;
  bool Equiv_ = false;
  bool StopAppend_ = false;
  bool Linked_ = false;

  if (StartLink_ && FwdTag_)
  {
    Equiv_ = EquivTag(this, FwdTag_, &Equal_, StackNodeInfo::FORWARD_ORDER);
    Linked_ = !BiDir_ && IsLinked(FwdTag_, false);
    StopAppend_ = BiDir_ && IsLinked(FwdTag_, true) ||
                  ((Equiv_ && !Equal_) && Preceeds(FwdTag_, false));
  }
  else if (!StartLink_ && BckTag_)
  {
    Equiv_ = EquivTag(BckTag_, this, &Equal_, StackNodeInfo::FORWARD_ORDER);
    Linked_ = !BiDir_ && BckTag_->IsLinked(this, false);
    StopAppend_ = BiDir_ && BckTag_->IsLinked(this, true) ||
                  ((Equiv_ && !Equal_) && BckTag_->Preceeds(this, false));
  }

  return (StopAppend_ || Linked_);
}

/****************************************************************************/
bool StackNodeInfo::IsLinked(StackNodeInfo* Tag_, bool BiDir_)
{
  return
  (
    BiDir_ ?
      ((_ForwardLink == Tag_ && Tag_->_BackLink == this) ||
       (Tag_->_ForwardLink == this && _BackLink == Tag_)):
      (_ForwardLink == Tag_)
  );
}

/****************************************************************************/
// _CurStkTag is the pointer to the Current Stack Tag
// 2nd argument to LinkForward indicates automatic LinkBackward
// action after checking for correct stack node values
//   StackNodeInfo::LinkBackward(StackNodeInfo* EncTag_, StackNodeInfo*);
//
// If current stack tag _CurStkTag is not of singleton tag type
// then LinkForward method will also set the closing tag for all
// intervening nodes between the linked tags to the ending tag node
// if the intervening node tags is of the optional or singleton or empty
// tag type with the method below:
//   SetTagEncountered(TagEnums::CLOSING, false);
//
// if (AutoLinkBack_)
//   EncounteredTag->LinkBackward(_CurStkTag);
//   _TagAppendDone = true;
//
StackNodeInfo* StackNodeInfo::LinkForward(StackNodeInfo* EndTag_, bool AutoLinkBack_)
{
  StackNodeInfo* ret = NULL;
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;

  if (!_TagAppendDone && EndTag_ && EndTag_ != this)
  {
    bool Equal_ = false;
    bool Equiv_ = EquivTag(this, EndTag_, &Equal_, StackNodeInfo::FORWARD_ORDER);
    bool ContentTag_ = _StdTagFound && !_SingletonTagFound && !_EmptyTagFound;

    _EndImplied = !ContentTag_ && _TagTypeFound != TagEnums::END_TAG;
    AutoLinkBack_ = ((Equiv_ && !Equal_) && Preceeds(EndTag_, false)) ||
                    (AutoLinkBack_ && ContentTag_ &&
                     EndTag_->_TagTypeFound == TagEnums::END_TAG);

    _ForwardLink = EndTag_;
    SetLinkType(FindLinkType(this, EndTag_));

    if (AutoLinkBack_ && !_EndImplied)
    {
      EndTag_->LinkBackward(this);
      EndTag_->SetChildrenRef(EndTag_->_BackLink->GetChildrenRef());
      _NodeErased = PopAppContStr(false, EndTag_, true);
    }
    else
      EndTag_->LinkBackward(NULL);

    _TagAppendDone = _EndImplied || AutoLinkBack_;
    StackNodeInfo* Ptr_ = _Prev;

    if (AutoLinkBack_ && !_EndImplied)
      while (Ptr_ && !Ptr_->_TagAppendDone &&
             Ptr_->IsOptionalOrNoContentTag())
      {
        Ptr_->SetTagEncountered(EndTag_, TagEnums::CLOSING, false);
        Ptr_ = Ptr_->_Prev;
      }

    ret = (AutoLinkBack_ && !_EndImplied) ? EndTag_:this;

    if (_MatchInfoLink)
    {
      if (AutoLinkBack_ && !_EndImplied)
        EndTag_->LinkBackward(NULL);

      return _MatchInfoLink->LinkForward(EndTag_, AutoLinkBack_);
    }
    else
    {
      const char* Istr_ =
           (_ForwardLink && _ForwardLink->_BackLink) ? "BackLink":"CurrLink";

      #if ((TAGREADER_INTERNALSTACK_DEBUG)|(TAGREADER_SHOW_LINKFORWARD))
        if (TagTypeInfo::ShowDebugDump())
        {
          #if TAGREADER_INTERNALSTACK_DEBUG
            *os_ <<"------------------------ debug: LinkForward : START -----------------------";
          #elif TAGREADER_SHOW_LINKFORWARD
            *os_ <<"--------------------------- LinkForward : START ---------------------------";
          #endif

          *os_ <<endl <<"LinkForward(StackNodeInfo*, bool)" <<endl;
          StackNodeInfo::ShowStackAddr(this, StackNodeInfo::BACKFORW, Istr_);

          #if TAGREADER_INTERNALSTACK_DEBUG
            *os_ <<"------------------------- debug: LinkForward : END ------------------------" <<endl;
          #elif TAGREADER_SHOW_LINKFORWARD
            *os_ <<"---------------------------- LinkForward : END ----------------------------" <<endl;
          #endif
        }
      #endif
    }
  }

  return ret;
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::LinkBackward(StackNodeInfo* StartTag_)
{
  StackNodeInfo* ret = NULL;

  if (!StartTag_)
  {
    _BackLink = NULL;
    _TagAppendDone = false;
  }
  else if (!_TagAppendDone && StartTag_ &&
           _TagTypeFound == TagEnums::END_TAG &&
           StartTag_ != this)
  {
    _BackLink = StartTag_;
    _TagAppendDone = true;
    ret = this;

    if (_MatchInfoLink)
      return _MatchInfoLink->LinkBackward(StartTag_);
  }

  return ret;
}

/****************************************************************************/
long* StackNodeInfo::SetChildrenCount(long val)
{
  if (!_PallocChildren)
    _PallocChildren = new_long(val);

  return _PallocChildren;
}

/****************************************************************************/
long* StackNodeInfo::IncChildrenCount()
{
  if (_PallocChildren)
    (*_PallocChildren)++;

  return _PallocChildren;
}

/****************************************************************************/
long* StackNodeInfo::IncChildrenRef()
{
  if (_CrefChildren)
    (*_CrefChildren)++;

  return _CrefChildren;
}

/****************************************************************************/
void StackNodeInfo::SetCaseSensitive(bool Cs_)
{
  _StartBrkStr.SetCaseSensitive(Cs_);
  _TagElementStr.SetCaseSensitive(Cs_);
  _EndBrkStr.SetCaseSensitive(Cs_);
}

/****************************************************************************/
bool StackNodeInfo::TagStringEqual(StackNodeInfo* EncTag_)
{
  if (EncTag_)
    return
    (
      (_StartBrkStr == EncTag_->_StartBrkStr) &&
      (_TagElementStr == EncTag_->_TagElementStr) &&
      (_EndBrkStr == EncTag_->_EndBrkStr)
    );

  return false;
}

/****************************************************************************/
// EquivTag method matches same tag type as well as matching pair
// tag types as being equivalent
//
bool StackNodeInfo::EquivTag(StackNodeInfo* CurStkTag_, StackNodeInfo* FoundTag_,
                             bool* Equal_, int Order_)
{
  if (CurStkTag_ && FoundTag_)
  {
    if (CurStkTag_ == FoundTag_)
    {
      if (Equal_)
        *Equal_ = true;

      return true;
    }

    bool EqTag_ = false;
    ChrString EndTagStr_(ENDTAGOPEN_STR);
    CurStkTag_->SetCaseSensitive(false);
    FoundTag_->SetCaseSensitive(false);

    if (!CurStkTag_->Erased() && !FoundTag_->Erased() &&
        CurStkTag_->_EndBrkStr == FoundTag_->_EndBrkStr &&
        CurStkTag_->_TagElementStr == FoundTag_->_TagElementStr)
    {
      EqTag_ = FoundTag_->_TagTypeFound == CurStkTag_->_TagTypeFound;

      if (Equal_)
      {
        *Equal_ = EqTag_ &&
                  CurStkTag_->_StartBrkStr == FoundTag_->_StartBrkStr;
        EqTag_ = *Equal_;
      }
      else
        EqTag_ = EqTag_ &&
                 CurStkTag_->_StartBrkStr == FoundTag_->_StartBrkStr;

      if (Order_ == NO_ORDER)
      {
        if ((CurStkTag_->PairEnderFound() && FoundTag_->StdTagFound() &&
             CurStkTag_->_StartBrkStr == EndTagStr_) ||
            (FoundTag_->PairEnderFound() && CurStkTag_->StdTagFound() &&
             FoundTag_->_StartBrkStr == EndTagStr_) || EqTag_)
          return true;
      }
      else if (Order_ == FORWARD_ORDER)
      {
        if ((FoundTag_->PairEnderFound() && CurStkTag_->StdTagFound() &&
             FoundTag_->_StartBrkStr == EndTagStr_) || EqTag_)
          return true;
      }
      else if (Order_ == REVERSE_ORDER)
      {
        if ((CurStkTag_->PairEnderFound() && FoundTag_->StdTagFound() &&
             CurStkTag_->_StartBrkStr == EndTagStr_) || EqTag_)
          return true;
      }
    }
    else if (Equal_)
      *Equal_ = false;
  }

  return false;
}

/****************************************************************************/
// find latest preceeding tag which is either a pair or
// optional pair type tag that matches encountered closing tag
// third argument to TagIsEqual skips optional tags that are unmatched
//
// ::TagIsEqual(StackNodeInfo*, StackNodeInfo*, bool)
//   _StkTagFound = _CurStkTag != NULL && _FoundTag != NULL;
//   _FoundCurTag = _StkTagFound && EquivTag(_CurStkTag, _FoundTag);
//
// EquivTag method matches same tag type as well as matching pair
// tag types as being equivalent
//
bool StackNodeInfo::CursorStkTagIsEqual(TagTypeInfo* Parent_, bool SkipOptTags_)
{
  StackNodeInfo* Ptr_ = this;
  SetStkNodeSearchData(Parent_);

  if (_CursorStkTag && !_CursorStkTag->Erased() &&
      !_CursorStkTag->SingletonTagFound() &&
      !_CursorStkTag->EmptyTagFound())
  {
    while (Ptr_ && !EquivTag(_CursorStkTag, Ptr_))
      Ptr_ = Ptr_->_Next;

    _FoundTag = Ptr_;
    _StkTagFound = _FoundTag != NULL;
    _FoundCursorTag = _StkTagFound &&
                   (_CursorStkTag == _FoundTag ||
                   (_CursorStkTag && EquivTag(_CursorStkTag, _FoundTag)));
  }
  else
  {
    _FoundTag = NULL;
    _FoundCursorTag = _StkTagFound = false;
  }

  _Parent->Sequencer()->SetStkNodeSearchData(this);

  if (_FoundCursorTag)
    AssignGlobalStackTag();

  return _StkTagFound;
}

/****************************************************************************/
// find latest preceeding tag which is either a pair or
// optional pair type tag that matches encountered closing tag
// third argument to TagIsEqual skips optional tags that are unmatched
//
// ::TagIsEqual(StackNodeInfo*, StackNodeInfo*, bool)
//   _StkTagFound = _CurStkTag != NULL && _FoundTag != NULL;
//   _FoundCurTag = _StkTagFound && EquivTag(_CurStkTag, _FoundTag);
//
// EquivTag method matches same tag type as well as matching pair
// tag types as being equivalent
//
bool StackNodeInfo::EncTagIsEqual(StackNodeInfo* EncTag_, TagTypeInfo* Parent_, bool SkipOptTags_)
{
  StackNodeInfo* Ptr_ = this;
  SetStkNodeSearchData(Parent_);

  if (((_CursorStkTag && !_CursorStkTag->SingletonTagFound() &&
        !_CursorStkTag->EmptyTagFound()) || !_CursorStkTag) &&
       (EncTag_ && !EncTag_->Erased() &&
        !EncTag_->SingletonTagFound() &&
        !EncTag_->EmptyTagFound()))
  {
    while (Ptr_ && !EquivTag(EncTag_, Ptr_))
      Ptr_ = Ptr_->_Next;

    _FoundTag = Ptr_;
    _StkTagFound = _FoundTag != NULL;
    _FoundCursorTag = _StkTagFound &&
                   (_CursorStkTag == _FoundTag ||
                   (_CursorStkTag && EquivTag(_CursorStkTag, _FoundTag)));
  }
  else
  {
    _FoundTag = NULL;
    _FoundCursorTag = _StkTagFound = false;
  }

  _Parent->Sequencer()->SetStkNodeSearchData(this);

  if (_FoundCursorTag)
    AssignGlobalStackTag();

  return _StkTagFound;
}

/****************************************************************************/
// find latest preceeding tag which is either a pair or
// optional pair type tag that matches encountered closing tag
// third argument to TagIsEqual skips optional tags that are unmatched
//
// ::TagIsEqual(StackNodeInfo*, StackNodeInfo*, bool)
//   _StkTagFound = _CurStkTag != NULL && _FoundTag != NULL;
//   _FoundCurTag = _StkTagFound && EquivTag(_CurStkTag, _FoundTag);
//
// EquivTag method matches same tag type as well as matching pair
// tag types as being equivalent
//
bool StackNodeInfo::TagIsEqual(StackNodeInfo* EncTag_, TagTypeInfo* Parent_, bool SkipOptTags_)
{
  if (EncTag_ && _CursorStkTag == EncTag_)
    return CursorStkTagIsEqual(Parent_, SkipOptTags_);

  return EncTagIsEqual(EncTag_, Parent_, SkipOptTags_);
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::CursorTagStkPtr()
{
  return (_Parent ? _Parent->CursorTagStkPtr():NULL);
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::TagStackPtr()
{
  return (_Parent ? _Parent->TagStackPtr():NULL);
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::StackInfoPtr()
{
  return (_Parent ? _Parent->StackInfoPtr():NULL);
}

/****************************************************************************/
Subscript StackNodeInfo::ReadPos() const
{
  return (_Parent ? _Parent->ReadPos():NULL);
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::GiveNodeFound()
{
  return _FoundTag;
}

/****************************************************************************/
void StackNodeInfo::SetMismatchedEndPair(StackNodeInfo* EncTag_)
{
  _MismatchedEndPair = EncTag_;
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;

  if (_MatchInfoLink)
    _MatchInfoLink->SetMismatchedEndPair(EncTag_);
  else
  {
    const char* Istr_ = "CurrLink";

    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump())
      {
        *os_ <<"------------------------ debug: MismatchedEndPair : START -----------------------" <<endl
             <<"SetMismatchedEndPair(StackNodeInfo*)" <<endl;
        StackNodeInfo::ShowStackAddr(this, StackNodeInfo::MISMLINK, Istr_);
        *os_ <<"------------------------- debug: MismatchedEndPair : END ------------------------" <<endl;
      }
    #endif
  }
}

/****************************************************************************/
void StackNodeInfo::SetOverlappedPairs(StackNodeInfo* Prev_, StackNodeInfo* Next_)
{
  _OverlappedPrevPair = Prev_;
  _OverlappedNextPair = Next_;
  ostream* os_ = _Parent ? _Parent->GiveOutput():&std::cout;

  if (_MatchInfoLink)
    _MatchInfoLink->SetOverlappedPairs(Prev_, Next_);
  else
  {
    const char* Istr_ = "CurrLink";

    #if TAGREADER_INTERNALSTACK_DEBUG
      if (TagTypeInfo::ShowDebugDump())
      {
        *os_ <<"------------------------ debug: OverlappedPairs : START -----------------------" <<endl
             <<"SetOverlappedPairs(StackNodeInfo*, StackNodeInfo*)" <<endl;
        StackNodeInfo::ShowStackAddr(this, StackNodeInfo::PAIRLINK, Istr_);
        *os_ <<"------------------------- debug: OverlappedPairs : END ------------------------" <<endl;
      }
    #endif
  }
}

/****************************************************************************/
// Set tag nesting condition and setup TagTypeInfo data members and execute
// the action correlating to the specified nesting condition.
//
void StackNodeInfo::SetTagNestCond(long NestCond_, StackNodeInfo* EncTag_)
{
  if (_Parent->Sequencer()->ExecTagActionFails() < TagTypeInfo::EXEC_ACTION_MAX_FAIL)
  {
    _Parent->SetTagCondData(NestCond_, this, EncTag_);
    _Parent->ExecTagCondAction(NestCond_, this, EncTag_);
  }
}

/****************************************************************************/
bool StackNodeInfo::Unlink()
{
  if (_Erased && !_Copied)
  {
    TagElement* TagElmPtr_ = _Parent ? _Parent->GiveTagElementPtr():NULL;

    SetPrev(SetNext(NULL));
    _PrevCursor = _NextCursor = NULL;
    _BackLink = _ForwardLink = NULL;
    _ElementPtr = NULL;
    _Parent = NULL;

    _MatchInfoLink = 
    _MismatchedEndPair =
    _OverlappedPrevPair =
    _OverlappedNextPair = NULL;

    if (_TextNode)
    {
      _TextNode->SetRefNode(NULL);
      _TextNode = NULL;
    }

    if (_TagStates && !_TagStates->IsCopied())
    {
      if (TagElmPtr_)
        TagElmPtr_->SetTagElementState(NULL);

      _TagStates->SetParent(NULL);

      delete _TagStates;
      _TagStates = NULL;
    }

    delete this;
    return true;
  }

  return false;
}

/****************************************************************************/
bool StackNodeInfo::HasThisParent(const TagTypeInfo* Ptr_) const
{
  return (_Parent &&
          _Parent == Ptr_);
}

/****************************************************************************/
bool StackNodeInfo::HasThisParentAndSearchNum(const TagTypeInfo* Ptr_) const
{
  return (_Parent &&
          _Parent == Ptr_ &&
          _SearchRun == Ptr_->SearchRunNumber());
}

/****************************************************************************/
bool StackNodeInfo::IsParentOf(const TagElementStates* Child_) const
{
  return (Child_ ? Child_->HasThisParent(this):false);
}

/****************************************************************************/
bool StackNodeInfo::IsRefNodeOf(const AppTextNode* ReferFrom_) const
{
  return (ReferFrom_ ? ReferFrom_->HasThisRefNode(this):false);
}

/****************************************************************************/
QListNode* StackNodeInfo::GiveNext()
{
  return _Next;
}

/****************************************************************************/
const QListNode* StackNodeInfo::Next() const
{
  return _Next;
}

/****************************************************************************/
QListNode* StackNodeInfo::SetNext(QListNode* Ptr_)
{
  if (!Ptr_ || Ptr_->ClassType() == QListNode::STACKNODEINFO)
  {
    _Next = (StackNodeInfo*)Ptr_;
    _NextBase = _Next;
    return _Next;
  }

  _NextBase = Ptr_;
  return _NextBase;
}

/****************************************************************************/
QListNode* StackNodeInfo::GivePrev()
{
  return _Prev;
}

/****************************************************************************/
const QListNode* StackNodeInfo::Prev() const
{
  return _Prev;
}

/****************************************************************************/
QListNode* StackNodeInfo::SetPrev(QListNode* Ptr_)
{
  if (!Ptr_ || Ptr_->ClassType() == QListNode::STACKNODEINFO)
  {
    _Prev = (StackNodeInfo*)Ptr_;
    _PrevBase = _Prev;
    return _Prev;
  }

  _PrevBase = Ptr_;
  return _PrevBase;
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::GiveNextCursor()
{
  return _NextCursor;
}

/****************************************************************************/
const StackNodeInfo* StackNodeInfo::NextCursor() const
{
  return _NextCursor;
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::SetNextCursor(StackNodeInfo* Ptr_)
{
  _NextCursor = Ptr_;
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::GivePrevCursor()
{
  return _PrevCursor;
}

/****************************************************************************/
const StackNodeInfo* StackNodeInfo::PrevCursor() const
{
  return _PrevCursor;
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::SetPrevCursor(StackNodeInfo* Ptr_)
{
  _PrevCursor = Ptr_;
}

/****************************************************************************/
// Appended tag content text methods
/****************************************************************************/
TagElementStates* StackNodeInfo::MakeTagElementStates()
{
  _TagStates = new TagElementStates(this, NodeID());

  if (_TagStates)
  {
    _TagStates->SetCopied(false);
    return _TagStates;
  }

  return NULL;
}

/****************************************************************************/
AppTextNode* StackNodeInfo::MakeAppTextNode(bool HaveTagStk_)
{
  _TextNode = new AppTextNode(HaveTagStk_, this);

  if (_TextNode && HasThisParentAndSearchNum(_Parent) &&
      _ContentTypeTag && _HasTagContents)
  {
    ChrString RetStr_ = GiveTagContent();

    if (!RetStr_.IsEmpty())
    {
      if (IsContentShifted() || IsContentAppended())
        _TextNode->AppendTextStr(&RetStr_);
      else
        _TextNode->SetTextStr(&RetStr_);

      SetContentAppended();
    }
  }

  return _TextNode;
}

/****************************************************************************/
AppTextNode* StackNodeInfo::MakeAppTentTextNode(bool HaveTentStk_)
{
  AppTextNode* ContAppPtr_ = AppText();

  if (ContAppPtr_ && !ContAppPtr_->TransferNode())
  {
    _TextNode = new AppTextNode(HaveTentStk_, this);

    if (_TextNode && _Parent && _ContentTypeTag && _HasTagContents)
    {
      ChrString RetStr_ = GiveTagContent();

      if (!RetStr_.IsEmpty())
      {
        if (IsContentShifted() || IsContentAppended())
          _TextNode->AppendTextStr(&RetStr_);
        else
          _TextNode->SetTextStr(&RetStr_);

        SetContentAppended();
      }
    }

    if (_TextNode)
      _TextNode->SetTransferNode(ContAppPtr_);

    return _TextNode;
  }

  return NULL;
}

/****************************************************************************/
void StackNodeInfo::SetAppTextNode(AppTextNode* NodePtr_)
{
  _TextNode = NodePtr_;
}

/****************************************************************************/
// 1st param indicates string termination : AppContStr=0
//
bool StackNodeInfo::PopAppContStr(bool Flag_,
                                  StackNodeInfo* EncNode_, bool PostLink_)
{
  if (_TextNode && _Parent)
  {
    _NodeErased = _TextNode->PopAppContStr(Flag_, EncNode_, PostLink_);
    return _NodeErased;
  }

  return false;
}

/****************************************************************************/
void StackNodeInfo::SetOpStTagToUnmatchedEndData(StackNodeInfo* CurNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     if (Ume.BackLink->NumChildren) then
  //       SeqNum = Ume.BackLink->NumChildren;
  //     else if (Next.SeqNum)
  //       SeqNum = Next.SeqNum;
  //     else
  //       SeqNum = Ume.BackLink->SeqNum+1;
  //     SeqNum++;
  //
  //     CrefChildren = Ume.BackLink->CrefChildren;
  //
  //     if (Ume.BackLink->EncLevel > CurLevel) {
  //       CurLevel = Ume.BackLink->CurLevel = Ume.BackLink->EncLevel-1;
  //       ParLevel = Ume.BackLink->CurLevel + 1;
  //     }
  //     else if (EncLevel > Ume.BackLink->CurLevel) {
  //       ParLevel = Ume.BackLink->EncLevel
  //       CurLevel = EncLevel-1;
  //
  //       if (CurLevel < Ume.BackLink->EncLevel)
  //         Ume.BackLink->CurLevel = CurLevel
  //     }
  //     else {
  //       ParLevel = Ume.BackLink->EncLevel;
  //       CurLevel = Ume.BackLink->CurLevel;
  //     }
  //     EncLevel==0;
  //   else
  //     SeqNum = St.SeqNum;
  //     if (St.ParentNumChildren)
  //       SeqNum = St.ParentNumChildren;
  //     else
  //       SeqNum++;
  //
  //     CrefChildren = St.BackLink->CrefChildren;
  //
  //     ParLevel = St.CurLevel;
  //     CurLevel = (St.CurLevel > 0) ?
  //                    St.CurLevel-1:
  //                    St.CurLevel;
  //     EncLevel==0;
  //   endif
  //
  //   Tag=HeadAppText(true,true);
  //   OpSt.PopAppContStr(1);
  //   OpSt.AppContStr=OpSt.HeadAppContStr();
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Tag->_TransferNode && Ume==Tag then
  //     Ume.AppendTentToContStr(1);
  //   endif
  //   if (OpSt.AtReqTagCondStopState() &&
  //       OpSt.TagAtImpliedEnd()) then
  //     OpSt.PushAppTentStr(true,OpSt);
  //   endif
  // endif
  //
  // Example:
  //   <FONT SIZE=4>
  //     <P> ...
  //   </FONT>
  //

  if (_BackLink && !CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = _BackLink->NumChildren() ?
                       _BackLink->NumChildren():
                   _Next->AltZero_NextSequenceNumber() ?
                       _Next->AltZero_NextSequenceNumber():
                       (_BackLink->AltZero_NextSequenceNumber()+1);
    _SequenceNum++;
    SetChildrenRef(_BackLink->GetChildrenRef());

    if (_BackLink->_EncLevel > _CurLevel)
    {
      _CurLevel = _BackLink->_CurLevel = _BackLink->_EncLevel-1;
      _ChdLevel = _BackLink->_CurLevel + 1;
    }
    else if (_EncLevel > _BackLink->_CurLevel)
    {
      _ChdLevel = _BackLink->_EncLevel;
      _CurLevel = _EncLevel-1;

      if (_CurLevel < _BackLink->_EncLevel)
        _BackLink->_CurLevel = _CurLevel;
    }
    else
    {
      _ChdLevel = _BackLink->_EncLevel;
      _CurLevel = _BackLink->_CurLevel;
    }

    _EncLevel = 0;
  }
  else if (CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = CurNode_->ParentNumChildren() ?
                       CurNode_->ParentNumChildren():
                       CurNode_->AltZero_NextSequenceNumber();
    _SequenceNum++;
    SetChildrenRef(_BackLink->GetChildrenRef());

    _ChdLevel = CurNode_->_CurLevel;
    _CurLevel = (CurNode_->_CurLevel > 0) ?
                    CurNode_->_CurLevel-1:
                    CurNode_->_CurLevel;
    _EncLevel = 0;
  }
}

/****************************************************************************/
void StackNodeInfo::SetStdTagToUnmatchedEndData(StackNodeInfo* CurNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     if (Ume.BackLink->NumChildren) then
  //       SeqNum = Ume.BackLink->NumChildren;
  //     else if (Next.SeqNum)
  //       SeqNum = Next.SeqNum;
  //     else
  //       SeqNum = Ume.BackLink->SeqNum+1;
  //     SeqNum++;
  //
  //     CrefChildren = Ume.BackLink->CrefChildren;
  //
  //     if (Ume.BackLink->EncLevel > CurLevel) {
  //       CurLevel = Ume.BackLink->CurLevel = Ume.BackLink->EncLevel-1;
  //       ParLevel = Ume.BackLink->CurLevel + 1;
  //     }
  //     else if (EncLevel > Ume.BackLink->CurLevel) {
  //       ParLevel = Ume.BackLink->EncLevel
  //       CurLevel = EncLevel-1;
  //
  //       if (CurLevel < Ume.BackLink->EncLevel)
  //         Ume.BackLink->CurLevel = CurLevel
  //     }
  //     else {
  //       ParLevel = Ume.BackLink->EncLevel;
  //       CurLevel = Ume.BackLink->CurLevel;
  //     }
  //     EncLevel==0;
  //   else
  //     if (St.ParentNumChildren)
  //       SeqNum = St.ParentNumChildren;
  //     else
  //       SeqNum = St.SeqNum;
  //     SeqNum++;
  //
  //     CrefChildren = St.BackLink->CrefChildren;
  //
  //     ParLevel = St.CurLevel;
  //     CurLevel = (St.CurLevel > 0) ?
  //                    St.CurLevel-1:
  //                    St.CurLevel;
  //     EncLevel==0;
  //   endif
  //
  //   Tag=HeadAppText(true,true);
  //   St.PopAppContStr(0);
  //   St.AppContStr=St.HeadAppContStr();
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Tag->_TransferNode && Ume==Tag then
  //     Ume.AppendTentToContStr(1);
  //   endif
  // endif
  //
  // Example:
  //   <FONT SIZE=4>
  //     <A HREF="webpage.html"> ...
  //   </FONT>
  //

  if (_BackLink && !CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = _BackLink->NumChildren() ?
                       _BackLink->NumChildren():
                   _Next->AltZero_NextSequenceNumber() ?
                       _Next->AltZero_NextSequenceNumber():
                       (_BackLink->AltZero_NextSequenceNumber()+1);
    _SequenceNum++;
    SetChildrenRef(_BackLink->GetChildrenRef());

    if (_BackLink->_EncLevel > _CurLevel)
    {
      _CurLevel = _BackLink->_CurLevel = _BackLink->_EncLevel-1;
      _ChdLevel = _BackLink->_CurLevel + 1;
    }
    else if (_EncLevel > _BackLink->_CurLevel)
    {
      _ChdLevel = _BackLink->_EncLevel;
      _CurLevel = _EncLevel-1;

      if (_CurLevel < _BackLink->_EncLevel)
        _BackLink->_CurLevel = _CurLevel;
    }
    else
    {
      _ChdLevel = _BackLink->_EncLevel;
      _CurLevel = _BackLink->_CurLevel;
    }

    _EncLevel = 0;
  }
  else if (CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = CurNode_->ParentNumChildren() ?
                       CurNode_->ParentNumChildren():
                       CurNode_->AltZero_NextSequenceNumber();
    _SequenceNum++;
    SetChildrenRef(_BackLink->GetChildrenRef());

    _ChdLevel = CurNode_->_CurLevel;
    _CurLevel = (CurNode_->_CurLevel > 0) ?
                    CurNode_->_CurLevel-1:
                    CurNode_->_CurLevel;
    _EncLevel = 0;
  }
}

/****************************************************************************/
void StackNodeInfo::SetSglTagToUnmatchedEndData(StackNodeInfo* CurNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     if (Ume.BackLink->NumChildren) then
  //       SeqNum = Ume.BackLink->NumChildren;
  //     else if (Next.SeqNum)
  //       SeqNum = Next.SeqNum;
  //     else
  //       SeqNum = Ume.BackLink->SeqNum+1;
  //     SeqNum++;
  //
  //     CrefChildren = Ume.BackLink->CrefChildren;
  //
  //     if (Ume.BackLink->EncLevel > CurLevel) {
  //       CurLevel = Ume.BackLink->CurLevel = Ume.BackLink->EncLevel-1;
  //       ParLevel = Ume.BackLink->CurLevel + 1;
  //     }
  //     else if (EncLevel > Ume.BackLink->CurLevel) {
  //       ParLevel = Ume.BackLink->EncLevel
  //       CurLevel = EncLevel-1;
  //
  //       if (CurLevel < Ume.BackLink->EncLevel)
  //         Ume.BackLink->CurLevel = CurLevel
  //     }
  //     else {
  //       ParLevel = Ume.BackLink->EncLevel;
  //       CurLevel = Ume.BackLink->CurLevel;
  //     }
  //     EncLevel==0;
  //   else
  //     if (Sgl.ParentNumChildren)
  //       SeqNum = Sgl.ParentNumChildren;
  //     else
  //       SeqNum = Sgl.SeqNum;
  //     SeqNum++;
  //
  //     CrefChildren = Sgl.BackLink->CrefChildren;
  //
  //     ParLevel = St.CurLevel;
  //     CurLevel = (Sgl.CurLevel > 0) ?
  //                    Sgl.CurLevel-1:
  //                    Sgl.CurLevel;
  //     EncLevel==0;
  //   endif
  //
  //   Tag=HeadAppText(true,true);
  //   Sgl.PopAppContStr(1);
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Tag->_TransferNode && Ume==Tag then
  //     Ume.AppendTentToContStr(1);
  //   endif
  // endif
  //
  // Example:
  //   <FONT SIZE=4>
  //     <HR> ...
  //   </FONT>
  //

  if (_BackLink && !CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = _BackLink->NumChildren() ?
                       _BackLink->NumChildren():
                   _Next->AltZero_NextSequenceNumber() ?
                       _Next->AltZero_NextSequenceNumber():
                       (_BackLink->AltZero_NextSequenceNumber()+1);
    _SequenceNum++;
    SetChildrenRef(_BackLink->GetChildrenRef());

    if (_BackLink->_EncLevel > _CurLevel)
    {
      _CurLevel = _BackLink->_CurLevel = _BackLink->_EncLevel-1;
      _ChdLevel = _BackLink->_CurLevel + 1;
    }
    else if (_EncLevel > _BackLink->_CurLevel)
    {
      _ChdLevel = _BackLink->_EncLevel;
      _CurLevel = _EncLevel-1;

      if (_CurLevel < _BackLink->_EncLevel)
        _BackLink->_CurLevel = _CurLevel;
    }
    else
    {
      _ChdLevel = _BackLink->_EncLevel;
      _CurLevel = _BackLink->_CurLevel;
    }

    _EncLevel = 0;
  }
  else if (CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = CurNode_->ParentNumChildren() ?
                       CurNode_->ParentNumChildren():
                       CurNode_->AltZero_NextSequenceNumber();
    _SequenceNum++;
    SetChildrenRef(_BackLink->GetChildrenRef());

    _ChdLevel = CurNode_->_CurLevel;
    _CurLevel = (CurNode_->_CurLevel > 0) ?
                    CurNode_->_CurLevel-1:
                    CurNode_->_CurLevel;
    _EncLevel = 0;
  }
}

/****************************************************************************/
void StackNodeInfo::SetMatchedEndTagToUnmatchedEndData(StackNodeInfo* CurNode_)
{
  // if FindTag(Ume) then
  //   if (Ume->BackLink)
  //     if (Ume.BackLink->NumChildren) then
  //       SeqNum = Ume.BackLink->NumChildren;
  //     else if (Next.SeqNum)
  //       SeqNum = Next.SeqNum;
  //     else
  //       SeqNum = Ume.BackLink->SeqNum+1;
  //     SeqNum++;
  //
  //     CrefChildren = Ume.BackLink->CrefChildren;
  //
  //     ParLevel = _BackLink->_EncLevel;
  //     CurLevel = Ume.CurLevel;
  //     EncLevel = 0;
  //   else
  //     if (Me.ParentNumChildren)
  //       SeqNum = Me.ParentNumChildren;
  //     else if (Me.BackLink)
  //       SeqNum = Me.BackLink->ParentSeqNum;
  //     else
  //       SeqNum = Me.SeqNum;
  //     SeqNum++;
  //
  //     CrefChildren = Me.BackLink->CrefChildren;
  //
  //     ParLevel = Pe.CurLevel;
  //     CurLevel = (Pe.CurLevel > 0) ?
  //                    Pe.CurLevel-1:
  //                    Pe.CurLevel;
  //     EncLevel = 0;
  //   endif
  //
  //   Tag=Pe.PopAppContStr(0);
  //     Pe.AppContStr=Pe.HeadAppContStr();
  //   Ume.AppContStr=Ume.HeadAppContStr();
  //   Ume->BackLink.AppContStr=Ume->BackLink.HeadAppContStr();
  //   if Ume.SrchTentAppStk(Tag)==true then
  //     Ume.DetTentAppStk(Tag);
  //     Ume.AppendTentToContStr(1);
  //   endif
  // endif
  //
  // Example:
  //   <FONT SIZE=4>
  //     <A HREF="webpage.html"> ...
  //     </A>
  //   </FONT>
  //

  if (_BackLink && !CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = _BackLink->NumChildren() ?
                       _BackLink->NumChildren():
                   _Next->AltZero_NextSequenceNumber() ?
                       _Next->AltZero_NextSequenceNumber():
                       (_BackLink->AltZero_NextSequenceNumber()+1);
    _SequenceNum++;
    SetChildrenRef(_BackLink->GetChildrenRef());

    _ChdLevel = _BackLink->_EncLevel;
    _CurLevel = _BackLink->_CurLevel;
    _EncLevel = 0;
  }
  else if (CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = CurNode_->ParentNumChildren() ?
                       CurNode_->ParentNumChildren():
                   CurNode_->_BackLink ?
                       CurNode_->_BackLink->AltZero_ParentSequenceNumber():
                       CurNode_->AltZero_NextSequenceNumber();
    _SequenceNum++;
    SetChildrenRef(_BackLink->GetChildrenRef());

    _ChdLevel = CurNode_->_CurLevel;
    _CurLevel = (CurNode_->_CurLevel > 0) ?
                    CurNode_->_CurLevel-1:
                    CurNode_->_CurLevel;
    _EncLevel = 0;
  }
}

/****************************************************************************/
void StackNodeInfo::SetOpStTagToOpStTagData(StackNodeInfo* CurNode_, bool& NodePopped_)
{
  // SeqNum==0;
  // ParentSeqNum = OpSt.SeqNum;
  // ParentSeqNum++;
  //
  // OpSt.PallocChildren = 0;
  // CrefChildren = OpSt.PallocChildren;
  // CrefChildren++;
  // PallocChildren = 0;
  //
  // ParLevel = 0;
  // CurLevel = St->EncLevel;
  // IncNestLevel();
  // EncLevel = CurLevel++;
  //
  // OpSt.PopAppContStr(1);   // 1st param indicates string termination
  // OpSt_.PushAppContStr(HeadAppText,0);
  // OpSt_.AppContStr=OpSt_.HeadAppContStr();
  //   if (OpSt.AtReqTagCondStopState() &&
  //       OpSt.TagAtImpliedEnd()) then
  //     OpSt.PushAppTentStr(true,OpSt);      // 2nd param indicates OpSt tag
  //     OpSt.AppTentStr=OpSt.HeadAppTentStr();
  //   endif
  //
  // Example:
  //   <P> ...
  //     <P> ...
  //

  if (CurNode_ && _Parent)
  {
    _NextSequenceNum = 0;
    _SequenceNum = 0;
    _ParentSequenceNum = CurNode_->AltZero_NextSequenceNumber();
    _ParentSequenceNum++;

    SetChildrenRef(CurNode_->SetChildrenCount(0));
    IncChildrenRef();
    SetChildrenCount(0);

    _ChdLevel = 0;
    _CurLevel = (CurNode_->_EncLevel > CurNode_->_CurLevel) ?
                    CurNode_->_EncLevel:
                    CurNode_->_CurLevel+1;
    IncNestLevel();

    if (CurNode_->StdTagFound() &&
          ((CurNode_->AtReqTagCondStopState(false) &&
            CurNode_->RequiredTagCondMet()) ||
           (CurNode_->NoRequiredTag() &&
            CurNode_->CurrentTagCondMet())))
    {
      _NodeErased = CurNode_->PopAppContStr(true);
      NodePopped_ = _NodeErased;
    }
  }
}

/****************************************************************************/
void StackNodeInfo::SetSglTagToSglTagData(StackNodeInfo* CurNode_)
{
  // SeqNum = Sgl.SeqNum;
  // SeqNum++;
  // CrefChildren = Sgl.CrefChildren;
  // CrefChildren++;
  //
  // CurLevel = Sgl->CurLevel;
  // EncLevel = 0;
  // ParLevel = 0;
  // Sgl.PopAppContStr(1);
  // Sgl_.AppContStr=Sgl_.HeadAppContStr();
  //
  // Example:
  //   <img src="image.gif"> ...
  //   <br> ...
  //

  if (CurNode_ && _Parent)
  {
    _NextSequenceNum = 0;
    _SequenceNum = CurNode_->AltZero_NextSequenceNumber();
    _SequenceNum++;

    SetChildrenRef(CurNode_->GetChildrenRef());
    IncChildrenRef();

    _CurLevel = CurNode_->_CurLevel;
    _EncLevel = _ChdLevel = 0;

    if (CurNode_->SingletonTagFound())
      _NodeErased = CurNode_->PopAppContStr(true);
  }
}

/****************************************************************************/
void StackNodeInfo::SetSglTagToMatchedEndData(StackNodeInfo* CurNode_)
{
  // SeqNum = Sgl.SeqNum;
  // SeqNum++;
  // CrefChildren = Sgl.CrefChildren;
  // CrefChildren++;
  //
  // CurLevel = Sgl->CurLevel;
  // EncLevel = 0;
  // ParLevel = 0;
  // Sgl.PopAppContStr(1);
  // Me.AppContStr=Me.HeadAppContStr();
  // Me->BackLink.AppContStr=Me->BackLink.HeadAppContStr();
  //
  // Example:
  //   <img src="image.gif"> ...
  //   </img>
  //

  if (CurNode_ && _Parent)
  {
    _NextSequenceNum = 0;
    _SequenceNum = CurNode_->AltZero_NextSequenceNumber();
    _SequenceNum++;

    SetChildrenRef(CurNode_->GetChildrenRef());
    IncChildrenRef();

    _CurLevel = CurNode_->_CurLevel;
    _EncLevel = _ChdLevel = 0;

    if (CurNode_->SingletonTagFound())
      _NodeErased = CurNode_->PopAppContStr(true);
  }
}

/****************************************************************************/
void StackNodeInfo::SetStdTagToMatchedEndData(StackNodeInfo* CurNode_)
{
  // if (Me->BackLink)
  //   if (Me.BackLink->NumChildren) then
  //     SeqNum = Me.BackLink->NumChildren;
  //   else if (Next.SeqNum)
  //     SeqNum = Next.SeqNum;
  //   else
  //     SeqNum = Me.BackLink->SeqNum;
  //   SeqNum++;
  //
  //   CrefChildren = Me.BackLink->CrefChildren;
  //
  //   if (Me.BackLink->EncLevel > CurLevel) {
  //     CurLevel = Me.BackLink->CurLevel = Me.BackLink->EncLevel-1;
  //     ParLevel = Me.BackLink->CurLevel + 1;
  //   }
  //   else if (EncLevel > Me.BackLink->CurLevel) {
  //     ParLevel = Me.BackLink->EncLevel;
  //     CurLevel = EncLevel-1;
  //
  //     if (CurLevel < Me.BackLink->EncLevel)
  //       Me.BackLink->CurLevel = CurLevel
  //   }
  //   else {
  //     ParLevel = Me.BackLink->EncLevel;
  //     CurLevel = Me.BackLink->CurLevel;
  //   }
  //
  //   EncLevel==0;
  // else
  //   if (Pe.NumChildren)
  //     SeqNum = Pe.NumChildren;
  //   else
  //     SeqNum = Pe.SeqNum;
  //   SeqNum++;
  //
  //   CrefChildren = Pe.BackLink->CrefChildren;
  //
  //   if (Pe.EncLevel > CurLevel) {
  //     CurLevel = Pe.BackLink->EncLevel-1;
  //     ParLevel = CurLevel + 1;
  //   }
  //   else if (EncLevel > Pe.CurLevel) {
  //     ParLevel = Pe.EncLevel;
  //     CurLevel = EncLevel-1;
  //
  //     if (CurLevel < Pe.EncLevel)
  //       Pe.CurLevel = CurLevel
  //   }
  //   else {
  //     ParLevel = Pe.EncLevel;
  //     CurLevel = Pe.CurLevel;
  //   }
  //
  //   EncLevel==0;
  // endif
  //
  // St.PopAppContStr(1);
  // Me.AppContStr=Me.HeadAppContStr();
  // Me->BackLink.AppContStr=Me->BackLink.HeadAppContStr();
  //
  // Example:
  //   <A HREF="link.html">
  //      ....
  //   </A>
  //

  if (CurNode_)
  {
    if (_BackLink)
    {
      _NextSequenceNum = 0;
      _SequenceNum = _BackLink->NumChildren() ?
                         _BackLink->NumChildren():
                     _Next->AltZero_NextSequenceNumber() ?
                         _Next->AltZero_NextSequenceNumber():
                         _BackLink->AltZero_NextSequenceNumber();
      _SequenceNum++;
      SetChildrenRef(_BackLink->GetChildrenRef());

      if (_BackLink->_EncLevel > _CurLevel)
      {
        _CurLevel = _BackLink->_CurLevel = _BackLink->_EncLevel-1;
        _ChdLevel = _BackLink->_CurLevel + 1;
      }
      else if (_EncLevel > _BackLink->_CurLevel)
      {
        _ChdLevel = _BackLink->_EncLevel;
        _CurLevel = _EncLevel-1;

        if (_CurLevel < _BackLink->_EncLevel)
          _BackLink->_CurLevel = _CurLevel;
      }
      else
      {
        _ChdLevel = _BackLink->_EncLevel;
        _CurLevel = _BackLink->_CurLevel;
      }

      _EncLevel = 0;
    }
    else
    {
      _NextSequenceNum = 0;
      _SequenceNum = CurNode_->NumChildren() ?
                         CurNode_->NumChildren():
                         CurNode_->AltZero_NextSequenceNumber();
      _SequenceNum++;
      SetChildrenRef(_BackLink->GetChildrenRef());

      if (CurNode_->_EncLevel > _CurLevel)
      {
        _CurLevel = CurNode_->_EncLevel-1;
        _ChdLevel = _CurLevel + 1;
      }
      else if (_EncLevel > CurNode_->_CurLevel)
      {
        _ChdLevel = CurNode_->_EncLevel;
        _CurLevel = _EncLevel-1;

        if (_CurLevel < CurNode_->_EncLevel)
          CurNode_->_CurLevel = _CurLevel;
      }
      else
      {
        _ChdLevel = CurNode_->_EncLevel;
        _CurLevel = CurNode_->_CurLevel;
      }

      _EncLevel = 0;
    }

    if (CurNode_->StdTagFound())
      _NodeErased = CurNode_->PopAppContStr(true);
  }
}

/****************************************************************************/
void StackNodeInfo::SetXToSglTagData(StackNodeInfo* CurNode_)
{
  // if (X->ParentNumChildren)
  //   SeqNum = X.ParentNumChildren;
  // else if (X->BackLink)
  //   SeqNum = X.BackLink->ParentSeqNum;
  // else
  //   SeqNum = X.SeqNum;
  // SeqNum++;
  //
  // if (X->PallocChildren)
  //   CrefChildren = X.PallocChildren;
  // else
  //   CrefChildren = X.CrefChildren;
  // CrefChildren++;
  //
  // CurLevel = X->CurLevel;
  // EncLevel = 0;
  // ParLevel = 0;
  // Sgl.PushAppContStr(HeadAppText,1);
  // Sgl.AppContStr=Sgl.HeadAppContStr();
  //
  // Example:
  //   <P>
  //     <img src="image.gif"> ...
  //
  //   OR
  //
  //   </P>
  //     <img src="image.gif"> ...
  //
  if (CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = CurNode_->ParentNumChildren() ?
                       CurNode_->ParentNumChildren():
                   CurNode_->_BackLink ?
                       CurNode_->_BackLink->AltZero_ParentSequenceNumber():
                       CurNode_->AltZero_NextSequenceNumber();
    _SequenceNum++;

    SetChildrenRef(_BackLink->GetChildrenCount() ?
                       _BackLink->GetChildrenCount():
                       CurNode_->GetChildrenRef());
    IncChildrenRef();

    _CurLevel = CurNode_->_CurLevel;
    _EncLevel = _ChdLevel = 0;
  }
}

/****************************************************************************/
void StackNodeInfo::SetXToStdTagData(StackNodeInfo* CurNode_)
{
  // SeqNum==0;
  // if (X->ParentNumChildren)
  //   ParentSeqNum = X.ParentNumChildren;
  // else if (X->BackLink)
  //   ParentSeqNum = X.BackLink->ParentSeqNum;
  // else
  //   ParentSeqNum = X.SeqNum;
  // ParentSeqNum++;
  //
  // if (X->PallocChildren)
  //   CrefChildren = X.PallocChildren;
  // else
  //   CrefChildren = X.CrefChildren;
  // CrefChildren++;
  // PallocChildren = 0;
  //
  // ParLevel = 0;
  // CurLevel = X->CurLevel;
  // IncNestLevel(); EncLevel==CurLevel++;
  //   if (X == SglTag)
  //     SglTag.PopAppContStr(1);
  // St.PushAppContStr(HeadAppText,0);
  // St.AppContStr=St.HeadAppContStr();
  //
  // Example:
  //   <P>
  //     <FONT SIZE=4>
  //
  //   OR
  //
  //   </P>
  //     <FONT SIZE=4>
  //

  if (CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = 0;
    _ParentSequenceNum = CurNode_->ParentNumChildren() ?
                             CurNode_->ParentNumChildren():
                         CurNode_->_BackLink ?
                             CurNode_->_BackLink->ParentSequenceNumber():
                             CurNode_->AltZero_NextSequenceNumber();
    _ParentSequenceNum++;

    SetChildrenRef(_BackLink->GetChildrenCount() ?
                       _BackLink->GetChildrenCount():
                       CurNode_->GetChildrenRef());
    IncChildrenRef();
    SetChildrenCount(0);

    _ChdLevel = 0;
    _CurLevel = CurNode_->_CurLevel;
    IncNestLevel();

    if (CurNode_->SingletonTagFound())
      _NodeErased = CurNode_->PopAppContStr(true);
  }
}

/****************************************************************************/
void StackNodeInfo::SetStdTagToSglTagData(StackNodeInfo* CurNode_)
{
  // SeqNum = St.SeqNum;
  // SeqNum++;
  // CrefChildren = St.CrefChildren;
  // CrefChildren++;
  //
  // CurLevel = St->EncLevel;
  // EncLevel = 0;
  // ParLevel = 0;
  // Sgl.PushAppContStr(HeadAppText,1);
  // Sgl.AppContStr=Sgl.HeadAppContStr();
  //
  // Example:
  //   <FONT SIZE=4>
  //     <BR>
  //

  if (CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = CurNode_->AltZero_NextSequenceNumber();
    _SequenceNum++;

    SetChildrenRef(CurNode_->SetChildrenCount(0));
    IncChildrenRef();

    _CurLevel = (CurNode_->_EncLevel > CurNode_->_CurLevel) ?
                    CurNode_->_EncLevel:
                    CurNode_->_CurLevel+1;
    _EncLevel = _ChdLevel = 0;
  }
}

/****************************************************************************/
void StackNodeInfo::SetStdTagToStdTagData(StackNodeInfo* CurNode_)
{
  // SeqNum==0;
  // ParentSeqNum = OpSt.SeqNum;
  // ParentSeqNum++;
  //
  // OpSt.PallocChildren = 0;
  // CrefChildren = OpSt.PallocChildren;
  // CrefChildren++;
  // PallocChildren = 0;
  //
  // ParLevel = 0;
  // CurLevel = St->EncLevel;
  // IncNestLevel(); EncLevel==CurLevel++;
  // St_.PushAppContStr(HeadAppText,0);
  // St_.AppContStr=St_.HeadAppContStr();
  //
  // Example:
  //   <FONT SIZE=4> ...
  //     <H1 ALIGN="CENTER"> ...
  //

  if (CurNode_)
  {
    _NextSequenceNum = 0;
    _SequenceNum = 0;
    _ParentSequenceNum = CurNode_->AltZero_NextSequenceNumber();
    _ParentSequenceNum++;

    SetChildrenRef(CurNode_->SetChildrenCount(0));
    IncChildrenRef();
    SetChildrenCount(0);

    _ChdLevel = 0;
    _CurLevel = (CurNode_->_EncLevel > CurNode_->_CurLevel) ?
                    CurNode_->_EncLevel:
                    CurNode_->_CurLevel+1;
    IncNestLevel();
  }
}

/****************************************************************************/
void StackNodeInfo::SetStdTagData()
{
  // if (PrevUnorderedTagFound) {
  //   ParentSeqNum = NextSeqNum;
  //   PrevUnorderedTagFound = false;
  // }
  // else
  //   ParentSeqNum = 0;
  //
  // SeqNum = 0;
  // NextSeqNum = 0;
  // ParLevel = 0;
  // CurLevel = 0;
  // IncNestLevel(); EncLevel==CurLevel++;
  // Sgl.PushAppContStr(HeadAppText,0);
  // Sgl.AppContStr=Sgl.HeadAppContStr();

  if (_PrevUnorderedTagFound)
  {
    _ParentSequenceNum = _NextSequenceNum;
    _PrevUnorderedTagFound = false;
  }
  else
    _ParentSequenceNum = 0;

  _SequenceNum = 0;
  _NextSequenceNum = 0;
  SetChildrenCount(0);

  _ChdLevel = 0;
  _CurLevel = 0;
  IncNestLevel();
}

/****************************************************************************/
void StackNodeInfo::SetSingleTagData()
{
  // if (PrevUnorderedTagFound) {
  //   SeqNum = NextSeqNum;
  //   PrevUnorderedTagFound = false;
  // }
  // else
  //   SeqNum = 0;
  //
  // NextSeqNum = 0;
  // CurLevel = 0;
  // EncLevel = 0;
  // ParLevel = 0;
  // Sgl.PushAppContStr(HeadAppText,0);
  // Sgl.AppContStr=Sgl.HeadAppContStr();

  if (_PrevUnorderedTagFound)
  {
    _SequenceNum = _NextSequenceNum;
    _PrevUnorderedTagFound = false;
  }
  else
    _SequenceNum = 0;

  _NextSequenceNum = 0;
  _CurLevel = 0;
  _EncLevel = 0;
  _ChdLevel = 0;
}

/****************************************************************************/
void StackNodeInfo::SetNonContentTagData()
{
  // if (PrevUnorderedTagFound)
  //   SeqNum = NextSeqNum;
  // else
  //   SeqNum = 0;
  //
  // NextSeqNum = 0;
  // CurLevel = 0;
  // EncLevel = 0;
  // ParLevel = 0;

  if (_PrevUnorderedTagFound)
    _SequenceNum = _NextSequenceNum;
  else
    _SequenceNum = 0;

  _NextSequenceNum = 0;
  _CurLevel = 0;
  _EncLevel = 0;
  _ChdLevel = 0;
}

/****************************************************************************/
void StackNodeInfo::SetNonContentTagData(StackNodeInfo* CurNode_)
{
  // if (CurNode->ParentNumChildren)
  //   SeqNum = CurNode.ParentNumChildren;
  // else if (CurNode->BackLink)
  //   SeqNum = CurNode.BackLink->ParentSequenceNumber();
  // else if (CurNode->NumChildren)
  //   SeqNum = CurNode.NumChildren();
  // else
  //   SeqNum = CurNode.NextSequenceNumber();
  // SeqNum++;
  //
  // if (CurNode->PallocChildren)
  //   CrefChildren = CurNode.PallocChildren;
  // else
  //   CrefChildren = CurNode.CrefChildren;
  // CrefChildren++;
  //
  // CurLevel = CurNode->CurLevel;
  // EncLevel = 0;
  // ParLevel = 0;

  _SequenceNum = CurNode_->ParentNumChildren() ?
                     CurNode_->ParentNumChildren():
                 CurNode_->_BackLink ?
                     CurNode_->_BackLink->AltZero_ParentSequenceNumber():
                 CurNode_->NumChildren() ?
                     CurNode_->NumChildren():
                     CurNode_->NextSequenceNumber();
  _SequenceNum++;

  SetChildrenRef(CurNode_->GetChildrenCount() ?
                     CurNode_->GetChildrenCount():
                     CurNode_->GetChildrenRef());
  IncChildrenRef();

  CurNode_->_NCTagAddr = this;
  CurNode_->_NextSequenceNum = _SequenceNum;
  _NCTagAddr = this;
  _NextSequenceNum = _SequenceNum;

  _CurLevel = CurNode_->_CurLevel;
  _EncLevel = _ChdLevel = 0;
}

/****************************************************************************/
void StackNodeInfo::IncNestLevel()
{
  _EncLevel = _CurLevel + 1;
}

/****************************************************************************/
StackNodeInfo* StackNodeInfo::FindTagNode(StackNodeInfo* Node_)
{
  AppTextNode* AppTextPtr_ = NULL;
  StackNodeInfo* HomeNode_ = this;

  if (Node_ && Node_->AppText())
  {
    for (AppTextPtr_ = Node_->AppText();
         HomeNode_ &&
         (HomeNode_->_TextNode != AppTextPtr_ ||
          HomeNode_ != Node_);
         HomeNode_ = HomeNode_->_Next);

    return HomeNode_;
  }

  return NULL;
}

/****************************************************************************/
TagLinksSequencer* StackNodeInfo::Sequencer()
{
  if (_Parent)
    return _Parent->Sequencer();
}

/****************************************************************************/
const TagLinksSequencer* StackNodeInfo::Sequencer() const
{
  if (_Parent)
    return _Parent->Sequencer();
}

/****************************************************************************/
StackNodeInfo& StackNodeInfo::SetSequencer(TagLinksSequencer* Ptr_)
{
  if (_Parent)
    _Parent->SetSequencer(Ptr_);

  return *this;
}

/****************************************************************************/
HtmlSpecialCaseData* StackNodeInfo::HtmlSpecialCase()
{
  if (_Parent)
    return _Parent->HtmlSpecialCase();
}

/****************************************************************************/
const HtmlSpecialCaseData* StackNodeInfo::HtmlSpecialCase() const
{
  if (_Parent)
    return _Parent->HtmlSpecialCase();
}

/****************************************************************************/
StackNodeInfo& StackNodeInfo::SetHtmlSpecialCase(HtmlSpecialCaseData* Ptr_)
{
  if (_Parent)
    _Parent->SetHtmlSpecialCase(Ptr_);

  return *this;
}

/****************************************************************************/
void StackNodeInfo::SetUnlinked(bool Flag_)
{
  _Unlinked = Flag_;
}

/****************************************************************************/
int StackNodeInfo::ValidityChanged() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->ValidityChanged():0);
}

/****************************************************************************/
int StackNodeInfo::ReqTagCondChanged() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->ReqTagCondChanged():0);
}

/****************************************************************************/
int StackNodeInfo::ImpliedEndCondChanged() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->ImpliedEndCondChanged():0);
}

/****************************************************************************/
long StackNodeInfo::ReqTagFoundIndex() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->ReqTagFoundIndex():0);
}

/****************************************************************************/
bool StackNodeInfo::ImpliedTagFound() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->ImpliedTagFound():false);
}

/****************************************************************************/
int StackNodeInfo::ReqTagDirFound() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->ReqTagDirFound():0);
}

/****************************************************************************/
const TagElementStates* StackNodeInfo::EndTag() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->EndTag():NULL);
}

/****************************************************************************/
int StackNodeInfo::EndTagCond() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->EndTagCond():0);
}

/****************************************************************************/
bool StackNodeInfo::RequiredTagCondNotMet() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->RequiredTagCondNotMet():false);
}

/****************************************************************************/
bool StackNodeInfo::RequiredTagCondMet() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->RequiredTagCondMet():false);
}

/****************************************************************************/
bool StackNodeInfo::NoRequiredTag() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->NoRequiredTag():false);
}

/****************************************************************************/
bool StackNodeInfo::CurrentTagCondMet() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->CurrentTagCondMet():false);
}

/****************************************************************************/
bool StackNodeInfo::AtReqTagCondStopState(bool PartialMet_, bool FailOnly_) const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->AtReqTagCondStopState(PartialMet_, FailOnly_):false);
}

/****************************************************************************/
bool StackNodeInfo::TagAtImpliedEnd() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->TagAtImpliedEnd():false);
}

/****************************************************************************/
bool StackNodeInfo::TagIsValid() const
{
  SetTagElementState();
  return (_ElementPtr ? _ElementPtr->TagIsValid():false);
}

/****************************************************************************/
/****************************************************************************/
void StackNodeInfo::DumpAppTextNodeChkPtr(const StackNodeInfo* ptr_, ostream& os_, size_t fw_) const
{
  if (_TextNode && TagTypeInfo::ShowDebugDump() && ptr_ != this &&
      (!Erased() || TagTypeInfo::ShowErasedNodes()))
    _TextNode->DumpAppTextNode(ptr_, os_, fw_);
}

/****************************************************************************/
bool StackNodeInfo::HasShiftedTagContents() const
{
  bool Shifted_ = IsContentShifted() || IsContentAppended() || IsContentReady();
  StackNodeInfo* Ptr_ = _Next;

  while (Ptr_ && !Shifted_)
  {
    Shifted_ = Ptr_->IsContentShifted() ||
               Ptr_->IsContentAppended() ||
               Ptr_->IsContentReady();

    Ptr_ = Ptr_->_Next;
  }

  return Shifted_;
}

/****************************************************************************/
void StackNodeInfo::DumpKeyValuePairs(ostream& os_, size_t fw_, int Shift_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  size_t x;
  size_t max = _NumKeyValuePairs;
  char Padding_[256];
  char Buffer_[32];

  ChrString kvStr_;
  const KeyValuePairNode* kvPtr_ = NULL;
  const char* IndexTitleStr_ = _HasKVPairData ? "\tKVPair(":
                                                "\tIntraTagValue(";

  Padding_[0] = 0;
  if (Shift_)
    AddTrailing(Padding_, " ", Shift_);

  if (max && _TagKeyValuePairs)
  {
    if (_HasKVPairData || !_TagKeyValuePairs)
      memmove(Padding_+Shift_, "Tag Key Value Pairs:", strlen("Tag Key Value Pairs:")+1);
    else
      memmove(Padding_+Shift_, "IntraTag Values:", strlen("IntraTag Values:")+1);

    os_.width(fw_); os_ <<std::left <<Padding_ <<endl;

    for (x = 0; x < max; x++)
    {
      strcpy(Buffer_, IndexTitleStr_);
      UintToStr(x, Buffer_+strlen(Buffer_));

      if (_HasKVPairData)
      {
        kvPtr_ = _KVPairDataPtr->GiveKvPair(x);
        kvStr_ = kvPtr_ ? kvPtr_->GiveKVPairString():"NULL";

        strcat(Buffer_, "): ");
        memmove(Padding_+Shift_, Buffer_, strlen(Buffer_)+1);
        os_.width(fw_); os_ <<std::left <<Padding_ <<kvStr_ <<endl;
      }
      else if (_TagKeyValuePairs[x])
      {
        strcat(Buffer_, "): ");
        memmove(Padding_+Shift_, Buffer_, strlen(Buffer_)+1);
        os_.width(fw_); os_ <<std::left <<Padding_ <<(*_TagKeyValuePairs[x]) <<endl;
      }
      else
      {
        strcat(Buffer_, "): NULL");
        memmove(Padding_+Shift_, Buffer_, strlen(Buffer_)+1);
        os_.width(fw_); os_ <<std::left <<Padding_ <<endl;
      }
    }
  }
}

/****************************************************************************/
void StackNodeInfo::DumpTagAttrib(ostream& os_, size_t fw_, bool WalkDump_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  char Buffer_[256];
  int AbbrevDump_ = TAGREADER_ABBREV_DUMP;

  if (AbbrevDump_ && !WalkDump_)
  {
    os_ <<"******* Start : StackNodeInfo Dump *******" <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Position: "
                        <<TagBracketList::TagPositionToStr(_TagAttrib[TagEnums::TAGPOS])
                        <<endl;
  }
  else
  {
    os_ <<"******* Start : StackNodeInfo Dump *******" <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Validity: "
                        <<TagBracketList::TagValidityToStr(_TagAttrib[TagEnums::VALIDITY])
                        <<endl;

    os_.width(fw_); os_ <<std::left <<"Tag Attributes: "
                        <<TagBracketList::TagAttributesToStr(_TagAttrib[TagEnums::TAGATTR], Buffer_)
                        <<endl;

    os_.width(fw_); os_ <<std::left <<"Tag Nesting: "
                        <<TagNesting() <<endl;

    os_.width(fw_); os_ <<std::left <<"Tag Count: "
                        <<_TagAttrib[TagEnums::TAGCOUNT] <<endl;

    os_.width(fw_); os_ <<std::left <<"Tag Index: "
                        <<_TagAttrib[TagEnums::TAGINDEX] <<endl;

    os_.width(fw_); os_ <<std::left <<"Matching Index: "
                        <<_TagAttrib[TagEnums::MATCHINDEX] <<endl;

    os_.width(fw_); os_ <<std::left <<"Tag Position: "
                        <<TagBracketList::TagPositionToStr(_TagAttrib[TagEnums::TAGPOS])
                        <<endl;

    os_.width(fw_); os_ <<std::left <<"Tag List Seq.: "
                        <<_TagAttrib[TagEnums::LISTSEQ] <<endl;

    os_.width(fw_); os_ <<std::left <<"Tag Opt Tag Order: "
                        <<TagBracketList::TagOptReqTagOrderToStr(_TagAttrib[TagEnums::OPTREQTAGORDER])
                        <<endl;

    os_.width(fw_); os_ <<std::left <<"Tag Implied End: "
                        <<TagBracketList::TagImpliedEndToStr(_TagAttrib[TagEnums::IMPLIEDEND])
                        <<endl;

    os_.width(fw_); os_ <<std::left <<"Required Tag Cond: "
                        <<TagBracketList::TagReqTagCondToStr(_TagAttrib[TagEnums::REQTAGCOND], Buffer_)
                        <<endl;

    os_.width(fw_); os_ <<std::left <<"Implied End Cond: "
                        <<TagBracketList::TagImpliedEndCondToStr(_TagAttrib[TagEnums::IMPLIEDENDCOND], Buffer_)
                        <<endl;

    os_.width(fw_); os_ <<std::left <<"Validity State: "
                        <<TagBracketList::TagValidityToStr(_TagAttrib[TagEnums::VALIDITYSTATE])
                        <<endl;
  }
}

/****************************************************************************/
void StackNodeInfo::DumpShiftBack(ostream& os_, size_t fw_, StackNodeInfo* EncNode_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  os_ <<"*** Shifted Back Text Node Contents: Start ***" <<endl;
  if (EncNode_ && EncNode_->_TextNode)
    EncNode_->_TextNode->DumpAppTextNode(this, os_, fw_);
  os_ <<"*** Shifted Back Text Node Contents: End ***" <<endl <<endl;
}

/****************************************************************************/
void StackNodeInfo::DumpTagContentData(ostream& os_, size_t fw_, bool ShowUnshifted_, int Shift_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  int max = 256;
  int x1 = _ContentStr.strlen();
  int x2 = _IntraContentStr.strlen();
  max = (x1 > x2) ? ((x1 > 256) ? (x1+64):256):
                    ((x2 > 256) ? (x2+64):256);

  char* Padding_ = (char*)RawAllocateWith(MEMMATRIX, max*sizeof(char));
  bool ShiftedContent_ = IsContentShifted() || IsContentAppended() || IsContentReady();
  bool HasShiftedContent_ = ShiftedContent_ || HasShiftedTagContents();
  const ChrString* cPtr_ = _HasCommentData ?
                               _CommentDataPtr->CommentTagString():
                               NULL;

  Padding_[0] = 0;
  if (Shift_)
    AddTrailing(Padding_, " ", Shift_);

  if (Shift_ == 0 && (HasShiftedContent_ || ShowUnshifted_))
    os_ <<"******* Start : StackNodeInfo Tag Content Dump *******" <<endl;

  if (ShiftedContent_ || ShowUnshifted_)
  {
    memmove(Padding_+Shift_, "Content Type: ", strlen("Content Type: ")+1);
    os_.width(fw_); os_ <<std::left <<Padding_ <<_ContentType <<endl;
    memmove(Padding_+Shift_, "Has Tag Contents: ", strlen("Has Tag Contents: ")+1);
    os_.width(fw_); os_ <<std::left <<Padding_ <<_HasTagContents <<endl;
    memmove(Padding_+Shift_, "Has IntraTag Contents: ", strlen("Has IntraTag Contents: ")+1);
    os_.width(fw_); os_ <<std::left <<Padding_ <<_HasIntraTagContents <<endl;
    memmove(Padding_+Shift_, "Content String: ", strlen("Content String: ")+1);
    os_.width(fw_); os_ <<std::left <<Padding_ <<_ContentStr <<endl;
    memmove(Padding_+Shift_, "Intra Content String: ", strlen("Intra Content String: ")+1);
    os_.width(fw_); os_ <<std::left <<Padding_ <<_IntraContentStr <<endl;

    if (_HasKVPairData || !_TagKeyValuePairs)
      memmove(Padding_+Shift_, "Num Key Value Pairs: ", strlen("Num Key Value Pairs: ")+1);
    else
      memmove(Padding_+Shift_, "Num IntraTag Values: ", strlen("Num IntraTag Values: ")+1);

    os_.width(fw_); os_ <<std::left <<Padding_ <<_NumKeyValuePairs <<endl;
    DumpKeyValuePairs(os_, fw_, Shift_);

    memmove(Padding_+Shift_, "Tag KV Array Copied: ", strlen("Tag KV Array Copied: ")+1);
    os_.width(fw_); os_ <<std::left <<Padding_ <<_TagKVArrayCopied <<endl;
    memmove(Padding_+Shift_, "Content Shifted: ", strlen("Content Shifted: ")+1);
    os_.width(fw_); os_ <<std::left <<Padding_ <<ContentShiftedTypeToStr(_ContentShifted) <<endl;

    if (cPtr_)
    {
      memmove(Padding_+Shift_, "Comment Tag String: ", strlen("Comment Tag String: ")+1);
      os_.width(fw_); os_ <<std::left <<Padding_ <<*cPtr_ <<endl;
    }

    // StackNodeInfo Object data and pointer address
    memmove(Padding_+Shift_, "StackNode ObjectID: ", strlen("StackNode ObjectID: ")+1);
    os_.width(fw_); os_ <<std::left <<Padding_ <<_NodeID <<endl;
    memmove(Padding_+Shift_, "StackNode Addr: ", strlen("StackNode Addr: ")+1);
    os_.width(fw_); os_ <<std::left <<Padding_ <<this <<endl <<endl;

    if (_HasTagContents || _HasIntraTagContents || _NumKeyValuePairs)
      SetContentShown();
  }

  if (!ShiftedContent_ && (_Next && _Next != this))
  {
    if (HasShiftedContent_ || ShowUnshifted_)
    {
      Padding_[0] = 0;
      AddTrailing(Padding_, "*", Shift_+2);
      strcat(Padding_, " Shifted Tag Contents(");
      UintToStr((Shift_+2)/2, Padding_+strlen(Padding_));
      strcat(Padding_, "):");
      os_.width(fw_); os_ <<std::left <<Padding_ <<endl;
    }

    _Next->DumpTagContentData(os_, fw_, ShowUnshifted_, Shift_+2);
  }

  if (Shift_ == 0 && (HasShiftedContent_ || ShowUnshifted_))
    os_ <<"******* End : StackNodeInfo Tag Content Dump *******" <<endl <<endl;

  ::RawDeleteArray(Padding_);
}

/****************************************************************************/
void StackNodeInfo::DumpStackNodeInfo(ostream& os_, size_t fw_, bool WalkDump_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  // from TagElement Class
  if (_TagAttrib)
    DumpTagAttrib(os_, fw_, WalkDump_);
  else
    os_ <<"******* Start : StackNodeInfo Dump *******" <<endl;

  int AbbrevDump_ = TAGREADER_ABBREV_DUMP;

  if (AbbrevDump_ && !WalkDump_)
  {
    // from TagTypeInfo Class
    os_.width(fw_); os_ <<std::left <<"Element Ptr: " <<_ElementPtr <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Attr: " <<_TagAttr <<endl;

    os_.width(fw_); os_ <<std::left <<"Tag Found: " <<_TagFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Content Type Tag: " <<_ContentTypeTag <<endl;
    os_.width(fw_); os_ <<std::left <<"NonContent Type Tag: " <<_NonContentTypeTag <<endl;
    os_.width(fw_); os_ <<std::left <<"Has Pair Content: " <<_HasPairContent <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Type Found: " <<_TagTypeFound <<endl;

    os_.width(fw_); os_ <<std::left <<"Singleton Found: " <<_SingletonTagFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Empty Tag Found: " <<_EmptyTagFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Container Tag Found: " <<_ContainerTagFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Std Tag Found: " <<_StdTagFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Pair Ender Found: " <<_PairEnderFound <<endl <<endl;
  }
  else
  {
    // from Node Creation Parameters
    os_.width(fw_); os_ <<std::left <<"Doc. Type: " <<TagBracketList::TagDocFileTypeToStr(_DocType) <<endl;
    os_.width(fw_); os_ <<std::left <<"Search Type: " <<TagBracketList::TagSearchTypeToStr(_SearchType) <<endl;

    // from TagSearchResult Class
    os_.width(fw_); os_ <<std::left <<"Num Delims: " <<_NumDelims <<endl;
    os_.width(fw_); os_ <<std::left <<"Start Brk Found: " <<_StartBrkFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Start Brk Str: " <<_StartBrkStr <<endl;
    os_.width(fw_); os_ <<std::left <<"End Brk Found: " <<_EndBrkFound <<endl;
    os_.width(fw_); os_ <<std::left <<"End Brk Str: " <<_EndBrkStr <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Element Found: " <<_TagElementFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Element Str: " <<_TagElementStr <<endl;

    // from TagTypeInfo Class
    os_.width(fw_); os_ <<std::left <<"Element Ptr: " <<_ElementPtr <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Attr: " <<_TagAttr <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Brk Index: " <<_TagBrkIndex <<endl;
    os_.width(fw_); os_ <<std::left <<"End Brk Index: " <<_EndBrkIndex <<endl;
    os_.width(fw_); os_ <<std::left <<"Alt Index: " <<_AltIndex <<endl;

    os_.width(fw_); os_ <<std::left <<"Std Tag Count: " <<_StdTagCount <<endl;
    os_.width(fw_); os_ <<std::left <<"Nested Cmnt TagCnt: " <<_NestedCmntTagCount <<endl;
    os_.width(fw_); os_ <<std::left <<"C Open Cmnt TagCnt: " <<_CopenCmntTagCount <<endl;
    os_.width(fw_); os_ <<std::left <<"Nesting Error: " <<_NestingError <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Nest Cond: " <<_TagNestCond <<endl;

    os_.width(fw_); os_ <<std::left <<"Tag Found: " <<_TagFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Content Type Tag: " <<_ContentTypeTag <<endl;
    os_.width(fw_); os_ <<std::left <<"NonContent Type Tag: " <<_NonContentTypeTag <<endl;
    os_.width(fw_); os_ <<std::left <<"Has Pair Content: " <<_HasPairContent <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Type Found: " <<_TagTypeFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Link Type: " <<_LinkType <<endl;

    os_.width(fw_); os_ <<std::left <<"Search Run Number: " <<_SearchRun <<endl;
    os_.width(fw_); os_ <<std::left <<"Singleton Found: " <<_SingletonTagFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Empty Tag Found: " <<_EmptyTagFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Container Tag Found: " <<_ContainerTagFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Std Tag Found: " <<_StdTagFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Pair Ender Found: " <<_PairEnderFound <<endl <<endl;
    os_.width(fw_); os_ <<std::left <<"Unordered Tag Found: " <<_UnorderedTag <<endl <<endl;

    // From TagTypeInfo Class: Additional StackNodeInfo related data members
    os_.width(fw_); os_ <<std::left <<"*** Additional TagTypeInfo elements:" <<endl;
    os_.width(fw_); os_ <<std::left <<"Matching Info Link: " <<_MatchInfoLink <<endl;

    os_.width(fw_); os_ <<std::left <<"Found Cursor Tag: " <<_FoundCursorTag <<endl;
    os_.width(fw_); os_ <<std::left <<"Stk Tag Found: " <<_StkTagFound <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Content Valid: " <<_TagContValid <<endl;

    os_.width(fw_); os_ <<std::left <<"End Implied: " <<_EndImplied <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Append Done: " <<_TagAppendDone <<endl;
    os_.width(fw_); os_ <<std::left <<"Unlinked: " <<_Unlinked <<endl;
    os_.width(fw_); os_ <<std::left <<"Cursor Stack Node Found: " <<_CursorStkNodeFound <<endl;
  }

  if (_NCTagAddr)
    { os_.width(fw_); os_ <<std::left <<"NonContent Tag Addr: " <<_NCTagAddr <<endl; }
  else
    { os_.width(fw_); os_ <<std::left <<"NonContent Tag Addr: NULL" <<endl; }

  os_.width(fw_); os_ <<std::left <<"Parent Sequence Number: " <<_ParentSequenceNum <<endl;
  os_.width(fw_); os_ <<std::left <<"Next Sequence Number: " <<_NextSequenceNum <<endl;
  os_.width(fw_); os_ <<std::left <<"Sequence Number: " <<_SequenceNum <<endl;
  os_.width(fw_); os_ <<std::left <<"Parent Num Children: " <<ParentNumChildren() <<endl;
  os_.width(fw_); os_ <<std::left <<"Num Children: " <<NumChildren() <<endl;
  os_.width(fw_); os_ <<std::left <<"Current Level: " <<_CurLevel <<endl;
  os_.width(fw_); os_ <<std::left <<"Encountered Level: " <<_EncLevel <<endl;
  os_.width(fw_); os_ <<std::left <<"Child Level: " <<_ChdLevel <<endl;

  #if TAGREADER_SHOW_NODENESTING
    if (_NestedQ)
    {
      os_.width(fw_); os_ <<std::left <<"*NumNestedData : " <<_NumNestedData <<endl;
      ShowNodeNestingContents(NULL, os_, fw_);
    }
    else
    {
      os_.width(fw_); os_ <<std::left <<"(No NodeNesting Data)" <<endl;
    }
  #endif

  if (WalkDump_ && (_HasTagContents || _HasIntraTagContents || _NumKeyValuePairs))
    DumpTagContentData(os_, fw_, true);

  if (_TextNode)
  {
    os_ <<endl;
    _TextNode->DumpAppTextNode(this, os_, fw_);
  }
  else
  {
    os_ <<endl;
    os_.width(fw_); os_ <<std::left <<"(No AppTextNode Data)" <<endl <<endl;
  }

  // StackNodeInfo Object data
  if (!WalkDump_)
  {
    os_.width(fw_); os_ <<std::left <<"StackNode ObjectID: " <<_NodeID <<endl;
    os_.width(fw_); os_ <<std::left <<"StackNode Addr: " <<this <<endl;
    os_ <<"******* End : StackNodeInfo Dump *******" <<endl <<endl;
  }
}

/****************************************************************************/
void StackNodeInfo::TagInfoListWalkDump(ostream& os_, size_t fw_)
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  DumpStackNodeInfo(os_, fw_, true);

  // StackNodeInfo forward/backward links
  if (_ForwardLink || _BackLink)
  {
    const char* Istr_ =
       (_ForwardLink && _ForwardLink->_BackLink) ? "BackLink":"CurrLink";

    os_ <<"--------------------------- LinkForward : START ---------------------------";
    os_ <<endl <<"LinkForward(StackNodeInfo*, bool)" <<endl;
    StackNodeInfo::ShowStackAddr(this, StackNodeInfo::BACKFORW, Istr_);
    os_ <<"---------------------------- LinkForward : END ----------------------------" <<endl;
  }

  // StackNodeInfo Object data
  os_.width(fw_); os_ <<std::left <<"StackNode ObjectID: " <<_NodeID <<endl;
  os_.width(fw_); os_ <<std::left <<"StackNode Addr: " <<this <<endl;
  os_ <<"******* End : StackNodeInfo Dump *******" <<endl <<endl;
}

/****************************************************************************/
MEMORYOPS_DEFN(StackNodeInfo)

/****************************************************************************/
// TagDictionary class definition
/****************************************************************************/
TagDictionary::TagDictionary():
_TagTypeIndex(0),

_DocType(0),
_EofFound(false),
_TagType(TagEnums::HTMLSTYLE),

_StdVectorMax(0),
_StdVectorLength(0),

_EndVectorMax(0),
_EndVectorLength(0),

_TextBufferPtr(NULL),
_GenSearchTag(new TagTypeInfo()),
_Sequencer(new TagLinksSequencer()),
_SequencerCopied(false),
_HtmlSpecialCase(new HtmlSpecialCaseData()),
_HtmlSpecialCaseCopied(false),

_Fr(new FileReader),
_TagInfoMatrix((TagTypeInfo***)RawAllocateWith(MEMMATRIX, sizeof(TagTypeInfo**) * 2)),
_SearchResult(NULL),

_HasTag(false),
_DictSearchDone(false),
_ContentTypeTag(false),
_NonContentTypeTag(false),
_TagCreated(false),
_GenSearchDone(false),
_GenSearchFailed(NO_FAIL),
_SpecSearchDone(false),
_SpecSearchFailed(NO_FAIL),

_HtmlDictionaryCreated(false),
_PrevUnorderedTag(false),
_DumpTagContents(false)
{
  _TagStrIndex[0] = _TagStrIndex[1] = 0;
  _GenSearchTag->SetSequencer(Sequencer());
  _GenSearchTag->SetHtmlSpecialCase(HtmlSpecialCase());
  _GenSearchTag->InitAllHtmlTagSwitches(_TagSwitch);

  GrowVector(TagEnums::OPENING);
  GrowVector(TagEnums::CLOSING);
}

/****************************************************************************/
TagDictionary::TagDictionary(const TagDictionary& Obj_):
_TagTypeIndex(0),

_DocType(Obj_._DocType),
_EofFound(Obj_._EofFound),
_TagType(Obj_._TagType),

_StdVectorMax(0),
_StdVectorLength(0),

_EndVectorMax(0),
_EndVectorLength(0),

_TextBufferPtr(Obj_._TextBufferPtr),
_GenSearchTag(new TagTypeInfo()),
_Sequencer(Obj_._Sequencer),
_SequencerCopied(true),
_HtmlSpecialCase(Obj_._HtmlSpecialCase),
_HtmlSpecialCaseCopied(true),

_Fr(new FileReader),
_TagInfoMatrix((TagTypeInfo***)RawAllocateWith(MEMMATRIX, sizeof(TagTypeInfo**) * 2)),
_SearchResult(Obj_._SearchResult),

_HasTag(Obj_._HasTag),
_DictSearchDone(Obj_._DictSearchDone),
_ContentTypeTag(Obj_._ContentTypeTag),
_NonContentTypeTag(Obj_._NonContentTypeTag),
_TagCreated(Obj_._TagCreated),
_GenSearchDone(Obj_._GenSearchDone),
_GenSearchFailed(Obj_._GenSearchFailed),
_SpecSearchDone(Obj_._SpecSearchDone),
_SpecSearchFailed(Obj_._SpecSearchFailed),

_HtmlDictionaryCreated(Obj_._HtmlDictionaryCreated),
_PrevUnorderedTag(Obj_._PrevUnorderedTag),
_DumpTagContents(Obj_._DumpTagContents)
{
  _TagStrIndex[0] = 0;
  _TagStrIndex[1] = 0;
  _GenSearchTag->SetDocType(_DocType);
  _GenSearchTag->SetSequencer(Sequencer());
  _GenSearchTag->SetHtmlSpecialCase(HtmlSpecialCase());

  if (_TagType == TagEnums::HTMLSTYLE)
    _GenSearchTag->InitAllHtmlTagSwitches(_TagSwitch);
  else if (_TagType == TagEnums::XMLSTYLE)
    _GenSearchTag->InitAllXmlTagSwitches(_TagSwitch);
  else if (_TagType == TagEnums::CPPSTYLE)
    _GenSearchTag->InitCppLangTagSwitches(_TagSwitch);
  else if (_TagType == TagEnums::CSTYLE)
    _GenSearchTag->InitCLangTagSwitches(_TagSwitch);

  GrowVector(TagEnums::OPENING, Obj_._StdVectorMax);
  GrowVector(TagEnums::CLOSING, Obj_._EndVectorMax);
  CopyVectors(Obj_);
}

/****************************************************************************/
TagDictionary::~TagDictionary()
{
  delete _GenSearchTag;
  _GenSearchTag = NULL;

  delete _Fr;
  _Fr = NULL;

  if (!_SequencerCopied)
  {
    delete _Sequencer;
    _Sequencer = NULL;
    _SequencerCopied = true;
  }

  if (!_HtmlSpecialCaseCopied)
  {
    delete _HtmlSpecialCase;
    _HtmlSpecialCase = NULL;
    _HtmlSpecialCaseCopied = true;
  }

  EraseVectorContents();
  RawDeleteArray(_TagInfoMatrix[TagEnums::OPENING]);
  RawDeleteArray(_TagInfoMatrix[TagEnums::CLOSING]);
  RawDeleteArray(_TagInfoMatrix);
  _TagInfoMatrix = NULL;
}

/****************************************************************************/
TagDictionary* TagDictionary::Make()
{
  return (new TagDictionary());
}

/****************************************************************************/
TagDictionary* TagDictionary::Make(const TagDictionary& Obj_)
{
  return (new TagDictionary(Obj_));
}

/****************************************************************************/
TagTypeInfo* TagDictionary::SetRequiredTagCond(TagTypeInfo* TagPtr_,
                                               int ReqTagOrder_, int ImpliedEnd_,
                                               int TagCond_, int ImpliedEndCond_)
{
  if (TagPtr_)
    TagPtr_->SetRequiredCond(TagPtr_, ReqTagOrder_, ImpliedEnd_, TagCond_, ImpliedEndCond_);

  return TagPtr_;
}

/****************************************************************************/
TagTypeInfo* TagDictionary::SetHtmlContentTagType(TagTypeInfo* TagPtr_, int ContentType_)
{
  if (TagPtr_)
    TagPtr_->SetHtmlContentTagType(TagPtr_, ContentType_);

  return TagPtr_;
}

/****************************************************************************/
void TagDictionary::MakeHtmlTagDictionary()
{
  if (_HtmlDictionaryCreated)
    return;

  // Make html tag definitions for this tag dictionary object
  TagTypeInfo* vptr;

/**************************************/
    // a tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("A"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // a tag defintion end
/**************************************/
    // abbr tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("ABBR"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // abbr tag defintion end
/**************************************/
    // acronym tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("ACRONYM"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // acronym tag defintion end
/**************************************/
    // address tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("ADDRESS"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // address tag defintion end
/**************************************/
// applet tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("APPLET"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // applet tag defintion end
/**************************************/
    // area tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("AREA"), ">", NULL);
    // area tag definition end
/**************************************/
    // b tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("B"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // b tag defintion end
/**************************************/
    // base tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("BASE"), ">", NULL);
    // base tag definition end
/**************************************/
    // basefont tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("BASEFONT"), ">", NULL);
    // basefont tag definition end
/**************************************/
    // bdo tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("BDO"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // bdo tag defintion end
/**************************************/
    // big tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("BIG"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // big tag defintion end
/**************************************/
    // blockquote tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("BLOCKQUOTE"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // blockquote tag defintion end
/**************************************/
    // body tag defintion start
    //   (container tag)
    vptr = CreateTag(TagEnums::OPENING, false, true, 0,
                     "<", ChrString("BODY"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // body tag defintion end
/**************************************/
    // br tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("BR"), ">", NULL);
    // br tag definition end
/**************************************/
    // button tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("BUTTON"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // button tag defintion end
/**************************************/
    // caption tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("CAPTION"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // caption tag defintion end
/**************************************/
    // center tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("CENTER"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // center tag defintion end
/**************************************/
    // cite tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("CITE"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // cite tag defintion end
/**************************************/
    // code tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("CODE"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // code tag defintion end
/**************************************/
    // col tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("COL"), ">", NULL);
    // col tag definition end
/**************************************/
    // colgroup tag defintion start
    //   (container tag)
    vptr = CreateTag(TagEnums::OPENING, false, true, 0,
                     "<", ChrString("COLGROUP"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // colgroup tag defintion end
/**************************************/
    // dd tag defintion start
    //   (non container)
    vptr = CreateTag(TagEnums::OPENING, false, false, 0,
                     "<", ChrString("DD"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // dd tag defintion end
/**************************************/
    // del tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("DEL"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // del tag defintion end
/**************************************/
    // dfn tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("DFN"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // dfn tag defintion end
/**************************************/
    // dir tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("DIR"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // dir tag defintion end
/**************************************/
    // div tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("DIV"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // div tag defintion end
/**************************************/
    // dl tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("DL"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // dl tag defintion end
/**************************************/
    // dt tag defintion start
    //   (non container)
    vptr = CreateTag(TagEnums::OPENING, false, false, 0,
                     "<", ChrString("DT"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // dt tag defintion end
/**************************************/
    // em tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("EM"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // em tag defintion end
/**************************************/
    // fieldset tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("FIELDSET"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // fieldset tag defintion end
/**************************************/
    // font tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("FONT"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // font tag defintion end
/**************************************/
    // form tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("FORM"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // form tag defintion end
/**************************************/
    // frame tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("FRAME"), ">", NULL);
    // frame tag definition end
/**************************************/
    // frameset tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("FRAMESET"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // frameset tag defintion end
/**************************************/
    // h1 tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("H1"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // h1 tag defintion end
/**************************************/
    // h2 tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("H2"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // h2 tag defintion end
/**************************************/
    // h3 tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("H3"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // h3 tag defintion end
/**************************************/
    // h4 tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("H4"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // h4 tag defintion end
/**************************************/
    // h5 tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("H5"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // h5 tag defintion end
/**************************************/
    // h6 tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("H6"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // h6 tag defintion end
/**************************************/
    // head tag defintion start
    //   (container tag)
    vptr = CreateTag(TagEnums::OPENING, false, true, 0,
                     "<", ChrString("HEAD"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // head tag defintion end
/**************************************/
    // hr tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("HR"), ">", NULL);
    // hr tag definition end
/**************************************/
    // html tag defintion start
    //   (container tag)
    vptr = CreateTag(TagEnums::OPENING, false, true, 0,
                     "<", ChrString("HTML"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // html tag defintion end
/**************************************/
    // i tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("I"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // i tag defintion end
/**************************************/
    // iframe tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("IFRAME"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // iframe tag defintion end
/**************************************/
    // img tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("IMG"), ">", NULL);
    // img tag definition end
/**************************************/
    // input tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("INPUT"), ">", NULL);
    // input tag definition end
/**************************************/
    // ins tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("INS"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // ins tag defintion end
/**************************************/
    // isindex tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("ISINDEX"), ">", NULL);
    // isindex tag definition end
/**************************************/
    // kbd tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("KBD"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // kbd tag defintion end
/**************************************/
    // label tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("LABEL"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // label tag defintion end
/**************************************/
    // legend tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("LEGEND"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // legend tag defintion end
/**************************************/
    // li tag defintion start
    //   (non container)
    vptr = CreateTag(TagEnums::OPENING, false, false, 0,
                     "<", ChrString("LI"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // li tag defintion end
/**************************************/
    // link tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("LINK"), ">", NULL);
    // link tag definition end
/**************************************/
    // map tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("MAP"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // map tag defintion end
/**************************************/
    // menu tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("MENU"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // menu tag defintion end
/**************************************/
    // meta tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("META"), ">", NULL);
    // meta tag definition end
/**************************************/
    // noframes tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("NOFRAMES"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // noframes tag defintion end
/**************************************/
    // noscript tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("NOSCRIPT"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // noscript tag defintion end
/**************************************/
    // object tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("OBJECT"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // object tag defintion end
/**************************************/
    // ol tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("OL"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // ol tag defintion end
/**************************************/
    // optgroup tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("OPTGROUP"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // optgroup tag defintion end
/**************************************/
    // option tag defintion start
    //   (non container)
    vptr = CreateTag(TagEnums::OPENING, false, false, 0,
                     "<", ChrString("OPTION"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // option tag defintion end
/**************************************/
    // p tag defintion start
    //   (container tag)
    vptr = CreateTag(TagEnums::OPENING, false, true, 0,
                     "<", ChrString("P"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // p tag defintion end
/**************************************/
    // param tag definition start
    vptr = CreateTag(TagEnums::FORBIDDEN, true, false, 0,
                     "<", ChrString("PARAM"), ">", NULL);
    // param tag definition end
/**************************************/
    // pre tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("PRE"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // pre tag defintion end
/**************************************/
    // q tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("Q"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // q tag defintion end
/**************************************/
    // s tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("S"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // s tag defintion end
/**************************************/
    // samp tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("SAMP"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // samp tag defintion end
/**************************************/
    // script tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("SCRIPT"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // script tag defintion end
/**************************************/
    // select tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("SELECT"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // select tag defintion end
/**************************************/
    // small tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("SMALL"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // small tag defintion end
/**************************************/
    // span tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("SPAN"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // span tag defintion end
/**************************************/
    // strike tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("STRIKE"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // strike tag defintion end
/**************************************/
    // strong tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("STRONG"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // strong tag defintion end
/**************************************/
    // style tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("STYLE"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // style tag defintion end
/**************************************/
    // sub tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("SUB"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // sub tag defintion end
/**************************************/
    // sup tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("SUP"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // sup tag defintion end
/**************************************/
    // table tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("TABLE"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // table tag defintion end
/**************************************/
    // tbody tag defintion start
    //   (container tag)
    vptr = CreateTag(TagEnums::OPENING, false, true, 0,
                     "<", ChrString("TBODY"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // tbody tag defintion end
/**************************************/
    // td tag defintion start
    //   (non container)
    vptr = CreateTag(TagEnums::OPENING, false, false, 0,
                     "<", ChrString("TD"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // td tag defintion end
/**************************************/
    // textarea tag defintion start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("TEXTAREA"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // textarea tag defintion end
/**************************************/
    // tfoot tag defintion start
    //   (container tag)
    vptr = CreateTag(TagEnums::OPENING, false, true, 0,
                     "<", ChrString("TFOOT"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // tfoot tag defintion end
/**************************************/
    // th tag defintion start
    //   (container tag)
    vptr = CreateTag(TagEnums::OPENING, false, true, 0,
                     "<", ChrString("TH"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // th tag defintion end
/**************************************/
    // thead tag defintion start
    //   (container tag)
    vptr = CreateTag(TagEnums::OPENING, false, true, 0,
                     "<", ChrString("THEAD"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // thead tag defintion end
/**************************************/
    // title tag definition start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("TITLE"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // title tag definition end
/**************************************/
    // tr tag defintion start
    //   (container tag)
    vptr = CreateTag(TagEnums::OPENING, false, true, 0,
                     "<", ChrString("TR"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // tr tag defintion end
/**************************************/
    // tt tag definition start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("TT"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // tt tag definition end
/**************************************/
    // u tag definition start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("U"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // u tag definition end
/**************************************/
    // ul tag definition start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("UL"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // ul tag definition end
/**************************************/
    // var tag definition start
    vptr = CreateTag(TagEnums::OPENING, false, true,
                     TagEnums::PAIR_TAGATTR,
                     "<", ChrString("VAR"), ">", NULL);

    if (vptr)
      MakeMatchTag(vptr, true, true);
    // var tag definition end
/**************************************/

// Tag definition specializations
/**************************************/
// html tag defintion start
// HTML, Start tag: optional, End tag: optional, document root element
//
//   Must appear before <head>, </head>, <body> or </body> tag (if present),
//   end tag optional, implied at End Of File at: AT_EOF,
//   Validity: ignored
//   container tag: yes, no nested tags, duplicate tags disallowed
//   - html document tags
//
  TagStrLookupEqual(TagEnums::OPENING, "HTML");
  TagTypeInfo* TagPtr_ = DictionarySearchTag();
  SetupHtmlDocSpecialTags(TagPtr_);

/**************************************/
// head tag defintion start
// HEAD, Start tag: optional, End tag: optional, document head
//
//   Must appear after <html> tag (if present),
//   before </head>, <body> or </html> tag (if present),
//   end tag optional, implied at End Of File at: AT_EOF,
//   Validity: ignored
//   container tag: yes, no nested tags, duplicate tags disallowed
//   - html document tags
//
  TagStrLookupEqual(TagEnums::OPENING, "HEAD");
  TagPtr_ = DictionarySearchTag();
  SetupHtmlDocSpecialTags(TagPtr_);

/**************************************/
// body tag defintion start
// BODY, Start tag: optional, End tag: optional, document body
//
//   Must appear after <html> or <head> tag (if present),
//   before </body> or </html> tag (if present),
//   end tag optional, implied at End Of File at: AT_EOF
//   Validity: ignored
//   container tag: yes, no nested tags, duplicate tags disallowed
//   - html document tags
//
  TagStrLookupEqual(TagEnums::OPENING, "BODY");
  TagPtr_ = DictionarySearchTag();
  SetupHtmlDocSpecialTags(TagPtr_);

/**************************************/
// a tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "A");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// abbr tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "ABBR");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// acronym tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "ACRONYM");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// address tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "ADDRESS");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// b tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "B");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// basefont tag definition start
// end tag forbidden : implied end AT_NEXTTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "BASEFONT");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupBodyContentTag(TagPtr_, TagEnums::AT_NEXTTAG);

/**************************************/
// bdo tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "BDO");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupHtmlContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// big tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "BIG");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// blockquote tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "BLOCKQUOTE");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// br tag definition start
// end tag forbidden : implied end AT_NEXTTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "BR");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupBodyContentTag(TagPtr_, TagEnums::AT_NEXTTAG);

/**************************************/
// center tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "CENTER");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// cite tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "CITE");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// code tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "CODE");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// del tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "DEL");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// dfn tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "DFN");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// em tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "EM");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// font tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "FONT");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// h1 tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "H1");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// h2 tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "H2");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// h3 tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "H3");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// H4 tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "H4");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// h5 tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "H5");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// h6 tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "H6");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// hr tag definition start
// end tag forbidden : implied end AT_NEXTTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "HR");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupBodyContentTag(TagPtr_, TagEnums::AT_NEXTTAG);

/**************************************/
// i tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "I");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// ins tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "INS");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// kbd tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "KBD");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// meta tag definition start
// end tag forbidden : implied end AT_NEXTTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "META");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupHtmlContentTag(TagPtr_, TagEnums::AT_NEXTTAG);

/**************************************/
// pre tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "PRE");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// s tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "S");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// samp tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "SAMP");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// small tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "SMALL");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// strike tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "STRIKE");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// strong tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "STRONG");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// tt tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "TT");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// u tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "U");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// var tag definition start
//
  TagStrLookupEqual(TagEnums::OPENING, "VAR");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// applet tag defintion start
// APPLET, Start tag: required, End tag: required, java applet
//
//   Must appear after </head> tag or after <body> tag,
//   before </applet>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </applet> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "APPLET");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// area tag defintion start
// AREA, Start tag: required, End tag: forbidden, Empty tag,
//       client-side image map area
//
//   Must appear after <map> tag,
//   before </map> tag,
//   end tag forbidden,
//   Validity: Active only within <map> : </map> tags, otherwise inactive
//     otherwise inactive
//   - std tag pair content
//
  TagStrLookupEqual(TagEnums::OPENING, "AREA");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "MAP", TagEnums::AT_NEXTTAG);

/**************************************/
// base tag defintion start
// BASE, Start tag: Forbidden, Empty tag, document base URI
//
//   Must appear after <head> tag,
//   before </head>, <body> or </html> tag,
//   end tag forbidden,
//   Validity: Active only within <head> : </head>,<body> or </html> tags,
//     otherwise inactive
//   - head content
//
  TagStrLookupEqual(TagEnums::OPENING, "BASE");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupHeadContentTag(TagPtr_, TagEnums::AT_NEXTTAG);

/**************************************/
// button tag defintion start
// BUTTON, Start tag: required, End tag: required, push button
//
//   Must appear after <form> tag,
//   before </button> or </form> tag,
//   end tag required,
//   Validity: Active only within <form> : </form> tags, otherwise inactive
//     otherwise inactive
//   - </button> tag added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "BUTTON");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "FORM", TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// caption tag defintion start
// CAPTION, Start tag: required, End tag: required, table caption
//
//   Must appear after <table> tag,
//   before </caption> or </table> tag,
//   end tag required,
//   Validity: Active only within <table> : </table> tags, otherwise inactive
//     otherwise inactive
//   - </caption> tag added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "CAPTION");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "TABLE", TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// caption tag defintion start
// COLGROUP, Start tag: required, End tag: optional, table column group
//
//   Must appear after <table> tag,
//   before </table> tag, </colgroup> implied at </table>
//   end tag optional, implied at </table> at: AT_BEFORE_REQTAG
//   Validity: Active only within <table> : </table> tags, otherwise inactive
//     otherwise inactive
//   container tag: yes, no nested tags, duplicate tags allowed
//   - </colgroup> tag added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "COLGROUP");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "TABLE",
                              (TagEnums::AT_BEFORE_REQTAG | TagEnums::AT_IMPLIEDTAG));
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// col tag defintion start
// COL, Start tag: required, End Tag: forbidden, Table columnm
//
//   Must appear after <table> tag,
//   before </table> tag,
//   end tag forbidden,
//   Validity: Active only within <table> : </table> tags, otherwise inactive
//     otherwise inactive
//   - std tag pair content
//
  TagStrLookupEqual(TagEnums::OPENING, "COL");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "TABLE", TagEnums::AT_NEXTTAG);

/**************************************/
// dl tag defintion start
// DL, Start tag: required, End tag: required, definition list
//
//   Must appear after </head> tag or after <body> tag,
//   before </dl>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </dl> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "DL");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// dd tag defintion start
// DD, Start tag: required, End tag: optional, definition description
//
//   Must appear after <dl> tag,
//   before </dl> tag,
//   end tag optional, implied at <dt> or </dl> at: AT_IMPLIEDTAG
//   Validity: Active only within <dl> : <dt>, </dl> tags, otherwise inactive
//     otherwise inactive
//   container tag: no, no nested tags, duplicate tags allowed
//   - <dt>, </dd> tag added to : std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "DD");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "DL", TagEnums::AT_IMPLIEDTAG);
    AddRequiredBeforeTag(TagPtr_, TagEnums::OPENING, "DT");
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// dt tag defintion start
// DT, Start tag: required, End tag: optional, definition term
//
//   Must appear after <dl> tag,
//   before </dl> tag,
//   end tag optional, implied at <dd> or </dl> at: AT_IMPLIEDTAG
//   Validity: Active only within <dl> : <dd>, </dl> tags, otherwise inactive
//     otherwise inactive
//   container tag: no, no nested tags, duplicate tags allowed
//   - <dd>, </dt> tag added to : std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "DT");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "DL", TagEnums::AT_IMPLIEDTAG);
    AddRequiredBeforeTag(TagPtr_, TagEnums::OPENING, "DD");
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// dir tag defintion start
// DIR, Start tag: required, End tag: required, unordered list
//
//   Must appear after </head> tag or after <body> tag,
//   before </dir>, </body> or </html> tag,
//   end tag required
//   Validity: ignored
//   - </dir> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "DIR");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// div tag defintion start
// DIV, Start tag: required, End tag: required, generic language/style container
//
//   Must appear after </head> tag or after <body> tag,
//   before </div>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </div> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "DIV");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// fieldset tag defintion start
// FIELDSET, Start tag: required, End tag: required, form control group
//
//   Must appear after <form> tag,
//   before </fieldset>, </form> tag,
//   end tag required,
//   Validity: Active only within <form> : </form> tags, otherwise inactive
//     otherwise inactive
//   - </fieldset> tag added to: std tag pair content
//
  TagStrLookupEqual(TagEnums::OPENING, "FIELDSET");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "FORM", TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// form tag defintion start
// FORM, Start tag: required, End tag: required, interactive form
//
//   Must appear after </head> tag or after <body> tag,
//   before </form>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </form> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "FORM");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// frameset tag defintion start
// FRAMESET, Start tag: required, End tag: required, window subdivision
//
//   Must appear after </head> tag or after <body> tag,
//   before </frameset>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </frameset> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "FRAMESET");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// frame tag defintion start
// FRAME, Start tag: required, End tag: forbidden, subwindow
//
//   Must appear after <frameset> tag,
//   before </frameset> tag,
//   end tag forbidden,
//   Validity: Active only within <frameset> : </frameset> tags, otherwise inactive
//     otherwise inactive
//   - std tag pair content
//
  TagStrLookupEqual(TagEnums::OPENING, "FRAME");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "FRAMESET", TagEnums::AT_NEXTTAG);

/**************************************/
// iframe tag defintion start
// IFRAME, Start tag: required, End tag: required, inline subwindow
//
//   Must appear after </head> tag or after <body> tag,
//   before </iframe>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </iframe> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "IFRAME");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// img tag defintion start
// IMG, Start tag: required, End tag: forbidden, embedded image
//
//   Must appear after </head> tag or after <body> tag,
//   before </body> or </html> tag,
//   end tag forbidden,
//   Validity: ignored
//   - body content
//
  TagStrLookupEqual(TagEnums::OPENING, "IMG");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupBodyContentTag(TagPtr_, TagEnums::AT_NEXTTAG);

/**************************************/
// input tag defintion start
// INPUT, Start tag: required, End tag: forbidden, form control
//
//   Must appear after <form> tag,
//   before </form> tag,
//   end tag forbidden,
//   Validity: Active only within <form> : </form> tags, otherwise inactive
//     otherwise inactive
//   - std tag pair content
//
  TagStrLookupEqual(TagEnums::OPENING, "INPUT");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "FORM", TagEnums::AT_NEXTTAG);

/**************************************/
// isindex tag defintion start
// ISINDEX, Start tag: required, End tag: forbidden, single line prompt
//
//   Must appear after <form> tag,
//   before </form> tag,
//   end tag forbidden,
//   Validity: Active only within <form> : </form> tags, otherwise inactive
//     otherwise inactive
//   - std tag pair content
//
  TagStrLookupEqual(TagEnums::OPENING, "ISINDEX");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "FORM", TagEnums::AT_NEXTTAG);

/**************************************/
// label tag defintion start
// LABEL, Start tag: required, End tag: required, form field label text
//
//   Must appear after <form> tag,
//   before </label> or </form> tag,
//   end tag required,
//   Validity: Active only within <form> : </form> tags, otherwise inactive
//     otherwise inactive
//   - </label> tag added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "LABEL");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "FORM", TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// legend tag defintion start
// LEGEND, Start tag: required, End tag: required, fieldset legend
//
//   Must appear after <form> tag,
//   before </legend> or </form> tag,
//   end tag required,
//   Validity: Active only within <form> : </form> tags, otherwise inactive
//     otherwise inactive
//   - </legend> tag added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "LEGEND");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "FORM", TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// li tag defintion start
// LI, Start tag: required, End tag: optional, list item
//
//   Must appear after <ul> or <ol> tag,
//   before </ul> or </ol> tag,
//   end tag optional, implied at <li> at: AT_IMPLIEDTAG
//   Validity: Active only within <ul> : </ul> tags or
//                                <ol> : </ol> tags, otherwise inactive
//     otherwise inactive
//   container tag: no, no nested tags, duplicate tags allowed
//   - <ul> tag added to: std tag pair content : AFTER_REQTAG
//     </ul> tag added to: std tag pair content : BEFORE_REQTAG
//     </li> tag added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "LI");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "OL", TagEnums::AT_IMPLIEDTAG);
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "UL", TagEnums::AT_IMPLIEDTAG);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// link tag defintion start
// LINK, Start tag: required, End tag: forbidden, media independent link
//
//   Must appear after <head> tag,
//   before </head>, <body> or </html> tag,
//   end forbidden,
//   Validity: Active only within <head> : </head>,<body> or </html> tags,
//     otherwise inactive
//   - head content
//
  TagStrLookupEqual(TagEnums::OPENING, "LINK");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupHeadContentTag(TagPtr_, TagEnums::AT_NEXTTAG);

/**************************************/
// map tag defintion start
// MAP, Start tag: required, End tag: required, client-side image map
//
//   Must appear after </head> tag or after <body> tag,
//   before </map>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </map> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "MAP");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// menu tag defintion start
// MENU, Start tag: required, End tag: required, unordered list
//
//   Must appear after </head> tag or after <body> tag,
//   before </menu>, </body> or </html> tag,
//   end tag required
//   Validity: ignored
//   - </menu> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "MENU");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// noframes tag defintion start
// NOFRAMES, Start tag: required, End tag: required, alternate content container for non frame-based rendering
//
//   Must appear after <frameset> tag,
//   before </noframes> or </frameset> tag,
//   end tag required,
//   Validity: Active only within <frameset> : </frameset> tags, otherwise inactive
//     otherwise inactive
//   - </noframes> tag added to : std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "NOFRAMES");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "FRAMESET", TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// noscript tag defintion start
// NOSCRIPT, Start tag: required, End tag: required, alternate content container for non script-based rendering
//
//   Must appear after <html> or <head> or <body> tag,
//   before </noscript>, </head> or </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </noscript> tag added to : html doc content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "NOSCRIPT");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupHtmlContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// object tag defintion start
// OBJECT, Start tag: required, End tag: required, generic embedded object
//
//   Must appear after </head> tag or after <body> tag,
//   before </object>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </object> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "OBJECT");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// ol tag defintion start
// OL, Start tag: required, End tag: required, ordered list
//
//   Must appear after </head> tag or after <body> tag,
//   before </ol>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </ol> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "OL");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// optgroup tag defintion start
// OPTGROUP, Start tag: required, End tag: required, option group in selectable menu
//
//   Must appear after <select> tag,
//   before </optgroup> or </select> tag,
//   end tag required,
//   Validity: Active only within <select> : </select> tags, otherwise inactive
//     otherwise inactive
//   - </optgroup> tag added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "OPTGROUP");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "SELECT", TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// option tag defintion start
// OPTION, Start tag: required, End tag: optional, selectable choice in selectable menu
//
//   Must appear after <select> tag,
//   before </select> tag,
//   end tag optional, implied at <option> at: AT_IMPLIEDTAG
//   Validity: Active only within <select> : </select> tags, otherwise inactive
//     otherwise inactive
//   container tag: no, no nested tags, duplicate tags allowed
//   - </option> tag added to : std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "OPTION");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "SELECT", TagEnums::AT_IMPLIEDTAG);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// p tag definition start
// P, Start tag: required, End tag: optional, text paragraph
//
//   Must appear after </head> tag or after <body> tag,
//   before </q>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   container tag: yes, no nested tags, duplicate tags allowed
//   - </p> tag added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "P");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_,
                        (TagEnums::AT_BEFORE_REQTAG | TagEnums::AT_IMPLIEDTAG));
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// param tag defintion start
// PARAM, Start tag: required, End tag: forbidden, object initialization
//
//   Must appear after <object> tag,
//   before </object> tag,
//   end tag forbidden,
//   Validity: Active only within <object> : </object> tags, otherwise inactive
//     otherwise inactive
//   - std tag pair content
//
  TagStrLookupEqual(TagEnums::OPENING, "PARAM");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "OBJECT", TagEnums::AT_NEXTTAG);

/**************************************/
// q tag defintion start
// Q, Start tag: required, End tag: required, short inlined quotation
//
//   Must appear after </head> tag or after <body> tag,
//   before </q>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </q> tag added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "Q");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// script tag defintion start
// SCRIPT, Start tag: required, End tag: required, script statements
//
//   Must appear after <html> or <head> or <body> tag,
//   before </script>, </head> or </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </script> added to: html doc content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "SCRIPT");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupHtmlContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// select tag defintion start
// SELECT, Start tag: required, End tag: required, option selector
//
//   Must appear after </head> tag or after <body> tag,
//   before </select>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </select> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "SELECT");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// span tag defintion start
// SPAN, Start tag: required, End tag: required, generic language/style container
//
//   Must appear after <div> tag,
//   before </span> or </div> tag,
//   end tag required,
//   Validity: Active only within <div> : </div> tags, otherwise inactive
//     otherwise inactive
//   - </span> tag added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "SPAN");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "DIV", TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// style tag defintion start
// STYLE, Start tag: required, End tag: required, header style information
//
//   Must appear after <head> tag,
//   before </style>, </head>, <body> or </html> tag,
//   end tag required,
//   Validity: Active only within <head> : </head>,<body> or </html> tags,
//     otherwise inactive
//   - </style> tag added to: head content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "STYLE");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupHeadContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// style tag defintion start
// SUB, Start tag: required, End tag: required, subscript
//
//   Must appear after </head> tag or after <body> tag,
//   before </sub>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </sub> tag added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "SUB");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// style tag defintion start
// SUP, Start tag: required, End tag: required, superscript
//
//   Must appear after </head> tag or after <body> tag,
//   before </sup>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </sup> tag added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "SUP");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// style tag defintion start
// TABLE, Start tag: required, End tag: required, table
//
//   Must appear after </head> tag or after <body> tag,
//   before </table>, </body> or </html> tag,
//   end tag required,
//   Validity: ignored
//   - </table> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "TABLE");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// tbody tag defintion start
// TBODY, Start tag: optional, End tag: optional, table body tag
//
//   Must appear after <table> tag
//   before </table> tag,
//   end optional, implied at </table> at: AT_BEFORE_REQTAG
//   Validity: Active only within <table> : </table> tags, otherwise inactive
//     otherwise inactive
//   container tag: yes, no nested tags, duplicate tags allowed
//   - </tbody> tag added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "TBODY");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "TABLE",
                              (TagEnums::AT_BEFORE_REQTAG | TagEnums::AT_IMPLIEDTAG));
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// td tag defintion start
// TD, Start tag: required, End tag: optional, table data cell
//
//   Must appear after <table> tag
//   before </table> tag,
//   end optional, implied at <tr> or <td> at: AT_IMPLIEDTAG
//   Validity: Active only within <table> : </table> tags, otherwise inactive
//     otherwise inactive
//   container tag: no, no nested tags, duplicate tags allowed
//   - <tr>, </td> tag added to : std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "TD");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "TABLE", TagEnums::AT_IMPLIEDTAG);
    AddRequiredBeforeTag(TagPtr_, TagEnums::OPENING, "TR");
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// textarea tag defintion start
// TEXTAREA, Start tag: required, End tag: required, multiline text field
//
//   Must appear after <form> tag,
//   before </textarea> or </form> tag,
//   end tag required,
//   Validity: Active only within <form> : </form> tags, otherwise inactive
//     otherwise inactive
//   - </textarea> added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "TEXTAREA");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "FORM", TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// tfoot tag defintion start
// TFOOT, Start tag: required, End tag: optional, table footer
//
//   Must appear after <table> or <thead> tag,
//   before <tbody>, </tfoot> or </table> tag,
//   end tag optional, implied at </table> at: AT_BEFORE_REQTAG
//   Validity: Active only within <table> : </table> tags, otherwise inactive
//     otherwise inactive
//   container tag: yes, no nested tags, duplicate tags disallowed
//   - <tbody>, </tfoot> tag added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "TFOOT");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "TABLE", TagEnums::AT_BEFORE_REQTAG);
    AddRequiredBeforeTag(TagPtr_, TagEnums::OPENING, "TBODY");
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// th tag defintion start
// TH, Start tag: required, End tag: optional, table header cell
//
//   Must appear after <table> tag
//   before </table> tag,
//   end tag optional, implied at </table> at: AT_BEFORE_REQTAG
//   Validity: Active only within <table> : </table> tags, otherwise inactive
//     otherwise inactive
//   container tag: yes, no nested tags, duplicate tags allowed
//   - </th> tag added to : std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "TH");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "TABLE",
                              (TagEnums::AT_BEFORE_REQTAG | TagEnums::AT_IMPLIEDTAG));
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// thead tag defintion start
// THEAD, Start tag: required, End tag: optional, table header
//
//   Must appear after <table> tag,
//   before <tbody>, </thead> or </table> tag,
//   End tag optional, implied at </table> at: AT_BEFORE_REQTAG
//   Validity: Active only within <table> : </table> tags, otherwise inactive
//     otherwise inactive
//   container tag: yes, no nested tags, duplicate tags disallowed
//   - <tbody>, </thead> tags added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "THEAD");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "TABLE", TagEnums::AT_BEFORE_REQTAG);
    AddRequiredBeforeTag(TagPtr_, TagEnums::OPENING, "TBODY");
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// title tag defintion start
// TITLE, Start tag: required, End tag: required, document title
//
//   Must appear after <head> tag,
//   before </title>, </head>, <body> or </html> tag,
//   end tag required,
//   Validity: Active only within <head> : </head>,<body> or </html> tags,
//     otherwise inactive
//   - </title> tag added to: head content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "TITLE");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupHeadContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// tr tag defintion start
// TR, Start tag: required, End tag: optional, Table row
//
//   Must appear after <table> tag,
//   before </tr> or </table> tag,
//   End tag optional, implied at </table> at: AT_BEFORE_REQTAG
//   Validity: Active only within <table> : </table> tags, otherwise inactive
//     otherwise inactive
//   container tag: yes, no nested tags, duplicate tags allowed
//   - </tr> added to: std tag pair content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "TR");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING, "TABLE",
                              (TagEnums::AT_BEFORE_REQTAG | TagEnums::AT_IMPLIEDTAG));
    SetupMatchingEndTag(TagPtr_);
  }

/**************************************/
// ul tag defintion start
// UL, Start tag: required, End tag: required, unordered list
//
//   Must appear after </head> tag or after <body> tag,
//   before </ul>, </body> or </html> tag,
//   end tag required
//   Validity: ignored
//   - </ul> tags added to: body content : BEFORE_REQTAG
//
  TagStrLookupEqual(TagEnums::OPENING, "UL");
  TagPtr_ = DictionarySearchTag();

  if (TagPtr_)
  {
    SetupBodyContentTag(TagPtr_, TagEnums::NO_IMPLIEDEND);
    SetupMatchingEndTag(TagPtr_);
  }

  _HtmlDictionaryCreated = true;
}

/****************************************************************************/
TagTypeInfo* TagDictionary::SetupHtmlDocSpecialTags(TagTypeInfo* TagPtr_)
{
  // HTML, Start tag: optional, End tag: optional, document root element
  //
  //   Must appear before </html>, </body>, <body>, </head> or <head> tag (if present),
  //   end tag optional, implied at End Of File at: AT_EOF,
  //   Validity: ignored
  //   container tag: yes, no nested tags, duplicate tags disallowed
  //
  // HEAD, Start tag: optional, End tag: optional, document head
  //
  //   Must appear after <html> tag (if present),
  //   before </html>, </body>, <body> or </head> tag (if present),
  //   end tag optional, implied at End Of File at: AT_EOF,
  //   Validity: ignored
  //   container tag: yes, no nested tags, duplicate tags disallowed
  //
  // BODY, Start tag: optional, End tag: optional, document body
  //
  //   Must appear after <html> or <head> tag (if present),
  //   before </body> or </html> tag (if present),
  //   end tag optional, implied at End Of File at: AT_EOF
  //   Validity: ignored
  //   container tag: yes, no nested tags, duplicate tags disallowed
  //
  if (!TagPtr_)
    return TagPtr_;

  ChrString TagStr_ = TagPtr_->ElementStr();

  if (TagStr_ == ChrString("HTML"))
  {
    if (TagPtr_)
    {
      SetHtmlContentTagType(TagPtr_, TagEnums::HTMLCONTENT_SPECIAL);
      SetRequiredTagCond(TagPtr_, TagEnums::BEFORE_REQTAG, TagEnums::AT_EOF);
      TagPtr_ = SetupMatchingEndTag(TagPtr_);
    }

    if (TagPtr_)
      TagPtr_ = SetupStdTagPairContentTagAux(TagPtr_, TagEnums::OPENING, true,
                                             ChrString("BODY"), TagEnums::AT_EOF);

    if (TagPtr_)
      TagPtr_ = SetupStdTagPairContentTagAux(TagPtr_, TagEnums::OPENING, true,
                                             ChrString("HEAD"), TagEnums::AT_EOF);
  }
  else if (TagStr_ == ChrString("HEAD"))
  {
    if (TagPtr_)
    {
      SetHtmlContentTagType(TagPtr_, TagEnums::HTMLCONTENT_SPECIAL);
      SetRequiredTagCond(TagPtr_, TagEnums::BEFORE_REQTAG, TagEnums::AT_EOF);
      AddRequiredBeforeTag(TagPtr_, TagEnums::CLOSING, ChrString("HTML"));
    }

    if (TagPtr_)
      TagPtr_ = SetupStdTagPairContentTagAux(TagPtr_, TagEnums::OPENING, true,
                                             ChrString("BODY"), TagEnums::AT_EOF);

    if (TagPtr_)
      TagPtr_ = SetupMatchingEndTag(TagPtr_);
  }
  else if (TagStr_ == ChrString("BODY"))
  {
    if (TagPtr_)
    {
      SetHtmlContentTagType(TagPtr_, TagEnums::HTMLCONTENT_SPECIAL);
      SetRequiredTagCond(TagPtr_, TagEnums::BEFORE_REQTAG, TagEnums::AT_EOF);
      AddRequiredBeforeTag(TagPtr_, TagEnums::CLOSING, ChrString("HTML"));
    }

    if (TagPtr_)
      TagPtr_ = SetupMatchingEndTag(TagPtr_);
  }

  return TagPtr_;
}

/****************************************************************************/
TagTypeInfo* TagDictionary::SetupHtmlContentTag(TagTypeInfo* TagPtr_, int ImpliedEnd_)
{
  // For content type tags within both html document head and document body
  // Must appear after <html> or <head> or <body> tag,
  // before </head> or </body> or </html> tag,
  // end tag required or forbidden,
  // Validity: ignored
  //
  if (!TagPtr_)
    return TagPtr_;

  if (TagPtr_)
  {
    SetHtmlContentTagType(TagPtr_, TagEnums::HTMLCONTENT_HTML);
    TagPtr_ = SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING,
                                        ChrString("HEAD"), ImpliedEnd_);
  }

  if (TagPtr_)
    TagPtr_ = SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING,
                                        ChrString("BODY"), ImpliedEnd_);

  if (TagPtr_)
    return SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING,
                                     ChrString("HTML"), ImpliedEnd_);

  return TagPtr_;
}

/****************************************************************************/
TagTypeInfo* TagDictionary::SetupHeadContentTag(TagTypeInfo* TagPtr_, int ImpliedEnd_)
{
  // For content type tags within html document head
  // Must appear after <head> tag,
  // before </head>, <body> or </html> tag,
  // end tag: required or forbidden,
  // Validity: Active only within <head> : </head>,<body> or </html> tags,
  //   otherwise inactive
  //
  if (!TagPtr_)
    return TagPtr_;

  TagCountData* TagCnt_ = _GenSearchTag->GiveTagCountData();
  const char* StartBrk_ = TagCnt_->GiveTagBrk(TagEnums::END_DEX);
  const char* EndBrk_ = TagCnt_->GiveTagBrk(TagEnums::END_DEX+1);
  ChrString ElementStr_ = "HTML";
  int TagPos_ = TagEnums::CLOSING;

  TagTypeInfo* SurrTag_ = new TagTypeInfo();
  SurrTag_->SetTagBrk(StartBrk_, EndBrk_, NULL);
  SurrTag_->SetElementStr(true, new ChrString(ElementStr_), true);
  SurrTag_->SetLookupRefTag(true);

  if (TagPtr_ && TagLookupEqual(TagPos_, SurrTag_))
  {
    TagElement* TagElmPtr_;
    delete SurrTag_;
    SurrTag_ = DictionarySearchTag();
    SetRequiredTagCond(TagPtr_, TagEnums::SURROUND_REQTAG, ImpliedEnd_);

    TagElmPtr_ = SurrTag_->GiveMatchingElementPtr(TagEnums::CLOSING);
    if (TagElmPtr_)
    {
      SetHtmlContentTagType(TagPtr_, TagEnums::HTMLCONTENT_HEAD);
      TagPtr_ = TagPtr_->AddRequiredTagElement(TagPtr_, TagElmPtr_, TagEnums::BEFORE_REQTAG);
    }

    StartBrk_ = TagCnt_->GiveTagBrk(TagEnums::STD_DEX);
    EndBrk_ = TagCnt_->GiveTagBrk(TagEnums::STD_DEX+1);
    ElementStr_ = "BODY";
    TagPos_ = TagEnums::OPENING;

    SurrTag_ = new TagTypeInfo();
    SurrTag_->SetTagBrk(StartBrk_, EndBrk_, NULL);
    SurrTag_->SetElementStr(true, new ChrString(ElementStr_), true);
    SurrTag_->SetLookupRefTag(true);

    if (TagPtr_ && TagLookupEqual(TagPos_, SurrTag_))
    {
      delete SurrTag_;
      SurrTag_ = DictionarySearchTag();
      TagElmPtr_ = SurrTag_->GiveMatchingElementPtr(TagEnums::OPENING);

      if (TagElmPtr_)
        TagPtr_ = TagPtr_->AddRequiredTagElement(TagPtr_, TagElmPtr_, TagEnums::BEFORE_REQTAG);
    }
    else
    {
      delete SurrTag_;
      SurrTag_ = NULL;
    }
  }
  else
  {
    delete SurrTag_;
    SurrTag_ = NULL;
  }

  if (TagPtr_)
    return SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING,
                                     ChrString("HEAD"), ImpliedEnd_);

  return TagPtr_;
}

/****************************************************************************/
TagTypeInfo* TagDictionary::SetupBodyContentTag(TagTypeInfo* TagPtr_, int ImpliedEnd_)
{
  // For content type tags within html document body
  // Must appear after </head> tag or after <body> tag,
  // before </body> or </html> tag,
  // end tag: required or forbidden,
  // Validity: Active only within </head> or <body> : </body> or </html> tags,
  //   otherwise inactive
  //
  if (!TagPtr_)
    return TagPtr_;

  TagCountData* TagCnt_ = _GenSearchTag->GiveTagCountData();
  const char* StartBrk_ = TagCnt_->GiveTagBrk(TagEnums::END_DEX);
  const char* EndBrk_ = TagCnt_->GiveTagBrk(TagEnums::END_DEX+1);
  ChrString ElementStr_ = "HEAD";
  int TagPos_ = TagEnums::CLOSING;

  TagTypeInfo* SurrTag_ = new TagTypeInfo();
  SurrTag_->SetTagBrk(StartBrk_, EndBrk_, NULL);
  SurrTag_->SetElementStr(true, new ChrString(ElementStr_), true);
  SurrTag_->SetLookupRefTag(true);

  if (TagPtr_ && TagLookupEqual(TagPos_, SurrTag_))
  {
    TagElement* TagElmPtr_;
    delete SurrTag_;
    SurrTag_ = DictionarySearchTag();
    SetRequiredTagCond(TagPtr_, TagEnums::SURROUND_REQTAG, ImpliedEnd_);

    TagElmPtr_ = SurrTag_->GiveMatchingElementPtr(TagEnums::CLOSING);
    if (TagElmPtr_)
    {
      SetHtmlContentTagType(TagPtr_, TagEnums::HTMLCONTENT_BODY);
      TagPtr_ = TagPtr_->AddRequiredTagElement(TagPtr_, TagElmPtr_, TagEnums::AFTER_REQTAG);
    }

    ElementStr_ = "HTML";

    SurrTag_ = new TagTypeInfo();
    SurrTag_->SetTagBrk(StartBrk_, EndBrk_, NULL);
    SurrTag_->SetElementStr(true, new ChrString(ElementStr_), true);
    SurrTag_->SetLookupRefTag(true);

    if (TagPtr_ && TagLookupEqual(TagPos_, SurrTag_))
    {
      delete SurrTag_;
      SurrTag_ = DictionarySearchTag();
      TagElmPtr_ = SurrTag_->GiveMatchingElementPtr(TagEnums::CLOSING);

      if (TagElmPtr_)
        TagPtr_ = TagPtr_->AddRequiredTagElement(TagPtr_, TagElmPtr_, TagEnums::BEFORE_REQTAG);
    }
    else
    {
      delete SurrTag_;
      SurrTag_ = NULL;
    }
  }
  else
  {
    delete SurrTag_;
    SurrTag_ = NULL;
  }

  if (TagPtr_)
    return SetupStdTagPairContentTag(TagPtr_, TagEnums::OPENING,
                                     ChrString("BODY"), ImpliedEnd_);

  return TagPtr_;
}

/****************************************************************************/
TagTypeInfo* TagDictionary::SetupStdTagPairContentTag(TagTypeInfo* TagPtr_, int TagPos_,
                                                      const ChrString& ElementStr_, int ImpliedEnd_)
{
  return SetupStdTagPairContentTagAux(TagPtr_, TagPos_, false, ElementStr_,
                                      ImpliedEnd_, TagEnums::HTMLCONTENT_STDTAGPAIR);
}

/****************************************************************************/
TagTypeInfo* TagDictionary::SetupStdTagPairContentTagAux(TagTypeInfo* TagPtr_, int TagPos_, bool NoSurrPair_,
                                                         const ChrString& ElementStr_, int ImpliedEnd_,
                                                         int HtmlContentType_)
{
  // For content type tags within std tag pairs
  // Must appear after <tagname> tag,
  // before </tagname> tag
  // end tag : required or forbidden
  //
  // TagPos == OPENING
  // Validity: Active only within <tagname> : </tagname> tags,
  //   otherwise inactive
  //
  // TagPos == CLOSING
  // Validity: Active only within </tagname> : <tagname> tags,
  //   otherwise inactive
  //
  // TagPos == OPENING, NoSurrTag == true
  // Validity: Active only before <tagname>, </tagname> tags,
  //   otherwise inactive
  //
  // *tagname being the tag element name such as "script", etc...
  //
  if (!TagPtr_ ||
      (TagPos_ != TagEnums::OPENING &&
       TagPos_ != TagEnums::FORBIDDEN &&
       TagPos_ != TagEnums::CLOSING))
    return TagPtr_;

  int PrevTagPos_ = TagPos_;
  int TagIndex_ = (NoSurrPair_ && TagPos_ == TagEnums::OPENING) ? TagEnums::END_DEX:
                  (TagPos_ == TagEnums::CLOSING) ? TagEnums::END_DEX:TagEnums::STD_DEX;

  TagCountData* TagCnt_ = _GenSearchTag->GiveTagCountData();
  const char* StartBrk_ = TagCnt_->GiveTagBrk(TagIndex_);
  const char* EndBrk_ = TagCnt_->GiveTagBrk(TagIndex_+1);
  int TagReqCond_ = (NoSurrPair_ && TagPos_ == TagEnums::OPENING) ?
                        TagEnums::BEFORE_REQTAG:
                        TagEnums::SURROUND_REQTAG;

  TagTypeInfo* SurrTag_ = new TagTypeInfo();
  SurrTag_->SetTagBrk(StartBrk_, EndBrk_, NULL);
  SurrTag_->SetElementStr(true, new ChrString(ElementStr_), true);
  SurrTag_->SetLookupRefTag(true);

  if (NoSurrPair_ && TagPos_ == TagEnums::OPENING)
    TagPos_ = TagEnums::CLOSING;

  if (TagPtr_ && TagLookupEqual(TagPos_, SurrTag_))
  {
    TagElement* TagElmPtr_;
    delete SurrTag_;
    SurrTag_ = DictionarySearchTag();
    SetRequiredTagCond(TagPtr_, TagReqCond_, ImpliedEnd_);

    TagPos_ = (NoSurrPair_ && TagPos_ == TagEnums::CLOSING) ?
                  TagEnums::CLOSING:
                  PrevTagPos_;

    TagElmPtr_ = SurrTag_->GiveMatchingElementPtr(TagPos_);
    TagReqCond_ = (NoSurrPair_ && TagPos_ == TagEnums::CLOSING) ?
                      TagEnums::BEFORE_REQTAG:
                      TagEnums::AFTER_REQTAG;

    if (TagElmPtr_)
    {
      if (HtmlContentType_ > 0)
        SetHtmlContentTagType(TagPtr_, HtmlContentType_);

      TagPtr_ = TagPtr_->AddRequiredTagElement(TagPtr_, TagElmPtr_, TagReqCond_);
    }

    if (TagPtr_)
      if (!NoSurrPair_)
      {
        if (TagPos_ == TagEnums::CLOSING)
          TagElmPtr_ = SurrTag_->GiveMatchingElementPtr(TagEnums::OPENING);
        else
          TagElmPtr_ = SurrTag_->GiveMatchingElementPtr(TagEnums::CLOSING);

        if (TagElmPtr_)
          TagPtr_ = TagPtr_->AddRequiredTagElement(TagPtr_, TagElmPtr_, TagEnums::BEFORE_REQTAG);
      }
      else if (NoSurrPair_ && TagPos_ == TagEnums::CLOSING)
      {
        TagElmPtr_ = SurrTag_->GiveMatchingElementPtr(TagEnums::OPENING);

        if (TagElmPtr_)
          TagPtr_ = TagPtr_->AddRequiredTagElement(TagPtr_, TagElmPtr_, TagEnums::BEFORE_REQTAG);
      }
  }
  else
  {
    delete SurrTag_;
    SurrTag_ = NULL;
  }

  return TagPtr_;
}

/****************************************************************************/
TagTypeInfo* TagDictionary::SetupMatchingEndTag(TagTypeInfo* TagPtr_)
{
  if (!TagPtr_)
    return TagPtr_;

  TagCountData* TagCnt_ = _GenSearchTag->GiveTagCountData();
  const char* StartBrk_ = TagCnt_->GiveTagBrk(TagEnums::END_DEX);
  const char* EndBrk_ = TagCnt_->GiveTagBrk(TagEnums::END_DEX+1);
  TagElement* TagElmPtr_ = TagPtr_->GiveMatchingElementPtr(TagEnums::CLOSING);
  TagElement* MatchElmPtr_ = TagPtr_->GiveTagElementPtr();
  ChrString ElementStr_ = TagElmPtr_->TagName();

  TagTypeInfo* SurrTag_ = new TagTypeInfo();
  SurrTag_->SetTagBrk(StartBrk_, EndBrk_, NULL);
  SurrTag_->SetElementStr(true, new ChrString(ElementStr_), true);
  SurrTag_->SetLookupRefTag(true);

  if (TagPtr_ && MatchElmPtr_ &&
      TagLookupEqual(TagEnums::CLOSING, SurrTag_))
  {
    delete SurrTag_;
    SurrTag_ = DictionarySearchTag();
    TagElmPtr_ = SurrTag_->GiveMatchingElementPtr(TagEnums::CLOSING);

    if (TagElmPtr_ && MatchElmPtr_->IsEqualToMatchTag(*TagElmPtr_))
      TagPtr_ = TagPtr_->AddRequiredTagElement(TagPtr_, TagElmPtr_,
                                               TagEnums::BEFORE_REQTAG, true);
  }
  else
  {
    delete SurrTag_;
    SurrTag_ = NULL;
  }

  return TagPtr_;
}

/****************************************************************************/
TagTypeInfo* TagDictionary::AddRequiredAfterTag(TagTypeInfo* TagPtr_, int TagPos_, const ChrString& ElementStr_)
{
  if (!TagPtr_)
    return TagPtr_;

  int TagIndex_ = (TagPos_ == TagEnums::OPENING ||
                   TagPos_ == TagEnums::FORBIDDEN) ? TagEnums::STD_DEX:TagEnums::END_DEX;
  TagCountData* TagCnt_ = _GenSearchTag->GiveTagCountData();
  const char* StartBrk_ = TagCnt_->GiveTagBrk(TagIndex_);
  const char* EndBrk_ = TagCnt_->GiveTagBrk(TagIndex_+1);
  int TagReqCond_ = TagEnums::AFTER_REQTAG;

  TagTypeInfo* SurrTag_ = new TagTypeInfo();
  SurrTag_->SetTagBrk(StartBrk_, EndBrk_, NULL);
  SurrTag_->SetElementStr(true, new ChrString(ElementStr_), true);
  SurrTag_->SetLookupRefTag(true);

  if (TagPtr_ && TagLookupEqual(TagPos_, SurrTag_))
  {
    TagElement* TagElmPtr_;
    delete SurrTag_;
    SurrTag_ = DictionarySearchTag();
    TagElmPtr_ = SurrTag_->GiveMatchingElementPtr(TagPos_);

    if (TagElmPtr_)
      TagPtr_ = TagPtr_->AddRequiredTagElement(TagPtr_, TagElmPtr_, TagReqCond_);
  }
  else
  {
    delete SurrTag_;
    SurrTag_ = NULL;
  }

  return TagPtr_;
}

/****************************************************************************/
TagTypeInfo* TagDictionary::AddRequiredBeforeTag(TagTypeInfo* TagPtr_, int TagPos_, const ChrString& ElementStr_)
{
  if (!TagPtr_)
    return TagPtr_;

  int TagIndex_ = (TagPos_ == TagEnums::OPENING ||
                   TagPos_ == TagEnums::FORBIDDEN) ? TagEnums::STD_DEX:TagEnums::END_DEX;
  TagCountData* TagCnt_ = _GenSearchTag->GiveTagCountData();
  const char* StartBrk_ = TagCnt_->GiveTagBrk(TagIndex_);
  const char* EndBrk_ = TagCnt_->GiveTagBrk(TagIndex_+1);
  int TagReqCond_ = TagEnums::BEFORE_REQTAG;

  TagTypeInfo* SurrTag_ = new TagTypeInfo();
  SurrTag_->SetTagBrk(StartBrk_, EndBrk_, NULL);
  SurrTag_->SetElementStr(true, new ChrString(ElementStr_), true);
  SurrTag_->SetLookupRefTag(true);

  if (TagPtr_ && TagLookupEqual(TagPos_, SurrTag_))
  {
    TagElement* TagElmPtr_;
    delete SurrTag_;
    SurrTag_ = DictionarySearchTag();
    TagElmPtr_ = SurrTag_->GiveMatchingElementPtr(TagPos_);

    if (TagElmPtr_)
      TagPtr_ = TagPtr_->AddRequiredTagElement(TagPtr_, TagElmPtr_, TagReqCond_);
  }
  else
  {
    delete SurrTag_;
    SurrTag_ = NULL;
  }

  return TagPtr_;
}

/****************************************************************************/
void TagDictionary::EraseVectorContents()
{
  int x;

  for (x = 0; x < _StdVectorLength; x++)
  {
    delete _TagInfoMatrix[TagEnums::OPENING][x];
    _TagInfoMatrix[TagEnums::OPENING][x] = NULL;
  }

  for (x = 0; x < _EndVectorLength; x++)
  {
    delete _TagInfoMatrix[TagEnums::CLOSING][x];
    _TagInfoMatrix[TagEnums::CLOSING][x] = NULL;
  }

  _StdVectorLength =
  _EndVectorLength = 0;
  _HtmlDictionaryCreated = false;
}

/****************************************************************************/
TagTypeInfo** TagDictionary::GrowVector(int TagPos_, long Incr_)
{
  Subscript max = 0;
  Subscript tmax = 0;
  long Index_ = 0;
  long* MaxTags_ = NULL;
  TagTypeInfo** TagArray_ = NULL;

  if (TagPos_ == TagEnums::CLOSING)
  {
    _TagTypeIndex = Index_ = 0;
    TagArray_ = _TagInfoMatrix[0];
    MaxTags_ = &_EndVectorMax;
  }
  else
  {
    _TagTypeIndex = Index_ = 1;
    TagArray_ = _TagInfoMatrix[1];
    MaxTags_ = &_StdVectorMax;
  }

  if (!_TagInfoMatrix[Index_] || !(*MaxTags_))
  {
    max = tmax = Incr_;
    max *= sizeof(TagTypeInfo*);
    TagArray_ = (TagTypeInfo**)RawAllocateWith(MEMMATRIX, max);
    TagArray_ = TagTypeInfo::InitTagTypeArray(TagArray_, tmax);
    _TagInfoMatrix[Index_] = TagArray_;
    *MaxTags_ = Incr_;
  }
  else
  {
    max = tmax = *MaxTags_ + Incr_;
    max *= sizeof(TagTypeInfo*);
    TagArray_ = _TagInfoMatrix[Index_];
    _TagInfoMatrix[Index_] = (TagTypeInfo**)RawAllocateWith(MEMMATRIX, max);
    _TagInfoMatrix[Index_] = TagTypeInfo::InitTagTypeArray(_TagInfoMatrix[Index_], tmax);
    max = *MaxTags_ * sizeof(TagTypeInfo*);
    ::memmove(_TagInfoMatrix[Index_], TagArray_, max);
    *MaxTags_ += Incr_;
    RawDeleteArray(TagArray_);
  }

  return _TagInfoMatrix[Index_];
}

/****************************************************************************/
void TagDictionary::CopyVectors(const TagDictionary& Obj_)
{
  TagTypeInfo* TagPtr_ = NULL;
  TagTypeInfo* NewPtr_ = NULL;
  long x;

  long xTagMax_ = _StdVectorMax;
  long TagMax_ = Obj_._StdVectorMax;
  long TagLen_ = Obj_._StdVectorLength;
  long i = 1;
  bool HasEntries_ = TagLen_ && Obj_._TagInfoMatrix[i][TagLen_-1];
  _TagCreated = false;

  if (HasEntries_)
    for (_StdVectorLength = x = 0; x < TagLen_; x++)
    {
      if (x == xTagMax_ ||
          _StdVectorLength == xTagMax_)
        GrowVector(TagEnums::OPENING);

      TagPtr_ = Obj_._TagInfoMatrix[i][x];
      
      if (TagPtr_)
      {
        NewPtr_ = new TagTypeInfo(*TagPtr_);
        NewPtr_->SetSequencer(Sequencer());
        NewPtr_->SetHtmlSpecialCase(HtmlSpecialCase());
        _TagInfoMatrix[i][_StdVectorLength++] = NewPtr_;
        _TagTypeIndex = i;
        _TagStrIndex[i] = _StdVectorLength - 1;
        _TagCreated = true;
      }
    }

  xTagMax_ = _EndVectorMax;
  TagMax_ = Obj_._EndVectorMax;
  TagLen_ = Obj_._EndVectorLength;
  i = 0;
  HasEntries_ = TagLen_ && Obj_._TagInfoMatrix[i][TagLen_-1];
  _TagCreated = false;

  if (HasEntries_)
    for (_EndVectorLength = x = 0; x < TagLen_; x++)
    {
      if (x == xTagMax_ ||
          _EndVectorLength == xTagMax_)
        GrowVector(TagEnums::CLOSING);
    
      TagPtr_ = Obj_._TagInfoMatrix[i][x];
      
      if (TagPtr_)
      {
        NewPtr_ = new TagTypeInfo(*TagPtr_);
        NewPtr_->SetSequencer(Sequencer());
        NewPtr_->SetHtmlSpecialCase(HtmlSpecialCase());
        _TagInfoMatrix[i][_EndVectorLength++] = NewPtr_;
        _TagTypeIndex = i;
        _TagStrIndex[i] = _EndVectorLength - 1;
        _TagCreated = true;
      }
    }
}

/****************************************************************************/
TagTypeInfo* TagDictionary::CreatedTag()
{
  long i = _TagTypeIndex;
  long x = _TagStrIndex[i];
  bool Found_ = (_TagInfoMatrix && _TagInfoMatrix[i] &&
                 _TagInfoMatrix[i][x] && _TagCreated);

  return (Found_ ? _TagInfoMatrix[i][x]:NULL);
}

/****************************************************************************/
TagTypeInfo* TagDictionary::DictionarySearchTag()
{
  long i = _TagTypeIndex;
  long x = _TagStrIndex[i];
  bool Found_ = (_TagInfoMatrix && _TagInfoMatrix[i] &&
                 _TagInfoMatrix[i][x] && ContentTypeTagFound());

  return (Found_ ? _TagInfoMatrix[i][x]:NULL);
}

/****************************************************************************/
TagTypeInfo* TagDictionary::GeneralSearchTag()
{
  return ((ContentTypeTagFound() || NonContentTypeTagFound()) ? _GenSearchTag:NULL);
}

/****************************************************************************/
TagCountData* TagDictionary::GiveTagCountData(bool ChkTagFnd_)
{
  TagTypeInfo* InfoPtr_ = ChkTagFnd_ ? GeneralSearchTag():
                          _GenSearchTag ? _GenSearchTag:NULL;
  return (InfoPtr_ ? InfoPtr_->GiveTagCountData():NULL);
}

/****************************************************************************/
const TagTypeInfo* TagDictionary::DictionarySearchTag() const
{
  long i = _TagTypeIndex;
  long x = _TagStrIndex[i];
  bool Found_ = (_TagInfoMatrix && _TagInfoMatrix[i] &&
                 _TagInfoMatrix[i][x] && ContentTypeTagFound());

  return (Found_ ? _TagInfoMatrix[i][x]:NULL);
}

/****************************************************************************/
const TagTypeInfo* TagDictionary::GeneralSearchTag() const
{
  return ((ContentTypeTagFound() || NonContentTypeTagFound()) ? _GenSearchTag:NULL);
}

/****************************************************************************/
TagDictionary& TagDictionary::operator = (const TagDictionary& Obj_)
{
  if (&Obj_ != this)
  {
    _TagTypeIndex = 0;
    _DocType = Obj_._DocType;
    _EofFound = Obj_._EofFound;
    _TagType = Obj_._TagType;

    _Sequencer = Obj_._Sequencer;
    _SequencerCopied = true;

    _HtmlSpecialCase = Obj_._HtmlSpecialCase;
    _HtmlSpecialCaseCopied = true;

    if (Obj_._GenSearchTag)
    {
      delete _GenSearchTag;
      _GenSearchTag = new TagTypeInfo(*Obj_._GenSearchTag);
      _GenSearchTag->SetSequencer(Sequencer());
      _GenSearchTag->SetHtmlSpecialCase(HtmlSpecialCase());
    }

    _TextBufferPtr = Obj_._TextBufferPtr;
    _SearchResult = Obj_._SearchResult;

    _HasTag = Obj_._HasTag;
    _DictSearchDone = Obj_._DictSearchDone;
    _ContentTypeTag = Obj_._ContentTypeTag;
    _NonContentTypeTag = Obj_._NonContentTypeTag;
    _TagCreated = Obj_._TagCreated;

    _GenSearchDone = Obj_._GenSearchDone;
    _GenSearchFailed = Obj_._GenSearchFailed;
    _SpecSearchDone = Obj_._SpecSearchDone;
    _SpecSearchFailed = Obj_._SpecSearchFailed;

    _HtmlDictionaryCreated = Obj_._HtmlDictionaryCreated;
    _PrevUnorderedTag = Obj_._PrevUnorderedTag;
    _DumpTagContents = Obj_._DumpTagContents;

    _TagStrIndex[0] = 0;
    _TagStrIndex[1] = 0;
    _GenSearchTag->Reset();
    _GenSearchTag->SetDocType(_DocType);

    if (_TagType == TagEnums::HTMLSTYLE)
      _GenSearchTag->InitAllHtmlTagSwitches(_TagSwitch);
    else if (_TagType == TagEnums::XMLSTYLE)
      _GenSearchTag->InitAllXmlTagSwitches(_TagSwitch);
    else if (_TagType == TagEnums::CPPSTYLE)
      _GenSearchTag->InitCppLangTagSwitches(_TagSwitch);
    else if (_TagType == TagEnums::CSTYLE)
      _GenSearchTag->InitCLangTagSwitches(_TagSwitch);

    if (_StdVectorMax < Obj_._StdVectorMax)
      GrowVector(TagEnums::OPENING, Obj_._StdVectorMax - _StdVectorMax);

    if (_EndVectorMax < Obj_._EndVectorMax)
      GrowVector(TagEnums::CLOSING, Obj_._EndVectorMax - _EndVectorMax);

    if (Obj_._Fr && !_Fr)
      _Fr = new FileReader;

    EraseVectorContents();
    CopyVectors(Obj_);
  }

  return *this;
}

/****************************************************************************/
bool TagDictionary::TagStrLookupEqual(int TagPos_, const ChrString& TagElmStr_,
                                      ChrString* StartStr_, ChrString* EndStr_)
{
  bool HasTag_ = false;
  bool NoDictTag_ = TagPos_ != TagEnums::OPENING &&
                    TagPos_ != TagEnums::CLOSING &&
                    TagPos_ != TagEnums::FORBIDDEN &&
                    TagPos_ != TagEnums::INDETERMINATE;

  int TagIndex_ = (TagPos_ == TagEnums::OPENING ||
                   TagPos_ == TagEnums::FORBIDDEN) ? TagEnums::STD_DEX:
                  (TagPos_ == TagEnums::CLOSING ||
                   TagPos_ == TagEnums::INDETERMINATE) ? TagEnums::END_DEX:-1;
  TagCountData* TagCnt_ = _GenSearchTag->GiveTagCountData();
  const char* StartBrk_ = (TagIndex_ >= 0) ? TagCnt_->GiveTagBrk(TagIndex_):
                          StartStr_ ? StartStr_->c_str():NULL;
  const char* EndBrk_ = (TagIndex_ >= 0) ? TagCnt_->GiveTagBrk(TagIndex_+1):
                        EndStr_ ? EndStr_->c_str():NULL;

  TagTypeInfo* LookTag_ = (!NoDictTag_ &&
                           StartBrk_ && EndBrk_) ? (new TagTypeInfo()):NULL;

  if (LookTag_ && !NoDictTag_)
  {
    LookTag_->SetTagBrk(StartBrk_, EndBrk_, NULL);
    LookTag_->SetElementStr(true, new ChrString(TagElmStr_), true);
    LookTag_->SetLookupRefTag(true);

    HasTag_ = TagLookupEqual(TagPos_, LookTag_);
  }
  else if (StartBrk_ && EndBrk_ && NoDictTag_)
  {
    _ContentTypeTag = TagTypeInfo::IsContentTypeTag(StartBrk_, EndBrk_);
    _NonContentTypeTag = TagTypeInfo::IsNonContentTypeTag(StartBrk_, EndBrk_);
    _DictSearchDone = true;
  }

  delete LookTag_;
  return HasTag_;
}

/****************************************************************************/
bool TagDictionary::TagLookupEqual(int TagPos_, TagTypeInfo* TagInfop_)
{
  const char* StartBrk_ = TagInfop_ ? TagInfop_->StartBrk():NULL;
  const char* EndBrk_ = TagInfop_ ? TagInfop_->EndBrk():NULL;

  bool ret = false;
  bool undet_ = TagPos_ == TagEnums::INDETERMINATE ||
                TagPos_ == TagEnums::NO_TAGPOS;

  if (TagInfop_ && StartBrk_ && EndBrk_)
  {
    long x;
    long len = (TagPos_ == TagEnums::OPENING ||
                TagPos_ == TagEnums::FORBIDDEN) ? _StdVectorLength:
               (TagPos_ == TagEnums::CLOSING ||
                TagPos_ == TagEnums::INDETERMINATE) ? _EndVectorLength:-1;
    long i = (len == _StdVectorLength) ? 1:0;

    _HasTag = false;
    _DictSearchDone = (len == -1);

    if (TagInfop_)
      if (!TagTypeInfo::IsContentTypeTag(TagInfop_->StartBrk(), TagInfop_->EndBrk()))
      {
        _ContentTypeTag = false;
        _NonContentTypeTag = TagTypeInfo::IsNonContentTypeTag(TagInfop_->StartBrk(), TagInfop_->EndBrk());
        _DictSearchDone = true;

        return false;
      }
      else
      {
        _ContentTypeTag = true;
        _NonContentTypeTag = TagTypeInfo::IsNonContentTypeTag(TagInfop_->StartBrk(), TagInfop_->EndBrk());
      }

    for (x = 0; x < len; x++)
      if (_TagInfoMatrix[i][x])
      {
        if (_TagInfoMatrix[i][x]->TagLookupEqual(*TagInfop_))
        {
          _TagTypeIndex = i;
          _TagStrIndex[i] = x;
          _HasTag = true;
          _ContentTypeTag = true;
          _NonContentTypeTag = false;
          ret = true;
        }

        _DictSearchDone = true;
        if (ret)
          break;
      }

    if (!ret && TagPos_ == TagEnums::INDETERMINATE)
    {
      i = 1;
      len = _StdVectorLength;

      for (x = 0; x < len; x++)
        if (_TagInfoMatrix[i][x])
        {
          if (_TagInfoMatrix[i][x]->TagLookupEqual(*TagInfop_))
          {
            _TagTypeIndex = i;
            _TagStrIndex[i] = x;
            _HasTag = true;
            _ContentTypeTag = true;
            _NonContentTypeTag = false;
            ret = true;
          }

          _DictSearchDone = true;
          if (ret)
            break;
        }
    }
  }

  if (undet_ && !ret && TagInfop_)
    _NonContentTypeTag = TagTypeInfo::IsNonContentTypeTag(TagInfop_->StartBrk(), TagInfop_->EndBrk());

  return ret;
}

/****************************************************************************/
bool TagDictionary::HasThisTag(int TagPos_, TagTypeInfo* TagInfop_,
                               bool Insertion_, int DocType_, bool ForbiddenUnspec_)
{
  const char* StartBrk_ = TagInfop_ ? TagInfop_->StartBrk():NULL;
  const char* EndBrk_ = TagInfop_ ? TagInfop_->EndBrk():NULL;

  bool ret = false;
  bool undet_ = TagPos_ == TagEnums::INDETERMINATE;

  if (TagInfop_ && StartBrk_ && EndBrk_)
  {
    long x;
    long len = (TagPos_ == TagEnums::OPENING ||
                TagPos_ == TagEnums::FORBIDDEN) ? _StdVectorLength:
               (TagPos_ == TagEnums::CLOSING ||
                TagPos_ == TagEnums::INDETERMINATE) ? _EndVectorLength:-1;
    long i = (len == _StdVectorLength) ? 1:0;

    _HasTag = false;
    _DictSearchDone = (len == -1);

    if (TagInfop_)
      if (!TagTypeInfo::IsContentTypeTag(TagInfop_->StartBrk(), TagInfop_->EndBrk()))
      {
        _ContentTypeTag = false;
        _NonContentTypeTag = TagTypeInfo::IsNonContentTypeTag(TagInfop_->StartBrk(), TagInfop_->EndBrk());
        _DictSearchDone = true;

        if (!Insertion_)
          return true;
      }
      else
      {
        _ContentTypeTag = true;
        _NonContentTypeTag = TagTypeInfo::IsNonContentTypeTag(TagInfop_->StartBrk(), TagInfop_->EndBrk());
      }

    for (x = 0; x < len; x++)
      if (_TagInfoMatrix[i][x] &&
          (!DocType_ || _TagInfoMatrix[i][x]->InDocType(DocType_)))
      {
        if (ForbiddenUnspec_)
          _TagInfoMatrix[i][x]->SetForbiddenUnspecified(true);

        if (*_TagInfoMatrix[i][x] == *TagInfop_)
        {
          _TagTypeIndex = i;
          _TagStrIndex[i] = x;
          _HasTag = true;
          _ContentTypeTag = true;
          _NonContentTypeTag = false;
          ret = true;
        }

        if (ForbiddenUnspec_)
          _TagInfoMatrix[i][x]->SetForbiddenUnspecified(false);

        _DictSearchDone = true;
        if (ret)
          break;
      }

    if (!ret && TagPos_ == TagEnums::INDETERMINATE)
    {
      i = 1;
      len = _StdVectorLength;

      for (x = 0; x < len; x++)
        if (_TagInfoMatrix[i][x] &&
            (!DocType_ || _TagInfoMatrix[i][x]->InDocType(DocType_)))
        {
          if (ForbiddenUnspec_)
            _TagInfoMatrix[i][x]->SetForbiddenUnspecified(true);

          if (*_TagInfoMatrix[i][x] == *TagInfop_)
          {
            _TagTypeIndex = i;
            _TagStrIndex[i] = x;
            _HasTag = true;
            _ContentTypeTag = true;
            _NonContentTypeTag = false;
            ret = true;
          }

          if (ForbiddenUnspec_)
            _TagInfoMatrix[i][x]->SetForbiddenUnspecified(false);

          _DictSearchDone = true;
          if (ret)
            break;
        }
    }
  }

  if (undet_ && !ret && TagInfop_)
    _NonContentTypeTag = TagTypeInfo::IsNonContentTypeTag(TagInfop_->StartBrk(), TagInfop_->EndBrk());

  return ret;
}

/****************************************************************************/
void TagDictionary::SetHtmlTagErrors(StackNodeInfo* EncNode_)
{
  if ((!_DocType ||
       _DocType == TagEnums::HTML_FILE ||
       _DocType == TagEnums::XHTML_FILE) &&
      EncNode_ && EncNode_->_Parent)
  {
    TagTypeInfo* TagInfoPtr_ = EncNode_->_Parent;
    ChrString TagStr_ = TagInfoPtr_->SearchResult()->GiveTagString();
    _HtmlSpecialCase->CheckForSpecialHtmlTags(TagStr_);

    if (_HtmlSpecialCase->HasTagErrors())
      TagInfoPtr_->SetTagErrors(_HtmlSpecialCase->GiveTagErrors(), 0);
  }
}

/****************************************************************************/
void TagDictionary::ProcessRequiredTagCond(StackNodeInfo* EncNode_)
{
  // reseting tag found condition of previously discovered tag
  //
  // ReProcess TagElements found in TagLinksSequencer ptr in
  // _GenSearchTag ptr -- _Sequencer StackInfo stack _HeadInfo
  //

  if (EncNode_ && EncNode_->Sequencer() && EncNode_->_Parent)
  {
    bool AtNextTag_ = false;
    bool AtImpliedTag_ = false;
    bool AtBeforeReqTag_ = false;
    bool AtMatchingTag_ = false;

    TagElement* TagElmPtr_ = EncNode_->_ElementPtr;
    TagElement* TailCaller_ = NULL;

    TagTypeInfo* TagInfoPtr_ = NULL;
    StackNodeInfo* Head_ = NULL;
    StackNodeInfo* Tail_ = NULL;

    StackNodeInfo* TailEnd_ = EncNode_->Sequencer()->TailInfo(true, true);
    StackNodeInfo* HeadEnd_ = EncNode_->Sequencer()->HeadInfo(true, true);

    if (HeadEnd_ != EncNode_)
      return;

    // if AFTER_REQTAG or SURROUND_REQTAG tag requirement then
    //   for each ONE new StackNodeInfo tag newly appended to the head of the
    //   tag list _HeadInfo, call TagElement::TestAftReqCond(const TagElement* Ptr_)
    //   passing each StackNodeInfo::_ElementPtr in _TailInfo as the method
    //   argument, moving up through the tag list until reaching the newly
    //   appended StackNodeInfo tag.
    //   Use the pointer data member _ElementPtr as the calling pointer for
    //   TagElement::TestAftReqCond.
    //
    //   Example: 1st arg, nextarg, nextarg, nextarg as argument to method
    //   TagElement::TestAftReqCond of _HeadInfo->_ElementPtr, until _HeadInfo is
    //   reached
    //
    if ((TagElmPtr_ && (TagElmPtr_->TagReqTagCond() == TagEnums::AFTER_REQTAG ||
                        TagElmPtr_->TagReqTagCond() == TagEnums::SURROUND_REQTAG)) &&
        !TagElmPtr_->AtReqTagCondStopState(true))
    {
      Head_ = EncNode_;
      Tail_ = TailEnd_;

      if (Head_)
      {
        Head_->SetTagElementState();
        TagElmPtr_ = Head_->_ElementPtr;
      }

      if (Tail_)
        Tail_->SetSourceTagElementState();

      while (Tail_ && Tail_->_ElementPtr && Tail_ != Head_)
      {
        if (TagElmPtr_ && !TagElmPtr_->AtReqTagCondStopState(true))
        {
          TagElmPtr_->TestAftReqCond(*Tail_->_ElementPtr);

          if (TagElmPtr_->AtReqTagCondStopState(true))
            break;
        }
        else
          break;

        Tail_->SetSourceTagElementState(NULL);
        Tail_ = Tail_->Sequencer()->PrevStackInfo(false, false, true);

        if (Tail_)
          Tail_->SetSourceTagElementState();
      }

      if (Tail_)
        Tail_->SetSourceTagElementState(NULL);
    }

    // Test current tag requirment position with the node at the head of
    // the StackInfoNode list using TagElement::TestCurTagCond method passed
    // with the value TagEnums::CURRENT_REQTAG as the TestingCond_ argument
    //
    if (TagElmPtr_ && TagElmPtr_->TagReqTagCond() &&
        !TagElmPtr_->AtReqTagCondStopState(false))
    {
      Head_ = EncNode_;

      if (Head_)
      {
        Head_->SetTagElementState();
        Head_->SetSourceTagElementState();
        TagElmPtr_ = Head_->_ElementPtr;
      }

      TagElmPtr_->TestCurTagCond(*TagElmPtr_);

      if (Head_)
        Head_->SetSourceTagElementState(NULL);
    }

    // if BEFORE_REQTAG or SURROUND_REQTAG tag requirement then
    //   for every StackNodeInfo list node in the tag list _TailInfo up until
    //   the newly appended StackNodeInfo tag, call
    //   TagElement::TestBefReqCond(const TagElement* Ptr_) passing the newly appended
    //   StackNodeInfo::_ElementPtr in _HeadInfo as the method argument,
    //   up until the newly appended StackNodeInfo tag.
    //   use the pointer data member _ElementPtr as the calling pointer for
    //   TagElement::TestBefReqCond.
    //
    //   Example: 1st object caller, nextcaller, nextcaller, nextcaller of
    //   method TagElement::TestBefReqCond and passing _HeadInfo->_ElementPtr as
    //   method argument, until _HeadInfo is reached.
    //
    if (TagElmPtr_ && (TagElmPtr_->TagReqTagCond() == TagEnums::BEFORE_REQTAG ||
                       TagElmPtr_->TagReqTagCond() == TagEnums::SURROUND_REQTAG))
    {
      Head_ = EncNode_;
      Tail_ = HeadEnd_;

      if (Head_)
      {
        Head_->SetSourceTagElementState();
        TagElmPtr_ = Head_->_ElementPtr;
      }

      while (Tail_ && Tail_->_ElementPtr && Tail_ != TailEnd_)
      {
        Tail_ = Tail_->Sequencer()->NextStackInfo(false, false, true);

        if (Tail_)
        {
          Tail_->SetTagElementState();
          TailCaller_ = Tail_->_ElementPtr;

          if (TailCaller_ && !TailCaller_->AtReqTagCondStopState(false))
            TailCaller_->TestBefReqCond(*TagElmPtr_);
        }
      }

      if (Head_)
        Head_->SetSourceTagElementState(NULL);
    }

    // Report any REQTAGCOND changes which results in tag errors, traversing
    // from the tail of the StackNodeInfo list until head of list is reached
    //
    Head_ = EncNode_;
    Tail_ = HeadEnd_;

    while (Tail_ && Tail_->_ElementPtr && Tail_ != TailEnd_)
    {
      Tail_ = Tail_->Sequencer()->NextStackInfo(false, false, true);

      if (Tail_)
      {
        Tail_->SetTagElementState();
        TailCaller_ = Tail_->_ElementPtr;
        TagInfoPtr_ = Tail_->_Parent;

        if (TailCaller_ && TailCaller_->ReqTagCondChanged() &&
            TailCaller_->AtReqTagCondStopState(false, true))
        {
          // Required prerequisite tag(s) not found
          TagInfoPtr_->SetTagErrors(TagEnums::REQUIRED_TAG_NOT_FOUND, 0);
          TailCaller_->ResetState(true, false, false, false, false);
        }
      }
    }

    // Create any forward links for any optional tags that have any
    // corresponding matching end tags found for it.
    //
    Head_ = EncNode_;
    Tail_ = HeadEnd_;

    while (Tail_ && Tail_->_ElementPtr && Tail_ != TailEnd_)
    {
      Tail_ = Tail_->Sequencer()->NextStackInfo(false, false, true);

      if (Tail_)
      {
        Tail_->SetTagElementState();
        TailCaller_ = Tail_->_ElementPtr;

        AtNextTag_ = (TailCaller_->TagImpliedEnd() & TagEnums::AT_NEXTTAG);
        AtImpliedTag_ = (TailCaller_->TagImpliedEnd() & TagEnums::AT_IMPLIEDTAG);
        AtBeforeReqTag_ = (TailCaller_->TagImpliedEnd() & TagEnums::AT_BEFORE_REQTAG);
        AtMatchingTag_ = !TailCaller_->TagImpliedEnd() &&
                         (TailCaller_->TagImpliedEnd() & TagEnums::AT_MATCHENDTAG);

        if (TailCaller_ &&
            TailCaller_->ReqTagCondChanged() &&
            TailCaller_->AtReqTagCondStopState(false) &&
            TailCaller_->RequiredTagCondMet() &&
            !TailCaller_->TagAtReqTagCondNIE() &&
            !TailCaller_->NoRequiredTag())
        {
          if (!AtImpliedTag_ && !AtNextTag_ &&
              TailCaller_->AtReqTagCond(TagEnums::CURTAG_FOUND) &&
              TailCaller_->ReqTagDirFound() == TagEnums::BEFORE_REQTAG)
          {
            if (AtMatchingTag_ &&
                TailCaller_->TagAtMatchingEnd() &&
                TailCaller_->EndTag() &&
                  !Tail_->HasLinks(true, true) &&
                  !TailCaller_->EndTag()->_Parent->HasLinks(true, false) &&
                TailCaller_->EndTagCond() == TagEnums::AT_MATCHENDTAG)
            {
              // AT_MATCHENDTAG : at matching pair ender tag
              Tail_->LinkForward(TailCaller_->EndTag()->_Parent, true);
              TailCaller_->ResetState(true, false, false, false, false);
            }
            else if (AtBeforeReqTag_ &&
                     TailCaller_->TagAtBeforeReqTag() &&
                     TailCaller_->EndTag() &&
                       !TailCaller_->ImpliedTagFound() &&
                       !Tail_->HasLinks(true, true) &&
                     TailCaller_->EndTagCond() == TagEnums::AT_BEFORE_REQTAG)
            {
              // AT_BEFORE_REQTAG : at required BEFORE tag
              Tail_->LinkForward(TailCaller_->EndTag()->_Parent);
              TailCaller_->ResetState(true, false, false, false, false);
            }
          }
          else if (AtImpliedTag_ &&
                   TailCaller_->ImpliedTagFound() &&
                   TailCaller_->TagAtImpliedEnd() &&
                   TailCaller_->EndTag() &&
                     !Tail_->HasLinks(false, true) &&
                   TailCaller_->EndTagCond() == TagEnums::AT_IMPLIEDTAG)
          {
            // AT_IMPLIEDTAG : at optional implied end tag
            Tail_->LinkForward(TailCaller_->EndTag()->_Parent);
            TailCaller_->ResetState(true, false, false, false, false);
          }
          else if (AtNextTag_ &&
                   TailCaller_->TagAtImpliedEnd() &&
                   TailCaller_->EndTag() &&
                     !TailCaller_->ImpliedTagFound() &&
                     !Tail_->HasLinks(false, true) &&
                   TailCaller_->EndTagCond() == TagEnums::AT_NEXTTAG)
          {
            // AT_NEXTTAG : at next encountered tag
            Tail_->LinkForward(TailCaller_->EndTag()->_Parent);
            TailCaller_->ResetState(true, false, false, false, false);
          }
        }
      }
    }
  }
}

/****************************************************************************/
TagDictionary& TagDictionary::Reset(bool ReadPos_, bool TagSwitch_, int DocType_)
{
  if (!_TagInfoMatrix || (!ReadPos_ && !TagSwitch_))
    return *this;

  long x, y;
  long len[2];

  len[0] = _StdVectorLength;
  len[1] = _EndVectorLength;

  if (ReadPos_)
  {
    _GenSearchTag->Reset(TagTypeInfo::READPOS);

    for (y = 0; y < 2; y++)
      for (x = 0; x < len[y]; x++)
        if (_TagInfoMatrix[y] && _TagInfoMatrix[y][x])
          _TagInfoMatrix[y][x]->Reset(TagTypeInfo::READPOS);
  }

  if (TagSwitch_)
  {
    int TagBits_[TagTypeInfo::MAX_TAGINDEXES];
    ::memset(TagBits_, 0, TagTypeInfo::MAX_TAGINDEXES);

    if (DocType_ > 0)
      SetDocType(DocType_);
    else if (_GenSearchTag->DocTypeSpecified() != _DocType ||
             _GenSearchTag->DocTypeSpecified() == 0)
      _GenSearchTag->SetDocType(_DocType);

    if (_GenSearchTag->TagInitConfirmed())
      _GenSearchTag->RestoreTagSwitches();
    else
      _GenSearchTag->SetTagSwitches(_TagSwitch, TagBits_);
  }

  return *this;
}

/****************************************************************************/
TagDictionary& TagDictionary::SetDocType(int DocType_)
{
  if (!_TagInfoMatrix || DocType_ <= 0)
    return *this;

  long x, y;
  long len[2];

  len[0] = _StdVectorLength;
  len[1] = _EndVectorLength;
  
  _DocType = DocType_;
  _GenSearchTag->SetDocType(DocType_);

  for (y = 0; y < 2; y++)
    for (x = 0; x < len[y]; x++)
      if (_TagInfoMatrix[y] && _TagInfoMatrix[y][x] &&
          (_TagInfoMatrix[y][x]->DocTypeSpecified() != DocType_ ||
           _TagInfoMatrix[y][x]->DocTypeSpecified() == 0))
        _TagInfoMatrix[y][x]->SetDocType(DocType_);

  return *this;
}

/****************************************************************************/
TagDictionary& TagDictionary::SetEofFound(bool Eof_)
{
  if (!_TagInfoMatrix)
    return *this;

  long x, y;
  long len[2];

  len[0] = _StdVectorLength;
  len[1] = _EndVectorLength;
  
  _EofFound = Eof_;
  _GenSearchTag->SetEofFound(Eof_);

  for (y = 0; y < 2; y++)
    for (x = 0; x < len[y]; x++)
      if (_TagInfoMatrix[y] && _TagInfoMatrix[y][x])
        _TagInfoMatrix[y][x]->SetEofFound(Eof_);

  return *this;
}

/****************************************************************************/
TagDictionary& TagDictionary::SetTextBufferPtr(const ChrString* Ptr_)
{
  if (!_TagInfoMatrix)
    return *this;

  long x, y;
  long len[2];
  
  len[0] = _StdVectorLength;
  len[1] = _EndVectorLength;

  _TextBufferPtr = Ptr_;
  _GenSearchTag->SetTextBufferPtr(Ptr_);

  for (y = 0; y < 2; y++)
    for (x = 0; x < len[y]; x++)
      if (_TagInfoMatrix[y] && _TagInfoMatrix[y][x])
        _TagInfoMatrix[y][x]->SetTextBufferPtr(Ptr_);

  return *this;
}

/****************************************************************************/
void TagDictionary::DetermineTagAttrib(TagTypeInfo* TagInfop_,
            int TagPos_, int AppAttr_, int& TagAttr_,
            bool& EmptyTag_, bool& Singleton_, bool& Empty_, bool& Container_,
            bool& PairContent_, const char* StartBrk_, const char* EndBrk_)
{
  if ((EmptyTag_ || Singleton_) && TagPos_ != TagEnums::CLOSING)
  {
    Empty_ = TagBracketList::IsEmptyTag(StartBrk_, EndBrk_);
    if (!Singleton_ ||
          (StartBrk_ && strcmp(StartBrk_, STDTAGOPEN_STR) == 0 &&
           EndBrk_ && strcmp(EndBrk_, STDTAGCLOSE_STR) == 0))
      Singleton_ = !Empty_;

    TagInfop_->SetEmptyAllowed(true);
    TagInfop_->SetSingletonAllowed(Singleton_);
  }
  else if (TagPos_ == TagEnums::CLOSING)
    Singleton_ = Empty_ = false;

  if ((!AppAttr_ &&
       TagPos_ == TagEnums::OPENING &&
       TagAttr_ == TagEnums::PAIR_TAGATTR) ||
      (TagAttr_ == TagEnums::OPTIONALPAIR_TAGATTR))
  {
    if (!AppAttr_)
      TagAttr_ = TagEnums::TAG_SPECIFIC;
    else
      TagAttr_ |= TagEnums::TAG_SPECIFIC;

    if (Container_)
    {
      EmptyTag_ = false;
      TagAttr_ |= (TagEnums::OPTIONALPAIR_TAGATTR |
                   TagEnums::TAG_CONTAINER);
    }
    else if (EmptyTag_)
    {
      Container_ = false;
      TagAttr_ |= TagEnums::SINGLETON_TAGATTR;

      if (Empty_)
        TagAttr_ |= TagEnums::EMPTY_TAGATTR;
      else if (Singleton_)
      {
        TagAttr_ |= TagEnums::SINGLETON_TAGATTR;
        if (!_DocType ||
            _DocType == TagEnums::HTML_FILE ||
            _DocType == TagEnums::XHTML_FILE)
          TagAttr_ |= TagEnums::EMPTY_TAGATTR;
      }
    }
  }

  if (TagAttr_ == TagEnums::PAIR_TAGATTR)
  {
    EmptyTag_ = false;
    Container_ = true;
    TagAttr_ |= TagEnums::TAG_CONTAINER;
  }
  else if (TagPos_ == TagEnums::FORBIDDEN ||
           TagPos_ == TagEnums::CLOSING)
  {
    EmptyTag_ = true;
    Container_ = false;

    if (TagPos_ == TagEnums::CLOSING)
      Singleton_ = Empty_ = false;

    if (Empty_)
      TagAttr_ |= TagEnums::EMPTY_TAGATTR;
    else if (Singleton_)
    {
      TagAttr_ |= TagEnums::SINGLETON_TAGATTR;
      if (!_DocType ||
          _DocType == TagEnums::HTML_FILE ||
          _DocType == TagEnums::XHTML_FILE)
        TagAttr_ |= TagEnums::EMPTY_TAGATTR;
    }
  }

  if (!EmptyTag_)
    Singleton_ = Empty_ = false;

  PairContent_ = TagPos_ == TagEnums::OPENING &&
                 ((TagAttr_ & TagEnums::PAIR_TAGATTR) ||
                  (TagAttr_ & TagEnums::OPTIONALPAIR_TAGATTR));
}

/****************************************************************************/
bool TagDictionary::HasEntries(int TagPos_, TagTypeInfo* TagInfop_) const
{
  if (TagPos_ != TagEnums::OPENING &&
      TagPos_ != TagEnums::CLOSING &&
      TagPos_ != TagEnums::FORBIDDEN)
    return false;
    
  bool Updated_ = false;
  int IndexPos_ = _TagStrIndex[TagPos_];

  const char* StartBrk_ = TagInfop_ ? (!TagInfop_->ElementStr_Specified() ?
                                           TagInfop_->StartBrkFound():
                                           TagInfop_->StartBrk()):NULL;
  const char* EndBrk_ = TagInfop_ ? (!TagInfop_->ElementStr_Specified() ?
                                           TagInfop_->EndBrkFound():
                                           TagInfop_->EndBrk()):NULL;

  const long* TagMax_ = (TagPos_ == TagEnums::OPENING ||
                         TagPos_ == TagEnums::FORBIDDEN) ?
                     &_StdVectorMax:
                     &_EndVectorMax;
  const long* TagLen_ = (TagPos_ == TagEnums::OPENING ||
                         TagPos_ == TagEnums::FORBIDDEN) ?
                     &_StdVectorLength:
                     &_EndVectorLength;
  int Index_ = (TagLen_ == &_StdVectorLength) ? 1:0;

  if (StartBrk_ && EndBrk_ &&
      !TagTypeInfo::IsContentTypeTag(StartBrk_, EndBrk_))
    return true;

  bool HasEntries_ =
      ((_TagTypeIndex >= 0 && IndexPos_ >= 0) &&
       (((TagPos_ == TagEnums::OPENING ||
          TagPos_ == TagEnums::FORBIDDEN) && *TagLen_ && _TagInfoMatrix[Index_][*TagLen_-1]) ||
        (TagPos_ == TagEnums::CLOSING && *TagLen_ && _TagInfoMatrix[Index_][*TagLen_-1])));

  return (*TagMax_ && HasEntries_);
}

/****************************************************************************/
TagTypeInfo* TagDictionary::CreateTag(int TagPos_, bool Empty_, bool Container_, int AppAttr_,
                                      const char* StartBrk_, const ChrString& Element_,
                                      const char* EndBrk_, TagTypeInfo* MatchingTag_)
{
  _TagCreated = false;

  if ((TagPos_ != TagEnums::OPENING &&
       TagPos_ != TagEnums::CLOSING &&
       TagPos_ != TagEnums::FORBIDDEN) ||
      (!StartBrk_ || !EndBrk_))
    return NULL;

  bool Singleton_ = TagPos_ == TagEnums::FORBIDDEN;
  bool EmptyTag_ = Empty_;
  bool PairContent_ = false;

  bool Updated_ = false;
  int IndexPos_ = _TagStrIndex[TagPos_];
  int PairEnder_ = 0;
  TagTypeInfo::TagSwitch TagSwitch_;

  long* TagMax_ = (TagPos_ == TagEnums::OPENING ||
                   TagPos_ == TagEnums::FORBIDDEN) ?
                     &_StdVectorMax:
                     &_EndVectorMax;
  long* TagLen_ = (TagPos_ == TagEnums::OPENING ||
                   TagPos_ == TagEnums::FORBIDDEN) ?
                     &_StdVectorLength:
                     &_EndVectorLength;
  int Index_ = (TagLen_ == &_StdVectorLength) ? 1:0;

  bool HasEntries_ =
      ((_TagTypeIndex >= 0 && IndexPos_ >= 0) &&
       (((TagPos_ == TagEnums::OPENING ||
          TagPos_ == TagEnums::FORBIDDEN) && *TagLen_ && _TagInfoMatrix[Index_][*TagLen_-1]) ||
        (TagPos_ == TagEnums::CLOSING && *TagLen_ && _TagInfoMatrix[Index_][*TagLen_-1])));

  int BrkIndex_ = _GenSearchTag->BrkStringToIndex(StartBrk_, EndBrk_, PairEnder_);
  int TagType_ = (Index_ >= 0) ? _GenSearchTag->BrkIndexToTagType(BrkIndex_):-1;
  int attr = AppAttr_ ? AppAttr_:_GenSearchTag->GiveDefaultTagAttrib(TagType_, TagPos_);
  int TagBits_[TagTypeInfo::MAX_TAGINDEXES];
  ::memset(TagBits_, 0, TagTypeInfo::MAX_TAGINDEXES);

  if (*TagMax_)
  {
    TagElement* TagElmPtr_ = NULL;
    TagTypeInfo* TagInfop_ = new TagTypeInfo();

    if (_DocType > 0)
      TagInfop_->SetDocType(_DocType);

    DetermineTagAttrib(TagInfop_, TagPos_, AppAttr_, attr,
                       EmptyTag_, Singleton_, Empty_, Container_,
                       PairContent_, StartBrk_, EndBrk_);

    TagInfop_->SetSequencer(Sequencer());
    TagInfop_->SetHtmlSpecialCase(HtmlSpecialCase());
    TagInfop_->SetTagBrk(StartBrk_, EndBrk_, NULL);
    TagInfop_->SetElementStr(true, new ChrString(Element_), true);
    TagInfop_->SetEmptyTag(Empty_);
    TagInfop_->SetSingletonTag(Singleton_);
    TagInfop_->SetContainerTag(Container_);
    TagInfop_->SetTagAttr(attr);
    TagInfop_->SetContentTypeTag(StartBrk_, EndBrk_);
    TagInfop_->SetHasPairContent(PairContent_);
    TagInfop_->InitTagSwitches(TagSwitch_, TagInfop_->BrkIndexToTagType(BrkIndex_), PairEnder_);
    TagInfop_->SetTagSwitches(TagSwitch_, TagBits_);
    TagInfop_->SetDictionaryTag(true);

    bool AddToList_ = HasEntries_ ? !HasThisTag(TagPos_, TagInfop_, true):true;

    if (AddToList_)
    {
      if (TagInfop_->HasThisTag(TagPos_))
      {
        TagInfop_->SetThisTag(TagPos_);
        TagElmPtr_ = TagInfop_->GiveTagElementPtr();
        if (TagElmPtr_)
          TagInfop_->SetTagAttr(TagElmPtr_->TagAttrib());

        _TagInfoMatrix[Index_][(*TagLen_)++] = TagInfop_;
        _TagCreated = true;
        
        _TagTypeIndex = Index_;
        _TagStrIndex[Index_] = *TagLen_ - 1;
      }
      else
      {
        TagInfop_->AddToTagList(TagPos_, MatchingTag_, Updated_);
        TagElmPtr_ = TagInfop_->GiveTagElementPtr();
        if (TagElmPtr_)
          TagInfop_->SetTagAttr(TagElmPtr_->TagAttrib());

        _TagInfoMatrix[Index_][(*TagLen_)++] = TagInfop_;
        _TagCreated = true;

        _TagTypeIndex = Index_;
        _TagStrIndex[Index_] = *TagLen_ - 1;
      }

      if (*TagLen_ == *TagMax_)
        GrowVector(Index_);
    }
    else
    {
      delete TagInfop_;
      TagInfop_ = NULL;
    }
  }

  return CreatedTag();
}

/****************************************************************************/
TagTypeInfo* TagDictionary::MakeMatchTag(TagTypeInfo* StartTag_, bool SetMatchTag_, bool AddToList_)
{
  TagTypeInfo* MatchTag_ = NULL;
  TagTypeInfo::TagSwitch TagSwitch_;
  int TagBits_[TagTypeInfo::MAX_TAGINDEXES];
  ::memset(TagBits_, 0, TagTypeInfo::MAX_TAGINDEXES);
  long i = _TagTypeIndex;
  long x = _TagStrIndex[i];
  int TagPos_ = (i == 1) ? TagEnums::OPENING:TagEnums::CLOSING;
  int MatchPos_ = 0;
  int BrkIndex_;
  int PairEnder_;
  TagElement* ElmPtr_ = StartTag_ ? StartTag_->GiveTagElementPtr():NULL;
  bool Forbidden_ = ElmPtr_   ? (ElmPtr_->TagPosition() == TagEnums::FORBIDDEN):
                    StartTag_ ? (StartTag_->GiveTagPosition() == TagEnums::FORBIDDEN):true;

  _TagCreated = false;

  if (Forbidden_)
    return NULL;

  long* TagMax_ = (TagPos_ == TagEnums::OPENING) ?
                     &_StdVectorMax:
                     &_EndVectorMax;
  long* TagLen_ = (TagPos_ == TagEnums::OPENING) ?
                     &_StdVectorLength:
                     &_EndVectorLength;
  bool HasEntries_ =
      ((_TagTypeIndex >= 0 && x >= 0) &&
       (*TagLen_ && _TagInfoMatrix[i][*TagLen_-1]));

  if (StartTag_ && _GenSearchTag && *TagMax_)
  {
    MatchTag_ = _GenSearchTag->MakeMatchTag(StartTag_, SetMatchTag_, AddToList_);

    if (MatchTag_ &&
        (MatchTag_->DocTypeSpecified() != _DocType ||
         MatchTag_->DocTypeSpecified() == 0))
      MatchTag_->SetDocType(_DocType);

    if (MatchTag_)
    {
      BrkIndex_ = MatchTag_->BrkStringToIndex(MatchTag_->StartBrk(), MatchTag_->EndBrk(), PairEnder_);
      MatchTag_->SetPairEnderTag(PairEnder_);
      MatchTag_->InitTagSwitches(TagSwitch_, MatchTag_->BrkIndexToTagType(BrkIndex_), PairEnder_);
      MatchTag_->SetTagSwitches(TagSwitch_, TagBits_);
      MatchTag_->SetDictionaryTag(true);

      MatchPos_ = (TagPos_ == TagEnums::OPENING) ? TagEnums::CLOSING:TagEnums::OPENING;
      AddToList_ = HasEntries_ ? !HasThisTag(MatchPos_, MatchTag_, true):true;
    }

    if (AddToList_ && MatchTag_)
    {
      if (TagPos_ == TagEnums::OPENING)
      {
        i = (i == 1) ? 0:1;
        TagLen_ = (TagLen_ == &_StdVectorLength) ?
                      &_EndVectorLength:
                      &_StdVectorLength;
        TagMax_ = (TagMax_ == &_StdVectorMax) ?
                      &_EndVectorMax:
                      &_StdVectorMax;

        MatchTag_->SetHasPairContent(false);
        _TagInfoMatrix[i][(*TagLen_)++] = MatchTag_;
        _TagCreated = true;
        
        _TagTypeIndex = i;
        _TagStrIndex[i] = *TagLen_ - 1;
      }
      else
      {
        i = (i == 0) ? 1:0;
        TagLen_ = (TagLen_ == &_EndVectorLength) ?
                      &_StdVectorLength:
                      &_EndVectorLength;
        TagMax_ = (TagMax_ == &_EndVectorMax) ?
                      &_StdVectorMax:
                      &_EndVectorMax;

        MatchTag_->SetHasPairContent(true);
        _TagInfoMatrix[i][(*TagLen_)++] = MatchTag_;
        _TagCreated = true;

        _TagTypeIndex = i;
        _TagStrIndex[i] = *TagLen_ - 1;
      }

      if (*TagLen_ == *TagMax_)
        GrowVector(i);
    }
  }

  return CreatedTag();
}

/****************************************************************************/
const TagSearchResult* TagDictionary::SearchForTag(int TagPos_, bool Empty_, bool Container_,
                                                   int AppAttr_, const char* StartBrk_,
                                                   const ChrString& Element_, const char* EndBrk_)
{
  bool GenSrchWasDone_ = _GenSearchDone;
  bool SpecSrchWasDone_ = _SpecSearchDone;

  _GenSearchDone = false;
  _SpecSearchDone = false;

  if ((TagPos_ != TagEnums::OPENING &&
       TagPos_ != TagEnums::CLOSING &&
       TagPos_ != TagEnums::FORBIDDEN) ||
      (!StartBrk_ || !EndBrk_))
    return NULL;

  bool Singleton_ = TagPos_ == TagEnums::FORBIDDEN;
  bool EmptyTag_ = Empty_;
  bool PairContent_ = false;

  bool Updated_ = false;
  int IndexPos_ = _TagStrIndex[TagPos_];
  
  long* TagMax_ = (TagPos_ == TagEnums::OPENING ||
                   TagPos_ == TagEnums::FORBIDDEN) ?
                     &_StdVectorMax:
                     &_EndVectorMax;
  long* TagLen_ = (TagPos_ == TagEnums::OPENING ||
                   TagPos_ == TagEnums::FORBIDDEN) ?
                     &_StdVectorLength:
                     &_EndVectorLength;
  int Index_ = (TagLen_ == &_StdVectorLength) ? 1:0;

  bool HasEntries_ =
      ((_TagTypeIndex >= 0 && IndexPos_ >= 0) &&
       (((TagPos_ == TagEnums::OPENING ||
          TagPos_ == TagEnums::FORBIDDEN) && *TagLen_ && _TagInfoMatrix[Index_][*TagLen_-1]) ||
        (TagPos_ == TagEnums::CLOSING && *TagLen_ && _TagInfoMatrix[Index_][*TagLen_-1])));

  int PairEnder_ = 0;
  int BrkIndex_ = _GenSearchTag->BrkStringToIndex(StartBrk_, EndBrk_, PairEnder_);
  int TagType_ = (Index_ >= 0) ? _GenSearchTag->BrkIndexToTagType(BrkIndex_):-1;
  int attr = AppAttr_ ? AppAttr_:_GenSearchTag->GiveDefaultTagAttrib(TagType_, TagPos_);

  if (*TagMax_ && HasEntries_)
  {
    TagTypeInfo* TagInfop_ = new TagTypeInfo();
    if (_DocType)
      TagInfop_->SetDocType(_DocType);

    DetermineTagAttrib(TagInfop_, TagPos_, AppAttr_, attr,
                       EmptyTag_, Singleton_, Empty_, Container_,
                       PairContent_, StartBrk_, EndBrk_);

    TagInfop_->SetSequencer(Sequencer());
    TagInfop_->SetHtmlSpecialCase(HtmlSpecialCase());
    TagInfop_->SetTagBrk(StartBrk_, EndBrk_, NULL);
    TagInfop_->SetElementStr(true, new ChrString(Element_), true);
    TagInfop_->SetEmptyTag(Empty_);
    TagInfop_->SetSingletonTag(Singleton_);
    TagInfop_->SetContainerTag(Container_);
    TagInfop_->SetTagAttr(attr);

    bool Found_ = HasThisTag(TagPos_, TagInfop_, false);
    _SpecSearchDone = true;
    _GenSearchDone = false;

    if (Found_ && ContentTypeTagFound())
    {
      delete TagInfop_;
      TagInfop_ = NULL;

      int TagBits_[TagTypeInfo::MAX_TAGINDEXES];
      long i = _TagTypeIndex;
      long x = _TagStrIndex[i];

      ::memset(TagBits_, 0, TagTypeInfo::MAX_TAGINDEXES);
      _TagInfoMatrix[i][x]->SetDocType(_DocType);
      _TagInfoMatrix[i][x]->ShouldDumpContents(false, *_GenSearchTag->GiveOutput(), 20);

      Found_ = _TagInfoMatrix[i][x]->SearchForTag(TagBits_);

      if (Found_)
      {
        _SpecSearchFailed = NO_FAIL;
        _SearchResult = _TagInfoMatrix[i][x]->SearchResult();

        return _SearchResult;
      }
      else
        _SpecSearchFailed = NOT_A_TAG;
    }
    else
    {
      _SpecSearchFailed = UNREC_TAG;
      DumpUnrecognizedTag(TagInfop_);

      delete TagInfop_;
      TagInfop_ = NULL;
    }
  }
  else
  {
    _GenSearchDone = GenSrchWasDone_;
    _SpecSearchDone = SpecSrchWasDone_;
  }

  return NULL;
}

/****************************************************************************/
const TagSearchResult* TagDictionary::SearchForTag(int DocType_)
{
  bool GenSrchWasDone_ = _GenSearchDone;
  bool SpecSrchWasDone_ = _SpecSearchDone;

  _GenSearchDone = false;
  _SpecSearchDone = false;

  bool Updated_ = false;
  bool HasEntries_ = (_TagTypeIndex >= 0 &&
                      _StdVectorLength && _TagInfoMatrix[TagEnums::OPENING][_StdVectorLength-1]);

  if (_StdVectorMax && HasEntries_ && _GenSearchTag)
  {
    StackNodeInfo* CurNode_ = NULL;
    StackNodeInfo* EncNode_ = NULL;

    int TagPos_ = 0;
    int TagNestCond_ = 0;
    int CurrSearchRun_ = 0;
    int PrevSearchRun_ = _GenSearchTag->SearchRunNumber();
    int TagBits_[TagTypeInfo::MAX_TAGINDEXES];
    TagElement* ElmPtr_ = _GenSearchTag->GiveTagElementPtr();
    ::memset(TagBits_, 0, TagTypeInfo::MAX_TAGINDEXES);
    bool Found_ = false;
    bool PrevSearchDone_ = PrevSearchRun_ && GenSrchWasDone_ &&
                           _GenSearchFailed == NO_FAIL &&
                           (NonContentTypeTagFound() ||
                            ContentTypeTagFound());
    bool PrevTagFound_ = PrevSearchDone_ && ElmPtr_;

    if (PrevTagFound_)
      _GenSearchTag->SetPrevTagInfo(_GenSearchTag->HeadInfo(true, true));
    else if (!PrevSearchRun_)
      Sequencer()->ResetPrevTagInfo();
    else
      PrevSearchDone_ = Sequencer()->PrevTagInfoSet();

    _GenSearchTag->Reset(TagTypeInfo::SEARCH_PARAMS);
    _GenSearchTag->SetTagElementPtr(NULL);
    _GenSearchTag->SetDocType(_DocType);
    _GenSearchTag->ShouldDumpContents(_DumpTagContents, *_GenSearchTag->GiveOutput(), 20);

    _GenSearchTag->SetTagSwitches(_TagSwitch, TagBits_);
    Found_ = _GenSearchTag->SearchForTag(TagBits_);
    _GenSearchDone = true;
    _SpecSearchDone = false;

    _HasTag = false;
    _DictSearchDone = false;

    if (Found_)
    {
      bool EmptyTag_ = _GenSearchTag->IsEmptyTag();
      bool Singleton_ = _GenSearchTag->IsSingletonTag();
      bool Container_ = _GenSearchTag->IsContainerTag();
      bool Empty_ = false;
      bool PairContent_ = false;

      int attr = _GenSearchTag->TagAttribute();
      const char* StartBrk_ = _GenSearchTag->StartBrkFound();
      const char* EndBrk_ = _GenSearchTag->EndBrkFound();

      TagTypeInfo* TagInfop_ = new TagTypeInfo();
      if (_DocType)
        TagInfop_->SetDocType(_DocType);

      ElmPtr_ = TagInfop_->GiveTagElementPtr();
      TagPos_ = ElmPtr_ ? ElmPtr_->TagPosition():
                          _GenSearchTag->GiveTagPosition();

      DetermineTagAttrib(TagInfop_, TagPos_, 99, attr,
                         EmptyTag_, Singleton_, Empty_, Container_,
                         PairContent_, StartBrk_, EndBrk_);

      TagInfop_->SetSequencer(Sequencer());
      TagInfop_->SetHtmlSpecialCase(HtmlSpecialCase());
      TagInfop_->SetTagBrk(StartBrk_, EndBrk_, NULL);
      TagInfop_->SetElementStr(true, new ChrString(_GenSearchTag->ElementStr()), true);
      TagInfop_->SetEmptyTag(Empty_);
      TagInfop_->SetSingletonTag(Singleton_);
      TagInfop_->SetContainerTag(Container_);
      TagInfop_->SetTagAttr(attr);
      TagInfop_->SetTagTypeFound(_GenSearchTag->GiveTagBrkIndex(), false);

      Found_ = HasThisTag(TagPos_, TagInfop_, false, DocType_, true);

      if (Found_)
      {
        delete TagInfop_;
        TagInfop_ = NULL;
        _PrevUnorderedTag = false;

        if (ContentTypeTagFound())
        {
          TagInfop_ = DictionarySearchTag();
          TagPos_ = TagInfop_->GiveTagPosition();
          _GenSearchTag->AdjustTagCount(TagPos_);
          _GenSearchTag->SetTagInfoFound(TagInfop_, true);
          _GenSearchTag->UpdateTag(TagPos_);
        }

        _GenSearchFailed = NO_FAIL;
        CurNode_ = Sequencer()->PrevTagInfoNode();
        _GenSearchTag->CreateStackInfoNode(DocType_, NULL, false,
                                           _GenSearchDone, _SpecSearchDone);
        EncNode_ = _GenSearchTag->HeadInfo(true, true);
        CurrSearchRun_ = _GenSearchTag->SearchRunNumber();

        if (ContentTypeTagFound())
        {
          // Set HtmlTagErrors (html,head,body) in html type documents,
          // process RequiredTags conditions in linked list of tag elements
          SetHtmlTagErrors(EncNode_);
          ProcessRequiredTagCond(EncNode_);
          EncNode_ = _GenSearchTag->HeadInfo(true, true);
        }

        // Set tag nesting condition to determine tag content appending status
        if (CurNode_ && PrevSearchDone_)
        {
          _GenSearchTag->CompareWithPrevTagInfo(EncNode_);
          if (_DumpTagContents && TagTypeInfo::ShowDebugDump())
            CurNode_->DumpTagContentData(*_GenSearchTag->GiveOutput(), _GenSearchTag->GiveFieldWidth());

          _PrevUnorderedTag = false;
          TagNestCond_ = _GenSearchTag->GiveTagNestingCondition();
          CurNode_->SetTagNestCond(TagNestCond_, EncNode_);
        }
        else if (EncNode_ && CurrSearchRun_ &&
                 (NonContentTypeTagFound() || ContentTypeTagFound()))
        {
          TagNestCond_ = _GenSearchTag->GiveTagNestingCondition(true);
          EncNode_->SetPrevUnorderedTag(PrevSearchDone_, GenSrchWasDone_,
                                        _GenSearchFailed, PrevSearchRun_, CurrSearchRun_,
                                        NonContentTypeTagFound(), ContentTypeTagFound());

          _PrevUnorderedTag = EncNode_->PrevUnorderedTagFound();
          EncNode_->SetTagNestCond(TagNestCond_, EncNode_);
        }

        ElmPtr_ = _GenSearchTag->GiveTagElementPtr();
        PrevTagFound_ = CurrSearchRun_ &&
                        ElmPtr_ && TagNestCond_ &&
                        (NonContentTypeTagFound() || ContentTypeTagFound());

        if (PrevTagFound_)
          _GenSearchTag->SetPrevTagInfo(EncNode_);

        _GenSearchTag->ResetTagInfo();
        _SearchResult = _GenSearchTag->SearchResult();
        return _SearchResult;
      }
      else if (!NonContentTypeTagFound() && ContentTypeTagFound(false))
      {
        // shift back of contents from unrecognized content type tags
        DumpUnrecognizedTag(TagInfop_);

        _GenSearchFailed = UNREC_TAG;
        _GenSearchTag->TransferTagFoundDataTo(TagInfop_);
        _GenSearchTag->SetTagInfoFound(TagInfop_, true);
        _GenSearchTag->CreateStackInfoNode(DocType_, TagInfop_, true,
                                           _GenSearchDone, _SpecSearchDone);

        delete TagInfop_;
        TagInfop_ = NULL;

        _GenSearchTag->HeadInfo(true, true);
        _GenSearchTag->ResetTagInfo();
        _SearchResult = _GenSearchTag->SearchResult();
        return _SearchResult;
      }
    }
    else
      _GenSearchFailed = NOT_A_TAG;
  }
  else
  {
    _GenSearchDone = GenSrchWasDone_;
    _SpecSearchDone = SpecSrchWasDone_;
  }

  return NULL;
}

/****************************************************************************/
bool TagDictionary::TagFound(bool AllTags_) const
{
  return (((_SpecSearchDone && _SpecSearchFailed == NO_FAIL) ||
           (_GenSearchDone && _GenSearchFailed == NO_FAIL)) &&
          (ContentTypeTagFound() || (AllTags_ && NonContentTypeTagFound())));
}

/****************************************************************************/
TagDictionary& TagDictionary::AssignSearchResult(Subscript Index_, Subscript Length_,
                                                 Subscript MIndex_, Subscript MLength_,
                                                 Subscript EIndex_, Subscript ELength_)
{
  long i = _TagTypeIndex;
  long x = _TagStrIndex[i];

  if (_SearchResult)
    if (_GenSearchDone && _GenSearchFailed <= UNREC_TAG)
      _GenSearchTag->AssignSearchResult(Index_, Length_,
                                        MIndex_, MLength_,
                                        EIndex_, ELength_);
    else if (_SpecSearchDone && _SpecSearchFailed <= UNREC_TAG)
      _TagInfoMatrix[i][x]->AssignSearchResult(Index_, Length_,
                                               MIndex_, MLength_,
                                               EIndex_, ELength_);

  return *this;
}

/****************************************************************************/
TagDictionary& TagDictionary::AdvanceReadPos()
{
  long i = _TagTypeIndex;
  long x = _TagStrIndex[i];

  if (_SearchResult)
    if (_GenSearchDone && _GenSearchFailed <= UNREC_TAG)
    {
      _GenSearchTag->AdvanceReadPos();
      _GenSearchTag->GiveNextSwitches(_TagSwitch);
    }
    else if (_SpecSearchDone && _SpecSearchFailed <= UNREC_TAG)
      _TagInfoMatrix[i][x]->AdvanceReadPos();

  return *this;
}

/****************************************************************************/
TagDictionary& TagDictionary::ResetReadPos()
{
  long i = _TagTypeIndex;
  long x = _TagStrIndex[i];

  if (_SearchResult)
    if (_GenSearchDone && _GenSearchFailed <= UNREC_TAG)
      _GenSearchTag->ResetReadPos();
    else if (_SpecSearchDone && _SpecSearchFailed <= UNREC_TAG)
      _TagInfoMatrix[i][x]->ResetReadPos();

  return *this;
}

/****************************************************************************/
Subscript TagDictionary::ReadPos() const
{
  long i = _TagTypeIndex;
  long x = _TagStrIndex[i];

  if (_SearchResult)
    if (_GenSearchDone && _GenSearchFailed <= UNREC_TAG)
      return _GenSearchTag->ReadPos();
    else if (_SpecSearchDone && _SpecSearchFailed <= UNREC_TAG)
      return _TagInfoMatrix[i][x]->ReadPos();

  return 0;
}

/****************************************************************************/
TagLinksSequencer* TagDictionary::Sequencer()
{
  if (_GenSearchTag)
  {
    if (!_Sequencer)
    {
      _Sequencer = new TagLinksSequencer();
      _SequencerCopied = false;
    }

    if (!_GenSearchTag->Sequencer())
      _GenSearchTag->SetSequencer(_Sequencer);

    return _Sequencer;
  }
}

/****************************************************************************/
const TagLinksSequencer* TagDictionary::Sequencer() const
{
  if (_GenSearchTag)
  {
    if (!_Sequencer)
    {
      _Sequencer = new TagLinksSequencer();
      _SequencerCopied = false;
    }

    if (!_GenSearchTag->Sequencer())
      _GenSearchTag->SetSequencer(_Sequencer);

    return _Sequencer;
  }
}

/****************************************************************************/
TagDictionary& TagDictionary::SetSequencer(TagLinksSequencer* Ptr_)
{
  if (_GenSearchTag && Ptr_)
  {
    if (!_SequencerCopied)
      delete _Sequencer;

    _Sequencer = Ptr_;
    _SequencerCopied = true;
  }

  return *this;
}

/****************************************************************************/
StackNodeInfo* TagDictionary::ResetStackPtr(int StackType_, int EndType_, bool NotErased_)
{
  StackNodeInfo* ret = NULL;

  ret = (StackType_ == TagLinksSequencer::INFO_STACK) ?
          ((EndType_ == TagLinksSequencer::HEAD) ?
               Sequencer()->NextStackInfo(true, false, NotErased_):
           (EndType_ == TagLinksSequencer::TAIL) ?
               Sequencer()->PrevStackInfo(true, false, NotErased_):
               NULL):
        (StackType_ == TagLinksSequencer::CURSOR_STACK) ?
          ((EndType_ == TagLinksSequencer::HEAD) ?
               Sequencer()->NextCursorTag(true, false, NotErased_):
           (EndType_ == TagLinksSequencer::TAIL) ?
               Sequencer()->PrevCursorTag(true, false, NotErased_):
               NULL):
        (StackType_ == TagLinksSequencer::TAG_STACK) ?
          ((EndType_ == TagLinksSequencer::HEAD) ?
               Sequencer()->NextTagStack(true, false, NotErased_):
           (EndType_ == TagLinksSequencer::TAIL) ?
               Sequencer()->PrevTagStack(true, false, NotErased_):
               NULL):
          NULL;

  return ret;
}

/****************************************************************************/
StackNodeInfo* TagDictionary::PrevStackPtr(int StackType_, bool NotErased_)
{
  StackNodeInfo* ret = NULL;

  ret = (StackType_ == TagLinksSequencer::INFO_STACK) ?
            Sequencer()->PrevStackInfo(false, false, NotErased_):
        (StackType_ == TagLinksSequencer::CURSOR_STACK) ?
            Sequencer()->PrevCursorTag(false, false, NotErased_):
        (StackType_ == TagLinksSequencer::TAG_STACK) ?
            Sequencer()->PrevTagStack(false, false, NotErased_):
            NULL;

  return ret;
}

/****************************************************************************/
StackNodeInfo* TagDictionary::NextStackPtr(int StackType_, bool NotErased_)
{
  StackNodeInfo* ret = NULL;

  ret = (StackType_ == TagLinksSequencer::INFO_STACK) ?
            Sequencer()->NextStackInfo(false, false, NotErased_):
        (StackType_ == TagLinksSequencer::CURSOR_STACK) ?
            Sequencer()->NextCursorTag(false, false, NotErased_):
        (StackType_ == TagLinksSequencer::TAG_STACK) ?
            Sequencer()->NextTagStack(false, false, NotErased_):
            NULL;

  return ret;
}

/****************************************************************************/
HtmlSpecialCaseData* TagDictionary::HtmlSpecialCase()
{
  if (_GenSearchTag)
  {
    if (!_HtmlSpecialCase)
    {
      _HtmlSpecialCase = new HtmlSpecialCaseData();
      _HtmlSpecialCaseCopied = false;
    }

    if (!_GenSearchTag->HtmlSpecialCase())
      _GenSearchTag->SetHtmlSpecialCase(_HtmlSpecialCase);

    return _HtmlSpecialCase;
  }
}

/****************************************************************************/
const HtmlSpecialCaseData* TagDictionary::HtmlSpecialCase() const
{
  if (_GenSearchTag)
  {
    if (!_HtmlSpecialCase)
    {
      _HtmlSpecialCase = new HtmlSpecialCaseData();
      _HtmlSpecialCaseCopied = false;
    }

    if (!_GenSearchTag->HtmlSpecialCase())
      _GenSearchTag->SetHtmlSpecialCase(_HtmlSpecialCase);

    return _HtmlSpecialCase;
  }
}

/****************************************************************************/
TagDictionary& TagDictionary::SetHtmlSpecialCase(HtmlSpecialCaseData* Ptr_)
{
  if (_GenSearchTag && Ptr_)
  {
    if (!_HtmlSpecialCaseCopied)
      delete _HtmlSpecialCase;

    _HtmlSpecialCase = Ptr_;
    _HtmlSpecialCaseCopied = true;
  }

  return *this;
}

/****************************************************************************/
void TagDictionary::RestoreTagSwitches()
{
  if (_GenSearchTag)
    _GenSearchTag->RestoreTagSwitches();
}

/****************************************************************************/
void TagDictionary::InitCppLangTagSwitches()
{
  _GenSearchTag->InitCppLangTagSwitches(_TagSwitch);
  _TagType = TagEnums::CPPSTYLE;
}

/****************************************************************************/
void TagDictionary::InitCLangTagSwitches()
{
  _GenSearchTag->InitCppLangTagSwitches(_TagSwitch);
  _TagType = TagEnums::CSTYLE;
}

/****************************************************************************/
void TagDictionary::InitAllHtmlTagSwitches()
{
  _GenSearchTag->InitAllHtmlTagSwitches(_TagSwitch);
  _TagType = TagEnums::HTMLSTYLE;

  if (!_HtmlDictionaryCreated)
    MakeHtmlTagDictionary();
}

/****************************************************************************/
void TagDictionary::InitAllXmlTagSwitches()
{
  _GenSearchTag->InitAllXmlTagSwitches(_TagSwitch);
  _TagType = TagEnums::XMLSTYLE;
}

/****************************************************************************/
void TagDictionary::WriteHtmlTagDictionary(const char* OutFile_)
{
  /*
  static const char* TESTIN     = "text.in";
  static const char* TESTOUT    = "text.out";

  char Buffer_[128];
  FileReader Fr_;

  if (Fr_.Open(TESTIN, ios::in) && Fr_.Open(TESTOUT, ios::out))
    while (!Fr_.ReadLine(Buffer_, 128).EndOfFile())
      Fr_.WriteLine(Buffer_);
  */
}

/****************************************************************************/
void TagDictionary::ReadHtmlTagDictionary(const char* InFile_)
{
  /*
  static const char* TESTIN     = "text.in";
  static const char* TESTOUT    = "text.out";

  char Buffer_[128];
  FileReader Fr_;

  if (Fr_.Open(TESTIN, ios::in) && Fr_.Open(TESTOUT, ios::out))
    while (!Fr_.ReadLine(Buffer_, 128).EndOfFile())
      Fr_.WriteLine(Buffer_);
  */
}

/****************************************************************************/
void TagDictionary::RetrieveIntraTagContent(bool Flag_)
{
  _TagSwitch._RetrieveComment       = _TagSwitch._IsComment && Flag_;
  _TagSwitch._RetrieveNestedComment = _TagSwitch._IsNestedComment && Flag_;
  _TagSwitch._RetrieveScript        = _TagSwitch._IsScript && Flag_;
  _TagSwitch._RetrieveCLang         = _TagSwitch._IsCLang && Flag_;
  _TagSwitch._RetrieveCppLang       = _TagSwitch._IsCppLang && Flag_;
}

/****************************************************************************/
void TagDictionary::DumpUnrecognizedTag(TagTypeInfo* TagPtr_)
{
  if (!TagPtr_ || !TagTypeInfo::ShowDebugDump())
    return;

  const char* StartBrk_ = TagPtr_->StartBrk();
  const char* EndBrk_ = TagPtr_->EndBrk();

  *_GenSearchTag->GiveOutput() <<"Unrecognized Tag... ";

  if (TagPtr_->_ElementStr && StartBrk_ && EndBrk_)
    *_GenSearchTag->GiveOutput() <<StartBrk_
                                 <<(*TagPtr_->_ElementStr)
                                 <<EndBrk_ <<endl;
  else
    *_GenSearchTag->GiveOutput() <<endl;
}

/****************************************************************************/
void TagDictionary::DumpDictionary(ostream& os_, size_t fw_) const
{
  if (!_TagInfoMatrix)
    return;

  long x, y;
  long len[2];

  len[0] = _StdVectorLength;
  len[1] = _EndVectorLength;

  os_.width(fw_);
  os_ <<std::left <<"Std Vector Length: " <<len[0] <<endl;

  os_.width(fw_);
  os_ <<std::left <<"End Vector Length: " <<len[1] <<endl <<endl;

  TagElement* ElmPtr_;
  int TagPos_;

  const char* StdBrk_;
  const char* EndBrk_;
  const char* TagType_;

  for (y = 0; y < 2; y++)
    for (x = 0; x < len[y]; x++)
      if (_TagInfoMatrix[y] && _TagInfoMatrix[y][x])
        if (_TagInfoMatrix[y][x]->TagBrk_Specified() &&
            _TagInfoMatrix[y][x]->ElementStr_Specified())
        {
          ElmPtr_ = _TagInfoMatrix[y][x]->GiveTagElementPtr();
          TagPos_ = ElmPtr_ ? ElmPtr_->TagPosition():
                              _TagInfoMatrix[y][x]->GiveTagPosition();

          TagType_ =
            (TagPos_ == TagEnums::CLOSING)   ? "Close Tag":
            (TagPos_ == TagEnums::FORBIDDEN) ? "Singleton Tag":"Open Tag";
          StdBrk_ = _TagInfoMatrix[y][x]->StartBrkFound();
          EndBrk_ = _TagInfoMatrix[y][x]->EndBrkFound();

          os_.width(fw_);
          os_ <<std::left <<"Tag Type: "
              <<TagType_ <<endl;

          os_.width(fw_);
          os_ <<std::left <<"Tag String: "
              <<StdBrk_
              <<_TagInfoMatrix[y][x]->ElementStr()
              <<EndBrk_ <<endl <<endl;
        }
        else
        {
          os_.width(fw_);
          os_ <<std::left <<"*** Tag Unspecified ***" <<endl <<endl;
        }
}

/****************************************************************************/
void TagDictionary::DumpSearchResults(ostream& os_, size_t fw_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  const TagSearchResult* Result_ = SearchResult();
  ChrString RemStr_ = Result_->c_str();
  Subscript rPos_ = ReadPos();
  StackNodeInfo* StackNode_ = NULL;
  const TagTypeInfo* DictSrch_ = DictionarySearchTag();
  const TagTypeInfo* SrchPtr_ = (_GenSearchDone && _GenSearchFailed <= UNREC_TAG) ?
                                     GeneralSearchTag():NULL;

  if (_SpecSearchDone && _SpecSearchFailed <= UNREC_TAG &&
      DictSrch_ && DictSrch_->ElementStr_Specified())
  {
    os_.width(fw_);
    os_ <<std::left <<"Element String Specified: true" <<endl;
    os_.width(fw_);
    os_ <<std::left <<"Element String: " <<DictSrch_->ElementStr() <<endl;
    DictSrch_->ShouldDumpContents(_DumpTagContents, os_, fw_);
  }
  else if (_GenSearchDone && SrchPtr_)
    { SrchPtr_->ShouldDumpContents(_DumpTagContents, os_, fw_); }
  else
  {
    os_.width(fw_); os_ <<std::left <<"Search Failed... " <<endl;
    return;
  }

  if (!TagTypeInfo::ShowNewLines())
  {
    RemStr_.RemovePadding(" \t\n\r\f\v");
    SubstChar(RemStr_.c_str(), "\n\r", ' ');
    RemStr_.SimplifySpaces(false);
  }

  #if TAGREADER_ABBREV_DUMP
    os_.width(fw_); os_ <<std::left <<"Read Position: " <<rPos_ <<endl;
  #else
    os_.width(fw_); os_ <<std::left <<"Read Position: " <<rPos_ <<endl;
    os_.width(fw_); os_ <<std::left <<"Tag Index: " <<SrchPtr_->GiveTagBrkIndex() <<endl;
    os_.width(fw_); os_ <<std::left <<"Start Brk Found: " <<SrchPtr_->StartBrkFound() <<endl;
    os_.width(fw_); os_ <<std::left <<"End Brk Found: " <<SrchPtr_->EndBrkFound() <<endl;
    os_.width(fw_); os_ <<std::left <<"String Remaining: " <<RemStr_ <<endl;
  #endif

  Result_->DumpSearchResults(os_, fw_, TagTypeInfo::ShowNewLines());

  if (_DumpTagContents && !_SpecSearchDone &&
      _GenSearchDone && _GenSearchFailed <= UNREC_TAG)
    DumpIntraTagContentData(os_, fw_);
  else
    os_ <<endl;
}

/****************************************************************************/
void TagDictionary::DumpTagContentData(ostream& os_, size_t fw_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  const TagSearchResult* Result_ = SearchResult();
  Result_->DumpTagContentData(os_, fw_);
  os_ <<endl;
}

/****************************************************************************/
void TagDictionary::DumpIntraTagContentData(ostream& os_, size_t fw_) const
{
  if (!TagTypeInfo::ShowDebugDump())
    return;

  const TagSearchResult* Result_ = SearchResult();
  int ContentType_ = Result_->ContentType();
  bool HasIntTagCont_ = (ContentType_ & TagSearchResult::INTRA_TAG_CONTENT) ||
                        (ContentType_ & TagSearchResult::INTRA_NESTEDTAG_CONTENT);

  if (HasIntTagCont_)
    Result_->DumpIntraTagContentData(os_, fw_);
}

/****************************************************************************/
void TagDictionary::TagInfoListWalkDump(ostream& os_, size_t fw_)
{
  StackNodeInfo* TailEnd_ = Sequencer()->TailInfo(true, false);

  if (TagTypeInfo::ShowDebugDump())
  {
    os_ <<endl <<"------------------------- TagInfoListWalkDump Start -------------------------" <<endl;
    StackNodeInfo::ShowStackAddr(TailEnd_, StackNodeInfo::PREVNEXT,
                                 "TagInfoListWalkDump", -1, true);
    os_ <<endl <<"-------------------------- TagInfoListWalkDump End --------------------------" <<endl;
  }
}

/****************************************************************************/
void TagDictionary::ShouldDumpContents(bool Flag_) const
{
  _DumpTagContents = Flag_;

  const TagSearchResult* Result_ = SearchResult();
  const TagTypeInfo* DictSrch_ = DictionarySearchTag();
  const TagTypeInfo* SrchPtr_ = (_GenSearchDone && _GenSearchFailed <= UNREC_TAG) ?
                                     GeneralSearchTag():NULL;

  if (_SpecSearchDone && DictSrch_)
    { DictSrch_->ShouldDumpContents(_DumpTagContents, *_GenSearchTag->GiveOutput(), 20); }
  else if (_GenSearchDone && SrchPtr_)
    { SrchPtr_->ShouldDumpContents(_DumpTagContents, *_GenSearchTag->GiveOutput(), 20); }
}

/****************************************************************************/
MEMORYOPS_DEFN(TagDictionary)

/****************************************************************************/
// TagParseInfo class definition
/****************************************************************************/
TagParseInfo::TagParseInfo(TagStringData* Parent_):
_Parent(Parent_),
    
_TagInfo(NULL),
_Dictionary(NULL),
_TagCommentData(NULL),
_KeyValueData(NULL),
_SearchResult(NULL),
    
_DocType(0),
_EofFound(false),
  
_IsHtmlTag(false),
_IsXHtmlTag(false),
_IsXmlTag(false),
_IsCppTag(false),
_IsCTag(false),
_IsNonStdTag(false),
_IsSingletonTag(false),
_IsEmptyTag(false),
_IsContainerTag(false),

_TagError(0),
_AnchorType(0),
_FoundTagType(0),

_AltIndex(0),
_TagBrkIndex(0),
_EndBrkIndex(0),
_TagTypeIndex(0),
_TagSearchDone(false)
{
  _TagStrIndex[0] =
  _TagStrIndex[1] = 0;
}

/****************************************************************************/
TagParseInfo::TagParseInfo(const TagParseInfo& Obj_):
_Parent(Obj_._Parent),
    
_TagInfo(Obj_._TagInfo),
_Dictionary(Obj_._Dictionary),
_TagCommentData(Obj_._TagCommentData),
_KeyValueData(Obj_._KeyValueData),
_SearchResult(Obj_._SearchResult),
    
_DocType(Obj_._DocType),
_EofFound(Obj_._EofFound),
  
_IsHtmlTag(Obj_._IsHtmlTag),
_IsXHtmlTag(Obj_._IsXHtmlTag),
_IsXmlTag(Obj_._IsXmlTag),
_IsCppTag(Obj_._IsCppTag),
_IsCTag(Obj_._IsCTag),
_IsNonStdTag(Obj_._IsNonStdTag),
_IsSingletonTag(Obj_._IsSingletonTag),
_IsEmptyTag(Obj_._IsEmptyTag),
_IsContainerTag(Obj_._IsContainerTag),

_TagError(Obj_._TagError),
_AnchorType(Obj_._AnchorType),
_FoundTagType(Obj_._FoundTagType),

_AltIndex(Obj_._AltIndex),
_TagBrkIndex(Obj_._TagBrkIndex),
_EndBrkIndex(Obj_._EndBrkIndex),
_TagTypeIndex(Obj_._TagTypeIndex),
_TagSearchDone(Obj_._TagSearchDone)
{
  _TagStrIndex[0] = Obj_._TagStrIndex[0];
  _TagStrIndex[1] = Obj_._TagStrIndex[1];
}

/****************************************************************************/
TagParseInfo* TagParseInfo::Make(TagStringData* Parent_)
{
  return (new TagParseInfo(Parent_));
}

/****************************************************************************/
TagParseInfo* TagParseInfo::Make(const TagParseInfo& Obj_)
{
  return (new TagParseInfo(Obj_));
}

/****************************************************************************/
void TagParseInfo::SetShowTagDump(bool DebugDump_, bool ShowErased_, bool ContentDump_)
{
  TagTypeInfo::SetShowDebugDump(DebugDump_, ShowErased_);

  if (_Dictionary)
    _Dictionary->ShouldDumpContents(ContentDump_);
}

/****************************************************************************/
void TagParseInfo::SetShowNewLines(bool Flag_)
{
  TagTypeInfo::SetShowNewLines(Flag_);
}

/****************************************************************************/
TagParseInfo& TagParseInfo::operator = (const TagParseInfo& Obj_)
{
  if (&Obj_ != this)
  {
    _Parent = Obj_._Parent;
    
    _TagInfo = Obj_._TagInfo;
    _Dictionary = Obj_._Dictionary;
    _TagCommentData = Obj_._TagCommentData;
    _KeyValueData = Obj_._KeyValueData;
    _SearchResult = Obj_._SearchResult;
    
    _DocType = Obj_._DocType;
    _EofFound = Obj_._EofFound;
  
    _IsHtmlTag = Obj_._IsHtmlTag;
    _IsXHtmlTag = Obj_._IsXHtmlTag;
    _IsXmlTag = Obj_._IsXmlTag;
    _IsCppTag = Obj_._IsCppTag;
    _IsCTag = Obj_._IsCTag;
    _IsNonStdTag = Obj_._IsNonStdTag;
    _IsSingletonTag = Obj_._IsSingletonTag;
    _IsEmptyTag = Obj_._IsEmptyTag;
    _IsContainerTag = Obj_._IsContainerTag;

    _TagError = Obj_._TagError;
    _AnchorType = Obj_._AnchorType;
    _FoundTagType = Obj_._FoundTagType;

    _AltIndex = Obj_._AltIndex;
    _TagBrkIndex = Obj_._TagBrkIndex;
    _EndBrkIndex = Obj_._EndBrkIndex;
    _TagTypeIndex = Obj_._TagTypeIndex;
    _TagSearchDone = Obj_._TagSearchDone;

    _TagStrIndex[0] = Obj_._TagStrIndex[0];
    _TagStrIndex[1] = Obj_._TagStrIndex[1];
  }

  return *this;
}

/****************************************************************************/
const TagSearchResult* TagParseInfo::SearchForTag(int TagPos_, bool Empty_, bool Container_,
                                                  int AppAttr_, const char* StartBrk_,
                                                  const ChrString& Element_, const char* EndBrk_)
{
  if (_Dictionary)
  {
    _TagError = TagEnums::PENDING;
    _SearchResult = _Dictionary->SearchForTag(TagPos_, Empty_, Container_, AppAttr_,
                                              StartBrk_, Element_, EndBrk_);

    _TagInfo = _Dictionary->DictionarySearchTag();
    _TagSearchDone = _Dictionary->HasEntries(TagPos_, _TagInfo);

    if (_TagSearchDone && _TagInfo)
    {
      TagElement* ElmPtr_ = _TagInfo->GiveTagElementPtr();
      _AnchorType = ElmPtr_ ? ElmPtr_->TagPosition():
                              _TagInfo->GiveTagPosition();
      _FoundTagType = _TagInfo->TagTypeFound();
      _TagInfo->IndexFromBrkFound(&_TagBrkIndex, NULL);
      _TagInfo->IndexFromBrkFound(NULL, &_EndBrkIndex);
      _AltIndex - _TagInfo->GiveAltIndex(true);
      _TagTypeIndex = _Dictionary->TagTypeIndex();
      _TagStrIndex[_TagTypeIndex] = _Dictionary->TagStrIndex(_TagTypeIndex);

      _IsSingletonTag = _TagInfo->IsSingletonTag();
      _IsEmptyTag = _TagInfo->IsEmptyTag();
      _IsContainerTag = _TagInfo->IsContainerTag();
      SetTagType();
      SetTagError(true, TagPos_, Empty_);
    }

    return _SearchResult;
  }

  return NULL;
}

/****************************************************************************/
const TagSearchResult* TagParseInfo::SearchForTag()
{
  if (_Dictionary && _DocType)
  {
    _TagError = TagEnums::PENDING;
    _SearchResult = _Dictionary->SearchForTag(_DocType);

    _TagInfo = _Dictionary->GeneralSearchTag();
    _TagSearchDone = _Dictionary->HasEntries(TagEnums::OPENING, _TagInfo);

    if (_TagSearchDone && _TagInfo)
    {
      _AnchorType = _TagInfo->GiveTagPosition();
      _FoundTagType = _TagInfo->TagTypeFound();
      _TagInfo->IndexFromBrkFound(&_TagBrkIndex, NULL);
      _TagInfo->IndexFromBrkFound(NULL, &_EndBrkIndex);
      _AltIndex = _TagInfo->GiveAltIndex(true);
      _TagTypeIndex = _Dictionary->TagTypeIndex();
      _TagStrIndex[_TagTypeIndex] = _Dictionary->TagStrIndex(_TagTypeIndex);

      _IsSingletonTag = _TagInfo->IsSingletonTag();
      _IsEmptyTag = _TagInfo->IsEmptyTag();
      _IsContainerTag = _TagInfo->IsContainerTag();
      SetTagType();
      SetTagError(false, 0, false);
    }

    return _SearchResult;
  }

  return NULL;
}

/****************************************************************************/
void TagParseInfo::SetTagError(bool Specified_, int TagPos_, bool Empty_)
{
  if (!_TagInfo)
    return;

  bool EmptyTagSpecified_ = Specified_ && (TagPos_ == TagEnums::OPENING && Empty_);
  _TagError = 0;

  if (_TagInfo->EofFound())
  {
    if (_TagInfo->TagFound())
    {
      if (_TagInfo->StdTagCount() > 0 ||
          _TagInfo->NestedCmntTagCount() > 0 ||
          _TagInfo->CopenCmntTagCount() > 0)
        _TagError = _TagInfo->StdTagCount() ? TagEnums::ENDTAG_EXPECTED:
                    _TagInfo->NestedCmntTagCount() ? TagEnums::NESTEDCMNTTAG_EXPECTED:
                    _TagInfo->CopenCmntTagCount() ? TagEnums::CCLOSECMNTTAG_EXPECTED:0;
      else if (_TagInfo->StdTagCount() < 0 ||
               _TagInfo->NestedCmntTagCount() < 0 ||
               _TagInfo->CopenCmntTagCount() < 0)
        _TagError = (_TagInfo->NestingError() != 0) ? _TagInfo->NestingError():0;
      else
        _TagError = (_TagInfo->TagFound() &&
                     _TagInfo->TagTypeFound() &&
                     _TagInfo->EndBrkFound()) ? TagEnums::TAGVALID:
                    _TagInfo->EndBrkFound() ?
                        (EmptyTagSpecified_ ? TagEnums::EMPTYTAG_EXPECTED:
                                              TagEnums::TAG_NOTFOUND):
                        TagEnums::ENDBRK_MISSING;
    }
    else
      _TagError = EmptyTagSpecified_ ? TagEnums::EMPTYTAG_EXPECTED:
                                       TagEnums::TAG_NOTFOUND;
  }
  else
  {
    if (_TagInfo->TagFound())
    {
      if (_TagInfo->StdTagCount() < 0 ||
          _TagInfo->NestedCmntTagCount() < 0 ||
          _TagInfo->CopenCmntTagCount() < 0)
        _TagError = (_TagInfo->NestingError() != 0) ? _TagInfo->NestingError():0;
      else
        _TagError = (_TagInfo->TagFound() &&
                     _TagInfo->TagTypeFound() &&
                     _TagInfo->EndBrkFound()) ? TagEnums::TAGVALID:
                    _TagInfo->EndBrkFound() ?
                        (EmptyTagSpecified_ ? TagEnums::EMPTYTAG_EXPECTED:
                                              TagEnums::TAG_NOTFOUND):
                        TagEnums::ENDBRK_MISSING;
    }
    else
      _TagError = EmptyTagSpecified_ ? TagEnums::EMPTYTAG_EXPECTED:
                                       TagEnums::TAG_NOTFOUND;
  }
}

/****************************************************************************/
bool TagParseInfo::SetTagType()
{
  if (!_DocType || !_TagInfo)
    return false;

  bool CStyle_ = _DocType == TagEnums::C_FILE;
  bool CppStyle_ = _DocType == TagEnums::CPP_FILE;
  bool XmlStyle_ = _DocType == TagEnums::XML_FILE;
  bool HtmlStyle_ = !_DocType ||
                    _DocType == TagEnums::HTML_FILE ||
                    _DocType == TagEnums::XHTML_FILE;
  bool InDocType_ = _TagInfo->InDocType(_DocType);

  if (InDocType_)
  {
    if (HtmlStyle_)
    {
      _IsHtmlTag = (_DocType == TagEnums::HTML_FILE);
      _IsXHtmlTag = (_DocType == TagEnums::XHTML_FILE);
      _IsNonStdTag = false;
    }
    else if (XmlStyle_)
    {
      _IsXmlTag = true;
      _IsNonStdTag = false;
    }
    else if (CStyle_)
    {
      _IsCTag = true;
      _IsNonStdTag = true;
    }
    else if (CppStyle_)
    {
      _IsCppTag = true;
      _IsNonStdTag = true;
    }
  }

  return true;
}

/****************************************************************************/
int TagParseInfo::TagParseError() const
{
  return _TagError;
}

/****************************************************************************/
int TagParseInfo::TagTypeFound() const
{
  return _FoundTagType;
}

/****************************************************************************/
int TagParseInfo::AnchorTypeFound() const
{
  return _AnchorType;
}

/****************************************************************************/
TagStringData* TagParseInfo::Parent()
{
  return _Parent;
}

/****************************************************************************/
TagTypeInfo* TagParseInfo::GetTagInfo()
{
  return _TagInfo;
}

/****************************************************************************/
TagDictionary* TagParseInfo::GiveTagDictionary()
{
  return _Dictionary;
}

/****************************************************************************/
KeyValuePairData* TagParseInfo::GiveKeyValuePairData()
{
  return _KeyValueData;
}

/****************************************************************************/
TagCommentData* TagParseInfo::GiveCommentData()
{
  return _TagCommentData;
}

/****************************************************************************/
TagCountData* TagParseInfo::GiveTagCountData(bool ChkTagFnd_)
{
  return (_Dictionary ? _Dictionary->GiveTagCountData(ChkTagFnd_):NULL);
}

/****************************************************************************/
const ChrString* TagParseInfo::TextBufferPtr() const
{
  return (_Dictionary ? _Dictionary->TextBufferPtr():NULL);
}

/****************************************************************************/
TagParseInfo& TagParseInfo::SetParseHtmlTag(bool IsTag_)
{
  _IsHtmlTag = IsTag_;

  if (IsTag_)
  {
    _IsNonStdTag = false;
    _IsCTag =
    _IsCppTag =
    _IsXmlTag =
    _IsXHtmlTag = false;
  }

  int DocType_ = _IsCTag ? TagEnums::C_FILE:
                 _IsCppTag ? TagEnums::CPP_FILE:
                 _IsXmlTag ? TagEnums::XML_FILE:
                 _IsHtmlTag ? TagEnums::HTML_FILE:
                 _IsXHtmlTag ? TagEnums::XHTML_FILE:0;

  if (DocType_)
  {
    if (_Dictionary)
    {
      _Dictionary->SetDocType(DocType_);
      _Dictionary->InitAllHtmlTagSwitches();

      _EofFound = _Dictionary->EofFound();
      _DocType = _Dictionary->DocTypeSpecified();
      _TagInfo = _Dictionary->GeneralSearchTag();
    }

    if (_TagInfo)
      _TagInfo->SetDocType(DocType_);
  }

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::SetParseXHtmlTag(bool IsTag_)
{
  _IsXHtmlTag = IsTag_;

  if (IsTag_)
  {
    _IsNonStdTag = false;
    _IsCTag =
    _IsCppTag =
    _IsXmlTag =
    _IsHtmlTag = false;
  }

  int DocType_ = _IsCTag ? TagEnums::C_FILE:
                 _IsCppTag ? TagEnums::CPP_FILE:
                 _IsXmlTag ? TagEnums::XML_FILE:
                 _IsHtmlTag ? TagEnums::HTML_FILE:
                 _IsXHtmlTag ? TagEnums::XHTML_FILE:0;

  if (DocType_)
  {
    if (_Dictionary)
    {
      _Dictionary->SetDocType(DocType_);
      _Dictionary->InitAllHtmlTagSwitches();

      _EofFound = _Dictionary->EofFound();
      _DocType = _Dictionary->DocTypeSpecified();
      _TagInfo = _Dictionary->GeneralSearchTag();
    }

    if (_TagInfo)
      _TagInfo->SetDocType(DocType_);
  }

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::SetParseXmlTag(bool IsTag_)
{
  _IsXmlTag = IsTag_;

  if (IsTag_)
  {
    _IsNonStdTag = false;
    _IsCTag =
    _IsCppTag =
    _IsHtmlTag =
    _IsXHtmlTag = false;
  }

  int DocType_ = _IsCTag ? TagEnums::C_FILE:
                 _IsCppTag ? TagEnums::CPP_FILE:
                 _IsXmlTag ? TagEnums::XML_FILE:
                 _IsHtmlTag ? TagEnums::HTML_FILE:
                 _IsXHtmlTag ? TagEnums::XHTML_FILE:0;

  if (DocType_)
  {
    if (_Dictionary)
    {
      _Dictionary->SetDocType(DocType_);
      _Dictionary->InitAllXmlTagSwitches();

      _EofFound = _Dictionary->EofFound();
      _DocType = _Dictionary->DocTypeSpecified();
      _TagInfo = _Dictionary->GeneralSearchTag();
    }

    if (_TagInfo)
      _TagInfo->SetDocType(DocType_);
  }

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::SetParseCppTag(bool IsTag_)
{
  _IsCppTag = IsTag_;

  if (IsTag_)
  {
    _IsNonStdTag = true;
    _IsCTag =
    _IsXmlTag =
    _IsHtmlTag =
    _IsXHtmlTag = false;
  }

  int DocType_ = _IsCTag ? TagEnums::C_FILE:
                 _IsCppTag ? TagEnums::CPP_FILE:
                 _IsXmlTag ? TagEnums::XML_FILE:
                 _IsHtmlTag ? TagEnums::HTML_FILE:
                 _IsXHtmlTag ? TagEnums::XHTML_FILE:0;

  if (DocType_)
  {
    if (_Dictionary)
    {
      _Dictionary->SetDocType(DocType_);
      _Dictionary->InitCppLangTagSwitches();

      _EofFound = _Dictionary->EofFound();
      _DocType = _Dictionary->DocTypeSpecified();
      _TagInfo = _Dictionary->GeneralSearchTag();
    }

    if (_TagInfo)
      _TagInfo->SetDocType(DocType_);
  }

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::SetParseCTag(bool IsTag_)
{
  _IsCTag = IsTag_;

  if (IsTag_)
  {
    _IsNonStdTag = true;
    _IsCppTag =
    _IsXmlTag =
    _IsHtmlTag =
    _IsXHtmlTag = false;
  }

  int DocType_ = _IsCTag ? TagEnums::C_FILE:
                 _IsCppTag ? TagEnums::CPP_FILE:
                 _IsXmlTag ? TagEnums::XML_FILE:
                 _IsHtmlTag ? TagEnums::HTML_FILE:
                 _IsXHtmlTag ? TagEnums::XHTML_FILE:0;

  if (DocType_)
  {
    if (_Dictionary)
    {
      _Dictionary->SetDocType(DocType_);
      _Dictionary->InitCLangTagSwitches();

      _EofFound = _Dictionary->EofFound();
      _DocType = _Dictionary->DocTypeSpecified();
      _TagInfo = _Dictionary->GeneralSearchTag();
    }

    if (_TagInfo)
      _TagInfo->SetDocType(DocType_);
  }

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::SetTagDictionary(TagDictionary* Ptr_)
{
  if (Ptr_)
  {
    _Dictionary = Ptr_;

    _EofFound = _Dictionary->EofFound();
    _DocType = _Dictionary->DocTypeSpecified();
    _TagInfo = _Dictionary->GeneralSearchTag();

    _TagTypeIndex = _Dictionary->TagTypeIndex();
    _TagStrIndex[_TagTypeIndex] = _Dictionary->TagStrIndex(_TagTypeIndex);
  }
  
  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::SetKeyValuePairData(KeyValuePairData* Ptr_)
{
  if (Ptr_)
    _KeyValueData = Ptr_;

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::SetCommentData(TagCommentData* Ptr_)
{
  if (Ptr_)
    _TagCommentData = Ptr_;

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::SetTextBufferPtr(const ChrString* Ptr_)
{
  if (_Dictionary)
  {
    _Dictionary->SetTextBufferPtr(Ptr_);
    _EofFound = _Dictionary->EofFound();
    _DocType = _Dictionary->DocTypeSpecified();
    _TagInfo = _Dictionary->GeneralSearchTag();
  }

  return *this;
}

/****************************************************************************/
TagLinksSequencer* TagParseInfo::Sequencer()
{
  return (_Dictionary ? _Dictionary->Sequencer():NULL);
}

/****************************************************************************/
const TagLinksSequencer* TagParseInfo::Sequencer() const
{
  return (_Dictionary ? _Dictionary->Sequencer():NULL);
}

/****************************************************************************/
StackNodeInfo* TagParseInfo::ResetStackPtr(int StackType_, int EndType_, bool NotErased_)
{
  return
  (
    _Dictionary ?
        _Dictionary->ResetStackPtr(StackType_, EndType_, NotErased_):
        NULL
  );
}

/****************************************************************************/
StackNodeInfo* TagParseInfo::PrevStackPtr(int StackType_, bool NotErased_)
{
  return
  (
    _Dictionary ?
        _Dictionary->PrevStackPtr(StackType_, NotErased_):
        NULL
  );
}

/****************************************************************************/
StackNodeInfo* TagParseInfo::NextStackPtr(int StackType_, bool NotErased_)
{
  return
  (
    _Dictionary ?
        _Dictionary->NextStackPtr(StackType_, NotErased_):
        NULL
  );
}

/****************************************************************************/
HtmlSpecialCaseData* TagParseInfo::HtmlSpecialCase()
{
  return (_Dictionary ? _Dictionary->HtmlSpecialCase():NULL);
}

/****************************************************************************/
const HtmlSpecialCaseData* TagParseInfo::HtmlSpecialCase() const
{
  return (_Dictionary ? _Dictionary->HtmlSpecialCase():NULL);
}

/****************************************************************************/
TagParseInfo& TagParseInfo::Reset(bool ReadPos_, bool TagSwitch_, int DocType_)
{
  if (_Dictionary)
  {
    _Dictionary->Reset(ReadPos_, TagSwitch_, DocType_);
    _EofFound = _Dictionary->EofFound();
    _DocType = _Dictionary->DocTypeSpecified();
    _TagInfo = _Dictionary->GeneralSearchTag();
  }

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::SetDocType(int DocType_)
{
  if (_Dictionary && (DocType_ != _Dictionary->DocTypeSpecified()))
  {
    _Dictionary->SetDocType(DocType_);
    _EofFound = _Dictionary->EofFound();
    _DocType = _Dictionary->DocTypeSpecified();
    _TagInfo = _Dictionary->GeneralSearchTag();
  }

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::SetEofFound(bool Eof_)
{
  if (_Dictionary && (Eof_ != _Dictionary->EofFound()))
  {
    _Dictionary->SetEofFound(Eof_);
    _EofFound = _Dictionary->EofFound();
    _DocType = _Dictionary->DocTypeSpecified();
    _TagInfo = _Dictionary->GeneralSearchTag();
  }

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::AssignSearchResult(Subscript Index_, Subscript Length_,
                                               Subscript MIndex_, Subscript MLength_,
                                               Subscript EIndex_, Subscript ELength_)
{
  if (_Dictionary)
  {
    _Dictionary->AssignSearchResult(Index_, Length_,
                                    MIndex_, MLength_,
                                    EIndex_, ELength_);

    _EofFound = _Dictionary->EofFound();
    _DocType = _Dictionary->DocTypeSpecified();
    _TagInfo = _Dictionary->GeneralSearchTag();

    if (_TagInfo)
      _TagInfo->AssignSearchResult(Index_, Length_,
                                   MIndex_, MLength_,
                                   EIndex_, ELength_);
  }

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::AdvanceReadPos()
{
  if (_Dictionary)
  {
    _Dictionary->AdvanceReadPos();
    _EofFound = _Dictionary->EofFound();
    _DocType = _Dictionary->DocTypeSpecified();
    _TagInfo = _Dictionary->GeneralSearchTag();
  }

  return *this;
}

/****************************************************************************/
TagParseInfo& TagParseInfo::ResetReadPos()
{
  if (_Dictionary)
  {
    _Dictionary->ResetReadPos();
    _EofFound = _Dictionary->EofFound();
    _DocType = _Dictionary->DocTypeSpecified();
    _TagInfo = _Dictionary->GeneralSearchTag();

    if (_TagInfo)
      _TagInfo->ResetReadPos();
  }

  return *this;
}

/****************************************************************************/
Subscript TagParseInfo::ReadPos() const
{
  if (_Dictionary)
    return _Dictionary->ReadPos();

  return 0;
}

/****************************************************************************/
void TagParseInfo::RestoreTagSwitches()
{
  if (_Dictionary)
    _Dictionary->RestoreTagSwitches();
}

/****************************************************************************/
void TagParseInfo::InitCppLangTagSwitches()
{
  if (_Dictionary)
  {
    _Dictionary->InitCppLangTagSwitches();

    _IsCppTag = true;
    _IsNonStdTag = false;
    _IsCTag =
    _IsXmlTag =
    _IsHtmlTag =
    _IsXHtmlTag = false;
  }
}

/****************************************************************************/
void TagParseInfo::InitCLangTagSwitches()
{
  if (_Dictionary)
  {
    _Dictionary->InitCLangTagSwitches();

    _IsCTag = true;
    _IsNonStdTag = false;
    _IsCppTag =
    _IsXmlTag =
    _IsHtmlTag =
    _IsXHtmlTag = false;
  }
}

/****************************************************************************/
void TagParseInfo::InitAllHtmlTagSwitches()
{
  if (_Dictionary)
  {
    _Dictionary->InitAllHtmlTagSwitches();

    _IsHtmlTag = true;
    _IsNonStdTag = false;
    _IsCppTag =
    _IsCTag =
    _IsXmlTag =
    _IsXHtmlTag = false;
  }
}

/****************************************************************************/
void TagParseInfo::InitAllXmlTagSwitches()
{
  if (_Dictionary)
  {
    _Dictionary->InitAllXmlTagSwitches();

    _IsXmlTag = true;
    _IsNonStdTag = false;
    _IsCppTag =
    _IsCTag =
    _IsHtmlTag =
    _IsXHtmlTag = false;
  }
}

/****************************************************************************/
void TagParseInfo::TagInfoListWalkDump(ostream& os_, size_t fw_)
{
  if (_Dictionary)
    _Dictionary->TagInfoListWalkDump(os_, fw_);
}

/****************************************************************************/
void TagParseInfo::RetrieveIntraTagContent(bool Flag_)
{
  if (_Dictionary)
    _Dictionary->RetrieveIntraTagContent(Flag_);
}

/****************************************************************************/
MEMORYOPS_DEFN(TagParseInfo)

/****************************************************************************/
// TagCommentData class definition
/****************************************************************************/
TagCommentData::TagCommentData(TagStringData* Parent_):
_Parent(Parent_),
_IntCmntContent(NULL),
  
_NextCmntBlock(NULL),
_IntCmntEndBrk(NULL),
_IntCmntStartBrk(NULL),
_CmntBlkEnd(NULL),
_CmntBlkStart(NULL),
_IntCmntIndicator(NULL),
_SearchResult(NULL),

_CommentTagString(NULL),
_CommentTagSearch(NULL),

_CommentFound(false),
_NestedCmntTag(false),
_IntCmntTag(false),
_InComment(false),
  
_IntCmntStatus(TagEnums::IGNORED),
_IntCmntAnchor(TagEnums::NO_TAGPOS),
_CmntAnchor(TagEnums::NO_TAGPOS),
_CmntType(0),
_TagBrkIndex(-1),

_InclCCmntContent(true),
_InclCppCmntContent(true),
_InclScriptCmntContent(true),
_InclNestedCmntContent(true),
_InclHtmlCmntContent(true)
{
  ::memset(_MemPtrAlloc, 0, MAXCMNTTAGSIZE * sizeof(Subscript));
  _MemPtrAlloc[PARENTENUM] = Parent_ != NULL;
}

/****************************************************************************/
TagCommentData::TagCommentData(TagStringData* Parent_,
                               const char* StartBrk_, const char* EndBrk_,
                               const char* StartBlock_, const char* EndBlock_,
                               const char* Indicator_, int CmntType_):
_Parent(Parent_),
_IntCmntContent(NULL),
  
_NextCmntBlock(NULL),
_IntCmntEndBrk(EndBrk_ ? new ChrString(EndBrk_):NULL),
_IntCmntStartBrk(StartBrk_ ? new ChrString(StartBrk_):NULL),
_CmntBlkEnd(EndBlock_ ? new ChrString(EndBlock_):NULL),
_CmntBlkStart(StartBlock_ ? new ChrString(StartBlock_):NULL),
_IntCmntIndicator(Indicator_ ? new ChrString(Indicator_):NULL),
_SearchResult(NULL),

_CommentTagString(NULL),
_CommentTagSearch(NULL),

_CommentFound(StartBlock_ || EndBlock_),
_NestedCmntTag(StartBrk_ || EndBrk_),
_IntCmntTag(false),
_InComment(false),
  
_IntCmntStatus(TagEnums::IGNORED),
_IntCmntAnchor(TagEnums::NO_TAGPOS),
_CmntAnchor(TagEnums::NO_TAGPOS),
_CmntType(CmntType_),
_TagBrkIndex(-1),

_InclCCmntContent(true),
_InclCppCmntContent(true),
_InclScriptCmntContent(true),
_InclNestedCmntContent(true),
_InclHtmlCmntContent(true)
{
  ::memset(_MemPtrAlloc, 0, MAXCMNTTAGSIZE * sizeof(Subscript));
  _MemPtrAlloc[PARENTENUM] = Parent_ != NULL;

  int BrkEnder_;
  TagCountData* CntData_ = TagCountData::Instance();

  if (_CmntBlkStart)
    _TagBrkIndex = CntData_->BrkStringToIndex(_CmntBlkStart->c_str(), BrkEnder_);
}

/****************************************************************************/
TagCommentData::TagCommentData(const TagCommentData& Obj_):
_PrevCmntBlock(NULL),
_IntCmntContent(NULL),
  
_NextCmntBlock(Obj_._NextCmntBlock ? new TagCommentData(*Obj_._NextCmntBlock):NULL),
_IntCmntEndBrk(Obj_._IntCmntEndBrk ? new ChrString(*Obj_._IntCmntEndBrk):NULL),
_IntCmntStartBrk(Obj_._IntCmntStartBrk ? new ChrString(*Obj_._IntCmntStartBrk):NULL),
_CmntBlkEnd(Obj_._CmntBlkEnd ? new ChrString(*Obj_._CmntBlkEnd):NULL),
_CmntBlkStart(Obj_._CmntBlkStart ? new ChrString(*Obj_._CmntBlkStart):NULL),
_IntCmntIndicator(Obj_._IntCmntIndicator ? new ChrString(*Obj_._IntCmntIndicator):NULL),
_SearchResult(Obj_._SearchResult),

_CommentTagString(Obj_._CommentTagString ? (new ChrString(*Obj_._CommentTagString)):NULL),
_CommentTagSearch((Obj_._CommentFound && Obj_._CommentTagSearch) ?
                          (new TagSearchResult(*Obj_._CommentTagSearch)):
                          NULL),

_CommentFound(Obj_._CommentFound),
_NestedCmntTag(Obj_._NestedCmntTag),
_IntCmntTag(Obj_._IntCmntTag),
_InComment(Obj_._InComment),

_IntCmntStatus(Obj_._IntCmntStatus),
_IntCmntAnchor(Obj_._IntCmntAnchor),
_CmntAnchor(Obj_._CmntAnchor),
_CmntType(Obj_._CmntType),
_TagBrkIndex(Obj_._TagBrkIndex),

_InclCCmntContent(Obj_._InclCCmntContent),
_InclCppCmntContent(Obj_._InclCppCmntContent),
_InclScriptCmntContent(Obj_._InclScriptCmntContent),
_InclNestedCmntContent(Obj_._InclNestedCmntContent),
_InclHtmlCmntContent(Obj_._InclHtmlCmntContent)
{
  ::memset(_MemPtrAlloc, 0, MAXCMNTTAGSIZE * sizeof(Subscript));
  ::memmove(_MemPtrAlloc, Obj_._MemPtrAlloc, MAXCMNTTAGSIZE * sizeof(Subscript));

  if ((!_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::IGNORED) ||
      (_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::ACTIVE))
  {
    _CmntContent = Obj_._CmntContent ? (new ChrString(*Obj_._CmntContent)):NULL;
    _MemPtrAlloc[CMNTCNTENUM] = _CmntContent != NULL;
  }
  else if ((_IntCmntTag && !_InComment && _IntCmntStatus == TagEnums::ACTIVE) ||
           (!_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::INACTIVE))
  {
    _IntCmntContent = Obj_._IntCmntContent ? (new ChrString(*Obj_._IntCmntContent)):NULL;
    _MemPtrAlloc[INTCMNTCNTENUM] = _IntCmntContent != NULL;
  }

  if (_CmntAnchor == TagEnums::OPENING && _CmntBlkStart && Obj_._Parent)
  {
    _Parent = Obj_._Parent;
    _MemPtrAlloc[PARENTENUM] = 1;
  }
  else if ((_CmntAnchor != TagEnums::OPENING || !_CmntBlkStart) && Obj_._PrevCmntBlock)
  {
    _PrevCmntBlock = Obj_._PrevCmntBlock;
    _MemPtrAlloc[PREVCMNTBLKENUM] = 1;
  }
}

/****************************************************************************/
TagCommentData::~TagCommentData()
{
  delete _NextCmntBlock;
  delete _IntCmntEndBrk;
  delete _IntCmntStartBrk;
  delete _CmntBlkEnd;
  delete _CmntBlkStart;
  delete _IntCmntIndicator;

  _NextCmntBlock = NULL;
  _IntCmntEndBrk = NULL;
  _IntCmntStartBrk = NULL;
  _CmntBlkEnd = NULL;
  _CmntBlkStart = NULL;
  _IntCmntIndicator = NULL;

  if (_CommentFound)
  {
    if (_CommentTagSearch)
      delete _CommentTagSearch;

    if (_CommentTagString)
      delete _CommentTagString;

    _SearchResult = NULL;
    _CommentTagSearch = NULL;
    _CommentTagString = NULL;
    _CommentFound = false;
  }

  if (_MemPtrAlloc[CMNTCNTENUM])
  {
    if (_CmntContent)
      delete _CmntContent;

    _CmntContent = NULL;
    _MemPtrAlloc[CMNTCNTENUM] = 0;
  }
  else if (_MemPtrAlloc[INTCMNTCNTENUM])
  {
    if (_IntCmntContent)
      delete _IntCmntContent;

    _IntCmntContent = NULL;
    _MemPtrAlloc[INTCMNTCNTENUM] = 0;
  }

  if (_MemPtrAlloc[PARENTENUM])
  {
    _Parent = NULL;
    _MemPtrAlloc[PARENTENUM] = 0;
  }
  else if (_MemPtrAlloc[PREVCMNTBLKENUM])
  {
    _PrevCmntBlock = NULL;
    _MemPtrAlloc[PREVCMNTBLKENUM] = 0;
  }
}

/****************************************************************************/
TagCommentData* TagCommentData::Make(TagStringData* Parent_)
{
  return (new TagCommentData(Parent_));
}

/****************************************************************************/
TagCommentData* TagCommentData::Make(TagStringData* Parent_,
                                     const char* StartBrk_, const char* EndBrk_,
                                     const char* StartBlock_, const char* EndBlock_,
                                     const char* Indicator_, int CmntType_)
{
  return (new TagCommentData(Parent_, StartBrk_, EndBrk_,
                             StartBlock_, EndBlock_,
                             Indicator_, CmntType_));
}

/****************************************************************************/
TagCommentData* TagCommentData::Make(const TagCommentData& Obj_)
{
  return (new TagCommentData(Obj_));
}

/****************************************************************************/
void TagCommentData::SetShowCommentDataDump(bool Flag_)
{
  _ShowCommentDataDump = Flag_;
}

/****************************************************************************/
bool TagCommentData::ShowCommentDataDump()
{
  return _ShowCommentDataDump;
}

/****************************************************************************/
bool TagCommentData::TagContentsIncluded(int Index_, bool Nested_) const
{
  return
  (
    (Index_ == TagEnums::COMMENT_DEX && !Nested_) ? HtmlContentsIncluded():
    ((Nested_ &&
      Index_ == TagEnums::COMMENT_DEX) ||
     Index_ == TagEnums::NESTEDCOMMENT1_DEX ||
     Index_ == TagEnums::NESTEDCOMMENT2_DEX)      ? NestedContentsIncluded():
    (Index_ == TagEnums::SCRIPT_DEX)              ? ScriptContentsIncluded():
    (Index_ == TagEnums::CCOMMENTBLK_DEX ||
     Index_ == TagEnums::COPENCOMMENTBLK_DEX ||
     Index_ == TagEnums::CCLOSECOMMENTBLK_DEX)    ? CContentsIncluded():
    (Index_ == TagEnums::CPPCOMMENTLINEOPEN_DEX)  ? CppContentsIncluded():
                                                    false
  );
}

/****************************************************************************/
TagCommentData& TagCommentData::SetTextBufferPtr(const ChrString* Ptr_)
{
  if (_SearchResult)
    _SearchResult->SetTextBufferPtr(Ptr_);

  if (_CommentTagSearch)
    _CommentTagSearch->SetTextBufferPtr(Ptr_);
}

/****************************************************************************/
TagCommentData& TagCommentData::SetSearchResult(const TagSearchResult* SrchRes_)
{
  _SearchResult = SrchRes_;
  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetNextCmntBlock(TagCommentData* Ptr_)
{
  delete _NextCmntBlock;
  _NextCmntBlock = Ptr_;

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetPrevCmntBlock(TagCommentData* Ptr_)
{
  _PrevCmntBlock = Ptr_;
  _MemPtrAlloc[PREVCMNTBLKENUM] = Ptr_ != NULL;

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetInComment(bool Flag_)
{
  _InComment = Flag_;
  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetIntCmntTag(bool Flag_)
{
  _IntCmntTag = Flag_;
  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetCommentType(int Type_)
{
  if (Type_ >= 0)
    _CmntType = Type_;

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetCommentStatus(int Anchor_, int Type_)
{
  if (Anchor_ >= 0)
    _CmntAnchor = Anchor_;

  if (Type_ >= 0)
    _CmntType = Type_;

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetIntCommentStatus(int Anchor_, int Status_)
{
  if (Anchor_ >= 0)
    _IntCmntAnchor = Anchor_;

  if (Status_ >= 0)
    _IntCmntStatus = Status_;

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::EraseIntCmntBrk(bool StartBrk_, bool EndBrk_)
{
  if (StartBrk_)
  {
    delete _IntCmntStartBrk;
    _IntCmntStartBrk = NULL;
  }

  if (EndBrk_)
  {
    delete _IntCmntEndBrk;
    _IntCmntEndBrk = NULL;
  }

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetIntCmntBrk(const char* StartBrk_, const char* EndBrk_)
{
  if (StartBrk_)
  {
    delete _IntCmntStartBrk;
    _IntCmntStartBrk = new ChrString(StartBrk_);
  }

  if (EndBrk_)
  {
    delete _IntCmntEndBrk;
    _IntCmntEndBrk = new ChrString(EndBrk_);
  }

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::EraseCmntBlk(bool StartBlock_, bool EndBlock_, bool BrkIndex_)
{
  if (StartBlock_)
  {
    delete _CmntBlkStart;
    _CmntBlkStart = NULL;
  }

  if (EndBlock_)
  {
    delete _CmntBlkEnd;
    _CmntBlkEnd = NULL;
  }

  if (BrkIndex_)
    _TagBrkIndex = -1;

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetCmntBlk(const char* StartBlock_, const char* EndBlock_)
{
  if (StartBlock_)
  {
    delete _CmntBlkStart;
    _CmntBlkStart = new ChrString(StartBlock_);
  }

  if (EndBlock_)
  {
    delete _CmntBlkEnd;
    _CmntBlkEnd = new ChrString(EndBlock_);
  }

  int BrkEnder_;
  TagCountData* CntData_ = TagCountData::Instance();

  if (_CmntBlkStart)
    _TagBrkIndex = CntData_->BrkStringToIndex(_CmntBlkStart->c_str(), BrkEnder_);

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::EraseIntCmntIndicator(bool Indicator_)
{
  if (Indicator_)
  {
    delete _IntCmntIndicator;
    _IntCmntIndicator = NULL;
  }

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetIntCmntIndicator(const char* Indicator_)
{
  if (Indicator_)
  {
    delete _IntCmntIndicator;
    _IntCmntIndicator = new ChrString(Indicator_);
  }

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetIntCmntContent(ChrString* Ptr_, bool KeepPtr_)
{
  if (Ptr_)
  {
    if (KeepPtr_)
    {
      if (_MemPtrAlloc[CMNTCNTENUM])
      {
        if (_CmntContent)
          delete _CmntContent;

        _CmntContent = NULL;
        _MemPtrAlloc[CMNTCNTENUM] = 0;
      }
      else if (_MemPtrAlloc[INTCMNTCNTENUM])
      {
        if (_IntCmntContent)
          delete _IntCmntContent;

        _IntCmntContent = NULL;
        _MemPtrAlloc[INTCMNTCNTENUM] = 0;
      }

      _MemPtrAlloc[INTCMNTCNTENUM] = 1;
      _IntCmntContent = Ptr_;
      _IntCmntContent->SetCaseSensitive(FALSE);
    }
    else
    {
      if (_IntCmntContent)
        *_IntCmntContent = *Ptr_;
      else
        _IntCmntContent = new ChrString(*Ptr_);

      _MemPtrAlloc[INTCMNTCNTENUM] = 1;
      _IntCmntContent->SetCaseSensitive(FALSE);
    }
  }
  else if (!Ptr_)
  {
    if (_MemPtrAlloc[CMNTCNTENUM])
    {
      if (_CmntContent)
        delete _CmntContent;

      _CmntContent = NULL;
      _MemPtrAlloc[CMNTCNTENUM] = 0;
    }
    else if (_MemPtrAlloc[INTCMNTCNTENUM])
    {
      if (_IntCmntContent)
        delete _IntCmntContent;

      _IntCmntContent = NULL;
      _MemPtrAlloc[INTCMNTCNTENUM] = 0;
    }
  }

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::SetCmntContent(ChrString* Ptr_, bool KeepPtr_)
{
  if (Ptr_)
  {
    if (KeepPtr_)
    {
      if (_MemPtrAlloc[CMNTCNTENUM])
      {
        if (_CmntContent)
          delete _CmntContent;

        _CmntContent = NULL;
        _MemPtrAlloc[CMNTCNTENUM] = 0;
      }
      else if (_MemPtrAlloc[INTCMNTCNTENUM])
      {
        if (_IntCmntContent)
          delete _IntCmntContent;

        _IntCmntContent = NULL;
        _MemPtrAlloc[INTCMNTCNTENUM] = 0;
      }

      _MemPtrAlloc[CMNTCNTENUM] = 1;
      _CmntContent = Ptr_;
      _CmntContent->SetCaseSensitive(FALSE);
    }
    else
    {
      if (_CmntContent)
        *_CmntContent = *Ptr_;
      else
        _CmntContent = new ChrString(*Ptr_);

      _MemPtrAlloc[CMNTCNTENUM] = 1;
      _CmntContent->SetCaseSensitive(FALSE);
    }
  }
  else if (!Ptr_)
  {
    if (_MemPtrAlloc[CMNTCNTENUM])
    {
      if (_CmntContent)
        delete _CmntContent;

      _CmntContent = NULL;
      _MemPtrAlloc[CMNTCNTENUM] = 0;
    }
    else if (_MemPtrAlloc[INTCMNTCNTENUM])
    {
      if (_IntCmntContent)
        delete _IntCmntContent;

      _IntCmntContent = NULL;
      _MemPtrAlloc[INTCMNTCNTENUM] = 0;
    }
  }

  return *this;
}

/****************************************************************************/
TagCommentData& TagCommentData::operator = (const TagCommentData& Obj_)
{
  if (this != &Obj_)
  {
    delete _NextCmntBlock;
    delete _IntCmntEndBrk;
    delete _IntCmntStartBrk;
    delete _CmntBlkEnd;
    delete _CmntBlkStart;
    delete _IntCmntIndicator;

    _NextCmntBlock = Obj_._NextCmntBlock ? new TagCommentData(*Obj_._NextCmntBlock):NULL;
    _IntCmntEndBrk = Obj_._IntCmntEndBrk ? new ChrString(*Obj_._IntCmntEndBrk):NULL;
    _IntCmntStartBrk = Obj_._IntCmntStartBrk ? new ChrString(*Obj_._IntCmntStartBrk):NULL;
    _CmntBlkEnd = Obj_._CmntBlkEnd ? new ChrString(*Obj_._CmntBlkEnd):NULL;
    _CmntBlkStart = Obj_._CmntBlkStart ? new ChrString(*Obj_._CmntBlkStart):NULL;
    _IntCmntIndicator = Obj_._IntCmntIndicator ? new ChrString(*Obj_._IntCmntIndicator):NULL;

    _NestedCmntTag = Obj_._NestedCmntTag;
    _IntCmntTag = Obj_._IntCmntTag;
    _InComment = Obj_._InComment;
  
    _IntCmntStatus = Obj_._IntCmntStatus;
    _IntCmntAnchor = Obj_._IntCmntAnchor;
    _CmntAnchor = Obj_._CmntAnchor;
    _CmntType = Obj_._CmntType;

    int BrkEnder_;
    TagCountData* CntData_ = TagCountData::Instance();

    if (_CmntBlkStart)
      _TagBrkIndex = CntData_->BrkStringToIndex(_CmntBlkStart->c_str(), BrkEnder_);

    _InclCCmntContent = Obj_._InclCCmntContent;
    _InclCppCmntContent = Obj_._InclCppCmntContent;
    _InclScriptCmntContent = Obj_._InclScriptCmntContent;
    _InclNestedCmntContent = Obj_._InclNestedCmntContent;
    _InclHtmlCmntContent = Obj_._InclHtmlCmntContent;

    if (_CommentFound)
    {
      if (_CommentTagSearch)
        delete _CommentTagSearch;

      if (_CommentTagString)
        delete _CommentTagString;

      _SearchResult = NULL;
      _CommentTagString = NULL;
      _CommentTagSearch = NULL;
      _CommentFound = false;
    }

    _CommentFound = Obj_._CommentFound;
    _SearchResult = Obj_._SearchResult;

    _CommentTagString = Obj_._CommentTagString ? (new ChrString(*Obj_._CommentTagString)):NULL;
    _CommentTagSearch = (Obj_._CommentFound && Obj_._CommentTagSearch) ?
                                (new TagSearchResult(*Obj_._CommentTagSearch)):
                                Obj_._CommentTagSearch;

    if (_MemPtrAlloc[CMNTCNTENUM])
    {
      if (_CmntContent)
        delete _CmntContent;

      _CmntContent = NULL;
      _MemPtrAlloc[CMNTCNTENUM] = 0;
    }
    else if (_MemPtrAlloc[INTCMNTCNTENUM])
    {
      if (_IntCmntContent)
        delete _IntCmntContent;

      _IntCmntContent = NULL;
      _MemPtrAlloc[INTCMNTCNTENUM] = 0;
    }

    if (_MemPtrAlloc[PARENTENUM])
    {
      _Parent = NULL;
      _MemPtrAlloc[PARENTENUM] = 0;
    }
    else if (_MemPtrAlloc[PREVCMNTBLKENUM])
    {
      _PrevCmntBlock = NULL;
      _MemPtrAlloc[PREVCMNTBLKENUM] = 0;
    }

    if ((!_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::IGNORED) ||
        (_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::ACTIVE))
    {
      _CmntContent = Obj_._CmntContent ? (new ChrString(*Obj_._CmntContent)):NULL;
      _MemPtrAlloc[CMNTCNTENUM] = _CmntContent != NULL;
    }
    else if ((_IntCmntTag && !_InComment && _IntCmntStatus == TagEnums::ACTIVE) ||
             (!_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::INACTIVE))
    {
      _IntCmntContent = Obj_._IntCmntContent ? (new ChrString(*Obj_._IntCmntContent)):NULL;
      _MemPtrAlloc[INTCMNTCNTENUM] = _IntCmntContent != NULL;
    }

    if (_CmntAnchor == TagEnums::OPENING && _CmntBlkStart && Obj_._Parent)
    {
      _Parent = Obj_._Parent;
      _MemPtrAlloc[PARENTENUM] = 1;
    }
    else if ((_CmntAnchor != TagEnums::OPENING || !_CmntBlkStart) && Obj_._PrevCmntBlock)
    {
      _PrevCmntBlock = Obj_._PrevCmntBlock;
      _MemPtrAlloc[PREVCMNTBLKENUM] = 1;
    }

    ::memmove(&_MemPtrAlloc[NESTEDCMNTEND1],
              &Obj_._MemPtrAlloc[NESTEDCMNTEND1],
              (MAXCMNTTAGSIZE-NESTEDCMNTEND1) * sizeof(Subscript));
  }

  return *this;
}

/****************************************************************************/
long TagCommentData::ParseComment()
{
  if (_SearchResult && _SearchResult->StartBrkFound() && _SearchResult->EndBrkFound() &&
      CommentFound(_SearchResult->c_str(), _SearchResult->BracketIndex()))
  {
    const char* CmntStr_ = NULL;
    ChrString* SubStr_ = NULL;
    ChrString TagStr_ = _CommentTagSearch->GiveTagString();
    long Index_ = _CommentTagSearch->BracketLength() - 1;
    long TagOffset_ = _CommentTagSearch->BracketIndex();
    long len = 0;

    _NestedCmntTag = false;
    _CommentFound = true;
    TagSearchResult* NestedSearchResult_ = new TagSearchResult;
    NestedSearchResult_->SetTextBufferPtr(&TagStr_);
    CmntStr_ = NestedCommentClose_Str(NULL);

    if (CmntStr_)
    {
      NestedSearchResult_->SetSearchType(0);
      NestedSearchResult_->findnext(Index_, ChrString(CmntStr_));
    }

    if (CmntStr_ && NestedSearchResult_->TargetFound())
    {
      _MemPtrAlloc[NESTEDCMNTEND1] = TagOffset_ +
                                     NestedSearchResult_->SearchResultIndex();
      _MemPtrAlloc[NESTEDCMNTEND1+1] = NestedSearchResult_->SearchResultLength();

      if (_IntCmntIndicator)
      {
        Subscript len = _CommentTagSearch->BracketIndex() +
                        _CommentTagSearch->BracketLength();
        _CommentTagSearch->SetElementData(len, _MemPtrAlloc[NESTEDCMNTEND1] - len + 1, false);
      }

      Index_ = _MemPtrAlloc[NESTEDCMNTEND1] +
               _MemPtrAlloc[NESTEDCMNTEND1+1] - TagOffset_ - 1;
      CmntStr_ = NestedCommentOpen_Str(NULL);

      if (CmntStr_)
      {
        NestedSearchResult_->SetSearchType(0);
        NestedSearchResult_->findnext(Index_, ChrString(CmntStr_));
      }

      _NestedCmntTag = CmntStr_ && NestedSearchResult_->TargetFound();

      if (_NestedCmntTag)
      {
        _MemPtrAlloc[NESTEDCMNTSTART2] = TagOffset_ +
                                         NestedSearchResult_->SearchResultIndex();
        _MemPtrAlloc[NESTEDCMNTSTART2+1] = NestedSearchResult_->SearchResultLength();

        if (_IntCmntIndicator &&
            TagContentsIncluded(_TagBrkIndex, _NestedCmntTag))
        {
          Index_ = _MemPtrAlloc[NESTEDCMNTEND1] +
                   _MemPtrAlloc[NESTEDCMNTEND1+1];
          len = _MemPtrAlloc[NESTEDCMNTSTART2] - Index_;

          // include comment contents (internal nested comments only)
          // _IntCmntTag && !_InComment && _IntCmntStatus == TagEnums::ACTIVE
          SubStr_ = new ChrString(_CommentTagSearch->c_str());
          SetIntCmntContent(new ChrString(SubStr_->Mid(Index_, len)), true);
          SetIntCmntTag(true);
          SetInComment(false);
          SetIntCommentStatus(TagEnums::NO_TAGPOS, TagEnums::ACTIVE);
          delete SubStr_;
        }
      }

      delete NestedSearchResult_;
    }

    if (!_NestedCmntTag)
    {
      _MemPtrAlloc[NESTEDCMNTEND1] = 0;
      _MemPtrAlloc[NESTEDCMNTEND1+1] = 0;
      _MemPtrAlloc[NESTEDCMNTSTART2] = 0;
      _MemPtrAlloc[NESTEDCMNTSTART2+1] = 0;

      SetCommentStatus(TagEnums::OPENING, BrkIndexFileType());
      return EncounterOpen(false, false);
    }
    else
    {
      SetCommentStatus(TagEnums::OPENING, BrkIndexFileType());
      return EncounterOpen(false, true);
    }
  }

  return 0;
}

/****************************************************************************/
bool TagCommentData::CommentFound(const char* Str_, long Pos_)
{
  char ch[2];
  long len = 0;
  long Index_ = 0;
  bool StartDone_ = false;
  bool EndDone_ = false;
  bool CommFound_ = _CommentFound;
  const char* CmntStr_ = NULL;
  ChrString* SubStr_ = Str_ ? new ChrString(Str_):NULL;
  ch[0] = ch[1] = 0;

  int BrkEnder_;
  TagCountData* CntData_ = TagCountData::Instance();

  if (SubStr_ && CntData_ && Pos_ < SubStr_->strlen())
  {
    _TagBrkIndex = CntData_->BrkStringToIndex((*SubStr_)(Pos_), BrkEnder_, true);
    len = ::SafeStrLen(CommentOpen_Str());
  }

  if (SubStr_ && _SearchResult && len &&
      strncmp((*SubStr_)(Pos_), CommentOpen_Str(), len) == 0)
  {
    ChrString BrkStr_ = _SearchResult->GiveTargetBrkString();
    ChrString StartBrk_;
    ChrString EndBrk_;
    CmntStr_ = CommentOpen_Str();

    if (CmntStr_ && strcmp(BrkStr_.c_str(), CmntStr_) == 0)
    {
      if (_CmntBlkStart)
      {
        delete _CmntBlkStart;
        _CmntBlkStart = NULL;
      }

      StartBrk_ = BrkStr_;
      _CmntAnchor = TagEnums::OPENING;
      StartDone_ = true;

      if ((*SubStr_)[Pos_+len] == *NestedCommentOpen_Str(ch))
        SetIntCmntIndicator(NestedCommentOpen_Str(ch));
      else
        EraseIntCmntIndicator(true);
    }

    BrkStr_ = _SearchResult->GiveMatchingBrkString();
    CmntStr_ = CommentClose_Str();

    if (CmntStr_ && strcmp(BrkStr_.c_str(), CmntStr_) == 0)
    {
      if (_CmntBlkEnd)
      {
        delete _CmntBlkEnd;
        _CmntBlkEnd = NULL;
      }

      if (CommFound_)
      {
        if (_CommentTagSearch)
          delete _CommentTagSearch;

        if (_CommentTagString)
          delete _CommentTagString;

        _CommentTagSearch = NULL;
        _CommentTagString = NULL;
      }

      EndBrk_ = BrkStr_;
      EraseCmntBlk(true, true, false);
      SetCmntBlk(StartBrk_.c_str(), EndBrk_.c_str());
      _CommentTagString = new ChrString(_SearchResult->GiveTagString());
      _CommentTagSearch = new TagSearchResult(*_SearchResult);
      _CommentFound = true;
      EndDone_ = true;

      if (!_CmntBlkStart || !StartDone_)
        _CmntAnchor = TagEnums::CLOSING;

      if (!_IntCmntIndicator &&
          TagContentsIncluded(_TagBrkIndex, false))
      {
        Index_ = _CommentTagSearch->BracketIndex() +
                 _CommentTagSearch->BracketLength();
        len = _CommentTagSearch->MatchingIndex() - Index_;

        // include comment contents (no nested comments)
        // !_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::IGNORED
        delete SubStr_;
        SubStr_ = new ChrString(_CommentTagSearch->c_str());
        SetCmntContent(new ChrString(SubStr_->Mid(Index_, len)), true);
        SetIntCmntTag(false);
        SetInComment(true);
        SetIntCommentStatus(TagEnums::NO_TAGPOS, TagEnums::IGNORED);
      }
    }
  }

  delete SubStr_;
  return (StartDone_ && EndDone_);
}

/****************************************************************************/
void TagCommentData::MakeCommentTagData(TagCommentData* Rootp_)
{
  bool CommFound_ = _CommentFound;

  if (_SearchResult && Rootp_ && Rootp_->CommentFound())
  {
    if (CommFound_)
    {
      if (_CommentTagSearch)
        delete _CommentTagSearch;

      if (_CommentTagString)
        delete _CommentTagString;

      _CommentTagSearch = NULL;
      _CommentTagString = NULL;
    }

    _CommentTagString = new ChrString(_SearchResult->GiveTagString());
    _CommentTagSearch = new TagSearchResult(*_SearchResult);

    _CommentFound = true;
    _NestedCmntTag = Rootp_->NestedCmntTag();

    if (_MemPtrAlloc[CMNTCNTENUM])
    {
      if (_CmntContent)
        delete _CmntContent;

      _CmntContent = NULL;
      _MemPtrAlloc[CMNTCNTENUM] = 0;
    }
    else if (_MemPtrAlloc[INTCMNTCNTENUM])
    {
      if (_IntCmntContent)
        delete _IntCmntContent;

      _IntCmntContent = NULL;
      _MemPtrAlloc[INTCMNTCNTENUM] = 0;
    }

    if (_MemPtrAlloc[PARENTENUM])
    {
      _Parent = NULL;
      _MemPtrAlloc[PARENTENUM] = 0;
    }
    else if (_MemPtrAlloc[PREVCMNTBLKENUM])
    {
      _PrevCmntBlock = NULL;
      _MemPtrAlloc[PREVCMNTBLKENUM] = 0;
    }

    if ((!_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::IGNORED) ||
        (_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::ACTIVE))
    {
      _CmntContent = Rootp_->_CmntContent ? (new ChrString(*Rootp_->_CmntContent)):NULL;
      _MemPtrAlloc[CMNTCNTENUM] = _CmntContent != NULL;
    }
    else if ((_IntCmntTag && !_InComment && _IntCmntStatus == TagEnums::ACTIVE) ||
             (!_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::INACTIVE))
    {
      _IntCmntContent = Rootp_->_IntCmntContent ? (new ChrString(*Rootp_->_IntCmntContent)):NULL;
      _MemPtrAlloc[INTCMNTCNTENUM] = _IntCmntContent != NULL;
    }

    if (_CmntAnchor == TagEnums::OPENING && _CmntBlkStart && Rootp_->_Parent)
    {
      _Parent = Rootp_->_Parent;
      _MemPtrAlloc[PARENTENUM] = 1;
    }
    else if ((_CmntAnchor != TagEnums::OPENING || !_CmntBlkStart) && Rootp_->_PrevCmntBlock)
    {
      _PrevCmntBlock = Rootp_->_PrevCmntBlock;
      _MemPtrAlloc[PREVCMNTBLKENUM] = 1;
    }

    ::memmove(&_MemPtrAlloc[NESTEDCMNTEND1],
              &Rootp_->_MemPtrAlloc[NESTEDCMNTEND1],
              (MAXCMNTTAGSIZE-NESTEDCMNTEND1) * sizeof(Subscript));
  }
}

/****************************************************************************/
void TagCommentData::SetTagBrkIndex(long Index_, const char* StartBlock_)
{
  int BrkEnder_;
  TagCountData* CntData_ = TagCountData::Instance();

  if (StartBlock_)
    _TagBrkIndex = CntData_->BrkStringToIndex(StartBlock_, BrkEnder_);
  else if (Index_ >= 0)
    _TagBrkIndex = Index_;
}

/****************************************************************************/
const char* TagCommentData::NestedCommentClose_Str(char* chp) const
{
  TagBracketList* BrkList_ = TagBracketList::Instance();
  long Index_ = _TagBrkIndex;

  if (Index_ == TagEnums::COMMENT_DEX ||
      Index_ == TagEnums::COMMENT_DEX+1)
    Index_ = TagEnums::NESTEDCOMMENT1_DEX;
  else if (Index_ == TagEnums::CCOMMENTBLK_DEX ||
           Index_ == TagEnums::CCOMMENTBLK_DEX+1)
    Index_ = TagEnums::COPENCOMMENTBLK_DEX;

  if (Index_ >= 0 && !(Index_ % 2))
    ++Index_;

  const char* ret =
  (
    (BrkList_ && Index_ > 0) ?
        BrkList_->GiveNestedCommentStr(Index_):
    (Index_ == TagEnums::NESTEDCOMMENT1_DEX) ?
        INTNESTEDCOMMENTEND_STR:
    (Index_ == TagEnums::COPENCOMMENTBLK_DEX) ?
        INTCCOMMENTBLOCKEND_STR:
        INTNESTEDCOMMENTEND_STR
  );

  if (chp)
  {
    chp[0] = chp[1] = 0;

    if (BrkList_ && Index_ > 0)
      BrkList_->GiveNestedCommentChar(Index_, chp);
    else if (Index_ == TagEnums::NESTEDCOMMENT1_DEX)
      *chp = TagEnums::NESTEDCOMMENT_CLOSE_CH;
    else
      *chp = TagEnums::CCOMMENTBLOCK_CLOSE_CH;

    chp[1] = 0;
    return chp;
  }

  return ret;
}

/****************************************************************************/
const char* TagCommentData::NestedCommentOpen_Str(char* chp) const
{
  TagBracketList* BrkList_ = TagBracketList::Instance();
  long Index_ = _TagBrkIndex;

  if (Index_ == TagEnums::COMMENT_DEX ||
      Index_ == TagEnums::COMMENT_DEX+1)
    Index_ = TagEnums::NESTEDCOMMENT2_DEX;
  else if (Index_ == TagEnums::CCOMMENTBLK_DEX ||
           Index_ == TagEnums::CCOMMENTBLK_DEX+1)
    Index_ = TagEnums::CCLOSECOMMENTBLK_DEX;

  if (Index_ > 0 && Index_ % 2)
    --Index_;

  const char* ret =
  (
    (BrkList_ && Index_ >= 0) ?
        BrkList_->GiveNestedCommentStr(Index_):
    (Index_ == TagEnums::NESTEDCOMMENT2_DEX) ?
        INTNESTEDCOMMENTSTART_STR:
    (Index_ == TagEnums::CCLOSECOMMENTBLK_DEX) ?
        INTCCOMMENTBLOCKSTART_STR:
        INTNESTEDCOMMENTSTART_STR
  );

  if (chp)
  {
    chp[0] = chp[1] = 0;

    if (BrkList_ && Index_ >= 0)
      BrkList_->GiveNestedCommentChar(Index_, chp);
    else if (Index_ == TagEnums::NESTEDCOMMENT2_DEX)
      *chp = TagEnums::NESTEDCOMMENT_OPEN_CH;
    else
      *chp = TagEnums::CCOMMENTBLOCK_OPEN_CH;

    chp[1] = 0;
    return chp;
  }

  return ret;
}

/****************************************************************************/
const char* TagCommentData::CommentOpen_Str() const
{
  TagBracketList* BrkList_ = TagBracketList::Instance();
  long Index_ = _TagBrkIndex;

  return
  (
    (BrkList_ && _TagBrkIndex >= 0) ?
        BrkList_->GiveTagBrk(_TagBrkIndex):
    (Index_ == TagEnums::COMMENT_DEX ||
     Index_ == TagEnums::COMMENT_DEX+1) ?
        COMMENTOPEN_STR:
    (Index_ == TagEnums::CCOMMENTBLK_DEX ||
     Index_ == TagEnums::CCOMMENTBLK_DEX+1) ?
        CCOMMENTBLKOPEN_STR:
    (Index_ == TagEnums::CPPCOMMENTLINEOPEN_DEX ||
     Index_ == TagEnums::CPPCOMMENTLINEOPEN_DEX+1) ?
        CPPCOMMENTLINEOPEN_STR:
    (Index_ == TagEnums::SCRIPT_DEX ||
     Index_ == TagEnums::SCRIPT_DEX+1) ?
        SCRIPTOPEN_STR:
        COMMENTOPEN_STR
  );
}

/****************************************************************************/
const char* TagCommentData::CommentClose_Str() const
{
  TagBracketList* BrkList_ = TagBracketList::Instance();
  long Index_ = _TagBrkIndex;

  return
  (
    (BrkList_ && _TagBrkIndex >= 0) ?
        BrkList_->GiveTagBrk(_TagBrkIndex+1):
    (Index_ == TagEnums::COMMENT_DEX ||
     Index_ == TagEnums::COMMENT_DEX+1) ?
        COMMENTCLOSE_STR:
    (Index_ == TagEnums::CCOMMENTBLK_DEX ||
     Index_ == TagEnums::CCOMMENTBLK_DEX+1) ?
        CCOMMENTBLKCLOSE_STR:
    (Index_ == TagEnums::CPPCOMMENTLINEOPEN_DEX ||
     Index_ == TagEnums::CPPCOMMENTLINEOPEN_DEX+1) ?
        CPPCOMMENTLINECLOSE_STR:
    (Index_ == TagEnums::SCRIPT_DEX ||
     Index_ == TagEnums::SCRIPT_DEX+1) ?
        SCRIPTCLOSE_STR:
        COMMENTCLOSE_STR
  );
}

/****************************************************************************/
long TagCommentData::BrkIndexFileType() const
{
  return
  (
    (_TagBrkIndex < 0) ? TagEnums::HTMLSTYLE:
    (_TagBrkIndex == TagEnums::COMMENT_DEX ||
     _TagBrkIndex == TagEnums::COMMENT_DEX+1) ?
        TagEnums::HTMLSTYLE:
    (_TagBrkIndex == TagEnums::CCOMMENTBLK_DEX ||
     _TagBrkIndex == TagEnums::CCOMMENTBLK_DEX+1) ?
        TagEnums::CSTYLE:
    (_TagBrkIndex == TagEnums::CPPCOMMENTLINEOPEN_DEX ||
     _TagBrkIndex == TagEnums::CPPCOMMENTLINEOPEN_DEX+1) ?
        TagEnums::CPPSTYLE:
        TagEnums::HTMLSTYLE
  );
}

/****************************************************************************/
long TagCommentData::EncounterOpen(bool IntraCmnt_, bool BlockStart_)
{
  // returns number of nodes in comment linked list
  if (!_CommentTagSearch)
    return 0;

  char ch[2];
  ch[0] = ch[1] = 0;

  if (!IntraCmnt_ && !BlockStart_)
  {
    if (_CmntBlkStart && _CmntBlkEnd && _CmntAnchor == TagEnums::OPENING)
    {
      // <!--   : EncounterOpen(IntraCmnt=false, BlockStart=false)
      //              CmntBlkEnd=0, CmntBlkStart="<!--",
      //              IntCmntTag=false, InComment=false,
      //              IntCmntIndicator=0,
      //              CmntAnchor=OPENING,
      //              IntCmntStatus=IGNORED

      SetInComment(false);
      SetIntCmntTag(false);
      EraseIntCmntBrk(true, true);
      EraseIntCmntIndicator(true);
      SetCommentStatus(TagEnums::OPENING, BrkIndexFileType());
      SetIntCommentStatus(-1, TagEnums::IGNORED);

      SetNextCmntBlock(TagCommentData::Make(_Parent, NULL, NULL,
                                            NULL, _CmntBlkEnd->c_str(),
                                            NULL, TagEnums::HTMLSTYLE));

      TagSearchResult* NestedSearchResult_ =
          new TagSearchResult(*_CommentTagSearch);
      NestedSearchResult_->AssignSearchResult(_CommentTagSearch->MatchingIndex(),
                                              _CommentTagSearch->MatchingLength());

      _NextCmntBlock->SetTagBrkIndex(_TagBrkIndex, NULL);
      _NextCmntBlock->SetSearchResult(NestedSearchResult_);
      _NextCmntBlock->MakeCommentTagData(this);
      _NextCmntBlock->SetPrevCmntBlock(this);
      _NextCmntBlock->SetCommentStatus(TagEnums::CLOSING, BrkIndexFileType());

      return _NextCmntBlock->EncounterClose(false, false);
    }
    else if (!_CmntBlkStart && _CmntBlkEnd && _CmntAnchor == TagEnums::CLOSING)
    {
      SetInComment(true);
      SetIntCmntTag(false);
      EraseIntCmntBrk(true, true);
      EraseIntCmntIndicator(true);
      SetCommentStatus(TagEnums::CLOSING, BrkIndexFileType());
      SetIntCommentStatus(-1, TagEnums::IGNORED);
      return 2;
    }
  }
  else if (!IntraCmnt_ && BlockStart_)
  {
    if (_CmntBlkStart && _CmntBlkEnd && _IntCmntIndicator &&
        (*_IntCmntIndicator)[0] == *NestedCommentOpen_Str(ch) &&
        _CmntAnchor == TagEnums::OPENING)
    {
      // <!--[  : EncounterOpen(IntraCmnt=false, BlockStart=true),
      //              CmntBlkEnd=0, CmntBlkStart="<!--",
      //              IntCmntEndBrk=0, IntCmntStartBrk=0,
      //              IntCmntTag=false, InComment=false,
      //              IntCmntIndicator="[",
      //              CmntAnchor=OPENING,
      //              IntCmntStatus=INACTIVE

      SetInComment(false);
      SetIntCmntTag(false);
      EraseIntCmntBrk(true, true);
      SetCommentStatus(TagEnums::OPENING, BrkIndexFileType());
      SetIntCommentStatus(-1, TagEnums::INACTIVE);

      SetNextCmntBlock(TagCommentData::Make(_Parent, NULL, NestedCommentClose_Str(NULL),
                                            NULL, NULL, NULL, TagEnums::HTMLSTYLE));

      Subscript len = _MemPtrAlloc[NESTEDCMNTSTART2] -
                        (_MemPtrAlloc[NESTEDCMNTEND1] +
                         _MemPtrAlloc[NESTEDCMNTEND1+1]);
      TagSearchResult* NestedSearchResult_ =
          new TagSearchResult(*_CommentTagSearch);
      NestedSearchResult_->AssignSearchResult(_MemPtrAlloc[NESTEDCMNTEND1],
                                              _MemPtrAlloc[NESTEDCMNTEND1+1],
                                              _MemPtrAlloc[NESTEDCMNTSTART2],
                                              _MemPtrAlloc[NESTEDCMNTSTART2+1],
                                              (_MemPtrAlloc[NESTEDCMNTEND1] +
                                               _MemPtrAlloc[NESTEDCMNTEND1+1]),
                                              len);

      ChrString EndIntBrk_ = NestedSearchResult_->c_str();
      EndIntBrk_.Mid(_MemPtrAlloc[NESTEDCMNTEND1],
                     _MemPtrAlloc[NESTEDCMNTEND1+1]);

      _NextCmntBlock->SetTagBrkIndex(_TagBrkIndex, NULL);
      _NextCmntBlock->SetSearchResult(NestedSearchResult_);
      _NextCmntBlock->MakeCommentTagData(this);
      _NextCmntBlock->SetPrevCmntBlock(this);
      _NextCmntBlock->SetCommentStatus(-1, BrkIndexFileType());
      _NextCmntBlock->SetIntCmntBrk(NULL, EndIntBrk_.c_str());

      return _NextCmntBlock->EncounterClose(true, false);
    }
    else if (!_CmntBlkStart && _CmntBlkEnd && _IntCmntIndicator &&
             (*_IntCmntIndicator)[0] == *NestedCommentClose_Str(ch) &&
             _CmntAnchor == TagEnums::CLOSING)
    {
      SetInComment(true);
      SetIntCmntTag(false);
      EraseIntCmntBrk(true, true);
      SetCommentStatus(TagEnums::CLOSING, BrkIndexFileType());
      SetIntCommentStatus(-1, TagEnums::INACTIVE);
      return 4;
    }
  }
  else if (IntraCmnt_ && !BlockStart_)
  {
    if (_IntCmntStartBrk && !_IntCmntEndBrk)
    {
      // <![    : EncounterOpen(IntraCmnt=true, BlockStart=false)
      //              IntCmntEndBrk=0, IntCmntStartBrk="<![",
      //              IntCmntTag=true, InComment=true,
      //              IntCmntIndicator=0,
      //              IntCmntAnchor=OPENING,
      //              IntCmntStatus=ACTIVE

      EraseCmntBlk(true, true, false);
      EraseIntCmntIndicator(true);
      SetInComment(true);
      SetIntCmntTag(true);
      SetIntCommentStatus(TagEnums::OPENING, TagEnums::ACTIVE);

      SetNextCmntBlock(TagCommentData::Make(_Parent, NULL, NULL,
                                            NULL, CommentClose_Str(),
                                            NestedCommentClose_Str(ch),
                                            TagEnums::HTMLSTYLE));

      TagSearchResult* NestedSearchResult_ =
          new TagSearchResult(*_CommentTagSearch);
      NestedSearchResult_->AssignSearchResult(RootMatchingIndex(),
                                              RootMatchingLength());

      _NextCmntBlock->SetTagBrkIndex(_TagBrkIndex, NULL);
      _NextCmntBlock->SetSearchResult(NestedSearchResult_);
      _NextCmntBlock->MakeCommentTagData(this);
      _NextCmntBlock->SetPrevCmntBlock(this);
      _NextCmntBlock->SetCommentStatus(TagEnums::CLOSING, BrkIndexFileType());

      return _NextCmntBlock->EncounterClose(false, true);
    }
    else if (!_IntCmntStartBrk && _IntCmntEndBrk)
    {
      EraseCmntBlk(true, true, false);
      EraseIntCmntIndicator(true);
      SetInComment(false);
      SetIntCmntTag(true);
      SetIntCommentStatus(TagEnums::CLOSING, TagEnums::ACTIVE);

      SetNextCmntBlock(TagCommentData::Make(_Parent, NestedCommentOpen_Str(NULL),
                                            NULL, NULL, NULL, NULL, TagEnums::HTMLSTYLE));

      Subscript len = RootMatchingIndex() -
                        (_MemPtrAlloc[NESTEDCMNTSTART2] +
                         _MemPtrAlloc[NESTEDCMNTSTART2+1] - 1);
      TagSearchResult* NestedSearchResult_ =
          new TagSearchResult(*_CommentTagSearch);
      NestedSearchResult_->AssignSearchResult(_MemPtrAlloc[NESTEDCMNTSTART2],
                                              _MemPtrAlloc[NESTEDCMNTSTART2+1],
                                              RootMatchingIndex(),
                                              RootMatchingLength(),
                                              (_MemPtrAlloc[NESTEDCMNTSTART2] +
                                               _MemPtrAlloc[NESTEDCMNTSTART2+1] - 1),
                                              len);

      ChrString StartIntBrk_ = NestedSearchResult_->c_str();
      StartIntBrk_.Mid(_MemPtrAlloc[NESTEDCMNTSTART2],
                       _MemPtrAlloc[NESTEDCMNTSTART2+1]);

      _NextCmntBlock->SetTagBrkIndex(_TagBrkIndex, NULL);
      _NextCmntBlock->SetSearchResult(NestedSearchResult_);
      _NextCmntBlock->MakeCommentTagData(this);
      _NextCmntBlock->SetPrevCmntBlock(this);
      _NextCmntBlock->SetCommentStatus(-1, BrkIndexFileType());
      _NextCmntBlock->SetIntCmntBrk(StartIntBrk_.c_str(), NULL);

      return _NextCmntBlock->EncounterOpen(true, false);
    }
  }
}

/****************************************************************************/
long TagCommentData::EncounterClose(bool IntraCmnt_, bool BlockEnd_)
{
  // returns number of nodes in comment linked list
  if (!_CommentTagSearch)
    return 0;

  char ch[2];
  ch[0] = ch[1] = 0;

  if (!IntraCmnt_ && !BlockEnd_)
  {
    if (!_CmntBlkStart && _CmntBlkEnd && _CmntAnchor == TagEnums::CLOSING)
    {
      // -->    : EncounterClose(IntraCmnt=false, BlockStart=false)
      //              CmntBlkEnd="-->", CmntBlkStart=0,
      //              IntCmntTag=false, InComment=true,
      //              IntCmntIndicator=0,
      //              CmntAnchor=CLOSING,
      //              IntCmntStatus=IGNORED

      SetInComment(true);
      SetIntCmntTag(false);
      EraseIntCmntBrk(true, true);
      EraseIntCmntIndicator(true);
      SetCommentStatus(TagEnums::CLOSING, BrkIndexFileType());
      SetIntCommentStatus(-1, TagEnums::IGNORED);
      return 2;
    }
    else if (_CmntBlkStart && _CmntBlkEnd && _CmntAnchor == TagEnums::OPENING)
    {
      SetInComment(false);
      SetIntCmntTag(false);
      EraseIntCmntBrk(true, true);
      EraseIntCmntIndicator(true);
      SetCommentStatus(TagEnums::OPENING, BrkIndexFileType());
      SetIntCommentStatus(-1, TagEnums::IGNORED);

      SetNextCmntBlock(TagCommentData::Make(_Parent, NULL, NULL,
                                            NULL, _CmntBlkEnd->c_str(),
                                            NULL, TagEnums::HTMLSTYLE));

      TagSearchResult* NestedSearchResult_ =
          new TagSearchResult(*_CommentTagSearch);
      NestedSearchResult_->AssignSearchResult(_CommentTagSearch->MatchingIndex(),
                                              _CommentTagSearch->MatchingLength());

      _NextCmntBlock->SetTagBrkIndex(_TagBrkIndex, NULL);
      _NextCmntBlock->SetSearchResult(NestedSearchResult_);
      _NextCmntBlock->MakeCommentTagData(this);
      _NextCmntBlock->SetPrevCmntBlock(this);
      _NextCmntBlock->SetCommentStatus(TagEnums::CLOSING, BrkIndexFileType());

      return _NextCmntBlock->EncounterClose(false, false);
    }
  }
  else if (!IntraCmnt_ && BlockEnd_)
  {
    if (!_CmntBlkStart && _CmntBlkEnd && _IntCmntIndicator &&
        (*_IntCmntIndicator)[0] == *NestedCommentClose_Str(ch) &&
        _CmntAnchor == TagEnums::CLOSING)
    {
      // ]-->   : EncounterClose(IntraCmnt=false, BlockStart=true)
      //              CmntBlkEnd="-->", CmntBlkStart=0,
      //              IntCmntEndBrk=0, IntCmntStartBrk=0,
      //              IntCmntTag=false, InComment=true,
      //              IntCmntIndicator="]",
      //              CmntAnchor=CLOSING,
      //              IntCmntStatus=INACTIVE

      SetInComment(true);
      SetIntCmntTag(false);
      EraseIntCmntBrk(true, true);
      SetCommentStatus(TagEnums::CLOSING, BrkIndexFileType());
      SetIntCommentStatus(-1, TagEnums::INACTIVE);
      return 4;
    }
    else if (_CmntBlkStart && _CmntBlkEnd && _IntCmntIndicator &&
             (*_IntCmntIndicator)[0] == *NestedCommentOpen_Str(ch) &&
             _CmntAnchor == TagEnums::OPENING)
    {
      SetInComment(false);
      SetIntCmntTag(false);
      EraseIntCmntBrk(true, true);
      SetCommentStatus(TagEnums::OPENING, BrkIndexFileType());
      SetIntCommentStatus(-1, TagEnums::INACTIVE);

      SetNextCmntBlock(TagCommentData::Make(_Parent, NULL, NestedCommentClose_Str(NULL),
                                            NULL, NULL, NULL, TagEnums::HTMLSTYLE));

      Subscript len = _MemPtrAlloc[NESTEDCMNTSTART2] -
                        (_MemPtrAlloc[NESTEDCMNTEND1] +
                         _MemPtrAlloc[NESTEDCMNTEND1+1]);
      TagSearchResult* NestedSearchResult_ =
          new TagSearchResult(*_CommentTagSearch);
      NestedSearchResult_->AssignSearchResult(_MemPtrAlloc[NESTEDCMNTEND1],
                                              _MemPtrAlloc[NESTEDCMNTEND1+1],
                                              _MemPtrAlloc[NESTEDCMNTSTART2],
                                              _MemPtrAlloc[NESTEDCMNTSTART2+1],
                                              (_MemPtrAlloc[NESTEDCMNTEND1] +
                                               _MemPtrAlloc[NESTEDCMNTEND1+1]),
                                              len);

      ChrString EndIntBrk_ = NestedSearchResult_->c_str();
      EndIntBrk_.Mid(_MemPtrAlloc[NESTEDCMNTEND1],
                     _MemPtrAlloc[NESTEDCMNTEND1+1]);

      _NextCmntBlock->SetTagBrkIndex(_TagBrkIndex, NULL);
      _NextCmntBlock->SetSearchResult(NestedSearchResult_);
      _NextCmntBlock->MakeCommentTagData(this);
      _NextCmntBlock->SetPrevCmntBlock(this);
      _NextCmntBlock->SetCommentStatus(-1, BrkIndexFileType());
      _NextCmntBlock->SetIntCmntBrk(NULL, EndIntBrk_.c_str());

      return _NextCmntBlock->EncounterClose(true, false);
    }
  }
  else if (IntraCmnt_ && !BlockEnd_)
  {
    if (!_IntCmntStartBrk && _IntCmntEndBrk)
    {
      // ]>     : EncounterClose(IntraCmnt=true, BlockStart=false)
      //              IntCmntEndBrk="]>", IntCmntStartBrk=0,
      //              IntCmntTag=true, InComment=false,
      //              IntCmntIndicator=0,
      //              IntCmntAnchor=CLOSING,
      //              IntCmntStatus=ACTIVE

      EraseCmntBlk(true, true, false);
      EraseIntCmntIndicator(true);
      SetInComment(false);
      SetIntCmntTag(true);
      SetIntCommentStatus(TagEnums::CLOSING, TagEnums::ACTIVE);

      SetNextCmntBlock(TagCommentData::Make(_Parent, NestedCommentOpen_Str(NULL),
                                            NULL, NULL, NULL, NULL, TagEnums::HTMLSTYLE));

      Subscript len = RootMatchingIndex() -
                        (_MemPtrAlloc[NESTEDCMNTSTART2] +
                         _MemPtrAlloc[NESTEDCMNTSTART2+1] - 1);
      TagSearchResult* NestedSearchResult_ =
          new TagSearchResult(*_CommentTagSearch);
      NestedSearchResult_->AssignSearchResult(_MemPtrAlloc[NESTEDCMNTSTART2],
                                              _MemPtrAlloc[NESTEDCMNTSTART2+1],
                                              RootMatchingIndex(),
                                              RootMatchingLength(),
                                              (_MemPtrAlloc[NESTEDCMNTSTART2] +
                                               _MemPtrAlloc[NESTEDCMNTSTART2+1] - 1),
                                              len);

      ChrString StartIntBrk_ = NestedSearchResult_->c_str();
      StartIntBrk_.Mid(_MemPtrAlloc[NESTEDCMNTSTART2],
                       _MemPtrAlloc[NESTEDCMNTSTART2+1]);

      _NextCmntBlock->SetTagBrkIndex(_TagBrkIndex, NULL);
      _NextCmntBlock->SetSearchResult(NestedSearchResult_);
      _NextCmntBlock->MakeCommentTagData(this);
      _NextCmntBlock->SetPrevCmntBlock(this);
      _NextCmntBlock->SetCommentStatus(-1, BrkIndexFileType());
      _NextCmntBlock->SetIntCmntBrk(StartIntBrk_.c_str(), NULL);

      return _NextCmntBlock->EncounterOpen(true, false);
    }
    else if (_IntCmntStartBrk && !_IntCmntEndBrk)
    {
      EraseCmntBlk(true, true, false);
      EraseIntCmntIndicator(true);
      SetInComment(true);
      SetIntCmntTag(true);
      SetIntCommentStatus(TagEnums::OPENING, TagEnums::ACTIVE);

      SetNextCmntBlock(TagCommentData::Make(_Parent, NULL, NULL,
                                            NULL, CommentClose_Str(),
                                            NestedCommentClose_Str(ch),
                                            TagEnums::HTMLSTYLE));

      TagSearchResult* NestedSearchResult_ =
          new TagSearchResult(*_CommentTagSearch);
      NestedSearchResult_->AssignSearchResult(RootMatchingIndex(),
                                              RootMatchingLength());

      _NextCmntBlock->SetTagBrkIndex(_TagBrkIndex, NULL);
      _NextCmntBlock->SetSearchResult(NestedSearchResult_);
      _NextCmntBlock->MakeCommentTagData(this);
      _NextCmntBlock->SetPrevCmntBlock(this);
      _NextCmntBlock->SetCommentStatus(TagEnums::CLOSING, BrkIndexFileType());

      return _NextCmntBlock->EncounterClose(false, true);
    }
  }
}

/****************************************************************************/
const TagStringData* TagCommentData::Parent() const
{
  return
  (
    (_CmntAnchor == TagEnums::OPENING && _CmntBlkStart && _Parent) ?
        _Parent:
        NULL
  );
}

/****************************************************************************/
const TagCommentData* TagCommentData::NextCmntBlock() const
{
  return _NextCmntBlock;
}

/****************************************************************************/
const TagCommentData* TagCommentData::PrevCmntBlock() const
{
  return
  (
    ((_CmntAnchor != TagEnums::OPENING || !_CmntBlkStart) && _PrevCmntBlock) ?
        _PrevCmntBlock:
        NULL
  );
}

/****************************************************************************/
const ChrString* TagCommentData::IntCmntContent() const
{
  return
  (
    ((_IntCmntTag && !_InComment && _IntCmntStatus == TagEnums::ACTIVE) ||
     (!_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::INACTIVE)) ?
         _IntCmntContent:NULL
  );
}

/****************************************************************************/
const ChrString* TagCommentData::CmntContent() const
{
  return
  (
    ((!_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::IGNORED) ||
     (_IntCmntTag && _InComment && _IntCmntStatus == TagEnums::ACTIVE)) ?
         _CmntContent:NULL
  );
}

/****************************************************************************/
const ChrString* TagCommentData::IntCmntEndBrk() const
{
  return _IntCmntEndBrk;
}

/****************************************************************************/
const ChrString* TagCommentData::IntCmntStartBrk() const
{
  return _IntCmntStartBrk;
}

/****************************************************************************/
const ChrString* TagCommentData::CmntBlkEnd() const
{
  return _CmntBlkEnd;
}

/****************************************************************************/
const ChrString* TagCommentData::CmntBlkStart() const
{
  return _CmntBlkStart;
}

/****************************************************************************/
const ChrString* TagCommentData::IntCmntIndicator() const
{
  return _IntCmntIndicator;
}

/****************************************************************************/
const ChrString* TagCommentData::CommentTagString() const
{
  return _CommentTagString;
}

/****************************************************************************/
const TagCommentData* TagCommentData::Root() const
{
  const TagCommentData* Prev_ = PrevCmntBlock();
  while (Prev_ && Prev_->PrevCmntBlock())
    Prev_ = Prev_->PrevCmntBlock();

  return Prev_;
}

/****************************************************************************/
Subscript TagCommentData::RootIndex() const
{
  const TagCommentData* Root_ = Root();

  return
  (
    (Root_ && Root_->SearchResult()) ?
        Root_->SearchResult()->BracketIndex():0
  );
}

/****************************************************************************/
Subscript TagCommentData::RootLength() const
{
  const TagCommentData* Root_ = Root();

  return
  (
    (Root_ && Root_->SearchResult()) ?
        Root_->SearchResult()->BracketLength():0
  );
}

/****************************************************************************/
Subscript TagCommentData::RootMatchingIndex() const
{
  const TagCommentData* Root_ = Root();

  return
  (
    (Root_ && Root_->SearchResult()) ?
        Root_->SearchResult()->MatchingIndex():0
  );
}

/****************************************************************************/
Subscript TagCommentData::RootMatchingLength() const
{
  const TagCommentData* Root_ = Root();

  return
  (
    (Root_ && Root_->SearchResult()) ?
        Root_->SearchResult()->MatchingLength():0
  );
}

/****************************************************************************/
void TagCommentData::DumpParseResults(ostream& os_, size_t fw_) const
{
  if (!TagTypeInfo::ShowDebugDump() && !ShowCommentDataDump())
    return;

  const TagSearchResult* Result_ = SearchResult();
  ChrString RemStr_ = Result_->c_str();

  const ChrString* cPtr_ = IntCmntIndicator();
  if (cPtr_)
    { os_.width(fw_); os_ <<std::left <<"IntCmntIndicator: " <<*cPtr_ <<endl; }

  cPtr_ = IntCmntStartBrk();
  if (cPtr_)
    { os_.width(fw_); os_ <<std::left <<"IntCmntStartBrk: " <<*cPtr_ <<endl; }

  cPtr_ = IntCmntContent();
  if (cPtr_)
    { os_.width(fw_); os_ <<std::left <<"IntCmntContent: " <<*cPtr_ <<endl; }

  cPtr_ = IntCmntEndBrk();
  if (cPtr_)
    { os_.width(fw_); os_ <<std::left <<"IntCmntEndBrk: " <<*cPtr_ <<endl; }

  cPtr_ = CmntBlkStart();
  if (cPtr_)
    { os_.width(fw_); os_ <<std::left <<"CmntBlkStart: " <<*cPtr_ <<endl; }

  cPtr_ = CmntContent();
  if (cPtr_)
    { os_.width(fw_); os_ <<std::left <<"CmntContent: " <<*cPtr_ <<endl; }

  cPtr_ = CmntBlkEnd();
  if (cPtr_)
    { os_.width(fw_); os_ <<std::left <<"CmntBlkEnd: " <<*cPtr_ <<endl; }

  os_.width(fw_); os_ <<std::left <<"IntCmntTag: " <<IntCmntTag() <<endl;
  os_.width(fw_); os_ <<std::left <<"InComment: " <<InComment() <<endl;
  os_.width(fw_); os_ <<std::left <<"CommentFound: " <<CommentFound() <<endl;
  os_.width(fw_); os_ <<std::left <<"NestedCmntTag: " <<NestedCmntTag() <<endl;

  if (NestedCmntTag())
  {
    os_.width(fw_); os_ <<std::left <<"    NestedCmntEnd1Index: " <<NestedCmntEnd1Index() <<endl;
    os_.width(fw_); os_ <<std::left <<"    NestedCmntEnd1Length: " <<NestedCmntEnd1Length() <<endl;
    os_.width(fw_); os_ <<std::left <<"    NestedCmntStart2Index: " <<NestedCmntStart2Index() <<endl;
    os_.width(fw_); os_ <<std::left <<"    NestedCmntStart2Length: " <<NestedCmntStart2Length() <<endl;
  }

  os_.width(fw_); os_ <<std::left <<"IntCmntStatus: " <<IntCmntStatus() <<endl;
  os_.width(fw_); os_ <<std::left <<"IntCmntAnchor: " <<IntCmntAnchor() <<endl;
  os_.width(fw_); os_ <<std::left <<"CmntAnchor: " <<CmntAnchor() <<endl;
  os_.width(fw_); os_ <<std::left <<"CmntType: " <<CmntType() <<endl;

  Result_->DumpSearchResults(os_, fw_, TagTypeInfo::ShowNewLines());
  os_ <<endl;

  if (NextCmntBlock())
  {
    os_ <<"----------" <<endl;
    NextCmntBlock()->DumpParseResults(os_, fw_);
  }
}

/****************************************************************************/
MEMORYOPS_DEFN(TagCommentData)

/****************************************************************************/
// KeyValuePairNode class definition
/****************************************************************************/
KeyValuePairNode::KeyValuePairNode():
_Key(NULL),
_Value(NULL)
{}

/****************************************************************************/
KeyValuePairNode::KeyValuePairNode(ChrString* Key_, ChrString* Value_, bool KeepPtr_):
_Key((!KeepPtr_ && Key_) ? new ChrString(*Key_):Key_),
_Value((!KeepPtr_ && Value_) ? new ChrString(*Value_):Value_)
{}

/****************************************************************************/
KeyValuePairNode::~KeyValuePairNode()
{
  delete _Key;
  delete _Value;
  _Key = _Value = NULL;
}

/****************************************************************************/
KeyValuePairNode* KeyValuePairNode::Make()
{
  return (new KeyValuePairNode());
}

/****************************************************************************/
KeyValuePairNode* KeyValuePairNode::Make(ChrString* Key_, ChrString* Value_, bool KeepPtr_)
{
  return (new KeyValuePairNode(Key_, Value_, KeepPtr_));
}

/****************************************************************************/
KeyValuePairNode& KeyValuePairNode::AddPair(ChrString* Key_, ChrString* Value_, bool KeepPtr_)
{
  if (!KeepPtr_)
  {
    if (Key_)
      if (_Key)
        *_Key = *Key_;
      else
        _Key = new ChrString(*Key_);

    if (Value_)
      if (_Value)
        *_Value = *Value_;
      else
        _Value = new ChrString(*Value_);
  }
  else
  {
    delete _Key;
    _Key = Key_;

    delete _Value;
    _Value = Value_;
  }

  return *this;
}

/****************************************************************************/
ChrString KeyValuePairNode::GiveKVPairString(bool* Avail_) const
{
  bool HasStr_ = _Value && _Key;

  if (Avail_)
    *Avail_ = HasStr_;

  if (HasStr_)
    return ChrString(*_Key + ChrString(":") + *_Value);
}

/****************************************************************************/
bool KeyValuePairNode::IsKeyMatch(const ChrString& Str_) const
{
  return (_Key ? (*_Key == Str_):false);
}

/****************************************************************************/
MEMORYOPS_DEFN(KeyValuePairNode)

/****************************************************************************/
// KeyValuePairData class definition
/****************************************************************************/
KeyValuePairData::KeyValuePairData(TagStringData* Parent_):
_Parent(Parent_),
_KvPairs(NULL),

_PairsMax(0),
_PairsLen(0),

_KVDelim(NULL),
_Wspace(NULL),
_TagTerminators(NULL),
_SearchResult(NULL),
_TagTermIndex(0),
_TagFound(false),

_KVTagString(NULL),
_KVTagSearch(NULL)
{
  GrowVector();
}

/****************************************************************************/
KeyValuePairData::KeyValuePairData(TagStringData* Parent_,
                                   ChrString* Delim_, ChrString* Wspace_,
                                   ChrString* TagTerminator_, bool KeepPtr_):
_Parent(Parent_),
_KvPairs(NULL),

_PairsMax(0),
_PairsLen(0),

_KVDelim((!KeepPtr_ && Delim_) ? (new ChrString(*Delim_)):Delim_),
_Wspace((!KeepPtr_ && Wspace_) ? (new ChrString(*Wspace_)):Wspace_),
_TagTerminators((TagSearchResult::TAGTERMLISTTYPE)RawAllocateWith(MEMMATRIX, sizeof(ChrString*) * TagEnums::TAGTERM_MAX)),
_SearchResult(NULL),
_TagTermIndex(TagTerminator_ ? 1:0),
_TagFound(false),

_KVTagString(NULL),
_KVTagSearch(NULL)
{
  _TagTerminators = TagTypeInfo::InitChrStringPtrs(_TagTerminators, TagEnums::TAGTERM_MAX);

  if (_TagTermIndex && _TagTerminators)
    _TagTerminators[0] = (!KeepPtr_ && TagTerminator_) ?
                            (new ChrString(*TagTerminator_)):TagTerminator_;

  GrowVector();
}

/****************************************************************************/
KeyValuePairData::KeyValuePairData(const KeyValuePairData& Obj_):
_Parent(Obj_._Parent),
_KvPairs(NULL),

_PairsMax(Obj_._PairsMax),
_PairsLen(Obj_._PairsLen),

_KVDelim(Obj_._KVDelim ? (new ChrString(*Obj_._KVDelim)):NULL),
_Wspace(Obj_._Wspace ? (new ChrString(*Obj_._Wspace)):NULL),
_TagTerminators((TagSearchResult::TAGTERMLISTTYPE)RawAllocateWith(MEMMATRIX, sizeof(ChrString*) * TagEnums::TAGTERM_MAX)),
_SearchResult(Obj_._SearchResult ? (new TagSearchResult(*Obj_._SearchResult)):NULL),
_TagTermIndex(Obj_._TagTermIndex),
_TagFound(Obj_._TagFound),

_KVTagString(Obj_._KVTagString ? (new ChrString(*Obj_._KVTagString)):NULL),
_KVTagSearch(Obj_._KVTagSearch ? (new TagSearchResult(*Obj_._KVTagSearch)):NULL)
{
  Subscript x;
  Subscript max = _TagTermIndex;
  _TagTerminators = TagTypeInfo::InitChrStringPtrs(_TagTerminators, TagEnums::TAGTERM_MAX);

  if (_TagTermIndex && Obj_._TagTerminators)
    for (x = 0; x < max && Obj_._TagTerminators[x]; x++)
      _TagTerminators[x] = new ChrString(*Obj_._TagTerminators[x]);

  max = Obj_._PairsMax;

  if (max)
  {
    max *= sizeof(KeyValuePairNode*);
    KeyValuePairNode** KvPairs_ = (KeyValuePairNode**)RawAllocateWith(MEMMATRIX, max);
    _KvPairs = (KeyValuePairNode**)::memmove(KvPairs_, Obj_._KvPairs, max);
  }

  if (_SearchResult)
  {
    if (_KVDelim)
      _SearchResult->SetKVDelimPtr(_KVDelim);

    if (_Wspace)
      _SearchResult->SetWspacePtr(_Wspace);

    if (_TagTerminators && _TagTermIndex)
      _SearchResult->SetTagTerminatorPtr(_TagTerminators, _TagTermIndex);
  }
}

/****************************************************************************/
KeyValuePairData::~KeyValuePairData()
{
  delete _KVDelim;
  delete _Wspace;
  delete _SearchResult;
  delete _KVTagString;
  delete _KVTagSearch;

  _KVDelim = NULL;
  _Wspace = NULL;
  _SearchResult = NULL;
  _KVTagString = NULL;
  _KVTagSearch = NULL;

  Subscript x;
  Subscript max = _PairsLen;

  for (x = 0; x < max; x++)
  {
    delete _KvPairs[x];
    _KvPairs[x] = NULL;
  }

  max = _TagTermIndex;
  for (x = 0; x < max; x++)
  {
    delete _TagTerminators[x];
    _TagTerminators[x] = NULL;
  }

  RawDeleteArray(_TagTerminators);
  RawDeleteArray(_KvPairs);

  _TagTerminators = NULL;
  _KvPairs = NULL;
}

/****************************************************************************/
void KeyValuePairData::EraseKvPairs()
{
  Subscript x;
  Subscript max = _PairsLen;

  for (x = 0; x < max; x++)
  {
    delete _KvPairs[x];
    _KvPairs[x] = NULL;
  }

  _PairsLen = 0;
}

/****************************************************************************/
KeyValuePairData* KeyValuePairData::Make(TagStringData* Parent_)
{
  return (new KeyValuePairData(Parent_));
}

/****************************************************************************/
KeyValuePairData* KeyValuePairData::Make(TagStringData* Parent_,
                                         ChrString* Delim_, ChrString* Wspace_,
                                         ChrString* TagTerminator_, bool KeepPtr_)
{
  return (new KeyValuePairData(Parent_,
                               Delim_, Wspace_,
                               TagTerminator_, KeepPtr_));
}

/****************************************************************************/
KeyValuePairData* KeyValuePairData::Make(const KeyValuePairData& Obj_)
{
  return (new KeyValuePairData(Obj_));
}

/****************************************************************************/
KeyValuePairData& KeyValuePairData::operator = (const KeyValuePairData& Obj_)
{
  if (this != &Obj_)
  {
    _Parent = Obj_._Parent;

    _PairsMax = Obj_._PairsMax;
    _PairsLen = Obj_._PairsLen;

    Subscript x;
    Subscript max = _PairsLen;

    for (x = 0; x < max; x++)
    {
      delete _KvPairs[x];
      _KvPairs[x] = NULL;
    }

    max = _TagTermIndex;
    for (x = 0; x < max; x++)
    {
      delete _TagTerminators[x];
      _TagTerminators[x] = NULL;
    }

    RawDeleteArray(_TagTerminators);
    RawDeleteArray(_KvPairs);
    max = _PairsMax;

    if (max)
    {
      max *= sizeof(KeyValuePairNode*);
      KeyValuePairNode** KvPairs_ = (KeyValuePairNode**)RawAllocateWith(MEMMATRIX, max);
      _KvPairs = (KeyValuePairNode**)::memmove(KvPairs_, Obj_._KvPairs, max);
    }

    delete _KVDelim;
    delete _Wspace;
    delete _SearchResult;
    delete _KVTagString;
    delete _KVTagSearch;

    _KVDelim = Obj_._KVDelim ? (new ChrString(*Obj_._KVDelim)):NULL;
    _Wspace = Obj_._Wspace ? (new ChrString(*Obj_._Wspace)):NULL;
    _TagTerminators = (TagSearchResult::TAGTERMLISTTYPE)RawAllocateWith(MEMMATRIX, sizeof(ChrString*) * TagEnums::TAGTERM_MAX);
    _SearchResult = Obj_._SearchResult ? (new TagSearchResult(*Obj_._SearchResult)):NULL;
    _TagTermIndex = Obj_._TagTermIndex;
    _TagFound = Obj_._TagFound;

    max = _TagTermIndex;
    _TagTerminators = TagTypeInfo::InitChrStringPtrs(_TagTerminators, TagEnums::TAGTERM_MAX);

    if (_TagTermIndex && Obj_._TagTerminators)
      for (x = 0; x < max && Obj_._TagTerminators[x]; x++)
        _TagTerminators[x] = new ChrString(*Obj_._TagTerminators[x]);

    _KVTagString = Obj_._KVTagString ? (new ChrString(*Obj_._KVTagString)):NULL;
    _KVTagSearch = Obj_._KVTagSearch ? (new TagSearchResult(*Obj_._KVTagSearch)):NULL;

    if (_SearchResult)
    {
      if (_KVDelim)
        _SearchResult->SetKVDelimPtr(_KVDelim);

      if (_Wspace)
        _SearchResult->SetWspacePtr(_Wspace);

      if (_TagTerminators && _TagTermIndex)
        _SearchResult->SetTagTerminatorPtr(_TagTerminators, _TagTermIndex);
    }

    if (_KVTagSearch)
    {
      if (_KVDelim)
        _KVTagSearch->SetKVDelimPtr(_KVDelim);

      if (_Wspace)
        _KVTagSearch->SetWspacePtr(_Wspace);

      if (_TagTerminators && _TagTermIndex)
        _KVTagSearch->SetTagTerminatorPtr(_TagTerminators, _TagTermIndex);
    }
  }

  return *this;
}

/****************************************************************************/
KeyValuePairNode** KeyValuePairData::GrowVector(long Incr_)
{
  Subscript max = 0;
  KeyValuePairNode** KvPairs_ = NULL;

  if (!_KvPairs || !_PairsMax)
  {
    max = Incr_;
    max *= sizeof(KeyValuePairNode*);
    KvPairs_ = (KeyValuePairNode**)RawAllocateWith(MEMMATRIX, max);
    _KvPairs = KvPairs_;
    _PairsMax = Incr_;
  }
  else
  {
    max = _PairsMax + Incr_;
    max *= sizeof(KeyValuePairNode*);
    KvPairs_ = _KvPairs;
    _KvPairs = (KeyValuePairNode**)RawAllocateWith(MEMMATRIX, max);
    max = _PairsMax * sizeof(KeyValuePairNode*);
    ::memmove(_KvPairs, KvPairs_, max);
    _PairsMax += Incr_;
    RawDeleteArray(KvPairs_);
  }

  return _KvPairs;
}

/****************************************************************************/
long KeyValuePairData::ParseKVpair()
{
  Subscript x, y, max;
  ChrString KeyStr_;
  ChrString ValStr_;
  bool Found_;

  if (_SearchResult && _SearchResult->StartBrkFound() && _SearchResult->EndBrkFound() &&
      TagFound(_SearchResult->c_str(), _SearchResult->BracketIndex()))
  {  
    max = _KVTagSearch->DelimsFound();

    if (_KVTagSearch->StartBrkFound())
    {
      EncounterTagOpen(NULL, NULL);
      EraseKvPairs();
    }

    for (x = 0; x < max; x+=2)
    {
      y = x / 2;
      KeyStr_ = _KVTagSearch->TagKeyAt(y, &Found_);
      ValStr_ = _KVTagSearch->TagValueAt(y, &Found_);

      if (Found_)
      {
        _KvPairs[_PairsLen++] = KeyValuePairNode::Make(&KeyStr_, &ValStr_, false);

        if (_PairsLen == _PairsMax)
          GrowVector();
      }
    }

    if (_KVTagSearch->EndBrkFound())
    {
      EncounterTagClose(NULL, NULL);

      if (_KVDelim)
        _KVTagSearch->SetKVDelimPtr(_KVDelim);

      if (_Wspace)
        _KVTagSearch->SetWspacePtr(_Wspace);

      if (_TagTerminators && _TagTermIndex)
        _KVTagSearch->SetTagTerminatorPtr(_TagTerminators, _TagTermIndex);

      return _PairsLen;
    }
  }

  return 0;
}

/****************************************************************************/
bool KeyValuePairData::TagFound(const char* Str_, long Pos_)
{
  long len = 0;
  bool StartDone_ = false;
  bool EndDone_ = false;
  bool emtfound_ = false;
  bool TagFound_ = _TagFound;
  ChrString* SubStr_ = Str_ ? new ChrString(Str_):NULL;
  ChrString TestStr_(EMPTYCLOSE_STR);

  int TagBrkIndex_;
  int BrkEnder_;
  TagCountData* CntData_ = TagCountData::Instance();

  if (SubStr_ && CntData_ && Pos_ < SubStr_->strlen())
  {
    TagBrkIndex_ = CntData_->BrkStringToIndex((*SubStr_)(Pos_), BrkEnder_, true);
    len = ::SafeStrLen(CntData_->GiveTagBrk(TagBrkIndex_));
  }

  if (SubStr_ && _SearchResult && len &&
      strncmp((*SubStr_)(Pos_), STDTAGOPEN_STR, len) == 0)
  {
    ChrString BrkStr_ = _SearchResult->GiveTargetBrkString();
    StartDone_ = strcmp(BrkStr_.c_str(), STDTAGOPEN_STR) == 0;
    BrkStr_ = _SearchResult->GiveMatchingBrkString();
    EndDone_ = _SearchResult->StartBrkFound() &&
               _SearchResult->EndBrkFound() &&
               _SearchResult->TagElementFound() &&
               _SearchResult->DelimsFound();

    if (StartDone_ && EndDone_ &&
        _SearchResult->BracketIndex() == Pos_ &&
            (strcmp(BrkStr_.c_str(), STDTAGCLOSE_STR) == 0 ||
             (emtfound_=strcmp(BrkStr_.c_str(), EMPTYCLOSE_STR) == 0)))
    {
      if (TagFound_)
      {
        if (_KVTagSearch)
          delete _KVTagSearch;

        if (_KVTagString)
          delete _KVTagString;

        _KVTagSearch = NULL;
        _KVTagString = NULL;
      }

      if (emtfound_)
      {
        if (_TagTermIndex && _TagTerminators && _TagTerminators[0])
        {
          if (!HasTerm(TestStr_))
            AddTagTerminator(new ChrString(EMPTYCLOSE_STR), true);
        }
        else
        {
          ClearTagTerminators();
          AddTagTerminator(new ChrString(STDTAGCLOSE_STR), true);
          AddTagTerminator(new ChrString(EMPTYCLOSE_STR), true);
        }
      }
      else
      {
        ClearTagTerminators();
        AddTagTerminator(new ChrString(STDTAGCLOSE_STR), true);
      }

      _KVTagString = new ChrString(_SearchResult->GiveTagString());
      _KVTagSearch = new TagSearchResult(*_SearchResult);
      _TagFound = true;
    }
  }

  delete SubStr_;
  return (StartDone_ && EndDone_);
}

/****************************************************************************/
bool KeyValuePairData::EncounterTagOpen(Subscript* Index_, Subscript* Length_)
{
  if (_KVTagSearch && _KVTagSearch->StartBrkFound())
  {
    if (Index_)
      *Index_ = _KVTagSearch->BracketIndex();

    if (Length_)
      *Length_ = _KVTagSearch->BracketLength();

    return true;
  }

  return false;
}

/****************************************************************************/
bool KeyValuePairData::EncounterTagClose(Subscript* Index_, Subscript* Length_)
{
  if (_KVTagSearch && _KVTagSearch->TagElementFound() &&
      _KVTagSearch->EndBrkFound())
  {
    if (Index_)
      *Index_ = _KVTagSearch->MatchingIndex();

    if (Length_)
      *Length_ = _KVTagSearch->MatchingLength();

    return true;
  }

  return false;
}

/****************************************************************************/
KeyValuePairData& KeyValuePairData::SetKVDelimiter(ChrString* Ptr_, bool KeepPtr_)
{
  _KVDelim = (!KeepPtr_ && Ptr_) ?
                 (new ChrString(*Ptr_)):
                 Ptr_;

  if (_SearchResult)
    _SearchResult->SetKVDelimPtr(_KVDelim);

  if (_KVTagSearch)
    _KVTagSearch->SetKVDelimPtr(_KVDelim);

  return *this;
}

/****************************************************************************/
KeyValuePairData& KeyValuePairData::SetWspace(ChrString* Ptr_, bool KeepPtr_)
{
  _Wspace = (!KeepPtr_ && Ptr_) ?
                (new ChrString(*Ptr_)):
                Ptr_;

  if (_SearchResult)
    _SearchResult->SetWspacePtr(_Wspace);

  if (_KVTagSearch)
    _KVTagSearch->SetWspacePtr(_Wspace);

  return *this;
}

/****************************************************************************/
bool KeyValuePairData::HasTerm(const ChrString& Obj_)
{
  int x;
  int max = _TagTermIndex;

  for (x = 0; x < max; x++)
    if (_TagTerminators[x] &&
        *_TagTerminators[x] == Obj_)
      return true;

  return false;
}

/****************************************************************************/
KeyValuePairData& KeyValuePairData::AddTagTerminator(ChrString* Ptr_, bool KeepPtr_)
{
  ChrString* Terminator_ = (!KeepPtr_ && Ptr_) ?
                               (new ChrString(*Ptr_)):
                               Ptr_;

  if (_TagTermIndex < TagEnums::TAGTERM_MAX)
    _TagTerminators[_TagTermIndex++] = Terminator_;

  if (_SearchResult)
    _SearchResult->SetTagTerminatorPtr(_TagTerminators, _TagTermIndex);

  if (_KVTagSearch)
    _KVTagSearch->SetTagTerminatorPtr(_TagTerminators, _TagTermIndex);

  return *this;
}

/****************************************************************************/
KeyValuePairData& KeyValuePairData::ClearTagTerminators()
{
  int x;
  int max = _TagTermIndex;

  for (x = 0; x < max; x++)
  {
    delete _TagTerminators[x];
    _TagTerminators[x] = NULL;
  }

  _TagTermIndex = 0;

  if (_SearchResult)
    _SearchResult->SetTagTerminatorPtr(NULL, 0);

  if (_KVTagSearch)
    _KVTagSearch->SetTagTerminatorPtr(NULL, 0);
}

/****************************************************************************/
KeyValuePairData& KeyValuePairData::SetTextBufferPtr(const ChrString* Ptr_)
{
  if (_SearchResult)
    _SearchResult->SetTextBufferPtr(Ptr_);

  if (_KVTagSearch)
    _KVTagSearch->SetTextBufferPtr(Ptr_);
}

/****************************************************************************/
KeyValuePairData& KeyValuePairData::SetSearchResult(const TagSearchResult* SrchRes_)
{
  _SearchResult = SrchRes_ ? (new TagSearchResult(*SrchRes_)):NULL;

  if (_SearchResult)
  {
    if (_KVDelim)
      _SearchResult->SetKVDelimPtr(_KVDelim);

    if (_Wspace)
      _SearchResult->SetWspacePtr(_Wspace);

    if (_TagTerminators && _TagTermIndex)
      _SearchResult->SetTagTerminatorPtr(_TagTerminators, _TagTermIndex);
  }

  if (_KVTagSearch)
  {
    if (_KVDelim)
      _KVTagSearch->SetKVDelimPtr(_KVDelim);

    if (_Wspace)
      _KVTagSearch->SetWspacePtr(_Wspace);

    if (_TagTerminators && _TagTermIndex)
      _KVTagSearch->SetTagTerminatorPtr(_TagTerminators, _TagTermIndex);
  }

  return *this;
}

/****************************************************************************/
KeyValuePairData& KeyValuePairData::AddPair(ChrString* Key_, ChrString* Value_, bool KeepPtr_)
{
  if (_PairsLen < _PairsMax)
  {
    _KvPairs[_PairsLen++] = new KeyValuePairNode(Key_, Value_, KeepPtr_);

    if (_PairsLen == _PairsMax)
      GrowVector();
  }

  return *this;
}

/****************************************************************************/
const KeyValuePairNode* KeyValuePairData::GiveKvPair(long Index_, bool* Found_) const
{
  if (0 <= Index_ && Index_ < _PairsLen)
  {
    if (Found_)
      *Found_ = _KvPairs[Index_] != NULL;

    return _KvPairs[Index_];
  }

  if (Found_)
    *Found_ = false;

  return NULL;
}

/****************************************************************************/
bool KeyValuePairData::TagFound() const
{
  return _TagFound;
}

/****************************************************************************/
const ChrString* KeyValuePairData::KVDelimiter() const
{
  return _KVDelim;
}

/****************************************************************************/
const ChrString* KeyValuePairData::Wspace() const
{
  return _Wspace;
}

/****************************************************************************/
const ChrString* KeyValuePairData::TagTerminator(int Index_) const
{
  if (_TagTerminators && (0 <= Index_ && Index_ < _TagTermIndex))
    return _TagTerminators[Index_];
}

/****************************************************************************/
const ChrString* KeyValuePairData::KVTagString() const
{
  return _KVTagString;
}

/****************************************************************************/
bool KeyValuePairData::ShowKeyValuePairDump()
{
  return _ShowKeyValuePairDump;
}

/****************************************************************************/
void KeyValuePairData::SetShowKeyValuePairDump(bool Flag_)
{
  _ShowKeyValuePairDump = Flag_;
}

/****************************************************************************/
void KeyValuePairData::DumpParseResults(ostream& os_, size_t fw_) const
{
  if (!TagTypeInfo::ShowDebugDump() && !ShowKeyValuePairDump())
    return;

  const TagSearchResult* Result_ = SearchResult();
  ChrString RemStr_ = Result_->c_str();
  const KeyValuePairNode* Node_;

  Subscript x;
  Subscript max = _PairsLen;
  char String_[64];

  os_.width(fw_); os_ <<std::left <<"TagFound: " <<TagFound() <<endl;

  for (x = 0; x < max; x++)
  {
    Node_ = GiveKvPair(x);

    if (Node_)
    {
      if (Node_->_Key)
      {
        strcpy(String_, "Key[");
        LongToStr(x, String_+4);
        strcat(String_, "]: ");
        os_.width(fw_); os_ <<std::left <<String_ <<(*Node_->_Key) <<endl;
      }

      if (Node_->_Value)
      {
        strcpy(String_, "Value[");
        LongToStr(x, String_+6);
        strcat(String_, "]: ");
        os_.width(fw_); os_ <<std::left <<String_ <<(*Node_->_Value) <<endl;
      }
    }
  }

  Result_->DumpSearchResults(os_, fw_, TagTypeInfo::ShowNewLines());
  os_ <<endl;
}

/****************************************************************************/
MEMORYOPS_DEFN(KeyValuePairData)

/****************************************************************************/
#endif
