#ifndef TAGSTKNODE_H
#define TAGSTKNODE_H
#ifndef TAGELEMENT_H
  #include "tagelement.h"
#endif
#ifndef TEXTFILELINE_H
  #include "textfileline.h"
#endif

#define TAGREADER_DEBUG                 1  // set to 1 for debugging tests
#define TAGREADER_TAGTYPE_DEBUG         0  // tagdump1.txt

#define TAGREADER_TAGPARSE_DEBUG        1  // set to 1 for: td2 to td7
#define TAGREADER_TAGDICTDUMP_DEBUG     0  // tagdump2.txt (optional)
#define TAGREADER_TAGDICTIONARY_DEBUG   0  // tagdump2.txt

#define TAGREADER_TAGPARSEINFO_DEBUG    0  // testing TagParseInfo class
#define TAGREADER_TAGGENSEARCH_DEBUG    0  // general tag search subtests, tagdump3.txt
#define TAGREADER_USELONGSTR            0  // use long sample input string, tagdump3b.txt
#define TAGREADER_USESHORTSTR           0  // use shorten html string for tests
#define TAGREADER_USENESTEDTABLESTR     0  // use sample with nested tables, tagdump3c.txt
#define TAGREADER_TAGSPECSEARCH_DEBUG   0  // specific tag search subtests, tagdump3-4, tagdump4.txt
#define TAGREADER_TAGLNKSEQUENCER_DEBUG 0  // tag links sequencer subtests, tagdump3-4, tagdump4.txt
#define TAGREADER_TAGCOMMENT_DEBUG      0  // tagdump5.txt, TAGPARSEINFO_DEBUG == 1
#define TAGREADER_KEYVALUEPAIR_DEBUG    0  // tagdump6.txt, TAGPARSEINFO_DEBUG == 1

#define TAGREADER_ABBREV_DUMP           1  // set to 1 for abbreviated dump: tagdump3x.txt, tagdump4x.txt

#define TAGREADER_SHOW_LINKFORWARD      1  // dump results of forward linked tags
#define TAGREADER_SHOW_NODENESTING      1  // show node nesting contents in StackInfoNode
#define TAGREADER_INTERNALSTACK_DEBUG   0  // internal stack debugging, tagdump3a.txt, tagdump3-4a, tagdump4a.txt

// Tag opening/closing bracket types:
//
// valid for html, xhtml and xml tags
// DOCTYPEOPEN_STR                 "<!"   // valid for html, xhtml docs
// DOCTYPECLOSE_STR                ">"
// COMMENTOPEN_STR                 "<!--" // valid for html, xhtml comments
// COMMENTCLOSE_STR                "-->"
// NESTEDCOMMENT1OPEN_STR          "<!--" // valid for html comments
// NESTEDCOMMENT1CLOSE_STR         ">"
// NESTEDCOMMENT2OPEN_STR          "<!"   // valid for html comments
// NESTEDCOMMENT2CLOSE_STR         "-->"
// STDTAGOPEN_STR                  "<"    // valid for html, xml, xhtml
// STDTAGCLOSE_STR                 ">"
// ENDTAGOPEN_STR                  "</"   // valid for html, xml, xhtml
// ENDTAGCLOSE_STR                 ">"
// EMPTYOPEN_STR                   "<"    // valid for html, xml, xhtml
// EMPTYCLOSE_STR                  "/>"
// SCRIPTOPEN_STR                  "<?"   // valid for html, xml, xhtml
// SCRIPTCLOSE_STR                 "?>"
//
// valid only for C tags
// COPENCOMMENTBLKOPEN_STR         "/*"
// COPENCOMMENTBLKCLOSE_STR        "->"
// CCLOSECOMMENTBLKOPEN_STR        "<-"
// CCLOSECOMMENTBLKCLOSE_STR       "*/"
//
// valid only for Cpp tags
// CPPCOMMENTLINEOPEN_STR          "//"
// CPPCOMMENTLINECLOSE_STR         "=>"
//
// Tag Line Samples:
//
// <!DOCTYPE html PUBLIC
// "-//W3C//DTD HTML 4.01//EN"
// "http://www.w3.org/TR/html4/strict.dtd">
// <tag name1="value1" name2="value2">
// <!-- comment -->
//
// <!--[if lt IE 7]>
// <style>div.page img.alpha{display:none}</style>
// <![endif]-->
//
// <!--[nested comment]> content <![nested comment]-->
// </endtag>
// <empty tag/>
// <?script tag?>
//
// Tag Line deconstruction:
//
// <tagid name1="value1" name2="value2"> content </tagid>
// ^^   # ^   #^^      # ^   #^^      #^^       #^ ^   #^
// 17  7b 2  2b34     4b 2  2b34     4b56      6b1 7  7b5
//
// 1. tag starter: tag pos, tag type
// 2. tag key: key pos, 2b. key len, key array
// 3. assignment: assignment pos
// 4. tag value: value pos, 4b. value len, value array
// 5. tag ender: tag pos, tag type
// 6. tag content: content pos, 6b. content len
// 7. tag id: id pos, 7b. id len
//
// Tag conditions:
// nesting processing algorithm
//
// case 1: <start>5Jul07 </start>
// case 1a: <single>5Jul07 </single>
//
//   End tag detected:
//   _NestingCond == STENDNA_NESTCOND  // start-end tags, nesting active
//   _NestingCond == SGLEND_NESTCOND   // singleton-end tags
//   Save nesting level
//   Assign nesting value to zero
//   Pop currently stacked tag string and append to the current
//     accumulated read string. Resulting string should be:
//     <start>5Jul07 </start>
//   Save processed string into tag string result queue
//   restore nesting level
//   LenVectIndex = 0;
//   Process as normal and exit ReadUntilString procedure
//
// case 2: <single>5Jul07 <other>
//
//   Start tag encountered after singleton tag:
//   _NestingCond == SGLOTHST_NESTCOND  // singleton-otherstart tags
//   Save nesting level
//   Assign nesting value to zero
//   Pop currently stacked tag string and append to the current
//     accumulated read string. Resulting string should be:
//     <single>5Jul07
//   _HaveEndTag set to false
//   Save processed string into tag string result queue
//   restore nesting level
//   LenVectIndex = 1;
//   Process as normal and exit ReadUntilString procedure
//
// case 3: <start>5Jul07 <other>
//
//   Start tag encountered after normal tag:
//   _NestingCond == STOTHST_NESTCOND  // start-otherstart tags
//   Push tag and process normally
//
// case 3a: <start>5Jul07 </other>
//
//   _NestingCond == STOTHEND_NESTCOND  // start-otherend tags, ERROR
//   ERROR
//
// case 4: <single>5Jul07 </other>
//
//   Mismatched closing tag encountered after singleton tag:
//   _NestingCond == SGLOTHEND_NESTCOND  // singleton-otherend tags
//   Save tags. End tag start point and end tag end point is
//   assigned to end tag start point minus 1
//   Save nesting level
//   Assign nesting value to zero
//   Pop currently stacked tag string and append to the current
//     accumulated read string. Resulting string should be:
//     <single>5Jul07
//   _HaveEndTag set to false
//   Save processed string into tag string result queue
//   restore nesting level
//
//   Continue with saved string data at point of end tag start
//     with the string buffer assigned the end tag string </other>
//     then continue with logic as if end tag was parsed after
//     LenVectIndex == 1
//   LenVectIndex = 0;
//   Process as normal and exit ReadUntilString procedure
//
//----------------------------------------------------------------------------
// classes and struct included from strfilter.h:
//   struct TagEnums
//   class TagElement : public TagEnums
//
//   class TagCountData : public TagEnums
//   class TagFoundData
//
// structure and algorithm for tag storage
//
//   tag stack
//   includes:
//     scan mode: { tagread, tagcheck }
//     nesting number
//     tag string vector array
//
//   tag string vector
//   includes:
//     pointer to parent tag stack
//     tag sequence number
//     endtagattr: { open, close, forbidden }
//     matchingtag: matching sequence number of opening tag
//     pointer to tagstring
//     tag error: { valid, error, tagerr_type... }
//
//----------------------------------------------------------------------------
// Sample Parsing Output:
//
// <html>ooo<h1>xxx<h2>yyy</h2>zzz</h1>
// <applet>xxxxxoooooxxxxxooooo <font key1=val1 key2=val2>xxxxxoooooxxxxxooooo</font>
// xxxxxoooooxxxxxooooo <font key1=val1 key2=val2>xxxxxoooooxxxxxooooo</font>
// <head><title>Page Title</title></head>
// <body>
// <ul>
// <li>list item1
// <li>list item2
// <li>list item3
// </ul>
// </body>
// </applet>
// </html>
//
// parsing started:
//   TagStack: this=0x1, _NestLevel=1
//     TagVector[0]: _SeqNo=1, _MatchNo=0, _NextLevel=0x2
//       TagLine: this=0xB7, _StartTag=<html>, _EndTag=0, _ImpliedEnd=0, _TagContent=ooo, _TagStack=0x2
//         TagStack: this=0x2, _NestLevel=2
//           TagVector[0]: _SeqNo=1, _MatchNo=0, _NextLevel=0x3
//             TagLine: this=0xA1, _StartTag=<h1>, _EndTag=0, _ImpliedEnd=0, _TagContent=xxx, _TagStack=0x3
//               TagStack: this=0x3, _NestLevel=3
//                 TagVector[0]: _SeqNo=1, _MatchNo=0, _NextLevel=0
//                   TagLine: this=0xB2, _StartTag=<h2>, _EndTag=0, _ImpliedEnd=0, _TagContent=yyy, _TagStack=0
//                 TagVector[1]: _SeqNo=2, _MatchNo=1, _NextLevel=0
//                   TagLine: _StartTag=0, _EndTag=</h2>, _ImpliedEnd=0, _MatchingTag=0xB2, _TagContent=0, _TagStack=0
//           TagVector[1]: _SeqNo=2, _MatchNo=0, _NextLevel=0
//             TagLine: _StartTag=0, _EndTag=0, _ImpliedEnd=0, _TagContent=zzz, _TagStack=0
//           TagVector[2]: _SeqNo=3, _MatchNo=1, _NextLevel=0
//             TagLine: _StartTag=0, _EndTag=</h1>, _ImpliedEnd=0, _MatchingTag=0xA1, _TagContent=0, _TagStack=0
//           TagVector[3]: _SeqNo=4, _MatchNo=0, _NextLevel=0x3
//             TagLine: this=0xF2, _StartTag=<applet>, _EndTag=0, _ImpliedEnd=0, _TagContent=xxxxxoooooxxxxxooooo, _TagStack=0x3
//               TagStack: this=0x3, _NestLevel=3
//                 TagVector[0]: _SeqNo=1, _MatchNo=0, _NextLevel=0
//                   TagLine: this=0xC1, _StartTag=<font>, _EndTag=0, _ImpliedEnd=0, _TagContent=xxxxxoooooxxxxxooooo, _TagStack=0
//                     TagStringData: _KvPairs[0]._Key=key1, _KvPairs[0]._Value=val1,
//                                    _KvPairs[1]._Key=key2, _KvPairs[1]._Value=val2
//                 TagVector[1]: _SeqNo=2, _MatchNo=1, _NextLevel=0
//                   TagLine: _StartTag=0, _EndTag=</font>, _ImpliedEnd=0, _MatchingTag=0xC1, _TagContent=0, _TagStack=0
//                 TagVector[2]: _SeqNo=3, _MatchNo=0, _NextLevel=0
//                   TagLine: _StartTag=0, _EndTag=0, _ImpliedEnd=0, _TagContent=xxxxxoooooxxxxxooooo, _TagStack=0
//                 TagVector[3]: _SeqNo=4, _MatchNo=0, _NextLevel=0
//                   TagLine: this=0xD2, _StartTag=<font>, _EndTag=0, _ImpliedEnd=0, _TagContent=xxxxxoooooxxxxxooooo, _TagStack=0
//                     TagStringData: _KvPairs[0]._Key=key1, _KvPairs[0]._Value=val1,
//                                    _KvPairs[1]._Key=key2, _KvPairs[1]._Value=val2
//                 TagVector[4]: _SeqNo=5, _MatchNo=4, _NextLevel=0
//                   TagLine: _StartTag=0, _EndTag=</font>, _ImpliedEnd=0, _MatchingTag=0xD2, _TagContent=0, _TagStack=0
//                 TagVector[5]: _SeqNo=6, _MatchNo=0, _NextLevel=0x4
//                   TagLine: this=0xA3, _StartTag=<head>, _EndTag=0, _ImpliedEnd=0, _TagContent=0, _TagStack=0x4
//                     TagStack: this=0x4, _NestLevel=4
//                       TagVector[0]: _SeqNo=1, _MatchNo=0, _NextLevel=0
//                         TagLine: this=0xB4, _StartTag=<title>, _EndTag=0, _ImpliedEnd=0, _TagContent=Page Title, _TagStack=0
//                       TagVector[1]: _SeqNo=2, _MatchNo=1, _NextLevel=0
//                         TagLine: _StartTag=0, _EndTag=</title>, _ImpliedEnd=0, _MatchingTag=0xB4, _TagContent=0, _TagStack=0
//                 TagVector[6]: _SeqNo=7, _MatchNo=6, _NextLevel=0
//                   TagLine: _StartTag=0, _EndTag=</head>, _ImpliedEnd=0, _MatchingTag=0xA3, _TagContent=0, _TagStack=0
//                 TagVector[7]: _SeqNo=8, _MatchNo=0, _NextLevel=0x4
//                   TagLine: this=0xA5, _StartTag=<body>, _EndTag=0, _ImpliedEnd=0, _TagContent=0, _TagStack=0x4
//                     TagStack: this=0x4, _NestLevel=4
//                       TagVector[0]: _SeqNo=1, _MatchNo=0, _NextLevel=0x5
//                         TagLine: this=0xE1, _StartTag=<ul>, _EndTag=0, _ImpliedEnd=0, _TagContent=0, _TagStack=0x5
//                           TagStack: this=0x5, _NestLevel=5
//                             TagVector[0]: _SeqNo=1, _MatchNo=0, _NextLevel=0
//                               TagLine: _StartTag=<li>, _EndTag=0, _ImpliedEnd=1, _TagContent=list item1, _TagStack=0
//                             TagVector[1]: _SeqNo=2, _MatchNo=0, _NextLevel=0
//                               TagLine: _StartTag=<li>, _EndTag=0, _ImpliedEnd=1, _TagContent=list item2, _TagStack=0
//                             TagVector[2]: _SeqNo=3, _MatchNo=0, _NextLevel=0
//                               TagLine: _StartTag=<li>, _EndTag=0, _ImpliedEnd=1, _TagContent=list item3, _TagStack=0
//                       TagVector[1]: _SeqNo=2, _MatchNo=1, _NextLevel=0
//                         TagLine: _StartTag=0, _EndTag=</ul>, _ImpliedEnd=0, _MatchingTag=0xE1, _TagContent=0, _TagStack=0
//                 TagVector[8]: _SeqNo=9, _MatchNo=8, _NextLevel=0
//                   TagLine: _StartTag=0, _EndTag=</body>, _ImpliedEnd=0, _MatchingTag=0xA5, _TagContent=0, _TagStack=0
//           TagVector[4]: _SeqNo=5, _MatchNo=4, _NextLevel=0
//             TagLine: _StartTag=0, _EndTag=</applet>, _ImpliedEnd=0, _MatchingTag=0xF2, _TagContent=0, _TagStack=0
//     TagVector[1]: _SeqNo=2, _MatchNo=1, _NextLevel=0
//       TagLine: _StartTag=0, _EndTag=</html>, _ImpliedEnd=0, _MatchingTag=0xB7, _TagContent=0, _TagStack=0
//
//
// Contains data on tag search results with respective tag element positions
// in the following format:
//   <                 tag             ' 'key'='val..., >
//   ^                 ^               ^                ^
//   _Index + _Length, _ElementLength, _DelimsArray     _MatchIndex + _MatchLength
//
//
// Tag Bracket Nesting:
// --------------------
// DOCTYPEOPEN_STR                 "<!"   // PushBrkNesting(DocTypeOpen, DocTypeOpen, 1);
// DOCTYPECLOSE_STR                ">"    // PopBrkNesting(DocTypeOpen, DocTypeClose, 1);
//
// COMMENTOPEN_STR                 "<!--" // PushBrkNesting(CommentOpen, CommentOpen, 1);
// COMMENTCLOSE_STR                "-->"  // PopBrkNesting(CommentOpen, CommentClose, 1);
//
// NESTEDCOMMENT1OPEN_STR          "<!--" // PushBrkNesting(CommentOpen, NestedComment1Open, 1);
// NESTEDCOMMENT1CLOSE_STR         ">"    // PushBrkNesting(NestedComment1Open, NestedComment1Close, 1);
// NESTEDCOMMENT2OPEN_STR          "<!"   // if (nesting(CommentOpen) && nesting(Nested1CommentOpen))
//                                        //   PopBrkNesting(CommentOpen, NestedComment2Open, 1);
// NESTEDCOMMENT2CLOSE_STR         "-->"  // if (nesting(NestedComment1Open))
//                                        //   PopBrkNesting(NestedComment1Open, NestedComment2Close, 1);
//                                        // else
//                                             PopBrkNesting(CommentOpen, CommentClose, 1);
//
// STDTAGOPEN_STR                  "<"    // if (prev_tagattr == optionalpair)
//                                        //   PopBrkNesting(StdTagOpen, StdTagOpen, 1);
//                                        // PushBrkNesting(StdTagOpen, StdTagOpen, 1);
// STDTAGCLOSE_STR                 ">"    // if (tagattr == singleton)
//                                        //   PopBrkNesting(StdTagOpen, StdTagClose, 1);
//                                        // else
//                                        //   PopBrkNesting(StdTagOpen, StdTagClose, 0);
// ENDTAGOPEN_STR                  "</"   // if (tagattr == pair || tagattr == optionalpair)
//                                        //   PopBrkNesting(StdTagOpen, EndTagOpen, 1);
//                                        // else
//                                        //   PushBrkNesting(EndTagOpen, EndTagOpen, 1);
// ENDTAGCLOSE_STR                 ">"    // no action
//
// EMPTYOPEN_STR                   "<"    // if (prev_tagattr == optionalpair)
//                                        //   PopBrkNesting(StdTagOpen, EmptyTagOpen, 1);
//                                        // PushBrkNesting(EmptyTagOpen, EmptyTagOpen, 1);
// EMPTYCLOSE_STR                  "/>"   // if (tagattr == empty)
//                                        //   PopBrkNesting(EmptyTagOpen, EmptyTagClose, 1);
//                                        // else
//                                        //   PopBrkNesting(EmptyTagOpen, EmptyTagClose, 0);
//
// SCRIPTOPEN_STR                  "<?"   // PushBrkNesting(ScriptTagOpen, ScriptTagOpen, 1);
// SCRIPTCLOSE_STR                 "?>"   // PopBrkNesting(ScriptTagOpen, ScriptTagClose, 1);
//
// valid only for C tags
// COPENCOMMENTBLKOPEN_STR         "/*"   // PushBrkNesting(COpenBlkOpen, COpenBlkOpen, 1);
// COPENCOMMENTBLKCLOSE_STR        "->"   // PushBrkNesting(COpenBlkClose, COpenBlkClose, 1);
// CCLOSECOMMENTBLKOPEN_STR        "<-"   // if (nesting(COpenBlkOpen) && nesting(COpenBlkClose))
//                                        //   PopBrkNesting(COpenBlkOpen, CCloseBlkOpen, 1);
// CCLOSECOMMENTBLKCLOSE_STR       "*/"   // if (nesting(COpenBlkClose))
//                                        //   PopBrkNesting(COpenBlkClose, CCloseBlkClose, 1);
//
// valid only for Cpp tags
// CPPCOMMENTLINEOPEN_STR          "//"   // PushBrkNesting(CppCommentOpen, CppCommentOpen, 1);
// CPPCOMMENTLINECLOSE_STR         "=>"   // PopBrkNesting(CppCommentOpen, CppCommentClose, 1);
//
// Tag Bracket Attributes:
// -----------------------
// DOCTYPEOPEN_STR                 "<!"   // Singleton tag
// DOCTYPECLOSE_STR                ">"    // Singleton tag
// COMMENTOPEN_STR                 "<!--" // Singleton tag
// COMMENTCLOSE_STR                "-->"  // Singleton tag
// NESTEDCOMMENT1OPEN_STR          "<!--" // pair, matches with NestedComment2
// NESTEDCOMMENT1CLOSE_STR         ">"    // pair, matches with NestedComment2
// NESTEDCOMMENT2OPEN_STR          "<!"   // pair ender
// NESTEDCOMMENT2CLOSE_STR         "-->"  // pair ender
// STDTAGOPEN_STR                  "<"    // tag specific, matches with EndTag
// STDTAGCLOSE_STR                 ">"    // tag specific, matches with EndTag
// ENDTAGOPEN_STR                  "</"   // pair ender
// ENDTAGCLOSE_STR                 ">"    // pair ender
// EMPTYOPEN_STR                   "<"    // Singleton tag, empty tag
// EMPTYCLOSE_STR                  "/>"   // Singleton tag, empty tag
// SCRIPTOPEN_STR                  "<?"   // Singleton tag
// SCRIPTCLOSE_STR                 "?>"   // Singleton tag
// COPENCOMMENTBLKOPEN_STR         "/*"   // pair, matches with CCloseComment
// COPENCOMMENTBLKCLOSE_STR        "->"   // pair, matches with CCloseComment
// CCLOSECOMMENTBLKOPEN_STR        "<-"   // pair ender
// CCLOSECOMMENTBLKCLOSE_STR       "*/"   // pair ender
// CPPCOMMENTLINEOPEN_STR          "//"   // Singleton tag
// CPPCOMMENTLINECLOSE_STR         "=>"   // Singleton tag
//
class TagTypeInfo;

class TagSearchResult : public TagEnums
{
  public:
    typedef const ChrString* CONSTCHRSTRTYPE;
    typedef CONSTCHRSTRTYPE* TAGTERMLISTTYPE;

    typedef struct
    {
      Subscript _StartPos;
      Subscript _EndPos;
      ChrString _Str;
    } ContentDataType;

    enum
    {
      NO_CONTENT              = 0,
      TAG_ENCL_CONTENT        = 1,
      INTRA_TAG_CONTENT       = 2,
      INTRA_NESTEDTAG_CONTENT = 4
    };

    static const char* TagContentTypeToStr(int Type_);

  protected:
    // Store details of tag delimiter set sequence: ' ', '=', ' ', '=' ...
    //
    typedef struct
    {
      char _DelimChar;    // Tag delims can be either ' ' or '='
      Subscript _Index;   // Index position within string where delim is found
    } TagDelimiterSet;

    // TagTypeInfo parent pointer
    TagTypeInfo* _Parent;

    // Input data text file line and column position of search cursor
    //   xpos: column position
    //   ypos: row position
    Subscript _Xpos;
    Subscript _Ypos;

    // text buffer of read text line in tag file
    mutable const ChrString* _pTextBuffer;
    mutable TAGTERMLISTTYPE _pTagTerminator;
    mutable const ChrString* _pKVDelim;
    mutable const ChrString* _pWspace;
    mutable int _TagTermIndex;

    Subscript _ReadPos;
    Subscript _PrevReadPos;
    bool _SearchDone;         // Search operation on text buffer performed
    bool _EofFound;           // End of file marker found

    // fallible search result object
    mutable Fallible<Subscript> _SrchFall;

    long _TagBracketIndex;    // tag bracket index
    Subscript _Index;         // raw search result text line index pos
    Subscript _Length;        // raw search result text str length

    int _PrevReturnValue;     // Saved previous return value
    int _ReturnValue;         // Not Found        = 0,
                              // MATCH_TAGBRACKET = 2048,
                              // MATCH_TAGTYPE    = 4096,
                              // MATCH_TAGDELIMS  = 8192

    Subscript _BrkIndex;      // starting tag text line index pos
    Subscript _BrkLength;     // starting tag text str length

    Subscript _MatchIndex;    // ending tag text line index pos
    Subscript _MatchLength;   // ending tag text str length
    
    bool _AtStartBrk;         // 1: indicate indexes at expected positions
                              // 0: indicate ineexes at swapped positions

    Subscript _ElementIndex;  // tag element text line index pos
    Subscript _ElementLength; // tag element str length

    TagDelimiterSet* _DelimsArray; // delimiter positions of: ' ', '=' in tag
    Subscript _NumDelims;          // Number of delims found
    Subscript _MaxDelims;          // maximum capacity of delimiters array

    bool _AtKeyStart;    // At start of key-value pair so delim is: ' '
                         // else if false then at end of kvpair so delim is: '='
    bool _ValidDelimSet; // indicates if delimiters found are in valid
                         // delim characters and sequence order
                         
    bool _StartBrkFound;   // start bracket of tag found in search
    bool _EndBrkFound;     // end bracket of tag found in search
    bool _TagElementFound; // element name of tag found in search

    int _ContentType;      // Type of tag content that is retrieved
    ContentDataType _Content;    // Tag enclosed content data
    ContentDataType _IntContent; // Intra tag content data

    void InitContentData(const TagSearchResult* Ptr_=NULL);
    TagSearchResult::TagDelimiterSet* GrowVector(long Incr_=10);
    size_t FindToken(ChrString& SearchStr_, size_t& Index_, size_t& NextPos_, bool& Eos_);

    bool EndConditionReached(const ChrString& SearchStr_, Subscript StartPos_, Subscript EndPos_) const;
    bool IsWspace(char c);
    bool IsKVdelim(char c);
    char NextWspace(char c);
    char NextKVdelim(char c);

  public:
    TagSearchResult(TagTypeInfo* Parent_=NULL);
    TagSearchResult(const TagSearchResult& Obj_);
    ~TagSearchResult();

    TagSearchResult& operator = (const TagSearchResult& Obj_);
    TagSearchResult& SetParent(TagTypeInfo* Ptr_);
    TagSearchResult& SetFileXYpos(long Xpos_, long Ypos_);
    TagSearchResult& ResetFileXYpos();

    Fallible<Subscript> find(const ChrString& s);
    Fallible<Subscript> findlast(const ChrString& s);
    Fallible<Subscript> findnext(Subscript pos, const ChrString& s);
    Fallible<Subscript> findprevious(Subscript pos, const ChrString& s);

    Subscript SearchResultIndex() const;
    Subscript SearchResultLength() const;

    Subscript   strlen() const;
    const char* c_str()  const;
    const ChrString* txtbuf() const;

    TagSearchResult& SwapStartEndBrks();
    TagSearchResult& SetSearchType(int Type_);
    TagSearchResult& SetElementData(Subscript Index_, Subscript Length_, bool NoBlanks_);
    TagSearchResult& SetMatchingBrkData(Subscript Index_, Subscript Length_);
    TagSearchResult& SetStartingBrkData(Subscript Index_, Subscript Length_);
    TagSearchResult& AssignSearchResult(Subscript Index_, Subscript Length_,
                                        Subscript MIndex_=0, Subscript MLength_=0,
                                        Subscript EIndex_=0, Subscript ELength_=0,
                                        bool NoBlanks_=false);
                                        
    const TagSearchResult& SetTextBufferPtr(const ChrString* Ptr_) const;
    const TagSearchResult& AddTagTerminatorPtr(const ChrString* Ptr_) const;
    const TagSearchResult& SetTagTerminatorPtr(TAGTERMLISTTYPE Ptr_, int TagTermIndex_) const;
    const TagSearchResult& SetKVDelimPtr(const ChrString* Ptr_) const;
    const TagSearchResult& SetWspacePtr(const ChrString* Ptr_) const;

    TagSearchResult& SetEofFound(bool Flag_);
    TagSearchResult& AdvanceReadPos();
    TagSearchResult& ResetReadPos();

    int SearchReturn() const;
    bool TargetFound() const;
    bool AtStartBrk() const;
    bool AtEol() const;
    bool AtEotb() const;
    bool AtEof() const;

    Subscript Xstart() const;
    Subscript Xend() const;
    Subscript BracketIndex() const;
    Subscript BracketLength() const;
    Subscript MatchingIndex() const;
    Subscript MatchingLength() const;
    Subscript ElementIndex() const;
    Subscript ElementLength() const;
    Subscript TagContentStartPos() const;
    Subscript TagContentEndPos() const;
    Subscript TagContentLength() const;
    Subscript IntraTagContentStartPos() const;
    Subscript IntraTagContentEndPos() const;
    Subscript IntraTagContentLength() const;
    Subscript ReadPos() const;

    inline bool HasThisParent(const TagTypeInfo* Parent_) const
        { return (_Parent == Parent_); }
    inline Subscript Xpos() const
        { return _Xpos; }
    inline Subscript Ypos() const
        { return _Ypos; }
    inline int ContentType() const
        { return _ContentType; }
    inline long TagBracketIndex() const
        { return _TagBracketIndex; }
    inline bool SearchDone() const
        { return _SearchDone; }
    inline bool StartBrkFound() const
        { return _StartBrkFound; }
    inline bool EndBrkFound() const
        { return _EndBrkFound; }
    inline bool TagElementFound() const
        { return _TagElementFound; }
    inline Subscript DelimsFound() const
        { return (_ValidDelimSet ? _NumDelims:0); }
    inline TagSearchResult& ResetEofFound()
        { return SetEofFound(false); }
    inline TagTypeInfo* Parent()
        { return _Parent; }

    bool HasTagContentString() const;
    bool HasIntraTagContentString() const;
    bool RetrieveContents(bool PreserveNL_);
    bool RetrieveIntraContents(bool PreserveNL_,
                               const ChrString* StartStr_=NULL,
                               const ChrString* EndStr_=NULL);

    Subscript DelimPosAt(Subscript Index_, bool* Found_=NULL) const;
    char DelimCharAt(Subscript Index_, bool* Found_=NULL) const;
    ChrString TagKeyAt(Subscript Index_, bool* Found_=NULL) const;
    ChrString TagValueAt(Subscript Index_, bool* Found_=NULL) const;

    Subscript FindSingleDelims(Subscript StartPos_, Subscript EndPos_);
    Subscript FindDelims(Subscript StartPos_, Subscript EndPos_,
                         long BrkIndex_);
    Subscript FindIntraTagBrks(int TagIndex_, bool& Found_,
                          long* StartPosx_=NULL, long* EndPosx_=NULL,
                          ChrString* StartBrkStr_=NULL, ChrString* EndBrkStr_=NULL);

    void DumpXYCoordData(ostream& os_, size_t fw_=20) const;
    void DumpTargetBrkData(ostream& os_, size_t fw_=20) const;
    void DumpTagStringData(ostream& os_, size_t fw_=20, bool ShowNl_=false) const;
    void DumpTagElementData(ostream& os_, size_t fw_=20, bool ShowNl_=false) const;
    void DumpTagAttributeData(ostream& os_, size_t fw_=20, bool ShowNl_=false) const;
    void DumpMatchingBrkData(ostream& os_, size_t fw_=20) const;
    void DumpTagContentData(ostream& os_, size_t fw_=20, bool ShowNl_=false) const;
    void DumpIntraTagContentData(ostream& os_, size_t fw_=20, bool ShowNl_=false) const;
    void DumpSearchResults(ostream& os_, size_t fw_=20, bool ShowNl_=false) const;

    ChrString GiveTargetBrkString(Subscript* Indexp_=NULL, Subscript* Lenp_=NULL,
                                  bool FullNestedTag_=false) const;
    ChrString GiveTagString() const;
    ChrString GiveTagElementString() const;
    ChrString** GiveTagKeyValueString(size_t& ArrayMax_) const;
    ChrString** GiveTagIndexCharString(size_t& ArrayMax_) const;
    ChrString GiveMatchingBrkString(Subscript* Indexp_=NULL, Subscript* Lenp_=NULL,
                                    bool FullNestedTag_=false) const;
    ChrString GiveTagContentString() const;
    ChrString GiveIntraTagContentString() const;
    Subscript NumKeyValuePairs() const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
class AppTextNode;
class StackNodeInfo;
class TagDictionary;

struct TagInfoRecord
{
  int _TagAttr;              // User set: Singleton / Pair / OptionalPair / Empty / Pair_Ender
  int _TagTypeFound;         // Tag type found indicator, set after search

  long _NestingError;        // Tag nesting error indicattor (internal)
  long _TagNestCond;         // Tag nesting condition (internal)
  long _NCTagNestCond;       // NonContent Tag nesting condition (internal)

  bool _TagFound;            // Tag found indicator, set after search
  bool _ContentTypeTag;      // Tag contents allowed within tag brackets
  bool _NonContentTypeTag;   // Tag contents disallowed within tag brackets
  bool _HasPairContent;      // Has enclosed tag pair contents
  bool _EndBrkFound;         // End bracket found
  bool _TagSpcFound;         // Tag space found
  bool _SingletonTagFound;   // Singleton type tag found, set after search
  bool _EmptyTagFound;       // Empty type tag found, set after search
  bool _ContainerTagFound;   // Tag container type tag found
  bool _StdTagFound;         // Standard opening tag found
  bool _PairEnderFound;      // Ending tag of enclosing tag pair found
  bool _SingletonTagAllowed; // default == true
  bool _EmptyTagAllowed;     // default == true
  bool _SearchDone;          // Search operation was performed

  ChrString _TagName;        // Tag name copied from TagTypeInfo object
  bool _TagInfoSet;
  bool _MatchingTags;        // current and previous matching tags

  StackNodeInfo* _InfoNode;  // Valid stack info node for content type tags

  TagInfoRecord();
  void Reset();

  bool PrevTagInfoSet() const
      { return _TagInfoSet; }
  StackNodeInfo* PrevTagInfoNode()
      { return _InfoNode; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

class TagLinksSequencer : public TagEnums
{
  friend class TagDictionary;
  friend class StackNodeInfo;
  friend class AppTextNode;
  friend class TagTypeInfo;

  public:
    enum
    {
      INFO_STACK   = 10,
      CURSOR_STACK = 20,
      TAG_STACK    = 30,

      HEAD = 1,
      TAIL = 2
    };

  protected:
    TagTypeInfo* _Parent;
    Ulong _SearchRunNumber;   // Search Run Number saves number of search attempts

    long _StdTagCount;        // Standard open tag brk count (internal)
    long _NestedCmntTagCount; // Nested comment tag brk count (internal)
    long _CopenCmntTagCount;  // Open C comment tag brk count (internal)
    long _NestingError;       // Tag nesting error indicattor (internal)
    long _TagNestCond;        // Tag nesting condition (internal)
    long _NCTagNestCond;      // NonContent Tag nesting condition (internal)
    long _ExecTagActionFails; // Number of failed attempts at
                              // executing tag condition actions.

    // StackNodeInfo stack for for building linked list to be filled
    // with relevant tag stack helper information.
    StackNodeInfo* _HeadInfo;
    StackNodeInfo* _TailInfo;
    StackNodeInfo* _InfoPtr;
    bool _InfoPtrSet;

    // cursor tag from text processing stack
    StackNodeInfo* _HeadCursorTagStk;
    StackNodeInfo* _TailCursorTagStk;
    StackNodeInfo* _CursorTagStkPtr;
    bool _CursorTagStkPtrSet;

    // tag stack pointers for processing nested tags
    StackNodeInfo* _HeadTagStack;
    StackNodeInfo* _TailTagStack;
    StackNodeInfo* _TagStackPtr;
    bool _TagStackPtrSet;

    // tag stack pointers for processing nested tags
    AppTextNode* _HeadAppText;
    AppTextNode* _TailAppText;
    AppTextNode* _AppTextPtr;
    bool _AppTextPtrSet;

    TagInfoRecord* _PrevTagInfo;

    StackNodeInfo* _CursorStkTag; // Cursor matching tag at head of stack
    StackNodeInfo* _FoundTag;     // Found matching tag node from tag stack
    bool _FoundCursorTag;         // Found tag in search matches cursor stack tag
    bool _StkTagFound;            // Tag found within stack in search

  public:
    TagLinksSequencer();
    TagLinksSequencer(const TagLinksSequencer& Obj_);
    ~TagLinksSequencer();

    TagLinksSequencer& operator = (const TagLinksSequencer& Obj_);

    // Additional StackNodeInfo methods : start
    // methods for handing tag process stack for nested tags
    StackNodeInfo* PopTagStack(StackNodeInfo* FoundTagPtr_, StackNodeInfo* CurTagStkPtr_,
                               StackNodeInfo* EncNode_, TagTypeInfo* TagTypeInfoPtr_,
                               bool Erase_, bool ConfirmErased_=false);
    StackNodeInfo* PopTagStack(bool Erase_, bool ConfirmErased_=false);
    StackNodeInfo* NextTagStack(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    StackNodeInfo* PrevTagStack(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    StackNodeInfo* PushTagStack(StackNodeInfo* Node_, bool Create_);
    StackNodeInfo* HeadTagStack(bool Reset_, bool NotErased_);
    StackNodeInfo* TailTagStack(bool Reset_, bool NotErased_);
    StackNodeInfo* SetTagStackPtr(bool Set_);
    void ConfirmTagStackErased();

    // methods for handing cursor tag from text processing stack
    StackNodeInfo* PopCursorTag(bool Erase_, bool ConfirmErased_=false);
    StackNodeInfo* NextCursorTag(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    StackNodeInfo* PrevCursorTag(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    StackNodeInfo* PushCursorTag(StackNodeInfo* Node_, bool Create_);
    StackNodeInfo* HeadCursorTagStk(bool Reset_, bool NotErased_);
    StackNodeInfo* TailCursorTagStk(bool Reset_, bool NotErased_);
    StackNodeInfo* SetCursorTagStkPtr(bool Set_);
    void ConfirmCursorTagErased();

    // methods for handling StackInfoNode stack
    StackNodeInfo* PopStackInfo(bool Erase_, bool ConfirmErased_=false);
    StackNodeInfo* NextStackInfo(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    StackNodeInfo* PrevStackInfo(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    StackNodeInfo* PushStackInfo(StackNodeInfo* Node_, bool Create_, bool Show_=false);
    StackNodeInfo* HeadInfo(bool Reset_, bool NotErased_);
    StackNodeInfo* TailInfo(bool Reset_, bool NotErased_);
    StackNodeInfo* SetStackInfoPtr(bool Set_);
    void ConfirmStackInfoErased();

    // Appended tag content text methods
    AppTextNode* PopAppContStr(bool Erase_, bool ConfirmErased_,
                               StackNodeInfo* EncNode_, StackNodeInfo* ChkNode_,
                               bool* NodeErased_, bool* WasErased_);
    AppTextNode* NextAppText(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    AppTextNode* PrevAppText(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    AppTextNode* PushAppContStr(bool HasTagStk_, StackNodeInfo* Node_);
    AppTextNode* PushAppTentStr(bool HasTentStk_, StackNodeInfo* Node_);
    AppTextNode* HeadAppText(bool Reset_, bool NotErased_);
    AppTextNode* TailAppText(bool Reset_, bool NotErased_);
    AppTextNode* SetAppTextPtr(bool Set_);
    void ConfirmAppContStrErased();

    void SetOpStTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                      StackNodeInfo* EncNode_);
    void SetStdTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                     StackNodeInfo* EncNode_);
    void SetSglTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                     StackNodeInfo* EncNode_);
    void SetMatchedEndTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                            StackNodeInfo* EncNode_);
    void SetOpStTagToOpStTagData(StackNodeInfo* CurNode_,
                                 StackNodeInfo* EncNode_);
    void SetXToSglTagData(StackNodeInfo* CurNode_,
                          StackNodeInfo* EncNode_);
    void SetXToStdTagData(StackNodeInfo* CurNode_,
                          StackNodeInfo* EncNode_);
    void SetStdTagToStdTagData(StackNodeInfo* CurNode_,
                               StackNodeInfo* EncNode_);
    void SetStdTagToSglTagData(StackNodeInfo* CurNode_,
                               StackNodeInfo* EncNode_);
    void SetSglTagToMatchedEndData(StackNodeInfo* CurNode_,
                                   StackNodeInfo* EncNode_);
    void SetStdTagToMatchedEndData(StackNodeInfo* CurNode_,
                                   StackNodeInfo* EncNode_);
    void SetSglTagToSglTagData(StackNodeInfo* CurNode_,
                               StackNodeInfo* EncNode_);
    void SetStdTagData(StackNodeInfo* CurNode_);
    void SetSingleTagData(StackNodeInfo* CurNode_);
    void SetNonContentTagData(StackNodeInfo* CurNode_);
    void SetNonContentTagData(StackNodeInfo* CurNode_,
                              StackNodeInfo* EncNode_);

    inline void SetXToOpStTagData(StackNodeInfo* CurNode_,
                                  StackNodeInfo* EncNode_)
        { SetXToStdTagData(CurNode_, EncNode_); }
    inline void SetStdTagToOpStTagData(StackNodeInfo* CurNode_,
                                       StackNodeInfo* EncNode_)
        { SetStdTagToStdTagData(CurNode_, EncNode_); }
    inline void SetOpStTagToSglTagData(StackNodeInfo* CurNode_,
                                        StackNodeInfo* EncNode_)
        { SetStdTagToSglTagData(CurNode_, EncNode_); }
    inline void SetOpStTagToStdTagData(StackNodeInfo* CurNode_,
                                        StackNodeInfo* EncNode_)
        { SetOpStTagToOpStTagData(CurNode_, EncNode_); }
    inline void SetOpStTagToMatchedEndData(StackNodeInfo* CurNode_,
                                            StackNodeInfo* EncNode_)
        { SetStdTagToMatchedEndData(CurNode_, EncNode_); }
    inline void SetOpStTagData(StackNodeInfo* CurNode_)
        { SetStdTagData(CurNode_); }
    inline void SetParent(TagTypeInfo* Parent_)
        { _Parent = Parent_; }
    bool HasThisParent(const TagTypeInfo* Parent_) const
        { return (_Parent == Parent_); }

    void ResetLinks();
    void AssignGlobalStackTag()
        { _CursorTagStkPtr = _CursorStkTag; }
    void SetToGlobalStackTag()
        { _CursorStkTag = _CursorTagStkPtr; }
    StackNodeInfo* CursorStackTag()
        { return _CursorStkTag; }
    StackNodeInfo* FoundTag()
        { return _FoundTag; }
    bool FoundCursorTag()
        { return _FoundCursorTag; }
    bool StackTagFound()
        { return _StkTagFound; }

    StackNodeInfo* FindInfoNode(StackNodeInfo* Node_);
    StackNodeInfo* FindCursorTagNode(StackNodeInfo* Node_);
    StackNodeInfo* FindTagStackNode(StackNodeInfo* Node_);
    AppTextNode* FindAppTextNode(StackNodeInfo* Node_);
    StackNodeInfo* RecurseAppend(StackNodeInfo* CurNode_,
                                 StackNodeInfo* EncNode_, bool Shifted_=false);

    inline void ResetStdTagCount()
        { _StdTagCount = 0; }
    inline void IncStdTagCount(int Incr_=1)
        { _StdTagCount += Incr_; }
    inline void DecStdTagCount(int Decr_=1)
        { _StdTagCount -= Decr_; }
    inline void ResetNestedCmntTagCount()
        { _NestedCmntTagCount = 0; }
    inline void IncNestedCmntTagCount(int Incr_=1)
        { _NestedCmntTagCount += Incr_; }
    inline void DecNestedCmntTagCount(int Decr_=1)
        { _NestedCmntTagCount -= Decr_; }
    inline void ResetCopenCmntTagCount()
        { _CopenCmntTagCount = 0; }
    inline void IncCopenCmntTagCount(int Incr_=1)
        { _CopenCmntTagCount += Incr_; }
    inline void DecCopenCmntTagCount(int Decr_=1)
        { _CopenCmntTagCount -= Decr_; }
    inline void IncExecTagActionFails(int Incr_=1)
        { _ExecTagActionFails += Incr_; }
    inline void IncSearchRunNumber(int Incr_=1)
        { _SearchRunNumber += Incr_; }

    void SetStkNodeSearchData(StackNodeInfo* CurNode_);
    void DestroyAllStacks();

    void AppendNestingError(int Value_);
    void SetNestingError(int Value_);
    void ResetNestingError();
    void AppendTagNestCond(int Value_);
    void SetTagNestCond(int Value_);
    void ResetTagNestCond();
    void ResetSearchRunNumber();
    Ulong SearchRunNumber() const;
    void ResetExecTagActionFails();
    long ExecTagActionFails() const;
    inline long StdTagCount() const
        { return _StdTagCount; }
    inline long NestedCmntTagCount() const
        { return _NestedCmntTagCount; }
    inline long CopenCmntTagCount() const
        { return _CopenCmntTagCount; }
    inline long NestingError() const
        { return _NestingError; }
    inline long TagNestCondition() const
        { return _TagNestCond; }

    bool PrevTagInfoSet() const;
    void ErasePrevTagInfo();
    void ResetPrevTagInfo();
    void CompareWithPrevTagInfo(StackNodeInfo* EncNode_, bool PairEnderFound_);
    StackNodeInfo* PrevTagInfoNode();
    const StackNodeInfo* PrevTagInfoNode() const;

    inline StackNodeInfo* CursorTagStkPtr()
        { return _CursorTagStkPtr; }
    inline StackNodeInfo* TagStackPtr()
        { return _TagStackPtr; }
    inline StackNodeInfo* StackInfoPtr()
        { return _InfoPtr; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

class TagDumpStreamData
{
  private:
    ostream* _Output;    // output stream for auto output dump (output format)
                         // and writing to tag data file
    istream* _Input;     // input stream for reading from tag data file
    size_t _FieldWidth;  // field width for auto output dump (output format)

  public:
    TagDumpStreamData();
    TagDumpStreamData(const TagDumpStreamData* Ptr_);

    TagDumpStreamData& operator = (const TagDumpStreamData* Ptr_);

    inline size_t GiveFieldWidth() const
        { return _FieldWidth; }
    inline ostream* GiveOutput()
        { return _Output; }
    inline istream* GiveInput()
        { return _Input; }
    inline const ostream* GiveOutput() const
        { return _Output; }
    inline const istream* GiveInput() const
        { return _Input; }

    void SetOutput(ostream* os_);
    void SetInput(istream* is_);
    void SetFieldWidth(size_t fw_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

class TagTypeInfo : public TagEnums
{
  friend class TagLinksSequencer;
  friend class AppTextNode;
  friend class TagDictionary;
  friend class StackNodeInfo;

  public:
    struct TagSwitch
    {
      bool _IsHtml;
      bool _IsEnd;
      bool _IsScript;
      bool _IsDoc;
      bool _IsComment;
      bool _IsNestedComment;
      bool _IsSingle;
      bool _IsCLang;
      bool _IsCppLang;

      bool _AllowEnd;
      bool _AllowNestedComment;
      bool _AllowCLang;

      bool _HasEndError;
      bool _HasNestedCommentError;
      bool _HasCLangError;

      int _EndErrorCode;
      int _NestedCommentErrorCode;
      int _CLangErrorCode;

      bool _RetrieveEnd;
      bool _RetrieveComment;
      bool _RetrieveNestedComment;
      bool _RetrieveScript;
      bool _RetrieveCLang;
      bool _RetrieveCppLang;
    };

    enum
    {
      TAGEXCL_INDEX   = 0,
      TAGALLOW_INDEX  = 1,
      TAGERROR_INDEX  = 2,
      TAGRETR_INDEX   = 3,
      MAX_TAGINDEXES  = 4
    };

    enum
    {
      READPOS           = 1,
      TAGLIST           = 2,
      SEARCH_PARAMS     = 4,
      SEARCH_TARGETS    = 8,
      TAG_CONTENT_TYPE  = 16,
      ALL               = 31
    };

    enum
    {
      NO_ALTINDEX         = 0,
      SINGLE_EMPTY        = 1,
      EMPTY_SINGLE        = 2,
      COMMENT_NESTEDCMNT1 = 4,
      COMMENT_NESTEDCMNT2 = 8,
      CCMNT_COPENCMNTBLK  = 16,
      CCMNT_CCLOSECMNTBLK = 32
    };

  private:
    static size_t m_ObjectCnt;
    static bool _ShowDebugDump;   // Show debugging output dump
    static bool _ShowErasedNodes; // Show tag nodes even if erased
    static bool _ShowNewLines;    // Show new lines for completed tag searches

  protected:
    enum
    {
      ELEMENT_SPECIFIED     = 1,
      DOCTYPE_SPECIFIED     = 2,
      TAGATTR_SPECIFIED     = 4,
      TAGBRKINDEX_SPECIFIED = 8,
      ALL_SPECIFIED         = 15,
      EXEC_ACTION_MAX_FAIL  = 10
    };

    size_t _ObjectID;          // object unique ID number
    bool _TagSpecified;        // Specified html tag and bracket to find or none
    ChrString* _ElementStr;    // Element str confirmed or found if not specified
    TagElement* _ElementPtr;   // Tag element ptr of matching created TagElement object

    TagCountData* _TagCntData;      // Tag count data structure

    // User set: Tag attributes
    bool _LookupRefTag;             // default == false
    bool _DictionaryTag;            // default == false
    bool _SingletonTagAllowed;      // default == true
    bool _EmptyTagAllowed;          // default == true

    int _DocType;         // Document type: tag type allowed; html, xml, cpp...
    bool _EofFound;       // End of file marker found
    int _TagAttr;         // User set: Singleton / Pair / OptionalPair / Empty / Pair_Ender
    int _SavedTagAttr;    // Saved previous tag attributes
    int _TagTypeFound;    // Tag type found indicator, set after search

    // Tag bracket array, user specified
    // TagBrkIndex points to tag entry found in search results or
    // newest created tag entry.
    char** _TagBrkMatrix; // assigned pointer to _TagBrkList
    long _TagBrkIndex;    // in +2 increments for each tag, +1 for ending tag
    long _TagBrkMax;      // max entries in _TagBrkMatrix
    mutable long _AltIndex; // alternative index of found searched tag
                            // examples:
                            //     single(std)-->empty,
                            //     empty-->single(std),
                            //     comment-->nestedcomment1
                            //     comment-->nestedcomment2

    // Tag bracket extentions for nested type tags:
    //
    // _TagBrkIndex == TagEnums::COMMENT_DEX
    //    will assign _TagBrkExtend1 to TagEnums::NESTEDCOMMENT1_DEX   and
    //    will assign _TagBrkExtend2 to TagEnums::NESTEDCOMMENT2_DEX
    //
    // OR
    //
    // _TagBrkIndex == TagEnums::CCOMMENTBLK_DEX
    //    will assign _TagBrkExtend1 to TagEnums::COPENCOMMENTBLK_DEX  and
    //    will assign _TagBrkExtend2 to TagEnums::CCLOSECOMMENTBLK_DEX
    //
    mutable long _TagBrkExtend1; // extended close brk of opening tag in nested tag types
    mutable long _TagBrkExtend2; // extended open brk of closing tag in nested tag types
    TagSearchResult* _SearchResult;        // Tag search results data structure

    mutable TagLinksSequencer* _Sequencer; // Tag links sequencer
    mutable bool _SequencerSet;            // Links sequencer set from external class
    mutable bool _SequencerAlloc;          // Links sequenver allocated internally

    mutable HtmlSpecialCaseData* _HtmlSpecialCase; // Html tags special case data
    mutable bool _HtmlSpecialCaseSet;              // tags special case data set from external class
    mutable bool _HtmlSpecialCaseAlloc;            // tags special case data allocated internally

    long _DataSpecified;     // Checks for needed specified data from user
    bool _TagFound;          // Tag found indicator, set after search
    bool _ContentTypeTag;    // Tag contents allowed within tag brackets
    bool _NonContentTypeTag; // Tag contents disallowed within tag brackets
    bool _HasPairContent;    // Has enclosed tag pair contents
    bool _EndBrkFound;       // End bracket found
    bool _TagSpcFound;       // Tag space found
    bool _SingletonTagFound; // Singleton type tag found, set after search
    bool _EmptyTagFound;     // Empty type tag found, set after search
    bool _ContainerTagFound; // Tag container type tag found
    bool _StdTagFound;       // Standard opening tag found
    bool _PairEnderFound;    // Ending tag of enclosing tag pair found
    bool _SearchDone;        // Search operation was performed

    int _TagInitSelected;    // Saved tag switch initialization that was selected
    bool _TagInitConfirmed;  // Tag initialization confirmed and done
    TagSwitch _TagSwitch;    // Tag bracket exclusion/permission/error switches

    bool _ForbiddenUnspec;          // Forbidden pos. tags unspecified in equality function (internal)
    mutable bool _DumpTagContents;  // Dump tag contents after advancing read buffer position (output format)
    mutable TagDumpStreamData* _IO; // Input/Output streams and fieldwidth for tag dumps

    // Additional StackNodeInfo related data members
    long _TagContValid;        // Tag content valid

    // Additional StackNodeInfo methods : start
    // methods for handing tag process stack for nested tags
    void Exec_StartTagUnmatchedEnd(long NestCond_,
                                   StackNodeInfo* CurNode_, StackNodeInfo* EncNode_);
    void Exec_StartTagMatchedEnd(long NestCond_,
                                 StackNodeInfo* CurNode_, StackNodeInfo* EncNode_);
    void Exec_SingletonUnmatchedEnd(long NestCond_,
                                    StackNodeInfo* CurNode_, StackNodeInfo* EncNode_);
    void Exec_PairEndTagUnmatchedEnd(long NestCond_,
                                     StackNodeInfo* CurNode_, StackNodeInfo* EncNode_);

    void ExecTagCondAction(long NestCond_,
                           StackNodeInfo* CurNode_, StackNodeInfo* EncNode_);
    void SetTagCondData(long NestCond_,
                        StackNodeInfo* CurNode_, StackNodeInfo* EncNode_);

    // Appended tag content text methods
    void SetOpStTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                      StackNodeInfo* EncNode_);
    void SetStdTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                     StackNodeInfo* EncNode_);
    void SetSglTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                     StackNodeInfo* EncNode_);
    void SetMatchedEndTagToUnmatchedEndData(StackNodeInfo* CurNode_,
                                            StackNodeInfo* EncNode_);
    void SetOpStTagToOpStTagData(StackNodeInfo* CurNode_,
                                 StackNodeInfo* EncNode_);
    void SetXToSglTagData(StackNodeInfo* CurNode_,
                          StackNodeInfo* EncNode_);
    void SetXToStdTagData(StackNodeInfo* CurNode_,
                          StackNodeInfo* EncNode_);
    void SetStdTagToStdTagData(StackNodeInfo* CurNode_,
                               StackNodeInfo* EncNode_);
    void SetStdTagToSglTagData(StackNodeInfo* CurNode_,
                               StackNodeInfo* EncNode_);
    void SetSglTagToMatchedEndData(StackNodeInfo* CurNode_,
                                   StackNodeInfo* EncNode_);
    void SetStdTagToMatchedEndData(StackNodeInfo* CurNode_,
                                   StackNodeInfo* EncNode_);
    void SetSglTagToSglTagData(StackNodeInfo* CurNode_,
                               StackNodeInfo* EncNode_);
    void SetStdTagData(StackNodeInfo* CurNode_);
    void SetSingleTagData(StackNodeInfo* CurNode_);
    void SetNonContentTagData(StackNodeInfo* CurNode_);
    void SetNonContentTagData(StackNodeInfo* CurNode_,
                              StackNodeInfo* EncNode_);

    inline void SetPairEndToOpStTagData(StackNodeInfo* CurNode_,
                                        StackNodeInfo* EncNode_)
        { SetXToStdTagData(CurNode_, EncNode_); }
    inline void SetPairEndToStdTagData(StackNodeInfo* CurNode_,
                                       StackNodeInfo* EncNode_)
        { SetXToStdTagData(CurNode_, EncNode_); }
    inline void SetXToOpStTagData(StackNodeInfo* CurNode_,
                                  StackNodeInfo* EncNode_)
        { SetXToStdTagData(CurNode_, EncNode_); }
    inline void SetStdTagToOpStTagData(StackNodeInfo* CurNode_,
                                       StackNodeInfo* EncNode_)
        { SetStdTagToStdTagData(CurNode_, EncNode_); }
    inline void SetOpStTagToSglTagData(StackNodeInfo* CurNode_,
                                       StackNodeInfo* EncNode_)
        { SetStdTagToSglTagData(CurNode_, EncNode_); }
    inline void SetOpStTagToStdTagData(StackNodeInfo* CurNode_,
                                       StackNodeInfo* EncNode_)
        { SetOpStTagToOpStTagData(CurNode_, EncNode_); }
    inline void SetOpStTagToMatchedEndData(StackNodeInfo* CurNode_,
                                           StackNodeInfo* EncNode_)
        { SetStdTagToMatchedEndData(CurNode_, EncNode_); }
    inline void SetOpStTagData(StackNodeInfo* CurNode_)
        { SetStdTagData(CurNode_); }

    void SetPrevTagInfo(StackNodeInfo* InfoPtr_);
    void CompareWithPrevTagInfo(StackNodeInfo* EncNode_);

    TagLinksSequencer* CreateSequencer();
    HtmlSpecialCaseData* CreateHtmlSpecialCase();
    // Additional StackNodeInfo methods : end

    void ConfirmTagInit();
    void CopyFromReferenceObject(const TagTypeInfo& Obj_);
    bool SearchForElementStr(const ChrString* Str_=NULL);
    int GiveExcludedTags(bool IsHtml_, bool IsEnd_, bool IsScript_,
                         bool IsDoc_, bool IsComment_, bool IsNestedComment_,
                         bool IsSingle_, bool IsCLang_, bool IsCppLang_) const;
    bool FindIndexPositions(long* IndexPos_, long* TailIndexPos_, long x, long& FirstIndex_,
                            long& ClosestIndex_, long& ClosestLen_, long& ClosestTailIndex_, long& ClosestTailLen_,
                            long& MaxIndex_, long& MaxLen_, long& MaxTailIndex_, long& MaxTailLen_);

    ChrString* GiveEndTagTermStr(int x);
    void TransferTagFoundDataTo(TagTypeInfo* TagPtr_);
    void SetTagErrors(int ErrorTag_, long Index_);
    bool TagSequenceCheck(const char* ptr_, long* IndexPos_,
                          long cur_, long slen_, long endpt_);

    inline void SetTagElementPtr(TagElement* Ptr_)
        { _ElementPtr = Ptr_; }

    // Methods for returning binary tag switch values given a coded integer
    bool IsExcludedTag(int ExcludedTags_,
                       const char* start_, const char* end_, int* TagExcl_=NULL);
    bool IsAllowedTag(int AllowedTags_,
                      const char* start_, const char* end_, int* TagErr_=NULL);
    bool IsRetrievedTag(int RetrievedTags_,
                        const char* start_, const char* end_, int* TagRetr_=NULL);

    // Methods for returning tag switch values in a binary coded integer
    int GiveExcludedTagsForDocType(int ExcludedTags_) const;
    int GiveAllowedTags(int AllowedTags_) const;
    int GiveRetrievedTags(int RetrTags_) const;

    void ResetTagList();
    void ResetTagCount();
    void ResetSearchParams();
    void ResetSearchTargets();
    void ResetTagContentType();

    bool IsUnmatchedBrkPair(long ClosestIndex_, long ClosestLen_,
                            long ClosestTailIndex_, long ClosesTailLen_,
                            long MaxIndex_, long MaxLen_, long MaxTailLen_,
                            long* IndexPos_, long* TailIndexPos_,
                            long x, bool CppStyle_);

    // _ElementStr needs to be specified by SetElementStr(...)
    //     if _TagSpecified is true
    // _SingletonTagAllowed defaults to true
    //     if not specified by SetSingletonAllowed(...)
    // _EmptyTagAllowed defaults to true
    //     if not specified by SetEmptyAllowed(...)
    // _DocType needs to be specified by SetDocType(...)
    //     in all cases
    // _TagAttr needs to be specified by SetTagAttr(...)
    //     if _TagSpecified is true
    // _TagBrkIndex specified by SetTagBrk(...)
    //     if _TagSpecified is true

  public:
    TagTypeInfo();
    TagTypeInfo(const TagTypeInfo& Obj_);
    ~TagTypeInfo();

    static TagTypeInfo* Make();
    static TagTypeInfo* Make(const TagTypeInfo& Obj_);

    void ResetSearchRunNumber();
    Ulong SearchRunNumber() const;
    long ExecTagActionFails() const;
    long StdTagCount() const;
    long NestedCmntTagCount() const;
    long CopenCmntTagCount() const;
    long NestingError() const;
    long TagNestCondition() const;

    static TagSearchResult::TAGTERMLISTTYPE InitChrStringPtrs(TagSearchResult::TAGTERMLISTTYPE arrptr_, size_t max);
    static TagTypeInfo** InitTagTypeArray(TagTypeInfo** arrptr_, size_t max);
    static void SetShowDebugDump(bool Flag_, bool ShowErased_);
    static void SetShowNewLines(bool Flag_);
    static int GiveTagPosForTagType(int tagtype_);
    static bool IsContentTypeTag(const char* start_, const char* end_);
    static bool IsNonContentTypeTag(const char* start_, const char* end_);
    static bool IsOptionalOrNoContentTag(int TagAttr_, int TagTypeFound_,
                                         bool HasPairContent_, bool* PairPtr_=NULL);
    static bool IsOptionalTag(int TagAttr_, bool* PairPtr_=NULL);
    static bool ShowDebugDump();
    static bool ShowErasedNodes();
    static bool ShowNewLines();

    inline void ShouldDumpContents(bool Flag_) const
        { _DumpTagContents = Flag_; }
    inline bool ShouldDumpContents() const
        { return _DumpTagContents; }
    inline bool IsSingletonAllowed() const
        { return _SingletonTagAllowed; }
    inline bool IsSingletonTag() const
        { return ((_TagAttr & SINGLETON_TAGATTR) &&
                  !(_TagAttr & TAG_CONTAINER)); }
    inline bool IsEmptyAllowed() const
        { return _EmptyTagAllowed; }
    inline bool IsEmptyTag() const
        { return (_TagAttr & EMPTY_TAGATTR); }
    inline bool IsContainerAllowed() const
        { return ((_TagAttr & TagEnums::OPTIONALPAIR_TAGATTR) ||
                  (_TagAttr & TagEnums::PAIR_TAGATTR)); }
    inline bool IsContainerTag() const
        { return ((_TagAttr & TAG_CONTAINER) &&
                  !(_TagAttr & SINGLETON_TAGATTR)); }
    inline bool IsPairEnderTag() const
        { return (_TagAttr & PAIR_ENDER); }
    bool IsOptionalPairTag() const;
    inline int TagAttribute() const
        { return _TagAttr; }
    inline const TagInfoRecord* PrevTagAttribute()
        { return _Sequencer ? _Sequencer->_PrevTagInfo:NULL; }

    inline int DocTypeSpecified() const
        { return ((_DataSpecified & DOCTYPE_SPECIFIED) ? _DocType:0); }
    inline bool EofFound() const
        { return _EofFound; }
    inline int TagTypeFound() const
        { return _TagTypeFound; }
    inline bool TagFound() const
        { return _TagFound; }
    inline bool TagSpcFound() const
        { return _TagSpcFound; }
    inline bool PairEnderFound() const
        { return (_TagFound && _PairEnderFound); }
    inline bool StdTagFound() const
        { return (_TagFound && _StdTagFound); }
    inline bool SingletonTagFound() const
        { return (_TagFound && _SingletonTagFound); }
    inline bool EmptyTagFound() const
        { return (_TagFound && _EmptyTagFound); }
    inline bool ContainerTagFound() const
        { return (_TagFound && _ContainerTagFound); }
    inline bool IsContentTypeTag() const
        { return (_TagFound && _ContentTypeTag); }
    inline bool IsNonContentTypeTag() const
        { return (_TagFound && _NonContentTypeTag); }
    inline bool HasPairContent() const
        { return (_TagFound && _HasPairContent); }
    inline bool TagInitConfirmed() const
        { return _TagInitConfirmed; }

    inline void SetLookupRefTag(bool Flag_)
        { _LookupRefTag = Flag_; }
    inline bool LookupRefTag() const
        { return _LookupRefTag; }
    inline void SetDictionaryTag(bool Flag_)
        { _DictionaryTag = Flag_; }
    inline bool DictionaryTag() const
        { return _DictionaryTag; }
    inline void SetForbiddenUnspecified(bool Flag_)
        { _ForbiddenUnspec = Flag_; }
    inline bool ForbiddenUnspecified() const
        { return _ForbiddenUnspec; }

    inline size_t GiveFieldWidth() const
        { return (_IO ? _IO->GiveFieldWidth():20); }
    inline ostream* GiveOutput()
        { return (_IO ? _IO->GiveOutput():&std::cout); }
    inline istream* GiveInput()
        { return (_IO ? _IO->GiveInput():&std::cin); }
    inline const ostream* GiveOutput() const
        { return (_IO ? _IO->GiveOutput():&std::cout); }
    inline const istream* GiveInput() const
        { return (_IO ? _IO->GiveInput():&std::cin); }

    TagTypeInfo& operator = (const TagTypeInfo& Obj_);
    bool operator == (const TagTypeInfo& Obj_) const;

    bool InDocType(int DocType_) const;
    int GiveAltIndex(bool Real_) const;
    int TagIndexToTagType() const;
    int GiveTagPosition() const;
    int GiveTagNestingCondition(bool Initial_=false) const;
    int GiveDefaultTagAttrib(int TagType_, int TagPos_) const;
    int GiveMatchingTagAttrib(int TagType_, int TagPos_) const;
    bool GiveNextSwitches(TagSwitch& Switch_) const;

    // Methods for setting tag switch values into internal switch object
    int* SetTagSwitches(TagSwitch& Switch_, int* TagValArr_);
    int SetExcludedTags(TagSwitch& Switch_, int* TagErr_=NULL);
    int SetAllowedTags(TagSwitch& Switch_, int* TagErr_=NULL);
    int SetRetrievedTags(TagSwitch& Switch_, int* TagRetr_=NULL);

    void ResetTagInfo();
    bool IsEndTagTermType(int x);
    bool IsNestedTagType(int x, bool NestedCmnt_);
    bool IsNoElementNameTagType(int x);
    bool IsIntraContentTagType(int x);
    bool IsSingleDelimTagType(int x);
    bool SetContentTypeTag(const char* start_, const char* end_);
    bool SetTagInfoFound(TagTypeInfo* TagInfo_, bool Found_);
    bool SearchForTag(int* TagSwitchBits_, long* Index_=NULL);
    bool IsEqual(const TagTypeInfo& Obj_) const;
    bool TagLookupEqual(const TagTypeInfo& Obj_) const;
    bool IsParentOf(const TagSearchResult* Child_) const;
    bool IsParentOf(const AppTextNode* Child_) const;
    bool IsParentOf(const StackNodeInfo* Child_) const;
    bool IsParentOf(const TagLinksSequencer* Child_) const;
    bool IsParentOf(const HtmlSpecialCaseData* Child_) const;

    long IndexFromBrk(long* sindex_, long* eindex_, bool NestedTag_=false) const;
    const char* StartBrk(bool NestedTag_=false, long* index_=NULL) const;
    const char* EndBrk(bool NestedTag_=false, long* index_=NULL) const;

    long IndexFromBrkFound(long* sindex_, long* eindex_, bool NestedTag_=false) const;
    const char* StartBrkFound(bool NestedTag_=false, long* index_=NULL) const;
    const char* EndBrkFound(bool NestedTag_=false, long* index_=NULL) const;

    const char* TagBrkExtend1(bool StartingBrk_=false, long* index_=NULL) const;
    const char* TagBrkExtend2(bool EndingBrk_=false, long* index_=NULL) const;

    inline long GiveTagBrkIndex() const
        { return _TagBrkIndex; }
    inline char** GiveTagBrkMatrix()
        { return _TagBrkMatrix; }
    inline TagSearchResult* SearchResult()
        { return _SearchResult; }        
    inline const TagSearchResult* SearchResult() const
        { return _SearchResult; }
    inline TagCountData* GiveTagCountData()
        { return _TagCntData; }

    inline bool ElementStr_Specified() const
        { return (_TagSpecified && _ElementStr && (_DataSpecified & ELEMENT_SPECIFIED)); }
    inline bool TagAttr_Specified() const
        { return (_TagAttr && (_DataSpecified & TAGATTR_SPECIFIED)); }
    inline bool TagBrk_Specified() const
        { return (_TagBrkMatrix && (_DataSpecified & TAGBRKINDEX_SPECIFIED)); }
    inline TagTypeInfo& ResetEofFound()
        { return SetEofFound(false); }

    TagElement* GiveTagElementPtr();
    TagElement* GiveMatchingElementPtr(int TagPos_);

    TagTypeInfo& SetTagBrk(const char* Start_, const char* End_, long* Index_=NULL);
    TagTypeInfo& SetElementStr(bool Specified_, ChrString* Ptr_=NULL, bool KeepPtr_=false);
    TagTypeInfo& SetSingletonAllowed(bool Set_);
    TagTypeInfo& SetSingletonTag(bool Single_);
    TagTypeInfo& SetEmptyAllowed(bool Set_);
    TagTypeInfo& SetEmptyTag(bool Empty_);
    TagTypeInfo& SetContainerTag(bool Contain_);
    TagTypeInfo& SetPairEnderTag(bool PairEnder_);
    TagTypeInfo& SetDocType(int DocType_);
    TagTypeInfo& SetEofFound(bool Eof_);
    TagTypeInfo& SetTagAttr(int TagAttr_);
    TagTypeInfo& SetHasPairContent(bool Flag_);
    TagTypeInfo& AdjustTagCount(int DictTagPos_);
    TagTypeInfo& SetTagTypeFound(long Index_, bool SetTagCnt_);
    TagTypeInfo& SetTextBufferPtr(const ChrString* Ptr_);

    TagTypeInfo& CreateStackInfoNode(int DocType_, TagTypeInfo* TagPtr_, bool UnlinkTemp_,
                                     bool GenSearchDone_, bool SpecSearchDone_);
    TagTypeInfo& AssignSearchResult(Subscript Index_, Subscript Length_,
                                    Subscript MIndex_=0, Subscript MLength_=0,
                                    Subscript EIndex_=0, Subscript ELength_=0);

    TagTypeInfo& AdvanceReadPos();
    TagTypeInfo& ResetReadPos();
    TagTypeInfo& Reset(int Spec_=ALL);

    // Wrapper methods for TagCountData class methods : start
    const int* GiveTagCountArray(int Index_) const;
    int GiveTagCount(const char* TxtFndBrk_) const;
    int GiveBrkNesting(const char* TxtFndBrk_) const;
    int PushBrkNesting(int TagPos_, int Nest_);
    int PopBrkNesting(int TagPos_, int Nest_);
    int BrkIndexToTagType(int BrkIndex_) const;
    int BrkStringToIndex(const char* Brk_, int& BrkEnder_) const;
    int BrkStringToIndex(const char* StartBrk_, const char* EndBrk_,
                         int& PairEnder_) const;

    TagTypeInfo* AddToTagList(int TagPos_, TagTypeInfo* MatchingTag_, bool& Updated_);
    TagTypeInfo* MakeMatchTag(TagTypeInfo* StartTag_, bool SetMatchTag_, bool AddToList_);
    TagTypeInfo* ResetRequiredTags(TagTypeInfo* TagPtr_);
    TagTypeInfo* SetHtmlContentTagType(TagTypeInfo* TagPtr_, int ContentType_);
    TagTypeInfo* SetRequiredCond(TagTypeInfo* TagPtr_,
                                 int ReqTagOrder_, int ImpliedEnd_=-1,
                                 int TagCond_=0, int ImpliedEndCond_=0);
    TagTypeInfo* AddRequiredTag(TagTypeInfo* TagPtr_, TagTypeInfo* ReqTag_,
                                int ReqTagOrder_, bool MatchingEnd_=false);
    TagTypeInfo* AddRequiredTagElement(TagTypeInfo* TagPtr_, TagElement* ReqTag_,
                                       int ReqTagOrder_, bool MatchingEnd_=false);
    TagTypeInfo* SetRequiredTags(TagTypeInfo* TagPtr_, TagElement** Vect_,
                                 int ReqTagOrder_, bool MatchingEnd_=false);

    TagTypeInfo* UpdateTag(int TagPos_);
    TagTypeInfo* SetThisTag(int TagPos_);
    TagTypeInfo& ResetTagIterIndex();
    
    bool HasThisTag(int TagPos_) const;
    bool SetTagIterIndex(int TagIndex_);
    bool AdvanceTagIter();
    TagElement* GetCurrentTagElement();
    size_t TagListSize(int TagPos_) const;
    bool AtFirstTag() const;
    bool AtLastTag() const;
    // Wrapper methods for TagCountData class methods : end

    void RestoreTagSwitches();
    void InitCppLangTagSwitches(TagSwitch& Switch_);
    void InitCLangTagSwitches(TagSwitch& Switch_);
    void InitCommentTagSwitches(TagSwitch& Switch_);
    void InitNestedCommentTagSwitches(TagSwitch& Switch_);
    void InitSingleTagSwitches(TagSwitch& Switch_);
    void InitAllHtmlTagSwitches(TagSwitch& Switch_);
    void InitAllXmlTagSwitches(TagSwitch& Switch_);
    void InitStdTagSwitches(TagSwitch& Switch_);
    void InitEndTagSwitches(TagSwitch& Switch_);
    void InitScriptTagSwitches(TagSwitch& Switch_);
    void InitDocTagSwitches(TagSwitch& Switch_);
    void InitTagSwitches(TagSwitch& Switch_, int TagType_, int PairEnder_);
    void RetrieveIntraTagContent(bool Flag_);

    Subscript ReadPos() const;
    const ChrString& ElementStr() const;
    void DumpElementSeqListNode(ostream& os_, size_t fw_) const;
    void DumpStackNodeInfo(ostream& os_, size_t fw_=20) const;
    void DumpSearchResults(ostream& os_, size_t fw_=20) const;
    void DumpTagContentData(ostream& os_, size_t fw_=20) const;
    void DumpIntraTagContentData(ostream& os_, size_t fw_=20) const;
    void ShouldDumpContents(bool Flag_, ostream& os_, size_t fw_) const;

    // StackNodeInfo methods
    StackNodeInfo* HeadInfo(bool Reset_, bool NotErased_);
    StackNodeInfo* TailInfo(bool Reset_, bool NotErased_);
    StackNodeInfo* NextStackInfo(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    StackNodeInfo* PrevStackInfo(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    StackNodeInfo* SetStackInfoPtr(bool Set_);

    StackNodeInfo* CursorTagStkPtr();
    StackNodeInfo* TagStackPtr();
    StackNodeInfo* StackInfoPtr();

    void ShiftTagContents();

    // TagLinksSequencer methods
    TagLinksSequencer* Sequencer();
    const TagLinksSequencer* Sequencer() const;
    TagTypeInfo& SetSequencer(TagLinksSequencer* Ptr_);

    // HtmlSpecialCase methods
    HtmlSpecialCaseData* HtmlSpecialCase();
    const HtmlSpecialCaseData* HtmlSpecialCase() const;
    TagTypeInfo& SetHtmlSpecialCase(HtmlSpecialCaseData* Ptr_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
class QListNode
{
  public:
    enum
    {
      QLISTNODE         = 1,
      TEXTNODEDATA      = 2,
      NODENESTINGDATA   = 4,
      APPTEXTNODE       = 8,
      STACKNODEINFO     = 16,

      NO_DATA           = 0,
      INFONODE_DATA     = 16,
      STRING_DATA       = 32
    };

  protected:
    static size_t _NumErased; // number of erased stack nodes
    static size_t _StackSize; // global StackNodeInfo linked list size

    size_t _NodeID;           // Node ID of object
    QListNode* _NextBase;
    QListNode* _PrevBase;
    bool _Copied;
    bool _Erased;

  public:
    QListNode();
    QListNode(const QListNode& Obj_);
    virtual ~QListNode();

    // Returns TRUE if list is empty
    static bool IsEmpty();

    // Return size of list in number of elements
    static size_t Size();

    // Return number of stack nodes erased
    static size_t NumErased();

    virtual int ClassType() const;
    virtual QListNode& operator = (const QListNode& List_);

    virtual QListNode* Tail(int Type_=0);
    virtual const QListNode* Tail(int Type_=0) const;
    virtual QListNode* Head(int Type_=0);
    virtual const QListNode* Head(int Type_=0) const;

    virtual QListNode* RemoveHead(QListNode*& Head_);
    virtual QListNode* AppendHead(QListNode* Ptr_, bool NestTag_);

    virtual void SetNodeID(size_t ID_);
    virtual void SetCopied(bool Flag_);
    virtual void SetErased(bool Flag_);
    virtual bool Unlink();
    virtual void DeleteAll();

    virtual bool Preceeds(QListNode* EncTag_, bool OrSameAs_=false);
    virtual bool Follows(QListNode* EncTag_, bool OrSameAs_=false);
    virtual bool SameAs(QListNode* EncTag_);

    virtual QListNode* GiveNext();
    virtual const QListNode* Next() const;
    virtual QListNode* SetNext(QListNode* Ptr_);

    virtual QListNode* GivePrev();
    virtual const QListNode* Prev() const;
    virtual QListNode* SetPrev(QListNode* Ptr_);

    inline bool Erased() const
        { return _Erased; }
    inline bool IsCopied() const
        { return _Copied; }
    inline size_t NodeID() const
        { return _NodeID; }

    #if OVERLOAD_NEW
      void* operator new (size_t Bytes_);
      void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
    #endif
};

/****************************************************************************/
class TagCommentData;
class KeyValuePairData;

class TextNodeData : public QListNode
{
  protected:
    union
    {
      ChrString* _TextStr;
      StackNodeInfo* _NestedNode;
    };

    int _DataType;
    bool _DataSet;

  public:
    TextNodeData();
    TextNodeData(const TextNodeData& Obj_);
    virtual ~TextNodeData();

    static TextNodeData* MakeNode();
    static TextNodeData* MakeNode(const TextNodeData& Obj_);

    static TextNodeData* CopyNode(TextNodeData* TrgData_, TextNodeData* SrcData_);
    static TextNodeData* CopyTextNodeData(TextNodeData*& TrgData_,
                                          TextNodeData*& TrgDataHead_,
                                          size_t& TrgNumData_,
                                          TextNodeData* SrcData_,
                                          TextNodeData* SrcDataHead_,
                                          size_t SrcNumData_);

    virtual int ClassType() const;
    virtual QListNode& operator = (const QListNode& List_);

    void SetDataType(int Type_);
    void SetCTextStr(const char* Str_);
    void SetTextStr(const ChrString& Str_);
    void AppendCTextStr(const char* Str_);
    void AppendTextStr(const ChrString& Str_);
    void SetNestedNode(StackNodeInfo* NodePtr_);
    void ShowContents(const StackNodeInfo* ptr_, ostream& os_, size_t fw_=20, size_t* Index_=NULL) const;

    inline int DataType() const
        { return _DataType; }
    inline bool DataSet() const
        { return _DataSet; }
    inline void UnsetData()
        { _DataSet = false; }
    inline ChrString* GiveTextStr()
        { return ((_DataType == QListNode::STRING_DATA) ? _TextStr:NULL); }
    inline const ChrString* TextStr() const
        { return ((_DataType == QListNode::STRING_DATA) ? _TextStr:NULL); }

    inline StackNodeInfo* GiveNestedNode()
        { return ((_DataType == QListNode::INFONODE_DATA) ? _NestedNode:NULL); }
    inline const StackNodeInfo* NestedNode() const
        { return ((_DataType == QListNode::INFONODE_DATA) ? _NestedNode:NULL); }

    #if OVERLOAD_NEW
      void* operator new (size_t Bytes_);
      void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
    #endif
};

class AppTextNode : public QListNode
{
  protected:
    TagTypeInfo* _Parent;     // Referring Parent class object for this node
    StackNodeInfo* _RefNode;

    AppTextNode* _Next;
    AppTextNode* _Prev;

    mutable TextNodeData* _TextQ;
    mutable TextNodeData* _TextQhead;
    mutable TextNodeData* _TextQptr;
    mutable bool _TextQptrSet;
    size_t _NumTextData;

    bool _HasTagStk;     // TagLinksSequencer::_HeadTagStack != NULL
    bool _SglNode;       // Singleton or empty tag
    bool _StdNode;       // Standard opening tag of tag pair
    bool _AppendText;    // Append text string to _TextQhead->_TextStr
    bool _DoErase;       // Perform deferred node erasure
    int _PauseAppend;    // Pause appendind of text string upon encounter
                         // of a nested STDTAG and a nested ENDTAG

    bool _Transferred;
    AppTextNode* _TransferNode;

    void DumpThisTextNode(ostream& os_, size_t fw_=20,
                          const char* Str_=NULL) const;

  public:
    enum
    {
      TXTAPP_NOT_PAUSED = 0,
      NEED_STDTAG       = 1,
      STDTAG_APPENDED   = 2,
      NEED_ENDTAG       = 4,
      ENDTAG_APPENDED   = 8,
      TAG_COMPLETED     = 16
    };

    static const char* PauseAppendTypeToStr(int Type_);

    AppTextNode();
    AppTextNode(bool HasTagStk_, StackNodeInfo* RefNode_);
    AppTextNode(const AppTextNode& Obj_);
    virtual ~AppTextNode();

    // TextNodeData traversing methods : start
    TextNodeData* PopTextNode(bool Erase_, bool ConfirmErased_=false);
    TextNodeData* NextTextNode(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    TextNodeData* PrevTextNode(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    TextNodeData* PushTextNode(TextNodeData* Node_, bool Create_);
    TextNodeData* HeadTextNode(bool Reset_, bool NotErased_) const;
    TextNodeData* TailTextNode(bool Reset_, bool NotErased_) const;
    TextNodeData* SetTextNodePtr(bool Set_) const;

    void ConfirmTextNodeErased();

    void UnsetData(bool Head_=true, size_t Index_=0);
    bool DataSet(bool Head_=true, size_t Index_=0) const;
    ChrString* NodeTextStr(bool Head_=true, size_t Index_=0);
    StackNodeInfo* NodeNestedNode(bool Head_=true, size_t Index_=0);
    int NodeDataType(bool Head_=true, size_t Index_=0);
    TextNodeData* GiveTextNode(bool Head_=true, size_t Index_=0);
    const TextNodeData* TextNode(bool Head_=true, size_t Index_=0) const;

    inline bool HasThisRefNode(const StackNodeInfo* RefNode_) const
        { return (_RefNode == RefNode_); }
    inline bool HasThisParent(const TagTypeInfo* Parent_) const
        { return (_Parent == Parent_); }
    inline size_t TextQsize() const
        { return _NumTextData; }
    inline const TextNodeData* TextQhead() const
        { return _TextQhead; }
    inline const TextNodeData* TextQtail() const
        { return _TextQ; }
    inline TextNodeData* TextQhead()
        { return _TextQhead; }
    inline TextNodeData* TextQtail()
        { return _TextQ; }

    TextNodeData* CreateTextNodeData(ChrString* TextStr_=NULL,
                                     StackNodeInfo* NestedNode_=NULL,
                                     int DataType_=0);

    void ShowTextNodeContentsChkPtr(const StackNodeInfo* ptr_,
                                    ostream& os_, size_t fw_=20,
                                    bool ShowErased_=false) const;
    void ShowTextNodeContents(ostream& os_, size_t fw_=20,
                              bool ShowErased_=false) const;
    // TextNodeData traversing methods : end

    static AppTextNode* Make();
    static AppTextNode* Make(const AppTextNode& Obj_);

    // QListNode virtual methods : START
    virtual int ClassType() const;
    virtual QListNode& operator = (const QListNode& List_);

    // Set copied flag
    virtual void SetCopied(bool Flag_);

    // Set erase flag
    virtual void SetErased(bool Flag_);

    // Unlink and self-erase this node
    virtual bool Unlink();

    virtual QListNode* GiveNext();
    virtual const QListNode* Next() const;
    virtual QListNode* SetNext(QListNode* Ptr_);

    virtual QListNode* GivePrev();
    virtual const QListNode* Prev() const;
    virtual QListNode* SetPrev(QListNode* Ptr_);
    // QListNode virtual methods : END

    virtual StackNodeInfo* GiveRefNode();
    virtual const StackNodeInfo* RefNode() const;
    virtual StackNodeInfo* SetRefNode(StackNodeInfo* Ptr_);
    virtual const StackNodeInfo* RefNodeForwardLink() const;
    virtual const StackNodeInfo* RefNodeBackLink() const;

    void SetParent(TagTypeInfo* Ptr_);
    void SetDoErase(bool Flag_);
    void ExecDoErase();
    int LinkType() const;

    inline bool HasTentativeStr() const
        { return (_TransferNode && _RefNode && !_Transferred &&
                  _TextQhead && _TextQhead->DataType() == QListNode::STRING_DATA); }
    inline bool SglNode() const
        { return _SglNode; }
    inline bool StdNode() const
        { return _StdNode; }
    inline bool HasTagStk() const
        { return _HasTagStk; }
    inline bool Transferred() const
        { return _Transferred; }
    inline bool HasTransferNode() const
        { return (_TransferNode != NULL); }

    void SetTextStr(const ChrString* StrPtr_);
    void SetCTextStr(const char* StrPtr_);
    void AppendTextStr(const ChrString* StrPtr_);
    void AppendCTextStr(const char* StrPtr_);
    void SetNestedNode(StackNodeInfo* NodePtr_, bool Transfer_=false);
    void AppendNestedNode(StackNodeInfo* NodePtr_, bool Shifted_=false);

    // Current tag from text processing stack TagTypeInfo::_HeadCurTagStk;
    AppTextNode* AppTextPtr();

    // Appended tag content text methods : START
    void SetAppendText();
    void SetAppendText(bool Flag_);
    bool PopAppContStr(bool Flag_,
                       StackNodeInfo* EncNode_=NULL, bool PostLink_=false);
    void SetTransferNode(AppTextNode* Node_);
    void RemoveTentativeNode();
    void TransferText();
    AppTextNode* FindTagNode(StackNodeInfo* Node_);

    void ResumeTextAppend(StackNodeInfo* Ptr_);
    void PauseTextAppend(StackNodeInfo* Ptr_);

    inline const AppTextNode* TransferNode() const
        { return _TransferNode; }
    inline int AppendText() const
        { return _AppendText; }
    inline bool TagCompleted() const
        { return (_PauseAppend == TAG_COMPLETED); }
    inline bool TextAppendPaused() const
        { return (_PauseAppend == NEED_STDTAG ||
                  _PauseAppend == STDTAG_APPENDED ||
                  _PauseAppend == TAG_COMPLETED); }
    inline bool TextAppendResumed() const
        { return (_PauseAppend == NEED_ENDTAG ||
                  _PauseAppend == ENDTAG_APPENDED); }
    inline bool TextAppendNotPaused() const
        { return (_PauseAppend == TXTAPP_NOT_PAUSED); }
    inline void UnpauseTextAppend()
        { _PauseAppend = TXTAPP_NOT_PAUSED; }
    inline bool DoErase() const
        { return (!_Erased && _DoErase); }
    // Appended tag content text methods : END

    void DumpAppTextNode(const StackNodeInfo* ptr_, ostream& os_, size_t fw_=20) const;

    #if OVERLOAD_NEW
      void* operator new (size_t Bytes_);
      void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
    #endif
};

struct NodeDataTypes
{
  enum
  {
    THIS_TAG      = 1,
    START_TAG     = 2,
    END_TAG       = 3,
    PARENT_TAG    = 4
  };
};

class NodeNestingData : public NodeDataTypes, public QListNode
{
  protected:
    StackNodeInfo* _ThisTag;   // link to this current tag
    StackNodeInfo* _StartTag;  // link to pushed tag on tag stack for nested tags
    StackNodeInfo* _EndTag;    // link to popped end tag on tag stack for nested tags
    StackNodeInfo* _TagParent; // link to parent tag which contains this tag as nested tag

  public:
    NodeNestingData();
    NodeNestingData(const NodeNestingData& Obj_);
    virtual ~NodeNestingData();

    static NodeNestingData* MakeNode();
    static NodeNestingData* MakeNode(const NodeNestingData& Obj_);

    static NodeNestingData* CopyNode(NodeNestingData* TrgData_, NodeNestingData* SrcData_);
    static NodeNestingData* CopyNodeNestingData(NodeNestingData*& TrgData_,
                                                NodeNestingData*& TrgDataHead_,
                                                size_t& TrgNumData_,
                                                NodeNestingData* SrcData_,
                                                NodeNestingData* SrcDataHead_,
                                                size_t SrcNumData_);

    virtual int ClassType() const;
    virtual NodeNestingData& operator = (const QListNode& List_);

    void SetThisTag(StackNodeInfo* Ptr_);
    void SetStartTag(StackNodeInfo* Ptr_);
    void SetEndTag(StackNodeInfo* Ptr_);
    void SetTagParent(StackNodeInfo* Ptr_);

    void ShowContents(const StackNodeInfo* ptr_, ostream& os_, size_t fw_) const;

    inline StackNodeInfo* GiveThisTag()
        { return _ThisTag; }
    inline StackNodeInfo* GiveStartTag()
        { return _StartTag; }
    inline StackNodeInfo* GiveEndTag()
        { return _EndTag; }
    inline StackNodeInfo* GiveTagParent()
        { return _TagParent; }

    inline const StackNodeInfo* ThisTag() const
        { return _ThisTag; }
    inline const StackNodeInfo* StartTag() const
        { return _StartTag; }
    inline const StackNodeInfo* EndTag() const
        { return _EndTag; }
    inline const StackNodeInfo* TagParent() const
        { return _TagParent; }

    #if OVERLOAD_NEW
      void* operator new (size_t Bytes_);
      void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
    #endif
};

class StackNodeFinder : public NodeDataTypes
{
  private:
    size_t _NumNestedData;

    union
    {
      NodeNestingData* _NestedQ;
      const NodeNestingData* _ConstNestedQ;
    };

    union
    {
      StackNodeInfo* _StkNodePtr;
      const StackNodeInfo* _ConstStkNodePtr;
    };

  public:
    StackNodeFinder(StackNodeInfo* Ptr_);
    StackNodeFinder(const StackNodeInfo* cPtr_);

    NodeNestingData* FindNestingNode_MatchingPtr(int TagType_, StackNodeInfo* ptr,
                                                 StackNodeInfo* tpptr=NULL, bool* Found_=NULL);
    const NodeNestingData* FindNestingNode_MatchingPtr(int TagType_, const StackNodeInfo* ptr,
                                                       const StackNodeInfo* tpptr=NULL, bool* Found_=NULL) const;
    NodeNestingData* FindNestingNode_TagType(int TagType_, bool* Found_=NULL);
    const NodeNestingData* FindNestingNode_TagType(int TagType_, bool* Found_=NULL) const;

    NodeNestingData* FindNextNode_MatchingPtr(NodeNestingData* ptr, int TagType_, StackNodeInfo* snptr,
                                              StackNodeInfo* tpptr=NULL, bool* Found_=NULL);
    const NodeNestingData* FindNextNode_MatchingPtr(NodeNestingData* ptr, int TagType_, const StackNodeInfo* snptr,
                                                    const StackNodeInfo* tpptr=NULL, bool* Found_=NULL) const;
    NodeNestingData* FindNextNode_TagType(NodeNestingData* ptr, int TagType_, bool* Found_=NULL);
    const NodeNestingData* FindNextNode_TagType(NodeNestingData* ptr, int TagType_, bool* Found_=NULL) const;

    // NodeNestingData search functions
    inline NodeNestingData* FindThisNodeMatchingPtr(StackNodeInfo* ptr, bool* Found_=NULL)
        { return FindNestingNode_MatchingPtr(THIS_TAG, ptr, _StkNodePtr, Found_); }
    inline const NodeNestingData* FindThisNodeMatchingPtr(StackNodeInfo* ptr, bool* Found_=NULL) const
        { return FindNestingNode_MatchingPtr(THIS_TAG, ptr, _ConstStkNodePtr, Found_); }
    inline NodeNestingData* FindThisNestingNode(bool* Found_=NULL)
        { return FindNestingNode_TagType(THIS_TAG, Found_); }
    inline const NodeNestingData* FindThisNestingNode(bool* Found_=NULL) const
        { return FindNestingNode_TagType(THIS_TAG, Found_); }
    inline NodeNestingData* FindNextThisNode(NodeNestingData* ptr, bool* Found_=NULL)
        { return FindNextNode_TagType(ptr, THIS_TAG, Found_); }
    inline const NodeNestingData* FindNextThisNode(NodeNestingData* ptr, bool* Found_=NULL) const
        { return FindNextNode_TagType(ptr, THIS_TAG, Found_); }
    inline NodeNestingData* FindNextThisNodeMatchingPtr(NodeNestingData* ptr, StackNodeInfo* snptr, bool* Found_=NULL)
        { return FindNextNode_MatchingPtr(ptr, THIS_TAG, snptr, _StkNodePtr, Found_); }
    inline const NodeNestingData* FindNextThisNodeMatchingPtr(NodeNestingData* ptr, StackNodeInfo* snptr,bool* Found_=NULL) const
        { return FindNextNode_MatchingPtr(ptr, THIS_TAG, snptr, _ConstStkNodePtr, Found_); }

    inline NodeNestingData* FindStartTagNodeMatchingPtr(StackNodeInfo* ptr, bool* Found_=NULL)
        { return FindNestingNode_MatchingPtr(START_TAG, ptr, _StkNodePtr, Found_); }
    inline const NodeNestingData* FindStartTagNodeMatchingPtr(StackNodeInfo* ptr, bool* Found_=NULL) const
        { return FindNestingNode_MatchingPtr(START_TAG, ptr, _ConstStkNodePtr, Found_); }
    inline NodeNestingData* FindStartTagNestingNode(bool* Found_=NULL)
        { return FindNestingNode_TagType(START_TAG, Found_); }
    inline const NodeNestingData* FindStartTagNestingNode(bool* Found_=NULL) const
        { return FindNestingNode_TagType(START_TAG, Found_); }
    inline NodeNestingData* FindNextStartTagNode(NodeNestingData* ptr, bool* Found_=NULL)
        { return FindNextNode_TagType(ptr, START_TAG, Found_); }
    inline const NodeNestingData* FindNextStartTagNode(NodeNestingData* ptr, bool* Found_=NULL) const
        { return FindNextNode_TagType(ptr, START_TAG, Found_); }
    inline NodeNestingData* FindNextStartTagNodeMatchingPtr(NodeNestingData* ptr, StackNodeInfo* snptr, bool* Found_=NULL)
        { return FindNextNode_MatchingPtr(ptr, START_TAG, snptr, _StkNodePtr, Found_); }
    inline const NodeNestingData* FindNextStartTagNodeMatchingPtr(NodeNestingData* ptr, StackNodeInfo* snptr, bool* Found_=NULL) const
        { return FindNextNode_MatchingPtr(ptr, START_TAG, snptr, _ConstStkNodePtr, Found_); }

    inline NodeNestingData* FindEndTagNodeMatchingPtr(StackNodeInfo* ptr, bool* Found_=NULL)
        { return FindNestingNode_MatchingPtr(END_TAG, ptr, _StkNodePtr, Found_); }
    inline const NodeNestingData* FindEndTagNodeMatchingPtr(StackNodeInfo* ptr, bool* Found_=NULL) const
        { return FindNestingNode_MatchingPtr(END_TAG, ptr, _ConstStkNodePtr, Found_); }
    inline NodeNestingData* FindEndTagNestingNode(bool* Found_=NULL)
        { return FindNestingNode_TagType(END_TAG, Found_); }
    inline const NodeNestingData* FindEndTagNestingNode(bool* Found_=NULL) const
        { return FindNestingNode_TagType(END_TAG, Found_); }
    inline NodeNestingData* FindNextEndTagNode(NodeNestingData* ptr, bool* Found_=NULL)
        { return FindNextNode_TagType(ptr, END_TAG, Found_); }
    inline const NodeNestingData* FindNextEndTagNode(NodeNestingData* ptr, bool* Found_=NULL) const
        { return FindNextNode_TagType(ptr, END_TAG, Found_); }
    inline NodeNestingData* FindNextEndTagNodeMatchingPtr(NodeNestingData* ptr, StackNodeInfo* snptr, bool* Found_=NULL)
        { return FindNextNode_MatchingPtr(ptr, END_TAG, snptr, _StkNodePtr, Found_); }
    inline const NodeNestingData* FindNextEndTagNodeMatchingPtr(NodeNestingData* ptr, StackNodeInfo* snptr, bool* Found_=NULL) const
        { return FindNextNode_MatchingPtr(ptr, END_TAG, snptr, _ConstStkNodePtr, Found_); }

    inline NodeNestingData* FindParentNodeMatchingPtr(StackNodeInfo* ptr, bool* Found_=NULL)
        { return FindNestingNode_MatchingPtr(PARENT_TAG, ptr, _StkNodePtr, Found_); }
    inline const NodeNestingData* FindParentNodeMatchingPtr(StackNodeInfo* ptr, bool* Found_=NULL) const
        { return FindNestingNode_MatchingPtr(PARENT_TAG, ptr, _ConstStkNodePtr, Found_); }
    inline NodeNestingData* FindParentNestingNode(bool* Found_=NULL)
        { return FindNestingNode_TagType(PARENT_TAG, Found_); }
    inline const NodeNestingData* FindParentNestingNode(bool* Found_=NULL) const
        { return FindNestingNode_TagType(PARENT_TAG, Found_); }
    inline NodeNestingData* FindNextParentNode(NodeNestingData* ptr, bool* Found_=NULL)
        { return FindNextNode_TagType(ptr, PARENT_TAG, Found_); }
    inline const NodeNestingData* FindNextParentNode(NodeNestingData* ptr, bool* Found_=NULL) const
        { return FindNextNode_TagType(ptr, PARENT_TAG, Found_); }
    inline NodeNestingData* FindNextParentNodeMatchingPtr(NodeNestingData* ptr, StackNodeInfo* snptr, bool* Found_=NULL)
        { return FindNextNode_MatchingPtr(ptr, PARENT_TAG, snptr, _StkNodePtr, Found_); }
    inline const NodeNestingData* FindNextParentNodeMatchingPtr(NodeNestingData* ptr, StackNodeInfo* snptr, bool* Found_=NULL) const
        { return FindNextNode_MatchingPtr(ptr, PARENT_TAG, snptr, _ConstStkNodePtr, Found_); }

    #if OVERLOAD_NEW
      void* operator new (size_t Bytes_);
      void operator delete (void* Space_);

    #if	HAS_ARRAY_NEW
      void* operator new[] (size_t Bytes_);
      void operator delete[] (void* Space_);
    #endif
    #endif
};

// Tag stack node creation helper data class
class StackNodeInfo : public NodeDataTypes, public QListNode
{
  public:
    enum
    {
      PREVNEXT = 1,
      BACKFORW = 2,
      PAIRLINK = 4,
      MISMLINK = 8,
      NEXTCUR  = 16,
      APPTEXT  = 32,

      CONTENT_RESET       = 0,
      CONTENT_SHIFTED     = 2,
      CONTENT_SHOWN       = 4,
      CONTENT_APPENDED    = 8,
      CONTENT_TRANSFERRED = 16,
      CONTENT_RECEIVED    = 32,

      NO_ORDER      = 0,
      FORWARD_ORDER = 1,
      REVERSE_ORDER = 2
    };

    // from TagElement Class
    int _TagAttrib[TagEnums::MAX_FIELDS];

    // Set from passed parameters when creating StackNodeInfo object
    int _DocType;
    int _SearchType;

    // from TagSearchResult Class
    Subscript _NumDelims;          // Number of delims found
    bool _StartBrkFound;           // start bracket of tag found in search
    bool _EndBrkFound;             // end bracket of tag found in search
    bool _TagElementFound;         // element name of tag found in search
    ChrString _StartBrkStr;        // Target bracket string found in search
    ChrString _TagElementStr;      // Tag element string found in search
    ChrString _EndBrkStr;          // Matching bracket string found in search
    int _ContentType;              // Type of tag content that is retrieved
    bool _HasTagContents;          // Has tag contents
    bool _HasIntraTagContents;     // Has intratag contents
    ChrString _ContentStr;         // Tag content string
    ChrString _IntraContentStr;    // Intra Tag content string
    size_t _NumKeyValuePairs;      // Number of tag key value pairs
    ChrString** _TagKeyValuePairs; // Tag key value pair array
    bool _TagKVArrayCopied;        // Indicates pointer copy of KV pair array
    mutable int _ContentShifted;   // Indicates tag content string has been shifted

    // from TagTypeInfo Class
    TagTypeInfo* _Parent;             // Referring Parent class object for this node
    Ulong _SearchRun;                 // Search run number of referring parent object
    mutable TagElement* _ElementPtr;  // Tag element ptr of matching created TagElement object
    int _TagAttr;             // User set: Singleton / Pair / OptionalPair / Empty / Pair_Ender
    long _TagBrkIndex;        // in +2 increments for each tag, +1 for ending tag
    long _EndBrkIndex;        // Bracket index of ending tag bracket found in tag search
    long _AltIndex;           // Alternate bracket index

    long _StdTagCount;        // Standard open tag brk count (internal)
    long _NestedCmntTagCount; // Nested comment tag brk count (internal)
    long _CopenCmntTagCount;  // Open C comment tag brk count (internal)
    long _NestingError;       // Tag nesting error indicattor (internal)
    long _TagNestCond;        // Tag nesting condition

    bool _TagFound;           // Tag found indicator, set after search
    bool _ContentTypeTag;     // Tag contents allowed within tag brackets
    bool _NonContentTypeTag;  // Tag contents disallowed within tag brackets
    bool _HasPairContent;     // Has enclosed tag pair contents
    int _TagTypeFound;        // Tag type found indicator, set after search
    int _LinkType;            // ForwardLink-BackLink link type

    bool _SingletonTagFound; // Singleton type tag found, set after search
    bool _EmptyTagFound;     // Empty type tag found, set after search
    bool _ContainerTagFound; // Tag container type tag found
    bool _StdTagFound;       // Standard opening tag found
    bool _PairEnderFound;    // Ending tag of enclosing tag pair found
    bool _UnorderedTag;      // Unordered tag (non-content type tag)
    bool _PrevUnorderedTagFound;  // Previous search found an unordered tag (non-content type tag)

    // From TagTypeInfo Class: Additional StackNodeInfo related data members
    StackNodeInfo* _CursorStkTag; // Cursor matching tag at head of stack
    StackNodeInfo* _FoundTag;     // Found matching tag node from tag stack
    bool _FoundCursorTag;         // Found tag in search matches cursor stack tag
    bool _StkTagFound;            // Tag found within stack in search
    long _TagContValid;           // Tag content valid

    StackNodeInfo* _NCTagAddr; // address of next occuring non content tag
    long _NextSequenceNum;     // new sequence number to assign to the next encountered tag
    long _ParentSequenceNum;   // sequence number of this tag relative to enclosing parent tag
    long _SequenceNum;         // sequence number within horizontal sequence level
    long _CurLevel;            // current nesting level
    long _EncLevel;            // enclosed tags nesting level
    long _ChdLevel;            // previous parent nesting level
    long* _PallocChildren;     // number of enclosed children tags (parent allocated)
    long* _CrefChildren;       // number of enclosed children tags (child referenced)

    AppTextNode* _TextNode;       // Append tag content text list node
    TagElementStates* _TagStates; // TagStates structure created for each TagTypeInfo
                                  // that's made for each encountered tag in the document

    // StackNodeInfo class data members
    StackNodeInfo* _Prev;     // previous tag node pointer
    StackNodeInfo* _Next;     // next tag node pointer

    StackNodeInfo* _BackLink;    // back link to matching opening tag from closing tag
    StackNodeInfo* _ForwardLink; // forward link to matching closing tag from opening tag

    StackNodeInfo* _PrevCursor;  // previous cursor tag node pointer
    StackNodeInfo* _NextCursor;  // next cursor tag node pointer

    mutable NodeNestingData* _NestedQ;
    mutable NodeNestingData* _NestedQhead;
    mutable NodeNestingData* _NestedQptr;
    mutable bool _NestedQptrSet;
    size_t _NumNestedData;

    StackNodeInfo* _OverlappedPrevPair; // tag overlapped error, previous tag pair
    StackNodeInfo* _OverlappedNextPair; // tag overlapped error, next tag pair
    StackNodeInfo* _MismatchedEndPair;  // tag mismatch error, end pair with no matching start
    StackNodeInfo* _MatchInfoLink;      // matching info stack link

    bool _EndImplied;         // implied end set true if forward link is open tag
    bool _TagAppendDone;      // Appending text content within enclosed tag completed
    bool _Unlinked;           // Node is unlinked from stack linked list
    bool _CursorStkNodeFound; // _CursorStkTag node matches _FoundTag in search
    bool _NodeErased;         // Forward linked AppTextNode nodes are erased

    bool _HasCommentData;
    bool _HasKVPairData;

    union
    {
      TagCommentData* _CommentDataPtr;
      KeyValuePairData* _KVPairDataPtr;
    };

    static bool EquivTag(StackNodeInfo* CurStkTag_, StackNodeInfo* FoundTag_,
                         bool* Equal_=NULL, int Order_=0);
    static int FindLinkType(StackNodeInfo* StartTag_, StackNodeInfo* EndTag_);

    inline bool TagCompleted() const
        { return (_StartBrkFound && _EndBrkFound &&
                  _TagElementFound && _TagFound); }
    inline void SetCursorStkNodeFound(bool Flag_)
        { _CursorStkNodeFound = Flag_; }
    inline void SetMatchInfoLink(StackNodeInfo* Ptr_)
        { _MatchInfoLink = Ptr_; }
    inline void SetLinkType(int Type_)
        { _LinkType = Type_; }
    inline StackNodeInfo* GiveIfExistMatchingInfo()
        { return (_MatchInfoLink ? _MatchInfoLink:this); }

    void SetContentReceived() const;
    void SetContentTransferred() const;
    void SetContentAppended() const;
    void SetContentShown() const;
    void SetContentShifted() const;
    void ResetContentShifted() const;
    void SetUnlinked(bool Flag_);
    void DumpTagAttrib(ostream& os_, size_t fw_=20, bool WalkDump_=false) const;
    void DumpShiftBack(ostream& os_, size_t fw_=20, StackNodeInfo* EncNode_=NULL) const;
    void DumpKeyValuePairs(ostream& os_, size_t fw_=20, int Shift_=0) const;
    void SetCaseSensitive(bool Cs_);
    void SetTagEncountered(StackNodeInfo* EndTag_, int TagPos_, bool AutoLinkBack_);
    void ParseKVPairCommentData(TagSearchResult* Ptr_);

    bool EncTagIsEqual(StackNodeInfo* EncTag_, TagTypeInfo* Parent_, bool SkipOptTags_);
    bool CursorStkTagIsEqual(TagTypeInfo* Parent_, bool SkipOptTags_);
    bool IsOptionalOrNoContentTag() const;
    bool HasShiftedTagContents() const;

    long* SetChildrenCount(long val);
    inline long* SetChildrenRef(long* ref)
        { return ((_CrefChildren = ref)); }
    long* IncChildrenCount();
    long* IncChildrenRef();
    inline long* GetChildrenCount()
        { return _PallocChildren; }
    inline long* GetChildrenRef()
        { return _CrefChildren; }

    // NodeNestingData traversing methods : start
    NodeNestingData* PopNestingNode(bool Erase_, bool ConfirmErased_=false);
    NodeNestingData* NextNestingNode(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    NodeNestingData* PrevNestingNode(bool Reset_=false, bool Peek_=false, bool NotErased_=false);
    NodeNestingData* PushNestingNode(NodeNestingData* Node_, bool Create_);
    NodeNestingData* HeadNestingNode(bool Reset_, bool NotErased_) const;
    NodeNestingData* TailNestingNode(bool Reset_, bool NotErased_) const;
    NodeNestingData* SetNestingNodePtr(bool Set_) const;
    void ConfirmNestingNodeErased();

    StackNodeInfo* NestedTag(bool Head_=true, size_t Index_=0);
    StackNodeInfo* NestedEndTag(bool Head_=true, size_t Index_=0);
    StackNodeInfo* NestedTagParent(bool Head_=true, size_t Index_=0);

    NodeNestingData* GiveNestingNode(bool Head_=true, size_t Index_=0);
    const NodeNestingData* NestingNode(bool Head_=true, size_t Index_=0) const;

    inline size_t NestedQsize() const
        { return _NumNestedData; }
    inline const NodeNestingData* NestedQhead() const
        { return _NestedQhead; }
    inline const NodeNestingData* NestedQtail() const
        { return _NestedQ; }
    inline NodeNestingData* NestedQhead()
        { return _NestedQhead; }
    inline NodeNestingData* NestedQtail()
        { return _NestedQ; }
    inline long AltZero_NextSequenceNumber() const
        { return (_NextSequenceNum ? _NextSequenceNum:_SequenceNum); }
    inline long AltZero_ParentSequenceNumber() const
        { return (_ParentSequenceNum ? _ParentSequenceNum:_SequenceNum); }

    NodeNestingData* CreateNestingNodeData(StackNodeInfo* ThisTag_=NULL,
                                           StackNodeInfo* Tag_=NULL,
                                           StackNodeInfo* EndTag_=NULL,
                                           StackNodeInfo* TagParent_=NULL);

    void ShowNodeNestingContents(const StackNodeInfo* ptr_, ostream& os_, size_t fw_=20) const;
    // NodeNestingData traversing methods : end

  public:
    StackNodeInfo();
    StackNodeInfo(const StackNodeInfo& Obj_);
    virtual ~StackNodeInfo();

    static StackNodeInfo* Make();
    static StackNodeInfo* Make(const StackNodeInfo& Obj_);

    static const char* ContentShiftedTypeToStr(int Type_);
    static ChrString GiveTagStrFromSearch(StackNodeInfo* Ptr_);
    static ChrString GiveTagContentFromSearch(StackNodeInfo* Ptr_);
    static ChrString GiveIntraTagContentFromSearch(StackNodeInfo* Ptr_);
    static void ShowAppTextStr(ostream& os_, size_t fw_,
                               AppTextNode* Ptr_, int Type_);
    static void ShowTagStr(ostream& os_, size_t fw_, StackNodeInfo* Ptr_,
                           int Type_, const char* LinkStr_=NULL,
                           bool ShowObjectID_=false);
    static void ShowStackAddr(QListNode* StkPtr_, int Type_, const char* StkStr_,
                              int Dir_=1, bool DumpInfo_=false);

    // QListNode virtual methods : START
    virtual int ClassType() const;
    virtual QListNode& operator = (const QListNode& List_);

    virtual bool Unlink();

    virtual QListNode* RemoveHead(QListNode*& Head_);
    virtual QListNode* AppendHead(QListNode* Ptr_, bool NestTag_);

    virtual QListNode* Tail(int Type_=0);
    virtual const QListNode* Tail(int Type_=0) const;
    virtual QListNode* Head(int Type_=0);
    virtual const QListNode* Head(int Type_=0) const;

    virtual QListNode* GiveNext();
    virtual const QListNode* Next() const;
    virtual QListNode* SetNext(QListNode* Ptr_);

    virtual QListNode* GivePrev();
    virtual const QListNode* Prev() const;
    virtual QListNode* SetPrev(QListNode* Ptr_);
    // QListNode virtual methods : END

    virtual StackNodeInfo* GiveNextCursor();
    virtual const StackNodeInfo* NextCursor() const;
    virtual StackNodeInfo* SetNextCursor(StackNodeInfo* Ptr_);

    virtual StackNodeInfo* GivePrevCursor();
    virtual const StackNodeInfo* PrevCursor() const;
    virtual StackNodeInfo* SetPrevCursor(StackNodeInfo* Ptr_);

    bool InComment() const;
    bool InNestedComment() const;
    bool InScript() const;
    bool IsDocTypeTag() const;
    bool IsEndTag() const;
    bool IsSingleton() const;
    bool IsStdTag() const;
    bool IsCTag() const;
    bool IsCppTag() const;

    ChrString GiveTagStr() const;
    ChrString GiveIntraTagContent() const;

    int TagNesting(bool PairOnly_=true) const;

    inline ChrString* GiveKeyValuePair(int x)
        { return ((0 <= x && x < _NumKeyValuePairs && _TagKeyValuePairs) ?
                     _TagKeyValuePairs[x]:NULL); }
    inline int TagCount() const
        { return _TagAttrib[TagEnums::TAGCOUNT]; }
    inline ChrString GiveTagContent() const
        { return _ContentStr; }
    inline bool PairEnderFound() const
        { return _PairEnderFound; }
    inline bool StdTagFound() const
        { return _StdTagFound; }
    inline bool SingletonTagFound() const
        { return _SingletonTagFound; }
    inline bool EmptyTagFound() const
        { return _EmptyTagFound; }
    inline bool ContainerTagFound() const
        { return _ContainerTagFound; }
    inline long TagBrkIndex() const
        { return _TagBrkIndex; }
    inline long EndBrkIndex() const
        { return _EndBrkIndex; }
    inline TagElement* GiveTagElementPtr()
        { return _ElementPtr; }
    inline int TagTypeFound() const
        { return _TagTypeFound; }
    inline int LinkType() const
        { return _LinkType; }

    void AssignGlobalStackTag();
    inline void SetToGlobalStackTag()
        { _CursorStkTag = _Parent ? _Parent->Sequencer()->_CursorTagStkPtr:NULL; }
    inline StackNodeInfo* CursorStackTag()
        { return _CursorStkTag; }
    inline StackNodeInfo* FoundTag()
        { return _FoundTag; }
    inline bool FoundCursorTag() const
        { return _FoundCursorTag; }
    inline bool StackTagFound() const
        { return _StkTagFound; }

    inline bool AppendToContentStr() const
        { return (_TextNode &&
                  !_TextNode->HasTransferNode() &&
                  _TextNode->AppendText()); }
    inline bool AppendToTentativeStr() const
        { return (_TextNode &&
                  _TextNode->HasTransferNode() &&
                  !_TextNode->Transferred()); }
    inline long TagContentValid() const
        { return _TagContValid; }
    inline bool AppendTentativeToContentStr() const
        { return (_TextNode &&
                  _TextNode->HasTransferNode() &&
                  _TextNode->Transferred()); }

    inline const AppTextNode* AppText() const
        { return _TextNode; }
    inline AppTextNode* AppText()
        { return _TextNode; }

    inline StackNodeInfo* ForwardLink()
        { return _ForwardLink; }
    inline StackNodeInfo* BackLink()
        { return _BackLink; }

    inline StackNodeInfo* OverlappedNextPair()
        { return _OverlappedNextPair; }
    inline StackNodeInfo* OverlappedPrevPair()
        { return _OverlappedPrevPair; }
    inline StackNodeInfo* MismatchedEndPair()
        { return _MismatchedEndPair; }

    inline bool EndImplied() const
        { return _EndImplied; }
    inline bool TagAppendDone() const
        { return _TagAppendDone; }
    inline bool Unlinked() const
        { return _Unlinked; }
    inline bool CursorStkNodeFound() const
        { return _CursorStkNodeFound; }
    inline bool NodeErased() const
        { return _NodeErased; }

    inline bool StartBrkFound() const
        { return _StartBrkFound; }
    inline bool EndBrkFound() const
        { return _EndBrkFound; }
    inline bool TagElementFound() const
        { return _TagElementFound; }
    inline bool HasTagContents() const
        { return _HasTagContents; }
    inline bool HasIntraTagContents() const
        { return _HasIntraTagContents; }
    inline size_t NumKeyValuePairs() const
        { return _NumKeyValuePairs; }

    inline long ParentSequenceNumber() const
        { return _ParentSequenceNum; }
    inline long NextSequenceNumber() const
        { return _NextSequenceNum; }
    inline long SequenceNumber() const
        { return _SequenceNum; }
    inline long ParentNumChildren() const
        { return (_CrefChildren ? *_CrefChildren:0); }
    inline long NumChildren() const
        { return (_PallocChildren ? *_PallocChildren:0); }
    inline long ChildNestLevel() const
        { return _ChdLevel; }
    inline long NextNestLevel() const
        { return _EncLevel; }
    inline long NestLevel() const
        { return _CurLevel; }

    inline bool IsContentReset() const
        { return (_ContentShifted == CONTENT_RESET); }
    inline bool IsContentShifted() const
        { return (_ContentShifted & CONTENT_SHIFTED); }
    inline bool IsContentShown() const
        { return (_ContentShifted & CONTENT_SHOWN); }
    inline bool IsContentAppended() const
        { return (_ContentShifted & CONTENT_APPENDED); }
    inline bool IsContentTransferred() const
        { return (_ContentShifted & CONTENT_TRANSFERRED); }
    inline bool IsContentReceived() const
        { return (_ContentShifted & CONTENT_RECEIVED); }
    inline bool IsContentReady() const
        { return (IsContentReceived() &&
                 (HasTagContents() || HasIntraTagContents() || NumKeyValuePairs())); }
    inline bool IsContentTypeTag() const
        { return (_TagFound && _ContentTypeTag); }
    inline bool IsNonContentTypeTag() const
        { return (_TagFound && _NonContentTypeTag); }
    inline bool PrevUnorderedTagFound() const
        { return _PrevUnorderedTagFound; }
    inline bool UnorderedTag() const
        { return _UnorderedTag; }

    inline bool HasCommentData() const
        { return _HasCommentData; }
    inline bool HasKVPairData() const
        { return _HasKVPairData; }
    inline KeyValuePairData* GiveKeyValuePairData()
        { return (_HasKVPairData ? _KVPairDataPtr:NULL); }
    inline TagCommentData* GiveCommentData()
        { return (_HasCommentData ? _CommentDataPtr:NULL); }

    inline size_t GiveFieldWidth() const
        { return (_Parent ? _Parent->GiveFieldWidth():20); }
    inline ostream* GiveOutput()
        { return (_Parent ? _Parent->GiveOutput():&std::cout); }
    inline istream* GiveInput()
        { return (_Parent ? _Parent->GiveInput():&std::cin); }
    inline const ostream* GiveOutput() const
        { return (_Parent ? _Parent->GiveOutput():&std::cout); }
    inline const istream* GiveInput() const
        { return (_Parent ? _Parent->GiveInput():&std::cin); }

    const TagElementStates* TagElementState(bool FromElmPtr_) const;
    TagElementStates* GiveTagElementState(bool FromElmPtr_);
    const TagElementStates* SourceTagElementState(bool FromElmPtr_) const;
    TagElementStates* GiveSourceTagElementState(bool FromElmPtr_);

    void CopyFromTagElement(TagElement* Ptr_);
    void CopyFromTagSearchResult(TagSearchResult* Ptr_);
    void CopyFromTagTypeInfo(TagTypeInfo* Ptr_);
    void SetFromParameters(int DocType_, bool GenSearchDone_, bool SpecSearchDone_);

    void CopyTagTypeStackNodeData(TagTypeInfo* Ptr_);
    void SetStkNodeSearchData(TagTypeInfo* Ptr_);
    void CopyStackNodeDataFromPtr(StackNodeInfo* Ptr_);
    void SetStkNodeSearchDataFromPtr(StackNodeInfo* Ptr_);

    // Item/Node Removal and Insertion methods
    StackNodeInfo* RemoveHeadCursor(StackNodeInfo*& Head_);
    void RemovePrevTentStr(); // new

    // Calls the Push method to add the node at the head of the list.
    StackNodeInfo* ShiftBackContents(StackNodeInfo* EncNode_);
    StackNodeInfo* ShiftTagContents(StackNodeInfo* EncNode_,
                                    int TagEndDir_, int TagEndVal_, int Level_);
    StackNodeInfo* AppendHeadCursor(StackNodeInfo* Ptr_);
    void ResumeTextAppend(StackNodeInfo* Ptr_);
    void PauseTextAppend(StackNodeInfo* Ptr_);

    // Matching tags linking methods
    bool HasLinks(bool BiDir_, bool StartLink_);
    bool IsLinked(StackNodeInfo* Tag_, bool BiDir_);
    StackNodeInfo* LinkForward(StackNodeInfo* EndTag_, bool AutoLinkBack_=false);
    StackNodeInfo* LinkBackward(StackNodeInfo* StartTag_);

    // Tag comparison method
    bool TagIsEqual(StackNodeInfo* EncTag_, TagTypeInfo* Parent_, bool SkipOptTags_);
    bool TagStringEqual(StackNodeInfo* EncTag_);

    // Current tag from text processing stack TagTypeInfo::_HeadCurTagStk;
    StackNodeInfo* CursorTagStkPtr();
    StackNodeInfo* TagStackPtr();
    StackNodeInfo* StackInfoPtr();
    Subscript ReadPos() const;

    // Tag node found from TagIsEqual comparison method execution
    StackNodeInfo* GiveNodeFound();

    // Set tag nesting condition
    void SetTagNestCond(long NestCond_, StackNodeInfo* EncTag_);

    // SetOverlapped tag pairs
    void SetOverlappedPairs(StackNodeInfo* Prev_, StackNodeInfo* Next_);
    void SetMismatchedEndPair(StackNodeInfo* EncTag_);

    // Current node matches with referring parent object pointer
    bool HasThisParent(const TagTypeInfo* Ptr_) const;
    bool HasThisParentAndSearchNum(const TagTypeInfo* Ptr_) const;
    bool IsParentOf(const TagElementStates* Child_) const;
    bool IsRefNodeOf(const AppTextNode* ReferFrom_) const;

    // Appended tag content text methods : START
    TagElementStates* MakeTagElementStates();
    AppTextNode* MakeAppTextNode(bool HaveTagStk_);
    AppTextNode* MakeAppTentTextNode(bool HaveTentStk_);
    void SetAppTextNode(AppTextNode* NodePtr_);
    bool PopAppContStr(bool Flag_,
                       StackNodeInfo* EncNode_=NULL, bool PostLink_=false);
    void SetOpStTagToUnmatchedEndData(StackNodeInfo* CurNode_);
    void SetStdTagToUnmatchedEndData(StackNodeInfo* CurNode_);
    void SetSglTagToUnmatchedEndData(StackNodeInfo* CurNode_);
    void SetMatchedEndTagToUnmatchedEndData(StackNodeInfo* CurNode_);
    void SetOpStTagToOpStTagData(StackNodeInfo* CurNode_, bool& NodePopped_);
    void SetXToSglTagData(StackNodeInfo* CurNode_);
    void SetXToStdTagData(StackNodeInfo* CurNode_);
    void SetStdTagToStdTagData(StackNodeInfo* CurNode_);
    void SetStdTagToSglTagData(StackNodeInfo* CurNode_);
    void SetSglTagToSglTagData(StackNodeInfo* CurNode_);
    void SetSglTagToMatchedEndData(StackNodeInfo* CurNode_);
    void SetStdTagToMatchedEndData(StackNodeInfo* CurNode_);
    void SetStdTagData();
    void SetSingleTagData();
    void SetNonContentTagData();
    void SetNonContentTagData(StackNodeInfo* CurNode_);
    void IncNestLevel();

    void SetTagElementState() const;
    void SetSourceTagElementState() const;
    void SetSourceTagElementState(TagElementStates* Ptr_) const;
    void SetPrevUnorderedTag(bool PrevSearchDone_, bool GenSrchWasDone_,
                             int GenSearchFailed_, Ulong PrevSearchRun_,
                             Ulong CurrSearchRun_, bool NonContentTypeTagFound_,
                             bool ContentTypeTagFound_);
    StackNodeInfo* FindTagNode(StackNodeInfo* Node_);

    int ValidityChanged() const;
    int ReqTagCondChanged() const;
    int ImpliedEndCondChanged() const;

    long ReqTagFoundIndex() const;
    bool ImpliedTagFound() const;
    int ReqTagDirFound() const;
    const TagElementStates* EndTag() const;
    int EndTagCond() const;

    bool RequiredTagCondNotMet() const;
    bool RequiredTagCondMet() const;
    bool NoRequiredTag() const;
    bool CurrentTagCondMet() const;

    bool AtReqTagCondStopState(bool PartialMet_, bool FailOnly_=false) const;
    bool TagAtImpliedEnd() const;
    bool TagIsValid() const;
    // Appended tag content text methods : END

    // TagLinksSequencer methods
    TagLinksSequencer* Sequencer();
    const TagLinksSequencer* Sequencer() const;
    StackNodeInfo& SetSequencer(TagLinksSequencer* Ptr_);

    // HtmlSpecialCase methods
    HtmlSpecialCaseData* HtmlSpecialCase();
    const HtmlSpecialCaseData* HtmlSpecialCase() const;
    StackNodeInfo& SetHtmlSpecialCase(HtmlSpecialCaseData* Ptr_);

    void DumpAppTextNodeChkPtr(const StackNodeInfo* ptr_, ostream& os_, size_t fw_) const;
    void DumpStackNodeInfo(ostream& os_, size_t fw_=20, bool WalkDump_=false) const;
    void DumpTagContentData(ostream& os_, size_t fw_=20, bool ShowUnshifted_=false, int Shift_=0) const;
    void TagInfoListWalkDump(ostream& os_, size_t fw_=20);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
class TagDictionary : public TagEnums
{
  public:
    enum
    {
      NO_FAIL   = 0,
      UNREC_TAG = 1,
      NOT_A_TAG = 2
    };

  protected:
    long _TagStrIndex[2];  // Index of vector for tags of same type (search result)
    long _TagTypeIndex;    // Index of matrix for tag type vectors
                           //   1 : start tag vector (TagEnums::OPENING)
                           //   0 : end tag vector (TagEnums::CLOSING)

    int _DocType;          // Document type: tag type allowed; html, xml, cpp...
    bool _EofFound;        // End of file marker found
    int _TagType;          // Html, Xml, Cpp or C style tag switches
    
    long _StdVectorMax;    // Maximum capacity of start tag vector
    long _StdVectorLength; // Vector length of array containing start tags

    long _EndVectorMax;    // Maximum capacity of end tag vector
    long _EndVectorLength; // Vector length of array containing end tags

    TagTypeInfo::TagSwitch _TagSwitch;
    const ChrString* _TextBufferPtr;
    TagTypeInfo* _GenSearchTag;
    mutable TagLinksSequencer* _Sequencer;
    mutable bool _SequencerCopied;
    mutable HtmlSpecialCaseData* _HtmlSpecialCase;
    mutable bool _HtmlSpecialCaseCopied;

    FileReader* _Fr;
    TagTypeInfo*** _TagInfoMatrix;        // TagMatrix containing array of tag vectors (size == 2)
    TagSearchResult* _SearchResult;       // Search results for unspecified or specific tags
    bool _HasTag;                         // Has searched for tag in dictionary
    bool _DictSearchDone;                 // Indicates search of tag dictionary is done
    bool _ContentTypeTag;                 // Tag contents allowed within tag brackets
    bool _NonContentTypeTag;              // Tag contents disallowed within tag brackets
    bool _TagCreated;                     // Tag or matching tag created successfully
    bool _GenSearchDone;                  // General tag search performed successfully
    int  _GenSearchFailed;                // General search failure code
    bool _SpecSearchDone;                 // Specific tag search performed successfully
    int  _SpecSearchFailed;               // Specific tag search failure code
    bool _HtmlDictionaryCreated;          // Indicates if html tag dictionary has been created
    bool _PrevUnorderedTag;               // Previous search found an unordered tag (non-content type tag)
    mutable bool _DumpTagContents;        // Dump tag contents after advancing read buffer position

    void DumpUnrecognizedTag(TagTypeInfo* TagPtr_);
    void SetHtmlTagErrors(StackNodeInfo* EncNode_);
    void ProcessRequiredTagCond(StackNodeInfo* EncNode_);
    void MakeHtmlTagDictionary();
    void DetermineTagAttrib(TagTypeInfo* TagInfop_,
            int TagPos_, int AppAttr_, int& TagAttr_,
            bool& EmptyTag_, bool& Singleton_, bool& Empty_, bool& Container_,
            bool& PairContent_, const char* StartBrk_, const char* EndBrk_);

    TagTypeInfo* SetHtmlContentTagType(TagTypeInfo* TagPtr_, int ContentType_);
    TagTypeInfo* SetRequiredTagCond(TagTypeInfo* TagPtr_,
                                    int ReqTagOrder_, int ImpliedEnd_=-1,
                                    int TagCond_=0, int ImpliedEndCond_=0);
    TagTypeInfo* SetupStdTagPairContentTagAux(TagTypeInfo* TagPtr_, int TagPos_, bool NoSurrPair_,
                                              const ChrString& ElementStr_, int ImpliedEnd_=-1,
                                              int HtmlContentType_=-1);

    void EraseVectorContents();
    void CopyVectors(const TagDictionary& Obj_);
    TagTypeInfo** GrowVector(int TagPos_, long Incr_=10);

    bool HasThisTag(int TagPos_, TagTypeInfo* TagInfop_, bool Insertion_,
                    int TagType_=0, bool ForbiddenUnspec_=false);
    TagTypeInfo* CreatedTag();

  public:
    TagDictionary();
    TagDictionary(const TagDictionary& Obj_);
    ~TagDictionary();

    static TagDictionary* Make();
    static TagDictionary* Make(const TagDictionary& Obj_);

    inline long TagsVectorMax(int TagPos_) const
        { return ((TagPos_ == TagEnums::OPENING) ?
                       _StdVectorMax:
                       _EndVectorMax); }
    inline long TagsVectorLength(int TagPos_) const
        { return ((TagPos_ == TagEnums::OPENING) ?
                       _StdVectorLength:
                       _EndVectorLength); }
    inline long TagStrIndex(int TagPos_) const
        { return ((TagPos_ == TagEnums::OPENING ||
                   TagPos_ == TagEnums::CLOSING) ?
                       _TagStrIndex[TagPos_]:0); }
    inline long TagTypeIndex() const
        { return _TagTypeIndex; }
    inline TagTypeInfo*** GiveTagInfoMatrix()
        { return _TagInfoMatrix; }

    inline int DocTypeSpecified() const
        { return _DocType; }
    inline bool EofFound() const
        { return _EofFound; }
    inline bool ShouldDumpContents() const
        { return _DumpTagContents; }
    inline const ChrString* TextBufferPtr() const
        { return _TextBufferPtr; }
    inline TagSearchResult* SearchResult()
        { return _SearchResult; }
    inline const TagSearchResult* SearchResult() const
        { return _SearchResult; }
    inline bool GenSearchDone() const
        { return _GenSearchDone; }
    inline bool SpecSearchDone() const
        { return _SpecSearchDone; }
    inline TagDictionary& ResetEofFound()
        { return SetEofFound(false); }
    inline bool NonContentTypeTagFound(bool Found_=true) const
        { return (((Found_ && _DictSearchDone && !_HasTag) || !Found_) &&
                  !_ContentTypeTag && _NonContentTypeTag); }
    inline bool ContentTypeTagFound(bool Found_=true) const
        { return (((Found_ && _DictSearchDone && _HasTag) || !Found_) &&
                  _ContentTypeTag && !_NonContentTypeTag); }
    inline bool PrevUnorderedTag() const
        { return _PrevUnorderedTag; }
    inline bool PrevTagInfoSet() const
        { return Sequencer()->PrevTagInfoSet(); }

    bool TagFound(bool AllTags_=false) const;

    TagDictionary& operator = (const TagDictionary& Obj_);

    TagTypeInfo* GeneralSearchTag();
    TagTypeInfo* DictionarySearchTag();
    TagCountData* GiveTagCountData(bool ChkTagFnd_=true);
    const TagTypeInfo* GeneralSearchTag() const;
    const TagTypeInfo* DictionarySearchTag() const;
    bool HasEntries(int TagPos_, TagTypeInfo* TagInfop_) const;

    bool TagStrLookupEqual(int TagPos_, const ChrString& TagElmStr_,
                           ChrString* StartBrk_=NULL, ChrString* EndBrk_=NULL);
    bool TagLookupEqual(int TagPos_, TagTypeInfo* TagInfop_);

    TagDictionary& Reset(bool ReadPos_, bool TagSwitch_, int DocType_=0);
    TagDictionary& SetDocType(int DocType_);
    TagDictionary& SetEofFound(bool Eof_);
    TagDictionary& SetTextBufferPtr(const ChrString* Ptr_);

    TagTypeInfo* CreateTag(int TagPos_, bool Empty_, bool Container_, int AppAttr_,
                           const char* StartBrk_, const ChrString& Element_,
                           const char* EndBrk_, TagTypeInfo* MatchingTag_);
    TagTypeInfo* MakeMatchTag(TagTypeInfo* StartTag_, bool SetMatchTag_, bool AddToList_);

    const TagSearchResult* SearchForTag(int TagType_=0);
    const TagSearchResult* SearchForTag(int TagPos_, bool Empty_, bool Container_, int AppAttr_,
                                        const char* StartBrk_, const ChrString& Element_,
                                        const char* EndBrk_);

    TagTypeInfo* SetupHtmlDocSpecialTags(TagTypeInfo* TagPtr_);
    TagTypeInfo* SetupHtmlContentTag(TagTypeInfo* TagPtr_, int ImpliedEnd_=-1);
    TagTypeInfo* SetupHeadContentTag(TagTypeInfo* TagPtr_, int ImpliedEnd_=-1);
    TagTypeInfo* SetupBodyContentTag(TagTypeInfo* TagPtr_, int ImpliedEnd_=-1);
    TagTypeInfo* SetupStdTagPairContentTag(TagTypeInfo* TagPtr_, int TagPos_,
                                           const ChrString& ElementStr_, int ImpliedEnd_=-1);
    TagTypeInfo* SetupMatchingEndTag(TagTypeInfo* TagPtr_);
    TagTypeInfo* AddRequiredAfterTag(TagTypeInfo* TagPtr_, int TagPos_, const ChrString& ElementStr_);
    TagTypeInfo* AddRequiredBeforeTag(TagTypeInfo* TagPtr_, int TagPos_, const ChrString& ElementStr_);

    void DumpDictionary(ostream& os_, size_t fw_=20) const;
    void DumpSearchResults(ostream& os_, size_t fw_=20) const;
    void DumpTagContentData(ostream& os_, size_t fw_=20) const;
    void DumpIntraTagContentData(ostream& os_, size_t fw_=20) const;
    void TagInfoListWalkDump(ostream& os_, size_t fw_=20);
    void ShouldDumpContents(bool Flag_) const;

    TagDictionary& AssignSearchResult(Subscript Index_, Subscript Length_,
                                      Subscript MIndex_=0, Subscript MLength_=0,
                                      Subscript EIndex_=0, Subscript ELength_=0);
    TagDictionary& AdvanceReadPos();
    TagDictionary& ResetReadPos();
    Subscript ReadPos() const;

    // TagLinksSequencer methods
    TagLinksSequencer* Sequencer();
    const TagLinksSequencer* Sequencer() const;
    TagDictionary& SetSequencer(TagLinksSequencer* Ptr_);

    StackNodeInfo* ResetStackPtr(int StackType_, int EndType_, bool NotErased_=false);
    StackNodeInfo* PrevStackPtr(int StackType_, bool NotErased_=false);
    StackNodeInfo* NextStackPtr(int StackType_, bool NotErased_=false);

    // HtmlSpecialCase methods
    HtmlSpecialCaseData* HtmlSpecialCase();
    const HtmlSpecialCaseData* HtmlSpecialCase() const;
    TagDictionary& SetHtmlSpecialCase(HtmlSpecialCaseData* Ptr_);

    void RestoreTagSwitches();
    void InitCppLangTagSwitches();
    void InitCLangTagSwitches();
    void InitAllHtmlTagSwitches();
    void InitAllXmlTagSwitches();

    // Current tag from text processing stack TagTypeInfo::_HeadCurTagStk;
    inline StackNodeInfo* CursorTagStkPtr()
        { return (_GenSearchTag ? _GenSearchTag->CursorTagStkPtr():NULL); }
    inline StackNodeInfo* TagStackPtr()
        { return (_GenSearchTag ? _GenSearchTag->TagStackPtr():NULL); }
    inline StackNodeInfo* StackInfoPtr()
        { return (_GenSearchTag ? _GenSearchTag->StackInfoPtr():NULL); }

    // new
    void WriteHtmlTagDictionary(const char* OutFile_);
    void ReadHtmlTagDictionary(const char* InFile_);

    void RetrieveIntraTagContent(bool Flag_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
// class TagStringData forward declaration
//
class TagStringData;

class TagParseInfo : public TagEnums
{
  protected:
    TagStringData* _Parent;
    
    TagTypeInfo* _TagInfo;           // Retrieved from TagDictionary object
    TagDictionary* _Dictionary;      // pointer to tag dictionary object
    TagCommentData* _TagCommentData; // pointer to comment data object
    KeyValuePairData* _KeyValueData; // pointer to key value pair data object
    const TagSearchResult* _SearchResult;  // returned from search operation
    
    int _DocType;      // Document type: tag type allowed; html, xml, cpp...
    bool _EofFound;
  
    // Indicators set by object, parsing data returned
    bool _IsHtmlTag;
    bool _IsXHtmlTag;
    bool _IsXmlTag;
    bool _IsCppTag;
    bool _IsCTag;
    bool _IsNonStdTag;
    bool _IsSingletonTag;  // <tag> with no matching end tag </tag> allowed
    bool _IsEmptyTag;      // <tag/> with no matching end tag </tag> allowed
    bool _IsContainerTag;  // Tag container type tag

    int _TagError;     // Indicator: tag error: pending / valid / error / tagerr_type...
    int _AnchorType;   // Indicator: tag position: Opening / Closing / Forbidden
    int _FoundTagType; // Indicator: Actually found tag type == _TagInfo->_TagTypeFound

    long _AltIndex;        // Alternate index into brackets vector of found tag
    long _TagBrkIndex;     // Index into brackets vector of found tag
    long _EndBrkIndex;     // Bracket index of ending tag bracket
    long _TagStrIndex[2];  // Index of vector for tags of same type
    long _TagTypeIndex;    // Index of matrix for tag type vectors
    bool _TagSearchDone;

    bool SetTagType();
    void SetTagError(bool Specified_, int TagPos_, bool Empty_);

  public:
    TagParseInfo(TagStringData* Parent_);
    TagParseInfo(const TagParseInfo& Obj_);

    static TagParseInfo* Make(TagStringData* Parent_);
    static TagParseInfo* Make(const TagParseInfo& Obj_);

    TagParseInfo& operator = (const TagParseInfo& Obj_);

    inline void SetParent(TagStringData* Parent_)
        { _Parent = Parent_; }
    inline bool HasThisParent(const TagStringData* Parent_) const
        { return (_Parent == Parent_); }
    inline bool IsHtmlTag() const
        { return _IsHtmlTag; }
    inline bool IsXHtmlTag() const
        { return _IsXHtmlTag; }
    inline bool IsXmlTag() const
        { return _IsXmlTag; }
    inline bool IsCppTag() const
        { return _IsCppTag; }
    inline bool IsCTag() const
        { return _IsCTag; }
    inline bool IsNonStdTag() const
        { return _IsNonStdTag; }
    inline bool IsSingletonTag() const
        { return _IsSingletonTag; }
    inline bool IsEmptyTag() const
        { return _IsEmptyTag; }
    inline bool IsContainerTag() const
        { return _IsContainerTag; }
    inline int DocTypeSpecified() const
        { return _DocType; }
    inline bool EofFound() const
        { return _EofFound; }
    inline bool ShouldDumpContents() const
        { return (_Dictionary ? _Dictionary->ShouldDumpContents():false); }
    inline bool ShowDebugDump() const
        { return TagTypeInfo::ShowDebugDump(); }
    inline bool ShowErasedNodes() const
        { return TagTypeInfo::ShowErasedNodes(); }

    inline long TagStrIndex(int TagPos_) const
        { return ((TagPos_ == TagEnums::OPENING ||
                   TagPos_ == TagEnums::CLOSING) ?
                       _TagStrIndex[TagPos_]:0); }
    inline long TagTypeIndex() const
        { return _TagTypeIndex; }
    inline long TagBrkIndex() const
        { return _TagBrkIndex; }
    inline const TagSearchResult* SearchResult() const
        { return _SearchResult; }
    inline TagParseInfo& ResetEofFound()
        { return SetEofFound(false); }
    inline bool PrevUnorderedTag() const
        { return (_Dictionary ? _Dictionary->PrevUnorderedTag():false); }
    inline bool PrevTagInfoSet() const
        { return (_Dictionary ? _Dictionary->PrevTagInfoSet():false); }

    // Current tag from text processing stack TagTypeInfo::_HeadCurTagStk;
    inline StackNodeInfo* CursorTagStkPtr()
        { return (_Dictionary ? _Dictionary->CursorTagStkPtr():NULL); }
    inline StackNodeInfo* TagStackPtr()
        { return (_Dictionary ? _Dictionary->TagStackPtr():NULL); }
    inline StackNodeInfo* StackInfoPtr()
        { return (_Dictionary ? _Dictionary->StackInfoPtr():NULL); }

    int TagParseError() const;
    int TagTypeFound() const;
    int AnchorTypeFound() const;

    TagStringData* Parent();
    TagTypeInfo* GetTagInfo();
    TagDictionary* GiveTagDictionary();
    KeyValuePairData* GiveKeyValuePairData();
    TagCommentData* GiveCommentData();
    TagCountData* GiveTagCountData(bool ChkTagFnd_=true);
    const ChrString* TextBufferPtr() const;

    // TagLinksSequencer methods
    StackNodeInfo* ResetStackPtr(int StackType_, int EndType_, bool NotErased_=false);
    StackNodeInfo* PrevStackPtr(int StackType_, bool NotErased_=false);
    StackNodeInfo* NextStackPtr(int StackType_, bool NotErased_=false);

    TagLinksSequencer* Sequencer();
    const TagLinksSequencer* Sequencer() const;

    // HtmlSpecialCase methods
    HtmlSpecialCaseData* HtmlSpecialCase();
    const HtmlSpecialCaseData* HtmlSpecialCase() const;

    // Sets document type
    TagParseInfo& SetParseHtmlTag(bool IsTag_);
    TagParseInfo& SetParseXHtmlTag(bool IsTag_);
    TagParseInfo& SetParseXmlTag(bool IsTag_);
    TagParseInfo& SetParseCppTag(bool IsTag_);
    TagParseInfo& SetParseCTag(bool IsTag_);

    // Sets data member pointers
    TagParseInfo& SetTagDictionary(TagDictionary* Ptr_);
    TagParseInfo& SetKeyValuePairData(KeyValuePairData* Ptr_);
    TagParseInfo& SetCommentData(TagCommentData* Ptr_);

    TagParseInfo& Reset(bool ReadPos_, bool TagSwitch_, int DocType_=0);
    TagParseInfo& SetDocType(int DocType_);
    TagParseInfo& SetEofFound(bool Eof_);
    TagParseInfo& SetTextBufferPtr(const ChrString* Ptr_);
    TagParseInfo& AssignSearchResult(Subscript Index_, Subscript Length_,
                                     Subscript MIndex_=0, Subscript MLength_=0,
                                     Subscript EIndex_=0, Subscript ELength_=0);

    virtual TagParseInfo& AdvanceReadPos();
    virtual TagParseInfo& ResetReadPos();
    Subscript ReadPos() const;

    // Set tag debug and output dump options
    // Parsing output dump methods
    void SetShowTagDump(bool DebugDump_, bool ShowErased_, bool ContentDump_);
    void SetShowNewLines(bool Flag_);
    void TagInfoListWalkDump(ostream& os_, size_t fw_=20);
    void RetrieveIntraTagContent(bool Flag_);

    // Set tag and document parsing switches
    void RestoreTagSwitches();
    void InitCppLangTagSwitches();
    void InitCLangTagSwitches();
    void InitAllHtmlTagSwitches();
    void InitAllXmlTagSwitches();

    // search every vector in tag dictionary for matching tag string
    // or tag bracket.
    const TagSearchResult* SearchForTag();

    // Search method to look for specific html tag string and brackets
    const TagSearchResult* SearchForTag(int TagPos_, bool Empty_, bool Container_,
                                        int AppAttr_, const char* StartBrk_,
                                        const ChrString& Element_, const char* EndBrk_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
// Sample Comments Parsed:
//
// example 1, c comment block:
//   /*[C comment, 2015-06-28, start:
//      Notes:]->
//     C comment block contents
//   <-[C comment, 2015-06-28, end:]*/
//
// example 2: cpp comment line:
//   // Cpp comment, 2015-06-28, Notes: =>
//
// example 3: php script block:
//   <? phpinfo(); ?>
//
// example 4: html comment block:
//   <!-- comment -->
//
// example 5: html nested comment block:
//   <!--[if lt IE 7]>
//     <style>div.page img.alpha{display:none}</style>
//   <![endif]-->
//
// parsing started, for examples 2, 3, 4:
//   TagStringData: this=0x13, InComment=true, EndingTag=false
//     TagCommentData: Parent=0x13, IntCmntEndBrk=0, IntCmntStartBrk=0,
//                  CmntBlkEnd=0, CmntBlkStart="<!--", IntCmntIndicator=0,
//                  NestedCmntTag=false, IntCmntTag=false, InComment=false,
//                  IntCmntAnchor=NONE, CmntAnchor=OPENING, IntCmntStatus=IGNORED,
//                  CmntType=XMLSTYLE, NextCmntBlock=0x21
//       TagCommentData: PrevCmntBlck=0x21, CmntContent="comment",
//                    IntCmntEndBrk=0, IntCmntStartBrk=0,
//                    CmntBlkEnd="-->", CmntBlkStart=0, IntCmntIndicator=0,
//                    NestedCmntTag=false, IntCmntTag=false, InComment=true,
//                    IntCmntAnchor=NONE, CmntAnchor=CLOSING,
//                    IntCmntStatus=IGNORED, CmntType=XMLSTYLE,
//
// parsing started, for examples 1, 5:
//   TagStringData: this=0x3A, InComment=true, EndingTag=false
//     TagCommentData: Parent=0x3A, IntCmntEndBrk=0, IntCmntStartBrk=0, 
//                  CmntBlkEnd=0, CmntBlkStart="<!--", IntCmntIndicator="[",
//                  NestedCmntTag=false, IntCmntTag=false, InComment=false,
//                  IntCmntAnchor=NONE, CmntAnchor=OPENING, IntCmntStatus=INACTIVE,
//                  CmntType=XMLSTYLE, NextCmntBlock=0xA2
//       TagCommentData: PrevCmntBlck=0xA2, IntCmntContent="if lt IE 7", IntCmntEndBrk="]>",
//                    IntCmntStartBrk=0, CmntBlkEnd=0, CmntBlkStart=0, IntCmntIndicator=0,
//                    NestedCmntTag=false, IntCmntTag=true, InComment=false,
//                    IntCmntAnchor=CLOSING, CmntAnchor=NONE, IntCmntStatus=ACTIVE,
//                    CmntType=XMLSTYLE, NextCmntBlock=0x2B
//         TagCommentData: PrevCmntBlck=0x2B, CmntContent="<style>div.page img.alpha{display:none}</style>",
//                      IntCmntEndBrk=0, IntCmntStartBrk="<![", CmntBlkEnd=0, CmntBlkStart=0,
//                      IntCmntIndicator=0, NestedCmntTag=false, IntCmntTag=true, InComment=true,
//                      IntCmntAnchor=OPENING, CmntAnchor=NONE, IntCmntStatus=ACTIVE,
//                      CmntType=XMLSTYLE, NextCmntBlock=0x4D
//           TagCommentData: PrevCmntBlck=0x4D, IntCmntContent="endif", IntCmntEndBrk=0,
//                        IntCmntStartBrk=0, CmntBlkEnd="-->", CmntBlkStart=0,
//                        IntCmntIndicator="]", NestedCmntTag=false, IntCmntTag=false, InComment=true,
//                        IntCmntAnchor=NONE, CmntAnchor=CLOSING, IntCmntStatus=INACTIVE,
//                        CmntType=XMLSTYLE, NextCmntBlock=0
//
// Encounter method call sequence for examples 2, 3, 4:
//   <!--   : EncounterOpen(IntraCmnt=false, BlockStart=false)
//                CmntBlkEnd=0, CmntBlkStart="<!--",
//                NestedCmntTag=false, IntCmntTag=false, InComment=false,
//                IntCmntIndicator=0,
//                CmntAnchor=OPENING,
//                IntCmntStatus=IGNORED
//   -->    : EncounterClose(IntraCmnt=false, BlockStart=false)
//                CmntBlkEnd="-->", CmntBlkStart=0,
//                NestedCmntTag=false, IntCmntTag=false, InComment=true,
//                IntCmntIndicator=0,
//                CmntAnchor=CLOSING,
//                IntCmntStatus=IGNORED
//
// Encounter method call sequence for examples 1, 5:
//   <!--[  : EncounterOpen(IntraCmnt=false, BlockStart=true),
//                CmntBlkEnd=0, CmntBlkStart="<!--",
//                IntCmntEndBrk=0, IntCmntStartBrk=0,
//                NestedCmntTag=true, IntCmntTag=false, InComment=false,
//                IntCmntIndicator="[",
//                CmntAnchor=OPENING,
//                IntCmntStatus=INACTIVE
//   ]>     : EncounterClose(IntraCmnt=true, BlockStart=false)
//                IntCmntEndBrk="]>", IntCmntStartBrk=0,
//                NestedCmntTag=true, IntCmntTag=true, InComment=false,
//                IntCmntIndicator=0,
//                IntCmntAnchor=CLOSING,
//                IntCmntStatus=ACTIVE
//   <![    : EncounterOpen(IntraCmnt=true, BlockStart=false)
//                IntCmntEndBrk=0, IntCmntStartBrk="<![",
//                NestedCmntTag=true, IntCmntTag=true, InComment=true,
//                IntCmntIndicator=0,
//                IntCmntAnchor=OPENING,
//                IntCmntStatus=ACTIVE
//   ]-->   : EncounterClose(IntraCmnt=false, BlockStart=true)
//                CmntBlkEnd="-->", CmntBlkStart=0,
//                IntCmntEndBrk=0, IntCmntStartBrk=0,
//                NestedCmntTag=true, IntCmntTag=false, InComment=true,
//                IntCmntIndicator="]",
//                CmntAnchor=CLOSING,
//                IntCmntStatus=INACTIVE
//
class TagCommentData
{
  protected:
    enum
    {
      CMNTCNTENUM      = 0,
      INTCMNTCNTENUM   = 1,
      PARENTENUM       = 2,
      PREVCMNTBLKENUM  = 3,
      
      NESTEDCMNTEND1   = 4,
      NESTEDCMNTSTART2 = 6,
      MAXCMNTTAGSIZE   = 8
    };
  
    union
    {
      TagStringData* _Parent;         // parent reference object
      TagCommentData* _PrevCmntBlock; // previous node in linked list
    };

    union
    {
      ChrString* _IntCmntContent;   // internal comment content
      ChrString* _CmntContent;      // comment content
    };

    static bool _ShowCommentDataDump;

    TagCommentData* _NextCmntBlock; // next node in linked list
    ChrString* _IntCmntEndBrk;      // internal comment end brk, eg: "]>"
    ChrString* _IntCmntStartBrk;    // internal comment start brk, eg: "<!["
    ChrString* _CmntBlkEnd;         // comment block end, eg: "-->"
    ChrString* _CmntBlkStart;       // comment block start, eg: "<!--"
    ChrString* _IntCmntIndicator;   // internal comment indicator, eg: "[", "]"
    const TagSearchResult* _SearchResult; // search results, externally passed

    ChrString* _CommentTagString;       // point where enclosing comment tags are found
    TagSearchResult* _CommentTagSearch; // duplicated search result for comment parsing

    bool _CommentFound;             // Comment tags detected in parsing
    bool _NestedCmntTag;            // Nested comment tag found
    bool _IntCmntTag;               // Tag encountered is internal tag?
    bool _InComment;                // Parsing position is within comment?

    Subscript _MemPtrAlloc[8]; // memory allocation switch for union data members
    int _IntCmntStatus;  // active/inactive/ignored
    int _IntCmntAnchor;  // opening/closing/no_tagpos
    int _CmntAnchor;     // opening/closing/no_tagpos
    int _CmntType;       // cstyle/cppstyle/xmlstyle/htmlstyle
    long _TagBrkIndex;   // in +2 increments for each tag, +1 for ending tag

    // bool switches for toggling storing of content for various tag types
    bool _InclCCmntContent;
    bool _InclCppCmntContent;
    bool _InclScriptCmntContent;
    bool _InclNestedCmntContent;
    bool _InclHtmlCmntContent;

    bool TagContentsIncluded(int Index_, bool Nested_) const;
    bool CommentFound(const char* Str_, long Pos_);
    void SetTagBrkIndex(long Index_, const char* StartBlock_);
    void MakeCommentTagData(TagCommentData* Rootp_);
    long BrkIndexFileType() const;
    long EncounterOpen(bool IntraCmnt_, bool BlockStart_);
    long EncounterClose(bool IntraCmnt_, bool BlockEnd_);

    const char* CommentOpen_Str() const;
    const char* CommentClose_Str() const;

    const char* NestedCommentClose_Str(char* chp) const;
    const char* NestedCommentOpen_Str(char* chp) const;

    TagCommentData& EraseIntCmntBrk(bool StartBrk_, bool EndBrk_);
    TagCommentData& EraseCmntBlk(bool StartBlock_, bool EndBlock_, bool BrkIndex_);
    TagCommentData& EraseIntCmntIndicator(bool Indicator_);

    TagCommentData& SetIntCmntBrk(const char* StartBrk_, const char* EndBrk_);
    TagCommentData& SetCmntBlk(const char* StartBlock_, const char* EndBlock_);
    TagCommentData& SetIntCmntIndicator(const char* Indicator_);

    Subscript RootIndex() const;
    Subscript RootLength() const;

    Subscript RootMatchingIndex() const;
    Subscript RootMatchingLength() const;

  public:
    TagCommentData(TagStringData* Parent_);
    TagCommentData(TagStringData* Parent_,
                   const char* StartBrk_, const char* EndBrk_,
                   const char* StartBlock_, const char* EndBlock_,
                   const char* Indicator_, int CmntType_);
    TagCommentData(const TagCommentData& Obj_);
    ~TagCommentData();

    static TagCommentData* Make(TagStringData* Parent_);
    static TagCommentData* Make(TagStringData* Parent_,
                                const char* StartBrk_, const char* EndBrk_,
                                const char* StartBlock_, const char* EndBlock_,
                                const char* Indicator_, int CmntType_);
    static TagCommentData* Make(const TagCommentData& Obj_);

    static void SetShowCommentDataDump(bool Flag_);
    static bool ShowCommentDataDump();

    TagCommentData& operator = (const TagCommentData& Obj_);

    long ParseComment();

    TagCommentData& SetSearchResult(const TagSearchResult* SrchRes_);
    TagCommentData& SetTextBufferPtr(const ChrString* Ptr_);
    TagCommentData& SetNextCmntBlock(TagCommentData* Ptr_);
    TagCommentData& SetPrevCmntBlock(TagCommentData* Ptr_);
    TagCommentData& SetIntCmntContent(ChrString* Ptr_, bool KeepPtr_=false);
    TagCommentData& SetCmntContent(ChrString* Ptr_, bool KeepPtr_=false);

    TagCommentData& SetInComment(bool Flag_);
    TagCommentData& SetIntCmntTag(bool Flag_);
    TagCommentData& SetCommentType(int Type_);
    TagCommentData& SetCommentStatus(int Anchor_=-1, int Type_=-1);
    TagCommentData& SetIntCommentStatus(int Anchor_=-1, int Status_=-1);

    inline bool HasThisParent(const TagStringData* Parent_) const
        { return (_Parent == Parent_); }
    inline void IncludeCContents(bool Flag_=true)
        { _InclCCmntContent = Flag_; }
    inline void IncludeCppContents(bool Flag_=true)
        { _InclCppCmntContent = Flag_; }
    inline void IncludeScriptContents(bool Flag_=true)
        { _InclScriptCmntContent = Flag_; }
    inline void IncludeNestedContents(bool Flag_=true)
        { _InclNestedCmntContent = Flag_; }
    inline void IncludeHtmlContents(bool Flag_=true)
        { _InclHtmlCmntContent = Flag_; }

    inline bool CContentsIncluded() const
        { return _InclCCmntContent; }
    inline bool CppContentsIncluded() const
        { return _InclCppCmntContent; }
    inline bool ScriptContentsIncluded() const
        { return _InclScriptCmntContent; }
    inline bool NestedContentsIncluded() const
        { return _InclNestedCmntContent; }
    inline bool HtmlContentsIncluded() const
        { return _InclHtmlCmntContent; }

    inline bool IntCmntTag() const
        { return _IntCmntTag; }
    inline bool InComment() const
        { return _InComment; }
    inline bool CommentFound() const
        { return _CommentFound; }
    inline bool NestedCmntTag() const
        { return _NestedCmntTag; }
    inline Subscript NestedCmntStart2Index() const
        { return _MemPtrAlloc[NESTEDCMNTSTART2]; }
    inline Subscript NestedCmntStart2Length() const
        { return _MemPtrAlloc[NESTEDCMNTSTART2+1]; }
    inline Subscript NestedCmntEnd1Index() const
        { return _MemPtrAlloc[NESTEDCMNTEND1]; }
    inline Subscript NestedCmntEnd1Length() const
        { return _MemPtrAlloc[NESTEDCMNTEND1+1]; }

    inline long TagBrkIndex() const
        { return _TagBrkIndex; }
    inline int IntCmntStatus() const
        { return _IntCmntStatus; }
    inline int IntCmntAnchor() const
        { return _IntCmntAnchor; }
    inline int CmntAnchor() const
        { return _CmntAnchor; }
    inline int CmntType() const
        { return _CmntType; }
    inline const TagSearchResult* SearchResult() const
        { return ((_CommentFound && _CommentTagSearch) ?
                       _CommentTagSearch:
                       _SearchResult); }
    inline TagSearchResult* CommentTagSearch()
        { return ((_CommentFound && _CommentTagSearch) ?
                       _CommentTagSearch:
                       NULL); }

    const TagStringData* Parent() const;
    const TagCommentData* NextCmntBlock() const;
    const TagCommentData* PrevCmntBlock() const;
    const TagCommentData* Root() const;

    const ChrString* IntCmntContent() const;
    const ChrString* CmntContent() const;

    const ChrString* IntCmntEndBrk() const;
    const ChrString* IntCmntStartBrk() const;
    const ChrString* CmntBlkEnd() const;
    const ChrString* CmntBlkStart() const;
    const ChrString* IntCmntIndicator() const;
    const ChrString* CommentTagString() const;

    void DumpParseResults(ostream& os_, size_t fw_=20) const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
struct KeyValuePairNode
{
  ChrString* _Key;    // Key component of Key/Value pair node
  ChrString* _Value;  // Value component of Key/Value pair node

  KeyValuePairNode();
  KeyValuePairNode(ChrString* Key_, ChrString* Value_, bool KeepPtr_=false);
  ~KeyValuePairNode();

  static KeyValuePairNode* Make();
  static KeyValuePairNode* Make(ChrString* Key_, ChrString* Value_, bool KeepPtr_=false);

  KeyValuePairNode& AddPair(ChrString* Key_, ChrString* Value_, bool KeepPtr_=false);
  bool IsKeyMatch(const ChrString& Str_) const;
  ChrString GiveKVPairString(bool* Avail_=NULL) const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

// Sample tag with key, value pairs parsed
//
// str = "<tag1 key1=val1 key2=val2>xxxxxoooooxxxxxooooo</tag1>"
// TagFound(str) == true, parsed from "<tag1 " to ">"
//
// "<tag1 "  : EncounterTagOpen()
// key1=val1 : ParseKVpair(), KVDelim="=", Wspace=" "
// key2=val2 : ParseKVpair(), KVDelim="=", TagTerminator=">"
// ">"       : EncounterTagClose(), TagTerminator=">"
//
//
class KeyValuePairData
{
  protected:
    static bool _ShowKeyValuePairDump;

    TagStringData* _Parent;       // pointer to TagStringData parent object
    KeyValuePairNode** _KvPairs;  // vector of Key/Value pair pointers

    long _PairsMax;  // Key/Value pair vector upper limit
    long _PairsLen;  // Key/Value pair vector running pair count

    ChrString* _KVDelim;  // String containing valid KV pair delimiter chars
    ChrString* _Wspace;   // String containing valid white space chars
    TagSearchResult::TAGTERMLISTTYPE
      _TagTerminators;    // List of tag end bracket terminators
    int _TagTermIndex;    // Running count for end bracket terminators list
    bool _TagFound;       // Html tag with potential key/value pairs found

    // Search results returned from an external TagParseInfo object
    const TagSearchResult* _SearchResult;

    ChrString* _KVTagString;        // String containing Html tag with KV pairs
    TagSearchResult* _KVTagSearch;  // Duplicated search result object with KV pairs

    KeyValuePairNode** GrowVector(long Incr_=10);

    void EraseKvPairs();
    bool HasTerm(const ChrString& Obj_);

    bool TagFound(const char* Str_, long Pos_);
    bool EncounterTagOpen(Subscript* Index_, Subscript* Length_);
    bool EncounterTagClose(Subscript* Index_, Subscript* Length_);

  public:
    KeyValuePairData(TagStringData* Parent_);
    KeyValuePairData(TagStringData* Parent_,
                     ChrString* Delim_, ChrString* Wspace_,
                     ChrString* TagTerminator_, bool KeepPtr_=false);
    KeyValuePairData(const KeyValuePairData& Obj_);
    ~KeyValuePairData();

    static KeyValuePairData* Make(TagStringData* Parent_);
    static KeyValuePairData* Make(TagStringData* Parent_,
                                  ChrString* Delim_, ChrString* Wspace_,
                                  ChrString* TagTerminator_, bool KeepPtr_=false);
    static KeyValuePairData* Make(const KeyValuePairData& Obj_);

    static void SetShowKeyValuePairDump(bool Flag_);
    static bool ShowKeyValuePairDump();

    KeyValuePairData& operator = (const KeyValuePairData& Obj_);

    // returns index position into text buffer after parsing is finished
    long ParseKVpair();

    KeyValuePairData& SetKVDelimiter(ChrString* Ptr_, bool KeepPtr_=false);
    KeyValuePairData& SetWspace(ChrString* Ptr_, bool KeepPtr_=false);
    KeyValuePairData& AddTagTerminator(ChrString* Ptr_, bool KeepPtr_=false);
    KeyValuePairData& ClearTagTerminators();
    KeyValuePairData& SetSearchResult(const TagSearchResult* SrchRes_);
    KeyValuePairData& SetTextBufferPtr(const ChrString* Ptr_);
    
    KeyValuePairData& AddPair(ChrString* Key_, ChrString* Value_, bool KeepPtr_=false);
    const KeyValuePairNode* GiveKvPair(long Index_, bool* Found_=NULL) const;

    bool TagFound() const;
    inline bool HasThisParent(const TagStringData* Parent_) const
        { return (_Parent == Parent_); }
    inline int NumTagTerminators() const
        { return _TagTermIndex; }
    inline TagStringData* Parent()
        { return _Parent; }
    inline long NumPairs() const
        { return _PairsLen; }
    inline long PairsVectorMax() const
        { return _PairsMax; }

    inline const TagSearchResult* SearchResult() const
        { return ((_TagFound && _KVTagSearch) ?
                       _KVTagSearch:
                       _SearchResult); }
    inline TagSearchResult* KVTagSearch()
        { return ((_TagFound && _KVTagSearch) ?
                       _KVTagSearch:
                       NULL); }


    const ChrString* KVTagString() const;
    const ChrString* KVDelimiter() const;
    const ChrString* Wspace() const;
    const ChrString* TagTerminator(int Index_=0) const;

    void DumpParseResults(ostream& os_, size_t fw_=20) const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

/****************************************************************************/
#endif
