// Text file line buffer ADT class
//
#ifndef TEXTFILELINE_H
#define TEXTFILELINE_H
#ifndef INCL_STDIO_H
  #include <stdio.h>
  #define INCL_STDIO_H
#endif
#ifndef INCL_STDLIB_H
  #include <stdlib.h>
  #define INCL_STDLIB_H
#endif
#ifndef FILEREADER_H
  #include "filereader.h"
#endif
#ifndef SIMPLELIST_CPP
  #include "simplelist.cpp"
#endif
#ifndef CHRSTRING_H
  #include "chrstring.h"
#endif

#define TEXTFILELINE_DEBUG      0
#define TEXTFILELINE_DEBUG1     0  // starttag: <,=,> endtag: <,>,null
#define TEXTFILELINE_DEBUG2     0  // starttag: <,=,> endtag: </,>,null
#define TEXTFILELINE_DEBUG2b    0  // starttag: <,=,> endtag: </,>,null
#define TEXTFILELINE_DEBUG3     0  // starttag: null... endtag: null...
#define TEXTFILELINE_DEBUG4     0  // starttag: null... endtag: <,>,null
#define TEXTFILELINE_DEBUG5     0  // command line arguments

/****************************************************************************/
class FilePosition
{
  protected:
    long _BytesRead;
    bool _TextFileType;
    long _RowPos;
    long _ColPos;

  public:
    FilePosition(bool TextType_=false);

    FilePosition& SetTextFileType(bool TextType_);
    FilePosition& IncBytesRead(long Bytes_);
  
    FilePosition& IncColPos(long ColPos_=1);
    FilePosition& IncRowPos(long RowPos_=1);
    FilePosition& DecColPos(long ColPos_=1);
    FilePosition& DecRowPos(long RowPos_=1);

    FilePosition& SetColPos(long Row_);
    FilePosition& SetRowPos(long Col_);
    FilePosition& ResetColPos();
    FilePosition& ResetRowPos();

    inline long BytesRead() const
        { return _BytesRead; }
    inline bool TextFileType() const
        { return _TextFileType; }
    inline long RowPos() const
        { return _RowPos; }
    inline long ColPos() const
        { return _ColPos; }
};

/****************************************************************************/
struct DelimInfo
{
  int _DelimType;
  int _DelimIndex;
  char** _Delims;
  char** _DelimPtr;  
  char* _TextBuffer;
  bool _EraseBuf;
  bool _DelimFound[3];
  long _DelimPos[3];

  DelimInfo();
  DelimInfo(bool EraseBuf_, char* TextBuf_);
  DelimInfo(int DelimType_, int DelimIndex_, char** DelimPtr_, long* DelimPos_);
  DelimInfo(int DelimType_, char** Delims_, char** DelimPtr_,
            bool EraseBuf_, char* TextBuf_, bool* DelimFound_, long* DelimPos_);
  DelimInfo(const DelimInfo& Obj_);
  ~DelimInfo();

  static char** InitDelimsArray(char** ArrPtr_, size_t max);
  bool IsEqual(const DelimInfo& Obj_) const;
  bool IsTxtBufEqual(const DelimInfo& Obj_) const;
  bool Matches(int DelimType_, int Ends_, int Test_,
               char** Delims_, char** DelimPtr_,
               bool* DelimFound_, bool* DelimOptional_);
  DelimInfo& AssignTo(int DelimType_, char** Delims_, char** DelimPtr_,
                      bool* DelimFound_, long* DelimPos_);

  EXTERNALCLASSFNCS_STUB(DelimInfo, DelimInfo)
  CLONEFNCS_INLINE_DEFN(DelimInfo)

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif  
};

class DelimInfoComparer : public MatchableTraits<DelimInfo>
{
  STATICCLASSINSTANCE_VARS_DECL(DelimInfoComparer)

  public:  
    DelimInfoComparer();
    ~DelimInfoComparer();

    virtual MatchableTraits<DelimInfo>* Clone() const;
    virtual Boolean Equal(const DelimInfo& x, const DelimInfo& y) const;
    virtual Boolean NotEqual(const DelimInfo& x, const DelimInfo& y) const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

    EXTERNALCLASSFNCS_STUB(DelimInfoComparer, DelimInfoComparer)
    CLONEFNCS_INLINE_DEFN(DelimInfoComparer)
};

class DelimTxtBufComparer : public MatchableTraits<DelimInfo>
{
  STATICCLASSINSTANCE_VARS_DECL(DelimTxtBufComparer)

  public:  
    DelimTxtBufComparer();
    ~DelimTxtBufComparer();

    virtual MatchableTraits<DelimInfo>* Clone() const;
    virtual Boolean Equal(const DelimInfo& x, const DelimInfo& y) const;
    virtual Boolean NotEqual(const DelimInfo& x, const DelimInfo& y) const;

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif

    EXTERNALCLASSFNCS_STUB(DelimTxtBufComparer, DelimTxtBufComparer)
    CLONEFNCS_INLINE_DEFN(DelimTxtBufComparer)
};

// class to manage text file lines given an ascii string buffer of
// specified char size
//
// cases:
//
//  case1: text text text <delim> \n
//  case2: text <delim1> text <delim2> text \n
//  case3: text <delim1> text <delim2> text <delim3> text \n
//  ...
//  EOF
//
//  potential breaks for repeat read char process at delimiter points
//  within read file line are: <delim1> <delim2> or <delim3>
//
//  found delims can either be:
//  case1: near start of line
//  case2: near end of line
//  case3: first matches last since only single delim set found
//
//  Delims could be found either in text line which fits in buffer or not
//  eol character is: '\n'
//
//  Instead of reading from file to process text line, the text line can
//  be assigned as the given input line instead and placed in member
//  _GivenInputLine
//
// Line read attempt interruptions:
//
//  token1 <ws> token2 <delim> token3 <buflim> token4 <ws> token5 <eol>
//
//  series of char lengths from token1 ... token3 where buflim is reached
//  is stored in ReadLenQueue until <buflim> is reached.
//
//  series of array lens stored in array format in the form:
//  [arrsz][len1][len2] ... [len3] stored in ReadBufQueue until <eol>
//  is reached.
//
//  Call to method ResetBufferStatus will store newly created array from
//  data in ReadLenQueue into a new node of ReadBufQueue and then clear
//  the existing data in ReadLenQueue so queue is empty.
//
//  For the above example:
//    array1 == [rdattnum==3][len1==5][len2==5][len3==5],
//    array2 == [rdattnum==2][len4==5][len5==5]
//
//    order of operation for vector storage into queue:
//
//      ReadLenQeueu cleared
//      len1 ... len3 --> ReadLenQueue : for each read attempt before <buflim>
//      array1 --> ReadBufQueue : after call to method ResetBufferStatus
//      ReadLenQeuue cleared
//      len4 ... len5 --> ReadLenQueue : for each read attempt before <eol>
//      array2 --> ReadBufQueue : after call to method ResetBufferStatus
//      ReadLenQueue cleared
//      ...
//
//  Text file line record format:
//
//    <RowPos>:<LineFitInBuffer>:<SingleDelimSetFound>:<DelimSetTypeFound>:
//    <TimesLimitReached>:<ReadTimesRequired>:<EolFound>:<EolPos>:
//    <EofFound>:<EofPos>;\n
//    <LenQueueSize>;\n
//    <va1_size>:<va1_len0>:<va1_len1>...:<va1_len...x>;\n
//    <va2_size>:<va2_len0>:<va2_len1>...:<va2_len...x>;\n
//                   ...
//    <vax_size>:<vax_len0>:<vax_len1>...:<vax_len...x>;\n
//    <DelimQueueSize>;\n
//    <FirstDelimFound>[0],[1],[2]:
//    <FirstDelimPos>[0],[1],[2]:
//    <FirstDelims>[0],[1],[2];\n
//    ...
//    <MidDelimFound...>[0],[1],[2]:
//    <MidDelimPos...>[0],[1],[2]:
//    <MidDelims...>[0],[1],[2];\n
//    ...
//    <LastDelimFound>[0],[1],[2]:
//    <LastDelimPos>[0],[1],[2]:
//    <LastDelims>[0],[1],[2];\n
//
class TextFileLineBuffer;

class TextFileLineData
{
  friend class TextFileLineBuffer;

  protected:
    long _RowPos;
    bool _LineFitInBuffer;
    bool _SingleDelimSetFound;
    
    short _TimesLimitReached;
    short _ReadTimesRequired;
    short _DelimSetTypeFound;

    bool _EolFound;
    bool _EofFound;
    long _EndLinePos;
    long _EndFilePos;

    char** _FirstDelims;
    char** _FirstDelimPtr;
    bool _FirstDelimFound[3];
    long _FirstDelimPos[3];

    char** _LastDelims;
    char** _LastDelimPtr;
    bool _LastDelimFound[3];
    long _LastDelimPos[3];

    char* _TextBuffer;
    char* _TagBuffer;
    bool _SplitTag;
    bool _MultiLineTag;

    ChrString _TotalTextRead;

    SimpleQueue<long>* _ReadBufQueue;
    SimpleDeque<DelimInfo>* _DelimStk;
    SimpleDeque<DelimInfo>* _PartialDStk;
    SearchableList<DelimInfo>* _SrchList;

    char* FillTagBuffer(char* TagBuffer_, char** DelimPtr_,
                        long* DelimPos_, long fpos, long lpos);

  public:
    TextFileLineData(TextFileLineBuffer& Obj_);
    ~TextFileLineData();

    TextFileLineData& operator = (TextFileLineBuffer& Obj_);
    
    void SetTagBuffer(char* TagBuffer_);
    void DumpData(ostream& Out_, bool ColHead_);

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

class TextFileSearchData
{
  protected:
    int _ArrayIndex;
    int _StringIndex;
    int _DelimType;

  public:
    TextFileSearchData(int ArrayIndex_, int StringIndex_, int DelimType_);
    ~TextFileSearchData();

    int ArrayIndex() const
        { return _ArrayIndex; }
    int StringIndex() const
        { return _StringIndex; }
    int DelimType() const
        { return _DelimType; }

#if OVERLOAD_NEW
    void* operator new (size_t Bytes_);
    void operator delete (void* Space_);

#if	HAS_ARRAY_NEW
    void* operator new[] (size_t Bytes_);
    void operator delete[] (void* Space_);
#endif
#endif
};

class TextLineEnums
{
  public:
    enum
    {
      MAXTEXTBUFFER = 512,  // test with: 32, production use: 512
      MAXDELIMS     = 3,
      MAXCMMNTSTR   = 4,

      NONE       = 0,
      STARTING   = 1,
      ENDING     = 2,

      FIRST_SET = 1,
      LAST_SET  = 2,

      INCOMPLETE = 4,
      COMPLETE   = 8,

      FILE_INPUT  = 1,
      FILE_OUTPUT = 2
    };
};

class TextFileLineBuffer : public TextLineEnums
{
  friend class TextFileLineData;

  protected:
    char* _CurPosPtr;  
    char* _TextBuffer;
    char* _TagBuffer;
    
    bool _BufferFull;
    bool _LineFitInBuffer;
    bool _SingleDelimSetFound;
    bool _FreaderCreated;

    bool _QuitReadOnFirstDelimSetFound;
    bool _QuitReadOnLastDelimSetFound;
    bool _ReadUntilFirstDelimSetFound;
    bool _ReadUntilLastDelimSetFound;
    bool _FirstDelimSetFound;
    bool _LastDelimSetFound;
    bool _SimplifyTagSpaces;
    bool _SuppressSimplify;

    short _IoMode;
    short _DefaultDelim;
    short _LineEmpty;
    short _CharsRead;
    short _BufSize;
    short _ReadAttemptNumber;
    short _TimesLimitReached;
    short _ReadTimesRequired;
    short _DelimSetTypeFound;

    char* _NlChars;     // newline characters
    char* _WsChars;     // whitespace characters
    char* _WsxNlChars;  // whitespace characters excluding newlines

    char* _GivenInputLine;
    bool _GivenLineSet;

    bool _FileOpened;
    bool _BrkChFound;
    bool _WsFound;
    bool _EolFound;
    bool _EofFound;
    long _WsLinePos;
    long _EndLinePos;
    long _EndFilePos;

    char** _FirstDelims;
    char** _FirstDelimPtr;
    bool _FirstDelimFound[3];
    bool _FirstDelimOptional[3];
    long _FirstDelimPos[3];

    char** _LastDelims;
    char** _LastDelimPtr;
    bool _LastDelimFound[3];
    bool _LastDelimOptional[3];
    long _LastDelimPos[3];

    char _StartCmmntStr[5];
    char _EndCmmntStr[5];
    
    char _CommentSeg[5];
    int _CmmntSegIndex;
    bool _ResetCmmntSeg;
    int _TestCmmntSeg;
    int _DelimsToTest;
    bool _XmlTypeTag;
    bool _InComment;
    bool _SplitTag;
    bool _MultiLineTag;
    bool _TagBufferComplete;

    ChrString _TotalTextRead;
    bool _TextAppended;

    TextFileLineData* _LineData;
    TextFileSearchData* _SearchData;
    FilePosition* _FilePos;
    FileReader* _File;
    SimpleStack<long>* _ColStk;
    SimpleQueue<long>* _ReadLenQueue;
    SimpleQueue<long>* _ReadBufQueue;
    SimpleDeque<DelimInfo>* _DelimStk;
    SimpleDeque<DelimInfo>* _PartialDStk;

    char* FillTagBuffer();
    bool ReadUntilDelimFound() const;
    bool QuitReadDelimFound() const;
    bool IsEmptyLine(const char* InputStr_, int len);

    char CountColsRows(char ch);
    char UnCountColsRows(char ch);
    void CountColsRows(const char* Buf_, const char* Delim_);
    void CountColsRows(const char* Buf_, long Len_, const char* Delim_);
    char* AppendTextRead(char* TxtBuf_);
    char* TerminateTextBuf(char* TxtBuf_, int x, bool AppendTxt_,
                           bool AppendWs_, bool AppendNl_=false);

    void ResetDelimFound();
    void ResetCmmntSeg(bool Shift_);
    void TestCmmntSeg(char ch, bool Test_=true);

    int NumDelims(char** Delims_, bool SkipOptional_=false);
    int DetermineDelimSet(char** Delims_);
    bool SetMatchingDelims(char** Delims_);
    bool TransferDelims();
    int DelimSetComplete(char** Delims_,
                         bool* PartialMatch_=NULL, int* TxtBufMatch_=NULL) const;

    int SearchForFirstDelims(char ch, long PrevCol_, int& ArrayIndex_,
                             int& StringIndex_, int& DelimType_, bool& Found_);
    int SearchForLastDelims(char ch, long PrevCol_, int& ArrayIndex_,
                            int& StringIndex_, int& DelimType_, bool& Found_);
    int SearchForDelims(char** Delims_, char ch, long PrevCol_,
                        int& ArrayIndex_, int& StringIndex_, int& Type_);

  public:  
    TextFileLineBuffer(FileReader* Freader_,
                       short IoMode_, short BufLen_=0);
    ~TextFileLineBuffer();

    static const char* TagTypeStr(const char* EscStr_);
    TextFileLineBuffer& SetFileReader(FileReader* Freader_);
    TextFileLineBuffer& SetInputLine(const char* Input_);

    TextFileLineBuffer& SetDefaultToDelims(int Set_);
    TextFileLineBuffer& SetFirstDelims(long Pos_, const char* Str_, bool Optional_);
    TextFileLineBuffer& SetLastDelims(long Pos_, const char* Str_, bool Optional_);

    int FirstDelimFound(long Pos_, bool All_=false) const;
    int LastDelimFound(long Pos_, bool All_=false) const;
    long FirstDelimPos(long Pos_);
    long LastDelimPos(long Pos_);

    TextFileLineBuffer& ClearFirstDelims(long Pos_, bool All_=false);
    TextFileLineBuffer& ClearLastDelims(long Pos_, bool All_=false);
    TextFileLineBuffer& ClearAllDelims();
    TextFileLineBuffer& ResetFirstDelims(long Pos_, bool All_=false);
    TextFileLineBuffer& ResetLastDelims(long Pos_, bool All_=false);
    TextFileLineBuffer& ResetAllDelims();

    TextFileLineBuffer& ResetLineData(bool ResetTextRead_=true,
                                      bool ForceReset_=false);
    TextFileLineBuffer& ResetBufferStatus(bool ForceReset_=false);

    void ResetTotalTextRead();
    void SetXmlTypeTag(bool Flag_);
    void SimplifyTagSpaces(bool Flag_)
        { _SimplifyTagSpaces = Flag_; }
    void SetIoMode(short IoMode_)
        { _IoMode = IoMode_; }
    FileReader* GetFileReader()
        { return _File; }
    FilePosition* GetFilePosition()
        { return _FilePos; }
    bool SingleDelimSetFound() const
        { return _SingleDelimSetFound; }
    bool LineFitInBuffer() const
        { return _LineFitInBuffer; }
    bool BufferFull() const
        { return _BufferFull; }
    size_t TextBufferLen() const
        { return SafeStrLen(_TextBuffer); }
    size_t TagBufferLen() const
        { return SafeStrLen(_TagBuffer); }
    const char* TextBuffer() const
        { return _TextBuffer; }
    const char* TagBuffer() const
        { return _TagBuffer; }
    short BufferSize() const
        { return _BufSize; }
    short TimesLimitReached() const
        { return _TimesLimitReached; }
    short ReadTimesRequired() const
        { return _ReadTimesRequired; }
    short ReadAttemptNumber() const
        { return _ReadAttemptNumber; }
    short DelimSetTypeFound() const
        { return _DelimSetTypeFound; }
        
    bool SimplifyTagSpaces() const
        { return _SimplifyTagSpaces; }
    bool XmlTypeTag() const
        { return _XmlTypeTag; }
    bool DelimSetsFound() const
        { return (_FirstDelimSetFound || _LastDelimSetFound); }
    bool FirstDelimSetFound() const
        { return _FirstDelimSetFound; }
    bool LastDelimSetFound() const
        { return _LastDelimSetFound; }
    const char* GivenInputLine() const
        { return _GivenInputLine; }
    long WsLinePosition() const
        { return _WsLinePos; }
    long EolPosition() const
        { return _EndLinePos; }
    long EofPosition() const
        { return _EndFilePos; }
    bool WsFound() const
        { return _WsFound; }
    bool EolFound() const
        { return _EolFound; }
    bool EofFound() const
        { return _EofFound; }
    bool FileOpened() const
        { return _FileOpened; }
    bool ReadBufferLen() const
        { return (_TextBuffer ? ::strlen(_TextBuffer):0); }
    bool TagStrAvailable() const
        { return (::SafeStrLen(_TagBuffer) &&
                  DelimSetsFound() && !_SplitTag); }
    bool EmptyLine(bool Complete_) const
        { return (Complete_ ? (_LineEmpty == COMPLETE):
                              (_LineEmpty != NONE)); }

    TextFileLineBuffer& SetQuitReadOnFirstDelimSetFound(bool Flag_, bool UntilFound_);
    TextFileLineBuffer& SetQuitReadOnLastDelimSetFound(bool Flag_, bool UntilFound_);

    size_t TotalTextReadLen() const
        { return _TotalTextRead.strlen(); }
    ChrString TotalTextRead();
    bool QuitReadOnFirstDelimSetFound() const
        { return _QuitReadOnFirstDelimSetFound; }
    bool QuitReadOnLastDelimSetFound() const
        { return _QuitReadOnLastDelimSetFound; }

    // Set newline and whitespace delimiter strings
    TextFileLineBuffer& SetNewLineChars(const char* Sep_="\n\r");
    TextFileLineBuffer& SetWhiteSpaceChars(const char* Sep_=" \t\n\r");

    // output procedures
    TextFileLineBuffer& DumpLineRecord(bool ColHead_=false);

    // input procedures
    // Looping read methods until eol detected
    bool While_Read(char* Input_);
    bool While_ReadLine();
    bool While_ReadUntil(const char* CharSet_,
                         bool InSet_=true, bool PutBack_=false);
    
    // input operators (terminated on white space for char* string)
    TextFileLineBuffer& operator >> (char* InputStr_);
    TextFileLineBuffer& Read();

    // character oriented input
    TextFileLineBuffer& ReadUntil(const char* CharSet_,
                                  bool InSet_=true, bool PutBack_=false);

    // block oriented input
    TextFileLineBuffer& ReadLine();
    TextFileLineBuffer& ReadLineOrToEobuf();
    TextFileLineBuffer& TruncLine();
    TextFileLineBuffer& Ignore();
    TextFileLineBuffer& ReadBuffer(streambuf* Sb_);

    // I/O state accessor methods
    bool EndOfFile();
    bool Good();
    bool Bad();
    bool Fail();

    // file open procedure
    bool Open(const char* Fname_, int Mode_);

    // file close procedure
    bool Close();

    // Method to return the size of the file in bytes
    long FileSize(int Mode_);

    // Initialize standard tags
    void InitStdHtmlTags();
    void InitSingleTagDelim();
};

/****************************************************************************/
#endif

