#ifndef INCL_WINDOWS_H
  #define INCL_WINDOWS_H
  #include <windows.h>
#endif
#ifndef INCL_STDLIB_H
  #define INCL_STDLIB_H
  #include <stdlib.h>
#endif
#ifndef INCL_PROCESS_H
  #define INCL_PROCESS_H
  #include <process.h>
#endif
#ifndef WTHREAD_H
  #include "wthread.h"
#endif

/****************************************************************************/
static DWORD WINAPI ThreadFunction(LPVOID object)
{
  THREAD* thread = (THREAD*)object;
  return (DWORD)thread->Run(thread->Param);
}

/****************************************************************************/
THREAD::THREAD(ThreadEvent* pEvent_):
_event(pEvent_),
ThreadHandle(NULL),
_hExitEvent(NULL),
_hWorkEvent(NULL),

Started(FALSE),
Detached(FALSE),
ThreadID(0)
{
  _dwWorkInterval	= 0;
  _nStatus			= STATUS_INIT;
}

/****************************************************************************/
THREAD::~THREAD()
{
   if (STATUS_INIT == _nStatus)
	return;

  // resume if suspend
  if (STATUS_SUSPEND == _nStatus)
	Resume();

  // close thread...
  if (STATUS_CLOSING != _nStatus)
	Close();

  // waiting for close
  if (WAIT_TIMEOUT == ::WaitForSingleObject(ThreadHandle, TIME_WAITINGCLOSE))
    Stop();

  ThreadID = 0;
}

/****************************************************************************/
void* THREAD::Run(void*)
{
  return NULL;
}

/****************************************************************************/
int THREAD::Start(void* param)
{
  if (!Started)
  {
    Param = param;
    if (ThreadHandle = (HANDLE)::CreateThread(NULL, 0, ThreadFunction, this, 0, &ThreadID))
    {
      if (Detached)
		::CloseHandle(ThreadHandle);

      Started = TRUE;
      _nStatus = STATUS_RUNNING;
    }
  }

  return Started;
}

/****************************************************************************/
void* THREAD::Wait()
{
  DWORD status = (DWORD)NULL;
  if (Started && !Detached)
  {
    ::WaitForSingleObject(ThreadHandle, INFINITE);
    ::GetExitCodeThread(ThreadHandle, &status);
    ::CloseHandle(ThreadHandle);
    Detached = TRUE;
  }

  return (void*)status;
}

/****************************************************************************/
void THREAD::Detach()
{
  if (Started && !Detached)
    ::CloseHandle(ThreadHandle);
  Detached = TRUE;
}

/****************************************************************************/
void THREAD::Stop()
{
  if (Started && !Detached)
  {
    ::TerminateThread(ThreadHandle, 0);

    if (ThreadHandle)
    {
	  ::CloseHandle(ThreadHandle);  // winbase api call
	  ThreadHandle = NULL;
    }

    if (_hWorkEvent)
    {
	  ::CloseHandle(_hWorkEvent);  // winbase api call
	  _hWorkEvent = NULL;
    }

    if (_hExitEvent)
    {
	  ::CloseHandle(_hExitEvent);  // winbase api call
	  _hExitEvent = NULL;
    }

    Detached = TRUE;
    _nStatus = STATUS_CLOSED;
  }
}

/****************************************************************************/
bool THREAD::Init(bool bSuspend, DWORD dwWorkInterval)
{
  if (STATUS_INIT != _nStatus)
	return false;

  if (dwWorkInterval == DWORD(-1))
    dwWorkInterval = INFINITE;

  // exit event  // winbase api call
  _hExitEvent = ::CreateEvent(NULL,	    // no security
                              FALSE,	// auto reset
                              FALSE,	// initial event reset
                              NULL);	// no name
  if (!_hExitEvent)
	return false;

  // work event  // winbase api call
  _hWorkEvent = ::CreateEvent(NULL,	    // no security
                              FALSE,	// auto reset
			                  FALSE,	// initial event reset
			                  NULL);	// no name

  if (!_hWorkEvent)
    return false;

  // thread
  DWORD dwCreationFlags = 0;
  if (bSuspend)
    dwCreationFlags	= CREATE_SUSPENDED;

  typedef unsigned long	(__stdcall *THREAD_FUNC)(void *);
  // winbase api call
  ThreadHandle = ::CreateThread (NULL,						        // default security
	                             0,							        // default stack size
			                     (THREAD_FUNC)THREAD::RunThread,	// pointer to thread routine
			                     (void*)this,				        // argument for thread
			                     dwCreationFlags,			        // start it right away if 0, else suspend
			                     (unsigned long*)&ThreadID);

  if (!ThreadHandle)
  {
	::CloseHandle(_hExitEvent);  // winbase api call
	_hExitEvent = NULL;

	::CloseHandle(_hWorkEvent);  // winbase api call
	_hWorkEvent = NULL;

	return false;
  }

  _dwWorkInterval = dwWorkInterval;

  if (bSuspend)
	_nStatus = STATUS_SUSPEND;
  else
	_nStatus = STATUS_RUNNING;

  return true;
}

/****************************************************************************/
bool THREAD::Close()
{
  if (::SetEvent(_hExitEvent))  // winbase api call
  {
	_nStatus = STATUS_CLOSING;
	return true;
  }

  return false;
}

/****************************************************************************/
void THREAD::Resume(void)
{
  if (STATUS_SUSPEND != _nStatus)
	return;

  if (DWORD(-1) != ::ResumeThread(ThreadHandle))  // winbase api call
	_nStatus = STATUS_RUNNING;
}

/****************************************************************************/
void THREAD::Suspend(void)
{
  if (STATUS_RUNNING != _nStatus)
	return;

  if (DWORD(-1) != ::SuspendThread(ThreadHandle))  // winbase api call
	_nStatus = STATUS_SUSPEND;
}

/****************************************************************************/
// static
THREAD*  THREAD::CreateNew(ThreadEvent* pEvent_, bool bSuspend, DWORD dwWorkInterval)
{
  if (!pEvent_)
    return NULL;

  if (dwWorkInterval == DWORD(-1))
    dwWorkInterval = INFINITE;

  THREAD* pThread_ = new THREAD(pEvent_);
  if (!pThread_)
	return NULL;

  if (!pThread_->Init(bSuspend, dwWorkInterval))
  {
	delete pThread_;
	return NULL;
  }

  return pThread_;
}

/****************************************************************************/
// static
DWORD WINAPI THREAD::RunThread(LPVOID pThreadParameter)
{
  THREAD* pThread_ = (THREAD*)pThreadParameter;
  if (!pThread_ || !pThread_->_event)
	return 1;

  // init
  if (-1 == pThread_->_event->OnThreadCreate())
	return 2;

  // work now
  HANDLE hEvent[2] = { pThread_->_hExitEvent, pThread_->_hWorkEvent };
  while (true)
  {
	DWORD dwTimeWait = pThread_->_dwWorkInterval;
	// winbase api call
	int rval = ::WaitForMultipleObjects(2, hEvent, false, dwTimeWait);
	if (WAIT_OBJECT_0 == rval)
	{
	  // exit event is set
      break;
    }
	else if (WAIT_OBJECT_0+1 == rval)
	{
	  // work event is set
	  if (-1 == pThread_->_event->OnThreadEvent())
		break;
    }
	else if (WAIT_TIMEOUT == rval)
	{
	  // work interval is time out
	  if (-1 == pThread_->_event->OnThreadProcess())
		break;
    }
	else
	{
	  // here should log error msg
 	  int i = 6;
      break;
    }
  }

  pThread_->_nStatus = THREAD::STATUS_CLOSED;
  return pThread_->_event->OnThreadDestroy();
}

/****************************************************************************/




