#ifndef GAUSS_H
#define GAUSS_H

#ifndef INCL_STDIO_H
  #include <stdio.h>
  #define INCL_STDIO_H
#endif
#ifndef CHRSTRING_H
  #include "chrstring.h"
#endif

#define GAUSS_DEBUG     0
#if GAUSS_DEBUG
  #include <iostream>
#endif

/****************************************************************************/
template <class T>
class GVector
{
  friend GVector<T> operator * (const GVector<T>& v, const T& m)
      { return (GVector<T>(v) *= m); }
  
  friend GVector<T> operator * (const T& m, const GVector<T>& v)
      { return (GVector<T>(v) *= m); }

  friend GVector<T> operator - (const GVector<T>& d, const GVector<T>& s)
      { return (GVector<T>(d) -= s); }

  private:
    int _Size;
    T** _Array;
  
  public:
    GVector();
    GVector(int n);
    GVector(const GVector<T>& v);
    ~GVector();

    GVector<T>& operator = (const GVector<T>& v);
    GVector<T>& operator -= (const GVector<T>& v);
    
    GVector<T>& operator *= (const T& v);
    
    void SetVector(const char* Str_);
    T operator () (int pos_) const;
    void SetElement(int pos_, const T& v);
    T& Element(int pos_);
    int Size() const;

    T ConvertToNumber(const char* Str_);
    ChrString ConvertToString(int x);
    void WriteToString(ChrString& Answer_);

    #if GAUSS_DEBUG
      std::ostream& WriteResultVector(std::ostream& Os_) const;
    #endif
};

/****************************************************************************/
template <class T>
class GMatrix
{
  private:
    int _Size;
    GVector<T>** _Varray;

  public:
    GMatrix();
    GMatrix(int n);  
    GMatrix(const GMatrix<T>& m);
    ~GMatrix();

    GMatrix<T>& operator = (const GMatrix& m);
    void SetMatrix(GVector<T>** v);
    T operator () (int row_, int col_) const;    
    void SetRow(int row_, const GVector<T>& v);
    const GVector<T>& Row(int row_) const;
    GVector<T>& Row(int row_);
    int Size() const;
};

/****************************************************************************/
template <class T>
GMatrix<T>::GMatrix():
_Size(0),
_Varray(NULL)
{}

/****************************************************************************/
template <class T>
GMatrix<T>::GMatrix(int n):
_Size(n),
_Varray(new GVector<T>*[n])
{
  int i;
  for (i = 0; i < _Size; i++)
    _Varray[i] = NULL;
}

/****************************************************************************/
template <class T>
GMatrix<T>::GMatrix(const GMatrix<T>& m):
_Size(m.Size()),
_Varray(new GVector<T>*[m.Size()])
{
  int i;
  for (i = 0; i < _Size; i++)
    _Varray[i] = new GVector<T>(m.Row(i));
}

/****************************************************************************/
template <class T>
GMatrix<T>::~GMatrix()
{
  int i;
  for (i = 0; i < _Size; i++)
    delete _Varray[i];
  delete[] _Varray;
}

/****************************************************************************/
template <class T>
GMatrix<T>& GMatrix<T>::operator = (const GMatrix& m)
{
  int i;
  for (i = 0; i < _Size; i++)
    delete _Varray[i];
  delete[] _Varray;

  _Size = m.Size();
  _Varray = new GVector<T>*[m.Size()];
  
  for (i = 0; i < _Size; i++)
    _Varray[i] = new GVector<T>(m.Row(i));

  return *this;
}

/****************************************************************************/
template <class T>
void GMatrix<T>::SetMatrix(GVector<T>** v)
{
  int i;
  
  for (i = 0; v[i]; i++);
  if (i != _Size)
  {
    int max = i;
    for (i = 0; i < _Size; i++)
      delete _Varray[i];
    delete[] _Varray;

    _Size = max;
    _Varray = new GVector<T>*[max];
  }
  
  for (i = 0; i < _Size; i++)
    if (v[i])
    {
      delete _Varray[i];
      _Varray[i] = new GVector<T>(*v[i]);
    }
    else
      break;
}

/****************************************************************************/
template <class T>
T GMatrix<T>::operator () (int row_, int col_) const
{
  if (row_ < _Size)
    return _Varray[row_]->Element(col_);
  return 0;
}

/****************************************************************************/
template <class T>
void GMatrix<T>::SetRow(int row_, const GVector<T>& v)
{
  if (row_ < _Size)
  {
    if (_Varray[row_])
      delete _Varray[row_];

    _Varray[row_] = new GVector<T>(v);
  }
}

/****************************************************************************/
template <class T>
const GVector<T>& GMatrix<T>::Row(int row_) const
{
  static GVector<T> Dummy_(1);
  if (row_ < _Size)
    return *_Varray[row_];
  return Dummy_;
}

/****************************************************************************/
template <class T>
GVector<T>& GMatrix<T>::Row(int row_)
{
  static GVector<T> Dummy_(1);
  if (row_ < _Size)
    return *_Varray[row_];
  return Dummy_;
}

/****************************************************************************/
template <class T>
int GMatrix<T>::Size() const
{
  return _Size;
}

/****************************************************************************/
/****************************************************************************/
template <class T>
GVector<T>::GVector():
_Size(0),
_Array(NULL)
{}

/****************************************************************************/
template <class T>
GVector<T>::GVector(int n):
_Size(n),
_Array(new T*[n])
{
  int i;
  for (i = 0; i < _Size; i++)
    _Array[i] = NULL;
}

/****************************************************************************/
template <class T>
GVector<T>::GVector(const GVector<T>& v):
_Size(v.Size()),
_Array(new T*[v.Size()])
{
  int i;
  for (i = 0; i < _Size; i++)
    _Array[i] = new T(v(i));
}

/****************************************************************************/
template <class T>
GVector<T>::~GVector()
{
  int i;
  for (i = 0; i < _Size; i++)
    delete _Array[i];
  delete[] _Array;
}

/****************************************************************************/
template <class T>
GVector<T>& GVector<T>::operator *= (const T& v)
{
  int i;
  for (i = 0; i < _Size; i++)
    if (_Array[i])
      *_Array[i] *= v;

  return *this;
}

/****************************************************************************/
template <class T>
GVector<T>& GVector<T>::operator -= (const GVector& v)
{
  int i;
  for (i = 0; i < _Size; i++)
    if (_Array[i])
      *_Array[i] -= *(v._Array[i]);

  return *this;
}

/****************************************************************************/
template <class T>
GVector<T>& GVector<T>::operator = (const GVector& v)
{
  int i;
  for (i = 0; i < _Size; i++)
    delete _Array[i];
  delete[] _Array;

  _Size = v.Size();
  _Array = new T*[v.Size()];
  
  for (i = 0; i < _Size; i++)
    _Array[i] = new T(v(i));

  return *this;
}

/****************************************************************************/
template <class T>
T GVector<T>::ConvertToNumber(const char* Str_)
{
  T Value_;

  int Id_ = WhatIs(Value_);
  if (Id_ == TypeInfo::BUILTIN_LONG_DOUBLE)
    Value_ = strtold(Str_, NULL);
  else
    Value_ = atof(Str_);

  return Value_;
}

/****************************************************************************/
template <class T>
ChrString GVector<T>::ConvertToString(int x)
{
  char Buffer_[150];
  ChrString Res_;
  T Value_;  
  
  if (_Array[x])
  {  
    int Id_ = WhatIs(Value_);
    if (Id_ == TypeInfo::BUILTIN_LONG_DOUBLE)
      Res_ = ::RemovePadding(LongFloatToStr(*_Array[x], Buffer_, 127, 10), " ");
    else if (Id_ == TypeInfo::BUILTIN_DOUBLE)
      Res_ = ::RemovePadding(FloatToStr(*_Array[x], Buffer_, 127, 10), " ");
    else if (Id_ == TypeInfo::BUILTIN_FLOAT)
      Res_ = ::RemovePadding(FloatToStr(*_Array[x], Buffer_, 127, 7), " ");
  }

  return Res_;
}

/****************************************************************************/
template <class T>
void GVector<T>::SetVector(const char* Str_)
{
  if (!Str_)
    return;

  int x;
  int cnt = 0;
  char* Buffer_ = strcpy(new char[strlen(Str_) + 1], Str_);
  for (x = strlen(Buffer_); x && Buffer_[x - 1] == ')'; x--);
  Buffer_[x] = 0;
  for (x = 0; Buffer_[x] == '('; x++);
  if (x) memmove(Buffer_, &Buffer_[x], strlen(&Buffer_[x]) + 1);
  
  int len = strlen(Buffer_);
  for (x = 0; x < len; x++)
    if (Buffer_[x] == ',')
      cnt++;

  char* token_ = NULL;
  int max = 0;

  if (cnt)
  {
    token_ = strtok(Buffer_, ",");
    max = cnt + 1;
    cnt = 0;
  }
  else
  {
    for (x = 0; isspace(Buffer_[x]); x++);
    if (isdigit(Buffer_[x]))
    {
      token_ = Buffer_;
      max = 1;
      cnt = 0;
    }
  }

  if (max && max != _Size)
  {
    int i;
    for (i = 0; i < _Size; i++)
      delete _Array[i];
    delete[] _Array;

    _Size = max;
    _Array = new T*[max];
  }

  while (token_)
  {
    for (x = strlen(token_); x && isspace(token_[x - 1]); x--);
    token_[x] = 0;
    for (x = 0; isspace(token_[x]); x++);
    if (x) memmove(token_, &token_[x], strlen(&token_[x]) + 1);

    if (strlen(token_))
    {
      delete _Array[cnt];
      _Array[cnt++] = new T(ConvertToNumber(token_));
    }

    token_ = strtok(NULL, ",");
  }

  delete[] Buffer_;
}

/****************************************************************************/
template <class T>
T GVector<T>::operator () (int pos_) const
{
  if (pos_ < _Size)
    return (*_Array[pos_]);
  return 0;
}

/****************************************************************************/
template <class T>
void GVector<T>::SetElement(int pos_, const T& v)
{
  if (pos_ < _Size)
  {
    if (_Array[pos_])
      delete _Array[pos_];

    _Array[pos_] = new T(v);
  }
}

/****************************************************************************/
template <class T>
T& GVector<T>::Element(int pos_)
{
  static T Dummy_(1);
  if (pos_ < _Size)
    return *_Array[pos_];
  return Dummy_;
}

/****************************************************************************/
template <class T>
int GVector<T>::Size() const
{
  return _Size;
}

/****************************************************************************/
template <class T>
void GVector<T>::WriteToString(ChrString& Answer_)
{
  int x;
  char Buffer_[32];  

  Answer_ = "{";
  for (x = 0; x < _Size; x++)
    if (_Array[x])
    {
      Answer_ += ChrString("x") + ChrString(::IntToStr(x+1, Buffer_, 10)) + ChrString(" == ");
      Answer_ += ConvertToString(x);
      if (x < _Size - 1)
        Answer_ += ", ";
    }

  Answer_ += "}";
}

/****************************************************************************/
#if GAUSS_DEBUG
template <class T>
std::ostream& GVector<T>::WriteResultVector(std::ostream& Os_) const
{
  int x;
  for (x = 0; x < _Size; x++)
    if (_Array[x])
    {
      Os_ <<"x" <<(x+1) <<" = " <<(*_Array[x]);
      if (x < _Size - 1)
        Os_ <<", ";
    }

  Os_ <<"\n";
  return Os_;
}
#endif
/****************************************************************************/
/****************************************************************************/
template <class T>
int MaxPivot(const GMatrix<T>& a, int pos, int n)
{
  int row = pos;
  int rowno = row;
  
  T max = a(row, pos);
  T next = 0;
  
  for (++row; row < n; row++)
  {
    next = a(row, pos);
    if (next > max)
    {
      rowno = row;
      max = next;
    }
  }

  return (rowno - pos);
}

/****************************************************************************/
template <class T>
GVector<T> GaussPivot(GMatrix<T>& a, GVector<T>& b, int n)
{
  int h, i, k;

  for (i = 0; i < n - 1; i++)
  {
    k = MaxPivot(a, i, n);

    if (k > 0)
    {
      GVector<T> temp1 = a.Row(i);    T temp2 = b.Element(i);
      a.SetRow(i, a.Row(i+k));       b.SetElement(i, b.Element(i+k));
      a.SetRow(i+k, temp1);          b.SetElement(i+k, temp2);
    }

    for (h = i + 1; h < n; h++)
    {
      T m =  a(h, i) / a(i, i);
      a.SetRow(h, a.Row(h) - m * a.Row(i));
      b.SetElement(h, b.Element(h) - m * b.Element(i));
    }
  }

  GVector<T> x(n);
  T diff;

  x.SetElement(n - 1, b.Element(n - 1) / a(n - 1, n - 1));
  for (i = n - 2; i >= 0; i--)
  {
    diff = 0;
    for (k = i+1; k < n; k++)
       diff += (a(i, k) * x.Element(k));
  
    x.SetElement(i, (b.Element(i) - diff) / a(i, i));
  }

  #if GAUSS_DEBUG
    x.WriteResultVector(std::cout);
  #endif
  return x;
}

/****************************************************************************/
#endif
