/****************************************************************************/
// async mode server/client polling scheme:
//
// 0. inital file states
//
//    client alive file: IDLE
//    poll state file: IDLE
//    server alive file: IDLE
//
// 1. server-to-client
//
// 1a. i: check for server states in iostate file.
//     ii: check poll state file for IDLE state.
//     iii: check client alive file for IDLE state.
//     proceeed if conditions at i, ii and iii is true.
//     send polling signal to client
//
// 1b. client alive file:  IDLE
//     poll state file: SERVER_TO_CLIENT\n (cycle number)
//     server alive file: POLL_CLIENTALIVE\n (cycle number)
//
// 2. client receives server poll signal
//
// 2a. check for server states in iostate file, proceed if server states confirmed:
//     send back client alive signal
//
// 2b. client alive file: CLIENT_ALIVE\n (cycle number from server)
//     poll state file: CLIENT_RESPONSE
//     server alive file: POLL_CLIENTALIVE
//
// 3. server receives client alive signal and sends back acknowledgement
//    send back server acknowledged signal
//
// 3a. client alive file: CLIENT_ALIVE
//     poll state file: SERVER_TO_CLIENT
//     server alive file: SERVER_ACKNOWLEDGED\n (cycle number)
//
// 4. client receives server acknowledgement and sends reset polling timer signal
//    send back polling timer reset signal
//
// 4a. client resets own no-activity polling timer
//
// 4b. client alive file: RESET_POLLING\n (cycle number)
//     poll state file: CLIENT_RESPONSE
//     server alive file: SERVER_ACKNOWLEDGED
//
// 5. server receives reset polling timer signal and resets all polling timers (if timers exist)
//    Sets poll state file to IDLE state. Sets server alive file to IDLE state.
//
// 5a. server resets polling timer, server resets no-activity timer (if timers exist)
//
// 5b. client alive file: RESET_POLLING
//     poll state file: IDLE
//     server alive file: IDLE
//
// 6. client-to-server
//
// 6a. i: check for client states in iostate file.
//     ii: check poll state file for IDLE state.
//     iii: check server alive file for IDLE state.
//     proceeed if conditions at i, ii and iii is true.
//     send polling signal to server
//
// 6b. client alive file:  POLL_SERVERALIVE\n (cycle number)
//     poll state file: CLIENT_TO_SERVER\n (cycle number)
//     server alive file: IDLE
//
// 7. server receives client poll signal
//
// 7a. check for client states in iostate file, proceed if client states confirmed:
//     send back server alive signal
//
// 7b. client alive file: POLL_SERVERALIVE
//     poll state file: SERVER_RESPONSE
//     server alive file: SERVER_ALIVE\n (cycle number from client)
//
// 8. client receives server alive signal and sends back acknowledgement
//    send back client acknowledged signal
//
// 8a. client alive file: CLIENT_ACKNOWLEDGED\n (cycle number)
//     poll state file: CLIENT_TO_SERVER
//     server alive file: SERVER_ALIVE
//
// 9. server receives client acknowledgement and sends reset polling timer signal
//    send back polling timer reset signal
//
// 9a. server resets own no-activity polling timer (if timer exists)
//
// 9b. client alive file: CLIENT_ACKNOWLEDGED
//     poll state file: SERVER_RESPONSE
//     server alive file: RESET_POLLING\n (cycle number)
//
// 10. client receives reset polling timer signal and resets all polling timers
//     Sets poll state file to IDLE state. Sets client alive file to IDLE state.
//
// 10a. client resets polling timer, client resets no-activity timer
//
// 10b. client alive file: IDLE
//      poll state file: IDLE
//      server alive file: RESET_POLLING
//
/****************************************************************************/
// ServerPollingMgr class definition
/****************************************************************************/
ServerPollingMgr::ServerPollingMgr(CalculatorBase* Parent_, bool CreateBackup_):
_Parent(Parent_),
_StrBuffer((char*)::memset(new char[BUFSIZE], 0, BUFSIZE)),

_CycleNum(0),
_CycleActive(false),
_RetrievedCycle(0),

_ClientPollProcess(0),
_ServerPollProcess(0),
_PrevServerProcess(0),
_PrevClientProcess(0),

_RetryReadCount(0),
_RetryPollCount(0),
_PollState(0),
_ServerState(0),
_ClientState(0),

_AllowPollServer(false),
_SleepWaiting(false),
_ResetStatesOnDone(false),
_BreakOnPollError(false),
_KeepAlivePollError(false),
_ConfirmedClientAlive(true),
_ConfirmedServerAlive(true),
_ServerAliveConfSent(false),

_AsyncMode(Parent_ ? Parent_->InAsyncMode():false),
_ExitAtRetryMax(false),
_StopTracking(false),
_WaitUntilResponse(false),
_PausePollResults(false),
_ResetWhenNotReq(false),
_ForceReset(false),

#if CALCLIB_DEBUG10
  _Backup(CreateBackup_ ? (new ServerPollingMgr(Parent_, false)):NULL)
#else
  _Backup(NULL)
#endif
{}

/****************************************************************************/
ServerPollingMgr::~ServerPollingMgr()
{
  _Parent->ClosePollStateFile();
  _Parent->CloseClientAliveFile();
  _Parent->CloseServerAliveFile();

  delete[] _StrBuffer;
  _StrBuffer = NULL;

  if (_Backup)
  {
    delete _Backup;
    _Backup = NULL;
  }
}

/****************************************************************************/
ServerPollingMgr& ServerPollingMgr::operator = (const ServerPollingMgr& Obj_)
{
  if (&Obj_ != this)
  {
    strcpy(_StrBuffer, Obj_._StrBuffer);

    _CycleNum = Obj_._CycleNum;
    _CycleActive = Obj_._CycleActive;
    _RetrievedCycle = Obj_._RetrievedCycle;

    _ClientPollProcess = Obj_._ClientPollProcess;
    _ServerPollProcess = Obj_._ServerPollProcess;
    _PrevClientProcess = Obj_._PrevClientProcess;
    _PrevServerProcess = Obj_._PrevServerProcess;

    _RetryReadCount = Obj_._RetryReadCount;
    _RetryPollCount = Obj_._RetryPollCount;
    _PollState = Obj_._PollState;
    _ServerState = Obj_._ServerState;
    _ClientState = Obj_._ClientState;

    _AllowPollServer = Obj_._AllowPollServer;
    _SleepWaiting = Obj_._SleepWaiting;
    _ResetStatesOnDone = Obj_._ResetStatesOnDone;
    _BreakOnPollError = Obj_._BreakOnPollError;
    _KeepAlivePollError = Obj_._KeepAlivePollError;
    _ConfirmedClientAlive = Obj_._ConfirmedClientAlive;
    _ConfirmedServerAlive = Obj_._ConfirmedServerAlive;
    _ServerAliveConfSent = Obj_._ServerAliveConfSent;

    _AsyncMode = Obj_._AsyncMode;
    _ExitAtRetryMax = Obj_._ExitAtRetryMax;
    _StopTracking = Obj_._StopTracking;
    _WaitUntilResponse = Obj_._WaitUntilResponse;
    _PausePollResults = Obj_._PausePollResults;

    _ResetWhenNotReq = Obj_._ResetWhenNotReq;
    _ForceReset = Obj_._ForceReset;
  }

  return *this;
}

/****************************************************************************/
void ServerPollingMgr::ResetPollResults(bool& PollingServer_,
                                        bool& PollError_,
                                        bool& PollAnswered_)
{
  PollingServer_ = false;
  PollAnswered_ = false;
  PollError_ = false;
}

/****************************************************************************/
void ServerPollingMgr::ShowPollResults(bool PollingServer_,
                                       bool PollError_,
                                       bool PollAnswered_)
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ShowPollResults");
  #endif

  #if CALCLIB_DEBUG10
    #if CALCDRIVER_TEXT_VERSION
      std::cerr <<"Polling Server: " <<PollingServer_ <<endl
                <<"Poll Answered: " <<PollAnswered_ <<endl
                <<"Poll Error: " <<PollError_ <<endl;

      if (_PausePollResults)
        std::cin.get();
    #else
      _Parent->DbgPtr()->ShowInt(PollingServer_, "PollingServer_");
      _Parent->DbgPtr()->ShowInt(PollAnswered_, "PollAnswered_");
      _Parent->DbgPtr()->ShowInt(PollError_, "PollError_");
    #endif

    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void ServerPollingMgr::GivePollResults(bool& PollingServer_,
                                       bool& PollError_,
                                       bool& PollAnswered_)
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("GivePollResults");
  #endif

  if (!PollError_)
    PollingServer_ = ClientIsPolling() && PollServerInProcess();

  if (!PollingServer_)
    PollAnswered_ = _ConfirmedServerAlive &&
                    (PollServerInProcess() || PollServerCompleted(false));

  if (!PollingServer_ && !PollAnswered_)
    PollError_ = _KeepAlivePollError && _RetryPollCount > 0 &&
                 PollServerInProcess();

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(PollingServer_, "PollingServer_");
    _Parent->DbgPtr()->ShowInt(PollAnswered_, "PollAnswered_");
    _Parent->DbgPtr()->ShowInt(PollError_, "PollError_");

    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetSleepWaiting(bool flag_)
{
  _SleepWaiting = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetResetStatesOnDone(bool flag_)
{
  _ResetStatesOnDone = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetBreakOnPollError(bool flag_)
{
  _BreakOnPollError = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetAllowServerPolling(bool flag_)
{
  _AllowPollServer = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetExitAtRetryMax(bool flag_)
{
  _ExitAtRetryMax = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetWaitUntilResponse(bool flag_)
{
  _WaitUntilResponse = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetStopTracking(bool flag_)
{
  _StopTracking = flag_;

  #if CALCLIB_DEBUGGING
    if (_Parent && _Parent->DbgPtr())
    {
      _Parent->DbgPtr()->SetStopTracking(CalculatorBase::SERVERPOLLINGMGR, flag_);

      if (flag_)
        _Parent->DbgPtr()->SetTrackType(CalculatorBase::SERVERPOLLINGMGR);
      else
        _Parent->DbgPtr()->SetTrackType(0);
    }
  #endif

  if (_Backup)
    _Backup->_StopTracking = flag_;

  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetPausePollResults(bool flag_)
{
  _PausePollResults = flag_;
  return this;
}

/****************************************************************************/
bool ServerPollingMgr::PausePollResults() const
{
  return _PausePollResults;
}

/****************************************************************************/
bool ServerPollingMgr::StopTracking() const
{
  return _StopTracking;
}

/****************************************************************************/
bool ServerPollingMgr::ServerIsPolling() const
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ServerIsPolling");
    bool retval = ((_PollState == Mcalc_PollState::SERVER_TO_CLIENT ||
                    _PollState == Mcalc_PollState::CLIENT_RESPONSE) &&
                   _ServerState == Mcalc_KeepAliveState::POLL_CLIENTALIVE);
                   
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return ((_PollState == Mcalc_PollState::SERVER_TO_CLIENT ||
             _PollState == Mcalc_PollState::CLIENT_RESPONSE) &&
            _ServerState == Mcalc_KeepAliveState::POLL_CLIENTALIVE);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ClientIsAlive() const
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ClientIsAlive");
    bool ServerAcked_ = ServerAcked();
    bool retval = ((_PollState == Mcalc_PollState::CLIENT_RESPONSE ||
                    ServerAcked_) &&
                   _ClientState == Mcalc_KeepAliveState::CLIENT_ALIVE &&
                   _RetrievedCycle == _CycleNum);

    retval = retval || ServerAcked_;
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    
    return retval;
  #else
    bool ServerAcked_ = ServerAcked();
    bool retval = ((_PollState == Mcalc_PollState::CLIENT_RESPONSE ||
                    ServerAcked_) &&
                   _ClientState == Mcalc_KeepAliveState::CLIENT_ALIVE &&
                   _RetrievedCycle == _CycleNum);

    retval = retval || ServerAcked_;
    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ServerAcked() const
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ServerAcked");
    bool cond1 = ((_PollState == Mcalc_PollState::SERVER_TO_CLIENT ||
                   _PollState == Mcalc_PollState::CLIENT_RESPONSE) &&
                  _ServerState == Mcalc_KeepAliveState::SERVER_ACKNOWLEDGED &&
                  _RetrievedCycle == _CycleNum);
    bool cond2 = (_PollState == Mcalc_PollState::CLIENT_RESPONSE &&
                  _ServerState == Mcalc_KeepAliveState::SERVER_ACKNOWLEDGED &&
                  _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));
    bool retval = cond1 || cond2;

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    bool cond1 = ((_PollState == Mcalc_PollState::SERVER_TO_CLIENT ||
                   _PollState == Mcalc_PollState::CLIENT_RESPONSE) &&
                  _ServerState == Mcalc_KeepAliveState::SERVER_ACKNOWLEDGED &&
                  _RetrievedCycle == _CycleNum);
    bool cond2 = (_PollState == Mcalc_PollState::CLIENT_RESPONSE &&
                  _ServerState == Mcalc_KeepAliveState::SERVER_ACKNOWLEDGED &&
                  _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));
    bool retval = cond1 || cond2;

    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ResetSignalSent() const
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ResetSignalSent");
    bool ClientResponse_ = _PollState == Mcalc_PollState::CLIENT_RESPONSE &&
                           ClientReceiveCompleted(false);
    bool retval = ((ClientResponse_ || PollingResetByServer()) &&
                   _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
                   _RetrievedCycle == _CycleNum);

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    bool ClientResponse_ = _PollState == Mcalc_PollState::CLIENT_RESPONSE &&
                           ClientReceiveCompleted(false);
    return ((ClientResponse_ || PollingResetByServer()) &&
            _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
            _RetrievedCycle == _CycleNum);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::PollingResetByServer() const
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("PollingResetByServer");
    bool retval = (ClientReceiveCompleted(false) &&
                   _PollState == Mcalc_KeepAliveState::IDLE &&
                   _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
                   _ServerState == Mcalc_KeepAliveState::IDLE);

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return (ClientReceiveCompleted(false) &&
            _PollState == Mcalc_KeepAliveState::IDLE &&
            _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
            _ServerState == Mcalc_KeepAliveState::IDLE);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::AtInitialState()
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("AtInitialState");
    int IoState_ = _Parent->SetNoPolling(true)
                          ->Bare_GetIoState(0);
    bool PollServerDone_ = _ServerPollProcess == REQUIRED;
    bool PollClientDone_ = ClientReceiveCompleted(false) ||
                           _ClientPollProcess == INITIALSTATE;
    bool retval =
           (_Parent->IsClientStates(IoState_) &&
            PollServerDone_ && PollClientDone_ &&
            (_ServerState == Mcalc_KeepAliveState::IDLE ||
             _ServerState == Mcalc_KeepAliveState::RESET_POLLING) &&
            _PollState == Mcalc_KeepAliveState::IDLE &&
            (_ClientState == Mcalc_KeepAliveState::IDLE ||
             _ClientState == Mcalc_KeepAliveState::RESET_POLLING));

    _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
    _Parent->DbgPtr()->ShowInt(PollClientDone_, "PollClientDone_");
    _Parent->DbgPtr()->ShowInt(PollServerDone_, "PollServerDone_");
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    int IoState_ = _Parent->SetNoPolling(true)
                          ->Bare_GetIoState(0);
    bool PollServerDone_ = _ServerPollProcess == REQUIRED;
    bool PollClientDone_ = ClientReceiveCompleted(false) ||
                           _ClientPollProcess == INITIALSTATE;

    return (_Parent->IsClientStates(IoState_) &&
            PollServerDone_ && PollClientDone_ &&
            (_ServerState == Mcalc_KeepAliveState::IDLE ||
             _ServerState == Mcalc_KeepAliveState::RESET_POLLING) &&
            _PollState == Mcalc_KeepAliveState::IDLE &&
            (_ClientState == Mcalc_KeepAliveState::IDLE ||
             _ClientState == Mcalc_KeepAliveState::RESET_POLLING));
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ClientIsPolling() const
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ClientIsPolling");
    bool retval = ((_PollState == Mcalc_PollState::CLIENT_TO_SERVER ||
                    _PollState == Mcalc_PollState::SERVER_RESPONSE) &&
                   _ClientState == Mcalc_KeepAliveState::POLL_SERVERALIVE);

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return ((_PollState == Mcalc_PollState::CLIENT_TO_SERVER ||
             _PollState == Mcalc_PollState::SERVER_RESPONSE) &&
            _ClientState == Mcalc_KeepAliveState::POLL_SERVERALIVE);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ServerIsAlive() const
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ServerIsAlive");
    bool ClientAcked_ = ClientAcked();
    bool retval = ((_PollState == Mcalc_PollState::SERVER_RESPONSE ||
                    ClientAcked_) &&
                   _ServerState == Mcalc_KeepAliveState::SERVER_ALIVE &&
                   _RetrievedCycle == _CycleNum);

    retval = retval || ClientAcked_;
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    
    return retval;
  #else
    bool ClientAcked_ = ClientAcked();
    bool retval = ((_PollState == Mcalc_PollState::SERVER_RESPONSE ||
                    ClientAcked_) &&
                   _ServerState == Mcalc_KeepAliveState::SERVER_ALIVE &&
                   _RetrievedCycle == _CycleNum);

    retval = retval || ClientAcked_;
    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ClientAcked() const
{
  #if CALCLIB_DEBUG9
    _Parent->DbgPtr()->EnterLevel("ClientAcked");
    bool cond1 = ((_PollState == Mcalc_PollState::CLIENT_TO_SERVER ||
                   _PollState == Mcalc_PollState::SERVER_RESPONSE) &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _RetrievedCycle == _CycleNum);
    bool cond2 = (_PollState == Mcalc_PollState::SERVER_RESPONSE &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _ServerState == Mcalc_KeepAliveState::RESET_POLLING &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));
    bool retval = cond1 || cond2;

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    bool cond1 = ((_PollState == Mcalc_PollState::CLIENT_TO_SERVER ||
                   _PollState == Mcalc_PollState::SERVER_RESPONSE) &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _RetrievedCycle == _CycleNum);
    bool cond2 = (_PollState == Mcalc_PollState::SERVER_RESPONSE &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _ServerState == Mcalc_KeepAliveState::RESET_POLLING &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));
    return (cond1 || cond2);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ResetRequested() const
{
  #if CALCLIB_DEBUG9
    _Parent->DbgPtr()->EnterLevel("ResetRequested");
    bool PollingReset_ = PollingReset();
    bool retval = ((_PollState == Mcalc_PollState::SERVER_RESPONSE || PollingReset_) &&
                   _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                   _ServerState == Mcalc_KeepAliveState::RESET_POLLING &&
                   (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));

    retval = retval || PollingReset_;
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();

    return retval;
  #else
    bool PollingReset_ = PollingReset();
    bool retval = ((_PollState == Mcalc_PollState::SERVER_RESPONSE || PollingReset_) &&
                   _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                   _ServerState == Mcalc_KeepAliveState::RESET_POLLING &&
                   (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));

    retval = retval || PollingReset_;
    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::PollingReset() const
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("PollingReset");
    bool retval = (PollServerCompleted(false) &&
                   _PollState == Mcalc_KeepAliveState::IDLE &&
                   _ClientState == Mcalc_KeepAliveState::IDLE &&
                   _ServerState == Mcalc_KeepAliveState::RESET_POLLING);

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return (PollServerCompleted(false) &&
            _PollState == Mcalc_KeepAliveState::IDLE &&
            _ClientState == Mcalc_KeepAliveState::IDLE &&
            _ServerState == Mcalc_KeepAliveState::RESET_POLLING);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::WaitingForServerResponse(bool* FromClt_) const
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("WaitingForServerResponse");
    
    bool FromClient_ = SHOWMETHOD(_Parent, PollServerInProcess());
    bool FromServer_ = SHOWMETHOD(_Parent, ClientReceiveInProcess());

    _Parent->DbgPtr()->ShowInt(FromClient_, "FromClient_");
    _Parent->DbgPtr()->ShowInt(FromServer_, "FromServer_");
  #else
    bool FromClient_ = PollServerInProcess();
    bool FromServer_ = ClientReceiveInProcess();
  #endif

  if (FromClt_)
    *FromClt_ = FromClient_;

  bool retval = (FromClient_ ? (ClientIsPolling() || ClientAcked()):
                 FromServer_ ? (ClientIsAlive() || ResetSignalSent()):
                               false);

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return retval;
}

/****************************************************************************/
bool ServerPollingMgr::ResponseRequired(int PrevPollState_,
                                        int LoopGuard_, int LoopLimit_)
{
  LoopGuard_++;
  return (_WaitUntilResponse && LoopGuard_ < LoopLimit_ &&
          ((PrevPollState_ == Mcalc_PollState::CLIENT_TO_SERVER &&
            _PollState != Mcalc_PollState::SERVER_RESPONSE) ||
           (PrevPollState_ == Mcalc_PollState::CLIENT_RESPONSE &&
            _PollState != Mcalc_PollState::SERVER_TO_CLIENT &&
            _PollState != Mcalc_PollState::IDLE)));
}

/****************************************************************************/
int ServerPollingMgr::ReadServerState(int Sleep_)
{
  int mult_ = 0;
  int DelayLen_ = IOSTATE_DELAY_LENGTH * DELAY_MULT;
  int PrevPollState_ = _PollState;
  int LoopGuard_ = 0;
  int LoopMax_ = LOOP_GUARD_LIMIT;
  bool SleepDone_ = false;

  int x;
  char* testp = NULL;
  
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ReadServerState");
    
    FILE* Pfptr_ = SHOWPARENTMETHOD(_Parent, GetPollStateFile(READ, true));
    FILE* Sfptr_ = SHOWPARENTMETHOD(_Parent, GetServerAliveFile(READ, true));
  #else
    FILE* Pfptr_ = _Parent->GetPollStateFile(READ, true);
    FILE* Sfptr_ = _Parent->GetServerAliveFile(READ, true);
  #endif

  if (Pfptr_ && feof(Pfptr_))
    rewind(Pfptr_);

  if (Pfptr_ && !feof(Pfptr_))
  {
    mult_ = (Sleep_ > 1) ? Sleep_:1;
    SleepDone_ = true;
    
    #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
      Sleep(mult_ * DelayLen_);
    #else
      ::usleep(mult_ * DelayLen_);
    #endif
  }

  do
  {
    if (Pfptr_ && feof(Pfptr_))
      rewind(Pfptr_);
  
    if (Pfptr_ && !feof(Pfptr_))
    {
      ::memset(_StrBuffer, 0, BUFSIZE);
      testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);

      for (x = 0; !testp && x < LoopMax_; x++)
      {
        #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
          Sleep(DelayLen_);
        #else
          ::usleep(DelayLen_);
        #endif

        SleepDone_ = true;
        Pfptr_ = _Parent->GetPollStateFile(READ, true);

        if (Pfptr_ && feof(Pfptr_))
          rewind(Pfptr_);

        if (Pfptr_ && !feof(Pfptr_))
          testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);
      }

      if (testp)
        _PollState = atoi(testp);
    }

    // Wait some time before getting server state
    if (!SleepDone_ &&
        Sleep_ && (!_SleepWaiting || WaitingForServerResponse()))
    {
      mult_ = (Sleep_ > 1) ? Sleep_:1;
      SleepDone_ = true;

      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(mult_ * DelayLen_);
      #else
        ::usleep(mult_ * DelayLen_);
      #endif

      #if CALCTEST_DEBUG3
        _Parent->DbgPtr()->ShowMessage("WaitingForServerResponse == 1");
        _Parent->DbgPtr()->ShowInt(_SleepWaiting, "_SleepWaiting");
        _Parent->DbgPtr()->ShowInt(Sleep_, "Sleep_");
      #endif
    }

    SleepDone_ = false;
  }
  while (ResponseRequired(PrevPollState_, LoopGuard_, LoopMax_));

  if (Sfptr_ && feof(Sfptr_))
    rewind(Sfptr_);

  if (Sfptr_ && !feof(Sfptr_))
  {
    ::memset(_StrBuffer, 0, BUFSIZE);
    testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
    
    for (x = 0; !testp && x < LoopMax_; x++)
    {
      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(DelayLen_);
      #else
        ::usleep(DelayLen_);
      #endif
    
      Sfptr_ = _Parent->GetServerAliveFile(READ, true);

      if (Sfptr_ && feof(Sfptr_))
        rewind(Sfptr_);

      if (Sfptr_ && !feof(Sfptr_))
        testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
    }

    if (testp)
      _ServerState = atoi(testp);

    if (Sfptr_ && !feof(Sfptr_))
    {
      testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
      _RetrievedCycle = testp ? atoi(testp):0;
    }
    else
      _RetrievedCycle = 0;
  }
  
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(_PollState, "_PollState");
    _Parent->DbgPtr()->ShowInt(_ClientState, "_ServerState");
    _Parent->DbgPtr()->ShowInt(_RetrievedCycle, "_RetrievedCycle");
    
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return _ServerState;
}

/****************************************************************************/
int ServerPollingMgr::ReadClientState(bool RdPoll_)
{
  int DelayLen_ = IOSTATE_DELAY_LENGTH * DELAY_MULT;
  int LoopMax_ = LOOP_GUARD_LIMIT;

  int x;
  char* testp = NULL;

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ReadClientState");
    
    FILE* Pfptr_ = SHOWPARENTMETHOD(_Parent, GetPollStateFile(READ, true));
    FILE* Cfptr_ = SHOWPARENTMETHOD(_Parent, GetClientAliveFile(READ, true));
  #else
    FILE* Pfptr_ = _Parent->GetPollStateFile(READ, true);
    FILE* Cfptr_ = _Parent->GetClientAliveFile(READ, true);
  #endif

  if (Pfptr_ && feof(Pfptr_) && RdPoll_)
    rewind(Pfptr_);

  if (Pfptr_ && !feof(Pfptr_) && RdPoll_)
  {
    #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
      Sleep(DelayLen_);
    #else
      ::usleep(DelayLen_);
    #endif
  }

  if (Pfptr_ && !feof(Pfptr_) && RdPoll_)
  {
    ::memset(_StrBuffer, 0, BUFSIZE);
    testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);

    for (x = 0; !testp && x < LoopMax_; x++)
    {
      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(DelayLen_);
      #else
        ::usleep(DelayLen_);
      #endif
    
      Pfptr_ = _Parent->GetPollStateFile(READ, true);

      if (Pfptr_ && feof(Pfptr_))
        rewind(Pfptr_);

      if (Pfptr_ && !feof(Pfptr_))
        testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);
    }

    if (testp)
      _PollState = atoi(testp);
  }

  if (Cfptr_ && feof(Cfptr_))
    rewind(Cfptr_);

  if (Cfptr_ && !feof(Cfptr_))
  {
    ::memset(_StrBuffer, 0, BUFSIZE);
    testp = fgets(_StrBuffer, BUFSIZE, Cfptr_);

    for (x = 0; !testp && x < LoopMax_; x++)
    {
      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(DelayLen_);
      #else
        ::usleep(DelayLen_);
      #endif
    
      Cfptr_ = _Parent->GetClientAliveFile(READ, true);

      if (Cfptr_ && feof(Cfptr_))
        rewind(Cfptr_);

      if (Cfptr_ && !feof(Cfptr_))
        testp = fgets(_StrBuffer, BUFSIZE, Cfptr_);
    }

    if (testp)
      _ClientState = atoi(testp);
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(_PollState, "_PollState");
    _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
    _Parent->DbgPtr()->ShowInt(_RetrievedCycle, "_RetrievedCycle");
    
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return _ClientState;
}

/****************************************************************************/
int ServerPollingMgr::ReadClientState()
{
  return ReadClientState(true);
}

/****************************************************************************/
void ServerPollingMgr::ReadAllStates(int Sleep_)
{
  int mult_ = 0;
  int DelayLen_ = IOSTATE_DELAY_LENGTH * DELAY_MULT;
  int PrevPollState_ = _PollState;
  int LoopGuard_ = 0;
  int LoopMax_ = LOOP_GUARD_LIMIT;
  bool SleepDone_ = false;

  int x;
  char* testp = NULL;

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ReadAllStates");

    SHOWMETHOD(_Parent, ReadClientState(false));
    FILE* Pfptr_ = SHOWPARENTMETHOD(_Parent, GetPollStateFile(READ, true));
    FILE* Sfptr_ = SHOWPARENTMETHOD(_Parent, GetServerAliveFile(READ, true));
  #else
    ReadClientState(false);
    FILE* Pfptr_ = _Parent->GetPollStateFile(READ, true);
    FILE* Sfptr_ = _Parent->GetServerAliveFile(READ, true);
  #endif

  if (Pfptr_ && feof(Pfptr_))
    rewind(Pfptr_);

  if (Pfptr_ && !feof(Pfptr_))
  {
    mult_ = (Sleep_ > 1) ? Sleep_:1;
    SleepDone_ = true;
  
    #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
      Sleep(mult_ * DelayLen_);
    #else
      ::usleep(mult_ * DelayLen_);
    #endif
  }

  do
  {
    if (Pfptr_ && feof(Pfptr_))
      rewind(Pfptr_);
  
    if (Pfptr_ && !feof(Pfptr_))
    {
      ::memset(_StrBuffer, 0, BUFSIZE);
      testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);

      for (x = 0; !testp && x < LoopMax_; x++)
      {
        #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
          Sleep(DelayLen_);
        #else
          ::usleep(DelayLen_);
        #endif

        SleepDone_ = true;
        Pfptr_ = _Parent->GetPollStateFile(READ, true);

        if (Pfptr_ && feof(Pfptr_))
          rewind(Pfptr_);
        
        if (Pfptr_ && !feof(Pfptr_))
          testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);
      }

      if (testp)
        _PollState = atoi(testp);
    }

    // Wait some time before getting server state
    if (!SleepDone_ &&
        Sleep_ && (!_SleepWaiting || WaitingForServerResponse()))
    {
      mult_ = (Sleep_ > 1) ? Sleep_:1;
      SleepDone_ = true;

      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(mult_ * DelayLen_);
      #else
        ::usleep(mult_ * DelayLen_);
      #endif

      #if CALCTEST_DEBUG3
        _Parent->DbgPtr()->ShowMessage("WaitingForServerResponse == 1");
        _Parent->DbgPtr()->ShowInt(_SleepWaiting, "_SleepWaiting");
        _Parent->DbgPtr()->ShowInt(Sleep_, "Sleep_");
      #endif
    }

    SleepDone_ = false;
  }
  while (ResponseRequired(PrevPollState_, LoopGuard_, LoopMax_));

  if (Sfptr_ && feof(Sfptr_))
    rewind(Sfptr_);

  if (Sfptr_ && !feof(Sfptr_))
  {
    ::memset(_StrBuffer, 0, BUFSIZE);
    testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
    
    for (x = 0; !testp && x < LoopMax_; x++)
    {
      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(DelayLen_);
      #else
        ::usleep(DelayLen_);
      #endif

      Sfptr_ = _Parent->GetServerAliveFile(READ, true);

      if (Sfptr_ && feof(Sfptr_))
        rewind(Sfptr_);

      if (Sfptr_ && !feof(Sfptr_))
        testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
    }

    if (testp)
      _ServerState = atoi(testp);

    if (Sfptr_ && !feof(Sfptr_))
    {
      testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
      _RetrievedCycle = testp ? atoi(testp):0;
    }
    else
      _RetrievedCycle = 0;
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(_PollState, "_PollState");
    _Parent->DbgPtr()->ShowInt(_ServerState, "_ServerState");
    _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
    _Parent->DbgPtr()->ShowInt(_RetrievedCycle, "_RetrievedCycle");
    
    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::SetPollState(int State_)
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("SetPollState");
    FILE* Fptr_ = SHOWPARENTMETHOD(_Parent, GetPollStateFile(WRITE));
  #else
    FILE* Fptr_ = _Parent->GetPollStateFile(WRITE);
  #endif

  if (Fptr_)
  {
    fprintf(Fptr_, "%d\n%d", State_, IncCycleNum());
    _Parent->ClosePollStateFile();

    #if CALCLIB_DEBUG10
      _Parent->DbgPtr()->ShowInt(State_, "State_");
      _Parent->DbgPtr()->ShowInt(true, "retval");
      _Parent->DbgPtr()->LeaveLevel();
    #endif

    return true;
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(false, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return false;
}

/****************************************************************************/
bool ServerPollingMgr::SetClientState(int State_)
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("SetClientState");
    FILE* Fptr_ = SHOWPARENTMETHOD(_Parent, GetClientAliveFile(WRITE));
  #else
    FILE* Fptr_ = _Parent->GetClientAliveFile(WRITE);
  #endif

  if (Fptr_)
  {
    fprintf(Fptr_, "%d\n%d", State_, IncCycleNum());
    _Parent->CloseClientAliveFile();

    #if CALCLIB_DEBUG10
      _Parent->DbgPtr()->ShowInt(State_, "State_");
      _Parent->DbgPtr()->ShowInt(true, "retval");
      _Parent->DbgPtr()->LeaveLevel();
    #endif

    return true;
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(false, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return false;
}

/****************************************************************************/
void ServerPollingMgr::ResetPollingVars()
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ResetPollingVars");
    SHOWMETHOD(_Parent, ResetPollCount());
  #else
    ResetPollCount();
  #endif

  _KeepAlivePollError = false;
  _ConfirmedServerAlive = false;

  if (_ServerAliveConfSent)
  {
    #if CALCLIB_DEBUG10
      SHOWPARENTMETHOD(_Parent, SendServerAlive(false));
    #else
      _Parent->SendServerAlive(false);
    #endif

    _ServerAliveConfSent = false;
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void ServerPollingMgr::AcceptPollCycle()
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("AcceptPollCycle");
  #endif

  _CycleActive = true;
  _CycleNum = _RetrievedCycle;

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(_CycleActive, "_CycleActive");
    _Parent->DbgPtr()->ShowInt(_CycleNum, "_CycleNum");
    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void ServerPollingMgr::ResetPollCycle()
{
  _CycleActive = false;
}

/****************************************************************************/
void ServerPollingMgr::SetNewPollCycle()
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("SetNewPollCycle");
  #endif

  if (!_CycleActive)
  {
    _CycleActive = true;
    _CycleNum++;
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(_CycleActive, "_CycleActive");
    _Parent->DbgPtr()->ShowInt(_CycleNum, "_CycleNum");
    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
int ServerPollingMgr::IncCycleNum()
{
  if (!_CycleActive)
    return 0;

  return _CycleNum;
}

/****************************************************************************/
bool ServerPollingMgr::IsServerAliveStates(int State_)
{
  return
  (
    State_ == Mcalc_KeepAliveState::POLL_CLIENTALIVE ||
    State_ == Mcalc_KeepAliveState::SERVER_ALIVE
  );
}

/****************************************************************************/
bool ServerPollingMgr::IsClientAliveStates(int State_)
{
  return
  (
    State_ == Mcalc_KeepAliveState::CLIENT_ALIVE ||
    State_ == Mcalc_KeepAliveState::POLL_SERVERALIVE
  );
}

/****************************************************************************/
bool ServerPollingMgr::SendPollServerSignal()
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("SendPollServerSignal");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool Required_ = _ServerPollProcess == REQUIRED;
  bool DoPollServer_ = !_Parent->IsServerStates(IoState_) &&
                       AtInitialState() && Required_;

  if (DoPollServer_ && _Parent->GetClientAliveFile(WRITE))
  {
    ResetPollingVars();
  
    _ClientState = Mcalc_KeepAliveState::POLL_SERVERALIVE;
    _ServerPollProcess = INPROCESS;

    SetTrackingOnStateChange();

    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_PollState::CLIENT_TO_SERVER;
      #if CALCLIB_DEBUG10
        _Parent->DbgPtr()->ShowInt(Required_, "Required_");
        _Parent->DbgPtr()->ShowInt(DoPollServer_, "DoPollServer_");
        _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
        _Parent->DbgPtr()->ShowInt(_PollState, "_PollState");
        _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess");
        _Parent->DbgPtr()->LeaveLevel();
        _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
      #endif
      
      return SetPollState(_PollState);
    }
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(Required_, "Required_");
    _Parent->DbgPtr()->ShowInt(DoPollServer_, "DoPollServer_");
    _Parent->DbgPtr()->LeaveLevel();
    _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
  #endif
  return false;
}

/****************************************************************************/
void ServerPollingMgr::ConfirmServerAlive(bool ClientReceive_)
{
  _RetryPollCount = 0;
  _KeepAlivePollError = false;
  _ConfirmedServerAlive = ClientReceive_ ? ServerAcked():ServerIsAlive();
}

/****************************************************************************/
bool ServerPollingMgr::SendClientAckServerSignal()
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("SendClientAckServerSignal");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool DoAck_ = !_Parent->IsServerStates(IoState_) && ServerIsAlive() &&
                PollServerInProcess() && _ConfirmedServerAlive;
            
  if (DoAck_ && _Parent->GetClientAliveFile(WRITE))
  {
    _ClientState = Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED;
    _ServerPollProcess = INPROCESS;

    SetTrackingOnStateChange();
    
    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_PollState::CLIENT_TO_SERVER;
      #if CALCLIB_DEBUG10
        _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
        _Parent->DbgPtr()->ShowInt(DoAck_, "DoAck_");
        _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
        _Parent->DbgPtr()->ShowInt(_PollState, "_PollState");
        _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess");
        _Parent->DbgPtr()->LeaveLevel();
        _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
      #endif
      
      return SetPollState(_PollState);
    }
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
    _Parent->DbgPtr()->ShowInt(DoAck_, "DoAck_");
    _Parent->DbgPtr()->LeaveLevel();
    _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
  #endif
  return false;
}

/****************************************************************************/
bool ServerPollingMgr::ResetPolling()
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ResetPolling");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool DoReset_ = !_Parent->IsServerStates(IoState_) && ResetRequested() &&
                  PollServerInProcess() && _ConfirmedServerAlive;

  if (DoReset_ && _Parent->GetClientAliveFile(WRITE))
  {
    ResetPollCycle();
    _KeepAlivePollError = false;
    
    _ClientState = Mcalc_KeepAliveState::IDLE;
    _ServerPollProcess = COMPLETED;

    SetTrackingOnStateChange();

    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_KeepAliveState::IDLE;
      #if CALCLIB_DEBUG10
        _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
        _Parent->DbgPtr()->ShowInt(DoReset_, "DoReset_");
        _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
        _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
        _Parent->DbgPtr()->ShowInt(_PollState, "_PollState");
        _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess");
        _Parent->DbgPtr()->LeaveLevel();
        _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
      #endif
      
      bool retval = SetPollState(_PollState);
      bool Alive_ = _ConfirmedServerAlive && _ConfirmedClientAlive;

      // Reset client side no-activity timer
      if (!_ServerAliveConfSent)
      {
        #if CALCLIB_DEBUG10
          SHOWPARENTMETHOD(_Parent, SendServerAlive(Alive_));
        #else
          _Parent->SendServerAlive(Alive_);
        #endif

        _ServerAliveConfSent = true;
      }

      return retval;
    }
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
    _Parent->DbgPtr()->ShowInt(DoReset_, "DoReset_");
    _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
    _Parent->DbgPtr()->LeaveLevel();
    _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
  #endif
  return false;
}

/****************************************************************************/
bool ServerPollingMgr::SendClientAliveSignal()
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("SendClientAliveSignal");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool Required_ = _ClientPollProcess == REQUIRED;
  bool DoSendAlive_ = _Parent->IsServerStates(IoState_) &&
                      ServerIsPolling() && Required_;
            
  if (DoSendAlive_ && _Parent->GetClientAliveFile(WRITE))
  {
    _KeepAlivePollError = false;
    _ClientState = Mcalc_KeepAliveState::CLIENT_ALIVE;
    _ClientPollProcess = INPROCESS;

    SetTrackingOnStateChange();
    
    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_PollState::CLIENT_RESPONSE;
      #if CALCLIB_DEBUG10
        _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
        _Parent->DbgPtr()->ShowInt(Required_, "Required_");
        _Parent->DbgPtr()->ShowInt(DoSendAlive_, "DoSendAlive_");
        _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
        _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
        _Parent->DbgPtr()->ShowInt(_PollState, "_PollState");
        _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess");
        _Parent->DbgPtr()->LeaveLevel();
        _Parent->DbgPtr()->LeaveLevel();  // from ReceiveKeepAlivePoll()
      #endif

      return SetPollState(_PollState);
    }
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
    _Parent->DbgPtr()->ShowInt(Required_, "Required_");
    _Parent->DbgPtr()->ShowInt(DoSendAlive_, "DoSendAlive_");
    _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
    _Parent->DbgPtr()->LeaveLevel();
    _Parent->DbgPtr()->LeaveLevel();  // from ReceiveKeepAlivePoll()
  #endif
  return false;
}

/****************************************************************************/
void ServerPollingMgr::ConfirmClientAlive(bool PollServer_)
{
  _KeepAlivePollError = false;
  _ConfirmedClientAlive = PollServer_ ? ServerIsAlive():ClientIsAlive();
}

/****************************************************************************/
bool ServerPollingMgr::SendResetPollingSignal()
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("SendResetPollingSignal");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool DoReset_ = _Parent->IsServerStates(IoState_) && ServerAcked() &&
                  ClientReceiveInProcess() && _ConfirmedClientAlive;
            
  if (DoReset_ && _Parent->GetClientAliveFile(WRITE))
  {
    ResetPollCycle();
    _KeepAlivePollError = false;
    
    _ClientState = Mcalc_KeepAliveState::RESET_POLLING;
    _ClientPollProcess = COMPLETED;

    SetTrackingOnStateChange();
    
    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_PollState::CLIENT_RESPONSE;
      #if CALCLIB_DEBUG10
        _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
        _Parent->DbgPtr()->ShowInt(DoReset_, "DoReset_");
        _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
        _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
        _Parent->DbgPtr()->ShowInt(_PollState, "_PollState");
        _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess");
        _Parent->DbgPtr()->LeaveLevel();
        _Parent->DbgPtr()->LeaveLevel();  // from ReceiveKeepAlivePoll()
      #endif

      bool retval = SetPollState(_PollState);
      bool Alive_ = _ConfirmedServerAlive && _ConfirmedClientAlive;

      // Reset client side no-activity timer
      if (!_ServerAliveConfSent)
      {
        #if CALCLIB_DEBUG10
          SHOWPARENTMETHOD(_Parent, SendServerAlive(Alive_));
        #else
          _Parent->SendServerAlive(Alive_);
        #endif
        
        _ServerAliveConfSent = true;
      }

      return retval;
    }
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
    _Parent->DbgPtr()->ShowInt(DoReset_, "DoReset_");
    _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
    _Parent->DbgPtr()->LeaveLevel();
    _Parent->DbgPtr()->LeaveLevel();  // from ReceiveKeepAlivePoll()
  #endif
  return false;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::ResetWhenNotRequired(bool Forced_)
{
  _ResetWhenNotReq = true;
  _ForceReset = Forced_;
  SetStopTracking(true);
  
  return this;
}

/****************************************************************************/
bool ServerPollingMgr::PollServerRequired(int Sleep_, bool ResponseOnly_)
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("PollServerRequired");
    _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess(1)");
  #endif

  ReadAllStates(Sleep_);

  bool ServerResponse_ = _PollState == Mcalc_PollState::SERVER_RESPONSE;
  bool ResetReq_ = ResetRequested();
  bool Required_ = ((_AllowPollServer && !ResponseOnly_) ||
                    ServerResponse_ || ResetReq_ || _KeepAlivePollError);

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess(2)");
    _Parent->DbgPtr()->ShowInt(_ResetWhenNotReq, "_ResetWhenNotReq(1)");

    _Parent->DbgPtr()->ShowInt(_AllowPollServer, "_AllowPollServer(1)");
    _Parent->DbgPtr()->ShowInt(ResponseOnly_, "ResponseOnly_(1)");
    _Parent->DbgPtr()->ShowInt(ServerResponse_, "ServerResponse_(1)");
    _Parent->DbgPtr()->ShowInt(ResetReq_, "ResetReq_(1)");
    _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError(1)");
    _Parent->DbgPtr()->ShowInt(Required_, "Required_(1)");
  #endif

  if (_ServerPollProcess == INITIALSTATE ||
      _ServerPollProcess == COMPLETED)
    _ServerPollProcess = Required_ ? REQUIRED:NOTREQUIRED;

  bool retval = (_ServerPollProcess == REQUIRED ||
                 (_ServerPollProcess == INPROCESS && !_KeepAlivePollError));

  if (!retval)
    SetTrackingOnStateChange();

  #if CALCLIB_DEBUG10
    if (_ResetWhenNotReq && _ServerPollProcess == NOTREQUIRED)
    {
      _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess(3)");
      SHOWMETHOD(_Parent, SetStopTracking(true));
      SHOWMETHOD(_Parent, ResetServerPolling(_ForceReset));
      _ResetWhenNotReq = false;
    }

    _Parent->DbgPtr()->ShowInt(_ResetWhenNotReq, "_ResetWhenNotReq(2)");
    _Parent->DbgPtr()->ShowInt(_AllowPollServer, "_AllowPollServer(2)");
    _Parent->DbgPtr()->ShowInt(ResponseOnly_, "ResponseOnly_(2)");
    _Parent->DbgPtr()->ShowInt(ServerResponse_, "ServerResponse_(2)");
    _Parent->DbgPtr()->ShowInt(ResetReq_, "ResetReq_(2)");
    _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError(2)");
    _Parent->DbgPtr()->ShowInt(Required_, "Required_(2)");

    _Parent->DbgPtr()->ShowInt(_PollState, "_PollState");
    _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
    _Parent->DbgPtr()->ShowInt(_ServerState, "_ServerState");
    _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess(4)");
    _Parent->DbgPtr()->ShowInt(retval, "retval");

    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    if (_ResetWhenNotReq && _ServerPollProcess == NOTREQUIRED)
    {
      SetStopTracking(true);
      ResetServerPolling(_ForceReset);
      _ResetWhenNotReq = false;
    }

    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::PollServerNotRequired(bool InclAtInitial_) const
{
  return ((_ServerPollProcess == NOTREQUIRED ||
           (_ServerPollProcess == INITIALSTATE && InclAtInitial_)) && !_KeepAlivePollError);
}

/****************************************************************************/
bool ServerPollingMgr::PollServerInProcess() const
{
  return (_ServerPollProcess == INPROCESS && !_KeepAlivePollError);
}

/****************************************************************************/
bool ServerPollingMgr::PollServerCompleted_IMPL(bool InclNotReq_, bool InclPrev_) const
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("PollServerCompleted_IMPL");
  #endif

  bool ResetCond_ = false;
  bool CurrCond_ = false;
  bool PrevCond_ = false;

  if (_ResetStatesOnDone && InclPrev_)
  {
    ResetCond_ = _ServerPollProcess == INITIALSTATE;
    PrevCond_ = ((_PrevServerProcess == COMPLETED ||
                  (_PrevServerProcess == NOTREQUIRED && InclNotReq_)) && !_KeepAlivePollError);
  }

  CurrCond_ = ((_ServerPollProcess == COMPLETED ||
                (_ServerPollProcess == NOTREQUIRED && InclNotReq_)) && !_KeepAlivePollError);

  #if CALCLIB_DEBUG10
    bool retval = (CurrCond_ || (_ResetStatesOnDone && ResetCond_ && PrevCond_));
    _Parent->DbgPtr()->ShowInt(_ResetStatesOnDone, "_ResetStatesOnDone");
    _Parent->DbgPtr()->ShowInt(InclPrev_, "InclPrev_");
    _Parent->DbgPtr()->ShowInt(InclNotReq_, "InclNotReq_");
    _Parent->DbgPtr()->ShowInt(ResetCond_, "ResetCond_");
    _Parent->DbgPtr()->ShowInt(PrevCond_, "PrevCond_");
    _Parent->DbgPtr()->ShowInt(CurrCond_, "CurrCond_");
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return (CurrCond_ || (_ResetStatesOnDone && ResetCond_ && PrevCond_));
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::PollServerCompleted(bool InclNotReq_) const
{
  return PollServerCompleted_IMPL(InclNotReq_, true);
}

/****************************************************************************/
bool ServerPollingMgr::ClientReceiveRequired(int Sleep_)
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ClientReceiveRequired");
    _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess(1)");
  #endif

  ReadAllStates(Sleep_);

  bool ServerToClient_ = _PollState == Mcalc_PollState::SERVER_TO_CLIENT;
  bool ServerPolling_ = _ServerState == Mcalc_KeepAliveState::POLL_CLIENTALIVE;
  bool Required_ = ServerToClient_ || ServerPolling_;

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess(2)");
    _Parent->DbgPtr()->ShowInt(_ResetWhenNotReq, "_ResetWhenNotReq(1)");

    _Parent->DbgPtr()->ShowInt(ServerToClient_, "ServerToClient_(1)");
    _Parent->DbgPtr()->ShowInt(ServerPolling_, "ServerPolling_(1)");
    _Parent->DbgPtr()->ShowInt(Required_, "Required_(1)");
  #endif

  if (_ClientPollProcess == INITIALSTATE ||
      _ClientPollProcess == COMPLETED)
    _ClientPollProcess = Required_ ? REQUIRED:NOTREQUIRED;

  bool retval = (_ClientPollProcess == REQUIRED ||
                 (_ClientPollProcess == INPROCESS && !_KeepAlivePollError));

  if (!retval)
    SetTrackingOnStateChange();

  #if CALCLIB_DEBUG10
    if (_ResetWhenNotReq && _ClientPollProcess == NOTREQUIRED)
    {
      _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess(3)");
      SHOWMETHOD(_Parent, SetStopTracking(true));
      SHOWMETHOD(_Parent, ResetReceivePoll(_ForceReset));
      _ResetWhenNotReq = false;
    }

    _Parent->DbgPtr()->ShowInt(_ResetWhenNotReq, "_ResetWhenNotReq(2)");
    _Parent->DbgPtr()->ShowInt(ServerToClient_, "ServerToClient_(2)");
    _Parent->DbgPtr()->ShowInt(ServerPolling_, "ServerPolling_(2)");
    _Parent->DbgPtr()->ShowInt(Required_, "Required_(2)");

    _Parent->DbgPtr()->ShowInt(_PollState, "_PollState");
    _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
    _Parent->DbgPtr()->ShowInt(_ServerState, "_ServerState");
    _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess(4)");
    _Parent->DbgPtr()->ShowInt(retval, "retval");

    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    if (_ResetWhenNotReq && _ClientPollProcess == NOTREQUIRED)
    {
      SetStopTracking(true);
      ResetReceivePoll(_ForceReset);
      _ResetWhenNotReq = false;
    }

    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ClientReceiveNotRequired(bool InclAtInitial_) const
{
  return ((_ClientPollProcess == NOTREQUIRED ||
           (_ClientPollProcess == INITIALSTATE && InclAtInitial_)) && !_KeepAlivePollError);
}

/****************************************************************************/
bool ServerPollingMgr::ClientReceiveInProcess() const
{
  return (_ClientPollProcess == INPROCESS && !_KeepAlivePollError);
}

/****************************************************************************/
bool ServerPollingMgr::ClientReceiveCompleted_IMPL(bool InclNotReq_, bool InclPrev_) const
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ClientReceiveCompleted_IMPL");
  #endif

  bool ResetCond_ = false;
  bool CurrCond_ = false;
  bool PrevCond_ = false;

  if (_ResetStatesOnDone && InclPrev_)
  {
    ResetCond_ = _ClientPollProcess == INITIALSTATE;
    PrevCond_ = ((_PrevClientProcess == COMPLETED ||
                  (_PrevClientProcess == NOTREQUIRED && InclNotReq_)) && !_KeepAlivePollError);
  }

  CurrCond_ = ((_ClientPollProcess == COMPLETED ||
                (_ClientPollProcess == NOTREQUIRED && InclNotReq_)) && !_KeepAlivePollError);

  #if CALCLIB_DEBUG10
    bool retval = (CurrCond_ || (_ResetStatesOnDone && ResetCond_ && PrevCond_));
    _Parent->DbgPtr()->ShowInt(_ResetStatesOnDone, "_ResetStatesOnDone");
    _Parent->DbgPtr()->ShowInt(InclPrev_, "InclPrev_");
    _Parent->DbgPtr()->ShowInt(InclNotReq_, "InclNotReq_");
    _Parent->DbgPtr()->ShowInt(ResetCond_, "ResetCond_");
    _Parent->DbgPtr()->ShowInt(PrevCond_, "PrevCond_");
    _Parent->DbgPtr()->ShowInt(CurrCond_, "CurrCond_");
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return (CurrCond_ || (_ResetStatesOnDone && ResetCond_ && PrevCond_));
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ClientReceiveCompleted(bool InclNotReq_) const
{
  return ClientReceiveCompleted_IMPL(InclNotReq_, true);
}

/****************************************************************************/
void ServerPollingMgr::SetTrackingOnStateChange(bool Reset_)
{
  bool PollTestRetVal_ = false;

  if (_Backup)
  {
    bool ChangeFound_ = false;

    if (_Backup->_ClientPollProcess != _ClientPollProcess)
    {
      if (Reset_ && _ClientPollProcess == NOTREQUIRED)
      {
        ResetWhenNotRequired(true);
        SetStopTracking(true);
        
        #if CALCLIB_DEBUG9
          SHOWMETHOD(_Parent, ResetReceivePoll(_ForceReset));
        #else
          ResetReceivePoll(_ForceReset);
        #endif
      }
      else if (!_ResetWhenNotReq || _ClientPollProcess != NOTREQUIRED)
        SetStopTracking(false);
      
      _Backup->_ClientPollProcess = _ClientPollProcess;
      ChangeFound_ = true;
    }
    else if (!ChangeFound_ &&
             _Backup->_ClientPollProcess == _ClientPollProcess)
    {
      PollTestRetVal_ = (_ClientPollProcess == REQUIRED ||
                         (_ClientPollProcess == INPROCESS && !_KeepAlivePollError));

      if (!PollTestRetVal_)
        SetStopTracking(true);

      if (Reset_ && _ClientPollProcess == NOTREQUIRED)
      {
        ResetWhenNotRequired(true);
        
        #if CALCLIB_DEBUG9
          SHOWMETHOD(_Parent, ResetReceivePoll(_ForceReset));
        #else
          ResetReceivePoll(_ForceReset);
        #endif
      }
    }

    if (_Backup->_ServerPollProcess != _ServerPollProcess)
    {
      if (Reset_ && _ServerPollProcess == NOTREQUIRED)
      {
        ResetWhenNotRequired(true);
        SetStopTracking(true);
        
        #if CALCLIB_DEBUG9
          SHOWMETHOD(_Parent, ResetServerPolling(_ForceReset));
        #else
          ResetServerPolling(_ForceReset);
        #endif
      }
      else if (!_ResetWhenNotReq || _ClientPollProcess != NOTREQUIRED)
        SetStopTracking(false);

      _Backup->_ServerPollProcess = _ServerPollProcess;
      ChangeFound_ = true;
    }
    else if (!ChangeFound_ &&
             _Backup->_ServerPollProcess == _ServerPollProcess)
    {
      PollTestRetVal_ = (_ServerPollProcess == REQUIRED ||
                         (_ServerPollProcess == INPROCESS && !_KeepAlivePollError));

      if (!PollTestRetVal_)
        SetStopTracking(true);

      if (Reset_ && _ServerPollProcess == NOTREQUIRED)
      {
        ResetWhenNotRequired(true);
        
        #if CALCLIB_DEBUG9
          SHOWMETHOD(_Parent, ResetServerPolling(_ForceReset));
        #else
          ResetServerPolling(_ForceReset);
        #endif
      }
    }
  }
}

/****************************************************************************/
bool ServerPollingMgr::ResetServerPolling(bool ForceReset_)
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ResetServerPolling");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool NotInServerState_ = !_Parent->IsServerStates(IoState_) &&
                           !IsServerAliveStates(_ServerState);
  bool PollComplete_ = NotInServerState_ && PollServerCompleted_IMPL(true, false);
  ForceReset_ = NotInServerState_ && ForceReset_;

  if ((ForceReset_ || PollComplete_) &&
      _Parent->GetClientAliveFile(WRITE))
  {
    #if CALCLIB_DEBUG10
      SHOWMETHOD(_Parent, ResetPollCycle());
    #else
      ResetPollCycle();
    #endif
    _KeepAlivePollError = false;
    
    _ClientState = Mcalc_KeepAliveState::IDLE;
    _PrevServerProcess = _ServerPollProcess;
    _ServerPollProcess = INITIALSTATE;

    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_KeepAliveState::IDLE;
      #if CALCLIB_DEBUG10
        bool retval = SetPollState(_PollState);
        _Parent->DbgPtr()->ShowInt(_ServerState, "_ServerState");
        _Parent->DbgPtr()->ShowInt(_PrevClientProcess, "_PrevClientProcess");
        _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess");
        _Parent->DbgPtr()->ShowInt(retval, "retval");
        
        _Parent->DbgPtr()->LeaveLevel();
        return retval;
      #else
        return SetPollState(_PollState);
      #endif
    }
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(false, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return false;
}

/****************************************************************************/
bool ServerPollingMgr::ResetReceivePoll(bool ForceReset_)
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ResetReceivePoll");
  #endif

  bool ReceiveComplete_ = ClientReceiveCompleted_IMPL(true, false);

  if (ForceReset_ || ReceiveComplete_)
  {
    _ServerState = Mcalc_KeepAliveState::IDLE;
    _PrevClientProcess = _ClientPollProcess;
    _ClientPollProcess = INITIALSTATE;

    #if CALCLIB_DEBUG10
      _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
      _Parent->DbgPtr()->ShowInt(_PrevServerProcess, "_PrevServerProcess");
      _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess");
      _Parent->DbgPtr()->ShowInt(true, "retval");
      
      _Parent->DbgPtr()->LeaveLevel();
    #endif

    return true;
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(false, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return false;
}

/****************************************************************************/
bool ServerPollingMgr::ForceResetAllPolling()
{
  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ForceResetAllPolling");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool ForceReset_ = !_Parent->IsServerStates(IoState_) &&
                     !IsServerAliveStates(_ServerState);

  if (ForceReset_ && _Parent->GetClientAliveFile(WRITE))
  {
    ResetPollCycle();
    _KeepAlivePollError = false;
    
    _ClientState = Mcalc_KeepAliveState::IDLE;
    _PrevServerProcess = _ServerPollProcess;
    _ServerPollProcess = INITIALSTATE;
    
    _ServerState = Mcalc_KeepAliveState::IDLE;
    _PrevClientProcess = _ClientPollProcess;
    _ClientPollProcess = INITIALSTATE;

    #if CALCLIB_DEBUG10
      _Parent->DbgPtr()->ShowInt(_ServerState, "_ServerState");
      _Parent->DbgPtr()->ShowInt(_PrevClientProcess, "_PrevClientProcess");
      _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess");

      _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
      _Parent->DbgPtr()->ShowInt(_PrevServerProcess, "_PrevServerProcess");
      _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess");
    #endif

    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_KeepAliveState::IDLE;
      #if CALCLIB_DEBUG10
        bool retval = SetPollState(_PollState);
        _Parent->DbgPtr()->ShowInt(retval, "retval");
        
        _Parent->DbgPtr()->LeaveLevel();
        return retval;
      #else
        return SetPollState(_PollState);
      #endif
    }
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(false, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return false;
}

/****************************************************************************/
bool ServerPollingMgr::PollServerIfAlive_IMPL(int Sleep_, bool ResponseOnly_)
{
  _AsyncMode = _Parent ? _Parent->InAsyncMode():false;

  #if CALCLIB_DEBUG10
    SetStopTracking(false);  
    _Parent->DbgPtr()->EnterLevel("PollServerIfAlive_IMPL");
    _Parent->DbgPtr()->ShowInt(_AsyncMode, "_AsyncMode");
    _Parent->DbgPtr()->ShowMessage(_Parent->GetPollStateFilename());
    _Parent->DbgPtr()->ShowMessage("\n");
  #endif

  if (!_AsyncMode || !_Parent->GetPollStateFilename())
  {
    #if CALCLIB_DEBUG10
      _Parent->DbgPtr()->LeaveLevel();
    #endif

    return false;
  }

  // poll to check if calculator server is alive here
  if (PollServerRequired(Sleep_, ResponseOnly_))
  {
    bool RetryPoll_ = false;
    bool ServerResponse_ = _PollState == Mcalc_PollState::SERVER_RESPONSE;
    #if CALCLIB_DEBUG10
      _Parent->DbgPtr()->ShowInt(ServerResponse_, "ServerResponse_");
    #endif
    
    if (AtInitialState())
    {
      SetNewPollCycle();
      return SendPollServerSignal();
    }
    else if (ServerResponse_ && !ClientAcked())
    {
      if (ServerIsAlive())
      {
        #if CALCLIB_DEBUG10
          if (_Parent->DbgPtr()->StopTracking() &&
              _Backup && _Backup->_RetryPollCount+1 == _RetryPollCount)
        #else
          if (_Backup && _Backup->_RetryPollCount+1 == _RetryPollCount)
        #endif
        {
          SetStopTracking(false);

          #if CALCLIB_DEBUG10
            _Parent->DbgPtr()->EnterLevel("PollServerIfAlive_IMPL");
            _Parent->DbgPtr()->ShowInt(ServerResponse_, "ServerResponse_");
            ServerIsAlive();
          #endif
        }
      
        ConfirmServerAlive();
        ConfirmClientAlive(true);
        return SendClientAckServerSignal();
      }
      else
      {
        _KeepAlivePollError = true;
        RetryPoll_ = ReAttemptPolling();
        #if CALCLIB_DEBUG10
          _Parent->DbgPtr()->LeaveLevel();
        #endif

        if (_Backup)
          SetStopTracking(true);

        if (!RetryPoll_ && _ExitAtRetryMax)
          _Parent->ShowServerPollFailed(true);
        else
          return RetryPoll_;
      }
    }
    else if (ResetRequested() && ClientAcked())
    {
      #if CALCLIB_DEBUG10
        if (_Parent->DbgPtr()->StopTracking())
      #endif
      {
        if (_Backup)
          SetStopTracking(false);

        #if CALCLIB_DEBUG10
          _Parent->DbgPtr()->EnterLevel("PollServerIfAlive_IMPL");
          _Parent->DbgPtr()->ShowInt(ServerResponse_, "ServerResponse_");
          ResetRequested();
          ClientAcked();
        #endif
      }

      return ResetPolling();
    }
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  if (_Backup)
    SetStopTracking(true);

  return false;
}

/****************************************************************************/
bool ServerPollingMgr::ReceiveKeepAlivePoll_IMPL(int Sleep_)
{
  _AsyncMode = _Parent ? _Parent->InAsyncMode():false;

  #if CALCLIB_DEBUG10
    SetStopTracking(false);
    _Parent->DbgPtr()->EnterLevel("ReceiveKeepAlivePoll_IMPL");
    _Parent->DbgPtr()->ShowInt(_AsyncMode, "_AsyncMode");
    _Parent->DbgPtr()->ShowMessage(_Parent->GetPollStateFilename());
    _Parent->DbgPtr()->ShowMessage("\n");
  #endif

  if (!_AsyncMode || !_Parent->GetPollStateFilename())
  {
    #if CALCLIB_DEBUG10
      _Parent->DbgPtr()->LeaveLevel();
    #endif

    return false;
  }

  // receive check alive poll from calculator server here
  if (ClientReceiveRequired(Sleep_))
  {
    bool ServerPolling_ = _ServerState == Mcalc_KeepAliveState::POLL_CLIENTALIVE;
    #if CALCLIB_DEBUG10
      _Parent->DbgPtr()->ShowInt(ServerPolling_, "ServerPolling_");
    #endif
    
    if (ServerAcked())
    {
      #if CALCLIB_DEBUG10
        if (_Parent->DbgPtr()->StopTracking())
      #endif
      {
        if (_Backup)
          SetStopTracking(false);

        #if CALCLIB_DEBUG10
          _Parent->DbgPtr()->EnterLevel("ReceiveKeepAlivePoll_IMPL");
          _Parent->DbgPtr()->ShowInt(ServerPolling_, "ServerPolling_");
          ServerAcked();
        #endif
      }

      ConfirmServerAlive(true);
      ConfirmClientAlive();
      return SendResetPollingSignal();
    }
    else if (ServerPolling_)
    {
      if (!ClientIsAlive())
      {
        AcceptPollCycle();
        return SendClientAliveSignal();
      }
      else
      {
        if (_Backup)
          SetStopTracking(true);

        #if CALCLIB_DEBUG10
          _Parent->DbgPtr()->LeaveLevel();
        #endif
        return true;
      }
    }
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  if (_Backup)
    SetStopTracking(true);
  
  return false;
}

/****************************************************************************/
bool ServerPollingMgr::PollServerIfAlive(int Sleep_,
                                         bool UntilEnd_, bool ResponseOnly_)
{
  if (_Backup)
  {
    SetStopTracking(!UntilEnd_ || ResponseOnly_);
    *_Backup = *this;
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("PollServerIfAlive");
  #endif

  bool PollOk_ = true;
  long LoopGuard_ = 0;
  long Max = RETRY_READ_MAX * 2;
  
  int mult_ = 0;
  int DelayLen_ = IOSTATE_DELAY_LENGTH * DELAY_MULT;
  int PrevAttempt_;

  bool PollingServer_ = false;
  bool PollAnswered_ = false;
  bool PollError_ = false;
  bool PollRetVal_ = false;
  bool InProcess_ = PollServerInProcess();
  bool ResetOnNotReq_ = _ResetWhenNotReq;

  ResetServerPolling(UntilEnd_ || !InProcess_);
  
  for (LoopGuard_ = UntilEnd_ ? Max:1;
       LoopGuard_ && PollOk_ && PollServerRequired(0, ResponseOnly_);
       LoopGuard_--)
  {
    PrevAttempt_ = _RetryPollCount;
    PollOk_ = PollServerIfAlive_IMPL(Sleep_, ResponseOnly_);
    #if CALCTEST_TESTMOCKSERVER
      if (!PollServerCompleted(true))
        _Parent->NotifyParent(PollOk_, true);
    #endif
    GivePollResults(PollingServer_, PollError_, PollAnswered_);
    ResponseOnly_ = true;
    
    if (PollOk_ && PollError_)
    {
      if (_BreakOnPollError)
      {
        SetStopTracking(true);
        break;
      }
      else if (Sleep_)
      {
        mult_ = (Sleep_ > 1) ? Sleep_:1;

        #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
          Sleep(mult_ * DelayLen_);
        #else
          ::usleep(mult_ * DelayLen_);
        #endif
      }
      else if (_RetryPollCount != PrevAttempt_ && _KeepAlivePollError)
      {
        Sleep_ = 1;
        SetStopTracking(true);
        
        if (_Backup)
        {
          *_Backup = *this;
          _Backup->_RetryPollCount = PrevAttempt_;
        }
      }
    }
    else
    {
      SetTrackingOnStateChange(ResetOnNotReq_);

      if (PollServerCompleted(false))
        break;
    }
  }

  PollRetVal_ = PollOk_ && PollServerCompleted(true);

  if (_ResetStatesOnDone || _KeepAlivePollError)
  {
    if (PollServerNotRequired(false))
    {
      SetStopTracking(true);
      ResetServerPolling(UntilEnd_);
    }
    else if (PollRetVal_)
    {
      SetStopTracking(true);
      ResetServerPolling(false);
    }
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(PollRetVal_, "PollRetVal_");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return PollRetVal_;
}

/****************************************************************************/
bool ServerPollingMgr::ReceiveKeepAlivePoll(int Sleep_, bool UntilEnd_)
{
  if (_Backup)
  {
    SetStopTracking(!UntilEnd_);
    *_Backup = *this;
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->EnterLevel("ReceiveKeepAlivePoll");
  #endif

  bool PollOk_ = true;
  bool PollRetVal_ = false;
  bool InProcess_ = ClientReceiveInProcess();
  bool ResetOnNotReq_ = _ResetWhenNotReq;
  
  long LoopGuard_ = 0;
  long Max = RETRY_READ_MAX * 2;

  ResetReceivePoll(UntilEnd_ || !InProcess_);

  for (LoopGuard_ = UntilEnd_ ? Max:1;
       LoopGuard_ && PollOk_ && ClientReceiveRequired();
       LoopGuard_--)
  {
    PollOk_ = ReceiveKeepAlivePoll_IMPL(Sleep_);
    #if CALCLIB_TESTMOCKCLIENT
      if (!ClientReceiveCompleted(true))
        _Parent->NotifyParent(PollOk_, false);
    #endif
    SetTrackingOnStateChange(ResetOnNotReq_);

    if (ClientReceiveCompleted(false))
      break;
  }

  PollRetVal_ = PollOk_ && ClientReceiveCompleted(true);

  if (_ResetStatesOnDone || _KeepAlivePollError)
  {
    if (ClientReceiveNotRequired(false))
    {
      SetStopTracking(true);
      ResetReceivePoll(UntilEnd_);
    }
    else if (PollRetVal_)
    {
      SetStopTracking(true);
      ResetReceivePoll(false);
    }
  }

  #if CALCLIB_DEBUG10
    _Parent->DbgPtr()->ShowInt(PollRetVal_, "PollRetVal_");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return PollRetVal_;
}

/****************************************************************************/
void ServerPollingMgr::ResetRetryRead()
{
  _RetryReadCount = 0;
}

/****************************************************************************/
bool ServerPollingMgr::AtMaxReadAttempts() const
{
  return (_RetryReadCount >= RETRY_READ_MAX);
}

/****************************************************************************/
bool ServerPollingMgr::ReAttemptFileRead()
{
  if (_RetryReadCount < RETRY_READ_MAX)
    _RetryReadCount++;
  else if (_RetryReadCount == RETRY_READ_MAX)
    return false;

  return true;
}

/****************************************************************************/
void ServerPollingMgr::ResetPollCount()
{
  _RetryPollCount = 0;
}

/****************************************************************************/
bool ServerPollingMgr::AtMaxPollAttempts() const
{
  return (_RetryPollCount >= RETRY_READ_MAX);
}

/****************************************************************************/
bool ServerPollingMgr::ReAttemptPolling()
{
  if (_KeepAlivePollError && _RetryPollCount < RETRY_READ_MAX)
  {
    _RetryPollCount++;
    return (_RetryPollCount < RETRY_READ_MAX);
  }
  else if (_RetryPollCount >= RETRY_READ_MAX)
    return false;

  return true;
}

/****************************************************************************/
bool ServerPollingMgr::ServerPollingAllowed() const
{
  return _AllowPollServer;
}

/****************************************************************************/
bool ServerPollingMgr::ConfirmedServerAlive() const
{
  return _ConfirmedServerAlive;
}

/****************************************************************************/
bool ServerPollingMgr::ConfirmedClientAlive() const
{
  return _ConfirmedClientAlive;
}

/****************************************************************************/
bool ServerPollingMgr::ServerAliveConfirmedSent() const
{
  return _ServerAliveConfSent;
}

/****************************************************************************/
bool ServerPollingMgr::ExitAtRetryMax() const
{
  return _ExitAtRetryMax;
}

/****************************************************************************/
bool ServerPollingMgr::SleepWaiting() const
{
  return _SleepWaiting;
}

/****************************************************************************/
bool ServerPollingMgr::WaitUntilResponse() const
{
  return _WaitUntilResponse;
}

/****************************************************************************/
MEMORYOPS_DEFN(ServerPollingMgr)
