#include <qpainter.h>
#include <qstyle.h>
#include <qtoolbutton.h>

#include <cmath>
using namespace std;

#include "plotter.h"

/****************************************************************************/
Plotter::Plotter(QWidget *parent, const char *name, WFlags flags)
    : QWidget(parent, name, flags | WNoAutoErase)
{
    EPSILON = .00001;
    _InfPtRad = 0;
    _Xoffset = _Yoffset = 0;
    _Printer = NULL;
    _SetToPrint = false;
    _UserTextEdit = new QTextEdit(0);
    _UserTextEdit->hide();

    curveMap.clear();
    setBackgroundMode(PaletteDark);
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    setFocusPolicy(StrongFocus);
    rubberBandIsShown = false;
    _UseFileBuffers = true;
    _Tracing = false;
    _TraceIndex = 0;
    _CurveNum = 0;
    _Radius = 0;

    _ActiveCurve = 0;
    _NumCurves = 0;
    _Session = 0;

    QVBox* vbox = new QVBox(this);
    vbox->setMinimumWidth(MIN_WIDTH);
    vbox->setBackgroundMode(PaletteDark);
    _Xplot = new QLabel(vbox);
    _Yplot = new QLabel(vbox);
    _Xplot->setPaletteForegroundColor(Qt::white);
    _Yplot->setPaletteForegroundColor(Qt::white);
    _Xplot->setBackgroundMode(PaletteDark);
    _Yplot->setBackgroundMode(PaletteDark);
    _Xplot->hide();
    _Yplot->hide();

    zoomInButton = new QToolButton(this);
    zoomInButton->setIconSet(QPixmap::fromMimeSource("zoomin.png"));
    zoomInButton->adjustSize();
    connect(zoomInButton, SIGNAL(clicked()), this, SLOT(zoomIn()));

    zoomOutButton = new QToolButton(this);
    zoomOutButton->setIconSet(
            QPixmap::fromMimeSource("zoomout.png"));
    zoomOutButton->adjustSize();
    connect(zoomOutButton, SIGNAL(clicked()), this, SLOT(zoomOut()));

    setPlotSettings(PlotSettings());
}

/****************************************************************************/
void Plotter::Destruct()
{
    EraseDataFiles();
    clearCurves();

    delete _UserTextEdit;
    close();
}

/****************************************************************************/
bool Plotter::UseFileBuffers() const
{
    return _UseFileBuffers;
}

/****************************************************************************/
void Plotter::SetUseFileBuffers(bool Flag_)
{
    _UseFileBuffers = Flag_;

    if (!Flag_)
    {
      int x;
      int len = curveMap.size();
      for (x = 0; x < len; x++)
        ReadCurve(x);
    }
}

/****************************************************************************/
void Plotter::SetThePrinter(QPrinter* Printer_)
{
    _Printer = Printer_;
    if (_Printer)
      _Printer->setFullPage(TRUE);
}

/****************************************************************************/
void Plotter::SetToPaintOnPrinter()
{
    if (_Printer)
    {
      showMaximized();
      _SetToPrint = false;
      refreshPixmap();
      showMaximized();
      _SetToPrint = true;

      if (_Printer->setup(this))
        refreshPixmap();
    }
}

/****************************************************************************/
void Plotter::SetSession(int Session_)
{
    _Session = Session_;
}

/****************************************************************************/
void Plotter::SetInfPointRadius(double Rad_)
{
    _InfPtRad = Rad_;
}

/****************************************************************************/
void Plotter::SetErrorMargin(double Eps_)
{
    EPSILON = Eps_;
}

/****************************************************************************/
void Plotter::setPlotSettings(const PlotSettings &settings)
{
    zoomStack.resize(1);
    zoomStack[0] = settings;
    curZoom = 0;
    zoomInButton->hide();
    zoomOutButton->hide();
    refreshPixmap();
}

/****************************************************************************/
void Plotter::zoomOut()
{
    if (curZoom > 0) {
        --curZoom;
        zoomOutButton->setEnabled(curZoom > 0);
        zoomInButton->setEnabled(true);
        zoomInButton->show();
        refreshPixmap();
    }
}

/****************************************************************************/
void Plotter::zoomIn()
{
    if (curZoom < (int)zoomStack.size() - 1) {
        ++curZoom;
        zoomInButton->setEnabled(
                curZoom < (int)zoomStack.size() - 1);
        zoomOutButton->setEnabled(true);
        zoomOutButton->show();
        refreshPixmap();
    }
}

/****************************************************************************/
void Plotter::EraseDataFiles()
{
  QString FileName_;
  QString NameStub_ = QString("mcalc_plot") +
                      QString::number(_Session) +
                      QString("-");

  size_t x;
  for (x = 0; x < _NumCurves; x++)
  {
    FileName_ = NameStub_ + QString::number(x) + QString(".dat");
    if (QFile::exists(FileName_))
      QFile::remove(FileName_);
  }

  _NumCurves = 0;
}

/****************************************************************************/
void Plotter::ReadCurve(size_t id)
{
  if (!curveMap.size())
    return;

  if (!curveMap[id]._CurveData)
    curveMap[id]._CurveData = new CurveData;

  if (!curveMap[id]._InfinityPts)
    curveMap[id]._InfinityPts = new CurveData;

  QString FileName_ = QString("mcalc_plot") +
                      QString::number(_Session) +
                      QString("-") +
                      QString::number(id) +
                      QString(".dat");

  QString line;
  size_t len;
  size_t x;

  QFile Fin_(FileName_);
  _ActiveCurve = id;

  if (Fin_.open(IO_ReadOnly))
  {
    QTextStream instrm(&Fin_);

    line = instrm.readLine();
    len = line.toUInt();
    curveMap[id]._CurveData->resize(len);
    if (len)
      for (x = 0; x < len; x++)
      {
        line = instrm.readLine();
        (*curveMap[id]._CurveData)[x] = line.toDouble();
      }

    line = instrm.readLine();
    len = line.toUInt();
    curveMap[id]._InfinityPts->resize(len);
    if (len)
      for (x = 0; x < len; x++)
      {
        line = instrm.readLine();
        (*curveMap[id]._InfinityPts)[x] = line.toDouble();
      }

    Fin_.close();
  }
}

/****************************************************************************/
void Plotter::SaveCurve(size_t id)
{
  if (!curveMap.size() || !curveMap[id]._CurveData || !_UseFileBuffers)
    return;

  QString FileName_ = QString("mcalc_plot") +
                      QString::number(_Session) +
                      QString("-") +
                      QString::number(id) +
                      QString(".dat");

  size_t len;
  size_t x;

  QFile Fout_(FileName_);

  if (Fout_.open(IO_WriteOnly))
  {
    QTextStream outstrm(&Fout_);

    len = curveMap[id]._CurveData->size();
    outstrm <<len <<"\n";
    if (len)
    {
      for (x = 0; x < len; x++)
        outstrm <<(*curveMap[id]._CurveData)[x] <<"\n";

      curveMap[id]._CurveData->clear();
      delete curveMap[id]._CurveData;
      curveMap[id]._CurveData = NULL;
    }

    len = curveMap[id]._InfinityPts->size();
    outstrm <<len <<"\n";
    if (len)
    {
      for (x = 0; x < len; x++)
        outstrm <<(*curveMap[id]._InfinityPts)[x] <<"\n";

      curveMap[id]._InfinityPts->clear();
      delete curveMap[id]._InfinityPts;
      curveMap[id]._InfinityPts = NULL;
    }

    Fout_.close();
  }
}

/****************************************************************************/
void Plotter::initCurveData(size_t id, const QString& Function_)
{
   _UseFileBuffers = true;
   if (id != _ActiveCurve && curveMap.size())
     SaveCurve(_ActiveCurve);

   _ActiveCurve = id;
   curveMap[id]._Function = Function_;
   curveMap[id]._HasSecant = false;
   curveMap[id]._NumSecants = 0;
   curveMap[id]._CurveData = NULL;
   curveMap[id]._InfinityPts = NULL;

   int x;
   for (x = 0; x < 8; x++)
     curveMap[id]._SecantData[x] =
     curveMap[id]._SecantData2[x] =
     curveMap[id]._SecantData3[x] = 0;

   if (_NumCurves < curveMap.size())
     _NumCurves = curveMap.size();

   if (Function_.length())
   {
     if (curveMap.size() == 1)
       setCaption(curveMap[_ActiveCurve]._Function);
     else if (!_Tracing)
       setCaption("Plotter");
   }
}

/****************************************************************************/
void Plotter::setCurveData(size_t id, CurveData *data, CurveData *infdata)
{
    delete curveMap[id]._CurveData;
    delete curveMap[id]._InfinityPts;
    curveMap[id]._CurveData = data;
    curveMap[id]._InfinityPts = infdata;
    refreshPixmap();
}

/****************************************************************************/
void Plotter::popSecantData(size_t id)
{
  int x;
  for (x = 0; x < 8; x++)
  {
    curveMap[id]._SecantData[x] = curveMap[id]._SecantData2[x];
    curveMap[id]._SecantData2[x] = curveMap[id]._SecantData3[x];
    curveMap[id]._SecantData3[x] = 0;
  }
}

/****************************************************************************/
void Plotter::pushSecantData(size_t id)
{
  int x;

  if (curveMap[id]._NumSecants == 2)
  {
    for (x = 0; x < 8; x++)
    {
      curveMap[id]._SecantData3[x] = curveMap[id]._SecantData2[x];
      curveMap[id]._SecantData2[x] = curveMap[id]._SecantData[x];
    }
  }
  else if (curveMap[id]._NumSecants == 1)
  {
    for (x = 0; x < 8; x++)
      curveMap[id]._SecantData2[x] = curveMap[id]._SecantData[x];
  }
}

/****************************************************************************/
void Plotter::setSecantData(size_t id, bool HasSecant_,
                            double Pt1x, double Pt1y,
                            double Pt2x, double Pt2y,
                            double Pt3x, double Pt3y,
                            double Pt4x, double Pt4y)
{
  curveMap[id]._HasSecant = HasSecant_;

  if (HasSecant_)
  {
    curveMap[id]._NumSecants++;
    curveMap[id]._SecantData[0] = Pt1x;
    curveMap[id]._SecantData[1] = Pt1y;
    curveMap[id]._SecantData[2] = Pt2x;
    curveMap[id]._SecantData[3] = Pt2y;
    curveMap[id]._SecantData[4] = Pt3x;
    curveMap[id]._SecantData[5] = Pt3y;
    curveMap[id]._SecantData[6] = Pt4x;
    curveMap[id]._SecantData[7] = Pt4y;
  }
}

/****************************************************************************/
void Plotter::clearCurves()
{
    int x;
    int len = curveMap.size();
    for (x = 0; x < len; x++)
    {
      delete curveMap[x]._CurveData;
      delete curveMap[x]._InfinityPts;
      curveMap[x]._CurveData = NULL;
      curveMap[x]._InfinityPts = NULL;
    }

    curveMap.clear();
    _ActiveCurve = 0;
    ClearTracePt();
}

/****************************************************************************/
void Plotter::clearCurve(size_t id)
{
    curveMap.erase(id);

    if (curveMap.size() == 1)
      setCaption(curveMap[0]._Function);
    else if (curveMap.size() == 0)
      setCaption("Plotter");

    if (_ActiveCurve == id)
      _ActiveCurve = 0;

    refreshPixmap();
}

/****************************************************************************/
const CurveData& Plotter::GetCurveData(size_t id)
{
    static CurveData Dummy_;

    if (!curveMap.size())
      return Dummy_;

    if (!_Tracing && id != _ActiveCurve && _UseFileBuffers)
    {
      SaveCurve(_ActiveCurve);
      ReadCurve(id);
    }

    if (!curveMap[id]._CurveData)
      return Dummy_;

    return (*curveMap[id]._CurveData);
}

/****************************************************************************/
size_t Plotter::NumCurves() const
{
  return curveMap.size();
}

/****************************************************************************/
QSize Plotter::minimumSizeHint() const
{
    return QSize(4 * Margin, 4 * Margin);
}

/****************************************************************************/
QSize Plotter::sizeHint() const
{
    return QSize(6 * Margin, 6 * Margin);
}

/****************************************************************************/
const PlotSettings& Plotter::CurrentSettings() const
{
    return zoomStack[curZoom];
}

/****************************************************************************/
void Plotter::paintEvent(QPaintEvent *event)
{
    QMemArray<QRect> rects = event->region().rects();
    for (int i = 0; i < (int)rects.size(); ++i)
        bitBlt(this, rects[i].topLeft(), &pixmap, rects[i]);

    QPainter painter(this);

    if (rubberBandIsShown) {
        painter.setPen(colorGroup().light());
        painter.drawRect(rubberBandRect.normalize());
    }
    if (hasFocus()) {
        style().drawPrimitive(QStyle::PE_FocusRect, &painter,
                              rect(), colorGroup(),
                              QStyle::Style_FocusAtBorder,
                              colorGroup().dark());
    }
}

/****************************************************************************/
void Plotter::resizeEvent(QResizeEvent *)
{
    int x = width() - (zoomInButton->width()
                       + zoomOutButton->width() + 10);
    zoomInButton->move(x, 5);
    zoomOutButton->move(x + zoomInButton->width() + 5, 5);
    refreshPixmap();
}

/****************************************************************************/
void Plotter::mousePressEvent(QMouseEvent *event)
{
    if (event->button() == LeftButton)
    {
        rubberBandIsShown = true;
        rubberBandRect.setTopLeft(event->pos());
        rubberBandRect.setBottomRight(event->pos());
        updateRubberBandRegion();
        setCursor(crossCursor);
    }
}

/****************************************************************************/
void Plotter::mouseMoveEvent(QMouseEvent *event)
{
    if (event->state() & LeftButton)
    {
        updateRubberBandRegion();
        rubberBandRect.setBottomRight(event->pos());
        updateRubberBandRegion();
    }
}

/****************************************************************************/
void Plotter::mouseReleaseEvent(QMouseEvent *event)
{
    if (event->button() == LeftButton)
    {
        rubberBandIsShown = false;
        updateRubberBandRegion();
        unsetCursor();

        QRect rect = rubberBandRect.normalize();
        if (rect.width() < 4 || rect.height() < 4)
            return;
        rect.moveBy(-Margin, -Margin);

        PlotSettings prevSettings = zoomStack[curZoom];
        PlotSettings settings;
        double dx = prevSettings.spanX() / (width() - 2 * Margin);
        double dy = prevSettings.spanY() / (height() - 2 * Margin);
        settings.minX = prevSettings.minX + dx * rect.left();
        settings.maxX = prevSettings.minX + dx * rect.right();
        settings.minY = prevSettings.maxY - dy * rect.bottom();
        settings.maxY = prevSettings.maxY - dy * rect.top();
        settings.adjust();

        zoomStack.resize(curZoom + 1);
        zoomStack.push_back(settings);
        zoomIn();
    }
}

/****************************************************************************/
void Plotter::ScrollGraph(int Horz_, int Vert_)
{
  if (Horz_)
  {
    zoomStack[curZoom].scroll(Horz_ - _Xoffset, 0);
    _Xoffset = Horz_;
  }

  if (Vert_)
  {
    zoomStack[curZoom].scroll(0, Vert_ - _Yoffset);
    _Yoffset = Vert_;
  }

  refreshPixmap();
}

/****************************************************************************/
void Plotter::ClearTracePt()
{
  _Tracing = false;

  if (curveMap.size() != 1)
    setCaption("Plotter");

  _Xplot->clear();
  _Yplot->clear();
  _Xplot->hide();
  _Yplot->hide();

  if (curveMap.size())
    refreshPixmap();
}

/****************************************************************************/
void Plotter::ShowTracePt(size_t CurveNum_, size_t Index_,
                          double Ptx, double Pty, double Radius_)
{
  SetTracePt(CurveNum_, Index_, Ptx, Pty, Radius_);
  refreshPixmap();
}

/****************************************************************************/
void Plotter::keyPressEvent(QKeyEvent *event)
{
    double ptx;

    switch (event->key()) {
    case Key_Plus:
        zoomIn();
        break;

    case Key_Minus:
        zoomOut();
        break;

    case Key_PageUp:
        if (_Tracing && _CurveNum < curveMap.size() &&
            curveMap.size() && curveMap[_CurveNum]._CurveData)
        {
          if (_TraceIndex > 0)
          {
            ptx = ::floor((*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)]);
            if (_TraceIndex > 0 && ptx < (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)])
              _TraceIndex--;

            for (;_TraceIndex > 0 && ptx == (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)]; _TraceIndex--);
            ptx = ::floor((*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)]);
            for (;_TraceIndex > 0 && ptx < (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)]; _TraceIndex--);
            SetTracePt(_CurveNum, _TraceIndex,
                       (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)],
                       (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex + 1)],
                       _Radius);
            refreshPixmap();
          }
        }
        break;

    case Key_PageDown:
        if (_Tracing && _CurveNum < curveMap.size() &&
            curveMap.size() && curveMap[_CurveNum]._CurveData)
        {
          size_t Max_ = (curveMap[_CurveNum]._CurveData->size() - 1) / 2;

          if (_TraceIndex < Max_ - 1)
          {
            ptx = ::ceil((*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)]);
            if (_TraceIndex < Max_ - 1 && ptx > (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)])
              _TraceIndex++;

            for (;_TraceIndex < Max_ - 1 && ptx == (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)]; _TraceIndex++);
            ptx = ::ceil((*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)]);
            for (;_TraceIndex < Max_ - 1 && ptx > (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)]; _TraceIndex++);
            SetTracePt(_CurveNum, _TraceIndex,
                       (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)],
                       (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex + 1)],
                       _Radius);
            refreshPixmap();
          }
        }
        break;

    case Key_Left:
        if (_Tracing && _CurveNum < curveMap.size() &&
            curveMap.size() && curveMap[_CurveNum]._CurveData)
        {
          if (_TraceIndex > 0)
          {
            _TraceIndex--;
            SetTracePt(_CurveNum, _TraceIndex,
                       (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)],
                       (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex + 1)],
                       _Radius);
            refreshPixmap();
          }
        }
        else
        {
          _Xoffset -= 1;
          zoomStack[curZoom].scroll(-1, 0);
          refreshPixmap();
        }
        break;

    case Key_Right:
        if (_Tracing && _CurveNum < curveMap.size() &&
            curveMap.size() && curveMap[_CurveNum]._CurveData)
        {
          size_t Max_ = (curveMap[_CurveNum]._CurveData->size() - 1) / 2;

          if (_TraceIndex < Max_ - 1)
          {
            _TraceIndex++;
            SetTracePt(_CurveNum, _TraceIndex,
                       (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex)],
                       (*curveMap[_CurveNum]._CurveData)[1 + (2 * _TraceIndex + 1)],
                       _Radius);
            refreshPixmap();
          }
        }
        else
        {
          _Xoffset += 1;
          zoomStack[curZoom].scroll(+1, 0);
          refreshPixmap();
        }
        break;

    case Key_Down:
        _Yoffset -= 1;
        zoomStack[curZoom].scroll(0, -1);
        refreshPixmap();
        break;

    case Key_Up:
        _Yoffset += 1;
        zoomStack[curZoom].scroll(0, +1);
        refreshPixmap();
        break;

    default:
        QWidget::keyPressEvent(event);
    }
}

/****************************************************************************/
void Plotter::wheelEvent(QWheelEvent *event)
{
    int numDegrees = event->delta() / 8;
    int numTicks = numDegrees / 15;

    if (event->orientation() == Horizontal)
    {
        _Xoffset += numTicks;
        zoomStack[curZoom].scroll(numTicks, 0);
    }
    else
    {
        _Yoffset += numTicks;
        zoomStack[curZoom].scroll(0, numTicks);
    }
    refreshPixmap();
}

/****************************************************************************/
void Plotter::updateRubberBandRegion()
{
    QRect rect = rubberBandRect.normalize();
    update(rect.left(), rect.top(), rect.width(), 1);
    update(rect.left(), rect.top(), 1, rect.height());
    update(rect.left(), rect.bottom(), rect.width(), 1);
    update(rect.right(), rect.top(), 1, rect.height());
}

/****************************************************************************/
void Plotter::SetPrintMetrics(QPainter* p)
{
    _UserTextEdit->setText(caption() + QString("\n") +
                           _Xplot->text() + QString("\n") +
                           _Yplot->text() + QString("\n"));
    _UserTextEdit->hide();

    double Mult_ = .5;
    if (_Xplot->text().length() || _Yplot->text().length())
      Mult_ = 1.0;

    QPaintDeviceMetrics metrics( p->device() );
    int dpiy = metrics.logicalDpiY();
    int margin = (int) ( (2/2.54)*dpiy ); // 2 cm margins
    QRect body( margin, margin, metrics.width() - 2*margin, metrics.height() - 2*margin );

    // setup painter to draw on printer
    _Xtrans = body.left() + Margin;
    _Ytrans = body.top() + (Mult_ * margin);
    _PrintArea = body;
}

/****************************************************************************/
void Plotter::PrintPlotterElements(QPainter* p)
{
    QSimpleRichText richText( QStyleSheet::convertFromPlainText(_UserTextEdit->text()),
                              QFont(),
                              _UserTextEdit->context(),
                              _UserTextEdit->styleSheet(),
                              _UserTextEdit->mimeSourceFactory(),
                              _PrintArea.height() );

    _UserTextEdit->hide();
    richText.setWidth( p, _PrintArea.width() );
    QRect view( _PrintArea );
    richText.draw( p, _PrintArea.left(), _PrintArea.top(), view, colorGroup() );
}

/****************************************************************************/
void Plotter::refreshPixmap()
{
    if (_Tracing)
      AdjustForTracePt();

    QPainter* painter;
    if (_SetToPrint && _Printer)
    {
      painter = new QPainter;

      if (!painter->begin(_Printer))
      {
        delete painter;
        return;
      }

      SetPrintMetrics(painter);
      PrintPlotterElements(painter);

      painter->setFont(QFont("times", 6, QFont::Bold));
      painter->translate(_Xtrans, _Ytrans);
      painter->setClipping(false);
    }
    else
    {
      pixmap.resize(size());
      pixmap.fill(this, 0, 0);
      painter = new QPainter(&pixmap, this);
    }

    drawGrid(painter);

    if (curveMap.size())
    {
      painter->setClipping(true);
      drawCurves(painter);
    }

    update();
    painter->end();
    delete painter;

    if (_Tracing)
    {
      _Xplot->show();
      _Yplot->show();
    }

    if (_SetToPrint)
    {
      _SetToPrint = false;
      refreshPixmap();
    }
}

/****************************************************************************/
void Plotter::drawGrid(QPainter *painter)
{
    QRect rect(Margin, Margin,
               width() - 2 * Margin, height() - 2 * Margin);
    PlotSettings settings = zoomStack[curZoom];
    QPen quiteDark = colorGroup().dark().light();
    QPen light = colorGroup().light();

    for (int i = 0; i <= settings.numXTicks; ++i) {
        int x = rect.left() + (i * (rect.width() - 1)
                                 / settings.numXTicks);
        double label = settings.minX + (i * settings.spanX()
                                          / settings.numXTicks);
        if (fabs(label) < EPSILON)
          label = 0;

        if (_SetToPrint)
        {
          quiteDark.setWidth(3);
          light.setWidth(3);
        }

        painter->setPen(quiteDark);
        painter->drawLine(x, rect.top(), x, rect.bottom());
        painter->setPen(light);
        painter->drawLine(x, rect.bottom(), x, rect.bottom() + 5);

        if (i % 2 == 0)
          painter->drawText(x - 50, rect.bottom() + 5, 100, 15,
                            AlignHCenter | AlignTop,
                            QString::number(label));
    }
    for (int j = 0; j <= settings.numYTicks; ++j) {
        int y = rect.bottom() - (j * (rect.height() - 1)
                                   / settings.numYTicks);
        double label = settings.minY + (j * settings.spanY()
                                          / settings.numYTicks);
        if (fabs(label) < EPSILON)
          label = 0;

        if (_SetToPrint)
        {
          quiteDark.setWidth(3);
          light.setWidth(3);
        }

        painter->setPen(quiteDark);
        painter->drawLine(rect.left(), y, rect.right(), y);
        painter->setPen(light);
        painter->drawLine(rect.left() - 5, y, rect.left(), y);

        if (j % 2 == 0)
          painter->drawText(rect.left() - Margin, y - 10,
                            Margin - 5, 20,
                            AlignRight | AlignVCenter,
                            QString::number(label));
    }
    painter->drawRect(rect);
}

/****************************************************************************/
void Plotter::FillSubinterval(QPainter* painter, const QPoint& Pt1, const QPoint& Pt2, int Zero_)
{
  painter->setPen(Qt::green);
  painter->setBrush(Qt::SolidPattern);
  painter->setBrush(Qt::green);

  if ((Pt1.y() == 0 && Pt2.y() < 0) ||
      (Pt1.y() == 0 && Pt2.y() > 0) ||
      (Pt1.y() < 0 && Pt2.y() > 0))
  {
     QPointArray intpoints(3);
     intpoints[0] = Pt1;
     intpoints[1] = QPoint(Pt2.x(), Pt1.y());
     intpoints[2] = Pt2;
     painter->drawPolygon(intpoints);
  }
  else if ((Pt1.y() < 0 && Pt2.y() == 0) ||
           (Pt1.y() > 0 && Pt2.y() == 0) ||
           (Pt1.y() > 0 && Pt2.y() < 0))
  {
     QPointArray intpoints(3);
     intpoints[0] = Pt1;
     intpoints[1] = QPoint(Pt1.x(), Pt2.y());
     intpoints[2] = Pt2;
     painter->drawPolygon(intpoints);
  }
  else if (Pt1.y() && Pt2.y())
  {
     QPointArray intpoints(4);
     intpoints[0] = QPoint(Pt1.x(), Zero_);
     intpoints[1] = Pt1;
     intpoints[2] = Pt2;
     intpoints[3] = QPoint(Pt2.x(), Zero_);
     painter->drawPolygon(intpoints);
  }
}

/****************************************************************************/
void Plotter::AdjustForTracePt()
{
  double x = _TracePtx;
  double y = _TracePty;

  while (x <= zoomStack[curZoom].minX)
  {
    _Xoffset -= 1;
    zoomStack[curZoom].scroll(-1, 0);
  }

  while (x >= zoomStack[curZoom].maxX)
  {
    _Xoffset += 1;
    zoomStack[curZoom].scroll(+1, 0);
  }

  while (y <= zoomStack[curZoom].minY)
  {
    _Yoffset -= 1;
    zoomStack[curZoom].scroll(0, -1);
  }

  while (y >= zoomStack[curZoom].maxY)
  {
    _Yoffset += 1;
    zoomStack[curZoom].scroll(0, +1);
  }
}

/****************************************************************************/
void Plotter::SetTracePt(size_t CurveNum_, size_t Index_,
                         double Ptx, double Pty, double Radius_)
{
  if (fabs(Ptx) < 32768 && fabs(Pty) < 32768)
  {
    if (CurveNum_ < curveMap.size() && curveMap.size())
      setCaption(curveMap[CurveNum_]._Function);

    if (!_Tracing && _UseFileBuffers)
    {
      int x;
      int len = curveMap.size();
      for (x = 0; x < len; x++)
        ReadCurve(x);
    }

    _CurveNum = CurveNum_;
    _TraceIndex = Index_;
    _Tracing = true;
    _TracePtx = Ptx;
    _TracePty = Pty;
    _Radius = Radius_;

    _Xplot->setText(QString("X = ") + QString::number(Ptx));
    _Yplot->setText(QString("Y = ") + QString::number(Pty));
    _Xplot->show();
    _Yplot->show();

    _TracePts[0] = Ptx - Radius_;
    _TracePts[1] = Pty;
    _TracePts[2] = Ptx + Radius_;
    _TracePts[3] = Pty;
    _TracePts[4] = Ptx;
    _TracePts[5] = Pty - Radius_;
    _TracePts[6] = Ptx;
    _TracePts[7] = Pty + Radius_;
  }
}

/****************************************************************************/
void Plotter::DrawGraphMarkers(QPainter *painter, const double* points, const QColor& color_)
{
  // points[0] : -secant point (-2)
  // points[1] : +secant point (+2)
  // points[2] : -intersect point (-.5)
  // points[3] : +intersect point (+.5)

  int i;
  PlotSettings settings = zoomStack[curZoom];
  QPointArray realpoints(4);
  QRect rect(Margin, Margin,
             width() - 2 * Margin, height() - 2 * Margin);

  if (_SetToPrint)
    painter->setClipRect(_PrintArea.x() + 1, _PrintArea.y() + 1,
                         _PrintArea.width() - 2, _PrintArea.height() - 2);
  else
    painter->setClipRect(rect.x() + 1, rect.y() + 1,
                         rect.width() - 2, rect.height() - 2);

  for (i = 0; i < 4; i++)
  {
    double dx = points[2 * i] - settings.minX;
    double dy = points[2 * i + 1] - settings.minY;
    double x = rect.left() + (dx * (rect.width() - 1)
                                 / settings.spanX());
    double y = rect.bottom() - (dy * (rect.height() - 1)
                                   / settings.spanY());

    realpoints[i].setX((int)x);
    realpoints[i].setY((int)y);
  }

  if (_SetToPrint)
    painter->setPen(QPen(color_, 3));
  else
    painter->setPen(color_);

  painter->drawLine(realpoints[0], realpoints[1]);
  painter->drawLine(realpoints[2], realpoints[3]);
}

/****************************************************************************/
void Plotter::PlotTracePt(QPainter *painter, const double* points)
{
  QPoint Cpt1_;
  QPoint Cpt2_;
  PlotSettings settings = zoomStack[curZoom];
  PlotSettings settings0 = zoomStack[0];
  QRect rect(Margin, Margin,
             width() - 2 * Margin, height() - 2 * Margin);

  if (_SetToPrint)
    painter->setClipRect(_PrintArea.x() + 1, _PrintArea.y() + 1,
                         _PrintArea.width() - 2, _PrintArea.height() - 2);
  else
    painter->setClipRect(rect.x() + 1, rect.y() + 1,
                         rect.width() - 2, rect.height() - 2);

  double dx = (_TracePtx - (_InfPtRad * settings.spanX() / settings0.spanX())) - settings.minX;
  double dy = (_TracePty - (_InfPtRad * settings.spanY() / settings0.spanY())) - settings.minY;
  double x = rect.left() + (dx * (rect.width() - 1) / settings.spanX());
  double y = rect.bottom() - (dy * (rect.height() - 1) / settings.spanY());

  if (fabs(x) < 32768 && fabs(y) < 32768)
    Cpt1_ = QPoint((int)x, (int)y);

  dx = (_TracePtx + (_InfPtRad * 2 * settings.spanX() / settings0.spanX())) - settings.minX;
  dy = (_TracePty + (_InfPtRad * 2 * settings.spanY() / settings0.spanY())) - settings.minY;
  x = (rect.left() + (dx * (rect.width() - 1) / settings.spanX())) - x;
  y = (rect.bottom() - (dy * (rect.height() - 1) / settings.spanY())) - y;

  if (fabs(x) < 32768 && fabs(y) < 32768)
    Cpt2_ = QPoint((int)x, (int)y);

  if (_SetToPrint)
    painter->setPen(QPen(Qt::blue, 3));
  else
    painter->setPen(Qt::blue);

  painter->setBrush(QBrush(Qt::blue, Qt::SolidPattern));
  painter->drawEllipse(Cpt1_.x(), Cpt1_.y(), Cpt2_.x(), Cpt2_.y());
  DrawGraphMarkers(painter, points, Qt::blue);
}

/****************************************************************************/
void Plotter::PlotSecantLine(QPainter *painter, const double* points)
{
  DrawGraphMarkers(painter, points, Qt::red);
}

/****************************************************************************/
void Plotter::drawCurves(QPainter *painter)
{
    static const QColor colorForIds[3] = {
        yellow, magenta, cyan
    };
    PlotSettings settings = zoomStack[curZoom];
    PlotSettings settings0 = zoomStack[0];
    QRect rect(Margin, Margin,
               width() - 2 * Margin, height() - 2 * Margin);

    if (_SetToPrint)
      painter->setClipRect(_PrintArea.x() + 1, _PrintArea.y() + 1,
                           _PrintArea.width() - 2, _PrintArea.height() - 2);
    else
      painter->setClipRect(rect.x() + 1, rect.y() + 1,
                           rect.width() - 2, rect.height() - 2);

    map<int, GraphData>::const_iterator it = curveMap.begin();
    QPointArray infpoints;
    QPointArray points;
    
    int fill;
    int maxPoints;
    int maxInfPts;
    int numPoints;
    double dx;
    double dy;
    double xpt;
    double ypt;
    
    bool* ptsdrawn_ = NULL;
    bool* infptsvalid = NULL;
    bool ptsins_;
    bool drawn_;

    while (it != curveMap.end())
    {
        size_t id = (*it).first;
        if (!_Tracing && id != _ActiveCurve && _UseFileBuffers)
        {
          SaveCurve(_ActiveCurve);
          ReadCurve(id);
        }

        const GraphData &gdata = (*it).second;
        if (!gdata._CurveData || !gdata._InfinityPts)
          break;

        const CurveData &data = *gdata._CurveData;
        const CurveData &infdata = *gdata._InfinityPts;

        fill = data[0] ? 1:0;
        maxPoints = (data.size() - 1) / 2;
        maxInfPts = infdata.size() / 2;
        numPoints = 0;
	ptsins_ = false;

        #if PLOTTER_DEBUG
          QMessageBox::warning(0, "DEBUG",
                               QString("max infinity pts:") +
                               QString("\nmax = ") + QString::number(maxInfPts),
                               QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
        #endif

        QPoint Pt1;
        QPoint Pt2;
        QPoint Cpt1_;
        QPoint Cpt2_;

        if (maxPoints)
        {
          int i;
          ptsdrawn_ = new bool[maxPoints];
          ::memset(ptsdrawn_, 0, sizeof(bool) * maxPoints);
  
          points = QPointArray(maxPoints);
          numPoints = 0;

          for (i = 0; i < maxPoints; ++i)
          {
              dx = data[1 + (2 * i)] - settings.minX;
              dy = data[1 + (2 * i + 1)] - settings.minY;
              double x = rect.left() + (dx * (rect.width() - 1)
                                           / settings.spanX());
              double y = rect.bottom() - (dy * (rect.height() - 1)
                                             / settings.spanY());
              double zero = rect.bottom() - (-settings.minY * (rect.height() - 1)
                                             / settings.spanY());

              if (fabs(x) < 32768 && fabs(y) < 32768)
              {
                  points[numPoints] = QPoint((int)x, (int)y);
                  ++numPoints;

                  if (fill)
                  {
                    if (numPoints % 2)
                    {
                      Pt1.setX(int(x));
                      Pt1.setY(int(y));

                      if (numPoints > 2)
                      {
                        QPoint temp = Pt2;
                        Pt2 = Pt1;
                        Pt1 = temp;
                        FillSubinterval(painter, Pt1, Pt2, int(zero));
                      }
                    }
                    else
                    {
                      Pt2.setX(int(x));
                      Pt2.setY(int(y));
                      FillSubinterval(painter, Pt1, Pt2, int(zero));
                    }
                  }
              }
          }

          points.truncate(numPoints);
          maxPoints = numPoints;
          numPoints = 0;

          if (maxPoints && maxInfPts)
          {
	    infptsvalid = new bool[maxInfPts * 3];
	    ::memset(infptsvalid, 0, sizeof(bool) * maxInfPts * 3);
	    
            infpoints = QPointArray(maxInfPts * 3);
            numPoints = 0;

            for (i = 0; i < maxInfPts; ++i)
            {
                ptsins_ = false;
		infptsvalid[numPoints] = false;
                dx = infdata[2 * i] - settings.minX;
                dy = infdata[2 * i + 1] - settings.minY;
                double x = rect.left() + (dx * (rect.width() - 1) / settings.spanX());
                double y = rect.bottom() - (dy * (rect.height() - 1) / settings.spanY());

                if (fabs(x) < 32768 && fabs(y) < 32768)
                {
                    infpoints[numPoints] = QPoint((int)x, (int)y);
                    infptsvalid[numPoints] = true;
                    ++numPoints;
                    ptsins_ = true;
                }
                
                if (ptsins_)
                {
                    dx = (infdata[2 * i] - (_InfPtRad * settings.spanX() / settings0.spanX())) - settings.minX;
                    dy = (infdata[2 * i + 1] - (_InfPtRad * settings.spanY() / settings0.spanY())) - settings.minY;
                    x = rect.left() + (dx * (rect.width() - 1) / settings.spanX());
                    y = rect.bottom() - (dy * (rect.height() - 1) / settings.spanY());
                }

                if (ptsins_ && fabs(x) < 32768 && fabs(y) < 32768)
                {
                    infpoints[numPoints] = QPoint((int)x, (int)y);
                    infptsvalid[numPoints] = true;
                    ++numPoints;
                }
                else if (ptsins_)
                {
                    infpoints[numPoints] = QPoint(0, 0);
                    infptsvalid[numPoints] = false;
                    ++numPoints;
                }

                if (ptsins_)
                {
                    dx = (infdata[2 * i] + (_InfPtRad * 2 * settings.spanX() / settings0.spanX())) - settings.minX;
                    dy = (infdata[2 * i + 1] + (_InfPtRad * 2 * settings.spanY() / settings0.spanY())) - settings.minY;
                    x = (rect.left() + (dx * (rect.width() - 1) / settings.spanX())) - x;
                    y = (rect.bottom() - (dy * (rect.height() - 1) / settings.spanY())) - y;
                }

                if (ptsins_ && fabs(x) < 32768 && fabs(y) < 32768)
                {
                    infpoints[numPoints] = QPoint((int)x, (int)y);
                    infptsvalid[numPoints] = true;
                    ++numPoints;

                    #if PLOTTER_DEBUG
                       QMessageBox::warning(0, "DEBUG",
                          QString("\ninfinity pt saved:") +
                          QString("\nx = ") + QString::number(infpoints[numPoints-3].x()) +
                          QString("\ny = ") + QString::number(infpoints[numPoints-3].y()),
                          QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
                    #endif
                }
                else if (ptsins_)
                {
                    infpoints[numPoints] = QPoint(0, 0);
                    infptsvalid[numPoints] = false;
                    ++numPoints;
                }
            }

            infpoints.truncate(numPoints);
            maxInfPts = numPoints;
            numPoints = 0;

            if (_SetToPrint)
              painter->setPen(QPen(colorForIds[(uint)id % 3], 3));
            else
              painter->setPen(colorForIds[(uint)id % 3]);

            if (maxPoints && maxInfPts)
            {
              int x, y;
              drawn_ = false;
      
              for (x = i = 0; x < maxPoints; x++)
                for (y = 0; y < maxInfPts; y += 3)
                  if (infptsvalid[y] && points[x].x() == infpoints[y].x() && points[x].y() == infpoints[y].y())
                  {
		    if (i <= x)
                    {
                      if (x - i + 1 > 1 && i < maxPoints-1)
		      {
                        if (_SetToPrint)
                          painter->setPen(QPen(colorForIds[(uint)id % 3], 3));
                        else
                          painter->setPen(colorForIds[(uint)id % 3]);

                        painter->drawPolyline(points, i, x - i + 1);
                        ::memset(&ptsdrawn_[i], 1, sizeof(bool) * (x - i));
                        drawn_ = true;
		      }
		      else if (x - i + 1 > 0 && i <= maxPoints-1 && infptsvalid[y+1] && infptsvalid[y+2])
		      {
			ptsins_ = false;
                        dx = (points[i].x() - (_InfPtRad * settings.spanX() / settings0.spanX())) - settings.minX;
                        dy = (points[i].y() - (_InfPtRad * settings.spanY() / settings0.spanY())) - settings.minY;
                        xpt = rect.left() + (dx * (rect.width() - 1) / settings.spanX());
                        ypt = rect.bottom() - (dy * (rect.height() - 1) / settings.spanY());

                        if (fabs(xpt) < 32768 && fabs(ypt) < 32768)
			{
                          Cpt1_ = QPoint((int)xpt, (int)ypt);
			  ptsins_ = true;
			}

			if (ptsins_)
			{
                          dx = (points[i].x() + (_InfPtRad * 2 * settings.spanX() / settings0.spanX())) - settings.minX;
                          dy = (points[i].y() + (_InfPtRad * 2 * settings.spanY() / settings0.spanY())) - settings.minY;
                          xpt = (rect.left() + (dx * (rect.width() - 1) / settings.spanX())) - xpt;
                          ypt = (rect.bottom() - (dy * (rect.height() - 1) / settings.spanY())) - ypt;
			}

                        if (ptsins_ && fabs(xpt) < 32768 && fabs(ypt) < 32768)
                          Cpt2_ = QPoint((int)xpt, (int)ypt);
			else
			  ptsins_ = false;

			if (ptsins_)
			{
                          #if PLOTTER_DEBUG
                             QMessageBox::warning(0, "DEBUG",
                                QString("\nplotting infinity pt:") +
                                QString("\nx = ") + QString::number(Cpt1_.x()) +
                                QString("\ny = ") + QString::number(Cpt1_.y()) +
                                QString("\nxrad = ") + QString::number(Cpt2_.x()) +
                                QString("\nyrad = ") + QString::number(Cpt2_.y()),
                                QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
                          #endif

                          if (_SetToPrint)
                            painter->setPen(QPen(Qt::red, 3));
                          else
                            painter->setPen(Qt::red);

                          painter->setBrush(QBrush(Qt::red, Qt::SolidPattern));
                          painter->drawEllipse(Cpt1_.x(), Cpt1_.y(), Cpt2_.x(), Cpt2_.y());
                        }
                      }
                    }

                    if (drawn_ || (infptsvalid[y+1] && infptsvalid[y+2]))
                    {
                      i = x + 1;
                      drawn_ = false;
                    }
                  }

              if (i <= x && x - i + 1 > 1 && i < maxPoints-1)
              {
                if (_SetToPrint)
                  painter->setPen(QPen(colorForIds[(uint)id % 3], 3));
                else
                  painter->setPen(colorForIds[(uint)id % 3]);

                if (x >= maxPoints)
                {
                  painter->drawPolyline(points, i, maxPoints - i);
                  ::memset(&ptsdrawn_[i], 1, sizeof(bool) * (maxPoints - i));
                }
                else
                {
                  painter->drawPolyline(points, i, x - i + 1);
                  ::memset(&ptsdrawn_[i], 1, sizeof(bool) * (x - i));
                }
              }
            }
            else if (maxPoints)
            {
              painter->drawPolyline(points);
	      ::memset(ptsdrawn_, 1, sizeof(bool) * maxPoints);
            }

            if (gdata._HasSecant)
            {
              i = 1;
              PlotSecantLine(painter, gdata._SecantData);

              if (i < gdata._NumSecants)
              {
                i = 2;
                PlotSecantLine(painter, gdata._SecantData2);

                if (i < gdata._NumSecants)
                  PlotSecantLine(painter, gdata._SecantData3);
              }
            }

            if (maxPoints && maxInfPts)
            {
              int xi, yi;
              int xp, yp;
              int xrd;
              int yrd;
	      int segcnt;

              for (xi = 0; xi < maxPoints; xi++)
                for (yi = 0; yi < maxInfPts; yi += 3)
                  if (infptsvalid[yi] && points[xi].x() == infpoints[yi].x() && points[xi].y() == infpoints[yi].y())
                  {
                    xrd = infpoints[yi+2].x();
                    yrd = infpoints[yi+2].y();
		    drawn_ = false;

                    #if PLOTTER_DEBUG
                       QMessageBox::warning(0, "DEBUG",
                          QString("\nplotting infinity pt:") +
                          QString("\nx = ") + QString::number(infpoints[yi].x()) +
                          QString("\ny = ") + QString::number(infpoints[yi].y()) +
                          QString("\nxrad = ") + QString::number(xrd) +
                          QString("\nyrad = ") + QString::number(yrd),
                          QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
                    #endif

                    if (infptsvalid[yi+1] && infptsvalid[yi+2])
                    {
                      if (_SetToPrint)
                        painter->setPen(QPen(Qt::red, 3));
                      else
                        painter->setPen(Qt::red);

                      painter->setBrush(QBrush(Qt::red, Qt::SolidPattern));
                      painter->drawEllipse(infpoints[yi+1].x(), infpoints[yi+1].y(), xrd, yrd);
		      drawn_ = true;
                    }
                    
                    // make sure connecting line segments to infinity point is drawn
                    // make sure lead-in line segment is drawn
                    if (drawn_ && xi > 0)
                    {
                      xp = xi - 1;
		      segcnt = 1;
      
                      if (_SetToPrint)
                        painter->setPen(QPen(colorForIds[(uint)id % 3], 3));
                      else
                        painter->setPen(colorForIds[(uint)id % 3]);

                      do
                      {
                        for (yp = 0; yp < maxInfPts; yp += 3)
                          if (infptsvalid[yp] && points[xp].x() == infpoints[yp].x() && points[xp].y() == infpoints[yp].y())
                            break;
  
                        if (yp >= maxInfPts)
                        {
                          --xp;
                          ++segcnt;
                        }
                        else
                        {
                          ++xp;
                          --segcnt;
                          break;
                        }
                      }
                      while (xp >= 0 && segcnt < 9);
		      
                      if (segcnt > 0)
                      {
                        painter->drawPolyline(points, xp, segcnt+1);
                        ::memset(&ptsdrawn_[xp], 1, sizeof(bool) * (segcnt+1));
                      }
                    }                   
		    
                    // make sure lead-out line segment is drawn
                    if (drawn_ && xi < maxPoints-1)
                    {
                      xp = xi + 1;
		      segcnt = 1;

                      if (_SetToPrint)
                        painter->setPen(QPen(colorForIds[(uint)id % 3], 3));
                      else
                        painter->setPen(colorForIds[(uint)id % 3]);

                      do
                      {
                        for (yp = 0; yp < maxInfPts; yp += 3)
                          if (infptsvalid[yp] && points[xp].x() == infpoints[yp].x() && points[xp].y() == infpoints[yp].y())
                            break;

                        if (yp >= maxInfPts)
                        {
                          ++xp;
                          ++segcnt;
                        }
                        else
                        {
                          --xp;
                          --segcnt;
                          break;
                        }
                      }
                      while (xp < maxPoints && segcnt < 9);
		      
                      if (segcnt > 0)
                      {
                        if (xp == maxPoints)
                        {
                          --xp;
                          --segcnt;
                        }
			
                        painter->drawPolyline(points, xp-segcnt, segcnt+1);
                        ::memset(&ptsdrawn_[xp-segcnt], 1, sizeof(bool) * (segcnt+1));
                      }
                    }
                  }
            }
                
            if (maxPoints && maxInfPts)
            {
              delete[] infptsvalid;
              infptsvalid = NULL;
      
              delete[] ptsdrawn_;
              ptsdrawn_ = NULL;
      
              if (_SetToPrint)
                painter->setPen(QPen(colorForIds[(uint)id % 3], 3));
              else
                painter->setPen(colorForIds[(uint)id % 3]);
            }
          }
          else if (maxPoints)
          {
            if (_SetToPrint)
              painter->setPen(QPen(colorForIds[(uint)id % 3], 3));
            else
              painter->setPen(colorForIds[(uint)id % 3]);

            painter->drawPolyline(points);

            if (gdata._HasSecant)
            {
              i = 1;
              PlotSecantLine(painter, gdata._SecantData);

              if (i < gdata._NumSecants)
              {
                i = 2;
                PlotSecantLine(painter, gdata._SecantData2);

                if (i < gdata._NumSecants)
                  PlotSecantLine(painter, gdata._SecantData3);
              }
            }
          }
        }

        ++it;
    }

    if (_Tracing && curveMap.size())
      PlotTracePt(painter, _TracePts);
}

/****************************************************************************/
PlotSettings::PlotSettings()
{
    minX = 0.0;
    maxX = 10.0;
    numXTicks = 10;

    minY = 0.0;
    maxY = 10.0;
    numYTicks = 10;
}

/****************************************************************************/
void PlotSettings::scroll(int dx, int dy)
{
    double stepX = spanX() / numXTicks;
    minX += dx * stepX;
    maxX += dx * stepX;

    double stepY = spanY() / numYTicks;
    minY += dy * stepY;
    maxY += dy * stepY;
}

/****************************************************************************/
void PlotSettings::adjust()
{
    adjustAxis(minX, maxX, numXTicks);
    adjustAxis(minY, maxY, numYTicks);
}

/****************************************************************************/
void PlotSettings::adjustAxis(double &min, double &max,
                              int &numTicks)
{
    const int MinTicks = 4;
    double grossStep = (max - min) / MinTicks;
    double step = pow(10.0, floor(log10(grossStep)));

    if (5 * step < grossStep)
        step *= 5;
    else if (2 * step < grossStep)
        step *= 2;

    numTicks = (int)(ceil(max / step) - floor(min / step));
    min = floor(min / step) * step;
    max = ceil(max / step) * step;
}
