#ifndef CALCDRIVER_H
  #include "calcdriver.h"
#endif

// CalcDriver debugging data
#if CALCDRIVER_DEBUGGING
size_t CalcDriver::_errline = 0;
size_t CalcDriver::_errcol = 0;
int CalcDriver::_errcode = 0;
Debugging<CalcDriver>* CalcDriver::_dbgptr = NULL;
#endif

/****************************************************************************/
// async mode server/client polling scheme:
//
// 0. inital file states
//
//    client alive file: IDLE
//    poll state file: IDLE
//    server alive file: IDLE
//
// 1. server-to-client
//
// 1a. i: check for server states in iostate file.
//     ii: check poll state file for IDLE state.
//     iii: check client alive file for IDLE state.
//     proceeed if conditions at i, ii and iii is true.
//     send polling signal to client
//
// 1b. client alive file:  IDLE
//     poll state file: SERVER_TO_CLIENT\n (cycle number)
//     server alive file: POLL_CLIENTALIVE\n (cycle number)
//
// 2. client receives server poll signal
//
// 2a. check for server states in iostate file, proceed if server states confirmed:
//     send back client alive signal
//
// 2b. client alive file: CLIENT_ALIVE\n (cycle number from server)
//     poll state file: CLIENT_RESPONSE
//     server alive file: POLL_CLIENTALIVE
//
// 3. server receives client alive signal and sends back acknowledgement
//    send back server acknowledged signal
//
// 3a. client alive file: CLIENT_ALIVE
//     poll state file: SERVER_TO_CLIENT
//     server alive file: SERVER_ACKNOWLEDGED\n (cycle number)
//
// 4. client receives server acknowledgement and sends reset polling timer signal
//    send back polling timer reset signal
//
// 4a. client resets own no-activity polling timer
//
// 4b. client alive file: RESET_POLLING\n (cycle number)
//     poll state file: CLIENT_RESPONSE
//     server alive file: SERVER_ACKNOWLEDGED
//
// 5. server receives reset polling timer signal and resets all polling timers (if timers exist)
//    Sets poll state file to IDLE state. Sets server alive file to IDLE state.
//
// 5a. server resets polling timer, server resets no-activity timer (if timers exist)
//
// 5b. client alive file: RESET_POLLING
//     poll state file: IDLE
//     server alive file: IDLE
//
// 6. client-to-server
//
// 6a. i: check for client states in iostate file.
//     ii: check poll state file for IDLE state.
//     iii: check server alive file for IDLE state.
//     proceeed if conditions at i, ii and iii is true.
//     send polling signal to server
//
// 6b. client alive file:  POLL_SERVERALIVE\n (cycle number)
//     poll state file: CLIENT_TO_SERVER\n (cycle number)
//     server alive file: IDLE
//
// 7. server receives client poll signal
//
// 7a. check for client states in iostate file, proceed if client states confirmed:
//     send back server alive signal
//
// 7b. client alive file: POLL_SERVERALIVE
//     poll state file: SERVER_RESPONSE
//     server alive file: SERVER_ALIVE\n (cycle number from client)
//
// 8. client receives server alive signal and sends back acknowledgement
//    send back client acknowledged signal
//
// 8a. client alive file: CLIENT_ACKNOWLEDGED\n (cycle number)
//     poll state file: CLIENT_TO_SERVER
//     server alive file: SERVER_ALIVE
//
// 9. server receives client acknowledgement and sends reset polling timer signal
//    send back polling timer reset signal
//
// 9a. server resets own no-activity polling timer (if timer exists)
//
// 9b. client alive file: CLIENT_ACKNOWLEDGED
//     poll state file: SERVER_RESPONSE
//     server alive file: RESET_POLLING\n (cycle number)
//
// 10. client receives reset polling timer signal and resets all polling timers
//     Sets poll state file to IDLE state. Sets client alive file to IDLE state.
//
// 10a. client resets polling timer, client resets no-activity timer
//
// 10b. client alive file: IDLE
//      poll state file: IDLE
//      server alive file: RESET_POLLING
//
/****************************************************************************/
// ServerPollingMgr class definition
/****************************************************************************/
ServerPollingMgr::ServerPollingMgr(CalcDriver* Parent_, bool CreateBackup_):
_Parent(Parent_),
_StrBuffer((char*)::memset(new char[BUFSIZE], 0, BUFSIZE)),

_CycleNum(0),
_CycleActive(false),
_RetrievedCycle(0),

_ClientPollProcess(0),
_ServerPollProcess(0),
_PrevServerProcess(0),
_PrevClientProcess(0),

_RetryReadCount(0),
_RetryPollCount(0),
_PollState(0),
_ServerState(0),
_ClientState(0),
_RdClientPoll(true),

_AllowPollServer(false),
_SleepWaiting(false),
_ResetStatesOnDone(false),
_BreakOnPollError(false),
_KeepAlivePollError(false),
_ConfirmedClientAlive(true),
_ConfirmedServerAlive(true),
_ServerAliveConfSent(false),

_AsyncMode(Parent_ ? Parent_->InAsyncMode():false),
_ExitAtRetryMax(false),
_StopTracking(false),
_WaitUntilResponse(false),
_PausePollResults(false),
_ResetWhenNotReq(false),
_ForceReset(false),

#if CALCTEST_DEBUG3
  _Backup(CreateBackup_ ? (new ServerPollingMgr(Parent_, false)):NULL)
#else
  _Backup(NULL)
#endif
{}

/****************************************************************************/
ServerPollingMgr::~ServerPollingMgr()
{
  _Parent->ClosePollStateFile();
  _Parent->CloseClientAliveFile();
  _Parent->CloseServerAliveFile();

  delete[] _StrBuffer;
  _StrBuffer = NULL;

  if (_Backup)
  {
    delete _Backup;
    _Backup = NULL;
  }
}

/****************************************************************************/
ServerPollingMgr& ServerPollingMgr::operator = (const ServerPollingMgr& Obj_)
{
  if (&Obj_ != this)
  {
    strcpy(_StrBuffer, Obj_._StrBuffer);

    _CycleNum = Obj_._CycleNum;
    _CycleActive = Obj_._CycleActive;
    _RetrievedCycle = Obj_._RetrievedCycle;

    _ClientPollProcess = Obj_._ClientPollProcess;
    _ServerPollProcess = Obj_._ServerPollProcess;
    _PrevClientProcess = Obj_._PrevClientProcess;
    _PrevServerProcess = Obj_._PrevServerProcess;

    _RetryReadCount = Obj_._RetryReadCount;
    _RetryPollCount = Obj_._RetryPollCount;
    _PollState = Obj_._PollState;
    _ServerState = Obj_._ServerState;
    _ClientState = Obj_._ClientState;

    _AllowPollServer = Obj_._AllowPollServer;
    _SleepWaiting = Obj_._SleepWaiting;
    _ResetStatesOnDone = Obj_._ResetStatesOnDone;
    _BreakOnPollError = Obj_._BreakOnPollError;
    _KeepAlivePollError = Obj_._KeepAlivePollError;
    _ConfirmedClientAlive = Obj_._ConfirmedClientAlive;
    _ConfirmedServerAlive = Obj_._ConfirmedServerAlive;
    _ServerAliveConfSent = Obj_._ServerAliveConfSent;

    _AsyncMode = Obj_._AsyncMode;
    _ExitAtRetryMax = Obj_._ExitAtRetryMax;
    _StopTracking = Obj_._StopTracking;
    _WaitUntilResponse = Obj_._WaitUntilResponse;
    _PausePollResults = Obj_._PausePollResults;

    _ResetWhenNotReq = Obj_._ResetWhenNotReq;
    _ForceReset = Obj_._ForceReset;
  }

  return *this;
}

/****************************************************************************/
const char* ServerPollingMgr::KeepAliveStateToStr(int State_)
{
  return
  (
    (State_ == Mcalc_KeepAliveState::IDLE)                ? "IDLE":
    (State_ == Mcalc_KeepAliveState::RESET_POLLING)       ? "RESET_POLLING":
    
    (State_ == Mcalc_KeepAliveState::POLL_SERVERALIVE)    ? "POLL_SERVERALIVE":
    (State_ == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED) ? "CLIENT_ACKNOWLEDGED":
    (State_ == Mcalc_KeepAliveState::CLIENT_ALIVE)        ? "CLIENT_ALIVE":

    (State_ == Mcalc_KeepAliveState::POLL_CLIENTALIVE)    ? "POLL_CLIENTALIVE":
    (State_ == Mcalc_KeepAliveState::SERVER_ACKNOWLEDGED) ? "SERVER_ACKNOWLEDGED":
    (State_ == Mcalc_KeepAliveState::SERVER_ALIVE)        ? "SERVER_ALIVE":""
  );
}

/****************************************************************************/
const char* ServerPollingMgr::PollStateToStr(int State_)
{
  return
  (
    (State_ == Mcalc_PollState::IDLE)             ? "IDLE":
    (State_ == Mcalc_PollState::SERVER_TO_CLIENT) ? "SERVER_TO_CLIENT":
    (State_ == Mcalc_PollState::CLIENT_RESPONSE)  ? "CLIENT_RESPONSE":
    (State_ == Mcalc_PollState::CLIENT_TO_SERVER) ? "CLIENT_TO_SERVER":
    (State_ == Mcalc_PollState::SERVER_RESPONSE)  ? "SERVER_RESPONSE":""
  );
}

/****************************************************************************/
void ServerPollingMgr::ResetPollResults(bool& PollingServer_,
                                        bool& PollError_,
                                        bool& PollAnswered_)
{
  PollingServer_ = false;
  PollAnswered_ = false;
  PollError_ = false;
}

/****************************************************************************/
void ServerPollingMgr::ShowPollResults(bool PollingServer_,
                                       bool PollError_,
                                       bool PollAnswered_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ShowPollResults");
  #endif

  #if CALCTEST_DEBUG3
    #if CALCDRIVER_TEXT_VERSION
      std::cerr <<"Polling Server: " <<PollingServer_ <<endl
                <<"Poll Answered: " <<PollAnswered_ <<endl
                <<"Poll Error: " <<PollError_ <<endl;

      if (_PausePollResults)
        std::cin.get();
    #else
      _Parent->DbgPtr()->ShowInt(PollingServer_, "PollingServer_");
      _Parent->DbgPtr()->ShowInt(PollAnswered_, "PollAnswered_");
      _Parent->DbgPtr()->ShowInt(PollError_, "PollError_");
    #endif

    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void ServerPollingMgr::GivePollResults(bool& PollingServer_,
                                       bool& PollError_,
                                       bool& PollAnswered_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("GivePollResults");
  #endif

  if (!PollError_)
    PollingServer_ = ClientIsPolling() && PollServerInProcess();

  if (!PollingServer_)
    PollAnswered_ = _ConfirmedServerAlive &&
                    (PollServerInProcess() || PollServerCompleted(false));

  if (!PollingServer_ && !PollAnswered_)
    PollError_ = _KeepAlivePollError && _RetryPollCount > 0 &&
                 PollServerInProcess();

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(PollingServer_, "PollingServer_");
    _Parent->DbgPtr()->ShowInt(PollAnswered_, "PollAnswered_");
    _Parent->DbgPtr()->ShowInt(PollError_, "PollError_");

    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetSleepWaiting(bool flag_)
{
  _SleepWaiting = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetResetStatesOnDone(bool flag_)
{
  _ResetStatesOnDone = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetBreakOnPollError(bool flag_)
{
  _BreakOnPollError = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetAllowServerPolling(bool flag_)
{
  _AllowPollServer = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetExitAtRetryMax(bool flag_)
{
  _ExitAtRetryMax = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetWaitUntilResponse(bool flag_)
{
  _WaitUntilResponse = flag_;
  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetStopTracking(bool flag_)
{
  _StopTracking = flag_;

  #if CALCDRIVER_DEBUGGING
    if (_Parent && _Parent->DbgPtr())
    {
      _Parent->DbgPtr()->SetStopTracking(CalcDriver::SERVERPOLLINGMGR, flag_);

      if (flag_)
        _Parent->DbgPtr()->SetTrackType(CalcDriver::SERVERPOLLINGMGR);
      else
        _Parent->DbgPtr()->SetTrackType(0);
    }
  #endif

  if (_Backup)
    _Backup->_StopTracking = flag_;

  return this;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::SetPausePollResults(bool flag_)
{
  _PausePollResults = flag_;
  return this;
}

/****************************************************************************/
bool ServerPollingMgr::PausePollResults() const
{
  return _PausePollResults;
}

/****************************************************************************/
bool ServerPollingMgr::StopTracking() const
{
  return _StopTracking;
}

/****************************************************************************/
bool ServerPollingMgr::ServerIsPolling() const
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ServerIsPolling");
    bool retval = ((_PollState == Mcalc_PollState::SERVER_TO_CLIENT ||
                    _PollState == Mcalc_PollState::CLIENT_RESPONSE) &&
                   _ServerState == Mcalc_KeepAliveState::POLL_CLIENTALIVE);
                   
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return ((_PollState == Mcalc_PollState::SERVER_TO_CLIENT ||
             _PollState == Mcalc_PollState::CLIENT_RESPONSE) &&
            _ServerState == Mcalc_KeepAliveState::POLL_CLIENTALIVE);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ClientIsAlive() const
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ClientIsAlive");
    bool ServerAcked_ = ServerAcked();
    bool retval = ((_PollState == Mcalc_PollState::CLIENT_RESPONSE ||
                    ServerAcked_) &&
                   _ClientState == Mcalc_KeepAliveState::CLIENT_ALIVE &&
                   _RetrievedCycle == _CycleNum);

    retval = retval || ServerAcked_;
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    
    return retval;
  #else
    bool ServerAcked_ = ServerAcked();
    bool retval = ((_PollState == Mcalc_PollState::CLIENT_RESPONSE ||
                    ServerAcked_) &&
                   _ClientState == Mcalc_KeepAliveState::CLIENT_ALIVE &&
                   _RetrievedCycle == _CycleNum);

    retval = retval || ServerAcked_;
    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ServerAcked() const
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ServerAcked");
    bool cond1 = ((_PollState == Mcalc_PollState::SERVER_TO_CLIENT ||
                   _PollState == Mcalc_PollState::CLIENT_RESPONSE) &&
                  _ServerState == Mcalc_KeepAliveState::SERVER_ACKNOWLEDGED &&
                  _RetrievedCycle == _CycleNum);
    bool cond2 = ((_PollState == Mcalc_PollState::CLIENT_RESPONSE ||
                   _PollState == Mcalc_PollState::IDLE) &&
                  _ServerState == Mcalc_KeepAliveState::SERVER_ACKNOWLEDGED &&
                  _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));
    bool retval = cond1 || cond2;

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    bool cond1 = ((_PollState == Mcalc_PollState::SERVER_TO_CLIENT ||
                   _PollState == Mcalc_PollState::CLIENT_RESPONSE) &&
                  _ServerState == Mcalc_KeepAliveState::SERVER_ACKNOWLEDGED &&
                  _RetrievedCycle == _CycleNum);
    bool cond2 = ((_PollState == Mcalc_PollState::CLIENT_RESPONSE ||
                   _PollState == Mcalc_PollState::IDLE) &&
                  _ServerState == Mcalc_KeepAliveState::SERVER_ACKNOWLEDGED &&
                  _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));
    bool retval = cond1 || cond2;

    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ResetSignalSent()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ResetSignalSent");
    bool ClientResponse_ = _PollState == Mcalc_PollState::CLIENT_RESPONSE &&
                           ClientReceiveCompleted(false);
    bool retval = ((ClientResponse_ || PollingResetByServer()) &&
                   _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
                   _RetrievedCycle == _CycleNum);

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    bool ClientResponse_ = _PollState == Mcalc_PollState::CLIENT_RESPONSE &&
                           ClientReceiveCompleted(false);
    return ((ClientResponse_ || PollingResetByServer()) &&
            _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
            _RetrievedCycle == _CycleNum);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::PollingResetByServer()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("PollingResetByServer");
    bool retval = (ClientReceiveCompleted(false) &&
                   _PollState == Mcalc_KeepAliveState::IDLE &&
                   _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
                   _ServerState == Mcalc_KeepAliveState::IDLE);

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return (ClientReceiveCompleted(false) &&
            _PollState == Mcalc_KeepAliveState::IDLE &&
            _ClientState == Mcalc_KeepAliveState::RESET_POLLING &&
            _ServerState == Mcalc_KeepAliveState::IDLE);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::AtInitialState()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("AtInitialState");
    int IoState_ = _Parent->SetNoPolling(true)
                          ->Bare_GetIoState(0);
    bool PollServerDone_ = _ServerPollProcess == REQUIRED;
    bool PollClientDone_ = ClientReceiveCompleted(false) ||
                           _ClientPollProcess == INITIALSTATE;
    bool retval =
           (_Parent->IsClientStates(IoState_) &&
            PollServerDone_ && PollClientDone_ &&
            (_ServerState == Mcalc_KeepAliveState::IDLE ||
             _ServerState == Mcalc_KeepAliveState::RESET_POLLING) &&
            _PollState == Mcalc_KeepAliveState::IDLE &&
            (_ClientState == Mcalc_KeepAliveState::IDLE ||
             _ClientState == Mcalc_KeepAliveState::RESET_POLLING));

    _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
    _Parent->DbgPtr()->ShowInt(PollClientDone_, "PollClientDone_");
    _Parent->DbgPtr()->ShowInt(PollServerDone_, "PollServerDone_");
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    int IoState_ = _Parent->SetNoPolling(true)
                          ->Bare_GetIoState(0);
    bool PollServerDone_ = _ServerPollProcess == REQUIRED;
    bool PollClientDone_ = ClientReceiveCompleted(false) ||
                           _ClientPollProcess == INITIALSTATE;

    return (_Parent->IsClientStates(IoState_) &&
            PollServerDone_ && PollClientDone_ &&
            (_ServerState == Mcalc_KeepAliveState::IDLE ||
             _ServerState == Mcalc_KeepAliveState::RESET_POLLING) &&
            _PollState == Mcalc_KeepAliveState::IDLE &&
            (_ClientState == Mcalc_KeepAliveState::IDLE ||
             _ClientState == Mcalc_KeepAliveState::RESET_POLLING));
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ClientIsPolling() const
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ClientIsPolling");
    bool retval = ((_PollState == Mcalc_PollState::CLIENT_TO_SERVER ||
                    _PollState == Mcalc_PollState::SERVER_RESPONSE) &&
                   _ClientState == Mcalc_KeepAliveState::POLL_SERVERALIVE);

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return ((_PollState == Mcalc_PollState::CLIENT_TO_SERVER ||
             _PollState == Mcalc_PollState::SERVER_RESPONSE) &&
            _ClientState == Mcalc_KeepAliveState::POLL_SERVERALIVE);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ServerIsAlive() const
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ServerIsAlive");
    bool ClientAcked_ = ClientAcked();
    bool retval = ((_PollState == Mcalc_PollState::SERVER_RESPONSE ||
                    ClientAcked_) &&
                   _ServerState == Mcalc_KeepAliveState::SERVER_ALIVE &&
                   _RetrievedCycle == _CycleNum);

    retval = retval || ClientAcked_;
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    
    return retval;
  #else
    bool ClientAcked_ = ClientAcked();
    bool retval = ((_PollState == Mcalc_PollState::SERVER_RESPONSE ||
                    ClientAcked_) &&
                   _ServerState == Mcalc_KeepAliveState::SERVER_ALIVE &&
                   _RetrievedCycle == _CycleNum);

    retval = retval || ClientAcked_;
    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ClientAcked() const
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ClientAcked");
    bool cond1 = ((_PollState == Mcalc_PollState::CLIENT_TO_SERVER ||
                   _PollState == Mcalc_PollState::SERVER_RESPONSE) &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _RetrievedCycle == _CycleNum);
    bool cond2 = ((_PollState == Mcalc_PollState::SERVER_RESPONSE ||
                   _PollState == Mcalc_PollState::IDLE) &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _ServerState == Mcalc_KeepAliveState::RESET_POLLING &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));
    bool cond3 = (_PollState == Mcalc_PollState::IDLE &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _ServerState == Mcalc_KeepAliveState::IDLE &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));

    bool retval = cond1 || cond2 || cond3;

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    bool cond1 = ((_PollState == Mcalc_PollState::CLIENT_TO_SERVER ||
                   _PollState == Mcalc_PollState::SERVER_RESPONSE) &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _RetrievedCycle == _CycleNum);
    bool cond2 = ((_PollState == Mcalc_PollState::SERVER_RESPONSE ||
                   _PollState == Mcalc_PollState::IDLE) &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _ServerState == Mcalc_KeepAliveState::RESET_POLLING &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));
    bool cond3 = (_PollState == Mcalc_PollState::IDLE &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _ServerState == Mcalc_KeepAliveState::IDLE &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));
                  
    return (cond1 || cond2 || cond3);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ResetRequested()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ResetRequested");
    bool PollingReset_ = PollingReset();
    bool cond1 = ((_PollState == Mcalc_PollState::SERVER_RESPONSE ||
                   _PollState == Mcalc_PollState::IDLE || PollingReset_) &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _ServerState == Mcalc_KeepAliveState::RESET_POLLING &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));
    bool cond2 = (_PollState == Mcalc_PollState::IDLE &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _ServerState == Mcalc_KeepAliveState::IDLE &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));

    bool retval = cond1 || cond2 || PollingReset_;
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();

    return retval;
  #else
    bool PollingReset_ = PollingReset();
    bool cond1 = ((_PollState == Mcalc_PollState::SERVER_RESPONSE ||
                   _PollState == Mcalc_PollState::IDLE || PollingReset_) &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _ServerState == Mcalc_KeepAliveState::RESET_POLLING &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));
    bool cond2 = (_PollState == Mcalc_PollState::IDLE &&
                  _ClientState == Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED &&
                  _ServerState == Mcalc_KeepAliveState::IDLE &&
                  (_RetrievedCycle == _CycleNum || _RetrievedCycle == 0));

    bool retval = cond1 || cond2 || PollingReset_;
    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::PollingReset()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("PollingReset");
    bool retval = (PollServerCompleted(false) &&
                   _PollState == Mcalc_KeepAliveState::IDLE &&
                   _ClientState == Mcalc_KeepAliveState::IDLE &&
                   _ServerState == Mcalc_KeepAliveState::RESET_POLLING);

    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return (PollServerCompleted(false) &&
            _PollState == Mcalc_KeepAliveState::IDLE &&
            _ClientState == Mcalc_KeepAliveState::IDLE &&
            _ServerState == Mcalc_KeepAliveState::RESET_POLLING);
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::WaitingForServerResponse(bool* FromClt_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("WaitingForServerResponse");
    
    bool FromClient_ = SHOWMETHOD(_Parent, PollServerInProcess());
    bool FromServer_ = SHOWMETHOD(_Parent, ClientReceiveInProcess());

    _Parent->DbgPtr()->ShowInt(FromClient_, "FromClient_");
    _Parent->DbgPtr()->ShowInt(FromServer_, "FromServer_");
  #else
    bool FromClient_ = PollServerInProcess();
    bool FromServer_ = ClientReceiveInProcess();
  #endif

  if (FromClt_)
    *FromClt_ = FromClient_;

  bool retval = (FromClient_ ? (ClientIsPolling() || ClientAcked()):
                 FromServer_ ? (ClientIsAlive() || ResetSignalSent()):
                               false);

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return retval;
}

/****************************************************************************/
bool ServerPollingMgr::ResponseRequired(int PrevPollState_,
                                        int& LoopGuard_, int LoopLimit_)
{
  LoopGuard_++;
  return (_WaitUntilResponse && LoopGuard_ < LoopLimit_ &&
          ((PrevPollState_ == Mcalc_PollState::CLIENT_TO_SERVER &&
            _PollState != Mcalc_PollState::SERVER_RESPONSE) ||
           (PrevPollState_ == Mcalc_PollState::CLIENT_RESPONSE &&
            _PollState != Mcalc_PollState::SERVER_TO_CLIENT &&
            _PollState != Mcalc_PollState::IDLE)));
}

/****************************************************************************/
bool ServerPollingMgr::ServerSetPollInProcess(bool* PollStateOk_) const
{
  int ResetState_ = _ServerState == Mcalc_KeepAliveState::RESET_POLLING ?
                       Mcalc_PollState::IDLE:-1;
  int NewState_ = (_ServerState == Mcalc_KeepAliveState::POLL_CLIENTALIVE ||
                   _ServerState == Mcalc_KeepAliveState::SERVER_ACKNOWLEDGED) ?
                       Mcalc_PollState::SERVER_TO_CLIENT:-1;

  bool PollStateMatch_ = _PollState == NewState_ ||
                         _PollState == ResetState_;
  if (PollStateOk_)
    *PollStateOk_ = PollStateMatch_;
  
  return (_ServerState == Mcalc_KeepAliveState::POLL_CLIENTALIVE ||
          _ServerState == Mcalc_KeepAliveState::SERVER_ACKNOWLEDGED ||
          _ServerState == Mcalc_KeepAliveState::RESET_POLLING);
}

/****************************************************************************/
bool ServerPollingMgr::ClientSetPollReset(bool* PollStateOk_) const
{
  int ResetState_ = _ClientState == Mcalc_KeepAliveState::RESET_POLLING ?
                       Mcalc_PollState::IDLE:-1;
  int NewState_ = (_ClientState == Mcalc_KeepAliveState::RESET_POLLING ||
                   _ClientState == Mcalc_KeepAliveState::CLIENT_ALIVE) ?
                       Mcalc_PollState::CLIENT_RESPONSE:-1;

  bool PollStateMatch_ = _PollState == NewState_ ||
                         _PollState == ResetState_;
  if (PollStateOk_)
    *PollStateOk_ = PollStateMatch_;
  
  return (_ClientState == Mcalc_KeepAliveState::RESET_POLLING ||
          _ClientState == Mcalc_KeepAliveState::CLIENT_ALIVE);
}

/****************************************************************************/
int ServerPollingMgr::ReadClientPollProcess(bool RefreshStates_)
{
  bool Reset_ = false;
  bool PsOk_ = false;
  _RdClientPoll = false;

  if (RefreshStates_)
  {
    ReadServerState(0);
    ReadClientState(false);
  }

  if (ClientSetPollReset(&PsOk_))
  {
    if (!PsOk_)
    {
      _PollState = Mcalc_PollState::CLIENT_RESPONSE;
      PsOk_ = SetPollState(_PollState);
    }

    Reset_ = _ClientState == Mcalc_KeepAliveState::RESET_POLLING;

    if (Reset_)
    {
      _ClientPollProcess = COMPLETED;
      _PollState = Mcalc_PollState::IDLE;
    }
    else
      _ClientPollProcess = INPROCESS;
  }
  else if (ServerSetPollInProcess(&PsOk_))
  {
    if (!PsOk_)
    {
      Reset_ = _ServerState == Mcalc_KeepAliveState::RESET_POLLING;
      _PollState = Reset_ ? Mcalc_PollState::IDLE:
                            Mcalc_PollState::SERVER_TO_CLIENT;

      PsOk_ = SetPollState(_PollState);
    }

    Reset_ = _ServerState == Mcalc_KeepAliveState::RESET_POLLING;

    if (Reset_)
    {
      _ClientPollProcess = COMPLETED;
      _PollState = Mcalc_PollState::IDLE;
    }
    else
      _ClientPollProcess = INPROCESS;
  }

  _RdClientPoll = true;
  return PsOk_;
}

/****************************************************************************/
int ServerPollingMgr::ReadPollState(int Sleep_)
{
  int mult_ = 0;
  int PrevPollState_ = _PollState;
  int LoopGuard_ = 0;
  int LoopMax_ = LOOP_GUARD_LIMIT;

  int x;
  char* testp = NULL;

  #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  #else
    // microsec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MICRO_DELAY_MULT;
  #endif
  
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ReadServerState");
    FILE* Pfptr_ = SHOWPARENTMETHOD(_Parent, GetPollStateFile(READ, true));
  #else
    FILE* Pfptr_ = _Parent->GetPollStateFile(READ, true);
  #endif

  if (Pfptr_ && feof(Pfptr_))
    rewind(Pfptr_);

  if (Pfptr_ && !feof(Pfptr_))
  {
    mult_ = (Sleep_ > 1) ? Sleep_:1;
    
    #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
      Sleep(mult_ * DelayLen_);
    #else
      ::usleep(mult_ * DelayLen_);
    #endif
  }

  do
  {
    if (Pfptr_ && feof(Pfptr_))
      rewind(Pfptr_);
  
    if (Pfptr_ && !feof(Pfptr_))
    {
      ::memset(_StrBuffer, 0, BUFSIZE);
      testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);

      for (x = 0; !testp && x < LoopMax_; x++)
      {
        #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
          Sleep(DelayLen_);
        #else
          ::usleep(DelayLen_);
        #endif

        Pfptr_ = _Parent->GetPollStateFile(READ, true);

        if (Pfptr_ && feof(Pfptr_))
          rewind(Pfptr_);

        if (Pfptr_ && !feof(Pfptr_))
          testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);
      }

      if (testp)
        _PollState = atoi(testp);
    }
  }
  while (ResponseRequired(PrevPollState_, LoopGuard_, LoopMax_));

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowStr(PollStateToStr(_PollState), "_PollState");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return _PollState;
}

/****************************************************************************/
int ServerPollingMgr::ReadServerState(int Sleep_)
{
  int mult_ = 0;
  int PrevPollState_ = _PollState;
  int LoopGuard_ = 0;
  int LoopMax_ = LOOP_GUARD_LIMIT;
  bool SleepDone_ = false;

  int x;
  char* testp = NULL;

  #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  #else
    // microsec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MICRO_DELAY_MULT;
  #endif
  
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ReadServerState");
    
    FILE* Pfptr_ = SHOWPARENTMETHOD(_Parent, GetPollStateFile(READ, true));
    FILE* Sfptr_ = SHOWPARENTMETHOD(_Parent, GetServerAliveFile(READ, true));
  #else
    FILE* Pfptr_ = _Parent->GetPollStateFile(READ, true);
    FILE* Sfptr_ = _Parent->GetServerAliveFile(READ, true);
  #endif

  if (Pfptr_ && feof(Pfptr_))
    rewind(Pfptr_);

  if (Pfptr_ && !feof(Pfptr_))
  {
    mult_ = (Sleep_ > 1) ? Sleep_:1;
    SleepDone_ = true;
    
    #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
      Sleep(mult_ * DelayLen_);
    #else
      ::usleep(mult_ * DelayLen_);
    #endif
  }

  do
  {
    if (Pfptr_ && feof(Pfptr_))
      rewind(Pfptr_);
  
    if (Pfptr_ && !feof(Pfptr_))
    {
      ::memset(_StrBuffer, 0, BUFSIZE);
      testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);

      for (x = 0; !testp && x < LoopMax_; x++)
      {
        #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
          Sleep(DelayLen_);
        #else
          ::usleep(DelayLen_);
        #endif

        SleepDone_ = true;
        Pfptr_ = _Parent->GetPollStateFile(READ, true);

        if (Pfptr_ && feof(Pfptr_))
          rewind(Pfptr_);

        if (Pfptr_ && !feof(Pfptr_))
          testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);
      }

      if (testp)
        _PollState = atoi(testp);
    }

    // Wait some time before getting server state
    if (!SleepDone_ &&
        Sleep_ && (!_SleepWaiting || WaitingForServerResponse()))
    {
      mult_ = (Sleep_ > 1) ? Sleep_:1;
      SleepDone_ = true;

      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(mult_ * DelayLen_);
      #else
        ::usleep(mult_ * DelayLen_);
      #endif

      #if CALCTEST_DEBUG3
        _Parent->DbgPtr()->ShowMessage("WaitingForServerResponse == 1");
        _Parent->DbgPtr()->ShowInt(_SleepWaiting, "_SleepWaiting");
        _Parent->DbgPtr()->ShowInt(Sleep_, "Sleep_");
      #endif
    }

    SleepDone_ = false;
  }
  while (ResponseRequired(PrevPollState_, LoopGuard_, LoopMax_));

  if (Sfptr_ && feof(Sfptr_))
    rewind(Sfptr_);

  if (Sfptr_ && !feof(Sfptr_))
  {
    ::memset(_StrBuffer, 0, BUFSIZE);
    testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
    
    for (x = 0; !testp && x < LoopMax_; x++)
    {
      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(DelayLen_);
      #else
        ::usleep(DelayLen_);
      #endif
    
      Sfptr_ = _Parent->GetServerAliveFile(READ, true);

      if (Sfptr_ && feof(Sfptr_))
        rewind(Sfptr_);

      if (Sfptr_ && !feof(Sfptr_))
        testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
    }

    if (testp)
      _ServerState = atoi(testp);

    if (Sfptr_ && !feof(Sfptr_))
    {
      testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
      _RetrievedCycle = testp ? atoi(testp):0;
    }
    else
      _RetrievedCycle = 0;
  }
  
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowStr(PollStateToStr(_PollState), "_PollState");
    _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ServerState), "_ServerState");
    _Parent->DbgPtr()->ShowInt(_RetrievedCycle, "_RetrievedCycle");
    
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return _ServerState;
}

/****************************************************************************/
int ServerPollingMgr::ReadClientState(bool RdPoll_)
{
  int LoopMax_ = LOOP_GUARD_LIMIT;
  int x;
  char* testp = NULL;

  #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  #else
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MICRO_DELAY_MULT;
  #endif

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ReadClientState");
    
    FILE* Pfptr_ = SHOWPARENTMETHOD(_Parent, GetPollStateFile(READ, true));
    FILE* Cfptr_ = SHOWPARENTMETHOD(_Parent, GetClientAliveFile(READ, true));
  #else
    FILE* Pfptr_ = _Parent->GetPollStateFile(READ, true);
    FILE* Cfptr_ = _Parent->GetClientAliveFile(READ, true);
  #endif

  if (Pfptr_ && feof(Pfptr_) && RdPoll_)
    rewind(Pfptr_);

  if (Pfptr_ && !feof(Pfptr_) && RdPoll_)
  {
    #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
      Sleep(DelayLen_);
    #else
      ::usleep(DelayLen_);
    #endif
  }

  if (Pfptr_ && !feof(Pfptr_) && RdPoll_)
  {
    ::memset(_StrBuffer, 0, BUFSIZE);
    testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);

    for (x = 0; !testp && x < LoopMax_; x++)
    {
      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(DelayLen_);
      #else
        ::usleep(DelayLen_);
      #endif
    
      Pfptr_ = _Parent->GetPollStateFile(READ, true);

      if (Pfptr_ && feof(Pfptr_))
        rewind(Pfptr_);

      if (Pfptr_ && !feof(Pfptr_))
        testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);
    }

    if (testp)
      _PollState = atoi(testp);
  }

  if (Cfptr_ && feof(Cfptr_))
    rewind(Cfptr_);

  if (Cfptr_ && !feof(Cfptr_))
  {
    ::memset(_StrBuffer, 0, BUFSIZE);
    testp = fgets(_StrBuffer, BUFSIZE, Cfptr_);

    for (x = 0; !testp && x < LoopMax_; x++)
    {
      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(DelayLen_);
      #else
        ::usleep(DelayLen_);
      #endif
    
      Cfptr_ = _Parent->GetClientAliveFile(READ, true);

      if (Cfptr_ && feof(Cfptr_))
        rewind(Cfptr_);

      if (Cfptr_ && !feof(Cfptr_))
        testp = fgets(_StrBuffer, BUFSIZE, Cfptr_);
    }

    if (testp)
      _ClientState = atoi(testp);
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowStr(PollStateToStr(_PollState), "_PollState");
    _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ClientState), "_ClientState");
    _Parent->DbgPtr()->ShowInt(_RetrievedCycle, "_RetrievedCycle");
    
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return _ClientState;
}

/****************************************************************************/
int ServerPollingMgr::ReadClientState()
{
  return ReadClientState(true);
}

/****************************************************************************/
void ServerPollingMgr::ReadAllStates(int Sleep_)
{
  int mult_ = 0;
  int PrevPollState_ = _PollState;
  int LoopGuard_ = 0;
  int LoopMax_ = LOOP_GUARD_LIMIT;
  bool SleepDone_ = false;

  int x;
  char* testp = NULL;

  #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  #else
    // microsec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MICRO_DELAY_MULT;
  #endif

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ReadAllStates");

    SHOWMETHOD(_Parent, ReadClientState(false));
    FILE* Pfptr_ = SHOWPARENTMETHOD(_Parent, GetPollStateFile(READ, true));
    FILE* Sfptr_ = SHOWPARENTMETHOD(_Parent, GetServerAliveFile(READ, true));
  #else
    ReadClientState(false);
    FILE* Pfptr_ = _Parent->GetPollStateFile(READ, true);
    FILE* Sfptr_ = _Parent->GetServerAliveFile(READ, true);
  #endif

  if (Pfptr_ && feof(Pfptr_))
    rewind(Pfptr_);

  if (Pfptr_ && !feof(Pfptr_))
  {
    mult_ = (Sleep_ > 1) ? Sleep_:1;
    SleepDone_ = true;
  
    #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
      Sleep(mult_ * DelayLen_);
    #else
      ::usleep(mult_ * DelayLen_);
    #endif
  }

  do
  {
    if (Pfptr_ && feof(Pfptr_))
      rewind(Pfptr_);
  
    if (Pfptr_ && !feof(Pfptr_))
    {
      ::memset(_StrBuffer, 0, BUFSIZE);
      testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);

      for (x = 0; !testp && x < LoopMax_; x++)
      {
        #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
          Sleep(DelayLen_);
        #else
          ::usleep(DelayLen_);
        #endif

        SleepDone_ = true;
        Pfptr_ = _Parent->GetPollStateFile(READ, true);

        if (Pfptr_ && feof(Pfptr_))
          rewind(Pfptr_);
        
        if (Pfptr_ && !feof(Pfptr_))
          testp = fgets(_StrBuffer, BUFSIZE, Pfptr_);
      }

      if (testp)
        _PollState = atoi(testp);
    }

    // Wait some time before getting server state
    if (!SleepDone_ &&
        Sleep_ && (!_SleepWaiting || WaitingForServerResponse()))
    {
      mult_ = (Sleep_ > 1) ? Sleep_:1;
      SleepDone_ = true;

      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(mult_ * DelayLen_);
      #else
        ::usleep(mult_ * DelayLen_);
      #endif

      #if CALCTEST_DEBUG3
        _Parent->DbgPtr()->ShowMessage("WaitingForServerResponse == 1");
        _Parent->DbgPtr()->ShowInt(_SleepWaiting, "_SleepWaiting");
        _Parent->DbgPtr()->ShowInt(Sleep_, "Sleep_");
      #endif
    }

    SleepDone_ = false;
  }
  while (ResponseRequired(PrevPollState_, LoopGuard_, LoopMax_));

  if (Sfptr_ && feof(Sfptr_))
    rewind(Sfptr_);

  if (Sfptr_ && !feof(Sfptr_))
  {
    ::memset(_StrBuffer, 0, BUFSIZE);
    testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
    
    for (x = 0; !testp && x < LoopMax_; x++)
    {
      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(DelayLen_);
      #else
        ::usleep(DelayLen_);
      #endif

      Sfptr_ = _Parent->GetServerAliveFile(READ, true);

      if (Sfptr_ && feof(Sfptr_))
        rewind(Sfptr_);

      if (Sfptr_ && !feof(Sfptr_))
        testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
    }

    if (testp)
      _ServerState = atoi(testp);

    if (Sfptr_ && !feof(Sfptr_))
    {
      testp = fgets(_StrBuffer, BUFSIZE, Sfptr_);
      _RetrievedCycle = testp ? atoi(testp):0;
    }
    else
      _RetrievedCycle = 0;
  }

  if (_RdClientPoll)
    ReadClientPollProcess(false);

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowStr(PollStateToStr(_PollState), "_PollState");
    _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ServerState), "_ServerState");
    _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ClientState), "_ClientState");
    _Parent->DbgPtr()->ShowInt(_RetrievedCycle, "_RetrievedCycle");
    
    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::SetPollState(int State_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("SetPollState");
    FILE* Fptr_ = SHOWPARENTMETHOD(_Parent, GetPollStateFile(WRITE));
  #else
    FILE* Fptr_ = _Parent->GetPollStateFile(WRITE);
  #endif

  if (Fptr_)
  {
    fprintf(Fptr_, "%d\n%d", State_, IncCycleNum());
    _Parent->ClosePollStateFile();

    #if CALCTEST_DEBUG3
      _Parent->DbgPtr()->ShowInt(State_, "State_");
      _Parent->DbgPtr()->ShowInt(true, "retval");
      _Parent->DbgPtr()->LeaveLevel();
    #endif

    return true;
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(false, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return false;
}

/****************************************************************************/
bool ServerPollingMgr::SetClientState(int State_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("SetClientState");
    FILE* Fptr_ = SHOWPARENTMETHOD(_Parent, GetClientAliveFile(WRITE));
  #else
    FILE* Fptr_ = _Parent->GetClientAliveFile(WRITE);
  #endif

  if (Fptr_)
  {
    fprintf(Fptr_, "%d\n%d", State_, IncCycleNum());
    _Parent->CloseClientAliveFile();

    #if CALCTEST_DEBUG3
      _Parent->DbgPtr()->ShowInt(State_, "State_");
      _Parent->DbgPtr()->ShowInt(true, "retval");
      _Parent->DbgPtr()->LeaveLevel();
    #endif

    return true;
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(false, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return false;
}

/****************************************************************************/
bool ServerPollingMgr::SetServerState(int State_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("SetServerState");
    FILE* Fptr_ = SHOWPARENTMETHOD(_Parent, GetServerAliveFile(WRITE));
  #else
    FILE* Fptr_ = _Parent->GetServerAliveFile(WRITE);
  #endif

  if (Fptr_)
  {
    fprintf(Fptr_, "%d\n%d", State_, IncCycleNum());
    _Parent->CloseServerAliveFile();

    #if CALCTEST_DEBUG3
      _Parent->DbgPtr()->ShowInt(State_, "State_");
      _Parent->DbgPtr()->ShowInt(true, "retval");
      _Parent->DbgPtr()->LeaveLevel();
    #endif

    return true;
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(false, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return false;
}

/****************************************************************************/
void ServerPollingMgr::ResetPollingVars()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ResetPollingVars");
    SHOWMETHOD(_Parent, ResetPollCount());
  #else
    ResetPollCount();
  #endif

  _KeepAlivePollError = false;
  _ConfirmedServerAlive = false;

  if (_ServerAliveConfSent)
  {
    #if CALCTEST_DEBUG3
      SHOWPARENTMETHOD(_Parent, SendServerAlive(false));
    #else
      _Parent->SendServerAlive(false);
    #endif

    _ServerAliveConfSent = false;
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void ServerPollingMgr::AcceptPollCycle()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("AcceptPollCycle");
  #endif

  _CycleActive = true;
  _CycleNum = _RetrievedCycle;

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(_CycleActive, "_CycleActive");
    _Parent->DbgPtr()->ShowInt(_CycleNum, "_CycleNum");
    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void ServerPollingMgr::ResetPollCycle()
{
  _CycleActive = false;
}

/****************************************************************************/
void ServerPollingMgr::SetNewPollCycle()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("SetNewPollCycle");
  #endif

  if (!_CycleActive)
  {
    _CycleActive = true;
    _CycleNum++;
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(_CycleActive, "_CycleActive");
    _Parent->DbgPtr()->ShowInt(_CycleNum, "_CycleNum");
    _Parent->DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
int ServerPollingMgr::IncCycleNum()
{
  if (!_CycleActive)
    return 0;

  return _CycleNum;
}

/****************************************************************************/
bool ServerPollingMgr::IsServerAliveStates(int State_)
{
  return
  (
    State_ == Mcalc_KeepAliveState::POLL_CLIENTALIVE ||
    State_ == Mcalc_KeepAliveState::SERVER_ALIVE
  );
}

/****************************************************************************/
bool ServerPollingMgr::IsClientAliveStates(int State_)
{
  return
  (
    State_ == Mcalc_KeepAliveState::CLIENT_ALIVE ||
    State_ == Mcalc_KeepAliveState::POLL_SERVERALIVE
  );
}

/****************************************************************************/
bool ServerPollingMgr::SendPollServerSignal()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("SendPollServerSignal");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool Required_ = _ServerPollProcess == REQUIRED;
  bool DoPollServer_ = !_Parent->IsServerStates(IoState_) &&
                       AtInitialState() && Required_;

  if (DoPollServer_ && _Parent->GetClientAliveFile(WRITE))
  {
    ResetPollingVars();
  
    _ClientState = Mcalc_KeepAliveState::POLL_SERVERALIVE;
    _ServerPollProcess = INPROCESS;

    #if CALCTEST_DEBUG5
      SetTrackingOnStateChange();
    #endif

    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_PollState::CLIENT_TO_SERVER;
      #if CALCTEST_DEBUG3
        _Parent->DbgPtr()->ShowInt(Required_, "Required_");
        _Parent->DbgPtr()->ShowInt(DoPollServer_, "DoPollServer_");
        _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ClientState), "_ClientState");
        _Parent->DbgPtr()->ShowStr(PollStateToStr(_PollState), "_PollState");
        _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess");
        _Parent->DbgPtr()->LeaveLevel();
        _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
      #endif
      
      return SetPollState(_PollState);
    }
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(Required_, "Required_");
    _Parent->DbgPtr()->ShowInt(DoPollServer_, "DoPollServer_");
    _Parent->DbgPtr()->LeaveLevel();
    _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
  #endif
  return false;
}

/****************************************************************************/
void ServerPollingMgr::ConfirmServerAlive(bool ClientReceive_)
{
  _RetryPollCount = 0;
  _KeepAlivePollError = false;
  _ConfirmedServerAlive = ClientReceive_ ? ServerAcked():ServerIsAlive();
}

/****************************************************************************/
bool ServerPollingMgr::SendClientAckServerSignal()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("SendClientAckServerSignal");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool DoAck_ = !_Parent->IsServerStates(IoState_) && ServerIsAlive() &&
                PollServerInProcess() && _ConfirmedServerAlive;
            
  if (DoAck_ && _Parent->GetClientAliveFile(WRITE))
  {
    _ClientState = Mcalc_KeepAliveState::CLIENT_ACKNOWLEDGED;
    _ServerPollProcess = INPROCESS;

    #if CALCTEST_DEBUG5
      SetTrackingOnStateChange();
    #endif
    
    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_PollState::CLIENT_TO_SERVER;
      #if CALCTEST_DEBUG3
        _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
        _Parent->DbgPtr()->ShowInt(DoAck_, "DoAck_");
        _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ClientState), "_ClientState");
        _Parent->DbgPtr()->ShowStr(PollStateToStr(_PollState), "_PollState");
        _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess");
        _Parent->DbgPtr()->LeaveLevel();
        _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
      #endif
      
      return SetPollState(_PollState);
    }
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
    _Parent->DbgPtr()->ShowInt(DoAck_, "DoAck_");
    _Parent->DbgPtr()->LeaveLevel();
    _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
  #endif
  return false;
}

/****************************************************************************/
bool ServerPollingMgr::ResetPolling()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ResetPolling");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool DoReset_ = !_Parent->IsServerStates(IoState_) && ResetRequested() &&
                  PollServerInProcess() && _ConfirmedServerAlive;

  if (DoReset_ && _Parent->GetClientAliveFile(WRITE))
  {
    ResetPollCycle();
    _KeepAlivePollError = false;
    
    _ClientState = Mcalc_KeepAliveState::IDLE;
    _ServerPollProcess = COMPLETED;

    #if CALCTEST_DEBUG5
      SetTrackingOnStateChange();
    #endif

    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_KeepAliveState::IDLE;
      #if CALCTEST_DEBUG3
        _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
        _Parent->DbgPtr()->ShowInt(DoReset_, "DoReset_");
        _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
        _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ClientState), "_ClientState");
        _Parent->DbgPtr()->ShowStr(PollStateToStr(_PollState), "_PollState");
        _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess");
        _Parent->DbgPtr()->LeaveLevel();
        _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
      #endif
      
      bool retval = SetPollState(_PollState);
      bool Alive_ = _ConfirmedServerAlive && _ConfirmedClientAlive;

      // Reset client side no-activity timer
      if (!_ServerAliveConfSent)
      {
        #if CALCTEST_DEBUG3
          SHOWPARENTMETHOD(_Parent, SendServerAlive(Alive_));
        #else
          _Parent->SendServerAlive(Alive_);
        #endif

        _ServerAliveConfSent = true;
      }

      return retval;
    }
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
    _Parent->DbgPtr()->ShowInt(DoReset_, "DoReset_");
    _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
    _Parent->DbgPtr()->LeaveLevel();
    _Parent->DbgPtr()->LeaveLevel();  // from PollServerIfAlive()
  #endif
  return false;
}

/****************************************************************************/
bool ServerPollingMgr::SendClientAliveSignal()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("SendClientAliveSignal");
  #endif

  if (_RdClientPoll)
    ReadClientPollProcess();

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool Required_ = _ClientPollProcess == REQUIRED ||
                   ServerSetPollInProcess();
  bool DoSendAlive_ = _Parent->IsServerStates(IoState_) &&
                      ServerIsPolling() && Required_;
            
  if (DoSendAlive_ && _Parent->GetClientAliveFile(WRITE))
  {
    _KeepAlivePollError = false;
    _ClientState = Mcalc_KeepAliveState::CLIENT_ALIVE;
    _ClientPollProcess = INPROCESS;

    #if CALCTEST_DEBUG5
      SetTrackingOnStateChange();
    #endif
    
    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_PollState::CLIENT_RESPONSE;
      #if CALCTEST_DEBUG3
        _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
        _Parent->DbgPtr()->ShowInt(Required_, "Required_");
        _Parent->DbgPtr()->ShowInt(DoSendAlive_, "DoSendAlive_");
        _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
        _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ClientState), "_ClientState");
        _Parent->DbgPtr()->ShowStr(PollStateToStr(_PollState), "_PollState");
        _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess");
        _Parent->DbgPtr()->LeaveLevel();
        _Parent->DbgPtr()->LeaveLevel();  // from ReceiveKeepAlivePoll()
      #endif

      return SetPollState(_PollState);
    }
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
    _Parent->DbgPtr()->ShowInt(Required_, "Required_");
    _Parent->DbgPtr()->ShowInt(DoSendAlive_, "DoSendAlive_");
    _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
    _Parent->DbgPtr()->LeaveLevel();
    _Parent->DbgPtr()->LeaveLevel();  // from ReceiveKeepAlivePoll()
  #endif
  return false;
}

/****************************************************************************/
void ServerPollingMgr::ConfirmClientAlive(bool PollServer_)
{
  _KeepAlivePollError = false;
  _ConfirmedClientAlive = PollServer_ ? ServerIsAlive():ClientIsAlive();
}

/****************************************************************************/
bool ServerPollingMgr::SendResetPollingSignal()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("SendResetPollingSignal");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool DoReset_ = _Parent->IsServerStates(IoState_) && ServerAcked() &&
                  ClientReceiveInProcess() && _ConfirmedClientAlive;
            
  if (DoReset_ && _Parent->GetClientAliveFile(WRITE))
  {
    ResetPollCycle();
    _KeepAlivePollError = false;
    
    _ClientState = Mcalc_KeepAliveState::RESET_POLLING;
    _ClientPollProcess = COMPLETED;

    #if CALCTEST_DEBUG5
      SetTrackingOnStateChange();
    #endif
    
    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_PollState::CLIENT_RESPONSE;
      #if CALCTEST_DEBUG3
        _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
        _Parent->DbgPtr()->ShowInt(DoReset_, "DoReset_");
        _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
        _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ClientState), "_ClientState");
        _Parent->DbgPtr()->ShowStr(PollStateToStr(_PollState), "_PollState");
        _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess");
        _Parent->DbgPtr()->LeaveLevel();
        _Parent->DbgPtr()->LeaveLevel();  // from ReceiveKeepAlivePoll()
      #endif

      bool retval = SetPollState(_PollState);
      bool Alive_ = _ConfirmedServerAlive && _ConfirmedClientAlive;

      // Reset client side no-activity timer
      if (!_ServerAliveConfSent)
      {
        #if CALCTEST_DEBUG3
          SHOWPARENTMETHOD(_Parent, SendServerAlive(Alive_));
        #else
          _Parent->SendServerAlive(Alive_);
        #endif
        
        _ServerAliveConfSent = true;
      }

      return retval;
    }
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(IoState_, "IoState_");
    _Parent->DbgPtr()->ShowInt(DoReset_, "DoReset_");
    _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError");
    _Parent->DbgPtr()->LeaveLevel();
    _Parent->DbgPtr()->LeaveLevel();  // from ReceiveKeepAlivePoll()
  #endif
  return false;
}

/****************************************************************************/
ServerPollingMgr* ServerPollingMgr::ResetWhenNotRequired(bool Forced_)
{
  _ResetWhenNotReq = true;
  _ForceReset = Forced_;

  #if CALCTEST_DEBUG5
    SetStopTracking(true);
  #endif
  
  return this;
}

/****************************************************************************/
bool ServerPollingMgr::PollServerRequired(int Sleep_, bool ResponseOnly_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("PollServerRequired");
    _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess(1)");
  #endif

  ReadAllStates(Sleep_);

  bool BreakOrQuit_ = _Parent->QuitAll() || _Parent->BreakAll();
  bool ServerResponse_ = _PollState == Mcalc_PollState::SERVER_RESPONSE;
  bool ResetReq_ = BreakOrQuit_ || ResetRequested();
  bool Required_ = ((_AllowPollServer && !ResponseOnly_) ||
                    ServerResponse_ || ResetReq_ || _KeepAlivePollError);

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess(2)");
    _Parent->DbgPtr()->ShowInt(_ResetWhenNotReq, "_ResetWhenNotReq(1)");

    _Parent->DbgPtr()->ShowInt(_AllowPollServer, "_AllowPollServer(1)");
    _Parent->DbgPtr()->ShowInt(ResponseOnly_, "ResponseOnly_(1)");
    _Parent->DbgPtr()->ShowInt(ServerResponse_, "ServerResponse_(1)");
    _Parent->DbgPtr()->ShowInt(ResetReq_, "ResetReq_(1)");
    _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError(1)");
    _Parent->DbgPtr()->ShowInt(Required_, "Required_(1)");
  #endif

  if (_ServerPollProcess == INITIALSTATE ||
      _ServerPollProcess == COMPLETED)
    _ServerPollProcess = _KeepAlivePollError ? NOTREQUIRED:
                         ResetReq_           ? COMPLETED:
                         Required_           ? REQUIRED:NOTREQUIRED;
  else if (_KeepAlivePollError)
    _ServerPollProcess = NOTREQUIRED;
  else if (ResetReq_)
    _ServerPollProcess = COMPLETED;

  bool retval = (_ServerPollProcess == REQUIRED ||
                 (_ServerPollProcess == INPROCESS && !_KeepAlivePollError));

  #if CALCTEST_DEBUG5
    if (!retval)
      SetTrackingOnStateChange();
  #endif

  #if CALCTEST_DEBUG3
    if (_ResetWhenNotReq && _ServerPollProcess == NOTREQUIRED)
    {
      _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess(3)");
      #if CALCTEST_DEBUG5
        SHOWMETHOD(_Parent, SetStopTracking(true));
      #endif
      SHOWMETHOD(_Parent, ResetServerPolling(_ForceReset));
      _ResetWhenNotReq = false;
    }

    _Parent->DbgPtr()->ShowInt(_ResetWhenNotReq, "_ResetWhenNotReq(2)");
    _Parent->DbgPtr()->ShowInt(_AllowPollServer, "_AllowPollServer(2)");
    _Parent->DbgPtr()->ShowInt(ResponseOnly_, "ResponseOnly_(2)");
    _Parent->DbgPtr()->ShowInt(ServerResponse_, "ServerResponse_(2)");
    _Parent->DbgPtr()->ShowInt(ResetReq_, "ResetReq_(2)");
    _Parent->DbgPtr()->ShowInt(_KeepAlivePollError, "_KeepAlivePollError(2)");
    _Parent->DbgPtr()->ShowInt(Required_, "Required_(2)");

    _Parent->DbgPtr()->ShowStr(PollStateToStr(_PollState), "_PollState");
    _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ClientState), "_ClientState");
    _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ServerState), "_ServerState");
    _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess(4)");
    _Parent->DbgPtr()->ShowInt(retval, "retval");

    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    if (_ResetWhenNotReq && _ServerPollProcess == NOTREQUIRED)
    {
      #if CALCTEST_DEBUG5
        SetStopTracking(true);
      #endif
      ResetServerPolling(_ForceReset);
      _ResetWhenNotReq = false;
    }

    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::PollServerNotRequired(bool InclAtInitial_)
{
  if (_RdClientPoll)
    ReadClientPollProcess(false);

  return ((_ServerPollProcess == NOTREQUIRED ||
           (_ServerPollProcess == INITIALSTATE && InclAtInitial_)) && !_KeepAlivePollError);
}

/****************************************************************************/
bool ServerPollingMgr::PollServerInProcess()
{
  if (_RdClientPoll)
    ReadClientPollProcess(false);

  return (_ServerPollProcess == INPROCESS && !_KeepAlivePollError);
}

/****************************************************************************/
bool ServerPollingMgr::PollServerCompleted_IMPL(bool InclNotReq_, bool InclPrev_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("PollServerCompleted_IMPL");
  #endif

  bool ResetCond_ = false;
  bool CurrCond_ = false;
  bool PrevCond_ = false;
  bool BreakOrQuit_ = _Parent->QuitAll() || _Parent->BreakAll();

  if (_RdClientPoll)
    ReadClientPollProcess(false);

  if (_ResetStatesOnDone && InclPrev_)
  {
    ResetCond_ = _ServerPollProcess == INITIALSTATE;
    PrevCond_ = ((_PrevServerProcess == COMPLETED ||
                  (_PrevServerProcess == NOTREQUIRED && InclNotReq_)) && !_KeepAlivePollError);
  }
  else if (BreakOrQuit_)
    _ServerPollProcess = COMPLETED;

  CurrCond_ = ((_ServerPollProcess == COMPLETED ||
                (_ServerPollProcess == NOTREQUIRED && InclNotReq_)) && !_KeepAlivePollError);

  #if CALCTEST_DEBUG3
    bool retval = (CurrCond_ || (_ResetStatesOnDone && ResetCond_ && PrevCond_));
    _Parent->DbgPtr()->ShowInt(_ResetStatesOnDone, "_ResetStatesOnDone");
    _Parent->DbgPtr()->ShowInt(InclPrev_, "InclPrev_");
    _Parent->DbgPtr()->ShowInt(InclNotReq_, "InclNotReq_");
    _Parent->DbgPtr()->ShowInt(ResetCond_, "ResetCond_");
    _Parent->DbgPtr()->ShowInt(PrevCond_, "PrevCond_");
    _Parent->DbgPtr()->ShowInt(CurrCond_, "CurrCond_");
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return (CurrCond_ || (_ResetStatesOnDone && ResetCond_ && PrevCond_));
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::PollServerCompleted(bool InclNotReq_)
{
  return PollServerCompleted_IMPL(InclNotReq_, true);
}

/****************************************************************************/
bool ServerPollingMgr::ClientReceiveRequired(int Sleep_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ClientReceiveRequired");
    _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess(1)");
  #endif

  ReadAllStates(Sleep_);

  bool BreakOrQuit_ = _Parent->QuitAll() || _Parent->BreakAll();
  bool ServerToClient_ = _PollState == Mcalc_PollState::SERVER_TO_CLIENT;
  bool ServerPolling_ = _ServerState == Mcalc_KeepAliveState::POLL_CLIENTALIVE;
  bool Required_ = ServerToClient_ || ServerPolling_;
  bool Completed_ = BreakOrQuit_ ||
                    (!_PollState && !_ServerState &&
                     (!_ClientState ||
                      _ClientState == Mcalc_KeepAliveState::RESET_POLLING));

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess(2)");
    _Parent->DbgPtr()->ShowInt(_ResetWhenNotReq, "_ResetWhenNotReq(1)");

    _Parent->DbgPtr()->ShowInt(ServerToClient_, "ServerToClient_(1)");
    _Parent->DbgPtr()->ShowInt(ServerPolling_, "ServerPolling_(1)");
    _Parent->DbgPtr()->ShowInt(Required_, "Required_(1)");
  #endif

  if (_ClientPollProcess == INITIALSTATE ||
      _ClientPollProcess == COMPLETED ||
      ServerSetPollInProcess())
    _ClientPollProcess = Completed_ ? COMPLETED:
                         Required_  ? REQUIRED:NOTREQUIRED;
  else if (Completed_)
    _ClientPollProcess = COMPLETED;

  bool retval = _ClientPollProcess != COMPLETED &&
                (_ClientPollProcess == REQUIRED || ServerSetPollInProcess() ||
                 (_ClientPollProcess == INPROCESS && !_KeepAlivePollError));

  #if CALCTEST_DEBUG5
    if (!retval)
      SetTrackingOnStateChange();
  #endif

  #if CALCTEST_DEBUG3
    if (_ResetWhenNotReq && _ClientPollProcess == NOTREQUIRED)
    {
      _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess(3)");
      #if CALCTEST_DEBUG5
        SHOWMETHOD(_Parent, SetStopTracking(true));
      #endif
      SHOWMETHOD(_Parent, ResetReceivePoll(_ForceReset));
      _ResetWhenNotReq = false;
    }

    _Parent->DbgPtr()->ShowInt(_ResetWhenNotReq, "_ResetWhenNotReq(2)");
    _Parent->DbgPtr()->ShowInt(ServerToClient_, "ServerToClient_(2)");
    _Parent->DbgPtr()->ShowInt(ServerPolling_, "ServerPolling_(2)");
    _Parent->DbgPtr()->ShowInt(Required_, "Required_(2)");

    _Parent->DbgPtr()->ShowStr(PollStateToStr(_PollState), "_PollState");
    _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ClientState), "_ClientState");
    _Parent->DbgPtr()->ShowStr(KeepAliveStateToStr(_ServerState), "_ServerState");
    _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess(4)");
    _Parent->DbgPtr()->ShowInt(retval, "retval");

    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    if (_ResetWhenNotReq && _ClientPollProcess == NOTREQUIRED)
    {
      #if CALCTEST_DEBUG5
        SetStopTracking(true);
      #endif
      ResetReceivePoll(_ForceReset);
      _ResetWhenNotReq = false;
    }

    return retval;
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ClientReceiveNotRequired(bool InclAtInitial_)
{
  if (_RdClientPoll)
    ReadClientPollProcess(false);

  return ((_ClientPollProcess == NOTREQUIRED ||
           (_ClientPollProcess == INITIALSTATE && InclAtInitial_)) && !_KeepAlivePollError);
}

/****************************************************************************/
bool ServerPollingMgr::ClientReceiveInProcess()
{
  if (_RdClientPoll)
    ReadClientPollProcess(false);

  return (_ClientPollProcess == INPROCESS && !_KeepAlivePollError);
}

/****************************************************************************/
bool ServerPollingMgr::ClientReceiveCompleted_IMPL(bool InclNotReq_, bool InclPrev_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ClientReceiveCompleted_IMPL");
  #endif

  bool ResetCond_ = false;
  bool CurrCond_ = false;
  bool PrevCond_ = false;
  bool BreakOrQuit_ = _Parent->QuitAll() || _Parent->BreakAll();

  if (_RdClientPoll)
    ReadClientPollProcess(false);

  if (_ResetStatesOnDone && InclPrev_)
  {
    ResetCond_ = _ClientPollProcess == INITIALSTATE;
    PrevCond_ = ((_PrevClientProcess == COMPLETED ||
                  (_PrevClientProcess == NOTREQUIRED && InclNotReq_)) && !_KeepAlivePollError);
  }
  else if (BreakOrQuit_)
    _ClientPollProcess = COMPLETED;

  CurrCond_ = ((_ClientPollProcess == COMPLETED ||
                (_ClientPollProcess == NOTREQUIRED && InclNotReq_)) && !_KeepAlivePollError);

  #if CALCTEST_DEBUG3
    bool retval = (CurrCond_ || (_ResetStatesOnDone && ResetCond_ && PrevCond_));
    _Parent->DbgPtr()->ShowInt(_ResetStatesOnDone, "_ResetStatesOnDone");
    _Parent->DbgPtr()->ShowInt(InclPrev_, "InclPrev_");
    _Parent->DbgPtr()->ShowInt(InclNotReq_, "InclNotReq_");
    _Parent->DbgPtr()->ShowInt(ResetCond_, "ResetCond_");
    _Parent->DbgPtr()->ShowInt(PrevCond_, "PrevCond_");
    _Parent->DbgPtr()->ShowInt(CurrCond_, "CurrCond_");
    _Parent->DbgPtr()->ShowInt(retval, "retval");
    
    _Parent->DbgPtr()->LeaveLevel();
    return retval;
  #else
    return (CurrCond_ || (_ResetStatesOnDone && ResetCond_ && PrevCond_));
  #endif
}

/****************************************************************************/
bool ServerPollingMgr::ClientReceiveCompleted(bool InclNotReq_)
{
  return ClientReceiveCompleted_IMPL(InclNotReq_, true);
}

/****************************************************************************/
void ServerPollingMgr::SetTrackingOnStateChange(bool Reset_)
{
  #if CALCTEST_DEBUG5
    bool PollTestRetVal_ = false;
  #endif

  if (_RdClientPoll)
    ReadClientPollProcess();

  if (_Backup)
  {
    bool ChangeFound_ = false;

    if (_Backup->_ClientPollProcess != _ClientPollProcess)
    {
      if (Reset_ && _ClientPollProcess == NOTREQUIRED)
      {
        ResetWhenNotRequired(true);        
        #if CALCTEST_DEBUG5
          SetStopTracking(true);
        #endif
        
        #if CALCTEST_DEBUG3
          SHOWMETHOD(_Parent, ResetReceivePoll(_ForceReset));
        #else
          ResetReceivePoll(_ForceReset);
        #endif
      }
      else if (!_ResetWhenNotReq || _ClientPollProcess != NOTREQUIRED)
        SetStopTracking(false);
      
      _Backup->_ClientPollProcess = _ClientPollProcess;
      ChangeFound_ = true;
    }
    else if (!ChangeFound_ &&
             _Backup->_ClientPollProcess == _ClientPollProcess)
    {
      #if CALCTEST_DEBUG5
        PollTestRetVal_ = (_ClientPollProcess == REQUIRED || ServerSetPollInProcess() ||
                           (_ClientPollProcess == INPROCESS && !_KeepAlivePollError));

        if (!PollTestRetVal_)
          SetStopTracking(true);
      #endif

      if (Reset_ && _ClientPollProcess == NOTREQUIRED)
      {
        ResetWhenNotRequired(true);
        
        #if CALCTEST_DEBUG3
          SHOWMETHOD(_Parent, ResetReceivePoll(_ForceReset));
        #else
          ResetReceivePoll(_ForceReset);
        #endif
      }
    }

    if (_Backup->_ServerPollProcess != _ServerPollProcess)
    {
      if (Reset_ && _ServerPollProcess == NOTREQUIRED)
      {
        ResetWhenNotRequired(true);
        #if CALCTEST_DEBUG5
          SetStopTracking(true);
        #endif
        
        #if CALCTEST_DEBUG3
          SHOWMETHOD(_Parent, ResetServerPolling(_ForceReset));
        #else
          ResetServerPolling(_ForceReset);
        #endif
      }
      else if (!_ResetWhenNotReq || _ClientPollProcess != NOTREQUIRED)
        SetStopTracking(false);

      _Backup->_ServerPollProcess = _ServerPollProcess;
      ChangeFound_ = true;
    }
    else if (!ChangeFound_ &&
             _Backup->_ServerPollProcess == _ServerPollProcess)
    {
      #if CALCTEST_DEBUG5
        PollTestRetVal_ = (_ServerPollProcess == REQUIRED ||
                           (_ServerPollProcess == INPROCESS && !_KeepAlivePollError));

        if (!PollTestRetVal_)
          SetStopTracking(true);
      #endif

      if (Reset_ && _ServerPollProcess == NOTREQUIRED)
      {
        ResetWhenNotRequired(true);
        
        #if CALCTEST_DEBUG3
          SHOWMETHOD(_Parent, ResetServerPolling(_ForceReset));
        #else
          ResetServerPolling(_ForceReset);
        #endif
      }
    }
  }
}

/****************************************************************************/
bool ServerPollingMgr::ResetServerPolling(bool ForceReset_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ResetServerPolling");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool NotInServerState_ = !_Parent->IsServerStates(IoState_) &&
                           !IsServerAliveStates(_ServerState);
  bool PollComplete_ = NotInServerState_ && PollServerCompleted_IMPL(true, false);
  bool retval = false;
  ForceReset_ = NotInServerState_ && ForceReset_;

  if ((ForceReset_ || PollComplete_) &&
      _Parent->GetClientAliveFile(WRITE))
  {
    #if CALCTEST_DEBUG3
      SHOWMETHOD(_Parent, ResetPollCycle());
    #else
      ResetPollCycle();
    #endif
    _KeepAlivePollError = false;
    
    _ClientState = Mcalc_KeepAliveState::IDLE;
    _PrevServerProcess = _ServerPollProcess;
    _ServerPollProcess = INITIALSTATE;

    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_KeepAliveState::IDLE;
      retval = SetPollState(_PollState);
      
      #if CALCTEST_DEBUG3
        _Parent->DbgPtr()->ShowInt(_ServerState, "_ServerState");
        _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
        _Parent->DbgPtr()->ShowInt(_PrevClientProcess, "_PrevClientProcess");
        _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess");
        _Parent->DbgPtr()->ShowInt(retval, "retval");        
        _Parent->DbgPtr()->LeaveLevel();
      #endif

      return retval;
    }
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(false, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return false;
}

/****************************************************************************/
bool ServerPollingMgr::ResetReceivePoll(bool ForceReset_)
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ResetReceivePoll");
  #endif

  bool retval = true;
  bool ReceiveComplete_ = ClientReceiveCompleted_IMPL(true, false);

  if (ForceReset_ || ReceiveComplete_)
  {
    _ServerState = Mcalc_KeepAliveState::IDLE;
    
    _PrevClientProcess = _ClientPollProcess;
    _ClientPollProcess = INITIALSTATE;

    if (!_PollState && !_ServerState &&
        _ClientState == Mcalc_KeepAliveState::RESET_POLLING)
    {
      _ClientState = Mcalc_KeepAliveState::IDLE;
      retval = SetClientState(_ClientState);
    }
    else
      _ClientState = Mcalc_KeepAliveState::IDLE;

    #if CALCTEST_DEBUG3
      _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
      _Parent->DbgPtr()->ShowInt(_PrevServerProcess, "_PrevServerProcess");
      _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess");
      _Parent->DbgPtr()->ShowInt(true, "retval");
      
      _Parent->DbgPtr()->LeaveLevel();
    #endif

    return retval;
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(false, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return false;
}

/****************************************************************************/
bool ServerPollingMgr::ForceResetAllPolling()
{
  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ForceResetAllPolling");
  #endif

  int IoState_ = _Parent->SetNoPolling(true)
                        ->Bare_GetIoState(0);
  bool ForceReset_ = !_Parent->IsServerStates(IoState_) &&
                     !IsServerAliveStates(_ServerState);

  if (ForceReset_ && _Parent->GetClientAliveFile(WRITE))
  {
    ResetPollCycle();
    _KeepAlivePollError = false;
    
    _ClientState = Mcalc_KeepAliveState::IDLE;
    _PrevServerProcess = _ServerPollProcess;
    _ServerPollProcess = INITIALSTATE;
    
    _ServerState = Mcalc_KeepAliveState::IDLE;
    _PrevClientProcess = _ClientPollProcess;
    _ClientPollProcess = INITIALSTATE;

    #if CALCTEST_DEBUG3
      _Parent->DbgPtr()->ShowInt(_ServerState, "_ServerState");
      _Parent->DbgPtr()->ShowInt(_PrevClientProcess, "_PrevClientProcess");
      _Parent->DbgPtr()->ShowInt(_ClientPollProcess, "_ClientPollProcess");

      _Parent->DbgPtr()->ShowInt(_ClientState, "_ClientState");
      _Parent->DbgPtr()->ShowInt(_PrevServerProcess, "_PrevServerProcess");
      _Parent->DbgPtr()->ShowInt(_ServerPollProcess, "_ServerPollProcess");
    #endif

    if (SetClientState(_ClientState))
    {
      _PollState = Mcalc_KeepAliveState::IDLE;
      #if CALCTEST_DEBUG3
        bool retval = SetPollState(_PollState);
        _Parent->DbgPtr()->ShowInt(retval, "retval");
        
        _Parent->DbgPtr()->LeaveLevel();
        return retval;
      #else
        return SetPollState(_PollState);
      #endif
    }
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(false, "retval");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return false;
}

/****************************************************************************/
bool ServerPollingMgr::PollServerIfAlive_IMPL(int Sleep_, bool ResponseOnly_)
{
  _AsyncMode = _Parent ? _Parent->InAsyncMode():false;

  #if CALCTEST_DEBUG3
    SetStopTracking(false);  
    _Parent->DbgPtr()->EnterLevel("PollServerIfAlive_IMPL");
    _Parent->DbgPtr()->ShowInt(_AsyncMode, "_AsyncMode");
    _Parent->DbgPtr()->ShowMessage(_Parent->GetPollStateFilename());
    _Parent->DbgPtr()->ShowMessage("\n");
  #endif

  if (!_AsyncMode || !_Parent->GetPollStateFilename())
  {
    #if CALCTEST_DEBUG3
      _Parent->DbgPtr()->LeaveLevel();
    #endif

    return false;
  }

  // poll to check if calculator server is alive here
  if (PollServerRequired(Sleep_, ResponseOnly_))
  {
    bool RetryPoll_ = false;
    bool ServerResponse_ = _PollState == Mcalc_PollState::SERVER_RESPONSE;
    #if CALCTEST_DEBUG3
      _Parent->DbgPtr()->ShowInt(ServerResponse_, "ServerResponse_");
    #endif
    
    if (AtInitialState())
    {
      SetNewPollCycle();
      return SendPollServerSignal();
    }
    else if (ServerResponse_ && !ClientAcked())
    {
      if (ServerIsAlive())
      {
        #if CALCTEST_DEBUG3
          if (_Backup && _Backup->_RetryPollCount+1 == _RetryPollCount &&
              _Parent->DbgPtr() && _Parent->DbgPtr()->StopTracking())
          {
            SetStopTracking(false);
            _Parent->DbgPtr()->EnterLevel("PollServerIfAlive_IMPL");
            _Parent->DbgPtr()->ShowInt(ServerResponse_, "ServerResponse_");
            ServerIsAlive();
          }
        #endif
      
        ConfirmServerAlive();
        ConfirmClientAlive(true);
        return SendClientAckServerSignal();
      }
      else
      {
        _KeepAlivePollError = true;
        RetryPoll_ = ReAttemptPolling();
        #if CALCTEST_DEBUG3
          _Parent->DbgPtr()->LeaveLevel();
        #endif

        #if CALCTEST_DEBUG5
          if (_Backup)
            SetStopTracking(true);
        #endif

        if (!RetryPoll_ && _ExitAtRetryMax)
          _Parent->ShowServerPollFailed(true);
        else
          return RetryPoll_;
      }
    }
    else if (ResetRequested() && ClientAcked())
    {
      #if CALCTEST_DEBUG3
        if (_Backup && _Parent->DbgPtr() &&
            _Parent->DbgPtr()->StopTracking())
        {
          SetStopTracking(false);
          _Parent->DbgPtr()->EnterLevel("PollServerIfAlive_IMPL");
          _Parent->DbgPtr()->ShowInt(ServerResponse_, "ServerResponse_");
          ResetRequested();
          ClientAcked();
        }
      #endif

      return ResetPolling();
    }
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  #if CALCTEST_DEBUG5
    if (_Backup)
      SetStopTracking(true);
  #endif

  return false;
}

/****************************************************************************/
bool ServerPollingMgr::ReceiveKeepAlivePoll_IMPL(int Sleep_)
{
  _AsyncMode = _Parent ? _Parent->InAsyncMode():false;

  #if CALCTEST_DEBUG3
    SetStopTracking(false);
    _Parent->DbgPtr()->EnterLevel("ReceiveKeepAlivePoll_IMPL");
    _Parent->DbgPtr()->ShowInt(_AsyncMode, "_AsyncMode");
    _Parent->DbgPtr()->ShowMessage(_Parent->GetPollStateFilename());
    _Parent->DbgPtr()->ShowMessage("\n");
  #endif

  if (!_AsyncMode || !_Parent->GetPollStateFilename())
  {
    #if CALCTEST_DEBUG3
      _Parent->DbgPtr()->LeaveLevel();
    #endif

    return false;
  }

  // receive check alive poll from calculator server here
  if (ClientReceiveRequired(Sleep_))
  {
    bool ServerPolling_ = _ServerState == Mcalc_KeepAliveState::POLL_CLIENTALIVE;
    #if CALCTEST_DEBUG3
      _Parent->DbgPtr()->ShowInt(ServerPolling_, "ServerPolling_");
    #endif
    
    if (ServerAcked())
    {
      #if CALCTEST_DEBUG3
        if (_Backup && _Parent->DbgPtr() &&
            _Parent->DbgPtr()->StopTracking())
        {
          SetStopTracking(false);
          _Parent->DbgPtr()->EnterLevel("ReceiveKeepAlivePoll_IMPL");
          _Parent->DbgPtr()->ShowInt(ServerPolling_, "ServerPolling_");
          ServerAcked();
        }
      #endif

      ConfirmServerAlive(true);
      ConfirmClientAlive();
      return SendResetPollingSignal();
    }
    else if (ServerPolling_)
    {
      if (!ClientIsAlive())
      {
        AcceptPollCycle();
        return SendClientAliveSignal();
      }
      else
      {
        #if CALCTEST_DEBUG5
          if (_Backup)
            SetStopTracking(true);
        #endif

        #if CALCTEST_DEBUG3
          _Parent->DbgPtr()->LeaveLevel();
        #endif
        return true;
      }
    }
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  #if CALCTEST_DEBUG5
    if (_Backup)
      SetStopTracking(true);
  #endif
  
  return false;
}

/****************************************************************************/
bool ServerPollingMgr::PollServerIfAlive(int Sleep_,
                                         bool UntilEnd_, bool ResponseOnly_)
{
  if (_Backup)
  {
    SetStopTracking(!UntilEnd_ || ResponseOnly_);
    *_Backup = *this;
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("PollServerIfAlive");
  #endif

  bool PollOk_ = true;
  long LoopGuard_ = 0;
  long Max = LOOP_GUARD_LIMIT;  // RETRY_READ_MAX * 2;
  
  int mult_ = 0;
  int PrevAttempt_;

  bool PollingServer_ = false;
  bool PollAnswered_ = false;
  bool PollError_ = false;
  bool PollRetVal_ = false;
  bool InProcess_ = PollServerInProcess();
  #if CALCTEST_DEBUG5
    bool ResetOnNotReq_ = _ResetWhenNotReq;
  #endif

  #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  #else
    // microsec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MICRO_DELAY_MULT;
  #endif

  ResetServerPolling(UntilEnd_ || !InProcess_);
  
  for (LoopGuard_ = UntilEnd_ ? Max:1;
       LoopGuard_ && PollOk_ && PollServerRequired(0, ResponseOnly_);
       LoopGuard_--)
  {
    PrevAttempt_ = _RetryPollCount;
    PollOk_ = PollServerIfAlive_IMPL(Sleep_, ResponseOnly_);
    GivePollResults(PollingServer_, PollError_, PollAnswered_);
    ResponseOnly_ = true;
    
    if (PollOk_ && PollError_)
    {
      if (_BreakOnPollError)
      {
        SetStopTracking(true);
        break;
      }
      else if (Sleep_)
      {
        mult_ = (Sleep_ > 1) ? Sleep_:1;

        #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
          Sleep(mult_ * DelayLen_);
        #else
          ::usleep(mult_ * DelayLen_);
        #endif
      }
      else if (_RetryPollCount != PrevAttempt_ && _KeepAlivePollError)
      {
        Sleep_ = 1;
        SetStopTracking(true);
        
        if (_Backup)
        {
          *_Backup = *this;
          _Backup->_RetryPollCount = PrevAttempt_;
        }
      }
    }
    else
    {
      #if CALCTEST_DEBUG5
        SetTrackingOnStateChange(ResetOnNotReq_);
      #endif

      if (PollServerCompleted(false))
        break;
    }
  }

  PollRetVal_ = PollOk_ && PollServerCompleted(true);

  if (_ResetStatesOnDone || _KeepAlivePollError)
  {
    if (PollServerNotRequired(false))
    {
      SetStopTracking(true);
      ResetServerPolling(UntilEnd_);
    }
    else if (PollRetVal_)
    {
      SetStopTracking(true);
      ResetServerPolling(false);
    }
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(PollRetVal_, "PollRetVal_");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return PollRetVal_;
}

/****************************************************************************/
bool ServerPollingMgr::ReceiveKeepAlivePoll(int Sleep_, bool UntilEnd_)
{
  if (_Backup)
  {
    SetStopTracking(!UntilEnd_);
    *_Backup = *this;
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->EnterLevel("ReceiveKeepAlivePoll");
  #endif

  bool PollOk_ = true;
  bool PollRetVal_ = false;
  bool InProcess_ = ClientReceiveInProcess();
  #if CALCTEST_DEBUG5
    bool ResetOnNotReq_ = _ResetWhenNotReq;
  #endif
  
  long LoopGuard_ = 0;
  long Max = LOOP_GUARD_LIMIT;  // RETRY_READ_MAX * 2;

  ResetReceivePoll(UntilEnd_ || !InProcess_);

  for (LoopGuard_ = UntilEnd_ ? Max:1;
       LoopGuard_ && PollOk_ && ClientReceiveRequired();
       LoopGuard_--)
  {
    PollOk_ = ReceiveKeepAlivePoll_IMPL(Sleep_);
    #if CALCTEST_DEBUG5
      SetTrackingOnStateChange(ResetOnNotReq_);
    #endif

    if (ClientReceiveCompleted(false))
      break;
  }

  PollRetVal_ = PollOk_ && ClientReceiveCompleted(true);

  if (_ResetStatesOnDone || _KeepAlivePollError)
  {
    if (ClientReceiveNotRequired(false))
    {
      SetStopTracking(true);
      ResetReceivePoll(UntilEnd_);
    }
    else if (PollRetVal_)
    {
      SetStopTracking(true);
      ResetReceivePoll(false);
    }
  }

  #if CALCTEST_DEBUG3
    _Parent->DbgPtr()->ShowInt(PollRetVal_, "PollRetVal_");
    _Parent->DbgPtr()->LeaveLevel();
  #endif

  return PollRetVal_;
}

/****************************************************************************/
void ServerPollingMgr::ResetRetryRead()
{
  _RetryReadCount = 0;
}

/****************************************************************************/
bool ServerPollingMgr::AtMaxReadAttempts() const
{
  return (_RetryReadCount >= RETRY_READ_MAX);
}

/****************************************************************************/
bool ServerPollingMgr::ReAttemptFileRead()
{
  if (_RetryReadCount < RETRY_READ_MAX)
    _RetryReadCount++;
  else if (_RetryReadCount == RETRY_READ_MAX)
    return false;

  return true;
}

/****************************************************************************/
void ServerPollingMgr::ResetPollCount()
{
  _RetryPollCount = 0;
}

/****************************************************************************/
bool ServerPollingMgr::AtMaxPollAttempts() const
{
  return (_RetryPollCount >= RETRY_READ_MAX);
}

/****************************************************************************/
bool ServerPollingMgr::ReAttemptPolling()
{
  if (_KeepAlivePollError && _RetryPollCount < RETRY_READ_MAX)
  {
    _RetryPollCount++;
    return (_RetryPollCount < RETRY_READ_MAX);
  }
  else if (_RetryPollCount >= RETRY_READ_MAX)
    return false;

  return true;
}

/****************************************************************************/
bool ServerPollingMgr::ServerPollingAllowed() const
{
  return _AllowPollServer;
}

/****************************************************************************/
bool ServerPollingMgr::ConfirmedServerAlive() const
{
  return _ConfirmedServerAlive;
}

/****************************************************************************/
bool ServerPollingMgr::ConfirmedClientAlive() const
{
  return _ConfirmedClientAlive;
}

/****************************************************************************/
bool ServerPollingMgr::ServerAliveConfirmedSent() const
{
  return _ServerAliveConfSent;
}

/****************************************************************************/
bool ServerPollingMgr::ExitAtRetryMax() const
{
  return _ExitAtRetryMax;
}

/****************************************************************************/
bool ServerPollingMgr::SleepWaiting() const
{
  return _SleepWaiting;
}

/****************************************************************************/
bool ServerPollingMgr::WaitUntilResponse() const
{
  return _WaitUntilResponse;
}

/****************************************************************************/
// ExpectedSignalsHandler class definitions
/****************************************************************************/
ExpectedSignalsHandler::ExpectedSignalsHandler(CalcDriver* Parent_):
_Parent(Parent_),
_FromServer(0),
_ClientExpected(0),
_PendingResponse(0),

_FromClient(0),
_Matched(false)
{}

/****************************************************************************/
bool ExpectedSignalsHandler::IsMatching() const
{
  _Matched = (_FromClient & _ClientExpected) != 0;
  return _Matched;
}

/****************************************************************************/
bool ExpectedSignalsHandler::ResponseReceived(int ResponseSignal_)
{
  SetFromClient(ResponseSignal_);
  
  return (IsMatching() ||
          (_FromClient == Mcalc_IOState::CALC_HALT ||
           _FromClient == Mcalc_IOState::BREAK_PROGRAM) ||
          ((_ClientExpected & Mcalc_IOState::PROCESS) &&
           (ResponseSignal_ == Mcalc_IOState::BREAK_PROGRAM ||
            ResponseSignal_ == Mcalc_IOState::CALC_HALT)));
}

/****************************************************************************/
bool ExpectedSignalsHandler::ExpectedStatesSet() const
{
  return (_FromServer &&
          _FromServer != Mcalc_IOState::IDLE_STATE &&
          _ClientExpected);
}

/****************************************************************************/
bool ExpectedSignalsHandler::Cleared() const
{
  return (_FromServer == Mcalc_IOState::IDLE_STATE &&
          !_ClientExpected);
}

/****************************************************************************/
bool ExpectedSignalsHandler::IsProcessNext() const
{
  return ((_FromServer & Mcalc_IOState::CALC_RESPONSE) ||
          (_FromServer & Mcalc_IOState::CALC_ERROR) ||
          (_FromServer & Mcalc_IOState::BREAK_PROGRAM) ||
          
          (_FromClient & Mcalc_IOState::INPUT_RECEIVED) ||
          (_FromClient & Mcalc_IOState::OUTPUT_FETCHED) ||
          (_FromClient & Mcalc_IOState::BATCHFILE_ENDED_ACK) ||
          (_FromClient & Mcalc_IOState::BREAK_PROGRAM) ||
          (_FromClient & Mcalc_IOState::PROCESS_DONE_ACK));
}

/****************************************************************************/
void ExpectedSignalsHandler::Reset(int Signals_)
{
  if (Signals_ & FROM_SERVER)
    _FromServer = Mcalc_IOState::IDLE_STATE;

  if (Signals_ & CLIENT_EXPECTED)
    _ClientExpected = 0;

  if (Signals_ & FROM_CLIENT)
    _FromClient = 0;

  if (Signals_ & PENDING_RESPONSE)
    _PendingResponse = 0;
}

/****************************************************************************/
void ExpectedSignalsHandler::ClearSignalsIfMatched()
{
  if (_Matched)
    Reset();
}

/****************************************************************************/
bool ExpectedSignalsHandler::MatchesClientExpected(int Signal_, bool InBatch_)
{
  return ((_ClientExpected & Signal_) != 0 &&
          (!InBatch_ || _Parent->InBatchFile()));
}

/****************************************************************************/
bool ExpectedSignalsHandler::MatchesFromServer(int Signal_, bool InBatch_)
{
  return ((_FromServer & Signal_) != 0 &&
          (!InBatch_ || _Parent->InBatchFile()));
}

/****************************************************************************/
// CalcDriver class definitions
/****************************************************************************/
CalcDriver::CalcDriver():
_PollServerMgr(NULL),
_ServerProgramAlive(true),
_IoStateFout(NULL),
_SessionFout(NULL),
_ElapsedStateFout(NULL),

_ServerAliveFile(NULL),
_ClientAliveFile(NULL),
_PollStateFile(NULL),
_WaitCycle(1),
_ServerAliveMode(0),
_ClientAliveMode(0),
_PollStateMode(0),
_CalcPollingInactive(0),
_SuspendPolling(0),
_ResumePolling(0),

filestatus_ServerAliveState(0),
filestatus_ClientAliveState(0),
filestatus_PollState(0),
_OpenServerAliveFileRetries(0),
_OpenClientAliveFileRetries(0),
_OpenPollStateFileRetries(0),

_AttemptFile(0),
_AttemptMode(NULL),
_AttemptFileName(NULL),
_FromMethod(0),
_SavedState(0),
_SavedElapsedState(0),
_SavedFilePos(0),
_SavedLen(0),
_OutputReady(0),
_ProgramExited(0),
_ResponseOnly(true),
_Halted(false),
_InitProgressBar(false),
_EndProgressBar(false),
_ProgressBarInitDone(false),
_ProgressBarEndDone(false),
_InputDataFnameDone(false),
_OutputDataFnameDone(false),
_MatchingStates(false),
_RetryReadIoDone(false),
_ExpectedSigSent(false),
_EnterKeyActive(true),

_Numer(0),
_NumerSet(false),
_Denom(0),
_DenomSet(false),
_InputDataFname(NULL),
_InputFnameSet(false),
_OutputDataFname(NULL),
_OutputFnameSet(false),
_ReadBuffer(NULL),
_ReadOk(false),
_IoStateRead(false),
_IostateFileReadError(false),
_ProgressOutDone(false),
_CalcOutDone(false),
_InputFileDoneLvl(-1),
_FetchError(false),
_ServerCalledHalt(false),

_GrOutputDataFname(NULL),
_GraphProgressOutDone(false),
_GraphOperationOutDone(false),
_GrOutputDataFnameDone(false),
_GrOutputFnameSet(false),
_GrNumer(0),
_GrNumerSet(false),
_GrDenom(0),
_GrDenomSet(false),
_GraphPlotOpType(0),
_GraphType(0),
_GraphSubtype(0),
_GraphOpTypeSet(false),
_GrProgressFetched(false),
_GrOperationFetched(false),
_GrProgressDataShown(true),
_GrOperationDataShown(true),
_GrReadBuffer(NULL),
_GrReadOk(false),

_OutputBufferSize(256),
_OutputBuffer(new char[256]),
_DoQuit(0),
_QuitAll(0),
_BreakAll(0),
_SessionNum(0),
_Processing(0),
_IoState(0),
_ElapsedState(0),
_PrevIoState(0),
_NoWaitReadCnt(0),
_BatchExecLevel(0),
_NoPolling(true),
_InBatchFile(false),
_EmptyBuffer(false),
_ResendLastSig(false),
_SendResendLastSig(false),
_RequestResend(false),
_ResendSigDone(false),
_ResendSigUnchanged(false),
_AcceptProcessAckDone(false),
_SendingEndBatchFileAck(false),

_ExpectedSignals(NULL),
_ExecuteProgram(NULL),

_InputRequested(true),
_FatalErrCount(0),
_Executed(0),
_Spawned(0),

LOG_FILE(NULL),
SPAWNARG_FILE(NULL),
CURRENTERROR_FILE(NULL),
USER_INPUT_FILE(NULL),
USER_PROMPT_FILE(NULL),
PROG_OUTPUT_FILE(NULL),
PROG_IOSTATE_FILE(NULL),
SERVERALIVE_FILE(NULL),
CLIENTALIVE_FILE(NULL),
POLLSTATE_FILE(NULL),
PROGRESS_FILE(NULL),
GRAPH_PROGRESS_FILE(NULL),
GRAPH_OPERATION_FILE(NULL),
ELAPSED_TIME_FILE(NULL)
{
  _PollServerMgr = new ServerPollingMgr(this, true);
  if (_PollServerMgr)
    _PollServerMgr->SetExitAtRetryMax(true)
                  ->SetSleepWaiting(true)
                  ->SetAllowServerPolling(true)
                  ->SetBreakOnPollError(false)
                  ->SetResetStatesOnDone(true);

  #if CALCDRIVER_GUI_VERSION
    _Output = new QTextEdit();
  #else
    _Output = new char[256];
    _Input = new char[256];
  #endif
    
  #if CALCDRIVER_TEXT_VERSION
    SetupDataFiles();
    ReadIoState();
  #endif

  connect(&_ReadIoStateTimer, SIGNAL(timeout()),
          this, SLOT(RetryReadIoState()));
  connect(&_FileOpenRetryTimer, SIGNAL(timeout()),
          this, SLOT(RetryFileOpen()));
  connect(&_TerminationTimer, SIGNAL(timeout()),
          this, SLOT(ResetSession()));
  
#if CALCDRIVER_TEXT_VERSION
  connect(this, SIGNAL(DoQuit()),
          this, SLOT(DoHalt()));
  connect(this, SIGNAL(InputNeeded()),
          this, SLOT(SendToInput()));
  connect(this, SIGNAL(OutputReady()),
          this, SLOT(ShowOutput()));
#endif

#if CALCDRIVER_DEBUGGING
  _dbgptr = new Debugging<CalcDriver>(this);
  
  #if CALCTEST_DEBUG3
    _dbgptr->SetFilename("clientsig-dbg.txt");
  #else
    _dbgptr->SetFilename("calcdriver-dbg.txt");
  #endif
  _dbgptr->SetBasePtr(this);
  _dbgptr->SetStopTracking(SERVERPOLLINGMGR, true)
         ->SetTrackType(SERVERPOLLINGMGR);

  Debugging<CalcDriver>::SetCurrentDebugger(_dbgptr);
  #if CALCTEST_DEBUG4
    _dbgptr->SetStopTracking(SERVERPOLLINGMGR, false);
  #endif
  #if ((CALCTEST_DEBUG6a)|(CALCTEST_DEBUG6c)|\
       (CALCTEST_DEBUG1c)|(CALCTEST_DEBUG1d)|(CALCTEST_DEBUG8))
    _dbgptr->StartTrackingAll();
  #endif
  
  #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
    if (DbgPtr(false))
      DbgPtr()->ShowMessage("Debugging Ptr Setup Done:\n\n");
  #endif
#endif

  _NoPolling = _PollServerMgr == NULL;
  _ExpectedSignals = new ExpectedSignalsHandler(this);
  _ExpectedSignals->Reset();

  #if IMPLEMENT_KEEP_ALIVE
    ResumeCalcPolling();
  #else
    PauseCalcPolling();
  #endif
}

/****************************************************************************/
CalcDriver::~CalcDriver()
{
  delete[] _OutputBuffer;
  #if CALCDRIVER_GUI_VERSION
    delete _Output;
  #else
    delete[] _Output;
    delete[] _Input;
  #endif

    delete[] SPAWNARG_FILE;
    delete[] LOG_FILE;
    delete[] CURRENTERROR_FILE;
    delete[] USER_INPUT_FILE;
    delete[] USER_PROMPT_FILE;
    delete[] PROG_OUTPUT_FILE;
    delete[] PROG_IOSTATE_FILE;
    
    delete[] SERVERALIVE_FILE;
    delete[] CLIENTALIVE_FILE;
    delete[] POLLSTATE_FILE;
    
    delete[] GRAPH_PROGRESS_FILE;
    delete[] GRAPH_OPERATION_FILE;
    delete[] PROGRESS_FILE;
    delete[] ELAPSED_TIME_FILE;

    SPAWNARG_FILE =
    LOG_FILE =
    CURRENTERROR_FILE =
    USER_INPUT_FILE =
    USER_PROMPT_FILE =
    PROG_OUTPUT_FILE =
    PROG_IOSTATE_FILE =
    
    SERVERALIVE_FILE =
    CLIENTALIVE_FILE =
    POLLSTATE_FILE =
    
    GRAPH_PROGRESS_FILE =
    GRAPH_OPERATION_FILE =
    PROGRESS_FILE =
    ELAPSED_TIME_FILE = NULL;

    delete _ExecuteProgram;
    _ExecuteProgram = NULL;

    delete _ExpectedSignals;
    _ExpectedSignals = NULL;

    #if CALCDRIVER_DEBUGGING
      Debugging<CalcDriver>::SetCurrentDebugger(NULL);
      delete _dbgptr;
      _dbgptr = NULL;
    #endif
}

/****************************************************************************/
const char* CalcDriver::IoStateToStr(int State_)
{
  return
  (
    (State_ == Mcalc_IOState::PROCESS)        ? "PROCESS":
    (State_ == Mcalc_IOState::OUTPUT_READY)   ? "OUTPUT_READY":
    (State_ == Mcalc_IOState::CALC_HALT)      ? "CALC_HALT":
    (State_ == Mcalc_IOState::CALC_RESPONSE)  ? "CALC_RESPONSE":
    (State_ == Mcalc_IOState::INPUT_REQUIRED) ? "INPUT_REQUIRED":
    (State_ == Mcalc_IOState::OUTPUT_FETCHED) ? "OUTPUT_FETCHED":
    (State_ == Mcalc_IOState::INPUT_RECEIVED) ? "INPUT_RECEIVED":
    (State_ == Mcalc_IOState::BREAK_PROGRAM)  ? "BREAK_PROGRAM":
    (State_ == Mcalc_IOState::IDLE_STATE)     ? "IDLE_STATE":
    (State_ == Mcalc_IOState::GRAPH_OUTPUT)   ? "GRAPH_OUTPUT":
    (State_ == Mcalc_IOState::GRAPH_WAIT)     ? "GRAPH_WAIT":
    (State_ == Mcalc_IOState::CALC_ERROR)     ? "CALC_ERROR":
    (State_ == Mcalc_IOState::PROGRESS_READY) ? "PROGRESS_READY":
    (State_ == Mcalc_IOState::ERROR_FETCH)    ? "ERROR_FETCH":
    (State_ == Mcalc_IOState::GRAPH_PROGRESS) ? "GRAPH_PROGRESS":

    (State_ == Mcalc_IOState::INFILE_PROGRESS_ACK) ? "INFILE_PROGRESS_ACK":
    (State_ == Mcalc_IOState::GRAPH_PROGRESS_ACK)  ? "GRAPH_PROGRESS_ACK":
    (State_ == Mcalc_IOState::GRAPH_WAIT_ACK)      ? "GRAPH_WAIT_ACK":

    (State_ == Mcalc_IOState::SPAWN_NEW_MCALC) ? "SPAWN_NEW_MCALC":
    (State_ == Mcalc_IOState::MCALC_SPAWNED)   ? "MCALC_SPAWNED":
    
    (State_ == Mcalc_IOState::BATCHFILE_ENDED)     ? "BATCHFILE_ENDED":
    (State_ == Mcalc_IOState::BATCHFILE_ENDED_ACK) ? "BATCHFILE_ENDED_ACK":
    (State_ == Mcalc_IOState::PROCESS_DONE)        ? "PROCESS_DONE":
    (State_ == Mcalc_IOState::PROCESS_DONE_ACK)    ? "PROCESS_DONE_ACK":

    (State_ == Mcalc_IOState::RESENDLASTSIG)       ? "RESENDLASTSIG":
    (State_ == Mcalc_IOState::CLIENTPING)          ? "CLIENTPING":
    (State_ == Mcalc_IOState::CLIENTPING_ACK)      ? "CLIENTPING_ACK":"0"
  );
}

/****************************************************************************/
const char* CalcDriver::TimeoutStateToStr(int State_)
{
  return
  (
    (State_ == Mcalc_ElapsedTime::RESET)        ? "RESET":
    (State_ == Mcalc_ElapsedTime::SENT)         ? "SENT":
    (State_ == Mcalc_ElapsedTime::ACKNOWLEDGED) ? "ACKNOWLEDGED":"0"
  );
}

/****************************************************************************/
#if CALCDRIVER_DEBUGGING
void CalcDriver::SetErrorData(int errcode_, const char* file_,
                              size_t line_, size_t col_, bool terminate_)
{
  _errline = line_;
  _errcol = col_;
  _errcode = errcode_;
  char* buffer_ = NULL;
  char* fmtstr_;

  if (file_)
  {
    buffer_ = ::strcpy(new char[strlen(file_) + 100], file_);
    strcat(buffer_, ": ");
    strcat(buffer_, ERRMSG_ERROR_IN_LINE);
  }
  else
  {
    buffer_ = new char[100];
    strcpy(buffer_, ERRMSG_ERROR_IN_LINE);
  }

  FILE* fp = fopen("calcdriver_errorout.txt", "w");

  fmtstr_ = strdup(buffer_);
  sprintf(buffer_, fmtstr_, _errline, _errcol, _errcode);
  fputs(buffer_, fp);
  fflush(fp);
  fclose(fp);
  free(fmtstr_);
  delete[] buffer_;

  if (terminate_)
    exit(_errcode);
}
#endif

/****************************************************************************/
#if CALCDRIVER_DEBUGGING
void CalcDriver::Terminate(bool setcode_, int errcode_)
{
  if (setcode_)
    _errcode = errcode_;

  exit(_errcode);
}
#endif

/****************************************************************************/
#if CALCDRIVER_DEBUGGING
void CalcDriver::SetCurrentDebugger(Debugging<CalcDriver>* ptr)
{
  Debugging<CalcDriver>::SetCurrentDebugger(ptr);
}
#endif

/****************************************************************************/
#if CALCDRIVER_DEBUGGING
Debugging<CalcDriver>* CalcDriver::DbgPtr(bool Deref_)
{
  Debugging<CalcDriver>* ptr =
    Debugging<CalcDriver>::CurrentDebugger(Deref_);

  if (Deref_ && !ptr)
    SetErrorData(1);

  return ptr;
}
#endif

/****************************************************************************/
char* CalcDriver::NewString(const char* src)
{
  char* ns = NULL;
  return ((src && (ns = new char[strlen(src)+1])) ? strcpy(ns, src):NULL);
}

/****************************************************************************/
void CalcDriver::SetupDataFiles()
{
  GetFilenames();
  InitializeFiles();
}

/****************************************************************************/
void CalcDriver::ResizeOutputBuffer(size_t newsize_)
{
  size_t x;
  size_t OldSize_ = _OutputBufferSize;
  char* OldBuffer_ = _OutputBuffer;

  if (newsize_)
  {
    if (newsize_ < OldSize_)
      return;
    
    if (newsize_ > OldSize_ + 64)
      _OutputBufferSize = newsize_ + 5;
    else
      _OutputBufferSize += 128;
  }   
  else
    _OutputBufferSize += 128;
  
  _OutputBuffer = new char[_OutputBufferSize];

  for (x = 0; x < OldSize_; x++)
    _OutputBuffer[x] = OldBuffer_[x];

  for (;x < _OutputBufferSize; x++)
    _OutputBuffer[x] = 0;

  delete[] OldBuffer_;
  OldBuffer_ = NULL;
}

/****************************************************************************/
long CalcDriver::FileSize(FILE* Fptr_)
{
  if (Fptr_)
  {
    fseek(Fptr_, 0, SEEK_END);
    long Pos_ = ftell(Fptr_);
    rewind(Fptr_);
    return Pos_;    
  }

  return 0;
}

/****************************************************************************/
void CalcDriver::GetSessionNumbers(FILE* Fptr_, int** SesNumArray_, int Max_, int& NumSes_)
{
  char* buffer = new char[128];
  char* token1;
  int x = 0;

  int OldMax_;
  int* OldArray_;  
  int* Array_ = *SesNumArray_;
  char* RetStr_ = NULL;

  while (!feof(Fptr_))
  {
    RetStr_ = fgets(buffer, 128, Fptr_);
    if (!RetStr_)
      break;

    token1 = strtok(RetStr_, ",");
    Array_[x++] = atoi(token1);

    if (x == Max_)
    {
      OldMax_ = Max_;
      Max_ += 128;
      OldArray_ = Array_;
      Array_ = new int[Max_];

      for (x = 0; x < OldMax_; x++)
        Array_[x] = OldArray_[x];

      delete[] OldArray_;
    }
  }

  Array_[x] = 0;
  NumSes_ = x;

  delete[] buffer;
  *SesNumArray_ = Array_;
}

/****************************************************************************/
int CalcDriver::FindUnusedEntry(FILE** SesFptr_)
{
  FILE* Fptr_ = *SesFptr_;
  char* buffer = new char[128];
  char* token1;
  char* token2;
  long Fpos_;
  int StateVal_;
  int Found_ = 0;
  size_t Len_;  
  char* RetStr_ = NULL;
  size_t x;

  #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  #else
    // microsec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MICRO_DELAY_MULT;
  #endif

  while (!feof(Fptr_))
  {
    Fpos_ = ftell(Fptr_);
    RetStr_ = fgets(buffer, 128, Fptr_);
    if (!RetStr_)
      break;
    
    Len_ = strlen(RetStr_);
    token1 = strtok(RetStr_, ",");
    token2 = strtok(NULL, ",");
    for (x = strlen(token2); x && isspace(token2[x - 1]); x--);
    token2[x] = 0;    
    
    StateVal_ = atoi(token2);
    Found_ = StateVal_ == Mcalc_SessionStates::UnUsed;
    
    if (Found_)
    {
      _SessionNum = atoi(token1);
      if (Fptr_)
        fclose(Fptr_);
      
      Fptr_ = fopen(MCALC_SESSION_FILE, "r+");
      while (!Fptr_)
      {
        #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
          Sleep(DelayLen_);
        #else
          ::usleep(DelayLen_);
        #endif
        Fptr_ = fopen(MCALC_SESSION_FILE, "r+");        
      }
      
      fseek(Fptr_, Fpos_, SEEK_SET);
      strcpy(buffer, token1);
      strcat(buffer, ",");
      sprintf(&buffer[strlen(buffer)], "%d", Mcalc_SessionStates::InUse);
      --Len_;
      while (strlen(buffer) != Len_)
        strcat(buffer, " ");
      strcat(buffer, "\n");
      fputs(buffer, Fptr_);
      break;
    }
  }

  *SesFptr_ = Fptr_;
  delete[] buffer;
  return Found_;
}

/****************************************************************************/
int CalcDriver::HasSessionNumber(int* Array_, int NumSesNumbers_, int& Number_)
{
  int TestNumber_;
  int HasTestNum_;
  int HasArgNum_;
  
  if (!NumSesNumbers_)
    return FALSE;

  HasArgNum_ = HasTestNum_ = 0;
  TestNumber_ = Array_[NumSesNumbers_ - 1] + 1;
  int x;

  for (x = 0; x < NumSesNumbers_; x++)
  {
    if (TestNumber_ == Array_[x])
      HasTestNum_ = 1;

    if (Number_ == Array_[x])
      HasArgNum_ = 1;
  }

  if (HasArgNum_)
  {
    if (!HasTestNum_)
    {
      Number_ = TestNumber_;
      return FALSE;
    }
  }
  else
    return FALSE;

  return TRUE;
}

/****************************************************************************/
void CalcDriver::GetFilenames()
{
        int* SesNumArray_ = new int[128];
        char* buffer = new char[128];
        char* RetStr_ = NULL;
        int SesState_;
        long Pos_;
        QString TempStr_;
        FILE* SesFptr_ = fopen(MCALC_SESSION_FILE, "r");
        int Found_ = 0;
        int NumSesNumbers_ = 0;

        if (FileSize(SesFptr_))
          Found_ = FindUnusedEntry(&SesFptr_);

        if (!Found_)
        {
          if (SesFptr_)
          {
            rewind(SesFptr_);
            GetSessionNumbers(SesFptr_, &SesNumArray_, 128, NumSesNumbers_);
            fclose(SesFptr_);
          }
          
          SesFptr_ = fopen(MCALC_SESSION_COUNT_FILE, "r");
          if (FileSize(SesFptr_))
          {          
            RetStr_ = fgets(buffer, 128, SesFptr_);
            _SessionNum = atoi(RetStr_);
          }
          else
            _SessionNum = 0;
          
          _SessionNum += 1;
          while (HasSessionNumber(SesNumArray_, NumSesNumbers_, _SessionNum))
            _SessionNum += 1;

          if (SesFptr_)
            fclose(SesFptr_);
            
          SesFptr_ = fopen(MCALC_SESSION_COUNT_FILE, "w");
          sprintf(buffer, "%d", _SessionNum);
          TempStr_ = buffer;
          fputs(TempStr_.ascii(), SesFptr_);

          if (SesFptr_)
            fclose(SesFptr_);
            
          SesState_ = Mcalc_SessionStates::InUse;
          TempStr_ = buffer;
          sprintf(buffer, "%d", SesState_);          
          TempStr_ += ",";
          TempStr_ += buffer;
          TempStr_ += "\n";
          SesFptr_ = fopen(MCALC_SESSION_FILE, "a");
          fputs(TempStr_.ascii(), SesFptr_);
        }
        
        if (SesFptr_)
        {
          sprintf(buffer, "%d", _SessionNum);

          TempStr_ = MCALC_LOG_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          LOG_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_SPAWNARG_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          SPAWNARG_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_CURRENT_ERROR_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          CURRENTERROR_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_USER_INPUT_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          USER_INPUT_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_USER_PROMPT_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          USER_PROMPT_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_PROG_OUTPUT_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          PROG_OUTPUT_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_IOSTATE_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          PROG_IOSTATE_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_SERVERALIVE_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          SERVERALIVE_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_CLIENTALIVE_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          CLIENTALIVE_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_POLLSTATE_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          POLLSTATE_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_PROGRESS_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          PROGRESS_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_GRAPH_PROGRESS_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          GRAPH_PROGRESS_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_GRAPH_OPERATION_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          GRAPH_OPERATION_FILE = NewString(TempStr_.ascii());

          TempStr_ = MCALC_ELAPSEDTIME_FILE;
          Pos_ = TempStr_.find('.');
          TempStr_.insert(Pos_, buffer);
          ELAPSED_TIME_FILE = NewString(TempStr_.ascii());

          if (SesFptr_)
            fclose(SesFptr_);
        }

        delete[] buffer;
        delete[] SesNumArray_;
}

/****************************************************************************/
void CalcDriver::InitializeFiles()
{
  FILE* Fout_;

  Fout_ = fopen(USER_INPUT_FILE, "w");
  fprintf(Fout_, " ");
  fclose(Fout_);

  Fout_ = fopen(PROG_IOSTATE_FILE, "w");
  fprintf(Fout_, "%d", Mcalc_IOState::IDLE_STATE);  
  fclose(Fout_);
  _IoState = Mcalc_IOState::IDLE_STATE;

  Fout_ = fopen(ELAPSED_TIME_FILE, "w");
  fprintf(Fout_, "%d", Mcalc_ElapsedTime::RESET);
  fclose(Fout_);
  _ElapsedState = Mcalc_ElapsedTime::RESET;

  Fout_ = fopen(SERVERALIVE_FILE, "w");
  fprintf(Fout_, "%d\n", Mcalc_KeepAliveState::IDLE);
  // fprintf(Fout_, "%d\n", 0);
  fclose(Fout_);

  Fout_ = fopen(CLIENTALIVE_FILE, "w");
  fprintf(Fout_, "%d\n", Mcalc_KeepAliveState::IDLE);
  // fprintf(Fout_, "%d\n", 0);
  fclose(Fout_);

  Fout_ = fopen(POLLSTATE_FILE, "w");
  fprintf(Fout_, "%d", Mcalc_PollState::IDLE);
  fclose(Fout_);

  Fout_ = fopen(SPAWNARG_FILE, "w");
  fclose(Fout_);  
}

/*****************************************************************************
Procedure to compare two strings for equality. If both strings are NULL
then procedure returns 1, otherwise the strings are compared character
by character up to max characters same as strncmp.
PARAMETERS:
  char* s1 : the first string
  char* s2 : the second string
  int max : the maximum number of characters to compare in each string
  bool SkipWs_ : Skip leading whitespaces for first string argument s1
RETURNED:
  returns 1 if both string matches in characters this also include the
  the case in which both strings are zero length with an initial NULL char
  ,otherwise returns 0 if the strings have different characters.
*/
int CalcDriver::icasestrcompn(const char* s1, const char* s2, size_t max, bool SkipWs_)
{
  size_t i;
  
  if (SkipWs_ && s1)
    while (*s1)
      if (isspace(*s1))
        s1++;
      else
        break;

  for (i = 0; i < max; i++)
  {
    if (!(*s1) && !(*s2))
      return 1;
    else if (!(*s1) || !(*s2))
      return 0;

    if (isalpha(*s1) && isalpha(*s2))
    {
      if (tolower(*s1) != tolower(*s2))
        return 0;
    }
    else if ((*s1) != (*s2))
      return 0;

    s1++;
    s2++;
  }

  return 1;
}

/*****************************************************************************
Procedure to compare two strings for equality. If both strings are NULL
then procedure returns 1
PARAMETERS:
  char* s1 : the first string
  char* s2 : the second string
  bool SkipWs_ : Skip leading whitespaces for first string argument s1
RETURNED:
  returns 1 if both string matches in characters this also include the
  the case in which both strings are zero length with an initial NULL char
  ,otherwise returns 0 if the strings have different characters.
*/
int CalcDriver::icasestrcomp(const char* s1, const char* s2, bool SkipWs_)
{
  size_t i;
  size_t max = s1 ? ::strlen(s1):0;

  if (SkipWs_ && s1)
    while (*s1)
      if (isspace(*s1))
        s1++;
      else
        break;
  
  if (s2 && ::strlen(s2) > max)
    max = ::strlen(s2);

  for (i = 0; i < max; i++)
  {
    if (!(*s1) && !(*s2))
      return 1;
    else if (!(*s1) || !(*s2))
      return 0;

    if (isalpha(*s1) && isalpha(*s2))
    {
      if (tolower(*s1) != tolower(*s2))
        return 0;
    }
    else if ((*s1) != (*s2))
      return 0;

    s1++;
    s2++;
  }

  return 1;
}

/****************************************************************************/
bool CalcDriver::ShouldEnableEnter()
{
  bool Active_ = _ReadIoStateTimer.isActive();
  return (_EnterKeyActive && !Active_);
}

/****************************************************************************/
bool CalcDriver::IoStateTimer(int Action_, int Period_)
{
  bool SignalSent_ = false;
  bool WasActive_ = false;

  if (Action_ == STOP)
  {
    WasActive_ = _ReadIoStateTimer.isActive();
    _ReadIoStateTimer.stop();

    if (WasActive_ && !_EnterKeyActive)
    {
      _EnterKeyActive = true;
      emit EnableEnterKey();

      SignalSent_ = true;
    }
  }
  else if (Action_ == START)
  {
    if (_EnterKeyActive)
    {
      _EnterKeyActive = false;
      emit DisableEnterKey();

      SignalSent_ = true;
    }

    // Period in milliseconds
    _ReadIoStateTimer.start(Period_);
  }

  return SignalSent_;
}

/****************************************************************************/
void CalcDriver::StopAllTimers()
{
  IoStateTimer(STOP);
  _FileOpenRetryTimer.stop();
  _TerminationTimer.stop();

  _RequestResend = false;
  _ResendSigDone = false;
  _ResendLastSig = false;
  _ResendSigUnchanged = false;
}

/****************************************************************************/
#if CALCDRIVER_GUI_VERSION
void CalcDriver::SetOutput(QTextEdit* Output_)
{
  _Output = Output_;
}
#else
void CalcDriver::SetOutput(const char* Output_)
{
  strcpy(_Output, Output_);
}
#endif

/****************************************************************************/
void CalcDriver::SendToInput()
{
  #if CALCDRIVER_TEXT_VERSION
    #if CALCTEST_DEBUG3
      if (DbgPtr(false))
        DbgPtr()->EnterLevel("CalcDriver::SendToInput");
    #endif

    _IoStateRead =
    _ProgressOutDone =
    _CalcOutDone = false;
    
    int PrevState_ = _IoState;
    bool ProgressShown_ = false;
    bool GraphDataShown_ = false;
    bool CalcOutReady_ = false;
          
    if (_IoState == Mcalc_IOState::OUTPUT_READY || _IoState == Mcalc_IOState::ERROR_FETCH)
      ShowOutput();
    else if (_IoState == Mcalc_IOState::PROGRESS_READY)
    {
      ShowProgress();
      ProgressShown_ = true;
    }
    else if (_IoState == Mcalc_IOState::GRAPH_PROGRESS && !_GrProgressFetched)
    {
      ShowGraphProgress();
      ProgressShown_ = true;
    }
    else if (_IoState == Mcalc_IOState::GRAPH_WAIT && !_GrOperationFetched)
    {
      FetchGraphOperationData();
      GraphDataShown_ = true;
    }

    if ((!ProgressShown_ && !GraphDataShown_ &&
         (PrevState_ != Mcalc_IOState::PROGRESS_READY ||
          PrevState_ != Mcalc_IOState::GRAPH_PROGRESS)) ||
        (ProgressShown_ && _EndProgressBar && _ProgressBarEndDone && _InputFileDoneLvl == 1) ||
        (WaitForOutput(1, RETRY_ATTEMPTS) &&
         (IoState() == Mcalc_IOState::OUTPUT_READY ||
          IoState() == Mcalc_IOState::CALC_RESPONSE ||
          IoState() == Mcalc_IOState::PROCESS_DONE ||
          IoState() == Mcalc_IOState::CALC_ERROR ||
          IoState() == Mcalc_IOState::ERROR_FETCH ||
          IoState() == Mcalc_IOState::INPUT_REQUIRED ||
          IoState() == Mcalc_IOState::GRAPH_OUTPUT)))
    {
      _FetchError = IoState() == Mcalc_IOState::ERROR_FETCH;
      
      if (_InputRequested)
      {
        std::cout <<"in>> ";
        std::cin.getline(_Input, 256, '\n');
        _Output[0] = 0;
      }
      else
      {
        std::cin.getline(_Input, 256, '\n');
        _Input[0] = 0;
        _Output[0] = 0;
      }
    }

    _BreakAll = false;
    _FetchError = _IoState == Mcalc_IOState::ERROR_FETCH;
    
    bool Quit_ = strcmp(_Input, "quit") == 0;
    bool Break_ = strcmp(_Input, "break") == 0;

    if (Quit_ || Break_)
    {
      if (Quit_)
        ProcessExited();
      
      _FetchError = false;
      _InputRequested = false;
      _BreakAll = true;
      
      SendUserResponse(QString(_Input), true);
    }
    else if (_Input && strlen(_Input))
    {
      _FetchError = false;
      ClearOutput();
      std::cout <<"output sent to: " <<GetProgramOutputFilename() <<std::endl;
      
      _InputRequested = false;
      SendUserResponse(QString(_Input));
    }
    else if (IoState() == Mcalc_IOState::INPUT_REQUIRED ||
             IoState() == Mcalc_IOState::CALC_ERROR || _FetchError)
    {
      _InputRequested = true;
      SendUserResponse("\n");
    }
    else if ((!ProgressShown_ && !GraphDataShown_ &&
              (PrevState_ != Mcalc_IOState::PROGRESS_READY ||
               PrevState_ != Mcalc_IOState::GRAPH_PROGRESS)) ||
             (ProgressShown_ && _EndProgressBar && _ProgressBarEndDone && _InputFileDoneLvl == 1) ||
             (WaitForOutput(1, RETRY_ATTEMPTS) &&
              (IoState() == Mcalc_IOState::OUTPUT_READY ||
               IoState() == Mcalc_IOState::CALC_RESPONSE ||
               IoState() == Mcalc_IOState::PROCESS_DONE ||
               IoState() == Mcalc_IOState::CALC_ERROR ||
               IoState() == Mcalc_IOState::ERROR_FETCH ||
               IoState() == Mcalc_IOState::INPUT_REQUIRED ||
               IoState() == Mcalc_IOState::GRAPH_OUTPUT)))
    {
      _FetchError = IoState() == Mcalc_IOState::ERROR_FETCH;
      _InputRequested = true;
    }

    #if CALCTEST_DEBUG3
      if (DbgPtr(false))
        DbgPtr()->LeaveLevel();
    #endif
  #endif
}

/****************************************************************************/
void CalcDriver::ShowGraphProgress()
{
  #if CALCDRIVER_TEXT_VERSION
    if (_ProgramExited)
      exit(0);
  #endif

  #if CALCTEST_DEBUG6c
    if (DbgPtr(false))
      DbgPtr()->EnterLevel("CalcDriver::ShowGraphProgress");
  #endif

  char* buffer = new char[128];
  ::memset(buffer, 0, 128);
  FILE* fp = fopen(GRAPH_PROGRESS_FILE, "r");
  _ReadOk = false;
  
  if (fp && buffer)
  {
    _GrReadOk = fgets(buffer, 128, fp) != NULL;
    fclose(fp);

    #if CALCTEST_DEBUG6c
      if (DbgPtr(false))
        DbgPtr()->ShowInt(_GrReadOk, "_GrReadOk");
    #endif
  }

  QString Output_;
  QString CalcOut_ = _GrReadOk ? buffer:"";

  if (_GrReadOk && buffer)
  {
    if (_GrReadBuffer)
    {
      delete[] _GrReadBuffer;
      _GrReadBuffer = NULL;
    }
    
    _GrReadBuffer = strcpy(new char[strlen(buffer) + 1], buffer);
    FetchGraphProgressData();

    #if CALCTEST_DEBUG6c
      if (DbgPtr(false))
      {
        DbgPtr()->ShowStr(_GrReadBuffer, "_GrReadBuffer");
        DbgPtr()->ShowMessage("FetchGraphProgressData()\n");
      }
    #endif
  }

  delete[] buffer;
  SendGraphProgressFetched();

  #if CALCDRIVER_TEXT_VERSION
    if (_GrProgressFetched || !_GrProgressDataShown)
    {
      ShowGraphProgressData();
      HandleGraphProgressData();
    }
  #endif

  #if CALCTEST_DEBUG6c
    if (DbgPtr(false))
    {
      DbgPtr()->ShowMessage("SendGraphProgressFetched()\n");
      DbgPtr()->LeaveLevel();
    }
  #endif
}

/****************************************************************************/
void CalcDriver::ShowProgress()
{
  #if CALCDRIVER_TEXT_VERSION
    if (_ProgramExited)
      exit(0);
  #endif

  char* buffer = new char[128];
  ::memset(buffer, 0, 128);
  FILE* fp = fopen(PROGRESS_FILE, "r");
  _ReadOk = false;
  
  #if CALCDRIVER_TEXT_VERSION
    bool SendNewLine_ = false;
  #endif
  
  if (fp && buffer)
  {
    _ReadOk = fgets(buffer, 128, fp) != NULL;
    fclose(fp);
  }

  QString Output_;
  QString CalcOut_ = _ReadOk ? buffer:"";

  if (_ReadOk && buffer)
  {
    if (_ReadBuffer)
    {
      delete[] _ReadBuffer;
      _ReadBuffer = NULL;
    }
    
    _ReadBuffer = strcpy(new char[strlen(buffer) + 1], buffer);
    FetchProgressData();
  }

  if (CalcOut_.length())
  {
    if (IoState() == Mcalc_IOState::CALC_ERROR || _FetchError)
      Output_ = QString("# ") + CalcOut_;
    else if (IoState() == Mcalc_IOState::INPUT_REQUIRED ||
             IoState() == Mcalc_IOState::PROGRESS_READY)
      Output_ = CalcOut_;
    else
      Output_ = QString(" = ") + CalcOut_;
  }
  else
    Output_ = QString("");

  if (IoState() == Mcalc_IOState::CALC_ERROR || _FetchError ||
      (IoState() != Mcalc_IOState::INPUT_REQUIRED && CalcOut_.length()))
  {
    Output_ += "\n";

    #if CALCDRIVER_TEXT_VERSION
      if (IoState() == Mcalc_IOState::CALC_ERROR || _FetchError)
        SendNewLine_ = true;
    #endif
  }

  #if CALCDRIVER_TEXT_VERSION
    if (Output_.length())
    {
      strcpy(_Output, Output_.ascii());
      
      if (IoState() == Mcalc_IOState::PROGRESS_READY && !_ProgressOutDone)
      {
        std::cout <<"done>> " <<_Output <<std::endl;
        _ProgressOutDone = true;
      }
    }
  #endif
  
  delete[] buffer;
  SendProgressFetched();
  
  #if CALCDRIVER_TEXT_VERSION
    if (_InitProgressBar && !_ProgressBarInitDone && _InputFileDoneLvl == -1)
    {
      HandleProgressData();
      ++_InputFileDoneLvl;      
    }
    else if (_EndProgressBar && !_ProgressBarEndDone && _InputFileDoneLvl == 0)
    {
      HandleProgressData();
      ++_InputFileDoneLvl;      
    }

  if (SendNewLine_)
    _InputRequested = true;
  #endif
}

/****************************************************************************/
void CalcDriver::ShowOutput()
{
  #if CALCDRIVER_TEXT_VERSION
    if (_ProgramExited)
      exit(0);
  #endif

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->EnterLevel("CalcDriver::ShowOutput");
  #endif

  QString Output_;
  bool SendNewLine_ = false;
  
  #if CALCDRIVER_GUI_VERSION
    QString CalcOut_ = _Output->text();
  #else
    QString CalcOut_ = _Output;
  #endif

  if (CalcOut_.length())
  {
    if (IoState() == Mcalc_IOState::CALC_ERROR || _FetchError)
      Output_ = QString("# ") + CalcOut_;
    else if (IoState() == Mcalc_IOState::INPUT_REQUIRED ||
             IoState() == Mcalc_IOState::OUTPUT_READY)
      Output_ = CalcOut_;
    else
      Output_ = QString(" = ") + CalcOut_;
  }
  else
    Output_ = QString("");

  if (IoState() == Mcalc_IOState::CALC_ERROR || _FetchError ||
      (IoState() != Mcalc_IOState::INPUT_REQUIRED && CalcOut_.length()))
  {
    Output_ += "\n";

    if (IoState() == Mcalc_IOState::CALC_ERROR || _FetchError)
      SendNewLine_ = true;
  }

  #if CALCDRIVER_TEXT_VERSION
    if (Output_.length())
    {
      strcpy(_Output, Output_.ascii());
      
      if (IoState() != Mcalc_IOState::PROGRESS_READY &&
          IoState() != Mcalc_IOState::GRAPH_PROGRESS && !_CalcOutDone)
      {
        std::cout <<"out>> " <<_Output <<std::endl;
        _CalcOutDone = true;
      }
    }
    
    if (_InputFileDoneLvl == 1 && _ProgressBarInitDone && _ProgressBarEndDone)
    {      
      ShowResultsFile(); // Show Results in file edit window
      HandleProgressData();
      _InputFileDoneLvl *= -1;
    }
    else if (_GrProgressFetched || !_GrProgressDataShown)
    {
      ShowGraphProgressData();
      HandleGraphProgressData();
    }
  #endif

  #if CALCDRIVER_TEXT_VERSION
    if (IoState() == Mcalc_IOState::PROGRESS_READY)
      // should have been executed in SendToInput method
      ShowProgress();
    else if (IoState() == Mcalc_IOState::GRAPH_PROGRESS && !_GrProgressFetched)
      // should have been executed in SendToInput method
      ShowGraphProgress();
    else if (IoState() == Mcalc_IOState::GRAPH_WAIT && !_GrOperationFetched)
      // should have been executed in SendToInput method
      FetchGraphOperationData();
    else
      SendOutputFetched();
  #else
    if ((IoState() != Mcalc_IOState::PROGRESS_READY) &&
        (IoState() != Mcalc_IOState::GRAPH_PROGRESS || _GrProgressFetched) &&
        (IoState() != Mcalc_IOState::GRAPH_WAIT || _GrOperationFetched))
      SendOutputFetched();
  #endif

  if (SendNewLine_)
    _InputRequested = true;

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void CalcDriver::ShowResultsFile()
{
  // Replace with process execution code
  #if defined(__linux__) | defined(__unix__)
    #if CALCTEST_EXEC_CURRENTDIR
      QString ProgName_ = QString("./calcedit");
    #else
      QString ProgName_ = QString("calcedit");
    #endif
  #else
    #if CALCTEST_EXEC_CURRENTDIR
      QString ProgName_ = QString(".\\calcedit.exe");
    #else
      QString ProgName_ = QString("calcedit.exe");
    #endif
  #endif

  _ProcPtr = new QProcess(this);
  _ProcPtr->addArgument(ProgName_);
  _ProcPtr->addArgument(_InputDataFname);
  _ProcPtr->addArgument(_OutputDataFname);
  _ProcPtr->setCommunication(QProcess::Stdin | QProcess::Stdout);

  connect(_ProcPtr, SIGNAL(processExited()),
          this, SLOT(ProcessExited()));

  if (!_ProcPtr->start())
  {
    #if CALCTEST_EXEC_CURRENTDIR
      #if defined(__linux__) | defined(__unix__)
        ProgName_ = QString("calcedit");
      #else
        ProgName_ = QString("calcedit.exe");
      #endif

      delete _ProcPtr;
      _ProcPtr = new QProcess(this);
      _ProcPtr->addArgument(ProgName_);
      _ProcPtr->addArgument(_InputDataFname);
      _ProcPtr->addArgument(_OutputDataFname);
      _ProcPtr->setCommunication(QProcess::Stdin | QProcess::Stdout);
      
      if (!_ProcPtr->start())
      {
        if (_FatalErrCount >= MAX_FATAL_ERRORS)
        {
          exit(1);
          return;
        }

        #if CALCTEST_SHOW_MSGBOX
          QMessageBox::critical(0, tr("Fatal Error"),
                                   tr("Could not start external text editor program"),
                                   tr("Quit"));
        #endif

        _FatalErrCount++;
        DoHalt();
        emit FatalError();
      }
      else
        _FatalErrCount = 0;
      
    #else
      #if defined(__linux__) | defined(__unix__)
        ProgName_ = QString("./calcedit");
      #else
        ProgName_ = QString(".\\calcedit.exe");
      #endif

      delete _ProcPtr;
      _ProcPtr = new QProcess(this);
      _ProcPtr->addArgument(ProgName_);
      _ProcPtr->addArgument(_InputDataFname);
      _ProcPtr->addArgument(_OutputDataFname);
      _ProcPtr->setCommunication(QProcess::Stdin | QProcess::Stdout);

      if (!_ProcPtr->start())
      {
        if (_FatalErrCount >= MAX_FATAL_ERRORS)
        {
          exit(1);
          return;
        }

        #if CALCTEST_SHOW_MSGBOX
          QMessageBox::critical(0, tr("Fatal Error"),
                                   tr("Could not start external text editor program"),
                                   tr("Quit"));
        #endif

        _FatalErrCount++;
        DoHalt();
        emit FatalError();
      }
      else
        _FatalErrCount = 0;
    #endif
  }
}

/****************************************************************************/
int CalcDriver::SendUserResponse(const QString& Buffer_, bool ExecCmd_)
{
  // stop retry reading of IoState
  IoStateTimer(STOP);
  bool WasActive_ = _ReadIoStateTimer.isActive();
  
  _RetryReadIoDone = false;

  // millisec sleep
  int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  int EvalRetval_ = false;

  bool DoWait_;
  bool NoWait_;
  bool EmptyBatchFileResponse_ =
            InBatchFile() &&
            (_IoState == Mcalc_IOState::CALC_ERROR ||
             _IoState == Mcalc_IOState::ERROR_FETCH ||
             _IoState == Mcalc_IOState::INPUT_REQUIRED ||
             _IoState == Mcalc_IOState::OUTPUT_READY ||
             _IoState == Mcalc_IOState::CALC_RESPONSE);

  bool BlanksAllowed_ = (IsServerStates(_IoState) ||
                         EmptyBatchFileResponse_ ||
                         _IoState == Mcalc_IOState::PROCESS ||
                         _IoState == Mcalc_IOState::PROCESS_DONE ||
                         _IoState == Mcalc_IOState::BREAK_PROGRAM);

  bool IntermediateState_ = IsIntermediateStates(_IoState);
  bool WaitForServerSend_ = WaitingForServerSend(_IoState);

  _SavedState = _IoState;
  _MatchingStates = WaitForServerSend_ &&
                    IsMatchingStates(_PrevIoState, _IoState, DoWait_, NoWait_);

  if (!WaitForServerSend_ || EmptyBatchFileResponse_)
  {
    _SavedState = 0;
    _NoWaitReadCnt = 0;
    
    DoWait_ = false;
  }

  if (DoWait_ && IntermediateState_)
  {
    if (!EmptyBuffer())
    {
      DelayLen_ *= _WaitCycle--;
      if (_WaitCycle == 0)
        _WaitCycle = 2;
    }

    _NoWaitReadCnt = 0;
    _RetryReadIoDone = false;

    IoStateTimer(START, EmptyBuffer() ? LONG_WAIT:DelayLen_);
    return false;
  }

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("CalcDriver::SendUserResponse");
      DbgPtr()->ShowStr(IoStateToStr(_IoState), "_IoState");
      DbgPtr()->ShowInt(_FetchError, "_FetchError");
    }
  #endif

  // no need to allocate new memory for ascii buffer
  // char* AsciiBufPtr_ = NewString(Buffer_.ascii());
  
  const char* AsciiBufPtr_ = Buffer_.ascii();
  char* sptr = NewString(AsciiBufPtr_);
  
  int x = sptr ? strlen(sptr):0;
  for (--x; x >= 0; x--)
    if (isspace(sptr[x]))
      sptr[x] = 0;
  
  bool BatchQuit_ = sptr && strlen(sptr) == 1 &&
                    toupper(sptr[0]) == 'Q';
  bool AllSpaces_ = sptr && strlen(sptr) == 0;
  bool Quit_ = sptr && strcmp(sptr, "quit") == 0;
  bool Break_ = sptr && strcmp(sptr, "break") == 0;
  bool Excmd_ = sptr && strncmp(sptr, "exec", 4) == 0 &&
                        isspace(sptr[4]);
  bool Grcmd_ = sptr && strncmp(sptr, "graph", 5) == 0 &&
                        (isspace(sptr[5]) || sptr[5] == '(');
  bool Pspollimp_ = Excmd_ || Grcmd_;

  bool Pspoll_ = sptr && strcmp(sptr, "pausepolling") == 0;
  bool Rspoll_ = sptr && strcmp(sptr, "resumepolling") == 0;

  delete[] sptr;
  sptr = NULL;  

  if (Quit_ || Break_ || Pspoll_ || Rspoll_)
  {
    if (Quit_)
      ProcessExited();

    ExecCmd_ = true;

    if (Quit_ || Break_)
      _BreakAll = true;
  }
  else
  {
    _BreakAll = false;

    if (Pspollimp_)
      SetSuspendPolling(true);
  }

  // IDLE_STATE          --> PROCESS
  // CALC_RESPONSE       --> PROCESS
  // INPUT_RECEIVED      --> PROCESS
  // OUTPUT_FETCHED      --> PROCESS
  // CALC_ERROR          --> PROCESS
  // ERROR_FETCH         --> PROCESS
  // BREAK_PROGRAM       --> PROCESS
  // BATCHFILE_ENDED_ACK --> PROCESS
  
  // Input Data Batch File Mode:
  // ---------------------------
  // If calculator is executing a batch file or program,
  // the following special cases apply:
  //
  //   INPUT_RECEIVED      --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
  //   OUTPUT_FETCHED      --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
  //   BREAK_PROGRAM       --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
  //   BATCHFILE_ENDED_ACK --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
  //
  //   CALC_RESPONSE       --> INPUT_RECEIVED
  //   CALC_ERROR          --> INPUT_RECEIVED
  //   ERROR_FETCH         --> INPUT_RECEIVED
  //
  //   PROCESS_DONE        --> PROCESS_DONE_ACK
  //   PROCESS_DONE_ACK    --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE

  // INPUT_REQUIRED      --> INPUT_RECEIVED
  // GRAPH_WAIT          --> INPUT_RECEIVED

  // OUTPUT_READY        --> OUTPUT_FETCHED --> PROCESS
  // GRAPH_OUTPUT        --> OUTPUT_FETCHED --> PROCESS
  // PROCESS_DONE        --> PROCESS_DONE_ACK --> PROCESS

  // SPAWN_NEW_MCALC     --> MCALC_SPAWNED --> PROCESS

  if (_IoState == Mcalc_IOState::SPAWN_NEW_MCALC)
  {
    // - sent by calculator to notify client to spawn another mcalc process
    EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::MCALC_SPAWNED, ExecCmd_);
  }
  else if (InBatchFile() &&
           (_FetchError ||
            _IoState == Mcalc_IOState::CALC_RESPONSE ||
            _IoState == Mcalc_IOState::CALC_ERROR))
  {
    if (ResumePolling())
    {
      ResumeCalcPolling();
      SetResumePolling(false);
    }
  
    EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::INPUT_RECEIVED, ExecCmd_);
  }
  else if (_IoState == Mcalc_IOState::INPUT_REQUIRED ||
           _IoState == Mcalc_IOState::CALC_ERROR || _FetchError)
  {
    if (ResumePolling())
    {
      ResumeCalcPolling();
      SetResumePolling(false);
    }
  
    if (_IoState == Mcalc_IOState::INPUT_REQUIRED)
      EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::INPUT_RECEIVED, ExecCmd_);
    else
      EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::PROCESS, ExecCmd_);
  }
  else if (_IoState == Mcalc_IOState::OUTPUT_READY ||
           _IoState == Mcalc_IOState::GRAPH_OUTPUT)
  {
    if (ResumePolling())
    {
      ResumeCalcPolling();
      SetResumePolling(false);
    }
  
    EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::OUTPUT_FETCHED, ExecCmd_);
  }
  else if (_IoState == Mcalc_IOState::PROGRESS_READY)
    EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::INFILE_PROGRESS_ACK, ExecCmd_);
  else if (_IoState == Mcalc_IOState::PROCESS_DONE)
    EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::PROCESS_DONE_ACK, ExecCmd_);
  else if (_IoState == Mcalc_IOState::GRAPH_WAIT)
  {
    if (BatchQuit_)
    {
      EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::INPUT_RECEIVED, ExecCmd_);
      SetInBatchFile(false, true);
    }
    else
      EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::GRAPH_WAIT_ACK, ExecCmd_);
  }
  else if (_IoState == Mcalc_IOState::GRAPH_PROGRESS)
    EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::GRAPH_PROGRESS_ACK, ExecCmd_);
  else if (_IoState == Mcalc_IOState::BREAK_PROGRAM &&
           (Buffer_.length() == 0 || AllSpaces_))
    EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::BREAK_PROGRAM, ExecCmd_);
  else if (!AllSpaces_ || BlanksAllowed_)
  {
    if (ResumePolling())
    {
      ResumeCalcPolling();
      SetResumePolling(false);
    }

    if (IsCalcResponseSignals(_IoState))
      _ExpectedSignals->SetFromServer(_IoState);
    else if (WaitingForServerSend(_IoState))
      _ExpectedSignals->SetFromClient(_IoState);

    if (_ExpectedSignals->IsProcessNext() &&
        !_ExpectedSignals->PendingResponse())
    {
      _ExpectedSignals->Reset();
      _ExpectedSignals->SetClientExpected(Mcalc_IOState::PROCESS);
    }
      
    EvalRetval_ = EvalExpression(AsciiBufPtr_, Mcalc_IOState::PROCESS, ExecCmd_);
  }
  else if (WasActive_ && WaitForServerSend_)
  {
    if (!EmptyBuffer())
    {
      DelayLen_ *= _WaitCycle--;
      if (_WaitCycle == 0)
        _WaitCycle = 2;
    }

    _NoWaitReadCnt = 0;
    _RetryReadIoDone = false;

    IoStateTimer(START, EmptyBuffer() ? LONG_WAIT:DelayLen_);
  }

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif

  return EvalRetval_;
}

/****************************************************************************/
void CalcDriver::RetryFileOpen()
{
  #if CALCDRIVER_TEXT_VERSION
    if (_ProgramExited)
      exit(0);
  #endif

  #if ((IMPLEMENT_KEEP_ALIVE)|(IDLE_RESPONSE_ONLY_POLL))
    int Sleep_ = 0;
  #endif
  FILE* Fptr_ = NULL;
  
  if (_FromMethod == READFROMOUTPUT && _PollServerMgr)
  {
    #if IMPLEMENT_KEEP_ALIVE
    if (CalcPollingActive())
    {
      _PollServerMgr->SetExitAtRetryMax(true)
                    ->SetSleepWaiting(true)
                    ->SetAllowServerPolling(true)
                    ->SetBreakOnPollError(false);

      #if CALCTEST_DEBUG4
        if (DbgPtr(false))
        {
          _PollServerMgr->SetStopTracking(false);
          DbgPtr()->ShowMessage("RetryFileOpen:: _PollServerMgr->PollServerRequired(0, _ResponseOnly)\n");
        }
      #endif

      if (!_NoPolling && _PollServerMgr &&
          _PollServerMgr->SetAllowServerPolling(!_ResponseOnly)
                        ->ResetWhenNotRequired(true)
                        ->PollServerRequired(0, _ResponseOnly))
      {
        #if CALCTEST_DEBUG4b
          if (DbgPtr(false))
            DbgPtr()->ShowMessage("RetryFileOpen:: _PollServerMgr->WaitingForServerResponse()\n");
        #endif
      
        Sleep_ = _PollServerMgr->WaitingForServerResponse() ? 1:0;
        _PollServerMgr->ResetServerPolling(!_ResponseOnly);
        _PollServerMgr->SetAllowServerPolling(!_ResponseOnly);
        _PollServerMgr->PollServerIfAlive(Sleep_, true, _ResponseOnly);
      }
    }
    #endif

    ReadFromOutput(_SavedState != 0, _SavedState);
    return;
  }

  switch (_AttemptFile)
  {
    case IOSTATE:
      #if CALCTEST_DEBUG
        QMessageBox::warning(0, "DEBUG",
                             QString("Error: opening ") + QString(PROG_IOSTATE_FILE) +
                             QString(" for writing, client side"),
                             QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
      #endif
      
      Fptr_ =
      _IoStateFout = fopen(PROG_IOSTATE_FILE, "w");
      break;

    case ELAPSEDSTATE:
      #if CALCTEST_DEBUG
        QMessageBox::warning(0, "DEBUG",
                             QString("Error: opening ") + QString(ELAPSED_TIME_FILE) +
                             QString(" for writing, client side"),
                             QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
      #endif
      
      Fptr_ =
      _ElapsedStateFout = fopen(ELAPSED_TIME_FILE, "w");
      break;

    case SESSION:
      Fptr_ =
      _SessionFout = fopen(MCALC_SESSION_FILE, "r+");
      break;

    case SERVERALIVE:
      ++_OpenServerAliveFileRetries;
      Fptr_ =
      _ServerAliveFile = GetServerAliveFile(_ServerAliveMode == 'w', true);
      if (_OpenServerAliveFileRetries >= RETRY_READ_MAX)
        _FileOpenRetryTimer.stop();
      break;

    case CLIENTALIVE:
      ++_OpenClientAliveFileRetries;
      Fptr_ =
      _ClientAliveFile = GetClientAliveFile(_ClientAliveMode == 'w', true);
      if (_OpenClientAliveFileRetries >= RETRY_READ_MAX)
        _FileOpenRetryTimer.stop();
      break;

    case POLLSTATE:
      ++_OpenPollStateFileRetries;
      Fptr_ =
      _PollStateFile = GetPollStateFile(_PollStateMode == 'w', true);
      if (_OpenPollStateFileRetries >= RETRY_READ_MAX)
        _FileOpenRetryTimer.stop();
      break;
  }

  if (Fptr_)
  {
    _FileOpenRetryTimer.stop();

    switch (_FromMethod)
    {
      case SETIOSTATE:
        SetIoState(_SavedState);
        break;

      case SETCALCELAPSEDTIME:
        SetCalcElapsedTimeState(_SavedElapsedState, "CalcDriver::RetryFileOpen()");
        break;

      case DOHALT:
        DoHalt();
        break;

      case SETSESSIONENTRY:
        SetSessionEntry(_SavedFilePos, _SavedLen);
        break;

      case POLLSERVERIFALIVE:
        #if ((IMPLEMENT_KEEP_ALIVE)|(IDLE_RESPONSE_ONLY_POLL))
        if (CalcPollingActive())
        {
          #if CALCTEST_DEBUG4
            if (DbgPtr(false))
            {
              _PollServerMgr->SetStopTracking(false);
              DbgPtr()->ShowMessage("RetryFileOpen:: _PollServerMgr->PollServerRequired(0, _ResponseOnly)\n");
            }
          #endif
        
          if (!_NoPolling && _PollServerMgr &&
              _PollServerMgr->SetAllowServerPolling(!_ResponseOnly)
                            ->ResetWhenNotRequired(true)
                            ->PollServerRequired(0, _ResponseOnly))
          {
            #if CALCTEST_DEBUG4b
              if (DbgPtr(false))
                DbgPtr()->ShowMessage("RetryFileOpen:: _PollServerMgr->WaitingForServerResponse()\n");
            #endif

            Sleep_ = _PollServerMgr->WaitingForServerResponse() ? 1:0;
            _PollServerMgr->ResetServerPolling(!_ResponseOnly);
            _PollServerMgr->SetAllowServerPolling(!_ResponseOnly);
            _PollServerMgr->PollServerIfAlive(Sleep_, true, _ResponseOnly);
          }
        }
        #endif
        break;

      case RECEIVEKEEPALIVE:
        #if ((IMPLEMENT_KEEP_ALIVE)|(IDLE_RESPONSE_ONLY_POLL))
          #if CALCTEST_DEBUG4
            if (DbgPtr(false))
            {
              _PollServerMgr->SetStopTracking(false);
              DbgPtr()->ShowMessage("RetryFileOpen:: _PollServerMgr->ClientReceiveRequired(0)\n");
            }
          #endif
        
          if (!_NoPolling && _PollServerMgr &&
              _PollServerMgr->ResetWhenNotRequired(true)
                            ->ClientReceiveRequired(0))
          {
            #if CALCTEST_DEBUG4b
              if (DbgPtr(false))
                DbgPtr()->ShowMessage("RetryFileOpen:: _PollServerMgr->WaitingForServerResponse()\n");
            #endif
          
            Sleep_ = _PollServerMgr->WaitingForServerResponse() ? 1:0;
            _PollServerMgr->ReceiveKeepAlivePoll(Sleep_, true);
          }
        #endif
        break;

      case OPENFILE:
        Fptr_ =
        (_AttemptFile == CLIENTALIVE) ? _ClientAliveFile:
        (_AttemptFile == SERVERALIVE) ? _ServerAliveFile:
        (_AttemptFile == POLLSTATE)   ? _PollStateFile:NULL;

        if (_AttemptFile)
          openfile(&Fptr_, _AttemptFileName, _AttemptMode, true);
    }    
  }
}

/****************************************************************************/
CalcDriver* CalcDriver::SetNoPolling(bool flag_)
{
  _NoPolling = flag_;
  return this;
}

/****************************************************************************/
int CalcDriver::NoPolling() const
{
  return _NoPolling;
}

/****************************************************************************/
void CalcDriver::SetCalcElapsedTimeState(int State_, const char* From_)
{
  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("CalcDriver::SetCalcElapsedTimeState");
      DbgPtr()->ShowStr(IoStateToStr(State_), "State_");
    }
  #endif

  const char* ElapsedStateFile_ = GetCalcElapsedTimeFilename();
  
  // millisec sleep
  int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;

  #if CALCTEST_DEBUG1e
    QMessageBox::warning(0, "DEBUG",
                         QString("opening ") + QString(ElapsedStateFile_) +
                         QString(" for writing 1st attempt, client side"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #endif

  if (!_ElapsedStateFout)
  {
    _ElapsedStateFout = fopen(ElapsedStateFile_, "w");

    #if CALCTEST_DEBUG1e
      QMessageBox::warning(0, "DEBUG",
                           QString("opening ") + QString(ElapsedStateFile_),
                           QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
    #endif

    if (!_ElapsedStateFout)
    {
      _AttemptFile = ELAPSEDSTATE;
      _AttemptMode = "w";
      _AttemptFileName = ElapsedStateFile_;
      SetFromMethod(SETCALCELAPSEDTIME);
      
      _SavedElapsedState = State_;
      _FileOpenRetryTimer.start(DelayLen_);
      return;
    }
  }

  if (State_ >= 0)
  {
    fprintf(_ElapsedStateFout, "%d\n", State_);
    _ElapsedState = State_;
  }

  fclose(_ElapsedStateFout);
  _ElapsedStateFout = NULL;

  #if CALCTEST_DEBUG1e
    if (From_)
      QMessageBox::warning(0, "DEBUG SetElapsedState()",
                           QString("\n_ElapsedState == ") +
                           QString(TimeoutStateToStr(_ElapsedState)) +
                           QString("\nFrom: ") + From_,
                           QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
    else
      QMessageBox::warning(0, "DEBUG SetElapsedState()",
                           QString("\n_ElapsedState == ") +
                           QString(TimeoutStateToStr(_ElapsedState)),
                           QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #elif CALCTEST_DEBUG4
    DbgPtr()->ShowStr(TimeoutStateToStr(_ElapsedState), "_ElapsedState");
  #endif

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void CalcDriver::SetIoState(int State_, bool SetStateOnly_, bool IgnoreDup_)
{
  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("CalcDriver::SetIoState");
      DbgPtr()->ShowStr(IoStateToStr(State_), "State_");
      DbgPtr()->ShowInt(SetStateOnly_, "SetStateOnly_");
    }
  #endif

  const char* IoStateFile_ = GetProgramIostateFilename();
  const int resendsig_ = Mcalc_IOState::RESENDLASTSIG;

  int Duplicate_ = IgnoreDup_ ? (State_ == IoState()):false;
  
  // millisec sleep
  int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;

  #if CALCTEST_DEBUG
    QMessageBox::warning(0, "DEBUG",
                         QString("opening ") + QString(IoStateFile_) +
                         QString(" for writing 1st attempt, client side"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);  
  #endif

  if (!_IoStateFout && !Duplicate_)
  {
    _IoStateFout = fopen(IoStateFile_, "w");

    #if CALCTEST_DEBUG
      QMessageBox::warning(0, "DEBUG",
                           QString("opening ") + QString(IoStateFile_),
                           QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
    #endif

    if (!_IoStateFout)
    {
      _AttemptFile = IOSTATE;
      _AttemptMode = "w";
      _AttemptFileName = IoStateFile_;
      SetFromMethod(SETIOSTATE);
      
      _SavedState = State_;
      _FileOpenRetryTimer.start(DelayLen_);
      _NoPolling = false;
      return;
    }
  }

  #if CALCTEST_DEBUG
    QMessageBox::warning(0, "DEBUG",
                         QString("opening ") + QString(IoStateFile_) +
                         QString(" for writing:") + QString::number(State_) +
                         QString(" successful, client side"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);  
  #endif

  if (State_ && !Duplicate_)
  {
    if (_ResendLastSig)
    {
      _ResendLastSig = false;
      _RequestResend = State_ == resendsig_;
    }
  
    fprintf(_IoStateFout, "%d\n", State_);

    if (State_ == Mcalc_IOState::RESENDLASTSIG ||
        State_ == Mcalc_IOState::BATCHFILE_ENDED_ACK)
    {
      ReadIoState(1, 0);
      State_ = IoState();

      if (_SendingEndBatchFileAck)
      {
        _SendingEndBatchFileAck = false;
        SetStateOnly_ = !InBatchFile();
      }
    }
  }

  if (!Duplicate_)
  {
    fclose(_IoStateFout);
    _IoStateFout = NULL;
  }
  
  if (!SetStateOnly_)
    _Processing = TRUE;

  // Reset last output acknowledgement signal if confirmed by client
  if (InBatchFile() ||
      _ExpectedSignals->ResponseReceived(State_))
    _ExpectedSignals->Reset();

  #if ((IMPLEMENT_KEEP_ALIVE)|(IDLE_RESPONSE_ONLY_POLL))
    int mult_;
    bool ResponseSent_ = IsServerStates(State_);
  #endif
  
  #if CALCTEST_DEBUG3p
    if (_PollServerMgr)
    {
      _PollServerMgr->ResetPollCount();
      _PollServerMgr->SetWaitUntilResponse(true)
                    ->SetAllowServerPolling(true)
                    ->SetSleepWaiting(true)
                    ->SetResetStatesOnDone(true)
                    ->SetBreakOnPollError(false)
                    ->SetStopTracking(false);
    }
  
    if (!ResponseSent_ && !_NoPolling && _PollServerMgr &&
        _PollServerMgr->ResetWhenNotRequired(true)
                      ->PollServerRequired(0, false))
      SimulatePolling();
  #else
    #if ((IMPLEMENT_KEEP_ALIVE)|(IDLE_RESPONSE_ONLY_POLL))
    if (CalcPollingActive())
    {
      #if CALCTEST_DEBUG4
        if (DbgPtr(false))
        {
          _PollServerMgr->SetStopTracking(false);
          DbgPtr()->ShowInt(ResponseSent_, "ResponseSent_");
          DbgPtr()->ShowInt(_NoPolling, "_NoPolling");
          DbgPtr()->ShowMessage("SetIoState:: _PollServerMgr->PollServerRequired(0, true)\n");
        }
      #endif
    
      if (!ResponseSent_ && !_NoPolling && _PollServerMgr &&
          _PollServerMgr->ResetWhenNotRequired(true)
                        ->PollServerRequired(0, true))
      {
        #if CALCTEST_DEBUG4b
          if (DbgPtr(false))
            DbgPtr()->ShowMessage("SetIoState:: _PollServerMgr->WaitingForServerResponse()\n");
        #endif
      
        mult_ = _PollServerMgr->WaitingForServerResponse() ? 1:0;
        _ResponseOnly = true;
        _PollServerMgr->SetStopTracking(false);
        _PollServerMgr->ResetServerPolling(false);
        _PollServerMgr->PollServerIfAlive(mult_, false, true); // Response Only
      }
    }
    #endif
  #endif

  #if CALCTEST_DEBUG1b
    QMessageBox::warning(0, "DEBUG-cdrv SetIoState()",
                         QString("\n_IoState == ") + QString::number(_IoState),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #elif CALCTEST_DEBUG4
    DbgPtr()->ShowStr(IoStateToStr(_IoState), "_IoState");
  #endif

  if (SetStateOnly_)
  {
    if (State_ && !_RequestResend)
      _IoState = State_;
  }
  else
    ReadFromOutput(true, _RequestResend ? _IoState:State_);

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void CalcDriver::SendResendLastSig()
{
  #if CALCTEST_DEBUG4
    if (DbgPtr(false))
      DbgPtr()->EnterLevel("CalcDriver::SendResendLastSig");
  #endif

  if (!_RequestResend && !_ResendSigDone && !_ResendSigUnchanged)
  {
    _FatalErrCount = 0;
    _NoWaitReadCnt = 0;
    _ResendLastSig = true;
    _SendResendLastSig = true;

    SetIoState(Mcalc_IOState::RESENDLASTSIG, true);
  }

  #if CALCTEST_DEBUG4
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void CalcDriver::RunCalcServerCalledHalt()
{
  _ServerCalledHalt = true;  
  SetIoState(Mcalc_IOState::IDLE_STATE, true);
  emit DoQuit();
}

/****************************************************************************/
// Replace with GUI method
// busy wait replaced with timer code here
//
void CalcDriver::DoHalt()
{
  const char* IoStateFile_ = GetProgramIostateFilename();
  int TermDelay_ = IOSTATE_DELAY_LENGTH * RETRY_READ_MAX;
  
  _Halted = true;

  #if CALCTEST_DEBUG
    QMessageBox::warning(0, "DEBUG",
                         QString("Calling DoHalt"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #endif

  #if CALCTEST_DEBUG
    QMessageBox::warning(0, "DEBUG",
                         QString("opening ") + QString(IoStateFile_) +
                         QString(" for writing, client side"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #endif
  
  if (!_IoStateFout)
  {
    _IoStateFout = fopen(IoStateFile_, "w");
    
    #if CALCDRIVER_TEXT_VERSION
      if (!_IoStateFout)
      {
        ResetSession();
        exit(0);
      }
    #else
      // millisec sleep
      int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
    
      if (!_IoStateFout)
      {
        _AttemptFile = IOSTATE;
        _AttemptMode = "w";
        _AttemptFileName = IoStateFile_;
        SetFromMethod(DOHALT);
        
        _FileOpenRetryTimer.start(DelayLen_);
        return;
      }
    #endif
  }

  #if CALCTEST_DEBUG
    QMessageBox::warning(0, "DEBUG",
                         QString("opening ") + QString(IoStateFile_) +
                         QString(" for writing successful, client side"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);  
  #endif
  
  fprintf(_IoStateFout, "%d\n", Mcalc_IOState::CALC_HALT);
  fclose(_IoStateFout);
  _IoStateFout = NULL;

  // make sure calculator server program terminates
  if (!_ProgramExited)
    _TerminationTimer.start(TermDelay_, TRUE);
  else
    ResetSession();
}

/****************************************************************************/
void CalcDriver::SetSessionEntry(long Fpos_, size_t Len_)
{
      // millisec sleep
      int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;

      if (!_SessionFout)
      {
        _SessionFout = fopen(MCALC_SESSION_FILE, "r+");
    
        if (!_SessionFout)
        {
          _AttemptFile = SESSION;
          _AttemptMode = "r+";
          _AttemptFileName = MCALC_SESSION_FILE;
          SetFromMethod(SETSESSIONENTRY);
          
          _SavedFilePos = Fpos_;
          _SavedLen = Len_;
          _FileOpenRetryTimer.start(DelayLen_);
          return;
        }
      }

      char* buffer = new char[128];      
      fseek(_SessionFout, Fpos_, SEEK_SET);
      sprintf(buffer, "%d", _SessionNum);
      strcat(buffer, ",");
      sprintf(&buffer[strlen(buffer)], "%d", Mcalc_SessionStates::UnUsed);
      --Len_;
      while (strlen(buffer) != Len_)
        strcat(buffer, " ");
      strcat(buffer, "\n");
      fputs(buffer, _SessionFout);
      delete[] buffer;
      fclose(_SessionFout);
      _SessionFout = NULL;
}

/****************************************************************************/
// Replace with GUI method
// busy wait replaced with timer code here
//
void CalcDriver::ResetSession()
{
  FILE* Fptr_ = fopen(MCALC_SESSION_FILE, "r");
  char* buffer = new char[128];
  char* token1;
  long Fpos_;
  int SessionVal_;
  int Found_ = 0;
  size_t Len_;  
  char* RetStr_ = NULL;

  while (Fptr_ && !feof(Fptr_))
  {
    Fpos_ = ftell(Fptr_);
    RetStr_ = fgets(buffer, 128, Fptr_);
    if (!RetStr_)
      break;
    
    Len_ = strlen(RetStr_);
    token1 = strtok(RetStr_, ",");    
    SessionVal_ = atoi(token1);
    Found_ = SessionVal_ == _SessionNum;
    
    if (Found_)
    {
      fclose(Fptr_);      
      SetSessionEntry(Fpos_, Len_);
      Fptr_ = fopen(LOG_FILE, "r");
      
      if (Fptr_)
      {
        Len_ = FileSize(Fptr_);
        fclose(Fptr_);
        
        if (!Len_)
          unlink(LOG_FILE);
      }
      
      Fptr_ = fopen(CURRENTERROR_FILE, "r");
      
      if (Fptr_)
      {
        Len_ = FileSize(Fptr_);
        fclose(Fptr_);

        if (!Len_)
          unlink(CURRENTERROR_FILE);
      }
      
      break;
    }
  }

  delete[] buffer;
  
  #if CALCDRIVER_TEXT_VERSION
    if (!_Halted)
      emit DoQuit();
  #else
    emit DoQuit();
  #endif
}

/****************************************************************************/
int CalcDriver::SimulatePolling()
{
  #if IMPLEMENT_KEEP_ALIVE
  int LoopGuard_ = 20;

  bool PollingServer_ = false;
  bool PollError_ = false;
  bool PollAnswered_ = false;
  bool PollingAllowed_ = false;
  
  if (CalcPollingActive())
  {
    if (_PollServerMgr)
    {
      PollingAllowed_ = _PollServerMgr->ServerPollingAllowed();
      _PollServerMgr->ResetPollCount();
      _PollServerMgr->ResetPollResults(PollingServer_, PollError_, PollAnswered_);
      _PollServerMgr->SetWaitUntilResponse(true)
                    ->SetAllowServerPolling(true)
                    ->SetSleepWaiting(true)
                    ->SetResetStatesOnDone(true)
                    ->SetBreakOnPollError(false)
                    ->SetStopTracking(false);
    }

    #if CALCTEST_DEBUG3
      if (DbgPtr(false))
        DbgPtr()->EnterLevel("SimulatePolling");
    #endif

    if (_PollServerMgr)
    for (LoopGuard_ = RETRY_READ_MAX;
         !_PollServerMgr->AtMaxPollAttempts() && LoopGuard_;
         LoopGuard_--)
    {
      #if CALCTEST_DEBUG3
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("SimulatePolling: _PollClientMgr->PollClientIfAlive(1, true, false);\n");
      #endif
      SetNoPolling(false);
      _PollServerMgr->ResetWhenNotRequired(true)
                    ->PollServerIfAlive(1, true, false); // Initiate Polling

      #if CALCTEST_DEBUG5
        _PollServerMgr->SetStopTracking(false);
      #endif
                    
      _PollServerMgr->GivePollResults(PollingServer_, PollError_, PollAnswered_);

      if (PollAnswered_ || PollError_)
      {
        _PollServerMgr->SetAllowServerPolling(PollingAllowed_);
        if (PollError_)
          _PollServerMgr->ResetServerPolling(true);

        break;
      }
      else
        _PollServerMgr->ResetServerPolling(true);
    }

    if (_PollServerMgr)
      _PollServerMgr->SetWaitUntilResponse(false);
  }
  #endif

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
    {
      DbgPtr()->LeaveLevel();
      DbgPtr()->ResetLevels();
    }
  #endif

  _PollServerMgr->SetAllowServerPolling(false);
  SetNoPolling(true);
  int IoState_ = GetIoState(0);
  SetNoPolling(false);
  
  return IoState_;
}

/****************************************************************************/
bool CalcDriver::ReadElapsedTimeState(int ResponseSent_, char* Buffer_, const char* From_)
{
  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("ReadElapsedTimeState");
      DbgPtr()->ShowInt(ResponseSent_, "ResponseSent_");
      DbgPtr()->ShowStr(Buffer_, "Buffer_");
      DbgPtr()->ShowStr(From_, "From_");
    }
  #endif

  const char* ElapsedFile_ = GetCalcElapsedTimeFilename();
  bool ReadElapsedOk_ = false;
  bool NullBuf_ = Buffer_ == NULL;
  int BufSz_ = 128;

  if (NullBuf_)
    Buffer_ = new char[BufSz_];

  if (!From_)
    From_ = "CalcDriver::ReadElapsedTimeState";

  FILE* ElapsedFin_ = fopen(ElapsedFile_, "r");

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->ShowInt(ElapsedFin_!=NULL, "ElapsedFin_!=NULL");
  #endif

  if (ElapsedFin_)
  {
    if (!ResponseSent_)
    {
      ReadElapsedOk_ = fgets(Buffer_, BufSz_, ElapsedFin_) != NULL;

      if (ReadElapsedOk_)
      {
        _ElapsedState = atoi(Buffer_);

        #if CALCTEST_DEBUG1e
          if (_ElapsedState != Mcalc_ElapsedTime::RESET)
            QMessageBox::warning(0, From_,
                                 QString("\n_ElapsedState == ") + QString(TimeoutStateToStr(_ElapsedState)),
                                 QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
        #elif CALCTEST_DEBUG3
          DbgPtr()->ShowInt(_ElapsedState, "_ElapsedState");
        #endif
      
        if (_ElapsedState == Mcalc_ElapsedTime::SENT)
        {
          #if CALCTEST_DEBUG1e
            QMessageBox::warning(0, From_,
                                 QString("\nemit CalcServerTimeout();"),
                                 QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
          #elif CALCTEST_DEBUG3
            DbgPtr()->ShowMessage("emit CalcServerTimeout();");
          #endif
        
          emit CalcServerTimeout();
        }
      }
    }
    
    fclose(ElapsedFin_);
  }

  if (NullBuf_)
    delete[] Buffer_;

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
    {
      DbgPtr()->ShowInt(ReadElapsedOk_, "ReadElapsedOk_");
      DbgPtr()->LeaveLevel();
    }
  #endif

  return ReadElapsedOk_;
}

/****************************************************************************/
// Replace with GUI method
// busy wait replaced with timer code here
//
// int delay_ : delay pulse multiplier
// int retry_ : number of times to retry reading until ResponseSent_ == TRUE
//              status is received from calculator server before quitting
//
// RETURN:
//   Returns true if response from calculator server is sent, otherwise
//   returns false.
//
int CalcDriver::WaitForOutput(int delay_, int retry_, int* ReadyToSendp_)
{
  #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("CalcDriver::WaitForOutput");
      DbgPtr()->ShowInt(delay_, "delay_");
      DbgPtr()->ShowInt(retry_, "retry_");
      DbgPtr()->ShowStr(IoStateToStr(_IoState), "_IoState");
    }
  #endif

  if (_EnterKeyActive)
  {
    _EnterKeyActive = false;
    emit DisableEnterKey();
  }

  #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  #else
    // microsec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MICRO_DELAY_MULT;
  #endif
  
  if (delay_)
  {
    #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
      Sleep(DelayLen_ * delay_);
    #else
      ::usleep(DelayLen_ * delay_);
    #endif
  }
  
  char Buffer_[256];
  const char* IostateFile_ = GetProgramIostateFilename();
  
  const int resendsig_ = Mcalc_IOState::RESENDLASTSIG;
  
  FILE* StateFin_ = fopen(IostateFile_, "r");

  bool ClientPingReceived_ = false;
  bool EndBatchFileReceived_ = false;
  bool BatchLevelReceived_ = false;
  bool PollingActive_ = false;
  bool PollingAllowed_ = _PollServerMgr->ServerPollingAllowed();

  #if ((IMPLEMENT_KEEP_ALIVE)|(CALCTEST_ALLOWPOLL)|(IDLE_RESPONSE_ONLY_POLL))
    int mult_;
    int polldone_ = 0;
  #endif

  int BatchLevel_ = 0;
  int LastState_ = _IoState;
  int PrevState_ = (_PrevIoState == 0) ? _IoState:_PrevIoState;
  int PrevGuiRequest_ = IsGuiReadyToSend(PrevState_, true, true);
  
  int Eof_ = 0;
  int retval = 0;
  int ReadyToSend_ = 0;
  int ErrCond_ = StateFin_ == NULL ||
                 (Eof_ = feof(StateFin_)) ||
                 ferror(StateFin_);

  if (ErrCond_ || !FileSize(StateFin_))
  {
    if (ErrCond_)
    {
      if (StateFin_)
      {      
        clearerr(StateFin_);
        if (Eof_)
          fclose(StateFin_);
          
        StateFin_ = NULL;
      }
    }
    else
      fclose(StateFin_);

    // altered : 2018-10-04
    #if LIMIT_READ_RETRIES
      if (_PollServerMgr && _IostateFileReadError &&
          !_PollServerMgr->ReAttemptFileRead())
      {
        if (_FatalErrCount < MAX_FATAL_ERRORS)
        {
          sprintf(Buffer_, ERRMSG_OPENSTATEFILE, IostateFile_);

          #if CALCTEST_SHOW_MSGBOX
            QMessageBox::critical(0, "ERROR", QString(Buffer_),
                                  QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
          #endif

          _FatalErrCount++;
          DoHalt();
          emit FatalError();
        }
        else
          exit(1);
      }
      else
        _FatalErrCount = 0;
    #endif

    if (retry_ > RETRY_ATTEMPTS / 10)
      retry_ = 1;
    else if (retry_ == 0 && ErrCond_)
      _IoState = Mcalc_IOState::IDLE_STATE;

  #if CALCTEST_ALLOWPOLL
    if (_PollServerMgr)
    {
      _PollServerMgr->SetStopTracking(false);
      _PollServerMgr->SetExitAtRetryMax(true)
                    ->SetSleepWaiting(true)
                    ->SetAllowServerPolling(true)
                    ->SetBreakOnPollError(true);
    }

    #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
      if (DbgPtr(false))
        DbgPtr()->ShowMessage("ErrCond_ || !FileSize(StateFin_)\n"
                              "_PollServerMgr->PollServerRequired(0, false)\n");
    #endif

    #if ((CALCTEST_DEBUG4)|(CALCTEST_DEBUG1c))
      if (DbgPtr(false))
      {
        _PollServerMgr->SetStopTracking(false);
        DbgPtr()->ShowInt(_NoPolling, "_NoPolling");
        DbgPtr()->ShowMessage("WaitForOutput:: _PollServerMgr->PollServerRequired(0, false)\n");
      }
    #endif

    if (!_NoPolling && _PollServerMgr && retry_ > 0 &&
        _PollServerMgr->SetAllowServerPolling(true)
                      ->ResetWhenNotRequired(true)
                      ->PollServerRequired(0, false))
    {
      #if ((CALCTEST_DEBUG4b)|(CALCTEST_DEBUG1c))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("WaitForOutput:: _PollServerMgr->WaitingForServerResponse()\n");
      #endif

      PollingActive_ = CalcPollingActive();
      ResumeCalcPolling();

      mult_ = (!delay_ && _PollServerMgr->WaitingForServerResponse()) ? 1:0;
      _ResponseOnly = false;
      _PollServerMgr->SetAllowServerPolling(true);
      _PollServerMgr->ResetServerPolling(true);
      _PollServerMgr->PollServerIfAlive(mult_, true, false); // Initiate Polling

      retval = WaitForOutput(1, 0, ReadyToSendp_ ? ReadyToSendp_:&ReadyToSend_);

      if (!PollingActive_)
        PauseCalcPolling();

      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
        if (DbgPtr(false))
          DbgPtr()->LeaveLevel();
      #endif

      if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
      {
        _EnterKeyActive = true;
        emit EnableEnterKey();
      }

      return retval;
    }
    else if (CalcPollingActive() &&
             !_NoPolling && _PollServerMgr && retry_ == 0 &&
             _PollServerMgr->ResetWhenNotRequired(true)
                           ->PollServerRequired(0, true))
    {
      #if ((CALCTEST_DEBUG4b)|(CALCTEST_DEBUG1c))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("WaitForOutput:: _PollServerMgr->WaitingForServerResponse()\n");
      #endif
    
      mult_ = _PollServerMgr->WaitingForServerResponse() ? 1:0;
      _ResponseOnly = true;
      _PollServerMgr->SetStopTracking(false);
      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("_PollServerMgr->PollServerIfAlive(mult_, false, true);\n"
                                "-Called From: WaitForOutput\n");
      #endif
      _PollServerMgr->ResetServerPolling(false);
      _PollServerMgr->PollServerIfAlive(mult_, false, true); // Response Only

      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
        if (DbgPtr(false))
          DbgPtr()->LeaveLevel();
      #endif

      return WaitForOutput(1, -1, ReadyToSendp_ ? ReadyToSendp_:&ReadyToSend_);
    }
  #endif

    _IostateFileReadError = TRUE;  // version: 2.7.3
    _NoPolling = false;
    _PollServerMgr->SetAllowServerPolling(PollingAllowed_);

    #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
      if (DbgPtr(false))
        DbgPtr()->LeaveLevel();
    #endif

    #if ((CALCTEST_ALLOWPOLL)|(IDLE_RESPONSE_ONLY_POLL))
      if (CalcPollingActive() && retry_ > 0)
        return WaitForOutput(1, retry_-1, ReadyToSendp_ ? ReadyToSendp_:&ReadyToSend_);
      else
      {
        if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
        {
          _EnterKeyActive = true;
          emit EnableEnterKey();
        }
      
        return 0;
      }
    #else
      if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
      {
        _EnterKeyActive = true;
        emit EnableEnterKey();
      }
    
      return 0;
    #endif
  }
  else if (_PollServerMgr)
  {  
    _IostateFileReadError = FALSE;
    PollingAllowed_ = _PollServerMgr->ServerPollingAllowed();
    _PollServerMgr->SetAllowServerPolling(false)
                  ->ResetRetryRead();
  }

  if (retry_ < 0)
    retry_ = 0;

  do
  {
    _ReadOk = fgets(Buffer_, 256, StateFin_) != NULL;
    
    _IoState = _ReadOk ? atoi(Buffer_):LastState_;
    ClientPingReceived_ = _ReadOk && _IoState == Mcalc_IOState::CLIENTPING;
    EndBatchFileReceived_ = _ReadOk && (_IoState == Mcalc_IOState::BATCHFILE_ENDED ||
                                        _IoState == Mcalc_IOState::BATCHFILE_ENDED_ACK);
  
    if (ClientPingReceived_)
    {
      fclose(StateFin_);
      SetIoState(Mcalc_IOState::CLIENTPING_ACK, true);
      StateFin_ = fopen(IostateFile_, "r");
      _IoState = LastState_;

      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(LONG_WAIT * MILLI_DELAY_MULT);
      #else
        ::usleep(LONG_WAIT * MICRO_DELAY_MULT);
      #endif
    }
    else if (EndBatchFileReceived_ || _RequestResend)
    {    
      #if CALCTEST_DEBUG4
        DbgPtr()->EnterLevel("if (EndBatchFileReceived_ || _RequestResend)...");
      #endif
    
      _ReadOk = fgets(Buffer_, 256, StateFin_) != NULL;
      fclose(StateFin_);
      
      BatchLevel_ = _ReadOk ? atoi(Buffer_):-1;
      BatchLevelReceived_ = _ReadOk && BatchLevel_ >= 0;

      #if CALCTEST_DEBUG4
        DbgPtr()->ShowInt(EndBatchFileReceived_, "EndBatchFileReceived_");
        DbgPtr()->ShowInt(_RequestResend, "_RequestResend");
        DbgPtr()->ShowInt(BatchLevel_, "BatchLevel_");
        DbgPtr()->ShowInt(BatchLevelReceived_, "BatchLevelReceived_");
        DbgPtr()->ShowInt(_BatchExecLevel, "_BatchExecLevel(1)");
      #endif

      if (BatchLevelReceived_ && !InProgressDataFile())
      {
        if (_BatchExecLevel < BatchLevel_)
        {
          while (_BatchExecLevel < BatchLevel_)
            IncrBatchLevel();
        }
        else if (_BatchExecLevel > BatchLevel_)
        {
          while (_BatchExecLevel > BatchLevel_)
            DecrBatchLevel();
        }
      }

      #if CALCTEST_DEBUG4
        DbgPtr()->ShowInt(_BatchExecLevel, "_BatchExecLevel(2)");
        DbgPtr()->LeaveLevel();
      #endif
    }
    else
      fclose(StateFin_);
  }
  while (ClientPingReceived_);

  int WaitForServerSend_ = WaitingForServerSend(_IoState);
  int GuiReadyToSend_ = IsGuiReadyToSend(_IoState, false, false);
  int ResponseSent_ = IsCalcResponseSignals(_IoState);

  if (PrevState_ != _IoState &&
      PrevGuiRequest_ && !ResponseSent_)
    _PrevIoState = PrevState_;

  if (ResponseSent_)
    _Processing = FALSE;

  if (_ReadOk)
  {
    _FetchError = IoState() == Mcalc_IOState::ERROR_FETCH;
    _ResendSigDone = _RequestResend &&
                     _IoState != LastState_ && _IoState != resendsig_;
    _RequestResend = _RequestResend ? !_ResendSigDone:false;

    if (_RequestResend &&
        (_IoState == resendsig_ || _IoState == LastState_))
    {
      _IoState = LastState_;
      _ResendSigUnchanged = true;
    }    
  }

  if (!WaitForServerSend_)
  {
    _FatalErrCount = 0;
    _NoWaitReadCnt = 0;
  }

  ReadElapsedTimeState(ResponseSent_, Buffer_, "DEBUG WaitForOutput()");

  if (_SendResendLastSig)
  {
    _RequestResend = false;
    _ResendSigDone = false;
    _ResendLastSig = false;
    _ResendSigUnchanged = false;
    _NoWaitReadCnt = 0;
    _SendResendLastSig = false;
  }

  #if ((IMPLEMENT_KEEP_ALIVE)|(CALCTEST_ALLOWPOLL)|(IDLE_RESPONSE_ONLY_POLL))
    #if ((CALCTEST_DEBUG4)|(CALCTEST_DEBUG1c))
      if (DbgPtr(false))
      {
        _PollServerMgr->SetStopTracking(false);
        DbgPtr()->ShowStr(Buffer_, "Buffer_");
        DbgPtr()->ShowInt(ResponseSent_, "ResponseSent_");
        DbgPtr()->ShowInt(_NoPolling, "_NoPolling");
        DbgPtr()->ShowInt(_ReadOk, "_ReadOk");
        DbgPtr()->ShowStr(IoStateToStr(_IoState), "_IoState");
        DbgPtr()->ShowMessage("WaitForOutput:: _PollServerMgr->ClientReceiveRequired()\n");
      }
    #endif

    if (!_NoPolling && _ReadOk &&
        ResponseSent_ && _PollServerMgr &&
        _PollServerMgr->ResetWhenNotRequired(true)
                      ->ClientReceiveRequired())
    {
      #if ((CALCTEST_DEBUG4b)|(CALCTEST_DEBUG1c))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("WaitForOutput:: _PollServerMgr->WaitingForServerResponse()\n");
      #endif

      polldone_ = 1;
      mult_ = (!delay_ && _PollServerMgr->WaitingForServerResponse()) ? 1:0;
      _PollServerMgr->SetStopTracking(false);
      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("_PollServerMgr->ReceiveKeepAlivePoll(mult_, false);\n"
                                "-Called From WaitForOutput\n");
      #endif
      _PollServerMgr->ResetReceivePoll(false);
      _PollServerMgr->ReceiveKeepAlivePoll(mult_, false);

      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
        if (DbgPtr(false))
          DbgPtr()->LeaveLevel();
      #endif

      if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
      {
        _EnterKeyActive = true;
        emit EnableEnterKey();
      }

      return ResponseSent_;
    }

    #if ((CALCTEST_DEBUG4)|(CALCTEST_DEBUG1c))
      if (DbgPtr(false))
      {
        _PollServerMgr->SetStopTracking(false);
        DbgPtr()->ShowInt(polldone_, "polldone_");
        DbgPtr()->ShowMessage("WaitForOutput:: _PollServerMgr->PollServerRequired(0, true)\n");
      }
    #endif

    if (CalcPollingActive() &&
        !polldone_ && !_NoPolling && _ReadOk &&
        !ResponseSent_ && _PollServerMgr &&
        _PollServerMgr->ResetWhenNotRequired(true)
                      ->PollServerRequired(0, true))
    {
      #if ((CALCTEST_DEBUG4b)|(CALCTEST_DEBUG1c))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("WaitForOutput:: _PollServerMgr->WaitingForServerResponse()\n");
      #endif
    
      mult_ = _PollServerMgr->WaitingForServerResponse() ? 1:0;
      _ResponseOnly = true;
      _PollServerMgr->SetStopTracking(false);
      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("_PollServerMgr->PollServerIfAlive(mult_, false, true);\n"
                                "-Called From: WaitForOutput\n");
      #endif
      _PollServerMgr->ResetServerPolling(false);
      _PollServerMgr->PollServerIfAlive(mult_, false, true); // Response Only

      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
        if (DbgPtr(false))
          DbgPtr()->LeaveLevel();
      #endif

      if (!ResponseSent_ && retry_ > 0)
      {
        retval = WaitForOutput(1, retry_-1, ReadyToSendp_ ? ReadyToSendp_:&ReadyToSend_);

        if (ReadyToSendp_)
          ReadyToSend_ = *ReadyToSendp_;
      }
      else
      {
        retval = ResponseSent_;
        ReadyToSend_ = GuiReadyToSend_;

        if (ReadyToSendp_)
          *ReadyToSendp_ = ReadyToSend_;
      }

      if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
      {
        _EnterKeyActive = true;
        emit EnableEnterKey();
      }

      return retval;
    }
  #endif

  if (!ResponseSent_ && retry_ > 0)
  {
    if (WaitForServerSend_ && retry_ > RETRY_READ_MAX)
      retval = WaitForOutput(1, RETRY_READ_MAX-1, ReadyToSendp_ ? ReadyToSendp_:&ReadyToSend_);
    else
      retval = WaitForOutput(1, retry_-1, ReadyToSendp_ ? ReadyToSendp_:&ReadyToSend_);

    if (ReadyToSendp_)
      ReadyToSend_ = *ReadyToSendp_;
  }
  else
  {
    retval = ResponseSent_;
    ReadyToSend_ = GuiReadyToSend_;

    if (ReadyToSendp_)
      *ReadyToSendp_ = ReadyToSend_;
  }

  _NoPolling = false;
  _PollServerMgr->SetAllowServerPolling(PollingAllowed_);

  #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1c))
    if (DbgPtr(false))
    {
      DbgPtr()->ShowInt(retval, "retval");
      DbgPtr()->LeaveLevel();
    }
  #endif

  if (ReadyToSend_ && ResumePolling())
  {
    ResumeCalcPolling();
    SetResumePolling(false);
  }

  if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
  {
    _EnterKeyActive = true;
    emit EnableEnterKey();
  }
  
  return retval;
}

/****************************************************************************/
int CalcDriver::GetIoState(int Sleep_)
{
  int mult_;
  #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  #else
    // microsec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MICRO_DELAY_MULT;
  #endif

  if (Sleep_)
  {
    mult_ = (Sleep_ > 1) ? Sleep_:1;

    #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
      Sleep(mult_ * DelayLen_);
    #else
      ::usleep(mult_ * DelayLen_);
    #endif
  }

  ReadIoState();
  return IoState();
}

/****************************************************************************/
int CalcDriver::GetCalcElapsedTimeState(int Sleep_)
{
  int mult_;
  #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  #else
    // microsec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MICRO_DELAY_MULT;
  #endif

  if (Sleep_)
  {
    mult_ = (Sleep_ > 1) ? Sleep_:1;

    #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
      Sleep(mult_ * DelayLen_);
    #else
      ::usleep(mult_ * DelayLen_);
    #endif
  }

  ReadElapsedTimeState(false, NULL, "CalcDriver::GetCalcElapsedTimeState");
  return ElapsedState();
}

/****************************************************************************/
// Valid IO States for client to start sending data: 
//   IDLE_STATE         : calculator is in idle or initial starting state, set after CALC_HALT received
//   CALC_RESPONSE      : sent by calculator to indicate post processing response
//   PROCESS_DONE       : sent by calculator to indicate process acknowledgement of non-output directives
//   INPUT_REQUIRED     : sent by calculator in script program or interactive text mode to
//   OUTPUT_READY       : sent by calculator to indicate output data ready
//   CALC_ERROR         : sent by calculator to indicate calculator error from invalid input or other conditions
//   BATCHFILE_ENDED    : sent by calculator to indicate end of input batch file processing
//
// IO States that are handled elsewhere in the McalcDialog class
//   GRAPH_WAIT          : Handled by WaitForGraphOutput(), shows twirly bar on calc. input window. Sends OUTPUT_FETCHED
//   GRAPH_OUTPUT        : Handled by FetchAndGraphOutput(), sends OUTPUT_FETCHED to calc. after plotting graph
//   GRAPH_PROGRESS      : Handled by WaitForGraphProgress(), shows percentage completed of graph plot. Sends OUTPUT_FETCHED
//   PROGRESS_READY      : Handled by FetchProgress(), sends OUTPUT_FETCHED to calc. after showing progress
//   ERROR_FETCH         : Handled by _Driver object, _Driver sends CALC_ERROR signal to McalcDialog
//   BATCHFILE_ENDED     : Handled by AckEndOfBatchData(), Acknowledge BATCHFILE_ENDED signal received
//
// IO States where only calculator server program has permission to send data:
//   PROCESS             : send process input data request from client program to calculator
//   CALC_HALT           : sent by client program to tell calculator to halt execution and quit
//   OUTPUT_FETCHED      : sent by client program to indicate literal text string or graph plotting data fetched
//   PROCESS_DONE_ACK    : sent by client to acknowledge server directive execution
//   INPUT_RECEIVED      : sent by client program to indicate required input data sent to calculator
//   BREAK_PROGRAM       : sent by client program to tell calculator to break program and exit program mode
//   BATCHFILE_ENDED_ACK : sent by client to acknowledge and of batch data file signal
//
int CalcDriver::ReadIoState(int delay_, int retry_,
                            int* AltGuiStates_, int* CalcOnlyStates_)
{
  #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1d))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("CalcDriver::ReadIoState");
      DbgPtr()->ShowInt(delay_, "delay_");
      DbgPtr()->ShowInt(retry_, "retry_");
      DbgPtr()->ShowStr(IoStateToStr(_IoState), "_IoState");
      
      if (AltGuiStates_)
        DbgPtr()->ShowInt(*AltGuiStates_, "AltGuiStates_");
      if (CalcOnlyStates_)
        DbgPtr()->ShowInt(*CalcOnlyStates_, "CalcOnlyStates_");
    }
  #endif

  if (_EnterKeyActive)
  {
    _EnterKeyActive = false;
    emit DisableEnterKey();
  }

  #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  #else
    // microsec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MICRO_DELAY_MULT;
  #endif

  if (delay_)
  {
    #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
      Sleep(DelayLen_ * delay_);
    #else
      ::usleep(DelayLen_ * delay_);
    #endif
  }

  char Buffer_[256];
  const char* IostateFile_ = GetProgramIostateFilename();
  
  const int resendsig_ = Mcalc_IOState::RESENDLASTSIG;

  FILE* StateFin_ = fopen(IostateFile_, "r");

  bool ClientPingReceived_ = false;
  bool EndBatchFileReceived_ = false;
  bool BatchLevelReceived_ = false;
  bool PollingActive_ = false;
  bool PollingAllowed_ = _PollServerMgr->ServerPollingAllowed();

  #if ((IMPLEMENT_KEEP_ALIVE)|(CALCTEST_ALLOWPOLL)|(IDLE_RESPONSE_ONLY_POLL))
    int mult_;
    int polldone_ = 0;
  #endif

  int BatchLevel_ = 0;
  int LastState_ = _IoState;
  int PrevState_ = (_PrevIoState == 0) ? _IoState:_PrevIoState;
  int PrevGuiRequest_ = IsGuiReadyToSend(PrevState_, true, true);
  
  int Eof_ = 0;
  int retval = 0;
  int ErrCond_ = StateFin_ == NULL ||
                 (Eof_ = feof(StateFin_)) ||
                 ferror(StateFin_);

  if (ErrCond_ || !FileSize(StateFin_))
  {
    if (ErrCond_)
    {
      if (StateFin_)
      {      
        clearerr(StateFin_);
        if (Eof_)
          fclose(StateFin_);
          
        StateFin_ = NULL;
      }
    }
    else
      fclose(StateFin_);

    // altered: 2018-10-04
    #if LIMIT_READ_RETRIES
      if (_PollServerMgr && _IostateFileReadError &&
          !_PollServerMgr->ReAttemptFileRead())
      {
        if (_FatalErrCount < MAX_FATAL_ERRORS)
        {
          sprintf(Buffer_, ERRMSG_OPENSTATEFILE, IostateFile_);

          #if CALCTEST_SHOW_MSGBOX
            QMessageBox::critical(0, "ERROR", QString(Buffer_),
                                  QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
          #endif

          _FatalErrCount++;
          DoHalt();
          emit FatalError();
        }
        else
          exit(1);
      }
      else
        _FatalErrCount = 0;
    #endif

    if (retry_ > RETRY_ATTEMPTS / 10)
      retry_ = 1;
    else if (retry_ == 0 && ErrCond_)
      _IoState = Mcalc_IOState::IDLE_STATE;

  #if CALCTEST_ALLOWPOLL
    if (_PollServerMgr)
    {
      _PollServerMgr->SetStopTracking(false);
      _PollServerMgr->SetExitAtRetryMax(true)
                    ->SetSleepWaiting(true)
                    ->SetAllowServerPolling(true)
                    ->SetBreakOnPollError(true);
    }

    #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1d))
      if (DbgPtr(false))
      {
        _PollServerMgr->SetStopTracking(false);
        DbgPtr()->ShowMessage("ErrCond_ || !FileSize(StateFin_)\n"
                              "_PollServerMgr->PollServerRequired(0, false)\n");
        DbgPtr()->ShowInt(_NoPolling, "_NoPolling");
      }
    #endif                  

    if (!_NoPolling && _PollServerMgr && retry_ > 0 &&
        _PollServerMgr->SetAllowServerPolling(true)
                      ->ResetWhenNotRequired(true)
                      ->PollServerRequired(0, false))
    {
      #if ((CALCTEST_DEBUG4b)|(CALCTEST_DEBUG1d))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("ReadIoState:: _PollServerMgr->WaitingForServerResponse()\n");
      #endif

      PollingActive_ = CalcPollingActive();
      ResumeCalcPolling();
      
      mult_ = _PollServerMgr->WaitingForServerResponse() ? 1:0;
      _ResponseOnly = false;
      _PollServerMgr->SetAllowServerPolling(true);
      _PollServerMgr->ResetServerPolling(true);
      _PollServerMgr->PollServerIfAlive(mult_, true, false); // Initiate Polling

      retval = ReadIoState(1, 0, AltGuiStates_, CalcOnlyStates_);

      if (!PollingActive_)
        PauseCalcPolling();

      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1d))
        if (DbgPtr(false))
          DbgPtr()->LeaveLevel();
      #endif

      if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
      {
        _EnterKeyActive = true;
        emit EnableEnterKey();
      }

      return retval;
    }
    else if (CalcPollingActive() &&
             !_NoPolling && _PollServerMgr && retry_ == 0 &&
             _PollServerMgr->ResetWhenNotRequired(true)
                           ->PollServerRequired(0, true))
    {
      #if ((CALCTEST_DEBUG4b)|(CALCTEST_DEBUG1d))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("ReadIoState:: _PollServerMgr->WaitingForServerResponse()\n");
      #endif
    
      mult_ = _PollServerMgr->WaitingForServerResponse() ? 1:0;
      _ResponseOnly = true;
      _PollServerMgr->SetStopTracking(false);
      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1d))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("_PollServerMgr->PollServerIfAlive(mult_, false, true);\n"
                                "-Called From: ReadIoState\n");
      #endif
      _PollServerMgr->ResetServerPolling(false);
      _PollServerMgr->PollServerIfAlive(mult_, false, true); // Response Only

      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1d))
        if (DbgPtr(false))
          DbgPtr()->LeaveLevel();
      #endif

      return ReadIoState(1, -1, AltGuiStates_, CalcOnlyStates_);
    }
  #endif

    _IostateFileReadError = TRUE;  // version: 2.7.3
    _NoPolling = false;
    _PollServerMgr->SetAllowServerPolling(PollingAllowed_);

    #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1d))
      if (DbgPtr(false))
        DbgPtr()->LeaveLevel();
    #endif

    #if ((CALCTEST_ALLOWPOLL)|(IDLE_RESPONSE_ONLY_POLL))
      if (CalcPollingActive() && retry_ > 0)
        ReadIoState(1, retry_-1, AltGuiStates_, CalcOnlyStates_);
      else
      {
        if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
        {
          _EnterKeyActive = true;
          emit EnableEnterKey();
        }
      
        return 0;
      }
    #else
      if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
      {
        _EnterKeyActive = true;
        emit EnableEnterKey();
      }
    
      return 0;
    #endif
  }
  else if (_PollServerMgr)
  {
    _IostateFileReadError = FALSE;
    PollingAllowed_ = _PollServerMgr->ServerPollingAllowed();
    _PollServerMgr->SetAllowServerPolling(false)
                  ->ResetRetryRead();
  }

  if (retry_ < 0)
    retry_ = 0;

  do
  {
    _ReadOk = fgets(Buffer_, 256, StateFin_) != NULL;

    _IoState = _ReadOk ? atoi(Buffer_):LastState_;
    ClientPingReceived_ = _ReadOk && _IoState == Mcalc_IOState::CLIENTPING;
    EndBatchFileReceived_ = _ReadOk && (_IoState == Mcalc_IOState::BATCHFILE_ENDED ||
                                        _IoState == Mcalc_IOState::BATCHFILE_ENDED_ACK);
  
    if (ClientPingReceived_)
    {
      fclose(StateFin_);
      SetIoState(Mcalc_IOState::CLIENTPING_ACK, true);
      StateFin_ = fopen(IostateFile_, "r");
      _IoState = LastState_;

      #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
        Sleep(LONG_WAIT * MILLI_DELAY_MULT);
      #else
        ::usleep(LONG_WAIT * MICRO_DELAY_MULT);
      #endif
    }
    else if (EndBatchFileReceived_ || _RequestResend)
    {
      #if CALCTEST_DEBUG4
        DbgPtr()->EnterLevel("if (EndBatchFileReceived_ || _RequestResend)...");
      #endif
    
      _ReadOk = fgets(Buffer_, 256, StateFin_) != NULL;
      fclose(StateFin_);

      BatchLevel_ = _ReadOk ? atoi(Buffer_):-1;
      BatchLevelReceived_ = _ReadOk && BatchLevel_ >= 0;

      #if CALCTEST_DEBUG4
        DbgPtr()->ShowInt(EndBatchFileReceived_, "EndBatchFileReceived_");
        DbgPtr()->ShowInt(_RequestResend, "_RequestResend");
        DbgPtr()->ShowInt(BatchLevel_, "BatchLevel_");
        DbgPtr()->ShowInt(BatchLevelReceived_, "BatchLevelReceived_(1)");
        DbgPtr()->ShowInt(_BatchExecLevel, "_BatchExecLevel");
      #endif

      if (BatchLevelReceived_ && !InProgressDataFile())
      {
        if (_BatchExecLevel < BatchLevel_)
        {
          while (_BatchExecLevel < BatchLevel_)
            IncrBatchLevel();
        }
        else if (_BatchExecLevel > BatchLevel_)
        {
          while (_BatchExecLevel > BatchLevel_)
            DecrBatchLevel();
        }
      }

      #if CALCTEST_DEBUG4
        DbgPtr()->ShowInt(_BatchExecLevel, "_BatchExecLevel(2)");
        DbgPtr()->LeaveLevel();
      #endif
    }
    else
      fclose(StateFin_);
  }
  while (ClientPingReceived_);

  int WaitForServerSend_ = WaitingForServerSend(_IoState);
  int GuiReadyToSend_ = IsGuiReadyToSend(_IoState, true, false);

  // Gui I/O states that need to be processed by the client program,
  // so Client Gui is not ready to send user commands to the server.
  int AltGuiCond_ = IsGuiNeedToProcessSignals(_IoState);

  // I/O states intended for calculator server processing only
  int CalcOnlyCond_ = IsCalcOnlySignals(_IoState);

  // Response by calculator is GuiReadyToSend OR AltGuiCond conditions
  int ResponseSent_ = IsCalcResponseSignals(_IoState);
  int ProcessAckDone_ = _AcceptProcessAckDone ? IsProcessAckDone(_IoState):FALSE;

  if (AltGuiStates_)
    *AltGuiStates_ = AltGuiCond_;

  if (CalcOnlyStates_)
    *CalcOnlyStates_ = CalcOnlyCond_;

  if (PrevState_ != _IoState &&
      PrevGuiRequest_ && !ResponseSent_)
    _PrevIoState = PrevState_;

  if (_ReadOk)
  {
    _FetchError = IoState() == Mcalc_IOState::ERROR_FETCH;
    _ResendSigDone = _RequestResend &&
                     _IoState != LastState_ && _IoState != resendsig_;
    _RequestResend = _RequestResend ? !_ResendSigDone:false;

    if (_RequestResend &&
        (_IoState == resendsig_ || _IoState == LastState_))
    {
      _IoState = LastState_;
      _ResendSigUnchanged = true;
    }
  }

  if (!WaitForServerSend_)
  {
    _FatalErrCount = 0;
    _NoWaitReadCnt = 0;
  }

  ReadElapsedTimeState(ResponseSent_, Buffer_, "DEBUG ReadIoState()");

  if (_SendResendLastSig)
  {
    _RequestResend = false;
    _ResendSigDone = false;
    _ResendLastSig = false;
    _ResendSigUnchanged = false;
    _NoWaitReadCnt = 0;
    _SendResendLastSig = false;
  }

  #if ((IMPLEMENT_KEEP_ALIVE)|(CALCTEST_ALLOWPOLL)|(IDLE_RESPONSE_ONLY_POLL))
    #if ((CALCTEST_DEBUG4)|(CALCTEST_DEBUG1d))
      if (DbgPtr(false))
      {
        _PollServerMgr->SetStopTracking(false);
        DbgPtr()->ShowStr(Buffer_, "Buffer_");
        DbgPtr()->ShowInt(GuiReadyToSend_, "GuiReadyToSend_");
        DbgPtr()->ShowInt(AltGuiCond_, "AltGuiCond_");
        DbgPtr()->ShowInt(CalcOnlyCond_, "CalcOnlyCond_");
        DbgPtr()->ShowInt(ResponseSent_, "ResponseSent_");
      
        DbgPtr()->ShowInt(_NoPolling, "_NoPolling");
        DbgPtr()->ShowInt(_ReadOk, "_ReadOk");
        DbgPtr()->ShowStr(IoStateToStr(_IoState), "_IoState");
        DbgPtr()->ShowMessage("ReadIoState:: _PollServerMgr->ClientReceiveRequired()\n");
      }
    #endif

    if (!_NoPolling && _ReadOk &&
        !CalcOnlyCond_ && _PollServerMgr &&
        _PollServerMgr->ResetWhenNotRequired(true)
                      ->ClientReceiveRequired())
    {
      #if ((CALCTEST_DEBUG4b)|(CALCTEST_DEBUG1d))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("ReadIoState:: _PollServerMgr->WaitingForServerResponse()\n");
      #endif

      polldone_ = 1;
      mult_ = _PollServerMgr->WaitingForServerResponse() ? 1:0;
      _PollServerMgr->SetStopTracking(false);
      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1d))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("_PollServerMgr->ReceiveKeepAlivePoll(mult_, false);\n"
                                "-Called From: ReadIoState\n");
      #endif
      _PollServerMgr->ResetReceivePoll(false);
      _PollServerMgr->ReceiveKeepAlivePoll(mult_, false);

      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1d))
        if (DbgPtr(false))
          DbgPtr()->LeaveLevel();
      #endif

      if (!(ResponseSent_ ||
            (_AcceptProcessAckDone && ProcessAckDone_)) && retry_ > 0)
        retval = ReadIoState(1, retry_-1,
                             AltGuiStates_, CalcOnlyStates_);
      else
        retval = GuiReadyToSend_;

      _AcceptProcessAckDone = false;

      if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
      {
        _EnterKeyActive = true;
        emit EnableEnterKey();
      }
      
      return retval;
    }

    #if ((CALCTEST_DEBUG4)|(CALCTEST_DEBUG1d))
      if (DbgPtr(false))
      {
        _PollServerMgr->SetStopTracking(false);
        DbgPtr()->ShowInt(polldone_, "polldone_");
        DbgPtr()->ShowMessage("ReadIoState:: _PollServerMgr->PollServerRequired(0, true)\n");
      }
    #endif

    if (CalcPollingActive() &&
        !polldone_ && !_NoPolling &&  _ReadOk &&
        CalcOnlyCond_ && _PollServerMgr &&
        _PollServerMgr->ResetWhenNotRequired(true)
                      ->PollServerRequired(0, true))
    {
      #if ((CALCTEST_DEBUG4b)|(CALCTEST_DEBUG1d))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("ReadIoState:: _PollServerMgr->WaitingForServerResponse()\n");
      #endif
    
      mult_ = _PollServerMgr->WaitingForServerResponse() ? 1:0;
      _ResponseOnly = true;
      _PollServerMgr->SetStopTracking(false);
      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1d))
        if (DbgPtr(false))
          DbgPtr()->ShowMessage("_PollServerMgr->PollServerIfAlive(mult_, false, true);\n"
                                "-Called From: ReadIoState\n");
      #endif
      _PollServerMgr->ResetServerPolling(false);
      _PollServerMgr->PollServerIfAlive(mult_, false, true); // Response Only

      #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1d))
        if (DbgPtr(false))
          DbgPtr()->LeaveLevel();
      #endif

      if (!(ResponseSent_ ||
            (_AcceptProcessAckDone && ProcessAckDone_)) && retry_ > 0)
        retval = ReadIoState(1, retry_-1,
                             AltGuiStates_, CalcOnlyStates_);
      else
        retval = GuiReadyToSend_;

      _AcceptProcessAckDone = false;

      if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
      {
        _EnterKeyActive = true;
        emit EnableEnterKey();
      }
      
      return retval;
    }
  #endif

  if (!(ResponseSent_ ||
        (_AcceptProcessAckDone && ProcessAckDone_)) && retry_ > 0)
  {
    if (WaitForServerSend_ && retry_ > RETRY_READ_MAX)
      retval = ReadIoState(1, RETRY_READ_MAX-1,
                           AltGuiStates_, CalcOnlyStates_);
    else
      retval = ReadIoState(1, retry_-1,
                           AltGuiStates_, CalcOnlyStates_);
  }
  else
    retval = GuiReadyToSend_;

  _NoPolling = false;
  _PollServerMgr->SetAllowServerPolling(PollingAllowed_);

  #if ((CALCTEST_DEBUG3)|(CALCTEST_DEBUG1d))
    if (DbgPtr(false))
    {
      DbgPtr()->ShowInt(retval, "retval");
      DbgPtr()->LeaveLevel();
    }
  #endif

  if (retval && ResumePolling())
  {
    ResumeCalcPolling();
    SetResumePolling(false);
  }

  _AcceptProcessAckDone = false;

  if (!_ReadIoStateTimer.isActive() && !_EnterKeyActive)
  {
    _EnterKeyActive = true;
    emit EnableEnterKey();
  }
  
  return retval;
}

/****************************************************************************/
bool CalcDriver::SetPollState(int State_)
{
  FILE* Fptr_ = GetPollStateFile(1);
  
  if (Fptr_)
  {
    fprintf(Fptr_, "%d", State_);
    ClosePollStateFile();

    return true;
  }

  return false;
}

/*****************************************************************************/
void CalcDriver::SetSuspendPolling(bool flag_)
{
  _SuspendPolling = flag_;
}

/*****************************************************************************/
bool CalcDriver::SuspendPolling()
{
  return _SuspendPolling;
}

/*****************************************************************************/
void CalcDriver::SetResumePolling(bool flag_)
{
  _ResumePolling = flag_;
}

/*****************************************************************************/
bool CalcDriver::ResumePolling()
{
  return _ResumePolling;
}

/*****************************************************************************/
void CalcDriver::PauseCalcPolling()
{
  if (_SuspendPolling && !_CalcPollingInactive)
  {
    _CalcPollingInactive = TRUE;
    SetPollState(Mcalc_PollState::POLLING_PAUSED);
  }
}

/*****************************************************************************/
void CalcDriver::ResumeCalcPolling()
{
  if (_ResumePolling && _CalcPollingInactive)
  {
    _CalcPollingInactive = FALSE;
    SetPollState(Mcalc_PollState::IDLE);
  }
}

/*****************************************************************************/
bool CalcDriver::CalcPollingActive()
{
  int Pstate_ = _PollServerMgr->ReadPollState(0);

  if (Pstate_ == Mcalc_PollState::POLLING_PAUSED && !_CalcPollingInactive)
  {
    SetSuspendPolling(true);

    if (SuspendPolling())
    {
      PauseCalcPolling();
      SetSuspendPolling(false);
      SetResumePolling(true);
    }
  }

  return !_CalcPollingInactive;
}

/****************************************************************************/
bool CalcDriver::IsClientStates(int State_)
{
  return
  (
    State_ == Mcalc_IOState::CALC_HALT ||
    State_ == Mcalc_IOState::PROCESS ||
    State_ == Mcalc_IOState::OUTPUT_FETCHED ||
    State_ == Mcalc_IOState::INPUT_RECEIVED ||
    State_ == Mcalc_IOState::BREAK_PROGRAM ||
    State_ == Mcalc_IOState::GRAPH_WAIT_ACK ||
    State_ == Mcalc_IOState::GRAPH_PROGRESS_ACK ||
    State_ == Mcalc_IOState::INFILE_PROGRESS_ACK ||
    State_ == Mcalc_IOState::BATCHFILE_ENDED_ACK ||
    State_ == Mcalc_IOState::PROCESS_DONE_ACK ||
    State_ == Mcalc_IOState::MCALC_SPAWNED
  );
}

/****************************************************************************/
bool CalcDriver::IsServerStates(int State_)
{
  return
  (
    State_ == Mcalc_IOState::OUTPUT_READY ||
    State_ == Mcalc_IOState::CALC_RESPONSE ||
    State_ == Mcalc_IOState::INPUT_REQUIRED ||
    State_ == Mcalc_IOState::IDLE_STATE ||
    State_ == Mcalc_IOState::GRAPH_OUTPUT ||
    State_ == Mcalc_IOState::GRAPH_WAIT ||
    State_ == Mcalc_IOState::CALC_ERROR ||
    State_ == Mcalc_IOState::PROGRESS_READY ||
    State_ == Mcalc_IOState::ERROR_FETCH ||
    State_ == Mcalc_IOState::GRAPH_PROGRESS ||
    State_ == Mcalc_IOState::BATCHFILE_ENDED ||
    State_ == Mcalc_IOState::SPAWN_NEW_MCALC ||
    State_ == Mcalc_IOState::PROCESS_DONE
  );
}

/****************************************************************************/
bool CalcDriver::IsCalcResponseSignals(int State_)
{
  bool CalcResponseSignals_ =
      (State_ == Mcalc_IOState::CALC_HALT)        ||
      (State_ == Mcalc_IOState::CALC_ERROR)       ||
      (State_ == Mcalc_IOState::ERROR_FETCH)      ||
      (State_ == Mcalc_IOState::CALC_RESPONSE)    ||
      (State_ == Mcalc_IOState::PROCESS_DONE)     ||
      (State_ == Mcalc_IOState::OUTPUT_READY)     ||
      (State_ == Mcalc_IOState::PROGRESS_READY)   ||
      (State_ == Mcalc_IOState::GRAPH_PROGRESS)   ||
      (State_ == Mcalc_IOState::GRAPH_OUTPUT)     ||
      (State_ == Mcalc_IOState::GRAPH_WAIT)       ||
      (State_ == Mcalc_IOState::SPAWN_NEW_MCALC)  ||
      (State_ == Mcalc_IOState::BATCHFILE_ENDED)  ||
      (State_ == Mcalc_IOState::BREAK_PROGRAM)    ||
      (State_ == Mcalc_IOState::INPUT_REQUIRED);

  return CalcResponseSignals_;
}

/****************************************************************************/
bool CalcDriver::IsGuiReadyToSend(int State_,
                                  bool IncludeIdle_, bool IncludeErrorFetch_)
{
  bool GuiReadyToSend_ = ((IncludeIdle_ &&
                           State_ == Mcalc_IOState::IDLE_STATE) ||
                          State_ == Mcalc_IOState::CALC_RESPONSE ||
                          State_ == Mcalc_IOState::PROCESS_DONE ||
                          State_ == Mcalc_IOState::INPUT_REQUIRED ||
                          State_ == Mcalc_IOState::OUTPUT_READY ||
                          State_ == Mcalc_IOState::SPAWN_NEW_MCALC ||
                          State_ == Mcalc_IOState::BATCHFILE_ENDED ||
                          State_ == Mcalc_IOState::CALC_ERROR ||
                          (IncludeErrorFetch_ &&
                           State_ == Mcalc_IOState::ERROR_FETCH));

  return GuiReadyToSend_;
}

/****************************************************************************/
bool CalcDriver::IsProcessAckDone(int State_)
{
  bool ProcessAckDone_ = (State_ == Mcalc_IOState::OUTPUT_FETCHED ||
                          State_ == Mcalc_IOState::INPUT_RECEIVED ||
                          State_ == Mcalc_IOState::BREAK_PROGRAM ||
                          State_ == Mcalc_IOState::BATCHFILE_ENDED_ACK ||
                          State_ == Mcalc_IOState::PROCESS_DONE_ACK ||
                          State_ == Mcalc_IOState::MCALC_SPAWNED);

  return ProcessAckDone_;
}

/****************************************************************************/
bool CalcDriver::IsGuiNeedToProcessSignals(int State_)
{
  bool GuiNeedToProcess_ =
      (State_ == Mcalc_IOState::GRAPH_WAIT)       ||
      (State_ == Mcalc_IOState::GRAPH_OUTPUT)     ||
      (State_ == Mcalc_IOState::GRAPH_PROGRESS)   ||
      (State_ == Mcalc_IOState::PROGRESS_READY)   ||
      (State_ == Mcalc_IOState::ERROR_FETCH);

  return GuiNeedToProcess_;
}

/****************************************************************************/
bool CalcDriver::IsCalcOnlySignals(int State_)
{
  bool CalcOnlySignals_ =
      (State_ == Mcalc_IOState::PROCESS)          ||
      (State_ == Mcalc_IOState::CALC_HALT)        ||
      (State_ == Mcalc_IOState::OUTPUT_FETCHED)   ||
      (State_ == Mcalc_IOState::PROCESS_DONE_ACK) ||
      (State_ == Mcalc_IOState::INPUT_RECEIVED)   ||
      (State_ == Mcalc_IOState::BREAK_PROGRAM);

  return CalcOnlySignals_;
}

/****************************************************************************/
bool CalcDriver::WaitingForServerSend(int State_)
{
  return
  (
    State_ == Mcalc_IOState::PROCESS             ||
    State_ == Mcalc_IOState::INPUT_RECEIVED      ||
    State_ == Mcalc_IOState::OUTPUT_FETCHED      ||
    State_ == Mcalc_IOState::GRAPH_WAIT_ACK      ||
    State_ == Mcalc_IOState::GRAPH_PROGRESS_ACK  ||
    State_ == Mcalc_IOState::INFILE_PROGRESS_ACK ||
    State_ == Mcalc_IOState::BATCHFILE_ENDED_ACK ||
    State_ == Mcalc_IOState::PROCESS_DONE_ACK ||
    State_ == Mcalc_IOState::MCALC_SPAWNED
  );
}

/****************************************************************************/
bool CalcDriver::LongWaitForServerSend(int State_)
{
  return
  (
    State_ == Mcalc_IOState::PROCESS             ||
    State_ == Mcalc_IOState::INPUT_RECEIVED      ||
    State_ == Mcalc_IOState::GRAPH_WAIT_ACK      ||
    State_ == Mcalc_IOState::GRAPH_PROGRESS_ACK  ||
    State_ == Mcalc_IOState::INFILE_PROGRESS_ACK ||
    State_ == Mcalc_IOState::PROCESS_DONE_ACK
  );
}

/****************************************************************************/
bool CalcDriver::IsIntermediateStates(int State_)
{
  // Client acknowledgement signals sent in the middle of a process
  // comprising a series of further server/client signals that is expected
  return
  (
    State_ == Mcalc_IOState::ERROR_FETCH ||
    State_ == Mcalc_IOState::GRAPH_WAIT_ACK ||
    State_ == Mcalc_IOState::GRAPH_PROGRESS_ACK
  );
}

/****************************************************************************/
void CalcDriver::SetFromMethod(int Method_, bool SetResponse_, bool ResponseOnly_)
{
  _FromMethod = Method_;
  
  if (SetResponse_)
    _ResponseOnly = ResponseOnly_;
}

/****************************************************************************/
void CalcDriver::FetchGraphOperationData()
{
  #if CALCDRIVER_TEXT_VERSION
    if (_ProgramExited)
      exit(0);
  #endif

  #if CALCTEST_DEBUG6a
    if (DbgPtr(false))
      DbgPtr()->EnterLevel("CalcDriver::FetchGraphOperationData()");
  #endif
  
  char* buffer = new char[128];
  ::memset(buffer, 0, 128);
  FILE* fp = fopen(GRAPH_OPERATION_FILE, "r");
  _ReadOk = false;
  
  if (fp && buffer)
    _GrReadOk = fgets(buffer, 128, fp) != NULL;

  QString Output_;
  QString CalcOut_ = _GrReadOk ? buffer:"";
  _GraphOpTypeSet = _GrReadOk && buffer &&
                    strlen(buffer) && isdigit(buffer[0]);

  #if CALCTEST_DEBUG6a
    if (DbgPtr(false))
    {
      DbgPtr()->ShowStr(buffer, "buffer");
      DbgPtr()->ShowInt(_GrReadOk, "_GrReadOk(1)");
      DbgPtr()->ShowInt(_GraphOpTypeSet, "_GraphOpTypeSet(1)");
    }
  #endif

  if (_GraphOpTypeSet)
  {
    if (_GrReadBuffer)
      delete[] _GrReadBuffer;
    _GrReadBuffer = buffer;

    _GraphPlotOpType = atoi(buffer);    
    _GrReadOk = fgets(buffer, 128, fp) != NULL;
    _GraphOpTypeSet = _GrReadOk && buffer &&
                      strlen(buffer) && isdigit(buffer[0]);

    if (_GraphOpTypeSet)
    {
      _GraphType = atoi(buffer);
      _GrReadOk = fgets(buffer, 128, fp) != NULL;
      _GraphOpTypeSet = _GrReadOk && buffer &&
                        strlen(buffer) && isdigit(buffer[0]);

      if (_GraphOpTypeSet)
      {
        _GraphSubtype = atoi(buffer);

        _GrOperationFetched = true;
        _GrOperationDataShown = false;
        _GraphOperationOutDone = false;
      }
    }

    fclose(fp);
  }
    
  SendGraphWaitFetched();

  #if CALCTEST_DEBUG6a
    if (DbgPtr(false))
    {
      DbgPtr()->ShowMessage("SendGraphWaitFetched()\n");
      DbgPtr()->ShowInt(_GrOperationFetched, "_GrOperationFetched");
      DbgPtr()->ShowInt(_GrOperationDataShown, "_GrOperationDataShown");
      DbgPtr()->ShowInt(_GraphOperationOutDone, "_GraphOperationOutDone");
      DbgPtr()->ShowInt(_GraphPlotOpType, "_GraphPlotOpType");
      DbgPtr()->ShowInt(_GraphType, "_GraphType");
      DbgPtr()->ShowInt(_GraphSubtype, "_GraphSubtype");
      DbgPtr()->ShowInt(_GrReadOk, "_GrReadOk(2)");
      DbgPtr()->ShowInt(_GraphOpTypeSet, "_GraphOpTypeSet(2)");
    }
  #endif

  #if CALCTEST_DEBUG6a
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void CalcDriver::FetchGraphProgressData()
{
  char* OutFname = _GrReadOk ? strtok(_GrReadBuffer, ":"):NULL;
  char* numer = (_GrReadOk && OutFname) ? strtok(NULL, "/"):NULL;
  char* denom = NULL;

  _GrProgressFetched = false;
  _GrOutputFnameSet = OutFname && strlen(OutFname) && strstr(OutFname, ".out");
  _GrDenomSet = _GrNumerSet = numer && strlen(numer) && isdigit(numer[0]);

  #if CALCTEST_DEBUG6c
    if (DbgPtr(false))
    {
      DbgPtr()->ShowInt(_GrProgressFetched, "_GrProgressFetched");
      DbgPtr()->ShowInt(_GrOutputFnameSet, "_GrOutputFnameSet");
      DbgPtr()->ShowInt(_GrDenomSet, "_GrDenomSet");
      DbgPtr()->EnterLevel("CalcDriver::FetchGraphProgressData()");
    }
  #endif
  
  if (_GrOutputFnameSet && !_GrOutputDataFnameDone &&
      (!_GrOutputDataFname || strcmp(_GrOutputDataFname, OutFname) != 0))
  {
    delete[] _GrOutputDataFname;
    _GrOutputDataFname = strcpy(new char[strlen(OutFname) + 1], OutFname);
  }

  if (_GrNumerSet)
  {
    _GrNumer = atoi(numer);
    denom = strtok(NULL, "/");
    _GrDenomSet = denom && strlen(denom) && isdigit(denom[0]);
    
    if (_GrDenomSet)
    {
      _GrDenom = atoi(denom);
      
      _GrProgressFetched = true;
      _GrProgressDataShown = false;
      _GraphProgressOutDone = false;
    }
  }

  if (_GrReadBuffer &&
      _GraphProgressOutDone && _GrProgressDataShown)
  {
    delete[] _GrReadBuffer;
    _GrReadBuffer = NULL;
  }

  #if CALCTEST_DEBUG6c
    if (DbgPtr(false))
    {
      DbgPtr()->ShowInt(_GrOutputFnameSet, "_GrOutputFnameSet(2)");
      DbgPtr()->ShowInt(_GrOutputDataFnameDone, "_GrOutputDataFnameDone");
      DbgPtr()->ShowStr(_GrOutputDataFname, "_GrOutputDataFname");
      DbgPtr()->ShowInt(_GrNumerSet, "_GrNumerSet");
      DbgPtr()->ShowInt(_GrNumer, "_GrNumer");
      DbgPtr()->ShowInt(_GrDenomSet, "_GrDenomSet");
      DbgPtr()->ShowInt(_GrDenom, "_GrDenom");
      DbgPtr()->ShowStr(_GrReadBuffer, "_GrReadBuffer");
      DbgPtr()->ShowInt(_GrProgressFetched, "_GrProgressFetched");
      DbgPtr()->ShowInt(_GrProgressDataShown, "_GrProgressDataShown");
      DbgPtr()->ShowInt(_GraphProgressOutDone, "_GraphProgressOutDone");
    }
  #endif
}

/****************************************************************************/
void CalcDriver::FetchProgressData()
{
  #if CALCTEST_DEBUG7
    if (DbgPtr(false))
      DbgPtr()->EnterLevel("FetchProgressData");
  #endif

  const char* InFname = _ReadOk ? strtok(_ReadBuffer, ">"):NULL;
  const char* OutFname = (_ReadOk && InFname) ? strtok(NULL, ":"):NULL;
  const char* numer = (_ReadOk && OutFname) ? strtok(NULL, "/"):NULL;
  const char* denom = NULL;
  
  _InputFnameSet = InFname && strlen(InFname) && strstr(InFname, ".in");
  _OutputFnameSet = OutFname && strlen(OutFname) && strstr(OutFname, ".out");
  _DenomSet = _NumerSet = numer && strlen(numer) && isdigit(numer[0]);
  
  if (_InputFnameSet && !_InputDataFnameDone &&
      (!_InputDataFname || strcmp(_InputDataFname, InFname) != 0))
  {
    delete[] _InputDataFname;
    _InputDataFname = strcpy(new char[strlen(InFname) + 1], InFname);
  }

  if (_OutputFnameSet && !_OutputDataFnameDone &&
      (!_OutputDataFname || strcmp(_OutputDataFname, OutFname) != 0))
  {
    delete[] _OutputDataFname;
    _OutputDataFname = strcpy(new char[strlen(OutFname) + 1], OutFname);
  }

  if (_NumerSet)
  {
    if (!_ProgressBarInitDone)
      _InitProgressBar = numer[0] == '0' && strlen(numer) == 1;
    
    _Numer = atoi(numer);
    denom = strtok(NULL, "/");
    _DenomSet = denom && strlen(denom) && isdigit(denom[0]);
    
    if (_DenomSet)
    {
      _Denom = atoi(denom);
      
      if (!_ProgressBarEndDone)
        _EndProgressBar = _Numer == _Denom;
    }
  }
        
  if (!_NumerSet && !_ProgressBarInitDone)
    _InitProgressBar = false;
  
  if (!_DenomSet && !_ProgressBarEndDone)
    _EndProgressBar = false;
  
  delete[] _ReadBuffer;
  _ReadBuffer = NULL;

  #if CALCTEST_DEBUG7
    if (DbgPtr(false))
    {
      DbgPtr()->ShowStr(_InputDataFname, "_InputDataFname");
      DbgPtr()->ShowStr(_OutputDataFname, "_OutputDataFname");
      DbgPtr()->ShowInt(_Numer, "_Numer");
      DbgPtr()->ShowInt(_Denom, "_Denom");

      DbgPtr()->ShowInt(_ProgressBarInitDone, "_ProgressBarInitDone");
      DbgPtr()->ShowInt(_ProgressBarEndDone, "_ProgressBarEndDone");
      DbgPtr()->ShowInt(_InitProgressBar, "_InitProgressBar");
      DbgPtr()->ShowInt(_EndProgressBar, "_EndProgressBar");
      
      DbgPtr()->ShowInt(_InputFnameSet, "_InputFnameSet");
      DbgPtr()->ShowInt(_OutputFnameSet, "_OutputFnameSet");
      DbgPtr()->ShowInt(_NumerSet, "_NumerSet");
      DbgPtr()->ShowInt(_DenomSet, "_DenomSet");
    
      DbgPtr()->LeaveLevel();
    }
  #endif
}

/****************************************************************************/
void CalcDriver::RetryReadIoState()
{
  // execute call to method:
  // int WaitForOutput(int delay_, int retry_, int* ReadyToSendp_)
  IoStateTimer(STOP);

  bool SigReset_ = false;
  bool NoWait_ = !_ResendSigUnchanged &&
                 !_RequestResend && !_ResendSigDone;

  // millisec sleep
  int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;

  if (!_OutputReady)
  {
    _OutputReady = WaitForOutput(0, 0);

    if (_OutputReady)
    {
      if (_ResendSigDone &&
          IsGuiReadyToSend(_IoState, true, true))
      {
        _PrevIoState = _IoState;
        SigReset_ = true;
      }
    
      _RetryReadIoDone = true;
      _Processing = true;
      _RequestResend = false;
      _ResendSigDone = false;
      _ResendLastSig = false;
      _ResendSigUnchanged = false;
      _NoWaitReadCnt = 0;

      if (_SavedState == 0)
        _GraphProgressOutDone =
        _ProgressOutDone =
        _CalcOutDone = false;

      ReadFromOutput(true, _IoState, SigReset_ ? 0:_PrevIoState);
    }
    else
    {
      if (!EmptyBuffer())
      {
        DelayLen_ *= _WaitCycle--;
        if (_WaitCycle == 0)
          _WaitCycle = 2;
      }

      if ((NoWait_ && _NoWaitReadCnt < RETRY_READ_MAX) ||
          LongWaitForServerSend(_IoState))
        _NoWaitReadCnt++;
      else
      {
        if (!_RequestResend && !_ResendSigDone && !_ResendSigUnchanged)
        {
          _FatalErrCount = 0;
          _NoWaitReadCnt = 0;
          _ResendLastSig = true;

          SetIoState(Mcalc_IOState::RESENDLASTSIG, true);
        }
        else if (_RequestResend &&
                 !_ResendSigDone && _ResendSigUnchanged &&
                 (_NoWaitReadCnt < RETRY_READ_MAX || LongWaitForServerSend(_IoState)))
        {
          _FatalErrCount = 0;
          _NoWaitReadCnt++;
        }
        else if (_ResendSigDone ||
                 (_RequestResend && _ResendSigUnchanged && _NoWaitReadCnt >= RETRY_READ_MAX))
        {
          if (_FatalErrCount >= MAX_FATAL_ERRORS)
          {
            exit(1);
            return;
          }

          #if CALCTEST_SHOW_MSGBOX
            QMessageBox::critical(0, tr("Fatal Error"),
                                     tr("Server signal resend error"),
                                     tr("Quit"));
          #endif

          _FatalErrCount++;
          DoHalt();
          emit FatalError();
        }
      }

      _RetryReadIoDone = false;
      IoStateTimer(START, EmptyBuffer() ? LONG_WAIT:DelayLen_);
    }
  }
  else
  {
    if (_ResendSigDone &&
        IsGuiReadyToSend(_IoState, true, true))
    {
      _PrevIoState = _IoState;
      SigReset_ = true;
    }

    _RetryReadIoDone = true;
    _Processing = true;
    _RequestResend = false;
    _ResendSigDone = false;
    _ResendLastSig = false;
    _ResendSigUnchanged = false;
    _NoWaitReadCnt = 0;

    if (_SavedState == 0)
      _GraphProgressOutDone =
      _ProgressOutDone =
      _CalcOutDone = false;

    ReadFromOutput(true, _IoState, SigReset_ ? 0:_PrevIoState);
  }
}

/****************************************************************************/
bool CalcDriver::IsMatchingStates(int PrevState_, int CurState_,
                                  bool& DoWait_, bool& NoWait_)
{
  int WaitForServerSend_ = WaitingForServerSend(_IoState);
  
  bool ErrState_ = CurState_ == Mcalc_IOState::CALC_ERROR;
  bool InBatchFile_ = InBatchFile();
  
  bool CritStates_ = CurState_ == Mcalc_IOState::CALC_HALT ||
                     CurState_ == Mcalc_IOState::IDLE_STATE;

  bool ErrorFlip_ = (PrevState_ == Mcalc_IOState::CALC_ERROR &&
                     CurState_ == Mcalc_IOState::ERROR_FETCH) ||
                    (CurState_ == Mcalc_IOState::CALC_ERROR &&
                     PrevState_ == Mcalc_IOState::ERROR_FETCH);

  bool CalcResponseSignals_ = CurState_ == Mcalc_IOState::INPUT_REQUIRED ||
                              CurState_ == Mcalc_IOState::OUTPUT_READY ||
                              CurState_ == Mcalc_IOState::CALC_RESPONSE ||
                              CurState_ == Mcalc_IOState::PROCESS_DONE;

  bool InBatchCond_ =
        (ErrorFlip_ ||
         (InBatchFile_ &&
          (((PrevState_ == Mcalc_IOState::INPUT_RECEIVED ||
             PrevState_ == Mcalc_IOState::OUTPUT_FETCHED) &&
            CalcResponseSignals_) ||

           ((PrevState_ == Mcalc_IOState::BREAK_PROGRAM ||
             PrevState_ == Mcalc_IOState::BATCHFILE_ENDED_ACK) &&
            (CalcResponseSignals_ ||
             CurState_ == Mcalc_IOState::PROCESS)) ||

           (((PrevState_ == Mcalc_IOState::INPUT_REQUIRED ||
              PrevState_ == Mcalc_IOState::GRAPH_WAIT) ||
             (PrevState_ == Mcalc_IOState::CALC_RESPONSE ||
              PrevState_ == Mcalc_IOState::CALC_ERROR ||
              PrevState_ == Mcalc_IOState::ERROR_FETCH)) &&
            CurState_ == Mcalc_IOState::INPUT_RECEIVED) ||

           (PrevState_ == Mcalc_IOState::GRAPH_WAIT &&
            CurState_ == Mcalc_IOState::INPUT_RECEIVED) ||
           (PrevState_ == Mcalc_IOState::GRAPH_WAIT &&
            CurState_ == Mcalc_IOState::GRAPH_WAIT_ACK) ||
           (PrevState_ == Mcalc_IOState::GRAPH_PROGRESS &&
            CurState_ == Mcalc_IOState::GRAPH_PROGRESS_ACK) ||
              
           (PrevState_ == Mcalc_IOState::PROGRESS_READY &&
            CurState_ == Mcalc_IOState::INFILE_PROGRESS_ACK) ||

           (PrevState_ == Mcalc_IOState::PROCESS_DONE &&
            CurState_ == Mcalc_IOState::PROCESS_DONE_ACK) ||

           (PrevState_ == Mcalc_IOState::PROCESS_DONE_ACK &&
            CurState_ == Mcalc_IOState::PROCESS_DONE) ||

           ((PrevState_ == Mcalc_IOState::OUTPUT_READY ||
             PrevState_ == Mcalc_IOState::GRAPH_OUTPUT) &&
            (CurState_ == Mcalc_IOState::OUTPUT_FETCHED ||
             CurState_ == Mcalc_IOState::PROCESS)) ||

           (PrevState_ == Mcalc_IOState::PROCESS &&
            CurState_ == Mcalc_IOState::PROCESS_DONE) ||

           (PrevState_ == Mcalc_IOState::BATCHFILE_ENDED &&
            (CurState_ == Mcalc_IOState::BATCHFILE_ENDED_ACK ||
             CurState_ == Mcalc_IOState::PROCESS)) ||

           (PrevState_ == Mcalc_IOState::SPAWN_NEW_MCALC &&
            (CurState_ == Mcalc_IOState::MCALC_SPAWNED ||
             CurState_ == Mcalc_IOState::PROCESS)) ||

           CurState_ == Mcalc_IOState::BREAK_PROGRAM)));

  bool NoBatchCond_ =
           (ErrorFlip_ ||
            (!InBatchFile_ && WaitForServerSend_ &&
            ((PrevState_ == Mcalc_IOState::OUTPUT_READY &&
              (CurState_ == Mcalc_IOState::OUTPUT_FETCHED ||
               CurState_ == Mcalc_IOState::PROCESS)) ||
              
             (PrevState_ == Mcalc_IOState::GRAPH_OUTPUT &&
              (CurState_ == Mcalc_IOState::OUTPUT_FETCHED ||
               CurState_ == Mcalc_IOState::PROCESS)) ||
              
             (PrevState_ == Mcalc_IOState::INPUT_REQUIRED &&
              CurState_ == Mcalc_IOState::INPUT_RECEIVED) ||
              
             (PrevState_ == Mcalc_IOState::GRAPH_WAIT &&
              CurState_ == Mcalc_IOState::INPUT_RECEIVED) ||
             (PrevState_ == Mcalc_IOState::GRAPH_WAIT &&
              CurState_ == Mcalc_IOState::GRAPH_WAIT_ACK) ||
             (PrevState_ == Mcalc_IOState::GRAPH_PROGRESS &&
              CurState_ == Mcalc_IOState::GRAPH_PROGRESS_ACK) ||
              
             (PrevState_ == Mcalc_IOState::PROGRESS_READY &&
              CurState_ == Mcalc_IOState::INFILE_PROGRESS_ACK) ||

             (PrevState_ == Mcalc_IOState::PROCESS_DONE &&
              CurState_ == Mcalc_IOState::PROCESS_DONE_ACK) ||

             (PrevState_ == Mcalc_IOState::PROCESS_DONE_ACK &&
              CurState_ == Mcalc_IOState::PROCESS_DONE) ||

             (PrevState_ == Mcalc_IOState::PROCESS &&
              CurState_ == Mcalc_IOState::PROCESS_DONE) ||

             (PrevState_ == Mcalc_IOState::BATCHFILE_ENDED &&
              (CurState_ == Mcalc_IOState::BATCHFILE_ENDED_ACK ||
               CurState_ == Mcalc_IOState::PROCESS)) ||
              
             (PrevState_ == Mcalc_IOState::SPAWN_NEW_MCALC &&
              (CurState_ == Mcalc_IOState::MCALC_SPAWNED ||
               CurState_ == Mcalc_IOState::PROCESS)))));

  bool retval = InBatchCond_ || NoBatchCond_;

  if (retval && WaitForServerSend_)
  {
    DoWait_ = CurState_ == Mcalc_IOState::OUTPUT_FETCHED;

    NoWait_ = ErrorFlip_ ||
              CurState_ == Mcalc_IOState::INPUT_RECEIVED ||
              CurState_ == Mcalc_IOState::GRAPH_WAIT_ACK ||              
              CurState_ == Mcalc_IOState::GRAPH_PROGRESS_ACK ||
              CurState_ == Mcalc_IOState::PROCESS_DONE_ACK ||
              CurState_ == Mcalc_IOState::INFILE_PROGRESS_ACK;

    if (DoWait_)
      NoWait_ = false;
    else if (NoWait_)
      DoWait_ = false;
  }
  else
  {
    if (CritStates_)
    {
      DoWait_ = false;
      NoWait_ = false;
    }
    else if (CurState_ == Mcalc_IOState::PROCESS)
    {
      DoWait_ = true;
      NoWait_ = false;
    }
    else if ((ErrState_ && !ErrorFlip_) ||
             CurState_ == Mcalc_IOState::CALC_RESPONSE ||
             CurState_ == Mcalc_IOState::BREAK_PROGRAM)
    {
      DoWait_ = false;
      NoWait_ = true;
    }
    else
    {
      DoWait_ = IsClientStates(CurState_);

      if (DoWait_)
        NoWait_ = false;
      else
        NoWait_ = IsServerStates(CurState_);
    }
  }

  return retval;
}

/****************************************************************************/
// Replace with GUI method
// busy wait replaced with timer code here
//
void CalcDriver::ReadFromOutput(bool SetNewState_,
                                int CurrentState_, int PreviousState_)
{
  #if CALCDRIVER_TEXT_VERSION
    if (_ProgramExited)
      exit(0);
  #endif

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->EnterLevel("CalcDriver::ReadFromOutput");
  #endif

  #if CALCTEST_DEBUG1b
    QMessageBox::warning(0, "DEBUG-cdrv ReadFromOutput",
                         QString("\n_Processing = ") + QString::number(_Processing),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #elif ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG4))
    DbgPtr()->ShowInt(_Processing, "_Processing");
  #endif

  if (_Processing)
  {
    // millisec sleep
    int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
    int TermDelay_ = IOSTATE_DELAY_LENGTH * RETRY_READ_MAX;
    
    int PrevState_ = (PreviousState_ == 0) ? _IoState:PreviousState_;
    int PrevGuiRequest_ = IsGuiReadyToSend(PrevState_, true, true);
    
    int WaitForGraphSend_ = 0;
    int RetryMax_ = RETRY_ATTEMPTS;
    int x;

    bool WasActive_ = false;
    bool DoWait_ = false;
    bool NoWait_ = false;
    
    const char* IoStateFile_ = GetProgramIostateFilename();

    #if ((CALCTEST_DEBUG)|(CALCTEST_DEBUG1b))
      QMessageBox::warning(0, "DEBUG-cdrv ReadFromOutput",
                           QString("opening ") + QString(IoStateFile_) +
                           QString(" for reading 1st attempt, client side") +
                           QString("\n_OutputReady = ") + QString::number(_OutputReady),
                           QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
    #endif

    if (SetNewState_ && CurrentState_)
      _IoState = CurrentState_;

    WaitForGraphSend_ =
      (_IoState == Mcalc_IOState::GRAPH_WAIT_ACK)      ||
      (_IoState == Mcalc_IOState::GRAPH_PROGRESS_ACK);

    if (!_OutputReady || _RetryReadIoDone)
    {
      if (!_FileOpenRetryTimer.isActive() && _PollServerMgr)
        _PollServerMgr->ResetRetryRead();

      if (!_RetryReadIoDone)
      {
        RetryMax_ = WaitForGraphSend_ ? RETRY_ATTEMPTS:
                                        RETRY_READ_MAX;
      
        for (x = 0; !_OutputReady && x < RetryMax_; x+=RETRY_READ_MAX)
          _OutputReady = WaitForOutput(0, RETRY_READ_MAX);

        if (!_OutputReady)
        {
          if (_RequestResend &&
              !_ResendSigDone && _ResendSigUnchanged &&
              (_NoWaitReadCnt < RETRY_READ_MAX || LongWaitForServerSend(_IoState)))
          {
            _FatalErrCount = 0;
            _NoWaitReadCnt++;
          }
          else if (_ResendSigDone ||
                   (_RequestResend && _ResendSigUnchanged && _NoWaitReadCnt >= RETRY_READ_MAX))
          {
            if (_FatalErrCount >= MAX_FATAL_ERRORS)
            {
              exit(1);
              return;
            }

            #if CALCTEST_SHOW_MSGBOX
              QMessageBox::critical(0, tr("Fatal Error"),
                                       tr("Server signal resend error"),
                                       tr("Quit"));
            #endif

            _FatalErrCount++;
            DoHalt();
            emit FatalError();
          }
        }
        else
        {
          _RequestResend = false;
          _ResendSigDone = false;
          _ResendLastSig = false;
          _ResendSigUnchanged = false;
          _NoWaitReadCnt = 0;
        }
      }

      #if CALCTEST_DEBUG1b
        QMessageBox::warning(0, "DEBUG-cdrv ReadFromOutput",
                             QString("\n_OutputReady = ") + QString::number(_OutputReady),
                             QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
      #endif

      #if ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG4))
        DbgPtr()->ShowInt(_OutputReady, "_OutputReady");
      #endif

      if (_OutputReady || _IoState == Mcalc_IOState::CALC_HALT)
      {
        _RetryReadIoDone = false;
        _NoWaitReadCnt = 0;
        _RequestResend = false;
        _ResendSigDone = false;
        _ResendLastSig = false;
        _ResendSigUnchanged = false;

        IoStateTimer(STOP);
        _FileOpenRetryTimer.stop();

        if (_IoState == Mcalc_IOState::CALC_HALT)
          _TerminationTimer.start(TermDelay_, TRUE);
      }
      else if (_IoState != Mcalc_IOState::CALC_HALT)
      {
        int Eof_ = 0;
        const char* IostateFile_ = GetProgramIostateFilename();
        FILE* StateFin_ = fopen(IostateFile_, "r");
      
        int ErrCond_ = StateFin_ == NULL ||
                       (Eof_ = feof(StateFin_)) ||
                       ferror(StateFin_);

        if (ErrCond_ || !FileSize(StateFin_))
        {
          if (ErrCond_)
          {
            if (StateFin_)
            {
              clearerr(StateFin_);
              if (Eof_)
                fclose(StateFin_);
          
              StateFin_ = NULL;
            }
          }
          else
            fclose(StateFin_);
        
          if (!_FileOpenRetryTimer.isActive())
          {
            _AttemptFile = IOSTATE;
            _AttemptMode = "r";
            _AttemptFileName = IoStateFile_;
            SetFromMethod(READFROMOUTPUT);
            _SavedState = SetNewState_ ? CurrentState_:0;

            _FileOpenRetryTimer.start(DelayLen_);
          }

          #if CALCTEST_DEBUG3
            if (DbgPtr(false))
              DbgPtr()->LeaveLevel();
          #endif

          return;
        }
      }
    }
    #if ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG4))
      else
        DbgPtr()->ShowInt(_OutputReady, "_OutputReady");
    #endif
    
    #if ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG4))
      DbgPtr()->ShowStr(IoStateToStr(_IoState), "_IoState");
      DbgPtr()->ShowStr(IoStateToStr(PrevState_), "PrevState_");
    #endif

    if (PrevState_ != _IoState &&
        PrevGuiRequest_ && !_OutputReady)
      _PrevIoState = PrevState_;

    _MatchingStates = !_OutputReady &&
                      IsMatchingStates(_PrevIoState, _IoState, DoWait_, NoWait_);

    if (_OutputReady)
    {
      DoWait_ = false;
      NoWait_ = true;
    }

    #if CALCTEST_DEBUG1b
      QMessageBox::warning(0, "DEBUG-cdrv ReadFromOutput",
                           QString("\n_IoState = ") + QString::number(_IoState) +
                           QString("\nPrevState_ = ") + QString::number(PrevState_),
                           QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
    #endif

    QString Temp_;
    FILE* OutFin_ = NULL;

    if (_IoState == Mcalc_IOState::CALC_HALT &&
        PrevState_ != Mcalc_IOState::CALC_HALT && !_ServerCalledHalt)
    {
      #if CALCTEST_DEBUG4b
        DbgPtr()->ShowMessage("RunCalcServerCalledHalt()\n");
      #endif
      
      RunCalcServerCalledHalt();

      #if CALCTEST_DEBUG3
        if (DbgPtr(false))
          DbgPtr()->LeaveLevel();
      #endif
      return;
    }
    else if (_IoState == Mcalc_IOState::INPUT_REQUIRED)
    {
      _FetchError = false;
      const char* UserPromptFile_ = GetUserPromptFilename();
      OutFin_ = fopen(UserPromptFile_, "r");

      #if ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG4b))
        DbgPtr()->ShowMessage("OutFin_ = fopen(UserPromptFile_, \"r\");\n");
      #endif
    }
    else if (_IoState == Mcalc_IOState::PROGRESS_READY)
    {
      _FetchError = false;
      const char* ProgressFile_ = GetProgressFilename();
      OutFin_ = fopen(ProgressFile_, "r");

      #if CALCTEST_DEBUG4b
        DbgPtr()->ShowMessage("OutFin_ = fopen(ProgressFile_, \"r\");\n");
      #endif
    }
    else if (_IoState == Mcalc_IOState::GRAPH_PROGRESS)
    {
      _FetchError = false;
      const char* GraphProgressFile_ = GetGraphProgressFilename();
      OutFin_ = fopen(GraphProgressFile_, "r");

      #if ((CALCTEST_DEBUG4b)|(CALCTEST_DEBUG6c))
        DbgPtr()->ShowMessage("OutFin_ = fopen(GraphProgressFile_, \"r\");\n");
      #endif
    }
    else if (_IoState == Mcalc_IOState::ERROR_FETCH)
    {
      _FetchError = true;
      const char* ErrMsgFile_ = GetCurrentErrorFilename();
      OutFin_ = fopen(ErrMsgFile_, "r");

      #if ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG4b))
        DbgPtr()->ShowMessage("OutFin_ = fopen(ErrMsgFile_, \"r\");\n");
      #endif
    }
    else if (_IoState == Mcalc_IOState::SPAWN_NEW_MCALC)
    {
      _FetchError = false;
      const char* SpawnArgFile_ = GetSpawnArgFilename();
      OutFin_ = fopen(SpawnArgFile_, "r");

      #if ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG4b))
        DbgPtr()->ShowMessage("OutFin_ = fopen(SpawnArgFile_, \"r\");\n");
      #endif
    }
    else if (_OutputReady && (_IoState != Mcalc_IOState::PROCESS_DONE))
    {
      _FetchError = false;
      const char* ProgramOutFile_ = GetProgramOutputFilename();
      OutFin_ = fopen(ProgramOutFile_, "r");

      #if ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG4b))
        DbgPtr()->ShowMessage("OutFin_ = fopen(ProgramOutFile_, \"r\");\n");
      #endif
    }
    else
      OutFin_ = NULL;

    unsigned long Size_ = (_OutputReady && OutFin_) ? FileSize(OutFin_):0;
    _SavedState = _OutputReady ? 0:_IoState;
    _ReadOk = false;

    if (!OutFin_ || !Size_)
    {
      strcpy(_OutputBuffer, "");
      
      if (OutFin_)
        fclose(OutFin_);
    }
    else
    {    
      if (Size_ >= _OutputBufferSize)
        ResizeOutputBuffer(Size_);
      
      ::memset(_OutputBuffer, 0, Size_);
      _ReadOk = fread(_OutputBuffer, 1, Size_, OutFin_);

      #if ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG4))
        DbgPtr()->ShowMessage("_ReadOk = fread(_OutputBuffer, 1, Size_, OutFin_);\n");
        DbgPtr()->ShowInt(Size_, "Size_");
        DbgPtr()->ShowInt(_ReadOk, "_ReadOk");
        DbgPtr()->ShowStr(IoStateToStr(_SavedState), "_SavedState");
      #endif

      if (_ReadOk)
      {
        _OutputBuffer[Size_] = 0;
        Temp_ = _OutputBuffer;
        Temp_ = Temp_.stripWhiteSpace();
        strcpy(_OutputBuffer, Temp_.ascii());
      }
      
      fclose(OutFin_);

      #if ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG4))
        DbgPtr()->ShowStr(_OutputBuffer, "_OutputBuffer");
      #endif
    }
    
    _OutputReady = FALSE;

    if (DoWait_ && !_ResendSigDone)
    {
      if (!EmptyBuffer())
      {
        DelayLen_ *= _WaitCycle--;
        if (_WaitCycle == 0)
          _WaitCycle = 2;
      }

      _NoWaitReadCnt = 0;
      _RetryReadIoDone = false;

      IoStateTimer(START, EmptyBuffer() ? LONG_WAIT:DelayLen_);
    }
    else if (NoWait_ || _ResendSigDone)
      EvalOutput();
  }

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void CalcDriver::WriteToInput(const char* Str_)
{
  const char* UserInputFile_ = GetUserInputFilename();

  #if CALCTEST_DEBUG
      QMessageBox::warning(0, "DEBUG",
                           QString("opening ") + QString(UserInputFile_) +
                           QString(" for writing 1st attempt, client side"),
                           QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #endif

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("CalcDriver::WriteToInput");
      if (Str_)
        DbgPtr()->ShowStr(Str_, "Str_");
      else
        DbgPtr()->ShowMessage("Str_ == NULL\n");
    }
  #endif
  
  FILE* InFout_ = fopen(UserInputFile_, "w");
  size_t Len_ = (Str_ ? ::strlen(Str_):0);
  if (Len_)
    fprintf(InFout_, "%s;\n", Str_);

  if (InFout_)
    fclose(InFout_);

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
// Input Data Batch File Mode:
// ---------------------------
// If calculator is executing a batch file or program,
// the following special cases apply:
//
//   INPUT_RECEIVED      --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
//   OUTPUT_FETCHED      --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
//   BREAK_PROGRAM       --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
//   BATCHFILE_ENDED_ACK --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
//
//   CALC_RESPONSE       --> INPUT_RECEIVED
//   CALC_ERROR          --> INPUT_RECEIVED
//   ERROR_FETCH         --> INPUT_RECEIVED
//
//   PROCESS_DONE        --> PROCESS_DONE_ACK
//   PROCESS_DONE_ACK    --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
//
void CalcDriver::EvalOutput()
{
  #if CALCDRIVER_TEXT_VERSION
    if (_ProgramExited)
      exit(0);
  #endif

  // millisec sleep
  int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  size_t len = _OutputBuffer ? ::strlen(_OutputBuffer):0;

  int WaitForServerSend_ = WaitingForServerSend(_IoState);

  bool SignalSent_ = false;
  bool Break_ = _IoState == Mcalc_IOState::BREAK_PROGRAM;
  bool EndOfBatchAck_ = _IoState == Mcalc_IOState::BATCHFILE_ENDED_ACK;
  
  bool CritStates_ = _IoState == Mcalc_IOState::CALC_HALT ||
                     _IoState == Mcalc_IOState::IDLE_STATE;
  bool ErrorFlip_ = (_PrevIoState == Mcalc_IOState::CALC_ERROR &&
                     _IoState == Mcalc_IOState::ERROR_FETCH) ||
                    (_IoState == Mcalc_IOState::CALC_ERROR &&
                     _PrevIoState == Mcalc_IOState::ERROR_FETCH);

  int NoWait_ = EmptyBuffer() || EndOfBatchAck_ || Break_ ||
                (_MatchingStates &&
                   (ErrorFlip_ ||
                    _IoState == Mcalc_IOState::INPUT_RECEIVED ||
                    _IoState == Mcalc_IOState::GRAPH_WAIT_ACK ||
                    _IoState == Mcalc_IOState::GRAPH_PROGRESS_ACK ||
                    _IoState == Mcalc_IOState::INFILE_PROGRESS_ACK ||
                    _IoState == Mcalc_IOState::PROCESS_DONE_ACK));

  #if ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG3))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("CalcDriver::EvalOutput");
      DbgPtr()->ShowStr(_OutputBuffer, "_OutputBuffer");
      DbgPtr()->ShowInt(len, "len");
      DbgPtr()->ShowStr(IoStateToStr(_IoState), "_IoState");
      DbgPtr()->ShowInt(_RequestResend, "_RequestResend");
      DbgPtr()->ShowInt(_ResendLastSig, "_ResendLastSig");
      DbgPtr()->ShowInt(_ResendSigDone, "_ResendSigDone");
      DbgPtr()->ShowInt(_ResendSigUnchanged, "_ResendSigUnchanged");
    }
  #endif

  #if CALCTEST_DEBUG
    QMessageBox::warning(0, "DEBUG",
                         QString("Evaluating output: ") + QString(_OutputBuffer),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #endif

  if (len)
  {
    // output function to send output from calculator
    #if CALCDRIVER_GUI_VERSION
      _Output->append(QString(_OutputBuffer));
    #else
      strcat(_Output, _OutputBuffer);
      
      if (IoState() == Mcalc_IOState::PROGRESS_READY && !_ProgressOutDone)
      {
        std::cout <<"done>> " <<_Output <<std::endl;
        _ProgressOutDone = true;
        _Output[0] = 0;
      }
      else if (IoState() == Mcalc_IOState::GRAPH_PROGRESS && !_GraphProgressOutDone)
      {
        std::cout <<"plot>> " <<_GrReadBuffer <<std::endl;
        _GraphProgressOutDone = true;
        _GrReadBuffer[0] = 0;
      }      
      else if (IoState() != Mcalc_IOState::PROGRESS_READY &&
               IoState() != Mcalc_IOState::GRAPH_PROGRESS && !_CalcOutDone)
      {
        std::cout <<"out>> " <<_Output <<std::endl;
        _CalcOutDone = true;
        _Output[0] = 0;
      }
    #endif
  }

  if (InBatchFile())
    _ExpectedSignals->Reset();

  if (_IoState == Mcalc_IOState::SPAWN_NEW_MCALC)
  {
    SignalSent_ = true;
    _ExpectedSignals->SetFromServer(_IoState);

    // We don't need to emit an CreateNewMcalc() signal
    // if the MCALC_SPAWNED I/O state was resent by the server
    if (!_ResendSigDone ||
        !_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::MCALC_SPAWNED))
    {
      _ExpectedSignals->SetClientExpected(Mcalc_IOState::MCALC_SPAWNED);
      _Spawned = FALSE;

      #if CALCTEST_DEBUG8
        DbgPtr()->ShowMessage("SpawnNewMcalc();\n");
        DbgPtr()->ShowMessage("expecting MCALC_SPAWNED response from client\n");
        DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
      #endif

      #if QMCALC_WIN64VER
        SpawnNewMcalc();
      #else
        _Spawned = TRUE;
        emit CreateNewMcalc();
      #endif
      // expecting MCALC_SPAWNED response from client
    }
    else
      _ResendSigDone = false;
  }
  else if (_IoState == Mcalc_IOState::INPUT_REQUIRED)
  {
    // len != 0 NOT required
    SignalSent_ = true;
    _ExpectedSignals->SetFromServer(_IoState);

    // We don't need to emit an InputNeeded() signal
    // if the INPUT_REQUIRED I/O state was resent by the server
    if (!_ResendSigDone ||
        !_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::INPUT_RECEIVED))
    {
      _ExpectedSignals->SetClientExpected(Mcalc_IOState::INPUT_RECEIVED);

      #if CALCTEST_DEBUG8
        DbgPtr()->ShowMessage("emit InputNeeded()\n");
        DbgPtr()->ShowMessage("expecting INPUT_RECEIVED response from client\n");
        DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
      #endif

      emit InputNeeded();
      // expecting INPUT_RECEIVED response from client
    }
    else
      _ResendSigDone = false;

    #if CALCDRIVER_TEXT_VERSION
      if (!_IoStateRead)
      {
        _IoStateRead = true;
        ReadIoState();
      }
    #endif
  }
  else if (_IoState == Mcalc_IOState::OUTPUT_READY)
  {
    // output buffer should not be empty
    if (len || _ResendSigDone)
    {
      SignalSent_ = true;
      _ExpectedSignals->SetFromServer(_IoState);

      // We don't need to emit a OutputReady() signal
      // if the OUTPUT_READY I/O state was resent by the server
      if (!_ResendSigDone ||
          (len && !_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::OUTPUT_FETCHED)))
      {
        _ExpectedSignals->SetClientExpected(Mcalc_IOState::OUTPUT_FETCHED);

        #if CALCTEST_DEBUG8
          DbgPtr()->ShowMessage("emit OutputReady()\n");
          DbgPtr()->ShowMessage("expecting OUTPUT_FETCHED response from client\n");
          DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
        #endif

        emit OutputReady();
        // expecting OUTPUT_FETCHED response from client
      }
      else
        _ResendSigDone = false;
    }

    #if CALCDRIVER_TEXT_VERSION
      if (!_IoStateRead)
      {
        _IoStateRead = true;
        ReadIoState();
      }
    #endif
  }
  else if (_IoState == Mcalc_IOState::PROGRESS_READY)
  {
    if (_ReadBuffer)
    {
      delete[] _ReadBuffer;
      _ReadBuffer = NULL;
    }

    if (_OutputBuffer)
      _ReadBuffer = strcpy(new char[strlen(_OutputBuffer) + 1], _OutputBuffer);
    else
      _ReadBuffer = strcpy(new char[5], "");

    SignalSent_ = true;
    _ResendSigDone = false;

    _ExpectedSignals->SetFromServer(_IoState);
    _ExpectedSignals->SetClientExpected(Mcalc_IOState::INFILE_PROGRESS_ACK);

    #if CALCTEST_DEBUG8
      DbgPtr()->ShowStr(_ReadBuffer, "_ReadBuffer");
      DbgPtr()->ShowMessage("emit ProgressReady()\n");
      DbgPtr()->ShowMessage("Expecting INFILE_PROGRESS_ACK response from client\n");
      DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
      DbgPtr()->ShowInt(SignalSent_, "SignalSent_");
      DbgPtr()->ShowInt(_ResendSigDone, "_ResendSigDone");
    #endif
    
    emit ProgressReady();
    // Expecting INFILE_PROGRESS_ACK response from client
  }
  else if (_IoState == Mcalc_IOState::ERROR_FETCH)
  {
    #if CALCDRIVER_GUI_VERSION
      _FetchError = false;
      SetIoState(Mcalc_IOState::CALC_ERROR, true);
    #endif

    SignalSent_ = true;
    _ExpectedSignals->SetFromServer(_IoState);

    // We don't need to emit a ErrorFound() signal
    // if the ERROR_FETCH I/O state was resent by the server
    if (!_ResendSigDone ||
        (!_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::PROCESS) &&
         !_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::INPUT_RECEIVED, true)))
    {
      #if CALCTEST_DEBUG8
        DbgPtr()->ShowMessage("emit ErrorFound()\n");
        DbgPtr()->ShowMessage("expecting PROCESS, BREAK_PROGRAM or CALC_HALT response from client\n");
        DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
        DbgPtr()->ShowInt(SignalSent_, "SignalSent_");
        DbgPtr()->ShowInt(_ResendSigDone, "_ResendSigDone");
      #endif

      _ExpectedSignals->SetClientExpected(Mcalc_IOState::PROCESS);
      if (InBatchFile())
        _ExpectedSignals->SetClientExpected(Mcalc_IOState::INPUT_RECEIVED);
                                        ;
      emit ErrorFound();
      // expecting PROCESS, BREAK_PROGRAM or CALC_HALT response from client
    }
    else
      _ResendSigDone = false;
  }
  else if (_IoState == Mcalc_IOState::CALC_ERROR)
  {
    SignalSent_ = true;
    _ExpectedSignals->SetFromServer(_IoState);

    // We don't need to emit a ErrorFound() signal
    // if the ERROR_FETCH I/O state was resent by the server
    if (!_ResendSigDone ||
        (!_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::PROCESS) &&
         !_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::INPUT_RECEIVED, true)))
    {
      #if CALCTEST_DEBUG8
        DbgPtr()->ShowMessage("emit ErrorFound()\n");
        DbgPtr()->ShowMessage("expecting PROCESS, BREAK_PROGRAM or CALC_HALT response from client\n");
        DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
        DbgPtr()->ShowInt(SignalSent_, "SignalSent_");
        DbgPtr()->ShowInt(_ResendSigDone, "_ResendSigDone");
      #endif

      _ExpectedSignals->SetClientExpected(Mcalc_IOState::PROCESS);
      if (InBatchFile())
        _ExpectedSignals->SetClientExpected(Mcalc_IOState::INPUT_RECEIVED);

      emit ErrorFound();
      // expecting PROCESS, BREAK_PROGRAM or CALC_HALT response from client
    }
    else
      _ResendSigDone = false;
  }
  else if (_IoState == Mcalc_IOState::CALC_RESPONSE)
  {
    // zero length output allowed for CALC_RESPONSE if explicitly specified
    SignalSent_ = true;
    _ExpectedSignals->SetFromServer(_IoState);

    // We don't need to emit an InputNeeded() signal
    // if the CALC_RESPONSE I/O state was resent by the server
    if (!_ResendSigDone ||
        (!_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::PROCESS) &&
         !_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::INPUT_RECEIVED, true)))
    {
      #if CALCTEST_DEBUG8
        DbgPtr()->ShowMessage("emit InputNeeded()\n");
        DbgPtr()->ShowMessage("expecting PROCESS, BREAK_PROGRAM or CALC_HALT response from client\n");
        DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
        DbgPtr()->ShowInt(SignalSent_, "SignalSent_");
        DbgPtr()->ShowInt(_ResendSigDone, "_ResendSigDone");
      #endif

      _ExpectedSignals->SetClientExpected(Mcalc_IOState::PROCESS);
      if (InBatchFile())
        _ExpectedSignals->SetClientExpected(Mcalc_IOState::INPUT_RECEIVED);

      emit InputNeeded();
      // expecting PROCESS, BREAK_PROGRAM or CALC_HALT response from client
    }
    else
      _ResendSigDone = false;

    #if CALCDRIVER_TEXT_VERSION
      if (!_IoStateRead)
      {
        _IoStateRead = true;
        ReadIoState();
      }
    #endif
  }
  else if (_IoState == Mcalc_IOState::PROCESS_DONE)
  {
    // output buffer should not be empty
    SignalSent_ = true;
    _ExpectedSignals->SetFromServer(_IoState);

    // We don't need to emit a OutputReady() signal
    // if the OUTPUT_READY I/O state was resent by the server
    if (!_ResendSigDone ||
        !_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::PROCESS_DONE_ACK))
    {
      _ExpectedSignals->SetClientExpected(Mcalc_IOState::PROCESS_DONE_ACK);

      #if CALCTEST_DEBUG8
        DbgPtr()->ShowMessage("emit ProcessAcked()\n");
        DbgPtr()->ShowMessage("expecting PROCESS_DONE_ACK response from client\n");
        DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
        DbgPtr()->ShowInt(SignalSent_, "SignalSent_");
        DbgPtr()->ShowInt(_ResendSigDone, "_ResendSigDone");
      #endif

      emit ProcessAcked();
      // expecting PROCESS_DONE_ACK response from client
    }
    else
      _ResendSigDone = false;

    #if CALCDRIVER_TEXT_VERSION
      if (!_IoStateRead)
      {
        _IoStateRead = true;
        ReadIoState();
      }
    #endif
  }
  else if (_IoState == Mcalc_IOState::GRAPH_WAIT)
  {
    #if CALCTEST_DEBUG6b
      QMessageBox::warning(0, "DEBUG emit GraphWait()",
                           QString("\nemit GraphWait()"),
                           QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
    #endif

    SignalSent_ = true;
    _ExpectedSignals->SetFromServer(_IoState);

    // We don't need to emit a GraphWait() signal
    // if the GRAPH_WAIT I/O state was resent by the server
    if (!_ResendSigDone ||
        (!_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::GRAPH_WAIT_ACK) &&
         !_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::INPUT_RECEIVED)))
    {
      #if CALCTEST_DEBUG8
        DbgPtr()->ShowMessage("emit GraphWait()\n");
        DbgPtr()->ShowMessage("expecting GRAPH_WAIT_ACK or INPUT_RECEIVED response from client\n");
        DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
        DbgPtr()->ShowInt(SignalSent_, "SignalSent_");
        DbgPtr()->ShowInt(_ResendSigDone, "_ResendSigDone");        
      #endif

      _ExpectedSignals->SetClientExpected(Mcalc_IOState::GRAPH_WAIT_ACK|
                                          Mcalc_IOState::INPUT_RECEIVED);
      emit GraphWait();
      // expecting GRAPH_WAIT_ACK response from client
    }
    else
      _ResendSigDone = false;
  }
  else if (_IoState == Mcalc_IOState::GRAPH_OUTPUT)
  {
    #if CALCTEST_DEBUG6b
      QMessageBox::warning(0, "DEBUG emit GraphOutput()",
                           QString("\nemit GraphOutput()"),
                           QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
    #endif

    SignalSent_ = true;
    _ExpectedSignals->SetFromServer(_IoState);

    // We don't need to emit a GraphOutput() signal
    // if the GRAPH_OUTPUT I/O state was resent by the server
    if (!_ResendSigDone ||
        !_ExpectedSignals->MatchesClientExpected(Mcalc_IOState::OUTPUT_FETCHED))
    {
      #if CALCTEST_DEBUG8
        DbgPtr()->ShowMessage("emit GraphOutput()\n");
        DbgPtr()->ShowMessage("expecting OUTPUT_FETCHED response from client\n");
        DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
        DbgPtr()->ShowInt(SignalSent_, "SignalSent_");
        DbgPtr()->ShowInt(_ResendSigDone, "_ResendSigDone");
      #endif

      _ExpectedSignals->SetClientExpected(Mcalc_IOState::OUTPUT_FETCHED);
      emit GraphOutput();
      // expecting OUTPUT_FETCHED response from client
    }
    else
      _ResendSigDone = false;
  }
  else if (_IoState == Mcalc_IOState::GRAPH_PROGRESS)
  {
    if (_GrReadBuffer &&
        _GraphProgressOutDone && _GrProgressDataShown)
    {
      delete[] _GrReadBuffer;
      _GrReadBuffer = NULL;
    }

    #if CALCTEST_DEBUG8
      DbgPtr()->ShowMessage("emit GraphProgress()\n");
      DbgPtr()->ShowMessage("expecting GRAPH_PROGRESS_ACK response from client\n");
      DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
      DbgPtr()->ShowInt(SignalSent_, "SignalSent_");
      DbgPtr()->ShowInt(_ResendSigDone, "_ResendSigDone");
    #endif

    SignalSent_ = true;
    _ResendSigDone = false;

    _ExpectedSignals->SetFromServer(_IoState);
    _ExpectedSignals->SetClientExpected(Mcalc_IOState::GRAPH_PROGRESS_ACK);
    
    emit GraphProgress();
    // expecting GRAPH_PROGRESS_ACK response from client
  }
  else if (_IoState == Mcalc_IOState::BATCHFILE_ENDED)
  {
    #if CALCTEST_DEBUG8
      DbgPtr()->ShowMessage("emit EndOfBatchData()\n");
      DbgPtr()->ShowMessage("expecting BATCHFILE_ENDED_ACK response from client\n");
      DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
      DbgPtr()->ShowInt(SignalSent_, "SignalSent_");
      DbgPtr()->ShowInt(_ResendSigDone, "_ResendSigDone");
    #endif

    SignalSent_ = true;
    _ResendSigDone = false;

    _ExpectedSignals->SetFromServer(_IoState);
    _ExpectedSignals->SetClientExpected(Mcalc_IOState::BATCHFILE_ENDED_ACK);
    
    emit EndOfBatchData();
    // expecting BATCHFILE_ENDED_ACK response from client
  }

  #if CALCTEST_DEBUG8
    if (SignalSent_ && !_ResendSigDone)
    {
      DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetFromServer()), "GetFromServer()");
      DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
      DbgPtr()->ShowInt(SignalSent_, "SignalSent_");
      DbgPtr()->ShowInt(_ResendSigDone, "_ResendSigDone");
    }
  #endif

  if (!SignalSent_ &&
      (!len ||
       IsIntermediateStates(_IoState) || WaitForServerSend_ || CritStates_))
  {
    // poll to check if calculator server is alive here
    #if IMPLEMENT_KEEP_ALIVE
    if (CalcPollingActive())
    {
      #if CALCTEST_DEBUG4
        if (DbgPtr(false))
        {
          _PollServerMgr->SetStopTracking(false);
          DbgPtr()->ShowMessage("CalcDriver::EvalOutput: len == 0\n");
          DbgPtr()->ShowInt(_NoPolling, "_NoPolling");
          DbgPtr()->ShowMessage("EvalOutput:: _PollServerMgr->PollServerRequired(0, false)\n");
        }
      #endif

      if (!_NoPolling && _PollServerMgr &&
          _PollServerMgr->SetAllowServerPolling(true)
                        ->ResetWhenNotRequired(true)
                        ->PollServerRequired(0, false))
      {

        #if CALCTEST_DEBUG4b
          if (DbgPtr(false))
            DbgPtr()->ShowMessage("EvalOutput:: _PollServerMgr->PollServerIfAlive(1, true, false)\n");
        #endif
    
        _PollServerMgr->SetWaitUntilResponse(true)
                      ->SetExitAtRetryMax(true)
                      ->SetSleepWaiting(true)
                      ->SetAllowServerPolling(true)
                      ->SetBreakOnPollError(false);

        _ResponseOnly = false;
        _PollServerMgr->ResetServerPolling(true);
        _PollServerMgr->PollServerIfAlive(1, true, false); // Initiate Polling
      }
    }
    #endif

    if (IsCalcResponseSignals(_IoState))
      _ExpectedSignals->SetFromServer(_IoState);
    else if (WaitingForServerSend(_IoState))
      _ExpectedSignals->SetFromClient(_IoState);

    if (_ExpectedSignals->IsProcessNext() &&
        !_ExpectedSignals->PendingResponse())
    {
      _ExpectedSignals->Reset();
      _ExpectedSignals->SetClientExpected(Mcalc_IOState::PROCESS);
    }

    if (!_ReadIoStateTimer.isActive())
    {
      Break_ = _IoState == Mcalc_IOState::BREAK_PROGRAM;
      NoWait_ = NoWait_ &&
                !_ResendSigUnchanged && !_ResendLastSig &&
                !_RequestResend && !_ResendSigDone &&
                (!_OutputReady || Break_) && _Processing;

      if ((!WaitForServerSend_ || CritStates_) ||
          ((NoWait_ && _NoWaitReadCnt < RETRY_READ_MAX) ||
           LongWaitForServerSend(_IoState)))
      {
        if (_SavedState == 0)
          _GraphProgressOutDone =
          _ProgressOutDone =
          _CalcOutDone = false;

        if (NoWait_)
          _NoWaitReadCnt++;

        ReadElapsedTimeState(false, NULL, "CalcDriver::EvalOutput()");
        ReadFromOutput(_SavedState != 0, _SavedState);
      }
      else if ((WaitForServerSend_ || Break_) ||
               _ResendLastSig || _ResendSigDone ||
               (NoWait_ && _NoWaitReadCnt >= RETRY_READ_MAX))
      {
        if (!EmptyBuffer())
        {
          DelayLen_ *= _WaitCycle--;
          if (_WaitCycle == 0)
            _WaitCycle = 2;
        }

        _NoWaitReadCnt = 0;
        _RetryReadIoDone = false;

        if (_ResendSigDone)
          RetryReadIoState();
        else
        {
          if (_ResendLastSig)
            SetIoState(Mcalc_IOState::RESENDLASTSIG, true);

          IoStateTimer(START,
                       (EmptyBuffer() || EndOfBatchAck_) ? LONG_WAIT:DelayLen_);
        }
      }
    }
  }

  #if ((CALCTEST_DEBUG8)|(CALCTEST_DEBUG3))
    if (DbgPtr(false))
    {
      DbgPtr()->LeaveLevel();

      if (_ExpectedSigSent)
      {
        DbgPtr()->LeaveLevel();
        _ExpectedSigSent = false;
      }
    }
  #endif
}

/****************************************************************************/
void CalcDriver::ShowGraphProgressData()
{
  #if CALCDRIVER_TEXT_VERSION
    static const char* GrPlotTypes_[] = { "NO_GRAPH",
                                          "FUNCTION_PLOT",
                                          "APPLIED_FNCGRAPH",
                                          "BATCH_GRAPH_PLOT" };
    static const char* GrTypes_[] = { "FUNCTION",
                                      "EXTREMA",
                                      "DERIVATIVE",
                                      "INTEGRAL",
                                      "ROOTSOLVE",
                                      "INTERSECT1",
                                      "INTERSECT2" };
  #endif

  if (_GrProgressFetched && !_GrProgressDataShown)
  {
    if (_GrOperationFetched && !_GrOperationDataShown)
    {
      #if CALCDRIVER_TEXT_VERSION
        std::cout <<"Graph Operation Data:" <<std::endl;
        std::cout <<"   " <<GrPlotTypes_[_GraphPlotOpType] <<std::endl;
        std::cout <<"   " <<GrTypes_[_GraphType] <<std::endl;

        if (_GraphSubtype == Mcalc_GraphType::MIN_EXTREMA)
          std::cout <<"   " <<"MIN_EXTREMA" <<std::endl;
        else if (_GraphSubtype == Mcalc_GraphType::MAX_EXTREMA)
          std::cout <<"   " <<"MAX_EXTREMA" <<std::endl;

        if (_GraphSubtype == Mcalc_GraphType::FIRST_DERIV)
          std::cout <<"   " <<"FIRST_DERIV" <<std::endl;
        else if (_GraphSubtype == Mcalc_GraphType::SECOND_DERIV)
          std::cout <<"   " <<"SECOND_DERIV" <<std::endl;
      #endif

      _GraphOperationOutDone = true;
      _GrOperationDataShown = true;
    }

    if (_GrNumerSet && _GrNumer && _GrDenomSet && _GrDenom)
    {
      #if CALCDRIVER_TEXT_VERSION
        std::cout <<"Graph Progress Data:" <<std::endl;
        std::cout <<"   " <<_GrNumer <<"/" <<_GrDenom <<std::endl;
      #endif
      
      _GraphProgressOutDone = true;
      _GrProgressDataShown = _GrNumer == _GrDenom;
    }
  }
}

/****************************************************************************/
void CalcDriver::HandleGraphProgressData()
{
  #if CALCTEST_DEBUG6c
    if (DbgPtr(false))
      DbgPtr()->EnterLevel("CalcDriver::HandleGraphProgressData");
  #endif

  if ((IoState() == Mcalc_IOState::GRAPH_PROGRESS ||
       IoState() == Mcalc_IOState::OUTPUT_READY) &&
      _GrProgressFetched && _GraphProgressOutDone &&
      _GrOperationFetched && _GrOperationDataShown)
  {
    _GrNumer = 0;
    _GrNumerSet = false;
    _GrDenom = 0;
    _GrDenomSet = false;
            
    if (_GrReadBuffer &&
        _GraphProgressOutDone && _GrProgressDataShown)
    {
      delete[] _GrReadBuffer;
      _GrReadBuffer = NULL;

      delete[] _GrOutputDataFname;
      _GrOutputDataFname = NULL;
      
      _GraphProgressOutDone = true;
      _GraphOperationOutDone = true;
      
      _GrOutputDataFnameDone = false;
      _GrOutputFnameSet = false;
           
      _GraphPlotOpType = 0;
      _GraphType = 0;
      _GraphSubtype = 0;
      _GraphOpTypeSet = false;
      _GrProgressFetched = false;
      _GrOperationFetched = false;
      _GrReadOk = false;
    }
  }

  #if CALCTEST_DEBUG6c
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void CalcDriver::HandleProgressData()
{
  if (_InitProgressBar && !_ProgressBarInitDone &&
      _DenomSet && _Denom && _InputFileDoneLvl == -1)
  {
    _ProgressBarInitDone = true;
    
    if (_InputFnameSet && !_InputDataFnameDone)
      _InputDataFnameDone = true;
    
    if (_OutputFnameSet && !_OutputDataFnameDone)
      _OutputDataFnameDone = true;
  }
  else if (_EndProgressBar && !_ProgressBarEndDone &&
           _NumerSet && _Numer && _DenomSet && _Denom &&
           _InputFileDoneLvl == 0)
  {
    _ProgressBarEndDone = true;    
  }
  
  if (IoState() == Mcalc_IOState::OUTPUT_READY &&
      _EndProgressBar && _ProgressBarEndDone && _InputFileDoneLvl == 1)
  {
    _InitProgressBar = false;
    _EndProgressBar = false;
    _ProgressBarInitDone = false;
    _ProgressBarEndDone = false;
    _InputDataFnameDone = false;
    _OutputDataFnameDone = false;
    _Numer = 0;
    _NumerSet = false;
    _Denom = 0;
    _DenomSet = false;
    _InputFnameSet = false;
    _OutputFnameSet = false;
    
    if (_InputDataFname)
    {
      delete[] _InputDataFname;
      _InputDataFname = NULL;
    }
      
    if (_OutputDataFname)
    {
      delete[] _OutputDataFname;
      _OutputDataFname = NULL;
    }
      
    if (_ReadBuffer)
    {
      delete[] _ReadBuffer;
      _ReadBuffer = NULL;
    }
  }
}

/****************************************************************************/
void CalcDriver::SendProgressReady()
{
  // SIGNAL: SendCalcProgress(int);    | SLOT : ReceiveCalcProgress(int);
  // SIGNAL: void EndCalcProgress();   | SLOT : DoEndCalcProgress();
  // SIGNAL: void SendTotalSteps(int); | SLOT : SetTotalSteps(int);

  bool ValidCase_ = _OutputBuffer &&
                    ((_NumerSet && _Numer) || (_DenomSet && _Denom));

  #if CALCTEST_DEBUG7
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("SendProgressReady");
      DbgPtr()->ShowInt(ValidCase_, "ValidCase_(1)");
      DbgPtr()->ShowInt(_ProgressBarEndDone, "_ProgressBarEndDone");
      DbgPtr()->ShowInt(_EndProgressBar, "_EndProgressBar");
      DbgPtr()->ShowInt(_Numer, "_Numer");
      DbgPtr()->ShowInt(_NumerSet, "_NumerSet");
      DbgPtr()->ShowInt(_Denom, "_Denom");
      DbgPtr()->ShowInt(_DenomSet, "_DenomSet");
      DbgPtr()->ShowStr(IoStateToStr(IoState()), "IoState()");
    }
  #endif
                    
  if (!ValidCase_)
  {
    if (IoState() == Mcalc_IOState::PROGRESS_READY)
      SendProgressFetched();
    
    return;
  }
  else
    ValidCase_ = false;    
  
  if (_InitProgressBar && !_ProgressBarInitDone && _DenomSet && _Denom)
  {
    ValidCase_ = true;
    _ProgressBarInitDone = true;
    emit SendTotalSteps(_Denom);
    
    if (_InputFnameSet && !_InputDataFnameDone)
    {
      _InputDataFnameDone = true;
      emit SendInputFilename(_InputDataFname);
    }
    
    if (_OutputFnameSet && !_OutputDataFnameDone)
    {
      _OutputDataFnameDone = true;
      emit SendOutputFilename(_OutputDataFname);
    }

    #if CALCTEST_DEBUG7
      if (DbgPtr(false))
      {
        DbgPtr()->ShowInt(ValidCase_, "ValidCase_(2)");
        DbgPtr()->ShowMessage("emit SendTotalSteps(_Denom);\n");
        if (_InputDataFnameDone)
          DbgPtr()->ShowMessage("emit SendInputFilename(_InputDataFname);\n");
        if (_OutputDataFnameDone)
          DbgPtr()->ShowMessage("emit SendOutputFilename(_OutputDataFname);\n");
      }
    #endif
  }
  else if (_EndProgressBar && !_ProgressBarEndDone &&
           _NumerSet && _Numer && _DenomSet && _Denom)
  {
    ValidCase_ = true;
    _ProgressBarEndDone = true;
    QString OutFile_(_OutputDataFname);
    QString InFile_(_InputDataFname);
    
    emit EndCalcProgress();
    emit ResultsFileReady(InFile_, OutFile_);

    #if CALCTEST_DEBUG7
      if (DbgPtr(false))
      {
        DbgPtr()->ShowInt(ValidCase_, "ValidCase_(3)");
        DbgPtr()->ShowInt(_ProgressBarEndDone, "_ProgressBarEndDone");
        DbgPtr()->ShowMessage("emit EndCalcProgress();\n");
        DbgPtr()->ShowMessage("emit ResultsFileReady(InFile_, OutFile_);\n");
        DbgPtr()->ShowStr(_InputDataFname, "InFile_");
        DbgPtr()->ShowStr(_OutputDataFname, "OutFile_");
      }
    #endif
  }
  else if (_NumerSet && _Numer)
  {
    ValidCase_ = true;
    
    if (!_ProgressBarInitDone && _DenomSet && _Denom)
    {
      _ProgressBarInitDone = true;
      emit SendTotalSteps(_Denom);
    }
        
    emit SendCalcProgress(_Numer);

    #if CALCTEST_DEBUG7
      if (DbgPtr(false))
      {
        DbgPtr()->ShowInt(ValidCase_, "ValidCase_(4)");
        DbgPtr()->ShowInt(_ProgressBarInitDone, "_ProgressBarInitDone");
        if (_ProgressBarInitDone)
          DbgPtr()->ShowMessage("emit SendTotalSteps(_Denom);\n");
        DbgPtr()->ShowMessage("emit SendCalcProgress(_Numer);\n");
      }
    #endif
  }

  if (ValidCase_ || IoState() == Mcalc_IOState::PROGRESS_READY)
  {
    SendProgressFetched();
    
    if (_EndProgressBar && _ProgressBarEndDone)
    {
      _InitProgressBar = false;
      _EndProgressBar = false;
      _ProgressBarInitDone = false;
      _ProgressBarEndDone = false;
      _InputDataFnameDone = false;
      _OutputDataFnameDone = false;
      _Numer = 0;
      _NumerSet = false;
      _Denom = 0;
      _DenomSet = false;
      _InputFnameSet = false;
      _OutputFnameSet = false;
      
      if (_InputDataFname)
      {
        delete[] _InputDataFname;
        _InputDataFname = NULL;
      }
      
      if (_OutputDataFname)
      {
        delete[] _OutputDataFname;
        _OutputDataFname = NULL;
      }
      
      if (_ReadBuffer)
      {
        delete[] _ReadBuffer;
        _ReadBuffer = NULL;
      }
    }

    #if CALCTEST_DEBUG7
      DbgPtr()->ShowMessage("SendProgressFetched\n");
      DbgPtr()->ShowInt(_EndProgressBar, "_EndProgressBar");
      DbgPtr()->ShowInt(_ProgressBarEndDone, "_ProgressBarEndDone");
    #endif
  }

  #if CALCTEST_DEBUG7
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
bool CalcDriver::HasExpectedStates()
{
  bool retval_ = _ExpectedSignals->ExpectedStatesSet();
  #if CALCTEST_DEBUG8
    DbgPtr()->EnterLevel("HasExpectedStates");
    DbgPtr()->ShowInt(retval_, "ExpectedStatesSet()");
    DbgPtr()->LeaveLevel();
  #endif
  return retval_;
}

/****************************************************************************/
void CalcDriver::PendingOutputFetched()
{
  #if CALCTEST_DEBUG8
    DbgPtr()->EnterLevel("PendingOutputFetched");
    DbgPtr()->ShowMessage("SetPendingResponse(Mcalc_IOState::OUTPUT_FETCHED)\n");
    DbgPtr()->LeaveLevel();
  #endif
  _ExpectedSignals->SetPendingResponse(Mcalc_IOState::OUTPUT_FETCHED);
}

/****************************************************************************/
void CalcDriver::PendingInputReceived()
{
  #if CALCTEST_DEBUG8
    DbgPtr()->EnterLevel("PendingInputReceived");
    DbgPtr()->ShowMessage("SetPendingResponse(Mcalc_IOState::INPUT_RECEIVED)\n");
    DbgPtr()->LeaveLevel();
  #endif
  _ExpectedSignals->SetPendingResponse(Mcalc_IOState::INPUT_RECEIVED);
}

/****************************************************************************/
void CalcDriver::PendingProcessSignal()
{
  #if CALCTEST_DEBUG8
    DbgPtr()->EnterLevel("PendingProcessSignal");
    DbgPtr()->ShowMessage("SetPendingResponse(Mcalc_IOState::PROCESS)\n");
    DbgPtr()->LeaveLevel();
  #endif
  _ExpectedSignals->SetPendingResponse(Mcalc_IOState::PROCESS);
}

/****************************************************************************/
void CalcDriver::AckErrorFetched()
{
  if (IoState() == Mcalc_IOState::ERROR_FETCH)
    SetIoState(Mcalc_IOState::CALC_ERROR, true);
}

/****************************************************************************/
void CalcDriver::SendMcalcSpawned(bool* Received_)
{
  #if ((CALCTEST_DEBUG6a)|(CALCTEST_DEBUG8))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("SendMcalcSpawned");
      _ExpectedSigSent = true;
    }
  #endif

  #if CALCTEST_DEBUG6b
    QMessageBox::warning(0, "DEBUG SendMcalcSpawned()",
                         QString("\nSendMcalcSpawned()"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #elif CALCTEST_DEBUG8
    DbgPtr()->ShowMessage("SendMcalcSpawned()\n");
    DbgPtr()->ShowMessage("expecting Mcalc_IOState::MCALC_SPAWNED response from client\n");
    DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
  #endif

  int PrevState_ = 0;
  int PrevGuiRequest_ = 0;
  int ClientResponse_ = 0;
  
  int ExpectedState_ = Mcalc_IOState::SPAWN_NEW_MCALC;
  int ResponseState_ = Mcalc_IOState::MCALC_SPAWNED;

  bool ResponseReceived_ = _ExpectedSignals->ResponseReceived(ResponseState_);

  // Reset last output acknowledgement signal if confirmed by client
  if (ResponseReceived_)
  {
    _ExpectedSignals->Reset();

    PrevState_ = (_PrevIoState == 0) ? _IoState:_PrevIoState;
    PrevGuiRequest_ = _IoState == ExpectedState_;
    ClientResponse_ = IsClientStates(ResponseState_);

    if (PrevState_ != ResponseState_ &&
        PrevGuiRequest_ && ClientResponse_)
      _PrevIoState = ResponseState_;
  }

  if (Received_)
    *Received_ = ResponseReceived_;

  #if CALCTEST_DEBUG8
    bool PrevIoSet_ = ResponseReceived_ &&
                      PrevState_ != ResponseState_ &&
                      PrevGuiRequest_ && ClientResponse_;
    
    DbgPtr()->ShowInt(ResponseReceived_, "ResponseReceived_");
    DbgPtr()->ShowInt(PrevIoSet_, "PrevIoSet_");
    DbgPtr()->ShowStr(IoStateToStr(_PrevIoState), "_PrevIoState");
  #endif

  ClearOutput();
  SetIoState(ResponseState_);
}

/****************************************************************************/
void CalcDriver::SendEndBatchFileAcked(bool* Received_)
{
  #if ((CALCTEST_DEBUG6a)|(CALCTEST_DEBUG8))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("SendEndBatchFileAcked");
      _ExpectedSigSent = true;
    }
  #endif

  #if CALCTEST_DEBUG6b
    QMessageBox::warning(0, "DEBUG SendEndBatchFileAcked()",
                         QString("\nSendEndBatchFileAcked()"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #elif CALCTEST_DEBUG8
    DbgPtr()->ShowMessage("SendEndBatchFileAcked()\n");
    DbgPtr()->ShowMessage("expecting Mcalc_IOState::BATCHFILE_ENDED_ACK response from client\n");
    DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
  #endif

  int PrevState_ = 0;
  int PrevGuiRequest_ = 0;
  int ClientResponse_ = 0;
  
  int ExpectedState_ = Mcalc_IOState::BATCHFILE_ENDED;
  int ResponseState_ = Mcalc_IOState::BATCHFILE_ENDED_ACK;

  bool ResponseReceived_ = _ExpectedSignals->ResponseReceived(ResponseState_);

  // Reset last output acknowledgement signal if confirmed by client
  if (ResponseReceived_)
  {
    _ExpectedSignals->Reset();

    PrevState_ = (_PrevIoState == 0) ? _IoState:_PrevIoState;
    PrevGuiRequest_ = _IoState == ExpectedState_;
    ClientResponse_ = IsClientStates(ResponseState_);

    if (!InProgressDataFile())
      SetInBatchFile(false, true);

    if (PrevState_ != ResponseState_ &&
        PrevGuiRequest_ && ClientResponse_)
      _PrevIoState = ResponseState_;
  }

  if (Received_)
  {
    *Received_ = ResponseReceived_;

    if (!InBatchFile())
      emit ProcessSignal();
  }

  #if CALCTEST_DEBUG8
    bool PrevIoSet_ = ResponseReceived_ &&
                      PrevState_ != ResponseState_ &&
                      PrevGuiRequest_ && ClientResponse_;
    
    DbgPtr()->ShowInt(ResponseReceived_, "ResponseReceived_");
    DbgPtr()->ShowInt(PrevIoSet_, "PrevIoSet_");
    DbgPtr()->ShowStr(IoStateToStr(_PrevIoState), "_PrevIoState");
  #endif

  ClearOutput();
  _SendingEndBatchFileAck = true;
  SetIoState(ResponseState_, !InBatchFile());
}

/****************************************************************************/
void CalcDriver::SendInputReceived(bool* Received_)
{
  #if ((CALCTEST_DEBUG6a)|(CALCTEST_DEBUG8))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("SendInputReceived");
      _ExpectedSigSent = true;
    }
  #endif

  #if CALCTEST_DEBUG6b
    QMessageBox::warning(0, "DEBUG SendInputReceived()",
                         QString("\nSendInputReceived()"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #elif CALCTEST_DEBUG8
    DbgPtr()->ShowMessage("SendInputReceived()\n");
    DbgPtr()->ShowMessage("expecting Mcalc_IOState::INPUT_RECEIVED response from client\n");
    DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
  #endif

  int PrevState_ = 0;
  int PrevGuiRequest_ = 0;
  int ClientResponse_ = 0;
  
  int ExpectedState1_ = Mcalc_IOState::INPUT_REQUIRED;
  int ExpectedState2_ = InBatchFile() ? Mcalc_IOState::CALC_RESPONSE:0;
  int ExpectedState3_ = InBatchFile() ? Mcalc_IOState::CALC_ERROR:0;
  int ExpectedState4_ = InBatchFile() ? Mcalc_IOState::ERROR_FETCH:0;
  int ExpectedState5_ = _ExpectedSignals->PendingResponse() ?
                          _ExpectedSignals->GetFromServer():0;

  int ResponseState_ = Mcalc_IOState::INPUT_RECEIVED;

  bool PendingResponse_ = _ExpectedSignals->PendingResponse();
  bool ResponseReceived_ = _ExpectedSignals->ResponseReceived(ResponseState_);

  // Reset last output acknowledgement signal if confirmed by client
  if (ResponseReceived_)
  {
    _ExpectedSignals->Reset();

    PrevState_ = (_PrevIoState == 0) ? _IoState:_PrevIoState;
    PrevGuiRequest_ = (_IoState == ExpectedState1_ ||
                       _IoState == ExpectedState2_ ||
                       _IoState == ExpectedState3_ ||
                       _IoState == ExpectedState4_ ||
                       _IoState == ExpectedState5_);
    ClientResponse_ = IsClientStates(ResponseState_);

    if (PrevState_ != ResponseState_ &&
        PrevGuiRequest_ && ClientResponse_)
      _PrevIoState = ResponseState_;
  }

  if (Received_)
    *Received_ = ResponseReceived_;

  #if CALCTEST_DEBUG8
    bool PrevIoSet_ = ResponseReceived_ &&
                      PrevState_ != ResponseState_ &&
                      PrevGuiRequest_ && ClientResponse_;

    DbgPtr()->ShowInt(PendingResponse_, "PendingResponse_");
    DbgPtr()->ShowInt(ResponseReceived_, "ResponseReceived_");
    DbgPtr()->ShowInt(PrevIoSet_, "PrevIoSet_");
    DbgPtr()->ShowStr(IoStateToStr(_PrevIoState), "_PrevIoState");
  #endif

  if (PendingResponse_)
  {
    _ExpectedSignals->ResetPendingResponse();

    #if CALCTEST_DEBUG8
      DbgPtr()->LeaveLevel();
    #endif
  }
  else
    SetIoState(ResponseState_);
}

/****************************************************************************/
void CalcDriver::SendClientProcessSignal(bool* Received_)
{
  #if ((CALCTEST_DEBUG6a)|(CALCTEST_DEBUG8))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("SendClientProcessSignal");
      _ExpectedSigSent = true;
    }
  #endif

  #if CALCTEST_DEBUG6b
    QMessageBox::warning(0, "DEBUG SendClientProcessSignal()",
                         QString("\nSendClientProcessSignal()"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #elif CALCTEST_DEBUG8
    DbgPtr()->ShowMessage("SendClientProcessSignal()\n");
    DbgPtr()->ShowMessage("expecting Mcalc_IOState::PROCESS response from client\n");
    DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
  #endif

  int PrevState_ = 0;
  int PrevGuiRequest_ = 0;
  int ClientResponse_ = 0;

  int ExpectedState1_ = Mcalc_IOState::CALC_RESPONSE;
  int ExpectedState2_ = Mcalc_IOState::CALC_ERROR;
  int ExpectedState3_ = Mcalc_IOState::ERROR_FETCH;
  int ExpectedState4_ = _ExpectedSignals->PendingResponse() ?
                          _ExpectedSignals->GetFromServer():0;
  
  int ResponseState_ = Mcalc_IOState::PROCESS;

  bool PendingResponse_ = _ExpectedSignals->PendingResponse();
  bool ResponseReceived_ = _ExpectedSignals->ResponseReceived(ResponseState_);

  // Reset last output acknowledgement signal if confirmed by client
  if (ResponseReceived_)
  {
    _ExpectedSignals->Reset();

    PrevState_ = (_PrevIoState == 0) ? _IoState:_PrevIoState;
    PrevGuiRequest_ = (_IoState == ExpectedState1_ ||
                       _IoState == ExpectedState2_ ||
                       _IoState == ExpectedState3_ ||
                       _IoState == ExpectedState4_);
    ClientResponse_ = IsClientStates(ResponseState_);

    if (PrevState_ != ResponseState_ &&
        PrevGuiRequest_ && ClientResponse_)
      _PrevIoState = ResponseState_;
  }

  if (Received_)
  {
    *Received_ = ResponseReceived_;
    emit ProcessSignal();
  }

  #if CALCTEST_DEBUG8
    bool PrevIoSet_ = ResponseReceived_ &&
                      PrevState_ != ResponseState_ &&
                      PrevGuiRequest_ && ClientResponse_;

    DbgPtr()->ShowInt(PendingResponse_, "PendingResponse_");
    DbgPtr()->ShowInt(ResponseReceived_, "ResponseReceived_");
    DbgPtr()->ShowInt(PrevIoSet_, "PrevIoSet_");
    DbgPtr()->ShowStr(IoStateToStr(_PrevIoState), "_PrevIoState");
  #endif
  
  if (PendingResponse_)
  {
    _ExpectedSignals->ResetPendingResponse();

    #if CALCTEST_DEBUG8
      DbgPtr()->LeaveLevel();
    #endif    
  }
  else
    SetIoState(ResponseState_);
}

/****************************************************************************/
void CalcDriver::SendProcessDoneAcked(bool* Received_)
{
  #if ((CALCTEST_DEBUG6a)|(CALCTEST_DEBUG8))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("SendProcessDoneAcked");
      _ExpectedSigSent = true;
    }
  #endif

  #if CALCTEST_DEBUG6b
    QMessageBox::warning(0, "DEBUG SendProcessDoneAcked()",
                         QString("\nSendProcessDoneAcked()"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #elif CALCTEST_DEBUG8
    DbgPtr()->ShowMessage("SendProcessDoneAcked()\n");
    DbgPtr()->ShowMessage("expecting Mcalc_IOState::PROCESS_DONE_ACK response from client\n");
    DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
  #endif

  int PrevState_ = 0;
  int PrevGuiRequest_ = 0;
  int ClientResponse_ = 0;
  
  int ExpectedState_ = Mcalc_IOState::PROCESS_DONE;
  int ResponseState_ = Mcalc_IOState::PROCESS_DONE_ACK;

  bool ResponseReceived_ = _ExpectedSignals->ResponseReceived(ResponseState_);

  // Reset last output acknowledgement signal if confirmed by client
  if (ResponseReceived_)
  {
    _ExpectedSignals->Reset();

    PrevState_ = (_PrevIoState == 0) ? _IoState:_PrevIoState;
    PrevGuiRequest_ = _IoState == ExpectedState_;
    ClientResponse_ = IsClientStates(ResponseState_);

    if (PrevState_ != ResponseState_ &&
        PrevGuiRequest_ && ClientResponse_)
      _PrevIoState = ResponseState_;
  }

  if (Received_)
  {
    *Received_ = ResponseReceived_;

    if (!InBatchFile())
      emit ProcessSignal();    
  }

  #if CALCTEST_DEBUG8
    bool PrevIoSet_ = ResponseReceived_ &&
                      PrevState_ != ResponseState_ &&
                      PrevGuiRequest_ && ClientResponse_;
    
    DbgPtr()->ShowInt(ResponseReceived_, "ResponseReceived_");
    DbgPtr()->ShowInt(PrevIoSet_, "PrevIoSet_");
    DbgPtr()->ShowStr(IoStateToStr(_PrevIoState), "_PrevIoState");
  #endif

  SetIoState(ResponseState_, !InBatchFile());
}

/****************************************************************************/
void CalcDriver::SendOutputFetched(bool* Received_)
{
  #if ((CALCTEST_DEBUG6a)|(CALCTEST_DEBUG8))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("SendOutputFetched");
      _ExpectedSigSent = true;
    }
  #endif

  #if CALCTEST_DEBUG6b
    QMessageBox::warning(0, "DEBUG SendOutputFetched()",
                         QString("\nSendOutputFetched()"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #elif CALCTEST_DEBUG8
    DbgPtr()->ShowMessage("SendOutputFetched()\n");
    if (_FetchError)
      DbgPtr()->ShowMessage("expecting Mcalc_IOState::CALC_ERROR response from client\n");
    else
      DbgPtr()->ShowMessage("expecting Mcalc_IOState::OUTPUT_FETCHED response from client\n");
    DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
  #endif

  int PrevState_ = 0;
  int PrevGuiRequest_ = 0;
  int ClientResponse_ = 0;
  
  int ExpectedState1_ = Mcalc_IOState::OUTPUT_READY;
  int ExpectedState2_ = Mcalc_IOState::GRAPH_OUTPUT;
  int ResponseState_ = !_FetchError ? Mcalc_IOState::OUTPUT_FETCHED:
                                      Mcalc_IOState::CALC_ERROR;

  bool PendingResponse_ = _ExpectedSignals->PendingResponse();
  bool ResponseReceived_ = _ExpectedSignals->ResponseReceived(ResponseState_);

  // Reset last output acknowledgement signal if confirmed by client
  if (ResponseReceived_)
  {
    _ExpectedSignals->Reset();

    PrevState_ = (_PrevIoState == 0) ? _IoState:_PrevIoState;
    PrevGuiRequest_ = (_FetchError ||
                       _IoState == ExpectedState1_ ||
                       _IoState == ExpectedState2_);
    ClientResponse_ = IsClientStates(ResponseState_);

    if (PrevState_ != ResponseState_ &&
        PrevGuiRequest_ && ClientResponse_)
      _PrevIoState = ResponseState_;
  }

  if (Received_)
  {
    *Received_ = ResponseReceived_;

    if (_FetchError || !InBatchFile())
      emit ProcessSignal();    
  }

  #if CALCTEST_DEBUG8
    bool PrevIoSet_ = ResponseReceived_ &&
                      PrevState_ != ResponseState_ &&
                      PrevGuiRequest_ && ClientResponse_;

    DbgPtr()->ShowInt(PendingResponse_, "PendingResponse_");
    DbgPtr()->ShowInt(ResponseReceived_, "ResponseReceived_");
    DbgPtr()->ShowInt(PrevIoSet_, "PrevIoSet_");
    DbgPtr()->ShowStr(IoStateToStr(_PrevIoState), "_PrevIoState");
  #endif
  
  if (!_FetchError)
  {
    ClearOutput();

    if (PendingResponse_)
    {
      _ExpectedSignals->ResetPendingResponse();

      #if CALCTEST_DEBUG8
        DbgPtr()->LeaveLevel();
      #endif
    }
    else
      SetIoState(ResponseState_, !InBatchFile());
  }
  else
  {
    _FetchError = false;

    if (PendingResponse_)
    {
      _ExpectedSignals->ResetPendingResponse();

      #if CALCTEST_DEBUG8
        DbgPtr()->LeaveLevel();
      #endif
    }
    else
      SetIoState(ResponseState_, true);
  }
}

/****************************************************************************/
void CalcDriver::SendGraphWaitFetched(bool* Received_)
{
  #if ((CALCTEST_DEBUG6a)|(CALCTEST_DEBUG8))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("SendGraphWaitFetched");
      _ExpectedSigSent = true;
    }
  #endif

  #if CALCTEST_DEBUG6b
    QMessageBox::warning(0, "DEBUG SendGraphWaitFetched()",
                         QString("\nSendGraphWaitFetched()"),
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #elif CALCTEST_DEBUG8
    DbgPtr()->ShowMessage("SendGraphWaitFetched()\n");
    DbgPtr()->ShowMessage("expecting Mcalc_IOState::GRAPH_WAIT_ACK response from client\n");
    DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
  #endif

  int PrevState_ = 0;
  int PrevGuiRequest_ = 0;
  int ClientResponse_ = 0;
  
  int ExpectedState_ = Mcalc_IOState::GRAPH_WAIT;
  int ResponseState_ = Mcalc_IOState::GRAPH_WAIT_ACK;

  bool ResponseReceived_ = _ExpectedSignals->ResponseReceived(ResponseState_);

  // Reset last output acknowledgement signal if confirmed by client
  if (ResponseReceived_)
  {
    _ExpectedSignals->Reset();

    PrevState_ = (_PrevIoState == 0) ? _IoState:_PrevIoState;
    PrevGuiRequest_ = _IoState == ExpectedState_;
    ClientResponse_ = IsClientStates(ResponseState_);

    if (PrevState_ != ResponseState_ &&
        PrevGuiRequest_ && ClientResponse_)
      _PrevIoState = ResponseState_;
  }

  if (Received_)
    *Received_ = ResponseReceived_;

  #if CALCTEST_DEBUG8
    bool PrevIoSet_ = ResponseReceived_ &&
                      PrevState_ != ResponseState_ &&
                      PrevGuiRequest_ && ClientResponse_;
    
    DbgPtr()->ShowInt(ResponseReceived_, "ResponseReceived_");
    DbgPtr()->ShowInt(PrevIoSet_, "PrevIoSet_");
    DbgPtr()->ShowStr(IoStateToStr(_PrevIoState), "_PrevIoState");
  #endif

  ClearOutput();
  SetIoState(ResponseState_);
}

/****************************************************************************/
void CalcDriver::SendGraphProgressFetched(bool* Received_)
{
  #if ((CALCTEST_DEBUG6c)|(CALCTEST_DEBUG8))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("SendGraphProgressFetched");
      _ExpectedSigSent = true;
    }
  #endif

  #if CALCTEST_DEBUG8
    DbgPtr()->ShowMessage("SendGraphProgressFetched()\n");
    DbgPtr()->ShowMessage("expecting Mcalc_IOState::GRAPH_PROGRESS_ACK response from client\n");
    DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
  #endif

  int PrevState_ = 0;
  int PrevGuiRequest_ = 0;
  int ClientResponse_ = 0;
  
  int ExpectedState_ = Mcalc_IOState::GRAPH_PROGRESS;
  int ResponseState_ = Mcalc_IOState::GRAPH_PROGRESS_ACK;

  bool ResponseReceived_ = _ExpectedSignals->ResponseReceived(ResponseState_);

  // Reset last output acknowledgement signal if confirmed by client
  if (ResponseReceived_)
  {
    _ExpectedSignals->Reset();

    PrevState_ = (_PrevIoState == 0) ? _IoState:_PrevIoState;
    PrevGuiRequest_ = _IoState == ExpectedState_;
    ClientResponse_ = IsClientStates(ResponseState_);

    if (PrevState_ != ResponseState_ &&
        PrevGuiRequest_ && ClientResponse_)
      _PrevIoState = ResponseState_;
  }

  if (Received_)
    *Received_ = ResponseReceived_;

  #if CALCTEST_DEBUG8
    bool PrevIoSet_ = ResponseReceived_ &&
                      PrevState_ != ResponseState_ &&
                      PrevGuiRequest_ && ClientResponse_;
    
    DbgPtr()->ShowInt(ResponseReceived_, "ResponseReceived_");
    DbgPtr()->ShowInt(PrevIoSet_, "PrevIoSet_");
    DbgPtr()->ShowStr(IoStateToStr(_PrevIoState), "_PrevIoState");
  #endif

  ClearOutput();
  SetIoState(ResponseState_);
}

/****************************************************************************/
void CalcDriver::SendProgressFetched(bool* Received_)
{
  #if ((CALCTEST_DEBUG6a)|(CALCTEST_DEBUG8))
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("SendProgressFetched");
      _ExpectedSigSent = true;
    }
  #endif
  
  #if CALCTEST_DEBUG8
    DbgPtr()->ShowMessage("SendProgressFetched()\n");
    DbgPtr()->ShowMessage("expecting Mcalc_IOState::INFILE_PROGRESS_ACK response from client\n");
    DbgPtr()->ShowStr(IoStateToStr(_ExpectedSignals->GetClientExpected()), "GetClientExpected()");
  #endif

  int PrevState_ = 0;
  int PrevGuiRequest_ = 0;
  int ClientResponse_ = 0;
  
  int ExpectedState_ = Mcalc_IOState::PROGRESS_READY;
  int ResponseState_ = Mcalc_IOState::INFILE_PROGRESS_ACK;

  bool ResponseReceived_ = _ExpectedSignals->ResponseReceived(ResponseState_);

  // Reset last output acknowledgement signal if confirmed by client
  if (ResponseReceived_)
  {
    _ExpectedSignals->Reset();

    PrevState_ = (_PrevIoState == 0) ? _IoState:_PrevIoState;
    PrevGuiRequest_ = _IoState == ExpectedState_;
    ClientResponse_ = IsClientStates(ResponseState_);

    if (PrevState_ != ResponseState_ &&
        PrevGuiRequest_ && ClientResponse_)
      _PrevIoState = ResponseState_;
  }

  if (Received_)
    *Received_ = ResponseReceived_;

  #if CALCTEST_DEBUG8
    bool PrevIoSet_ = ResponseReceived_ &&
                      PrevState_ != ResponseState_ &&
                      PrevGuiRequest_ && ClientResponse_;
    
    DbgPtr()->ShowInt(ResponseReceived_, "ResponseReceived_");
    DbgPtr()->ShowInt(PrevIoSet_, "PrevIoSet_");
    DbgPtr()->ShowStr(IoStateToStr(_PrevIoState), "_PrevIoState");
  #endif

  ClearOutput();
  SetIoState(ResponseState_);
}

/****************************************************************************/
void CalcDriver::ClearInput()
{
  WriteToInput("");
}

/****************************************************************************/
void CalcDriver::ClearOutput()
{
  #if CALCDRIVER_GUI_VERSION
    _Output->clear();
  #else
    _Output[0] = 0;
  #endif
}

/*****************************************************************************/
void CalcDriver::IncrBatchLevel()
{
  _BatchExecLevel++;
  _InBatchFile = true;
}

/*****************************************************************************/
void CalcDriver::DecrBatchLevel()
{
  if (_BatchExecLevel > 0)
  {
    _BatchExecLevel--;

    if (!_BatchExecLevel)
      _InBatchFile = false;
  }
  else if (!_BatchExecLevel)
    _InBatchFile = false;
}

/*****************************************************************************/
void CalcDriver::SetInBatchFile(bool ExecFound_, bool Reset_)
{
  int x;
  int FileLen_ = 0;
  
  bool Set_ = false;
  bool PauseSwitch_ = false;
  bool InvalidSwitch_ = false;

  const char* CommaPt_ = NULL;

  QString ExecCmdStr_;
  QString ExecOptions_;
  ExecutingProgramName* OldNode_;

  if (Reset_)
  {
    if (_ExecuteProgram)
    {
      _ExecuteProgram = _ExecuteProgram->PopLevel(OldNode_);
      
      if (OldNode_)
      {
        delete OldNode_;
        _InBatchFile = true;

        DecrBatchLevel();
      }
      else
      {
        delete _ExecuteProgram;
        _ExecuteProgram = NULL;

        DecrBatchLevel();
      }
    }
  }
  else if (!_InBatchFile)
  {
    _InBatchFile = Set_ = ExecFound_;

    if (_InBatchFile)
      IncrBatchLevel();
  }

  if (Set_ && _ExecuteProgram)
  {
    FileLen_ = _ExecuteProgram->NameLength();
    CommaPt_ = ::strchr(_ExecuteProgram->Name(), ',');

    if (FileLen_ && CommaPt_)
    {
      if (CommaPt_[1])
      {
        ExecOptions_ = &CommaPt_[1];
        ExecOptions_.lower();
        ExecOptions_.stripWhiteSpace();

        if (ExecOptions_ == "nopause")
          PauseSwitch_ = FALSE;
        else if (ExecOptions_ == "pause")
          PauseSwitch_ = TRUE;
        else
          InvalidSwitch_ = TRUE;
      }
      else
      {
        ExecOptions_ = " ";
        InvalidSwitch_ = TRUE;
      }

      if (!InvalidSwitch_)
      {
        x = CommaPt_ - _ExecuteProgram->Name();
        _ExecuteProgram->NameIndexAt(x, 0);

        ExecCmdStr_ = _ExecuteProgram->Name();
        ExecCmdStr_.lower();
        ExecCmdStr_.stripWhiteSpace();
        _ExecuteProgram->ReplaceName(ExecCmdStr_.ascii());
        FileLen_ = _ExecuteProgram->NameLength();
      }
    }
    else if (FileLen_)
    {
      PauseSwitch_ = TRUE;

      ExecCmdStr_ = _ExecuteProgram->Name();
      ExecCmdStr_.lower();
      ExecCmdStr_.stripWhiteSpace();
      _ExecuteProgram->ReplaceName(ExecCmdStr_.ascii());
      FileLen_ = _ExecuteProgram->NameLength();
    }

    if (InvalidSwitch_)
    {
      if (_ExecuteProgram)
      {
        _ExecuteProgram = _ExecuteProgram->PopLevel(OldNode_);

        if (OldNode_)
        {
          delete OldNode_;
          _InBatchFile = true;

          DecrBatchLevel();
        }
        else
        {
          delete _ExecuteProgram;
          _ExecuteProgram = NULL;

          DecrBatchLevel();
        }
      }
    }
    else
    {
      if (icasestrcomp(_ExecuteProgram->Name()+FileLen_-3, ".in"))
      {
        _ExecuteProgram->SetAsInputFile(true);
        _ExecuteProgram->SetPauseSwitch(PauseSwitch_);
      }
      else
        _ExecuteProgram->SetAsInputFile(false);
    }
  }
}

/*****************************************************************************/
const char* CalcDriver::ProgramName(char* Name_)
{
  size_t x;
  size_t len = Name_ ? ::strlen(Name_):0;

  if (len)
  {
    for (x = ::strlen(Name_); x && isspace(Name_[x - 1]); x--);
    Name_[x] = 0;
    for (x = 0; isspace(Name_[x]); x++);
    if (x && Name_ && (&Name_[x]))
      memmove(Name_, &Name_[x], strlen(&Name_[x]) + 1);

    if (strlen(Name_) > 0)
      return Name_;
  }

  return NULL;
}

/*****************************************************************************/
const char* CalcDriver::IsExecCmd(char* Str_)
{
  bool ExecFound_ = false;

  const char* ExecPrgPt_ = NULL;
  char* pt1 = NULL;
  char* pt2 = NULL;

  if (icasestrcompn(Str_, "exec", 4, true))
  {
    pt1 = strchr(Str_, 'e');
    pt2 = strchr(Str_, 'E');
    
    if (!pt1 || (pt2 && pt2 < pt1))
      pt1 = pt2;
    
    ExecFound_ = pt1 && (ExecPrgPt_ = ProgramName(pt1+4));
    
    if (!ExecFound_)
      ExecPrgPt_ = NULL;
  }
  
  if (ExecFound_ && ExecPrgPt_)
  {
    if (_ExecuteProgram)
      _ExecuteProgram->SetNewProgramName(ExecPrgPt_);
    else
      _ExecuteProgram = new ExecutingProgramName(ExecPrgPt_);
      
    return ExecPrgPt_;
  }

  return NULL;
}

/****************************************************************************/
char* CalcDriver::RemovePadSpace(char* str)
{
  if (!str)
    return str;
    
  int x, l;

  for (l = 0; str[l] && isspace(str[l]); l++);
  for (x = 0; (str[x] = str[l]); x++, l++);

  for (l = strlen(str); l;)
    if (isspace(str[--l]))
      str[l] = '\0';
    else
      break;

  return str;
}

/****************************************************************************/
int CalcDriver::EvalExpression(const char* Buffer_, int InputState_, bool ExecCmd_)
{
  #if CALCTEST_DEBUG
      QMessageBox::warning(0, "DEBUG",
                           QString("Evaluating expr: ") + QString(Buffer_),
                           QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #endif

  #if CALCDRIVER_TEXT_VERSION
    if (_ProgramExited && (_Halted || strcmp(Buffer_, "quit") != 0))
      exit(0);
  #endif
  
  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
    {
      DbgPtr()->EnterLevel("CalcDriver::EvalExpression");
      DbgPtr()->ShowStr(Buffer_, "Buffer_");
    }
  #endif

  int ResponseSignal_ = 0;

  if (_ExpectedSignals->PendingResponse())
  {
    ResponseSignal_ = _ExpectedSignals->PendingResponse();
    
    if (ResponseSignal_ & Mcalc_IOState::OUTPUT_FETCHED)
      SendOutputFetched();
    else if (ResponseSignal_ & Mcalc_IOState::INPUT_RECEIVED)
      SendInputReceived();
    else if (ResponseSignal_ & Mcalc_IOState::PROCESS)
      SendClientProcessSignal();
  }

  bool EmptyBuffer_ = !Buffer_ || strlen(Buffer_) == 0;
  char* StrBuffer_ = !EmptyBuffer_ ? NewString(Buffer_):NULL;
  
  if (StrBuffer_)
  {
    StrBuffer_ = RemovePadSpace(StrBuffer_);
    EmptyBuffer_ = strlen(StrBuffer_) == 0;
  }

  bool InBatch_ = InBatchFile();
  bool ExecCalcCmdFound_ = !EmptyBuffer_ ? (IsExecCmd(StrBuffer_) != NULL):false;
  bool SetStateOnly_ =
         (InputState_ == Mcalc_IOState::OUTPUT_FETCHED && !InBatch_) ||
         (InputState_ == Mcalc_IOState::PROCESS_DONE_ACK && !InBatch_) ||
         (InputState_ == Mcalc_IOState::BATCHFILE_ENDED_ACK && !InBatch_);
  bool SetStateAndWaitForIo_ =
         (InputState_ == Mcalc_IOState::OUTPUT_FETCHED && InBatch_) ||
         (InputState_ == Mcalc_IOState::PROCESS_DONE_ACK && InBatch_) ||
         (InputState_ == Mcalc_IOState::BATCHFILE_ENDED_ACK && InBatch_);
  
  bool ReadyForProcessSignal_ =
         SetStateOnly_ ||
         InputState_ == Mcalc_IOState::MCALC_SPAWNED ||
         InputState_ == Mcalc_IOState::BREAK_PROGRAM;

  if (!EmptyBuffer_)
  {
    strcpy(StrBuffer_, Buffer_);
    StrBuffer_ = RemovePadSpace(StrBuffer_);
    EmptyBuffer_ = strlen(StrBuffer_) == 0;
  }

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
    {
      DbgPtr()->ShowStr(StrBuffer_, "StrBuffer_");
      DbgPtr()->ShowInt(EmptyBuffer_, "EmptyBuffer_");
      DbgPtr()->ShowInt(ExecCalcCmdFound_, "ExecCalcCmdFound_");
      DbgPtr()->ShowInt(ReadyForProcessSignal_, "ReadyForProcessSignal_");
      DbgPtr()->ShowStr(IoStateToStr(InputState_), "InputState_");
      DbgPtr()->ShowInt(ExecCmd_, "ExecCmd_");
    }
  #endif

  if ((!EmptyBuffer_ && ExecCmd_ && strcmp(StrBuffer_, "quit") == 0) ||
      InputState_ == Mcalc_IOState::CALC_HALT)
  {
    #if CALCDRIVER_TEXT_VERSION
      WriteToInput(StrBuffer_);
      DoHalt();
      exit(0);
    #else
      DoHalt();
      delete[] StrBuffer_;
      StrBuffer_ = NULL;
      
      return FALSE;
    #endif
  }
  else if (!EmptyBuffer_ && ExecCmd_ && strcmp(StrBuffer_, "break") == 0 &&
             (InputState_ != Mcalc_IOState::BREAK_PROGRAM &&
              ReadyForProcessSignal_))
  {
    _RequestResend = false;
    _ResendSigDone = false;
    _ResendLastSig = false;
    _ResendSigUnchanged = false;

    delete[] StrBuffer_;
    StrBuffer_ = NULL;
    SetIoState(Mcalc_IOState::BREAK_PROGRAM, true);
  }
  else if (!EmptyBuffer_ && ExecCmd_ && strcmp(StrBuffer_, "pausepolling") == 0)
  {
    if (ReadyForProcessSignal_)
      InputState_ = Mcalc_IOState::PROCESS;

    PauseCalcPolling();

    if (InputState_ == Mcalc_IOState::PROCESS)
    {
      emit ProcessSignal();
      _RequestResend = false;
      _ResendSigDone = false;
      _ResendLastSig = false;
      _ResendSigUnchanged = false;
    
      SetInBatchFile(ExecCalcCmdFound_, false);
      WriteToInput(StrBuffer_);

      delete[] StrBuffer_;
      StrBuffer_ = NULL;
      SetIoState(InputState_);

      return TRUE;
    }
    else
      return FALSE;
  }
  else if (!EmptyBuffer_ && ExecCmd_ && strcmp(StrBuffer_, "resumepolling") == 0)
  {
    if (ReadyForProcessSignal_)
      InputState_ = Mcalc_IOState::PROCESS;

    ResumeCalcPolling();

    if (InputState_ == Mcalc_IOState::PROCESS)
    {
      emit ProcessSignal();
      _RequestResend = false;
      _ResendSigDone = false;
      _ResendLastSig = false;
      _ResendSigUnchanged = false;
    
      SetInBatchFile(ExecCalcCmdFound_, false);
      WriteToInput(StrBuffer_);

      delete[] StrBuffer_;
      StrBuffer_ = NULL;
      SetIoState(InputState_);

      return TRUE;
    }
    else
      return FALSE;
  }
  else
  {
    if (ReadyForProcessSignal_ && !SetStateAndWaitForIo_)
      InputState_ = Mcalc_IOState::PROCESS;

    if (SuspendPolling())
    {
      PauseCalcPolling();
      SetSuspendPolling(false);
      SetResumePolling(true);
    }

    if (!IsServerStates(InputState_) &&
        !(EmptyBuffer_ && InputState_ == Mcalc_IOState::PROCESS))
    {
      if (InputState_ == Mcalc_IOState::PROCESS)
        emit ProcessSignal();

      _RequestResend = false;
      _ResendSigDone = false;
      _ResendLastSig = false;
      _ResendSigUnchanged = false;
    
      SetInBatchFile(ExecCalcCmdFound_, false);
      WriteToInput(EmptyBuffer_ ? "":StrBuffer_);
      SetEmptyBuffer(EmptyBuffer_);
      
      delete[] StrBuffer_;
      StrBuffer_ = NULL;
      SetIoState(InputState_);

      return TRUE;
    }
    else
      return FALSE;
  }

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif

  return TRUE;
}

/****************************************************************************/
void CalcDriver::SpawnNewMcalc()
{
  QString Num_;
  Num_ = Num_.setNum(_SessionNum);
  
  if (_Spawned)
    return;

  #if CALCTEST_DEBUG2
    QMessageBox::warning(0, "DEBUG",
                         QString("calling mcalc with session # = ") + Num_,
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);  
  #endif

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->EnterLevel("CalcDriver::SpawnNewMcalc");
  #endif

  // Replace with process execution code
  #if defined(__linux__) | defined(__unix__)
    #if CALCTEST_EXEC_CURRENTDIR
      QString ProgName_ = QString("./mcalc");
    #else
      QString ProgName_ = QString("mcalc");
    #endif
  #else
    #if CALCTEST_EXEC_CURRENTDIR
      QString ProgName_ = QString(".\\mcalc.exe");
    #else
      QString ProgName_ = QString("mcalc.exe");
    #endif
  #endif

  #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
    wchar_t* envstr_;
    wchar_t* envval_;
    
    LPCWSTR pwenvstr_;
    LPCWSTR pwenvval_;
  #endif

  int x;
  int cnt;
  int spos = 0;
  int lines = 0;
  int argvpos = 0;
  int envspos = 0;
  int nxline = 0;
  int cmpres_ = 0;
  int argc = 0;

  bool argsfnd = false;
  bool allargsfnd = false;

  const char* ptr = strchr(_OutputBuffer, '\n');

  _ProcPtr = NULL;
  
  if (ptr)
  {
    spos = ptr - _OutputBuffer;
    _OutputBuffer[spos] = 0;
    
    cnt = atoi(_OutputBuffer);
    lines = cnt;
    argsfnd = lines > 0;
    _OutputBuffer[spos++] = '\n';
  }

  if (argsfnd)
  {
    ptr = strchr(&_OutputBuffer[spos], ':');
    argsfnd = false;

    if (ptr)
    {
      x = spos;
      spos = ptr - _OutputBuffer;
      _OutputBuffer[spos] = 0;

      argc = atoi(&_OutputBuffer[x]);
      argsfnd = argc > 0;
      _OutputBuffer[spos++] = ':';
    }

    if (argsfnd)
    {
      _ProcPtr = new QProcess(this);
      _ProcPtr->addArgument(ProgName_);

      #if CALCTEST_DEBUG3
        DbgPtr()->ShowStr(ProgName_, "_ProcPtr->addArgument(ProgName_)");
      #endif
      
      argvpos = spos;
      cnt = 0;
      ptr = strtok(&_OutputBuffer[argvpos], " ");
      
      for (x = 0; ptr && x < argc; x++)
      {
        cmpres_ = strcmp(ptr, "mcalc");
        
        // omit adding "mcalc" program name argument because already added
        if (cmpres_ != 0 || !strstr(ProgName_, ptr))
        {
          _ProcPtr->addArgument(ptr);

          #if CALCTEST_DEBUG3
            DbgPtr()->ShowStr(ptr, "_ProcPtr->addArgument(ptr)");
            DbgPtr()->ShowInt(cmpres_, "strcmp(ptr, \"mcalc\")");
          #endif
        }

        ptr = strtok(NULL, " ");

        if (ptr)
          ++cnt;
      }

      allargsfnd = lines && cnt && argc && x == argc;

      if (allargsfnd)
      {
        ptr = strchr(&_OutputBuffer[argvpos], '\n');
        
        if (ptr && lines > 1)
        {
          ++ptr;

          if (*ptr)
            envspos = ptr - _OutputBuffer;
          else
            envspos = 0;

          if (envspos)
          {
            ptr = strchr(&_OutputBuffer[envspos], '=');
            x = ptr - &_OutputBuffer[envspos];
            _OutputBuffer[envspos+x] = 0;
            x++;
            #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
              mbstowcs(envstr_, &_OutputBuffer[envspos], strlen(&_OutputBuffer[envspos]));
              mbstowcs(envval_, &_OutputBuffer[envspos+x], strlen(&_OutputBuffer[envspos+x]));
              pwenvstr_ = envstr_;
              pwenvval_ = envval_;
              SetEnvironmentVariable(pwenvstr_, pwenvval_);
            #else
              setenv(&_OutputBuffer[envspos],
                     &_OutputBuffer[envspos+x], 1);
            #endif
            _OutputBuffer[envspos+x-1] = '=';

            if (lines > 2)
            {
              ptr = strchr(&_OutputBuffer[envspos], '\n');
              ++ptr;

              if (*ptr)
              {
                nxline = ptr - &_OutputBuffer[envspos];
              
                ptr = strchr(&_OutputBuffer[envspos+nxline], '=');
                x = ptr - &_OutputBuffer[envspos+nxline];
                _OutputBuffer[envspos+nxline+x] = 0;
                x++;
                #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
                  mbstowcs(envstr_, &_OutputBuffer[envspos+nxline], strlen(&_OutputBuffer[envspos+nxline]));
                  mbstowcs(envval_, &_OutputBuffer[envspos+nxline+x], strlen(&_OutputBuffer[envspos+nxline]));
                  pwenvstr_ = envstr_;
                  pwenvval_ = envval_;
                  SetEnvironmentVariable(pwenvstr_, pwenvval_);
                #else
                  setenv(&_OutputBuffer[envspos+nxline],
                         &_OutputBuffer[envspos+nxline+x], 1);
                #endif
                _OutputBuffer[envspos+nxline+x-1] = '=';
              }
              else
                lines = 2;
            }
          }
        }
      }
    }
  }

  if (_ProcPtr)
  {
    _ProcPtr->setCommunication(QProcess::Stdin | QProcess::Stdout);

    connect(_ProcPtr, SIGNAL(processExited()),
            this, SLOT(ProcessExited()));

    #if CALCTEST_DEBUG3
      DbgPtr()->ShowMessage("_ProcPtr->start()\n");
    #endif

    if (!_ProcPtr->start() && allargsfnd)
    {
      #if CALCTEST_EXEC_CURRENTDIR
        #if defined(__linux__) | defined(__unix__)
          ProgName_ = QString("mcalc");
        #else
          ProgName_ = QString("mcalc.exe");
        #endif

        #if CALCTEST_DEBUG3
          DbgPtr()->ShowMessage("_ProcPtr->start() == false\n");
        #endif

        delete _ProcPtr;
        _ProcPtr = new QProcess(this);
        _ProcPtr->addArgument(ProgName_);

        #if CALCTEST_DEBUG3
          DbgPtr()->ShowStr(ProgName_, "_ProcPtr->addArgument(ProgName_)");
        #endif

        ptr = strtok(&_OutputBuffer[argvpos], " ");
      
        for (x = 0; ptr && x < argc; x++)
        {
          cmpres_ = strcmp(ptr, "mcalc");

          // omit adding "mcalc" program name argument because already added
          if (cmpres_ != 0 || !strstr(ProgName_, ptr))
          {
            _ProcPtr->addArgument(ptr);

            #if CALCTEST_DEBUG3
              DbgPtr()->ShowStr(ptr, "_ProcPtr->addArgument(ptr)");
              DbgPtr()->ShowInt(cmpres_, "strcmp(ptr, \"mcalc\")");
            #endif
          }

          ptr = strtok(NULL, " ");
        }
        
        _ProcPtr->setCommunication(QProcess::Stdin | QProcess::Stdout);

        if (envspos)
        {
          ptr = strchr(&_OutputBuffer[envspos], '=');
          x = ptr - &_OutputBuffer[envspos];
          _OutputBuffer[envspos+x] = 0;
          x++;
          #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
            mbstowcs(envstr_, &_OutputBuffer[envspos], strlen(&_OutputBuffer[envspos]));
            mbstowcs(envval_, &_OutputBuffer[envspos+x], strlen(&_OutputBuffer[envspos+x]));
            pwenvstr_ = envstr_;
            pwenvval_ = envval_;
            SetEnvironmentVariable(pwenvstr_, pwenvval_);
          #else
            setenv(&_OutputBuffer[envspos],
                   &_OutputBuffer[envspos+x], 1);
          #endif
          _OutputBuffer[envspos+x-1] = '=';

          if (lines > 2)
          {
            ptr = strchr(&_OutputBuffer[envspos], '\n');
            ++ptr;

            if (*ptr)
            {
              nxline = ptr - &_OutputBuffer[envspos];
              
              ptr = strchr(&_OutputBuffer[envspos+nxline], '=');
              x = ptr - &_OutputBuffer[envspos+nxline];
              _OutputBuffer[envspos+nxline+x] = 0;
              x++;
              #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
                mbstowcs(envstr_, &_OutputBuffer[envspos+nxline], strlen(&_OutputBuffer[envspos+nxline]));
                mbstowcs(envval_, &_OutputBuffer[envspos+nxline+x], strlen(&_OutputBuffer[envspos+nxline+x]));
                pwenvstr_ = envstr_;
                pwenvval_ = envval_;
                SetEnvironmentVariable(pwenvstr_, pwenvval_);
              #else
                setenv(&_OutputBuffer[envspos+nxline],
                       &_OutputBuffer[envspos+nxline+x], 1);
              #endif
              _OutputBuffer[envspos+nxline+x-1] = '=';
            }
            else
              lines = 2;
          }
        }
        
        if (!_ProcPtr->start())
        {
          if (_FatalErrCount >= MAX_FATAL_ERRORS)
          {
            #if CALCTEST_DEBUG3
              if (DbgPtr(false))
                DbgPtr()->LeaveLevel();
            #endif

            exit(1);
            return;
          }

          #if CALCTEST_SHOW_MSGBOX
            QMessageBox::critical(0, tr("Fatal Error"),
                                     tr("Could not start external command line calculator"),
                                   tr("Quit"));
          #endif

          _FatalErrCount++;
          DoHalt();
          emit FatalError();
        }
        else
        {
          _FatalErrCount = 0;
          _Spawned = TRUE;
        }
      #else
        #if defined(__linux__) | defined(__unix__)
          ProgName_ = QString("./mcalc");
        #else
          ProgName_ = QString(".\\mcalc.exe");
        #endif

        #if CALCTEST_DEBUG3
          DbgPtr()->ShowMessage("_ProcPtr->start() == false\n");
        #endif

        delete _ProcPtr;
        _ProcPtr = new QProcess(this);
        _ProcPtr->addArgument(ProgName_);

        #if CALCTEST_DEBUG3
          DbgPtr()->ShowStr(ProgName_, "_ProcPtr->addArgument(ProgName_)");
        #endif

        ptr = strtok(&_OutputBuffer[argvpos], " ");
      
        for (x = 0; ptr && x < argc; x++)
        {
          cmpres_ = strcmp(ptr, "mcalc");
          
          // omit adding "mcalc" program name argument because already added
          if (cmpres_ != 0 || !strstr(ProgName_, ptr))
          {
            _ProcPtr->addArgument(ptr);

            #if CALCTEST_DEBUG3
              DbgPtr()->ShowStr(ptr, "_ProcPtr->addArgument(ptr)");
              DbgPtr()->ShowInt(cmpres_, "strcmp(ptr, \"mcalc\")");
            #endif
          }

          ptr = strtok(NULL, " ");
        }
        
        _ProcPtr->setCommunication(QProcess::Stdin | QProcess::Stdout);

        if (envspos)
        {
          ptr = strchr(&_OutputBuffer[envspos], '=');
          x = ptr - &_OutputBuffer[envspos];
          _OutputBuffer[envspos+x] = 0;
          x++;
          #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
            mbstowcs(envstr_, &_OutputBuffer[envspos], strlen(&_OutputBuffer[envspos]));
            mbstowcs(envval_, &_OutputBuffer[envspos+x], strlen(&_OutputBuffer[envspos+x]));
            pwenvstr_ = envstr_;
            pwenvval_ = envval_;
            SetEnvironmentVariable(pwenvstr_, pwenvstr_);
          #else
            setenv(&_OutputBuffer[envspos],
                   &_OutputBuffer[envspos+x], 1);
          #endif
          _OutputBuffer[envspos+x-1] = '=';

          if (lines > 2)
          {
            ptr = strchr(&_OutputBuffer[envspos], '\n');
            ++ptr;

            if (*ptr)
            {
              nxline = ptr - &_OutputBuffer[envspos];
              
              ptr = strchr(&_OutputBuffer[envspos+nxline], '=');
              x = ptr - &_OutputBuffer[envspos+nxline];
              _OutputBuffer[envspos+nxline+x] = 0;
              x++;
              #if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
                mbstowcs(envstr_, &_OutputBuffer[envspos+nxline], strlen(&_OutputBuffer[envspos+nxline]));
                mbstowcs(envval_, &_OutputBuffer[envspos+nxline+x], strlen(&_OutputBuffer[envspos+nxline+x]));
                pwenvstr_ = envstr_;
                pwenvval_ = envval_;
                SetEnvironmentVariable(pwenvstr_, pwenvval_);
              #else
                setenv(&_OutputBuffer[envspos+nxline],
                       &_OutputBuffer[envspos+nxline+x], 1);
              #endif
              _OutputBuffer[envspos+nxline+x-1] = '=';
            }
            else
              lines = 2;
          }
        }

        if (!_ProcPtr->start())
        {
          if (_FatalErrCount >= MAX_FATAL_ERRORS)
          {
            #if CALCTEST_DEBUG3
              if (DbgPtr(false))
                DbgPtr()->LeaveLevel();
            #endif

            exit(1);
            return;
          }

          #if CALCTEST_SHOW_MSGBOX
            QMessageBox::critical(0, tr("Fatal Error"),
                                     tr("Could not start external command line calculator"),
                                     tr("Quit"));
          #endif

          _FatalErrCount++;
          DoHalt();
          emit FatalError();
        }
        else
        {
          _FatalErrCount = 0;
          _Spawned = TRUE;
        }
      #endif
    }
    else
      _Spawned = TRUE;
  }

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif

  if (_Spawned)
    emit CreateNewMcalc();
}

/****************************************************************************/
void CalcDriver::Execute()
{
  QString Num_;
  Num_ = Num_.setNum(_SessionNum);
  
  if (_Executed)
    return;

  #if CALCTEST_DEBUG2
    QMessageBox::warning(0, "DEBUG",
                         QString("calling mcalc with session # = ") + Num_,
                         QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);  
  #endif

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->EnterLevel("CalcDriver::Execute");
  #endif

  // Replace with process execution code
  #if defined(__linux__) | defined(__unix__)
    #if CALCTEST_EXEC_CURRENTDIR
      QString ProgName_ = QString("./mcalc");
    #else
      QString ProgName_ = QString("mcalc");
    #endif
  #else
    #if CALCTEST_EXEC_CURRENTDIR
      QString ProgName_ = QString(".\\mcalc.exe");
    #else
      QString ProgName_ = QString("mcalc.exe");
    #endif
  #endif

  _ProcPtr = new QProcess(this);
  _ProcPtr->addArgument(ProgName_);
  _ProcPtr->addArgument("-x");
  _ProcPtr->addArgument(Num_);
  _ProcPtr->setCommunication(QProcess::Stdin | QProcess::Stdout);

  connect(_ProcPtr, SIGNAL(processExited()),
          this, SLOT(ProcessExited()));

  if (!_ProcPtr->start())
  {
    #if CALCTEST_EXEC_CURRENTDIR
      #if defined(__linux__) | defined(__unix__)
        ProgName_ = QString("mcalc");
      #else
        ProgName_ = QString("mcalc.exe");
      #endif

      delete _ProcPtr;
      _ProcPtr = new QProcess(this);
      _ProcPtr->addArgument(ProgName_);
      _ProcPtr->addArgument("-x");
      _ProcPtr->addArgument(Num_);
      _ProcPtr->setCommunication(QProcess::Stdin | QProcess::Stdout);
      
      if (!_ProcPtr->start())
      {
        if (_FatalErrCount >= MAX_FATAL_ERRORS)
        {
          #if CALCTEST_DEBUG3
            if (DbgPtr(false))
              DbgPtr()->LeaveLevel();
          #endif

          exit(1);
          return;
        }

        #if CALCTEST_SHOW_MSGBOX
          QMessageBox::critical(0, tr("Fatal Error"),
                                   tr("Could not start external command line calculator"),
                                   tr("Quit"));
        #endif

        _FatalErrCount++;
        DoHalt();
        emit FatalError();
      }
      else
      {
        _FatalErrCount = 0;
        _Executed = TRUE;
      }
    #else
      #if defined(__linux__) | defined(__unix__)
        ProgName_ = QString("./mcalc");
      #else
        ProgName_ = QString(".\\mcalc.exe");
      #endif

      delete _ProcPtr;
      _ProcPtr = new QProcess(this);
      _ProcPtr->addArgument(ProgName_);
      _ProcPtr->addArgument("-x");
      _ProcPtr->addArgument(Num_);
      _ProcPtr->setCommunication(QProcess::Stdin | QProcess::Stdout);

      if (!_ProcPtr->start())
      {
        if (_FatalErrCount >= MAX_FATAL_ERRORS)
        {
          #if CALCTEST_DEBUG3
            if (DbgPtr(false))
              DbgPtr()->LeaveLevel();
          #endif

          exit(1);
          return;
        }

        #if CALCTEST_SHOW_MSGBOX
          QMessageBox::critical(0, tr("Fatal Error"),
                                   tr("Could not start external command line calculator"),
                                   tr("Quit"));
        #endif

        _FatalErrCount++;
        DoHalt();
        emit FatalError();
      }
      else
      {
        _FatalErrCount = 0;
        _Executed = TRUE;
      }
    #endif
  }
  else
    _Executed = TRUE;
  
  #if CALCDRIVER_TEXT_VERSION
    while (!_ProgramExited)
    {
      SendToInput();
      if (!_ProgramExited)
        ReadFromOutput(false, 0);
    }
  #endif

  #if CALCTEST_DEBUG3
    if (DbgPtr(false))
      DbgPtr()->LeaveLevel();
  #endif
}

/****************************************************************************/
void CalcDriver::ProcessExited()
{
  _ProgramExited = TRUE;
}

/****************************************************************************/
// Methods related to server polling : start
/****************************************************************************/
void CalcDriver::SendServerAlive(bool flag_)
{
  _ServerProgramAlive = flag_;
  
  if (flag_)
    emit ServerAliveConfirmed();
}

/****************************************************************************/
void CalcDriver::ShowServerPollFailed(bool Exit_)
{
  #if CALCTEST_SHOW_MSGBOX
    QMessageBox::critical(0, "ERROR", QString(ERRMSG_SERVERNOTFOUND),
                          QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton);
  #elif CALCDRIVER_TEXT_VERSION
    std::cout <<ERRMSG_SERVERNOTFOUND;
    std::cerr <<ERRMSG_SERVERNOTFOUND;
  #endif

  if (Exit_)
    exit(1);
}

/****************************************************************************/
bool CalcDriver::PollServerIfAlive(int Sleep_)
{
  int retval = 0;

  if (_PollServerMgr)
  {
    bool PollingActive_ = CalcPollingActive();
    ResumeCalcPolling();
  
    _PollServerMgr->SetWaitUntilResponse(true)
                  ->SetExitAtRetryMax(true)
                  ->SetSleepWaiting(true)
                  ->SetAllowServerPolling(true)
                  ->SetBreakOnPollError(false);

    _ResponseOnly = false;
    _PollServerMgr->ResetServerPolling(true);
    retval = _PollServerMgr->ResetWhenNotRequired(true)
                           ->PollServerIfAlive(Sleep_, true, false); // Initiate Polling

    if (!PollingActive_)
      PauseCalcPolling();
  }

  return retval;
}

/****************************************************************************/
int CalcDriver::openfile(FILE** file, const char* name, const char* mode,
                         int Retry_, int ExitOnFail_, int MaxRetries_)
{
  int retval = 1;

  // millisec sleep
  int DelayLen_ = IOSTATE_DELAY_LENGTH * MILLI_DELAY_MULT;
  bool DoRetry_ = false;
  
  size_t* Retryp_ =
  (name == CLIENTALIVE_FILE) ? &_OpenClientAliveFileRetries:
  (name == SERVERALIVE_FILE) ? &_OpenServerAliveFileRetries:
  (name == POLLSTATE_FILE)   ? &_OpenPollStateFileRetries:
                               NULL;

  if ((*file = fopen(name,mode))==NULL)
  {
    retval = 0;
    DoRetry_ = !Retryp_ || *Retryp_ < size_t(MaxRetries_);

    if (Retry_ && MaxRetries_ && DoRetry_)
    {
      _AttemptFileName = name;
      _AttemptMode = mode;
      _AttemptFile =
      (name == CLIENTALIVE_FILE) ? CLIENTALIVE:
      (name == SERVERALIVE_FILE) ? SERVERALIVE:
      (name == POLLSTATE_FILE)   ? POLLSTATE:0;

      if (!_FromMethod)
        SetFromMethod(OPENFILE);
      
      _FileOpenRetryTimer.start(DelayLen_);
      return true;
    }

    if (ExitOnFail_)
      exit(1);
  }
  else if (Retryp_)
    *Retryp_ = 0;

  return retval;
}

/****************************************************************************/
FILE* CalcDriver::GetClientAliveFile(int Write_, bool Rewind_)
{
  char Mode_ = Write_ ? 'w':'r';
  bool Diff_ = Mode_ != _ClientAliveMode && _ClientAliveMode;

    if (!filestatus_ClientAliveState ||
        !_ClientAliveFile || Diff_)
    {
      if (Diff_)
        CloseClientAliveFile();
    
      if (Write_)
      {
        filestatus_ClientAliveState = openfile(&_ClientAliveFile,CLIENTALIVE_FILE,"w",true);
        _ClientAliveMode = 'w';
      }
      else
      {
        filestatus_ClientAliveState = openfile(&_ClientAliveFile,CLIENTALIVE_FILE,"r",true);
        _ClientAliveMode = 'r';
      }
    }
    else if (!Diff_ && Mode_ == 'r' && Rewind_)
      rewind(_ClientAliveFile);
  
  if (_ClientAliveFile && _ClientAliveMode == 'w')
    fflush(_ClientAliveFile);

  return _ClientAliveFile;
}

/*****************************************************************************/
FILE* CalcDriver::GetServerAliveFile(int Write_, bool Rewind_)
{
  char Mode_ = Write_ ? 'w':'r';
  bool Diff_ = Mode_ != _ServerAliveMode && _ServerAliveMode;

    if (!filestatus_ServerAliveState ||
        !_ServerAliveFile || Diff_)
    {
      if (Diff_)
        CloseServerAliveFile();

      if (Write_)
      {
        filestatus_ServerAliveState = openfile(&_ServerAliveFile,SERVERALIVE_FILE,"w",true);
        _ServerAliveMode = 'w';
      }
      else
      {
        filestatus_ServerAliveState = openfile(&_ServerAliveFile,SERVERALIVE_FILE,"r",true);
        _ServerAliveMode = 'r';
      }
    }
    else if (!Diff_ && Mode_ == 'r' && Rewind_)
      rewind(_ServerAliveFile);
  
  if (_ServerAliveFile && _ServerAliveMode == 'w')
    fflush(_ServerAliveFile);

  return _ServerAliveFile;
}

/*****************************************************************************/
FILE* CalcDriver::GetPollStateFile(int Write_, bool Rewind_)
{
  char Mode_ = Write_ ? 'w':'r';
  bool Diff_ = Mode_ != _PollStateMode && _PollStateMode;

    if (!filestatus_PollState ||
        !_PollStateFile || Diff_)
    {
      if (Diff_)
        ClosePollStateFile();

      if (Write_)
      {
        filestatus_PollState = openfile(&_PollStateFile,POLLSTATE_FILE,"w",true);
        _PollStateMode = 'w';
      }
      else
      {
        filestatus_PollState = openfile(&_PollStateFile,POLLSTATE_FILE,"r",true);
        _PollStateMode = 'r';
      }
    }
    else if (!Diff_ && Mode_ == 'r' && Rewind_)
      rewind(_PollStateFile);
  
  if (_PollStateFile && _PollStateMode == 'w')
    fflush(_PollStateFile);

  return _PollStateFile;
}

/*****************************************************************************/
void CalcDriver::CloseClientAliveFile()
{
    if (_ClientAliveFile)
      fclose(_ClientAliveFile);

    _ClientAliveFile = NULL;
    filestatus_ClientAliveState = 0;
}

/*****************************************************************************/
void CalcDriver::CloseServerAliveFile()
{
    if (_ServerAliveFile)
      fclose(_ServerAliveFile);

    _ServerAliveFile = NULL;
    filestatus_ServerAliveState = 0;
}

/*****************************************************************************/
void CalcDriver::ClosePollStateFile()
{
    if (_PollStateFile)
      fclose(_PollStateFile);

    _PollStateFile = NULL;
    filestatus_PollState = 0;
}

/****************************************************************************/
// Methods related to server polling : end
/****************************************************************************/
const char* CalcDriver::GetSpawnArgFilename() const
{
  return SPAWNARG_FILE;
}

/****************************************************************************/
const char* CalcDriver::GetLogFilename() const
{
  return LOG_FILE;  
}

/****************************************************************************/
const char* CalcDriver::GetCurrentErrorFilename() const
{
  return CURRENTERROR_FILE;  
}

/****************************************************************************/
const char* CalcDriver::GetUserInputFilename() const
{
  return USER_INPUT_FILE;
}

/****************************************************************************/
const char* CalcDriver::GetUserPromptFilename() const
{
  return USER_PROMPT_FILE;  
}

/****************************************************************************/
const char* CalcDriver::GetProgressFilename() const
{
  return PROGRESS_FILE;
}

/****************************************************************************/
const char* CalcDriver::GetGraphProgressFilename() const
{
  return GRAPH_PROGRESS_FILE;
}

/****************************************************************************/
const char* CalcDriver::GetGraphOperationFilename() const
{
  return GRAPH_OPERATION_FILE;
}

/****************************************************************************/
const char* CalcDriver::GetProgramOutputFilename() const
{
  return PROG_OUTPUT_FILE;
}

/****************************************************************************/
const char* CalcDriver::GetProgramIostateFilename() const
{
  return PROG_IOSTATE_FILE;
}

/****************************************************************************/
const char* CalcDriver::GetServerAliveFilename() const
{
  return SERVERALIVE_FILE;
}

/****************************************************************************/
const char* CalcDriver::GetClientAliveFilename() const
{
  return CLIENTALIVE_FILE;
}

/****************************************************************************/
const char* CalcDriver::GetPollStateFilename() const
{
  return POLLSTATE_FILE;
}

/****************************************************************************/
const char* CalcDriver::GetCalcElapsedTimeFilename() const
{
  return ELAPSED_TIME_FILE;
}

/****************************************************************************/
/****************************************************************************/
#if CALCDRIVER_TEXT_VERSION
int main()
{
    CalcDriver Prog_;
    Prog_.Execute();
    return 0;
}
#endif
/****************************************************************************/
