#ifndef CALCDRIVER_H
#define CALCDRIVER_H

#define CALCDRIVER_GUI_VERSION          0

#if CALCDRIVER_GUI_VERSION
  #define CALCDRIVER_TEXT_VERSION       0
  #define CALCTEST_SHOW_MSGBOX          1
  #define CALCTEST_EXEC_CURRENTDIR      0
  #define CALCTEST_FIX_SESSION_DATA     0
#else
  #define CALCDRIVER_TEXT_VERSION       1
  #define CALCTEST_SHOW_MSGBOX          0
  #define CALCTEST_EXEC_CURRENTDIR      1
  #define CALCTEST_FIX_SESSION_DATA     1
#endif

#define CALCTEST_DEBUG                  0    // gui mode only
#define CALCTEST_DEBUG1b                0    // calculator response signal testing, gui mode only
#define CALCTEST_DEBUG1c                0    // trace execution for method: WaitForOutput
#define CALCTEST_DEBUG1d                0    // trace execution for method: ReadIoState
#define CALCTEST_DEBUG1e                0    // trace elapsed time state execution: gui mode only
#define CALCTEST_ALLOWPOLL              0    // activate polling code

#define CALCTEST_DEBUG2                 0    // session mode testing, gui mode only
#define CALCTEST_DEBUG3                 0    // test client/server polling
#define CALCTEST_DEBUG3p                0    // do client/server simulation polling

// additional method tracking debugging
// information in CalcDriver class
#define CALCTEST_DEBUG4                 0
#define CALCTEST_DEBUG4b                0
#define CALCTEST_DEBUG5                 0    // allow conditional stoppage of method
                                             // tracking in ServerPollingMgr class

#define CALCTEST_DEBUG6a                0    // trace execution for Graph Operation methods
#define CALCTEST_DEBUG6b                0    // trace graph operation methods, gui mode only
#define CALCTEST_DEBUG6c                0    // trace execution for Graph Progress methods
#define CALCTEST_DEBUG7                 0    // trace execution for input batch file methods
#define CALCTEST_DEBUG8                 0    // emit signal and server I/O response testing
#define CALCDRIVER_DEBUGGING            0

#ifndef DEBUGUTILS_H
  #include "debugutils.h"
#endif

#if CALCDRIVER_DEBUGGING
  #undef IN_DEBUGGING_MODE
  #define IN_DEBUGGING_MODE             1
  #include "debugmacros.h"
#else
  #undef IN_DEBUGGING_MODE
  #define IN_DEBUGGING_MODE             0
#endif

#define IMPLEMENT_KEEP_ALIVE            0  // implement client polling for keep alive state
#define IDLE_RESPONSE_ONLY_POLL         0  // implement client polling for only idle time server response
#define RETRY_READ_MAX                  5
#define LIMIT_READ_RETRIES              0

#define ERRMSG_SERVERNOTFOUND           "ERROR: Server program for calculator not responding\n"
#define ERRMSG_ERROR_IN_LINE            "error in line: %d, col: %d, code: %d\n"

#if CALCDRIVER_GUI_VERSION
  // Qt libraries
  #ifndef MCHEADERS_H
    #include "mcheaders.h"
  #endif
  #ifndef EXECPRGNAME_H
    #include "execprgname.h"
  #endif
#else
  #include <qeventloop.h>
  #include <qapplication.h>
  #include <qvariant.h>
  #include <qtimer.h>
  #include <qfile.h>
  #include <qobject.h>
  #include <qprocess.h>
  #include <qstring.h>
  
  #include <iostream>

  #ifndef EXECPRGNAME_H
    #include "execprgname.h"
  #endif
#endif

/* standard libraries */
#ifndef INCL_STDIO_H
  #include <stdio.h>
  #define INCL_STDIO_H
#endif
#ifndef INCL_STRING_H
  #include <string.h>
  #define INCL_STRING_H
#endif
#ifndef INCL_CTYPE_H
  #include <ctype.h>
  #define INCL_CTYPE_H
#endif
#ifndef INCL_ASSERT_H
  #include <assert.h>
  #define INCL_ASSERT_H
#endif
#ifndef INCL_STDLIB_H
  #include <stdlib.h>
  #define INCL_STDLIB_H
#endif
#ifndef INCL_MATH_H
  #include <math.h>
  #define INCL_MATH_H
#endif
#if (defined(__DJGPP__) | defined(__linux__))
  #ifndef INCL_UNISTD_H
    #include <unistd.h>
    #define INCL_UNISTD_H
  #endif
#endif
#if (!defined(__DJGPP__) & !defined(__linux__) & !defined(__unix__) & !defined(__GNUC__))
  #include <Windows.h>
#endif

#define IOSTATE_PRE_SET_DELAY         0    // delay before setting I/O state value into IoState file
#define IOSTATE_POST_SET_DELAY        0    // delay after setting I/O state value into IoState file
#define IOSTATE_DELAY_LENGTH          200  // delay length in milliseconds
#define MILLI_DELAY_MULT              1    // delay length multiplier for millisec sleep
#define MICRO_DELAY_MULT              1000 // delay length multiplier for microsec sleep
#define LONG_WAIT                     3000 // delay length for:
                                           //   empty user input buffer returned OR
                                           //   after BATCHFILE_ENDED signal received OR
                                           //   after sending CLIENTPING_ACK signal to server

// Default file names
#define MCALC_CONFIG_FILE               "mcalc_fileio.cfg"        // configuration file for using file input/output
#define MCALC_SESSION_COUNT_FILE        "mcalc_sescnt.dat"        // main session counter file
#define MCALC_SESSION_FILE              "mcalc_session.dat"       // main session file
#define MCALC_SPAWNARG_FILE             "mcalc_spawnargs.log"     // spawn calculator arguments file
#define MCALC_LOG_FILE                  "mcalc_errors.log"        // error log file
#define MCALC_CURRENT_ERROR_FILE        "mcalc_currenterr.log"    // current error message output file
#define MCALC_USER_PROMPT_FILE          "mcalc_userprompt.dat"    // user prompt file
#define MCALC_USER_INPUT_FILE           "mcalc_userinput.dat"     // user input file
#define MCALC_PROG_OUTPUT_FILE          "mcalc_programout.dat"    // program output file
#define MCALC_IOSTATE_FILE              "mcalc_iostate.dat"       // program I/O state file
#define MCALC_SERVERALIVE_FILE          "mcalc_serveralive.dat"   // server keep alive state file
#define MCALC_CLIENTALIVE_FILE          "mcalc_clientalive.dat"   // client keep alive state file
#define MCALC_PROGRESS_FILE             "mcalc_progress.dat"      // completion progress of input data file processing
#define MCALC_ELAPSEDTIME_FILE          "mcalc_elapsedtime.dat"   // elapsed time file used for calc. server timeout notifications
#define MCALC_GRAPH_PROGRESS_FILE       "mcalc_graphprogress.dat" // graph plotting progress file
#define MCALC_GRAPH_OPERATION_FILE      "mcalc_graphoper.dat"     // graph plotting operation type data file
#define MCALC_POLLSTATE_FILE            "mcalc_pollstate.dat"     // current keep alive poll state file

// Error messages
#define ERRMSG_OPENSTATEFILE            "Error opening I/O state file: %s, calculator client\n"
#define ERRMSG_SERVERNOTFOUND           "ERROR: Server program for calculator not responding\n"

#define MSG_PROGRAM_ENDED               "<Program Ended>"
#define MSG_INPUT_FILE_ENDED            "<Input File Ended>"

#if (!(CALCDRIVER_GUI_VERSION))
  using std::cout;
  using std::cerr;
  using std::endl;
#endif

class Mcalc_SessionStates
{
  public:
  enum
  {
    InUse = 1,
    UnUsed = 2
  };
};

class Mcalc_GraphOpType
{
  public:
  enum
  {
    NO_GRAPH          = 0,  // No graph operations specified
    
    FUNCTION_PLOT     = 1,  // straight function plot of math expression
    APPLIED_FNCGRAPH  = 2,  // extra applied functions: deriv, integ, etc...
    BATCH_GRAPH_PLOT  = 3   // batch input of graph operations
  };
};

class Mcalc_GraphType
{
  public:
  enum
  {
    NO_GRAPH     = 0,
    
    FUNCTION     = 1,
    EXTREMA      = 2,
    DERIVATIVE   = 3,
    INTEGRAL     = 4,
    ROOTSOLVE    = 5,
    INTERSECT1   = 6,
    INTERSECT2   = 7,

    MIN_EXTREMA  = 101,
    MAX_EXTREMA  = 102,
    
    FIRST_DERIV  = 201,
    SECOND_DERIV = 202
  };
};

class Mcalc_IOState
{
  public:
  enum
  {
    PROCESS         = 1,     // - send process input data request from client program to calculator
    OUTPUT_READY    = 2,     // - sent by calculator to indicate output data ready
                             //     used to indicate text messages sent that are not meant as calculated answers
                             //     examples: error messages, status messages, executed disp program commands...
    CALC_HALT       = 4,     // - sent by client program to tell calculator to halt execution and quit
    CALC_RESPONSE   = 8,     // - sent by calculator to indicate post processing response
    INPUT_REQUIRED  = 16,    // - sent by calculator in script program or interactive text mode to
                             //   indicate prompt from stdio/console input for additional required input data
    OUTPUT_FETCHED  = 32,    // - sent by client program to indicate literal text string or graph plotting data fetched
    INPUT_RECEIVED  = 64,    // - sent by client program to indicate required input data sent to calculator
    BREAK_PROGRAM   = 128,   // - sent by client program to tell calculator to break program and exit program mode
                             //   execution of program script if it is in program mode
    IDLE_STATE      = 256,   // - calculator is in idle or initial starting state, set after CALC_HALT received
    GRAPH_OUTPUT    = 512,   // - function graphing output is completed and ready to be read
    GRAPH_WAIT      = 1024,  // - Function graphing is in process. Data points being plotted.
    CALC_ERROR      = 2048,  // - sent by calculator to indicate calculator error from invalid input or other conditions
    PROGRESS_READY  = 4096,  // - input data file -- calculation progress is ready and sent to gui client
    ERROR_FETCH     = 8192,  // - set by calculator to indicate calculator error. Fetch output from error log file.
    GRAPH_PROGRESS  = 16384, // - plotted data points -- graph plotting progress is ready and sent to gui client

    INFILE_PROGRESS_ACK = 32768,  // - Input file processing progress acknowledged/fetched by client
    GRAPH_PROGRESS_ACK  = 65536,  // - Graph plotting processing progress acknowledged/fetched by client
    GRAPH_WAIT_ACK      = 131072, // - Graph wait signal acknowledged, graph plot operation data fetched

    SPAWN_NEW_MCALC = 262144, // - sent by calculator to notify client to spawn another mcalc process
    MCALC_SPAWNED   = 524288, // - sent by client program to indicate request to spawn mcalc process acknowledged and done

    BATCHFILE_ENDED     = 1048576, // sent by calculator to indicate end of batch data input file processing
    BATCHFILE_ENDED_ACK = 2097152, // sent by client to acknowledge and of batch data file signal
    PROCESS_DONE        = 4194304, // sent by calculator to indicate process acknowledgement of non-output directives
    PROCESS_DONE_ACK    = 8388608, // sent by client to acknowledge server directive execution

    RESENDLASTSIG       = 16777216,  // Resend last signal sent by server
    CLIENTPING          = 33554432, // Server ping signal to client
    CLIENTPING_ACK      = 67108864, // Client acknowledgement of client ping signal
    
    IOSTATE_ENDVAL      = 67108864  // IO state ending value to mark end of iostate enumerated list
  };

  // expected valid IOState sequence
  // 1. Client-->Calc : PROCESS, calculator input data sent
  // non-graph output:
  // 1.1 Calc-->Client : CALC_RESPONSE, Answer as calculated output returned to client         OR
  //        Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  // 1.2a Calc-->Client : INPUT_REQUIRED, Additional input data requested                      OR
  //         Client-->Calc : INPUT_RECEIVED, Additional data sent by client
  //            Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  // 1.2b Calc-->Client : INPUT_REQUIRED, Additional input data requested                      OR
  //         Client-->Calc : INPUT_RECEIVED, Additional data sent by client
  //            Calc-->Client : INPUT_REQUIRED, Additional input data requested
  // 1.3a Calc-->Client : OUTPUT_READY, Answer as non-calculated text string                    OR
  //         Client-->Calc : OUTPUT_FETCHED, Non-calculated text string fetched
  //            Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  // 1.3b Calc-->Client : OUTPUT_READY, Answer as non-calculated text string                    OR
  //         Client-->Calc : OUTPUT_FETCHED, Non-calculated text string fetched
  //            Client-->Calc : OUTPUT_READY, Additional non-calculated text string output
  // 1.4 Calc-->Client : CALC_ERROR, Calculator error from detected invalid input data
  //        Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  // 1.5 Calc-->Client : ERROR_FETCH, Calculator error from detected invalid input data
  //        Client : Fetch output from error log file and set state to: CALC_ERROR
  //        Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  //
  // graphing output:                                                                          OR
  // 1.6a Calc-->Client : GRAPH_WAIT, Function graph data points being plotted,
  //                                  graph operation data sent to client.
  //         Client-->Calc : GRAPH_WAIT_ACK, Graph plotting operation type and data fetched
  // 1.6b Calc-->Client : GRAPH_WAIT, Function graph data points being plotted,
  //                                  graph operation data sent to client.
  //         Client-->Calc : INPUT_RECEIVED, quit graph plotting signal sent by client
  //
  // 1.7 Calc-->Client : GRAPH_PROGRESS, Function graph plotting progress data sent
  //        Client-->Calc : GRAPH_PROGRESS_ACK, Function graph plotting progress data fetched
  // 1.8a Calc-->Client : GRAPH_OUTPUT, Graph plotting data is completed                       OR
  //         Client-->Calc : OUTPUT_FETCHED, Graph plotting data points fetched
  //            Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  // 1.8b Calc-->Client : GRAPH_OUTPUT, Graph plotting data is completed
  //         Client-->Calc : OUTPUT_FETCHED, Graph plotting data points fetched
  //            Calc-->Client : BATCHFILE_ENDED, Batch data file for Graph plotting ended
  //               Client-->Calc : BATCHFILE_ENDED_ACK, Acknowledge BATCHFILE_ENDED signal received
  //                  Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  //
  // 1.9 Calc-->Client : CALC_ERROR, Calculator error from detected invalid input data
  //        Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  // 1.10 Calc-->Client : ERROR_FETCH, Calculator error from detected invalid input data
  //         Client : Fetch output from error log file and set state to: CALC_ERROR
  //         Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  //
  // 2. Client-->Calc : BREAK_PROGRAM, request calculator to break out from program mode
  //       Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  //
  // 3. Client-->Calc : CALC_HALT, tell calculator to halt execution and quit
  //       Calc->Client : IDLE_STATE, Calculator set to idle state before program termination
  //
  // 4. Calc-->Client : PROGRESS_READY, Send progress status of input data file processing
  //       Client-->Calc : INFILE_PROGRESS_ACK, Non-calculated progress status text fetched
  //          Calc-->Client : BATCHFILE_ENDED, Batch data file for math expressions ended
  //             Client-->Calc : BATCHFILE_ENDED_ACK, Acknowledge BATCHFILE_ENDED signal received
  //                Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  //
  // 5. Calc-->Client : SPAWN_NEW_MCALC, notify client to spawn another mcalc process
  //       Client-->Calc : MCALC_SPAWNED, request to spawn mcalc process acknowledged and done
  //
  // 6a. Client-->Calc : PROCESS, process signal sent by client to process non-output calc. directive
  //        Calc-->Client : PROCESS_DONE, send acknowledgement signal to client of directive execution
  //           Client-->Calc : PROCESS_DONE_ACK, server directive execution acknowledgement from client
  //              Client-->Calc : (PROCESS|CALC_HALT|BREAK_PROGRAM), process, quit or break
  // 6b. Client-->Calc : PROCESS, process signal sent by client to process non-output calc. directive
  //        Calc-->Client : PROCESS_DONE, send acknowledgement signal to client of directive execution
  //           Client-->Calc : PROCESS_DONE_ACK, sent by client to acknowledge server directive execution
  //              Client-->Calc : PROCESS_DONE, send additional acknowledgement signals to client of directive execution
  //
  // 7. Calc-->Client : CALC_HALT, calculator has self terminated and CALC_HALT signal is sent
  //       Client : IDLE_STATE, Client sets calculator state file to idle state before quitting
  //
  // Input Data Batch File Mode:
  // ---------------------------
  // If calculator is executing a batch file or program,
  // the following special cases apply:
  //
  //   INPUT_RECEIVED      --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
  //   OUTPUT_FETCHED      --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
  //   BREAK_PROGRAM       --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
  //   BATCHFILE_ENDED_ACK --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
  //
  //   CALC_RESPONSE       --> INPUT_RECEIVED
  //   CALC_ERROR          --> INPUT_RECEIVED
  //   ERROR_FETCH         --> INPUT_RECEIVED
  //
  //   PROCESS_DONE        --> PROCESS_DONE_ACK
  //   PROCESS_DONE_ACK    --> INPUT_REQUIRED|OUTPUT_READY|CALC_RESPONSE|PROCESS|PROCESS_DONE
  //
  // Control Signals Only (Not Saved As IO State):
  // ---------------------------------------------
  // 8. Client-->Calc : RESENDLASTSIG, sent by client to request resending of latest sent IO signal
  //       Calc-->Client : Resending of last sent IO signal, (RESENDLASTSIG not saved)
  //
  // 9. Calc-->Client : CLIENTPING, sent by calculator to verify if client program is still active
  //       Client-->Calc : CLIENTPING_ACK, response sent by client to verify that it is still active
  //                       Calculator reverts back to previous IO state prior to sending CLIENTPING.
  //                       Calculator waits for PROCESS, BREAK_PROGRAM or CALC_HALT signal from client.
  //
};

class Mcalc_ElapsedTime
{
  public:
  enum
  {
    RESET = 0,
    SENT = 1,
    ACKNOWLEDGED = 2
  };
};

class Mcalc_KeepAliveState
{
  public:
  enum
  {
    IDLE                = 0,  // idle state
    RESET_POLLING       = 1,  // reset polling timer when received

    POLL_SERVERALIVE    = 2,  // poll signal sent to server to check if process is alive
    CLIENT_ACKNOWLEDGED = 4,  // reply to server, client acknowledged server alive signal
    CLIENT_ALIVE        = 8,  // sent from client to confirm that process is alive

    POLL_CLIENTALIVE    = 16, // poll signal sent to client to check if process is alive
    SERVER_ACKNOWLEDGED = 32, // reply to client, server acknowledged client alive signal
    SERVER_ALIVE        = 64  // sent from server to confirm that proces is alive
  };
};

class Mcalc_PollState
{
  public:
  enum
  {
    IDLE                = 0,
    SERVER_TO_CLIENT    = 1,
    CLIENT_RESPONSE     = 2,
    CLIENT_TO_SERVER    = 4,
    SERVER_RESPONSE     = 8,
    POLLING_PAUSED      = 16
  };

// async mode server/client polling scheme:
//
// 0. inital file states
//
//    client alive file: IDLE
//    poll state file: IDLE
//    server alive file: IDLE
//
// 1. server-to-client
//
// 1a. i: check for server states in iostate file.
//     ii: check poll state file for IDLE state.
//     iii: check client alive file for IDLE state.
//     proceeed if conditions at i, ii and iii is true.
//     send polling signal to client
//
// 1b. client alive file:  IDLE
//     poll state file: SERVER_TO_CLIENT\n (cycle number)
//     server alive file: POLL_CLIENTALIVE\n (cycle number)
//
// 2. client receives server poll signal
//
// 2a. check for server states in iostate file, proceed if server states confirmed:
//     send back client alive signal
//
// 2b. client alive file: CLIENT_ALIVE\n (cycle number from server)
//     poll state file: CLIENT_RESPONSE
//     server alive file: POLL_CLIENTALIVE
//
// 3. server receives client alive signal and sends back acknowledgement
//    send back server acknowledged signal
//
// 3a. client alive file: CLIENT_ALIVE
//     poll state file: SERVER_TO_CLIENT
//     server alive file: SERVER_ACKNOWLEDGED\n (cycle number)
//
// 4. client receives server acknowledgement and sends reset polling timer signal
//    send back polling timer reset signal
//
// 4a. client resets own no-activity polling timer
//
// 4b. client alive file: RESET_POLLING\n (cycle number)
//     poll state file: CLIENT_RESPONSE
//     server alive file: SERVER_ACKNOWLEDGED
//
// 5. server receives reset polling timer signal and resets all polling timers (if timers exist)
//    Sets poll state file to IDLE state. Sets server alive file to IDLE state.
//
// 5a. server resets polling timer, server resets no-activity timer (if timers exist)
//
// 5b. client alive file: RESET_POLLING
//     poll state file: IDLE
//     server alive file: IDLE
//
// 6. client-to-server
//
// 6a. i: check for client states in iostate file.
//     ii: check poll state file for IDLE state.
//     iii: check server alive file for IDLE state.
//     proceeed if conditions at i, ii and iii is true.
//     send polling signal to server
//
// 6b. client alive file:  POLL_SERVERALIVE\n (cycle number)
//     poll state file: CLIENT_TO_SERVER\n (cycle number)
//     server alive file: IDLE
//
// 7. server receives client poll signal
//
// 7a. check for client states in iostate file, proceed if client states confirmed:
//     send back server alive signal
//
// 7b. client alive file: POLL_SERVERALIVE
//     poll state file: SERVER_RESPONSE
//     server alive file: SERVER_ALIVE\n (cycle number from client)
//
// 8. client receives server alive signal and sends back acknowledgement
//    send back client acknowledged signal
//
// 8a. client alive file: CLIENT_ACKNOWLEDGED\n (cycle number)
//     poll state file: CLIENT_TO_SERVER
//     server alive file: SERVER_ALIVE
//
// 9. server receives client acknowledgement and sends reset polling timer signal
//    send back polling timer reset signal
//
// 9a. server resets own no-activity polling timer (if timer exists)
//
// 9b. client alive file: CLIENT_ACKNOWLEDGED
//     poll state file: SERVER_RESPONSE
//     server alive file: RESET_POLLING\n (cycle number)
//
// 10. client receives reset polling timer signal and resets all polling timers
//     Sets poll state file to IDLE state. Sets client alive file to IDLE state.
//
// 10a. client resets polling timer, client resets no-activity timer
//
// 10b. client alive file: IDLE
//      poll state file: IDLE
//      server alive file: RESET_POLLING
};

/****************************************************************************/
class CalcDriver;

class ServerPollingMgr
{
  friend class CalcDriver;

  protected:
    enum
    {
      READ    = 0,
      WRITE   = 1,
      BUFSIZE = 64,
      LOOP_GUARD_LIMIT = 200,

      INITIALSTATE = 0,
      NOTREQUIRED  = 1,
      REQUIRED     = 2,
      INPROCESS    = 4,
      COMPLETED    = 8
    };

    CalcDriver* _Parent;        // Parent calculator driver object
    char* _StrBuffer;           // string buffer for state file date reading

    int _CycleNum;              // Poll process cycle number
    bool _CycleActive;          // Poll process cycle in active state
    int _RetrievedCycle;        // Retrieved process cycle number from client

    int _ClientPollProcess;     // Client polling process steps
    int _ServerPollProcess;     // Server polling process steps
    int _PrevServerProcess;     // Saved previous server polling process value
    int _PrevClientProcess;     // Saved previous receive from server poll process value
    
    int _RetryReadCount;        // Counter for retry attempts to read I/O state file
    int _RetryPollCount;        // Retry attempts counter for client program "keep alive" polling
    int _PollState;             // Client polling state
    int _ServerState;           // Server keep alive state
    int _ClientState;           // client keep alive state

    bool _RdClientPoll;         // Read Client Poll info allowed
    bool _AllowPollServer;      // Allow polling of server program
    bool _SleepWaiting;         // Sleep only when waiting for client response to server signal
    bool _ResetStatesOnDone;    // Reset poll process state after poll process completed
    bool _BreakOnPollError;     // Break out of polling loop on encounter of polling error
    bool _KeepAlivePollError;   // Error indicator for client program "keep alive" polling
    bool _ConfirmedClientAlive; // Confirmed client program is alive after polling client
    bool _ConfirmedServerAlive; // Confirmed server program is alive after client polled server
    bool _ServerAliveConfSent;  // Server alive confirmation sent to calculator object

    bool _AsyncMode;
    bool _ExitAtRetryMax;
    bool _StopTracking;         // debug mode method and data tracking
    bool _WaitUntilResponse;    // Wait indefinitely until client response
    bool _PausePollResults;     // Pause after showing poll results
    bool _ResetWhenNotReq;      // Reset polling/receiving when condition not required
    bool _ForceReset;           // Force reset polling/receiving on condition

    ServerPollingMgr* _Backup;  // backup version to compare differences in polling results data

    // server-polling-client process steps
    bool ServerIsPolling() const;
    bool ClientIsAlive() const;
    bool ServerAcked() const;
    bool ResetSignalSent();
    bool PollingResetByServer();

    // client-polling-server process steps
    bool AtInitialState();
    bool ClientIsPolling() const;
    bool ServerIsAlive() const;
    bool ClientAcked() const;
    bool ResetRequested();
    bool PollingReset();

    // polling variables and poll cycle number
    void ResetPollingVars();
    void ResetPollCycle();
    void SetNewPollCycle();
    void AcceptPollCycle();
    int IncCycleNum();

    // methods for polling server in order (top to bottom)
    bool PollServerIfAlive_IMPL(int Sleep_, bool ResponseOnly_);
    bool SendPollServerSignal();
    void ConfirmServerAlive(bool ReceiveClient_=false);
    bool SendClientAckServerSignal();
    bool ResetPolling();

    // methods for receiving client poll in order (top to bottom)
    bool ReceiveKeepAlivePoll_IMPL(int Sleep_);
    bool SendClientAliveSignal();
    void ConfirmClientAlive(bool PollServer_=false);
    bool SendResetPollingSignal();

    bool PollServerCompleted_IMPL(bool InclNotReq_, bool InclPrev_);
    bool ClientReceiveCompleted_IMPL(bool InclNotReq_, bool InclPrev_);

    bool SetPollState(int State_);
    bool SetClientState(int State_);
    bool SetServerState(int State_);
    
    int ReadClientState(bool RdPoll_);
    int ReadClientPollProcess(bool RefreshStates_=true);
    bool ServerSetPollInProcess(bool* PollStateOk_=NULL) const;
    bool ClientSetPollReset(bool* PollStateOk_=NULL) const;
    bool ResponseRequired(int PrevPollState_,
                          int& LoopGuard_, int LoopLimit_);
    void SetTrackingOnStateChange(bool Reset_=false);

  // public slots:
  public:
    void ResetPollResults(bool& PollingServer_,
                          bool& PollError_,
                          bool& PollAnswered_);
    void GivePollResults(bool& PollingServer_,
                         bool& PollError_,
                         bool& PollAnswered_);
    void ShowPollResults(bool PollingServer_,
                         bool PollError_,
                         bool PollAnswered_);

    void ReadAllStates(int Sleep_);
    int ReadPollState(int Sleep_);
    int ReadServerState(int Sleep_);
    int ReadClientState();

    bool WaitingForServerResponse(bool* FromClt_=NULL);
    bool PollServerIfAlive(int Sleep_=0,
                           bool UntilEnd_=false, bool ResponseOnly_=true);
    bool ReceiveKeepAlivePoll(int Sleep_=0, bool UntilEnd_=false);

    bool ResetServerPolling(bool ForceReset_=false);
    bool ResetReceivePoll(bool ForceReset_=false);
    bool ForceResetAllPolling();

    void ResetRetryRead();
    void ResetPollCount();

  public:
    ServerPollingMgr(CalcDriver* Parent_, bool CreateBackup_=true);
    ~ServerPollingMgr();

    static bool IsServerAliveStates(int State_);
    static bool IsClientAliveStates(int State_);

    static const char* KeepAliveStateToStr(int State_);
    static const char* PollStateToStr(int State_);

    ServerPollingMgr& operator = (const ServerPollingMgr& Obj_);

    ServerPollingMgr* SetAllowServerPolling(bool flag_);
    ServerPollingMgr* SetSleepWaiting(bool flag_);
    ServerPollingMgr* SetResetStatesOnDone(bool flag_);
    ServerPollingMgr* SetBreakOnPollError(bool flag_);
    ServerPollingMgr* SetExitAtRetryMax(bool flag_);
    ServerPollingMgr* SetWaitUntilResponse(bool flag_);
    ServerPollingMgr* SetStopTracking(bool flag_);
    ServerPollingMgr* SetPausePollResults(bool flag_);
    ServerPollingMgr* ResetWhenNotRequired(bool Forced_);
    
    bool PollServerRequired(int Sleep_=0, bool ResponseOnly_=true);
    bool PollServerNotRequired(bool InclAtInitial_);
    bool PollServerInProcess();
    bool PollServerCompleted(bool InclNotReq_);

    bool ClientReceiveRequired(int Sleep_=0);
    bool ClientReceiveNotRequired(bool InclAtInitial_);
    bool ClientReceiveInProcess();
    bool ClientReceiveCompleted(bool InclNotReq_);

    bool ServerPollingAllowed() const;
    bool ConfirmedServerAlive() const;
    bool ConfirmedClientAlive() const;
    bool ServerAliveConfirmedSent() const;
    bool ExitAtRetryMax() const;
    bool SleepWaiting() const;
    bool WaitUntilResponse() const;
    bool StopTracking() const;
    bool PausePollResults() const;

    bool AtMaxReadAttempts() const;
    bool ReAttemptFileRead();

    bool AtMaxPollAttempts() const;
    bool ReAttemptPolling();
};

/****************************************************************************/
class CalcDriver;

class ExpectedSignalsHandler
{
  protected:
    CalcDriver* _Parent;
  
    int _FromServer;
    int _ClientExpected;
    int _PendingResponse;

    int _FromClient;
    mutable bool _Matched;

  public:
    enum
    {
      FROM_SERVER      = 1,
      CLIENT_EXPECTED  = 2,
      FROM_CLIENT      = 4,
      PENDING_RESPONSE = 8,
      ALL              = 15
    };
  
    ExpectedSignalsHandler(CalcDriver* Parent_);

    bool ResponseReceived(int ResponseSignal_);
    bool Cleared() const;
    bool IsMatching() const;
    bool IsProcessNext() const;
    bool ExpectedStatesSet() const;

    void Reset(int Signals_=ALL);
    void ClearSignalsIfMatched();

    bool MatchesClientExpected(int Signal_, bool InBatch_=false);
    bool MatchesFromServer(int Signal_, bool InBatch_=false);

    inline void SetFromClient(int Signal_)
        { _FromClient = Signal_; }
    inline void SetFromServer(int Signal_)
        { _FromServer = Signal_; }
    inline void SetClientExpected(int Signal_)
        { _ClientExpected |= Signal_; }
    inline void SetPendingResponse(int Signal_)
        { _PendingResponse |= Signal_; }

    inline int GetFromClient() const
        { return _FromClient; }
    inline int GetFromServer() const
        { return _FromServer; }
    inline int GetClientExpected() const
        { return _ClientExpected; }
    inline int PendingResponse() const
        { return _PendingResponse; }

    inline void ResetPendingResponse()
        { Reset(PENDING_RESPONSE); }
};

/****************************************************************************/
class CalcDriver : public QObject
{
  Q_OBJECT
  
  friend class ServerPollingMgr;

  protected:  
    enum
    {
      STOP  = 0,
      START = 1,
    
      LINEINCR = 20,
      MAX_FATAL_ERRORS = 10,
      RETRY_ATTEMPTS = 200,

      SETIOSTATE = 1,
      DOHALT = 2,
      SETSESSIONENTRY = 3,
      RECEIVEKEEPALIVE = 4,
      POLLSERVERIFALIVE = 5,
      READFROMOUTPUT = 6,
      OPENFILE = 7,
      SETCALCELAPSEDTIME = 8,

      IOSTATE = 101,
      SESSION = 102,
      CLIENTALIVE = 103,
      SERVERALIVE = 104,
      POLLSTATE = 105,
      ELAPSEDSTATE = 106
    };

    QTimer _FileOpenRetryTimer;
    QTimer _TerminationTimer;
    QTimer _ReadIoStateTimer;
    QProcess* _ProcPtr;
    
    ServerPollingMgr* _PollServerMgr;
    bool _ServerProgramAlive;

    FILE* _IoStateFout;
    FILE* _SessionFout;
    FILE* _ElapsedStateFout;

    // Server polling methods : start
    FILE* _ServerAliveFile;
    FILE* _ClientAliveFile;
    FILE* _PollStateFile;

    int _WaitCycle;
    int _ServerAliveMode;                        // server alive status file open mode
    int _ClientAliveMode;                        // client alive status file open mode
    int _PollStateMode;                          // current server/client alive poll state file open mode
    int _CalcPollingInactive;                    // Client polling of calculator server is deactivated
    int _SuspendPolling;                         // Temporarily suspend polling operations
    int _ResumePolling;                          // Resume polling operations that was temporarily suspended

    int filestatus_ServerAliveState;
    int filestatus_ClientAliveState;
    int filestatus_PollState;

    size_t _OpenServerAliveFileRetries;
    size_t _OpenClientAliveFileRetries;
    size_t _OpenPollStateFileRetries;
    // Server polling methods : end

    int _AttemptFile;
    const char* _AttemptMode;
    const char* _AttemptFileName;
    int _FromMethod;    
    int _SavedState;
    int _SavedElapsedState;
    long _SavedFilePos;
    size_t _SavedLen;
    int _OutputReady;
    int _ProgramExited;
    bool _ResponseOnly;
    bool _Halted;
    bool _InitProgressBar;
    bool _EndProgressBar;
    bool _ProgressBarInitDone;
    bool _ProgressBarEndDone;
    bool _InputDataFnameDone;    
    bool _OutputDataFnameDone;
    bool _MatchingStates;
    bool _RetryReadIoDone;
    bool _ExpectedSigSent;
    bool _EnterKeyActive;
    
    int _Numer;
    bool _NumerSet;
    int _Denom;
    bool _DenomSet;
    char* _InputDataFname;
    bool _InputFnameSet;
    char* _OutputDataFname;
    bool _OutputFnameSet;
    char* _ReadBuffer;
    bool _ReadOk;
    bool _IoStateRead;
    bool _IostateFileReadError;
    bool _ProgressOutDone;
    bool _CalcOutDone;
    short _InputFileDoneLvl;
    bool _FetchError;
    bool _ServerCalledHalt;

    char* _GrOutputDataFname;
    bool _GraphProgressOutDone;    
    bool _GraphOperationOutDone;
    bool _GrOutputDataFnameDone;
    bool _GrOutputFnameSet;
    int _GrNumer;
    bool _GrNumerSet;
    int _GrDenom;
    bool _GrDenomSet;
    int _GraphPlotOpType;
    int _GraphType;
    int _GraphSubtype;
    bool _GraphOpTypeSet;
    bool _GrProgressFetched;
    bool _GrOperationFetched;
    bool _GrProgressDataShown;
    bool _GrOperationDataShown;
    char* _GrReadBuffer;    
    bool _GrReadOk;

    size_t _OutputBufferSize;
    char* _OutputBuffer;
    int _DoQuit;
    int _QuitAll;
    int _BreakAll;
    int _SessionNum;
    int _Processing;
    int _IoState;
    int _ElapsedState;
    int _PrevIoState;
    int _NoWaitReadCnt;
    int _BatchExecLevel;
    bool _NoPolling;
    bool _InBatchFile;
    bool _EmptyBuffer;
    bool _ResendLastSig;
    bool _SendResendLastSig;
    bool _RequestResend;
    bool _ResendSigDone;
    bool _ResendSigUnchanged;
    bool _AcceptProcessAckDone;
    bool _SendingEndBatchFileAck;

    ExpectedSignalsHandler* _ExpectedSignals;
    ExecutingProgramName* _ExecuteProgram;
    
    int _InputRequested;
    int _FatalErrCount;
    size_t _Executed;
    size_t _Spawned;

    #if CALCDRIVER_DEBUGGING
      static size_t _errline;
      static size_t _errcol;
      static int _errcode;
      static Debugging<CalcDriver>* _dbgptr;
    #endif    

    // default file names
    char* LOG_FILE;                             // Error log file name
    char* SPAWNARG_FILE;                        // Spawn calc. arguments file
    char* CURRENTERROR_FILE;                    // Current error message output file
    char* USER_INPUT_FILE;                      // User input file
    char* USER_PROMPT_FILE;                     // User prompt file
    char* PROG_OUTPUT_FILE;                     // Program output file
    char* PROG_IOSTATE_FILE;                    // Program I/O state file
    char* SERVERALIVE_FILE;                     // Server keep alive state for use in async. mode
    char* CLIENTALIVE_FILE;                     // Client keep alive state for use in async. mode
    char* POLLSTATE_FILE;                       // Current keep alive polling state
    char* PROGRESS_FILE;                        // Completion progress of input data file processing
    char* GRAPH_PROGRESS_FILE;                  // graph plotting progress file
    char* GRAPH_OPERATION_FILE;                 // graph plotting operation data file
    char* ELAPSED_TIME_FILE;                    // elapsed time file used for calc. server timeout notifications

    #if CALCDRIVER_GUI_VERSION
      void SetOutput(QTextEdit* Output_);
    #else
      void SetOutput(const char* Output_);
    #endif

    bool IoStateTimer(int Action_, int Period_=LONG_WAIT);
    bool ReadElapsedTimeState(int ResponseSent_,
                              char* Buffer_=NULL, const char* From_=NULL);
    bool SetPollState(int State_);
    void SetSuspendPolling(bool flag_);
    void SetResumePolling(bool flag_);

    void SetSessionEntry(long Fpos_, size_t Len_);
    void WriteToInput(const char* Str_);
    bool IsMatchingStates(int PrevState_, int CurState_,
                          bool& DoWait_, bool& NoWait_);
    int WaitForOutput(int delay_=0, int retry_=RETRY_READ_MAX,
                      int* ReadyToSendp_=NULL);
    void EvalOutput();
    void HandleProgressData();
    void HandleGraphProgressData();
    void ShowResultsFile();
    void ShowGraphProgressData();
    void RunCalcServerCalledHalt();

    void GetFilenames();
    long FileSize(FILE* Fptr_);
    void GetSessionNumbers(FILE* Fptr_, int** SesNumArray_, int Max_, int& NumSes_);
    int HasSessionNumber(int* Array_, int NumSesNumbers_, int& Number_);    
    int FindUnusedEntry(FILE** SesFptr_);
    void ResizeOutputBuffer(size_t newsize_=0);
    void InitializeFiles();
    CalcDriver* SetNoPolling(bool flag_);
    int NoPolling() const;

    int icasestrcompn(const char* s1, const char* s2, size_t max, bool SkipWs_=false);
    int icasestrcomp(const char* s1, const char* s2, bool SkipWs_=false);

    char* RemovePadSpace(char* str);
    const char* IsExecCmd(char* Str_);
    const char* ProgramName(char* Name_);

    inline void SetEmptyBuffer(bool flag_)
        { _EmptyBuffer = flag_; }
    inline bool EmptyBuffer() const
        { return _EmptyBuffer; }

    int SimulatePolling();
    int EvalExpression(const char* Buffer_, int InputState_, bool ExecCmd_);

  protected slots:
    // Timer time-out methods
    void RetryReadIoState(); // connected to _ReadIoStateTimer
    void RetryFileOpen();    // connected to _FileOpenRetryTimer
    void ResetSession();     // connected to _TerminationTimer
    void ProcessExited();
    
  public slots:
    void ShowProgress();
    void ShowGraphProgress();
    void ShowOutput();
    void SendToInput();
    void DoHalt();

  public:
    enum
    {
      CALCDRIVER       = 8,
      SERVERPOLLINGMGR = 16,
      CLIENTPOLLINGMGR = 32
    };
  
    #if CALCDRIVER_DEBUGGING
      static size_t GetErrline() { return _errline; }
      static size_t GetErrcol() { return _errcol; }
      static int GetErrcode() { return _errcode; }
      static void SetCurrentDebugger(Debugging<CalcDriver>* ptr);
      static Debugging<CalcDriver>* DbgPtr(bool Deref_=true);
      static void SetErrorData(int errcode_, const char* file_=__FILE__,
                               size_t line_=__LINE__, size_t col_=0, bool terminate_=false);
      static void Terminate(bool setcode_=false, int errcode_=0);
    #endif
  
    #if CALCDRIVER_GUI_VERSION
      QTextEdit* _Output;
    #else
      char* _Output;
      char* _Input;
    #endif
  
    CalcDriver();
    CalcDriver(const char* Name_, int UseNewCalc_=TRUE, int SessionNum_=0);
    virtual ~CalcDriver();

    static const char* IoStateToStr(int State_);
    static const char* TimeoutStateToStr(int State_);
    static char* NewString(const char* src);

    static bool IsClientStates(int State_);
    static bool IsServerStates(int State_);
    static bool LongWaitForServerSend(int State_);
    static bool WaitingForServerSend(int State_);
    static bool IsIntermediateStates(int State_);
    static bool IsGuiReadyToSend(int State_,
                                 bool IncludeIdle_=true,
                                 bool IncludeErrorFetch_=true);
    static bool IsProcessAckDone(int State_);
    static bool IsGuiNeedToProcessSignals(int State_);
    static bool IsCalcOnlySignals(int State_);
    static bool IsCalcResponseSignals(int State_);

    bool SuspendPolling();
    bool ResumePolling();
    void PauseCalcPolling();
    void ResumeCalcPolling();
    bool CalcPollingActive();

    // File name retrieval
    const char* GetLogFilename() const;
    const char* GetSpawnArgFilename() const;
    const char* GetCurrentErrorFilename() const;
    const char* GetUserInputFilename() const;
    const char* GetUserPromptFilename() const;
    const char* GetProgramOutputFilename() const;
    const char* GetProgramIostateFilename() const;
    const char* GetProgressFilename() const;
    const char* GetGraphProgressFilename() const;
    const char* GetGraphOperationFilename() const;
    const char* GetServerAliveFilename() const;
    const char* GetClientAliveFilename() const;
    const char* GetPollStateFilename() const;
    const char* GetCalcElapsedTimeFilename() const;

    int GetIoState(int Sleep_);
    int GetCalcElapsedTimeState(int Sleep_);
    void SetCalcElapsedTimeState(int State_, const char* From_=NULL);
    void SetIoState(int State_,
                    bool SetStateOnly_=false, bool IgnoreDup_=false);
    inline void SetAcceptProcessAckDone(bool flag_)
        { _AcceptProcessAckDone = flag_; }
    inline int Bare_GetIoState(int Sleep_)
        { return GetIoState(Sleep_); }
    int ReadIoState(int delay_=0, int retry_=RETRY_READ_MAX,
                    int* AltGuiStates_=NULL, int* CalcOnlyStates_=NULL);

    void FetchProgressData();
    void FetchGraphProgressData();
    void FetchGraphOperationData();
    void SetupDataFiles();
    void Execute();
    void SpawnNewMcalc();
    void ClearInput();
    void ClearOutput();
    void ReadFromOutput(bool SetNewState_,
                        int CurrentState_, int PreviousState_=0);

    void StopAllTimers();
    void IncrBatchLevel();
    void DecrBatchLevel();
    void SetInBatchFile(bool ExecFound_, bool Reset_);
    void PendingProcessSignal();
    void PendingOutputFetched();
    void PendingInputReceived();
    void AckErrorFetched();
    void SendMcalcSpawned(bool* Received_=NULL);
    void SendEndBatchFileAcked(bool* Received_=NULL);
    void SendInputReceived(bool* Received_=NULL);
    void SendClientProcessSignal(bool* Received_=NULL);
    void SendProcessDoneAcked(bool* Received_=NULL);
    void SendOutputFetched(bool* Received_=NULL);
    void SendProgressFetched(bool* Received_=NULL);
    void SendGraphWaitFetched(bool* Received_=NULL);
    void SendGraphProgressFetched(bool* Received_=NULL);
    void SendProgressReady();
    void SendResendLastSig();

    bool ShouldEnableEnter();
    bool HasExpectedStates();
    int SendUserResponse(const QString& Buffer_, bool ExecCmd_=false);

    // Server polling methods : start
    void SendServerAlive(bool flag_);
    void ShowServerPollFailed(bool Exit_);
    bool PollServerIfAlive(int Sleep_);

    int openfile(FILE** file, const char* name, const char* mode,
                 int Retry_=FALSE, int ExitOnFail_=TRUE,
                 int MaxRetries_=RETRY_READ_MAX);
    
    FILE* GetClientAliveFile(int Write_, bool Rewind_=false);
    FILE* GetServerAliveFile(int Write_, bool Rewind_=false);
    FILE* GetPollStateFile(int Write_, bool Rewind_=false);

    void CloseClientAliveFile();
    void CloseServerAliveFile();
    void ClosePollStateFile();

    bool InAsyncMode() const
        { return true; }

    bool ConfirmedServerAlive() const
        { return _ServerProgramAlive; }
    bool ConfirmedClientAlive() const
        { return (_PollServerMgr ? _PollServerMgr->ConfirmedClientAlive():false); }

    bool AtInitialState()
        { return (_PollServerMgr ? _PollServerMgr->AtInitialState():false); }
    bool ClientIsPolling() const
        { return (_PollServerMgr ? _PollServerMgr->ClientIsPolling():false); }
    bool ServerIsAlive() const
        { return (_PollServerMgr ? _PollServerMgr->ServerIsAlive():false); }
    bool ClientAcked() const
        { return (_PollServerMgr ? _PollServerMgr->ClientAcked():false); }
    bool ResetRequested() const
        { return (_PollServerMgr ? _PollServerMgr->ResetRequested():false); }
    bool PollingReset() const
        { return (_PollServerMgr ? _PollServerMgr->PollingReset():false); }
    bool ResetSignalSent()
        { return (_PollServerMgr ? _PollServerMgr->ResetSignalSent():false); }
    bool PollingResetByServer()
        { return (_PollServerMgr ? _PollServerMgr->PollingResetByServer():false); }

    bool PollServerRequired(int Sleep_=0, bool ResponseOnly_=true)
        { return (_PollServerMgr ? _PollServerMgr->PollServerRequired(Sleep_, ResponseOnly_):false); }
    bool PollServerNotRequired(bool InclAtInitial_) const
        { return (_PollServerMgr ? _PollServerMgr->PollServerNotRequired(InclAtInitial_):false); }
    bool PollServerInProcess() const
        { return (_PollServerMgr ? _PollServerMgr->PollServerInProcess():false); }
    bool PollServerCompleted(bool InclNotReq_) const
        { return (_PollServerMgr ? _PollServerMgr->PollServerCompleted(InclNotReq_):false); }
    // Server polling methods : end

    void SetFromMethod(int Method_,
                       bool SetResponse_=false,
                       bool ResponseOnly_=true);

    int QuitAll() const
        { return _QuitAll; }
    int BreakAll() const
        { return _BreakAll; }
        
    int IsProcessing() const
        { return _Processing; }    
    int IsOutputReady() const
        { return _OutputReady; }
    int IsProgramExited() const
        { return _ProgramExited; }
    int IoState() const
        { return _IoState; }
    int ElapsedState() const
        { return _ElapsedState; }
    int SessionNum() const
        { return _SessionNum; }

    bool WaitingForServerResponse() const
        { return _ReadIoStateTimer.isActive(); }
    bool InBatchFile() const
        { return (_InBatchFile &&
                  ((_ExecuteProgram && _ExecuteProgram->Level()) ||
                   _BatchExecLevel)); }
    bool InInputFile() const
        { return _ExecuteProgram ? _ExecuteProgram->InInputFile():false; }
    bool InProgramFile() const
        { return _ExecuteProgram ? _ExecuteProgram->InProgramFile():false; }
    bool InProgressDataFile() const
        { return (!_ProgressBarEndDone &&
                  _NumerSet && _Numer && _DenomSet && _Denom); }

  signals:
    void DoQuit();
    void DisableEnterKey();
    void EnableEnterKey();
    void CalcServerTimeout();
    void ProcessSignal();
    void ProcessAcked();
    void InputNeeded();
    void OutputReady();
    void FatalError();
    void EndOfBatchData();
    void GraphProgress();
    void GraphOutput();
    void GraphWait();
    void ErrorFound();
    void ProgressReady();
    void CreateNewMcalc();
    void ServerAliveConfirmed();
    void SendCalcProgress(int);
    void EndCalcProgress();
    void SendTotalSteps(int);
    void SendCalcProgressLabel(const QString&);
    void SendInputFilename(const QString&);
    void SendOutputFilename(const QString&);
    void ResultsFileReady(const QString&, const QString&);
};

/****************************************************************************/
#endif


