#ifndef DEBUGUTILS_H
#define DEBUGUTILS_H
#ifndef INCL_STDIO_H
  #include <stdio.h>
  #define INCL_STDIO_H
#endif
#ifndef INCL_STDLIB_H
  #include <stdlib.h>
  #define INCL_STDLIB_H
#endif
#ifndef INCL_STRING_H
  #include <string.h>
  #define INCL_STRING_H
#endif

#define ERRMSG_NULLDEBUGPTR "Error: Null Debug Class Object Pointer\n"

struct DebuggingLevelStack
{
  char* _desc;
  int _level;
  DebuggingLevelStack* _next;
      
  DebuggingLevelStack(const char* desc_, int level_):
    _desc(desc_ ? ::strcpy(new char[strlen(desc_) + 1], desc_):NULL),
    _level(level_),
    _next(NULL) {}
  
  ~DebuggingLevelStack()
  {
    delete[] _desc;
    _desc = NULL;
  
    delete _next;
    _next = NULL;
  }
      
  DebuggingLevelStack* Push(DebuggingLevelStack* Old_)
  {
    _next = Old_;
    return this;
  }
  
  DebuggingLevelStack* Pop(DebuggingLevelStack*& New_)
  {
    New_ = _next;
    _next = NULL; 
    return this;
  }
};

template <class T>
class Debugging
{
  public:
    enum
    {
      NULL_PTR     = 1,
      NON_NULL_PTR = 2,
      
      INDENTLENGTH = 2
    };
    
  private:        
    T* _base;
    size_t _linenum;
    size_t _colnum;
    int _ErrCode;
    const char* _ErrFile;
    
    FILE* _fp;    
    char* _buffer;
    size_t _bufsz;
    DebuggingLevelStack* _Level;
    size_t _Levels;
    int _StopTracking;
    int _TrackingType;

    bool _ShowError;
    bool _RepeatInStdout;
    bool _RepeatInStderr;
    
    static Debugging<T>* _CurrentDebugger;

    int DecimalPointsToPrec(double Val_, int Pts_);
    char* ConvertFloatToStr(double value, char* result, int width, int prec, char fmt='g');
    void ShowVar(void* rawdata_, const char* Name_, const char* fmtspec_, const char* FormatStr_);
    void CheckArgLengths(const char* Name_, const char* FormatStr_=NULL);
    char* FlexResizeBuffer(size_t newsize_);
    void IndentLevel();
    void ShowLevel();
  
  public:
    Debugging(T* base_);
    ~Debugging();
    
    FILE* SetFilename(const char* Fname_);
    void SetBasePtr(T* base_);
    
    T* BasePtr();
    const T* BasePtr() const;

    char* ShowMessage(const char* msg_);
    char* ShowStr(const char* data_, const char* Name_, const char* FormatStr_=NULL);
    char* ShowChar(char data_, const char* Name_, const char* FormatStr_=NULL);
    char* ShowInt(long data_, const char* Name_, const char* FormatStr_=NULL);
    char* ShowFloat(double data_, const char* Name_, const char* FormatStr_=NULL);
    char* ShowAddr(void* data_, const char* Name_, const char* FormatStr_=NULL);
    
    bool PtrCheck(void* ptr_, bool NotNull_,
                  bool SetFile_=false, const char* Fname_=NULL,
                  bool SetPos_=false, size_t row_=0, size_t col_=0);
    bool ShowPtrCheck(void* ptr_, const char* Name_, bool NotNull_,
                      bool SetFile_=false, const char* Fname_=NULL,
                      bool SetPos_=false, size_t row_=0, size_t col_=0);

    void ResetLevels(bool ShowRemaining_=false);
    void EnterLevel(const char* str_, int Type_=0);    
    void LeaveLevel();

    Debugging<T>* SetShowError(bool Flag_);
    Debugging<T>* SetRepeatInStdout(bool Flag_);
    Debugging<T>* SetRepeatInStderr(bool Flag_);
    
    Debugging<T>* SetTrackType(int Type_);
    Debugging<T>* StopTrackingAll();
    Debugging<T>* StartTrackingAll();
    Debugging<T>* SetStopTracking(int Type_, bool Flag_);
    
    int StopTracking(int Type_=0) const;
    inline bool ShowError() const
        { return _ShowError; }
    inline bool RepeatInStdout() const
        { return _RepeatInStdout; }
    inline bool RepeatInStderr() const
        { return _RepeatInStderr; }
    
    static void SetCurrentDebugger(Debugging<T>* ptr);
    static Debugging<T>* CurrentDebugger(bool Deref_=true);
};

#define IN_DEBUGGING_MODE             0

#include "debugmacros.h"

template <class T>
Debugging<T>* Debugging<T>::_CurrentDebugger = NULL;

/****************************************************************************/
template <class T>
Debugging<T>::Debugging(T* base_):
_base(base_),
_linenum(0),
_colnum(0),
_ErrCode(0),
_ErrFile(NULL),

_fp(NULL),
_buffer(new char[128]),
_bufsz(128),
_Level(NULL),
_Levels(0),

_StopTracking(0),
_TrackingType(0),

_ShowError(true),
_RepeatInStdout(false),
_RepeatInStderr(false)
{}

/****************************************************************************/
template <class T>
Debugging<T>::~Debugging()
{
  fclose(_fp);
  delete[] _buffer;
  
  delete _Level;
  _Level = NULL;
}

/****************************************************************************/
template <class T>
void Debugging<T>::SetCurrentDebugger(Debugging<T>* ptr)
{
  _CurrentDebugger = ptr;
}

/****************************************************************************/
template <class T>
Debugging<T>* Debugging<T>::CurrentDebugger(bool Deref_)
{
  if (Deref_ && !_CurrentDebugger)
    fputs(ERRMSG_NULLDEBUGPTR, stderr);
    
  return _CurrentDebugger;
}

/****************************************************************************/
template <class T>
T* Debugging<T>::BasePtr()
{
  return _base;
}

/****************************************************************************/
template <class T>
const T* Debugging<T>::BasePtr() const
{
  return _base;
}

/****************************************************************************/
template <class T>
void Debugging<T>::SetBasePtr(T* base_)
{
  if (base_)
    _base = base_;
}

/****************************************************************************/
template <class T>
FILE* Debugging<T>::SetFilename(const char* Fname_)
{
  if (Fname_ && strlen(Fname_))
  {
    _fp = fopen(Fname_, "w");
    return _fp;
  }
  
  return NULL;
}

/****************************************************************************/
template <class T>
char* Debugging<T>::FlexResizeBuffer(size_t newsize_)
{
  size_t bufsize_ = _bufsz;
  
  if (newsize_)
  {
    if (newsize_ < _bufsz)
      return _buffer;
    
    if (newsize_ > _bufsz + 64)
      bufsize_ = newsize_ + 5;
    else
      bufsize_ += 128;
  }
  else
    bufsize_ += 128;
  
  char* oldbuf_ = _buffer;
  _buffer = new char[bufsize_];
  ::memset(_buffer, 0, bufsize_);
  ::memmove(_buffer, oldbuf_, _bufsz);
  _bufsz = bufsize_;
  
  return _buffer;
}

/****************************************************************************/
template <class T>
void Debugging<T>::CheckArgLengths(const char* Name_, const char* FormatStr_)
{
  size_t len = 0;
  
  if (Name_)
    len += strlen(Name_);
  
  if (FormatStr_)
    len += strlen(FormatStr_);
  
  if (len + 32 >= _bufsz)
    FlexResizeBuffer(len + 32);
}

/****************************************************************************/
template <class T>
bool Debugging<T>::PtrCheck(void* ptr_, bool NotNull_,
                            bool SetFile_, const char* Fname_,
                            bool SetPos_, size_t row_, size_t col_)
{
  if (SetFile_)
    _ErrFile = Fname_;
  
  if (SetPos_)
  {
    _linenum = row_;
    _colnum = col_;
    _ErrCode = 0;
  }

  if (!ptr_ && NotNull_)
    if (_base)
    {
      _ErrCode = NULL_PTR;
      _base->SetErrorData(_ErrCode, _ErrFile, _linenum, _colnum);
      return false;
    }
    
  if (ptr_ && !NotNull_)
    if (_base)
    {
      _ErrCode = NON_NULL_PTR;
      _base->SetErrorData(_ErrCode, _ErrFile, _linenum, _colnum);
      return false;
    }
    
  return true;
}

/****************************************************************************/
template <class T>
bool Debugging<T>::ShowPtrCheck(void* ptr_, const char* Name_, bool NotNull_,
                                bool SetFile_, const char* Fname_,
                                bool SetPos_, size_t row_, size_t col_)
{
  if (SetFile_)
    _ErrFile = Fname_;
  
  if (SetPos_)
  {
    _linenum = row_;
    _colnum = col_;    
    _ErrCode = 0;
  }
  
  if (!ptr_ && NotNull_)
    if (_base && !StopTracking())
    {
      strcpy(_buffer, "Ptr Check: ");
      strcat(_buffer, Name_);
      strcat(_buffer, " : null (error)\n");
      fputs(_buffer, _fp);
      if (_RepeatInStdout)
        fputs(_buffer, stdout);
      if (_RepeatInStderr)
        fputs(_buffer, stderr);

      _ErrCode = NULL_PTR;
      _base->SetErrorData(_ErrCode, _ErrFile, _linenum, _colnum);
      return false;
    }
    
  if (ptr_ && !NotNull_ && !StopTracking())
    if (_base)
    {
      strcpy(_buffer, "Ptr Check: ");
      strcat(_buffer, Name_);
      strcat(_buffer, " : not null (error)\n");
      fputs(_buffer, _fp);
      if (_RepeatInStdout)
        fputs(_buffer, stdout);
      if (_RepeatInStderr)
        fputs(_buffer, stderr);

      _ErrCode = NON_NULL_PTR;
      _base->SetErrorData(_ErrCode, _ErrFile, _linenum, _colnum);
      return false;
    }
    
  return true;
}

/****************************************************************************/
template <class T>
char* Debugging<T>::ShowMessage(const char* msg_)
{
  if (!_fp)
    return NULL;

  if (StopTracking())
    return _buffer;

  if (_Levels)
    IndentLevel();
  
  fputs(msg_, _fp);
  fflush(_fp);

  if (_RepeatInStdout)
    fputs(msg_, stdout);
  if (_RepeatInStderr)
    fputs(msg_, stderr);
  
  strcpy(_buffer, msg_);
  return _buffer;
}

/****************************************************************************/
template <class T>
void Debugging<T>::ShowVar(void* rawdata_, const char* Name_, const char* fmtspec_, const char* FormatStr_)
{
  if (!_fp)
    return;
  
  CheckArgLengths(Name_, FormatStr_);
  char* fmtstr_ = NULL;
  
  char* cdata = NULL;
  long* ldata = NULL;
  char* sdata = NULL;
  double* fdata = NULL;
  void* adata = NULL;
  
  bool nulldata_ = false;
  bool floatdata_ = false;
  
  if (strcmp(fmtspec_, "%c") == 0)
  {
    cdata = (char*)rawdata_;
    nulldata_ = !cdata;
  }
  else if (strcmp(fmtspec_, "%d") == 0)
  {
    ldata = (long*)rawdata_;
    nulldata_ = !ldata;
  }
  else if (strcmp(fmtspec_, "%s") == 0)
  {
    sdata = (char*)rawdata_;
    nulldata_ = !sdata;
  }
  else if (strcmp(fmtspec_, "%p") == 0)
  {
    adata = rawdata_;
    nulldata_ = !adata;
  }
  else if (strcmp(fmtspec_, "%lg") == 0)
  {
    fdata = (double*)rawdata_;
    nulldata_ = !fdata;
    floatdata_ = true;
  }
    
  if (Name_ && !StopTracking())
  {
    if (floatdata_)
    {
      if (!nulldata_)
      {
        _buffer = ConvertFloatToStr(*fdata, _buffer, 8, 2);
        fmtstr_ = strdup(_buffer);
      }
      
      strcpy(_buffer, Name_);
      strcat(_buffer, ": ");
      if (nulldata_)
        strcat(_buffer, "(null)");
      else
        strcat(_buffer, fmtstr_);
      strcat(_buffer, "\n");
      
      if (_Levels)
        IndentLevel();
      
      fputs(_buffer, _fp);
      if (_RepeatInStdout)
        fputs(_buffer, stdout);
      if (_RepeatInStderr)
        fputs(_buffer, stderr);
      
      if (!nulldata_)
        free(fmtstr_);
    }
    else
    {
      strcpy(_buffer, Name_);
      strcat(_buffer, ": ");
      if (nulldata_)
        strcat(_buffer, "(null)");
      else
        strcat(_buffer, fmtspec_);
      strcat(_buffer, "\n");
      
      if (!nulldata_)
      {
        fmtstr_ = strdup(_buffer);

        if (cdata)
          sprintf(_buffer, fmtstr_, *cdata);
        else if (ldata)
          sprintf(_buffer, fmtstr_, *ldata);
        else if (sdata)
          sprintf(_buffer, fmtstr_, sdata);
        else if (adata)
          sprintf(_buffer, fmtstr_, adata);
      }

      if (_Levels)
        IndentLevel();
      
      fputs(_buffer, _fp);   
      if (_RepeatInStdout)
        fputs(_buffer, stdout);
      if (_RepeatInStderr)
        fputs(_buffer, stderr);

      if (!nulldata_)
        free(fmtstr_);
    }
  }
  else if (!nulldata_ && !StopTracking())
  {
    if (floatdata_)
    {    
      _buffer = ConvertFloatToStr(*fdata, _buffer, 8, 2);
      fputs(_buffer, _fp);
      if (_RepeatInStdout)
        fputs(_buffer, stdout);
      if (_RepeatInStderr)
        fputs(_buffer, stderr);
    }    
    else if (FormatStr_)
    {
      if (cdata)
        sprintf(_buffer, FormatStr_, *cdata);
      else if (ldata)
        sprintf(_buffer, FormatStr_, *ldata);
      else if (sdata)
        sprintf(_buffer, FormatStr_, sdata);
      else if (adata)
        sprintf(_buffer, FormatStr_, adata);
      
      if (_Levels)
        IndentLevel();
      
      fputs(_buffer, _fp);
      if (_RepeatInStdout)
        fputs(_buffer, stdout);
      if (_RepeatInStderr)
        fputs(_buffer, stderr);
    }
  }
  
  fflush(_fp);
}

/****************************************************************************/
template <class T>
char* Debugging<T>::ShowStr(const char* data_, const char* Name_, const char* FormatStr_)
{
  ShowVar((void*)data_, Name_, "%s", FormatStr_);
  return ((_buffer && strlen(_buffer)) ? _buffer:NULL);
}

/****************************************************************************/
template <class T>
char* Debugging<T>::ShowChar(char data_, const char* Name_, const char* FormatStr_)
{
  ShowVar(&data_, Name_, "%c", FormatStr_);
  return ((_buffer && strlen(_buffer)) ? _buffer:NULL);
}

/****************************************************************************/
template <class T>
char* Debugging<T>::ShowInt(long data_, const char* Name_, const char* FormatStr_)
{
  ShowVar(&data_, Name_, "%d", FormatStr_);
  return ((_buffer && strlen(_buffer)) ? _buffer:NULL);
}

/****************************************************************************/
template <class T>
char* Debugging<T>::ShowFloat(double data_, const char* Name_, const char* FormatStr_)
{
  ShowVar(&data_, Name_, "%lg", FormatStr_);
  return ((_buffer && strlen(_buffer)) ? _buffer:NULL);
}

/****************************************************************************/
template <class T>
char* Debugging<T>::ShowAddr(void* data_, const char* Name_, const char* FormatStr_)
{
  ShowVar(data_, Name_, "%p", FormatStr_);
  return ((_buffer && strlen(_buffer)) ? _buffer:NULL);
}

/****************************************************************************/
// Converts desired decimal points to the precision value that is used in
// C printf functions.
//
template <class T>
int Debugging<T>::DecimalPointsToPrec(double Val_, int Pts_)
{
  int x;
  long Dval_ = (long)Val_;

  for (x = 0; Dval_; x++)
    Dval_ /= 10;

  Pts_ += x;
  return Pts_;
}

/****************************************************************************/
template <class T>
char* Debugging<T>::ConvertFloatToStr(double value, char* result, int width, int prec, char fmt)
{
  if (!result)
    result = new char[128];

  char fmtstr[10];
  strcpy(fmtstr, "%-*.*");
  fmtstr[5] = fmt;
  fmtstr[6] = 0;

  if (fmt == 'f' || fmt =='g')
  {
    if (prec < 0)
      prec = 0;

    if (width < 0)
      width = 0;

    int oldv = prec;
    if (fmt == 'g')
      prec = DecimalPointsToPrec(value, prec);

    if (!width)
      width = oldv ? (prec + 1):prec;
  }
  else if (!width)
    width = prec ? (prec + 2):DecimalPointsToPrec(value, 0);
  
  sprintf(result, fmtstr, width, prec, value);
  return result;
}

/****************************************************************************/
template <class T>
void Debugging<T>::IndentLevel()
{
  if (_Level && !StopTracking())
  {
    int x;
    int max;
    char* indent_ = new char[INDENTLENGTH + 5];
    char* buffer_ = strcpy(new char[_Level->_level + strlen(_Level->_desc) + 64], "   ");
    
    x = strlen(buffer_);
    sprintf(&buffer_[3], "%d", _Level->_level);
    max = strlen(buffer_);
    
    for (max += _Levels; x < max; x++)
      buffer_[x] = ' ';

    buffer_[max] = 0;
    ::memset(indent_, ' ', INDENTLENGTH);
    indent_[INDENTLENGTH] = 0;
    strcat(buffer_, indent_);
    
    fputs(buffer_, _fp);
    fflush(_fp);

    if (_RepeatInStdout)
      fputs(buffer_, stdout);
    if (_RepeatInStderr)
      fputs(buffer_, stderr);
    
    delete[] buffer_;
    delete[] indent_;
  }
}

/****************************************************************************/
template <class T>
void Debugging<T>::EnterLevel(const char* str_, int Type_)
{
  if (str_)
  {
    if (Type_)
      _TrackingType = Type_;
  
    ++_Levels;
    if (!_Level)
      _Level = new DebuggingLevelStack(str_, _Levels);
    else
    {
      DebuggingLevelStack* New_ = new DebuggingLevelStack(str_, _Levels);
      _Level = New_->Push(_Level);
    }

    if (!StopTracking())
    {
      int x;
      int max;
      char* buffer_ = strcpy(new char[_Levels + strlen(str_) + 64], "lv:");
      sprintf(&buffer_[3], "%d", _Levels);
      x = max = strlen(buffer_);
    
      for (max += _Levels; x < max; x++)
        buffer_[x] = '-';
      
      buffer_[max] = 0;
      strcat(buffer_, "Entering: ");
      strcat(buffer_, str_);
      strcat(buffer_, "\n");
    
      fputs(buffer_, _fp);
      fflush(_fp);
      
      if (_RepeatInStdout)
        fputs(buffer_, stdout);
      if (_RepeatInStderr)
        fputs(buffer_, stderr);
    
      delete[] buffer_;
    }
  }
}

/****************************************************************************/
template <class T>
void Debugging<T>::LeaveLevel()
{
  if (_Levels)
  {
    if (_Level && !StopTracking())
    {
      int x;
      int max;
      char* buffer_ = strcpy(new char[_Level->_level + strlen(_Level->_desc) + 64], "lv:");
      sprintf(&buffer_[3], "%d", _Level->_level);
      x = max = strlen(buffer_);
      
      for (max += _Levels; x < max; x++)
        buffer_[x] = '-';
    
      buffer_[max] = 0;
      strcat(buffer_, "Leaving: ");
      strcat(buffer_, _Level->_desc);
      strcat(buffer_, "\n");
    
      fputs(buffer_, _fp);
      fflush(_fp);

      if (_RepeatInStdout)
        fputs(buffer_, stdout);
      if (_RepeatInStderr)
        fputs(buffer_, stderr);
      
      delete[] buffer_;
    }    
    
    --_Levels;    
    DebuggingLevelStack* New_;
    DebuggingLevelStack* Old_ = _Level->Pop(New_);
    delete Old_;
    _Level = New_;    
  }
}

/****************************************************************************/
template <class T>
void Debugging<T>::ResetLevels(bool ShowRemaining_)
{
  if (ShowRemaining_)
    StartTrackingAll();
  else
    StopTrackingAll();

  while (_Levels)
  {
    if (_Level && !StopTracking())
    {
      int x;
      int max;
      char* buffer_ = strcpy(new char[_Level->_level + strlen(_Level->_desc) + 64], "lv:");
      sprintf(&buffer_[3], "%d", _Level->_level);
      x = max = strlen(buffer_);
      
      for (max += _Levels; x < max; x++)
        buffer_[x] = '-';
    
      buffer_[max] = 0;
      strcat(buffer_, "Leaving: ");
      strcat(buffer_, _Level->_desc);
      strcat(buffer_, "\n");
    
      fputs(buffer_, _fp);
      fflush(_fp);

      if (_RepeatInStdout)
        fputs(buffer_, stdout);
      if (_RepeatInStderr)
        fputs(buffer_, stderr);
      
      delete[] buffer_;
    }    

    --_Levels;    
    DebuggingLevelStack* New_;
    DebuggingLevelStack* Old_ = _Level->Pop(New_);
    delete Old_;
    _Level = New_;
  }
}

/****************************************************************************/
template <class T>
Debugging<T>* Debugging<T>::SetShowError(bool Flag_)
{
  _ShowError = Flag_;
  return this;
}

/****************************************************************************/
template <class T>
Debugging<T>* Debugging<T>::SetRepeatInStdout(bool Flag_)
{
  _RepeatInStdout = Flag_;
  return this;
}

/****************************************************************************/
template <class T>
Debugging<T>* Debugging<T>::SetRepeatInStderr(bool Flag_)
{
  _RepeatInStderr = Flag_;
  return this;
}

/****************************************************************************/
template <class T>
Debugging<T>* Debugging<T>::SetTrackType(int Type_)
{
  _TrackingType = Type_;
  return this;
}

/****************************************************************************/
template <class T>
Debugging<T>* Debugging<T>::StartTrackingAll()
{
  _StopTracking = 0;
  return this;
}

/****************************************************************************/
template <class T>
Debugging<T>* Debugging<T>::StopTrackingAll()
{
  _StopTracking = (~(1 << (sizeof (int) * 8 - 1)));
  return this;
}

/****************************************************************************/
template <class T>
Debugging<T>* Debugging<T>::SetStopTracking(int Type_, bool Flag_)
{
  if (Flag_)
    _StopTracking |= Type_;
  else
    _StopTracking &= ~Type_;

  return this;
}

/****************************************************************************/
template <class T>
int Debugging<T>::StopTracking(int Type_) const
{
  return (Type_ ? (_StopTracking & Type_):
                  (_StopTracking & _TrackingType));
}

/****************************************************************************/
#endif

